/*
 *
 *		Linux MegaRAID driver for SAS based RAID controllers
 *
 * Copyright (c) 2003-2005  LSI Logic Corporation.
 *
 *     This program is free software; you can redistribute it and/or
 *	   modify it under the terms of the GNU General Public License
 *	   as published by the Free Software Foundation; either version
 *	   2 of the License, or (at your option) any later version.
 *
 * FILE		: megaraid_sas.c
 * Version	: v00.00.03.19
 *
 * Authors:
 *	(email-id : megaraidlinux@lsi.com)
 * 	Sreenivas Bagalkote
 * 	Sumant Patro
 *	Bo Yang
 *
 * List of supported controllers
 *
 * OEM	Product Name			VID	DID	SSVID	SSID
 * ---	------------			---	---	----	----
 */
#include <linux/kernel.h>
#include <linux/types.h>
#include <linux/pci.h>
#include <linux/list.h>
#include <linux/version.h>
#include <linux/module.h>
#include <linux/spinlock.h>
#include <linux/interrupt.h>
#include <linux/delay.h>
#include <linux/uio.h>
#include <asm/uaccess.h>

#include <linux/module.h>
#include <linux/reboot.h>
#include <linux/proc_fs.h>
#include <linux/blk.h>
#include <linux/notifier.h>

#if defined (__x86_64__)
#include <asm/ioctl32.h>
#endif
#ifdef __VMKERNEL_MODULE__
#include <linux/poll.h>
#endif

#include "scsi.h"
#include "hosts.h"
#include "sd.h"

#include "megaraid_sas.h"

/*
 * Modules parameters
 */

/*
 * Fast driver load option, skip scanning for physical devices during load.
 * This would result in physical devices being skipped during driver load
 * time.
 */
static unsigned int fast_load = 0;
MODULE_PARM(fast_load, "i");
MODULE_PARM_DESC(fast_load,
	"megasas: Faster loading of the driver, skips physical devices! \
	 (default=0)");

/*
 * Number of sectors per IO command
 * Will be set in megasas_init_mfi if user does not provide
 */
static unsigned int max_sector = 0;
MODULE_PARM(max_sector, "i");
MODULE_PARM_DESC(max_sector,
	"Maximum number of sectors per IO command");

/*
 * Number of cmds per logical unit
 */
static unsigned int cmd_per_lun = MEGASAS_DEFAULT_CMD_PER_LUN;
MODULE_PARM(cmd_per_lun, "i");
MODULE_PARM_DESC(cmd_per_lun,
	"Maximum number of commands per logical unit (default=128)");

/*
 * poll_mode_io:1- schedule complete completion from q cmd
 */
static unsigned int poll_mode_io;
MODULE_PARM(poll_mode_io, "i");
MODULE_PARM_DESC(poll_mode_io,
	"Complete cmds from IO path, (default=0)");


#ifdef __VMKERNEL_MODULE__
spinlock_t io_request_lock;
static DECLARE_WAIT_QUEUE_HEAD ( megasas_poll_wait );
static int megasas_wait_aen_poll;
spinlock_t aen_poll_lock;
#endif

MODULE_LICENSE("GPL");
#ifdef MODULE_VERSION
MODULE_VERSION(MEGASAS_VERSION);
#endif
MODULE_AUTHOR("megaraidlinux@lsi.com");
MODULE_DESCRIPTION("LSI Logic MegaRAID SAS Driver");

static const char *megasas_info(struct Scsi_Host *);
static void msleep(unsigned long);

/*
 * PCI ID table for all supported controllers
 */
static struct pci_device_id megasas_pci_table[] = {

	{
	 PCI_VENDOR_ID_LSI_LOGIC,
	 PCI_DEVICE_ID_LSI_SAS1064R, // xscale IOP
	 PCI_ANY_ID,
	 PCI_ANY_ID,
	 },
	{
	 PCI_VENDOR_ID_LSI_LOGIC,
	 PCI_DEVICE_ID_LSI_SAS1078R, // ppc IOP
	 PCI_ANY_ID,
	 PCI_ANY_ID,
	},
        {
         PCI_VENDOR_ID_LSI_LOGIC,
         PCI_DEVICE_ID_LSI_SAS1078DE, // ppc IOP
         PCI_ANY_ID,
         PCI_ANY_ID,
        },
	{
	 PCI_VENDOR_ID_LSI_LOGIC,
	 PCI_DEVICE_ID_LSI_VERDE_ZCR,	// vega
	 PCI_ANY_ID,
	 PCI_ANY_ID,
	 },
	{
	 PCI_VENDOR_ID_DELL,
	 PCI_DEVICE_ID_DELL_PERC5, // xscale IOP
	 PCI_ANY_ID,
	 PCI_ANY_ID,
	 },
	{0}			/* Terminating entry */

};

MODULE_DEVICE_TABLE(pci, megasas_pci_table);

static int megasas_mgmt_majorno;
static struct megasas_mgmt_info megasas_mgmt_info;
static struct fasync_struct *megasas_async_queue;
static DECLARE_MUTEX(megasas_async_queue_mutex);
static u32 megasas_dbg_lvl;

/*for IOCTL mem pool*/
int arr[MAX_IOCTL_MEM_POOL] = {MAX_4K_BUFF,MAX_8K_BUFF,MAX_64K_BUFF};

int arr_size[MAX_IOCTL_MEM_POOL] = {MEGASAS_INIT_IOCTL_MEM_SIZE/*4k*/,
				2*MEGASAS_INIT_IOCTL_MEM_SIZE/*8k*/,
				16*MEGASAS_INIT_IOCTL_MEM_SIZE/*64k*/};

#ifdef CONFIG_PROC_FS
static struct proc_dir_entry *megasas_proc_root;
static struct proc_dir_entry *megasas_proc_hba_map;
static struct proc_dir_entry *megasas_proc_version;
static struct proc_dir_entry *megasas_proc_release_date;
static struct proc_dir_entry *megasas_proc_dbg_lvl;
static struct proc_dir_entry *megasas_proc_poll_mode_io;
#endif

static void megasas_complete_cmd_q(struct megasas_instance *instance,
			u8 alt_status);
static inline void megasas_complete_cmd(struct megasas_instance *instance,
			struct megasas_cmd *cmd, u8 alt_status);

static const char *megasas_info(struct Scsi_Host *host)
{
	return "LSI Logic Corporation MegaRAID SAS Driver";
}

static inline unsigned long msecs_to_jiffies(unsigned long msecs)
{
	return ((HZ * msecs + 999) / 1000);
}

static void msleep(unsigned long msecs)
{
#ifdef __VMKERNEL_MODULE__
    /* only schedule out if sleeping for longer than a second */
    if (msecs > 1000) {
#endif
set_current_state(TASK_UNINTERRUPTIBLE);
schedule_timeout(msecs_to_jiffies(msecs) + 1);
#ifdef __VMKERNEL_MODULE__
  } else {
  mdelay(msecs);
	}
#endif
}

/**
 * megasas_start_timer - Initializes a timer object
 * @instance:		Adapter soft state
 * @timer:		timer object to be initialized
 * @fn:			timer function
 * @interval:		time interval between timer function call
 *
 */
static inline void
megasas_start_timer(struct megasas_instance *instance,
			struct timer_list *timer,
			void *fn, unsigned long interval)
{
	init_timer(timer);
	timer->expires = jiffies + interval;
	timer->data = (unsigned long)instance;
	timer->function = fn;
	add_timer(timer);
}

/**
 * megasas_io_completion_timer - Timer fn
 * @instance_addr:	Address of adapter soft state
 *
 * Schedules tasklet for cmd completion 
 * if poll_mode_io is set
 */
static void
megasas_io_completion_timer(unsigned long instance_addr)
{
	struct megasas_instance *instance = 
			(struct megasas_instance *)instance_addr;

	if (atomic_read(&instance->fw_outstanding))
		megasas_complete_cmd_q(instance, DID_OK);

	/* Restart timer */
	if (poll_mode_io)
		mod_timer(&instance->io_completion_timer,
			jiffies + MEGASAS_COMPLETION_TIMER_INTERVAL);
}

static int
megasas_proc_show_version(char *page, char **start, off_t offset, int count,
			  int *eof, void *data)
{
	*eof = 1;
	return (snprintf(page, strlen(MEGASAS_VERSION) + 2, "%s\n",
			 MEGASAS_VERSION));
}

static int
megasas_proc_show_release_date(char *page, char **start, off_t offset,
			       int count, int *eof, void *data)
{
	*eof = 1;
	return (snprintf(page, strlen(MEGASAS_RELDATE) + 2, "%s\n",
			 MEGASAS_RELDATE));
}

static int 
megasas_proc_set_dbg_lvl(struct file *file, const char *page,
                                     size_t count, loff_t *ppos)
{
	int retval = count;
	
	return retval;

}

static int
megasas_proc_show_max_sectors(char *page, char **start, off_t offset,
			       int count, int *eof, void *data)
{
	*eof = 1;
	return sprintf(page,"%u",max_sector);

}

static int
megasas_proc_show_bus_devfn(char *page, char **start, off_t offset,
			    int count, int *eof, void *data)
{
	int i;
	uint32_t bus_devfn = 0;

	struct megasas_instance *instance = (struct megasas_instance *)data;

	for (i = 0; i < megasas_mgmt_info.max_index; i++)
		if (instance == megasas_mgmt_info.instance[i])
			bus_devfn = ((instance->pdev->bus->number) << 16 |
				     (PCI_SLOT(instance->pdev->devfn)) << 8 |
				     (PCI_FUNC(instance->pdev->devfn))) &
			    0xFFFFFF;

	*eof = 1;
	return (sprintf(page, "%d", bus_devfn));
}

static void megasas_create_proc_entry(struct megasas_instance *instance)
{
	unsigned char string[8] = { 0 };

	sprintf(string, "%d", instance->host->host_no);
	instance->hba_proc_dir = proc_mkdir(string, megasas_proc_hba_map);

	if (!instance->hba_proc_dir) {
		printk(KERN_WARNING "megasas: proc creation failed\n");
		return;
	}

	create_proc_read_entry("bus_devfn", S_IRUSR | S_IFREG,
			       instance->hba_proc_dir,
			       megasas_proc_show_bus_devfn, instance);

	create_proc_read_entry("max_sector", S_IRUSR | S_IFREG,
			       instance->hba_proc_dir,
			       megasas_proc_show_max_sectors, NULL);

}

static void megasas_remove_proc_entry(struct megasas_instance *instance)
{
	unsigned char string[8] = { 0 };

	remove_proc_entry("bus_devfn", instance->hba_proc_dir);
	remove_proc_entry("max_sector", instance->hba_proc_dir);

	sprintf(string, "%d", instance->host->host_no);
	remove_proc_entry(string, megasas_proc_hba_map);
}

static ssize_t
megasas_proc_show_dbg_lvl(struct file *file, char *buf,
					size_t count, loff_t *ppos)
 {
	char arr[10];
	int len;
	loff_t pos = *ppos;
	len = sprintf(arr,"%u\n", megasas_dbg_lvl);
	if (pos < 0 || pos >= len)
		return 0;
	if (copy_to_user(buf,(void *)arr,len))
		return -EFAULT;
	*ppos = pos + len;
	return len;
}

static struct file_operations proc_dbg_lvl_ops = {
        .write          = megasas_proc_set_dbg_lvl,
	.read		= megasas_proc_show_dbg_lvl,
};

static ssize_t
megasas_proc_show_poll_mode_io(struct file *file, char *buf,
                                       size_t count, loff_t *ppos)
{
       char arr[10];
       int len;
        loff_t pos = *ppos;
       len = sprintf(arr,"%u\n",poll_mode_io);
        if (pos < 0 || pos >= len)
                return 0;
        if (copy_to_user(buf,(void *)arr,len))
                return -EFAULT;
       *ppos = pos + len;
        return len;
}

static ssize_t
megasas_proc_set_poll_mode_io(struct file *file, const char *page,
                                     size_t count, loff_t *ppos)
{
       int retval = count;
       int tmp = poll_mode_io;
       int i;
       struct megasas_instance *instance;

      /* if(sscanf(page,"%u",&poll_mode_io) < 1){
               printk(KERN_ERR "megasas: could not set poll_mode_io\n");
               retval = -EINVAL;
       }*/

       /*
        * Check if poll_mode_io is already set or is same as previous value
        */
       if ((tmp && poll_mode_io) || (tmp == poll_mode_io))
               goto out;

       if (poll_mode_io) {
               /*
                * Start timers for all adapters
                */
               for (i = 0; i < megasas_mgmt_info.max_index; i++) {
                       instance = megasas_mgmt_info.instance[i];
                       if (instance) {
                               megasas_start_timer(instance, &instance->io_completion_timer,
                                       megasas_io_completion_timer,
                                       MEGASAS_COMPLETION_TIMER_INTERVAL);
                       }
               }
       }
       else {
               /*
                * Delete timers for all adapters
                */
               for (i = 0; i < megasas_mgmt_info.max_index; i++) {
                       instance = megasas_mgmt_info.instance[i];
                       if (instance)
                               del_timer_sync(&instance->io_completion_timer);
               }
       }

out:
       return retval;
}

static struct file_operations proc_poll_mode_ops = {
        .write          = megasas_proc_set_poll_mode_io,
       .read           = megasas_proc_show_poll_mode_io,
};

/**
 * megasas_get_cmd -	Get a command from the free pool
 * @instance:		Adapter soft state
 *
 * Returns a free command from the pool
 */
static inline struct megasas_cmd *megasas_get_cmd(struct megasas_instance
						  *instance)
{
	unsigned long flags;
	struct megasas_cmd *cmd = NULL;

	spin_lock_irqsave(&instance->cmd_pool_lock, flags);

	if (!list_empty(&instance->cmd_pool)) {
		cmd = list_entry((&instance->cmd_pool)->next,
				 struct megasas_cmd, list);
		list_del_init(&cmd->list);
	} else {
		printk(KERN_ERR "megasas: Command pool empty!\n");
	}

	spin_unlock_irqrestore(&instance->cmd_pool_lock, flags);
	return cmd;
}

/**
 * megasas_return_cmd -	Return a cmd to free command pool
 * @instance:		Adapter soft state
 * @cmd:		Command packet to be returned to free command pool
 */
static inline void
megasas_return_cmd(struct megasas_instance *instance, struct megasas_cmd *cmd)
{
	unsigned long flags;

	spin_lock_irqsave(&instance->cmd_pool_lock, flags);

	cmd->scmd = NULL;
	list_add_tail(&cmd->list, &instance->cmd_pool);

	spin_unlock_irqrestore(&instance->cmd_pool_lock, flags);
}


/**
*	The following functions are defined for xscale
*	(deviceid : 1064R, PERC5) controllers
*/

/**
 * megasas_enable_intr_xscale -	Enables interrupts
 * @regs:			MFI register set
 */
static inline void
megasas_enable_intr_xscale(struct megasas_register_set __iomem * regs)
{
	writel(1, &(regs)->outbound_intr_mask);

	/* Dummy readl to force pci flush */
	readl(&regs->outbound_intr_mask);
}

/**
 * megasas_disable_intr_xscale -Disables interrupt
 * @regs:			MFI register set
 */
static inline void
megasas_disable_intr_xscale(struct megasas_register_set __iomem * regs)
{
	u32 mask = 0x1f;
	writel(mask, &regs->outbound_intr_mask);
	/* Dummy readl to force pci flush */
	readl(&regs->outbound_intr_mask);
}

/**
 * megasas_read_fw_status_reg_xscale - returns the current FW status value
 * @regs:			MFI register set
 */
static u32
megasas_read_fw_status_reg_xscale(struct megasas_register_set __iomem * regs)
{
	return readl(&(regs)->outbound_msg_0);
}

/**
 * megasas_clear_interrupt_xscale -	Check & clear interrupt
 * @regs:				MFI register set
 */
static int
megasas_clear_intr_xscale(struct megasas_register_set __iomem * regs)
{
	u32 status;
	/*
	 * Check if it is our interrupt
	 */
	status = readl(&regs->outbound_intr_status);

	if (!(status & MFI_OB_INTR_STATUS_MASK)) {
		return 1;
	}

	/*
	 * Clear the interrupt by writing back the same value
	 */
	writel(status, &regs->outbound_intr_status);

	return 0;
}

/**
 * megasas_fire_cmd_xscale -	Sends command to the FW
 * @frame_phys_addr :		Physical address of cmd
 * @frame_count :		Number of frames for the command
 * @regs :			MFI register set
 */
static inline void
megasas_fire_cmd_xscale(dma_addr_t frame_phys_addr,u32 frame_count, struct megasas_register_set __iomem *regs)
{
	writel((frame_phys_addr >> 3)|(frame_count),
	       &(regs)->inbound_queue_port);
}

static struct megasas_instance_template megasas_instance_template_xscale = {

	.fire_cmd = megasas_fire_cmd_xscale,
	.enable_intr = megasas_enable_intr_xscale,
	.disable_intr = megasas_disable_intr_xscale,
	.clear_intr = megasas_clear_intr_xscale,
	.read_fw_status_reg = megasas_read_fw_status_reg_xscale,
};

/**
*	This is the end of set of functions & definitions specific
*	to xscale (deviceid : 1064R, PERC5) controllers
*/

/**
*	The following functions are defined for ppc (deviceid : 0x60)
* 	controllers
*/

/**
 * megasas_enable_intr_ppc -	Enables interrupts
 * @regs:			MFI register set
 */
static inline void
megasas_enable_intr_ppc(struct megasas_register_set __iomem * regs)
{
	writel(0xFFFFFFFF, &(regs)->outbound_doorbell_clear);

	writel(~0x80000004, &(regs)->outbound_intr_mask);

	/* Dummy readl to force pci flush */
	readl(&regs->outbound_intr_mask);
}

/**
 * megasas_disable_intr_ppc -	Disables interrupt
 * @regs:			MFI register set
 */
static inline void
megasas_disable_intr_ppc(struct megasas_register_set __iomem * regs)
{
	u32 mask = 0xFFFFFFFF;
	writel(mask, &regs->outbound_intr_mask);
	/* Dummy readl to force pci flush */
	readl(&regs->outbound_intr_mask);
}

/**
 * megasas_read_fw_status_reg_ppc - returns the current FW status value
 * @regs:			MFI register set
 */
static u32
megasas_read_fw_status_reg_ppc(struct megasas_register_set __iomem * regs)
{
	return readl(&(regs)->outbound_scratch_pad);
}

/**
 * megasas_clear_interrupt_ppc -	Check & clear interrupt
 * @regs:				MFI register set
 */
static int
megasas_clear_intr_ppc(struct megasas_register_set __iomem * regs)
{
	u32 status;
	/*
	 * Check if it is our interrupt
	 */
	status = readl(&regs->outbound_intr_status);

	if (!(status & MFI_REPLY_1078_MESSAGE_INTERRUPT)) {
		return 1;
	}

	/*
	 * Clear the interrupt by writing back the same value
	 */
	writel(status, &regs->outbound_doorbell_clear);

	return 0;
}
/**
 * megasas_fire_cmd_ppc -	Sends command to the FW
 * @frame_phys_addr :		Physical address of cmd
 * @frame_count :		Number of frames for the command
 * @regs :			MFI register set
 */
static inline void
megasas_fire_cmd_ppc(dma_addr_t frame_phys_addr, u32 frame_count, struct megasas_register_set __iomem *regs)
{
	writel((frame_phys_addr | (frame_count<<1))|1,
			&(regs)->inbound_queue_port);
}

static struct megasas_instance_template megasas_instance_template_ppc = {

	.fire_cmd = megasas_fire_cmd_ppc,
	.enable_intr = megasas_enable_intr_ppc,
	.disable_intr = megasas_disable_intr_ppc,
	.clear_intr = megasas_clear_intr_ppc,
	.read_fw_status_reg = megasas_read_fw_status_reg_ppc,
};

/**
*	This is the end of set of functions & definitions
* 	specific to ppc (deviceid : 0x60) controllers
*/

/**
 * megasas_issue_polled -	Issues a polling command
 * @instance:			Adapter soft state
 * @cmd:			Command packet to be issued
 *
 * For polling, MFI requires the cmd_status to be set to 0xFF before posting.
 */
static int
megasas_issue_polled(struct megasas_instance *instance, struct megasas_cmd *cmd)
{
	int i;
	u32 msecs = MFI_POLL_TIMEOUT_SECS * 1000;

	struct megasas_header *frame_hdr = &cmd->frame->hdr;

	frame_hdr->cmd_status = 0xFF;
	frame_hdr->flags |= MFI_FRAME_DONT_POST_IN_REPLY_QUEUE;

	/*
	 * Issue the frame using inbound queue port
	 */
	instance->instancet->fire_cmd(cmd->frame_phys_addr ,0,instance->reg_set);

	/*
	 * Wait for cmd_status to change
	 */
	for (i = 0; (i < msecs) && (frame_hdr->cmd_status == 0xff); i++) {
		rmb();
		msleep(1);
	}

	if (frame_hdr->cmd_status == 0xff)
		return -ETIME;

	return 0;
}

/**
 * megasas_issue_blocked_cmd -	Synchronous wrapper around regular FW cmds
 * @instance:			Adapter soft state
 * @cmd:			Command to be issued
 *
 * This function waits on an event for the command to be returned from ISR.
 * Used to issue ioctl commands.
 */
static int
megasas_issue_blocked_cmd(struct megasas_instance *instance,
			  struct megasas_cmd *cmd)
{
	cmd->cmd_status = ENODATA;

	instance->instancet->fire_cmd(cmd->frame_phys_addr ,0,instance->reg_set);

	wait_event(instance->int_cmd_wait_q, cmd->cmd_status != ENODATA);

	return 0;
}

/**
 * megasas_issue_blocked_abort_cmd -	Aborts previously issued cmd
 * @instance:				Adapter soft state
 * @cmd_to_abort:			Previously issued cmd to be aborted
 *
 * MFI firmware can abort previously issued AEN comamnd (automatic event
 * notification). The megasas_issue_blocked_abort_cmd() issues such abort
 * cmd and waits for return status.
 */
static int
megasas_issue_blocked_abort_cmd(struct megasas_instance *instance,
				struct megasas_cmd *cmd_to_abort)
{
	struct megasas_cmd *cmd;
	struct megasas_abort_frame *abort_fr;

	cmd = megasas_get_cmd(instance);

	if (!cmd)
		return -1;

	abort_fr = &cmd->frame->abort;

	/*
	 * Prepare and issue the abort frame
	 */
	abort_fr->cmd = MFI_CMD_ABORT;
	abort_fr->cmd_status = 0xFF;
	abort_fr->flags = 0;
	abort_fr->abort_context = cmd_to_abort->index;
	abort_fr->abort_mfi_phys_addr_lo = cmd_to_abort->frame_phys_addr;
	abort_fr->abort_mfi_phys_addr_hi = 0;

	cmd->sync_cmd = 1;
	cmd->cmd_status = 0xFF;

	instance->instancet->fire_cmd(cmd->frame_phys_addr ,0,instance->reg_set);

	/*
	 * Wait for this cmd to complete
	 */
	wait_event(instance->abort_cmd_wait_q, cmd->cmd_status != 0xFF);

	megasas_return_cmd(instance, cmd);
	return 0;
}

/**
 * megasas_make_sgl32 -	Prepares 32-bit SGL
 * @instance:		Adapter soft state
 * @scp:		SCSI command from the mid-layer
 * @mfi_sgl:		SGL to be filled in
 *
 * If successful, this function returns the number of SG elements. Otherwise,
 * it returnes -1.
 */
static inline int
megasas_make_sgl32(struct megasas_instance *instance, struct scsi_cmnd *scp,
		   union megasas_sgl *mfi_sgl)
{
	int i;
	int sge_count;
	struct scatterlist *os_sgl;

	/*
	 * Return 0 if there is no data transfer
	 */
	if (!scp->request_buffer || !scp->request_bufflen)
		return 0;

	if (!scp->use_sg) {
#ifdef __VMKERNEL_MODULE__
		mfi_sgl->sge32[0].phys_addr = scp->request_bufferMA;
#else
		mfi_sgl->sge32[0].phys_addr = pci_map_single(instance->pdev,
							     scp->
							     request_buffer,
							     scp->
							     request_bufflen,
							     scp->
							     sc_data_direction);
#endif
		mfi_sgl->sge32[0].length = scp->request_bufflen;

		return 1;
	}

	os_sgl = (struct scatterlist *)scp->request_buffer;
#ifdef __VMKERNEL_MODULE__
	sge_count = scp->use_sg;
#else
	sge_count = pci_map_sg(instance->pdev, os_sgl, scp->use_sg,
			       scp->sc_data_direction);
#endif

	for (i = 0; i < sge_count; i++, os_sgl++) {
		mfi_sgl->sge32[i].length = sg_dma_len(os_sgl);
		mfi_sgl->sge32[i].phys_addr = sg_dma_address(os_sgl);
	}

	return sge_count;
}

/**
 * megasas_make_sgl64 -	Prepares 64-bit SGL
 * @instance:		Adapter soft state
 * @scp:		SCSI command from the mid-layer
 * @mfi_sgl:		SGL to be filled in
 *
 * If successful, this function returns the number of SG elements. Otherwise,
 * it returnes -1.
 */
static inline int
megasas_make_sgl64(struct megasas_instance *instance, struct scsi_cmnd *scp,
		   union megasas_sgl *mfi_sgl)
{
	int i;
	int sge_count;
	struct scatterlist *os_sgl;

	/*
	 * Return 0 if there is no data transfer
	 */
/*
#ifdef __VMKERNEL_MODULE__
  if (!scp->request_bufflen)
#else
*/
	if (!scp->request_buffer || !scp->request_bufflen)
/*
#endif
*/
		return 0;

	if (!scp->use_sg) {
#ifdef __VMKERNEL_MODULE__
		mfi_sgl->sge64[0].phys_addr = scp->request_bufferMA;
#else
		mfi_sgl->sge64[0].phys_addr = pci_map_single(instance->pdev,
							     scp->
							     request_buffer,
							     scp->
							     request_bufflen,
							     scp->
							     sc_data_direction);

#endif
		mfi_sgl->sge64[0].length = scp->request_bufflen;

		return 1;
	}

	os_sgl = (struct scatterlist *)scp->request_buffer;
#ifdef __VMKERNEL_MODULE__
	sge_count = scp->use_sg;
#else
	sge_count = pci_map_sg(instance->pdev, os_sgl, scp->use_sg,
			       scp->sc_data_direction);
#endif

	for (i = 0; i < sge_count; i++, os_sgl++) {
		mfi_sgl->sge64[i].length = sg_dma_len(os_sgl);
		mfi_sgl->sge64[i].phys_addr = sg_dma_address(os_sgl);
	}

	return sge_count;
}

 /**
 * megasas_get_frame_count - Computes the number of frames
 * @sge_count		: number of sg elements
 * @frame_type		: type of frame- io or pthru frame
 *
 * Returns the number of frames required for numnber of sge's (sge_count)
 */

static u32 megasas_get_frame_count(u8 sge_count, u8 frame_type)
{
	int num_cnt;
	int sge_bytes;
	u32 sge_sz;
	u32 frame_count=0;

	sge_sz = (IS_DMA64) ? sizeof(struct megasas_sge64) :
	    sizeof(struct megasas_sge32);

	/*
	 * Main frame can contain 2 SGEs for 64-bit SGLs and
	 * 3 SGEs for 32-bit SGLs for ldio &
	 * 1 SGEs for 64-bit SGLs and
	 * 2 SGEs for 32-bit SGLs for pthru frame
	 */
	if (unlikely(frame_type == PTHRU_FRAME)) {
		if (IS_DMA64)
			num_cnt = sge_count - 1;
		else
			num_cnt = sge_count - 2;
	} else {
		if (IS_DMA64)
			num_cnt = sge_count - 2;
		else
			num_cnt = sge_count - 3;
	}

	if(num_cnt>0){
		sge_bytes = sge_sz * num_cnt;

		frame_count = (sge_bytes / MEGAMFI_FRAME_SIZE) +
		    ((sge_bytes % MEGAMFI_FRAME_SIZE) ? 1 : 0) ;
	}
	/* Main frame */
	frame_count +=1;

	if (frame_count > 7)
		frame_count = 8;
	return frame_count;
}

/**
 * megasas_build_dcdb -	Prepares a direct cdb (DCDB) command
 * @instance:		Adapter soft state
 * @scp:		SCSI command
 * @cmd:		Command to be prepared in
 *
 * This function prepares CDB commands. These are typcially pass-through
 * commands to the devices.
 */
static inline int
megasas_build_dcdb(struct megasas_instance *instance, struct scsi_cmnd *scp,
		   struct megasas_cmd *cmd)
{
	u32 is_logical;
	u32 device_id;
	u16 flags = 0;
	struct megasas_pthru_frame *pthru;

	is_logical = MEGASAS_IS_LOGICAL(scp);
	device_id = MEGASAS_DEV_INDEX(instance, scp);
	pthru = (struct megasas_pthru_frame *)cmd->frame;

	if (scp->sc_data_direction == PCI_DMA_TODEVICE)
		flags = MFI_FRAME_DIR_WRITE;
	else if (scp->sc_data_direction == PCI_DMA_FROMDEVICE)
		flags = MFI_FRAME_DIR_READ;
	else if (scp->sc_data_direction == PCI_DMA_NONE)
		flags = MFI_FRAME_DIR_NONE;

	/*
	 * Prepare the DCDB frame
	 */
	pthru->cmd = (is_logical) ? MFI_CMD_LD_SCSI_IO : MFI_CMD_PD_SCSI_IO;
	pthru->cmd_status = 0x0;
	pthru->scsi_status = 0x0;
	pthru->target_id = device_id;
	pthru->lun = scp->lun;
	pthru->cdb_len = scp->cmd_len;
	pthru->timeout = 0;
	pthru->flags = flags;
	pthru->data_xfer_len = scp->request_bufflen;

	memcpy(pthru->cdb, scp->cmnd, scp->cmd_len);

	/*
	 * Construct SGL
	 */
	if (IS_DMA64) {
		pthru->flags |= MFI_FRAME_SGL64;
		pthru->sge_count = megasas_make_sgl64(instance, scp,
						      &pthru->sgl);
	} else
		pthru->sge_count = megasas_make_sgl32(instance, scp,
						      &pthru->sgl);

	/*
	 * Sense info specific
	 */
	pthru->sense_len = SCSI_SENSE_BUFFERSIZE;
	pthru->sense_buf_phys_addr_hi = 0;
	pthru->sense_buf_phys_addr_lo = cmd->sense_phys_addr;

	/*
	 * Compute the total number of frames this command consumes. FW uses
	 * this number to pull sufficient number of frames from host memory.
	 */
	cmd->frame_count = megasas_get_frame_count(pthru->sge_count,
							PTHRU_FRAME);

	return cmd->frame_count;
}

/**
 * megasas_build_ldio -	Prepares IOs to logical devices
 * @instance:		Adapter soft state
 * @scp:		SCSI command
 * @cmd:		Command to to be prepared
 *
 * Frames (and accompanying SGLs) for regular SCSI IOs use this function.
 */
static inline int
megasas_build_ldio(struct megasas_instance *instance, struct scsi_cmnd *scp,
		   struct megasas_cmd *cmd)
{
	u32 device_id;
	u8 sc = scp->cmnd[0];
	u16 flags = 0;
	struct megasas_io_frame *ldio;

	device_id = MEGASAS_DEV_INDEX(instance, scp);
	ldio = (struct megasas_io_frame *)cmd->frame;

	if (scp->sc_data_direction == PCI_DMA_TODEVICE)
		flags = MFI_FRAME_DIR_WRITE;
	else if (scp->sc_data_direction == PCI_DMA_FROMDEVICE)
		flags = MFI_FRAME_DIR_READ;

	/*
	 * Prepare the Logical IO frame: 2nd bit is zero for all read cmds
	 */
	ldio->cmd = (sc & 0x02) ? MFI_CMD_LD_WRITE : MFI_CMD_LD_READ;
	ldio->cmd_status = 0x0;
	ldio->scsi_status = 0x0;
	ldio->target_id = device_id;
	ldio->timeout = 0;
	ldio->reserved_0 = 0;
	ldio->pad_0 = 0;
	ldio->flags = flags;
	ldio->start_lba_hi = 0;
	ldio->access_byte = (scp->cmd_len != 6) ? scp->cmnd[1] : 0;

	/*
	 * 6-byte READ(0x08) or WRITE(0x0A) cdb
	 */
	if (scp->cmd_len == 6) {
		ldio->lba_count = (u32) scp->cmnd[4];
		ldio->start_lba_lo = ((u32) scp->cmnd[1] << 16) |
		    ((u32) scp->cmnd[2] << 8) | (u32) scp->cmnd[3];

		ldio->start_lba_lo &= 0x1FFFFF;
	}

	/*
	 * 10-byte READ(0x28) or WRITE(0x2A) cdb
	 */
	else if (scp->cmd_len == 10) {
		ldio->lba_count = (u32) scp->cmnd[8] |
		    ((u32) scp->cmnd[7] << 8);
		ldio->start_lba_lo = ((u32) scp->cmnd[2] << 24) |
		    ((u32) scp->cmnd[3] << 16) |
		    ((u32) scp->cmnd[4] << 8) | (u32) scp->cmnd[5];
	}

	/*
	 * 12-byte READ(0xA8) or WRITE(0xAA) cdb
	 */
	else if (scp->cmd_len == 12) {
		ldio->lba_count = ((u32) scp->cmnd[6] << 24) |
		    ((u32) scp->cmnd[7] << 16) |
		    ((u32) scp->cmnd[8] << 8) | (u32) scp->cmnd[9];

		ldio->start_lba_lo = ((u32) scp->cmnd[2] << 24) |
		    ((u32) scp->cmnd[3] << 16) |
		    ((u32) scp->cmnd[4] << 8) | (u32) scp->cmnd[5];
	}

	/*
	 * 16-byte READ(0x88) or WRITE(0x8A) cdb
	 */
	else if (scp->cmd_len == 16) {
		ldio->lba_count = ((u32) scp->cmnd[10] << 24) |
		    ((u32) scp->cmnd[11] << 16) |
		    ((u32) scp->cmnd[12] << 8) | (u32) scp->cmnd[13];

		ldio->start_lba_lo = ((u32) scp->cmnd[6] << 24) |
		    ((u32) scp->cmnd[7] << 16) |
		    ((u32) scp->cmnd[8] << 8) | (u32) scp->cmnd[9];

		ldio->start_lba_hi = ((u32) scp->cmnd[2] << 24) |
		    ((u32) scp->cmnd[3] << 16) |
		    ((u32) scp->cmnd[4] << 8) | (u32) scp->cmnd[5];

	}

	/*
	 * Construct SGL
	 */
	if (IS_DMA64) {
		ldio->flags |= MFI_FRAME_SGL64;
		ldio->sge_count = megasas_make_sgl64(instance, scp, &ldio->sgl);
	} else
		ldio->sge_count = megasas_make_sgl32(instance, scp, &ldio->sgl);

	/*
	 * Sense info specific
	 */
	ldio->sense_len = SCSI_SENSE_BUFFERSIZE;
	ldio->sense_buf_phys_addr_hi = 0;
	ldio->sense_buf_phys_addr_lo = cmd->sense_phys_addr;

	/*
	 * Compute the total number of frames this command consumes. FW uses
	 * this number to pull sufficient number of frames from host memory.
	 */
	cmd->frame_count = megasas_get_frame_count(ldio->sge_count, IO_FRAME);

	return cmd->frame_count;
}

/**
 * megasas_is_ldio -		Checks if the cmd is for logical drive
 * @scmd:			SCSI command
 *
 * Called by megasas_queue_command to find out if the command to be queued
 * is a logical drive command
 */
static inline int megasas_is_ldio(struct scsi_cmnd *cmd)
{
	if (!MEGASAS_IS_LOGICAL(cmd))
		return 0;
	switch (cmd->cmnd[0]) {
	case READ_10:
	case WRITE_10:
	case READ_12:
	case WRITE_12:
	case READ_6:
	case WRITE_6:
	case READ_16:
	case WRITE_16:
		return 1;
	default:
		return 0;
	}
}

/**
 * megasas_dump_pending_frames -	Dumps the frame address of all pending cmds
 *                              	in FW
 * @instance:				Adapter soft state
 */
static inline void
megasas_dump_pending_frames(struct megasas_instance *instance)
{
	struct megasas_cmd *cmd;
	int i,n;
	union megasas_sgl *mfi_sgl;
	struct megasas_io_frame *ldio;
	struct megasas_pthru_frame *pthru;
	u32 sgcount;
	u32 max_cmd = instance->max_fw_cmds;

	printk(KERN_ERR "\nmegasas[%d]: Dumping Frame Phys Address of all pending cmds in FW\n",instance->host->host_no);
	printk(KERN_ERR "megasas[%d]: Total OS Pending cmds : %d\n",instance->host->host_no,atomic_read(&instance->fw_outstanding));
	if (IS_DMA64)
		printk(KERN_ERR "\nmegasas[%d]: 64 bit SGLs were sent to FW\n",instance->host->host_no);
	else
		printk(KERN_ERR "\nmegasas[%d]: 32 bit SGLs were sent to FW\n",instance->host->host_no);

	printk(KERN_ERR "megasas[%d]: Pending OS cmds in FW : \n",instance->host->host_no);
	for (i = 0; i < max_cmd; i++) {
		cmd = instance->cmd_list[i];
		if(!cmd->scmd)
			continue;
		printk(KERN_ERR "megasas[%d]: Frame addr :0x%08lx : ",instance->host->host_no,(unsigned long)cmd->frame_phys_addr);
		if (megasas_is_ldio(cmd->scmd)){
			ldio = (struct megasas_io_frame *)cmd->frame;
			mfi_sgl = &ldio->sgl;
			sgcount = ldio->sge_count;
			printk(KERN_ERR "megasas[%d]: frame count : 0x%x, Cmd : 0x%x, Tgt id : 0x%x, lba lo : 0x%x, lba_hi : 0x%x, sense_buf addr : 0x%x,sge count : 0x%x\n",instance->host->host_no, cmd->frame_count,ldio->cmd,ldio->target_id, ldio->start_lba_lo,ldio->start_lba_hi,ldio->sense_buf_phys_addr_lo,sgcount);
		}
		else {
			pthru = (struct megasas_pthru_frame *) cmd->frame;
			mfi_sgl = &pthru->sgl;
			sgcount = pthru->sge_count;
			printk(KERN_ERR "megasas[%d]: frame count : 0x%x, Cmd : 0x%x, Tgt id : 0x%x, lun : 0x%x, cdb_len : 0x%x, data xfer len : 0x%x, sense_buf addr : 0x%x,sge count : 0x%x\n",instance->host->host_no,cmd->frame_count,pthru->cmd,pthru->target_id,pthru->lun,pthru->cdb_len , pthru->data_xfer_len,pthru->sense_buf_phys_addr_lo,sgcount);
		}
	if(megasas_dbg_lvl & MEGASAS_DBG_LVL){
		for (n = 0; n < sgcount; n++){
			if (IS_DMA64)
				printk(KERN_ERR "megasas: sgl len : 0x%x, sgl addr : 0x%08lx ",mfi_sgl->sge64[n].length , (unsigned long)mfi_sgl->sge64[n].phys_addr) ;
			else
				printk(KERN_ERR "megasas: sgl len : 0x%x, sgl addr : 0x%x ",mfi_sgl->sge32[n].length , mfi_sgl->sge32[n].phys_addr) ;
			}
		}
		printk(KERN_ERR "\n");
	} /*for max_cmd*/
	printk(KERN_ERR "\nmegasas[%d]: Pending Internal cmds in FW : \n",instance->host->host_no);
	for (i = 0; i < max_cmd; i++) {

		cmd = instance->cmd_list[i];

		if(cmd->sync_cmd == 1){
			printk(KERN_ERR "0x%08lx : ", (unsigned long)cmd->frame_phys_addr);
		}
	}
	printk(KERN_ERR "megasas[%d]: Dumping Done.\n\n",instance->host->host_no);
}

/**
 * megasas_queue_command -	Queue entry point
 * @scmd:			SCSI command to be queued
 * @done:			Callback entry point
 */
static int
megasas_queue_command(struct scsi_cmnd *scmd, void (*done) (struct scsi_cmnd *))
{
	u32 frame_count;
	struct megasas_cmd *cmd;
	struct megasas_instance *instance;
	unsigned long sec;

	instance = (struct megasas_instance *)
	    scmd->host->hostdata;
	scmd->scsi_done = done;
	scmd->result = 0;

	/*Don't process if we have already declared adapter dead*/
	if(instance->hw_crit_error)
		return SCSI_MLQUEUE_HOST_BUSY;

	if (MEGASAS_IS_LOGICAL(scmd) &&
#ifdef __VMKERNEL_MODULE__
	    (MEGASAS_TARGET_TO_DEV(scmd->target) >= MEGASAS_MAX_LD ||
        scmd->device->lun)) {
#else
	    (scmd->device->id >= MEGASAS_MAX_LD || scmd->device->lun)) {
#endif
		scmd->result = DID_BAD_TARGET << 16;
		goto out_done;
	}

#ifdef __VMKERNEL_MODULE__
        /* 
         * PR155896: return DID_OK for SCSI2 reserve/release commands for 1078R
         * card. 1078R's firmware doesn't support SCSI reservation, and esx 
         * needs SCSI2 reservation for VMFS.
         */
        if (instance->pdev->device == PCI_DEVICE_ID_LSI_SAS1078R &&
            (scmd->cmnd[0] == RESERVE || scmd->cmnd[0] == RELEASE)) {
            scmd->result = DID_OK << 16;
            goto out_done;
        }
#endif

#ifdef __VMKERNEL_MODULE__
        /* 
         * Fix for PR131932: DELL RAID controller may expose an ENCLOSURE LUN.
         * Send REPORT LUNS command to the ENCLOSURE LUN target id with a non-zero
         * LUN # will hang the controller firmware. Send REPORT LUNS to LUN 0 is OK.
         * The current rescan code sends REPORT LUNS to non-zero LUN #.
         * Hence force any REPORT LUNS command to LUN 0 as a workaround.
         * Eventually this need to be fixed in the scan code or in the megaraid
         * controller firmware and this workaround fix shall be removed. (PR142089)
         */
        if(scmd->cmnd[0] == REPORT_LUNS && scmd->lun != 0) {
                printk(KERN_NOTICE "megasas: REPORT LUNS to target %4x change lun id %04x to 0\n",
                        scmd->target, scmd->lun);
                scmd->lun = 0;
        }
#endif

	if (fast_load) {
		if (!MEGASAS_IS_LOGICAL(scmd)) {
			scmd->result = DID_BAD_TARGET << 16;
			goto out_done;
		}
	}

	switch (scmd->cmnd[0]) {
		case SYNCHRONIZE_CACHE:
#ifndef __VMKERNEL_MODULE__
#if defined(CONFIG_DISKDUMP) || defined(CONFIG_DISKDUMP_MODULE)
			if (!crashdump_mode())
#endif
#endif /* __VMKERNEL_MODULE__ */
			{
				scmd->result = DID_OK << 16;
				goto out_done;
			}
	}

	/* If FW is busy donot accept any more cmds */
	if(instance->is_busy){
		sec = (jiffies - instance->last_time) / HZ;
		if(sec<10)
			return SCSI_MLQUEUE_HOST_BUSY;
		else{
			instance->is_busy=0;
			instance->last_time=0;
		}
	}

	if(scmd->retries>1){
		instance->is_busy=1;
		instance->last_time=jiffies;
	}

	cmd = megasas_get_cmd(instance);
	if (!cmd)
		return SCSI_MLQUEUE_HOST_BUSY;

	/*
	 * Logical drive command
	 */
	if (megasas_is_ldio(scmd))
		frame_count = megasas_build_ldio(instance, scmd, cmd);
	else
		frame_count = megasas_build_dcdb(instance, scmd, cmd);

	if (!frame_count)
		goto out_return_cmd;

	cmd->scmd = scmd;
	scmd->SCp.ptr = (char *)cmd;
	scmd->SCp.sent_command = jiffies;

	/*
	 * Issue the command to the FW
	 */
	atomic_inc(&instance->fw_outstanding);

	instance->instancet->fire_cmd(cmd->frame_phys_addr ,cmd->frame_count-1,instance->reg_set);

	/*
	 * Check if we have pend cmds to be completed
	 */
	if (poll_mode_io && atomic_read(&instance->fw_outstanding))
		megasas_complete_cmd_q(instance, DID_OK);

	return 0;

 out_return_cmd:
	megasas_return_cmd(instance, cmd);
 out_done:
	done(scmd);
	return 0;
}

static void
megasas_complete_cmd_q(struct megasas_instance *instance, u8 alt_status)
{
	u32 producer;
	u32 consumer;
	u32 context;
	struct megasas_cmd *cmd;
	unsigned long flags;

	spin_lock_irqsave(&instance->completion_lock, flags);

	producer = *instance->producer;
	consumer = *instance->consumer;

	while (consumer != producer) {
		context = instance->reply_queue[consumer];

		cmd = instance->cmd_list[context];

		megasas_complete_cmd(instance, cmd, alt_status);

		consumer++;
		if (consumer == (instance->max_fw_cmds + 1)) {
			consumer = 0;
		}
	}

	*instance->consumer = producer;

	spin_unlock_irqrestore(&instance->completion_lock, flags);
}

/**
 * megasas_wait_for_outstanding -	Wait for all outstanding cmds
 * @instance:				Adapter soft state
 *
 * This function waits for upto MEGASAS_RESET_WAIT_TIME seconds for FW to
 * complete all its outstanding commands. Returns error if one or more IOs
 * are pending after this time period. It also marks the controller dead.
 */
static int megasas_wait_for_outstanding(struct megasas_instance *instance)
{
	int i;
	u32 wait_time = MEGASAS_RESET_WAIT_TIME;

	for (i = 0; i < wait_time; i++) {

		int outstanding = atomic_read(&instance->fw_outstanding);

		if (!outstanding)
			break;

#ifdef __VMKERNEL_MODULE__
		/* don't log in the abort/reset path (PR #51280) */ ;
#else
		if (!(i % MEGASAS_RESET_NOTICE_INTERVAL)) {
			printk(KERN_NOTICE "megasas: [%2d]waiting for %d "
			       "commands to complete\n",i,outstanding);
		}
#endif
		/*
		 * Call cmd completion routine. Cmd to be 
		 * be completed directly without depending on isr.
		 */
		megasas_complete_cmd_q(instance, DID_OK);

		msleep(1000);
	}

	if (atomic_read(&instance->fw_outstanding)) {
		/*
		* Send signal to FW to stop processing any pending cmds.
		* The controller will be taken offline by the OS now.
		*/
		writel(MFI_STOP_ADP,
				&instance->reg_set->inbound_doorbell);
		megasas_dump_pending_frames(instance);
		instance->hw_crit_error = 1;
		return FAILED;
	}

	return SUCCESS;
}

/**
 * megasas_generic_reset -	Generic reset routine
 * @scmd:			Mid-layer SCSI command
 *
 * This routine implements a generic reset handler for device, bus and host
 * reset requests. Device, bus and host specific reset handlers can use this
 * function after they do their specific tasks.
 */
static int megasas_generic_reset(struct scsi_cmnd *scmd)
{
	int ret_val;
	struct megasas_instance *instance;

	instance = (struct megasas_instance *)scmd->host->hostdata;

	printk(KERN_NOTICE "megasas: RESET -%ld cmd=%x <c=%d t=%d l=%d> retries=%x\n",
	       scmd->serial_number, scmd->cmnd[0], scmd->channel,
	       scmd->target, scmd->lun, scmd->retries);

	if (instance->hw_crit_error) {
#ifdef __VMKERNEL_MODULE__
		/* don't log in the abort/reset path (PR #51280) */ ;
#else
		printk(KERN_ERR "megasas: cannot recover from previous reset "
		       "failures\n");
#endif
		return FAILED;
	}

#ifdef __VMKERNEL_MODULE__
	spin_unlock(&io_request_lock);
#else
	spin_unlock(scmd->host->host_lock);
#endif

	ret_val = megasas_wait_for_outstanding(instance);

#ifdef __VMKERNEL_MODULE__
	/* don't log in the abort/reset path (PR #51280) */

	spin_lock(&io_request_lock);
#else
	if (ret_val == SUCCESS)
		printk(KERN_NOTICE "megasas: reset successful \n");
	else
		printk(KERN_ERR "megasas: failed to do reset\n");
	spin_lock(scmd->host->host_lock);
#endif

	return ret_val;
}

/**
 * megasas_reset_device -	Device reset handler entry point
 */
static int megasas_reset_device(struct scsi_cmnd *scmd)
{
	int ret;

	/*
	 * First wait for all commands to complete
	 */
	ret = megasas_generic_reset(scmd);

	return ret;
}

/**
 * megasas_reset_bus_host -	Bus & host reset handler entry point
 */
static int megasas_reset_bus_host(struct scsi_cmnd *scmd)
{
	int ret;

	/*
	 * First wait for all commands to complete
	 */
	ret = megasas_generic_reset(scmd);

	return ret;
}

/**
 * megasas_service_aen -	Processes an event notification
 * @instance:			Adapter soft state
 * @cmd:			AEN command completed by the ISR
 *
 * For AEN, driver sends a command down to FW that is held by the FW till an
 * event occurs. When an event of interest occurs, FW completes the command
 * that it was previously holding.
 *
 * This routines sends SIGIO signal to processes that have registered with the
 * driver for AEN.
 */
static void
megasas_service_aen(struct megasas_instance *instance, struct megasas_cmd *cmd)
{
	/*
	 * Don't signal app if it is just an aborted previously registered aen
	 */
        if (!cmd->abort_aen) {
#ifdef __VMKERNEL_MODULE__
		megasas_wait_aen_poll = 1;

	        wake_up_interruptible(&megasas_poll_wait);
#endif
		kill_fasync(&megasas_async_queue, SIGIO, POLL_IN);
	}else
		cmd->abort_aen = 0;

	instance->aen_cmd = NULL;
	megasas_return_cmd(instance, cmd);
}

#ifdef __VMKERNEL_MODULE__
/*
 * megasas_detect - Detect stub to force 2.4 kernels to rescan luns
 */
static int megasas_detect(Scsi_Host_Template *template)
{
   return megasas_mgmt_info.count;
}
#endif

/*
 * Scsi host template for megaraid_sas driver
 */
static Scsi_Host_Template megasas_template = {

	.name = "MegaRAID SAS",
	.proc_name = "megaraid_sas",
	.info = megasas_info,
	.queuecommand = megasas_queue_command,
	.eh_device_reset_handler = megasas_reset_device,
	.eh_bus_reset_handler = megasas_reset_bus_host,
	.eh_host_reset_handler = megasas_reset_bus_host,
	.use_clustering = ENABLE_CLUSTERING,
	.use_new_eh_code = 1,
#ifndef __VMWARE__NO_ESX
	.highmem_io = 1,
	.vary_io = 1,
#endif
#ifdef __VMKERNEL_MODULE__
	.detect = megasas_detect,
#endif
};

/**
 * megasas_complete_int_cmd -	Completes an internal command
 * @instance:			Adapter soft state
 * @cmd:			Command to be completed
 *
 * The megasas_issue_blocked_cmd() function waits for a command to complete
 * after it issues a command. This function wakes up that waiting routine by
 * calling wake_up() on the wait queue.
 */
static void
megasas_complete_int_cmd(struct megasas_instance *instance,
			 struct megasas_cmd *cmd)
{
	cmd->cmd_status = cmd->frame->io.cmd_status;

	if (cmd->cmd_status == ENODATA) {
		cmd->cmd_status = 0;
	}
	wake_up(&instance->int_cmd_wait_q);
}

/**
 * megasas_complete_abort -	Completes aborting a command
 * @instance:			Adapter soft state
 * @cmd:			Cmd that was issued to abort another cmd
 *
 * The megasas_issue_blocked_abort_cmd() function waits on abort_cmd_wait_q
 * after it issues an abort on a previously issued command. This function
 * wakes up all functions waiting on the same wait queue.
 */
static void
megasas_complete_abort(struct megasas_instance *instance,
		       struct megasas_cmd *cmd)
{
	if (cmd->sync_cmd) {
		cmd->sync_cmd = 0;
		cmd->cmd_status = 0;
		wake_up(&instance->abort_cmd_wait_q);
	}

	return;
}

/**
 * megasas_unmap_sgbuf -	Unmap SG buffers
 * @instance:			Adapter soft state
 * @cmd:			Completed command
 */
static inline void
megasas_unmap_sgbuf(struct megasas_instance *instance, struct megasas_cmd *cmd)
{
	dma_addr_t buf_h;
	u8 opcode;

#ifndef __VMKERNEL_MODULE__
	if (cmd->scmd->use_sg) {
		pci_unmap_sg(instance->pdev, cmd->scmd->request_buffer,
			     cmd->scmd->use_sg, cmd->scmd->sc_data_direction);
		return;
	}

	if (!cmd->scmd->request_bufflen)
		return;

	opcode = cmd->frame->hdr.cmd;

	if ((opcode == MFI_CMD_LD_READ) || (opcode == MFI_CMD_LD_WRITE)) {
		if (IS_DMA64)
			buf_h = cmd->frame->io.sgl.sge64[0].phys_addr;
		else
			buf_h = cmd->frame->io.sgl.sge32[0].phys_addr;
	} else {
		if (IS_DMA64)
			buf_h = cmd->frame->pthru.sgl.sge64[0].phys_addr;
		else
			buf_h = cmd->frame->pthru.sgl.sge32[0].phys_addr;
	}

	pci_unmap_single(instance->pdev, buf_h, cmd->scmd->request_bufflen,
			 cmd->scmd->sc_data_direction);
#endif
	return;
}

/**
 * megasas_complete_cmd -	Completes a command
 * @instance:			Adapter soft state
 * @cmd:			Command to be completed
 * @alt_status:			If non-zero, use this value as status to
 * 				SCSI mid-layer instead of the value returned
 * 				by the FW. This should be used if caller wants
 * 				an alternate status (as in the case of aborted
 * 				commands)
 */
static inline void
megasas_complete_cmd(struct megasas_instance *instance, struct megasas_cmd *cmd,
		     u8 alt_status)
{
	int exception = 0;
	struct megasas_header *hdr = &cmd->frame->hdr;
	int outstanding;
	if (cmd->scmd) {
		cmd->scmd->SCp.ptr = (char *)0;
	}


	switch (hdr->cmd) {

	case MFI_CMD_PD_SCSI_IO:
	case MFI_CMD_LD_SCSI_IO:

		/*
		 * MFI_CMD_PD_SCSI_IO and MFI_CMD_LD_SCSI_IO could have been
		 * issued either through an IO path or an IOCTL path. If it
		 * was via IOCTL, we will send it to internal completion.
		 */
		if (cmd->sync_cmd) {
			cmd->sync_cmd = 0;
			megasas_complete_int_cmd(instance, cmd);
			break;
		}

		/*
		 * Don't export physical disk devices to mid-layer.
		 */
		if (!MEGASAS_IS_LOGICAL(cmd->scmd) &&
		    (hdr->cmd_status == MFI_STAT_OK) &&
		    (cmd->scmd->cmnd[0] == INQUIRY)) {

#ifdef __VMKERNEL_MODULE__
			void *vaddr = vmk_phys_to_kmap(
						cmd->scmd->request_bufferMA,
						1);
			if (((*(u8 *) vaddr) & 0x1F) == TYPE_DISK) {
#else
			if (((*(u8 *) cmd->scmd->request_buffer) & 0x1F) ==
			    TYPE_DISK) {
#endif
				cmd->scmd->result = DID_BAD_TARGET << 16;
				exception = 1;
			}
#ifdef __VMKERNEL_MODULE__
			vmk_phys_to_kmap_free(vaddr);
#endif
		}

	case MFI_CMD_LD_READ:
	case MFI_CMD_LD_WRITE:

		if (alt_status) {
			cmd->scmd->result = alt_status << 16;
			exception = 1;
		}

		if (exception) {

			atomic_dec(&instance->fw_outstanding);

			megasas_unmap_sgbuf(instance, cmd);
			cmd->scmd->scsi_done(cmd->scmd);
			megasas_return_cmd(instance, cmd);

			break;
		}

		switch (hdr->cmd_status) {

		case MFI_STAT_OK:
			cmd->scmd->result = DID_OK << 16;
			break;

		case MFI_STAT_SCSI_IO_FAILED:
		case MFI_STAT_LD_INIT_IN_PROGRESS:
			cmd->scmd->result =
			    (DID_ERROR << 16) | hdr->scsi_status;
			break;

		case MFI_STAT_SCSI_DONE_WITH_ERROR:

			cmd->scmd->result = (DID_OK << 16) | hdr->scsi_status;

			if (hdr->scsi_status == SAM_STAT_CHECK_CONDITION) {
				memset(cmd->scmd->sense_buffer, 0,
				       SCSI_SENSE_BUFFERSIZE);
				memcpy(cmd->scmd->sense_buffer, cmd->sense,
				       hdr->sense_len);

				cmd->scmd->result |= DRIVER_SENSE << 24;
			}

			break;

		case MFI_STAT_LD_OFFLINE:
		case MFI_STAT_DEVICE_NOT_FOUND:
			cmd->scmd->result = DID_BAD_TARGET << 16;
			break;

		default:
			printk(KERN_DEBUG "megasas: MFI FW status %#x\n",
			       hdr->cmd_status);
			cmd->scmd->result = DID_ERROR << 16;
			break;
		}

		atomic_dec(&instance->fw_outstanding);

		megasas_unmap_sgbuf(instance, cmd);
		cmd->scmd->scsi_done(cmd->scmd);
		megasas_return_cmd(instance, cmd);

		break;

	case MFI_CMD_SMP:
	case MFI_CMD_STP:
	case MFI_CMD_DCMD:

		/*
		 * See if got an event notification
		 */
		if (cmd->frame->dcmd.opcode == MR_DCMD_CTRL_EVENT_WAIT)
			megasas_service_aen(instance, cmd);
		else
			megasas_complete_int_cmd(instance, cmd);

		break;

	case MFI_CMD_ABORT:
		/*
		 * Cmd issued to abort another cmd returned
		 */
		megasas_complete_abort(instance, cmd);
		break;

	default:
		printk("megasas: Unknown command completed! [0x%X]\n",
		       hdr->cmd);
		break;
	}
	if(instance->is_busy){
		outstanding = atomic_read(&instance->fw_outstanding);
		if(outstanding<17)
			instance->is_busy=0;
	}
}

/**
 * megasas_deplete_reply_queue -	Processes all completed commands
 * @instance:				Adapter soft state
 * @alt_status:				Alternate status to be returned to
 * 					SCSI mid-layer instead of the status
 * 					returned by the FW
 */
static inline void
megasas_deplete_reply_queue(struct megasas_instance *instance, u8 alt_status)
{
	/*
	 * Check if it is our interrupt
	 * Clear the interrupt
	 */
	if(instance->instancet->clear_intr(instance->reg_set))
		return IRQ_NONE;

	if(instance->hw_crit_error)
		goto out_done;

	megasas_complete_cmd_q(instance, alt_status);
out_done:
	return IRQ_HANDLED;

}

/**
 * megasas_isr - isr entry point
 */
static irqreturn_t megasas_isr(int irq, void *devp, struct pt_regs *regs)
{
	return megasas_deplete_reply_queue((struct megasas_instance *)devp,
					   DID_OK);
}

/**
 * megasas_transition_to_ready -	Move the FW to READY state
 * @instance:				Adapter soft state
 *
 * During the initialization, FW passes can potentially be in any one of
 * several possible states. If the FW in operational, waiting-for-handshake
 * states, driver must take steps to bring it to ready state. Otherwise, it
 * has to wait for the ready state.
 */
static int
megasas_transition_to_ready(struct megasas_instance* instance)
{
	int i;
	u8 max_wait;
	u32 fw_state;
	u32 cur_state;

	fw_state = instance->instancet->read_fw_status_reg(instance->reg_set) & MFI_STATE_MASK;

	if (fw_state != MFI_STATE_READY)
 		printk(KERN_INFO "megasas: Waiting for FW to come to ready"
 		       " state\n");

	while (fw_state != MFI_STATE_READY) {

		switch (fw_state) {

		case MFI_STATE_FAULT:

			printk(KERN_DEBUG "megasas: FW in FAULT state!!\n");
			return -ENODEV;

		case MFI_STATE_WAIT_HANDSHAKE:
			/*
			 * Set the CLR bit in inbound doorbell
			 */
			writel(MFI_INIT_CLEAR_HANDSHAKE|MFI_INIT_HOTPLUG,
				&instance->reg_set->inbound_doorbell);

			max_wait = 2;
			cur_state = MFI_STATE_WAIT_HANDSHAKE;
			break;

		case MFI_STATE_BOOT_MESSAGE_PENDING:
			writel(MFI_INIT_HOTPLUG,
				&instance->reg_set->inbound_doorbell);

			max_wait = 10;
			cur_state = MFI_STATE_BOOT_MESSAGE_PENDING;
			break;

		case MFI_STATE_OPERATIONAL:
			/*
			 * Bring it to READY state; assuming max wait 10 secs
			 */
			instance->instancet->disable_intr(instance->reg_set);
			writel(MFI_RESET_FLAGS, &instance->reg_set->inbound_doorbell);

			max_wait = 10;
			cur_state = MFI_STATE_OPERATIONAL;
			break;

		case MFI_STATE_UNDEFINED:
			/*
			 * This state should not last for more than 2 seconds
			 */
			max_wait = 2;
			cur_state = MFI_STATE_UNDEFINED;
			break;

		case MFI_STATE_BB_INIT:
			max_wait = 2;
			cur_state = MFI_STATE_BB_INIT;
			break;

		case MFI_STATE_FW_INIT:
			max_wait = 20;
			cur_state = MFI_STATE_FW_INIT;
			break;

		case MFI_STATE_FW_INIT_2:
			max_wait = 20;
			cur_state = MFI_STATE_FW_INIT_2;
			break;

		case MFI_STATE_DEVICE_SCAN:
			max_wait = 20;
			cur_state = MFI_STATE_DEVICE_SCAN;
			break;

		case MFI_STATE_FLUSH_CACHE:
			max_wait = 20;
			cur_state = MFI_STATE_FLUSH_CACHE;
			break;

		default:
			printk(KERN_DEBUG "megasas: Unknown state 0x%x\n",
			       fw_state);
			return -ENODEV;
		}

		/*
		 * The cur_state should not last for more than max_wait secs
		 */
		for (i = 0; i < (max_wait * 1000); i++) {
			fw_state = instance->instancet->read_fw_status_reg(instance->reg_set) &
					MFI_STATE_MASK ;

			if (fw_state == cur_state) {
				msleep(1);
			} else
				break;
		}

		/*
		 * Return error if fw_state hasn't changed after max_wait
		 */
		if (fw_state == cur_state) {
			printk(KERN_DEBUG "FW state [%d] hasn't changed "
			       "in %d secs\n", fw_state, max_wait);
			return -ENODEV;
		}
	};
 	printk(KERN_INFO "megasas: FW now in Ready state\n");

	return 0;
}

/**
 * megasas_teardown_frame_pool -	Destroy the cmd frame DMA pool
 * @instance:				Adapter soft state
 */
static void megasas_teardown_frame_pool(struct megasas_instance *instance)
{
	int i;
	u32 max_cmd = instance->max_fw_cmds;
	struct megasas_cmd *cmd;

	if (!instance->frame_dma_pool)
		return;

	/*
	 * Return all frames to pool
	 */
	for (i = 0; i < max_cmd; i++) {

		cmd = instance->cmd_list[i];

		if (cmd->frame)
			pci_pool_free(instance->frame_dma_pool, cmd->frame,
				      cmd->frame_phys_addr);

		if (cmd->sense)
			pci_pool_free(instance->sense_dma_pool, cmd->sense,
				      cmd->sense_phys_addr);
	}

	/*
	 * Now destroy the pool itself
	 */
	pci_pool_destroy(instance->frame_dma_pool);
	pci_pool_destroy(instance->sense_dma_pool);

	instance->frame_dma_pool = NULL;
	instance->sense_dma_pool = NULL;
}

/**
 * megasas_create_frame_pool -	Creates DMA pool for cmd frames
 * @instance:			Adapter soft state
 *
 * Each command packet has an embedded DMA memory buffer that is used for
 * filling MFI frame and the SG list that immediately follows the frame. This
 * function creates those DMA memory buffers for each command packet by using
 * PCI pool facility.
 */
static int megasas_create_frame_pool(struct megasas_instance *instance)
{
	int i;
	u32 max_cmd;
	u32 sge_sz;
	u32 sgl_sz;
	u32 total_sz;
	u32 frame_count;
	struct megasas_cmd *cmd;

	max_cmd = instance->max_fw_cmds;

	/*
	 * Size of our frame is 64 bytes for MFI frame, followed by max SG
	 * elements and finally SCSI_SENSE_BUFFERSIZE bytes for sense buffer
	 */
	sge_sz = (IS_DMA64) ? sizeof(struct megasas_sge64) :
	    sizeof(struct megasas_sge32);

	/*
	 * Calculated the number of 64byte frames required for SGL
	 */
	sgl_sz = sge_sz * instance->max_num_sge;
	frame_count = (sgl_sz + MEGAMFI_FRAME_SIZE - 1) / MEGAMFI_FRAME_SIZE;

	/*
	 * We need one extra frame for the MFI command
	 */
	frame_count++;

	total_sz = MEGAMFI_FRAME_SIZE * frame_count;
	/*
	 * Use DMA pool facility provided by PCI layer
	 */
	instance->frame_dma_pool = pci_pool_create("megasas frame pool",
						   instance->pdev, total_sz, 64,
						   0, GFP_DMA);

	if (!instance->frame_dma_pool) {
		printk(KERN_DEBUG "megasas: failed to setup frame pool\n");
		return -ENOMEM;
	}

	instance->sense_dma_pool = pci_pool_create("megasas sense pool",
						   instance->pdev, 128, 4, 0,
						   GFP_DMA);

	if (!instance->sense_dma_pool) {
		printk(KERN_DEBUG "megasas: failed to setup sense pool\n");

		pci_pool_destroy(instance->frame_dma_pool);
		instance->frame_dma_pool = NULL;

		return -ENOMEM;
	}

	/*
	 * Allocate and attach a frame to each of the commands in cmd_list.
	 * By making cmd->index as the context instead of the &cmd, we can
	 * always use 32bit context regardless of the architecture
	 */
	for (i = 0; i < max_cmd; i++) {

		cmd = instance->cmd_list[i];

		cmd->frame = pci_pool_alloc(instance->frame_dma_pool,
					    GFP_KERNEL, &cmd->frame_phys_addr);

		cmd->sense = pci_pool_alloc(instance->sense_dma_pool,
					    GFP_KERNEL, &cmd->sense_phys_addr);

		/*
		 * megasas_teardown_frame_pool() takes care of freeing
		 * whatever has been allocated
		 */
		if (!cmd->frame || !cmd->sense) {
			printk(KERN_DEBUG "megasas: pci_pool_alloc failed \n");
			megasas_teardown_frame_pool(instance);
			return -ENOMEM;
		}

		cmd->frame->io.context = cmd->index;
	}

	return 0;
}

/**
 * megasas_free_cmds -	Free all the cmds in the free cmd pool
 * @instance:		Adapter soft state
 */
static void megasas_free_cmds(struct megasas_instance *instance)
{
	int i;
	/* First free the MFI frame pool */
	megasas_teardown_frame_pool(instance);

	/* Free all the commands in the cmd_list */
	for (i = 0; i < instance->max_fw_cmds; i++)
		kfree(instance->cmd_list[i]);

	/* Free the cmd_list buffer itself */
	kfree(instance->cmd_list);
	instance->cmd_list = NULL;

	INIT_LIST_HEAD(&instance->cmd_pool);
}

/**
 * megasas_alloc_cmds -	Allocates the command packets
 * @instance:		Adapter soft state
 *
 * Each command that is issued to the FW, whether IO commands from the OS or
 * internal commands like IOCTLs, are wrapped in local data structure called
 * megasas_cmd. The frame embedded in this megasas_cmd is actually issued to
 * the FW.
 *
 * Each frame has a 32-bit field called context (tag). This context is used
 * to get back the megasas_cmd from the frame when a frame gets completed in
 * the ISR. Typically the address of the megasas_cmd itself would be used as
 * the context. But we wanted to keep the differences between 32 and 64 bit
 * systems to the mininum. We always use 32 bit integers for the context. In
 * this driver, the 32 bit values are the indices into an array cmd_list.
 * This array is used only to look up the megasas_cmd given the context. The
 * free commands themselves are maintained in a linked list called cmd_pool.
 */
static int megasas_alloc_cmds(struct megasas_instance *instance)
{
	int i;
	int j;
	u32 max_cmd;
	struct megasas_cmd *cmd;

	max_cmd = instance->max_fw_cmds;

	/*
	 * instance->cmd_list is an array of struct megasas_cmd pointers.
	 * Allocate the dynamic array first and then allocate individual
	 * commands.
	 */
	instance->cmd_list = kmalloc(sizeof(struct megasas_cmd *) * max_cmd,
				     GFP_KERNEL);

	if (!instance->cmd_list) {
		printk(KERN_DEBUG "megasas: out of memory\n");
		return -ENOMEM;
	}

	memset(instance->cmd_list, 0, sizeof(struct megasas_cmd *) * max_cmd);

	for (i = 0; i < max_cmd; i++) {
		instance->cmd_list[i] = kmalloc(sizeof(struct megasas_cmd),
						GFP_KERNEL);

		if (!instance->cmd_list[i]) {

			for (j = 0; j < i; j++)
				kfree(instance->cmd_list[j]);

			kfree(instance->cmd_list);
			instance->cmd_list = NULL;

			return -ENOMEM;
		}
	}

	/*
	 * Add all the commands to command pool (instance->cmd_pool)
	 */
	for (i = 0; i < max_cmd; i++) {
		cmd = instance->cmd_list[i];
		memset(cmd, 0, sizeof(struct megasas_cmd));
		cmd->index = i;
		cmd->instance = instance;

		list_add_tail(&cmd->list, &instance->cmd_pool);
	}

	/*
	 * Create a frame pool and assign one frame to each cmd
	 */
	if (megasas_create_frame_pool(instance)) {
		printk(KERN_DEBUG "megasas: Error creating frame DMA pool\n");
		megasas_free_cmds(instance);
	}

	return 0;
}

/**
 * megasas_get_controller_info -	Returns FW's controller structure
 * @instance:				Adapter soft state
 * @ctrl_info:				Controller information structure
 *
 * Issues an internal command (DCMD) to get the FW's controller structure.
 * This information is mainly used to find out the maximum IO transfer per
 * command supported by the FW.
 */
static int
megasas_get_ctrl_info(struct megasas_instance *instance,
		      struct megasas_ctrl_info *ctrl_info)
{
	int ret = 0;
	struct megasas_cmd *cmd;
	struct megasas_dcmd_frame *dcmd;
	struct megasas_ctrl_info *ci;
	dma_addr_t ci_h = 0;

	cmd = megasas_get_cmd(instance);

	if (!cmd) {
		printk(KERN_DEBUG "megasas: Failed to get a free cmd\n");
		return -ENOMEM;
	}

	dcmd = &cmd->frame->dcmd;

	ci = pci_alloc_consistent(instance->pdev,
				  sizeof(struct megasas_ctrl_info), &ci_h);

	if (!ci) {
		printk(KERN_DEBUG "Failed to alloc mem for ctrl info\n");
		megasas_return_cmd(instance, cmd);
		return -ENOMEM;
	}

	memset(ci, 0, sizeof(*ci));
	memset(dcmd->mbox.b, 0, MFI_MBOX_SIZE);

	dcmd->cmd = MFI_CMD_DCMD;
	dcmd->cmd_status = 0xFF;
	dcmd->sge_count = 1;
	dcmd->flags = MFI_FRAME_DIR_READ;
	dcmd->timeout = 0;
	dcmd->data_xfer_len = sizeof(struct megasas_ctrl_info);
	dcmd->opcode = MR_DCMD_CTRL_GET_INFO;
	dcmd->sgl.sge32[0].phys_addr = ci_h;
	dcmd->sgl.sge32[0].length = sizeof(struct megasas_ctrl_info);

	if (!megasas_issue_polled(instance, cmd)) {
		ret = 0;
		memcpy(ctrl_info, ci, sizeof(struct megasas_ctrl_info));
	} else {
		ret = -1;
	}

	pci_free_consistent(instance->pdev, sizeof(struct megasas_ctrl_info),
			    ci, ci_h);

	megasas_return_cmd(instance, cmd);
	return ret;
}

/**
 * megasas_get_ioctl_mem -	Get a buff from the free ioctl memory pool
 * @instance:			Adapter soft state
 * @i:				mem_pool_index
 *
 * Returns a free buff from the pool
 */
static inline struct megasas_ioctl_mm *
megasas_get_ioctl_mem(struct megasas_instance* instance, u8 i)
{
	unsigned long flags;
	struct megasas_ioctl_mm *ioctl_mm = NULL;

	if(i >= MAX_IOCTL_MEM_POOL)
		goto out;

	spin_lock_irqsave(&instance->ioctl_memory_pool_lock, flags);

	if (!list_empty(&instance->ioctl_memory_pool[i])) {
		ioctl_mm = list_entry((&instance->ioctl_memory_pool[i])->next,
				 struct megasas_ioctl_mm, list);
		list_del_init(&ioctl_mm->list);
	} else
		printk(KERN_ERR "megasas: ioctl memory pool empty!\n");

	spin_unlock_irqrestore(&instance->ioctl_memory_pool_lock, flags);
out:
	return ioctl_mm;
}


/**
 * megasas_return_ioctl_mm -	Return a memory to ioctl memory pool
 * @instance:		Adapter soft state
 * @megasas_ioctl_mm:	ioctl mem block
 * @i:			mem_pool_index
 *
 */
static inline void
megasas_return_ioctl_mem(struct megasas_instance *instance,
			struct megasas_ioctl_mm *ioctl_mm, u8 i)
{
	unsigned long flags;
	if(!ioctl_mm)
	{
		printk(KERN_DEBUG "megasas: Trying to return NULL to mem_pool\n");
		return;
	}
	spin_lock_irqsave(&instance->ioctl_memory_pool_lock, flags);
	list_add_tail(&ioctl_mm->list, &instance->ioctl_memory_pool[i]);
	spin_unlock_irqrestore(&instance->ioctl_memory_pool_lock, flags);
}

/**
 * megasas_free_ioctl_mem_pools -	Free all the memory in the ioctl memory pool
 * @instance:		Adapter soft state
 */
static void
megasas_free_ioctl_mem_pools(struct megasas_instance *instance)
{
	struct megasas_ioctl_mm	*ioctl_mm;
	u32 i,j;
	u32 mem_size;
	if(instance->mem_pool_empty)
		return;
	for (i=0; i < MAX_IOCTL_MEM_POOL; i++) {
		mem_size = arr_size[i];
		for (j = 0; j < arr[i]; j++) {
			ioctl_mm = megasas_get_ioctl_mem(instance, i);

			if (ioctl_mm){
				if(ioctl_mm->vaddr)
					pci_free_consistent(instance->pdev,
							mem_size,
							ioctl_mm->vaddr,
						 	ioctl_mm->buf_handle);
				kfree(ioctl_mm);
			}
		}
	}
}

/*
 * The memory pool will have 8x4K, 4*8K and 1x64K memory
 * ioctl_memory_pool[0] ----> 4k memory list
 * ioctl_memory_pool[1] ----> 8K memory list
 * ioctl_memory_pool[2] ----> 64K memory list
 */

/**
 * megasas_setup_mem_pools - Free all the memory in the ioctl memory pool
 * @instance:		Adapter soft state
 */
static int
megasas_setup_mem_pools(struct megasas_instance *instance)
{
	struct megasas_ioctl_mm	*ioctl_mm;
	dma_addr_t buf_handle;
	void *vaddr;
	u32 i, j;
	u32 mem_size;
	for (i=0; i < MAX_IOCTL_MEM_POOL; i++) {
		mem_size = arr_size[i];

		for (j = 0; j < arr[i]; j++) {

			ioctl_mm = kmalloc(sizeof(struct megasas_ioctl_mm), GFP_KERNEL);
			if(!ioctl_mm){
				printk(KERN_DEBUG "megasas:Failed to alloc init memory for ioctl \n");
				goto failed_to_alloc_mem_ioctl;
			}
			vaddr = pci_alloc_consistent(instance->pdev,mem_size,
							    &buf_handle);
			if (!vaddr) {
				printk(KERN_DEBUG "megasas:Failed to alloc init memory for ioctl \n");
				goto failed_to_alloc_mem_ioctl;
			} else {
				ioctl_mm->vaddr	= vaddr;
				ioctl_mm->buf_handle = (u32)buf_handle;

				list_add_tail(&ioctl_mm->list,
					&instance->ioctl_memory_pool[i]);
			}
		}
	}
	return 0;
failed_to_alloc_mem_ioctl:
	megasas_free_ioctl_mem_pools(instance);
	return 1;
}

int
megasas_get_buff_for_sge(struct megasas_iocpacket *ioc,
			struct megasas_instance *instance,
			struct megasas_cmd *cmd)
{
	int i,n;
	u8 mem_type;
	struct megasas_ioctl_mm* ioctl_mm;
	if(instance->mem_pool_empty)
		goto empty;
	for (i = 0; i < ioc->sge_count; i++) {
		/*
		 * Check if we have buffer to use from our mem_pool
		 * If we donot have enough to satisfy for all sge's then
		 * free that has already been attached to the ioc
		 */
		mem_type=0xff;
		if (ioc->sgl[i].iov_len <= MEGASAS_INIT_IOCTL_MEM_SIZE)
			mem_type=0;
		else if ((ioc->sgl[i].iov_len > MEGASAS_INIT_IOCTL_MEM_SIZE) &&
			(ioc->sgl[i].iov_len <= 2*MEGASAS_INIT_IOCTL_MEM_SIZE))
			mem_type=1;
		else if ((ioc->sgl[i].iov_len > 2*MEGASAS_INIT_IOCTL_MEM_SIZE) &&
			(ioc->sgl[i].iov_len <= 16*MEGASAS_INIT_IOCTL_MEM_SIZE))
			mem_type=2;

		ioctl_mm = megasas_get_ioctl_mem(instance, mem_type);

		if (ioctl_mm) {
			cmd->ioctl_mem[i]=ioctl_mm;
			cmd->ioctl_mem_pool_index[i] = mem_type;
		} else {
			/* Not enough buffer in mem pool
			 * Free all allocated buffer for this ioc
			 */
			goto out;
		}
	}
	return 0;
out:
	for (n = 0; n < i; n++) {
		if ((struct megasas_ioctl_mm *)cmd->ioctl_mem[i]) {
			megasas_return_ioctl_mem(instance,
					(struct megasas_ioctl_mm *)cmd->ioctl_mem[i],
					cmd->ioctl_mem_pool_index[i]);
			cmd->ioctl_mem[i]=NULL;
			cmd->ioctl_mem_pool_index[i] = 0xff;
		}
	}
empty:
	return 1;
}

/**
 * megasas_init_mfi -	Initializes the FW
 * @instance:		Adapter soft state
 *
 * This is the main function for initializing MFI firmware.
 */
static int megasas_init_mfi(struct megasas_instance *instance)
{
	u32 context_sz;
	u32 reply_q_sz;
	u32 max_sectors_1;
	u32 max_sectors_2;
	u32 tmp_sectors;
	struct megasas_register_set __iomem *reg_set;

	struct megasas_cmd *cmd;
	struct megasas_ctrl_info *ctrl_info;

	struct megasas_init_frame *init_frame;
	struct megasas_init_queue_info *initq_info;
	dma_addr_t init_frame_h;
	dma_addr_t initq_info_h;

	/*
	 * Map the message registers
	 */
	instance->base_addr = pci_resource_start(instance->pdev, 0);

	if (pci_request_regions(instance->pdev, "megasas: LSI Logic")) {
		printk(KERN_DEBUG "megasas: IO memory region busy!\n");
		return -EBUSY;
	}

	instance->reg_set = ioremap_nocache(instance->base_addr, 8192);

	if (!instance->reg_set) {
		printk(KERN_DEBUG "megasas: Failed to map IO mem\n");
		goto fail_ioremap;
	}

	reg_set = instance->reg_set;

	switch(instance->pdev->device)
	{
		case PCI_DEVICE_ID_LSI_SAS1078R:
		case PCI_DEVICE_ID_LSI_SAS1078DE:
			instance->instancet = &megasas_instance_template_ppc;
			break;
		case PCI_DEVICE_ID_LSI_SAS1064R:
		case PCI_DEVICE_ID_DELL_PERC5:
		default:
			instance->instancet = &megasas_instance_template_xscale;
			break;
	}

	/*
	 * We expect the FW state to be READY
	 */
	if (megasas_transition_to_ready(instance))
		goto fail_ready_state;

	/*
	 * Get various operational parameters from status register
	 */
	instance->max_fw_cmds = instance->instancet->read_fw_status_reg(reg_set) & 0x00FFFF;
	/*
	 * Reduce the max supported cmds by 1. This is to ensure that the
	 * reply_q_sz (1 more than the max cmd that driver may send)
	 * does not exceed max cmds that the FW can support
	 */
	instance->max_fw_cmds = instance->max_fw_cmds-1;
	instance->max_num_sge = (instance->instancet->read_fw_status_reg(reg_set) & 0xFF0000) >>
					0x10;
	/*
	 * Create a pool of commands
	 */
	if (megasas_alloc_cmds(instance))
		goto fail_alloc_cmds;

	/*
	 * Allocate memory for reply queue. Length of reply queue should
	 * be _one_ more than the maximum commands handled by the firmware.
	 *
	 * Note: When FW completes commands, it places corresponding contex
	 * values in this circular reply queue. This circular queue is a fairly
	 * typical producer-consumer queue. FW is the producer (of completed
	 * commands) and the driver is the consumer.
	 */
	context_sz = sizeof(u32);
	reply_q_sz = context_sz * (instance->max_fw_cmds + 1);

	instance->reply_queue = pci_alloc_consistent(instance->pdev,
						     reply_q_sz,
						     &instance->reply_queue_h);

	if (!instance->reply_queue) {
		printk(KERN_DEBUG "megasas: Out of DMA mem for reply queue\n");
		goto fail_reply_queue;
	}

	/*
	 * Prepare a init frame. Note the init frame points to queue info
	 * structure. Each frame has SGL allocated after first 64 bytes. For
	 * this frame - since we don't need any SGL - we use SGL's space as
	 * queue info structure
	 *
	 * We will not get a NULL command below. We just created the pool.
	 */
	cmd = megasas_get_cmd(instance);

	init_frame = (struct megasas_init_frame *)cmd->frame;
	initq_info = (struct megasas_init_queue_info *)
	    ((unsigned long)init_frame + 64);

	init_frame_h = cmd->frame_phys_addr;
	initq_info_h = init_frame_h + 64;

	memset(init_frame, 0, MEGAMFI_FRAME_SIZE);
	memset(initq_info, 0, sizeof(struct megasas_init_queue_info));

	initq_info->reply_queue_entries = instance->max_fw_cmds + 1;
	initq_info->reply_queue_start_phys_addr_lo = instance->reply_queue_h;

	initq_info->producer_index_phys_addr_lo = instance->producer_h;
	initq_info->consumer_index_phys_addr_lo = instance->consumer_h;

	init_frame->cmd = MFI_CMD_INIT;
	init_frame->cmd_status = 0xFF;
	init_frame->queue_info_new_phys_addr_lo = initq_info_h;

	init_frame->data_xfer_len = sizeof(struct megasas_init_queue_info);

	/*
	 * disable the intr before firing the init frame to FW
	 */
	instance->instancet->disable_intr(instance->reg_set);

	/*
	 * Issue the init frame in polled mode
	 */
	if (megasas_issue_polled(instance, cmd)) {
		printk(KERN_DEBUG "megasas: Failed to init firmware\n");
		goto fail_fw_init;
	}

	megasas_return_cmd(instance, cmd);

	ctrl_info = kmalloc(sizeof(struct megasas_ctrl_info), GFP_KERNEL);

	/*
	 * Compute the max allowed sectors per IO: The controller info has two
	 * limits on max sectors. Driver should use the minimum of these two.
	 *
	 * 1 << stripe_sz_ops.min = max sectors per strip
	 *
	 * Note that older firmwares ( < FW ver 30) didn't report information
	 * to calculate max_sectors_1. So the number ended up as zero always.
	 */
	tmp_sectors = 0;
	if (ctrl_info && !megasas_get_ctrl_info(instance, ctrl_info)) {

		max_sectors_1 = (1 << ctrl_info->stripe_sz_ops.min) *
		    ctrl_info->max_strips_per_io;
		max_sectors_2 = ctrl_info->max_request_size;

		tmp_sectors = (max_sectors_1 < max_sectors_2)
		    ? max_sectors_1 : max_sectors_2;
	}

	instance->max_sectors_per_req = instance->max_num_sge *
		    PAGE_SIZE / 512;
	if (tmp_sectors && (instance->max_sectors_per_req > tmp_sectors))
		instance->max_sectors_per_req = tmp_sectors;

	kfree(ctrl_info);


#ifdef __VMKERNEL_MODULE__
	megasas_wait_aen_poll = 0;
	init_waitqueue_head(&megasas_poll_wait);
#endif

	/* Initialize the cmd completion timer */
	if (poll_mode_io)
		megasas_start_timer(instance, &instance->io_completion_timer,
				megasas_io_completion_timer,
				MEGASAS_COMPLETION_TIMER_INTERVAL);
	return 0;

      fail_fw_init:
	megasas_return_cmd(instance, cmd);

	pci_free_consistent(instance->pdev, reply_q_sz,
			    instance->reply_queue, instance->reply_queue_h);
      fail_reply_queue:
	megasas_free_cmds(instance);

      fail_alloc_cmds:
      fail_ready_state:
	iounmap(instance->reg_set);

      fail_ioremap:
	pci_release_regions(instance->pdev);

	return -EINVAL;
}

/**
 * megasas_release_mfi -	Reverses the FW initialization
 * @intance:			Adapter soft state
 */
static void megasas_release_mfi(struct megasas_instance *instance)
{
	u32 reply_q_sz = sizeof(u32) * (instance->max_fw_cmds + 1);

	pci_free_consistent(instance->pdev, reply_q_sz,
			    instance->reply_queue, instance->reply_queue_h);

	megasas_free_cmds(instance);

	iounmap(instance->reg_set);

	pci_release_regions(instance->pdev);
}

/**
 * megasas_get_seq_num -	Gets latest event sequence numbers
 * @instance:			Adapter soft state
 * @eli:			FW event log sequence numbers information
 *
 * FW maintains a log of all events in a non-volatile area. Upper layers would
 * usually find out the latest sequence number of the events, the seq number at
 * the boot etc. They would "read" all the events below the latest seq number
 * by issuing a direct fw cmd (DCMD). For the future events (beyond latest seq
 * number), they would subsribe to AEN (asynchronous event notification) and
 * wait for the events to happen.
 */
static int
megasas_get_seq_num(struct megasas_instance *instance,
		    struct megasas_evt_log_info *eli)
{
	struct megasas_cmd *cmd;
	struct megasas_dcmd_frame *dcmd;
	struct megasas_evt_log_info *el_info;
	dma_addr_t el_info_h = 0;

	cmd = megasas_get_cmd(instance);

	if (!cmd) {
		return -ENOMEM;
	}

	dcmd = &cmd->frame->dcmd;
	el_info = pci_alloc_consistent(instance->pdev,
				       sizeof(struct megasas_evt_log_info),
				       &el_info_h);

	if (!el_info) {
		megasas_return_cmd(instance, cmd);
		return -ENOMEM;
	}

	memset(el_info, 0, sizeof(*el_info));
	memset(dcmd->mbox.b, 0, MFI_MBOX_SIZE);

	dcmd->cmd = MFI_CMD_DCMD;
	dcmd->cmd_status = 0x0;
	dcmd->sge_count = 1;
	dcmd->flags = MFI_FRAME_DIR_READ;
	dcmd->timeout = 0;
	dcmd->data_xfer_len = sizeof(struct megasas_evt_log_info);
	dcmd->opcode = MR_DCMD_CTRL_EVENT_GET_INFO;
	dcmd->sgl.sge32[0].phys_addr = el_info_h;
	dcmd->sgl.sge32[0].length = sizeof(struct megasas_evt_log_info);

	megasas_issue_blocked_cmd(instance, cmd);

	/*
	 * Copy the data back into callers buffer
	 */
	memcpy(eli, el_info, sizeof(struct megasas_evt_log_info));

	pci_free_consistent(instance->pdev, sizeof(struct megasas_evt_log_info),
			    el_info, el_info_h);

	megasas_return_cmd(instance, cmd);

	return 0;
}

/**
 * megasas_register_aen -	Registers for asynchronous event notification
 * @instance:			Adapter soft state
 * @seq_num:			The starting sequence number
 * @class_locale:		Class of the event
 *
 * This function subscribes for AEN for events beyond the @seq_num. It requests
 * to be notified if and only if the event is of type @class_locale
 */
static int
megasas_register_aen(struct megasas_instance *instance, u32 seq_num,
		     u32 class_locale_word)
{
	int ret_val;
	struct megasas_cmd *cmd;
	struct megasas_dcmd_frame *dcmd;
	union megasas_evt_class_locale curr_aen;
	union megasas_evt_class_locale prev_aen;

	/*
	 * If there an AEN pending already (aen_cmd), check if the
	 * class_locale of that pending AEN is inclusive of the new
	 * AEN request we currently have. If it is, then we don't have
	 * to do anything. In other words, whichever events the current
	 * AEN request is subscribing to, have already been subscribed
	 * to.
	 *
	 * If the old_cmd is _not_ inclusive, then we have to abort
	 * that command, form a class_locale that is superset of both
	 * old and current and re-issue to the FW
	 */

	curr_aen.word = class_locale_word;

	if (instance->aen_cmd) {

		prev_aen.word = instance->aen_cmd->frame->dcmd.mbox.w[1];

		/*
		 * A class whose enum value is smaller is inclusive of all
		 * higher values. If a PROGRESS (= -1) was previously
		 * registered, then a new registration requests for higher
		 * classes need not be sent to FW. They are automatically
		 * included.
		 *
		 * Locale numbers don't have such hierarchy. They are bitmap
		 * values
		 */
		if ((prev_aen.members.class <= curr_aen.members.class) &&
		    !((prev_aen.members.locale & curr_aen.members.locale) ^
		      curr_aen.members.locale)) {
			/*
			 * Previously issued event registration includes
			 * current request. Nothing to do.
			 */
			return 0;
		} else {
			curr_aen.members.locale |= prev_aen.members.locale;

			if (prev_aen.members.class < curr_aen.members.class)
				curr_aen.members.class = prev_aen.members.class;

			instance->aen_cmd->abort_aen = 1;
			ret_val = megasas_issue_blocked_abort_cmd(instance,
								  instance->
								  aen_cmd);

			if (ret_val) {
				printk(KERN_DEBUG "megasas: Failed to abort "
				       "previous AEN command\n");
				return ret_val;
			}
		}
	}

	cmd = megasas_get_cmd(instance);

	if (!cmd)
		return -ENOMEM;

	dcmd = &cmd->frame->dcmd;

	memset(instance->evt_detail, 0, sizeof(struct megasas_evt_detail));

	/*
	 * Prepare DCMD for aen registration
	 */
	memset(dcmd->mbox.b, 0, MFI_MBOX_SIZE);

	dcmd->cmd = MFI_CMD_DCMD;
	dcmd->cmd_status = 0x0;
	dcmd->sge_count = 1;
	dcmd->flags = MFI_FRAME_DIR_READ;
	dcmd->timeout = 0;
	dcmd->data_xfer_len = sizeof(struct megasas_evt_detail);
	dcmd->opcode = MR_DCMD_CTRL_EVENT_WAIT;
	dcmd->mbox.w[0] = seq_num;
	dcmd->mbox.w[1] = curr_aen.word;
	dcmd->sgl.sge32[0].phys_addr = (u32) instance->evt_detail_h;
	dcmd->sgl.sge32[0].length = sizeof(struct megasas_evt_detail);

	/*
	 * Store reference to the cmd used to register for AEN. When an
	 * application wants us to register for AEN, we have to abort this
	 * cmd and re-register with a new EVENT LOCALE supplied by that app
	 */
	instance->aen_cmd = cmd;

	/*
	 * Issue the aen registration frame
	 */
	instance->instancet->fire_cmd(cmd->frame_phys_addr ,0,instance->reg_set);

	return 0;
}

/**
 * megasas_start_aen -	Subscribes to AEN during driver load time
 * @instance:		Adapter soft state
 */
static int megasas_start_aen(struct megasas_instance *instance)
{
	struct megasas_evt_log_info eli;
	union megasas_evt_class_locale class_locale;

	/*
	 * Get the latest sequence number from FW
	 */
	memset(&eli, 0, sizeof(eli));

	if (megasas_get_seq_num(instance, &eli))
		return -1;

	/*
	 * Register AEN with FW for latest sequence number plus 1
	 */
	class_locale.members.reserved = 0;
	class_locale.members.locale = MR_EVT_LOCALE_ALL;
	class_locale.members.class = MR_EVT_CLASS_DEBUG;

	return megasas_register_aen(instance, eli.newest_seq_num + 1,
				    class_locale.word);
}

/**
 * megasas_io_attach -	Attaches this driver to SCSI mid-layer
 * @instance:		Adapter soft state
 */
static int megasas_io_attach(struct megasas_instance *instance)
{
	struct Scsi_Host *host = instance->host;

	/*
	 * Export parameters required by SCSI mid-layer
	 */
	scsi_set_pci_device(host, instance->pdev);

#ifndef __VMKERNEL_MODULE__
	host->host_lock = &instance->host_lock;
#endif
	host->irq = instance->pdev->irq;
	host->unique_id = instance->unique_id;
	host->can_queue = instance->max_fw_cmds - MEGASAS_INT_CMDS;
	host->this_id = instance->init_id;
	host->sg_tablesize = instance->max_num_sge;
	
	/*
	 * Check if the module parameter value for max_sectors can be used
	 */
	if (max_sector && max_sector < instance->max_sectors_per_req)
		instance->max_sectors_per_req = max_sector;
	else {
		if (max_sector)
			printk(KERN_INFO "megasas: max_sector should be > 0 and"
			 	"<= %d\n",instance->max_sectors_per_req);
	}
	max_sector = instance->max_sectors_per_req;
	host->max_sectors = instance->max_sectors_per_req;

	/*
	 * Check if the module parameter value for cmd_per_lun can be used
	 */
	instance->cmd_per_lun = MEGASAS_DEFAULT_CMD_PER_LUN;
	if (cmd_per_lun && cmd_per_lun <= MEGASAS_DEFAULT_CMD_PER_LUN)
		instance->cmd_per_lun = cmd_per_lun;
	else
		printk(KERN_INFO "megasas: cmd_per_lun should be > 0 and"
			 	"<= %d\n",MEGASAS_DEFAULT_CMD_PER_LUN);
		
	cmd_per_lun = instance->cmd_per_lun;
	host->cmd_per_lun = instance->cmd_per_lun;

	printk(KERN_DEBUG "megasas: max_sector : 0x%x, cmd_per_lun : 0x%x\n",
			instance->max_sectors_per_req, instance->cmd_per_lun);

#ifdef __VMKERNEL_MODULE__
	host->max_channel = 0;
	host->max_id = MEGASAS_MAX_DEV_PER_CHANNEL * MEGASAS_MAX_CHANNELS;
#else
	host->max_channel = MEGASAS_MAX_CHANNELS - 1;
	host->max_id = MEGASAS_MAX_DEV_PER_CHANNEL;
#endif
	host->max_lun = MEGASAS_MAX_LUN;
	host->max_cmd_len = 16;

#ifndef __VMKERNEL_MODULE__
	/*
	 * Notify the mid-layer about the new controller
	 */
	if (scsi_add_host(host, &instance->pdev->dev)) {
		printk(KERN_DEBUG "megasas: scsi_add_host failed\n");
		return -ENODEV;
	}

	/*
	 * Trigger SCSI to scan our drives
	 */
	scsi_scan_host(host);
#endif
	return 0;
}

/**
 * megasas_probe_one -	PCI hotplug entry point
 * @pdev:		PCI device structure
 * @id:			PCI ids of supported hotplugged adapter
 */
static int __devinit
megasas_probe_one(struct pci_dev *pdev, const struct pci_device_id *id)
{
	int rval, i;
	struct Scsi_Host *host;
	struct megasas_instance *instance;

	/*
	 * Announce PCI information
	 */
	printk(KERN_INFO "megasas: %#4.04x:%#4.04x:%#4.04x:%#4.04x: ",
	       pdev->vendor, pdev->device, pdev->subsystem_vendor,
	       pdev->subsystem_device);

	printk("bus %d:slot %d:func %d\n",
	       pdev->bus->number, PCI_SLOT(pdev->devfn), PCI_FUNC(pdev->devfn));

	/*
	 * PCI prepping: enable device set bus mastering and dma mask
	 */
	rval = pci_enable_device(pdev);

	if (rval) {
		return rval;
	}

	pci_set_master(pdev);

	/*
	 * All our contollers are capable of performing 64-bit DMA
	 */
	if (IS_DMA64) {
		if (pci_set_dma_mask(pdev, DMA_64BIT_MASK) != 0) {

			if (pci_set_dma_mask(pdev, DMA_32BIT_MASK) != 0)
				goto fail_set_dma_mask;
		}
	} else {
		if (pci_set_dma_mask(pdev, DMA_32BIT_MASK) != 0)
			goto fail_set_dma_mask;
	}

#ifdef __VMKERNEL_MODULE__
	/* -- FIXME: PR #128009 --
	 *
	 * Too many people hit this bug. Disable PAE temporarily
	 * on this driver until the bug is fixed.
	 */
	if (pci_set_dma_mask(pdev, DMA_32BIT_MASK) != 0)
	   goto fail_set_dma_mask;

	host = vmk_scsi_register(&megasas_template,
				 sizeof(struct megasas_instance),
				 pdev->bus->number, pdev->devfn);
#else

	host = scsi_host_alloc(&megasas_template,
			       sizeof(struct megasas_instance));
#endif

	if (!host) {
		printk(KERN_DEBUG "megasas: scsi_host_alloc failed\n");
		goto fail_alloc_instance;
	}

	instance = (struct megasas_instance *)host->hostdata;
	memset(instance, 0, sizeof(*instance));

#ifdef __VMKERNEL_MODULE__
	/* Don't waste a whole page for just a word of data */
	instance->producer = (u32 *)kmalloc(sizeof(u32), GFP_ATOMIC | GFP_DMA);
	instance->producer_h = virt_to_bus(instance->producer);
	instance->consumer = (u32 *)kmalloc(sizeof(u32), GFP_ATOMIC | GFP_DMA);
	instance->consumer_h = virt_to_bus(instance->consumer);
#else
	instance->producer = pci_alloc_consistent(pdev, sizeof(u32),
						  &instance->producer_h);
	instance->consumer = pci_alloc_consistent(pdev, sizeof(u32),
						  &instance->consumer_h);
#endif

	if (!instance->producer || !instance->consumer) {
		printk(KERN_DEBUG "megasas: Failed to allocate memory for "
		       "producer, consumer\n");
		goto fail_alloc_dma_buf;
	}

	*instance->producer = 0;
	*instance->consumer = 0;

	instance->evt_detail = pci_alloc_consistent(pdev,
						    sizeof(struct
							   megasas_evt_detail),
						    &instance->evt_detail_h);

	if (!instance->evt_detail) {
		printk(KERN_DEBUG "megasas: Failed to allocate memory for "
		       "event detail structure\n");
		goto fail_alloc_dma_buf;
	}

	/*
	 * Initialize locks and queues
	 */
	INIT_LIST_HEAD(&instance->cmd_pool);

	atomic_set(&instance->fw_outstanding,0);

	init_waitqueue_head(&instance->int_cmd_wait_q);
	init_waitqueue_head(&instance->abort_cmd_wait_q);

#ifndef __VMKERNEL_MODULE__
	spin_lock_init(&instance->host_lock);
#endif
	spin_lock_init(&instance->cmd_pool_lock);
	spin_lock_init(&instance->ioctl_memory_pool_lock);
	spin_lock_init(&instance->completion_lock);

	sema_init(&instance->aen_mutex, 1);
	sema_init(&instance->ioctl_sem, MEGASAS_INT_CMDS);

	/**
	for-ioctl: initialize ioctl memory list and memory lock
	**/

	for (i=0; i<MAX_IOCTL_MEM_POOL; i++) {
		INIT_LIST_HEAD(&instance->ioctl_memory_pool[i]);
	}

	if(megasas_setup_mem_pools(instance))
		instance->mem_pool_empty=1;

	/* end for-ioctl */

	/*
	 * Initialize PCI related and misc parameters
	 */
	instance->pdev = pdev;
	instance->host = host;
	instance->unique_id = pdev->bus->number << 8 | pdev->devfn;
	instance->init_id = MEGASAS_DEFAULT_INIT_ID;

	megasas_dbg_lvl = 0;
	instance->is_busy=0;
	instance->last_time=0;

	/*
	 * Initialize MFI Firmware
	 */
	if (megasas_init_mfi(instance))
		goto fail_init_mfi;

	/*
	 * Register IRQ
	 */
	if (request_irq(pdev->irq, megasas_isr, SA_SHIRQ, "megasas", instance)) {
		printk(KERN_DEBUG "megasas: Failed to register IRQ\n");
		goto fail_irq;
	}

	instance->instancet->enable_intr(instance->reg_set);

#ifdef __VMKERNEL_MODULE__
	vmk_scsi_register_uinfo(host, pdev->bus->number, pdev->devfn,
				(void *)instance);
#endif
	/*
	 * Store instance in PCI softstate
	 */
	pci_set_drvdata(pdev, instance);

	/*
	 * Add this controller to megasas_mgmt_info structure so that it
	 * can be exported to management applications
	 */
	megasas_mgmt_info.count++;
	megasas_mgmt_info.instance[megasas_mgmt_info.max_index] = instance;
	megasas_mgmt_info.max_index++;

	/*
	 * Initiate AEN (Asynchronous Event Notification)
	 */
	if (megasas_start_aen(instance)) {
		printk(KERN_DEBUG "megasas: start aen failed\n");
		goto fail_start_aen;
	}

	/*
	 * Register with SCSI mid-layer
	 */
	if (megasas_io_attach(instance))
		goto fail_io_attach;
#if CONFIG_PROC_FS
	if (megasas_proc_hba_map)
		megasas_create_proc_entry(instance);
#endif

	return 0;

      fail_start_aen:
      fail_io_attach:
	megasas_mgmt_info.count--;
	megasas_mgmt_info.instance[megasas_mgmt_info.max_index] = NULL;
	megasas_mgmt_info.max_index--;

	pci_set_drvdata(pdev, NULL);
	instance->instancet->disable_intr(instance->reg_set);
	free_irq(instance->pdev->irq, instance);

	megasas_release_mfi(instance);

      fail_irq:
      fail_init_mfi:
      fail_alloc_dma_buf:
	/* Free the ioctl mem pool */
	megasas_free_ioctl_mem_pools(instance);
	if (instance->evt_detail)
		pci_free_consistent(pdev, sizeof(struct megasas_evt_detail),
				    instance->evt_detail,
				    instance->evt_detail_h);

	if (instance->producer)
#ifdef __VMKERNEL_MODULE__
		kfree(instance->producer);
#else
		pci_free_consistent(pdev, sizeof(u32), instance->producer,
				    instance->producer_h);
#endif
	if (instance->consumer)
#ifdef __VMKERNEL_MODULE__
		kfree(instance->consumer);
#else
		pci_free_consistent(pdev, sizeof(u32), instance->consumer,
				    instance->consumer_h);
#endif
	scsi_host_put(host);

      fail_alloc_instance:
      fail_set_dma_mask:
	pci_disable_device(pdev);

	return -ENODEV;
}

/**
 * megasas_flush_cache -	Requests FW to flush all its caches
 * @instance:			Adapter soft state
 */
static void megasas_flush_cache(struct megasas_instance *instance)
{
	struct megasas_cmd *cmd;
	struct megasas_dcmd_frame *dcmd;

	cmd = megasas_get_cmd(instance);

	if (!cmd)
		return;

	dcmd = &cmd->frame->dcmd;

	memset(dcmd->mbox.b, 0, MFI_MBOX_SIZE);

	dcmd->cmd = MFI_CMD_DCMD;
	dcmd->cmd_status = 0x0;
	dcmd->sge_count = 0;
	dcmd->flags = MFI_FRAME_DIR_NONE;
	dcmd->timeout = 0;
	dcmd->data_xfer_len = 0;
	dcmd->opcode = MR_DCMD_CTRL_CACHE_FLUSH;
	dcmd->mbox.b[0] = MR_FLUSH_CTRL_CACHE | MR_FLUSH_DISK_CACHE;

	megasas_issue_blocked_cmd(instance, cmd);

	megasas_return_cmd(instance, cmd);

	return;
}

/**
 * megasas_reboot_notify-	Flush adapter cache
 * @this:			Our notifier block
 * @code:			The event notified
 * @unused:			Unused
 */
static int
megasas_reboot_notify (struct notifier_block *this, unsigned long code,
		void *unused)
{
	struct megasas_instance *instance;
	int i;

	for (i = 0; i < megasas_mgmt_info.max_index; i++) {
		instance = megasas_mgmt_info.instance[i];
		if (instance) {
			megasas_flush_cache(instance);
		}
	}

	return NOTIFY_DONE;
}

/**
 * notifier block to get notification on system halt/reboot/shutdown/power off
 */
static struct notifier_block megasas_notifier = {
	.notifier_call = megasas_reboot_notify
};

/**
 * megasas_shutdown_controller -	Instructs FW to shutdown the controller
 * @instance:				Adapter soft state
 */
static void megasas_shutdown_controller(struct megasas_instance *instance)
{
	struct megasas_cmd *cmd;
	struct megasas_dcmd_frame *dcmd;

	cmd = megasas_get_cmd(instance);

	if (!cmd)
		return;

	if (instance->aen_cmd)
		megasas_issue_blocked_abort_cmd(instance, instance->aen_cmd);

	dcmd = &cmd->frame->dcmd;

	memset(dcmd->mbox.b, 0, MFI_MBOX_SIZE);

	dcmd->cmd = MFI_CMD_DCMD;
	dcmd->cmd_status = 0x0;
	dcmd->sge_count = 0;
	dcmd->flags = MFI_FRAME_DIR_NONE;
	dcmd->timeout = 0;
	dcmd->data_xfer_len = 0;
	dcmd->opcode = MR_DCMD_CTRL_SHUTDOWN;

	megasas_issue_blocked_cmd(instance, cmd);

	megasas_return_cmd(instance, cmd);

	return;
}

/**
 * megasas_detach_one -	PCI hot"un"plug entry point
 * @pdev:		PCI device structure
 */
static void megasas_detach_one(struct pci_dev *pdev)
{
	int i;
	struct Scsi_Host *host;
	struct megasas_instance *instance;

	instance = pci_get_drvdata(pdev);
	host = instance->host;

	if (poll_mode_io)
		del_timer_sync(&instance->io_completion_timer);

#if CONFIG_PROC_FS
	if (megasas_proc_root && instance->hba_proc_dir)
		megasas_remove_proc_entry(instance);
#endif

	scsi_remove_host(instance->host);
	megasas_flush_cache(instance);
	megasas_shutdown_controller(instance);

	/*
	 * Take the instance off the instance array. Note that we will not
	 * decrement the max_index. We let this array be sparse array
	 */
	for (i = 0; i < megasas_mgmt_info.max_index; i++) {
		if (megasas_mgmt_info.instance[i] == instance) {
			megasas_mgmt_info.count--;
			megasas_mgmt_info.instance[i] = NULL;

			break;
		}
	}

	pci_set_drvdata(instance->pdev, NULL);

	instance->instancet->disable_intr(instance->reg_set);

	free_irq(instance->pdev->irq, instance);

	megasas_release_mfi(instance);

	/* Free IOCTL mem pool */
	megasas_free_ioctl_mem_pools(instance);

	pci_free_consistent(pdev, sizeof(struct megasas_evt_detail),
			    instance->evt_detail, instance->evt_detail_h);

	pci_free_consistent(pdev, sizeof(u32), instance->producer,
			    instance->producer_h);

	pci_free_consistent(pdev, sizeof(u32), instance->consumer,
			    instance->consumer_h);

#ifdef __VMKERNEL_MODULE__
	destroy_waitqueue_head(&instance->int_cmd_wait_q);
	destroy_waitqueue_head(&instance->abort_cmd_wait_q);

	spin_lock_destroy(&instance->cmd_pool_lock);
#endif

	scsi_host_put(host);

	pci_set_drvdata(pdev, NULL);

	pci_disable_device(pdev);

	return;
}

/**
 * megasas_mgmt_open -	char node "open" entry point
 */
static int megasas_mgmt_open(struct inode *inode, struct file *filep)
{
	/*
	 * Allow only those users with admin rights
	 */
	if (!capable(CAP_SYS_ADMIN))
		return -EACCES;

	return 0;
}

/**
 * megasas_mgmt_release - char node "release" entry point
 */
static int megasas_mgmt_release(struct inode *inode, struct file *filep)
{
	filep->private_data = NULL;
	fasync_helper(-1, filep, 0, &megasas_async_queue);

	return 0;
}

/**
 * megasas_mgmt_fasync -	Async notifier registration from applications
 *
 * This function adds the calling process to a driver global queue. When an
 * event occurs, SIGIO will be sent to all processes in this queue.
 */
static int megasas_mgmt_fasync(int fd, struct file *filep, int mode)
{
	int rc;

	down(&megasas_async_queue_mutex);

	rc = fasync_helper(fd, filep, mode, &megasas_async_queue);

	up(&megasas_async_queue_mutex);

	if (rc >= 0) {
		/* For sanity check when we get ioctl */
		filep->private_data = filep;
		return 0;
	}

	printk(KERN_DEBUG "megasas: fasync_helper failed [%d]\n", rc);

	return rc;
}
#ifdef __VMKERNEL_MODULE__
/**
 * megasas_mgmt_poll -	char node "poll" entry point
 */
static unsigned int megasas_mgmt_poll(struct file *file, poll_table *wait) {
   unsigned int             mask = 0;
   unsigned long	    flags;

   poll_wait(file, &megasas_poll_wait, wait);
   if (megasas_wait_aen_poll == 1) {
	spin_lock_irqsave(&aen_poll_lock, flags);
	if (megasas_wait_aen_poll == 1) {
		megasas_wait_aen_poll = 0;
        	mask |= (POLLIN | POLLRDNORM);
	}
	spin_unlock_irqrestore(&aen_poll_lock, flags);
   }

   return mask;
}
#endif
/**
 * megasas_mgmt_fw_ioctl -	Issues management ioctls to FW
 * @instance:			Adapter soft state
 * @ioc:			Kernel virtual address ioctl packet
 */
static int
megasas_mgmt_fw_ioctl(struct megasas_instance *instance,
		      struct megasas_iocpacket *ioc)
{
	struct megasas_sge32 *kern_sge32;
	struct megasas_ioctl_mm *ioctl_mem = NULL;
	struct megasas_cmd *cmd;
	void *kbuff_arr[MAX_IOCTL_SGE];
	dma_addr_t buf_handle = 0;
	int error = 0, i;
	void *sense = NULL;
	dma_addr_t sense_handle;
	u32 *sense_ptr;
	unsigned long *sense_buff;
	u8 from_pool=0;

	memset(kbuff_arr, 0, sizeof(kbuff_arr));

	if (ioc->sge_count > MAX_IOCTL_SGE) {
		printk(KERN_DEBUG "megasas: SGE count [%d] >  max limit [%d]\n",
		       ioc->sge_count, MAX_IOCTL_SGE);
		return -EINVAL;
	}

	cmd = megasas_get_cmd(instance);
	if (!cmd) {
		printk(KERN_DEBUG "megasas: Failed to get a cmd packet\n");
		return -ENOMEM;
	}

	/*
	 * User's IOCTL packet has 2 frames (maximum). Copy those two
	 * frames into our cmd's frames. cmd->frame's context will get
	 * overwritten when we copy from user's frames. So set that value
	 * alone separately
	 */
	memcpy(cmd->frame, ioc->frame.raw, 2 * MEGAMFI_FRAME_SIZE);
	cmd->frame->hdr.context = cmd->index;

	/*
	 * The management interface between applications and the fw uses
	 * MFI frames. E.g, RAID configuration changes, LD property changes
	 * etc are accomplishes through different kinds of MFI frames. The
	 * driver needs to care only about substituting user buffers with
	 * kernel buffers in SGLs. The location of SGL is embedded in the
	 * struct iocpacket itself.
	 */
	kern_sge32 = (struct megasas_sge32 *)
	    ((unsigned long)cmd->frame + ioc->sgl_off);

	/*
	 * Check if we have buffer to use from our mem_pool
	 * If we donot have then try to allocate new buffer
	 */
	if(!megasas_get_buff_for_sge(ioc,instance,cmd))
		from_pool=1;

	/*
	 * For each user buffer, create a mirror buffer and copy in
	 */
	for (i = 0; i < ioc->sge_count; i++) {
		if (from_pool) {
			kbuff_arr[i] = cmd->ioctl_mem[i]->vaddr;
			buf_handle = cmd->ioctl_mem[i]->buf_handle;
		} else {
			kbuff_arr[i] = pci_alloc_consistent(instance->pdev,
						    ioc->sgl[i].iov_len,
						    &buf_handle);
			if (!kbuff_arr[i]) {
				printk(KERN_DEBUG "megasas: Failed to alloc "
				       "kernel SGL buffer for IOCTL \n");
				error = -ENOMEM;
				goto out;
			}
		}

		/*
		 * We don't change the dma_coherent_mask, so
		 * pci_alloc_consistent only returns 32bit addresses
		 */
		kern_sge32[i].phys_addr = (u32) buf_handle;
		kern_sge32[i].length = ioc->sgl[i].iov_len;

		/*
		 * We created a kernel buffer corresponding to the
		 * user buffer. Now copy in from the user buffer
		 */
		if (copy_from_user(kbuff_arr[i], ioc->sgl[i].iov_base,
				   (u32) (ioc->sgl[i].iov_len))) {
			error = -EFAULT;
			goto out;
		}
	}

	if (ioc->sense_len) {
		sense = pci_alloc_consistent(instance->pdev, ioc->sense_len,
					     &sense_handle);
		if (!sense) {
			error = -ENOMEM;
			goto out;
		}

		sense_ptr =
		    (u32 *) ((unsigned long)cmd->frame + ioc->sense_off);
		*sense_ptr = sense_handle;
	}

	/*
	 * Set the sync_cmd flag so that the ISR knows not to complete this
	 * cmd to the SCSI mid-layer
	 */
	cmd->sync_cmd = 1;
	megasas_issue_blocked_cmd(instance, cmd);
	cmd->sync_cmd = 0;

	/*
	 * copy out the kernel buffers to user buffers
	 */
	for (i = 0; i < ioc->sge_count; i++) {
		if (copy_to_user(ioc->sgl[i].iov_base, kbuff_arr[i],
				 ioc->sgl[i].iov_len)) {
			error = -EFAULT;
			goto out;
		}
	}

	/*
	 * copy out the sense
	 */
	if (ioc->sense_len) {
		/*
		 * sense_ptr points to the location that has the user
		 * sense buffer address
		 */
                sense_buff = (unsigned long *) ((unsigned long)ioc->frame.raw +
					ioc->sense_off);
		sense_ptr = (u32 *) ((unsigned long)ioc->frame.raw +
					ioc->sense_off);
#if defined(__ia64__)
		if (copy_to_user((void __user *)((unsigned long)(*sense_buff)),
                                 sense, ioc->sense_len)) {
#else
		if (copy_to_user((void __user *)((unsigned long)(*sense_ptr)),
                                 sense, ioc->sense_len)) {
#endif
                        printk(KERN_ERR "megasas: Failed to copy out to user"
                               "sense data\n");
                        error = -EFAULT;
                        goto out;
                }
	}

	/*
	 * copy the status codes returned by the fw
	 */
	ioc->frame.hdr.cmd_status = cmd->frame->hdr.cmd_status;

      out:
	if (sense) {
		pci_free_consistent(instance->pdev, ioc->sense_len,
				    sense, sense_handle);
	}

	for (i = 0; i < ioc->sge_count && kbuff_arr[i]; i++) {

		if (from_pool) {
			/* Return to the mem pool */
			if ((struct megasas_ioctl_mm *)cmd->ioctl_mem[i]) {
				megasas_return_ioctl_mem(instance,
					(struct megasas_ioctl_mm *)cmd->ioctl_mem[i],
					cmd->ioctl_mem_pool_index[i]);
				cmd->ioctl_mem_pool_index[i]=0xff;
				cmd->ioctl_mem[i]=NULL;
			}
		} else {
			pci_free_consistent(instance->pdev,
					kern_sge32[i].length,
					kbuff_arr[i], kern_sge32[i].phys_addr);
		}
	}

	megasas_return_cmd(instance, cmd);
	return error;
}

static struct megasas_instance *megasas_lookup_instance(u16 host_no)
{
	int i;

	for (i = 0; i < megasas_mgmt_info.max_index; i++) {

		if ((megasas_mgmt_info.instance[i]) &&
		    (megasas_mgmt_info.instance[i]->host->host_no == host_no))
			return megasas_mgmt_info.instance[i];
	}

	return NULL;
}

static int megasas_mgmt_ioctl_fw(struct file *file, unsigned long arg)
{
	struct megasas_iocpacket __user *user_ioc =
	    (struct megasas_iocpacket __user *)arg;
	struct megasas_iocpacket *ioc;
	struct megasas_instance *instance;
	int error;

	ioc = kmalloc(sizeof(*ioc), GFP_KERNEL);
	if (!ioc)
		return -ENOMEM;

	if (copy_from_user(ioc, user_ioc, sizeof(*ioc))) {
		error = -EFAULT;
		goto out_kfree_ioc;
	}

	instance = megasas_lookup_instance(ioc->host_no);
	if (!instance) {
		error = -ENODEV;
		goto out_kfree_ioc;
	}

	/*
	 * We will allow only MEGASAS_INT_CMDS number of parallel ioctl cmds
	 */
	if (down_interruptible(&instance->ioctl_sem)) {
		error = -ERESTARTSYS;
		goto out_kfree_ioc;
	}

	error = megasas_mgmt_fw_ioctl(instance, ioc);

	if (copy_to_user(&user_ioc->frame.hdr.cmd_status,
			 &ioc->frame.hdr.cmd_status, sizeof(u8))) {

		printk(KERN_DEBUG "megasas: Error copying out cmd_status\n");
		error = -EFAULT;
	}

	up(&instance->ioctl_sem);

      out_kfree_ioc:
	kfree(ioc);
	return error;
}

static int megasas_mgmt_ioctl_aen(struct file *file, unsigned long arg)
{
	struct megasas_instance *instance;
	struct megasas_aen aen;
	int error;

	if (copy_from_user(&aen, (void __user *)arg, sizeof(aen)))
		return -EFAULT;

	instance = megasas_lookup_instance(aen.host_no);

	if (!instance)
		return -ENODEV;

	down(&instance->aen_mutex);
	error = megasas_register_aen(instance, aen.seq_num,
				     aen.class_locale_word);
	up(&instance->aen_mutex);
	return error;
}

/**
 * megasas_mgmt_ioctl -	char node ioctl entry point
 */
#ifdef __VMWARE__NO_BUG_FIX
static long
#else
static int
#endif
megasas_mgmt_ioctl(struct inode *inode, struct file *file, unsigned int cmd,
		   unsigned long arg)
{
	switch (cmd) {
	case MEGASAS_IOC_FIRMWARE:
		return megasas_mgmt_ioctl_fw(file, arg);
	case MEGASAS_IOC_GET_AEN:
		return megasas_mgmt_ioctl_aen(file, arg);
	}

	return -ENOTTY;
}

#ifdef __x86_64__

static inline void
megasas_convert_compat_iocpacket(struct megasas_iocpacket *ioc,
			struct compat_megasas_iocpacket* kcioc)
{
	int i = 0;
	compat_uptr_t ptr;

	ioc->host_no = kcioc->host_no;
	ioc->sgl_off = kcioc->sgl_off;
	ioc->sense_off = kcioc->sense_off;
	ioc->sense_len = kcioc->sense_len;
	ioc->sge_count = kcioc->sge_count;

	memcpy(ioc->frame.raw, kcioc->frame.raw, 128);

	for (i = 0; i < MAX_IOCTL_SGE; i++) {
		compat_uptr_t ptr;
		ptr = kcioc->sgl[i].iov_base;
		ioc->sgl[i].iov_len = kcioc->sgl[i].iov_len;
		ioc->sgl[i].iov_base = ptr;
	}
}


static int
megasas_mgmt_compat_ioctl_fw(struct file *file, unsigned long arg)
{
	int i = 0;
	int error = 0;

	struct megasas_instance *instance = NULL;
	struct megasas_iocpacket *ioc = NULL;
	struct compat_megasas_iocpacket *kcioc = NULL;
	struct compat_megasas_iocpacket *ucioc =
	    (struct compat_megasas_iocpacket *)arg;


	ioc = kmalloc(sizeof(struct megasas_iocpacket), GFP_KERNEL);
	kcioc = kmalloc(sizeof(struct compat_megasas_iocpacket), GFP_KERNEL);

	if (!ioc || !kcioc) {
		printk(KERN_DEBUG "megasas: memalloc error\n");
		return ENOMEM;
	}

	if (copy_from_user(kcioc, ucioc, sizeof(*kcioc))) {
		printk(KERN_DEBUG "megasas: cp from usr failed in compat\n");
		error = -EFAULT;
		goto out;
	}

	/*
	 * The application has sent a struct compat_megasas_iocpacket
	 * (it is a 32 bit application). Convert it to regular 64 bit
	 * struct megasas_iocpacket
	 */
	megasas_convert_compat_iocpacket(ioc, kcioc);

	instance = megasas_lookup_instance(ioc->host_no);
	if (!instance) {
		ioc = kmalloc(sizeof(*ioc), GFP_KERNEL);
		error = -ENODEV;
		goto out;
	}

	/*
	 * Now send the ioc through regular ioctl path. Note that we allow
	 * only MEGASAS_INT_CMDS number of parallel ioctl cmds
	 */
	if (down_interruptible(&instance->ioctl_sem)) {
		error = -ERESTARTSYS;
		goto out;
	}
	error = megasas_mgmt_fw_ioctl(instance, ioc);

	up(&instance->ioctl_sem);
	/*
	 * copy the status codes returned by the fw
	 */
	if (copy_to_user(&ucioc->frame.hdr.cmd_status,
 			 &ioc->frame.hdr.cmd_status, sizeof(u8))) {
		printk(KERN_DEBUG "megasas: Error copying out cmd_status\n");
		error = -EFAULT;
 	}
out:
	kfree(ioc);
	kfree(kcioc);
 	return error;
 }
static int
megasas_mgmt_compat_ioctl(unsigned int fd, unsigned int cmd,
			  unsigned long arg, struct file *file)
{
	switch (cmd) {
	case MEGASAS_IOC_FIRMWARE32:
		return megasas_mgmt_compat_ioctl_fw(file, arg);
	case MEGASAS_IOC_GET_AEN:
		return megasas_mgmt_ioctl_aen(file, arg);
	}

	return -ENOTTY;
}
#endif

/*
 * File operations structure for management interface
 */
static struct file_operations megasas_mgmt_fops = {
	.owner = THIS_MODULE,
	.open = megasas_mgmt_open,
	.release = megasas_mgmt_release,
	.fasync = megasas_mgmt_fasync,
	.ioctl = megasas_mgmt_ioctl,
#ifdef __VMKERNEL_MODULE__
	.poll = megasas_mgmt_poll
#endif
};

/*
 * PCI hotplug support registration structure
 */
static struct pci_driver megasas_pci_driver = {

	.name = "megaraid_sas",
	.id_table = megasas_pci_table,
	.probe = megasas_probe_one,
	.remove = __devexit_p(megasas_detach_one),
};

/**
 * megasas_init - Driver load entry point
 */
static int __init megasas_init(void)
{
	int rval;

#ifdef __VMKERNEL_MODULE__
	if (!vmk_set_module_version(MEGASAS_VERSION)) {
		return 0;
	}

	spin_lock_init(&io_request_lock);
	spin_lock_init(&aen_poll_lock);
	megasas_template.driverLock = &io_request_lock;
#endif

	/*
	 * Announce driver version and other information
	 */
	printk(KERN_INFO "megasas: %s %s\n", MEGASAS_VERSION,
	       MEGASAS_EXT_VERSION);

	memset(&megasas_mgmt_info, 0, sizeof(megasas_mgmt_info));

	/*
	 * Register character device node
	 */
	rval = register_chrdev(0, "megaraid_sas_ioctl", &megasas_mgmt_fops);

	if (rval < 0) {
		printk(KERN_DEBUG "megasas: failed to open device node\n");
		return rval;
	}

	megasas_mgmt_majorno = rval;
#ifdef CONFIG_PROC_FS
	megasas_proc_root = proc_mkdir("megaraid_sas", proc_scsi);

	if (megasas_proc_root) {
		megasas_proc_version = create_proc_read_entry("version",
							      S_IRUSR | S_IFREG,
							      megasas_proc_root,
							      megasas_proc_show_version,
							      NULL);

		megasas_proc_release_date =
		    create_proc_read_entry("release_date", S_IRUSR | S_IFREG,
					   megasas_proc_root,
					   megasas_proc_show_release_date,
					   NULL);
		megasas_proc_dbg_lvl = create_proc_entry("dbg_lvl",
					S_IRUSR|S_IFREG|S_IWUSR,
					megasas_proc_root);
		if (megasas_proc_dbg_lvl)
			megasas_proc_dbg_lvl->proc_fops = 
						&proc_dbg_lvl_ops;
		megasas_proc_poll_mode_io = create_proc_entry("poll_mode_io",
					S_IRUSR | S_IWUSR | S_IFREG,
					megasas_proc_root);
		if (megasas_proc_poll_mode_io)
			megasas_proc_poll_mode_io->proc_fops =
						&proc_poll_mode_ops;
		megasas_proc_hba_map = proc_mkdir("hba_map", megasas_proc_root);
	}
#endif

	/*
	 * Register ourselves as PCI hotplug module
	 */
	rval = pci_module_init(&megasas_pci_driver);

	if (rval) {
		printk(KERN_DEBUG "megasas: PCI hotplug regisration failed \n");
#if 1 || defined(__VMWARE__)
#ifdef CONFIG_PROC_FS
		if (megasas_proc_root) {
			remove_proc_entry("version", megasas_proc_root);
			remove_proc_entry("release_date", megasas_proc_root);
			remove_proc_entry("dbg_lvl", megasas_proc_root);
			remove_proc_entry("poll_mode_io", megasas_proc_root);
			remove_proc_entry("hba_map", megasas_proc_root);
			remove_proc_entry("megaraid_sas", proc_scsi);
		}
#endif
#endif
		unregister_chrdev(megasas_mgmt_majorno, "megaraid_sas_ioctl");
#ifdef __VMKERNEL_MODULE__
		spin_lock_destroy(&io_request_lock);
		spin_lock_destroy(&aen_poll_lock);
#endif
		return rval;
	}
#ifdef __VMKERNEL_MODULE__
	else {
		scsi_register_module(MODULE_SCSI_HA, &megasas_template);
	}
#endif

#ifdef __x86_64__
	register_ioctl32_conversion(MEGASAS_IOC_FIRMWARE32,
				    megasas_mgmt_compat_ioctl);
	register_ioctl32_conversion(MEGASAS_IOC_GET_AEN,
				    megasas_mgmt_compat_ioctl);
#endif
	if(register_reboot_notifier(&megasas_notifier)) {
		printk("megasas: reboot notify routine registration failed!!\n");
	}

	return rval;
}

/**
 * megasas_exit - Driver unload entry point
 */
static void __exit megasas_exit(void)
{

#ifdef __x86_64__
	unregister_ioctl32_conversion(MEGASAS_IOC_FIRMWARE32);
	unregister_ioctl32_conversion(MEGASAS_IOC_GET_AEN);
#endif
	pci_unregister_driver(&megasas_pci_driver);

#ifdef CONFIG_PROC_FS
	if (megasas_proc_root) {
		remove_proc_entry("version", megasas_proc_root);
		remove_proc_entry("release_date", megasas_proc_root);
                remove_proc_entry("poll_mode_io", megasas_proc_root);
		remove_proc_entry("hba_map", megasas_proc_root);
		remove_proc_entry("megaraid_sas", proc_scsi);
	}
#endif

	unregister_chrdev(megasas_mgmt_majorno, "megaraid_sas_ioctl");
	unregister_reboot_notifier(&megasas_notifier);
#ifdef __VMKERNEL_MODULE__
	scsi_unregister_module(MODULE_SCSI_HA, &megasas_template);
	spin_lock_destroy(&io_request_lock);
	spin_lock_destroy(&aen_poll_lock);
#endif
}

module_init(megasas_init);
module_exit(megasas_exit);
