/********************************************************************************
* QLogic ISP2xxx Firmware Dump Reader
* Copyright (C) 2006 QLogic Corporation
* (www.qlogic.com)
*
* This program is free software; you can redistribute it and/or modify it
* under the terms of the GNU General Public License as published by the
* Free Software Foundation; either version 2, or (at your option) any
* later version.
*
* This program is distributed in the hope that it will be useful, but
* WITHOUT ANY WARRANTY; without even the implied warranty of
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
* General Public License for more details.
*
******************************************************************************/

/* QLogic Firmware Dump Reader
 * Usage: ./qla_gdm <host_no>. 
 * The host_no can be obtained from /proc/scsi/qla2300/
 * Original author: AV
*/

#include <stdio.h>
#include <stdlib.h>
#include <stddef.h>
#include <stdint.h>
#include <stdarg.h>
#include <errno.h>
#include <unistd.h>
#include <fcntl.h>
#include <string.h>
#include <arpa/inet.h>
#include <sys/time.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <time.h>
#include <ctype.h>

#include "qla_gdmp.h"
#include "../exioctln.h"

#define IS_QLA2100(d)  (ntohl((d)->device) == 0x2100)
#define IS_QLA2200(d)  (ntohl((d)->device) == 0x2200)
#define IS_QLA2300(d)  (ntohl((d)->device) == 0x2300)
#define IS_QLA2312(d)  (ntohl((d)->device) == 0x2312)
#define IS_QLA2322(d)  (ntohl((d)->device) == 0x2322)
#define IS_QLA6312(d)  (ntohl((d)->device) == 0x6312)
#define IS_QLA6322(d)  (ntohl((d)->device) == 0x6322)
#define IS_QLA2422(d)  (ntohl((d)->device) == 0x2422)
#define IS_QLA2432(d)  (ntohl((d)->device) == 0x2432)
#define IS_QLA5422(d)  (ntohl((d)->device) == 0x5422)
#define IS_QLA5432(d)  (ntohl((d)->device) == 0x5432)
#define IS_QLA23XX(d)  (IS_QLA2300(d) || IS_QLA2312(d) || IS_QLA2322(d) || \
                             IS_QLA6312(d) || IS_QLA6322(d))
#define IS_QLA24XX(d)  (IS_QLA2422(d) || IS_QLA2432(d))
#define IS_QLA54XX(d)  (IS_QLA5422(d) || IS_QLA5432(d))

extern UINT32 qlapi_get_host_no_by_handle(UINT32, UINT16 *);

#define PATH_MAX 4096
unsigned char *buffer = NULL;

static void
report(const char *prefix, const char *err, va_list params)
{
	fputs(prefix, stderr);
	vfprintf(stderr, err, params);
	fputs("\n", stderr);
}

void
die(const char *err, ...)
{
	va_list params;

	va_start(params, err);
	report("Fatal: ", err, params);
	va_end(params);
	exit(128);
}

int
error(const char *err, ...)
{
	va_list params;

	va_start(params, err);
	report("Error: ", err, params);
	va_end(params);
	return -1;
}

int
info(const char *err, ...)
{
	va_list params;

	va_start(params, err);
	report("Info: ", err, params);
	va_end(params);
	return 0;
}

#define DEFAULT_ALLOC 0x100000

static inline void *
xmalloc(size_t size)
{
	void *ret = malloc(size);
	if (!ret && !size)
		ret = malloc(1);
	if (!ret)
		die("Out of memory, malloc failed");
	return ret;
}

static inline void *
xrealloc(void *ptr, size_t size)
{
	void *ret = realloc(ptr, size);
	if (!ret && !size)
		ret = realloc(ptr, 1);
	if (!ret)
		die("Out of memory, realloc failed");
	return ret;
}

static inline ssize_t
verify_dump(struct qla2xxx_fw_dump *fw_dump)
{
	unsigned long isize;
	uint32_t val;

	if (fw_dump->signature[0] != 'Q' &&
	    fw_dump->signature[1] != 'L' &&
	    fw_dump->signature[2] != 'G' &&
	    fw_dump->signature[3] != 'C') {
		error("invalid signature %02x%02x%02x%02x",
		    fw_dump->signature[0], fw_dump->signature[1],
		    fw_dump->signature[2], fw_dump->signature[3]);
		return 1;
	}
	info("input signature %02x%02x%02x%02x", fw_dump->signature[0],
	    fw_dump->signature[1], fw_dump->signature[2],
	    fw_dump->signature[3]);

	val = ntohl(fw_dump->header_size);
	if (val != offsetof(struct qla2xxx_fw_dump, isp)) {
		error("invalid header size %d", val);
		return 1;
	}
	info("input header size %d", val);

	val = ntohl(fw_dump->version);
	if (val != 1) {
		error("unsupported version %d", val);
		return 1;
	}
	info("input version %d", val);

	info("input fw version %d.%02d.%02d [%04x]",
	    ntohl(fw_dump->fw_major_version),
	    ntohl(fw_dump->fw_minor_version),
	    ntohl(fw_dump->fw_subminor_version),
	    ntohl(fw_dump->fw_attributes));

	info("input dev info %04x %04x %04x %04x", ntohl(fw_dump->vendor),
	    ntohl(fw_dump->device), ntohl(fw_dump->subsystem_vendor),
	    ntohl(fw_dump->subsystem_device));

	info("input layout - fixed size %d", ntohl(fw_dump->fixed_size));
	info("input layout - mem size   %d", ntohl(fw_dump->mem_size));
	val = ntohl(fw_dump->req_q_size);
	info("input layout - req q size %d (%d)", val, val / 64);
	val = ntohl(fw_dump->rsp_q_size);
	info("input layout - rsp q size %d (%d)", val, val / 64);

	val = ntohl(fw_dump->eft_size);
	if (val && (val % EFT_BYTES_PER_BUFFER)) {
		error("invalid eft size %d", val);
		return 1;
	}
	info("input layout - eft size   %d", val);
	info("input layout - eft addr   %08x%08x", ntohl(fw_dump->eft_addr_h),
	    ntohl(fw_dump->eft_addr_l));

	isize = ntohl(fw_dump->header_size);
	isize += ntohl(fw_dump->fixed_size) + ntohl(fw_dump->mem_size) +
	    ntohl(fw_dump->req_q_size) + ntohl(fw_dump->rsp_q_size) +
	    ntohl(fw_dump->eft_size);

	info("input size %d", isize);

	return 0;
}

static inline void *
dump_queues(struct qla2xxx_fw_dump *fw_dump, void *ptr)
{
	uint8_t *dmp = ptr;
	uint32_t i, j;
	struct queue {
		char *name;
		uint32_t cnt;
	} *e,  l[] = {
		{ "Request Queue", ntohl(fw_dump->req_q_size) },
		{ "\nResponse Queue", ntohl(fw_dump->rsp_q_size) },
		{ NULL, },
	};

	for (e = l; e->name; e++) {
		printf("\n%s", e->name);
		for (i = 0; i < e->cnt / 64; i++) {
			printf("\n%04x:", i);
			for (j = 0; j < 64; j++) {
				if (j && (j % 32 == 0))
					printf("\n      ");
				else if (j % 4 == 0)
					printf(" ");
				printf("%02x", *dmp++);
			}
		}
	}
	printf("\n");

	return dmp;
}

dump_fw_header(struct qla2xxx_fw_dump *fw_dump)
{
	printf("ISP FW Version %d.%02d.%02d Attributes %04x\n",
	    ntohl(fw_dump->fw_major_version), ntohl(fw_dump->fw_minor_version),
	    ntohl(fw_dump->fw_subminor_version),
	    ntohl(fw_dump->fw_attributes));
}

struct entry {
	char *name;
	void *ptr;
	uint32_t cnt;
};

void dump_reg16(struct entry *l)
{
	uint32_t i;
	uint16_t *p;
	struct entry *e;

	for (e = l, p = e->ptr; e->name; e++) {
		printf("%s", e->name);
		for (i = 0; i < e->cnt; i++) {
			if (i % 8 == 0)
				printf("\n");
			else
				printf(" ");
			printf("%04x", ntohs(*p++));
		}
	}
}

void dump_reg16w(struct entry *l)
{
	uint32_t i;
	uint16_t *p;
	struct entry *e;

	for (e = l, p = e->ptr; e->name; e++) {
		printf("%s", e->name);
		for (i = 0; i < e->cnt; i++) {
			if (i % 16 == 0)
				printf("\n");
			else
				printf(" ");
			printf("%04x", ntohs(*p++));
		}
	}
}

void dump_reg32(struct entry *l)
{
	uint32_t i;
	uint32_t *p;
	struct entry *e;

	for (e = l, p = e->ptr; e->name; e++) {
		printf("%s", e->name);
		for (i = 0; i < e->cnt; i++) {
			if (i % 8 == 0)
				printf("\n");
			else
				printf(" ");
			printf("%08x", ntohl(*p++));
		}
	}
}

struct ram {
	char *name;
	void *ptr;
	uint32_t cnt;
	uint32_t offset;
};

void dump_ram16(struct ram *l)
{
	uint32_t i;
	uint16_t *p;
	struct ram *e;
	int wide;

	for (e = l, p = e->ptr; e->name; e++) {
		wide = e->offset & 0xF0000;
		printf("%s", e->name);
		for (i = 0; i < e->cnt; i++) {
			if (i % 8 == 0)
				printf(wide ? "\n%05x: ": "\n%04x: ", i + e->offset);
			else
				printf(" ");
			printf("%04x", ntohs(*p++));
		}
	}
}

void dump_ram32(struct ram *l)
{
	uint32_t i;
	uint32_t *p;
	struct ram *e;

	for (e = l, p = e->ptr; e->name; e++) {
		printf("%s", e->name);
		for (i = 0; i < e->cnt; i++) {
			if (i % 8 == 0)
				printf("\n%08x: ", i + e->offset);
			else
				printf(" ");
			printf("%08x", ntohl(*p++));
		}
	}
}

void
do_dump_2100(struct qla2xxx_fw_dump *fw_dump)
{
	struct qla2100_fw_dump *fw = &fw_dump->isp.isp21;
	struct entry l[] = {
		{ "PBIU Registers:", fw->pbiu_reg, sizeof (fw->pbiu_reg) / 2 },
		{ "\n\nMailbox Registers:", fw->mailbox_reg, sizeof (fw->mailbox_reg) / 2 },
		{ "\n\nDMA Registers:", fw->dma_reg, sizeof (fw->dma_reg) / 2 },
		{ "\n\nRISC Hardware Registers:", fw->risc_hdw_reg, sizeof (fw->risc_hdw_reg) / 2 },
		{ "\n\nRISC GP0 Registers:", fw->risc_gp0_reg, sizeof (fw->risc_gp0_reg) / 2 },
		{ "\n\nRISC GP1 Registers:", fw->risc_gp1_reg, sizeof (fw->risc_gp1_reg) / 2 },
		{ "\n\nRISC GP2 Registers:", fw->risc_gp2_reg, sizeof (fw->risc_gp2_reg) / 2 },
		{ "\n\nRISC GP3 Registers:", fw->risc_gp3_reg, sizeof (fw->risc_gp3_reg) / 2 },
		{ "\n\nRISC GP4 Registers:", fw->risc_gp4_reg, sizeof (fw->risc_gp4_reg) / 2 },
		{ "\n\nRISC GP5 Registers:", fw->risc_gp5_reg, sizeof (fw->risc_gp5_reg) / 2 },
		{ "\n\nRISC GP6 Registers:", fw->risc_gp6_reg, sizeof (fw->risc_gp6_reg) / 2 },
		{ "\n\nRISC GP7 Registers:", fw->risc_gp7_reg, sizeof (fw->risc_gp7_reg) / 2 },
		{ "\n\nFrame Buffer Hardware Registers:", fw->frame_buf_hdw_reg, sizeof (fw->frame_buf_hdw_reg) / 2 },
		{ "\n\nFPM B0 Registers:", fw->fpm_b0_reg, sizeof (fw->fpm_b0_reg) / 2 },
		{ "\n\nFPM B1 Registers:", fw->fpm_b1_reg, sizeof (fw->fpm_b1_reg) / 2 },
		{ NULL, },
	};
	struct ram r[] = {
		{ "\n\nRISC SRAM:", fw->risc_ram, sizeof (fw->risc_ram) / 2, 0x1000 },
		{ NULL, },
	};

	dump_fw_header(fw_dump);

	printf("\n[==>BEG]\n");

	printf("HCCR Register:\n%04x\n\n", ntohs(fw->hccr));

	dump_reg16(l);
	dump_ram16(r);

	printf("\n\n[<==END] ISP Debug Dump\n");

	dump_queues(fw_dump, &fw->risc_ram[r[0].cnt]);
}


void
do_dump_2300(struct qla2xxx_fw_dump *fw_dump)
{
	struct qla2300_fw_dump *fw = &fw_dump->isp.isp23;
	struct entry l[] = {
		{ "PBIU Registers:", fw->pbiu_reg, sizeof (fw->pbiu_reg) / 2 },
		{ "\n\nReqQ-RspQ-Risc2Host Status registers:", fw->risc_host_reg, sizeof (fw->risc_host_reg) / 2 },
		{ "\n\nMailbox Registers:", fw->mailbox_reg, sizeof (fw->mailbox_reg) / 2 },
		{ "\n\nAuto Request Response DMA Registers:", fw->resp_dma_reg, sizeof (fw->resp_dma_reg) / 2 },
		{ "\n\nDMA Registers:", fw->dma_reg, sizeof (fw->dma_reg) / 2 },
		{ "\n\nRISC Hardware Registers:", fw->risc_hdw_reg, sizeof (fw->risc_hdw_reg) / 2 },
		{ "\n\nRISC GP0 Registers:", fw->risc_gp0_reg, sizeof (fw->risc_gp0_reg) / 2 },
		{ "\n\nRISC GP1 Registers:", fw->risc_gp1_reg, sizeof (fw->risc_gp1_reg) / 2 },
		{ "\n\nRISC GP2 Registers:", fw->risc_gp2_reg, sizeof (fw->risc_gp2_reg) / 2 },
		{ "\n\nRISC GP3 Registers:", fw->risc_gp3_reg, sizeof (fw->risc_gp3_reg) / 2 },
		{ "\n\nRISC GP4 Registers:", fw->risc_gp4_reg, sizeof (fw->risc_gp4_reg) / 2 },
		{ "\n\nRISC GP5 Registers:", fw->risc_gp5_reg, sizeof (fw->risc_gp5_reg) / 2 },
		{ "\n\nRISC GP6 Registers:", fw->risc_gp6_reg, sizeof (fw->risc_gp6_reg) / 2 },
		{ "\n\nRISC GP7 Registers:", fw->risc_gp7_reg, sizeof (fw->risc_gp7_reg) / 2 },
		{ "\n\nFrame Buffer Hardware Registers:", fw->frame_buf_hdw_reg, sizeof (fw->frame_buf_hdw_reg) / 2 },
		{ "\n\nFPM B0 Registers:", fw->fpm_b0_reg, sizeof (fw->fpm_b0_reg) / 2 },
		{ "\n\nFPM B1 Registers:", fw->fpm_b1_reg, sizeof (fw->fpm_b1_reg) / 2 },
		{ NULL, },
	};
	struct ram r[] = {
		{ "\n\nCode RAM Dump:", fw->risc_ram, sizeof (fw->risc_ram) / 2, 0x0800 },
		{ "\n\nStack RAM Dump:", fw->stack_ram, sizeof (fw->stack_ram) / 2, 0x10000 },
		{ "\n\nData RAM Dump:", fw->data_ram, ntohl(fw_dump->mem_size) / 2, 0x11000 },
		{ NULL, },
	};

	dump_fw_header(fw_dump);

	printf("\n[==>BEG]\n");

	printf("HCCR Register:\n%04x\n\n", ntohs(fw->hccr));

	dump_reg16(l);
	dump_ram16(r);

	printf("\n\n[<==END] ISP Debug Dump\n");

	dump_queues(fw_dump, &fw->data_ram[r[2].cnt]);
}

void
do_dump_24xx(struct qla2xxx_fw_dump *fw_dump)
{
	uint32_t cnt;
	struct qla24xx_fw_dump *fw = &fw_dump->isp.isp24;
	uint8_t *eft;
	uint32_t eft_h;
	uint64_t eft_start;
	struct entry l[] = {
		{ "\nHost Interface Registers", fw->host_reg, sizeof(fw->host_reg) / 4 },
		{ "\n\nShadow Registers", fw->shadow_reg, sizeof(fw->shadow_reg) / 4 },
		{ NULL, },
	};
	struct entry l2[] = {
		{ "\n\nMailbox Registers", fw->mailbox_reg, sizeof (fw->mailbox_reg) / 2 },
		{ NULL, },
	};
	struct entry l3[] = {
		{"\n\nXSEQ GP Registers", fw->xseq_gp_reg, sizeof(fw->xseq_gp_reg) / 4 },
		{"\n\nXSEQ-0 Registers", fw->xseq_0_reg, sizeof(fw->xseq_0_reg) / 4 },
		{"\n\nXSEQ-1 Registers", fw->xseq_1_reg, sizeof(fw->xseq_1_reg) / 4 },
		{"\n\nRSEQ GP Registers", fw->rseq_gp_reg, sizeof(fw->rseq_gp_reg) / 4 },
		{"\n\nRSEQ-0 Registers", fw->rseq_0_reg, sizeof(fw->rseq_0_reg) / 4 },
		{"\n\nRSEQ-1 Registers", fw->rseq_1_reg, sizeof(fw->rseq_1_reg) / 4 },
		{"\n\nRSEQ-2 Registers", fw->rseq_2_reg, sizeof(fw->rseq_2_reg) / 4 },
		{"\n\nCommand DMA Registers", fw->cmd_dma_reg, sizeof(fw->cmd_dma_reg) / 4 },
		{"\n\nRequest0 Queue DMA Channel Registers", fw->req0_dma_reg, sizeof(fw->req0_dma_reg) / 4 },
		{"\n\nResponse0 Queue DMA Channel Registers", fw->resp0_dma_reg, sizeof(fw->resp0_dma_reg) / 4 },
		{"\n\nRequest1 Queue DMA Channel Registers", fw->req1_dma_reg, sizeof(fw->req1_dma_reg) / 4 },
		{"\n\nXMT0 Data DMA Registers", fw->xmt0_dma_reg, sizeof(fw->xmt0_dma_reg) / 4 },
		{"\n\nXMT1 Data DMA Registers", fw->xmt1_dma_reg, sizeof(fw->xmt1_dma_reg) / 4 },
		{"\n\nXMT2 Data DMA Registers", fw->xmt2_dma_reg, sizeof(fw->xmt2_dma_reg) / 4 },
		{"\n\nXMT3 Data DMA Registers", fw->xmt3_dma_reg, sizeof(fw->xmt3_dma_reg) / 4 },
		{"\n\nXMT4 Data DMA Registers", fw->xmt4_dma_reg, sizeof(fw->xmt4_dma_reg) / 4 },
		{"\n\nXMT Data DMA Common Registers", fw->xmt_data_dma_reg, sizeof(fw->xmt_data_dma_reg) / 4 },
		{"\n\nRCV Thread 0 Data DMA Registers", fw->rcvt0_data_dma_reg, sizeof(fw->rcvt0_data_dma_reg) / 4 },
		{"\n\nRCV Thread 1 Data DMA Registers", fw->rcvt1_data_dma_reg, sizeof(fw->rcvt1_data_dma_reg) / 4 },
		{"\n\nRISC GP Registers", fw->risc_gp_reg, sizeof(fw->risc_gp_reg) / 4 },
		{"\n\nLMC Registers", fw->lmc_reg, sizeof(fw->lmc_reg) / 4 },
		{"\n\nFPM Hardware Registers", fw->fpm_hdw_reg, sizeof(fw->fpm_hdw_reg) / 4 },
		{"\n\nFB Hardware Registers", fw->fb_hdw_reg, sizeof(fw->fb_hdw_reg) / 4 },
		{ NULL, },
	};
	struct ram r[] = {
		{ "\n\nCode RAM", fw->code_ram, sizeof (fw->code_ram) / 4, 0x20000 },
		{ "\n\nExternal Memory", fw->ext_mem, ntohl(fw_dump->mem_size) / 4, 0x100000 },
		{ NULL, },
	};

	dump_fw_header(fw_dump);

	printf("\nR2H Status Register\n%04x\n", ntohl(fw->host_status));

	dump_reg32(l);
	dump_reg16w(l2);
	dump_reg32(l3);
	dump_ram32(r);

	printf("\n\n[<==END] ISP Debug Dump\n");

	eft = dump_queues(fw_dump, &fw->ext_mem[r[1].cnt]);

	if (!ntohl(fw_dump->eft_size))
		return;

	printf("\nExtended Trace Buffer");
	eft_h = !!ntohl(fw_dump->eft_addr_h);
	eft_start = (((uint64_t)ntohl(fw_dump->eft_addr_h) << 16 << 16) |
	    (uint64_t)ntohl(fw_dump->eft_addr_l));
	for (cnt = 0; cnt < ntohl(fw_dump->eft_size); cnt++) {
		if (cnt % 32 == 0)
			printf(eft_h ? "\n%016llx: ": "\n%08llx: ",
			    (unsigned long long)(cnt / 4 + eft_start));
		else if (cnt % 4 == 0)
			printf(" ");
		printf("%02x", *eft++);
	}
	printf("\n");
}

int
do_dump(struct qla2xxx_fw_dump *fw_dump)
{
	if (IS_QLA2100(fw_dump) || IS_QLA2200(fw_dump))
		do_dump_2100(fw_dump);
	else if (IS_QLA23XX(fw_dump))
		do_dump_2300(fw_dump);
	else if (IS_QLA24XX(fw_dump) || IS_QLA54XX(fw_dump))
		do_dump_24xx(fw_dump);
	else
		return 1;

	return 0;
}

int
get_dump(SDMGT_HANDLE handle, int host_no)
{
	int ret;
	SD_UINT32 result, osize, size;
	SD_UINT8 *buf;

	ret = 1;
	osize = size = DEFAULT_ALLOC;
	buf = xmalloc(size);
retry:
	result = SDGetFwDump(handle, 0, buf, &size);
	switch (result) {
	case SDMGT_NONE:
		buffer = xmalloc(size);
		memcpy(buffer, buf, size);
		break;
	case SDMGT_HBA_NOT_READY:
		info("No firmware dump available at host %d", host_no);
		break;
	case SDMGT_BUFFER_TOO_SMALL:
		info("Firmware dump on %d -- expanding buffer %d --> %d", host_no,
		    osize, size);
		osize = size;
		size += DEFAULT_ALLOC;
		buf = realloc(buf, size);
		goto retry;
	default:
		error("Unable to retrieve dump on %d -- (%s)", host_no,
		    SDGetErrorString(result));
		ret = 1;
		break;
	}
	free(buf);
	return ret;
}

void
help(char* argv)
{
	//[RC]: Add both info and get FW dump help
	printf("QLogic Firmware Dump Reader\n");
	printf("Usage:\n");
	printf("To get fw_dump file\n");
	printf("%s <host_no>\n",argv);
	printf("To get information only\n");
	printf("%s [-i] <host_no>\n", argv);
	exit (128);
}

int
main(int argc, const char **argv, char **envp)
{
	int host_no, done, error_flag;
 	SD_UINT16 hno;
	SD_UINT32 result, inst, size;
	struct qla2xxx_fw_dump *fw_dump;
	unsigned long long_size;
	int info_only = 0;
	char time[30];
	char path[PATH_MAX];
	struct timeval tv;
	time_t curtime;
	FILE *fd;

	SDMGT_HANDLE handle;

	if ((argc < 2) || (argc > 3)) {
		help((char *)argv[0]);
	}

	done = 0;
	inst = 0;
	error_flag=0;

//	Parse command line arguments
	if(isdigit(argv[1][0])) {
		host_no = strtol(argv[1], NULL, 10);
		if(argc == 3) {
			if(!(strcmp(argv[2], "-i")))
				info_only=1;
			else 
				help((char *)argv[0]);
		}
	}
	else if(isalpha(argv[1][1])) {
		if (argc < 3) {
			help((char *)argv[0]);
		}
		if(!(strcmp(argv[1], "-i")))
			info_only=1;
		else {
			help((char *)argv[0]);
		}
		if(isdigit(argv[2][0]))
			host_no = strtol(argv[1], NULL, 10);
		else {
			help((char *)argv[0]);
		}
	}
	else {
		help((char *)argv[0]);
	}
		
	do {
		result = SDOpenDevice(inst, &handle);
		switch (result) {
		case SDMGT_NONE:
			if (qlapi_get_host_no_by_handle((SD_UINT32)handle,
			    &hno))
				error("No host_no for instance %d", inst);
			if (hno != host_no)
				break;

			get_dump(handle, host_no);
			done = 1;
			break;
		case SDMGT_UNSUPPORTED_VERSION:
			error("Unsupported instance %d", inst);
			break;
		case SDMGT_DEV_NOT_FOUND:
		default:
			error("host_no %d not found", host_no);
			done = 1;
			error_flag= 1;
			break;
		}

//		 Close the device 
		if (result == SDMGT_NONE) {
			SDCloseDevice(handle);
		}
		inst++;
	} while (!done);

	if (error_flag) {
		return ;
	}		

	gettimeofday(&tv, NULL);
	curtime=tv.tv_sec;
	strftime(time, 30, "%Y%m%d_%H%M%S", localtime(&curtime));
	snprintf(path, PATH_MAX, "fw_dump_%d_%s.txt", host_no, time);
	
	if (buffer == NULL) {
		printf("fw_dump is not avalable\n");
		return 0;
	}

	fw_dump = (struct qla2xxx_fw_dump *) buffer;
	if (verify_dump(fw_dump))
		die("unable to validate input data");
	if (info_only)
		return 0;

//	Redirect stdout to fw_dump* file	
	fd = freopen(path, "w", stdout);
	if ( fd == NULL) {
		error("Can't create file %s for writing %d",path, errno);
		return ;
	}
	info("writing firmware dump to file: %s", path);
	if (do_dump(fw_dump))
		die("unable to dump data");

	fclose(fd);
	return 0;
}
