/*
 * QLogic Fibre Channel HBA Driver
 * Copyright (c)  2003-2005 QLogic Corporation
 *
 * See LICENSE.qla2xxx for copyright and licensing details.
 */


/*
 * QLogic ISP2x00 Multi-path LUN Support Driver 
 * Linux specific functions
 *
 */
#include <linux/vmalloc.h>
#include "qlfo.h"
#include "qla_cfg.h"
#include "qla_gbl.h"

#define MAX_SEARCH_STR_SIZE	512

/*
 * qla2x00_set_lun_data_from_config
 * Set lun_data byte from the configuration parameters.
 * The lun data are preferred lun and disabled lun. The lun data will be store
 * in chunks of 256 lun (one bit = one lun).
 * Info in the conf file will be like this
 * scsi-qla0-tgt-0-di-0-preferred=00000000000000000000000000000000000000000000\
 *                                                        00000000000000000000
 * If there are more than 256 LUN, entry for next 256 CHUNK will be available
 * (256 to 511)
 * scsi-qla0-tgt-0-di-0-lun_preferred-256=000000000000000000000000000000000000\
 *                                                0000000000000000000000000000
 * (512 to 767)
 * scsi-qla0-tgt-0-di-0-lun_preferred-512=000000000000000000000000000000000000\
 *                                                0000000000000000000000000000
 * (768 to 1023)
 * scsi-qla0-tgt-0-di-0-lun_preferred-768=000000000000000000000000000000000000\
 *                                                0000000000000000000000000000
 * And so on upto MAX_LUNS
 * Similary iformation for disabled lun will be maintained. The disabled line
 * will be available only if the lun is disabled.
 * scsi-qla0-tgt-0-di-0-lun-disabled=ff000000000000000ff00000000000000ff000000\
 *                                                     00000000000000000000000
 * If there are more than 256 LUN, and LUN in the next 256 chunk is disabled
 * (256 to 511)
 * scsi-qla0-tgt-0-di-0-lun_disabled-256=0000000000000fffffffffff000000000000\
 *
 * Input:
 * host -- pointer to host adapter structure.
 * port -- pointer to port
 * tgt  -- target number
 * dev_no  -- device number
 */
void
qla2x00_set_lun_data_from_config(mp_host_t *host, fc_port_t *port,
    uint16_t tgt, uint16_t dev_no)
{
	char		*propbuf;  /* As big as largest search string */
	int		rval, idx;
	int		lun, l;
	scsi_qla_host_t *ha = host->ha;
	mp_device_t	*dp;
	lun_bit_mask_t	*plun_mask;
	lun_bit_mask_t  *mask_ptr;
	mp_path_list_t	*pathlist;
	int		lun_mask_len;

	mp_path_t *path;

	if (qla2x00_get_ioctl_scrap_mem(ha, (void **)&propbuf,
	    MAX_SEARCH_STR_SIZE)) {
		/* not enough memory */
		DEBUG9_10(printk("%s(%ld): inst=%ld scrap not big enough. "
		    "propbuf requested=%d.\n",
		    __func__, ha->host_no, ha->instance,
		    MAX_SEARCH_STR_SIZE));
		return;
	}

	/* Allocate 32 bytes (256 bits) at a time */
	lun_mask_len = (LUN_CHUNK_SIZE >> 3);
	if (qla2x00_get_ioctl_scrap_mem(ha, (void **)&plun_mask,
	    lun_mask_len)) {
		/* not enough memory */
		DEBUG9_10(printk("%s(%ld): inst=%ld scrap not big enough. "
		    "lun_mask requested=%ld.\n",
		    __func__, ha->host_no, ha->instance,
		    (ulong)lun_mask_len));
		qla2x00_free_ioctl_scrap_mem(ha);
		return;
	}
	mask_ptr = plun_mask;

	dp = host->mp_devs[tgt];
	if (dp == NULL) {
		printk("qla2x00_set_lun_data_from_config: Target %d "
		    "not found for hba %d\n",tgt, host->instance);
		qla2x00_free_ioctl_scrap_mem(ha);
		return;
	}
	if ((pathlist = dp->path_list) == NULL) {
		printk("qla2x00_set_lun_data_from_config: path list "
		    "not found for target %d\n", tgt);
		qla2x00_free_ioctl_scrap_mem(ha);
		return;
	}

	if ((path = qla2x00_find_path_by_name(host, pathlist,
	    port->port_name)) == NULL) {
		printk("qla2x00_set_lun_data_from_config: No path found "
		    "for target %d\n", tgt);
		qla2x00_free_ioctl_scrap_mem(ha);
		return;
	}
	for (idx = 0; idx < TOTAL_256_LUN_CHUNKS; idx++) {

		/* Get "target-N-device-N-preferred" as a 256 bit lun_mask*/
		if (idx == 0) {
			l = 0;
			/* Default case 0 to 255 LUN */
			PERSIST_STRING("scsi-qla%ld-tgt-%d-di-%d-preferred",
			    "%ld-%d-%d-f");
		} else {
			l = (idx * LUN_CHUNK_SIZE);
			PERSIST_STRING_LUN(
			    "scsi-qla%ld-tgt-%d-di-%d-lun_preferred-%d",
			    "%ld-%d-%d-%d-lf");
		}
		lun = LUN_CHUNK_SIZE - 1;
		DEBUG3(printk("build_tree: %s\n",propbuf));

		rval = qla2x00_get_prop_xstr(ha, propbuf,
		    (uint8_t *)(plun_mask), lun_mask_len);

		if (rval == -1) {
			/* EMPTY */
			DEBUG2(printk("%s(%ld): no preferred mask entry found"
			    " for path id %d , LUN > %d on port "
			    "%02x%02x%02x%02x%02x%02x%02x%02x.\n",
			    __func__, ha->host_no, path->id, l,
			    path->portname[0], path->portname[1],
			    path->portname[2], path->portname[3],
			    path->portname[4], path->portname[5],
			    path->portname[6], path->portname[7]));
		} else {
			if (rval != lun_mask_len) {
				/* EMPTY */
				printk("qla2x00_set_lun_data_from_config: "
				    "Preferred mask len %d is incorrect.\n",
				    rval);
			}

			DEBUG3(printk("%s(%ld): reading Preferred Mask for path id %d "
			    "on port %02x%02x%02x%02x%02x%02x%02x%02x:\n",
			    __func__, ha->host_no, path->id,
			    path->portname[0], path->portname[1],
			    path->portname[2], path->portname[3],
			    path->portname[4], path->portname[5],
			    path->portname[6], path->portname[7]));
			DEBUG3(qla2x00_dump_buffer((char *)plun_mask,
			    lun_mask_len));

			for (; lun >= 0; lun--, l++) {
				if (EXT_IS_LUN_BIT_SET(mask_ptr, lun)) {
					path->lun_data.data[l] |=
					    LUN_DATA_PREFERRED_PATH;
					pathlist->current_path[l] = path->id;
				} else {
					path->lun_data.data[l] &=
					    ~LUN_DATA_PREFERRED_PATH;
				}
			}
		}
	} /* for */

	for (idx = 0; idx < TOTAL_256_LUN_CHUNKS; idx++) {

		/* Get "target-N-device-N-lun-disabled" as a 256 bit lun_mask*/
		if (idx == 0) {
			l = 0;
			/* Default case 0 to 255 LUN */
			PERSIST_STRING(
			    "scsi-qla%ld-tgt-%d-di-%d-lun-disabled",
			    "%ld-%d-%d-d");
			lun = LUN_CHUNK_SIZE - 1;
		} else {
			l = (idx * LUN_CHUNK_SIZE);
			PERSIST_STRING_LUN(
			    "scsi-qla%ld-tgt-%d-di-%d-lun_disabled-%d",
			    "%ld-%d-%d-%d-ld");
			lun = l + (LUN_CHUNK_SIZE - 1);
		}
		DEBUG3(printk("build_tree: %s\n",propbuf));

		rval = qla2x00_get_prop_xstr(ha, propbuf,
		    (uint8_t *)plun_mask, lun_mask_len);
		if (rval == -1) {
			/* default: all luns enabled */
			printk("%s(%ld): no entry found for path id %d. "
			    "Assume all LUNs enabled on port "
			    "%02x%02x%02x%02x%02x%02x%02x%02x.\n",
			    __func__, ha->host_no, path->id,
			    path->portname[0], path->portname[1],
			    path->portname[2], path->portname[3],
			    path->portname[4], path->portname[5],
			    path->portname[6], path->portname[7]);

			for (; l <= lun; l++) {
				path->lun_data.data[l] |= LUN_DATA_ENABLED;
			}
		} else {
			if (rval != lun_mask_len) {
				printk("qla2x00_set_lun_data_from_config: "
				    "Enable mask has wrong size %d != %d\n",
				    rval, lun_mask_len);
			} else {
				lun = LUN_CHUNK_SIZE - 1;
				for (; lun >= 0; lun--, l++) {
					/* our bit mask is inverted */
					if (!EXT_IS_LUN_BIT_SET(
					    mask_ptr, lun)) {
						path->lun_data.data[l] |=
						    LUN_DATA_ENABLED;
					} else {
						path->lun_data.data[l] &=
						    ~LUN_DATA_ENABLED;
					}
				}
				DEBUG3(printk("%s(%ld): got lun mask for path "
				    "id %d port "
				    "%02x%02x%02x%02x%02x%02x%02x%02x:\n",
				    __func__, ha->host_no, path->id,
				    path->portname[0], path->portname[1],
				    path->portname[2], path->portname[3],
				    path->portname[4], path->portname[5],
				    path->portname[6], path->portname[7]));
				DEBUG3(qla2x00_dump_buffer(
				    (uint8_t *)&path->lun_data.data[0], 64));
			}
		}
	} /* For */

	DEBUG3(printk("qla2x00_set_lun_data_from_config: Luns data for "
	    "device %p, instance %d, path id=%d\n",
	    dp,host->instance,path->id));
	DEBUG3(qla2x00_dump_buffer((char *)&path->lun_data.data[0], 64));

	qla2x00_free_ioctl_scrap_mem(ha);
	LEAVE("qla2x00_set_lun_data_from_config");
}



/*
 * qla2x00_cfg_build_path_tree
 *	Find all path properties and build a path tree. The
 *  resulting tree has no actual port assigned to it
 *  until the port discovery is done by the lower level.
 *
 * Input:
 *	ha = adapter block pointer.
 *
 * Context:
 *	Kernel context.
 */
void
qla2x00_cfg_build_path_tree(scsi_qla_host_t *ha)
{
	char		*propbuf;
	uint8_t		node_name[WWN_SIZE];
	uint8_t		port_name[WWN_SIZE];
	fc_port_t	*port;
	uint16_t	dev_no = 0, tgt;
	int		instance, rval;
	mp_host_t	*host = NULL;
	int		done;
	uint8_t         control_byte;


	ENTER("qla2x00_cfg_build_path_tree");

	printk(KERN_INFO
	    "qla02%d: ConfigRequired is set. \n", (int)ha->instance);
	DEBUG(printk("qla2x00_cfg_build_path_tree: hba =%d",
	    (int)ha->instance));

	if (qla2x00_get_ioctl_scrap_mem(ha, (void **)&propbuf,
	    MAX_SEARCH_STR_SIZE)) {
		/* not enough memory */
		DEBUG9_10(printk("%s(%ld): inst=%ld scrap not big enough. "
		    "propbuf requested=%d.\n",
		    __func__, ha->host_no, ha->instance,
		    MAX_SEARCH_STR_SIZE));
		return;
	}

	/* Look for adapter nodename in properties */
	if (!ql2xdevflag) 
		sprintf(propbuf, "scsi-qla%ld-adapter-port", ha->instance);
	else
		sprintf(propbuf, "%ld-w", ha->instance);
	DEBUG(printk("build_tree: %s\n",propbuf));

	rval = qla2x00_get_prop_xstr(ha, propbuf, port_name, WWN_SIZE);
	if (rval != WWN_SIZE) {
		qla2x00_free_ioctl_scrap_mem(ha);
		return;
	}

	/* Does nodename match the host adapter nodename? */
	if (!qla2x00_is_nodename_equal(ha->port_name, port_name)) {
		printk(KERN_INFO
		    "scsi(%d): Adapter nodenames don't match - ha = %p.\n",
		    (int)ha->instance,ha);
		DEBUG(printk("qla(%d): Adapter nodenames don't match - "
		    "ha=%p. port name=%02x%02x%02x%02x%02x%02x%02x%02x\n",
		    (int)ha->instance,ha,
		    ha->port_name[0], ha->port_name[1], ha->port_name[2], 
		    ha->port_name[3], ha->port_name[4], ha->port_name[5],
		    ha->port_name[6], ha->port_name[7]));

		qla2x00_free_ioctl_scrap_mem(ha);
		return;
	}

	DEBUG(printk("%s: found entry for adapter port %02x%02x%02x%02x"
	    "%02x%02x%02x%02x.\n",
	    __func__,
	    port_name[0], port_name[1], port_name[2],
	    port_name[3], port_name[4], port_name[5],
	    port_name[6], port_name[7]));

	instance = ha->instance;
	if ((host = qla2x00_alloc_host(ha)) == NULL) {
		printk(KERN_INFO
		    "scsi(%d): Couldn't allocate host - ha = %p.\n",
		    (int)instance,ha);
	} else {
		/* create a dummy port */
		port = (fc_port_t *)KMEM_ZALLOC(sizeof (fc_port_t),9);
		if (port == NULL) {
			printk(KERN_INFO
			    "scsi(%d): Couldn't allocate port.\n",
			    (int)instance);
			DEBUG(printk("qla(%d): Couldn't allocate port.\n",
			    (int)host->instance));
			/* remove host */
			qla2x00_free_ioctl_scrap_mem(ha);
			return;
		}

		done = 0;

		/* For each target on the host bus adapter */
		for (tgt = 0; tgt < MAX_MP_DEVICES &&
		    !done; tgt++) {

			/* get all paths for this target */
			for (dev_no = 0; dev_no < MAX_PATHS_PER_DEVICE &&
			    !done ; dev_no++) {

				/*
				 * O(N*M) scan, should ideally check if there
				 * are any tgt entries present, if not, then
				 * continue.
				 *
				 *   sprintf(propbuf,
				 * 		"scsi-qla%d-tgt-%d-",
				 *		instance, tgt);
				 *   if (strstr(ha->cmdline, propbuf) == NULL)
				 *	continue;
				 *
				 */
				memset(port, 0, sizeof (fc_port_t));

				/*
				 * Get "target-N-device-N-node" is a 16-chars
				 * number
				 */
				PERSIST_STRING("scsi-qla%ld-tgt-%d-di-%d-node", "%ld-%d-%d-n");

				rval = qla2x00_get_prop_xstr(ha, propbuf,
				    node_name, WWN_SIZE);
				if (rval != WWN_SIZE)
					/* di values may not be contiguous for
					 * override case.
					 */
					continue;

				DEBUG(printk("build_tree: %s\n",propbuf));
				memcpy(port->node_name, node_name, WWN_SIZE);

				/*
				 * Get "target-N-device-N-port" is a 16-chars
				 * number
				 */
				PERSIST_STRING("scsi-qla%ld-tgt-%d-di-%d-port", "%ld-%d-%d-p");

				rval = qla2x00_get_prop_xstr(ha, propbuf,
				    port_name, WWN_SIZE);
				if (rval != WWN_SIZE)
					continue;

				DEBUG(printk("build_tree: %s\n",propbuf));
				memcpy(port->node_name, node_name, WWN_SIZE);
				memcpy(port->port_name, port_name, WWN_SIZE);
				port->flags |= FC_CONFIG;

				/*
				 * Get "target-N-device-N-control" if property 
				 * is present then all luns are visible.
				 */
				PERSIST_STRING("scsi-qla%ld-tgt-%d-di-%d-control", "%ld-%d-%d-c");
				rval = qla2x00_get_prop_xstr(ha, propbuf,
				    (uint8_t *)(&control_byte),
				    sizeof(control_byte));
				if (rval == -1) {
					/* error getting string. go to next. */
					DEBUG2(printk(
					    "%s: string parsing failed.\n",
					    __func__));
					continue;
				}

				DEBUG3(printk("build_tree: %s\n",propbuf));

				DEBUG(printk("build_tree: control byte 0x%x\n",
				    control_byte));

				port->mp_byte = control_byte;
				DEBUG(printk("%s(%ld): calling update_mp_device"
				    " for host %p port %p-%02x%02x%02x%02x%02x"
				    "%02x%02x%02x tgt=%d mpbyte=%02x.\n",
				    __func__, ha->host_no, host, port,
				    port->port_name[0], port->port_name[1],
				    port->port_name[2], port->port_name[3],
				    port->port_name[4], port->port_name[5],
				    port->port_name[6], port->port_name[7],
				    tgt, port->mp_byte));
#if  0	/* not supported */
				/* To do the lun binding: Create a fclun for each
				 * lun the user has specified, so a mp_lun can be
				 * created for each.
				 */
				for (lun = 0; lun < MAX_LUNS_PER_DEVICE && ; lun++) {
					if (!ql2xdevflag) 
	 					sprintf(propbuf, "scsi-qla%ld-tgt-%d-lun-%d-lunid", 
	 						ha->instance, tgt, lun);
					else 
						sprintf(propbuf, "%ld-%d-%d-l", ha->instance, 
							tgt, lun);

					DEBUG(printk("build_tree: %s\n",propbuf));

					/* allocate space for mp_lun and fclun */

					rval = qla2x00_get_prop_xstr(ha, propbuf,
				    	mplun->wwuln, 32);
					if (rval != 32)
						continue;

				}
#endif

				qla2x00_update_mp_device(host, port, tgt,
				    dev_no);

				/* free any mplun info */

				qla2x00_set_lun_data_from_config(host,
				    port, tgt, dev_no);
			}
		}
		KMEM_FREE(port, sizeof (fc_port_t));
	}

	qla2x00_free_ioctl_scrap_mem(ha);

	LEAVE("qla2x00_cfg_build_path_tree");
	DEBUG(printk("Leaving: qla2x00_cfg_build_path_tree\n"));
}

/* Supporting function for qla2x00_cfg_display_devices */
inline void
qla2x00_cfg_display_lun_data(mp_path_list_t *path_list,
    mp_path_t *path, int instance, int id)
{
	int mask_set = 0;
	int l;
	int i;
	int cnt;
	int lun_mask_size;
	lun_bit_mask_t *plun_mask;

	lun_mask_size = sizeof(lun_bit_mask_t);

	plun_mask = (lun_bit_mask_t *)vmalloc(lun_mask_size);
	if (plun_mask == NULL) {
		/* not enough memory */
		DEBUG9_10(printk("%s: Out of memory"
		    " lun_mask requested=%d.\n",
		    __func__, lun_mask_size));
		return;
	}

	/*
	 * Build preferred bit mask for this
	 * path */
	memset(plun_mask, 0, lun_mask_size);
	mask_set = 0;
	cnt = 0;
	for (i = 0; i < MAX_LUNS; i++) {
		l = (i & LUN_MASK);
		if (path_list->current_path[l] == path->id) {
			EXT_SET_LUN_BIT(plun_mask, l);
			mask_set++;
		}

		if ((i % LUN_CHUNK_SIZE) == (LUN_CHUNK_SIZE - 1)) {
			if (mask_set) {
				if (i < 256) {
					printk(KERN_INFO
					    "scsi-qla%d-tgt-%d-di-%d-preferred",
					    instance,  id, path->id);
				} else {
					printk(KERN_INFO
					    "scsi-qla%d-tgt-%d-di-%d-"
					    "lun_preferred-%d",
					    instance,  id, path->id,
					    (l - (LUN_CHUNK_SIZE - 1)));
				}
				printk(KERN_INFO
				    "=%08x%08x%08x%08x%08x%08x%08x%08x\\;\n",
                                    *((uint32_t *) &plun_mask->mask[cnt+28]),
                                    *((uint32_t *) &plun_mask->mask[cnt+24]),
                                    *((uint32_t *) &plun_mask->mask[cnt+20]),
                                    *((uint32_t *) &plun_mask->mask[cnt+16]),
                                    *((uint32_t *) &plun_mask->mask[cnt+12]),
                                    *((uint32_t *) &plun_mask->mask[cnt+8]),
                                    *((uint32_t *) &plun_mask->mask[cnt+4]),
                                    *((uint32_t *) &plun_mask->mask[cnt+0]));
			}
			mask_set = 0;
			cnt += 32; /* 256 bits */
		}
	}
	/*
	 * Build disable bit mask for this path
	 */

	memset(plun_mask, 0, lun_mask_size);
	mask_set = 0;
	cnt = 0;
	for (i = 0; i < MAX_LUNS; i++) {
		l = (i & LUN_MASK);

		if (!(path->lun_data.data[l] & LUN_DATA_ENABLED)) {
			EXT_SET_LUN_BIT(plun_mask, l);
			mask_set++;
		}

		if ((i % LUN_CHUNK_SIZE) == (LUN_CHUNK_SIZE - 1)) {
			if (mask_set) {
				if (i < 256) {
					printk(KERN_INFO
					    "scsi-qla%d-tgt-%d-di-%d-lun-"
					    "disabled", instance, id, path->id);
				} else {
					printk(KERN_INFO
					    "scsi-qla%d-tgt-%d-di-%d-"
					    "lun_disabled-%d",
					    instance, id, path->id,
						(l - (LUN_CHUNK_SIZE - 1)));
				}
				printk(KERN_INFO
				    "=%08x%08x%08x%08x%08x%08x%08x%08x\\;\n",
                                    *((uint32_t *) &plun_mask->mask[cnt+28]),
                                    *((uint32_t *) &plun_mask->mask[cnt+24]),
                                    *((uint32_t *) &plun_mask->mask[cnt+20]),
                                    *((uint32_t *) &plun_mask->mask[cnt+16]),
                                    *((uint32_t *) &plun_mask->mask[cnt+12]),
                                    *((uint32_t *) &plun_mask->mask[cnt+8]),
                                    *((uint32_t *) &plun_mask->mask[cnt+4]),
                                    *((uint32_t *) &plun_mask->mask[cnt+0]));
			}
			mask_set = 0;
			cnt += 32; /* 256 bits */
		}
	}

	vfree(plun_mask);
	return;
}

/*
 * qla2x00_cfg_display_devices
 *      This routine will the node names of the different devices found
 *      after port inquiry.
 *
 * Input:
 *
 * Returns:
 *      None.
 */
void
qla2x00_cfg_display_devices(int flag)
{
	mp_host_t     *host;
	int     id;
	mp_device_t	*dp;
	mp_path_t  *path;
	mp_path_list_t	*path_list;
	int cnt, i, dev_no;
	int instance;
	mp_lun_t	*lun;
	unsigned char 	tmp_buf[32];

	for (host = mp_hosts_base; (host); host = host->next) {

		instance = (int) host->instance;
		/* Display the node name for adapter */
		printk(KERN_INFO
			"scsi-qla%d-adapter-port="
			"%02x%02x%02x%02x%02x%02x%02x%02x\\;\n",
			instance,
			host->portname[0],
			host->portname[1],
			host->portname[2],
			host->portname[3],
			host->portname[4],
			host->portname[5],
			host->portname[6],
			host->portname[7]);

		for (id = 0; id < MAX_MP_DEVICES; id++) {
			if ((dp = host->mp_devs[id]) == NULL)
				continue;

			path_list = dp->path_list;

			if ((path = path_list->last) == NULL) {
				continue;
			}
			/* Print out device port names */
			path = path->next; /* first path */
			for (dev_no = 0, cnt = 0; cnt < path_list->path_cnt;
			    path = path->next, cnt++) {

				/* skip others if not our host */
				if (host != path->host)
					continue;
				printk(KERN_INFO
				    "scsi-qla%d-tgt-%d-di-%d-node="
				    "%02x%02x%02x%02x%02x%02x%02x%02x\\;\n",
				    instance, id, path->id,
				    dp->nodename[0],
				    dp->nodename[1],
				    dp->nodename[2],
				    dp->nodename[3],
				    dp->nodename[4],
				    dp->nodename[5],
				    dp->nodename[6],
				    dp->nodename[7]);

				/* port_name */
				printk(KERN_INFO
				    "scsi-qla%d-tgt-%d-di-%d-port="
				    "%02x%02x%02x%02x%02x%02x%02x%02x\\;\n",
				    instance, id, path->id,
				    path->portname[0],
				    path->portname[1],
				    path->portname[2],
				    path->portname[3],
				    path->portname[4],
				    path->portname[5],
				    path->portname[6],
				    path->portname[7]);

				/* control byte */
				printk(KERN_INFO
				    "scsi-qla%d-tgt-%d-di-%d-control=%02x\\;\n",
				    instance, id, path->id, path->mp_byte);

				qla2x00_cfg_display_lun_data(path_list,
				    path, instance, id);

				/* display lun wwuln */
				if (flag)
				for (lun = dp->luns; lun != NULL;
				    lun = lun->next) {
					printk(KERN_INFO
					    "scsi-qla%d-tgt-%d-di-%d-"
					    "lun-%d-lunid=", instance,
					    id, path->id, lun->number);
					for (i = 0; i < lun->siz; i++) {
						sprintf(tmp_buf + i, "%02x",
						    lun->wwuln[i]);
					}
					printk(KERN_INFO "%s:%02d;\n", tmp_buf,
					    lun->siz);
				}
				dev_no++;
			}
		}
	}
}
