/*
 * QLogic Fibre Channel HBA Driver
 * Copyright (c)  2003-2005 QLogic Corporation
 *
 * See LICENSE.qla2xxx for copyright and licensing details.
 */
#include <linux/delay.h>

static inline void
qla2xxx_prep_dump(scsi_qla_host_t *ha, struct qla2xxx_fw_dump *fw_dump)
{
	fw_dump->fw_major_version = htonl(ha->fw_version[0]);
	fw_dump->fw_minor_version = htonl(ha->fw_version[1]);
	fw_dump->fw_subminor_version = htonl(ha->fw_version[2]);
	fw_dump->fw_attributes = htonl(ha->fw_attributes);

	fw_dump->vendor = htonl(ha->pdev->vendor);
	fw_dump->device = htonl(ha->pdev->device);
	fw_dump->subsystem_vendor = htonl(ha->pdev->subsystem_vendor);
	fw_dump->subsystem_device = htonl(ha->pdev->subsystem_device);
}

static inline void *
qla2xxx_copy_queues(scsi_qla_host_t *ha, void *ptr)
{
	/* Request queue. */
	memcpy(ptr, ha->request_ring, ha->req_ring_length *
	    sizeof(request_t));

	/* Response queue. */
	ptr += ha->req_ring_length * sizeof(request_t);
	memcpy(ptr, ha->response_ring, ha->rsp_ring_length  *
	    sizeof(response_t));

	return ptr + (ha->rsp_ring_length * sizeof(response_t));
}

static int
qla2xxx_dump_memory(scsi_qla_host_t *ha, uint32_t *code_ram,
    uint32_t cram_size, uint32_t *ext_mem, void **nxt)
{
	int rval;
	uint32_t cnt, stat, timer, risc_address, ext_mem_cnt;
	uint16_t mb[4];
	struct device_reg_24xx  *reg =
	    (struct device_reg_24xx  *)ha->iobase;

	rval = QLA2X00_SUCCESS;
	risc_address = ext_mem_cnt = 0;
	memset(mb, 0, sizeof(mb));

	/* Code RAM. */
	risc_address = 0x20000;
	WRT_REG_WORD(&reg->mailbox0, MBC_READ_RAM_EXTENDED);
	clear_bit(MBX_INTERRUPT, &ha->mbx_cmd_flags);

	for (cnt = 0; cnt < cram_size / 4 && rval == QLA2X00_SUCCESS;
	    cnt++, risc_address++) {
		WRT_REG_WORD(&reg->mailbox1, LSW(risc_address));
		WRT_REG_WORD(&reg->mailbox8, MSW(risc_address));
		RD_REG_WORD(&reg->mailbox8);
		WRT_REG_DWORD(&reg->hccr, HCCRX_SET_HOST_INT);

		for (timer = 6000000; timer; timer--) {
			/* Check for pending interrupts. */
			stat = RD_REG_DWORD(&reg->host_status);
			if (stat & HSRX_RISC_INT) {
				stat &= 0xff;

				if (stat == 0x1 || stat == 0x2 ||
				    stat == 0x10 || stat == 0x11) {
					set_bit(MBX_INTERRUPT,
					    &ha->mbx_cmd_flags);

					mb[0] = RD_REG_WORD(&reg->mailbox0);
					mb[2] = RD_REG_WORD(&reg->mailbox2);
					mb[3] = RD_REG_WORD(&reg->mailbox3);

					WRT_REG_DWORD(&reg->hccr,
					    HCCRX_CLR_RISC_INT);
					RD_REG_DWORD(&reg->hccr);
					break;
				}

				/* Clear this intr; it wasn't a mailbox intr */
				WRT_REG_DWORD(&reg->hccr, HCCRX_CLR_RISC_INT);
				RD_REG_DWORD(&reg->hccr);
			}
			udelay(5);
		}

		if (test_and_clear_bit(MBX_INTERRUPT, &ha->mbx_cmd_flags)) {
			rval = mb[0] & MBS_MASK;
			code_ram[cnt] = htonl((mb[3] << 16) | mb[2]);
		} else {
			rval = QLA2X00_FUNCTION_FAILED;
		}
	}

	if (rval == QLA2X00_SUCCESS) {
		/* External Memory. */
		risc_address = 0x100000;
		ext_mem_cnt = ha->fw_memory_size - 0x100000 + 1;
		WRT_REG_WORD(&reg->mailbox0, MBC_READ_RAM_EXTENDED);
		clear_bit(MBX_INTERRUPT, &ha->mbx_cmd_flags);
	}
	for (cnt = 0; cnt < ext_mem_cnt && rval == QLA2X00_SUCCESS;
	    cnt++, risc_address++) {
		WRT_REG_WORD(&reg->mailbox1, LSW(risc_address));
		WRT_REG_WORD(&reg->mailbox8, MSW(risc_address));
		RD_REG_WORD(&reg->mailbox8);
		WRT_REG_DWORD(&reg->hccr, HCCRX_SET_HOST_INT);

		for (timer = 6000000; timer; timer--) {
			/* Check for pending interrupts. */
			stat = RD_REG_DWORD(&reg->host_status);
			if (stat & HSRX_RISC_INT) {
				stat &= 0xff;

				if (stat == 0x1 || stat == 0x2 ||
				    stat == 0x10 || stat == 0x11) {
					set_bit(MBX_INTERRUPT,
					    &ha->mbx_cmd_flags);

					mb[0] = RD_REG_WORD(&reg->mailbox0);
					mb[2] = RD_REG_WORD(&reg->mailbox2);
					mb[3] = RD_REG_WORD(&reg->mailbox3);

					WRT_REG_DWORD(&reg->hccr,
					    HCCRX_CLR_RISC_INT);
					RD_REG_DWORD(&reg->hccr);
					break;
				}

				/* Clear this intr; it wasn't a mailbox intr */
				WRT_REG_DWORD(&reg->hccr, HCCRX_CLR_RISC_INT);
				RD_REG_DWORD(&reg->hccr);
			}
			udelay(5);
		}

		if (test_and_clear_bit(MBX_INTERRUPT, &ha->mbx_cmd_flags)) {
			rval = mb[0] & MBS_MASK;
			ext_mem[cnt] = htonl((mb[3] << 16) | mb[2]);
		} else {
			rval = QLA2X00_FUNCTION_FAILED;
		}
	}

	*nxt = rval == QLA2X00_SUCCESS ? &ext_mem[cnt]: NULL;
	return rval;
}

static uint32_t *
qla24xx_read_window(struct device_reg_24xx *reg, uint32_t iobase,
    uint32_t count, uint32_t *buf)
{
	uint32_t *dmp_reg;

	WRT_REG_DWORD(&reg->iobase_addr, iobase);
	dmp_reg = &reg->iobase_window;
	while (count--)
		*buf++ = htonl(RD_REG_DWORD(dmp_reg++));

	return buf;
}
#if defined(ISP2100) || defined(ISP2200)
/**
 * qla2100_fw_dump() - Dumps binary data from the 2100/2200 firmware.
 * @ha: HA context
 * @hardware_locked: Called with the hardware_lock
 */
void
qla2100_fw_dump(scsi_qla_host_t *ha, int hardware_locked)
{
	int		rval;
	uint32_t	cnt, timer;
	uint16_t	risc_address;
	uint16_t	mb0, mb2;
	device_reg_t  *reg = ha->iobase;
	uint16_t  *dmp_reg;
	unsigned long	flags;
	struct qla2100_fw_dump	*fw;
	uint16_t	wd;

	risc_address = 0;
	mb0 = mb2 = 0;
	flags = 0;

	if (!hardware_locked)
		spin_lock_irqsave(&ha->hardware_lock, flags);

	if (!ha->fw_dump) {
		printk(KERN_WARNING
		    "scsi(%ld): No buffer available for dump!!!\n",
		    ha->host_no);
		goto qla2100_fw_dump_failed;
	}

	if (ha->fw_dumped) {
		printk(KERN_WARNING
		    "scsi(%ld): Firmware has been previously dumped (%p) "
		    "-- ignoring request...\n", ha->host_no, ha->fw_dump);
		goto qla2100_fw_dump_failed;
	}
	fw = &ha->fw_dump->isp.isp21;
	qla2xxx_prep_dump(ha, ha->fw_dump);

	rval = QLA2X00_SUCCESS;
	fw->hccr = htons(RD_REG_WORD(&reg->host_cmd));

	/* Pause RISC. */
	WRT_REG_WORD(&reg->host_cmd, HC_PAUSE_RISC);
	for (cnt = 30000; (RD_REG_WORD(&reg->host_cmd) & HC_RISC_PAUSE) == 0 &&
	    rval == QLA2X00_SUCCESS; cnt--) {
		if (cnt)
			udelay(100);
		else
			rval = QL_STATUS_TIMEOUT;
	}
	if (rval == QLA2X00_SUCCESS) {
		dmp_reg = (uint16_t  *)(reg + 0);
		for (cnt = 0; cnt < sizeof(fw->pbiu_reg) / 2; cnt++)
			fw->pbiu_reg[cnt] = htons(RD_REG_WORD(dmp_reg++));

		dmp_reg = (uint16_t  *)((uint8_t  *)reg + 0x10);
		for (cnt = 0; cnt < MAILBOX_REGISTER_COUNT; cnt++) {
			if (cnt == 8) {
				dmp_reg = (uint16_t  *)
					((uint8_t  *)reg + 0xe0);
			}
			fw->mailbox_reg[cnt] = htons(RD_REG_WORD(dmp_reg++));
		}

		dmp_reg = (uint16_t  *)((uint8_t  *)reg + 0x20);
		for (cnt = 0; cnt < sizeof(fw->dma_reg) / 2; cnt++)
			fw->dma_reg[cnt] = htons(RD_REG_WORD(dmp_reg++));

		WRT_REG_WORD(&reg->ctrl_status, 0x00);
		dmp_reg = (uint16_t  *)((uint8_t  *)reg + 0xA0);
		for (cnt = 0; cnt < sizeof(fw->risc_hdw_reg) / 2; cnt++)
			fw->risc_hdw_reg[cnt] = htons(RD_REG_WORD(dmp_reg++));

		WRT_REG_WORD(&reg->pcr, 0x2000);
		dmp_reg = (uint16_t  *)((uint8_t  *)reg + 0x80);
		for (cnt = 0; cnt < sizeof(fw->risc_gp0_reg) / 2; cnt++)
			fw->risc_gp0_reg[cnt] = htons(RD_REG_WORD(dmp_reg++));

		WRT_REG_WORD(&reg->pcr, 0x2100);
		dmp_reg = (uint16_t  *)((uint8_t  *)reg + 0x80);
		for (cnt = 0; cnt < sizeof(fw->risc_gp1_reg) / 2; cnt++)
			fw->risc_gp1_reg[cnt] = htons(RD_REG_WORD(dmp_reg++));

		WRT_REG_WORD(&reg->pcr, 0x2200);
		dmp_reg = (uint16_t  *)((uint8_t  *)reg + 0x80);
		for (cnt = 0; cnt < sizeof(fw->risc_gp2_reg) / 2; cnt++)
			fw->risc_gp2_reg[cnt] = htons(RD_REG_WORD(dmp_reg++));

		WRT_REG_WORD(&reg->pcr, 0x2300);
		dmp_reg = (uint16_t  *)((uint8_t  *)reg + 0x80);
		for (cnt = 0; cnt < sizeof(fw->risc_gp3_reg) / 2; cnt++)
			fw->risc_gp3_reg[cnt] = htons(RD_REG_WORD(dmp_reg++));

		WRT_REG_WORD(&reg->pcr, 0x2400);
		dmp_reg = (uint16_t  *)((uint8_t  *)reg + 0x80);
		for (cnt = 0; cnt < sizeof(fw->risc_gp4_reg) / 2; cnt++)
			fw->risc_gp4_reg[cnt] = htons(RD_REG_WORD(dmp_reg++));

		WRT_REG_WORD(&reg->pcr, 0x2500);
		dmp_reg = (uint16_t  *)((uint8_t  *)reg + 0x80);
		for (cnt = 0; cnt < sizeof(fw->risc_gp5_reg) / 2; cnt++)
			fw->risc_gp5_reg[cnt] = htons(RD_REG_WORD(dmp_reg++));

		WRT_REG_WORD(&reg->pcr, 0x2600);
		dmp_reg = (uint16_t  *)((uint8_t  *)reg + 0x80);
		for (cnt = 0; cnt < sizeof(fw->risc_gp6_reg) / 2; cnt++)
			fw->risc_gp6_reg[cnt] = htons(RD_REG_WORD(dmp_reg++));

		WRT_REG_WORD(&reg->pcr, 0x2700);
		dmp_reg = (uint16_t  *)((uint8_t  *)reg + 0x80);
		for (cnt = 0; cnt < sizeof(fw->risc_gp7_reg) / 2; cnt++)
			fw->risc_gp7_reg[cnt] = htons(RD_REG_WORD(dmp_reg++));

		WRT_REG_WORD(&reg->ctrl_status, 0x10);
		dmp_reg = (uint16_t  *)((uint8_t  *)reg + 0x80);
		for (cnt = 0; cnt < sizeof(fw->frame_buf_hdw_reg) / 2; cnt++)
			fw->frame_buf_hdw_reg[cnt] =
			    htons(RD_REG_WORD(dmp_reg++));

		WRT_REG_WORD(&reg->ctrl_status, 0x20);
		dmp_reg = (uint16_t  *)((uint8_t  *)reg + 0x80);
		for (cnt = 0; cnt < sizeof(fw->fpm_b0_reg) / 2; cnt++)
			fw->fpm_b0_reg[cnt] = htons(RD_REG_WORD(dmp_reg++));

		WRT_REG_WORD(&reg->ctrl_status, 0x30);
		dmp_reg = (uint16_t  *)((uint8_t  *)reg + 0x80);
		for (cnt = 0; cnt < sizeof(fw->fpm_b1_reg) / 2; cnt++)
			fw->fpm_b1_reg[cnt] = htons(RD_REG_WORD(dmp_reg++));

		/* Reset the ISP. */
		WRT_REG_WORD(&reg->ctrl_status, CSR_ISP_SOFT_RESET);
		pci_read_config_word(ha->pdev, PCI_COMMAND, &wd);
	}

	for (cnt = 30000; RD_REG_WORD(&reg->mailbox0) != 0 &&
	    rval == QLA2X00_SUCCESS; cnt--) {
		if (cnt)
			udelay(100);
		else
			rval = QL_STATUS_TIMEOUT;
	}

	/* Pause RISC. */
	if (rval == QLA2X00_SUCCESS &&
	    (RD_REG_WORD(&reg->mctr) & (BIT_1 | BIT_0)) != 0) {
		WRT_REG_WORD(&reg->host_cmd, HC_PAUSE_RISC);
		for (cnt = 30000;
		    (RD_REG_WORD(&reg->host_cmd) & HC_RISC_PAUSE) == 0 &&
		    rval == QLA2X00_SUCCESS; cnt--) {
			if (cnt)
				udelay(100);
			else
				rval = QL_STATUS_TIMEOUT;
		}
		if (rval == QLA2X00_SUCCESS) {
			/* Set memory configuration and timing. */
#if defined(ISP2100)
			WRT_REG_WORD(&reg->mctr, 0xf1);
#elif defined(ISP2200)
			WRT_REG_WORD(&reg->mctr, 0xf2);
#endif
			RD_REG_WORD(&reg->mctr);	/* PCI Posting. */

			/* Release RISC. */
			WRT_REG_WORD(&reg->host_cmd, HC_RELEASE_RISC);
			RD_REG_WORD(&reg->host_cmd);	/* PCI Posting. */
		}
	}

	if (rval == QLA2X00_SUCCESS) {
		/* Get RISC SRAM. */
		risc_address = 0x1000;
		WRT_REG_WORD(&reg->mailbox0, MBC_READ_RAM_WORD);
		clear_bit(MBX_INTERRUPT, &ha->mbx_cmd_flags);
	}
	for (cnt = 0; cnt < sizeof(fw->risc_ram) / 2 && rval == QLA2X00_SUCCESS;
	    cnt++, risc_address++) {
                WRT_REG_WORD(&reg->mailbox1, risc_address);
		RD_REG_WORD(&reg->mailbox1);
		WRT_REG_WORD(&reg->host_cmd, HC_SET_HOST_INT);

		for (timer = 6000000; timer != 0; timer--) {
			/* Check for pending interrupts. */
			if (RD_REG_WORD(&reg->istatus) & RISC_INT) {
				if (RD_REG_WORD(&reg->semaphore) & BIT_0) {
					set_bit(MBX_INTERRUPT,
					    &ha->mbx_cmd_flags);

					mb0 = RD_REG_WORD(&reg->mailbox0);
					mb2 = RD_REG_WORD(&reg->mailbox2);

					WRT_REG_WORD(&reg->semaphore, 0);
					WRT_REG_WORD(&reg->host_cmd,
					    HC_CLR_RISC_INT);
					RD_REG_WORD(&reg->host_cmd);
					break;
				}
				WRT_REG_WORD(&reg->host_cmd, HC_CLR_RISC_INT);
				RD_REG_WORD(&reg->host_cmd);
			}
			udelay(5);
		}

		if (test_and_clear_bit(MBX_INTERRUPT, &ha->mbx_cmd_flags)) {
			rval = mb0 & MBS_MASK;
			fw->risc_ram[cnt] = htons(mb2);
		} else {
			rval = QLA2X00_FUNCTION_FAILED;
		}
	}

	if (rval == QLA2X00_SUCCESS)
		qla2xxx_copy_queues(ha, &fw->risc_ram[cnt]);

	if (rval != QLA2X00_SUCCESS) {
		printk(KERN_WARNING
		    "scsi(%ld): Failed to dump firmware (%x)!!!\n",
		    ha->host_no, rval);
		ha->fw_dumped = 0;
	} else {
		printk(KERN_INFO
		    "scsi(%ld): Firmware dump saved to temp buffer "
		    "(%ld/%p).\n", ha->host_no, ha->host_no, ha->fw_dump);
		ha->fw_dumped = 1;
	}

qla2100_fw_dump_failed:
	if (!hardware_locked)
		spin_unlock_irqrestore(&ha->hardware_lock, flags);
}

#elif defined(ISP2300)
/**
 * qla2300_fw_dump() - Dumps binary data from the 2300 firmware.
 * @ha: HA context
 * @hardware_locked: Called with the hardware_lock
 */
void
qla2300_fw_dump(scsi_qla_host_t *ha, int hardware_locked)
{
	int		rval;
	uint32_t	cnt, timer;
	uint32_t	risc_address;
	uint16_t	mb0, mb2;

	uint16_t	stat;
	device_reg_t  *reg = ha->iobase;
	uint16_t  *dmp_reg;
	unsigned long	flags;
	struct qla2300_fw_dump	*fw;
	uint32_t	data_ram_cnt;
	uint16_t	wd;

	risc_address = data_ram_cnt = 0;
	mb0 = mb2 = 0;
	flags = 0;

	if (!hardware_locked)
		spin_lock_irqsave(&ha->hardware_lock, flags);

	if (!ha->fw_dump) {
		printk(KERN_WARNING
		    "scsi(%ld): No buffer available for dump!!!\n",
		    ha->host_no);
		goto qla2300_fw_dump_failed;
	}

	if (ha->fw_dumped) {
		printk(KERN_WARNING
		    "scsi(%ld): Firmware has been previously dumped (%p) "
		    "-- ignoring request...\n", ha->host_no, ha->fw_dump);
		goto qla2300_fw_dump_failed;
	}
	fw = &ha->fw_dump->isp.isp23;
	qla2xxx_prep_dump(ha, ha->fw_dump);

	rval = QLA2X00_SUCCESS;
	fw->hccr = htons(RD_REG_WORD(&reg->host_cmd));

	/* Pause RISC. */
	WRT_REG_WORD(&reg->host_cmd, HC_PAUSE_RISC);
	if (ha->device_id == QLA2300_DEVICE_ID) {
		for (cnt = 30000;
		    (RD_REG_WORD(&reg->host_cmd) & HC_RISC_PAUSE) == 0 &&
			rval == QLA2X00_SUCCESS; cnt--) {
			if (cnt)
				udelay(100);
			else
				rval = QL_STATUS_TIMEOUT;
		}
	} else {
		RD_REG_WORD(&reg->host_cmd);		/* PCI Posting. */
		udelay(10);
	}

	if (rval == QLA2X00_SUCCESS) {
		dmp_reg = (uint16_t  *)(reg + 0);
		for (cnt = 0; cnt < sizeof(fw->pbiu_reg) / 2; cnt++)
			fw->pbiu_reg[cnt] = htons(RD_REG_WORD(dmp_reg++));

		dmp_reg = (uint16_t  *)((uint8_t  *)reg + 0x10);
		for (cnt = 0; cnt < sizeof(fw->risc_host_reg) / 2; cnt++)
			fw->risc_host_reg[cnt] = htons(RD_REG_WORD(dmp_reg++));

		dmp_reg = (uint16_t  *)((uint8_t  *)reg + 0x40);
		for (cnt = 0; cnt < sizeof(fw->mailbox_reg) / 2; cnt++)
			fw->mailbox_reg[cnt] = htons(RD_REG_WORD(dmp_reg++));

		WRT_REG_WORD(&reg->ctrl_status, 0x40);
		dmp_reg = (uint16_t  *)((uint8_t  *)reg + 0x80);
		for (cnt = 0; cnt < sizeof(fw->resp_dma_reg) / 2; cnt++)
			fw->resp_dma_reg[cnt] = htons(RD_REG_WORD(dmp_reg++));

		WRT_REG_WORD(&reg->ctrl_status, 0x50);
		dmp_reg = (uint16_t  *)((uint8_t  *)reg + 0x80);
		for (cnt = 0; cnt < sizeof(fw->dma_reg) / 2; cnt++)
			fw->dma_reg[cnt] = htons(RD_REG_WORD(dmp_reg++));

		WRT_REG_WORD(&reg->ctrl_status, 0x00);
		dmp_reg = (uint16_t  *)((uint8_t  *)reg + 0xA0);
		for (cnt = 0; cnt < sizeof(fw->risc_hdw_reg) / 2; cnt++)
			fw->risc_hdw_reg[cnt] = htons(RD_REG_WORD(dmp_reg++));

		WRT_REG_WORD(&reg->pcr, 0x2000);
		dmp_reg = (uint16_t  *)((uint8_t  *)reg + 0x80);
		for (cnt = 0; cnt < sizeof(fw->risc_gp0_reg) / 2; cnt++)
			fw->risc_gp0_reg[cnt] = htons(RD_REG_WORD(dmp_reg++));

		WRT_REG_WORD(&reg->pcr, 0x2200);
		dmp_reg = (uint16_t  *)((uint8_t  *)reg + 0x80);
		for (cnt = 0; cnt < sizeof(fw->risc_gp1_reg) / 2; cnt++)
			fw->risc_gp1_reg[cnt] = htons(RD_REG_WORD(dmp_reg++));

		WRT_REG_WORD(&reg->pcr, 0x2400);
		dmp_reg = (uint16_t  *)((uint8_t  *)reg + 0x80);
		for (cnt = 0; cnt < sizeof(fw->risc_gp2_reg) / 2; cnt++)
			fw->risc_gp2_reg[cnt] = htons(RD_REG_WORD(dmp_reg++));

		WRT_REG_WORD(&reg->pcr, 0x2600);
		dmp_reg = (uint16_t  *)((uint8_t  *)reg + 0x80);
		for (cnt = 0; cnt < sizeof(fw->risc_gp3_reg) / 2; cnt++)
			fw->risc_gp3_reg[cnt] = htons(RD_REG_WORD(dmp_reg++));

		WRT_REG_WORD(&reg->pcr, 0x2800);
		dmp_reg = (uint16_t  *)((uint8_t  *)reg + 0x80);
		for (cnt = 0; cnt < sizeof(fw->risc_gp4_reg) / 2; cnt++)
			fw->risc_gp4_reg[cnt] = htons(RD_REG_WORD(dmp_reg++));

		WRT_REG_WORD(&reg->pcr, 0x2A00);
		dmp_reg = (uint16_t  *)((uint8_t  *)reg + 0x80);
		for (cnt = 0; cnt < sizeof(fw->risc_gp5_reg) / 2; cnt++)
			fw->risc_gp5_reg[cnt] = htons(RD_REG_WORD(dmp_reg++));

		WRT_REG_WORD(&reg->pcr, 0x2C00);
		dmp_reg = (uint16_t  *)((uint8_t  *)reg + 0x80);
		for (cnt = 0; cnt < sizeof(fw->risc_gp6_reg) / 2; cnt++)
			fw->risc_gp6_reg[cnt] = htons(RD_REG_WORD(dmp_reg++));

		WRT_REG_WORD(&reg->pcr, 0x2E00);
		dmp_reg = (uint16_t  *)((uint8_t  *)reg + 0x80);
		for (cnt = 0; cnt < sizeof(fw->risc_gp7_reg) / 2; cnt++)
			fw->risc_gp7_reg[cnt] = htons(RD_REG_WORD(dmp_reg++));

		WRT_REG_WORD(&reg->ctrl_status, 0x10);
		dmp_reg = (uint16_t  *)((uint8_t  *)reg + 0x80);
		for (cnt = 0; cnt < sizeof(fw->frame_buf_hdw_reg) / 2; cnt++)
			fw->frame_buf_hdw_reg[cnt] =
			    htons(RD_REG_WORD(dmp_reg++));

		WRT_REG_WORD(&reg->ctrl_status, 0x20);
		dmp_reg = (uint16_t  *)((uint8_t  *)reg + 0x80);
		for (cnt = 0; cnt < sizeof(fw->fpm_b0_reg) / 2; cnt++)
			fw->fpm_b0_reg[cnt] = htons(RD_REG_WORD(dmp_reg++));

		WRT_REG_WORD(&reg->ctrl_status, 0x30);
		dmp_reg = (uint16_t  *)((uint8_t  *)reg + 0x80);
		for (cnt = 0; cnt < sizeof(fw->fpm_b1_reg) / 2; cnt++)
			fw->fpm_b1_reg[cnt] = htons(RD_REG_WORD(dmp_reg++));

		/* Reset RISC. */
		WRT_REG_WORD(&reg->ctrl_status, CSR_ISP_SOFT_RESET);
		pci_read_config_word(ha->pdev, PCI_COMMAND, &wd);
		for (cnt = 0; cnt < 30000; cnt++) {
			udelay(10);
			if ((RD_REG_WORD(&reg->ctrl_status) &
			    CSR_ISP_SOFT_RESET) == 0)
				break;
		}
	}

	if (ha->device_id != QLA2300_DEVICE_ID) {
		for (cnt = 30000; RD_REG_WORD(&reg->mailbox0) != 0 &&
		    rval == QLA2X00_SUCCESS; cnt--) {
			if (cnt)
				udelay(100);
			else
				rval = QL_STATUS_TIMEOUT;
		}
	}

	if (rval == QLA2X00_SUCCESS) {
		/* Get RISC SRAM. */
		risc_address = 0x800;
		WRT_REG_WORD(&reg->mailbox0, MBC_READ_RAM_WORD);
		clear_bit(MBX_INTERRUPT, &ha->mbx_cmd_flags);
	}
	for (cnt = 0; cnt < sizeof(fw->risc_ram) / 2 && rval == QLA2X00_SUCCESS;
	    cnt++, risc_address++) {
		WRT_REG_WORD(&reg->mailbox1, (uint16_t)risc_address);
		RD_REG_WORD(&reg->mailbox1);
		WRT_REG_WORD(&reg->host_cmd, HC_SET_HOST_INT);

		for (timer = 6000000; timer; timer--) {
			/* Check for pending interrupts. */
                        stat = RD_REG_WORD(&reg->host_status_lo);
			if (stat & HOST_STATUS_INT) {
				stat &= 0xff;

				if (stat == 0x1 || stat == 0x2) {
					set_bit(MBX_INTERRUPT,
					    &ha->mbx_cmd_flags);

					mb0 = RD_REG_WORD(&reg->mailbox0);
					mb2 = RD_REG_WORD(&reg->mailbox2);

					/* Release mailbox registers. */
					WRT_REG_WORD(&reg->semaphore, 0);
					WRT_REG_WORD(&reg->host_cmd,
					    HC_CLR_RISC_INT);
					RD_REG_WORD(&reg->host_cmd);
					break;
				} else if (stat == 0x10 || stat == 0x11) {
					set_bit(MBX_INTERRUPT,
					    &ha->mbx_cmd_flags);

					mb0 = RD_REG_WORD(&reg->mailbox0);
					mb2 = RD_REG_WORD(&reg->mailbox2);

					WRT_REG_WORD(&reg->host_cmd,
					    HC_CLR_RISC_INT);
					RD_REG_WORD(&reg->host_cmd);
					break;
				}

				/* clear this intr; it wasn't a mailbox intr */
				WRT_REG_WORD(&reg->host_cmd, HC_CLR_RISC_INT);
				RD_REG_WORD(&reg->host_cmd);
			}
			udelay(5);
		}

		if (test_and_clear_bit(MBX_INTERRUPT, &ha->mbx_cmd_flags)) {
			rval = mb0 & MBS_MASK;
			fw->risc_ram[cnt] = htons(mb2);
		} else {
			rval = QLA2X00_FUNCTION_FAILED;
		}
	}

	if (rval == QLA2X00_SUCCESS) {
		/* Get stack SRAM. */
		risc_address = 0x10000;
		WRT_REG_WORD(&reg->mailbox0, MBC_READ_RAM_EXTENDED);
		clear_bit(MBX_INTERRUPT, &ha->mbx_cmd_flags);
	}
	for (cnt = 0; cnt < sizeof(fw->stack_ram) / 2 && rval == QLA2X00_SUCCESS;
	    cnt++, risc_address++) {
		WRT_REG_WORD(&reg->mailbox1, LSW(risc_address));
		WRT_REG_WORD(&reg->mailbox8, MSW(risc_address));
		RD_REG_WORD(&reg->mailbox8);
		WRT_REG_WORD(&reg->host_cmd, HC_SET_HOST_INT);

		for (timer = 6000000; timer; timer--) {
			/* Check for pending interrupts. */
                        stat = RD_REG_WORD(&reg->host_status_lo);
			if (stat & HOST_STATUS_INT) {
				stat &= 0xff;

				if (stat == 0x1 || stat == 0x2) {
					set_bit(MBX_INTERRUPT,
					    &ha->mbx_cmd_flags);

					mb0 = RD_REG_WORD(&reg->mailbox0);
					mb2 = RD_REG_WORD(&reg->mailbox2);

					/* Release mailbox registers. */
					WRT_REG_WORD(&reg->semaphore, 0);
					WRT_REG_WORD(&reg->host_cmd,
					    HC_CLR_RISC_INT);
					RD_REG_WORD(&reg->host_cmd);
					break;
				} else if (stat == 0x10 || stat == 0x11) {
					set_bit(MBX_INTERRUPT,
					    &ha->mbx_cmd_flags);

					mb0 = RD_REG_WORD(&reg->mailbox0);
					mb2 = RD_REG_WORD(&reg->mailbox2);

					WRT_REG_WORD(&reg->host_cmd,
					    HC_CLR_RISC_INT);
					RD_REG_WORD(&reg->host_cmd);
					break;
				}

				/* clear this intr; it wasn't a mailbox intr */
				WRT_REG_WORD(&reg->host_cmd, HC_CLR_RISC_INT);
				RD_REG_WORD(&reg->host_cmd);
			}
			udelay(5);
		}

		if (test_and_clear_bit(MBX_INTERRUPT, &ha->mbx_cmd_flags)) {
			rval = mb0 & MBS_MASK;
			fw->stack_ram[cnt] = htons(mb2);
		} else {
			rval = QLA2X00_FUNCTION_FAILED;
		}
	}

	if (rval == QLA2X00_SUCCESS) {
		/* Get data SRAM. */
		risc_address = 0x11000;
		data_ram_cnt = ha->fw_memory_size - risc_address + 1;
		WRT_REG_WORD(&reg->mailbox0, MBC_READ_RAM_EXTENDED);
		clear_bit(MBX_INTERRUPT, &ha->mbx_cmd_flags);
	}
	for (cnt = 0; cnt < data_ram_cnt && rval == QLA2X00_SUCCESS;
	    cnt++, risc_address++) {
		WRT_REG_WORD(&reg->mailbox1, LSW(risc_address));
		WRT_REG_WORD(&reg->mailbox8, MSW(risc_address));
		RD_REG_WORD(&reg->mailbox8);
		WRT_REG_WORD(&reg->host_cmd, HC_SET_HOST_INT);

		for (timer = 6000000; timer; timer--) {
			/* Check for pending interrupts. */
                        stat = RD_REG_WORD(&reg->host_status_lo);
			if (stat & HOST_STATUS_INT) {
				stat &= 0xff;

				if (stat == 0x1 || stat == 0x2) {
					set_bit(MBX_INTERRUPT,
					    &ha->mbx_cmd_flags);

					mb0 = RD_REG_WORD(&reg->mailbox0);
					mb2 = RD_REG_WORD(&reg->mailbox2);

					/* Release mailbox registers. */
					WRT_REG_WORD(&reg->semaphore, 0);
					WRT_REG_WORD(&reg->host_cmd,
					    HC_CLR_RISC_INT);
					RD_REG_WORD(&reg->host_cmd);
					break;
				} else if (stat == 0x10 || stat == 0x11) {
					set_bit(MBX_INTERRUPT,
					    &ha->mbx_cmd_flags);

					mb0 = RD_REG_WORD(&reg->mailbox0);
					mb2 = RD_REG_WORD(&reg->mailbox2);

					WRT_REG_WORD(&reg->host_cmd,
					    HC_CLR_RISC_INT);
					RD_REG_WORD(&reg->host_cmd);
					break;
				}

				/* clear this intr; it wasn't a mailbox intr */
				WRT_REG_WORD(&reg->host_cmd, HC_CLR_RISC_INT);
				RD_REG_WORD(&reg->host_cmd);
			}
			udelay(5);
		}

		if (test_and_clear_bit(MBX_INTERRUPT, &ha->mbx_cmd_flags)) {
			rval = mb0 & MBS_MASK;
			fw->data_ram[cnt] = htons(mb2);
		} else {
			rval = QLA2X00_FUNCTION_FAILED;
		}
	}

	if (rval == QLA2X00_SUCCESS)
		qla2xxx_copy_queues(ha, &fw->data_ram[cnt]);

	if (rval != QLA2X00_SUCCESS) {
		printk(KERN_WARNING
		    "scsi(%ld): Failed to dump firmware (%x)!!!\n",
		    ha->host_no, rval);
		ha->fw_dumped = 0;
	} else {
		printk(KERN_INFO
		    "scsi(%ld): Firmware dump saved to temp buffer "
		    "(%ld/%p).\n", ha->host_no, ha->host_no, ha->fw_dump);
		ha->fw_dumped = 1;
	}

qla2300_fw_dump_failed:
	if (!hardware_locked)
		spin_unlock_irqrestore(&ha->hardware_lock, flags);
}

void
qla24xx_fw_dump(scsi_qla_host_t *ha, int hardware_locked)
{
	int		rval;
	uint32_t	cnt;
	uint32_t	risc_address;
	uint16_t	mb0, wd;

	struct device_reg_24xx  *reg =
	    (struct device_reg_24xx  *)ha->iobase;
	uint32_t  *dmp_reg;
	uint32_t	*iter_reg;
	uint16_t  *mbx_reg;
	unsigned long	flags;
	struct qla24xx_fw_dump *fw;
	uint32_t	ext_mem_cnt;
	void		*nxt;

	risc_address = ext_mem_cnt = 0;
	flags = 0;

	if (!hardware_locked)
		spin_lock_irqsave(&(to_qla_parent(ha))->hardware_lock, flags);

	if (!ha->fw_dump) {
		printk(KERN_WARNING
		    "scsi(%ld): No buffer available for dump!!!\n",
		    ha->host_no);
		goto qla24xx_fw_dump_failed;
	}

	if (ha->fw_dumped) {
		printk(KERN_WARNING
		    "scsi(%ld): Firmware has been previously dumped (%p) "
		    "-- ignoring request...\n", ha->host_no, ha->fw_dump);
		goto qla24xx_fw_dump_failed;
	}
	fw = &ha->fw_dump->isp.isp24;
	qla2xxx_prep_dump(ha, ha->fw_dump);

	rval = QLA2X00_SUCCESS;
	fw->host_status = htonl(RD_REG_DWORD(&reg->host_status));

	/* Pause RISC. */
	if ((RD_REG_DWORD(&reg->hccr) & HCCRX_RISC_PAUSE) == 0) {
		WRT_REG_DWORD(&reg->hccr, HCCRX_SET_RISC_PAUSE);
		for (cnt = 30000;
		    (RD_REG_DWORD(&reg->hccr) & HCCRX_RISC_PAUSE) == 0 &&
		    rval == QLA2X00_SUCCESS; cnt--) {
			if (cnt)
				udelay(100);
			else
				rval = QL_STATUS_TIMEOUT;
		}
	}

	if (rval == QLA2X00_SUCCESS) {
		/* Host interface registers. */
		dmp_reg = (uint32_t  *)(reg + 0);
		for (cnt = 0; cnt < sizeof(fw->host_reg) / 4; cnt++)
			fw->host_reg[cnt] = htonl(RD_REG_DWORD(dmp_reg++));

		/* Disable interrupts. */
		WRT_REG_DWORD(&reg->ictrl, 0);
		RD_REG_DWORD(&reg->ictrl);

		/* Shadow registers. */
		WRT_REG_DWORD(&reg->iobase_addr, 0x0F70);
		RD_REG_DWORD(&reg->iobase_addr);
		dmp_reg = (uint32_t  *)((uint8_t  *)reg + 0xF0);
		WRT_REG_DWORD(dmp_reg, 0xB0000000);
		dmp_reg = (uint32_t  *)((uint8_t  *)reg + 0xFC);
		fw->shadow_reg[0] = htonl(RD_REG_DWORD(dmp_reg));

		dmp_reg = (uint32_t  *)((uint8_t  *)reg + 0xF0);
		WRT_REG_DWORD(dmp_reg, 0xB0100000);
		dmp_reg = (uint32_t  *)((uint8_t  *)reg + 0xFC);
		fw->shadow_reg[1] = htonl(RD_REG_DWORD(dmp_reg));

		dmp_reg = (uint32_t  *)((uint8_t  *)reg + 0xF0);
		WRT_REG_DWORD(dmp_reg, 0xB0200000);
		dmp_reg = (uint32_t  *)((uint8_t  *)reg + 0xFC);
		fw->shadow_reg[2] = htonl(RD_REG_DWORD(dmp_reg));

		dmp_reg = (uint32_t  *)((uint8_t  *)reg + 0xF0);
		WRT_REG_DWORD(dmp_reg, 0xB0300000);
		dmp_reg = (uint32_t  *)((uint8_t  *)reg + 0xFC);
		fw->shadow_reg[3] = htonl(RD_REG_DWORD(dmp_reg));

		dmp_reg = (uint32_t  *)((uint8_t  *)reg + 0xF0);
		WRT_REG_DWORD(dmp_reg, 0xB0400000);
		dmp_reg = (uint32_t  *)((uint8_t  *)reg + 0xFC);
		fw->shadow_reg[4] = htonl(RD_REG_DWORD(dmp_reg));

		dmp_reg = (uint32_t  *)((uint8_t  *)reg + 0xF0);
		WRT_REG_DWORD(dmp_reg, 0xB0500000);
		dmp_reg = (uint32_t  *)((uint8_t  *)reg + 0xFC);
		fw->shadow_reg[5] = htonl(RD_REG_DWORD(dmp_reg));

		dmp_reg = (uint32_t  *)((uint8_t  *)reg + 0xF0);
		WRT_REG_DWORD(dmp_reg, 0xB0600000);
		dmp_reg = (uint32_t  *)((uint8_t  *)reg + 0xFC);
		fw->shadow_reg[6] = htonl(RD_REG_DWORD(dmp_reg));

		/* Mailbox registers. */
		mbx_reg = (uint16_t  *)((uint8_t  *)reg + 0x80);
		for (cnt = 0; cnt < sizeof(fw->mailbox_reg) / 2; cnt++)
			fw->mailbox_reg[cnt] = htons(RD_REG_WORD(mbx_reg++));

		/* Transfer sequence registers. */
		iter_reg = fw->xseq_gp_reg;
		iter_reg = qla24xx_read_window(reg, 0xBF00, 16, iter_reg);
		iter_reg = qla24xx_read_window(reg, 0xBF10, 16, iter_reg);
		iter_reg = qla24xx_read_window(reg, 0xBF20, 16, iter_reg);
		iter_reg = qla24xx_read_window(reg, 0xBF30, 16, iter_reg);
		iter_reg = qla24xx_read_window(reg, 0xBF40, 16, iter_reg);
		iter_reg = qla24xx_read_window(reg, 0xBF50, 16, iter_reg);
		iter_reg = qla24xx_read_window(reg, 0xBF60, 16, iter_reg);
		qla24xx_read_window(reg, 0xBF70, 16, iter_reg);

		qla24xx_read_window(reg, 0xBFE0, 16, fw->xseq_0_reg);
		qla24xx_read_window(reg, 0xBFF0, 16, fw->xseq_1_reg);

		/* Receive sequence registers. */
		iter_reg = fw->rseq_gp_reg;
		iter_reg = qla24xx_read_window(reg, 0xFF00, 16, iter_reg);
		iter_reg = qla24xx_read_window(reg, 0xFF10, 16, iter_reg);
		iter_reg = qla24xx_read_window(reg, 0xFF20, 16, iter_reg);
		iter_reg = qla24xx_read_window(reg, 0xFF30, 16, iter_reg);
		iter_reg = qla24xx_read_window(reg, 0xFF40, 16, iter_reg);
		iter_reg = qla24xx_read_window(reg, 0xFF50, 16, iter_reg);
		iter_reg = qla24xx_read_window(reg, 0xFF60, 16, iter_reg);
		qla24xx_read_window(reg, 0xFF70, 16, iter_reg);

		qla24xx_read_window(reg, 0xFFD0, 16, fw->rseq_0_reg);
		qla24xx_read_window(reg, 0xFFE0, 16, fw->rseq_1_reg);
		qla24xx_read_window(reg, 0xFFF0, 16, fw->rseq_2_reg);

		/* Command DMA registers. */
		qla24xx_read_window(reg, 0x7100, 16, fw->cmd_dma_reg);

		/* Queues. */
		iter_reg = fw->req0_dma_reg;
		WRT_REG_DWORD(&reg->iobase_addr, 0x7200);
		dmp_reg = (uint32_t  *)((uint8_t  *)reg + 0xC0);
		for (cnt = 0; cnt < 8; cnt++)
			*iter_reg++ = htonl(RD_REG_DWORD(dmp_reg++));

		dmp_reg = (uint32_t  *)((uint8_t  *)reg + 0xE4);
		for (cnt = 0; cnt < 7; cnt++)
			*iter_reg++ = htonl(RD_REG_DWORD(dmp_reg++));

		iter_reg = fw->resp0_dma_reg;
		WRT_REG_DWORD(&reg->iobase_addr, 0x7300);
		dmp_reg = (uint32_t  *)((uint8_t  *)reg + 0xC0);
		for (cnt = 0; cnt < 8; cnt++)
			*iter_reg++ = htonl(RD_REG_DWORD(dmp_reg++));

		dmp_reg = (uint32_t  *)((uint8_t  *)reg + 0xE4);
		for (cnt = 0; cnt < 7; cnt++)
			*iter_reg++ = htonl(RD_REG_DWORD(dmp_reg++));

		iter_reg = fw->req1_dma_reg;
		WRT_REG_DWORD(&reg->iobase_addr, 0x7400);
		dmp_reg = (uint32_t  *)((uint8_t  *)reg + 0xC0);
		for (cnt = 0; cnt < 8; cnt++)
			*iter_reg++ = htonl(RD_REG_DWORD(dmp_reg++));

		dmp_reg = (uint32_t  *)((uint8_t  *)reg + 0xE4);
		for (cnt = 0; cnt < 7; cnt++)
			*iter_reg++ = htonl(RD_REG_DWORD(dmp_reg++));

		/* Transmit DMA registers. */
		iter_reg = fw->xmt0_dma_reg;
		iter_reg = qla24xx_read_window(reg, 0x7600, 16, iter_reg);
		qla24xx_read_window(reg, 0x7610, 16, iter_reg);

		iter_reg = fw->xmt1_dma_reg;
		iter_reg = qla24xx_read_window(reg, 0x7620, 16, iter_reg);
		qla24xx_read_window(reg, 0x7630, 16, iter_reg);

		iter_reg = fw->xmt2_dma_reg;
		iter_reg = qla24xx_read_window(reg, 0x7640, 16, iter_reg);
		qla24xx_read_window(reg, 0x7650, 16, iter_reg);

		iter_reg = fw->xmt3_dma_reg;
		iter_reg = qla24xx_read_window(reg, 0x7660, 16, iter_reg);
		qla24xx_read_window(reg, 0x7670, 16, iter_reg);

		iter_reg = fw->xmt4_dma_reg;
		iter_reg = qla24xx_read_window(reg, 0x7680, 16, iter_reg);
		qla24xx_read_window(reg, 0x7690, 16, iter_reg);

		qla24xx_read_window(reg, 0x76A0, 16, fw->xmt_data_dma_reg);

		/* Receive DMA registers. */
		iter_reg = fw->rcvt0_data_dma_reg;
		iter_reg = qla24xx_read_window(reg, 0x7700, 16, iter_reg);
		qla24xx_read_window(reg, 0x7710, 16, iter_reg);

		iter_reg = fw->rcvt1_data_dma_reg;
		iter_reg = qla24xx_read_window(reg, 0x7720, 16, iter_reg);
		qla24xx_read_window(reg, 0x7730, 16, iter_reg);

		/* RISC registers. */
		iter_reg = fw->risc_gp_reg;
		iter_reg = qla24xx_read_window(reg, 0x0F00, 16, iter_reg);
		iter_reg = qla24xx_read_window(reg, 0x0F10, 16, iter_reg);
		iter_reg = qla24xx_read_window(reg, 0x0F20, 16, iter_reg);
		iter_reg = qla24xx_read_window(reg, 0x0F30, 16, iter_reg);
		iter_reg = qla24xx_read_window(reg, 0x0F40, 16, iter_reg);
		iter_reg = qla24xx_read_window(reg, 0x0F50, 16, iter_reg);
		iter_reg = qla24xx_read_window(reg, 0x0F60, 16, iter_reg);
		qla24xx_read_window(reg, 0x0F70, 16, iter_reg);

		/* Local memory controller registers. */
		iter_reg = fw->lmc_reg;
		iter_reg = qla24xx_read_window(reg, 0x3000, 16, iter_reg);
		iter_reg = qla24xx_read_window(reg, 0x3010, 16, iter_reg);
		iter_reg = qla24xx_read_window(reg, 0x3020, 16, iter_reg);
		iter_reg = qla24xx_read_window(reg, 0x3030, 16, iter_reg);
		iter_reg = qla24xx_read_window(reg, 0x3040, 16, iter_reg);
		iter_reg = qla24xx_read_window(reg, 0x3050, 16, iter_reg);
		qla24xx_read_window(reg, 0x3060, 16, iter_reg);

		/* Fibre Protocol Module registers. */
		iter_reg = fw->fpm_hdw_reg;
		iter_reg = qla24xx_read_window(reg, 0x4000, 16, iter_reg);
		iter_reg = qla24xx_read_window(reg, 0x4010, 16, iter_reg);
		iter_reg = qla24xx_read_window(reg, 0x4020, 16, iter_reg);
		iter_reg = qla24xx_read_window(reg, 0x4030, 16, iter_reg);
		iter_reg = qla24xx_read_window(reg, 0x4040, 16, iter_reg);
		iter_reg = qla24xx_read_window(reg, 0x4050, 16, iter_reg);
		iter_reg = qla24xx_read_window(reg, 0x4060, 16, iter_reg);
		iter_reg = qla24xx_read_window(reg, 0x4070, 16, iter_reg);
		iter_reg = qla24xx_read_window(reg, 0x4080, 16, iter_reg);
		iter_reg = qla24xx_read_window(reg, 0x4090, 16, iter_reg);
		iter_reg = qla24xx_read_window(reg, 0x40A0, 16, iter_reg);
		qla24xx_read_window(reg, 0x40B0, 16, iter_reg);

		/* Frame Buffer registers. */
		iter_reg = fw->fb_hdw_reg;
		iter_reg = qla24xx_read_window(reg, 0x6000, 16, iter_reg);
		iter_reg = qla24xx_read_window(reg, 0x6010, 16, iter_reg);
		iter_reg = qla24xx_read_window(reg, 0x6020, 16, iter_reg);
		iter_reg = qla24xx_read_window(reg, 0x6030, 16, iter_reg);
		iter_reg = qla24xx_read_window(reg, 0x6040, 16, iter_reg);
		iter_reg = qla24xx_read_window(reg, 0x6100, 16, iter_reg);
		iter_reg = qla24xx_read_window(reg, 0x6130, 16, iter_reg);
		iter_reg = qla24xx_read_window(reg, 0x6150, 16, iter_reg);
		iter_reg = qla24xx_read_window(reg, 0x6170, 16, iter_reg);
		iter_reg = qla24xx_read_window(reg, 0x6190, 16, iter_reg);
		qla24xx_read_window(reg, 0x61B0, 16, iter_reg);

		/* Reset RISC. */
		WRT_REG_DWORD(&reg->ctrl_status,
		    CSRX_DMA_SHUTDOWN|MWB_4096_BYTES);
		for (cnt = 0; cnt < 30000; cnt++) {
			if ((RD_REG_DWORD(&reg->ctrl_status) &
			    CSRX_DMA_ACTIVE) == 0)
				break;

			udelay(10);
		}

		WRT_REG_DWORD(&reg->ctrl_status,
		    CSRX_ISP_SOFT_RESET|CSRX_DMA_SHUTDOWN|MWB_4096_BYTES);
		pci_read_config_word(ha->pdev, PCI_COMMAND, &wd);

		udelay(100);
		/* Wait for firmware to complete NVRAM accesses. */
		mb0 = (uint32_t) RD_REG_WORD(&reg->mailbox0);
		for (cnt = 10000 ; cnt && mb0; cnt--) {
			udelay(5);
			mb0 = (uint32_t) RD_REG_WORD(&reg->mailbox0);
			barrier();
		}

		/* Wait for soft-reset to complete. */
		for (cnt = 0; cnt < 30000; cnt++) {
			if ((RD_REG_DWORD(&reg->ctrl_status) &
			    CSRX_ISP_SOFT_RESET) == 0)
				break;

			udelay(10);
		}
		WRT_REG_DWORD(&reg->hccr, HCCRX_CLR_RISC_RESET);
		RD_REG_DWORD(&reg->hccr);             /* PCI Posting. */
	}

	for (cnt = 30000; RD_REG_WORD(&reg->mailbox0) != 0 &&
	    rval == QLA2X00_SUCCESS; cnt--) {
		if (cnt)
			udelay(100);
		else
			rval = QL_STATUS_TIMEOUT;
	}

	if (rval == QLA2X00_SUCCESS)
		rval = qla2xxx_dump_memory(ha, fw->code_ram,
		    sizeof(fw->code_ram), fw->ext_mem, &nxt);

	if (rval == QLA2X00_SUCCESS) {
		nxt = qla2xxx_copy_queues(ha, nxt);
		if (ha->eft)
			memcpy(nxt, ha->eft, ntohl(ha->fw_dump->eft_size));
	}
	if (rval != QLA2X00_SUCCESS) {
		printk(KERN_WARNING
		    "scsi(%ld): Failed to dump firmware (%x)!!!\n",
		    ha->host_no, rval);
		ha->fw_dumped = 0;
	} else {
		printk(KERN_INFO
		    "scsi(%ld): Firmware dump saved to temp buffer "
		    "(%ld/%p).\n", ha->host_no, ha->host_no, ha->fw_dump);
		ha->fw_dumped = 1;
	}

qla24xx_fw_dump_failed:
	if (!hardware_locked)
		spin_unlock_irqrestore(&(to_qla_parent(ha))->hardware_lock, flags);
}

void
qla25xx_fw_dump(scsi_qla_host_t *ha, int hardware_locked)
{
	int		rval;
	uint32_t	cnt;
	uint32_t	risc_address;
	uint16_t	mb0, wd;

	struct device_reg_24xx  *reg =
	    (struct device_reg_24xx  *)ha->iobase;
	uint32_t  *dmp_reg;
	uint32_t  *iter_reg;
	uint16_t  *mbx_reg;
	unsigned long	flags;
	struct qla25xx_fw_dump *fw;
	uint32_t	ext_mem_cnt;
	void		*nxt;

	risc_address = ext_mem_cnt = 0;
	flags = 0;

	if (!hardware_locked)
		spin_lock_irqsave(&(to_qla_parent(ha))->hardware_lock, flags);

	if (!ha->fw_dump) {
		printk(KERN_WARNING
		    "No buffer available for dump!!!\n");
		goto qla25xx_fw_dump_failed;
	}

	if (ha->fw_dumped) {
		printk(KERN_WARNING
		    "Firmware has been previously dumped (%p) -- ignoring "
		    "request...\n", ha->fw_dump);
		goto qla25xx_fw_dump_failed;
	}
	fw = &ha->fw_dump->isp.isp25;
	qla2xxx_prep_dump(ha, ha->fw_dump);
	ha->fw_dump->version = __constant_htonl(2);

	rval = QLA2X00_SUCCESS;
	fw->host_status = htonl(RD_REG_DWORD(&reg->host_status));

	/* Pause RISC. */
	if ((RD_REG_DWORD(&reg->hccr) & HCCRX_RISC_PAUSE) == 0) {
		WRT_REG_DWORD(&reg->hccr, HCCRX_SET_RISC_PAUSE);
		for (cnt = 30000;
		    (RD_REG_DWORD(&reg->hccr) & HCCRX_RISC_PAUSE) == 0 &&
		    rval == QLA2X00_SUCCESS; cnt--) {
			if (cnt)
				udelay(100);
			else
				rval = QL_STATUS_TIMEOUT;
		}
	}

	if (rval == QLA2X00_SUCCESS) {
		/* Host/Risc registers. */
		iter_reg = fw->host_risc_reg;
		iter_reg = qla24xx_read_window(reg, 0x7000, 16, iter_reg);
		qla24xx_read_window(reg, 0x7010, 16, iter_reg);

		/* PCIe registers. */
		WRT_REG_DWORD(&reg->iobase_addr, 0x7C00);
		RD_REG_DWORD(&reg->iobase_addr);
		WRT_REG_DWORD(&reg->iobase_window, 0x01);
		dmp_reg = &reg->iobase_c4;
		fw->pcie_regs[0] = htonl(RD_REG_DWORD(dmp_reg++));
		fw->pcie_regs[1] = htonl(RD_REG_DWORD(dmp_reg++));
		fw->pcie_regs[2] = htonl(RD_REG_DWORD(dmp_reg));
		fw->pcie_regs[3] = htonl(RD_REG_DWORD(&reg->iobase_window));
		WRT_REG_DWORD(&reg->iobase_window, 0x00);
		RD_REG_DWORD(&reg->iobase_window);

		/* Host interface registers. */
		dmp_reg = (uint32_t  *)(reg + 0);
		for (cnt = 0; cnt < sizeof(fw->host_reg) / 4; cnt++)
			fw->host_reg[cnt] = htonl(RD_REG_DWORD(dmp_reg++));

		/* Disable interrupts. */
		WRT_REG_DWORD(&reg->ictrl, 0);
		RD_REG_DWORD(&reg->ictrl);

		/* Shadow registers. */
		WRT_REG_DWORD(&reg->iobase_addr, 0x0F70);
		RD_REG_DWORD(&reg->iobase_addr);
		dmp_reg = (uint32_t  *)((uint8_t  *)reg + 0xF0);
		WRT_REG_DWORD(dmp_reg, 0xB0000000);
		dmp_reg = (uint32_t  *)((uint8_t  *)reg + 0xFC);
		fw->shadow_reg[0] = htonl(RD_REG_DWORD(dmp_reg));

		dmp_reg = (uint32_t  *)((uint8_t  *)reg + 0xF0);
		WRT_REG_DWORD(dmp_reg, 0xB0100000);
		dmp_reg = (uint32_t  *)((uint8_t  *)reg + 0xFC);
		fw->shadow_reg[1] = htonl(RD_REG_DWORD(dmp_reg));

		dmp_reg = (uint32_t  *)((uint8_t  *)reg + 0xF0);
		WRT_REG_DWORD(dmp_reg, 0xB0200000);
		dmp_reg = (uint32_t  *)((uint8_t  *)reg + 0xFC);
		fw->shadow_reg[2] = htonl(RD_REG_DWORD(dmp_reg));

		dmp_reg = (uint32_t  *)((uint8_t  *)reg + 0xF0);
		WRT_REG_DWORD(dmp_reg, 0xB0300000);
		dmp_reg = (uint32_t  *)((uint8_t  *)reg + 0xFC);
		fw->shadow_reg[3] = htonl(RD_REG_DWORD(dmp_reg));

		dmp_reg = (uint32_t  *)((uint8_t  *)reg + 0xF0);
		WRT_REG_DWORD(dmp_reg, 0xB0400000);
		dmp_reg = (uint32_t  *)((uint8_t  *)reg + 0xFC);
		fw->shadow_reg[4] = htonl(RD_REG_DWORD(dmp_reg));

		dmp_reg = (uint32_t  *)((uint8_t  *)reg + 0xF0);
		WRT_REG_DWORD(dmp_reg, 0xB0500000);
		dmp_reg = (uint32_t  *)((uint8_t  *)reg + 0xFC);
		fw->shadow_reg[5] = htonl(RD_REG_DWORD(dmp_reg));

		dmp_reg = (uint32_t  *)((uint8_t  *)reg + 0xF0);
		WRT_REG_DWORD(dmp_reg, 0xB0600000);
		dmp_reg = (uint32_t  *)((uint8_t  *)reg + 0xFC);
		fw->shadow_reg[6] = htonl(RD_REG_DWORD(dmp_reg));

		dmp_reg = (uint32_t  *)((uint8_t  *)reg + 0xF0);
		WRT_REG_DWORD(dmp_reg, 0xB0700000);
		dmp_reg = (uint32_t  *)((uint8_t  *)reg + 0xFC);
		fw->shadow_reg[7] = htonl(RD_REG_DWORD(dmp_reg));

		dmp_reg = (uint32_t  *)((uint8_t  *)reg + 0xF0);
		WRT_REG_DWORD(dmp_reg, 0xB0800000);
		dmp_reg = (uint32_t  *)((uint8_t  *)reg + 0xFC);
		fw->shadow_reg[8] = htonl(RD_REG_DWORD(dmp_reg));

		dmp_reg = (uint32_t  *)((uint8_t  *)reg + 0xF0);
		WRT_REG_DWORD(dmp_reg, 0xB0900000);
		dmp_reg = (uint32_t  *)((uint8_t  *)reg + 0xFC);
		fw->shadow_reg[9] = htonl(RD_REG_DWORD(dmp_reg));

		dmp_reg = (uint32_t  *)((uint8_t  *)reg + 0xF0);
		WRT_REG_DWORD(dmp_reg, 0xB0A00000);
		dmp_reg = (uint32_t  *)((uint8_t  *)reg + 0xFC);
		fw->shadow_reg[10] = htonl(RD_REG_DWORD(dmp_reg));

		/* RISC I/O register. */
		WRT_REG_DWORD(&reg->iobase_addr, 0x0010);
		RD_REG_DWORD(&reg->iobase_addr);
		dmp_reg = (uint32_t  *)((uint8_t  *)reg + 0xC0);
		fw->risc_io_reg = htonl(RD_REG_DWORD(dmp_reg));

		/* Mailbox registers. */
		mbx_reg = (uint16_t  *)((uint8_t  *)reg + 0x80);
		for (cnt = 0; cnt < sizeof(fw->mailbox_reg) / 2; cnt++)
			fw->mailbox_reg[cnt] = htons(RD_REG_WORD(mbx_reg++));

		/* Transfer sequence registers. */
		iter_reg = fw->xseq_gp_reg;
		iter_reg = qla24xx_read_window(reg, 0xBF00, 16, iter_reg);
		iter_reg = qla24xx_read_window(reg, 0xBF10, 16, iter_reg);
		iter_reg = qla24xx_read_window(reg, 0xBF20, 16, iter_reg);
		iter_reg = qla24xx_read_window(reg, 0xBF30, 16, iter_reg);
		iter_reg = qla24xx_read_window(reg, 0xBF40, 16, iter_reg);
		iter_reg = qla24xx_read_window(reg, 0xBF50, 16, iter_reg);
		iter_reg = qla24xx_read_window(reg, 0xBF60, 16, iter_reg);
		qla24xx_read_window(reg, 0xBF70, 16, iter_reg);

		iter_reg = fw->xseq_0_reg;
		iter_reg = qla24xx_read_window(reg, 0xBFC0, 16, iter_reg);
		iter_reg = qla24xx_read_window(reg, 0xBFD0, 16, iter_reg);
		qla24xx_read_window(reg, 0xBFE0, 16, iter_reg);

		qla24xx_read_window(reg, 0xBFF0, 16, fw->xseq_1_reg);

		/* Receive sequence registers. */
		iter_reg = fw->rseq_gp_reg;
		iter_reg = qla24xx_read_window(reg, 0xFF00, 16, iter_reg);
		iter_reg = qla24xx_read_window(reg, 0xFF10, 16, iter_reg);
		iter_reg = qla24xx_read_window(reg, 0xFF20, 16, iter_reg);
		iter_reg = qla24xx_read_window(reg, 0xFF30, 16, iter_reg);
		iter_reg = qla24xx_read_window(reg, 0xFF40, 16, iter_reg);
		iter_reg = qla24xx_read_window(reg, 0xFF50, 16, iter_reg);
		iter_reg = qla24xx_read_window(reg, 0xFF60, 16, iter_reg);
		qla24xx_read_window(reg, 0xFF70, 16, iter_reg);

		iter_reg = fw->rseq_0_reg;
		iter_reg = qla24xx_read_window(reg, 0xFFC0, 16, iter_reg);
		qla24xx_read_window(reg, 0xFFD0, 16, iter_reg);

		qla24xx_read_window(reg, 0xFFE0, 16, fw->rseq_1_reg);
		qla24xx_read_window(reg, 0xFFF0, 16, fw->rseq_2_reg);

		/* Auxiliary sequence registers. */
		iter_reg = fw->aseq_gp_reg;
		iter_reg = qla24xx_read_window(reg, 0xB000, 16, iter_reg);
		iter_reg = qla24xx_read_window(reg, 0xB010, 16, iter_reg);
		iter_reg = qla24xx_read_window(reg, 0xB020, 16, iter_reg);
		iter_reg = qla24xx_read_window(reg, 0xB030, 16, iter_reg);
		iter_reg = qla24xx_read_window(reg, 0xB040, 16, iter_reg);
		iter_reg = qla24xx_read_window(reg, 0xB050, 16, iter_reg);
		iter_reg = qla24xx_read_window(reg, 0xB060, 16, iter_reg);
		qla24xx_read_window(reg, 0xB070, 16, iter_reg);

		iter_reg = fw->aseq_0_reg;
		iter_reg = qla24xx_read_window(reg, 0xB0C0, 16, iter_reg);
		qla24xx_read_window(reg, 0xB0D0, 16, iter_reg);

		qla24xx_read_window(reg, 0xB0E0, 16, fw->aseq_1_reg);
		qla24xx_read_window(reg, 0xB0F0, 16, fw->aseq_2_reg);

		/* Command DMA registers. */
		qla24xx_read_window(reg, 0x7100, 16, fw->cmd_dma_reg);

		/* Queues. */
		iter_reg = fw->req0_dma_reg;
		WRT_REG_DWORD(&reg->iobase_addr, 0x7200);
		dmp_reg = (uint32_t  *)((uint8_t  *)reg + 0xC0);
		for (cnt = 0; cnt < 8; cnt++)
			*iter_reg++ = htonl(RD_REG_DWORD(dmp_reg++));

		dmp_reg = (uint32_t  *)((uint8_t  *)reg + 0xE4);
		for (cnt = 0; cnt < 7; cnt++)
			*iter_reg++ = htonl(RD_REG_DWORD(dmp_reg++));

		iter_reg = fw->resp0_dma_reg;
		WRT_REG_DWORD(&reg->iobase_addr, 0x7300);
		dmp_reg = (uint32_t  *)((uint8_t  *)reg + 0xC0);
		for (cnt = 0; cnt < 8; cnt++)
			*iter_reg++ = htonl(RD_REG_DWORD(dmp_reg++));

		dmp_reg = (uint32_t  *)((uint8_t  *)reg + 0xE4);
		for (cnt = 0; cnt < 7; cnt++)
			*iter_reg++ = htonl(RD_REG_DWORD(dmp_reg++));

		iter_reg = fw->req1_dma_reg;
		WRT_REG_DWORD(&reg->iobase_addr, 0x7400);
		dmp_reg = (uint32_t  *)((uint8_t  *)reg + 0xC0);
		for (cnt = 0; cnt < 8; cnt++)
			*iter_reg++ = htonl(RD_REG_DWORD(dmp_reg++));

		dmp_reg = (uint32_t  *)((uint8_t  *)reg + 0xE4);
		for (cnt = 0; cnt < 7; cnt++)
			*iter_reg++ = htonl(RD_REG_DWORD(dmp_reg++));

		/* Transmit DMA registers. */
		iter_reg = fw->xmt0_dma_reg;
		iter_reg = qla24xx_read_window(reg, 0x7600, 16, iter_reg);
		qla24xx_read_window(reg, 0x7610, 16, iter_reg);

		iter_reg = fw->xmt1_dma_reg;
		iter_reg = qla24xx_read_window(reg, 0x7620, 16, iter_reg);
		qla24xx_read_window(reg, 0x7630, 16, iter_reg);

		iter_reg = fw->xmt2_dma_reg;
		iter_reg = qla24xx_read_window(reg, 0x7640, 16, iter_reg);
		qla24xx_read_window(reg, 0x7650, 16, iter_reg);

		iter_reg = fw->xmt3_dma_reg;
		iter_reg = qla24xx_read_window(reg, 0x7660, 16, iter_reg);
		qla24xx_read_window(reg, 0x7670, 16, iter_reg);

		iter_reg = fw->xmt4_dma_reg;
		iter_reg = qla24xx_read_window(reg, 0x7680, 16, iter_reg);
		qla24xx_read_window(reg, 0x7690, 16, iter_reg);

		qla24xx_read_window(reg, 0x76A0, 16, fw->xmt_data_dma_reg);

		/* Receive DMA registers. */
		iter_reg = fw->rcvt0_data_dma_reg;
		iter_reg = qla24xx_read_window(reg, 0x7700, 16, iter_reg);
		qla24xx_read_window(reg, 0x7710, 16, iter_reg);

		iter_reg = fw->rcvt1_data_dma_reg;
		iter_reg = qla24xx_read_window(reg, 0x7720, 16, iter_reg);
		qla24xx_read_window(reg, 0x7730, 16, iter_reg);

		/* RISC registers. */
		iter_reg = fw->risc_gp_reg;
		iter_reg = qla24xx_read_window(reg, 0x0F00, 16, iter_reg);
		iter_reg = qla24xx_read_window(reg, 0x0F10, 16, iter_reg);
		iter_reg = qla24xx_read_window(reg, 0x0F20, 16, iter_reg);
		iter_reg = qla24xx_read_window(reg, 0x0F30, 16, iter_reg);
		iter_reg = qla24xx_read_window(reg, 0x0F40, 16, iter_reg);
		iter_reg = qla24xx_read_window(reg, 0x0F50, 16, iter_reg);
		iter_reg = qla24xx_read_window(reg, 0x0F60, 16, iter_reg);
		qla24xx_read_window(reg, 0x0F70, 16, iter_reg);

		/* Local memory controller registers. */
		iter_reg = fw->lmc_reg;
		iter_reg = qla24xx_read_window(reg, 0x3000, 16, iter_reg);
		iter_reg = qla24xx_read_window(reg, 0x3010, 16, iter_reg);
		iter_reg = qla24xx_read_window(reg, 0x3020, 16, iter_reg);
		iter_reg = qla24xx_read_window(reg, 0x3030, 16, iter_reg);
		iter_reg = qla24xx_read_window(reg, 0x3040, 16, iter_reg);
		iter_reg = qla24xx_read_window(reg, 0x3050, 16, iter_reg);
		iter_reg = qla24xx_read_window(reg, 0x3060, 16, iter_reg);
		qla24xx_read_window(reg, 0x3070, 16, iter_reg);

		/* Fibre Protocol Module registers. */
		iter_reg = fw->fpm_hdw_reg;
		iter_reg = qla24xx_read_window(reg, 0x4000, 16, iter_reg);
		iter_reg = qla24xx_read_window(reg, 0x4010, 16, iter_reg);
		iter_reg = qla24xx_read_window(reg, 0x4020, 16, iter_reg);
		iter_reg = qla24xx_read_window(reg, 0x4030, 16, iter_reg);
		iter_reg = qla24xx_read_window(reg, 0x4040, 16, iter_reg);
		iter_reg = qla24xx_read_window(reg, 0x4050, 16, iter_reg);
		iter_reg = qla24xx_read_window(reg, 0x4060, 16, iter_reg);
		iter_reg = qla24xx_read_window(reg, 0x4070, 16, iter_reg);
		iter_reg = qla24xx_read_window(reg, 0x4080, 16, iter_reg);
		iter_reg = qla24xx_read_window(reg, 0x4090, 16, iter_reg);
		iter_reg = qla24xx_read_window(reg, 0x40A0, 16, iter_reg);
		qla24xx_read_window(reg, 0x40B0, 16, iter_reg);

		/* Frame Buffer registers. */
		iter_reg = fw->fb_hdw_reg;
		iter_reg = qla24xx_read_window(reg, 0x6000, 16, iter_reg);
		iter_reg = qla24xx_read_window(reg, 0x6010, 16, iter_reg);
		iter_reg = qla24xx_read_window(reg, 0x6020, 16, iter_reg);
		iter_reg = qla24xx_read_window(reg, 0x6030, 16, iter_reg);
		iter_reg = qla24xx_read_window(reg, 0x6040, 16, iter_reg);
		iter_reg = qla24xx_read_window(reg, 0x6100, 16, iter_reg);
		iter_reg = qla24xx_read_window(reg, 0x6130, 16, iter_reg);
		iter_reg = qla24xx_read_window(reg, 0x6150, 16, iter_reg);
		iter_reg = qla24xx_read_window(reg, 0x6170, 16, iter_reg);
		iter_reg = qla24xx_read_window(reg, 0x6190, 16, iter_reg);
		iter_reg = qla24xx_read_window(reg, 0x61B0, 16, iter_reg);
		qla24xx_read_window(reg, 0x6F00, 16, iter_reg);

		/* Reset RISC. */
		WRT_REG_DWORD(&reg->ctrl_status,
		    CSRX_DMA_SHUTDOWN|MWB_4096_BYTES);
		for (cnt = 0; cnt < 30000; cnt++) {
			if ((RD_REG_DWORD(&reg->ctrl_status) &
			    CSRX_DMA_ACTIVE) == 0)
				break;

			udelay(10);
		}

		WRT_REG_DWORD(&reg->ctrl_status,
		    CSRX_ISP_SOFT_RESET|CSRX_DMA_SHUTDOWN|MWB_4096_BYTES);
		pci_read_config_word(ha->pdev, PCI_COMMAND, &wd);

		udelay(100);
		/* Wait for firmware to complete NVRAM accesses. */
		mb0 = (uint32_t) RD_REG_WORD(&reg->mailbox0);
		for (cnt = 10000 ; cnt && mb0; cnt--) {
			udelay(5);
			mb0 = (uint32_t) RD_REG_WORD(&reg->mailbox0);
			barrier();
		}

		/* Wait for soft-reset to complete. */
		for (cnt = 0; cnt < 30000; cnt++) {
			if ((RD_REG_DWORD(&reg->ctrl_status) &
			    CSRX_ISP_SOFT_RESET) == 0)
				break;

			udelay(10);
		}
		WRT_REG_DWORD(&reg->hccr, HCCRX_CLR_RISC_RESET);
		RD_REG_DWORD(&reg->hccr);             /* PCI Posting. */
	}

	for (cnt = 30000; RD_REG_WORD(&reg->mailbox0) != 0 &&
	    rval == QLA2X00_SUCCESS; cnt--) {
		if (cnt)
			udelay(100);
		else
			rval = QL_STATUS_TIMEOUT;
	}

	if (rval == QLA2X00_SUCCESS)
		rval = qla2xxx_dump_memory(ha, fw->code_ram,
		    sizeof(fw->code_ram), fw->ext_mem, &nxt);

	if (rval == QLA2X00_SUCCESS) {
		nxt = qla2xxx_copy_queues(ha, nxt);
		if (ha->eft)
			memcpy(nxt, ha->eft, ntohl(ha->fw_dump->eft_size));
	}

	if (rval != QLA2X00_SUCCESS) {
		printk(KERN_WARNING
		    "Failed to dump firmware (%x)!!!\n", rval);
		ha->fw_dumped = 0;

	} else {
		printk(KERN_INFO
		    "Firmware dump saved to temp buffer (%ld/%p).\n",
		    ha->host_no, ha->fw_dump);
		ha->fw_dumped = 1;
	}

qla25xx_fw_dump_failed:
	if (!hardware_locked)
		spin_unlock_irqrestore(&(to_qla_parent(ha))->hardware_lock, flags);
}
#endif
