/*
 * QLogic Fibre Channel HBA Driver
 * Copyright (c)  2003-2005 QLogic Corporation
 *
 * See LICENSE.qla2xxx for copyright and licensing details.
 */


#include <linux/vmalloc.h>
#include "qla2x00.h"
#include "inioct.h"

#define VPD_HBAPARAM_SIZE_25XX	0x800

extern int qla2x00_loopback_test(scsi_qla_host_t *ha, INT_LOOPBACK_REQ *req,
    uint16_t *ret_mb);

int qla24xx_read_nvram(scsi_qla_host_t *, EXT_IOCTL *, int);
int qla2x00_read_nvram(scsi_qla_host_t *, EXT_IOCTL *, int);
int qla2x00_update_nvram(scsi_qla_host_t *, EXT_IOCTL *, int);
int qla2x00_write_nvram_word(scsi_qla_host_t *, uint8_t, uint16_t);
int qla2x00_send_loopback(scsi_qla_host_t *, EXT_IOCTL *, int);
int qla2x00_read_option_rom(scsi_qla_host_t *, EXT_IOCTL *, int);
int qla2x00_read_option_rom_ext(scsi_qla_host_t *, EXT_IOCTL *, int);
int qla2x00_update_option_rom(scsi_qla_host_t *, EXT_IOCTL *, int);
int qla2x00_update_option_rom_ext(scsi_qla_host_t *, EXT_IOCTL *, int);
int qla2x00_get_option_rom_layout(scsi_qla_host_t *, EXT_IOCTL *, int);
static void qla2x00_get_option_rom_table(scsi_qla_host_t *,
    INT_OPT_ROM_REGION **, unsigned long *);
int qla2x00_get_fw_dump(scsi_qla_host_t *, EXT_IOCTL *, int);


#define FLASH_CHUNK_SZ	0x20000		/* 128 KBytes */

/* Option ROM definitions. */
INT_OPT_ROM_REGION OptionRomTable2312[] = 
{
    {INT_OPT_ROM_REGION_ALL, INT_OPT_ROM_SIZE_2312,
	    0, INT_OPT_ROM_SIZE_2312-1},
    {INT_OPT_ROM_REGION_PHBIOS_FCODE_EFI_CFW, INT_OPT_ROM_SIZE_2312,
	    0, INT_OPT_ROM_SIZE_2312-1},
    {INT_OPT_ROM_REGION_NONE, 0, 0, 0 }
};

INT_OPT_ROM_REGION OptionRomTable6312[] = // 128k x20000
{
    {INT_OPT_ROM_REGION_ALL,    INT_OPT_ROM_SIZE_2312,
	    0, INT_OPT_ROM_SIZE_2312-1},
    {INT_OPT_ROM_REGION_PHBIOS_CFW, INT_OPT_ROM_SIZE_2312,
	    0, INT_OPT_ROM_SIZE_2312-1},
    {INT_OPT_ROM_REGION_NONE, 0, 0, 0 }
};

INT_OPT_ROM_REGION OptionRomTableHp[] = // 128k x20000
{
    {INT_OPT_ROM_REGION_ALL, INT_OPT_ROM_SIZE_2312,
	    0, INT_OPT_ROM_SIZE_2312-1},
    {INT_OPT_ROM_REGION_PHEFI_PHECFW_PHVPD, INT_OPT_ROM_SIZE_2312,
	    0, INT_OPT_ROM_SIZE_2312-1},
    {INT_OPT_ROM_REGION_NONE, 0, 0, 0 }
};

INT_OPT_ROM_REGION  OptionRomTable2322[] = // 1 M x100000
{
    {INT_OPT_ROM_REGION_ALL, INT_OPT_ROM_SIZE_2322,
	    0, INT_OPT_ROM_SIZE_2322-1},
    {INT_OPT_ROM_REGION_PHBIOS_PHFCODE_PHEFI_FW, INT_OPT_ROM_SIZE_2322,
	    0, INT_OPT_ROM_SIZE_2322-1},
    {INT_OPT_ROM_REGION_NONE, 0, 0, 0 }
};

INT_OPT_ROM_REGION  OptionRomTable6322[] = // 1 M x100000
{
    {INT_OPT_ROM_REGION_ALL, INT_OPT_ROM_SIZE_2322,
	    0, INT_OPT_ROM_SIZE_2322-1},
    {INT_OPT_ROM_REGION_PHBIOS_FW, INT_OPT_ROM_SIZE_2322,
	    0, INT_OPT_ROM_SIZE_2322-1},
    {INT_OPT_ROM_REGION_NONE, 0, 0, 0 }
};

INT_OPT_ROM_REGION OptionRomTable2422[] = // 1 M x100000
{
    {INT_OPT_ROM_REGION_ALL, INT_OPT_ROM_SIZE_2422,
	    0, INT_OPT_ROM_SIZE_2422-1},
    {INT_OPT_ROM_REGION_PHBIOS_PHFCODE_PHEFI, 0x40000,
	    0, 0x40000-1 },
	{INT_OPT_ROM_REGION_NPIV_CONFIG_0, 0x4000,
		0x58000, 0x5C000-1},
	{INT_OPT_ROM_REGION_NPIV_CONFIG_1, 0x4000,
		0x5C000, 0x60000-1},
    {INT_OPT_ROM_REGION_FW, 0x80000,
	    0x80000, INT_OPT_ROM_SIZE_2422-1},
    {INT_OPT_ROM_REGION_NONE, 0, 0, 0 }
};

INT_OPT_ROM_REGION OptionRomTable25XX[] = // 1 M + 64K x130000
{
    {INT_OPT_ROM_REGION_ALL, INT_OPT_ROM_SIZE_25XX,
	    0, INT_OPT_ROM_SIZE_25XX-1},
    {INT_OPT_ROM_REGION_PHBIOS_PHFCODE_PHEFI, 0x80000,
	    0, 0x80000-1 },
    {INT_OPT_ROM_REGION_FW, 0x80000,
	    0x80000, INT_OPT_ROM_SIZE_25XX-1},
    {INT_OPT_ROM_REGION_VPD_HBAPARAM, 0x10000,
	    0x120000, 0x130000-1 },
    {INT_OPT_ROM_REGION_FW_DATA, 0x20000,
	    0x100000, 0x120000-1},
	{INT_OPT_ROM_REGION_SERDES, 0x8000,
		0x148000, 0x150000-1},
	{INT_OPT_ROM_REGION_NPIV_CONFIG_0, 0x4000,
		0x170000, 0x174000-1},
	{INT_OPT_ROM_REGION_NPIV_CONFIG_1, 0x4000,
		0x174000, 0x178000-1},
    {INT_OPT_ROM_REGION_NONE, 0, 0, 0 }
};
#if defined(ISP2300)
/*****************************************************************************/
/* Flash Manipulation Routines                                               */
/*****************************************************************************/
static inline uint32_t
flash_data_to_access_addr(uint32_t faddr)
{
	return FARX_ACCESS_FLASH_DATA | faddr;
}

static inline uint32_t
flash_conf_to_access_addr(uint32_t faddr)
{
	return FARX_ACCESS_FLASH_CONF | faddr;
}

static inline uint32_t
nvram_conf_to_access_addr(uint32_t naddr)
{
	return FARX_ACCESS_NVRAM_CONF | naddr;
}

static inline uint32_t
nvram_data_to_access_addr(uint32_t naddr)
{
	return FARX_ACCESS_NVRAM_DATA | naddr;
}

int
qla24xx_write_flash_dword(scsi_qla_host_t *ha, uint32_t addr, uint32_t data)
{
	int rval;
	uint32_t cnt;
	struct device_reg_24xx *reg = (struct device_reg_24xx *)ha->iobase;

	WRT_REG_DWORD(&reg->flash_data, data);
	RD_REG_DWORD(&reg->flash_data);         /* PCI Posting. */
	WRT_REG_DWORD(&reg->flash_addr, addr | FARX_DATA_FLAG);
	/* Wait for Write cycle to complete. */
	rval = QLA2X00_SUCCESS;
	for (cnt = 500000; (RD_REG_DWORD(&reg->flash_addr) & FARX_DATA_FLAG) &&
	    rval == QLA2X00_SUCCESS; cnt--) {
		if (cnt)
			udelay(10);
		else
			rval = QLA2X00_FUNCTION_FAILED;
	}
	return rval;
}

int
qla24xx_read_nvram(scsi_qla_host_t *ha, EXT_IOCTL *pext, int mode)
{
	int     ret = 0;
	char    *ptmp_buf;
	uint32_t transfer_size;
	unsigned long flags;
	uint32_t i;
	uint32_t *dwptr, naddr;
	struct device_reg_24xx *reg;

	DEBUG9(printk("%s: entered.\n",__func__));

	if (qla2x00_get_ioctl_scrap_mem(ha, (void **)&ptmp_buf,
	    ha->nvram_size)) {
		/* not enough memory */
		pext->Status = EXT_STATUS_NO_MEMORY;
		DEBUG9_10(printk("%s(%ld): inst=%ld scrap not big enough. "
		    "size requested=%d.\n",
		    __func__, ha->host_no, ha->instance,
		    ha->nvram_size));
		return (ret);
	}

	transfer_size = ha->nvram_size;
	if (pext->ResponseLen < ha->nvram_size)
		transfer_size = pext->ResponseLen;

	reg = (struct device_reg_24xx *)ha->iobase;

	/* Dump NVRAM. */
	spin_lock_irqsave(&(to_qla_parent(ha))->hardware_lock, flags);

	/* Dword reads to flash. */
	naddr = ha->nvram_base;
	dwptr = (uint32_t *)ptmp_buf;
	if (check_25xx_device_ids(ha)) {
		for (i = 0; i < ha->nvram_size >> 2; i++, naddr++)
			dwptr[i] = cpu_to_le32(qla24xx_read_flash_dword(ha,
			    flash_data_to_access_addr(FA_VPD_NVRAM_ADDR |
			    naddr)));
	} else if (check_24xx_or_54xx_device_ids(ha)) {
		for (i = 0; i < ha->nvram_size >> 2; i++, naddr++)
			dwptr[i] = cpu_to_le32(qla24xx_read_flash_dword(ha,
			    nvram_data_to_access_addr(naddr)));
	}

	spin_unlock_irqrestore(&(to_qla_parent(ha))->hardware_lock, flags);

	DEBUG9(printk("%s(%ld): transfer_size=%d.\n",
	    __func__, ha->host_no, transfer_size));
	DEBUG9(qla2x00_dump_buffer((uint8_t *)ptmp_buf, transfer_size));

	ret = copy_to_user(pext->ResponseAdr, ptmp_buf, transfer_size);
	if (ret) {
		pext->Status = EXT_STATUS_COPY_ERR;
		DEBUG9_10(printk("%s(%ld): inst=%ld ERROR copy rsp buffer.\n",
		    __func__, ha->host_no, ha->instance));
			qla2x00_free_ioctl_scrap_mem(ha);
		return (-EFAULT);
	}

	pext->Status       = EXT_STATUS_OK;
	pext->DetailStatus = EXT_STATUS_OK;

	qla2x00_free_ioctl_scrap_mem(ha);

	DEBUG9(printk("%s: exiting.\n",__func__));

	return (ret);
}
#endif

int
qla2x00_read_nvram(scsi_qla_host_t *ha, EXT_IOCTL *pext, int mode)
{
	char		*ptmp_buf;
	int		ret = 0;

#if defined(ISP2300)
	device_reg_t	*reg = ha->iobase;
	uint16_t	data;
	uint32_t	wait_cnt;
#endif
#if defined(ISP2100)
	uint32_t	nvram_size = sizeof(nvram21_t);
#else
	uint32_t	nvram_size = sizeof(nvram22_t);
#endif
	uint16_t	cnt, base;
 	uint16_t	*wptr;
	uint32_t	transfer_size;

	DEBUG9(printk("qla2x00_read_nvram: entered.\n"));

#if defined(ISP2300)
	if (check_24xx_or_54xx_device_ids(ha) || check_25xx_device_ids(ha))
		return qla24xx_read_nvram(ha, pext, mode);
#endif
	if (qla2x00_get_ioctl_scrap_mem(ha, (void **)&ptmp_buf,
	    nvram_size)) {
		/* not enough memory */
		pext->Status = EXT_STATUS_NO_MEMORY;
		DEBUG9_10(printk("%s(%ld): inst=%ld scrap not big enough. "
		    "size requested=%d.\n",
		    __func__, ha->host_no, ha->instance,
		    nvram_size));
		return (ret);
	}

	if (pext->ResponseLen < nvram_size)
		transfer_size = pext->ResponseLen / 2;
	else
		transfer_size = nvram_size / 2;

	/* Dump NVRAM. */
#if defined(ISP2300)
	if (ha->device_id == QLA2312_DEVICE_ID ||
	    ha->device_id == QLA2322_DEVICE_ID ||
	    ha->device_id == QLA6312_DEVICE_ID ||
	    ha->device_id == QLA6322_DEVICE_ID) { 	    
		data = RD_REG_WORD(&reg->ctrl_status);
		if ((data >> 14) == 1)
			base = 0x80;
		else
			base = 0;

		wait_cnt = 20000;
		data = RD_REG_WORD(&reg->nvram);
		while (data & NV_BUSY) {
			wait_cnt--;
			if (wait_cnt == 0) { 
				DEBUG9_10(printk("%s(%ld) NVRAM Busy\n",
						__func__,ha->host_no));
				return ret;
			}
			UDELAY(100);
			data = RD_REG_WORD(&reg->nvram);
		}

		/* Lock resource */
		WRT_REG_WORD(&reg->host_semaphore, 0x1);
		UDELAY(5);

		data = RD_REG_WORD(&reg->host_semaphore);
		while ((data & BIT_0) == 0) {
			/* Lock failed */
			UDELAY(100);
			WRT_REG_WORD(&reg->host_semaphore, 0x1);
			UDELAY(5);
			data = RD_REG_WORD(&reg->host_semaphore);
		}
	} else {
		base = 0;
	}
#else
	base = 0;
#endif

 	wptr = (uint16_t *)ptmp_buf;
 	for (cnt = 0; cnt < transfer_size; cnt++) {
		*wptr = cpu_to_le16(qla2x00_get_nvram_word(ha, (cnt+base)));
		wptr++;
 	}

#if defined(ISP2300) 
	if (ha->device_id == QLA2312_DEVICE_ID ||
	    ha->device_id == QLA2322_DEVICE_ID ||
	    ha->device_id == QLA6312_DEVICE_ID ||
	    ha->device_id == QLA6322_DEVICE_ID) { 	    
		/* Unlock resource */
		WRT_REG_WORD(&reg->host_semaphore, 0);
		PCI_POSTING(&reg->host_semaphore);
	}
#endif

	ret = copy_to_user(pext->ResponseAdr, ptmp_buf, transfer_size * 2);
	if (ret) {
		pext->Status = EXT_STATUS_COPY_ERR;
		DEBUG9_10(printk("%s(%ld): inst=%ld ERROR copy rsp buffer.\n",
		    __func__, ha->host_no, ha->instance));
		qla2x00_free_ioctl_scrap_mem(ha);
		return (-EFAULT);
	}

	pext->Status       = EXT_STATUS_OK;
	pext->DetailStatus = EXT_STATUS_OK;

	qla2x00_free_ioctl_scrap_mem(ha);

	DEBUG9(printk("qla2x00_read_nvram: exiting.\n"));
	return (ret);
}

#if defined(ISP2300)

/*
 * qla24xx_update_nvram
 *	Write data to NVRAM.
 *
 * Input:
 *	ha = adapter block pointer.
 *	pext = pointer to driver internal IOCTL structure.
 *
 * Returns:
 *
 * Context:
 *	Kernel context.
 */
int
qla24xx_update_nvram(scsi_qla_host_t *ha, EXT_IOCTL *pext, int mode)
{
	uint8_t cnt;
	uint8_t *usr_tmp, *kernel_tmp, *porg_nv;
	struct nvram_24xx *pnew_nv;
	uint32_t transfer_size;
	int ret = 0;
	unsigned long flags;
	uint32_t *iter;
	uint32_t chksum;
	uint32_t i;
	uint32_t *dwptr, naddr;
	struct device_reg_24xx *reg = (struct device_reg_24xx *)ha->iobase;


	DEBUG9(printk("qla2x00_update_nvram: entered.\n"));

	if (pext->RequestLen < ha->nvram_size)
		transfer_size = pext->RequestLen;
	else
		transfer_size = ha->nvram_size;

	if (qla2x00_get_ioctl_scrap_mem(ha, (void **)&pnew_nv,
	    ha->nvram_size)) {
		/* not enough memory */
		pext->Status = EXT_STATUS_NO_MEMORY;
		DEBUG9_10(printk("%s(%ld): inst=%ld scrap not big enough. "
		    "size requested=%d.\n",
		    __func__, ha->host_no, ha->instance, ha->nvram_size));
		return (ret);
	}

	if (check_25xx_device_ids(ha)){
		if (qla2x00_get_ioctl_scrap_mem(ha, (void **)&porg_nv,
		    VPD_HBAPARAM_SIZE_25XX)) {
			/* not enough memory */
			pext->Status = EXT_STATUS_NO_MEMORY;
			DEBUG9_10(printk("%s(%ld): inst=%ld scrap not big enough. "
    			    "size requested=%d.\n",
    			    __func__, ha->host_no, ha->instance,
    			    VPD_HBAPARAM_SIZE_25XX));
			return (ret);
		}
	}

	/* Read from user buffer */
	kernel_tmp = (uint8_t *)pnew_nv;
	usr_tmp = (uint8_t *)pext->RequestAdr;

	ret = copy_from_user(kernel_tmp, usr_tmp, transfer_size);
	if (ret) {
		pext->Status = EXT_STATUS_COPY_ERR;
		DEBUG9_10(printk(
		    "qla2x00_update_nvram: ERROR in buffer copy READ. "
		    "RequestAdr=%p\n", pext->RequestAdr));
		qla2x00_free_ioctl_scrap_mem(ha);
		return (-EFAULT);
	}

	/* Checksum NVRAM. */

	iter = (uint32_t *)pnew_nv;
	chksum = 0;
	for (cnt = 0; cnt < ((ha->nvram_size >> 2) - 1); cnt++)
		chksum += le32_to_cpu(*iter++);
	chksum = ~chksum + 1;
	*iter = cpu_to_le32(chksum);

	ret = QLA2X00_SUCCESS;
	/* Write NVRAM. */
	spin_lock_irqsave(&(to_qla_parent(ha))->hardware_lock, flags);

	if (check_25xx_device_ids(ha)){
		qla24xx_read_nvram_data(ha, (uint32_t *)porg_nv, FA_NVRAM_VPD0_ADDR,
		    VPD_HBAPARAM_SIZE_25XX);

		memcpy((porg_nv + (ha->nvram_base * 4)), pnew_nv,
		    transfer_size);

		qla24xx_write_nvram_data(ha, porg_nv, FA_NVRAM_VPD0_ADDR,
		    VPD_HBAPARAM_SIZE_25XX);
	} else {
		/* Enable flash write. */
		WRT_REG_DWORD(&reg->ctrl_status,
		    RD_REG_DWORD(&reg->ctrl_status) | CSRX_FLASH_ENABLE);
		RD_REG_DWORD(&reg->ctrl_status);        /* PCI Posting. */

		/* Disable NVRAM write-protection. */
		qla24xx_write_flash_dword(ha, nvram_conf_to_access_addr(0x101), 0);
		qla24xx_write_flash_dword(ha, nvram_conf_to_access_addr(0x101), 0);

		/* Dword writes to flash. */
		dwptr = (uint32_t *)pnew_nv;
		naddr = ha->nvram_base;
		for (i = 0; i < transfer_size >> 2; i++, naddr++, dwptr++) {
			if (qla24xx_write_flash_dword(ha,
			    nvram_data_to_access_addr(naddr), cpu_to_le32(*dwptr)) !=
			    QLA2X00_SUCCESS) {
				DEBUG9(printk("%s(%ld) Unable to program "
				    "nvram address=%x data=%x.\n", __func__,
				    ha->host_no, naddr, *dwptr));
				ret = QLA2X00_FUNCTION_FAILED;
				break;
			}
		}

		/* Enable NVRAM write-protection. */
		qla24xx_write_flash_dword(ha, nvram_conf_to_access_addr(0x101),
		    0x8c);

		/* Disable flash write. */
		WRT_REG_DWORD(&reg->ctrl_status,
		    RD_REG_DWORD(&reg->ctrl_status) & ~CSRX_FLASH_ENABLE);
		RD_REG_DWORD(&reg->ctrl_status);        /* PCI Posting. */
	}

	spin_unlock_irqrestore(&(to_qla_parent(ha))->hardware_lock, flags);

	pext->Status       = EXT_STATUS_OK;
	pext->DetailStatus = EXT_STATUS_OK;

	qla2x00_free_ioctl_scrap_mem(ha);

	/* Schedule DPC to restart the RISC */
	set_bit(ISP_ABORT_NEEDED, &ha->dpc_flags);
	up(ha->dpc_wait);

	if (qla2x00_wait_for_hba_online(ha) != QLA2X00_SUCCESS) {
		pext->Status = EXT_STATUS_ERR;
	}

	DEBUG9(printk("qla2x00_update_nvram: exiting.\n"));
	return ret;
}
#endif

static void
qla2x00_set_nvram_protection(scsi_qla_host_t *ha, int stat)
{
	device_reg_t *reg;
	uint32_t word;
	uint32_t	wait_cnt;

	reg = ha->iobase;

	if (stat != QLA2X00_SUCCESS)
		return;

	/* Set NVRAM write protection. */
	/* Write enable. */
	qla2x00_nv_write(ha, NV_DATA_OUT);
	qla2x00_nv_write(ha, 0);
	qla2x00_nv_write(ha, 0);
	for (word = 0; word < 8; word++)
		qla2x00_nv_write(ha, NV_DATA_OUT);

	qla2x00_nv_deselect(ha);

	/* Enable protection register. */
	qla2x00_nv_write(ha, NV_PR_ENABLE | NV_DATA_OUT);
	qla2x00_nv_write(ha, NV_PR_ENABLE);
	qla2x00_nv_write(ha, NV_PR_ENABLE);
	for (word = 0; word < 8; word++)
		qla2x00_nv_write(ha, NV_DATA_OUT | NV_PR_ENABLE);

	qla2x00_nv_deselect(ha);

	/* Enable protection register. */
	qla2x00_nv_write(ha, NV_PR_ENABLE | NV_DATA_OUT);
	qla2x00_nv_write(ha, NV_PR_ENABLE);
	qla2x00_nv_write(ha, NV_PR_ENABLE | NV_DATA_OUT);
	for (word = 0; word < 8; word++)
		qla2x00_nv_write(ha, NV_PR_ENABLE);

	qla2x00_nv_deselect(ha);

	/* Wait for NVRAM to become ready. */
	wait_cnt = 20000;
	WRT_REG_WORD(&reg->nvram, NV_SELECT);
	PCI_POSTING(&reg->nvram);
	do {
		wait_cnt--;
		if (wait_cnt == 0) { 
			DEBUG9_10(printk(
			    "%s(%ld) Wait for NVRAM Ready Over\n",
			    __func__,ha->host_no));
			return;
		}
		NVRAM_DELAY();
		word = RD_REG_WORD(&reg->nvram);
	} while ((word & NV_DATA_IN) == 0);
}

/*
 * qla2x00_update_nvram
 *	Write data to NVRAM.
 *
 * Input:
 *	ha = adapter block pointer.
 *	pext = pointer to driver internal IOCTL structure.
 *
 * Returns:
 *
 * Context:
 *	Kernel context.
 */
int
qla2x00_update_nvram(scsi_qla_host_t *ha, EXT_IOCTL *pext, int mode)
{
#if defined(ISP2300) 
	device_reg_t	*reg = ha->iobase;
	uint32_t	word;
	uint16_t	wprot, wprot_old;
	uint32_t	wait_cnt;
#endif
#if defined(ISP2100)
	nvram21_t	*pnew_nv;
	uint32_t	nvram_size = sizeof(nvram21_t);
#else
	nvram22_t	*pnew_nv;
	uint32_t	nvram_size = sizeof(nvram22_t);
#endif
	uint8_t		chksum = 0;
	uint8_t		*usr_tmp, *kernel_tmp;
	uint16_t	i, cnt, base;
	uint16_t	data;
	uint16_t	*wptr;
	uint32_t	transfer_size;
	int		ret = 0;
	uint8_t		stat = 0;

	// FIXME: Endianess?
	DEBUG9(printk("qla2x00_update_nvram: entered.\n"));

#if defined(ISP2300)
	if (check_24xx_or_54xx_device_ids(ha) || check_25xx_device_ids(ha))
		return qla24xx_update_nvram(ha, pext, mode);
#endif

	if (pext->RequestLen < nvram_size)
		transfer_size = pext->RequestLen;
	else
		transfer_size = nvram_size;

	if (qla2x00_get_ioctl_scrap_mem(ha, (void **)&pnew_nv,
	    nvram_size)) {
		/* not enough memory */
		pext->Status = EXT_STATUS_NO_MEMORY;
		DEBUG9_10(printk("%s(%ld): inst=%ld scrap not big enough. "
		    "size requested=%d.\n",
		    __func__, ha->host_no, ha->instance,
		    nvram_size));
		return (ret);
	}

	/* Read from user buffer */
	kernel_tmp = (uint8_t *)pnew_nv;
	usr_tmp = (uint8_t *)pext->RequestAdr;

	ret = copy_from_user(kernel_tmp, usr_tmp, transfer_size);
	if (ret) {
		DEBUG9_10(printk(
		    "qla2x00_update_nvram: ERROR in buffer copy READ. "
		    "RequestAdr=%p\n", pext->RequestAdr));
		pext->Status = EXT_STATUS_COPY_ERR;
		qla2x00_free_ioctl_scrap_mem(ha);
		return (-EFAULT);
	}

	kernel_tmp = (uint8_t *)pnew_nv;

	/* we need to checksum the nvram */
	for (i = 0; i < nvram_size - 1; i++) {
		chksum += *kernel_tmp;
		kernel_tmp++;
	}

	chksum = ~chksum + 1;

	*kernel_tmp = chksum;

	/* Write to NVRAM */
#if defined(ISP2300)
	if (ha->device_id == QLA2312_DEVICE_ID ||
	    ha->device_id == QLA2322_DEVICE_ID ||
	    ha->device_id == QLA6312_DEVICE_ID ||
	    ha->device_id == QLA6322_DEVICE_ID) { 	    
		data = RD_REG_WORD(&reg->ctrl_status);
		if ((data >> 14) == 1)
			base = 0x80;
		else
			base = 0;

		wait_cnt = 20000;
		data = RD_REG_WORD(&reg->nvram);
		while (data & NV_BUSY) {
			wait_cnt--;
			if (wait_cnt == 0) { 
				DEBUG9_10(printk("%s(%ld) NVRAM Busy\n",
						__func__,ha->host_no));
				return ret;
			}
			UDELAY(100);
			data = RD_REG_WORD(&reg->nvram);
		}

		/* Lock resource */
		WRT_REG_WORD(&reg->host_semaphore, 0x1);
		UDELAY(5);

		wait_cnt = 20000;
		data = RD_REG_WORD(&reg->host_semaphore);
		while ((data & BIT_0) == 0) {
			wait_cnt--;
			if (wait_cnt == 0) { 
				DEBUG9_10(printk(
				    "%s(%ld) Unable to set sem_lock\n",
				    __func__,ha->host_no));
				return ret;
			}
			/* Lock failed */
			UDELAY(100);
			WRT_REG_WORD(&reg->host_semaphore, 0x1);
			UDELAY(5);
			data = RD_REG_WORD(&reg->host_semaphore);
		}
	} else {
		base = 0;
	}

	/* Clear write protection, if necessary. */
	wprot_old = cpu_to_le16(qla2x00_get_nvram_word(ha, base));
	stat = qla2x00_write_nvram_word(ha, base, 
					__constant_cpu_to_le16(0x1234));
	wprot = cpu_to_le16(qla2x00_get_nvram_word(ha, base));
	if (stat != QLA2X00_SUCCESS || wprot != 0x1234) {
		/* write enable */
		qla2x00_nv_write(ha, NV_DATA_OUT);
		qla2x00_nv_write(ha, 0);
		qla2x00_nv_write(ha, 0);

		for (word = 0; word < 8; word++) {
			qla2x00_nv_write(ha, NV_DATA_OUT);
		}

		qla2x00_nv_deselect(ha);

		/* enable protection register */
		qla2x00_nv_write(ha, NV_PR_ENABLE | NV_DATA_OUT);
		qla2x00_nv_write(ha, NV_PR_ENABLE);
		qla2x00_nv_write(ha, NV_PR_ENABLE);

		for (word = 0; word < 8; word++) {
			qla2x00_nv_write(ha, NV_DATA_OUT | NV_PR_ENABLE);
		}

		qla2x00_nv_deselect(ha);

		/* clear protection register (ffff is cleared) */
		qla2x00_nv_write(ha, NV_PR_ENABLE | NV_DATA_OUT);
		qla2x00_nv_write(ha, NV_PR_ENABLE | NV_DATA_OUT);
		qla2x00_nv_write(ha, NV_PR_ENABLE | NV_DATA_OUT);

		for (word = 0; word < 8; word++) {
			qla2x00_nv_write(ha, NV_DATA_OUT | NV_PR_ENABLE);
		}
		qla2x00_nv_deselect(ha);

		/* Wait for NVRAM to become ready */
		wait_cnt = 20000;
		WRT_REG_WORD(&reg->nvram, NV_SELECT);
		PCI_POSTING(&reg->nvram);
		do {
			wait_cnt--;
			if (wait_cnt == 0) { 
				DEBUG9_10(printk(
				    "%s(%ld) Wait for NVRAM Ready Over\n",
				    __func__,ha->host_no));
				return ret;
			}
			NVRAM_DELAY();
			word = RD_REG_WORD(&reg->nvram);
		} while ((word & NV_DATA_IN) == 0);
	} else
		qla2x00_write_nvram_word(ha, base, wprot_old);
#else
	base = 0;
#endif

	wptr = (uint16_t *)pnew_nv;
	for (cnt = 0; cnt < transfer_size / 2; cnt++) {
		data = cpu_to_le16(*wptr++);
		qla2x00_write_nvram_word(ha, (cnt+base), data);
	}
	/* Enable NVRAM write-protection if cleared. */
	qla2x00_set_nvram_protection(ha, stat);

#if defined(ISP2300)
	if (ha->device_id == QLA2312_DEVICE_ID ||
	    ha->device_id == QLA2322_DEVICE_ID ||
	    ha->device_id == QLA6312_DEVICE_ID ||
	    ha->device_id == QLA6322_DEVICE_ID) { 	    
		/* Unlock resource */
		WRT_REG_WORD(&reg->host_semaphore, 0);
		PCI_POSTING(&reg->host_semaphore);
	}
#endif

	pext->Status       = EXT_STATUS_OK;
	pext->DetailStatus = EXT_STATUS_OK;

	qla2x00_free_ioctl_scrap_mem(ha);

	DEBUG9(printk("qla2x00_update_nvram: exiting.\n"));

	/* Schedule DPC to restart the RISC */
	set_bit(ISP_ABORT_NEEDED, &ha->dpc_flags);
	up(ha->dpc_wait);

	ret = qla2x00_wait_for_hba_online(ha);
	return ret;
}

int
qla2x00_write_nvram_word(scsi_qla_host_t *ha, uint8_t addr, uint16_t data)
{
	int count;
	uint16_t word;
	uint32_t nv_cmd;
	device_reg_t *reg = ha->iobase;
	uint32_t	wait_cnt;

	DEBUG9(printk("%s entered\n",__func__));

	qla2x00_nv_write(ha, NV_DATA_OUT);
	qla2x00_nv_write(ha, 0);
	qla2x00_nv_write(ha, 0);

	for (word = 0; word < 8; word++)
		qla2x00_nv_write(ha, NV_DATA_OUT);

	qla2x00_nv_deselect(ha);

	/* Write data */
	nv_cmd = (addr << 16) | NV_WRITE_OP;
	nv_cmd |= data;
	nv_cmd <<= 5;
	for (count = 0; count < 27; count++) {
		if (nv_cmd & BIT_31)
			qla2x00_nv_write(ha, NV_DATA_OUT);
		else
			qla2x00_nv_write(ha, 0);

		nv_cmd <<= 1;
	}

	qla2x00_nv_deselect(ha);

	/* Wait for NVRAM to become ready */
	wait_cnt = 20000;
	WRT_REG_WORD(&reg->nvram, NV_SELECT);
	PCI_POSTING(&reg->nvram);
	do {
		wait_cnt--;
		if (wait_cnt == 0) { 
			DEBUG9_10(printk("%s(%ld) Wait for NVRAM Ready Over\n",
					__func__,ha->host_no));
			return 1;
		}
		NVRAM_DELAY();
		word = RD_REG_WORD(&reg->nvram);
	} while ((word & NV_DATA_IN) == 0);

	qla2x00_nv_deselect(ha);

	/* Disable writes */
	qla2x00_nv_write(ha, NV_DATA_OUT);
	for (count = 0; count < 10; count++)
		qla2x00_nv_write(ha, 0);

	qla2x00_nv_deselect(ha);

	DEBUG9(printk("qla2x00_write_nvram_word: exiting.\n"));

	return 0;
}

int
qla2x00_send_loopback(scsi_qla_host_t *ha, EXT_IOCTL *pext, int mode)
{

#define MAX_LOOPBACK_BUFFER_SIZE		(128 * 1024)

	int		status;
	int 		rval = 0;
	uint16_t	ret_mb[MAILBOX_REGISTER_COUNT];
	INT_LOOPBACK_REQ req;
	INT_LOOPBACK_RSP rsp;

	DEBUG9(printk("qla2x00_send_loopback: entered.\n"));


	if (pext->RequestLen != sizeof(INT_LOOPBACK_REQ)) {
		pext->Status = EXT_STATUS_INVALID_PARAM;
		DEBUG9_10(printk(
		    "qla2x00_send_loopback: invalid RequestLen =%d.\n",
		    pext->RequestLen));
		return 0;
	}

	if (pext->ResponseLen != sizeof(INT_LOOPBACK_RSP)) {
		pext->Status = EXT_STATUS_INVALID_PARAM;
		DEBUG9_10(printk(
		    "qla2x00_send_loopback: invalid ResponseLen =%d.\n",
		    pext->ResponseLen));
		return 0;
	}

	status = copy_from_user(&req, pext->RequestAdr, pext->RequestLen);
	if (status) {
		pext->Status = EXT_STATUS_COPY_ERR;
		DEBUG9_10(printk("qla2x00_send_loopback: ERROR copy read of "
		    "request buffer.\n"));
		return (-EFAULT);
	}

	status = copy_from_user(&rsp, pext->ResponseAdr, pext->ResponseLen);
	if (status) {
		pext->Status = EXT_STATUS_COPY_ERR;
		DEBUG9_10(printk("qla2x00_send_loopback: ERROR verify read of "
		    "response buffer.\n"));
		return (-EFAULT);
	}

	if (req.TransferCount > MAX_LOOPBACK_BUFFER_SIZE ||
	    req.TransferCount > req.BufferLength ||
	    req.TransferCount > rsp.BufferLength) {

		/* Buffer lengths not large enough. */
		pext->Status = EXT_STATUS_INVALID_PARAM;

		DEBUG9_10(printk(
		    "qla2x00_send_loopback: invalid TransferCount =%d. "
		    "req BufferLength =%d rspBufferLength =%d.\n",
		    req.TransferCount, req.BufferLength, rsp.BufferLength));

		return 0;
	}

	if (req.TransferCount > ha->ioctl_mem_size) {
		if (qla2x00_get_new_ioctl_dma_mem(ha, req.TransferCount) !=
		    QLA2X00_SUCCESS) {
			DEBUG9_10(printk("%s(%ld): inst=%ld ERROR cannot alloc "
			    "requested DMA buffer size %x.\n",
			    __func__, ha->host_no, ha->instance,
			    req.TransferCount));

			pext->Status = EXT_STATUS_NO_MEMORY;
			return rval;
		}
	}

	status = copy_from_user(ha->ioctl_mem, req.BufferAddress,
	    req.TransferCount);
	if (status) {
		pext->Status = EXT_STATUS_COPY_ERR;
		DEBUG9_10(printk("qla2x00_send_loopback: ERROR copy read of "
		    "user loopback data buffer.\n"));
		return (-EFAULT);
	}

	DEBUG9(printk("qla2x00_send_loopback: req -- bufadr=%p, buflen=%x, "
	    "xfrcnt=%x, rsp -- bufadr=%p, buflen=%x.\n",
	    req.BufferAddress, req.BufferLength, req.TransferCount,
	    rsp.BufferAddress, rsp.BufferLength));

	/*
	 * AV - the caller of this IOCTL expects the FW to handle
	 * a loopdown situation and return a good status for the
	 * call function and a LOOPDOWN status for the test operations
	 */
	/*if (ha->loop_state != LOOP_READY || */
	if (
	    (test_bit(CFG_ACTIVE, &ha->cfg_flags)) ||
	    (test_bit(ABORT_ISP_ACTIVE, &ha->dpc_flags)) ||
	    ABORTS_ACTIVE || ha->dpc_active) {

		pext->Status = EXT_STATUS_BUSY;
		DEBUG9_10(printk("qla2x00_send_loopback(%ld): "
		    "loop not ready.\n", ha->host_no));
		return 0;
	}

	if (ha->current_topology == ISP_CFG_F) {
#if defined(ISP2300)
		status = qla2x00_echo_test(ha, &req, ret_mb);
#else
		pext->Status = EXT_STATUS_INVALID_REQUEST ;
		DEBUG9_10(printk("qla2x00_send_loopback: ERROR "
		    "command only supported for QLA23xx.\n"));
		return 0 ;
#endif
	} else {
		status = qla2x00_loopback_test(ha, &req, ret_mb);
	}

	if (status) {
		if (status == QL_STATUS_TIMEOUT) {
			pext->Status = EXT_STATUS_BUSY;
			DEBUG9_10(printk("qla2x00_send_loopback: ERROR "
			    "command timed out.\n"));
			return 0;
		} else {
			/* EMPTY. Just proceed to copy back mailbox reg
			 * values for users to interpret.
			 */
			DEBUG10(printk("qla2x00_send_loopback: ERROR "
			    "loopback command failed 0x%x.\n", ret_mb[0]));
		}
	}

	DEBUG9(printk("qla2x00_send_loopback: loopback mbx cmd ok. "
	    "copying data.\n"));

	/* put loopback return data in user buffer */
	status = copy_to_user(rsp.BufferAddress, ha->ioctl_mem,
	    req.TransferCount);
	if (status) {
		pext->Status = EXT_STATUS_COPY_ERR;
		DEBUG9_10(printk("qla2x00_send_loopback: ERROR verify "
		    "write of return data buffer.\n"));
		return (-EFAULT);
	}

	rsp.CompletionStatus = ret_mb[0];

	if (ha->current_topology == ISP_CFG_F) {
		rsp.CommandSent = INT_DEF_LB_ECHO_CMD;
	} else {
		if (rsp.CompletionStatus == INT_DEF_LB_COMPLETE ||
		    rsp.CompletionStatus == INT_DEF_LB_CMD_ERROR) {
			rsp.CrcErrorCount = ret_mb[1];
			rsp.DisparityErrorCount = ret_mb[2];
			rsp.FrameLengthErrorCount = ret_mb[3];
			rsp.IterationCountLastError =
			    (ret_mb[19] << 16) | ret_mb[18];
		}
	}

	status = copy_to_user(pext->ResponseAdr, &rsp, pext->ResponseLen);
	if (status) {
		pext->Status = EXT_STATUS_COPY_ERR;
		DEBUG9_10(printk("qla2x00_send_loopback: ERROR copy "
		    "write of response buffer.\n"));
		return (-EFAULT);
	}

	pext->Status       = EXT_STATUS_OK;
	pext->DetailStatus = EXT_STATUS_OK;

	DEBUG9(printk("qla2x00_send_loopback: exiting.\n"));

	return 0;
}

int
qla2x00_read_option_rom(scsi_qla_host_t *ha, EXT_IOCTL *pext, int mode)
{
	int	rval = 0;

	DEBUG9(printk("%s: entered.\n", __func__));

	if (pext->SubCode)
		return qla2x00_read_option_rom_ext(ha, pext, mode);

	/* These interfaces are not valid for 24xx and 25xx chips. */
	if (check_24xx_or_54xx_device_ids(ha) || check_25xx_device_ids(ha)) {
		pext->Status = EXT_STATUS_INVALID_REQUEST;
		return (rval);
        }

	if (pext->ResponseLen < FLASH_IMAGE_SIZE) {
		pext->Status = EXT_STATUS_BUFFER_TOO_SMALL;
		return (rval);
	}

	/* Dump FLASH. */
 	qla2x00_update_or_read_flash(ha, (uint8_t *)pext->ResponseAdr, 0, 
	    FLASH_IMAGE_SIZE, QLA2X00_READ); 

	pext->Status = EXT_STATUS_OK;
	pext->DetailStatus = EXT_STATUS_OK;

	DEBUG9(printk("%s: exiting.\n", __func__));

	return (rval);
}

int
qla2x00_read_option_rom_ext(scsi_qla_host_t *ha, EXT_IOCTL *pext, int mode)
{
	int		iter, found;
	int		rval = 0;
	uint8_t		*image_ptr;
	uint32_t	saddr, length;
	int		count, residual, cnt;

	DEBUG9(printk("%s: entered.\n", __func__));

	found = 0;
	saddr = length = 0;

	/* Retrieve region or raw starting address. */
	if (pext->SubCode == 0xFFFF) {
		saddr = pext->Reserved1;
		length = pext->ResponseLen;
#ifdef __VMKERNEL_MODULE__
		/* Limitation of copy_to_user */
		count = length / FLASH_CHUNK_SZ;
		residual = length % FLASH_CHUNK_SZ;	
#endif
		found++;
	} else {
		INT_OPT_ROM_REGION *OptionRomTable = NULL;
		unsigned long  OptionRomTableSize;

		/* Pick the right OptionRom table based on device id */
		qla2x00_get_option_rom_table(ha, &OptionRomTable, 
		    &OptionRomTableSize);
		for (iter = 0; OptionRomTable != NULL && iter <
		    (OptionRomTableSize / sizeof(INT_OPT_ROM_REGION));
		    iter++) {
			if (OptionRomTable[iter].Region == pext->SubCode) {
				saddr = OptionRomTable[iter].Beg;
				length = OptionRomTable[iter].Size;
#ifdef __VMKERNEL_MODULE__
				/* Limitation of copy_to_user */
				count = length / FLASH_CHUNK_SZ;
				residual = length % FLASH_CHUNK_SZ;	
#endif
				found++;
				break;
			}
		}
	}
	if (!found) {
		pext->Status = EXT_STATUS_ERR;
		return (rval);
	}
	if (pext->ResponseLen < length) {
		pext->Status = EXT_STATUS_BUFFER_TOO_SMALL;
		return (rval);
	}

	if (check_24xx_or_54xx_device_ids(ha) || check_25xx_device_ids(ha)) {
#ifdef __VMKERNEL_MODULE__
		image_ptr = KMEM_ZALLOC(length, 69);
#else
		image_ptr = vmalloc(length);
#endif
		if (image_ptr == NULL) {
			pext->Status = EXT_STATUS_NO_MEMORY;
			printk(KERN_WARNING
			    "%s: ERROR in flash allocation.\n", __func__);
			return (-ENOMEM);
		}
	} else {
		image_ptr = (uint8_t *)pext->ResponseAdr;
	}

	/* Dump FLASH. */
 	qla2x00_update_or_read_flash(ha, image_ptr, saddr, length,
	    QLA2X00_READ); 

	if (check_24xx_or_54xx_device_ids(ha) || check_25xx_device_ids(ha)) {
#ifdef __VMKERNEL_MODULE__
	for (cnt = 0; cnt <= count ; cnt++) {
		if (cnt < count) {
			rval = copy_to_user(pext->ResponseAdr + ((FLASH_CHUNK_SZ) * cnt), 
				image_ptr + ((FLASH_CHUNK_SZ) * cnt), FLASH_CHUNK_SZ);
		} else if (residual != 0) {
			rval = copy_to_user(pext->ResponseAdr + ((FLASH_CHUNK_SZ) * count), 
				image_ptr + ((FLASH_CHUNK_SZ) * count), residual);
		}
#else
	rval = copy_to_user(pext->ResponseAdr, image_ptr, length);
#endif
		if (rval) {
			pext->Status = EXT_STATUS_COPY_ERR;
			DEBUG9_10(printk(
			"%s(%ld): inst=%ld ERROR copy rsp buffer.\n",
			__func__, ha->host_no, ha->instance));
#ifdef __VMKERNEL_MODULE__
				KMEM_FREE(image_ptr, length);
#else
				vfree(image_ptr);
#endif	
			return (-EFAULT);
		}
#ifdef __VMKERNEL_MODULE__
		}
		KMEM_FREE(image_ptr, length);
#else
		vfree(image_ptr);
#endif
	}

	pext->Status = EXT_STATUS_OK;
	pext->DetailStatus = EXT_STATUS_OK;
	DEBUG9(printk("%s: exiting.\n", __func__));

	return rval;
}

int
qla2x00_update_option_rom(scsi_qla_host_t *ha, EXT_IOCTL *pext, int mode)
{
	int		ret = 0;
	uint8_t		*usr_tmp;
	uint8_t		*kern_tmp;
	uint16_t	status;
	uint32_t	length;

	DEBUG9(printk("%s: entered.\n", __func__));

	length = FLASH_IMAGE_SIZE;

	if (pext->SubCode)
		return qla2x00_update_option_rom_ext(ha, pext, mode);

	/* These interfaces are not valid for 24xx and 25xx chips. */
	if (check_24xx_or_54xx_device_ids(ha) || check_25xx_device_ids(ha)) {
                pext->Status = EXT_STATUS_INVALID_REQUEST;
                return (ret);
        }

	if (pext->RequestLen != FLASH_IMAGE_SIZE) {
		pext->Status = EXT_STATUS_INVALID_PARAM;
		return (ret);
	}

	pext->Status = EXT_STATUS_OK;
	pext->DetailStatus = EXT_STATUS_OK;

	/* Read from user buffer */
	usr_tmp = (uint8_t *)pext->RequestAdr;

#ifdef __VMKERNEL_MODULE__
	kern_tmp = (uint8_t *)KMEM_ZALLOC(length, 42);
#else
	kern_tmp = (uint8_t *)vmalloc(FLASH_IMAGE_SIZE);
#endif
	if (kern_tmp == NULL) {
		pext->Status = EXT_STATUS_NO_MEMORY;
		printk(KERN_WARNING
		    "%s: ERROR in flash allocation.\n", __func__);
		return (-ENOMEM);
	}

	ret = copy_from_user(kern_tmp, usr_tmp, FLASH_IMAGE_SIZE);
	if (ret) {
		KMEM_FREE(kern_tmp, FLASH_IMAGE_SIZE);
		pext->Status = EXT_STATUS_COPY_ERR;
		DEBUG9_10(printk("%s: ERROR in buffer copy READ. "
		    "RequestAdr=%p\n",
		    __func__, pext->RequestAdr));
#ifdef __VMKERNEL_MODULE__
			KMEM_FREE(kern_tmp, length);
#else
			vfree(kern_tmp);
#endif
		return (-EFAULT);
	}

	status = qla2x00_update_or_read_flash(ha, kern_tmp, 0, 
	    FLASH_IMAGE_SIZE, QLA2X00_WRITE); 

#ifdef __VMKERNEL_MODULE__
		KMEM_FREE(kern_tmp, length);
#else
		vfree(kern_tmp);
#endif

	if (status) {
		pext->Status = EXT_STATUS_COPY_ERR;
		ret = (-EFAULT);
		DEBUG9_10(printk("%s: ERROR updating flash.\n", __func__));
	}
	DEBUG9(printk("%s: exiting.\n", __func__));
	return (ret);
}

int
qla2x00_update_option_rom_ext(scsi_qla_host_t *ha, EXT_IOCTL *pext, int mode)
{
	int		iter, found;
	int		ret;
	uint8_t		*usr_tmp;
	uint8_t		*kern_tmp;
	uint16_t	status;
	uint32_t	saddr, length;
	int		count, residual, cnt;

	DEBUG9(printk("%s: entered.\n", __func__));

	found = 0;
	saddr = length = 0;
	/* Retrieve region or raw starting address. */
	if (pext->SubCode == 0xFFFF) {
		saddr = pext->Reserved1;
		length = pext->RequestLen;
		found++;
	} else {
		INT_OPT_ROM_REGION *OptionRomTable = NULL;
		unsigned long  OptionRomTableSize;

		/* Pick the right OptionRom table based on device id */
		qla2x00_get_option_rom_table(ha, &OptionRomTable, 
		    &OptionRomTableSize);
		for (iter = 0; OptionRomTable != NULL && iter <
		    (OptionRomTableSize / sizeof(INT_OPT_ROM_REGION));
		    iter++) {
			if (OptionRomTable[iter].Region == pext->SubCode) {
				saddr = OptionRomTable[iter].Beg;
				length = OptionRomTable[iter].Size;
#ifdef __VMKERNEL_MODULE__
				/* Limitation of copy_to_user */
				count = length / FLASH_CHUNK_SZ;
				residual = length % FLASH_CHUNK_SZ;	
#endif
				found++;
				break;
			}
		}
	}

	if (!found) {
		pext->Status = EXT_STATUS_ERR;
		return 0;
	}

	if (pext->RequestLen < length) {
		pext->Status = EXT_STATUS_COPY_ERR;
		return (-EFAULT);
	}

	pext->Status = EXT_STATUS_OK;
	pext->DetailStatus = EXT_STATUS_OK;

	/* Read from user buffer */
	usr_tmp = (uint8_t *)pext->RequestAdr;

#ifdef __VMKERNEL_MODULE__
	kern_tmp = (uint8_t *)KMEM_ZALLOC(length, 30);
#else
	kern_tmp = (uint8_t *)vmalloc(length);
#endif
	if (kern_tmp == NULL) {
		pext->Status = EXT_STATUS_NO_MEMORY;
		printk(KERN_WARNING
		    "%s: ERROR in flash allocation.\n", __func__);
		return (-ENOMEM);
	}

#ifdef __VMKERNEL_MODULE__
	/* 
	 * A copy to/from user of > 128 kb is causing 
	 * a kernel panic in vmware.(8.16.06)
	 */
	for (cnt = 0; cnt <= count ; cnt++) {
		if (cnt < count) {
			ret = copy_from_user(kern_tmp + ((FLASH_CHUNK_SZ) * cnt), 
				usr_tmp + ((FLASH_CHUNK_SZ) * cnt), FLASH_CHUNK_SZ);
		} else if (residual != 0) {
			ret = copy_from_user(kern_tmp + ((FLASH_CHUNK_SZ) * count), 
				usr_tmp + ((FLASH_CHUNK_SZ) * count), residual);
		}
		
#else
		ret = copy_from_user(kern_tmp , usr_tmp , length);
#endif
		if (ret) {
#ifdef __VMKERNEL_MODULE__
			KMEM_FREE(kern_tmp, length);
#else
			vfree(kern_tmp);
#endif
			pext->Status = EXT_STATUS_COPY_ERR;
			DEBUG9_10(printk("%s: ERROR in buffer copy READ. "
		    	"RequestAdr=%p\n", __func__, pext->RequestAdr));
			return (-EFAULT);
		}
#ifdef __VMKERNEL_MODULE__
	}
#endif

	/* Go with update */
	status = qla2x00_update_or_read_flash(ha, kern_tmp, saddr, 
	    length, QLA2X00_WRITE);

#ifdef __VMKERNEL_MODULE__
	KMEM_FREE(kern_tmp, length);
#else
	vfree(kern_tmp);
#endif

	if (status) {
		ret = -EFAULT;
		pext->Status = EXT_STATUS_COPY_ERR;
		DEBUG9_10(printk("%s: ERROR updating flash.\n", __func__));
#if defined(ISP2300)
	} else {
		uint8_t		*ptmp_mem = NULL;
		if (check_24xx_or_54xx_device_ids(ha) || check_25xx_device_ids(ha)) {
			if (qla2x00_get_ioctl_scrap_mem(ha,
			    (void **)&ptmp_mem, PAGE_SIZE)) {
				/* not enough memory */
				pext->Status = EXT_STATUS_NO_MEMORY;
				DEBUG9_10(printk("%s(%ld): inst=%ld scrap not "
				    "big enough. size requested=%ld.\n",
				    __func__, ha->host_no,
				    ha->instance, PAGE_SIZE));
			} else {
				if (qla24xx_refresh_flash_version(ha, ptmp_mem)) {
					pext->Status = EXT_STATUS_ERR;
					DEBUG9_10(printk( "%s: ERROR reading first updated "
					"flash versions.\n",
					__func__));
				}
				scsi_qla_host_t	*other_hba = NULL;	
				for (other_hba=qla2x00_hostlist; (other_hba != NULL); 
								other_hba=other_hba->next) {
					   if ((other_hba->pdev->bus->number == 
								ha->pdev->bus->number) &&
			    		    PCI_SLOT(other_hba->pdev->devfn) == 
								PCI_SLOT(ha->pdev->devfn) &&
				  	     PCI_FUNC(other_hba->pdev->devfn) != 
								PCI_FUNC(ha->pdev->devfn))  {
							
						if (qla24xx_refresh_flash_version(other_hba, ptmp_mem)) {
							pext->Status = EXT_STATUS_ERR;
							DEBUG9_10(printk( "%s: ERROR reading second updated "
				    			"flash versions.\n",
				    			__func__));
						}
					}	
				}
			}
			qla2x00_free_ioctl_scrap_mem(ha);
		}
#endif
	}

	DEBUG9(printk("%s: exiting.\n", __func__));
	return (ret);
}

int
qla2x00_get_option_rom_layout(scsi_qla_host_t *ha, EXT_IOCTL *pext, int mode)
{
	int			ret, iter;
	INT_OPT_ROM_REGION	*OptionRomTable = NULL;
	INT_OPT_ROM_LAYOUT	*optrom_layout;	
	unsigned long		OptionRomTableSize; 

	DEBUG9(printk("%s: entered.\n", __func__));

	/* Pick the right OptionRom table based on device id */
	qla2x00_get_option_rom_table(ha, &OptionRomTable, &OptionRomTableSize);

	if (OptionRomTable == NULL) {
		pext->Status = EXT_STATUS_DEV_NOT_FOUND;
		DEBUG9_10(printk("%s(%ld) Option Rom Table for device_id=0x%x "
		    "not defined\n", __func__,ha->host_no,ha->device_id));
		return 0;
	}

	if (pext->ResponseLen < OptionRomTableSize) {
		pext->Status = EXT_STATUS_BUFFER_TOO_SMALL;
		DEBUG9_10(printk("%s(%ld) buffer too small: response_len = %d "
		    "optrom_table_len=%ld.\n", __func__, ha->host_no,
		    pext->ResponseLen,OptionRomTableSize));
		return 0;
	}
	if (qla2x00_get_ioctl_scrap_mem(ha, (void **)&optrom_layout,
	    OptionRomTableSize)) {
		/* not enough memory */
		pext->Status = EXT_STATUS_NO_MEMORY;
		DEBUG9_10(printk("%s(%ld): inst=%ld scrap not big enough. "
		    "size requested=%ld.\n", __func__, ha->host_no,
		    ha->instance, OptionRomTableSize));
		return 0;
	}

	// Dont Count the NULL Entry.
	optrom_layout->NoOfRegions =
	    (OptionRomTableSize / sizeof(INT_OPT_ROM_REGION) - 1);

	for (iter = 0; iter < optrom_layout->NoOfRegions; iter++) {
		optrom_layout->Region[iter].Region =
		    OptionRomTable[iter].Region;
		optrom_layout->Region[iter].Size =
		    OptionRomTable[iter].Size;

		if (OptionRomTable[iter].Region == INT_OPT_ROM_REGION_ALL)
			optrom_layout->Size = OptionRomTable[iter].Size;
	}

	ret = copy_to_user(pext->ResponseAdr, optrom_layout,
	    OptionRomTableSize);
	if (ret) {
		pext->Status = EXT_STATUS_COPY_ERR;
		DEBUG9_10(printk("%s(%ld): inst=%ld ERROR copy rsp buffer.\n",
		    __func__, ha->host_no, ha->instance));
		qla2x00_free_ioctl_scrap_mem(ha);
		return (-EFAULT);
	}

	pext->Status       = EXT_STATUS_OK;
	pext->DetailStatus = EXT_STATUS_OK;

	qla2x00_free_ioctl_scrap_mem(ha);

	DEBUG9(printk("%s: exiting.\n", __func__));
	return ret;
}


/*
* qla2x00_get_option_rom_table
* 	This function returns the OptionRom Table for matching device id.
*/
static void
qla2x00_get_option_rom_table(scsi_qla_host_t *ha,
    INT_OPT_ROM_REGION **pOptionRomTable, unsigned long  *OptionRomTableSize)
{
	DEBUG9(printk("%s: entered.\n", __func__));

	switch (ha->device_id) {
	case QLA6312_DEVICE_ID:
		*pOptionRomTable = OptionRomTable6312;
		*OptionRomTableSize = sizeof(OptionRomTable6312);
		break;
	case QLA2312_DEVICE_ID:
		/* HBA Model 6826A - is 2312 V3 Chip */
		if (IS_OEM_1_HBA(ha->subsystem_vendor, ha->subsystem_device)) {
			*pOptionRomTable = OptionRomTableHp;
			*OptionRomTableSize = sizeof(OptionRomTableHp);
		} else {
			*pOptionRomTable = OptionRomTable2312;
			*OptionRomTableSize = sizeof(OptionRomTable2312);
		}
		break;
	case QLA2322_DEVICE_ID:
		*pOptionRomTable = OptionRomTable2322;
		*OptionRomTableSize = sizeof(OptionRomTable2322);
		break;
	case QLA6322_DEVICE_ID:
		*pOptionRomTable = OptionRomTable6322;
		*OptionRomTableSize = sizeof(OptionRomTable6322);
		break;
	case QLA2422_DEVICE_ID:
	case QLA2432_DEVICE_ID:
	case QLA5422_DEVICE_ID:
	case QLA5432_DEVICE_ID:
		*pOptionRomTable = OptionRomTable2422;
		*OptionRomTableSize = sizeof(OptionRomTable2422);
		break;
	case QLA2532_DEVICE_ID:
		*pOptionRomTable = OptionRomTable25XX;
		*OptionRomTableSize = sizeof(OptionRomTable25XX);
		break;
	default:
		DEBUG9_10(printk("%s(%ld) Option Rom Table for device_id=0x%x "
		    "not defined\n", __func__, ha->host_no,ha->device_id));
		break;
	}

	DEBUG9(printk("%s: exiting.\n", __func__));
}

#if !defined(ISP2100) && !defined(ISP2200)

int
qla24xx_write_nvram_data(scsi_qla_host_t *ha, uint8_t *pnew_nv, uint32_t naddr,
    uint32_t transfer_size)
{
	int			ret = 0;
	uint32_t		i;
	uint32_t		*dwptr;
	struct device_reg_24xx	*reg = (struct device_reg_24xx *)ha->iobase;

	if (check_25xx_device_ids(ha)) {
		ret = qla24xx_write_flash_data(ha, (uint32_t *)pnew_nv,
			    FA_VPD_NVRAM_ADDR | naddr, transfer_size >>2);
	} else if (check_24xx_or_54xx_device_ids(ha)) {
		/* Enable flash write. */
		WRT_REG_DWORD(&reg->ctrl_status,
		    RD_REG_DWORD(&reg->ctrl_status) | CSRX_FLASH_ENABLE);
		RD_REG_DWORD(&reg->ctrl_status);	/* PCI Posting. */

		/* Disable NVRAM write-protection. */
		qla24xx_write_flash_dword(ha, nvram_conf_to_access_addr(0x101), 0);
		qla24xx_write_flash_dword(ha, nvram_conf_to_access_addr(0x101), 0);

		/* Dword writes to flash. */
		dwptr = (uint32_t *)pnew_nv;
		for (i = 0; i < transfer_size >> 2; i++, naddr++, dwptr++) {

			if (qla24xx_write_flash_dword(ha,
			    nvram_data_to_access_addr(naddr), *dwptr) != 
			    QLA2X00_SUCCESS) {
				DEBUG9(printk("%s(%ld) Unable to program "
				    "nvram address=%x data=%x.\n", __func__,
				    ha->host_no, naddr, *dwptr));
				ret = QLA2X00_FUNCTION_FAILED;
				break;
			}
		}

		/* Enable NVRAM write-protection. */
		qla24xx_write_flash_dword(ha, nvram_conf_to_access_addr(0x101), 0x8c);

		/* Disable flash write. */
		WRT_REG_DWORD(&reg->ctrl_status,
		    RD_REG_DWORD(&reg->ctrl_status) & ~CSRX_FLASH_ENABLE);
		RD_REG_DWORD(&reg->ctrl_status);	/* PCI Posting. */
	}
	return (ret);
}

int
qla2x00_get_vpd(scsi_qla_host_t *ha, EXT_IOCTL *pext, int mode)
{
	int		ret = 0;
	uint8_t		*ptmp_buf;
	uint32_t	data_offset;
	uint32_t	tmp_data_offset;
	uint32_t	transfer_size;
	unsigned long	flags;

	if (!check_24xx_or_54xx_device_ids(ha) && !check_25xx_device_ids(ha)) {
		pext->Status = EXT_STATUS_INVALID_REQUEST;
		DEBUG9_10(printk(
		    "%s(%ld): inst=%ld not 24xx or 25xx. exiting.\n",
		    __func__, ha->host_no, ha->instance));
		return (ret);
	}

	DEBUG9(printk("%s(%ld): entered.\n", __func__, ha->host_no));

	transfer_size = FA_NVRAM_VPD_SIZE * 4; /* byte count */
	if (pext->ResponseLen < transfer_size) {
		pext->ResponseLen = transfer_size;
		pext->Status = EXT_STATUS_BUFFER_TOO_SMALL;
		DEBUG9_10(printk(
		    "%s(%ld): inst=%ld Response buffer too small.\n",
		    __func__, ha->host_no, ha->instance));
		return (ret);
	}

	if (qla2x00_get_ioctl_scrap_mem(ha, (void **)&ptmp_buf,
	    transfer_size)) {
		/* not enough memory */
		pext->Status = EXT_STATUS_NO_MEMORY;
		DEBUG9_10(printk("%s(%ld): inst=%ld scrap not big enough. "
		    "size requested=%d.\n",
		    __func__, ha->host_no, ha->instance,
		    ha->nvram_size));
		return (ret);
	}

	if (PCI_FUNC(ha->pdev->devfn))
		data_offset = FA_NVRAM_VPD1_ADDR;
	else
		data_offset = FA_NVRAM_VPD0_ADDR;

	tmp_data_offset = data_offset;

	/* Dump VPD region in NVRAM. */
	spin_lock_irqsave(&(to_qla_parent(ha))->hardware_lock, flags);
	qla24xx_read_nvram_data(ha, (uint32_t *)ptmp_buf, data_offset,
			transfer_size);
	spin_unlock_irqrestore(&(to_qla_parent(ha))->hardware_lock, flags);

	DEBUG2(printk(
	    "%s(%ld): using data_offset=0x%x transfer_size=0x%x.\n",
	    __func__, ha->host_no, data_offset, transfer_size));

	DEBUG2(qla2x00_dump_buffer((uint8_t *)ptmp_buf, transfer_size));
	ret = copy_to_user((void *)(pext->ResponseAdr), ptmp_buf,
	    transfer_size);
	if (ret) {
		pext->Status = EXT_STATUS_COPY_ERR;
		DEBUG9_10(printk("%s(%ld): inst=%ld ERROR copy rsp buffer.\n",
		    __func__, ha->host_no, ha->instance));
		qla2x00_free_ioctl_scrap_mem(ha);
		return (-EFAULT);
	}

	pext->Status       = EXT_STATUS_OK;
	pext->DetailStatus = EXT_STATUS_OK;

	qla2x00_free_ioctl_scrap_mem(ha);

	DEBUG9(printk("%s(%ld): exiting.\n", __func__, ha->host_no));

	return (ret);
}


int
qla2x00_update_vpd(scsi_qla_host_t *ha, EXT_IOCTL *pext, int mode)
{
	int		ret = 0;
	uint8_t		*usr_tmp, *kernel_tmp, *pnew_nv, *porg_nv;
	uint32_t	data_offset;
	uint32_t	transfer_size;
	unsigned long	flags;


	if (!check_24xx_or_54xx_device_ids(ha) && !check_25xx_device_ids(ha)) {
		pext->Status = EXT_STATUS_INVALID_REQUEST;
		DEBUG9_10(printk(
		    "%s(%ld): inst=%ld not 24xx or 25xx. exiting.\n",
		    __func__, ha->host_no, ha->instance));
		return (ret);
	}

	DEBUG9(printk("%s(%ld): entered DEBUG.\n", __func__, ha->host_no));

	transfer_size = FA_NVRAM_VPD_SIZE * 4; /* byte count */
	if (pext->RequestLen < transfer_size)
		transfer_size = pext->RequestLen;

	if (qla2x00_get_ioctl_scrap_mem(ha, (void **)&pnew_nv, transfer_size)) {
		/* not enough memory */
		pext->Status = EXT_STATUS_NO_MEMORY;
		DEBUG9_10(printk("%s(%ld): inst=%ld scrap not big enough. "
		    "size requested=%d.\n",
		    __func__, ha->host_no, ha->instance, transfer_size));
		return (ret);
	}

	if (check_25xx_device_ids(ha)){
		if (qla2x00_get_ioctl_scrap_mem(ha, (void **)&porg_nv,
		    VPD_HBAPARAM_SIZE_25XX)) {
			/* not enough memory */
			pext->Status = EXT_STATUS_NO_MEMORY;
			DEBUG9_10(printk("%s(%ld): inst=%ld scrap not big enough. "
			    "size requested=%d.\n",
			    __func__, ha->host_no, ha->instance,
			    VPD_HBAPARAM_SIZE_25XX));
			return (ret);
		}
	}

	/* Read from user buffer */
	kernel_tmp = (uint8_t *)pnew_nv;
	usr_tmp = (void *)(pext->RequestAdr);

	ret = copy_from_user(kernel_tmp, usr_tmp, transfer_size);
	if (ret) {
		pext->Status = EXT_STATUS_COPY_ERR;
		DEBUG9_10(printk(
		    "%s(%ld): ERROR in buffer copy READ. RequestAdr=%p\n",
		    __func__, ha->host_no, (void *)(pext->RequestAdr)));
		qla2x00_free_ioctl_scrap_mem(ha);
		return (-EFAULT);
	}

	if (PCI_FUNC(ha->pdev->devfn))
		data_offset = FA_NVRAM_VPD1_ADDR;
	else
		data_offset = FA_NVRAM_VPD0_ADDR;


	DEBUG(qla2x00_dump_buffer((uint8_t *)pnew_nv, transfer_size));

	DEBUG9(printk(
	    "%s(%ld): using data_offset=0x%x transfer_size=0x%x.\n",
	    __func__, ha->host_no, data_offset, transfer_size));


	DEBUG2(qla2x00_dump_buffer((uint8_t *)pnew_nv, transfer_size));

	/* Write NVRAM. */
	spin_lock_irqsave(&(to_qla_parent(ha))->hardware_lock, flags);

	if (check_25xx_device_ids(ha)) {
		qla24xx_read_nvram_data(ha, (uint32_t *)porg_nv, FA_NVRAM_VPD0_ADDR,
		    VPD_HBAPARAM_SIZE_25XX);

		memcpy((porg_nv + (data_offset * 4)), pnew_nv, transfer_size);

		qla24xx_write_nvram_data(ha, porg_nv, FA_NVRAM_VPD0_ADDR,
                       VPD_HBAPARAM_SIZE_25XX);
	} else {
		ret = qla24xx_write_nvram_data(ha, pnew_nv, data_offset, transfer_size);
		if (ret) {
			pext->Status       = EXT_STATUS_ERR;
			DEBUG9_10(printk("%s(%ld): inst=%ld nv_vpd 64K"
			    "read-modify-write failed.",
			    __func__, ha->host_no, ha->instance));
			qla2x00_free_ioctl_scrap_mem(ha);
			return (ret);
		}
	}

	spin_unlock_irqrestore(&(to_qla_parent(ha))->hardware_lock, flags);

	pext->Status       = EXT_STATUS_OK;
	pext->DetailStatus = EXT_STATUS_OK;

	qla2x00_free_ioctl_scrap_mem(ha);

	DEBUG9(printk("%s(%ld): exiting.\n", __func__, ha->host_no));

	/* No need to reset the 24xx. */
	return ret;
}

#define SFP_DEV_SIZE	256
#define SFP_BLOCK_SIZE	64

int
qla2x00_get_sfp_data(scsi_qla_host_t *ha, EXT_IOCTL *pext, int mode)
{
	int		ret = 0;
	uint8_t		*ptmp_buf, *ptmp_iter;
	uint32_t	transfer_size;
	uint16_t	iter, addr, offset;
	int		rval;

	if (!check_24xx_or_54xx_device_ids(ha) && !check_25xx_device_ids(ha)) {
		pext->Status = EXT_STATUS_INVALID_REQUEST;
		DEBUG9_10(printk(
		    "%s(%ld): inst=%ld not 24xx or 25xx. exiting.\n",
		    __func__, ha->host_no, ha->instance));
		return (ret);
	}

	DEBUG9(printk("%s(%ld): entered.\n", __func__, ha->host_no));

	transfer_size = SFP_DEV_SIZE * 2;
	if (pext->ResponseLen < transfer_size) {
		pext->ResponseLen = transfer_size;
		pext->Status = EXT_STATUS_BUFFER_TOO_SMALL;
		DEBUG9_10(printk(
		    "%s(%ld): inst=%ld Response buffer too small.\n",
		    __func__, ha->host_no, ha->instance));
		return (ret);
	}
        if (qla2x00_get_new_ioctl_dma_mem(ha, SFP_BLOCK_SIZE) !=
	    QL_STATUS_SUCCESS) {
		/* not enough memory */
		pext->Status = EXT_STATUS_NO_MEMORY;
		DEBUG9_10(printk("%s(%ld): inst=%ld scrap not big enough. "
		    "size requested=%d.\n",
		    __func__, ha->host_no, ha->instance,
		    ha->nvram_size));
		return (ret);
	}

	if (qla2x00_get_ioctl_scrap_mem(ha, (void **)&ptmp_buf,
	    transfer_size)) {
		/* not enough memory */
		pext->Status = EXT_STATUS_NO_MEMORY;
		DEBUG9_10(printk("%s(%ld): inst=%ld scrap not big enough. "
		    "size requested=%d.\n",
		    __func__, ha->host_no, ha->instance,
		    ha->nvram_size));
		return (ret);
	}

	ptmp_iter = ptmp_buf;
	addr = 0xa0;
	for (iter = 0, offset = 0; iter < (SFP_DEV_SIZE * 2) / SFP_BLOCK_SIZE;
	    iter++, offset += SFP_BLOCK_SIZE) {
		if (iter == 4) {
			/* Skip to next device address. */
			addr = 0xa2;
			offset = 0;
		}

		rval = qla2x00_read_sfp(ha, ha->ioctl_mem_phys, addr, offset,
		    SFP_BLOCK_SIZE);
		if (rval != QLA2X00_SUCCESS) {
			pext->Status = EXT_STATUS_COPY_ERR;
			DEBUG9_10(printk("%s(%ld): inst=%ld ERROR reading SFP "
			    "data (%x/%x/%x).\n",
			    __func__, ha->host_no, ha->instance, rval, addr,
			    offset));
			qla2x00_free_ioctl_scrap_mem(ha);
			return (-EFAULT);
		}
		memcpy(ptmp_iter, ha->ioctl_mem, SFP_BLOCK_SIZE);
		ptmp_iter += SFP_BLOCK_SIZE;
	}

	ret = copy_to_user((void *)(pext->ResponseAdr), ptmp_buf,
	    transfer_size);
	if (ret) {
		pext->Status = EXT_STATUS_COPY_ERR;
		DEBUG9_10(printk("%s(%ld): inst=%ld ERROR copy rsp buffer.\n",
		    __func__, ha->host_no, ha->instance));
		qla2x00_free_ioctl_scrap_mem(ha);
		return (-EFAULT);
	}

	pext->Status       = EXT_STATUS_OK;
	pext->DetailStatus = EXT_STATUS_OK;

	qla2x00_free_ioctl_scrap_mem(ha);

	DEBUG9(printk("%s(%ld): exiting.\n", __func__, ha->host_no));

	return (ret);
}

int
qla2x00_update_port_param(scsi_qla_host_t *ha, EXT_IOCTL *pext, int mode)
{
	int             ret = 0, rval, port_found;
	uint16_t        mb[MAILBOX_REGISTER_COUNT];
	uint16_t        idma_speed;
	fc_port_t       *fcport;
	INT_PORT_PARAM  *port_param;

	if (!check_24xx_or_54xx_device_ids(ha) && !check_25xx_device_ids(ha)) {
		pext->Status = EXT_STATUS_INVALID_REQUEST;
		DEBUG9_10(printk(
		    "%s(%ld): inst=%ld not 24xx or 25xx. exiting.\n",
		    __func__, ha->host_no, ha->instance));
		return (ret);
	}

	if (qla2x00_get_ioctl_scrap_mem(ha, (void **)&port_param,
	    sizeof(INT_PORT_PARAM))) {
		/* not enough memory */
		pext->Status = EXT_STATUS_NO_MEMORY;
		DEBUG9_10(printk("%s(%ld): inst=%ld scrap not big enough. "
		    "size requested=%Zd.\n",
		    __func__, ha->host_no, ha->instance,
		    sizeof(INT_PORT_PARAM)));
		return (ret);
	}

	/* Copy request buffer */
	ret = copy_from_user((uint8_t *)port_param, (uint8_t *)pext->RequestAdr,
	    sizeof(INT_PORT_PARAM));

	if (ret) {
		pext->Status = EXT_STATUS_COPY_ERR;
		DEBUG9_10(printk(
		    "%s(%ld): inst=%ld ERROR copy req buf ret=%d\n",
		    __func__, ha->host_no, ha->instance, ret));
		qla2x00_free_ioctl_scrap_mem(ha);
		return (-EFAULT);
	}

	if (port_param->FCScsiAddr.DestType != EXT_DEF_TYPE_WWPN) {
		pext->Status = EXT_STATUS_DEV_NOT_FOUND;
		DEBUG9_10(printk("%s(%ld): inst=%ld ERROR -wrong Dest "
		    "type.\n", __func__, ha->host_no, ha->instance));
		qla2x00_free_ioctl_scrap_mem(ha);
		return (ret);
	}

	port_found = 0;
	list_for_each_entry(fcport, &ha->fcports, list) {

		/* Do not process fcports that are discarded */
		if (fcport->flags & FC_DEVICE_DELETED) {
			if (!ql2xreloginfcport)
				continue;
		}

		if (memcmp(fcport->port_name,
		    port_param->FCScsiAddr.DestAddr.WWPN, WWN_SIZE))
			continue;

		port_found++;
		break;
	}
	if (!port_found) {
		pext->Status = EXT_STATUS_DEV_NOT_FOUND;
		DEBUG9_10(printk("%s(%ld): inst=%ld FC AddrFormat - DID NOT "
		    "FIND Port matching WWPN.\n",
		    __func__, ha->host_no, ha->instance));
		qla2x00_free_ioctl_scrap_mem(ha);
		return (ret);
	}
                      
	/* Go with operation. */
	if (port_param->Mode) {
		switch (port_param->Speed) {
		case EXT_DEF_PORTSPEED_1GBIT:
			idma_speed = PORT_SPEED_1GB;
			break;
		case EXT_DEF_PORTSPEED_2GBIT:
			idma_speed = PORT_SPEED_2GB;
			break;
		case EXT_DEF_PORTSPEED_4GBIT:
			idma_speed = PORT_SPEED_4GB;
			break;
		case EXT_DEF_PORTSPEED_8GBIT:
			idma_speed = PORT_SPEED_8GB;
			break;
		default:
			pext->Status = EXT_STATUS_INVALID_PARAM;
			DEBUG9_10(printk("%s(%ld): inst=%ld ERROR -invalid "
			    "speed.\n", __func__, ha->host_no, ha->instance));
			qla2x00_free_ioctl_scrap_mem(ha);
			return (ret);
		}
		
		rval = qla2x00_set_idma_speed(ha, fcport->loop_id, idma_speed,
		    mb);
		if (rval != QLA2X00_SUCCESS) {
			/* what does 0x09 represents in following test case */
			if (mb[0] == MBS_CMD_ERR && mb[1] == 0x09)
				pext->Status = EXT_STATUS_DEVICE_NOT_READY;
			else if (mb[0] == MBS_CMD_PARAM_ERR)
				pext->Status = EXT_STATUS_INVALID_PARAM;
			else
				pext->Status = EXT_STATUS_ERR;

			DEBUG9_10(printk("%s(%ld): inst=%ld set iDMA cmd "
			    "FAILED=%x.\n", __func__, ha->host_no,
			    ha->instance, mb[0]));
			qla2x00_free_ioctl_scrap_mem(ha);
			return (ret);
		}
	} else {
		rval = qla2x00_get_idma_speed(ha, fcport->loop_id,
		    &idma_speed, mb);
		if (rval != QLA2X00_SUCCESS) {
			/* what does 0x09 represents in following test case */
			if (mb[0] == MBS_CMD_ERR && mb[1] == 0x09)
				pext->Status = EXT_STATUS_DEVICE_NOT_READY;
			else if (mb[0] == MBS_CMD_PARAM_ERR)
				pext->Status = EXT_STATUS_INVALID_PARAM;
			else
				pext->Status = EXT_STATUS_ERR;

			DEBUG9_10(printk("%s(%ld): inst=%ld get iDMA cmd "
			    "FAILED=%x.\n", __func__, ha->host_no,
			    ha->instance, mb[0]));
			qla2x00_free_ioctl_scrap_mem(ha);
			return (ret);
		}

		switch (idma_speed) {
		case PORT_SPEED_1GB:
			port_param->Speed = EXT_DEF_PORTSPEED_1GBIT;
			break;
		case PORT_SPEED_2GB:
			port_param->Speed = EXT_DEF_PORTSPEED_2GBIT;
			break;
		case PORT_SPEED_4GB:
			port_param->Speed = EXT_DEF_PORTSPEED_4GBIT;
			break;
		case PORT_SPEED_8GB:
			port_param->Speed = EXT_DEF_PORTSPEED_8GBIT;
			break;
		default:
			port_param->Speed = 0xFFFF;
			break;
		}

		ret = copy_to_user((uint8_t*)pext->ResponseAdr,
		    (uint8_t *)port_param, sizeof(INT_PORT_PARAM));
		if (ret) {
			pext->Status = EXT_STATUS_COPY_ERR;
		DEBUG9_10(printk(
		    "%s(%ld): inst=%ld ERROR copy rsp buf ret=%d\n",
		    __func__, ha->host_no, ha->instance, ret));
		qla2x00_free_ioctl_scrap_mem(ha);
		return (-EFAULT);
		}
	}

	pext->Status       = EXT_STATUS_OK;
	pext->DetailStatus = EXT_STATUS_OK;

	qla2x00_free_ioctl_scrap_mem(ha);

	DEBUG9(printk("%s(%ld): exiting.\n", __func__, ha->host_no));

	return (ret);
}

#endif

int
qla2x00_get_fw_dump(scsi_qla_host_t *ha, EXT_IOCTL *pext, int mode)
{
	int		ret = 0;

	DEBUG9(printk("%s(%ld): entered.\n", __func__, ha->host_no));

	if (!ha->fw_dumped) {
		pext->Status = EXT_STATUS_HBA_NOT_READY;
		DEBUG9_10(printk(
		    "%s(%ld): inst=%ld No firmware dump available.\n",
		    __func__, ha->host_no, ha->instance));
		return (ret);
	}

	if (pext->ResponseLen < ha->fw_dump_len) {
		pext->ResponseLen = ha->fw_dump_len;
		pext->Status = EXT_STATUS_BUFFER_TOO_SMALL;
		DEBUG9_10(printk(
		    "%s(%ld): inst=%ld Response buffer too small.\n",
		    __func__, ha->host_no, ha->instance));
		return (ret);
	}

	ret = copy_to_user((void *)(pext->ResponseAdr), ha->fw_dump,
	    ha->fw_dump_len);
	if (ret) {
		pext->Status = EXT_STATUS_COPY_ERR;
		DEBUG9_10(printk("%s(%ld): inst=%ld ERROR copy rsp buffer.\n",
		    __func__, ha->host_no, ha->instance));
		return (-EFAULT);
	}

	printk(KERN_INFO
	    "scsi(%ld): Firmware dump cleared.\n", ha->host_no);
	ha->fw_dumped = 0;

	pext->Status       = EXT_STATUS_OK;
	pext->DetailStatus = EXT_STATUS_OK;
	pext->ResponseLen = ha->fw_dump_len;

	DEBUG9(printk("%s(%ld): exiting.\n", __func__, ha->host_no));

	return (ret);
}

