/*
 *                  QLOGIC LINUX SOFTWARE
 *
 * QLogic ISP2x00 device driver for Linux 2.6.x
 * Copyright (C) 2003-2005 QLogic Corporation
 * (www.qlogic.com)
 *
 * This program is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License as published by the
 * Free Software Foundation; either version 2, or (at your option) any
 * later version.
 *
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *
 */

#if defined(__VMKERNEL_MODULE__)
#include "npiv_api_dist.h"
#endif

#define MAX_RPI				2048
#define POISON_VALUE			0xDEAD
int qla24xx_create_vport(scsi_qla_host_t *, struct vport_data *);
int qla24xx_delete_vport(scsi_qla_host_t *);
int qla24xx_vport_info(scsi_qla_host_t *, struct vport_info *);
int qla24xx_tgt_remove(scsi_qla_host_t *, uint32, uint32);
void qla2x00_do_dpc_all_vps(scsi_qla_host_t *);
int qla24xx_use_vp_database(scsi_qla_host_t *);
STATIC void qla2x00_vp_cmpl_timeout(unsigned long);

extern void sp_get(srb_t *sp);
extern void sp_put(struct scsi_qla_host * ha, srb_t *sp);
extern struct semaphore instance_lock;

void qla2xxx_vha_init(scsi_qla_host_t *vha)
{
	atomic_set(&vha->refcount, 1);
	smp_mb();
}

static void qla2xxx_cleanup_pending_commands(scsi_qla_host_t *vha)
{
	unsigned long flags = 0;
	srb_t *sp = NULL;
	struct list_head *list = NULL, *temp = NULL;
	scsi_qla_host_t *pha = vha->parent;

	/* Cleaning up commands from pending queues */
	spin_lock_irqsave(&vha->list_lock,flags);
	list_for_each_safe(list, temp, &vha->pending_queue) {
		sp = list_entry(list, srb_t, list);

		if (sp->flags & SRB_ABORTED)
			continue;

		__del_from_pending_queue(vha, sp);

		sp->flags = 0;
		CMD_RESULT(sp->cmd) = DID_NO_CONNECT << 16;

		__add_to_done_queue(vha, sp);
	}

	/* Cleaning up commands from retry queues */
	list_for_each_safe(list, temp, &vha->retry_queue) {
		sp = list_entry(list, srb_t, list);

		if (sp->flags & SRB_ABORTED)
			continue;

		__del_from_retry_queue(vha, sp);

		sp->flags = 0;
		CMD_RESULT(sp->cmd) = DID_NO_CONNECT << 16;

		__add_to_done_queue(vha, sp);
	}

	/* Cleaning up commands from scsi_retry queues */
	list_for_each_safe(list, temp, &vha->scsi_retry_queue) {
		sp = list_entry(list, srb_t, list);

		if (sp->flags & SRB_ABORTED)
			continue;

		__del_from_scsi_retry_queue(vha, sp);

		sp->flags = 0;
		CMD_RESULT(sp->cmd) = DID_NO_CONNECT << 16;

		__add_to_done_queue(vha, sp);
	}
	spin_unlock_irqrestore(&vha->list_lock, flags);

	/* Cleaning up commands from done queues */
	qla2x00_done(vha);
	qla2x00_done(pha);
}

/*
 * qla24xx_deallocate_vp_id(): Move vport to vp_del_list.
 * @vha: Vport context.
 *
 * Note: The caller of this function needs to ensure that
 * the vport_lock of the corresponding physical port is
 * held while making this call.
 */
static void
qla24xx_deallocate_vp_id(scsi_qla_host_t *vha)
{
	uint16_t vp_id;
	scsi_qla_host_t *pha = vha->parent;
	struct timer_list	tmp_vp_timer;

	ENTER(__func__);	
#if !defined(SH_HAS_HOST_LOCK)
		spin_lock(&io_request_lock);
#else
		spin_lock(pha->host->host_lock);
#endif

	vp_id = vha->vp_idx;
	pha->num_vhosts--;
	clear_bit(vp_id, (unsigned long *)pha->vp_idx_map);
	pha->vhba[vp_id] = NULL;
	list_del(&vha->vp_list);
	/* Add vport to physical port's vp delete list. */
	list_add_tail(&vha->vp_list, &pha->vp_del_list);

#if !defined(SH_HAS_HOST_LOCK)
		spin_unlock(&io_request_lock);
#else
		spin_unlock(pha->host->host_lock);
#endif
	LEAVE(__func__);	
}

static void qla2xxx_vha_release(scsi_qla_host_t *vha)
{
	DEBUG(printk("Dealloacting virtual port host_no = %d..\n", vha->host_no));
	qla24xx_deallocate_vp_id(vha);

	set_bit(VPORT_CLEANUP_NEEDED, &vha->parent->dpc_flags);
	up(vha->parent->dpc_wait);
}

/*
 * qla2xxx_vha_get(): Grab a reference from the vport
 *
 * @vha: Vport context.
 *
 * 
 *
 */
void qla2xxx_vha_get(scsi_qla_host_t *vha)
{
	if (vha->parent) {
		atomic_inc(&vha->refcount);
		smp_mb__after_atomic_inc();
	}
}

/*
 * qla2xxx_vha_put(): Drop a reference from the vport.
 * If we are the last caller, free vha.
 * 
 * @vha: Vport context.
 *
 * Note: The caller of this function needs to ensure that
 * the vport_lock of the corresponding physical port is
 * not held while making this call.
 *
 */
int qla2xxx_vha_put(scsi_qla_host_t *vha)
{
	if (vha->parent) {
		if(atomic_dec_and_test(&vha->refcount)) {
			qla2xxx_vha_release(vha);
			return 1;
		}
	}

	return 0;
}

STATIC void
qla2x00_vp_cmpl_timeout(unsigned long data)
{
	struct completion	*vp_action = (struct completion *)data;

	printk("qla2x00_vp_cmpl_timeout: entered.\n");

	if (vp_action != NULL) {
		complete(vp_action);
	}

	printk("qla2x00_cp_cmpl_timeout: exiting.\n");
}

static uint32_t
qla24xx_allocate_vp_id(scsi_qla_host_t *vha)
{
	uint32_t vp_id;
	unsigned long flags;
	scsi_qla_host_t *pha = vha->parent;
	
	ENTER(__func__);
	/* Find an empty slot and assign an vp_id */
#if !defined(SH_HAS_HOST_LOCK)
		spin_lock(&io_request_lock);
#else
		spin_lock(pha->host->host_lock);
#endif
	spin_lock_irqsave(&pha->vport_lock, flags);
	vp_id = find_first_zero_bit((unsigned long *)pha->vp_idx_map,
				pha->max_multi_id_nports);
	if (vp_id >= pha->max_multi_id_nports)
		goto return_vp_id;

	set_bit(vp_id, (unsigned long *)pha->vp_idx_map);
	pha->num_vhosts++;
	vha->vp_idx = vp_id;
	pha->vhba[vp_id] = vha;

	list_add_tail(&vha->vp_list, &pha->vp_list);
return_vp_id:
	spin_unlock_irqrestore(&pha->vport_lock, flags);
#if !defined(SH_HAS_HOST_LOCK)
		spin_unlock(&io_request_lock);
#else
		spin_unlock(pha->host->host_lock);
#endif
	LEAVE(__func__);	
	
	return vp_id;
}

static scsi_qla_host_t *
qla24xx_find_vhost_by_name(scsi_qla_host_t *ha, uint8_t *wwpn)
{
	unsigned long flags;
	scsi_qla_host_t *vha;

	ENTER(__func__);
	/* Locate matching device in database. */
	spin_lock_irqsave(&ha->vport_lock, flags);
	list_for_each_entry(vha, &ha->vp_list, vp_list) {
		if (!memcmp(wwpn, vha->port_name, WWN_SIZE)) {
			spin_unlock_irqrestore(&ha->vport_lock, flags);
			return vha;
		}
	}
	spin_unlock_irqrestore(&ha->vport_lock, flags);
	LEAVE(__func__);	
	return NULL;
}


static scsi_qla_host_t *
qla24xx_find_vhost_by_index(scsi_qla_host_t *phys_ha, uint16_t vp_idx)
{
	scsi_qla_host_t *vha;
	unsigned long flags;

	/* return recieved ha if host is virtual or indexes the parent itself */
	if (phys_ha->parent == NULL && vp_idx == 0)
		return phys_ha;

	ENTER(__func__);
	/* Locate matching device in database. */
	spin_lock_irqsave(&phys_ha->vport_lock, flags);
	list_for_each_entry(vha, &phys_ha->vp_list, vp_list) {
		if (vha->vp_idx == vp_idx) {
			spin_unlock_irqrestore(&phys_ha->vport_lock, flags);
			return vha;
		}
	}
	spin_unlock_irqrestore(&phys_ha->vport_lock, flags);
	LEAVE(__func__);
	return NULL;
}

/*
 * qla2x00_mark_vp_devices_dead
 *	Updates fcport state when device goes offline.
 *
 * Input:
 *	ha = adapter block pointer.
 *	fcport = port structure pointer.
 *
 * Return:
 *	None.
 *
 * Context:
 */
void
qla2x00_mark_vp_devices_dead(scsi_qla_host_t *ha) 
{
	fc_port_t *fcport, *fctemp;
	scsi_qla_host_t *phys_ha = ha;

	ENTER(__func__);
	if (ha->parent)
		phys_ha = ha->parent;

#if !defined(SH_HAS_HOST_LOCK)
	spin_lock(&io_request_lock);
#else
	spin_lock(ha->host->host_lock);
#endif

	list_for_each_entry_safe(fcport, fctemp, &phys_ha->fcports, list) {

		/* The FC ports list still has the Initiator 
		 * entry corresponding to this VPort, invalidate them too. 
		 */
		#if 0
		if (fcport->port_type != FCT_TARGET)
			continue;
		#endif

		if (fcport->vp_idx != ha->vp_idx)
			continue;

		DEBUG(printk("scsi(%ld): Marking port dead, loop_id=0x%04x :%x\n",
			    ha->host_no, fcport->loop_id, fcport->vp_idx));

		/* As the fcport list is accessed without protection
  		 * set the fcport flag as deleted, but do not delete it.
		 * Set the vp index of the vport to an invalid value.
		 */
		fcport->vp_idx = POISON_VALUE; 
		fcport->flags |= FC_DEVICE_DELETED;
		atomic_set(&fcport->state, FC_DEVICE_DEAD);
		//list_del_init(&fcport->list);
	}

#if !defined(SH_HAS_HOST_LOCK)
	spin_unlock(&io_request_lock);
#else
	spin_unlock(ha->host->host_lock);
#endif
	LEAVE(__func__);	
}

static int
qla24xx_disable_vp (scsi_qla_host_t *vha)
{
	int cnt, ret;
	srb_t   *sp;
	struct scsi_cmnd *cmd;
	unsigned long flags;
	scsi_qla_host_t *pha = vha->parent, *tmp_ha = NULL;
	
	ENTER(__func__);	
	ret = qla24xx_control_vp(vha, VCE_COMMAND_DISABLE_VPS_LOGO_ALL);
	atomic_set(&vha->loop_state, LOOP_DEAD);

	/*
	 * Waiting for all commands for the designated target in the active
	 * array
	 */
	for (cnt = 1; cnt < MAX_OUTSTANDING_COMMANDS; cnt++) {
		spin_lock_irqsave(&pha->hardware_lock, flags);
		sp = pha->outstanding_cmds[cnt];
		/* Get a reference to the sp and drop the lock.*/
		if (sp)
			sp_get(sp);
		spin_unlock_irqrestore(&pha->hardware_lock, flags);

		if (sp) {
			cmd = sp->cmd;
			tmp_ha = (scsi_qla_host_t *)sp->cmd->host->hostdata;
			if (vha->vp_idx == tmp_ha->vp_idx) {
				if (qla2x00_eh_wait_on_command(pha, cmd, 1) == 0) {
					/* ERROR: Wait_on_command failed here */
					printk("Wait_on_command failed for scsi(%ld) sp=%p \n",
						vha->host_no, sp);
				}
			}
			/* Release the reference to sp */
			sp_put(vha, sp);
		}
	}

	atomic_set(&vha->vp_state, VP_FAILED);
	vha->flags.management_server_logged_in = 0;
	if (ret == QL_STATUS_SUCCESS) {
		printk(KERN_INFO "scsi(%ld) : VPort with id: %d disabled on host : %d\n", 
					vha->host_no, vha->vp_idx, vha->parent->host_no);
	} else {
		printk(KERN_INFO "scsi(%ld) : Failed to disable VPort %d\n", 
					vha->host_no, vha->vp_idx);
		return (VPORT_ERROR);
	}
	LEAVE(__func__);	
	return (0);
}


static int
qla24xx_enable_vp (scsi_qla_host_t *vha)
{
	int ret;
	scsi_qla_host_t *phys_ha = vha->parent;

	ENTER(__func__);
	/* Check if physical ha port is Up */
	if (atomic_read(&phys_ha->loop_state) == LOOP_DOWN  ||
		atomic_read(&phys_ha->loop_state) == LOOP_DEAD) {
		vha->vp_prev_err_state = vha->vp_err_state;
		vha->vp_err_state =  VP_ERR_PORTDWN;
		goto enable_failed;
	}

	/* Initialize the new vport unless it is a persistent port */
	if (vha->flags.persistent_bound) {
		memcpy(vha->node_name,
			((struct mid_init_cb_24xx *)phys_ha->init_cb)->entries[vha->vp_idx].node_name, WWN_SIZE);
		memcpy(vha->port_name,
			((struct mid_init_cb_24xx *)phys_ha->init_cb)->entries[vha->vp_idx].port_name, WWN_SIZE);
		ret = qla24xx_control_vp(vha, VCE_COMMAND_ENABLE_VPS);
	} else {
		ret = qla24xx_modify_vp_config(vha);
	}
	if (ret != QL_STATUS_SUCCESS) {
		goto enable_failed;
	}
	printk(KERN_INFO "scsi(%ld) : VPort with id: %d enabled on host: %d\n", 
				vha->host_no, vha->vp_idx, vha->parent->host_no);
	LEAVE(__func__);	
	return 0;
enable_failed:
	printk(KERN_ERR "scsi(%ld) : Failed to enable VPort with id: %d on host: %d\n", 
				vha->host_no, vha->vp_idx, vha->parent->host_no);
	return 1;
}

/*
 * qla24xx_use_vp_database
 *	Display the Vport database stored in the Firmware.
 *
 * Input:
 *	phys_ha = adapter block pointer.
 *
 * Return:
 *	Completion status.
 *
 * Context:
 */
int 
qla24xx_use_vp_database(scsi_qla_host_t *phys_ha)
{
	int rval = QL_STATUS_ERROR;
	int vp_count, status, vp;

	if (phys_ha->parent) {
		return rval;
	}

	DEBUG(printk("%s(%ld): Entered.\n",
			__func__, phys_ha->host_no));
	ENTER(__func__);

	rval = qla2x00_get_vp_database(phys_ha, phys_ha->mid_list, phys_ha->mid_list_dma, 
				&vp_count, &status);
	if (rval != QL_STATUS_SUCCESS) {
		return rval;
	}

	/* Information for Physical port is stored in index 0. */
	for(vp = 1; vp <= 4; vp++) {
		printk(KERN_INFO "scsi(%d) vp_status = %x port_id = %x%x%x \n", 
			phys_ha->host_no,
			phys_ha->mid_list->entries[vp].status,
			phys_ha->mid_list->entries[vp].port_id[2],
			phys_ha->mid_list->entries[vp].port_id[1],
			phys_ha->mid_list->entries[vp].port_id[0]);
	}
	DEBUG(printk("%s(%ld): Exiting.\n",
			__func__, phys_ha->host_no));
	LEAVE(__func__);
	return rval;
}

void
qla24xx_configure_vp(scsi_qla_host_t *phys_ha, uint16_t vp_idx)
{
	scsi_qla_host_t *vha = phys_ha->vhba[vp_idx];
	int ret;

	ENTER(__func__);
	if (vha == NULL)
		return;

	ret = qla2x00_send_change_request(vha, 0x3, vp_idx);
	if (ret != QL_STATUS_SUCCESS) {

		printk(KERN_ERR "Failed to enable receiving"
			" of RSCN requests: 0x%x\n", ret);
		goto config_vhba_error;
	} else {
		/* Corresponds to SCR enabled */
		clear_bit(VP_SCR_NEEDED, &vha->vp_flags);	
	}

	vha->flags.online = 1;
	if (qla24xx_configure_vhba(vha))
		goto config_vhba_error;

	atomic_set(&vha->vp_state, VP_ACTIVE);
	vha->vp_prev_err_state = vha->vp_err_state;
	vha->vp_err_state = VP_ERR_UNKWN;
	DEBUG2(printk("%s(%ld)Vport state VP_ACTIVE\n",__func__, vha->host_no));
	/* Complete the create_vport and return */
	if (!vha->init_done) {
		complete(vha->vp_create_wait);
		DEBUG2(printk("%s(%ld) signal for id acqusition completion.\n",
				__func__, vha->host_no));
	}
	LEAVE(__func__);	

config_vhba_error:
	return;

} 

void
qla2x00_alert_all_vps(scsi_qla_host_t *ha, uint32_t data, uint8_t *got_mbx)
{
	uint16_t     temp1;
	unsigned long flags;
	scsi_qla_host_t *vha = NULL, *tvha = NULL, *localvha = NULL;

	ENTER(__func__);
	/* skip if virtual port */
	if (ha->parent)
		return;

	temp1 = MSW(data);

	spin_lock_irqsave(&ha->vport_lock, flags);
	list_for_each_entry_safe(vha, tvha, &ha->vp_list, vp_list) {

		/* Skip if vport delete is in progress. */
		if(atomic_read(&vha->vp_delete_active) == 1)
			continue;
		qla2xxx_vha_get(vha);
		qla2xxx_vha_get(tvha);

		spin_unlock_irqrestore(&ha->vport_lock, flags);
		switch (temp1) {
			case MBA_LIP_OCCURRED:
			case MBA_LOOP_UP:
			case MBA_LOOP_DOWN:
			case MBA_LIP_RESET:
			case MBA_LINK_MODE_UP:
			case MBA_UPDATE_CONFIG:
			case MBA_PORT_UPDATE:
			case MBA_SCR_UPDATE:
				DEBUG(printk("scsi(%ld)%s: Async_event for"
					" vp index %d, data = 0x%x, vha=%p\n", 
					ha->host_no, __func__, vha->vp_idx, data, vha));
				
				qla24xx_isr(vha, data, got_mbx);
			break;
		}
		spin_lock_irqsave(&ha->vport_lock, flags);
		
		qla2xxx_vha_put(vha);
		localvha = tvha;
		/* Skip current entry as it is in the process of deletion. */
		if(atomic_read(&tvha->vp_delete_active) == 1)
			tvha = list_entry(tvha->vp_list.next, typeof(*tvha), vp_list);
		qla2xxx_vha_put(localvha);
	}
	spin_unlock_irqrestore(&ha->vport_lock, flags);
	LEAVE(__func__);	
}

static void
qla2x00_vp_abort_isp(scsi_qla_host_t *ha)
{
	ENTER(__func__);
	/*
	 * Physical port will do most of the abort and
	 * recovery work. We can just treat it as a
	 * loop down
	 */
	if (atomic_read(&ha->loop_state) != LOOP_DOWN) {
		atomic_set(&ha->loop_state, LOOP_DOWN);
		qla2x00_mark_all_devices_lost(ha);
	} else {
		if (!atomic_read(&ha->loop_down_timer))
			atomic_set(&ha->loop_down_timer, LOOP_DOWN_TIME);
	}
	DEBUG2(printk("scsi(%ld): Scheduling enable of Vport %d...\n",
	    ha->host_no, ha->vp_idx));
	qla24xx_enable_vp(ha);
	LEAVE(__func__);	
	// set_bit(LOOP_RESYNC_NEEDED, &ha->dpc_flags);
}

static void
qla2x00_vp_loop_resync(scsi_qla_host_t *vha)
{
	ENTER(__func__);

	if (!(test_and_set_bit(LOOP_RESYNC_ACTIVE,
	    &vha->dpc_flags))) {

		qla2x00_loop_resync(vha);

		clear_bit(LOOP_RESYNC_ACTIVE, &vha->dpc_flags);
	}

	LEAVE(__func__);	
}

static void
qla2x00_do_dpc_vp(scsi_qla_host_t *vha)
{

	ENTER(__func__);
	DEBUG(printk("scsi(%ld): called %s() - dpc->flags 0x%lx\n",
			vha->host_no, __func__,vha->dpc_flags));

	if (test_and_clear_bit(VP_IDX_ACQUIRED, &vha->vp_flags)) {
		/* VP acquired. complete port configuration */
		qla24xx_configure_vp(vha->parent, vha->vp_idx);
		return;
	}

	if (test_and_clear_bit(VP_SCR_NEEDED, &vha->vp_flags)) {
		/* Add code to retry for SCR */
	}	
	
	/* Process commands in retry queue */
	if (test_and_clear_bit(PORT_RESTART_NEEDED, &vha->dpc_flags)) {
		fc_port_t	*fcport = NULL;
		os_lun_t        *q;
		srb_t           *sp;
		unsigned long	flags = 0;
		struct list_head *list, *templist;
		int	dead_cnt, online_cnt;
		scsi_qla_host_t *ha = vha->parent;

		DEBUG(printk("scsi(%ld): port restart needed\n", vha->host_no));
		DEBUG3(printk(KERN_INFO"%s(%ld): (1) DPC checking retry_q. "
				"total=%d\n",
				__func__,
				vha->host_no,
				vha->retry_q_cnt));

		spin_lock_irqsave(&vha->list_lock, flags);
		dead_cnt = online_cnt = 0;
		list_for_each_safe(list, templist, &vha->retry_queue) {
			sp = list_entry(list, srb_t, list);
			q = sp->lun_queue;
			DEBUG3(printk("qla2x00_retry_q: pid=%ld "
					"sp=%p, spflags=0x%x, "
					"q_flag= 0x%lx\n",
					sp->cmd->serial_number,
					sp,
					sp->flags,
					q->q_flag));

			if (q == NULL)
				continue;
			fcport = q->fclun->fcport;

			/* Check for destination hba */
				if (test_bit(CFG_FAILOVER, &fcport->ha->cfg_flags)) {
				DEBUG2(printk("%s(%ld) cfg failover is active"
						" on this hba - skip processing"
						" retry_q\n", __func__, 
						fcport->ha->host_no));
				continue;
			}

			if (atomic_read(&fcport->state) == FC_DEVICE_DEAD ||
				atomic_read(&fcport->ha->loop_state) == LOOP_DEAD) {

				__del_from_retry_queue(vha, sp);
				CMD_RESULT(sp->cmd) = DID_NO_CONNECT << 16;
				if (atomic_read(&vha->loop_state) == LOOP_DEAD)
					sp->err_id = SRB_ERR_LOOP;
				else
					sp->err_id = SRB_ERR_PORT;
				CMD_HANDLE(sp->cmd) = (unsigned char *) NULL;

				add_to_done_queue(ha, sp);
				dead_cnt++;
			} else if (atomic_read(&fcport->state) != FC_DEVICE_LOST) {

				__del_from_retry_queue(vha, sp);
				CMD_RESULT(sp->cmd) = DID_BUS_BUSY << 16;
				CMD_HANDLE(sp->cmd) = (unsigned char *) NULL;
				add_to_done_queue(ha, sp);
				online_cnt++;
			} else if (qla2x00_cfg_is_lbenable(q->fclun)) {
				__del_from_retry_queue(vha, sp);
				sp->cmd->result = DID_NO_CONNECT << 16;
				sp->err_id = SRB_ERR_RETRY;
				sp->cmd->host_scribble =
					(unsigned char *) NULL;
				add_to_done_queue(ha, sp);
			}
		} /* list_for_each_safe() */
		spin_unlock_irqrestore(&vha->list_lock, flags);

		DEBUG2(printk(KERN_INFO "%s(%ld): (1) done processing retry queue - "
				"dead=%d, online=%d\n ",
				__func__,
				vha->host_no,
				dead_cnt,
				online_cnt));
	}

	/* Process commands in scsi retry queue */
	if (test_and_clear_bit(SCSI_RESTART_NEEDED, &vha->dpc_flags)) {
		os_lun_t        *q;
		os_tgt_t        *tq;
		srb_t           *sp;
		uint32_t        t;
		unsigned long	flags = 0;
		struct list_head *list, *templist;
		int	online_cnt, retry_cmds;

		DEBUG(printk("scsi(%ld): scsi restart needed\n", vha->host_no));

		/*
		 * Any requests we want to delay for some period is put
		 * in the scsi retry queue with a delay added. The
		 * timer will schedule a "scsi_restart_needed" every 
		 * second as long as there are requests in the scsi
		 * queue. 
		 */
		DEBUG2(printk(KERN_INFO"%s(%ld): (2) DPC checking scsi "
				"retry_q.total=%d\n",
				__func__,
				vha->host_no,
				vha->scsi_retry_q_cnt));

		online_cnt = 0;
		retry_cmds = 0;
		spin_lock_irqsave(&vha->list_lock, flags);
		list_for_each_safe(list, templist, &vha->scsi_retry_queue) {

			sp = list_entry(list, srb_t, list);
			q = sp->lun_queue;
			tq = sp->tgt_queue;

			DEBUG3(printk("qla2x00_scsi_retry_q: pid=%ld "
					"sp=%p, spflags=0x%x, "
					"q_flag= 0x%lx,q_state=%d\n",
					sp->cmd->serial_number,
					sp,
					sp->flags,
					q->q_flag,
					q->q_state));

			if (test_bit(TGT_SUSPENDED, &tq->q_flags)) {
				DEBUG2(printk("%s Dont flush scsi_retry_q"
					" since tgt(%d) is suspended\n", 
					__func__, SCSI_TCN_32(sp->cmd)));
				continue;
			}

			/* Was this lun suspended */
			if ((q->q_state != LUN_STATE_WAIT) &&
				 atomic_read(&tq->q_timer) == 0) {
				 DEBUG3(printk(KERN_INFO "qla2x00_scsi_retry_q: pid=%ld "
				"sp=%p, spflags=0x%x, "
				"q_flag= 0x%lx,q_state=%d, tgt_flags=0x%lx\n",
				sp->cmd->serial_number,
				sp,
				sp->flags,
				q->q_flag,
				q->q_state, tq->q_flags));
				online_cnt++;
				__del_from_scsi_retry_queue(vha, sp);
				if (test_bit(TGT_RETRY_CMDS, 
					&tq->q_flags)) {
					qla2x00_retry_command(vha,sp);
					retry_cmds++;
				} else 
					__add_to_retry_queue(vha,sp);
			}
		}
		spin_unlock_irqrestore(&vha->list_lock, flags);

		/* Clear all Target Unsuspended bits */
		for (t = 0; t < vha->max_targets; t++) {
			if ((tq = vha->otgt[t]) == NULL)
				continue;

			if (test_bit(TGT_RETRY_CMDS, &tq->q_flags))
				clear_bit(TGT_RETRY_CMDS, &tq->q_flags);
		}

		if (retry_cmds)
			qla2x00_next(vha);

		if (online_cnt > 0) {
			DEBUG3(printk(KERN_INFO "scsi%ld: dpc() (2) found scsi reqs "
					"to retry_q= %d, tgt retry cmds=%d\n",
					vha->host_no, online_cnt, retry_cmds));
		}
	}

	if (test_and_clear_bit(ISP_ABORT_NEEDED, &vha->dpc_flags)) {
		qla2x00_vp_abort_isp(vha);
	}

	if (test_and_clear_bit(RESET_MARKER_NEEDED, &vha->dpc_flags) &&
	    (!(test_and_set_bit(RESET_ACTIVE, &vha->dpc_flags)))) {
		DEBUG(printk("scsi(%ld): qla2x00_reset_marker()\n",
			vha->host_no));
		clear_bit(RESET_ACTIVE, &vha->dpc_flags);
	}

	if ((test_and_clear_bit(RELOGIN_NEEDED, &vha->dpc_flags)) &&
	    !test_bit(LOOP_RESYNC_NEEDED, &vha->dpc_flags) &&
	    atomic_read(&vha->loop_state) != LOOP_DOWN) {
		/* Everything is done in the parent's context */
	}
	if (test_and_clear_bit(LOOP_RESYNC_NEEDED, &vha->dpc_flags)) {
		qla2x00_vp_loop_resync(vha);
	}
	if (test_and_clear_bit(PORT_SCAN_NEEDED, &vha->dpc_flags)) {
		DEBUG(printk("scsi(%ld): scsi report rescan needed\n",vha->host_no));
	}
	LEAVE(__func__);	

}

void
qla2x00_do_dpc_all_vps(scsi_qla_host_t *phys_ha)
{
	unsigned long   flags;
	scsi_qla_host_t *vha = NULL, *tvha = NULL, *localvha = NULL;

	ENTER(__func__);
	DEBUG(printk("scsi(%ld): called %s() - dpc->flags 0x%lx\n",
			phys_ha->host_no, __func__, phys_ha->dpc_flags));

	/* skip if this is a virtual port */
	if (phys_ha->parent)
		return;

	/* skip if there are no virtual ports */
	if (list_empty(&phys_ha->vp_list))
		return;

	clear_bit(VPORT_ACTION_NEEDED, &phys_ha->dpc_flags);
	spin_lock_irqsave(&phys_ha->vport_lock, flags);
	list_for_each_entry_safe(vha, tvha, &phys_ha->vp_list, vp_list) {
		/* Vport in the process of deletion. */
		if(atomic_read(&vha->vp_delete_active) == 1)
			continue;
		qla2xxx_vha_get(vha);
		qla2xxx_vha_get(tvha);

		spin_unlock_irqrestore(&phys_ha->vport_lock, flags);
		qla2x00_do_dpc_vp(vha);
		spin_lock_irqsave(&phys_ha->vport_lock, flags);

		qla2xxx_vha_put(vha);
		localvha = tvha;
		/* Skip current vport entry as it is in the process of deletion. */
		if(atomic_read(&tvha->vp_delete_active) == 1)
			tvha = list_entry(tvha->vp_list.next, typeof(*tvha), vp_list);
		qla2xxx_vha_put(localvha);
	}
	spin_unlock_irqrestore(&phys_ha->vport_lock, flags);
	LEAVE(__func__);	
}

scsi_qla_host_t *
qla24xx_create_vhost(scsi_qla_host_t *phys_ha)
{
	scsi_qla_host_t *vha, *cur_ha;
	struct Scsi_Host *host;
	int	t;

	ENTER(__func__);
#ifdef __VMKERNEL_MODULE__
	/* 
	 * For virtual ports, the System starts assigning host IDs seperately. 
	 * To override the normal assignment, pass 0xfffe for bus number and 
	 * function number. 
	 */
	host = vmk_scsi_register(phys_ha->host->hostt, sizeof(scsi_qla_host_t),
			0xfffe, 0xfffe);

#else
	spin_lock_irq(&io_request_lock);
	host = scsi_register(phys_ha->host->hostt, sizeof(scsi_qla_host_t));
	spin_unlock_irq(&io_request_lock);
#endif //__VMKERNEL_MODULE__

	if (host == NULL) {
		printk(KERN_WARNING
		    "qla2xxx: Couldn't allocate host from scsi layer!\n");
		goto probe_failed;
	}

#ifdef __VMKERNEL_MODULE__
	vmk_scsi_register_uinfo(host, 0xfffe, 0xfffe, phys_ha);
  
	/* Start the fc transport */
        if (vmk_fc_attach_transport(host, &qla2xxx_transport_functions) != SUCCESS){
               printk("%s() : vmk_fc_attach_transport: failed!\n", __func__);
               scsi_unregister(host);
               // fatal error
	       goto probe_failed;
        }
#endif

	vha = (scsi_qla_host_t *)host->hostdata;

#if defined(CONFIG_VMNIX) && !defined(__VMKERNEL_MODULE__)
	host->bus = pdev->bus->number;
	host->devfn = pdev->devfn;
	host->devid = ha; 
#endif

	/* Clone the parent hba. */
	*vha = *phys_ha;

	/* New host Info */
	vha->host = host;
	vha->host_no = host->host_no;
	vha->parent = phys_ha;
	vha->init_done = 0;

	vha->ioctl = NULL;
	vha->ms_iocb = NULL;
	vha->ct_iu = NULL;
	vha->ioctl_mem = NULL;
	vha->ioctl_mem_size = 0;

#if LINUX_VERSION_CODE >= KERNEL_VERSION(2,4,4)
	scsi_set_pci_device(host, phys_ha->pdev);
#endif
	printk(KERN_INFO "scsi(%ld) : Creating VPort for host : %ld \n",
				vha->host_no, phys_ha->host_no);

	host->can_queue = max_srbs;
	set_bit(FDMI_REGISTER_NEEDED, &vha->fdmi_flags);
	set_bit(REGISTER_FC4_NEEDED, &vha->dpc_flags);
	atomic_set(&vha->loop_state, LOOP_DOWN);
	atomic_set(&vha->loop_down_timer, LOOP_DOWN_TIME);
	vha->device_flags |= DFLG_NO_CABLE;
	vha->dpc_flags = 0L;

	/* Initialized memory allocation pointers */
	INIT_LIST_HEAD(&vha->free_queue);
 	INIT_LIST_HEAD(&vha->done_queue);
	INIT_LIST_HEAD(&vha->retry_queue);
 	INIT_LIST_HEAD(&vha->scsi_retry_queue);
	INIT_LIST_HEAD(&vha->failover_queue);
	INIT_LIST_HEAD(&vha->pending_queue);

	INIT_LIST_HEAD(&vha->vp_fcports);
	INIT_LIST_HEAD(&vha->vp_list);
	INIT_LIST_HEAD(&vha->vp_del_list);
	INIT_LIST_HEAD(&vha->fcports);

	for (t = 0; t < MAX_FIBRE_DEVICES; t++)
		vha->otgt[t] = NULL;

	/* Allocated ioctl mem for the vport */
	if (qla2x00_mem_alloc(vha)) {
		printk(KERN_WARNING
		    "scsi(%d): [ERROR] Failed to allocate "
		    "memory for adapter\n", vha->host_no);
		qla2x00_mem_free(vha);

#ifdef __VMKERNEL_MODULE__
		vmk_fc_release_transport(host);
		scsi_unregister(host);
#else
		spin_lock_irq(&io_request_lock);
		scsi_unregister(host);
		spin_unlock_irq(&io_request_lock);
#endif //__VMKERNEL_MODULE__

		goto probe_failed;
	}

	/* load the F/W, read paramaters, and init the H/W */
	/* Is it needed since phys ha is used for vha as well ?? */
	init_MUTEX_LOCKED(&vha->mbx_intr_sem);

	/* Failover support required for vport ?? */
	if (ql2xfailover) {
		vha->flags.failover_enabled = 1;
	} else {
		vha->flags.failover_enabled = 0;
	}
	/*
	 * These locks are used to prevent more than one CPU
	 * from modifying the queue at the same time. The
 	 * higher level "host_lock" will reduce most
	 * contention for these locks.
	 */
#if defined(SH_HAS_HOST_LOCK)
	spin_lock_init(&ha->host_lock);
	host->host_lock = &ha->host_lock;
#endif
	spin_lock_init(&vha->mbx_bits_lock);
	spin_lock_init(&vha->mbx_reg_lock);
	spin_lock_init(&vha->mbx_q_lock);
	spin_lock_init(&vha->list_lock);
	vha->device_flags = 0;

#if defined(ISP2300)
	vha->dpc_notify = &qla2300_detect_sem;
#endif

	/* Currently, a single DPC thread is created */
#ifdef ENABLE_THIS_LATER 
	thread_pid = kernel_thread((int (*)(void *))qla2x00_do_dpc,
				(void *) vha, 0);
	if (thread_pid < 0) {
		printk(KERN_WARNING
		    "qla2x00(%ld): Failed to Create kernel" 
		    " thread thread_pid=%d\n",vha->host_no,
			thread_pid);
		DEBUG2(printk(KERN_WARNING
		    "qla2x00(%ld): Failed to Create kernel" 
		    " thread thread_pid=%d\n",hva->host_no,
			thread_pid));

		qla2x00_mem_free(vha);

		spin_lock_irq(&io_request_lock);
		scsi_unregister(host);
		spin_unlock_irq(&io_request_lock);

		goto probe_failed;
	}

#if defined(ISP2300)
	down(&qla2300_detect_sem);
#endif
#endif
	vha->dpc_notify = NULL;
	vha->next = NULL;

	host->can_queue = max_srbs;  /* default value:-MAX_SRBS(4096)  */
	host->cmd_per_lun = 1;
	host->select_queue_depths = qla2x00_select_queue_depth;
	host->n_io_port = 0xFF;
	host->base = 0;

#if MEMORY_MAPPED_IO
	host->base = (unsigned long)vha->mmio_address;
#endif

#if LINUX_VERSION_CODE >= KERNEL_VERSION(2,4,15)
	host->max_cmd_len = MAX_CMDSZ;
#endif
	/* fix: 07/31 host->max_lun = MAX_LUNS-1; */
	host->max_lun = phys_ha->max_luns;
	host->unique_id = phys_ha->instance;
	host->max_id = phys_ha->max_targets;

	/* set our host ID  (need to do something about our two IDs) */
	host->this_id = 255;

	/* Initialized the timer */
	START_TIMER(qla2x00_timer, vha, WATCH_INTERVAL);

	/* Insert new entry into the list of adapters */
	vha->next = NULL;

	/*
	 * To fix the issue of processing a parent's RSCN for the vport 
	 * before its SCR is complete.
	 */
	set_bit(VP_SCR_NEEDED, &vha->vp_flags);

	/* Modify it to use linux list helper routines */
	down(&instance_lock);
	vha->instance = find_first_zero_bit(host_instance_map, MAX_HBAS);
	if (vha->instance == MAX_HBAS)
		DEBUG9_10(printk("Host instance exhausted\n"));
	
	if (qla2x00_hostlist == NULL) {
		qla2x00_hostlist = vha;
	} else {
		cur_ha = qla2x00_hostlist;

		while (cur_ha->next != NULL) {
			cur_ha = cur_ha->next;
		}

		cur_ha->next = vha;
	}
	qla2xxx_vha_init(vha);
	set_bit(vha->instance, host_instance_map);
	num_hosts++;
	up(&instance_lock);


	LEAVE(__func__);	
	return (vha);

probe_failed:
	return (NULL);
}


#if	defined(__VMKERNEL_MODULE__)
/**
 * qla24xx_create_vport() - Adds a virtual port on the specified HBA.
 * @ha: HA context
 * @pvdata: pointer to buffer of virtual port parameters.
 *
 * Returns the virtual port handle, or MAX_MULTI_ID_NPORTS, if couldn't create.
 */
int
qla24xx_create_vport(scsi_qla_host_t *phys_ha, struct vport_data *pvdata)
{
	int ret = VPORT_OK; 
	uint16_t vp_id;
	scsi_qla_host_t *vha, *search_ha;
	struct timer_list	tmp_vp_timer;
	uint32_t tov = 20;
	unsigned long flags = 0;

	DECLARE_COMPLETION(vp_create_complete);
	ENTER(__func__);
	DEBUG9(printk("%s(): entered.\n", __func__));

	if (phys_ha == NULL) {
		DEBUG2(printk("%s(): Invalid Parameter \n", __func__));
		return VPORT_PARAMETER_ERR;
	}

	if (!(check_24xx_or_54xx_device_ids(phys_ha) || check_25xx_device_ids(phys_ha))) {
		DEBUG2(printk("Adapter does not support NPIV\n"));
		return VPORT_NORESOURCES;
	}

	if (qla24xx_find_vhost_by_name(phys_ha, &pvdata->port_name[0]) != NULL) {
                DEBUG2(printk("%s(): Virtual Port with WWPN: %02x%02x%02x%02x%02x%02x%02x%02x"
                                " already exists..\n",__func__,
                                pvdata->port_name[0], pvdata->port_name[1],
                                pvdata->port_name[2], pvdata->port_name[3],
                                 pvdata->port_name[4], pvdata->port_name[5],
                                pvdata->port_name[6], pvdata->port_name[7]));

		return VPORT_INVAL;
	}

	if (phys_ha->num_vhosts == phys_ha->max_multi_id_nports - 1) {
		DEBUG(printk("%s(): Maximum allowed Virtual Ports limit reached\n", 
					__func__));
		return VPORT_NORESOURCES;
	}

	for (search_ha = qla2x00_hostlist;
                (search_ha != NULL) && search_ha != phys_ha;
                search_ha = search_ha->next)
                continue;

        if (search_ha == NULL) {
                printk(KERN_INFO "%s(): Invalid Physical Host=%p specified",
                        __func__, phys_ha);
		return VPORT_INVAL;
        }

	
	vha = qla24xx_create_vhost(phys_ha);
	if (vha == NULL) {
		DEBUG2(printk("scsi(%ld): Unable to create vhost.\n",
			phys_ha->host_no));
		return VPORT_ERROR;
	}

	vha->vp_create_wait = &vp_create_complete;
	DEBUG2(printk(KERN_INFO "scsi(%ld): WWPN: "
	    	"%02x%02x%02x%02x%02x%02x%02x%02x "
	    	"WWNN: %02x%02x%02x%02x%02x%02x%02x%02x\n", vha->host_no,
		pvdata->port_name[0], pvdata->port_name[1],
		pvdata->port_name[2], pvdata->port_name[3],
		pvdata->port_name[4], pvdata->port_name[5],
		pvdata->port_name[6], pvdata->port_name[7],
	 	pvdata->node_name[0], pvdata->node_name[1],
		pvdata->node_name[2], pvdata->node_name[3],
		pvdata->node_name[4], pvdata->node_name[5],
		pvdata->node_name[6], pvdata->node_name[7]));	

	/* Add WWN to virtual port */
	memcpy(vha->node_name, pvdata->node_name, WWN_SIZE);
	memcpy(vha->port_name, pvdata->port_name, WWN_SIZE);

        /* get the wwns when they are available */
	fc_host_node_name(vha->host) = wwn_to_u64(vha->node_name);
	fc_host_port_name(vha->host) = wwn_to_u64(vha->port_name);

	vp_id = qla24xx_allocate_vp_id(vha);
	if (vp_id >= phys_ha->max_multi_id_nports) {
		DEBUG2(printk("scsi(%ld): Couldn't allocate vp_id.\n",
			phys_ha->host_no));
		return VPORT_NORESOURCES;
	}

	vha->qfull_retry_count = qfull_retry_count;
	vha->qfull_retry_delay = qfull_retry_delay;

	/* Initialized vport states */
	atomic_set(&vha->vp_state, VP_OFFLINE);
	vha->vp_err_state=  VP_ERR_PORTDWN;
	vha->vp_prev_err_state=  VP_ERR_UNKWN;
	/* Check if physical ha port is Up */

	/*
	 * If the physical link is up, initialize vport right away ,  if not
	 * schedule it for a discovery after the links comes up.
	 */
	if (atomic_read(&phys_ha->loop_state) == LOOP_DOWN  ||
		atomic_read(&phys_ha->loop_state) == LOOP_DEAD) {
		if ((pvdata->options == VPORT_OPT_AUTORETRY)) {
			printk(KERN_INFO "scsi(%ld): Virtual port id=%d AUTORETRY\n", 
						phys_ha->host_no, vha->vp_idx);
			atomic_set(&vha->vp_state, VP_OFFLINE);
		} else {
			/* Don't retry or attempt login of this virtual port */
			printk(KERN_INFO "scsi(%ld): Physical port down for VPort" 
					"id : %d\n", phys_ha->host_no, vha->vp_idx);
			atomic_set(&vha->vp_state, VP_FAILED);
		}
	}
	if (atomic_read(&vha->vp_state) != VP_FAILED) {

		/* Init timer */
		init_timer(&tmp_vp_timer);
		tmp_vp_timer.data = (unsigned long)vha->vp_create_wait;
		tmp_vp_timer.function =
		    (void (*)(unsigned long))qla2x00_vp_cmpl_timeout;
		tmp_vp_timer.expires = jiffies + MAX_LOOP_TIMEOUT * HZ;

		if (qla24xx_enable_vp(vha)) {
			// qla2x00_mem_free(vha);
			/* 
			 * Resources for the Vport are allocated.
			 * Appropriate err_state is set. Return OK so that
			 * the upper layer comes down for a delete, and free
			 * up the resources. 
			 */
			DEBUG2(printk("scsi(%ld): Failure enabling vport ... retrying.\n",
			phys_ha->host_no));
			pvdata->vport_shost = vha->host;
			return VPORT_OK;
		}

		add_timer(&tmp_vp_timer);

		/* Wait for the LIP to complete */
		wait_for_completion(vha->vp_create_wait);

		/* delete the timer */
		del_timer(&tmp_vp_timer);

		printk(KERN_INFO "scsi(%ld) : Virtual port Created on host=%ld with ID=%d\n",
				vha->host_no, phys_ha->host_no, vha->vp_idx);


		if (phys_ha->ioctl->flags &
				IOCTL_AEN_TRACKING_ENABLE) {
			/* Update AEN queue. */
			spin_lock_irqsave(&phys_ha->hardware_lock, flags);
			qla2x00_enqueue_aen(phys_ha,
					MBA_PORT_UPDATE, NULL);
			spin_unlock_irqrestore(&phys_ha->hardware_lock, flags);
		}

		/* Add a timer to timeout if completion fails to happen. */
	} else {
		atomic_set(&vha->loop_state, LOOP_DEAD);
		printk(KERN_INFO "scsi() : Virtual port id%d DISABLED because parent is OFFLINE.\n",
				 vha->vp_idx);
	}
	
	vha->init_done = 1;
	pvdata->vport_shost = vha->host;
	DEBUG9(printk("%s(): exiting. ret=%d.\n", __func__, ret));
	LEAVE(__func__);	
	return (ret);
}

/**
 * qla24xx_delete_vport() -  Removes the virtual fabric port's from the specified HBA  
 * @ha: HA context
 *		
 * Returns error code.
 */
int
qla24xx_delete_vport(scsi_qla_host_t *vha)
{
	int ret = VPORT_OK;
	scsi_qla_host_t *prev_ha; 
	scsi_qla_host_t *next_ha; 
	scsi_qla_host_t *nha; 
	scsi_qla_host_t *ha = vha->parent;
	unsigned long flags = 0;

	ENTER(__func__);
	DEBUG9(printk("%s: entered.\n", __func__));

	atomic_set(&vha->vp_delete_active, 1);
	
	/* Return Invalid parameter if the pointer is null, or a physical host. */
	if (vha == NULL || ha == NULL) {
		DEBUG2(printk("%s: Invalid Parameter \n", __func__));
		return VPORT_PARAMETER_ERR;
	}
	
	DEBUG(printk("Disabling virtual port host_no = %d..\n", vha->host_no));
	/* If the upper layer requests for it, disable it in any case */
	qla24xx_disable_vp(vha);

	DEBUG(printk("Remove from hostlist - virtual port host_no = %d..\n", vha->host_no));
	/* Modify it to use linux list helper routines */
	/* Access to hostlist should also be protected. */
	down(&instance_lock);
	/* Remove vha from hostlist. */
	if (qla2x00_hostlist == vha)  {
		qla2x00_hostlist = vha->next;
	} else {
		prev_ha = qla2x00_hostlist;
		while (prev_ha != NULL) {
			next_ha = prev_ha->next;
			if (next_ha == vha) {
				prev_ha->next = vha->next;
				num_hosts--;
				break;
			}
			prev_ha = next_ha;
		}
	}

	clear_bit(vha->instance, host_instance_map);
	up(&instance_lock);

	DEBUG(printk("Disabling timer for vport..\n"));
	if (vha->timer_active) {
		STOP_TIMER(qla2x00_timer,vha)
	}

	spin_lock_irqsave(&ha->vport_lock, flags);
	qla2xxx_vha_put(vha);
	spin_unlock_irqrestore(&ha->vport_lock, flags);

	if (ha->ioctl->flags &
			IOCTL_AEN_TRACKING_ENABLE) {
		/* Update AEN queue. */
		spin_lock_irqsave(&ha->hardware_lock, flags);
		qla2x00_enqueue_aen(ha,
				MBA_PORT_UPDATE, NULL);
		spin_unlock_irqrestore(&ha->hardware_lock, flags);
	}

	DEBUG9(printk("%s: exiting. ret=%d.\n", __func__, ret));
	LEAVE(__func__);	
	return (ret);
}


/**
 * qla24xx_vport_info() - Query information about virtual fabric port
 * @ha: HA context
 * @pvinfo: pointer to buffer of information about virtual port.
 *
 *		
 * Returns error code.
 */
int
qla24xx_vport_info(scsi_qla_host_t *ha, struct vport_info *pvinfo)
{
	int ret = VPORT_OK; 
	struct list_head *list;
	fc_port_t *fcport = NULL;

	ENTER(__func__);
	DEBUG9(printk("%s: entered.\n", __func__));

	/* Invalid hostdata pointer */
	if (ha == NULL) {
		DEBUG(printk("%s: Invalid Parameter \n", __func__));
		return VPORT_PARAMETER_ERR;
	}

	/* Non NPIV HBAs */
	if (!(check_24xx_or_54xx_device_ids(ha) || check_25xx_device_ids(ha))) {
		pvinfo->linktype = VPORT_TYPE_PHYSICAL;
		pvinfo->state = VPORT_STATE_OFFLINE;
		pvinfo->fail_reason = VPORT_FAIL_ADAP_NORESOURCES;
		pvinfo->prev_fail_reason = VPORT_FAIL_UNKNOWN;
		/* Values below based on API 1.00 */
		pvinfo->vports_max = VPORT_CNT_INVALID;
		pvinfo->vports_inuse = VPORT_CNT_INVALID;
		printk(KERN_INFO "scsi(%ld) : Non NPIV host\n", ha->host_no);
		return ret;
	}

	/* Non-NPIV fabric */
	if (!(ha->switch_cap & FLOGI_MID_SUPPORT)) {
		printk(KERN_INFO "scsi(%ld) : Non NPIV Fabric, "
			"Capability 0x%x\n", ha->host_no, ha->switch_cap);
		pvinfo->linktype = VPORT_TYPE_PHYSICAL;
		pvinfo->state = VPORT_STATE_OFFLINE;
		pvinfo->fail_reason = VPORT_FAIL_FAB_UNSUPPORTED;
		pvinfo->prev_fail_reason = VPORT_FAIL_UNKNOWN;
		pvinfo->vports_max = ha->max_multi_id_nports - 1;
		pvinfo->vports_inuse = ha->num_vhosts;
		return ret;
	}

	memset(pvinfo, 0, sizeof (vport_info));
	pvinfo->api_versions = VPORT_API_VERSION;

	if (ha->parent) {
		/* Virtual Port */
		pvinfo->linktype = VPORT_TYPE_VIRTUAL;
		switch ((int)atomic_read(&ha->vp_state)) {
			case VP_OFFLINE:
				pvinfo->state = VPORT_STATE_OFFLINE;
				break;
			case VP_ACTIVE:
				pvinfo->state = VPORT_STATE_ACTIVE;
				break;
			default:
				pvinfo->state = VPORT_STATE_FAILED;
				break;
		}
	} else {
		/* Physical Port */
		pvinfo->linktype = VPORT_TYPE_PHYSICAL;
		/* Physical ports only have two states, ACTIVE or OFFLINE 
		 * based of the fact if vports can be created at the given 
		 * point of time. 
		 */
		switch ((int)atomic_read(&ha->loop_state)) {
			case LOOP_DOWN:
			case LOOP_DEAD:
			case LOOP_TIMEOUT:
				pvinfo->state = VPORT_STATE_OFFLINE;
				break;
			default:
				pvinfo->state = VPORT_STATE_ACTIVE;
				break;
		}

		/* Max supported vports for fabric or loop enviroment. */
		if (ha->current_topology == ISP_CFG_F) {
			pvinfo->vports_max = ha->max_multi_id_nports - 1;
		} else {
			/* For now, we set it to 127 cause that's what we support */
			pvinfo->vports_max = ha->max_multi_id_nports - 1;
		}

		pvinfo->vports_inuse = ha->num_vhosts;

		/* Max Remote Port Instances */
		pvinfo->rpi_max = MAX_RPI;
 		list_for_each_entry(fcport, &ha->fcports, list) {
			/* 
			 * The rpi resource is considered free if 
			 * it has been deleted by mid-layer. 
			 */
			if (fcport->flags & FC_DEVICE_DELETED) 
				continue;
			/* Only consider target devices. */
			if (fcport->port_type != FCT_TARGET)
				continue;
			ha->free_rpi--;
		}
		pvinfo->rpi_inuse = MAX_RPI - ha->free_rpi;
	}

	/* World wide names. */
	memcpy(pvinfo->node_name , ha->node_name, WWN_SIZE);
	memcpy(pvinfo->port_name, ha->port_name, WWN_SIZE);

	/* Fail reasons. */
	switch (ha->vp_err_state) {
		case VP_ERR_UNKWN:
			pvinfo->fail_reason = VPORT_FAIL_UNKNOWN;
			break;
		case VP_ERR_PORTDWN:
			pvinfo->fail_reason = VPORT_FAIL_LINKDOWN;
			break;
		case VP_ERR_FAB_UNSUPPORTED:
			pvinfo->fail_reason = VPORT_FAIL_FAB_UNSUPPORTED;
			break;
		case VP_ERR_FAB_NORESOURCES:
			pvinfo->fail_reason = VPORT_FAIL_FAB_NORESOURCES;
			break;
		case VP_ERR_FAB_LOGOUT:
			pvinfo->fail_reason = VPORT_FAIL_FAB_LOGOUT;
			break;
		case VP_ERR_ADAP_NORESOURCES:
			pvinfo->fail_reason = VPORT_FAIL_ADAP_NORESOURCES;
			break;
		default:
			pvinfo->fail_reason = VPORT_FAIL_UNKNOWN;
			break;
	}

	switch (ha->vp_prev_err_state) {
		case VP_ERR_UNKWN:
			pvinfo->prev_fail_reason = VPORT_FAIL_UNKNOWN;
			break;
		case VP_ERR_PORTDWN:
			pvinfo->prev_fail_reason = VPORT_FAIL_LINKDOWN;
			break;
		case VP_ERR_FAB_UNSUPPORTED:
			pvinfo->prev_fail_reason = VPORT_FAIL_FAB_UNSUPPORTED;
			break;
		case VP_ERR_FAB_NORESOURCES:
			pvinfo->prev_fail_reason = VPORT_FAIL_FAB_NORESOURCES;
			break;
		case VP_ERR_FAB_LOGOUT:
			pvinfo->prev_fail_reason = VPORT_FAIL_FAB_LOGOUT;
			break;
		case VP_ERR_ADAP_NORESOURCES:
			pvinfo->prev_fail_reason = VPORT_FAIL_ADAP_NORESOURCES;
			break;
		default:
			pvinfo->prev_fail_reason = VPORT_FAIL_UNKNOWN;
			break;
	}

	DEBUG9(printk(KERN_INFO "%s: exiting. ret=%d.\n", __func__, ret));
	LEAVE(__func__);	
	return (ret);
}

/*
 * qla24xx_getinfo_vport() -  Query information about virtual fabric port
 * @ha: HA context
 * @vp_info: pointer to buffer of information about virtual port.
 * instance : instance of virtual port from 0 to MAX_MULTI_ID_NPORTS.
 * For Sansurfer Use only.
 *             
 * Returns error code.
 */
uint32_t
qla24xx_getinfo_vport(scsi_qla_host_t *ha, fc_vport_info_t *vp_info , uint32_t  instance        )
{
	unsigned long flags;
       scsi_qla_host_t *vha;
       int     i = 0;

	   if (instance >= ha->max_multi_id_nports) {
               DEBUG(printk(KERN_INFO "instance number out of range...\n"));
               return VP_RET_CODE_FATAL;
       }

       memset(vp_info, 0, sizeof (fc_vport_info_t));
       /* Always return these numbers */
	   vp_info->free_vpids = ha->max_multi_id_nports - ha->num_vhosts - 1;
       vp_info->used_vpids =  ha->num_vhosts; 
	   spin_lock_irqsave(&ha->vport_lock, flags);
       list_for_each_entry(vha, &ha->vp_list, vp_list) {
               if (i==instance) {
                               vp_info->vp_id = vha->vp_idx; 
                               vp_info->vp_state = atomic_read(&vha->vp_state);
               //              vp_info->vp_bound = vha->vsan != NULL;
                               memcpy(vp_info->node_name , vha->node_name, WWN_SIZE);
                               memcpy(vp_info->port_name, vha->port_name, WWN_SIZE);
							   spin_unlock_irqrestore(&ha->vport_lock, flags);
                               return VPORT_OK;
               }
               i++;
       }
	   spin_unlock_irqrestore(&ha->vport_lock, flags);
       DEBUG(printk(KERN_INFO "instance number not found..\n"));
       return VP_RET_CODE_FATAL;
}


/**
 * qla24xx_tgt_remove() - Remove specified target from virtual port target list
 * @ha: HA context.
 * @pvinfo: target ID of target to be removed.
 *
 *		
 * Returns error code.
 */
int
qla24xx_tgt_remove(scsi_qla_host_t *vha, uint32 channel, uint32 tgtid)
{
	int 		ret = VPORT_OK; 
	os_tgt_t	*tq;
	int		t;
	srb_t		*rp;
	unsigned long	flags;
	struct list_head *list, *temp;
	fc_lun_t	*fclun, *fcltemp;
	fc_port_t	*fcport, *fcptemp;
	ENTER(__func__);
	DEBUG9(printk("%s: entered.\n", __func__));

	for (t = 0; t < MAX_FIBRE_DEVICES; t++) {
		if ((tq = TGT_Q(vha, t)) == NULL)
			continue;
		if (tq->id == tgtid)
			break;
	}
	if (!tq) {
		printk(KERN_INFO"%s(%d) Couldn't find target with channel=%d and tgtid=%d\n",
				__func__, vha->host_no, channel, tgtid);
		ret = VPORT_INVAL;
		goto invalid_target_id;
	}

	/* Obtain the corresponding fcport entry */
  	list_for_each_entry_safe(fcport, fcptemp, &vha->vp_fcports, vp_fcport) {

		if (fcport->dev_id == tgtid)
			break;
	}
	if (!fcport) {
		printk(KERN_INFO"%s(%d) Couldn't find target with channel=%d and tgtid=%d\n",
				__func__, vha->host_no, channel, tgtid);
		ret = VPORT_INVAL;
		goto invalid_target_id;
	}

	fcport->flags |= FC_DEVICE_DELETED;

	qla2x00_fabric_logout(vha, fcport->loop_id, fcport->d_id.b.domain,
			 fcport->d_id.b.area, fcport->d_id.b.al_pa);

	DEBUG9(printk("%s: exiting. ret=%d.\n", __func__, ret));
	LEAVE(__func__);	

invalid_target_id:
	return (ret);
}
#endif // __VMKERNEL_MODULE__
