/******************************************************************************
 *                  QLOGIC LINUX SOFTWARE                                     *
 *                                                                            *
 * QLogic ISP4xxx device driver for Linux 2.5.x                               *
 * Copyright (C) 2004 Qlogic Corporation                                      *
 * (www.qlogic.com)                                                           *
 *                                                                            *
 * This program is free software; you can redistribute it and/or modify it    *
 * under the terms of the GNU General Public License as published by the      *
 * Free Software Foundation; either version 2, or (at your option) any        *
 * later version.                                                             *
 *                                                                            *
 * This program is distributed in the hope that it will be useful, but        *
 * WITHOUT ANY WARRANTY; without even the implied warranty of                 *
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU          *
 * General Public License for more details.                                   *
 *                                                                            *
 ******************************************************************************
 *             Please see release.txt for revision history.                   *
 *                                                                            *
 ******************************************************************************
 * Function Table of Contents:
 *      qla4xxx_take_semaphore
 *      qla4xxx_clear_semaphore
 ****************************************************************************/



//
// Semaphore register definitions
//
typedef enum
{
	SEM_HW_LOCK
	, SEM_GPO
	, SEM_SDRAM_INIT
	, SEM_PHY_GBIC
	, SEM_NVRAM
	, SEM_FLASH

	, SEM_COUNT // Not a real semaphore, just indicates how many there are
} ISP4010_SEMAPHORE;

#ifdef QLA4010
#define SEM_AVAILABLE        	0x00
#define SEM_OWNER_FIRMWARE   	0x01
#define SEM_OWNER_STORAGE    	0x02
#define SEM_OWNER_NETWORK    	0x03


//
// Private Semaphore definitions
//
#define SEM_MASK  0x3

typedef struct
{
	UINT32   semId;
	UINT32   semShift;
} isp4010SemInfo_t;

isp4010SemInfo_t semInfo[] = {
{ SEM_HW_LOCK,     4}
, { SEM_GPO,         6}
, { SEM_SDRAM_INIT,  8}
, { SEM_PHY_GBIC,   10}
, { SEM_NVRAM,      12}
, { SEM_FLASH,      14}
};

#define  SEM_READ(ha, semId)   ((RD_REG_DWORD(&ha->reg->NVRAM) >> semInfo[semId].semShift) & SEM_MASK)
#define  SEM_WRITE(ha, semId, owner)  (WRT_REG_DWORD(&ha->reg->NVRAM, (SEM_MASK << 16 << semInfo[semId].semShift) | (owner << semInfo[semId].semShift)))

/**************************************************************************
 * qla4xxx_take_semaphore
 *	This routine acquires the specified semaphore for the iSCSI
 *	storage driver.
 *
 * Input:
 * 	ha - Pointer to host adapter structure.
 *	sem - Indicates which semaphore.
 *	wait_flag - specifies type of wait to acquire semaphore
 *		    WAIT_FOREVER = wait indefinitely
 *		    TIMED_WAIT = wait for a specified amout of time
 *	            NO_WAIT = try once to acquire semaphore
 *
 * Returns:
 *	QLA_SUCCESS - Successfully acquired semaphore
 *	QLA_ERROR   - Failed to acquire semaphore
 *
 * Context:
 *	?? context.
 **************************************************************************/
#define NO_WAIT		0
#define WAIT_FOREVER	1
#define TIMED_WAIT	2
inline uint8_t
qla4xxx_take_semaphore(scsi_qla_host_t *ha, uint32_t sem, uint8_t wait_flag)
{
	uint32_t wait_time = SEMAPHORE_TOV;

	for (SEM_WRITE(ha, sem, SEM_OWNER_STORAGE);
	     (SEM_READ(ha, sem) != SEM_OWNER_STORAGE) && (wait_time--);
	     SEM_WRITE(ha, sem, SEM_OWNER_STORAGE))
	{
		if (wait_flag == NO_WAIT)
			return (QLA_ERROR);

		set_current_state(TASK_INTERRUPTIBLE);
		schedule_timeout(1 * HZ);
	}

	if (wait_time)
                return (QLA_SUCCESS);
	else
		return (QLA_ERROR);
}

/**************************************************************************
 * qla4xxx_clear_semaphore
 *	This routine restores the specified semaphore to the available
 *	state.
 *
 * Input:
 * 	ha - Pointer to host adapter structure.
 *	sem - Indicates which semaphore.
 *
 * Returns:
 *	QLA_SUCCESS - Successfully restored semaphore
 *	QLA_ERROR   - Failed to restore semaphore
 *
 * Context:
 *	?? context.
 **************************************************************************/
inline void
qla4xxx_clear_semaphore(scsi_qla_host_t *ha, uint32_t sem)
{
	if (SEM_READ(ha, sem) == SEM_OWNER_STORAGE)
                SEM_WRITE(ha, sem, SEM_AVAILABLE);
}

#endif

#ifdef QLA4000
#define qla4xxx_take_semaphore(ha,sem,wait_forever)   QLA_SUCCESS
#define qla4xxx_clear_semaphore(ha,sem)
#endif

