/******************************************************************************
 *                  QLOGIC LINUX SOFTWARE                                     *
 *                                                                            *
 * QLogic ISP4xxx device driver for Linux 2.4.x                               *
 * Copyright (C) 2002 Qlogic Corporation                                      *
 * (www.qlogic.com)                                                           *
 *                                                                            *
 * This program is free software; you can redistribute it and/or modify it    *
 * under the terms of the GNU General Public License as published by the      *
 * Free Software Foundation; either version 2, or (at your option) any        *
 * later version.                                                             *
 *                                                                            *
 * This program is distributed in the hope that it will be useful, but        *
 * WITHOUT ANY WARRANTY; without even the implied warranty of                 *
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU          *
 * General Public License for more details.                                   *
 *                                                                            *
 ******************************************************************************
 *             Please see release.txt for revision history.                   *
 *                                                                            *
 ******************************************************************************
 * Function Table of Contents:
 *        qla4xxx_detect
 *        qla4xxx_get_token
 *        qla4xxx_setup
 *        qla4xxx_alloc_dma_memory
 *        qla4xxx_free_dma_memory
 *        qla4xxx_resize_dma_buf
 *        qla4xxx_alloc_srb_pool
 *        qla4xxx_free_srb_pool
 *        qla4xxx_mem_alloc
 *        qla4xxx_free_lun
 *        qla4xxx_free_ddb
 *        qla4xxx_free_ddb_list
 *        qla4xxx_mem_free
 *        qla4xxx_register_resources
 *        qla4xxx_select_queue_depth
 *        qla4xxx_set_info
 *        qla4xxx_proc_dump_srb_info
 *        qla4xxx_proc_info
 *        qla4xxx_release
 *        qla4xxx_info
 *        qla4xxx_biosparam
 *        qla4xxx_init_rings
 *        qla4xxx_initialize_fw_cb
 *        qla4xxx_get_fw_version
 *        qla4xxx_validate_mac_address
 *        qla4xxx_get_firmware_state
 *        qla4xxx_init_firmware
 *        qla4xxx_login_device
 *        qla4xxx_logout_device
 *        qla4xxx_relogin_device
 *        qla4xxx_send_internal_scsi_passthru
 *        qla4xxx_send_inquiry_cmd
 *        qla4xxx_send_report_luns_cmd
 *        qla4xxx_get_ddb_entry
 *        qla4xxx_set_ddb_entry
 *        qla4xxx_update_ddb_entry
 *        qla4xxx_alloc_lun
 *        qla4xxx_discover_target_luns
 *        qla4xxx_map_targets_to_ddbs
 *        qla4xxx_alloc_ddb
 *        qla4xxx_build_ddb_list
 *        qla4xxx_initialize_ddb_list
 *        qla4xxx_reinitialize_ddb_list
 *        qla4010_start_firmware
 *        qla4xxx_initialize_adapter
 *        qla4xxx_get_req_pkt
 *        qla4xxx_send_marker_iocb
 *        add_to_active_array
 *        del_from_active_array
 *        qla4xxx_send_command_to_isp
 *        qla4xxx_lookup_lun_handle
 *        qla4xxx_lookup_ddb_by_SCSIID
 *        qla4xxx_lookup_ddb_by_fw_index
 *        qla4xxx_complete_request
 *        qla4xxx_queuecommand
 *        qla4xxx_extend_timeout
 *        qla4xxx_start_io
 *        qla4xxx_os_cmd_timeout
 *        qla4xxx_add_timer_to_cmd
 *        qla4xxx_delete_timer_from_cmd
 *        qla4xxx_suspend_lun
 *        qla4xxx_check_sense
 *        qla4xxx_status_entry
 *        qla4xxx_process_response_queue
 *        qla4xxx_interrupt_service_routine
 *        qla4xxx_mailbox_command
 *        qla4xxx_process_ddb_changed
 *        qla4xxx_process_aen
 *        qla4xxx_intr_handler
 *        qla4xxx_timer
 *        qla4xxx_do_dpc
 *        qla4xxx_panic
 *        qla4xxx_eh_wait_on_command
 *        qla4xxx_eh_abort
 *        qla4000_hard_reset
 *        qla4000_reset_firmware
 *        qla4010_soft_reset
 *        qla4xxx_soft_reset
 *        qla4xxx_hard_reset
 *        qla4xxx_cmd_wait
 *        qla4xxx_recover_adapter
 *        qla4xxx_eh_wait_for_active_target_commands
 *        qla4xxx_reset_lun
 *        qla4xxx_eh_device_reset
 *        qla4xxx_eh_bus_reset
 *        qla4xxx_reset_target
 *        qla4xxx_flush_active_srbs
 *        qla4xxx_eh_host_reset
 *        qla4xxx_enable_intrs
 *        qla4xxx_disable_intrs
 *        qla4xxx_get_adapter_handle
 *        qla4xxx_get_hba_count
 *
 ****************************************************************************/

#include <linux/config.h>
#include <linux/module.h>
#include <stdarg.h>
#include <asm/io.h>
#include <asm/irq.h>
#include <asm/segment.h>
#include <asm/byteorder.h>
#include <linux/spinlock.h>

#ifdef __VMKERNEL_MODULE__
#if !defined(LINUX_VERSION_CODE)
#include <linux/version.h>
#endif  /* LINUX_VERSION_CODE not defined */
#endif


#include <linux/types.h>
#include <linux/string.h>
#include <linux/errno.h>
#include <linux/kernel.h>
#include <linux/delay.h>
#include <linux/timer.h>
#include <linux/sched.h>
#include <linux/pci.h>
#include <linux/proc_fs.h>
#include <linux/blk.h>
#include <linux/tqueue.h>
#include <linux/wait.h>

#include <linux/init.h>

#ifdef __VMKERNEL_MODULE__
#include <scsi.h>
#include <sd.h>
#include <hosts.h>
#else
#include <../drivers/scsi/scsi.h>
#include <../drivers/scsi/sd.h>
#include <../drivers/scsi/hosts.h>
#endif

#include <linux/stat.h>
#include <linux/slab.h>  /* in lieu of malloc.h */

#define __KERNEL_SYSCALLS__
#include <linux/unistd.h>
#include <linux/smp_lock.h>
#include <asm/system.h>
#define SHUTDOWN_SIGS	(sigmask(SIGHUP))


#ifdef __VMKERNEL_MODULE__
#include "vmklinux_dist.h"
#include "../../public/vm_assert.h"
#endif

void code_start(void)
{
}
#include "qlisioct.h"
#include "qla4x_fw.h"
#include "ql4nvram.h"
#include "qla4xxx.h"
#include "ql4print.h"
#include "ql4lists.h"
#include "ql4isns.h"

/****************************************************************************/
/*    			     Function Prototypes			    */
/****************************************************************************/

/*
 *  Linux - iSCSI Driver Interface Function Prototypes.
 */
int         qla4xxx_detect(Scsi_Host_Template *);
int         qla4xxx_proc_info( char *, char **, off_t, int, int, int);
int         qla4xxx_release(struct Scsi_Host *);
const char *qla4xxx_info(struct Scsi_Host *);
int         qla4xxx_queuecommand(Scsi_Cmnd *, void (* done)(Scsi_Cmnd *));
int         qla4xxx_eh_abort(Scsi_Cmnd *);
int         qla4xxx_eh_device_reset(Scsi_Cmnd *);
int         qla4xxx_eh_bus_reset(Scsi_Cmnd *);
int         qla4xxx_eh_host_reset(Scsi_Cmnd *);
int         qla4xxx_biosparam(Disk *, kdev_t, int[]);
void        qla4xxx_intr_handler(int, void *, struct pt_regs *);
void        qla4xxx_select_queue_depth(struct Scsi_Host *, Scsi_Device *);
int         qla4xxx_set_info(char *, int , struct Scsi_Host *);
void        qla4xxx_timer(unsigned long data);

/*
 * Routines referenced from other modules
 */
extern int  qla4xxx_ioctl(Scsi_Device *dev, int cmd, void *arg);
extern void qla4xxx_ioctl_sem_init (scsi_qla_host_t *ha);
extern void qla4xxx_scsi_pass_done(Scsi_Cmnd *cmd);
extern int  apidev_init(struct Scsi_Host *host);
extern int  apidev_cleanup(void);

/*
 *  QLogic Driver Support Function Prototypes.
 */
STATIC uint8_t  qla4xxx_initialize_adapter(scsi_qla_host_t *ha,
					   uint8_t renew_ddb_list);
#define PRESERVE_DDB_LIST	0
#define REBUILD_DDB_LIST	1
#define NO_SOFT_RESET		0
#define FORCE_SOFT_RESET 	1

STATIC void     qla4xxx_mem_free(scsi_qla_host_t *ha);
static inline void __qla4xxx_enable_intrs(scsi_qla_host_t *);
static inline void __qla4xxx_disable_intrs(scsi_qla_host_t *);
static inline void qla4xxx_enable_intrs(scsi_qla_host_t *);
static inline void qla4xxx_disable_intrs(scsi_qla_host_t *);
STATIC void  qla4xxx_interrupt_service_routine(scsi_qla_host_t *ha,
					       uint32_t intr_status);
STATIC uint8_t  qla4xxx_mem_alloc(scsi_qla_host_t *ha);
STATIC uint8_t  qla4xxx_register_resources(scsi_qla_host_t *);
void qla4xxx_do_dpc(unsigned long p);
static uint8_t qla4xxx_complete_request(scsi_qla_host_t *ha, srb_t *srb);

inline lun_entry_t *qla4xxx_lookup_lun_handle(scsi_qla_host_t *ha,
					      ddb_entry_t *ddb_entry,
					      uint16_t lun);
inline ddb_entry_t *qla4xxx_lookup_ddb_by_SCSIID(scsi_qla_host_t *ha,
						 uint32_t bus,
						 uint32_t target);
inline ddb_entry_t *qla4xxx_lookup_ddb_by_fw_index(scsi_qla_host_t *ha,
						   uint32_t fw_ddb_index);
inline ddb_entry_t *qla4xxx_lookup_ddb_by_isns_info(scsi_qla_host_t *ha,
						    uint8_t *ip_addr,
						    uint8_t *alias);
uint8_t qla4xxx_reset_lun(scsi_qla_host_t *ha,
			  ddb_entry_t *ddb_entry,
			  lun_entry_t *lun_entry);
uint8_t qla4xxx_reset_target(scsi_qla_host_t *ha, ddb_entry_t *ddb_entry);
void    qla4xxx_start_io(scsi_qla_host_t *ha);
inline uint8_t qla4xxx_soft_reset(scsi_qla_host_t *ha);
STATIC void qla4xxx_flush_active_srbs(scsi_qla_host_t *ha);
uint8_t qla4xxx_recover_adapter(scsi_qla_host_t *ha, uint8_t renew_ddb_list);
static void qla4xxx_extend_timeout(Scsi_Cmnd *cmd, int timeout);
#if !defined(MODULE)
static int __init qla4xxx_setup (char *s);
#else
void qla4xxx_setup(char *s);
#endif
void qla4xxx_display_config(void);
void qla4xxx_process_aen(scsi_qla_host_t *ha, uint8_t flush_ddb_chg_aens);
srb_t *del_from_active_array(scsi_qla_host_t *ha, uint32_t index);
#ifdef QLA4010
uint8_t qla4010_soft_reset(scsi_qla_host_t *ha);
uint8_t qla4010_start_firmware(scsi_qla_host_t *ha, uint8_t force_soft_reset);
#endif
#if STOP_ON_ERROR
static void qla4xxx_panic(char *cp, scsi_qla_host_t *ha);
#endif
void qla4xxx_ioctl_error_recovery(scsi_qla_host_t *ha);
#if 1 || __VMKERNEL_MODULE__
static uint8_t qla4xxx_map_targets_to_ddbs(scsi_qla_host_t *ha, int assign_target_numbers);
#endif
#if 1 || __VMKERNEL_MODULE__
static void qla4xxx_relogin_ddb_list(scsi_qla_host_t *ha);
static void qla4xxx_wake_the_dead(scsi_qla_host_t *ha);
#endif


/****************************************************************************/
/*    					Global Data			    */
/****************************************************************************/


/*
 * Command line options
 *---------------------------------------------------------------------------*/
static int ql4xdiscoverywait=15;

#ifdef __VMKERNEL_MODULE__
static int ql4xportdownretrycount= 16;
#else
static int ql4xportdownretrycount= 60;
#endif
static int ql4xcmdretrycount = 20;
#ifdef __VMKERNEL_MODULE__
static int ql4xmaxqdepth = 2;
static int ql4xdeftimeout = 30;
#else
static int ql4xmaxqdepth = 2;
#endif
static char *ql4xdevconf = NULL;
static int displayConfig = 0;
static int extended_error_logging = 0;	/* 0 = off, 1 = log errors, 2 = debug logging */
#ifdef QLA4000
static int ql4xcmdtimeout = 5;
#endif

#ifdef MODULE
static char *ql4xopts = NULL;

/*
 * Just in case someone uses commas to separate items on the insmod
 * command line, we define a dummy buffer here to avoid having insmod
 * write wild stuff into our code segment
 */
static char dummy_buffer[60] =
"Please don't add commas in your insmod command!!\n";

MODULE_PARM(ql4xdiscoverywait, "i");
MODULE_PARM_DESC(ql4xdiscoverywait,
		 "Discovery wait time");

MODULE_PARM(ql4xportdownretrycount, "i");
MODULE_PARM_DESC(ql4xportdownretrycount,
		 "Port down retry count");

MODULE_PARM(ql4xcmdretrycount, "i");
MODULE_PARM_DESC(ql4xcmdretrycount,
		 "Command retry count");

MODULE_PARM(ql4xmaxqdepth, "i");
MODULE_PARM_DESC(ql4xmaxqdepth,
		 "Maximum queue depth to report for target devices.");

#ifdef __VMKERNEL_MODULE__
/* set the default timeout to this value */
MODULE_PARM(ql4xdeftimeout, "i");
MODULE_PARM_DESC(ql4xdeftimeout,
		 "Command timeout");
#endif

#ifdef QLA4000
MODULE_PARM(ql4xcmdtimeout, "i");
MODULE_PARM_DESC(ql4xcmdtimeout,
		 "Command timeout");
#endif

MODULE_PARM(extended_error_logging,"i");
MODULE_PARM_DESC(extended_error_logging,
		 "Option to enable extended error logging, "
		 "Default is 0 - no logging. 1 - log errors. 2 - debug logging");

MODULE_PARM(ql4xopts, "s");
MODULE_PARM_DESC(ql4xopts,
		 "Additional driver options.");

MODULE_PARM(displayConfig, "i");
MODULE_PARM_DESC(displayConfig,
		 "If 1 then display the configuration used in "
		 "/etc/modules.conf.");
#endif	/* MODULE */

#ifdef MODULE_LICENSE
MODULE_LICENSE("GPL");
#endif

/*
 * Detected host adapter variables
 *---------------------------------------------------------------------------*/
int qla4xxx_hba_count = 0;
#ifdef __VMKERNEL_MODULE__
int qla4xxx_in_detect = 0;
#endif
static scsi_qla_host_t *qla4xxx_hostlist = NULL;

/*
 * Proc info processing
 *---------------------------------------------------------------------------*/
struct info_str {
	char    *buffer;
	int     length;
	off_t   offset;
	int     pos;
};


/*
 * Declare global semaphore
 *---------------------------------------------------------------------------*/
DECLARE_MUTEX_LOCKED(qla4xxx_detect_sem);
DECLARE_MUTEX_LOCKED(qla4xxx_dpc_sem);

/*
 * String messages for various state values (used for print statements)
 *---------------------------------------------------------------------------*/
const char *host_sts_msg[] ={
	"DID_OK",
	"DID_NO_CONNECT",
	"DID_BUS_BUSY",
	"DID_TIME_OUT",
	"DID_BAD_TARGET",
	"DID_ABORT",
	"DID_PARITY",
	"DID_ERROR",
	"DID_RESET",
	"DID_BAD_INTR",
	NULL};

const char *ddb_state_msg[] ={
	"UNASSIGNED",
	"NO_CONNECTION_ACTIVE",
	"DISCOVERY",
	"NO_SESSION_ACTIVE",
	"SESSION_ACTIVE",
	"LOGGING_OUT",
	"SESSION_FAILED",
	NULL};

const char *srb_state_msg[] = SRB_STATE_TBL();
#if 0
const char *dev_state_msg[] = DEV_STATE_TBL();
const char *lun_state_msg[] = LUN_STATE_TBL();
const char *adapter_state_msg[] = ADAPTER_STATE_TBL();
#endif

#include "ql4_sem.h"
#ifdef QLA4010
#include "ql4isns.c"
#endif

/****************************************************************************/
/*  LINUX -  Loadable Module Functions.                                     */
/****************************************************************************/
#if 0
uint8_t qla4xxx_send_noop(scsi_qla_host_t *ha)
{
	uint32_t   mbox_cmd[MBOX_REG_COUNT];
	uint32_t   mbox_sts[MBOX_REG_COUNT];

	memset(&mbox_cmd, 0, sizeof(mbox_cmd));
	memset(&mbox_sts, 0, sizeof(mbox_sts));
	mbox_cmd[0] = MBOX_CMD_NOP;

	if (qla4xxx_mailbox_command(ha, 1, 1, &mbox_cmd[0], &mbox_sts[0])
	    != QLA_SUCCESS) {
		QL4PRINT(QLP2, printk(KERN_INFO "scsi%d: NOP failed\n", ha->host_no));
		return(QLA_ERROR);
	}
	else {
		QL4PRINT(QLP2, printk(KERN_INFO "scsi%d: NOP succeded\n", ha->host_no));
		return(QLA_SUCCESS);
	}
}

uint8_t qla4xxx_mbx_test(scsi_qla_host_t *ha)
{
	uint32_t   mbox_cmd[MBOX_REG_COUNT];
	uint32_t   mbox_sts[MBOX_REG_COUNT];
	int i;
	uint8_t status;

	memset(&mbox_cmd, 0, sizeof(mbox_cmd));
	memset(&mbox_sts, 0, sizeof(mbox_sts));
	mbox_cmd[0] = MBOX_CMD_REGISTER_TEST;
	mbox_cmd[1] = 0x11111111;
	mbox_cmd[2] = 0x22222222;
	mbox_cmd[3] = 0x33333333;
	mbox_cmd[4] = 0x44444444;
	mbox_cmd[5] = 0x55555555;
	mbox_cmd[6] = 0x66666666;
	mbox_cmd[7] = 0x77777777;

	if (qla4xxx_mailbox_command(ha, 8, 8, &mbox_cmd[0], &mbox_sts[0])
	    != QLA_SUCCESS) {
		QL4PRINT(QLP2, printk(KERN_INFO "scsi%d: REGISTER_TEST failed, mbox_sts = 0x)x\n",
				      ha->host_no, mbox_sts[0]));
		return(QLA_ERROR);
	}

	if (mobx_sts[1] != 0x11111111 ||
	    mobx_sts[2] != 0x22222222 ||
	    mobx_sts[3] != 0x33333333 ||
	    mobx_sts[4] != 0x44444444 ||
	    mobx_sts[5] != 0x55555555 ||
	    mobx_sts[6] != 0x66666666 ||
	    mobx_sts[7] != 0x77777777) {
		QL4PRINT(QLP2, printk(KERN_INFO "scsi%d: REGISTER_TEST failed\n", ha->host_no));
		status = QLA_ERROR;

	}
	else {
		QL4PRINT(QLP2, printk(KERN_INFO "scsi%d: REGISTER_TEST succeded\n", ha->host_no));
		status = QLA_SUCCESS;
	}

	for (i = 0; i < 8; i++) {
		QL4PRINT(QLP2, printk("scsi%d: %s: MBX%d = 0x%x\n",
				      ha->host_no, __func__, i, mbox_cmd[i]));
	}
	return(status);
}
#endif

#if 1 || __VMKERNEL_MODULE__
static inline int pci_set_dma_mask(struct pci_dev *dev, uint64_t mask);

static inline int
pci_set_dma_mask(struct pci_dev *dev, uint64_t mask)
#else
inline int pci_set_dma_mask(struct pci_dev *dev, uint64_t mask);

inline int
pci_set_dma_mask(struct pci_dev *dev, uint64_t mask)
#endif
{
	if (!pci_dma_supported(dev, mask))
		return(-EIO);

	dev->dma_mask = mask;
	return(0);
}

/**
 * qla4xxx_config_dma_addressing() - Configure OS DMA addressing method.
 * @ha: HA context
 *
 * At exit, the @ha's flags.enable_64bit_addressing set to indicated
 * supported addressing method.
 */
static inline void qla4xxx_config_dma_addressing(scsi_qla_host_t *ha);
static inline void
qla4xxx_config_dma_addressing(scsi_qla_host_t *ha)
{
	uint8_t enabled;
	/*
	 * Given the two variants pci_set_dma_mask(), allow the compiler to
	 * assist in setting the proper dma mask.
	 */
	if (sizeof(dma_addr_t) > 4) {
		enabled = 1;
		/* Update our PCI device dma_mask for full 64 bit mask */
		if (pci_set_dma_mask(ha->pdev, 0xffffffffffffffffULL)) {
			QL4PRINT(QLP2, printk("scsi%d: failed to set 64 bit PCI DMA mask, "
					      "using 32 bits\n", ha->host_no));
			enabled = 0;
			pci_set_dma_mask(ha->pdev, 0xffffffff);
		}
	}
	else {
		enabled = 0;
		pci_set_dma_mask(ha->pdev, 0xffffffff);
	}
	QL4PRINT(QLP2, printk(KERN_INFO
			      "scsi%d: %d Bit PCI Addressing Enabled.\n",
			      ha->host_no, (enabled ? 64 : 32)));
}

/**************************************************************************
 * qla4xxx_detect
 *    This routine will probe for Qlogic 4000 iSCSI host adapters.
 *    It returns the number of host adapters of a particular
 *    type that were found.  It also initializes all data necessary for
 *    the driver.  It is passed-in the host number, so that it
 *    knows where its first entry is in the scsi_hosts[] array.
 *
 * Input:
 *     SHT - pointer to SCSI Host Template
 *
 * Returns:
 *    qla4xxx_hba_count - number of host adapters found.
 *
 * Context:
 *	Kernel context.
 **************************************************************************/
int
qla4xxx_detect(Scsi_Host_Template *SHT)
{
	struct pci_dev   *pdev = NULL;
	int bar;
	uint8_t status;
	uint8_t init_retry_count;
#ifdef QLA4000
	struct list_head *list;
	struct pci_bus   *b_bus;
	struct pci_dev   *dev;
#endif

	ENTER("qla4xxx_detect");
	printk("qla4xxx: code starts at address = %p\n", code_start);

#if ISP_RESET_TEST
	printk(KERN_INFO "qla4xxx: Adapter Reset Test Enabled!  "
	       "Adapter Resets will be issued every 3 minutes!\n");
#endif

#ifdef __VMKERNEL_MODULE__
        if (!vmk_set_module_version(QLA4XXX_DRIVER_VERSION)) {
           return 0;
        }
        /* In the vmkernel, we do not hold the io_request lock during init,
         * so leave it unlocked and do not lock it before returning. */
	/* Also, set the qla4xxx_in_detect flag to prevent called
	 * functions from grabbing the locks. */
	qla4xxx_in_detect = 1;
#endif //__VMKERNEL_MODULE__

#ifndef __VMKERNEL_MODULE__
	HOST_SPIN_UNLOCK_IRQ(ha);
#endif

	switch (extended_error_logging) {
	case 0:
		// Default set in ql4print.c file
		break;
	case 1:
		ql_dbg_level = QLP1|QLP2;
		printk("qla4xxx: Extended Error Logging ENABLED\n");
		break;
	case 2:
		ql_dbg_level = QLP1|QLP2|QLP4|QLP7|QLP20;
		printk("qla4xxx: Debug Error Logging ENABLED\n");
		break;
	}

#ifdef MODULE
	/*
	* If we are called as a module, the qla4000 pointer may not be null
	* and it would point to our bootup string, just like on the lilo
	* command line.  If not NULL, then process this config string with
	* qla4xxx_setup
	*
	* Boot time Options
	* To add options at boot time add a line to your lilo.conf file like:
	* append="qla4000=verbose,max_tags:{{255,255,255,255},{255,255,255,255}}"
	* which will result in the first four devices on the first two
	* controllers being set to a tagged queue depth of 32.
	*/
	if (ql4xopts)
		qla4xxx_setup(ql4xopts);

	if (dummy_buffer[0] != 'P')
		printk(KERN_WARNING
		       "qla4xxx: Please read the file /usr/src/linux/drivers/scsi/%s/README.qla4xxx.txt\n"
		       "qla4xxx: to see the proper way to specify options to the qla4000 module\n"
		       "qla4xxx: Specifically, don't use any commas when passing arguments to\n"
		       "qla4xxx: insmod or else it might trash certain memory areas.\n",
		       QLA4XXX_PROC_NAME);
#endif

	if (!pci_present()) {
		QL4PRINT(QLP1, printk(KERN_INFO "qla4xxx_detect: PCI not present\n"));
#ifndef __VMKERNEL_MODULE__
                HOST_SPIN_LOCK_IRQ(ha);
#endif //__VMKERNEL_MODULE__
                return(0);
	} 

	qla4xxx_hostlist = NULL;

	QL4PRINT(QLP7, printk("qla4xxx_detect: searching for adapter %d, "
			      "vendor_id %x, device_id %x\n",
			      qla4xxx_hba_count,
			      QLA4XXX_VENDOR_ID, QLA4XXX_BOARD_ID));

	while ((pdev = pci_find_subsys(QLA4XXX_VENDOR_ID,
				       QLA4XXX_BOARD_ID,
				       PCI_ANY_ID,
				       PCI_ANY_ID,
				       pdev))
	       && (qla4xxx_hba_count < MAX_HBAS)
	      ) {
		struct Scsi_Host *host;
		scsi_qla_host_t *ha, *cur_ha;
		init_retry_count = 0;

		/* Only support iSCSI controller function 1
		 * (i.e. QLA4010 has both SCSI [function 1] and
		 * NetWork [function 0] controllers */
		if (PCI_FUNC(pdev->devfn) != 1)
			continue;

		/* Found an adapter */
		QL4PRINT(QLP7, printk("qla4xxx: Supported Device Found VID=%X "
				      "DID=%X SSVID=%X SSDID=%X, instance %d\n",
				      pdev->vendor, pdev->device,
				      pdev->subsystem_vendor,
				      pdev->subsystem_device,
				      qla4xxx_hba_count));

		if (pci_enable_device(pdev) != PCIBIOS_SUCCESSFUL) {
			QL4PRINT(QLP7,
				 printk("already found-- skipping\n"));
			continue;
		}


#ifdef __VMKERNEL_MODULE__
		printk("qla4xxx: at vmk_scsi_register\n");
                host = 	vmk_scsi_register(SHT, sizeof(scsi_qla_host_t), pdev->bus->number, pdev->devfn);
#else
		HOST_SPIN_LOCK_IRQ(ha);
		host = scsi_register(SHT, sizeof(scsi_qla_host_t));
#endif
		if (host == NULL) {
			QL4PRINT(QLP1,
				 printk(KERN_INFO "qla4xxx: Failed to get "
					"memory for host\n"));

#ifndef __VMKERNEL_MODULE__
			HOST_SPIN_UNLOCK_IRQ(ha);
#endif
			continue;
		}

		/* Clear our data area */
		ha = (scsi_qla_host_t *) host->hostdata;

#if defined(CONFIG_VMNIX) && !defined(__VMKERNEL_MODULE__)
                host->bus = pdev->bus->number;
                host->devfn = pdev->devfn;
                host->devid = ha; 
#endif
		memset(ha, 0, sizeof(scsi_qla_host_t));
#ifdef __VMKERNEL_MODULE__
		ha->in_detect = 1; 
                vmk_scsi_register_uinfo(host, pdev->bus->number, pdev->devfn, ha);
#endif
                

		/* Save the information from PCI BIOS.  */
		ha->pdev = pdev;
		ha->pci_bus = pdev->bus->number;
		ha->pci_device_fn = pdev->devfn;
		ha->pci_device_id = QLA4XXX_BOARD_ID;
		ha->host = host;
		ha->host_no = host->host_no;
		ha->instance = qla4xxx_hba_count;

		qla4xxx_config_dma_addressing(ha);

		/* Search for I/O register */
		for (bar = 0; bar <= 5; bar++) {
			unsigned long resource_flags;
			uint32_t      pci_base_address;

			pci_base_address = pci_resource_start(pdev, bar);

			resource_flags = pci_resource_flags(pdev, bar);
			resource_flags &= PCI_BASE_ADDRESS_SPACE;

#if MEMORY_MAPPED_IO
			if ((resource_flags) == PCI_BASE_ADDRESS_SPACE_MEMORY) {
				QL4PRINT(QLP7,
					 printk("scsi%d: Assigned to "
						"Memory I/O 0x%x in PCI "
						"BAR%d\n", ha->host_no,
						pci_base_address, bar));

				host->base = pci_base_address;
				break;
			}
#else
			if ((resource_flags) == PCI_BASE_ADDRESS_SPACE_IO) {
				QL4PRINT(QLP7,
					 printk("scsi%d: Assigned to "
						"I/O Port 0x%x in PCI BAR%d\n",
						ha->host_no, pci_base_address,
						bar));

				host->io_port = pci_base_address;
				break;
			}
#endif
		}

		/* Save host information */
		host->can_queue   = REQUEST_QUEUE_DEPTH/4;    /*   */
		host->cmd_per_lun = MAX_LINKED_CMDS_PER_LUN;
		host->select_queue_depths = qla4xxx_select_queue_depth;
		host->irq         = pdev->irq;
		host->max_channel = QLA4XXX_BOARD_PORTS-1;
#if 1 || defined(__VMKERNEL_MODULE__)
		host->max_lun     = MAX_LUNS;
		host->max_id      = MAX_TARGETS;
#else
		host->max_lun     = MAX_LUNS-1;
		host->max_id      = MAX_TARGETS-1;
#endif
		host->unique_id   = ha->instance;
		host->max_cmd_len = 16;

		/* Initialize lists and spinlocks */
		INIT_LIST_HEAD(&ha->ddb_list);
		INIT_LIST_HEAD(&ha->pending_srb_q);
		INIT_LIST_HEAD(&ha->retry_srb_q);
		INIT_LIST_HEAD(&ha->done_srb_q);
		INIT_LIST_HEAD(&ha->suspended_lun_q);
		INIT_LIST_HEAD(&ha->free_srb_q);

		init_MUTEX(&ha->mbox_sem);
		init_waitqueue_head(&ha->mailbox_wait_queue);
		init_MUTEX(&ha->ioctl_sem);
		init_MUTEX_LOCKED(&ha->ioctl_cmpl_sem);

		spin_lock_init(&ha->hardware_lock);
		spin_lock_init(&ha->aen_q_spinlock);
		spin_lock_init(&ha->pdu_spinlock);
		spin_lock_init(&ha->adapter_lock);
		spin_lock_init(&ha->list_lock);
#if 0 && __VMKERNEL_MODULE__
		spin_lock_init(&ha->dma_buf_spinlock);
#else
		spin_lock_init(&ha->ddb_list_spinlock);
		init_MUTEX(&ha->dma_buf_sem);
		init_MUTEX(&ha->targ_array_sem);
#endif

#if defined(SH_HAS_HOST_LOCK)
		spin_lock_init(&ha->host_lock);

		host->host_lock = &ha->host_lock;
#endif

#ifdef QLA4000

		/* Search for pointer to the Intel PCI Brigde on the QLA4000 */
#define QLPCI_BRIDGE_CLASS 0x060400
		b_bus=pdev->bus;
		list_for_each(list,&b_bus->devices)
		if ((dev = pci_dev_b(list)) != NULL) {
			if ((dev->class = QLPCI_BRIDGE_CLASS) &&
			    PCI_SLOT(dev->devfn) == PCI_SLOT(pdev->devfn) &&
			    PCI_FUNC(dev->devfn) != PCI_SLOT(pdev->devfn)) {
				ha->pci_brgdev=dev;
			}
		}
#endif

		HOST_SPIN_UNLOCK_IRQ(ha);

		/* Allocate memory for dma buffers */
		if (qla4xxx_mem_alloc(ha) == QLA_ERROR) {
			HOST_SPIN_LOCK_IRQ(ha);

			scsi_unregister(host);
			QL4PRINT(QLP1, printk(KERN_INFO
					      "scsi%d: Failed to get DMA memory\n",
					      ha->host_no));

			HOST_SPIN_UNLOCK_IRQ(ha);
			continue;
		}

		HOST_SPIN_LOCK_IRQ(ha);

		/* Register our resources with Linux
		 * (i.e. IRQ, I/O Port, etc.) */
		if (qla4xxx_register_resources(ha) == QLA_ERROR) {
			QL4PRINT(QLP1, printk(KERN_INFO
					      "scsi%d: Failed to register resources\n",
					      ha->host_no));
			qla4xxx_mem_free(ha);
			scsi_unregister(host);
			HOST_SPIN_UNLOCK_IRQ(ha);
			continue;
		}

		HOST_SPIN_UNLOCK_IRQ(ha);

		/* Initialize the Host adapter request/response queues and
		 * firmware */
		status = qla4xxx_initialize_adapter(ha, REBUILD_DDB_LIST);
		while ((status == QLA_ERROR) &&
		       test_bit(DPC_RESET_HA, &ha->dpc_flags) &&
		       (init_retry_count++ < MAX_INIT_RETRIES)) {
			QL4PRINT(QLP2, printk("scsi%d: %s: retrying adapter initialization (%d)\n",
					      ha->host_no, __func__, init_retry_count));

			HOST_SPIN_LOCK_IRQ(ha);
			qla4xxx_soft_reset(ha);
			HOST_SPIN_UNLOCK_IRQ(ha);
			status = qla4xxx_initialize_adapter(ha, REBUILD_DDB_LIST);
		}

		if (status == QLA_ERROR) {
			QL4PRINT(QLP2, printk(KERN_INFO
					      "scsi%d: Failed to initialize adapter\n",
					      ha->host_no));

			HOST_SPIN_LOCK_IRQ(ha);

#ifdef QLA4010
			spin_lock_irq(&ha->hardware_lock);
			WRT_REG_DWORD(&ha->reg->ISPControlStatus, CSRW_SOFT_RESET);
			spin_unlock_irq(&ha->hardware_lock);
#endif
			free_irq(host->irq, ha);
			if (host->io_port)
				release_region(host->io_port, 0xff);

			HOST_SPIN_UNLOCK_IRQ(ha);

			qla4xxx_mem_free(ha);

			HOST_SPIN_LOCK_IRQ(ha);

			scsi_unregister(host);

			HOST_SPIN_UNLOCK_IRQ(ha);
			continue;
		}

		/* Insert new entry into the list of adapters */
		ha->next = NULL;
		if (qla4xxx_hostlist == NULL) {
			cur_ha = qla4xxx_hostlist = ha;
		}
		else {
			cur_ha = qla4xxx_hostlist;
			while (cur_ha->next != NULL)
				cur_ha = cur_ha->next;
			cur_ha->next = ha;
		}

		set_bit(AF_INIT_DONE, &ha->flags);

		/* Start timer thread */
		QL4PRINT(QLP7,
			 printk("scsi%d: %s: Starting timer thread for adapter %d\n",
				ha->host_no, __func__, ha->instance));

		init_timer(&ha->timer);
		ha->timer.function = qla4xxx_timer;
		ha->timer.data     = (unsigned long) ha;
		ha->timer.expires  = jiffies + HZ;
		add_timer(&ha->timer);



		/* Startup the kernel thread for this host adapter */

#ifdef __VMKERNEL_MODULE__
			/*
			 * Initialize the extensions defined in ha to
			 * communicate with the DPC kernel thread.
			 */
                ha->should_die = FALSE;
#endif //__VMKERNEL_MODULE__
		QL4PRINT(QLP7,
			 printk("scsi%d: %s: Starting kernel thread for "
				"qla4xxx_dpc_%d\n", ha->host_no, __func__,
				ha->host_no));
		ha->dpc_notify = &qla4xxx_detect_sem;
		kernel_thread((int (*)(void *))qla4xxx_do_dpc, (void *) ha, 0);

		/* Now wait for the kernel dpc thread to initialize
		 * and go to sleep. */
		down(&qla4xxx_detect_sem);
		ha->dpc_notify = NULL;

		qla4xxx_hba_count++;
#ifdef __VMKERNEL_MODULE__
		/* Turn off initializion flag for this ha */
		ha->in_detect = 0;
#endif //__VMKERNEL_MODULE__

	}  /* end of WHILE */
        
#ifndef __VMKERNEL_MODULE__
        HOST_SPIN_LOCK_IRQ(ha);
#endif //__VMKERNEL_MODULE__
        
        
	QL4PRINT(QLP7, printk("scsi: Found %d hosts ...\n",
			      qla4xxx_hba_count));

	if (displayConfig)
		qla4xxx_display_config();

	LEAVE("qla4xxx_detect");
	return(qla4xxx_hba_count);
}

/**************************************************************************
* qla4xxx_get_line
* 	Copy a substring from the specified string. The substring
* 	consists of any number of chars seperated by white spaces
*	(i.e. spaces) and ending with a newline '\n' or a semicolon ';'.
*
* Enter:
* 	str - orig string
* 	line - substring
*
* Returns:
*   	cp - pointer to next string, or
*   	null - End of string
*
* Context:
*	Kernel context.
*************************************************************/
static char *
qla4xxx_get_line(char *str, char *line)
{
	register        char    *cp = str;
	register        char    *sp = line;

	/* skip preceeding spaces */
	while (*cp && *cp == ' ')
		++cp;
	while ((*cp) && *cp != '\n' && *cp != ';')   /* end of line */
		*sp++ = *cp++;

	*sp = '\0';

	QL4PRINT(QLP7, printk("%s: %s\n", __func__, line));

	if ((*cp)) {
		cp++;
		return(cp);
	}

	return(NULL);
}

/**************************************************************************
 * qla4xxx_get_tokens
 *	This routine retrieves a token from the command line.
 *
 * Input:
 *	line - Pointer to command line
 *	argv - Pointer to arguements
 *	str -  Pointer to starting point of symbol
 *
 * Output:
 *	count - Number of tokens retrieved
 *
 * Remarks:
 *	None
 *
 * Returns:
 *	Pointer to command Line after token is retrieved.
 *
 * Context:
 *	Kernel context.
 **************************************************************************/
static int
qla4xxx_get_tokens(char *line, char **argv, int maxargs )
{
	register        char    *cp = line;
	int     count = 0;

	while (*cp && count < maxargs) {
		/* skip preceeding spaces */
		while ((*cp) && *cp == ' ')
			++cp;
		/* symbol starts here */
		argv[count++] = cp;
		/* skip symbols */
		while ((*cp) && !(*cp == ' ' || *cp == ';' || *cp == ':'))
			cp++;
		/* replace comma or space with a null */
		if ((*cp) && (*cp ==' ' ) && argv[count-1] != cp)
			*cp++ = '\0';
	}
	return(count);
}

/**************************************************************************
 * qla4xxx_setup
 *	This routine handles Linux boot parameters. This routine allows for
 *	assigning a value to a parameter with a ':' between the parameter
 *	and the value. (ie. qla4000=max_reqs:0x0A,verbose)
 *
 * Input:
 *	s - Pointer to command line
 *
 * Returns:
 *	None
 *
 * Context:
 *	Kernel context.
 **************************************************************************/
#if !defined(MODULE)
static int __init
qla4xxx_setup (char *s)
#else
void
qla4xxx_setup(char *s)
#endif	
{
#define LINESIZE    256
#define MAXARGS      26
	char *cp, *np;
	static char     *slots[MAXARGS];
	char            **argv = &slots[0];
	static char buf[LINESIZE];
	int opts;

#if !defined(MODULE)
	if (s == NULL || *s == '\0')
		return(0);
#endif

	/*
	 * Determine if we have any properties.
	 */
	cp = s;
	opts = 1;
	while (*cp && (np = qla4xxx_get_line(cp, buf)) != NULL) {
		if (strncmp("scsi-qla",buf,8) == 0) {
			QL4PRINT(QLP7, printk("%s: devconf=%s\n",__func__, cp));

			ql4xdevconf = cp;
			(opts > 0)? opts-- : 0;
			break;
		}
		opts++;
		cp = np;
	}

	/*
	 * Parse the args before the properties
	 */
	if (opts) {
		int argc;

		opts = (opts > MAXARGS-1)? MAXARGS-1: opts;
		argc = qla4xxx_get_tokens(s, argv, opts);
		while (argc > 0) {
			cp = *argv;
			QL4PRINT(QLP7, printk("scsi: found cmd arg =[%s]\n", cp));

#if 0
			if (strcmp(cp, "verbose") == 0) {
				DEBUG(printk("qla2100: verbose\n");)
				qla2x00_verbose++;
			}
			else if (strcmp(cp, "quiet") == 0) {
				qla2x00_quiet = 1;
			}
			else if (strcmp(cp, "reinit_on_loopdown") == 0) {
				qla2x00_reinit++;
				DEBUG(printk("qla2100: reinit_on_loopdown\n");)
			}
#endif
			argc--, argv++;
		}
	}

#if !defined(MODULE)
	if (ql4xdevconf)
		return(1);
	else
		return(0);
#endif
}

#if !defined(MODULE)
__setup("ql4xopts=", qla4xxx_setup);
#endif

/**************************************************************************
 * qla4xxx_display_config
 *	This routine  displays the configuration information to be used in
 *	modules.conf.
 *
 * Input:
 *	ha - Pointer to host adapter structure
 *
 * Output:
 *	None
 *
 * Returns:
 *	None
 *
 * Context:
 *	Kernel context.
 **************************************************************************/
void
qla4xxx_display_config(void)
{
	scsi_qla_host_t *ha;

	for (ha = qla4xxx_hostlist; ha != NULL; ha = ha->next) {
		/* Display the M.A.C. Address for adapter */
		printk(KERN_INFO
		       "scsi-qla%d-mac=%02x%02x%02x%02x%02x%02x\\;\n",
		       (int)ha->instance,
		       ha->my_mac[0],
		       ha->my_mac[1],
		       ha->my_mac[2],
		       ha->my_mac[3],
		       ha->my_mac[4],
		       ha->my_mac[5]);
	}
}

/**************************************************************************
 * qla4xxx_alloc_dma_memory
 *	This routine allocates dma memory of the given size.
 *
 * Input:
 *	pdev    - Pointer to PCI device structure
 *	dma_buf - Pointer to structure containing dma buffer pointers, etc.
 *	size    - Size of dma buffer to allocate
 *
 * Output:
 *	dma_buf - virt_addr, phys_addr, and buf_len values filled in
 *
 * Returns:
 *	QLA_SUCCESS - Successfully allocates memory
 *	QLA_ERROR   - Failed to allocate memory
 *
 * Context:
 *	Kernel context.
 **************************************************************************/
inline uint8_t
qla4xxx_alloc_dma_memory(struct pci_dev *pdev,
			 dma_buf_t *dma_buf,
			 uint32_t size)
{
	dma_buf->virt_addr = pci_alloc_consistent(pdev,
						  size,
						  &dma_buf->phys_addr);
	if (dma_buf->virt_addr) {
		dma_buf->buf_len = size;
		return(QLA_SUCCESS);
	}
	else {
		return(QLA_ERROR);
	}
}

/**************************************************************************
 * qla4xxx_free_dma_memory
 *	This routine frees dma memory.
 *
 * Input:
 *	pdev    - Pointer to PCI device structure
 *	dma_buf - Pointer to structure containing dma buffer pointers, etc.
 *
 * Output:
 *	dma_buf - Structure cleared
 *
 * Returns:
 *	None.
 *
 * Context:
 *	Kernel context.
 **************************************************************************/
inline void
qla4xxx_free_dma_memory(struct pci_dev *pdev, dma_buf_t *dma_buf)
{
	pci_free_consistent(pdev,
			    dma_buf->buf_len,
			    dma_buf->virt_addr,
			    dma_buf->phys_addr);
	dma_buf->buf_len = 0;
	dma_buf->phys_addr = 0;
	dma_buf->virt_addr = 0;
}

/**************************************************************************
 * qla4xxx_resize_dma_buf
 *	This routine deallocates the dma_buf of the previous size and re-
 *	allocates the dma_buf with the given size.
 *
 * Input:
 *	ha      - Pointer to host adapter structure
 *	dma_buf - Pointer to structure containing dma buffer pointers, etc.
 *	size    - Size of dma buffer to allocate
 *
 * Output:
 *	dma_buf - virt_addr, phys_addr, and buf_len values filled in
 *
 * Returns:
 *	QLA_SUCCESS - Successfully re-allocates memory
 *	QLA_ERROR   - Failed to re-allocate memory
 *
 * Context:
 *	Kernel context.
 **************************************************************************/
uint8_t
qla4xxx_resize_dma_buf(scsi_qla_host_t *ha, dma_buf_t *dma_buf, uint32_t size)
{
	uint8_t status = QLA_ERROR;

	ENTER("qla4xxx_resize_dma_buf");

	if (dma_buf->buf_len) {
		QL4PRINT(QLP3,
			 printk("scsi%d: %s: deallocate old dma_buf, size=0x%x\n",
				ha->host_no, __func__, dma_buf->buf_len));
		qla4xxx_free_dma_memory(ha->pdev, dma_buf);
	}

	QL4PRINT(QLP3, printk("scsi%d: %s: allocate new dma_buf, size=0x%x\n",
			      ha->host_no, __func__, size));
	status =  qla4xxx_alloc_dma_memory(ha->pdev, dma_buf, size);

	if (status == QLA_ERROR)
		QL4PRINT(QLP2, printk("scsi%d: %s: ERROR allocating new dma_buf, "
				      "size=0x%x\n", ha->host_no, __func__, size));

	LEAVE("qla4xxx_resize_dma_buf");
	return(status);
}

/**************************************************************************
 * qla4xxx_alloc_srb_pool
 *	This routine is called during driver initialization to allocate
 *	memory for the local srb pool.
 *
 * Input:
 *	ha - Pointer to host adapter structure
 *
 * Returns:
 *	QLA_SUCCESS - Successfully allocated srbs
 *	QLA_ERROR   - Failed to allocate any srbs
 *
 * Context:
 *	Kernel context.
 **************************************************************************/
uint8_t qla4xxx_alloc_srb_pool(scsi_qla_host_t *ha)
{
	srb_t *srb;
	int i;
	uint8_t status = QLA_ERROR;
	unsigned long flags;
	ENTER("qla4xxx_alloc_srb_pool");

	spin_lock_irqsave(&ha->list_lock, flags);
	ha->num_srbs_allocated = 0;
	ha->free_srb_q_count = 0; /* incremented in add_to_free_srb_q routine */

	/*
	 * NOTE: Need to allocate each SRB separately, as Kernel 2.4.4
	 *       seems to have an error when allocating a large amount
	 *       of memory.
	 */
	for (i=0; i < MAX_SRBS; i++) {
		srb = (srb_t *) kmalloc(sizeof(srb_t), GFP_KERNEL);
		if (srb == NULL) {
			QL4PRINT(QLP2, printk(KERN_INFO "scsi%d: %s: failed to allocate memory, "
					      "count = %d\n",
					      ha->host_no, __func__, i));
		}
		else {
			ha->num_srbs_allocated++;
			memset(srb, 0, sizeof(srb_t));
			__add_to_free_srb_q(ha, srb);
			atomic_set(&srb->ref_count, 0);
			status = QLA_SUCCESS;
		}
	}

	if (ha->free_srb_q_count)
		status = QLA_SUCCESS;

	spin_unlock_irqrestore(&ha->list_lock, flags);

	QL4PRINT(QLP7, printk("scsi%d: %s: Allocated %d SRB(s)\n",
			      ha->host_no, __func__, ha->free_srb_q_count));

	LEAVE("qla4xxx_alloc_srb_pool");
	return(status);
}

/**************************************************************************
 * qla4xxx_free_srb_pool
 *	This routine is called during driver unload to deallocate the srb
 *	pool.
 *
 * Input:
 *	ha - Pointer to host adapter structure
 *
 * Returns:
 *	None
 *
 * Context:
 *	Kernel context.
 **************************************************************************/
void qla4xxx_free_srb_pool(scsi_qla_host_t *ha)
{
	struct list_head *ptr, *next;
	srb_t *srb;
	int cnt_free_srbs = 0;
	unsigned long flags;

	ENTER("qla4xxx_free_srb_pool");
	spin_lock_irqsave(&ha->list_lock, flags);
	list_for_each_safe(ptr, next, &ha->free_srb_q)
	{
		srb = list_entry(ptr, srb_t, list_entry);
		/* Remove srb from LUN queue. */
		__del_from_free_srb_q(ha, srb);
		kfree(srb);
		cnt_free_srbs++;
	}

	if (cnt_free_srbs != ha->num_srbs_allocated) {
		QL4PRINT(QLP2, printk(KERN_WARNING "scsi%d: Did not free all srbs, "
				      "Free'd srb count = %d, Alloc'd srb count %d\n",
				      ha->host_no, cnt_free_srbs, ha->num_srbs_allocated));
	}

	spin_unlock_irqrestore(&ha->list_lock, flags);
	LEAVE("qla4xxx_free_srb_pool");
}

/**************************************************************************
 * qla4xxx_mem_alloc
 *	This routine allocates memory use by the adapter.
 *
 * Input:
 *	ha - Pointer to host adapter structure
 *
 * Returns:
 *	QLA_SUCCESS - Successfully allocated adapter memory
 *	QLA_ERROR   - Failed to allocate adapter memory
 *
 * Context:
 *	Kernel context.
 **************************************************************************/
STATIC uint8_t
qla4xxx_mem_alloc(scsi_qla_host_t *ha)
{
	void *tmp_dma_memv;
	dma_addr_t tmp_dma_memp;

	ENTER("qla4xxx_mem_alloc");

	/*
	 * Allocate DMA memory (rounding up to nearest page)
	 *--------------------------------------------------*/
	ha->dma_mem_blk_raw_len = ((sizeof(dma_mem_blk_t) + MEM_ALIGN_VALUE) + (PAGE_SIZE-1)) & ~(PAGE_SIZE-1);

	tmp_dma_memv = pci_alloc_consistent(ha->pdev,
					    ha->dma_mem_blk_raw_len,
					    &tmp_dma_memp);
	if (tmp_dma_memv == NULL) {
		QL4PRINT(QLP7, printk("scsi%d: %s: Unable to allocate DMA memory\n",
				      ha->host_no, __func__));

#if 1 || __VMKERNEL_MODULE__
                ha->dma_mem_blk_raw_len = 0; /* So we don't deallocate */
#endif
		goto mem_alloc_error_exit;
	}

	ha->dma_mem_blk_rawv = (dma_mem_blk_t *) tmp_dma_memv;
	ha->dma_mem_blk_rawp = tmp_dma_memp;
	memset(ha->dma_mem_blk_rawv, 0, ha->dma_mem_blk_raw_len);

	if ((u_long) tmp_dma_memv & (MEM_ALIGN_VALUE-1)) {
#if 1 || defined(__VMKERNEL_MODULE__)
		u_long align_val = MEM_ALIGN_VALUE -
				   ((u_long)tmp_dma_memv &
				    (MEM_ALIGN_VALUE-1));
#else
		u_long align_val = MEM_ALIGN_VALUE -
				   ((u_long)tmp_dma_memv &
				    MEM_ALIGN_VALUE);
#endif

		tmp_dma_memv = (void *) ((u_long) tmp_dma_memv + align_val);
		tmp_dma_memp = tmp_dma_memp + align_val;
	}

	ha->dma_mem_blkv = (dma_mem_blk_t *) tmp_dma_memv;
	ha->dma_mem_blkp = tmp_dma_memp;

	QL4PRINT(QLP7, printk("scsi%d: %s: Raw DmaMemBlkv = 0x%p\n",
			      ha->host_no, __func__,
			      ha->dma_mem_blk_rawv));
	QL4PRINT(QLP7, printk("scsi%d: %s: Raw DmaMemBlkp = 0x%lx\n",
			      ha->host_no, __func__,
			      (unsigned long)ha->dma_mem_blk_rawp));
	QL4PRINT(QLP7, printk("scsi%d: %s: Aligned DmaMemBlkv = 0x%p\n",
			      ha->host_no, __func__,
			      ha->dma_mem_blkv));
	QL4PRINT(QLP7, printk("scsi%d: %s: Aligned DmaMemBlkp = 0x%lx\n",
			      ha->host_no, __func__,
			      (unsigned long)ha->dma_mem_blkp));


	/*
	 * FIll in variables to access dma data
	 *-------------------------------------------------*/
	ha->req_queue_headv          = &ha->dma_mem_blkv->request_queue[0];
	ha->rsp_queue_headv          = &ha->dma_mem_blkv->response_queue[0];

	#ifdef QLA4010
	ha->dma_regsv                = &ha->dma_mem_blkv->dma_registers;
	ha->dma_regsp                = ha->dma_mem_blkp + offsetof(dma_mem_blk_t, dma_registers);

	ha->pdu_buffsv               = (PDU_ENTRY *) &ha->dma_mem_blkv->pdu_buffs[0][0];
	ha->pdu_buffsp               = ha->dma_mem_blkp + offsetof(dma_mem_blk_t, pdu_buffs);
	ha->free_pdu_top             = ha->pdu_buffsv;
	#endif

	ha->req_queue_headp          = ha->dma_mem_blkp + offsetof(dma_mem_blk_t, request_queue);
	ha->rsp_queue_headp          = ha->dma_mem_blkp + offsetof(dma_mem_blk_t, response_queue);

	/*
	 * Allocate consistent memory for generic dma buffer
	 *------------------------------------------*/
	if (qla4xxx_alloc_dma_memory(ha->pdev,
				     &ha->dma_buf,
				     DMA_BUFFER_SIZE)
	    == QLA_ERROR) {
		goto mem_alloc_error_exit;
	}

	QL4PRINT(QLP7, printk("scsi%d: %s: dma buff.virt_addr = 0x%p\n",
			      ha->host_no, __func__,
			      ha->dma_buf.virt_addr));
	QL4PRINT(QLP7, printk("scsi%d: %s: dma buff.phys_addr = 0x%lx\n",
			      ha->host_no, __func__,
			      (unsigned long)ha->dma_buf.phys_addr));

	/*
	 * Allocate memory for srb pool
	 *-----------------------------*/
	if (qla4xxx_alloc_srb_pool(ha) == QLA_ERROR) {
		goto mem_alloc_error_exit;
	}

#if MEMORY_MAPPED_IO
	/*
	 * Map the Memory I/O register
	 *----------------------------*/
	{
		uint32_t  page_offset, base;
		struct Scsi_Host *host = ha->host;

		QL4PRINT(QLP7, printk("scsi%d: %s: base memory address = 0x%lX\n",
				      ha->host_no, __func__, host->base));

		/* Find proper memory chunk for memory map I/O reg. */
		base = host->base & PAGE_MASK;
		page_offset = host->base - base;

		/* Get virtual address for I/O registers. */
		ha->virt_mmapbase = ioremap(base, page_offset +
					    sizeof(*ha->reg));
		if (ha->virt_mmapbase == 0) {
			QL4PRINT(QLP2, printk("scsi%d: %s: I/O Remap Failed\n",
					      ha->host_no, __func__));
			goto mem_alloc_error_exit;
		}

		QL4PRINT(QLP7, printk("scsi%d: %s: virt memory_mapped_address = "
				      "0x%p\n", ha->host_no, __func__,
				      ha->virt_mmapbase));

		ha->reg = (ISP_REG_T *)(ha->virt_mmapbase + page_offset);
		QL4PRINT(QLP7, printk("scsi%d: %s: registers = 0x%p\n",
				      ha->host_no, __func__, ha->reg));
	}
#endif

	LEAVE("qla4xxx_mem_alloc");
	return(QLA_SUCCESS);

	mem_alloc_error_exit:
	qla4xxx_mem_free(ha);
	LEAVE("qla4xxx_mem_alloc");
	return(QLA_ERROR);
}

/**************************************************************************
 * qla4xxx_free_lun
 *	This routine deallocates and unlinks the specified lun from the
 *	ddb_entry's lun table.
 *
 * Input:
 *	ddb_entry - Pointer to device database entry
 *	lun - Lun to free
 *
 * Returns:
 *	None
 *
 * Context:
 *	Kernel context.
 **************************************************************************/
static inline void
qla4xxx_free_lun(ddb_entry_t *ddb_entry, uint16_t lun)
{
	lun_entry_t *lun_entry = ddb_entry->lun_table[lun];

	ENTER("qla4xxx_free_lun");
	if (lun_entry != 0) {
		QL4PRINT(QLP3, printk("%s: target %d, free lun %d\n",
				      __func__, ddb_entry->target, lun));

		ddb_entry->lun_table[lun] = NULL;
		ddb_entry->num_valid_luns--;
#ifdef __VMKERNEL_MODULE__
                spin_lock_destroy(&lun_entry->lun_lock);
#endif
		kfree(lun_entry);
	}

	LEAVE("qla4xxx_free_lun");
}

/**************************************************************************
 * qla4xxx_free_ddb
 *	This routine deallocates and unlinks the specified ddb_entry from the
 *	adapter's
 *
 * Input:
 * 	ha - Pointer to host adapter structure.
 *	ddb_entry - Pointer to device database entry
 *
 * Returns:
 *	None
 *
 * Context:
 *	Kernel context.
 **************************************************************************/
STATIC void
qla4xxx_free_ddb(scsi_qla_host_t *ha, ddb_entry_t *ddb_entry)
{
	uint16_t lun;
#if 1 || __VMKERNEL_MODULE__
	unsigned int flags;
#endif

	ENTER("qla4xxx_free_ddb");

#if 1 || __VMKERNEL_MODULE__
	spin_lock_irqsave(&ha->ddb_list_spinlock, flags);
#endif
	/* Remove device entry from list */
	list_del_init(&ddb_entry->list_entry);
#if 1 || __VMKERNEL_MODULE__
	spin_unlock_irqrestore(&ha->ddb_list_spinlock, flags);
#endif

	/* Remove device pointer from index mapping arrays */
	ha->fw_ddb_index_map[ddb_entry->fw_ddb_index] = (ddb_entry_t *) INVALID_ENTRY;
#if 1 || __VMKERNEL_MODULE__
	down(&ha->targ_array_sem);
	/*
         * Make sure te ddb has a valid target number and that the
         * entry in the target map points to this ddb.
         */
	if ((ddb_entry->target < MAX_DDB_ENTRIES) &&
		 (ha->target_map[ddb_entry->target] == ddb_entry)) {
			ha->target_map[ddb_entry->target] =
				(ddb_entry_t *) INVALID_ENTRY;
	}
	up(&ha->targ_array_sem);
#else
	if (ddb_entry->target < MAX_DDB_ENTRIES)
		ha->target_map[ddb_entry->target] = (ddb_entry_t *) INVALID_ENTRY;
#endif
	ha->tot_ddbs--;

	/* Free memory allocated for all luns */
	for (lun = 0; lun < MAX_LUNS; lun++)
		if (ddb_entry->lun_table[lun])
			qla4xxx_free_lun(ddb_entry, lun);

		/* Free memory for device entry */
	kfree(ddb_entry);
	LEAVE("qla4xxx_free_ddb");
}

/**************************************************************************
 * qla4xxx_free_ddb_list
 *	This routine deallocates and removes all devices on the sppecified
 *	adapter.
 *
 * Input:
 * 	ha - Pointer to host adapter structure.
 *
 * Returns:
 *	None
 *
 * Context:
 *	Kernel context.
 **************************************************************************/
STATIC void
qla4xxx_free_ddb_list(scsi_qla_host_t *ha)
{
	struct list_head *ptr;
	ddb_entry_t *ddb_entry;
#if 1 || __VMKERNEL_MODULE__
	unsigned long flags;
#endif


	ENTER("qla4xxx_free_ddb_list");

#if 1 || __VMKERNEL_MODULE__
	spin_lock_irqsave(&ha->ddb_list_spinlock, flags);
#endif
	while (!list_empty(&ha->ddb_list)) {
		/* Remove device entry from head of list */
		ptr = ha->ddb_list.next;
		list_del_init(ptr);

		/* Free memory for device entry */
		ddb_entry = list_entry(ptr, ddb_entry_t, list_entry);
		kfree(ddb_entry);
	}
#if 1 || __VMKERNEL_MODULE__
	spin_unlock_irqrestore(&ha->ddb_list_spinlock, flags);
#endif

	LEAVE("qla4xxx_free_ddb_list");
}

/**************************************************************************
 * qla4xxx_mem_free
 *	This routine frees adapter allocated memory
 *
 * Input:
 * 	ha - Pointer to host adapter structure.
 *
 * Returns:
 *	None
 *
 * Context:
 *	Kernel context.
 **************************************************************************/
STATIC void
qla4xxx_mem_free(scsi_qla_host_t *ha)
{
	ENTER("qla4xxx_mem_free");

	/* Free consistent memory allocated for DMA Memory Block */
	if (ha->dma_mem_blk_raw_len) {
		QL4PRINT(QLP7,
			 printk("scsi%d: %s: free DMA Memory Block\n",
				ha->host_no, __func__));
		pci_free_consistent(ha->pdev,
				    ha->dma_mem_blk_raw_len,
				    ha->dma_mem_blk_rawv,
				    ha->dma_mem_blk_rawp);

		ha->dma_mem_blk_raw_len = 0;
	}

	/* Free consistent memory allocated for dma buffer */
	if (ha->dma_buf.buf_len) {
		QL4PRINT(QLP7,
			 printk("scsi%d: %s: free dma buffer memory\n",
				ha->host_no, __func__));
		qla4xxx_free_dma_memory(ha->pdev, &ha->dma_buf);
	}

	/* Free srb pool */
	if (ha->num_srbs_allocated) {
		QL4PRINT(QLP7,
			 printk("scsi%d: %s: free srb pool\n",
				ha->host_no, __func__));
		qla4xxx_free_srb_pool(ha);
	}

	/* Free ddb list */
	if (!list_empty(&ha->ddb_list)) {
		QL4PRINT(QLP7,
			 printk("scsi%d: %s: free ddb list\n",
				ha->host_no, __func__));
		qla4xxx_free_ddb_list(ha);
	}

	/* Unmap Memory Mapped I/O region */
	if (ha->virt_mmapbase) {
		QL4PRINT(QLP7,
			 printk("scsi%d: %s: unmap mem io region\n",
				ha->host_no, __func__));
		iounmap(ha->virt_mmapbase);
	}

	LEAVE("qla4xxx_mem_free");
}

/**************************************************************************
 * qla4xxx_register_resources
 *	This routine registers the irq with Linux
 *
 * Input:
 * 	ha - Pointer to host adapter structure.
 *
 * Returns:
 *	QLA_SUCCESS - Sucessfully reserved resources.
 *	QLA_ERROR   - Failed to reserved a resource.
 *
 * Context:
 *	Kernel context.
 **************************************************************************/
STATIC uint8_t
qla4xxx_register_resources(scsi_qla_host_t *ha)
{
	uint8_t status = QLA_SUCCESS;
	struct Scsi_Host *host = ha->host;

	ENTER("qla4xxx_register_resources");

#if MEMORY_MAPPED_IO == 0

	/* Register the I/O space with Linux */
	if (check_region(host->io_port, 0xff)) {
		QL4PRINT(QLP1, printk(KERN_WARNING
				      "scsi%d: Failed to reserved i/o base region "
				      "0x%04lx-0x%04lx already in use\n",
				      ha->host_no, host->io_port, host->io_port + 0xff));
		LEAVE("qla4xxx_register_resources");
		return(QLA_ERROR);
	}

	QL4PRINT(QLP7,
		 printk("scsi%d: %s: I/O Port 0x%lx\n",
			ha->host_no, __func__, host->io_port));

	request_region(host->io_port, 0xff, QLA4XXX_PROC_NAME);
	ha->reg = (ISP_REG_T *) host->io_port;
#endif

	/* Register the IRQ with Linux (sharable) */
	if (request_irq(host->irq,
			qla4xxx_intr_handler,
			SA_INTERRUPT|SA_SHIRQ,
			QLA4XXX_PROC_NAME,
			ha) != QLA_SUCCESS) {
		QL4PRINT(QLP1, printk(KERN_WARNING
				      "scsi%d: Failed to reserved interrupt %d.  "
				      "Already in use\n", ha->host_no, host->irq));
		release_region(host->io_port, 0xff);
		status = QLA_ERROR;
	}
	else {
		QL4PRINT(QLP7, printk("scsi%d: %s: IRQ %d\n",
				      ha->host_no, __func__, host->irq));
	}

	LEAVE("qla4xxx_register_resources");
	return(status);
}

/**************************************************************************
 * qla4xxx_select_queue_depth
 *	This routine sets the queue depth for each SCSI device logged into
 *	the input host adapter.  We use a queue depth of 2 for devices that
 *	do not support tagged queueing.
 *
 * Input:
 *	host - Pointer to Linux's SCSI Host structure
 *	scsi_devs - Pointer to Linux's SCSI Device structure
 *
 * Returns:
 *	None
 *
 * Context:
 *	Kernel context.
 **************************************************************************/
void
qla4xxx_select_queue_depth(struct Scsi_Host *host, Scsi_Device *scsi_devs)
{
	Scsi_Device *device;
	scsi_qla_host_t  *ha = (scsi_qla_host_t *) host->hostdata;

	ENTER("qla4xxx_select_queue_depth");
	for (device = scsi_devs; device != NULL; device = device->next) {
		int target = device->id;
		ddb_entry_t *ddb_entry;


		if (device->host != host)
			continue;

		ddb_entry = qla4xxx_lookup_ddb_by_SCSIID(ha, 0, target);

		if (device->tagged_supported && (ddb_entry != NULL)) {
			device->tagged_queue = 1;
			device->current_tag = 0;
			device->queue_depth = ddb_entry->exe_throttle;

#if defined(MODULE)
			if (!(ql4xmaxqdepth == 0 || ql4xmaxqdepth > 256))
				device->queue_depth = ql4xmaxqdepth;
#endif

			QL4PRINT(QLP7, printk(KERN_INFO "scsi%d:%d:%d:%d: Enabled tagged "
					      "queuing, queue depth %d.\n",
					      (int)ha->host_no, device->channel, target,
					      device->lun, device->queue_depth));
		}
		else {
			device->queue_depth = 1;
		}
	}
	LEAVE("qla4xxx_select_queue_depth");
}

/**************************************************************************
 * qla4xxx_set_info
 *	This routine set parameters for the driver from the /proc filesystem.
 *
 * Input:
 *	Unused
 *
 * Returns:
 *	-ENOSYS - no-op
 *
 * Context:
 *	Kernel context.
 **************************************************************************/
int
qla4xxx_set_info(char *buffer, int length, struct Scsi_Host *HBAptr)
{
#ifdef __VMKERNEL_MODULE__

	scsi_qla_host_t *ha = (scsi_qla_host_t *) HBAptr->hostdata;
	if (length < 13 || strncmp("scsi-qla", buffer, 8))
	   return length;

	/*
	 * Usage: echo "scsi-qlascan " > /proc/scsi/qla4010 /<adapter-id>
	 *
	 * Use the driver instance number for the adapter id.
	 *
         * The device database gets updated when targets change and we
         * get an AEN.  However, the device database doesn't get updated
         * when a target comes online.  We need to explicitly login to any
         * targets that weren't around before or have come back.  If only
         * LUNs change, we need to rescan our targets to find the new ones
         * and delete the old ones.
         */
 	if (!strncmp("scan", buffer + 8, 4)) {

	        printk("qla4010-%ld: Scanning for new luns.... \n",
                    ha->host_no);
		printk(KERN_INFO
	           "qla4010-%ld: Scanning for new luns....  \n",
		   ha->host_no);
                /*
                 * Attempt to login to any dead target entries in the
                 * FW database.
                 */
                qla4xxx_wake_the_dead(ha);
                /*
                 * Devices that just we just issued a login to might not
                 * be caught by the following call.  The ddb change AEN
                 * that follows their successful login picks up targets
                 * and luns.  This discovers any new luns on already
                 * logged in entries.
                 */
		(void)qla4xxx_map_targets_to_ddbs(ha, 0);
	}

	return(length);
#else
	return(-ENOSYS);  /* Currently this is a no-op */
#endif
}


/*
 * The following support functions are adopted to handle
 * the re-entrant qla4xxx_proc_info correctly.
 */
STATIC void
copy_mem_info(struct info_str *info, char *data, int len)
{
	if (info->pos + len > info->offset + info->length)
		len = info->offset + info->length - info->pos;

	if (info->pos + len < info->offset) {
		info->pos += len;
		return;
	}

	if (info->pos < info->offset) {
		off_t partial;

		partial = info->offset - info->pos;
		data += partial;
		info->pos += partial;
		len  -= partial;
	}

	if (len > 0) {
		memcpy(info->buffer, data, len);
		info->pos += len;
		info->buffer += len;
	}
}

STATIC int
copy_info(struct info_str *info, char *fmt, ...)
{
	va_list args;
#if 1 || __VMKERNEL_MODULE__
	char buf[256];
#else
	static char buf[256];
#endif
	int len;

	va_start(args, fmt);
	len = vsprintf(buf, fmt, args);
	va_end(args);

	copy_mem_info(info, buf, len);

	return(len);
}

/**************************************************************************
 * qla4xxx_proc_dump_srb_info
 *	This routine displays srb information in the proc buffer.
 *
 * Input:
 *	len - length of proc buffer prior to this function's execution.
 *	srb - Pointer to srb to display.
 *
 * Remarks:
 *	This routine is dependent on the DISPLAY_SRBS_IN_PROC #define being
 *	set to 1.
 *
 * Returns:
 *	len - length of proc buffer after this function's execution.
 *
 * Context:
 *	Kernel context.
 **************************************************************************/
inline void
qla4xxx_proc_dump_srb_info(scsi_qla_host_t *ha, struct info_str *info, srb_t *srb)
{
	ddb_entry_t *ddb_entry;
	lun_entry_t *lun_entry;

	ddb_entry = qla4xxx_lookup_ddb_by_fw_index(ha, srb->fw_ddb_index);
	lun_entry = qla4xxx_lookup_lun_handle(ha, ddb_entry, srb->lun);

	copy_info(info, "srb %p", srb);

	if (ddb_entry && lun_entry && srb->cmd) {
		Scsi_Cmnd *cmd = srb->cmd;
		//int i;

		copy_info(info, ", b%d,t%d,l%d, SS=%d, DS=%d, LS=%d, "
			  "r_start=%ld, u_start=%ld",
			  cmd->channel, cmd->target, cmd->lun,
			  srb->state,
			  atomic_read(&ddb_entry->state),
			  lun_entry->lun_state,
			  srb->r_start,srb->u_start);

		//copy_info(info, ", cdb=");
		//for (i=0; i<cmd->cmd_len; i++)
		//        copy_info(info, "%02X ", cmd->cmnd[i]);

	}

	copy_info(info, "\n");
}

#ifndef QLA4000
/**************************************************************************
 * qla4xxx_proc_dump_discovered_devices
 *	This routine displays information for discovered devices in the proc
 *	buffer.
 *
 * Input:
 *	info - length of proc buffer prior to this function's execution.
 *
 * Remarks:
 *	This routine is dependent on the DISPLAY_SRBS_IN_PROC #define being
 *	set to 1.
 *
 * Returns:
 *	info - length of proc buffer after this function's execution.
 *
 * Context:
 *	Kernel context.
 **************************************************************************/
inline void
qla4xxx_proc_dump_discovered_devices(scsi_qla_host_t *ha, struct info_str *info)
{
	int i,j;

	copy_info(info, "SCSI discovered device Information:\n");
	copy_info(info, "Index: DID: NameString: Alias:\n");

	for (i=0; i < ha->isns_num_discovered_targets; i++) {
		ISNS_DISCOVERED_TARGET *isns_tgt =
		&ha->dma_mem_blkv->isns_discovered_target_database[i];

		copy_info(info, "%2d: %4d:   %s: %s\n",
			  i,
			  isns_tgt->DDID,
			  isns_tgt->NameString,
			  isns_tgt->Alias);

		for (j = 0; j < isns_tgt->NumPortals; j++) {
			ISNS_DISCOVERED_TARGET_PORTAL *isns_portal =
			&isns_tgt->Portal[j];

			copy_info(info, "            Port %d: IP %d.%d.%d.%d\n",
				  isns_portal->PortNumber,
				  isns_portal->IPAddr[0],
				  isns_portal->IPAddr[1],
				  isns_portal->IPAddr[2],
				  isns_portal->IPAddr[3]);
		}
	}
}
#endif

/**************************************************************************
 * qla4xxx_proc_dump_scanned_devices
 *	This routine displays information for scanned devices in the proc
 *	buffer.
 *
 * Input:
 *	info - length of proc buffer prior to this function's execution.
 *
 * Remarks:
 *	This routine is dependent on the DISPLAY_SRBS_IN_PROC #define being
 *	set to 1.
 *
 * Returns:
 *	info - length of proc buffer after this function's execution.
 *
 * Context:
 *	Kernel context.
 **************************************************************************/
inline void
qla4xxx_proc_dump_scanned_devices(scsi_qla_host_t *ha, struct info_str *info)
{
	struct list_head *ptr, *next; /* used for traversing lists */
#if 1 || __VMKERNEL_MODULE__
	unsigned long flags;
#endif


	copy_info(info, "SCSI scanned device Information:\n");
	copy_info(info, "  H: B: T: L:\n");

	/* scan for all equipment stats */
#if 1 || __VMKERNEL_MODULE__
	spin_lock_irqsave(&ha->ddb_list_spinlock, flags);
#endif
	list_for_each_safe(ptr, next, &ha->ddb_list)
	{
		int lun;

		/* Get ddb and make sure it's valid */
		ddb_entry_t *ddb_entry = list_entry(ptr, ddb_entry_t, list_entry);

		for (lun = 0; lun < MAX_LUNS; lun++) {
			lun_entry_t *lun_entry;

			lun_entry = qla4xxx_lookup_lun_handle(ha,
							      ddb_entry,
							      lun);
			if (lun_entry) {
				copy_info(info, "(%2d:%2d:%2d:%2d): "
					  "fw_index %d, ",
					  ha->host_no, ddb_entry->bus,
					  ddb_entry->target, lun,
					  ddb_entry->fw_ddb_index);

				/* current number of outstanding requests */
				copy_info(info, "Tot I/O %2d, ",
					  lun_entry->tot_io_count);
				copy_info(info, "Active reqs %2d, ",
					  lun_entry->out_count);
				copy_info(info, "DS=%d, ",
					  atomic_read(&ddb_entry->state));
				copy_info(info, "LS=%d, ",
					  lun_entry->lun_state);
				copy_info(info, "DDBS=%d",
					  ddb_entry->fw_ddb_device_state);
				copy_info(info, ", flgs=%x",
					  ddb_entry->flags);

				if (atomic_read(&ddb_entry->state) ==
				    DEV_STATE_MISSING) {
					copy_info(info,
						  ", PDTO= %d of %d",
						  atomic_read(&ddb_entry->port_down_timer),
						  ha->port_down_retry_count);
				}

				copy_info(info, "\n");

				if (info->pos >= info->offset + info->length) {
					/* No need to continue */
#if 1 || __VMKERNEL_MODULE__
					spin_unlock_irqrestore(&ha->ddb_list_spinlock, flags);
#endif
					return;
				}
			}
		}

		if (info->pos >= info->offset + info->length) {
			/* No need to continue */
			break;
		}
	}
#if 1 || __VMKERNEL_MODULE__
	spin_unlock_irqrestore(&ha->ddb_list_spinlock, flags);
#endif
}

/**************************************************************************
 * qla4xxx_proc_info
 *	This routine return information to handle /proc support for the driver
 *
 * Input:
 * Output:
 *	inout  - Decides on the direction of the dataflow and the meaning of
 *	         the variables.
 *	buffer - If inout==FALSE data is being written to it else read from
 *	         it (ptrs to a page buffer).
 *	*start - If inout==FALSE start of the valid data in the buffer.
 *	offset - If inout==FALSE offset from the beginning of the imaginary
 *	         file from which we start writing into the buffer.
 *	length - If inout==FALSE max number of bytes to be written into the
 *	         buffer else number of bytes in the buffer.
 *	hostno - Host number
 *
 * Remarks:
 *	None
 *
 * Returns:
 *	Size of proc buffer.
 *
 * Context:
 *	Kernel context.
 **************************************************************************/
int
qla4xxx_proc_info (char *buffer, char **start, off_t offset, int length,
		   int hostno, int inout)
{
	int             retval = -EINVAL;
	scsi_qla_host_t *ha;
	struct info_str info;
	struct list_head *ptr, *next; /* used for traversing lists */
	unsigned long   flags;

	QL4PRINT(QLP16, printk("scsi%d: Entering %s: buff_in=%p, "
			       "offset=0x%lx, length=0x%x\n",
			       hostno, __func__, buffer, offset, length));

	/* Find the host that was specified */
	for (ha=qla4xxx_hostlist;
	    (ha != NULL) && ha->host->host_no != hostno;
	    ha=ha->next) {
		continue;
	}

	/* if host wasn't found then exit */
	if (!ha) {
		QL4PRINT(QLP2,
			 printk("scsi%d: %s: Can't find adapter for host number %d\n",
				hostno, __func__, hostno));

		return(retval);
	}


	if (inout == TRUE) { /* Has data been written to the file? */
		QL4PRINT(QLP3, printk("scsi%d: %s: has data been written "
				      "to the file. \n", ha->host_no, __func__));
		return(qla4xxx_set_info(buffer, length, ha->host));
	}

	if (start) {
		*start = buffer;
	}

	info.buffer = buffer;
	info.length = length;
	info.offset = offset;
	info.pos    = 0;

	/* start building the print buffer */
	copy_info(&info, "QLogic iSCSI Adapter for ISP %s:\n",
		  QLA4XXX_BOARD_ID_STRING);
	copy_info(&info, "Driver version %s\n", QLA4XXX_DRIVER_VERSION);
	copy_info(&info, "Code starts at address = %p\n", code_start);
	copy_info(&info, "Firmware version %2d.%02d.%02d.%02d\n",
		  ha->firmware_version[0], ha->firmware_version[1],
		  ha->patch_number, ha->build_number);
	copy_info(&info, "SCSI Host Adapter Information: %s\n",
		  QLA4XXX_BOARD_NAME_STRING);
#if MEMORY_MAPPED_IO
	copy_info(&info, "Memory I/O = 0x%lx\n", ha->host->base);
#else
	copy_info(&info, "I/O Port = 0x%lx\n", ha->host->io_port);
#endif
	copy_info(&info, "IP Address = %d.%d.%d.%d\n",
		  ha->ip_address[0], ha->ip_address[1],
		  ha->ip_address[2], ha->ip_address[3]);
#ifdef QLA4010
	if (ha->tcp_options & TOPT_ISNS_ENABLE) {
		copy_info(&info, "iSNS IP Address = %d.%d.%d.%d\n",
			  ha->isns_ip_address[0], ha->isns_ip_address[1],
			  ha->isns_ip_address[2], ha->isns_ip_address[3]);
		copy_info(&info, "iSNS Server Port# = %d\n",
			  ha->isns_server_port_number);
	}
#endif
	copy_info(&info, "ReqQ DMA = 0x%x, ReqQ depth= 0x%x\n",
		  ha->req_queue_headp, REQUEST_QUEUE_DEPTH);
	copy_info(&info, "ComplQ DMA = 0x%x, ComplQ depth= 0x%x\n",
		  ha->rsp_queue_headp, RESPONSE_QUEUE_DEPTH);
#ifdef QLA4010
	copy_info(&info, "Shadow Regs Addr = 0x%x, size (bytes) = 0x%x\n",
		  ha->dma_regsp, sizeof(DMA_ISP_REGS));
	copy_info(&info, "PDU Buffer Addr = 0x%x, size (bytes) = 0x%x\n",
		  ha->dma_mem_blkv->pdu_buffs, sizeof(ha->dma_mem_blkv->pdu_buffs));
	copy_info(&info, "Discovered Target Database Addr = 0x%x, size (bytes) = 0x%x\n",
		  ha->dma_mem_blkv->isns_discovered_target_database,
		  sizeof(ha->dma_mem_blkv->isns_discovered_target_database));
#endif
	copy_info(&info, "Number of free request entries  = %d of %d\n",
		  ha->req_q_count, REQUEST_QUEUE_DEPTH);
	copy_info(&info, "Number of free aen entries    = %d of %d\n",
		  ha->aen_q_count, MAX_AEN_ENTRIES);
	copy_info(&info, "Number of Mailbox Timeouts = %d\n",
		  ha->mailbox_timeout_count);
	copy_info(&info, "Total H/W int = 0x%lx\n",
		  (unsigned long)ha->isr_count);
	copy_info(&info, "Total RISC int = 0x%lx\n",
		  (unsigned long)ha->total_io_count);
#if TRACK_SPURIOUS_INTRS
	copy_info(&info, "Number of Spurious interrupts = %d\n",
		  ha->spurious_int_count);
#endif
	copy_info(&info, "Seconds since last Interrupt = %d\n",
		  ha->seconds_since_last_intr);
	copy_info(&info, "Interrupt Status = %d\n", ISP_RD_INTR_STATUS(ha));
	copy_info(&info, "ReqQptr=%p, ReqIn=%d, ReqOut=%d\n",
		  ha->request_ptr, ha->request_in, ha->request_out);
	copy_info(&info, "Device queue depth = 0x%x\n",
		  (ql4xmaxqdepth == 0) ? 16 : ql4xmaxqdepth);
	copy_info(&info, "Adapter flags = 0x%x, DPC flags = 0x%x\n",
		  ha->flags, ha->dpc_flags);
	copy_info(&info, "Number of commands in pending_srb_q = %d\n",
		  ha->pending_srb_q_count);
	if (((ql_dbg_level & QLP16) != 0) && (ha->pending_srb_q_count)) {
		copy_info(&info, "\nDump pending_srb_q:\n");
		spin_lock_irqsave(&ha->list_lock, flags);
		list_for_each_safe(ptr, next, &ha->pending_srb_q)
		{
			srb_t *srb = list_entry(ptr, srb_t , list_entry);
			qla4xxx_proc_dump_srb_info(ha, &info, srb);
		}
		spin_unlock_irqrestore(&ha->list_lock, flags);
		copy_info(&info, "\n");
	}

	copy_info(&info, "Number of commands in retry_srb_q = %d\n",
		  ha->retry_srb_q_count);

	if (((ql_dbg_level & QLP16) != 0) && (ha->retry_srb_q_count)) {
		copy_info(&info, "\nDump retry_srb_q:\n");
		spin_lock_irqsave(&ha->list_lock, flags);
		list_for_each_safe(ptr, next, &ha->retry_srb_q)
		{
			srb_t *srb = list_entry(ptr, srb_t , list_entry);
			qla4xxx_proc_dump_srb_info(ha, &info, srb);
		}
		spin_unlock_irqrestore(&ha->list_lock, flags);
		copy_info(&info, "\n");
	}

	copy_info(&info, "Number of commands in done_srb_q = %d\n",
		  ha->done_srb_q_count);
	if (((ql_dbg_level & QLP16) != 0) && (ha->done_srb_q_count)) {
		copy_info(&info, "\nDump done_srb_q:\n");
		spin_lock_irqsave(&ha->list_lock, flags);
		list_for_each_safe(ptr, next, &ha->done_srb_q)
		{
			srb_t *srb = list_entry(ptr, srb_t , list_entry);
			qla4xxx_proc_dump_srb_info(ha, &info, srb);
		}
		spin_unlock_irqrestore(&ha->list_lock, flags);
		copy_info(&info, "\n");
	}

	copy_info(&info, "Number of active commands = %d\n",
		  ha->active_srb_count);

	if (((ql_dbg_level & QLP16) != 0) && (ha->active_srb_count)) {
		int i;

		spin_lock_irqsave(&ha->hardware_lock, flags);
		copy_info(&info, "\nDump active commands:\n");
		for (i=1; i<MAX_SRBS; i++) {
			srb_t *srb = ha->active_srb_array[i];
			if (srb) qla4xxx_proc_dump_srb_info(ha, &info, srb);
		}
		copy_info(&info, "\n");
		spin_unlock_irqrestore(&ha->hardware_lock, flags);
	}

	copy_info(&info, "Total number of IOCBs (used/max) "
		  "= (%d/%d)\n", ha->iocb_cnt, ha->iocb_hiwat);
	copy_info(&info, "Number of free srbs    = %d of %d\n",
		  ha->free_srb_q_count, ha->num_srbs_allocated);
	copy_info(&info, "\n");

	qla4xxx_proc_dump_scanned_devices(ha, &info);
	copy_info(&info, "\n");

#ifndef QLA4000
	if (test_bit(ISNS_FLAG_ISNS_ENABLED_IN_ISP, &ha->isns_flags))
		qla4xxx_proc_dump_discovered_devices(ha, &info);
#endif

	copy_info(&info, "\0");
	//if (info.offset == 0)
	//	QL4PRINT(QLP2, printk("%s\n", buffer));

	retval = info.pos > info.offset ? info.pos - info.offset : 0;

	QL4PRINT(QLP16, printk("scsi%d: Exiting %s: info.pos=%d, "
			       "offset=0x%lx, length=0x%x\n",
			       hostno, __func__, info.pos, offset, length));

	return(retval);
}

/**************************************************************************
 * qla4xxx_release
 *	This routine frees the passed in Scsi_Host memory structures prior
 *	to unloading the module.
 *
 * Input:
 * 	ha - Pointer to host adapter structure.
 *
 * Returns:
 *	QLA_SUCCESS - Always return successfull completion.
 *
 * Context:
 *	Kernel context.
 **************************************************************************/
int
qla4xxx_release(struct Scsi_Host *host)
{
	scsi_qla_host_t *ha = (scsi_qla_host_t *) host->hostdata;

	ENTER("qla4xxx_release");

	if (ha) {
		#ifdef QLA4010
		unsigned long flags;
		#if 0
		u_long wait_cnt;

		/* Deregister with the iSNS Server */
		if (test_bit(ISNS_FLAG_ISNS_SRV_REGISTERED, &ha->isns_flags)) {
			QL4PRINT(QLP7, printk("scsi%d: %s: deregister iSNS\n",
					      host->host_no, __func__));
			qla4xxx_isns_scn_dereg(ha);   //FIXME: KRH
			qla4xxx_isns_dev_dereg(ha);   //FIXME: KRH

			wait_cnt = jiffies + ISNS_DEREG_TOV * HZ;
			QLA4XXX_WAIT(wait_cnt,
				     (test_bit(ISNS_FLAG_ISNS_SRV_REGISTERED,
					       &ha->isns_flags) == 0));
		}
		#endif

		if (test_bit(ISNS_FLAG_ISNS_ENABLED_IN_ISP, &ha->isns_flags)) {
			QL4PRINT(QLP7, printk("scsi%d: %s: stop iSNS service\n",
					      host->host_no, __func__));
			qla4xxx_isns_disable(ha);
		}
		#endif

		/* turn-off interrupts on the card */
		qla4xxx_disable_intrs(ha);

#ifdef QLA4010
                
		/* Issue Soft Reset to put firmware in unknown state */
		QL4PRINT(QLP7, printk("scsi%d: %s: Soft Reset\n",
				      host->host_no, __func__));
		spin_lock_irqsave(&ha->hardware_lock, flags);
		WRT_REG_DWORD(&ha->reg->ISPControlStatus, CSRW_SOFT_RESET);
		spin_unlock_irqrestore(&ha->hardware_lock, flags);
	       #endif

		/* Detach interrupt */
		QL4PRINT(QLP7, printk("scsi%d: %s: IRQ %d\n", host->host_no,
				      __func__, host->irq));
		free_irq(host->irq, ha);

		/* release io space registers  */
		if (host->io_port)
			release_region(host->io_port, 0xff);

		/* Remove timer thread, if present */
#ifdef __VMKERNEL_MODULE__
#define timer_pending(x) (x)
#endif
		if (timer_pending(&ha->timer)) {
			QL4PRINT(QLP7,
				 printk("scsi%d: %s: Removing timer thread "
					"for adapter %d\n", host->host_no,
					__func__, ha->instance));

			del_timer_sync(&ha->timer);
			ha->timer.function = NULL;
		}

		/* Kill the kernel thread for this host */
		if (ha->dpc_handler != NULL) {
                   
#ifdef __VMKERNEL_MODULE__
			extern int vmk_shutting_down(void);
			if (vmk_shutting_down()) {
				printk("qla4xxx: vmkernel shutting down\n");
				goto vmware_shutdown;
			} else {
				printk("qla4xxx: killing thread and waiting\n");
				ha->should_die = TRUE;
				up(ha->dpc_wait);
			}
#endif //__VMKERNEL_MODULE__


                   
                   QL4PRINT(QLP7, printk("scsi%d: %s: kernel thread "
                                         "qla4xxx_dpc_%d\n", host->host_no,
                                         __func__, ha->host_no));
                   ha->dpc_notify = &qla4xxx_detect_sem;
                   
                   send_sig(SIGHUP, ha->dpc_handler, 1);
                   down(&qla4xxx_detect_sem);
                   
#ifdef __VMKERNEL_MODULE__
                   printk("qla4xxx: back from killing thread\n");
vmware_shutdown:
#endif //__VMKERNEL_MODULE__
                   ha->dpc_notify = NULL;
		}

		apidev_cleanup();

		/* Free	memory back to the OS and upmap I/O region */
		qla4xxx_mem_free(ha);

#ifdef __VMKERNEL_MODULE__ 
		spin_lock_destroy(&ha->hardware_lock);
		spin_lock_destroy(&ha->aen_q_spinlock);
		spin_lock_destroy(&ha->pdu_spinlock);
		spin_lock_destroy(&ha->adapter_lock);
		spin_lock_destroy(&ha->list_lock);
		spin_lock_destroy(&ha->ddb_list_spinlock);
#endif
	}
	else {
		QL4PRINT(QLP2, printk("scsi%d: %s: NULL host adapter\n",
				      host->host_no, __func__));
	}

	LEAVE("qla4xxx_release");
	return(QLA_SUCCESS);
}

/**************************************************************************
 * qla4xxx_info
 *	This routine initialized the IOCTL interface and returns a string
 *	describing the SCSI host adapter.
 *
 * Input:
 *	host - Pointer to Linux's SCSI Host structure
 *
 * Returns:
 *	bp - Pointer to statically defined string containing description of
 *	     SCSI host adapter.
 *
 * Context:
 *	Kernel context.
 **************************************************************************/
const char *
qla4xxx_info(struct Scsi_Host *host)
{
	static char qla4xxx_info_buffer[125];
	char *bp = &qla4xxx_info_buffer[0];
	scsi_qla_host_t *ha;

	/* Create node for IOCTL interface */
	apidev_init(host);

	ha = (scsi_qla_host_t *)host->hostdata;
	memset(bp, 0, sizeof(qla4xxx_info_buffer));
	sprintf(bp,
		"QLogic iSCSI Adapter: bus %d device %d irq %d\n"
		"       Firmware v%02d.%02d.%02d.%02d, Driver v%s",
		ha->pci_bus,
		(ha->pci_device_fn & 0xf8) >> 3,
		host->irq, ha->firmware_version[0], ha->firmware_version[1],
		ha->patch_number, ha->build_number, QLA4XXX_DRIVER_VERSION);
	return(bp);
}

/**************************************************************************
 * qla4xxx_biosparam
 *	This routine returns the disk geometry for the given SCSI device.
 *
 * Input:
 *	disk - Pointer to Linux's SCSI disk structure
 *	dev  - Unused
 *	geom - Structure used to return geometry information.
 *
 * Output:
 *	geom - Returns geometry information.
 *
 * Returns:
 *	0 - Always returns success.
 *
 * Context:
 *	Kernel context.
 **************************************************************************/
int
qla4xxx_biosparam(Disk *disk, kdev_t dev, int geom[])
{
	int heads, sectors, cylinders;
	int     ret;
	struct  buffer_head *bh;

#if LINUX_VERSION_CODE >= KERNEL_VERSION(2,4,17)
	bh = bread(MKDEV(MAJOR(dev), MINOR(dev) & ~0xf), 0, block_size(dev));
#else
	bh = bread(MKDEV(MAJOR(dev), MINOR(dev) & ~0xf), 0, 1024);
#endif

	if (bh) {
		ret = scsi_partsize(bh, disk->capacity,
				    &geom[2], &geom[0], &geom[1]);
		brelse(bh);
		if (ret != -1)
			return(ret);
	}

	heads = 64;
	sectors = 32;
	cylinders = disk->capacity / (heads * sectors);
	if (cylinders > 1024) {
		heads = 255;
		sectors = 63;
		cylinders = disk->capacity / (heads * sectors);
		/* if (cylinders > 1023)
		cylinders = 1023; */
	}

	geom[0] = heads;
	geom[1] = sectors;
	geom[2] = cylinders;

	QL4PRINT(QLP7, printk(KERN_DEBUG "qla4xxx: biosparam  heads %d, sectors %d, "
			      "cylinders %d\n", heads, sectors, cylinders));
	return(0);
}

/**************************************************************************
 * qla4xxx_init_rings
 *	This routine initializes the internal queues for the specified adapter.
 *
 * Input:
 * 	ha - Pointer to host adapter structure.
 *
 * Remarks:
 *	The QLA4010 requires us to restart the queues at index 0.
 *	The QLA4000 doesn't care, so just default to QLA4010's requirement.
 * Returns:
 *	QLA_SUCCESS - Always return success.
 *
 * Context:
 *	Kernel context.
 **************************************************************************/
STATIC uint8_t
qla4xxx_init_rings(scsi_qla_host_t *ha)
{
	uint16_t    i;
	QUEUE_ENTRY *response_queue;
	unsigned long flags = 0;

	ENTER("qla4xxx_init_rings");

	/* Initialize request queue. */
	spin_lock_irqsave(&ha->hardware_lock, flags);
	ha->request_out  = 0;
	ha->request_in   = 0;
	ha->request_ptr  = &ha->req_queue_headv[ha->request_in];
	ha->req_q_count    = REQUEST_QUEUE_DEPTH;

	/* Initialize response queue. */
	ha->response_in  = 0;
	ha->response_out = 0;
	ha->response_ptr = &ha->rsp_queue_headv[ha->response_out];

	QL4PRINT(QLP7, printk("scsi%d: %s response_ptr=%p\n",
			      ha->host_no, __func__, ha->response_ptr));

#ifdef QLA4010

	/* Initialize DMA Shadow registers.  The firmware is really supposed to
	 * take care of this, but on some uniprocessor systems, the shadow
	 * registers aren't cleared-- causing the interrupt_handler to think
	 * there are responses to be processed when there aren't */
	ha->dma_regsv->requestQueueOutPointer = 0;
	ha->dma_regsv->responseQueueInPointer = 0;

	WRT_REG_DWORD(&ha->reg->requestQueueInPointer, 0);
	WRT_REG_DWORD(&ha->reg->responseQueueOutPointer, 0);
#endif

	/* Initialize response queue entries */
	response_queue = ha->rsp_queue_headv;
	for (i=0; i < RESPONSE_QUEUE_DEPTH; i++, response_queue++)
		response_queue->signature = __constant_cpu_to_le32(RESPONSE_PROCESSED);

	/* Initialize active array */
	for (i=0; i<MAX_SRBS; i++)
		ha->active_srb_array[i] = 0;
	ha->active_srb_count = 0;

	/* Initialize aen queue */
	ha->aen_q_count = MAX_AEN_ENTRIES;

	spin_unlock_irqrestore(&ha->hardware_lock, flags);
	LEAVE("qla4xxx_init_rings");
	return(QLA_SUCCESS);
}

/**************************************************************************
 * qla4xxx_initialize_fw_cb
 *	This routine initializes the firmware control block for the
 *	specified adapter.
 *
 * Input:
 * 	ha - Pointer to host adapter structure.
 *
 * Returns:
 *	QLA_SUCCESS - Successfully initialized firmware ctrl block
 *	QLA_ERROR   - Failed to initialize firmware ctrl block
 *
 * Context:
 *	Kernel context.
 **************************************************************************/
STATIC uint8_t
qla4xxx_initialize_fw_cb(scsi_qla_host_t *ha)
{
	INIT_FW_CTRL_BLK  *init_fw_cb;
	uint32_t   mbox_cmd[MBOX_REG_COUNT];
	uint32_t   mbox_sts[MBOX_REG_COUNT];
	uint8_t           status = QLA_ERROR;

	ENTER("qla4xxx_initialize_fw_cb");

	init_fw_cb = (INIT_FW_CTRL_BLK *) ha->dma_buf.virt_addr;
	memset(init_fw_cb, 0, sizeof(INIT_FW_CTRL_BLK));

#if 0 && __VMKERNEL_MODULE__
	spin_lock(&ha->dma_buf_spinlock);
#else
	down(&ha->dma_buf_sem);
#endif

	/*
	 * Get Initialize Firmware Control Block
	 */
	memset(&mbox_cmd, 0, sizeof(mbox_cmd));
	memset(&mbox_sts, 0, sizeof(mbox_sts));
	mbox_cmd[0] = MBOX_CMD_GET_INIT_FW_CTRL_BLOCK;
	mbox_cmd[2] = LSDW(ha->dma_buf.phys_addr);
	mbox_cmd[3] = MSDW(ha->dma_buf.phys_addr);

	if (qla4xxx_mailbox_command(ha, 4, 1, &mbox_cmd[0], &mbox_sts[0])
	    != QLA_SUCCESS) {
		QL4PRINT(QLP2,
			 printk("scsi%d: %s: Failed to get init_fw_ctrl_blk\n",
				ha->host_no, __func__));
#if 0 && __VMKERNEL_MODULE__
		spin_unlock(&ha->dma_buf_spinlock);
#else
		up(&ha->dma_buf_sem);
#endif
		LEAVE("qla4xxx_initialize_fw_cb");
		return(status);
	}

	QL4PRINT(QLP10, printk("scsi%d: Init Fw Ctrl Blk\n", ha->host_no));
	qla4xxx_dump_bytes(QLP10, ha->dma_buf.virt_addr,
			   sizeof(INIT_FW_CTRL_BLK));

	/*
	 * Initialize request and response queues
	 */
	qla4xxx_init_rings(ha);

	/*
	 * Fill in the request and response queue information
	 */
	init_fw_cb->ReqQConsumerIndex  = __cpu_to_le16(ha->request_out);
	init_fw_cb->ComplQProducerIndex   = __cpu_to_le16(ha->response_in);
	init_fw_cb->ReqQLen     = __constant_cpu_to_le16(REQUEST_QUEUE_DEPTH);
	init_fw_cb->ComplQLen     = __constant_cpu_to_le16(RESPONSE_QUEUE_DEPTH);
	init_fw_cb->ReqQAddrLo  = __cpu_to_le32(LSDW(ha->req_queue_headp));
	init_fw_cb->ReqQAddrHi  = __cpu_to_le32(MSDW(ha->req_queue_headp));
	init_fw_cb->ComplQAddrLo  = __cpu_to_le32(LSDW(ha->rsp_queue_headp));
	init_fw_cb->ComplQAddrHi  = __cpu_to_le32(MSDW(ha->rsp_queue_headp));

#ifdef QLA4010
	init_fw_cb->ShadowRegBufAddrLo = __cpu_to_le32(LSDW(ha->dma_regsp));
	init_fw_cb->ShadowRegBufAddrHi = __cpu_to_le32(MSDW(ha->dma_regsp));
#endif

	/*
	 * Set up required options
	 */
	init_fw_cb->FwOptions |= (FWOPT_SESSION_MODE | FWOPT_INITIATOR_MODE);
	init_fw_cb->FwOptions &= ~FWOPT_TARGET_MODE;

	/*
	 * Save some info in adapter structure
	 */
	ha->firmware_options        = __le16_to_cpu(init_fw_cb->FwOptions);
	ha->tcp_options             = __le16_to_cpu(init_fw_cb->TCPOptions);
	ha->heartbeat_interval      = init_fw_cb->HeartbeatInterval;
	ha->isns_server_port_number = __le16_to_cpu(init_fw_cb->iSNSServerPortNumber);
	memcpy(ha->ip_address, init_fw_cb->IPAddr,
	       MIN(sizeof(ha->ip_address), sizeof(init_fw_cb->IPAddr)));
	memcpy(ha->isns_ip_address, init_fw_cb->iSNSIPAddr,
	       MIN(sizeof(ha->isns_ip_address), sizeof(init_fw_cb->iSNSIPAddr)));
	memcpy(ha->name_string, init_fw_cb->iSCSINameString,
	       MIN(sizeof(ha->name_string), sizeof(init_fw_cb->iSCSINameString)));
	memcpy(ha->alias, init_fw_cb->Alias,
	       MIN(sizeof(ha->alias), sizeof(init_fw_cb->Alias)));

	/* Save Command Line Paramater info */
	ha->port_down_retry_count = ql4xportdownretrycount;
	ha->discovery_wait = ql4xdiscoverywait;

	/* High-water mark of IOCBs */
	ha->iocb_hiwat = MAX_SRBS-2;

	/*
	 * Send Initialize Firmware Control Block
	 */
	QL4PRINT(QLP7, printk("scsi%d: %s: init_fw cmd sent\n",
			      ha->host_no, __func__));
	mbox_cmd[0] = MBOX_CMD_INITIALIZE_FIRMWARE;
	mbox_cmd[1] = 0;
	mbox_cmd[2] = LSDW(ha->dma_buf.phys_addr);
	mbox_cmd[3] = MSDW(ha->dma_buf.phys_addr);

	if (qla4xxx_mailbox_command(ha, 4, 1, &mbox_cmd[0], &mbox_sts[0])
	    == QLA_SUCCESS) {
		QL4PRINT(QLP10, printk("scsi%d: Init Fw Ctrl Blk\n",
				       ha->host_no));
		qla4xxx_dump_bytes(QLP10, ha->dma_buf.virt_addr,
				   sizeof(INIT_FW_CTRL_BLK));
		QL4PRINT(QLP7, printk("scsi%d: %s: IP Address      %d.%d.%d.%d\n",
				      ha->host_no, __func__,
				      ha->ip_address[0],
				      ha->ip_address[1],
				      ha->ip_address[2],
				      ha->ip_address[3]));
		QL4PRINT(QLP7, printk("scsi%d: %s: Subnet Mask     %d.%d.%d.%d\n",
				      ha->host_no, __func__,
				      init_fw_cb->SubnetMask[0],
				      init_fw_cb->SubnetMask[1],
				      init_fw_cb->SubnetMask[2],
				      init_fw_cb->SubnetMask[3]));
		QL4PRINT(QLP7, printk("scsi%d: %s: Default Gateway %d.%d.%d.%d\n",
				      ha->host_no, __func__,
				      init_fw_cb->GatewayIPAddr[0],
				      init_fw_cb->GatewayIPAddr[1],
				      init_fw_cb->GatewayIPAddr[2],
				      init_fw_cb->GatewayIPAddr[3]));

		#ifdef QLA4010
		QL4PRINT(QLP7, printk("scsi%d: %s: Auto-Negotiate  %s\n",
				      ha->host_no, __func__,
				      ((init_fw_cb->AddFwOptions & 0x10) != 0)
				      ? "ON" : "OFF"));
		QL4PRINT(QLP7, printk("scsi%d: %s: SLP Use DA Enable  %s\n",
				      ha->host_no, __func__,
				      ((ha->tcp_options & TOPT_SLP_USE_DA_ENABLE) != 0)
				      ? "ON" : "OFF"));
		QL4PRINT(QLP7, printk("scsi%d: %s: SLP UA Enable  %s\n",
				      ha->host_no, __func__,
				      ((ha->tcp_options & TOPT_SLP_UA_ENABLE) != 0)
				      ? "ON" : "OFF"));
		QL4PRINT(QLP7, printk("scsi%d: %s: DHCP Enable  %s\n",
				      ha->host_no, __func__,
				      ((ha->tcp_options & TOPT_DHCP_ENABLE) != 0)
				      ? "ON" : "OFF"));
		QL4PRINT(QLP7, printk("scsi%d: %s: DNS via DHCP Enable  %s\n",
				      ha->host_no, __func__,
				      ((ha->tcp_options & TOPT_GET_DNS_VIA_DHCP_ENABLE) != 0)
				      ? "ON" : "OFF"));
		QL4PRINT(QLP7, printk("scsi%d: %s: SLP via DHCP Enable  %s\n",
				      ha->host_no, __func__,
				      ((ha->tcp_options & TOPT_GET_SLP_VIA_DHCP_ENABLE) != 0)
				      ? "ON" : "OFF"));
		QL4PRINT(QLP7, printk("scsi%d: %s: Auto Discovery Enable  %s\n",
				      ha->host_no, __func__,
				      ((init_fw_cb->TCPOptions & TOPT_AUTO_DISCOVERY_ENABLE) != 0)
				      ? "ON" : "OFF"));
		QL4PRINT(QLP7|QLP20, printk("scsi%d: %s: iSNS Enable  %s\n",
					    ha->host_no, __func__,
					    ((init_fw_cb->TCPOptions & TOPT_ISNS_ENABLE) != 0)
					    ? "ON" : "OFF"));
		QL4PRINT(QLP7|QLP20, printk("scsi%d: %s: Learn iSNS IP Addr Enable  %s\n",
					    ha->host_no, __func__,
					    ((init_fw_cb->TCPOptions & TOPT_LEARN_ISNS_IP_ADDR_ENABLE) != 0)
					    ? "ON" : "OFF"));
		if (init_fw_cb->TCPOptions & TOPT_ISNS_ENABLE) {
			set_bit(ISNS_FLAG_ISNS_ENABLED_IN_ISP, &ha->isns_flags);

			QL4PRINT(QLP7|QLP20,
				 printk("scsi%d: %s: iSNS IP Address  %d.%d.%d.%d\n",
					ha->host_no, __func__,
					ha->isns_ip_address[0],
					ha->isns_ip_address[1],
					ha->isns_ip_address[2],
					ha->isns_ip_address[3]));
			QL4PRINT(QLP7|QLP20,
				 printk("scsi%d: %s: iSNS Server Port Number %d\n",
					ha->host_no, __func__,
					ha->isns_server_port_number));
		}

		QL4PRINT(QLP7, printk("scsi%d: %s: Heartbeat Enable  %s\n",
				      ha->host_no, __func__,
				      ((init_fw_cb->FwOptions & FWOPT_HEARTBEAT_ENABLE) != 0)
				      ? "ON" : "OFF"));
		if (init_fw_cb->FwOptions & FWOPT_HEARTBEAT_ENABLE)
			QL4PRINT(QLP7, printk("scsi%d: %s: Heartbeat Interval  %d\n",
					      ha->host_no, __func__, ha->heartbeat_interval));
		#endif
		status = QLA_SUCCESS;
	}
	else {
		QL4PRINT(QLP2,
			 printk("scsi%d: %s: MBOX_CMD_INITIALIZE_FIRMWARE "
				"failed w/ status %04X\n",
				ha->host_no, __func__, mbox_sts[0]));
	}

#if 0 && __VMKERNEL_MODULE__
	spin_unlock(&ha->dma_buf_spinlock);
#else
	up(&ha->dma_buf_sem);
#endif
	LEAVE("qla4xxx_initialize_fw_cb");
	return(status);
}

#ifdef QLA4000
/**************************************************************************
 * qla4xxx_get_fw_version
 *	This routine retrieves the firmware version for the specified adapter.
 *
 * Input:
 * 	ha - Pointer to host adapter structure.
 *
 * Output:
 *	None
 *
 * Remarks:
 *	In QLA4010, mailboxes 2 & 3 may hold an address for data.  Make sure
 *	that we write 0 to those mailboxes, if unused.
 *
 * Returns:
 *	QLA_SUCCESS - Successfully retrieved firmware version
 *	QLA_ERROR   - Failed to retrieve firmware version
 *
 * Context:
 *	Kernel context.
 **************************************************************************/
STATIC uint8_t
qla4xxx_get_fw_version(scsi_qla_host_t *ha)
{
	uint32_t mbox_cmd[MBOX_REG_COUNT];
	uint32_t mbox_sts[MBOX_REG_COUNT];

	/*
	 * Get firmware version
	 */
	memset(&mbox_cmd, 0, sizeof(mbox_cmd));
	memset(&mbox_sts, 0, sizeof(mbox_sts));
	mbox_cmd[0] = MBOX_CMD_ABOUT_FW;
	if (qla4xxx_mailbox_command(ha, 4, 5, &mbox_cmd[0], &mbox_sts[0])
	    != QLA_SUCCESS) {
		QL4PRINT(QLP2, printk("scsi%d: %s: MBOX_CMD_ABOUT_FW failed w/ "
				      "status %04X\n",
				      ha->host_no, __func__, mbox_sts[0]));
		return(QLA_ERROR);
	}

	/*
	 * Save firmware version information
	 */
	ha->firmware_version[0] = mbox_sts[1];
	ha->firmware_version[1] = mbox_sts[2];
	ha->patch_number        = mbox_sts[3];
	ha->build_number        = mbox_sts[4];

	QL4PRINT(QLP7, printk("scsi%d: FW Version %02d.%02d Patch %02d Build %02d\n",
			      ha->host_no, ha->firmware_version[0], ha->firmware_version[1],
			      ha->patch_number, ha->build_number));

	return(QLA_SUCCESS);
}
#endif

/**************************************************************************
 * qla4xxx_find_propname
 *	Get property in database.
 *
 * Input:
 *	ha = adapter structure pointer.
 *      db = pointer to database
 *      propstr = pointer to dest array for string
 *	propname = name of property to search for.
 *	siz = size of property
 *
 * Returns:
 *	0 = no property
 *      size = index of property
 *
 * Context:
 *	Kernel context.
 **************************************************************************/
STATIC uint8_t
qla4xxx_find_propname(scsi_qla_host_t *ha,
		      char *propname, char *propstr,
		      char *db, int siz)
{
	char    *cp;

	/* find the specified string */
	if (db) {
		/* find the property name */
		if ((cp = strstr(db,propname)) != NULL) {
			while ((*cp)  && *cp != '=')
				cp++;
			if (*cp) {
				strncpy(propstr, cp, siz+1);
				propstr[siz+1] = '\0';
				QL4PRINT(QLP7, printk("scsi%d: %s: found "
						      "property = {%s}\n",
						      ha->host_no, __func__,
						      propstr));
				return(siz);   /* match */
			}
		}
	}

	return(0);
}


/**************************************************************************
 * qla4xxx_get_prop_12chars
 *	Get a 6-byte property value for the specified property name by
 *      converting from the property string found in the configuration file.
 *      The resulting converted value is in big endian format (MSB at byte0).
 *
 * Input:
 *	ha = adapter state pointer.
 *	propname = property name pointer.
 *	propval  = pointer to location for the converted property val.
 *      db = pointer to database
 *
 * Returns:
 *	0 = value returned successfully.
 *
 * Context:
 *	Kernel context.
 **************************************************************************/
static int
qla4xxx_get_prop_12chars(scsi_qla_host_t *ha, uint8_t *propname,
			 uint8_t *propval, uint8_t *db)
{
	char            *propstr;
	int             i, k;
	int             rval;
	uint8_t         nval;
	uint8_t         *pchar;
	uint8_t         *ret_byte;
	uint8_t         *tmp_byte;
	uint8_t         *retval = (uint8_t*)propval;
	uint8_t         tmpval[6] = {0, 0, 0, 0, 0, 0};
	uint16_t        max_byte_cnt = 6; /* 12 chars = 6 bytes */
	uint16_t        max_strlen = 12;
	static char     buf[LINESIZE];

	rval = qla4xxx_find_propname(ha, propname, buf, db, max_strlen);

	propstr = &buf[0];
	if (*propstr == '=')
		propstr++;   /* ignore equal sign */

	if (rval == 0) {
		return(1);
	}

	/* Convert string to numbers. */
	pchar = (uint8_t *)propstr;
	tmp_byte = (uint8_t *)tmpval;

	rval = 0;
	for (i = 0; i < max_strlen; i++) {
		/*
		 * Check for invalid character, two at a time,
		 * then convert them starting with first byte.
		 */

		if ((pchar[i] >= '0') && (pchar[i] <= '9')) {
			nval = pchar[i] - '0';
		}
		else if ((pchar[i] >= 'A') && (pchar[i] <= 'F')) {
			nval = pchar[i] - 'A' + 10;
		}
		else if ((pchar[i] >= 'a') && (pchar[i] <= 'f')) {
			nval = pchar[i] - 'a' + 10;
		}
		else {
			/* invalid character */
			rval = 1;
			break;
		}

		if (i & 0x01) {
			*tmp_byte = *tmp_byte | nval;
			tmp_byte++;
		}
		else {
			*tmp_byte = *tmp_byte | nval << 4;
		}
	}

	if (rval != 0) {
		/* Encountered invalid character. */
		return(rval);
	}

	/* Copy over the converted value. */
	ret_byte = retval;
	tmp_byte = tmpval;

	i = max_byte_cnt;
	k = 0;
	while (i--) {
		*ret_byte++ = *tmp_byte++;
	}

	/* big endian retval[0]; */
	return(QLA_SUCCESS);
}

#define qla4xxx_mac_is_equal(mac1, mac2) (memcmp(mac1, mac2, MAC_ADDR_LEN) == 0)

/**************************************************************************
 * qla4xxx_validate_mac_address
 *	This routine validates the M.A.C. Address(es) of the adapter
 *
 * Input:
 * 	ha - Pointer to host adapter structure.
 *
 * Returns:
 *	QLA_SUCCESS - Successfully validated M.A.C. address
 *	QLA_ERROR   - Failed to validate M.A.C. address
 *
 * Context:
 *	Kernel context.
 **************************************************************************/
STATIC uint8_t
qla4xxx_validate_mac_address(scsi_qla_host_t *ha)
{
	uint32_t mbox_cmd[MBOX_REG_COUNT];
	uint32_t mbox_sts[MBOX_REG_COUNT];
	FLASH_SYS_INFO *sys_info;
	uint8_t status = QLA_ERROR;

	ENTER("qla4xxx_validate_mac_address");
#if 0 && __VMKERNEL_MODULE__
	spin_lock(&ha->dma_buf_spinlock);
#else
	down(&ha->dma_buf_sem);
#endif
	sys_info = (FLASH_SYS_INFO *) ha->dma_buf.virt_addr;
	memset(sys_info, 0, sizeof(*sys_info));

	/* Get flash sys info */
	memset(&mbox_cmd, 0, sizeof(mbox_cmd));
	memset(&mbox_sts, 0, sizeof(mbox_sts));
	mbox_cmd[0] = MBOX_CMD_READ_FLASH;
	mbox_cmd[1] = LSDW(ha->dma_buf.phys_addr);
	mbox_cmd[2] = MSDW(ha->dma_buf.phys_addr);
	mbox_cmd[3] = FLASH_OFFSET_SYS_INFO;
	mbox_cmd[4] = sizeof(*sys_info);

	if (qla4xxx_mailbox_command(ha, 5, 2, &mbox_cmd[0], &mbox_sts[0])
	    != QLA_SUCCESS) {
		QL4PRINT(QLP2, printk("scsi%d: %s: MBOX_CMD_READ_FLASH failed w/ "
				      "status %04X\n",
				      ha->host_no, __func__, mbox_sts[0]));
		goto exit_validate_mac;
	}

	/* Save M.A.C. address & serial_number */
	memcpy(ha->my_mac, &sys_info->physAddr[0].address[0],
	       MIN(sizeof(ha->my_mac), sizeof(sys_info->physAddr[0].address)));
	memcpy(ha->serial_number, &sys_info->acSerialNumber,
	       MIN(sizeof(ha->serial_number), sizeof(sys_info->acSerialNumber)));


	/* Display Debug Print Info */
	QL4PRINT(QLP10, printk("scsi%d: Flash Sys Info\n",
			       ha->host_no));
	qla4xxx_dump_bytes(QLP10, sys_info, sizeof(*sys_info));

	QL4PRINT(QLP7,
		 printk("scsi%d: MAC Addr0 %02x-%02x-%02x-"
			"%02x-%02x-%02x\n", ha->host_no,
			ha->my_mac[0],
			ha->my_mac[1],
			ha->my_mac[2],
			ha->my_mac[3],
			ha->my_mac[4],
			ha->my_mac[5]));

	/* If configuration information was specified on the command line,
	 * validate the mac address here.
	 */
	if (ql4xdevconf) {
		static char propbuf[LINESIZE];
		uint8_t cfg_mac[MAC_ADDR_LEN];

		/* Get mac address from configuration file. */
		sprintf(propbuf, "scsi-qla%d-mac", (int) ha->instance);
		qla4xxx_get_prop_12chars(ha, propbuf, &cfg_mac[0], ql4xdevconf);

		if (qla4xxx_mac_is_equal(&ha->my_mac, cfg_mac)) {
			QL4PRINT(QLP7, printk("scsi%d: %s: This is a registered "
					      "adapter.\n",
					      ha->host_no, __func__));
			status = QLA_SUCCESS;
		}
		else {
			QL4PRINT(QLP7, printk("scsi%d: %s: This is NOT a "
					      "registered adapter.\n",
					      ha->host_no, __func__));
		}
	}
	else {
		status = QLA_SUCCESS;
	}

	exit_validate_mac:
#if 0 && __VMKERNEL_MODULE__
	spin_unlock(&ha->dma_buf_spinlock);
#else
	up(&ha->dma_buf_sem);
#endif
	LEAVE("qla4xxx_validate_mac_address");
	return(status);
}

#ifndef QLA4000
/**************************************************************************
 * qla4xxx_init_local_data
 *	This routine initializes the local data for the specified adapter.
 *
 * Input:
 * 	ha - Pointer to host adapter structure.
 *
 * Returns:
 *	QLA_SUCCESS - Successfully initialized local data
 *	QLA_ERROR   - Failed to initialize local data
 *
 * Context:
 *	Kernel context.
 **************************************************************************/
uint8_t
qla4xxx_init_local_data(scsi_qla_host_t *ha)
{
	int i;

	/* Initialize passthru PDU list */
	memset(ha->pdu_buf_used, 0, sizeof(ha->pdu_buf_used));
	memset(ha->dma_mem_blkv->pdu_buffs, 0, sizeof(ha->dma_mem_blkv->pdu_buffs));
	for (i=0; i < (MAX_PDU_ENTRIES - 1); i++) {
		ha->pdu_queue[i].Next = &ha->pdu_queue[i+1];
	}
	ha->free_pdu_top = &ha->pdu_queue[0];
	ha->free_pdu_bottom = &ha->pdu_queue[MAX_PDU_ENTRIES - 1];
	ha->free_pdu_bottom->Next = NULL;
	ha->pdu_active = 0;

	/* Initialize local iSNS data */
	qla4xxx_isns_init_attributes(ha);
	ha->isns_flags = 0;
	atomic_set(&ha->isns_restart_timer, 0);
	ha->isns_connection_id = 0;
	ha->isns_remote_port_num = 0;
	ha->isns_scn_port_num = 0;
	ha->isns_esi_port_num = 0;
	ha->isns_nsh_port_num = 0;
	memset(ha->isns_entity_id, 0, sizeof(ha->isns_entity_id));
	ha->isns_num_discovered_targets = 0;

	return(QLA_SUCCESS);
}
#endif

/**************************************************************************
 * qla4xxx_get_firmware_state
 *	This routine retrieves the firmware state for the specified adapter.
 *
 * Input:
 * 	ha - Pointer to host adapter structure.
 *
 * Returns:
 *	QLA_SUCCESS - Successfully retrieved firmware state
 *	QLA_ERROR   - Failed to retrieve firmware state
 *
 * Context:
 *	Kernel context.
 **************************************************************************/
uint8_t
qla4xxx_get_firmware_state(scsi_qla_host_t *ha)
{
	uint32_t mbox_cmd[MBOX_REG_COUNT];
	uint32_t mbox_sts[MBOX_REG_COUNT];

	ENTER("qla4xxx_get_firmware_state");

	/* Get firmware version */
	memset(&mbox_cmd, 0, sizeof(mbox_cmd));
	memset(&mbox_sts, 0, sizeof(mbox_sts));
	mbox_cmd[0] = MBOX_CMD_GET_FW_STATE;
	if (qla4xxx_mailbox_command(ha, 1, 4, &mbox_cmd[0], &mbox_sts[0])
	    != QLA_SUCCESS) {
		QL4PRINT(QLP2,
			 printk("scsi%d: %s: MBOX_CMD_GET_FW_STATE failed w/ "
				"status %04X\n",
				ha->host_no, __func__, mbox_sts[0]));
		return(QLA_ERROR);
	}

	ha->firmware_state = mbox_sts[1];
	ha->board_id       = mbox_sts[2];
	ha->addl_fw_state  = mbox_sts[3];
	LEAVE("qla4xxx_get_firmware_state");
	return(QLA_SUCCESS);
}

/**************************************************************************
 * qla4xxx_init_firmware
 *	This routine initializes the firmware.
 *
 * Input:
 * 	ha - Pointer to host adapter structure.
 *
 * Returns:
 *	QLA_SUCCESS - Successfully initialized firmware
 *	QLA_ERROR   - Failed to initialize firmware
 *
 * Context:
 *	Kernel context.
 **************************************************************************/
STATIC uint8_t
qla4xxx_init_firmware(scsi_qla_host_t *ha)
{
	uint8_t  status = QLA_ERROR;
	uint32_t timeout_count;

	ENTER("qla4xxx_init_firmware");

	if (qla4xxx_initialize_fw_cb(ha) == QLA_ERROR) {
		LEAVE("qla4xxx_init_firmware");
		return(QLA_ERROR);
	}

#ifndef QLA4000
	/*
	 * If iSNS is enabled, start the iSNS service now.
	 */
	if ((ha->tcp_options & TOPT_ISNS_ENABLE) &&
	    !IPAddrIsZero(ha->isns_ip_address)) {
		uint32_t ip_addr = 0;
		IPAddr2Uint32(ha->isns_ip_address, &ip_addr);

		qla4xxx_isns_reenable(ha, ip_addr,
				      ha->isns_server_port_number);
	}
#endif

	for (timeout_count = ADAPTER_INIT_TOV;
	    timeout_count > 0;
	    timeout_count--) {

		/* Get firmware state */
		if (qla4xxx_get_firmware_state(ha) != QLA_SUCCESS) {
			QL4PRINT(QLP2,
				 printk("scsi%d: %s: unable to get firmware "
					"state\n", ha->host_no, __func__));
			LEAVE("qla4xxx_init_firmware");
			return(QLA_ERROR);
		}

		if (ha->firmware_state & FW_STATE_ERROR) {
			QL4PRINT(QLP1,
				 printk("scsi%d: %s: an unrecoverable "
					"error has occurred\n",
					ha->host_no, __func__));
			LEAVE("qla4xxx_init_firmware");
			return(QLA_ERROR);
		}
		if (ha->firmware_state & FW_STATE_CONFIG_WAIT) {
			/*
			 * The firmware has not yet been issued
			 * an Initialize Firmware command, so
			 * issue it now. */
			if (qla4xxx_initialize_fw_cb(ha) == QLA_ERROR) {
				LEAVE("qla4xxx_init_firmware");
				return(status);
			}

			/* Go back and test for ready state
			 * (without waiting) */
			continue;
		}

		if (ha->firmware_state & FW_STATE_WAIT_LOGIN) {
			QL4PRINT(QLP7,
				 printk("scsi%d: %s: waiting for firmware"
					" to initialize\n",
					ha->host_no, __func__));
		}

#ifdef QLA4010
		if (ha->firmware_state & FW_STATE_DHCP_IN_PROGRESS) {
			QL4PRINT(QLP7, printk("scsi%d: %s: DHCP in progress\n",
					      ha->host_no, __func__));
		}
#endif
		if (ha->firmware_state == FW_STATE_READY) {
			/* The firmware is ready to process
			 * SCSI commands. */
			qla4xxx_enable_intrs(ha);

			QL4PRINT(QLP7, printk("scsi%d: %s: FW STATE - READY\n",
					      ha->host_no, __func__));
			QL4PRINT(QLP7, printk("scsi%d: %s: MEDIA TYPE - %s\n",
					      ha->host_no, __func__,
					      ((ha->addl_fw_state & FW_ADDSTATE_OPTICAL_MEDIA) != 0)
					      ? "OPTICAL" : "COPPER"));
			#ifdef QLA4010
			QL4PRINT(QLP7, printk("scsi%d: %s: DHCP STATE Enabled  %s\n",
					      ha->host_no, __func__,
					      ((ha->addl_fw_state & FW_ADDSTATE_DHCP_ENABLED) != 0)
					      ? "YES" : "NO"));
			QL4PRINT(QLP7, printk("scsi%d: %s: DHCP STATE Lease Acquired  %s\n",
					      ha->host_no, __func__,
					      ((ha->addl_fw_state & FW_ADDSTATE_DHCP_LEASE_ACQUIRED) != 0)
					      ? "YES" : "NO"));
			QL4PRINT(QLP7, printk("scsi%d: %s: DHCP STATE Lease Expired  %s\n",
					      ha->host_no, __func__,
					      ((ha->addl_fw_state & FW_ADDSTATE_DHCP_LEASE_EXPIRED) != 0)
					      ? "YES" : "NO"));
			QL4PRINT(QLP7, printk("scsi%d: %s: LINK  %s\n",
					      ha->host_no, __func__,
					      ((ha->addl_fw_state & FW_ADDSTATE_LINK_UP) != 0)
					      ? "UP" : "DOWN"));
			QL4PRINT(QLP7, printk("scsi%d: %s: iSNS Service Enabled  %s\n",
					      ha->host_no, __func__,
					      ((ha->addl_fw_state & FW_ADDSTATE_ISNS_SVC_ENABLED) != 0)
					      ? "YES" : "NO"));
			#if QLA4040_SUPPORT_ENABLED
			QL4PRINT(QLP7, printk("scsi%d: %s: QLA4040 TopCat Initialized  %s\n",
					      ha->host_no, __func__,
					      ((ha->addl_fw_state & FW_ADDSTATE_TOPCAT_NOT_INITIALIZED) == 0)
					      ? "YES" : "NO"));
			#endif
			#endif

			goto exit_init_fw;
		}

		QL4PRINT(QLP7,
			 printk("scsi%d: %s: (%x/%x) delay 1 sec, time remaining %d\n",
				ha->host_no, __func__, ha->firmware_state, ha->addl_fw_state, timeout_count));
		set_current_state(TASK_INTERRUPTIBLE);
		schedule_timeout(1 * HZ);
	} /* for */

	QL4PRINT(QLP2,
		 printk("scsi%d: %s: FW Initialization timed out!\n",
			ha->host_no, __func__));

	exit_init_fw:
	set_bit(AF_ONLINE, &ha->flags);
	LEAVE("qla4xxx_init_firmware");
	return(QLA_SUCCESS);
}

/**************************************************************************
 * qla4xxx_login_device
 *	This routine is called by the login IOCTL to log in the specified
 *	device.
 *
 * Input:
 * 	ha - Pointer to host adapter structure.
 * 	fw_ddb_index - Index of the device to login
 * 	connection_id - Connection ID of the device to login
 *
 * Returns:
 *	QLA_SUCCESS - Successfully logged in device
 *	QLA_ERROR   - Failed to login device
 *
 * Context:
 *	Kernel context.
 **************************************************************************/
uint8_t
qla4xxx_login_device(scsi_qla_host_t *ha,
		     uint16_t fw_ddb_index,
		     uint16_t connection_id)
{
	ddb_entry_t *ddb_entry;
	uint32_t    mbox_cmd[MBOX_REG_COUNT];
	uint32_t    mbox_sts[MBOX_REG_COUNT];
	uint8_t     status = QLA_ERROR;


	ENTER("qla4xxx_login_device");

	QL4PRINT(QLP3, printk("scsi%d: %s: Login index [%d]\n",
			      ha->host_no, __func__, fw_ddb_index));

	ddb_entry = qla4xxx_lookup_ddb_by_fw_index(ha, fw_ddb_index);

	if (ddb_entry == NULL) {
		QL4PRINT(QLP2,
			 printk("scsi%d: %s: Invalid index [%d]\n",
				ha->host_no, __func__, fw_ddb_index));
		goto exit_login_device;
	}

	if (qla4xxx_get_ddb_entry(ha, fw_ddb_index,
				  NULL, NULL, NULL,
				  &ddb_entry->fw_ddb_device_state,
				  NULL, NULL, NULL) == QLA_ERROR) {
		QL4PRINT(QLP2, printk("scsi%d: %s: 1st get ddb entry failed\n",
				      ha->host_no, __func__));
		goto exit_login_device;
	}

	if (ddb_entry->fw_ddb_device_state == DDB_DS_SESSION_ACTIVE) {
		status = QLA_SUCCESS;
		QL4PRINT(QLP3,
			 printk("scsi%d: %s: login successful for index [%d]\n",
				ha->host_no, __func__, ddb_entry->fw_ddb_index));
		goto exit_login_device;
	}

	memset(&mbox_cmd, 0, sizeof(mbox_cmd));
	memset(&mbox_sts, 0, sizeof(mbox_sts));
	mbox_cmd[0] = MBOX_CMD_CONN_CLOSE_SESS_LOGOUT;
	mbox_cmd[1] = fw_ddb_index;
	mbox_cmd[2] = connection_id;
	mbox_cmd[3] = LOGOUT_OPTION_RELOGIN;

	if (qla4xxx_mailbox_command(ha, 4, 2, &mbox_cmd[0], &mbox_sts[0])
	    != QLA_SUCCESS) {
		QL4PRINT(QLP2,
			 printk("scsi%d: %s: MBOX_CMD_CONN_CLOSE_SESS_LOGOUT "
				"failed sts %04X",
				ha->host_no, __func__, mbox_sts[0]));

		if (mbox_sts[0] == 0x4005) {
			QL4PRINT(QLP2,
				 printk(", reason %04X\n", mbox_sts[1]));
		}
		else {
			QL4PRINT(QLP2, printk("\n"));
		}

		goto exit_login_device;
	}

	status = QLA_SUCCESS;

	exit_login_device:
	LEAVE("qla4xxx_login_device");
	return(status);
}

/**************************************************************************
 * qla4xxx_logout_device
 *	This routine is called by the logout IOCTL to log out the specified
 *    device.
 *
 * Input:
 * 	ha - Pointer to host adapter structure.
 * 	fw_ddb_index - Index of the device to logout
 * 	connection_id - Connection ID of the device to logout
 *
 * Returns:
 *	QLA_SUCCESS - Successfully logged out device
 *	QLA_ERROR   - Failed to logout device
 *
 * Context:
 *	Kernel context.
 **************************************************************************/
uint8_t
qla4xxx_logout_device(scsi_qla_host_t *ha,
		      uint16_t fw_ddb_index,
		      uint16_t connection_id)
{
	uint8_t     status = QLA_ERROR;
	uint32_t    mbox_cmd[MBOX_REG_COUNT];
	uint32_t    mbox_sts[MBOX_REG_COUNT];
	ddb_entry_t *ddb_entry;
	uint32_t    old_fw_ddb_device_state;

	ENTER("qla4xxx_logout_device");

	QL4PRINT(QLP3, printk("scsi%d: %s: Logout index [%d]\n",
			      ha->host_no, __func__, fw_ddb_index));

	ddb_entry = qla4xxx_lookup_ddb_by_fw_index(ha, fw_ddb_index);

	if (ddb_entry == NULL) {
		QL4PRINT(QLP2, printk("scsi%d: %s: Invalid index [%d]\n",
				      ha->host_no, __func__, fw_ddb_index));
		goto exit_logout_device;
	}

	if (qla4xxx_get_ddb_entry(ha, fw_ddb_index,
				  NULL, NULL, NULL,
				  &old_fw_ddb_device_state,
				  NULL, NULL, NULL) != QLA_SUCCESS) {
		QL4PRINT(QLP2, printk("scsi%d: %s: get_ddb_entry failed\n",
				      ha->host_no, __func__));
		goto exit_logout_device;
	}

	set_bit(DF_NO_RELOGIN, &ddb_entry->flags);

	memset(&mbox_cmd, 0, sizeof(mbox_cmd));
	memset(&mbox_sts, 0, sizeof(mbox_sts));
	mbox_cmd[0] = MBOX_CMD_CONN_CLOSE_SESS_LOGOUT;
	mbox_cmd[1] = fw_ddb_index;
	mbox_cmd[2] = connection_id;
	mbox_cmd[3] = LOGOUT_OPTION_CLOSE_SESSION;

	if (qla4xxx_mailbox_command(ha, 4, 2, &mbox_cmd[0], &mbox_sts[0])
	    != QLA_SUCCESS) {
		QL4PRINT(QLP2,
			 printk("scsi%d: %s: MBOX_CMD_CONN_CLOSE_SESS_LOGOUT "
				"failed sts %04X",
				ha->host_no, __func__, mbox_sts[0]));

		if (mbox_sts[0] == 0x4005) {
			QL4PRINT(QLP2, printk(", reason %04X\n",
					      mbox_sts[1]));
		}
		else {
			QL4PRINT(QLP2, printk("\n"));
		}

		goto exit_logout_device;
	}

	status = QLA_SUCCESS;

	exit_logout_device:
	LEAVE("qla4xxx_logout_device");
	return(status);
}

/**************************************************************************
 * qla4xxx_delete_device
 *	This routine is called by the logout IOCTL to delete the specified
 *      device.	 Send the LOGOUT and DELETE_DDB commands for the specified
 *      target, even if it's not in our internal database.
 *
 * Input:
 * 	ha - Pointer to host adapter structure.
 * 	fw_ddb_index - Index of the device to delete
 * 	connection_id - Connection ID of the device to delete
 *
 * Returns:
 *	QLA_SUCCESS - Successfully deleted device
 *	QLA_ERROR   - Failed to delete device
 *
 * Context:
 *	Kernel context.
 **************************************************************************/
uint8_t
qla4xxx_delete_device(scsi_qla_host_t *ha,
		      uint16_t fw_ddb_index,
		      uint16_t connection_id)
{
	uint8_t     status = QLA_ERROR;
	uint32_t    mbox_cmd[MBOX_REG_COUNT];
	uint32_t    mbox_sts[MBOX_REG_COUNT];
	uint32_t           fw_ddb_device_state = 0xFFFF;
	unsigned long      wait_count;
	ddb_entry_t *ddb_entry;
	uint8_t      mbox_status = QLA_SUCCESS;

	ENTER("qla4xxx_delete_device");

	QL4PRINT(QLP3, printk("scsi%d: %s: Delete index [%d]\n",
			      ha->host_no, __func__, fw_ddb_index));

	/* If the device is in our internal tables, set the NO_RELOGIN bit. */
	ddb_entry = qla4xxx_lookup_ddb_by_fw_index(ha, fw_ddb_index);
	if (ddb_entry != NULL) {
		QL4PRINT(QLP3, printk("scsi%d:%d:%d: %s:  setting NO_RELOGIN flag\n",
				      ha->host_no, ddb_entry->bus, ddb_entry->target, __func__));

		set_bit(DF_NO_RELOGIN, &ddb_entry->flags);
	}

	/* If the device state is already one that we can delete,
	 * bypass the logout command.
	 */
	qla4xxx_get_ddb_entry(ha, fw_ddb_index, NULL, NULL, NULL,
			      &fw_ddb_device_state, NULL, NULL, NULL);

	if (DELETEABLE_DDB_DS(fw_ddb_device_state))
		goto delete_ddb;

	/* First logout index */
	memset(&mbox_cmd, 0, sizeof(mbox_cmd));
	memset(&mbox_sts, 0, sizeof(mbox_sts));
	mbox_cmd[0] = MBOX_CMD_CONN_CLOSE_SESS_LOGOUT;
	mbox_cmd[1] = fw_ddb_index;
	mbox_cmd[2] = connection_id;
	mbox_cmd[3] = LOGOUT_OPTION_CLOSE_SESSION;
	if (qla4xxx_mailbox_command(ha, 4, 2, &mbox_cmd[0], &mbox_sts[0])
	    != QLA_SUCCESS) {
		QL4PRINT(QLP2,
			 printk("scsi%d: %s: MBOX_CMD_CONN_CLOSE_SESS_LOGOUT "
				"failed index [%d], sts %04X %04X\n",
				ha->host_no, __func__, fw_ddb_index,
				mbox_sts[0], mbox_sts[1]));
	}

	/* Wait enough time to complete logout */
	wait_count = jiffies + LOGOUT_TOV * HZ;
	while (qla4xxx_get_ddb_entry(ha, fw_ddb_index, NULL, NULL, NULL,
				     &fw_ddb_device_state, NULL, NULL, NULL) == QLA_SUCCESS) {
		if (wait_count <= jiffies)
			goto exit_delete_ddb;

		if (DELETEABLE_DDB_DS(fw_ddb_device_state))
			break;

		udelay(50);
	}

	delete_ddb:
	if (fw_ddb_device_state != DDB_DS_UNASSIGNED ) {
		/* Now delete index */
		memset(&mbox_cmd, 0, sizeof(mbox_cmd));
		memset(&mbox_sts, 0, sizeof(mbox_sts));
		mbox_cmd[0] = MBOX_CMD_CLEAR_DATABASE_ENTRY;
		mbox_cmd[1] = fw_ddb_index;

		mbox_status = qla4xxx_mailbox_command(ha, 2, 5, &mbox_cmd[0], &mbox_sts[0]);
	}
	if (mbox_status != QLA_SUCCESS) {
		QL4PRINT(QLP2,
			 printk("scsi%d: %s: MBOX_CMD_CLEAR_DATABASE_ENTRY "
				"failed sts %04X index [%d], state %04x\n",
				ha->host_no, __func__, mbox_sts[0], fw_ddb_index, mbox_sts[4]));
	}
	else {
		if (ddb_entry != NULL) {
			uint16_t lun;
			lun_entry_t *lun_entry;

			atomic_set(&ddb_entry->state, DEV_STATE_DEAD);

			for (lun = 0; lun < MAX_LUNS; lun++) {
				lun_entry = ddb_entry->lun_table[lun];
				if (lun_entry != NULL) {
					unsigned long cpu_flags;
					spin_lock_irqsave(&lun_entry->lun_lock, cpu_flags);

					QL4PRINT(QLP3, printk("scsi%d:%d:%d:%d: %s:  flushing srbs, pendq_cnt=%d, retryq_cnt=%d, activeq_cnt=%d\n",
							      ha->host_no, ddb_entry->bus, ddb_entry->target, lun, __func__,
							      ha->pending_srb_q_count, ha->retry_srb_q_count, ha->active_srb_count));

					qla4xxx_flush_all_srbs(ha, ddb_entry, lun_entry);
					if (lun_entry->lun_state == LS_LUN_SUSPENDED)
						del_from_suspended_lun_q(ha, lun_entry);

					spin_unlock_irqrestore(&lun_entry->lun_lock, cpu_flags);
				}
			}

			qla4xxx_free_ddb(ha, ddb_entry);
		}
		status = QLA_SUCCESS;
	}

	exit_delete_ddb:
	LEAVE("qla4xxx_delete_device");
	return(status);
}

void
qla4xxx_flush_all_srbs(scsi_qla_host_t *ha, ddb_entry_t *ddb_entry, lun_entry_t *lun_entry)
{
	srb_t       *srb;
	int         i;
	unsigned long flags;
	struct list_head *list, *temp; /* used for traversing lists */

	if (lun_entry == NULL || ddb_entry == NULL)
		return;

	spin_lock_irqsave(&ha->list_lock, flags);
	/* free pending commands */
	if (!list_empty(&ha->pending_srb_q)) {
		list_for_each_safe(list, temp, &ha->pending_srb_q)
		{
			srb = list_entry(list, srb_t, list_entry);

			QL4PRINT(QLP3,
				 printk("scsi%d:%d:%d:%d: %s: found srb %p in pending_q\n",
					ha->host_no, ddb_entry->bus, ddb_entry->target,
					lun_entry->lun, __func__, srb));

			if (srb) {
				if (srb->lun != lun_entry->lun)
					continue;

				QL4PRINT(QLP3,
					 printk("scsi%d:%d:%d:%d: %s: flushing srb %p from pending_q\n",
						ha->host_no, ddb_entry->bus, ddb_entry->target,
						lun_entry->lun, __func__, srb));
				__del_from_pending_srb_q(ha, srb);
				CMD_RESULT(srb->cmd) = (int) (DID_NO_CONNECT << 16);
				__add_to_done_srb_q(ha, srb);
			}
		}
	}

	/* free retry commands */
	if (!list_empty(&ha->retry_srb_q)) {
		list_for_each_safe(list, temp, &ha->retry_srb_q)
		{
			srb = list_entry(list, srb_t, list_entry);

			QL4PRINT(QLP3,
				 printk("scsi%d:%d:%d:%d: %s: found srb %p in retry_q\n",
					ha->host_no, ddb_entry->bus, ddb_entry->target,
					lun_entry->lun, __func__, srb));

			if (srb) {
				if (lun_entry != lun_entry)
					continue;

				QL4PRINT(QLP3,
					 printk("scsi%d:%d:%d:%d: %s: flushing srb %p from retry_q\n",
						ha->host_no, ddb_entry->bus, ddb_entry->target,
						lun_entry->lun, __func__, srb));
				__del_from_retry_srb_q(ha, srb);
				CMD_RESULT(srb->cmd) = (int) (DID_NO_CONNECT << 16);
				__add_to_done_srb_q(ha, srb);
			}
		}
	}
	spin_unlock_irqrestore(&ha->list_lock, flags);

	/* free active commands */
	spin_lock_irqsave(&ha->hardware_lock, flags);
	if (lun_entry->out_count != 0) {
		for (i = 1; i < MAX_SRBS; i++) {
			srb = ha->active_srb_array[i];

			if (srb) {
				QL4PRINT(QLP3,
					 printk("scsi%d:%d:%d:%d: %s: found srb %p in active_q\n",
						ha->host_no, ddb_entry->bus, ddb_entry->target,
						lun_entry->lun, __func__, srb));

				if (lun_entry == lun_entry) {
					QL4PRINT(QLP3,
						 printk("scsi%d:%d:%d:%d: %s: flushing srb %p from active_q\n",
							ha->host_no, ddb_entry->bus, ddb_entry->target,
							lun_entry->lun, __func__, srb));
					del_from_active_array(ha, i);
					CMD_RESULT(srb->cmd) = (int) (DID_NO_CONNECT << 16);
					add_to_done_srb_q(ha,srb);
				}
			}
		}
	}
	spin_unlock_irqrestore(&ha->hardware_lock, flags);

	/* Send all srbs back to OS */
	if (!list_empty(&ha->done_srb_q)) {
		while ((srb = del_from_done_srb_q_head(ha)) != NULL)
			qla4xxx_complete_request(ha, srb);
	}
}


/**************************************************************************
 * qla4xxx_relogin_device
 *	This routine does a session relogin with the specified device.
 *	The ddb entry must be assigned prior to making this call.
 *
 * Input:
 * 	ha - Pointer to host adapter structure.
 *	ddb_entry - Pointer to device database entry
 *
 * Returns:
 *    QLA_SUCCESS = Successfully relogged in device
 *    QLA_ERROR   = Failed to relogin device
 *
 * Context:
 *	Kernel context.
 **************************************************************************/
uint8_t
qla4xxx_relogin_device(scsi_qla_host_t *ha, ddb_entry_t *ddb_entry)
{
	uint16_t relogin_timer;

	ENTER("qla4xxx_relogin_device");

	relogin_timer = MAX(ddb_entry->default_relogin_timeout, RELOGIN_TOV);
	atomic_set(&ddb_entry->relogin_timer, relogin_timer);

	QL4PRINT(QLP3, printk(KERN_WARNING "scsi%d:%d:%d: Relogin index [%d]. TOV=%d\n",
			      ha->host_no, ddb_entry->bus, ddb_entry->target,
			      ddb_entry->fw_ddb_index, relogin_timer));


	qla4xxx_set_ddb_entry(ha, ddb_entry->fw_ddb_index, NULL);

	LEAVE("qla4xxx_relogin_device");
	return(QLA_SUCCESS);
}

/**************************************************************************
 * qla4xxx_send_internal_scsi_passthru
 *	This routine is used during (re)initialization to issue passthru
 * 	SCSI commands to the specified adapter.
 *
 * Input:
 * 	ha - Pointer to host adapter structure.
 *	scsi_pass - Pointer to SCSI Pass-through structure
 *	data_ptr - Pointer to data transfer buffer
 *	data_len - Length of data to transfer
 *
 * Output:
 *	scsi_pass - Updated with completion status
 *	data_ptr - Updated on Data in commands
 *
 * Returns:
 *	QLA_SUCCESS - Successfully completed SCSI pass-through command
 *	QLA_ERROR   - Failed to complete SCSI pass-through command
 *
 * Context:
 *	Kernel context.
 **************************************************************************/
uint8_t
qla4xxx_send_internal_scsi_passthru(scsi_qla_host_t *ha,
				    EXT_SCSI_PASSTHRU_ISCSI *scsi_pass,
				    void *data_ptr,
				    uint32_t data_len)
{
	uint8_t     status = QLA_ERROR;
	srb_t       *srb;
	ddb_entry_t *ddb_entry;
	Scsi_Device *scsi_device;
	Scsi_Cmnd   *cmd;
	int i;
#if 0 && __VMKERNEL_MODULE__
	unsigned long flags = 0;
#endif

	ENTER("qla4xxx_send_internal_scsi_passthru");
#if 0 && __VMKERNEL_MODULE__
	spin_lock_irqsave(&ha->dma_buf_spinlock, flags);
#else
	down(&ha->dma_buf_sem);
#endif

	/* ---- Allocate larger DMA buffer, if neccessary ---- */
	if (ha->dma_buf.buf_len < sizeof(*cmd) + sizeof(*scsi_device) &&
	    qla4xxx_resize_dma_buf(ha, &ha->dma_buf,
				   sizeof(*cmd) + sizeof(*scsi_device))
	    != QLA_SUCCESS) {
		QL4PRINT(QLP2,
			 printk("scsi%d: %s: ERROR cannot allocate requested "
				"DMA buffer size 0x%lx.\n",
				ha->host_no, __func__, (unsigned long)
				(sizeof(*cmd) + sizeof(*scsi_device))));
		goto exit_send_scsi_passthru;
	}

	memset(ha->dma_buf.virt_addr, 0, ha->dma_buf.buf_len);
	cmd = ha->dma_buf.virt_addr;
	scsi_device = ha->dma_buf.virt_addr + sizeof(*cmd);

	/* ---- Make sure device exists ---- */
	ddb_entry = qla4xxx_lookup_ddb_by_SCSIID(ha, scsi_pass->Addr.Bus,
						 scsi_pass->Addr.Target);
	if (ddb_entry == NULL) {
		QL4PRINT(QLP2,
			 printk("scsi%d: %s: invalid device (b%d,t%d,l%d) "
				"specified.\n", ha->host_no, __func__,
				scsi_pass->Addr.Bus,
				scsi_pass->Addr.Target,
				scsi_pass->Addr.Lun));
		goto exit_send_scsi_passthru;
	}

	/* ---- Make sure device is in an active state ---- */
	if (ddb_entry->fw_ddb_device_state != DDB_DS_SESSION_ACTIVE) {
		QL4PRINT(QLP2, printk("scsi%d: %s: device (b%d,t%d) not in "
				      "active state.\n",
				      ha->host_no, __func__,
				      scsi_pass->Addr.Bus,
				      scsi_pass->Addr.Target));
		goto exit_send_scsi_passthru;
	}

	/* ---- Retrieve srb from pool ---- */
	srb = del_from_free_srb_q_head(ha);
	if (srb == NULL) {
		QL4PRINT(QLP2, printk("scsi%d: %s: srb not available\n",
				      ha->host_no, __func__));
		goto exit_send_scsi_passthru;
	}

	/* ---- Fill in the SCSI command structure ---- */
	cmd->channel         = scsi_pass->Addr.Bus;
	cmd->target          = scsi_pass->Addr.Target;
	cmd->lun             = scsi_pass->Addr.Lun;
	cmd->device          = scsi_device;
	cmd->host            = ha->host;
	cmd->request_buffer  = (void *) data_ptr;
#ifdef __VMKERNEL_MODULE__
	cmd->request_bufferMA = (dma_addr_t)virt_to_phys(cmd->request_buffer);
#endif //__VMKERNEL_MODULE__
	cmd->request_bufflen = data_len;
	cmd->scsi_done       = qla4xxx_scsi_pass_done;
	CMD_TIMEOUT(cmd)     = INTERNAL_PASSTHRU__TOV * HZ;
	CMD_SP(cmd) = (char *) srb;
	srb->cmd = cmd;
	srb->fw_ddb_index = ddb_entry->fw_ddb_index;
	srb->lun = cmd->lun;
	srb->flags |= SRB_INTERNAL_CMD;


	if (scsi_pass->CdbLength == 6 ||
	    scsi_pass->CdbLength == 10 ||
	    scsi_pass->CdbLength == 12 ||
	    scsi_pass->CdbLength == 16) {
		cmd->cmd_len = scsi_pass->CdbLength;
	}
	else {
		QL4PRINT(QLP2,
			 printk("scsi%d: %s: Unsupported CDB length 0x%x \n",
				ha->host_no, __func__, cmd->cmd_len));
		goto exit_send_scsi_passthru;
	}

	switch (scsi_pass->Direction) {
	case EXT_DEF_SCSI_PASSTHRU_DATA_IN:
		cmd->sc_data_direction = SCSI_DATA_READ;
		break;
	case EXT_DEF_SCSI_PASSTHRU_DATA_OUT:
		cmd->sc_data_direction = SCSI_DATA_WRITE;
		break;
	default:
		cmd->sc_data_direction = SCSI_DATA_NONE;
		break;
	}

	memcpy(cmd->cmnd,      scsi_pass->Cdb, cmd->cmd_len);
	memcpy(cmd->data_cmnd, scsi_pass->Cdb, cmd->cmd_len);

	QL4PRINT(QLP5, printk("scsi%d:%d:%d:%d: %s  CDB = ",
			      ha->host_no, cmd->channel,
			      cmd->target, cmd->lun, __func__));
	for (i = 0; i < cmd->cmd_len; i++)
		QL4PRINT(QLP5, printk("%02X ", cmd->cmnd[i]));
	QL4PRINT(QLP5, printk("\n"));

	/* ---- prepare for receiving completion ---- */
	ha->ioctl_scsi_pass_in_progress = TRUE;
	ha->ioctl_tov = CMD_TIMEOUT(cmd);

	qla4xxx_ioctl_sem_init(ha);
	CMD_COMPL_STATUS(cmd)  = INVALID_ENTRY;
	CMD_PASSTHRU_TYPE(cmd) = (unsigned long)TRUE;

	/* ---- send command to adapter ---- */
	QL4PRINT(QLP5, printk("scsi%d:%d:%d:%d: %s: sending command.\n",
			      ha->host_no, cmd->channel,
			      cmd->target, cmd->lun, __func__));

	ha->ioctl_cmpl_timer.expires = jiffies + ha->ioctl_tov;
	add_timer(&ha->ioctl_cmpl_timer);

	if (qla4xxx_send_command_to_isp(ha, srb) != QLA_SUCCESS) {
		add_to_free_srb_q(ha, srb);
		QL4PRINT(QLP2,
			 printk("scsi%d: %s: error sending cmd to isp\n",
				ha->host_no, __func__));
		del_timer(&ha->ioctl_cmpl_timer);
		goto exit_send_scsi_passthru;
	}

	down(&ha->ioctl_cmpl_sem);

	/*******************************************************
	 *						       *
	 *             Passthru Completion                     *
	 *						       *
	 *******************************************************/
	del_timer(&ha->ioctl_cmpl_timer);

	/* ---- check for timeout --- */
	if (ha->ioctl_scsi_pass_in_progress == TRUE) {
		QL4PRINT(QLP2,
			 printk("scsi%d: %s: ERROR = command timeout.\n",
				ha->host_no, __func__));

		HOST_SPIN_LOCK_IRQ(ha);

		if ((srb != NULL) && (srb->active_array_index < MAX_SRBS)) {
			lun_entry_t *lun_entry;
			unsigned long wait_cnt;

			QL4PRINT(QLP2, printk("scsi%d: %s: timeout: srb state = 0x%02X\n",
					      ha->host_no, __func__, srb->state));

			if (srb->state != SRB_FREE_STATE)
				qla4xxx_delete_timer_from_cmd(srb);

			lun_entry = qla4xxx_lookup_lun_handle(ha,
							      ddb_entry,
							      srb->lun);
			if (lun_entry &&
			    (qla4xxx_reset_lun(ha, ddb_entry, lun_entry)
			     == QLA_ERROR)) {
				lun_entry->lun_state = LS_LUN_TIMEOUT;
			}

			/* Wait for command to get out of active state */
			QL4PRINT(QLP2, printk("scsi%d: %s: timeout: wait up to %d seconds "
					      "for command to return.\n",
					      ha->host_no, __func__, WAIT_CMD_TOV));
			wait_cnt  = jiffies + WAIT_CMD_TOV * HZ;
			QLA4XXX_WAIT(wait_cnt, (srb->flags != SRB_ACTIVE_STATE));
			QL4PRINT(QLP2, printk("scsi%d: %s: timeout: post wait, srb state = 0x%02X.\n",
					      ha->host_no, __func__, srb->state));

			/* If it's still in the active state, then clean it up here. */
			if (srb->flags != SRB_ACTIVE_STATE) {
				del_from_active_array(ha, (uint32_t)
						      srb->active_array_index);
				add_to_free_srb_q(ha, srb);
			}
		}

		ha->ioctl_scsi_pass_in_progress = FALSE;

		HOST_SPIN_UNLOCK_IRQ(ha);
		goto exit_send_scsi_passthru;
	}

	/* --- Return info from status entry --- */
	scsi_pass->Reserved[0] = (uint8_t) CMD_SCSI_STATUS(cmd);
	scsi_pass->Reserved[1] = (uint8_t) CMD_COMPL_STATUS(cmd);
	scsi_pass->Reserved[2] = (uint8_t) CMD_ACTUAL_SNSLEN(cmd);
	scsi_pass->Reserved[3] = (uint8_t) CMD_HOST_STATUS(cmd);
	scsi_pass->Reserved[6] = (uint8_t) CMD_ISCSI_RESPONSE(cmd);
	scsi_pass->Reserved[7] = (uint8_t) CMD_STATE_FLAGS(cmd);

	/* ---- Copy SCSI Passthru structure with updated sense buffer
	 *      to user space ---- */
	if (CMD_ACTUAL_SNSLEN(cmd)) {
		memcpy(scsi_pass->SenseData, CMD_SNSP(cmd),
		       MIN(CMD_ACTUAL_SNSLEN(cmd),
			   sizeof(scsi_pass->SenseData)));
	}

	/* ---- check for command completion --- */
	if (CMD_COMPL_STATUS(cmd) == INVALID_ENTRY) {
		QL4PRINT(QLP2,
			 printk("scsi%d: %s: ERROR = command not completed.\n",
				ha->host_no, __func__));
		goto exit_send_scsi_passthru;
	}
	else if (CMD_HOST_STATUS(cmd) == DID_OK) {
		status = QLA_SUCCESS;
	}
	else if (CMD_COMPL_STATUS(cmd) == SCS_DATA_UNDERRUN) {
		QL4PRINT(QLP2,
			 printk("scsi%d: %s: Data underrun.  Resid = 0x%x\n",
				ha->host_no, __func__, CMD_RESID_LEN(cmd)));

		scsi_pass->Reserved[4] = MSB(CMD_RESID_LEN(cmd));
		scsi_pass->Reserved[5] = LSB(CMD_RESID_LEN(cmd));
	}
	else if (CMD_COMPL_STATUS(cmd) == SCS_DATA_OVERRUN) {
		QL4PRINT(QLP2,
			 printk("scsi%d: %s: Data overrun.  Resid = 0x%x\n",
				ha->host_no, __func__, CMD_RESID_LEN(cmd)));

		scsi_pass->Reserved[4] = MSB(CMD_RESID_LEN(cmd));
		scsi_pass->Reserved[5] = LSB(CMD_RESID_LEN(cmd));
	}
	else {
		QL4PRINT(QLP2,
			 printk("scsi%d: %s: Command completed in ERROR. "
				"cs=%04x, ss=%04x\n", ha->host_no, __func__,
				CMD_COMPL_STATUS(cmd), CMD_SCSI_STATUS(cmd)));

		goto exit_send_scsi_passthru;
	}

	exit_send_scsi_passthru:
#if 0 && __VMKERNEL_MODULE__
	spin_unlock_irqrestore(&ha->dma_buf_spinlock, flags);
#else
	up(&ha->dma_buf_sem);
#endif
	LEAVE("qla4xxx_send_internal_scsi_passthru");
	return(status);
}

/**************************************************************************
 * qla4xxx_send_inquiry_cmd
 *	This routine issues a passthru SCSI inquiry command
 *
 * Input:
 * 	ha - Pointer to host adapter structure.
 *	target - SCSI target ID
 *	lun - SCSI lun
 *	inquiry - Pointer to inquiry data structure
 *
 * Output:
 *	inquiry - SCSI inquiry data returned
 *
 * Returns:
 *	QLA_SUCCESS - Successfully retrieved inquiry data
 *	QLA_ERROR   - Failed to retrieve inquiry data
 *
 * Context:
 *	Kernel context.
 **************************************************************************/
uint8_t
qla4xxx_send_inquiry_cmd(scsi_qla_host_t *ha,
			 uint16_t target,
			 uint16_t lun,
			 inquiry_t *inquiry)
{
	uint8_t     status = QLA_ERROR;
	EXT_SCSI_PASSTHRU_ISCSI *scsi_pass;

	ENTER("qla4xxx_send_inquiry_cmd");
	scsi_pass = (EXT_SCSI_PASSTHRU_ISCSI *) kmalloc(sizeof(*scsi_pass),
							GFP_KERNEL);
	if (scsi_pass) {
		/* set up SCSI Passthru structure */
		memset(scsi_pass, 0, sizeof(*scsi_pass));
		scsi_pass->Addr.Bus = 0;
		scsi_pass->Addr.Target = target;
		scsi_pass->Addr.Lun = lun;
		scsi_pass->Cdb[0] = INQUIRY;
		scsi_pass->Cdb[1] = lun << 5;
		scsi_pass->Cdb[4] = sizeof(*inquiry);
		scsi_pass->CdbLength = 6;
		scsi_pass->Direction = EXT_DEF_SCSI_PASSTHRU_DATA_IN;
		memset(inquiry, 0, sizeof(*inquiry));

		QL4PRINT(QLP7, printk("scsi%d:%d:%d:%d: %s:\n",
				      ha->host_no, 0, target, lun, __func__));

		status = qla4xxx_send_internal_scsi_passthru(ha, scsi_pass,
							     inquiry,
							     sizeof(*inquiry));

		kfree(scsi_pass);
	}

	LEAVE("qla4xxx_send_inquiry_cmd");
	return(status);
}

/**************************************************************************
 * qla4xxx_send_report_luns_cmd
 *	This routine issues a passthru SCSI Report Luns command
 *
 * Input:
 * 	ha - Pointer to host adapter structure.
 *	target - SCSI target ID
 *	report_luns - Pointer to report_luns data structure
 *
 * Output:
 *	report_luns - SCSI report_luns data returned
 *
 * Returns:
 *	QLA_SUCCESS - Successfully retrieved report_luns data
 *	QLA_ERROR   - Failed to retrieve report_luns data
 *
 * Context:
 *	Kernel context.
 **************************************************************************/
uint8_t
qla4xxx_send_report_luns_cmd(scsi_qla_host_t *ha,
			     uint16_t target,
			     report_luns_t *report_luns)
{
	uint8_t status = QLA_ERROR;
	EXT_SCSI_PASSTHRU_ISCSI *scsi_pass;

	ENTER("qla4xxx_send_report_luns_cmd");
	scsi_pass = (EXT_SCSI_PASSTHRU_ISCSI *) kmalloc(sizeof(*scsi_pass),
							GFP_KERNEL);
	if (scsi_pass) {
		/* set up SCSI Passthru structure */
		memset(scsi_pass, 0, sizeof(*scsi_pass));
		scsi_pass->Addr.Bus = 0;
		scsi_pass->Addr.Target = target;
		scsi_pass->Cdb[0] = REPORT_LUNS;
		scsi_pass->Cdb[8] = MSB(sizeof(*report_luns));
		scsi_pass->Cdb[9] = LSB(sizeof(*report_luns));
		scsi_pass->CdbLength = 12;
		scsi_pass->Direction = EXT_DEF_SCSI_PASSTHRU_DATA_IN;
		memset(report_luns, 0, sizeof(*report_luns));

		QL4PRINT(QLP7, printk("scsi%d:%d:%d: %s:\n",
				      ha->host_no, 0, target, __func__));
		status = qla4xxx_send_internal_scsi_passthru(
							    ha, scsi_pass, report_luns, sizeof(*report_luns));

		if ((status == QLA_SUCCESS) &&
		    (SENSE_KEY(scsi_pass->SenseData) != 0)) {
			status = QLA_ERROR;
		}

		kfree(scsi_pass);
	}

	LEAVE("qla4xxx_send_report_luns_cmd");
	return(status);
}


/**************************************************************************
 * qla4xxx_get_ddb_entry
 *	This routine retrieves the firmware's device database entry.
 *
 * Input:
 * 	ha - Pointer to host adapter structure.
 *      fw_ddb_index - Firmware's device database index
 *      fw_ddb_entry - Pointer to firmware's device database entry structure
 *      num_valid_ddb_entries - Pointer to number of valid ddb entries
 *      next_ddb_index - Pointer to next valid device database index
 *      fw_ddb_device_state - Pointer to device state
 *
 * Output:
 *      fw_ddb_entry - Fills in structure if pointer is supplied
 *      num_valid_ddb_entries - Fills in if pointer is supplied
 *      next_ddb_index - Fills in if pointer is supplied
 *      fw_ddb_device_state - Fills in if pointer is supplied
 *
 * Returns:
 *	QLA_SUCCESS - Successfully retrieved ddb info from firmware
 *	QLA_ERROR   - Failed to retrieve ddb info from firmware
 *
 * Context:
 *	Kernel context.
 **************************************************************************/
uint8_t
qla4xxx_get_ddb_entry(scsi_qla_host_t *ha,
		      uint16_t        fw_ddb_index,
		      DEV_DB_ENTRY    *fw_ddb_entry,
		      uint32_t        *num_valid_ddb_entries,
		      uint32_t        *next_ddb_index,
		      uint32_t        *fw_ddb_device_state,
		      uint32_t        *time2wait,
		      uint16_t        *tcp_source_port_num,
		      uint16_t        *connection_id)
{
	uint8_t         status = QLA_ERROR;
	uint32_t mbox_cmd[MBOX_REG_COUNT];
	uint32_t mbox_sts[MBOX_REG_COUNT];
	DEV_DB_ENTRY     *tmp_data = 0;
	dma_addr_t      tmp_data_phys_addr = 0;

	ENTER("qla4xxx_get_ddb_entry");

	/* Make sure the device index is valid */
	if (fw_ddb_index >= MAX_DDB_ENTRIES) {
		QL4PRINT(QLP2,
			 printk("scsi%d: %s: index [%d] out of range.\n",
				ha->host_no, __func__, fw_ddb_index));
		goto exit_get_ddb;
	}

	if (fw_ddb_entry) {
		tmp_data = pci_alloc_consistent(ha->pdev,
						sizeof(*tmp_data),
						&tmp_data_phys_addr);
		if (tmp_data == NULL) {
			QL4PRINT(QLP2,
				 printk("scsi%d: %s: index [%d] out of range.\n",
					ha->host_no, __func__, fw_ddb_index));
			goto exit_get_ddb;
		}

		memset(tmp_data, 0, sizeof(*tmp_data));
	}

	memset(&mbox_cmd, 0, sizeof(mbox_cmd));
	memset(&mbox_sts, 0, sizeof(mbox_sts));
	mbox_cmd[0] = MBOX_CMD_GET_DATABASE_ENTRY;
	mbox_cmd[1] = (uint32_t) fw_ddb_index;
	mbox_cmd[2] = LSDW(tmp_data_phys_addr);
	mbox_cmd[3] = MSDW(tmp_data_phys_addr);

	if (qla4xxx_mailbox_command(ha, 4, 7, &mbox_cmd[0], &mbox_sts[0])
	    == QLA_ERROR) {
		QL4PRINT(QLP2,
			 printk("scsi%d: %s: MBOX_CMD_GET_DATABASE_ENTRY failed "
				"with status 0x%04X\n",
				ha->host_no, __func__, mbox_sts[0]));
		goto exit_get_ddb;
	}

	if (fw_ddb_index != mbox_sts[1]) {
		QL4PRINT(QLP2,
			 printk("scsi%d: %s: index mismatch [%d] != [%d].\n",
				ha->host_no, __func__, fw_ddb_index,
				mbox_sts[1]));
		goto exit_get_ddb;
	}

	if (fw_ddb_entry) {
		memcpy(fw_ddb_entry, tmp_data, sizeof(*tmp_data));

		QL4PRINT(QLP10, printk("scsi%d: fw_ddb_entry = %p\n", ha->host_no, fw_ddb_entry));
		qla4xxx_dump_bytes(QLP10, fw_ddb_entry, sizeof(*fw_ddb_entry));

		QL4PRINT(QLP7,
			 printk("scsi%d: %s: index [%d] MB0 %04x Tot %d Next %d "
				"State %04x %d.%d.%d.%d:%04d \"%s\"\n",
				ha->host_no, __func__, fw_ddb_index,
				mbox_sts[0], mbox_sts[2], mbox_sts[3], mbox_sts[4],
				fw_ddb_entry->ipAddr[0],
				fw_ddb_entry->ipAddr[1],
				fw_ddb_entry->ipAddr[2],
				fw_ddb_entry->ipAddr[3],
				__le16_to_cpu(fw_ddb_entry->portNumber),
				fw_ddb_entry->iscsiName));
	}

	if (num_valid_ddb_entries)
		*num_valid_ddb_entries = mbox_sts[2];

	if (next_ddb_index)
		*next_ddb_index = mbox_sts[3];

	if (fw_ddb_device_state)
		*fw_ddb_device_state = mbox_sts[4];

	if (time2wait)
		*time2wait = mbox_sts[5];

	if (tcp_source_port_num)
		*tcp_source_port_num = (uint16_t) mbox_sts[6] >> 16;

	if (connection_id)
		*connection_id = (uint16_t) mbox_sts[6] & 0x00FF;

	status = QLA_SUCCESS;

	exit_get_ddb:
	if (tmp_data)
		pci_free_consistent(ha->pdev, sizeof(*tmp_data), tmp_data,
				    tmp_data_phys_addr);

	LEAVE("qla4xxx_get_ddb_entry");
	return(status);
}


/**************************************************************************
 * qla4xxx_set_ddb_entry
 *	This routine initializes or updates the adapter's device database
 *	entry for the specified device.
 *
 * Input:
 * 	ha - Pointer to host adapter structure.
 *      fw_ddb_index - Firmware's device database index
 *      fw_ddb_entry - Pointer to firmware's device database entry
 *		       structure, or NULL.
 *
 * Output:
 *	None
 *
 * Remarks:
 *	This routine also triggers a login for the specified device.
 *	Therefore, it may also be used as a secondary login routine when
 *	a NULL pointer is specified for the fw_ddb_entry.
 *
 * Returns:
 *	QLA_SUCCESS - Successfully set ddb_entry in firmware
 *	QLA_ERROR   - Failed to set ddb_entry in firmware
 *
 * Context:
 *	Kernel context.
 **************************************************************************/
uint8_t
qla4xxx_set_ddb_entry(scsi_qla_host_t *ha,
		      uint16_t        fw_ddb_index,
		      DEV_DB_ENTRY     *fw_ddb_entry)
{
	uint8_t         status = QLA_ERROR;
	uint32_t mbox_cmd[MBOX_REG_COUNT];
	uint32_t mbox_sts[MBOX_REG_COUNT];
	DEV_DB_ENTRY    *tmp_data = 0;
	dma_addr_t      tmp_data_phys_addr = 0;

	ENTER("qla4xxx_set_ddb_entry");

	if (fw_ddb_entry) {
		tmp_data = pci_alloc_consistent(ha->pdev,
						sizeof(*tmp_data),
						&tmp_data_phys_addr);
		if (tmp_data == NULL) {
			QL4PRINT(QLP2,
				 printk("scsi%d: %s: index [%d] out of range.\n",
					ha->host_no, __func__, fw_ddb_index));
			goto exit_set_ddb;
		}

		memcpy(tmp_data, fw_ddb_entry, sizeof(*tmp_data));
	}

	QL4PRINT(QLP7, printk("scsi%d: %s: index [%d]\n",
			      ha->host_no, __func__, fw_ddb_index));

	/* Do not wait for completion. The firmware will send us an
	 * ASTS_DATABASE_CHANGED (0x8014) to notify us of the login status.
	 */
	memset(&mbox_cmd, 0, sizeof(mbox_cmd));
	memset(&mbox_sts, 0, sizeof(mbox_sts));
	mbox_cmd[0] = MBOX_CMD_SET_DATABASE_ENTRY;
	mbox_cmd[1] = (uint32_t) fw_ddb_index;
	mbox_cmd[2] = LSDW(tmp_data_phys_addr);
	mbox_cmd[3] = MSDW(tmp_data_phys_addr);

#if 1 || __VMKERNEL_MODULE__
        /*
         * Asking for 0 words back from a mailbox_command() call (third
         * argument) is secret code for "don't wait for the mailbox
         * command to complete."  This could be appropriate when
         * requesting an adapter reset.  It is not appropriate when
         * setting a DDB entry because the next mailbox command
         * (often a GET_DATABASE_ENTRY in the next iteration of a
         * loop) will stomp this command before it has a chance to
         * complete.          *
         * In this case, we tell it we just want the status word back,
         * which we still ignore because any interesting info comes
         * back as a database change AEN.
         */
        if (qla4xxx_mailbox_command(ha, 4, 1, &mbox_cmd[0], &mbox_sts[0]) != QLA_SUCCESS) {
#else
	if (qla4xxx_mailbox_command(ha, 4, 0, &mbox_cmd[0], &mbox_sts[0]) != QLA_SUCCESS) {
#endif
		status = QLA_ERROR;
	}
	else {
		status = QLA_SUCCESS;
	}

	exit_set_ddb:
	if (tmp_data)
		pci_free_consistent(ha->pdev,
				    sizeof(*tmp_data),
				    tmp_data,
				    tmp_data_phys_addr);

	LEAVE("qla4xxx_set_ddb_entry");
	return(status);
}

/**************************************************************************
 * qla4xxx_update_ddb_entry
 *	This routine updates the driver's internal device database entry
 *	with information retrieved from the firmware's device database
 *	entry for the specified device.
 *
 * Input:
 * 	ha - Pointer to host adapter structure.
 *	ddb_entry - Pointer to device database entry
 *
 * Output:
 *	ddb_entry - Structure filled in.
 *
 * Remarks:
 *	The ddb_entry->fw_ddb_index field must be initialized prior to
 *	calling this routine
 *
 * Returns:
 *	QLA_SUCCESS - Successfully update ddb_entry
 *	QLA_ERROR   - Failed to update ddb_entry
 *
 * Context:
 *	Kernel context.
 **************************************************************************/
uint8_t
qla4xxx_update_ddb_entry(scsi_qla_host_t *ha, ddb_entry_t *ddb_entry)
{
	DEV_DB_ENTRY *fw_ddb_entry;
	uint32_t        fw_ddb_index;
	uint8_t status = QLA_ERROR;

	ENTER("qla4xxx_update_ddb_entry");
#if 0 && __VMKERNEL_MODULE__
	spin_lock(&ha->dma_buf_spinlock);
#else
	down(&ha->dma_buf_sem);
#endif

	if (ddb_entry == NULL) {
		QL4PRINT(QLP2, printk("scsi%d: %s: ddb_entry is NULL\n",
				      ha->host_no, __func__));
		goto exit_update_ddb;
	}


	/* Make sure the dma buffer is valid */
	if (ha->dma_buf.buf_len < sizeof(*fw_ddb_entry))
		qla4xxx_resize_dma_buf(ha, &ha->dma_buf, sizeof(*fw_ddb_entry));

	if (!ha->dma_buf.virt_addr) {
		QL4PRINT(QLP2, printk("scsi%d: %s: dma buffer inaccessible.\n",
				      ha->host_no, __func__));
		goto exit_update_ddb;
	}

	fw_ddb_entry = (DEV_DB_ENTRY *) ha->dma_buf.virt_addr;
	fw_ddb_index = ddb_entry->fw_ddb_index;
	if (qla4xxx_get_ddb_entry(ha, fw_ddb_index, fw_ddb_entry,
				  NULL, NULL,
				  &ddb_entry->fw_ddb_device_state,
				  &ddb_entry->default_time2wait,
				  &ddb_entry->tcp_source_port_num,
				  &ddb_entry->connection_id)
	    == QLA_ERROR) {
		QL4PRINT(QLP2,
			 printk("scsi%d: %s: failed get_ddb_entry for "
				"fw_ddb_index %d\n", ha->host_no, __func__,
				fw_ddb_index));

		goto exit_update_ddb;
	}

	status = QLA_SUCCESS;
	switch (ddb_entry->fw_ddb_device_state) {
	case DDB_DS_SESSION_ACTIVE:
		ddb_entry->target_session_id =
		__le16_to_cpu(fw_ddb_entry->TSID);
		ddb_entry->task_mgmt_timeout =
		__le16_to_cpu(fw_ddb_entry->taskMngmntTimeout);
		ddb_entry->CmdSn             = 0;
		ddb_entry->exe_throttle      =
		__le16_to_cpu(fw_ddb_entry->exeThrottle);
		ddb_entry->default_relogin_timeout   =
		__le16_to_cpu(fw_ddb_entry->taskMngmntTimeout);
		#ifdef QLA4000
		ddb_entry->connection_id     = 0;
		ddb_entry->default_time2wait =
		__le16_to_cpu(fw_ddb_entry->minTime2Wait);
		#endif

		#ifdef QLA4010

		//ddb_entry->port_number =
		//	__le16_to_cpu(fw_ddb_entry->portNumber);
		//memcpy(ddb_entry->iscsi_name, fw_ddb_entry->iscsiName,
		//       MAX(ddb_entry->iscsi_name, fw_ddb_entry->iscsiName));
		memcpy(&ddb_entry->ip_addr[0], &fw_ddb_entry->ipAddr[0],
		       MIN(sizeof(ddb_entry->ip_addr),
			   sizeof(fw_ddb_entry->ipAddr)));
		memcpy(&ddb_entry->iscsi_alias[0], &fw_ddb_entry->iSCSIAlias[0],
		       MIN(sizeof(ddb_entry->iscsi_alias),
			   sizeof(fw_ddb_entry->iSCSIAlias)));

		if (qla4xxx_is_discovered_target(ha, fw_ddb_entry->ipAddr,
						 fw_ddb_entry->iSCSIAlias,
						 fw_ddb_entry->iscsiName)
		    == QLA_SUCCESS) {
			set_bit(DF_ISNS_DISCOVERED, &ddb_entry->flags);
		}
		#endif

		//QL4PRINT(QLP7, printk("scsi%d: %s: index [%d] \"%s\"\n",
		//		      ha->host_no, __func__,
		//		      fw_ddb_index,
		//		      ddb_state_msg[ddb_entry->fw_ddb_device_state]));
		break;

	case DDB_DS_NO_CONNECTION_ACTIVE:
	case DDB_DS_NO_SESSION_ACTIVE:
	case DDB_DS_SESSION_FAILED:
		ddb_entry->target_session_id = 0;
		ddb_entry->task_mgmt_timeout = 0;
		ddb_entry->connection_id     = 0;
		ddb_entry->CmdSn             = 0;
		ddb_entry->exe_throttle      = 0;
		ddb_entry->default_time2wait = 0;

		//QL4PRINT(QLP7, printk("scsi%d: %s: index [%d] \"%s\"\n",
		//		      ha->host_no, __func__,
		//		      fw_ddb_index,
		//		      ddb_state_msg[ddb_entry->fw_ddb_device_state]));
		break;

	case DDB_DS_UNASSIGNED:
	case DDB_DS_DISCOVERY:
	case DDB_DS_LOGGING_OUT:
		//QL4PRINT(QLP7, printk("scsi%d: %s: index [%d] \"%s\"\n",
		//		      ha->host_no, __func__,
		//		      fw_ddb_index,
		//		      ddb_state_msg[ddb_entry->fw_ddb_device_state]));
		break;

		//default:
		//QL4PRINT(QLP7, printk("scsi%d: %s: index [%d] State %x. "
		//		      "Illegal state\n",
		//		      ha->host_no, __func__,
		//		      fw_ddb_index,
		//		      ddb_entry->fw_ddb_device_state));

	}

	exit_update_ddb:
#if 0 && __VMKERNEL_MODULE__
	spin_unlock(&ha->dma_buf_spinlock);
#else
	up(&ha->dma_buf_sem);
#endif
	LEAVE("qla4xxx_update_ddb_entry");
	return(status);
}

/**************************************************************************
 * qla4xxx_alloc_lun
 *	This routine allocates memory for a lun and links it to the
 *      specified device.
 *
 * Input:
 *	ddb_entry - Pointer to device database entry
 *	lun - SCSI lun
 *
 * Returns:
 *	None.
 *
 * Context:
 *	Kernel context.
 **************************************************************************/
static inline void
qla4xxx_alloc_lun(scsi_qla_host_t *ha, ddb_entry_t *ddb_entry, uint16_t lun)
{
	lun_entry_t *lun_entry;

	ENTER("qla4xxx_alloc_lun");
	if (ddb_entry->lun_table[lun] == 0) {
		QL4PRINT(QLP7, printk("scsi%d:%d:%d: %s: added lun %d\n",
				      ha->host_no, ddb_entry->bus, ddb_entry->target,
				      __func__, lun));

		lun_entry = kmalloc(sizeof(*lun_entry), GFP_KERNEL);
		if (lun_entry == 0) {
			QL4PRINT(QLP2,
				 printk("scsi%d:%d:%d: %s: Error allocating memory for "
					"lun %d\n", ha->host_no,
					ddb_entry->bus, ddb_entry->target,
					__func__, lun));

			goto exit_create_lun;
		}

		memset(lun_entry, 0, sizeof(*lun_entry));
		ddb_entry->lun_table[lun] = lun_entry;
		lun_entry->lun = lun;
		lun_entry->lun_state = LS_LUN_READY;
		ddb_entry->num_valid_luns++;
		spin_lock_init(&lun_entry->lun_lock);
	}

	exit_create_lun:
	LEAVE("qla4xxx_alloc_lun");
}

/**************************************************************************
 * qla4xxx_discover_target_luns
 *	This routine performs a discovery on the given target, as well as
 *	all possible luns, and links the ddb with the target mapping table.
 *
 * Input:
 * 	ha - Pointer to host adapter structure.
 *	ddb_entry - Pointer to device database entry
 *	target - SCSI target ID
 *
 * Returns:
 *	QLA_SUCCESS - Found SCSI target(s) and lun(s)
 *	QLA_ERROR   - Failed to locate SCSI target(s) and lun(s)
 *
 * Context:
 *	Kernel context.
 **************************************************************************/
static uint8_t
qla4xxx_discover_target_luns(scsi_qla_host_t *ha,
			     ddb_entry_t *ddb_entry,
			     uint32_t target)
{
	uint8_t status = QLA_ERROR;
	report_luns_t *report_luns;
	inquiry_t *inquiry;
	uint16_t lun = 0;
	uint8_t discovered_via_report_luns = FALSE;
	uint8_t discovered_via_inquiry = FALSE;

	ENTER("qla4xxx_discover_target_luns");

	/* Allocate structures to do discovery */
	report_luns = (report_luns_t*) kmalloc(sizeof(*report_luns),
					       GFP_KERNEL);
	inquiry = (inquiry_t *) kmalloc(sizeof(*inquiry), GFP_KERNEL);
	if ((!report_luns) || (!inquiry)) {
		QL4PRINT(QLP2, printk("scsi%d: %s: unable to allocate memory\n",
				      ha->host_no, __func__));
		goto exit_discover_target_and_luns;
	}

	/* Initialize ddb_entry for target/lun discovery */
	ddb_entry->target       = target;
	ha->target_map[target]  = ddb_entry;

	/* First create pseudo Lun 0 to issue SCSI commands */
	qla4xxx_alloc_lun(ha, ddb_entry, lun);

	/* First make sure that there is a valid device here. */
#if 1 || __VMKERNEL_MODULE__
        /* Well, a valid device here is simply one we can talk to. We issued this
         * INQUIRY to LUN 0 and if it is not a valid LUN this does not mean that no
         * other LUNs are valid on this target. Also, the test for qualifiers is wrong,
         * but we only need to look at them later (see further down in this function).
         */
	if (qla4xxx_send_inquiry_cmd(ha, target, lun, inquiry) != QLA_SUCCESS) {
		goto exit_no_target_lun;
	}
#else
	if ((qla4xxx_send_inquiry_cmd(ha, target, lun, inquiry) == QLA_SUCCESS) &&
	    (inquiry->device_type == 0x7f)) {
		goto exit_no_target_lun;
	}
#endif

	/* Discover via Report Luns */
	if (qla4xxx_send_report_luns_cmd(ha, target, report_luns)
	    == QLA_SUCCESS) {
		int index = 0;
		uint32_t num_luns;
#if 1 || __VMKERNEL_MODULE__
                uint32_t luns_seen = 1; /* We already (and always) have LUN 0 */
#endif

		num_luns = __be32_to_cpu(report_luns->lun_list_length) / 8;

		QL4PRINT(QLP7,
			 printk("scsi%d: %s: index[%d] mapped to target "
				"%d and has %d lun(s)\n",
				ha->host_no, __func__, ddb_entry->fw_ddb_index,
				target, num_luns));

		if (num_luns) {
			/* If lun 0 doesn't exist, keep pseudo lun 0,
			 * in addition to all luns from report_luns
			 */
#if 1 || __VMKERNEL_MODULE__
                        /* We already have a LUN 0, so skip it if we see it.
                         * Stop scanning when we reach the number of reported LUNs
                         * or if we have reached MAX_LUNs.
                         */
			for (index = 0; index < num_luns; index++) {
                                if (report_luns->lun[index].single_level_lun == 0) {
                                        continue; /* Already created this */
                                }
				qla4xxx_alloc_lun(ha, ddb_entry,
						  (uint16_t)
						  report_luns->lun[index].
						  single_level_lun);
                                if (++luns_seen == MAX_LUNS) {
                                        break; /* Drop any remaining LUNs */
                                }
			}
#else
			if (report_luns->lun[0].single_level_lun != 0) {
				num_luns++;
				index = 0;
			}
			else
				index = 1;

			if (num_luns > MAX_LUNS) num_luns = MAX_LUNS;

			for (; index < num_luns; index++) {
				qla4xxx_alloc_lun(ha, ddb_entry,
						  (uint16_t)
						  report_luns->lun[index].
						  single_level_lun);
			}
#endif

			discovered_via_report_luns = TRUE;
			status = QLA_SUCCESS;
		}
	}

	/* Discover via SCSI Inquiry */
#if 1 || __VMKERNEL_MODULE__
        /* See the comment for first call to qla4xxx_send_inquiry - applies here too.
         * Also, we know that the inquiry to LUN 0 has succeeded, so no point in
         * issuing one more here.
         */
        if (discovered_via_report_luns == FALSE) {
#else
	if ((discovered_via_report_luns == FALSE) &&
	    (qla4xxx_send_inquiry_cmd(ha, target, lun, inquiry) == QLA_SUCCESS)
	    &&
	    (inquiry->device_type != 0x7f)) {
#endif

		/* A target with lun 0 exists here.
		 * Add SCSI target/lun information */
		QL4PRINT(QLP7, printk("scsi%d: %s: index [%d] mapped to "
				      "target %d\n",
				      ha->host_no, __func__,
				      ddb_entry->fw_ddb_index, target));

		QL4PRINT(QLP7, printk("scsi%d: %s: found lun 0\n",
				      ha->host_no, __func__));

		discovered_via_inquiry = TRUE;
		status = QLA_SUCCESS;

		/* search for consecutive luns via inquiry command */
		for (lun = 1; lun < MAX_LUNS; lun++) {
			qla4xxx_alloc_lun(ha, ddb_entry, lun);
#if 1 || __VMKERNEL_MODULE__
                        /* We need to make a more precise selection here, since the
                         * inquiry->device_type consists of two values, namely the
                         * peripheral device type (lowest 5 bits) and the peripheral
                         * qualifier (top 3 bits).
                         * A peripheral qualifier of 0x1, 0x2 or 0x3 should disqualify
                         * the LUN for sure and so should a peripheral device type
                         * of 0x1f on it's own - see SPC2r20 page 83.
                         * Several devices mix and match these, which makes the
                         * check for 0x7f insufficient.
                         */
#define PQUAL_MASK 0xe0
#define PDEVT_MASK 0x1f
			if (qla4xxx_send_inquiry_cmd(ha, target, lun, inquiry)
                            == QLA_SUCCESS) {
                           int pQual = (inquiry->device_type & PQUAL_MASK) >> 5;
                           int pDevType = inquiry->device_type & PDEVT_MASK;
                           if ((pQual >= 0x01 && pQual <= 0x03) || pDevType == 0x1f) {
				/* No Logical Unit here, proceed to next LUN
				 */
				qla4xxx_free_lun(ddb_entry, lun);
                           } else {
                                QL4PRINT(QLP7, printk("scsi%d: %s: found lun %d\n",
                                                      ha->host_no, __func__, lun));
                           }
			} else {
				qla4xxx_free_lun(ddb_entry, lun);
                                QL4PRINT(QLP7, printk("scsi%d: %s: failed inquiry to lun %d\n",
                                                      ha->host_no, __func__, lun));
                        }
#else
			if ((qla4xxx_send_inquiry_cmd(ha, target, lun, inquiry)
			     == QLA_SUCCESS) &&
			    (inquiry->device_type == 0x7f)) {
				/* If a lun was NOT found, skip searching
				 * remaining luns.  Proceed to next target
				 */
				qla4xxx_free_lun(ddb_entry, lun);
				break;
			}

			QL4PRINT(QLP7, printk("scsi%d: %s: found lun %d\n",
					      ha->host_no, __func__, lun));
#endif
		}
	}

	if ((discovered_via_report_luns == FALSE) &&
	    (discovered_via_inquiry == FALSE)) {
		/* This target was not found, unlink it from target map */
		exit_no_target_lun:
		#if 0  //keep target and pseudo-lun 0
		qla4xxx_free_lun(ddb_entry, lun);
		ddb_entry->target       = INVALID_ENTRY;
		ha->target_map[target]  = (ddb_entry_t *) INVALID_ENTRY;
		#else
		status = QLA_SUCCESS;
		#endif
		atomic_set(&ddb_entry->state, DEV_STATE_DEAD);
	}

	exit_discover_target_and_luns:
	if (report_luns) kfree(report_luns);
	if (inquiry) kfree(inquiry);

	LEAVE("qla4xxx_discover_target_luns");
	return(status);
}

/**************************************************************************
 * qla4xxx_map_targets_to_ddbs
 *	This routine iterates through the existing list of ddb_entries
 *	and discovers the corresponding SCSI target and luns.
 *
 * Input:
 * 	ha - Pointer to host adapter structure.
 *
 * Returns:
 *	QLA_SUCCESS - Successfully discovered SCSI targets(s) and lun(s)
 *	QLA_ERROR   - Failed to discover SCSI targets(s) and lun(s)
 *
 * Context:
 *	Kernel context.
 **************************************************************************/
#if 1 || __VMKERNEL_MODULE__
static uint8_t
qla4xxx_map_targets_to_ddbs(scsi_qla_host_t *ha, int assign_target_numbers)
#else
static uint8_t
qla4xxx_map_targets_to_ddbs(scsi_qla_host_t *ha)
#endif
{
	uint8_t status = QLA_ERROR;
	struct list_head *ptr, *next; /* used for traversing ddb_list */
	uint16_t target = 0;
#if 1 || __VMKERNEL_MODULE__
	unsigned long flags;
	ddb_entry_t *ddb_entry;
	int i;
	uint16_t tnum;
#endif

	ENTER("qla4xxx_map_targets_to_ddbs");

#if 1 || __VMKERNEL_MODULE__
	/*
         * Mark each ddb entry as no discovery attempted
         */
	spin_lock_irqsave(&ha->ddb_list_spinlock, flags);
	list_for_each_safe(ptr, next, &ha->ddb_list) {
		ddb_entry = list_entry(ptr, ddb_entry_t, list_entry);
                ddb_entry->discover_done = 0;
	}
        spin_unlock_irqrestore(&ha->ddb_list_spinlock, flags);

	down(&ha->targ_array_sem);

	/*
         * Process at most MAX_TARGETS entries to avoid an infinite
         * loop. In reality we will never pass this, but even if we
         * do the worst case is that we will process any remaining
         * targets next time the DPC runs.
         */
        for (i=0; i < MAX_TARGETS; i++) {
                ddb_entry = NULL;
		spin_lock_irqsave(&ha->ddb_list_spinlock, flags);
                list_for_each_safe(ptr, next, &ha->ddb_list) {
			ddb_entry = list_entry(ptr, ddb_entry_t, list_entry);
                        if (!ddb_entry->discover_done) {
                                    /* Deal with this entry outside lock */
                                    ddb_entry->discover_done = 1;
                                    break;
                         }
                         ddb_entry = NULL; /* Use it as a flag */
		}
                spin_unlock_irqrestore(&ha->ddb_list_spinlock, flags);
                if (ddb_entry) {
		    /*
		     * If we're assigning targets for the first time, give
		     * this a new target number.  Otherwise, reassign it the
		     * target number it already has, to keep this number
		     * consistent.
		     */
		    tnum = assign_target_numbers ? target : ddb_entry->target;
	            if (qla4xxx_discover_target_luns(ha, ddb_entry, tnum)
		        == QLA_SUCCESS) {
			    status = QLA_SUCCESS;
			    target++;
		    }
                } else {
			break; /* Did not find anything, so we are done */
                }
	}
	up(&ha->targ_array_sem);
#else

	list_for_each_safe(ptr, next, &ha->ddb_list)
	{
		ddb_entry_t *ddb_entry;

		/* Get ddb_entry */
		ddb_entry = list_entry(ptr, ddb_entry_t, list_entry);

		/* Perform target/lun discovery */
		if (ddb_entry != NULL &&
		    qla4xxx_discover_target_luns(ha, ddb_entry, target)
		    == QLA_SUCCESS) {
			status = QLA_SUCCESS;
			target++;
		}

		if (target == MAX_TARGETS)
			break;
	}
#endif

	LEAVE("qla4xxx_map_targets_to_ddbs");
	return(status);

}

/**************************************************************************
 * qla4xxx_alloc_ddb
 *	This routine allocates a ddb_entry, ititializes some values, and
 *	inserts it into the ddb list.
 *
 * Input:
 * 	ha - Pointer to host adapter structure.
 *      fw_ddb_index - Firmware's device database index
 *
 * Returns:
 *	Pointer to internal device database structure
 *
 * Context:
 *	Kernel context.
 **************************************************************************/
static ddb_entry_t *
qla4xxx_alloc_ddb(scsi_qla_host_t *ha, uint32_t fw_ddb_index)
{
	ddb_entry_t     *ddb_entry;
#if 1 || __VMKERNEL_MODULE__
	unsigned long flags;
#endif

	QL4PRINT(QLP12, printk("scsi%d: %s: fw_ddb_index [%d]\n",
			       ha->host_no, __func__, fw_ddb_index));

	ddb_entry = (ddb_entry_t *) kmalloc(sizeof(*ddb_entry), GFP_KERNEL);
	if (ddb_entry == NULL) {
		QL4PRINT(QLP2,
			 printk("scsi%d: %s: Unable to allocate memory to "
				"add fw_ddb_index [%d]\n",
				ha->host_no, __func__, fw_ddb_index));
	}
	else {
		memset(ddb_entry, 0, sizeof(*ddb_entry));
		ddb_entry->fw_ddb_index = fw_ddb_index;
		atomic_set(&ddb_entry->port_down_timer,
			   ha->port_down_retry_count);
		atomic_set(&ddb_entry->retry_relogin_timer, INVALID_ENTRY);
		atomic_set(&ddb_entry->relogin_timer, 0);
		atomic_set(&ddb_entry->state, DEV_STATE_ONLINE);
#if 1 || __VMKERNEL_MODULE__
		spin_lock_irqsave(&ha->ddb_list_spinlock, flags);
#endif
		list_add_tail(&ddb_entry->list_entry, &ha->ddb_list);
#if 1 || __VMKERNEL_MODULE__
		spin_unlock_irqrestore(&ha->ddb_list_spinlock, flags);
#endif
		ha->fw_ddb_index_map[fw_ddb_index] = ddb_entry;
		ha->tot_ddbs++;
	}

	return(ddb_entry);
}

/**************************************************************************
 * qla4xxx_build_ddb_list
 *	This routine searches for all valid firmware ddb entries and builds
 *	an internal ddb list.
 *
 * Input:
 * 	ha - Pointer to host adapter structure.
 *
 * Remarks:
 *	Ddbs that are considered valid are those with a device state of
 *	SESSION_ACTIVE.
 *
 * Returns:
 *	QLA_SUCCESS - Successfully built internal ddb list
 *	QLA_ERROR   - Failed to build internal ddb list
 *
 * Context:
 *	Kernel context.
 **************************************************************************/
static uint8_t
qla4xxx_build_ddb_list(scsi_qla_host_t *ha)
{
	uint8_t         status = QLA_ERROR;
	uint32_t        fw_ddb_index = 0;
	uint32_t        next_fw_ddb_index = 0;
	uint32_t        ddb_state;
	ddb_entry_t     *ddb_entry;

	ENTER("qla4xxx_build_ddb_list");

	for (fw_ddb_index = 0;
	    fw_ddb_index < MAX_DDB_ENTRIES;
	    fw_ddb_index = next_fw_ddb_index) {
		/* First, let's see if a device exists here */
		if (qla4xxx_get_ddb_entry(ha, fw_ddb_index, NULL, NULL,
					  &next_fw_ddb_index, &ddb_state,
					  NULL, NULL, NULL)
		    == QLA_ERROR) {
			QL4PRINT(QLP2,
				 printk("scsi%d: %s: get_ddb_entry, "
					"fw_ddb_index %d failed",
					ha->host_no, __func__, fw_ddb_index));
			goto exit_build_ddb_list;
		}

		/* If the device is logged in (SESSION_ACTIVE) then
		 * add it to internal our ddb list. */
		if (ddb_state == DDB_DS_SESSION_ACTIVE) {
			/* Allocate a device structure */
			ddb_entry = qla4xxx_alloc_ddb(ha, fw_ddb_index);
			if (ddb_entry == NULL) {
				QL4PRINT(QLP2,
					 printk("scsi%d: %s: Unable to allocate "
						"memory for device at "
						"fw_ddb_index %d\n",
						ha->host_no, __func__,
						fw_ddb_index));
				goto exit_build_ddb_list;
			}

			/* Fill in the device structure */
			if (qla4xxx_update_ddb_entry(ha, ddb_entry)
			    == QLA_ERROR) {
				qla4xxx_free_ddb(ha, ddb_entry);
				QL4PRINT(QLP2,
					 printk("scsi%d: %s: update_ddb_index,  "
						"fw_ddb_index %d failed\n",
						ha->host_no, __func__,
						fw_ddb_index));
				goto exit_build_ddb_list;

			}

			/* if fw_ddb with session active state found,
			 * add to ddb_list */
			QL4PRINT(QLP7,
				 printk("scsi%d: %s: fw_ddb index [%d] "
					"added to ddb list\n",
					ha->host_no, __func__,
					fw_ddb_index));
		}
		else if (ddb_state == DDB_DS_SESSION_FAILED) {
			QL4PRINT(QLP7,
				 printk("scsi%d: %s: Attempt to login index [%d]\n",
					ha->host_no, __func__,
					fw_ddb_index));
			qla4xxx_set_ddb_entry(ha, fw_ddb_index, NULL);
		}

		/* We know we've reached the last device when
		 * next_fw_ddb_index is 0 */
		if (next_fw_ddb_index == 0)
			break;
	}

	/* tot_ddbs updated in alloc/free_ddb routines */
	if (ha->tot_ddbs)
		status = QLA_SUCCESS;


	exit_build_ddb_list:
	LEAVE("qla4xxx_build_ddb_list");
	return(status);
}

/**************************************************************************
 * qla4xxx_initialize_ddb_list
 *	This routine obtains device information from the F/W database during
 *	driver load time.  The device table is rebuilt from scratch.
 *
 * Input:
 * 	ha - Pointer to host adapter structure.
 *
 * Returns:
 *	QLA_SUCCESS - Successfully (re)built internal ddb list
 *	QLA_ERROR   - Failed to (re)build internal ddb list
 *
 * Context:
 *	Kernel context.
 **************************************************************************/
STATIC uint8_t
qla4xxx_initialize_ddb_list(scsi_qla_host_t *ha)
{
	uint16_t        target, fw_ddb_index;
	uint8_t         status = QLA_SUCCESS;

	ENTER("qla4xxx_initialize_ddb_list");

	/* reinitialize suspended lun queue */
	INIT_LIST_HEAD(&ha->suspended_lun_q);
	ha->suspended_lun_q_count = 0;

	/* free the ddb list if is not empty */
	if (!list_empty(&ha->ddb_list))
		qla4xxx_free_ddb_list(ha);

	/* Initialize internal DDB list and mappingss */
	for (target = 0; target < MAX_TARGETS; target++)
		ha->target_map[target] = (ddb_entry_t *) INVALID_ENTRY;

	for (fw_ddb_index = 0; fw_ddb_index < MAX_DDB_ENTRIES; fw_ddb_index++) {
		ha->fw_ddb_index_map[fw_ddb_index] =
		(ddb_entry_t *) INVALID_ENTRY;
	}

	ha->tot_ddbs = 0;

	if (ha->tcp_options & TOPT_SLP_UA_ENABLE) {
		QL4PRINT(QLP7, printk(KERN_INFO "scsi%d: Delay %d seconds while targets are "
				      "being discovered.\n",
				      ha->host_no, ql4xdiscoverywait));

		set_current_state(TASK_INTERRUPTIBLE);
		schedule_timeout(ql4xdiscoverywait * HZ);
	}

	/* First perform device discovery for active fw ddb indexes and
	 * build ddb list */
	qla4xxx_build_ddb_list(ha);

	/* Here we map a SCSI target to a fw_ddb_index and
	 * discover all possible luns. */
#if 1 || __VMKERNEL_MODULE__
	qla4xxx_map_targets_to_ddbs(ha, 1);
#else
	qla4xxx_map_targets_to_ddbs(ha);
#endif

	/* Targets can come online after the inital discovery,
	 * so processing the aens here will catch them. */
	qla4xxx_process_aen(ha, 0);

	if (!ha->tot_ddbs) {
		status = QLA_ERROR;
	}

	LEAVE("qla4xxx_initialize_ddb_list");
	return(status);
}

/**************************************************************************
 * qla4xxx_reinitialize_ddb_list
 *	This routine obtains device information from the F/W database after
 *	firmware or adapter resets.  The device table is preserved.
 *
 * Input:
 * 	ha - Pointer to host adapter structure.
 *
 * Returns:
 *	QLA_SUCCESS - Successfully updated internal ddb list
 *	QLA_ERROR   - Failed to update internal ddb list
 *
 * Context:
 *	Kernel context.
 **************************************************************************/
STATIC uint8_t
qla4xxx_reinitialize_ddb_list(scsi_qla_host_t *ha)
{
	uint8_t         status = QLA_SUCCESS;
	struct list_head *ptr, *next;
	ddb_entry_t *ddb_entry;

	ENTER("qla4xxx_reinitialize_ddb_list");

	/* Update the device information for all devices */
	list_for_each_safe(ptr, next, &ha->ddb_list)
	{
		ddb_entry = list_entry(ptr, ddb_entry_t, list_entry);

		if (ddb_entry) {
			qla4xxx_update_ddb_entry(ha, ddb_entry);

			if (ddb_entry->fw_ddb_device_state == DDB_DS_SESSION_ACTIVE) {
				atomic_set(&ddb_entry->state, DEV_STATE_ONLINE);

				QL4PRINT(QLP3|QLP7, printk(KERN_INFO "scsi%d:%d:%d: %s: index [%d] "
							   "marked ONLINE\n",
							   ha->host_no, ddb_entry->bus,
							   ddb_entry->target, __func__,
							   ddb_entry->fw_ddb_index));
			}
			else {
				if (atomic_read(&ddb_entry->state) ==
				    DEV_STATE_ONLINE) {
					qla4xxx_mark_device_missing(ha, ddb_entry);
				}
			}
		}
	}

	LEAVE("qla4xxx_reinitialize_ddb_list");
	return(status);
}

/**************************************************************************
 * qla4xxx_reset_relogin_timer
 *	This routine triggers a relogin.  After the relogin_timer expires,
 *	the relogin gets scheduled.  We must wait a minimum amount of time
 *	since receiving an 0x8014 AEN with failed device_state or a logout
 *	response before we can issue another relogin.
 *
 * Input:
 * 	ha - Pointer to host adapter structure.
 *	ddb_entry - Pointer to device database entry
 *
 * Returns:
 *	None
 *
 * Context:
 *	Kernel context.
 **************************************************************************/
inline void
qla4xxx_reset_relogin_timer(scsi_qla_host_t *ha, ddb_entry_t *ddb_entry)
{
	QL4PRINT(QLP3, printk(KERN_INFO "scsi%d:%d:%d: index [%d] "
			      "may relogin in %d seconds\n",
			      ha->host_no, ddb_entry->bus, ddb_entry->target,
			      ddb_entry->fw_ddb_index,
			      ddb_entry->default_time2wait));

#if 1 || __VMKERNEL_MODULE__
        /*
         * Make sure our time2wait timer is longer than the
         * firmware's, otherwise we just get an error returned from
         * the set_ddb mailbox call.  Also wipe out the relogin timer
         * until we actually try a relogin.
         */
        atomic_set(&ddb_entry->retry_relogin_timer, ddb_entry->default_time2wait + 4);
        atomic_set(&ddb_entry->relogin_timer, 0);
#else
        atomic_set(&ddb_entry->retry_relogin_timer, ddb_entry->default_time2wait);
#endif

}

/**************************************************************************
 * qla4xxx_mark_device_missing
 *	This routine marks a device missing and resets the relogin retry count.
 *
 * Input:
 * 	ha - Pointer to host adapter structure.
 *	ddb_entry - Pointer to device database entry
 *
 * Returns:
 *	None
 *
 * Context:
 *	Kernel context.
 **************************************************************************/
void
qla4xxx_mark_device_missing(scsi_qla_host_t *ha, ddb_entry_t *ddb_entry)
{
	atomic_set(&ddb_entry->state, DEV_STATE_MISSING);

	QL4PRINT(QLP3, printk(KERN_INFO "scsi%d:%d:%d: index [%d] "
			      "marked MISSING\n",
			      ha->host_no, ddb_entry->bus,
			      ddb_entry->target,
			      ddb_entry->fw_ddb_index));
}


#ifdef QLA4010
/**************************************************************************
 * qla4010_get_crash_record
 *	This routine retrieves a crash record from the QLA4010 after an
 *	8002h aen.
 *
 * Input:
 * 	ha - Pointer to host adapter structure.
 *
 * Returns:
 *	None
 *
 * Context:
 *	Kernel context.
 **************************************************************************/
void
qla4010_get_crash_record(scsi_qla_host_t *ha)
{
	uint32_t mbox_cmd[MBOX_REG_COUNT];
	uint32_t mbox_sts[MBOX_REG_COUNT];
	CRASH_RECORD    *crash_record;
	uint32_t        crash_record_size;

	ENTER("qla4010_get_crash_record");
	memset(&mbox_cmd, 0, sizeof(mbox_cmd));
	memset(&mbox_sts, 0, sizeof(mbox_cmd));

	/*
	 * Get size of crash record
	 */
	mbox_cmd[0] = MBOX_CMD_GET_CRASH_RECORD;

	if (qla4xxx_mailbox_command(ha, 5, 5, &mbox_cmd[0], &mbox_sts[0])
	    != QLA_SUCCESS) {
		QL4PRINT(QLP2, printk("scsi%d: %s: ERROR: Unable to retrieve size!\n",
				      ha->host_no, __func__));
		goto exit_get_crash_record;
	}

	crash_record_size = mbox_sts[4];
	if (crash_record_size == 0) {
		QL4PRINT(QLP2, printk("scsi%d: %s: ERROR: Crash record size is 0!\n",
				      ha->host_no, __func__));
		goto exit_get_crash_record;
	}

	/*
	 * Alloc Memory for Crash Record
	 */
#if 0 && __VMKERNEL_MODULE__
	spin_lock(&ha->dma_buf_spinlock);
#else
	down(&ha->dma_buf_sem);
#endif
	crash_record = (CRASH_RECORD *) ha->dma_buf.virt_addr;

	if ((ha->dma_buf.buf_len < crash_record_size) &&
	    qla4xxx_resize_dma_buf(ha, &ha->dma_buf, crash_record_size)
	    != QLA_SUCCESS) {
		QL4PRINT(QLP2, printk("scsi%d: %s: ERROR: Unable to allocate enough"
				      " memory (%d bytes) for crash record!\n",
				      ha->host_no, __func__, crash_record_size));
#if 0 && __VMKERNEL_MODULE__
		spin_unlock(&ha->dma_buf_spinlock);
#else
		up(&ha->dma_buf_sem);
#endif
		goto exit_get_crash_record;
	}

	/*
	 * Get Crash Record
	 */
	mbox_cmd[0] = MBOX_CMD_GET_CRASH_RECORD;
	mbox_cmd[2] = LSDW(ha->dma_buf.phys_addr);
	mbox_cmd[3] = MSDW(ha->dma_buf.phys_addr);
	mbox_cmd[4] = crash_record_size;

	if (qla4xxx_mailbox_command(ha, 5, 5, &mbox_cmd[0], &mbox_sts[0])
	    != QLA_SUCCESS) {
		QL4PRINT(QLP2, printk("scsi%d: %s: ERROR: Unable to retrieve crash"
				      " record!\n", ha->host_no, __func__));
#if 0 && __VMKERNEL_MODULE__
		spin_unlock(&ha->dma_buf_spinlock);
#else
		up(&ha->dma_buf_sem);
#endif
		goto exit_get_crash_record;
	}

	/*
	 * Dump Crash Record
	 */
	QL4PRINT(QLP1, printk(KERN_INFO "scsi%d: Crash Record Dump:\n",
			      ha->host_no));
	QL4PRINT( QLP1,
		  printk(KERN_INFO "Firmware Version: %02d.%02d.%02d.%02d\n",
			 crash_record->fw_major_version,
			 crash_record->fw_minor_version,
			 crash_record->fw_patch_version,
			 crash_record->fw_build_version));
	QL4PRINT(QLP1, printk(KERN_INFO "Build Date: %s\n",
			      crash_record->build_date));
	QL4PRINT(QLP1, printk(KERN_INFO "Build Time: %s\n",
			      crash_record->build_time));
	QL4PRINT(QLP1, printk(KERN_INFO "Build User: %s\n",
			      crash_record->build_user));
	QL4PRINT(QLP1, printk(KERN_INFO "Card Serial #: %s\n",
			      crash_record->card_serial_num));
	QL4PRINT(QLP1,
		 printk(KERN_INFO "Time of Crash (in seconds): %d (0x%x)\n",
			crash_record->time_of_crash_in_secs,
			crash_record->time_of_crash_in_secs));
	QL4PRINT(QLP1,
		 printk(KERN_INFO "Time of Crash (in milliseconds): "
			"%d (0x%x)\n",
			crash_record->time_of_crash_in_ms,
			crash_record->time_of_crash_in_ms));
	QL4PRINT(QLP1,
		 printk(KERN_INFO "# frames in OUT RISC processor stack dump: "
			"%d (0x%x)\n",
			crash_record->out_RISC_sd_num_frames,
			crash_record->out_RISC_sd_num_frames));
	QL4PRINT(QLP1,
		 printk(KERN_INFO "# words in OAP stack dump: %d (0x%x)\n",
			crash_record->OAP_sd_num_words,
			crash_record->OAP_sd_num_words));
	QL4PRINT(QLP1,
		 printk(KERN_INFO "# frames in IAP stack dump: %d (0x%x)\n",
			crash_record->IAP_sd_num_frames,
			crash_record->IAP_sd_num_frames));
	QL4PRINT(QLP1,
		 printk(KERN_INFO "# words in IN RISC processor stack dump: "
			"%d (0x%x)\n",
			crash_record->in_RISC_sd_num_words,
			crash_record->in_RISC_sd_num_words));
	QL4PRINT(QLP1,
		 printk(KERN_INFO "\nOUT RISC processor register dump:\n"));
	qla4xxx_dump_dwords(QLP1, &crash_record->out_RISC_reg_dump,
			    sizeof(crash_record->out_RISC_reg_dump));
	QL4PRINT(QLP1,
		 printk(KERN_INFO "\nIN RISC processor register dump:\n"));
	qla4xxx_dump_dwords(QLP1, &crash_record->in_RISC_reg_dump,
			    sizeof(crash_record->in_RISC_reg_dump));
	QL4PRINT(QLP1, printk(KERN_INFO "\nOUT RISC processor stack dump:\n"));
	qla4xxx_dump_dwords(QLP1, &crash_record->in_out_RISC_stack_dump,
			    crash_record->OAP_sd_num_words);
	QL4PRINT(QLP1, printk(KERN_INFO "\nIN RISC processor stack dump:\n"));
	qla4xxx_dump_dwords(QLP1, &crash_record->in_out_RISC_stack_dump[0] +
			    crash_record->OAP_sd_num_words,
			    crash_record->in_RISC_sd_num_words);

#if 0 && __VMKERNEL_MODULE__
	spin_unlock(&ha->dma_buf_spinlock);
#else
	up(&ha->dma_buf_sem);
#endif

	exit_get_crash_record:
	LEAVE("qla4010_get_crash_record");
}

#include "ql4nvram.c"

#if QLA4040_SUPPORT_ENABLED
/**************************************************************************
 * qla4010_topcat_soft_reset
 *	This routine determines if the QLA4040 TopCat chip is present.
 *
 * Input:
 * 	ha - Pointer to host adapter structure.
 *
 * Returns:
 *	None.
 *
 * Context:
 *	Kernel context.
 **************************************************************************/
void
qla4010_get_topcat_presence(scsi_qla_host_t *ha)
{
#if 0
	if (ha->pdev->subsystem_device == QLA4040_SSDID_ISCSI ||
	    ha->pdev->subsystem_device == QLA4040C_SSDID_ISCSI) {
		set_bit(AF_TOPCAT_CHIP_PRESENT, &ha->flags);
	}
#else
	unsigned int flags;
	uint16_t topcat_present;

	if (qla4xxx_take_semaphore(ha, SEM_NVRAM, TIMED_WAIT) != QLA_SUCCESS) {
		QL4PRINT(QLP2,
			 printk(KERN_WARNING "scsi%d: %s: Unable to take "
				"SEM_NVRAM semaphore\n",
				ha->host_no, __func__));
		return;
	}

	spin_lock_irqsave(&ha->hardware_lock, flags);

	topcat_present = ((RD_REG_WORD(&ha->reg->generalPurposeInput) & GPIR_TOPCAT_DISABLED) == 0);
	if (topcat_present)
		set_bit(AF_TOPCAT_CHIP_PRESENT, &ha->flags);
	else
		clear_bit(AF_TOPCAT_CHIP_PRESENT, &ha->flags);

	qla4xxx_clear_semaphore(ha, SEM_NVRAM);
	spin_unlock_irqrestore(&ha->hardware_lock, flags);
#endif
}
#endif

/**************************************************************************
 * qla4010_start_firmware
 *	This routine performs the neccessary steps to start the firmware for
 *	the QLA4010 adapter.
 *
 * Input:
 * 	ha - Pointer to host adapter structure.
 *
 * Returns:
 *	QLA_SUCCESS - Successfully started QLA4010 firmware
 *	QLA_ERROR   - Failed to start QLA4010 firmware
 *
 * Context:
 *	Kernel context.
 **************************************************************************/
uint8_t
qla4010_start_firmware(scsi_qla_host_t *ha, uint8_t force_soft_reset)
{
	uint32_t  max_wait_time;
	unsigned long flags = 0;
	uint32_t mbox_status;
	uint8_t status = QLA_ERROR;
	uint16_t  extHwConfig = 0;

	ENTER("qla4010_start_firmware");

	#if QLA4040_SUPPORT_ENABLED
	qla4010_get_topcat_presence(ha);
	#endif

	if (qla4xxx_take_semaphore(ha, SEM_HW_LOCK, TIMED_WAIT) != QLA_SUCCESS) {
		QL4PRINT(QLP2, printk(KERN_WARNING "scsi%d: %s: Unable to "
				      "take SEM_HW_LOCK semaphore (1)\n",
				      ha->host_no, __func__));
		return(QLA_ERROR);
	}

	/*
	 * Issue Soft Reset
	 */
	if (force_soft_reset == FALSE) {
		uint32_t port_status = RD_REG_DWORD(&ha->reg->portStatus);

		/* If the chip has not already been initialized,
		 * force a soft reset */
		QL4PRINT(QLP7, printk("scsi%d: %s: PortStatus = 0x%08X\n",
				      ha->host_no, __func__, port_status));

		if ((port_status & PSR_INIT_COMPLETE) == 0) {
			QL4PRINT(QLP7, printk("scsi%d: %s: Chip has not "
					      "already been initialized\n",
					      ha->host_no, __func__));
			force_soft_reset = TRUE;
			goto do_reset;
		}

		#if 1
		/* If the BIOS is loaded then the firmware is already initialized.
		 * Reinitializing it without first performing a reset is a NO-NO.
		 * We need to check here if the BIOS is loaded
		 * (i.e. FW_STATE_CONFIG_WAIT == 0).  If so, force a soft reset.
		 */
		if (qla4xxx_get_firmware_state(ha) == QLA_SUCCESS) {
			if ((ha->firmware_state & FW_STATE_CONFIG_WAIT) == 0) {
				QL4PRINT(QLP7, printk("scsi%d: %s: Firmware has been initialized by BIOS -- RESET\n",
						      ha->host_no, __func__));
				force_soft_reset = TRUE;
			}
		}
		else {
			QL4PRINT(QLP2, printk("scsi%d: %s: Error detecting if "
					      "firmware has already been "
					      "initialized-- RESET\n",
					      ha->host_no, __func__));
			force_soft_reset = TRUE;
		}
		#else
		// FIXME: KH 6/11/04 - Once this is supported in the firmware,
		// it should be used in place of the code above.
		{
			/* If the BIOS is loaded then the firmware is already initialized.
			 * Reinitializing it without first performing a reset is a NO-NO.
			 * We need to check here if the firmware is already initialized.
			 * If so, force a soft reset.
			 */
			uint32_t port_control = RD_REG_DWORD(&ha->reg->portControl);
			if ((port_control & PCR_FIRMWARE_INITIALIZED) != 0) {
				QL4PRINT(QLP7, printk("scsi%d: %s: Firmware has "
						      "already been initialized-- RESET\n",
						      ha->host_no, __func__));
				force_soft_reset = TRUE;
				goto do_reset;
			}
		}
		#endif
	}

	do_reset:
	if (force_soft_reset == TRUE) {
		QL4PRINT(QLP7, printk("scsi%d: %s: Issue Soft Reset\n",
				      ha->host_no, __func__));
		status = qla4xxx_soft_reset(ha);

		if (qla4xxx_take_semaphore(ha, SEM_HW_LOCK, TIMED_WAIT) != QLA_SUCCESS) {
			QL4PRINT(QLP2, printk(KERN_WARNING "scsi%d: %s: Unable to take SEM_HW_LOCK "
					      "semaphore (2)\n", ha->host_no, __func__));
			return(QLA_ERROR);
		}

		if (status == QLA_ERROR) {
			QL4PRINT(QLP3|QLP7, printk("scsi%d: %s: Soft Reset failed!\n",
						   ha->host_no, __func__));
			qla4xxx_clear_semaphore(ha, SEM_HW_LOCK);
			return(status);
		}
	}

	/*
	 * Set up memory management (Hardware Configuration Manager)
	 */
	QL4PRINT(QLP7, printk("scsi%d: %s: Set up Memory Management\n",
			      ha->host_no, __func__));

	if (qla4xxx_take_semaphore(ha, SEM_FLASH, TIMED_WAIT) != QLA_SUCCESS) {
		QL4PRINT(QLP2, printk(KERN_WARNING "scsi%d: %s: Unable to take SEM_FLASH "
				      "semaphore\n", ha->host_no, __func__));
		return(QLA_ERROR);
	}
	if (qla4xxx_take_semaphore(ha, SEM_NVRAM, TIMED_WAIT) != QLA_SUCCESS) {
		QL4PRINT(QLP2, printk(KERN_WARNING "scsi%d: %s: Unable to take SEM_NVRAM "
				      "semaphore\n", ha->host_no, __func__));
		return(QLA_ERROR);
	}

	extHwConfig = RD_NVRAM_WORD(ha, &ha->nvram->extHwConfig);
	QL4PRINT(QLP7, printk("scsi%d: %s: Setting extHwConfig to 0xFFFF%04x\n",
			      ha->host_no, __func__, extHwConfig));

	spin_lock_irqsave(&ha->hardware_lock, flags);
	WRT_REG_DWORD(&ha->reg->extHardwareConfiguration, ((0xFFFF << 16) | extHwConfig));
	qla4xxx_clear_semaphore(ha, SEM_NVRAM);
	qla4xxx_clear_semaphore(ha, SEM_FLASH);

	/*
	 * Start firmware from flash ROM
	 *
	 * WORKAROUND: Stuff a non-constant value that the firmware can use
	 * as a seed for a random number generator in MB7 prior to setting
	 * BOOT_ENABLE.  Fixes problem where the TCP connections use the
	 * same TCP ports after each reboot, causing some connections to
	 * not get re-established.
	 */
	QL4PRINT(QLP7, printk("scsi%d: %s: Start firmware from flash ROM\n",
			      ha->host_no, __func__));
	WRT_REG_DWORD(&ha->reg->mailboxIn[6], jiffies);
	WRT_REG_DWORD(&ha->reg->ISPControlStatus, CSRW_BOOT_ENABLE);
	spin_unlock_irqrestore(&ha->hardware_lock, flags);

	/*
	 * Check to see if firmware is UP
	 */
	max_wait_time = FIRMWARE_UP_TOV;
	do {
		spin_lock_irqsave(&ha->hardware_lock, flags);
		mbox_status = RD_REG_DWORD(&ha->reg->mailboxOut0);
		spin_unlock_irqrestore(&ha->hardware_lock, flags);

		if (mbox_status == MBOX_ASTS_SELF_TEST_FAILED) {
			QL4PRINT(QLP2,
				 printk("scsi%d: %s: Self Test failed with "
					"status 0x%x\n", ha->host_no, __func__,
					RD_REG_DWORD(&ha->reg->mailboxOut[0])));

			break;
		}
		else if (mbox_status == MBOX_STS_COMMAND_COMPLETE) {
			QL4PRINT(QLP7, printk("scsi%d: %s: Firmware has started\n",
					      ha->host_no, __func__));

			ha->firmware_version[0] = RD_REG_DWORD(&ha->reg->mailboxOut[0]);
			ha->firmware_version[1] = RD_REG_DWORD(&ha->reg->mailboxOut[1]);
			ha->patch_number        = RD_REG_DWORD(&ha->reg->mailboxOut[2]);
			ha->build_number        = RD_REG_DWORD(&ha->reg->mailboxOut[3]);

			QL4PRINT(QLP7, printk("scsi%d: FW Version %02d.%02d Patch %02d Build %02d\n",
					      ha->host_no, ha->firmware_version[0], ha->firmware_version[1],
					      ha->patch_number, ha->build_number));

			WRT_REG_DWORD(&ha->reg->ISPControlStatus,
				      CSRW_SCSI_PROCESSOR_INTR);

			status = QLA_SUCCESS;
			break;
		}

		QL4PRINT(QLP7, printk("scsi%d: %s: Waiting for firmware to come up... mbox_sts=0x%x, remaining=%d\n",
				      ha->host_no, __func__, mbox_status, max_wait_time));

		set_current_state(TASK_INTERRUPTIBLE);
		schedule_timeout(1 * HZ);

	} while ((max_wait_time--));

	if (status == QLA_SUCCESS) {
		/*
		 * Enable SCSI processor interrupts
		 */
		QL4PRINT(QLP7, printk("scsi%d: %s: Enable SCSI processor interrupts\n",
				      ha->host_no, __func__));
		qla4xxx_enable_intrs(ha);

		if (test_and_clear_bit(AF_GET_CRASH_RECORD, &ha->flags))
			qla4010_get_crash_record(ha);
	}
	else {
		QL4PRINT(QLP7, printk("scsi%d: %s: Firmware has NOT started\n",
				      ha->host_no, __func__));

		qla4xxx_dump_registers(QLP7, ha);
	}

	qla4xxx_clear_semaphore(ha, SEM_HW_LOCK);
	LEAVE("qla4010_start_firmware");
	return(status);
}
#endif

/**************************************************************************
 * qla4xxx_initialize_adapter
 *	This routine parforms all of the steps necessary to initialize the
 *	adapter.
 *
 * Input:
 * 	ha - Pointer to host adapter structure.
 *	renew_ddb_list - Indicates what to do with the adapter's ddb list
 *			after adapter recovery has completed.
 *			0=preserve ddb list, 1=destroy and rebuild ddb list
 *
 * Returns:
 *	QLA_SUCCESS - Successfully initialized adapter
 *	QLA_ERROR   - Failed to initialize adapter
 *
 * Context:
 *	Kernel context.
 **************************************************************************/
STATIC uint8_t
qla4xxx_initialize_adapter(scsi_qla_host_t *ha,
			   uint8_t renew_ddb_list)
{
	uint8_t      status;

	ENTER("qla4xxx_initialize_adapter");

	/* Initialize the Host adapter request/response queues and firmware */
#ifdef QLA4010
	if ((status = qla4010_start_firmware(ha, NO_SOFT_RESET)) == QLA_ERROR) {
		QL4PRINT(QLP2, printk(KERN_INFO "scsi%d: Failed to start QLA4010 "
				      "firmware\n", ha->host_no));
	}
#endif
#ifdef QLA4000
	if ((status = qla4xxx_get_fw_version(ha)) == QLA_ERROR) {
		QL4PRINT(QLP2, printk(KERN_WARNING "scsi%d: Failed to get firmware "
				      "version\n", ha->host_no));
	}
#endif

	else if ((status = qla4xxx_validate_mac_address(ha)) == QLA_ERROR) {
		QL4PRINT(QLP2, printk(KERN_WARNING "scsi%d: Failed to validate mac "
				      "address\n", ha->host_no));
	}
#ifndef QLA4000
	else if ((status = qla4xxx_init_local_data(ha)) == QLA_ERROR) {
		QL4PRINT(QLP2, printk(KERN_WARNING "scsi%d: Failed to initialize "
				      "local data\n", ha->host_no));
	}
#endif
	else if ((status = qla4xxx_init_firmware(ha)) == QLA_ERROR) {
		QL4PRINT(QLP2, printk(KERN_WARNING "scsi%d: Failed to initialize "
				      "firmware\n", ha->host_no));
	}
	else {
		if (renew_ddb_list == PRESERVE_DDB_LIST) {
			/*
			 * We want to preserve lun states (i.e. suspended, etc.)
			 * for recovery initiated by the driver.  So just update
			 * the device states for the existing ddb_list
			 */
			qla4xxx_reinitialize_ddb_list(ha);
		}
		else if (renew_ddb_list == REBUILD_DDB_LIST) {
			/*
			 * We want to build the ddb_list from scratch during
			 * driver initialization and recovery initiated by the
			 * INT_HBA_RESET IOCTL.
			 */
			qla4xxx_initialize_ddb_list(ha);
		}

#ifndef QLA4000
		if (test_bit(ISNS_FLAG_ISNS_ENABLED_IN_ISP, &ha->isns_flags)) {
			unsigned long wait_cnt = jiffies + ql4xdiscoverywait * HZ;

			if (!test_bit(ISNS_FLAG_ISNS_SRV_ENABLED,
				      &ha->isns_flags)) {
				QL4PRINT(QLP7,
					 printk(KERN_INFO "scsi%d: Delay up "
						"to %d seconds while targets "
						"are being discovered.\n",
						ha->host_no,
						ql4xdiscoverywait));

				QLA4XXX_WAIT(wait_cnt,
					     (test_bit(ISNS_FLAG_ISNS_SRV_ENABLED,
						       &ha->isns_flags)));
			}

			if (!test_bit(ISNS_FLAG_ISNS_SRV_ENABLED,
				      &ha->isns_flags)) {
				QL4PRINT(QLP2,
					 printk(KERN_WARNING "scsi%d: iSNS "
						"service failed to start\n",
						ha->host_no));
			}
		}
#endif

		if (!ha->tot_ddbs)
			QL4PRINT(QLP2,
				 printk(KERN_WARNING "scsi%d: Failed to "
					"initialize devices\n", ha->host_no));
	}

	LEAVE("qla4xxx_initialize_adapter");
	return(status);
}

/**************************************************************************
 * qla4xxx_get_req_pkt
 *	This routine performs the following tasks:
 *	- returns the current request_in pointer (if queue not full)
 *	- advances the request_in pointer
 *	- checks for queue full
 *
 * Input:
 * 	ha - Pointer to host adapter structure.
 *	queue_entry - Pointer to pointer to queue entry structure
 *
 * Output:
 *	queue_entry - Return pointer to next available request packet
 *
 * Returns:
 *	QLA_SUCCESS - Successfully retrieved request packet
 *	QLA_ERROR   - Failed to retrieve request packet
 *
 * Context:
 *	Kernel context.
 **************************************************************************/
uint8_t
qla4xxx_get_req_pkt(scsi_qla_host_t *ha,
		    QUEUE_ENTRY **queue_entry)
{
	uint16_t  request_in;
	uint8_t   status = QLA_SUCCESS;

	DBG(ENTER("qla4xxx_get_req_pkt");)

	*queue_entry = ha->request_ptr;

	/* get the latest request_in and request_out index */
	request_in = ha->request_in;

	ha->request_out  = ISP_RD_REQ_Q_OUT(ha);

	/* Advance request queue pointer and check for queue full */
	if (request_in == (REQUEST_QUEUE_DEPTH - 1)) {
		request_in = 0;
		ha->request_ptr = ha->req_queue_headv;
		DBG(QL4PRINT(QLP10, printk("scsi%d: %s: wraparound -- new request_in = "
					   "%04x, new request_ptr = %p\n",
					   ha->host_no, __func__,
					   request_in, ha->request_ptr));)
	}
	else {
		request_in++;
		ha->request_ptr++;
		DBG(QL4PRINT(QLP10, printk("scsi%d: %s: new request_in = %04x, "
					   "new request_ptr = %p\n",
					   ha->host_no, __func__,
					   request_in, ha->request_ptr));)
	}

	/* request queue is full, try again later */
	if ((ha->iocb_cnt + 1) >= ha->iocb_hiwat) {
		QL4PRINT(QLP2,
			 printk("scsi%d: %s: request queue is full, "
				"iocb_cnt=%d, iocb_hiwat=%d\n",
				ha->host_no, __func__,
				ha->iocb_cnt, ha->iocb_hiwat));

		/* restore request pointer */
		ha->request_ptr = *queue_entry;
		QL4PRINT(QLP2, printk("scsi%d: %s: restore request_ptr = %p, "
				      "request_in = %04x, request_out = %04x\n",
				      ha->host_no, __func__, ha->request_ptr,
				      ha->request_in, ha->request_out));
		status = QLA_ERROR;
	}
	else {
		ha->request_in = request_in;
		memset(*queue_entry, 0, sizeof(**queue_entry));
	}

	DBG(LEAVE("qla4xxx_get_req_pkt");)
	return(status);
}

/**************************************************************************
 * qla4xxx_send_marker_iocb
 *	This routine issues a marker IOCB.
 *
 * Input:
 * 	ha - Pointer to host adapter structure.
 *	ddb_entry - Pointer to device database entry
 *	lun - SCSI LUN
 *	marker_type - marker identifier
 *
 * Returns:
 *	QLA_SUCCESS - Successfully sent marker IOCB
 *	QLA_ERROR   - Failed to send marker IOCB
 *
 * Context:
 *	Kernel context.
 **************************************************************************/
STATIC uint8_t
qla4xxx_send_marker_iocb(scsi_qla_host_t *ha,
			 ddb_entry_t *ddb_entry,
			 lun_entry_t *lun_entry)
{
	MARKER_ENTRY *marker_entry;
	unsigned long flags = 0;
	uint8_t       status = QLA_SUCCESS;

	ENTER("qla4xxx_send_marker_iocb");

	/* Acquire hardware specific lock */
	spin_lock_irqsave(&ha->hardware_lock, flags);

	/* Get pointer to the queue entry for the marker */
	if (qla4xxx_get_req_pkt(ha, (QUEUE_ENTRY **) &marker_entry)
	    != QLA_SUCCESS) {
		QL4PRINT(QLP2,
			 printk("scsi%d: %s: request queue full, try again later\n",
				ha->host_no, __func__));

		status = QLA_ERROR;
		goto exit_send_marker;
	}

	/* Put the marker in the request queue */
	memset(marker_entry, 0, sizeof(*marker_entry));
	marker_entry->hdr.entryType  = ET_MARKER;
	marker_entry->hdr.entryCount = 1;
	marker_entry->target         = __cpu_to_le16(ddb_entry->fw_ddb_index);
	marker_entry->modifier       = __cpu_to_le16(MM_LUN_RESET);
	marker_entry->lun[1]         = lun_entry->lun;	 /*SAMII compliant lun*/

	wmb();

	QL4PRINT(QLP3, printk(KERN_INFO "scsi%d:%d:%d:%d: LUN_RESET Marker sent\n",
			      ha->host_no, ddb_entry->bus, ddb_entry->target, lun_entry->lun));

	/* Tell ISP it's got a new I/O request */
	WRT_REG_DWORD(&ha->reg->requestQueueInPointer, ha->request_in);

	lun_entry->lun_state = LS_LUN_READY;

	exit_send_marker:
	spin_unlock_irqrestore(&ha->hardware_lock, flags);
	LEAVE("qla4xxx_send_marker_iocb");
	return(status);
}

#ifdef QLA4010
PDU_ENTRY *
qla4xxx_get_pdu(scsi_qla_host_t *ha, uint32_t length)
{
	unsigned long flags;
	PDU_ENTRY       *pdu;
	PDU_ENTRY       *free_pdu_top;
	PDU_ENTRY       *free_pdu_bottom;
	uint16_t        pdu_active;

	uint16_t        i, j;
	uint16_t        num_pages;
	uint16_t        first_page;
	uint16_t        free_pages;
	uint8_t         pages_available;

	spin_lock_irqsave(&ha->pdu_spinlock, flags);

	if (ha->free_pdu_top == NULL) {
		QL4PRINT(QLP2|QLP19,
			 printk("scsi%d: %s: Out of PDUs!\n",
				ha->host_no, __func__));
		spin_unlock_irqrestore(&ha->pdu_spinlock, flags);
		return(NULL);
	}

	/* Save current state */
	free_pdu_top    = ha->free_pdu_top;
	free_pdu_bottom = ha->free_pdu_bottom;

	pdu = free_pdu_top;
	free_pdu_top = pdu->Next;

	if (free_pdu_top == NULL)
		free_pdu_bottom = NULL;

	pdu_active = ha->pdu_active + 1;

	QL4PRINT(QLP19,
		 printk("scsi%d: %s: Get PDU queue SUCCEEDED!  "
			"Top %p Bot %p PDU %p Active %d\n",
			ha->host_no, __func__,
			free_pdu_top, free_pdu_bottom, pdu, pdu_active));

	length = (length + (PAGE_SIZE-1)) & ~(PAGE_SIZE-1);
	num_pages = (uint16_t) (length / PAGE_SIZE);
	free_pages = 0;
	first_page = (uint16_t) (-1);
	pages_available = FALSE;

	/* Try to allocate contiguous free pages */
	for (i=0; i+num_pages <= MAX_PDU_ENTRIES; i++) {
		if (!ha->pdu_buf_used[i]) {
			first_page = i;
			pages_available = TRUE;
			for (j=i+1; j<num_pages; j++) {
				if (ha->pdu_buf_used[j]) {
					first_page = (uint16_t) (-1);
					pages_available = FALSE;
					i = j;
					break;
				}
			}
			if (pages_available)
				break;
		}
	}
	if (!pages_available) {
		QL4PRINT(QLP2|QLP19,
			 printk("scsi%d: %s: No contiguous pages available! "
				"Top %p Bot %p PDU %p Active %d\n",
				ha->host_no, __func__,
				free_pdu_top, free_pdu_bottom, pdu, pdu_active));
		spin_unlock_irqrestore(&ha->pdu_spinlock, flags);
		return(NULL);
	}

	for (i=0; i<num_pages; i++)
		ha->pdu_buf_used[first_page+i] = TRUE;

	ha->free_pdu_top = free_pdu_top;
	ha->free_pdu_bottom = free_pdu_bottom;
	ha->pdu_active = pdu_active;

	/* Fill in PDU */
	pdu->Buff = &((dma_mem_blk_t *)ha->dma_mem_blkv)->pdu_buffs[first_page][0];
	pdu->BuffLen = length;
	pdu->SendBuffLen = 0;
	pdu->RecvBuffLen = 0;
	pdu->Next = NULL;

	QL4PRINT(QLP19,
		 printk("scsi%d: %s: Get PDU buffers SUCCEEDED!  "
			"Top %p Bot %p PDU %p Buf %p Length %x Active %d\n",
			ha->host_no, __func__, free_pdu_top, free_pdu_bottom,
			pdu, pdu->Buff, pdu->BuffLen, pdu_active));
	spin_unlock_irqrestore(&ha->pdu_spinlock, flags);
	return(pdu);
}

void qla4xxx_free_pdu(scsi_qla_host_t *ha, PDU_ENTRY *pdu)
{
	unsigned long flags;
	uint16_t first_page;
	uint16_t num_pages;
	uint16_t i;

	spin_lock_irqsave(&ha->pdu_spinlock, flags);
	if (ha->free_pdu_bottom == NULL) {
		ha->free_pdu_top = pdu;
		ha->free_pdu_bottom = pdu;
	}
	else {
		ha->free_pdu_bottom->Next = pdu;
		ha->free_pdu_bottom = pdu;
	}

	/* Free buffer resources */
	first_page = (pdu->Buff - &((dma_mem_blk_t *)ha->dma_mem_blkv)->pdu_buffs[0][0]) / PAGE_SIZE;
	num_pages = (uint16_t) (pdu->BuffLen / PAGE_SIZE);
	for (i=0; i<num_pages; i++)
		ha->pdu_buf_used[first_page+i] = FALSE;
	ha->pdu_active--;

	QL4PRINT(QLP19,
		 printk("scsi%d: %s: Top %p Bot %p PDU %p Buf %p Length %x Active %d\n",
			ha->host_no, __func__, ha->free_pdu_top, ha->free_pdu_bottom,
			pdu, pdu->Buff, pdu->BuffLen, ha->pdu_active));

	/* Clear PDU */
	pdu->Buff = NULL;
	pdu->BuffLen = 0;
	pdu->SendBuffLen = 0;
	pdu->RecvBuffLen = 0;
	pdu->Next = NULL;

	spin_unlock_irqrestore(&ha->pdu_spinlock, flags);
}

/**************************************************************************
 * qla4xxx_send_passthru0_iocb
 *	This routine issues a passthru0 IOCB.
 *
 * Input:
 * 	ha - Pointer to host adapter structure.
 *
 * Remarks: hardware_lock acquired, interrupt context
 *
 * Returns:
 *	QLA_SUCCESS - Successfully sent marker IOCB
 *	QLA_ERROR   - Failed to send marker IOCB
 *
 * Context:
 *	Kernel context.
 **************************************************************************/
uint8_t
qla4xxx_send_passthru0_iocb(scsi_qla_host_t *ha,
			    uint16_t fw_ddb_index,
			    uint16_t connection_id,
			    uint8_t *pdu_data,
			    uint32_t send_len,
			    uint32_t recv_len,
			    uint16_t control_flags,
			    uint32_t handle)
{
	PASSTHRU0_ENTRY *passthru_entry;
	uint8_t         status = QLA_SUCCESS;
	uint8_t         *pdu_phys_data;

	ENTER("qla4xxx_send_passthru0_iocb");

	if (send_len) {
		QL4PRINT(QLP19, printk("PDU (0x%p) ->\n", pdu_data));
		qla4xxx_dump_bytes(QLP19, pdu_data, send_len);
	}

	/* Get pointer to the queue entry for the marker */
	if (qla4xxx_get_req_pkt(ha, (QUEUE_ENTRY **) &passthru_entry)
	    != QLA_SUCCESS) {
		QL4PRINT(QLP5|QLP2|QLP19,
			 printk("scsi%d: %s: request queue full, try again later\n",
				ha->host_no, __func__));

		status = QLA_ERROR;
		goto exit_send_pt0;
	}

	/* Fill in the request queue */
	memset(passthru_entry, 0, sizeof(*passthru_entry));
	passthru_entry->hdr.entryType  = ET_PASSTHRU0;
	passthru_entry->hdr.entryCount = 1;
	passthru_entry->handle         = __cpu_to_le32(handle);
	passthru_entry->target         = __cpu_to_le16(fw_ddb_index);
	passthru_entry->connectionID   = __cpu_to_le16(connection_id);
	passthru_entry->timeout        = PT_DEFAULT_TIMEOUT;

	pdu_phys_data = &((dma_mem_blk_t *)(unsigned long)
			  ha->dma_mem_blkp)->pdu_buffs[0][0] +
			(pdu_data - &ha->dma_mem_blkv->pdu_buffs[0][0]);

	if (send_len) {
		control_flags |= PT_FLAG_SEND_BUFFER;
		passthru_entry->outDataSeg64.base.addrHigh  =
		__cpu_to_le32(MSDW(pdu_phys_data));
		passthru_entry->outDataSeg64.base.addrLow   =
		__cpu_to_le32(LSDW(pdu_phys_data));
		passthru_entry->outDataSeg64.count          =
		__cpu_to_le32(send_len);
		QL4PRINT(QLP19,
			 printk("scsi%d: %s: sending 0x%X bytes, "
				"pdu_phys_data = %p\n",
				ha->host_no, __func__, send_len,
				pdu_phys_data));
	}

	if (recv_len) {
		passthru_entry->inDataSeg64.base.addrHigh = __cpu_to_le32(MSDW(pdu_phys_data));
		passthru_entry->inDataSeg64.base.addrLow  = __cpu_to_le32(LSDW(pdu_phys_data));
		passthru_entry->inDataSeg64.count         = __cpu_to_le32(recv_len);
		QL4PRINT(QLP19, printk("scsi%d: %s: receiving  0x%X bytes, pdu_phys_data = %p\n",
				       ha->host_no, __func__, recv_len, pdu_phys_data));
	}

	passthru_entry->controlFlags   = __cpu_to_le16(control_flags);

	wmb();

	QL4PRINT(QLP19, printk(KERN_INFO "scsi%d: Passthru0 IOCB type %x count %x In (%x) %p\n",
			       ha->host_no, passthru_entry->hdr.entryType,
			       passthru_entry->hdr.entryCount, ha->request_in, passthru_entry));
	qla4xxx_dump_bytes(QLP10, passthru_entry, sizeof(*passthru_entry));


	/* Tell ISP it's got a new I/O request */
	WRT_REG_DWORD(&ha->reg->requestQueueInPointer, ha->request_in);

	exit_send_pt0:
	LEAVE("qla4xxx_send_passthru0_iocb");
	return(status);
}
#endif

/**************************************************************************
 * add_to_active_array
 *	This routine adds an srb to the active array at the specified index.
 *
 * Input:
 * 	ha - Pointer to host adapter structure.
 *	index - index into to the active_array
 *	srb - Pointer to SCSI Request Block
 *
 * Returns:
 *	None
 *
 * Context:
 *	Kernel context.
 **************************************************************************/
void
add_to_active_array(scsi_qla_host_t *ha, uint32_t index, srb_t *srb)
{
	ddb_entry_t *ddb_entry = qla4xxx_lookup_ddb_by_fw_index(ha, srb->fw_ddb_index);
	lun_entry_t *lun_entry = qla4xxx_lookup_lun_handle(ha, ddb_entry, srb->lun);

	/* put command in active array */
	ha->active_srb_array[index] = srb;
	CMD_HANDLE(srb->cmd) = (unsigned char *) _32BIT_TO_ULONG(index);

	/* update counters */
	ha->active_srb_count++;
	ha->req_q_count -= srb->entry_count;
	if (ddb_entry) ddb_entry->out_count++;
	if (lun_entry) {
		lun_entry->out_count++;
		lun_entry->tot_io_count++;
	}
	srb->active_array_index = index;
	srb->state = SRB_ACTIVE_STATE;
}

/**************************************************************************
 * del_from_active_array
 *	This routine removes and returns the srb at the specified index
 *
 * Input:
 * 	ha - Pointer to host adapter structure.
 *	index - index into to the active_array
 *
 * Returns:
 *	Pointer to corresponding SCSI Request Block
 *
 * Context:
 *	Kernel/Interrupt context.
 **************************************************************************/
srb_t *
del_from_active_array(scsi_qla_host_t *ha, uint32_t index)
{
	srb_t *srb = NULL;

	/* validate handle and remove from active array */
	if (index < MAX_SRBS) {
		srb = ha->active_srb_array[index];
		ha->active_srb_array[index] = 0;

		if (srb) {
			ddb_entry_t *ddb_entry =
			qla4xxx_lookup_ddb_by_fw_index(ha,
						       srb->fw_ddb_index);
			lun_entry_t *lun_entry =
			qla4xxx_lookup_lun_handle(ha, ddb_entry, srb->lun);

			/* update counters */
			ha->req_q_count += srb->entry_count;
			ha->iocb_cnt -= srb->iocb_cnt;
			ha->active_srb_count--;
			if (ddb_entry) ddb_entry->out_count--;
			if (lun_entry) lun_entry->out_count--;
			srb->active_array_index = INVALID_ENTRY;
			CMD_HANDLE(srb->cmd) = NULL;
		}
		else
			QL4PRINT(QLP2,
				 printk(KERN_INFO "scsi%d: %s: array_index=%d "
					"already completed.\n",
					ha->host_no, __func__, index));
	}
	else
		QL4PRINT(QLP2, printk(KERN_INFO "scsi%d: %s: array_index=%d "
				      "exceeded max index of %d\n",
				      ha->host_no, __func__, index, MAX_SRBS));

	return(srb);
}

/**************************************************************************
 * qla4xxx_normalize_dma_addr()
 *	Normalize an DMA address.
 *
 * Input:
 * 	e_addr  - Raw DMA address
 * 	e_len   - Raw DMA length
 * 	ne_addr - Normalized second DMA address
 * 	ne_len  - Normalized second DMA length
 *
 * If the address does not span a 4GB page boundary, the contents of @ne_addr
 * and @ne_len are undefined.  @e_len is updated to reflect a normalization.
 *
 * Example:
 *
 * 	ffffabc0ffffeeee	(e_addr) start of DMA address
 * 	0000000020000000	(e_len)  length of DMA transfer
 *	ffffabc11fffeeed	end of DMA transfer
 *
 * Is the 4GB boundary crossed?
 *
 * 	ffffabc0ffffeeee	(e_addr)
 *	ffffabc11fffeeed	(e_addr + e_len - 1)
 *	00000001e0000003	((e_addr ^ (e_addr + e_len - 1))
 *	0000000100000000	((e_addr ^ (e_addr + e_len - 1)) & ~(0xffffffff)
 *
 * Compute start of second DMA segment:
 *
 * 	ffffabc0ffffeeee	(e_addr)
 *	ffffabc1ffffeeee	(0x100000000 + e_addr)
 *	ffffabc100000000	(0x100000000 + e_addr) & ~(0xffffffff)
 *	ffffabc100000000	(ne_addr)
 *	
 * Compute length of second DMA segment:
 *
 *	00000000ffffeeee	(e_addr & 0xffffffff)
 *	0000000000001112	(0x100000000 - (e_addr & 0xffffffff))
 *	000000001fffeeee	(e_len - (0x100000000 - (e_addr & 0xffffffff))
 *	000000001fffeeee	(ne_len)
 *
 * Adjust length of first DMA segment
 *
 * 	0000000020000000	(e_len)
 *	0000000000001112	(e_len - ne_len)
 *	0000000000001112	(e_len)
 *
 * Returns non-zero if the specified address was normalized, else zero.
 **************************************************************************/
static inline uint32_t
qla4xxx_normalize_dma_addr(
			  dma_addr_t *e_addr,  uint32_t *e_len,
			  dma_addr_t *ne_addr, uint32_t *ne_len)
{
	uint32_t normalized;

	normalized = 0;
	if ((*e_addr ^ (*e_addr + *e_len - 1)) & ~(0xFFFFFFFFULL)) {
		/* Compute normalized crossed address and len */
		*ne_addr = (0x100000000ULL + *e_addr) & ~(0xFFFFFFFFULL);
		*ne_len = *e_len - (0x100000000ULL - (*e_addr & 0xFFFFFFFFULL));
		*e_len -= *ne_len;

		normalized++;
	}
	return(normalized);
}

inline uint8_t
qla4xxx_alloc_cont_entry(scsi_qla_host_t *ha,
			 DATA_SEG_A64 **cur_dsd,
			 uint16_t *avail_dsds)
{
	CONTINUE_ENTRY *cont_entry;
	DBG(ENTER("qla4xxx_alloc_cont_entry");)

	/* Get request queue entry and adjust ring index. */
	if (qla4xxx_get_req_pkt(ha, (QUEUE_ENTRY **) &cont_entry) != QLA_SUCCESS) {
		QL4PRINT(QLP2, printk("scsi%d: %s: Unable to allocate "
				      "continuation packet\n",
				      ha->host_no, __func__));

		LEAVE("qla4xxx_alloc_cont_entry");
		return(QLA_ERROR);
	}

	cont_entry->hdr.entryType = ET_CONTINUE;
	cont_entry->hdr.entryCount = 1;
	cont_entry->hdr.systemDefined =
	(uint8_t) __cpu_to_le16(ha->request_in);
	*cur_dsd = (DATA_SEG_A64 *) &cont_entry->dataseg[0];
	*avail_dsds = CONTINUE_SEG;

	DBG(LEAVE("qla4xxx_alloc_cont_entry");)
	return(QLA_SUCCESS);
}

/**************************************************************************
 * qla4xxx_send_command_to_isp
 *	This routine is called by qla4xxx_queuecommand to build an ISP
 *	command and pass it to the ISP for execution.
 *
 * Input:
 * 	ha - Pointer to host adapter structure.
 *	srb - pointer to SCSI Request Block to be sent to ISP
 *
 * Output:
 *	None
 *
 * Remarks:
 *	None
 *
 * Returns:
 *	QLA_SUCCESS - Successfully sent command to ISP
 *	QLA_ERROR   - Failed to send command to ISP
 *
 * Context:
 *	Kernel context.
 **************************************************************************/
uint8_t
qla4xxx_send_command_to_isp(scsi_qla_host_t *ha, srb_t *srb)
{
	Scsi_Cmnd     *cmd = srb->cmd;
	ddb_entry_t   *ddb_entry;
	lun_entry_t   *lun_entry;
	COMMAND_ENTRY *cmd_entry;
	uint16_t      saved_request_in;
	QUEUE_ENTRY   *saved_request_ptr;

	uint16_t avail_dsds;
	DATA_SEG_A64 *cur_dsd;
	uint16_t      tot_dsds;	/* number of data segments */
				/* (sg entries, if sg request) */

	uint8_t       tot_iocbs;/* number of request queue entries */
				/* (commamd + continue) */
	unsigned long flags;
	uint16_t      i;
	uint32_t      index;
	uint8_t       found = FALSE;

	DBG(ENTER("qla4xxx_send_command_to_isp");)

	ddb_entry = qla4xxx_lookup_ddb_by_fw_index(ha, srb->fw_ddb_index);
	if (ddb_entry == NULL) {
		QL4PRINT(QLP2, printk("scsi%d: %s: invalid index [%d]\n",
				      ha->host_no, __func__, srb->fw_ddb_index));

		return(QLA_ERROR);
	}

	lun_entry = qla4xxx_lookup_lun_handle(ha, ddb_entry, srb->lun);
	if (lun_entry == NULL) {
		QL4PRINT(QLP2, printk("scsi%d:%d:%d: %s: invalid lun %d\n",
				      ha->host_no, cmd->channel, cmd->target,
				      __func__, cmd->lun));

		return(QLA_ERROR);
	}

	/*
	 * Send marker(s) if needed
	 *--------------------------*/
	if (ha->marker_needed == 1) {
		if (qla4xxx_send_marker_iocb(ha, ddb_entry, lun_entry)
		    != QLA_SUCCESS) {
			return(QLA_ERROR);
		}
		ha->marker_needed = 0;
	}

	/*
	 * Get iocb for the command
	 *--------------------------*/

	/* Acquire hardware specific lock */
	spin_lock_irqsave(&ha->hardware_lock, flags);

	/* Save some variables to undo things if an error occurs */
	saved_request_in  = ha->request_in;
	saved_request_ptr = ha->request_ptr;

	tot_dsds = 0;
	tot_iocbs = 1;
	avail_dsds = COMMAND_SEG;

	/* Get request queue entry and adjust ring index. */
	if (qla4xxx_get_req_pkt(ha, (QUEUE_ENTRY **) &cmd_entry) != QLA_SUCCESS) {
		QL4PRINT(QLP2,
			 printk("scsi%d:%d:%d:%d: %s: request queue is full, "
				"try again later\n",
				ha->host_no, cmd->channel, cmd->target,
				cmd->lun, __func__));

		goto exit_send_cmd;
	}

	/* Check for room in active srb array */
	index = ha->current_active_index;
	for (i=0; i<MAX_SRBS; i++) {
		index++;
		if (index == MAX_SRBS)
			index = 1;
		if (ha->active_srb_array[index] == 0) {
			found = 1;
			ha->current_active_index = index;
			break;
		}
	}
	if (!found) {
		QL4PRINT(QLP2,
			 printk("scsi%d:%d:%d:%d: %s: no room in active array, "
				"try again later\n",
				ha->host_no, cmd->channel, cmd->target,
				cmd->lun, __func__));

		goto exit_send_cmd_return_request;
	}

	/*
	 * Build command entry packet to send to ISP
	 *------------------------------------------*/

	/* If in connection mode, bump sequence number */
	if ((ha->firmware_options & FWOPT_SESSION_MODE) != 0)
		ddb_entry->CmdSn++;

	cmd_entry->hdr.entryType     = ET_COMMAND;
	cmd_entry->handle            = __cpu_to_le32(index);
	cmd_entry->target            = __cpu_to_le16(ddb_entry->fw_ddb_index);
	cmd_entry->connection_id     = __cpu_to_le16(ddb_entry->connection_id);
	cmd_entry->lun[1]            = LSB(cmd->lun); /*SAMII compliant lun*/
	cmd_entry->lun[2]            = MSB(cmd->lun);
	cmd_entry->cmdSeqNum         = __cpu_to_le32(ddb_entry->CmdSn);
	cmd_entry->ttlByteCnt        = __cpu_to_le32(cmd->request_bufflen);
	memcpy(cmd_entry->cdb, cmd->cmnd, cmd->cmd_len);

#ifdef __VMKERNEL_MODULE__
        /*
         * set a timeout for the command to make sure the FW doesn't
         * hang onto it forever.  If a command gets dropped, the
         * qla4010 will reset the session.
         */
	cmd_entry->timeout = ql4xdeftimeout;
#else
#if SET_IOCB_TIMEOUT

	/* Set firmware timeout to [target_mgmt_timeout + IOCB_TOV_MARGIN]
	 * seconds less than OS timeout.
	 * We want the firmware to time out the command first */
	cmd_entry->timeout           = CMD_TIMEOUT(cmd)/HZ;
	if (cmd_entry->timeout > ddb_entry->task_mgmt_timeout + IOCB_TOV_MARGIN)
		cmd_entry->timeout  -= (ddb_entry->task_mgmt_timeout + IOCB_TOV_MARGIN);
#else

	/* Do NOT time out commands in the RISC */
	cmd_entry->timeout           = 0;
#endif
#endif

	srb->iocb_tov = cmd_entry->timeout;
	srb->os_tov   = CMD_TIMEOUT(cmd)/HZ;

	DBG(QL4PRINT(QLP10,
		     printk("scsi%d:%d:%d:%d: %s: timeout set to %d seconds, \n",
			    ha->host_no, cmd->channel, cmd->target, cmd->lun,
			    __func__, cmd_entry->timeout));)

	/* Set data transfer direction control flags
	 * NOTE: Look at data_direction bits iff there is data to be
	 *       transferred, as the data direction bit is sometimed filled
	 *       in when there is no data to be transferred */
	if (cmd->request_bufflen) {
		switch (cmd->sc_data_direction) {
		case SCSI_DATA_READ:
			cmd_entry->control_flags = CF_READ;
			break;
		case SCSI_DATA_WRITE:
			cmd_entry->control_flags = CF_WRITE;
			break;
		case SCSI_DATA_NONE:
			cmd_entry->control_flags = CF_NO_DATA;
			break;
		case SCSI_DATA_UNKNOWN:
			if (cmd->request_bufflen != 0) {
				QL4PRINT(QLP2,
					 printk(KERN_WARNING
						"scsi%d: %s: Incorrect data "
						"direction - transfer "
						"length=%d, direction=%d, "
						"pid=%ld, opcode=%x\n",
						ha->host_no, __func__,
						cmd->request_bufflen,
						cmd->sc_data_direction,
						cmd->serial_number,
						cmd->cmnd[0]));
			}
		default:
			QL4PRINT(QLP2,
				 printk("scsi%d:%d:%d:%d: %s: No data "
					"direction specified!\n",
					ha->host_no, cmd->channel,
					cmd->target, cmd->lun, __func__));

			cmd_entry->control_flags = CF_NO_DATA;
		}
	}
	else {
		cmd_entry->control_flags = CF_NO_DATA;
	}


	/* Set tagged queueing control flags */
	switch (cmd->tag) {
	case HEAD_OF_QUEUE_TAG:
		cmd_entry->control_flags |= CF_HEAD_TAG;
		break;
	case ORDERED_QUEUE_TAG:
		cmd_entry->control_flags |= CF_ORDERED_TAG;
		break;
	case SIMPLE_QUEUE_TAG:
		cmd_entry->control_flags |= CF_SIMPLE_TAG;
		break;
	default:
		/*
		 * TEST_UNIT_READY commands from scsi_scan will fail
		 * due to "overlapped commands attempted" unless we
		 * setup at least a simple queue (midlayer will
		 * embelish this once it can do an INQUIRY command to
		 * the device)
		 */
		cmd_entry->control_flags |= CF_SIMPLE_TAG;  break;
	}

	cur_dsd = (DATA_SEG_A64 *) &(cmd_entry->dataseg[0]);

	/* Set data segments and byte counts */
	if (cmd->request_bufflen == 0 ||
	    cmd->sc_data_direction == SCSI_DATA_NONE) {
		/* No data being transferred */
		QL4PRINT(QLP5, printk("scsi%d:%d:%d:%d: %s: No data xfer\n",
				      ha->host_no, cmd->channel, cmd->target,
				      cmd->lun, __func__));

		tot_dsds = 0;
		cmd_entry->ttlByteCnt = __constant_cpu_to_le32(0);

	}
	else if (cmd->use_sg != 0) {
		struct  scatterlist *cur_seg;
		struct  scatterlist *end_seg;
		int     nseg;

		/* Data transfer with Scatter/Gather
		 *
		 * We must build an SG list in adapter format, as the kernel's
		 * SG list cannot be used directly because of data field size
		 * (__alpha__) differences and the kernel SG list uses virtual
		 * addresses where we need physical addresses.
		 */
		cur_seg = (struct scatterlist *) cmd->request_buffer;

#ifdef __VMKERNEL_MODULE__
                /*
                 * The dma addresses in sg have already been set up.
                 */
                nseg = cmd->use_sg;
#else
		nseg = pci_map_sg(ha->pdev, cur_seg, cmd->use_sg,
				  scsi_to_pci_dma_dir(cmd->sc_data_direction));
#endif

		if (nseg == 0)
			goto exit_send_cmd;
		end_seg = cur_seg + nseg;

		while (cur_seg < end_seg) {
			dma_addr_t      sle_dma;
			uint32_t        sle_len;
			dma_addr_t      nml_dma;
			uint32_t        nml_len;
			uint32_t        normalized;

			sle_dma = sg_dma_address(cur_seg);
			sle_len = sg_dma_len(cur_seg);

#ifdef __VMKERNEL_MODULE__
                        vmk_verify_memory_for_io(sle_dma, sle_len);
#endif
                        
                        normalized =
                           qla4xxx_normalize_dma_addr(&sle_dma, &sle_len,
                                                      &nml_dma, &nml_len);
                        
			/* Allocate continuation packet, if necessary,
			 * and point cur_dsd to 1st dsd entry */
			if (avail_dsds == 0) {
				tot_iocbs++;
				if (qla4xxx_alloc_cont_entry(ha, &cur_dsd,
							     &avail_dsds)
				    == QLA_ERROR) {
					QL4PRINT(QLP2,
						 printk("scsi%d:%d:%d:%d: %s: "
							"request queue full, "
							"unmap sg, try again "
							"later\n",
							ha->host_no,
							cmd->channel,
							cmd->target,
							cmd->lun, __func__));

					goto exit_send_cmd_return_dma;
				}
			}

			/* One entry always consumed */
			cur_dsd->base.addrLow = cpu_to_le32(LSDW(sle_dma));
			cur_dsd->base.addrHigh = cpu_to_le32(MSDW(sle_dma));
			cur_dsd->count = cpu_to_le32(sle_len);
			tot_dsds++;
			avail_dsds--;

			DBG(QL4PRINT(QLP5|QLP24,
				     printk("scsi%d:%d:%d:%d: %s: S/G DSD %p "
					    "phys_addr=%x:%08x, len=0x%x, "
					    "tot_dsd=0x%x, avail_dsd=0x%x\n",
					    ha->host_no, cmd->channel, cmd->target,
					    cmd->lun, __func__, cur_dsd,
					    cur_dsd->base.addrHigh,
					    cur_dsd->base.addrLow,
					    cur_dsd->count,
					    tot_dsds, avail_dsds));)
			cur_dsd++;

			if (normalized) {
				/* Allocate continuation packet,
				 * if necessary */
				if (avail_dsds == 0) {
					tot_iocbs++;
					if (qla4xxx_alloc_cont_entry(ha,
								     &cur_dsd,
								     &avail_dsds)
					    == QLA_ERROR) {
						goto exit_send_cmd_return_dma;
					}
				}

				/* One entry always consumed */
				cur_dsd->base.addrLow =
				cpu_to_le32(LSDW(nml_dma));
				cur_dsd->base.addrHigh =
				cpu_to_le32(MSDW(nml_dma));
				cur_dsd->count = cpu_to_le32(nml_len);
				tot_dsds++;
				avail_dsds--;

				DBG(QL4PRINT(QLP5|QLP24,
					     printk("scsi%d:%d:%d:%d: %s: "
						    "S/G Normalized DSD %p "
						    "phys_addr=%x:%08x, len=0x%x, "
						    "tot_dsd=0x%x, avail_dsd=0x%x\n",
						    ha->host_no, cmd->channel,
						    cmd->target, cmd->lun,
						    __func__, cur_dsd,
						    cur_dsd->base.addrHigh,
						    cur_dsd->base.addrLow,
						    cur_dsd->count,
						    tot_dsds, avail_dsds));)
				cur_dsd++;
			}
			cur_seg++;
		}
	}
	else {
		/* Data transfer without Scatter/Gather */
		dma_addr_t      req_dma;
		uint32_t        req_len;
		dma_addr_t      nml_dma;
		uint32_t        nml_len;
		uint32_t        normalized;


#ifdef __VMKERNEL_MODULE__
                /*
                 * We already have the machine address.
                 */
                req_dma = (unsigned long)cmd->request_bufferMA; 
                vmk_verify_memory_for_io(req_dma, cmd->request_bufflen);
#else
#if LINUX_VERSION_CODE >= KERNEL_VERSION(2,4,13)
		struct page *page = virt_to_page(cmd->request_buffer);
		unsigned long offset = ((unsigned long)
					cmd->request_buffer
					& ~PAGE_MASK);
                
		req_dma =
		pci_map_page(ha->pdev,
			     page,
			     offset,
			     cmd->request_bufflen,
			     scsi_to_pci_dma_dir(
						cmd->sc_data_direction));
#else
		req_dma =
		pci_map_single(ha->pdev,
			       cmd->request_buffer,
			       cmd->request_bufflen,
			       scsi_to_pci_dma_dir(
						cmd->sc_data_direction));
#endif
#endif
		srb->saved_dma_handle = req_dma;
		req_len = cmd->request_bufflen;

		if (!srb->saved_dma_handle) {
			QL4PRINT(QLP2,
				 printk("scsi%d:%d:%d:%d: %s: "
					"pci mapping failed!, "
					"try again later\n",
					ha->host_no, cmd->channel,
					cmd->target,
					cmd->lun, __func__));

			goto exit_send_cmd_return_dma;
		}

		normalized = qla4xxx_normalize_dma_addr(&req_dma, &req_len,
							&nml_dma, &nml_len);

		cur_dsd->base.addrLow  = cpu_to_le32(LSDW(req_dma));
		cur_dsd->base.addrHigh = cpu_to_le32(MSDW(req_dma));
		cur_dsd->count = req_len;
		tot_dsds++;
		avail_dsds--;

		DBG(QL4PRINT(QLP5,
			     printk("scsi%d:%d:%d:%d: %s: No S/G transfer, "
				    "DSD=%p cmd=%p "
				    "dma_addr=%x:%08x, len=%x, "
				    "tot_dsd=0x%x, avail_dsd=0x%x\n",
				    ha->host_no, cmd->channel,
				    cmd->target, cmd->lun,
				    __func__, cur_dsd, cmd,
				    cur_dsd->base.addrHigh,
				    cur_dsd->base.addrLow, cur_dsd->count,
				    tot_dsds, avail_dsds));)
		cur_dsd++;

		if (normalized) {
			/* Allocate continuation packet, if necessary,
			 * and point cur_dsd to 1st dsd entry */
			if (avail_dsds == 0) {
				tot_iocbs++;
				if (qla4xxx_alloc_cont_entry(ha,
							     &cur_dsd,
							     &avail_dsds)
				    == QLA_ERROR) {
					goto exit_send_cmd_return_dma;
				}
			}

			cur_dsd->base.addrLow  = cpu_to_le32(LSDW(nml_dma));
			cur_dsd->base.addrHigh = cpu_to_le32(MSDW(nml_dma));
			cur_dsd->count = nml_len;
			tot_dsds++;
			avail_dsds--;

			DBG(QL4PRINT(QLP5,
				     printk("scsi%d:%d:%d:%d: %s: No S/G transfer, "
					    "Normalized DSD=%p cmd=%p "
					    "dma_addr=%x:%08x, len=%x, "
					    "tot_dsd=0x%x, avail_dsd=0x%x\n",
					    ha->host_no, cmd->channel,
					    cmd->target, cmd->lun,
					    __func__, cur_dsd, cmd,
					    cur_dsd->base.addrHigh,
					    cur_dsd->base.addrLow, cur_dsd->count,
					    tot_dsds, avail_dsds));)
			cur_dsd++;
		}
	}


	cmd_entry->dataSegCnt        = __cpu_to_le16(tot_dsds);
	cmd_entry->hdr.entryCount    = srb->entry_count = tot_iocbs;


	/*
	 * Send command to ISP
	 *-----------------------------------------------*/
	add_to_active_array(ha, index, srb);
	srb->flags |= SRB_DMA_VALID;


	/* Track IOCB used */
	ha->iocb_cnt += tot_iocbs;
	srb->iocb_cnt = tot_iocbs;

	wmb();

	/* Debug print statements */
	#ifdef DEBUG
	QL4PRINT(QLP14, printk("scsi%d:%d:%d:%d: %s: CDB = ",
			       ha->host_no, cmd->channel,
			       cmd->target, cmd->lun, __func__));
	for (i = 0; i < cmd->cmd_len; i++)
		QL4PRINT(QLP14, printk("%02x ", cmd->cmnd[i]));
	QL4PRINT(QLP14, printk("\n"));

	QL4PRINT(QLP5, printk("scsi%d: %s: srb=%p, srb->index=0x%x, "
			      "cmd_entry->handle=0x%x "
			      "tot_dsds=%d, tot_iocbs=%d\n",
			      ha->host_no, __func__, srb,
			      srb->active_array_index,
			      cmd_entry->handle,
			      tot_dsds, tot_iocbs));

	QL4PRINT(QLP10|QLP24, printk("scsi%d: %s: cmd_entry 0x%p\n",
				     ha->host_no, __func__, cmd_entry));
	qla4xxx_dump_bytes(QLP10|QLP24, cmd_entry, sizeof(*cmd_entry));

	for (i=1; i<=tot_iocbs-1; i++) {
		CONTINUE_ENTRY *cont_entry = (CONTINUE_ENTRY *) cmd_entry+i;
		QL4PRINT(QLP10|QLP24,
			 printk("\nscsi%d: %s: cont_entry 0x%p\n",
				ha->host_no, __func__, cont_entry));
		qla4xxx_dump_bytes(QLP10|QLP24,
				   cont_entry, sizeof(*cont_entry));
	}

	QL4PRINT(QLP5, printk("scsi%d: %s: RequestQueueIn %x\n",
			      ha->host_no, __func__, ha->request_in));
	#endif

	/* Tell ISP that there's a new request */
	srb->u_start = jiffies;	/*Time we send the I/O to firmware*/
	ha->f_start = srb->u_start;
	WRT_REG_DWORD(&ha->reg->requestQueueInPointer, ha->request_in);

	spin_unlock_irqrestore(&ha->hardware_lock, flags);
	DBG(LEAVE("qla4xxx_send_command_to_isp");)
	return(QLA_SUCCESS);

	exit_send_cmd_return_dma:
	/* Unmap srb dma buffer */
#ifndef __VMKERNEL_MODULE__
	pci_unmap_sg(ha->pdev, (struct scatterlist *)cmd->request_buffer,
		     cmd->use_sg, scsi_to_pci_dma_dir(cmd->sc_data_direction));
#endif

	exit_send_cmd_return_request:
	/* restore request queue in pointers */
	ha->request_in = saved_request_in;
	ha->request_ptr = saved_request_ptr;

	exit_send_cmd:
	/* Release hardware specific lock */
	spin_unlock_irqrestore(&ha->hardware_lock, flags);

	DBG(LEAVE("qla4xxx_send_command_to_isp");)
	return(QLA_ERROR);
}

/**************************************************************************
 * qla4xxx_lookup_lun_handle
 *	This routine locates a lun handle given the device handle and lun
 *	number.
 *
 * Input:
 * 	ha - Pointer to host adapter structure.
 *	ddb_entry - Pointer to device database entry
 *	lun - SCSI LUN
 *
 * Returns:
 *	Pointer to corresponding lun_entry structure
 *
 * Context:
 *	Kernel context.
 **************************************************************************/
inline lun_entry_t *
qla4xxx_lookup_lun_handle(scsi_qla_host_t *ha,
			  ddb_entry_t *ddb_entry,
			  uint16_t lun)
{
	lun_entry_t *lun_entry = NULL;

	if (ddb_entry && lun < MAX_LUNS) {
		lun_entry = ddb_entry->lun_table[lun];
	}

	DBG(QL4PRINT(QLP3, printk("scsi%d: %s: lun %d, lun_entry = %p\n",
				  ha->host_no, __func__, lun, lun_entry));)
	return(lun_entry);
}

/**************************************************************************
 * qla4xxx_lookup_ddb_by_SCSIID
 *	This routine locates a device handle given the SCSI bus and
 *	target IDs.  If device doesn't exist, returns NULL.
 *
 * Input:
 * 	ha - Pointer to host adapter structure.
 *	bus - SCSI bus number
 *	target - SCSI target ID.
 *
 * Returns:
 *	Pointer to the corresponding internal device database structure
 *
 * Context:
 *	Kernel context.
 **************************************************************************/
inline ddb_entry_t *
qla4xxx_lookup_ddb_by_SCSIID(scsi_qla_host_t *ha,
			     uint32_t bus,
			     uint32_t target)
{
	ddb_entry_t *ddb_entry = NULL;


	if ((target < MAX_DDB_ENTRIES) &&
	    (ha->target_map[target] != (ddb_entry_t *) INVALID_ENTRY)) {
		ddb_entry = ha->target_map[target];
	}

	DBG(QL4PRINT(QLP3, printk("scsi%d: %s: b%d:t%d, ddb_entry = %p\n",
				  ha->host_no, __func__, bus, target, ddb_entry));)
	return(ddb_entry);
}

/**************************************************************************
 * qla4xxx_lookup_ddb_by_fw_index
 *	This routine locates a device handle given the firmware device
 *	database index.  If device doesn't exist, returns NULL.
 *
 * Input:
 * 	ha - Pointer to host adapter structure.
 *      fw_ddb_index - Firmware's device database index
 *
 * Returns:
 *	Pointer to the corresponding internal device database structure
 *
 * Context:
 *	Kernel context.
 **************************************************************************/
inline ddb_entry_t *
qla4xxx_lookup_ddb_by_fw_index(scsi_qla_host_t *ha, uint32_t fw_ddb_index)
{
	ddb_entry_t *ddb_entry = NULL;


	if ((fw_ddb_index < MAX_DDB_ENTRIES) &&
	    (ha->fw_ddb_index_map[fw_ddb_index] !=
	     (ddb_entry_t *) INVALID_ENTRY)) {
		ddb_entry = ha->fw_ddb_index_map[fw_ddb_index];
	}

	DBG(QL4PRINT(QLP3, printk("scsi%d: %s: index [%d], ddb_entry = %p\n",
				  ha->host_no, __func__, fw_ddb_index, ddb_entry));)
	return(ddb_entry);
}


#ifndef QLA4000
/**************************************************************************
 * qla4xxx_is_discovered_target
 *	This routine locates a device handle given iSNS information.
 *	If device doesn't exist, returns NULL.
 *
 * Input:
 * 	ha - Pointer to host adapter structure.
 *      ip_addr - Pointer to IP address
 *      alias - Pointer to iSCSI alias
 *
 * Returns:
 *	Pointer to the corresponding internal device database structure
 *
 * Context:
 *	Kernel context.
 **************************************************************************/
inline uint8_t
qla4xxx_is_discovered_target(scsi_qla_host_t *ha,
			     uint8_t *ip_addr,
			     uint8_t *alias,
			     uint8_t *name_str)
{
	ISNS_DISCOVERED_TARGET *discovered_target = NULL;
	int i,j;

	for (i=0; i < ha->isns_num_discovered_targets; i++) {
		discovered_target =
		&ha->dma_mem_blkv->isns_discovered_target_database[i];

		for (j = 0; j < discovered_target->NumPortals; j++) {
			if (memcmp(discovered_target->Portal[j].IPAddr,
				   ip_addr,
				   MIN(sizeof(discovered_target->Portal[j].IPAddr),
				       sizeof(*ip_addr)))
			    &&
			    memcmp(discovered_target->Alias, alias,
				   MIN(sizeof(discovered_target->Alias),
				       sizeof(*alias)))
			    &&
			    memcmp(discovered_target->NameString,
				   name_str,
				   MIN(sizeof(discovered_target->Alias),
				       sizeof(*name_str)))
			   ) {
				return(QLA_SUCCESS);
			}
		}
	}

	return(QLA_ERROR);
}
#endif

/**************************************************************************
 * qla4xxx_complete_request
 *	This routine returns a command to the caller via the done_fn
 *	specified in the cmd structure.
 *
 * Input:
 * 	ha - Pointer to host adapter structure.
 *	srb - Pointer to SCSI Request Block
 *
 * Remarks:
 *    The srb pointer should be guaranteed to be nonzero before calling
 *    this function.  The caller should also ensure that the list_lock is
 *    released before calling this function.
 *
 * Returns:
 *	QLA_SUCCESS - Successfully completed request
 *	QLA_ERROR   - Failed to complete request
 *
 * Context:
 *	Kernel/Interrupt context.
 **************************************************************************/
uint8_t
qla4xxx_complete_request(scsi_qla_host_t *ha, srb_t *srb)
{
	uint8_t status = QLA_ERROR;
	Scsi_Cmnd *cmd;
	unsigned long flags;

	//ENTER("qla4xxx_complete_request");

	/* Make sure the cmd pointer is valid */
	if (srb == NULL) {
		QL4PRINT(QLP2, printk("scsi%d: %s: ERROR: NULL srb \n",
				      ha->host_no, __func__));
		goto exit_complete_request;
	}

	if ((srb->flags & SRB_FREE_STATE) == 0)
		qla4xxx_delete_timer_from_cmd(srb);

	cmd = srb->cmd;
	if (cmd == NULL) {
		QL4PRINT(QLP2,
			 printk("scsi%d: %s: ERROR: NULL cmd pointer in "
				"srb=%p\n", ha->host_no, __func__, srb));

		goto exit_complete_request;
	}

	/* Release memory used for this I/O */
	if ((srb->flags & SRB_DMA_VALID) != 0) {
		srb->flags &= ~SRB_DMA_VALID;

#ifndef __VMKERNEL_MODULE__
		/* Release memory used for this I/O */
		if (cmd->use_sg) {
			DBG(QL4PRINT(QLP5,
				     printk("scsi%d: %s: S/G unmap_sg cmd=%p\n",
					    ha->host_no, __func__, cmd));)

			pci_unmap_sg(ha->pdev,
				     cmd->request_buffer,
				     cmd->use_sg,
				     scsi_to_pci_dma_dir(
							cmd->sc_data_direction));
		}
		else if (cmd->request_bufflen) {
			DBG(QL4PRINT(QLP5,
				     printk("scsi%d: %s: No S/G unmap_single "
					    "cmd=%p saved_dma_handle=%x\n",
					    ha->host_no, __func__, cmd,
					    (uint32_t) srb->saved_dma_handle));)

			pci_unmap_single(ha->pdev,srb->saved_dma_handle,
					 cmd->request_bufflen,
					 scsi_to_pci_dma_dir(
							    cmd->sc_data_direction));
		}

#endif
		ha->total_mbytes_xferred += cmd->request_bufflen / 1024;
	}

	if (host_byte(CMD_RESULT(cmd)) == DID_OK) {
		if (!(srb->flags & SRB_GOT_SENSE)) {
			ddb_entry_t *ddb_entry =
			qla4xxx_lookup_ddb_by_fw_index(ha, srb->fw_ddb_index);

			lun_entry_t *lun_entry =
			qla4xxx_lookup_lun_handle(ha, ddb_entry, srb->lun);

			if (lun_entry) {
				/*
				 * If lun was not ready (suspended or timeout)
				 * then change state to "READY".
				 */
				spin_lock_irqsave(&lun_entry->lun_lock, flags);
				if (lun_entry->lun_state != LS_LUN_READY) {
					del_from_suspended_lun_q(ha, lun_entry);
					lun_entry->lun_state = LS_LUN_READY;
				}
				spin_unlock_irqrestore(&lun_entry->lun_lock, flags);
			}
		}
	}

#ifdef DEBUG
	/* debug prints */

	if (host_byte(CMD_RESULT(cmd)) == DID_OK) {
		switch (cmd->cmnd[0]) {
		case TEST_UNIT_READY:
			QL4PRINT(QLP13,
				 printk("scsi%d:%d:%d:%d: %s: "
					"TEST_UNIT_READY "
					"status = 0x%x\n",
					ha->host_no, cmd->channel,
					cmd->target, cmd->lun, __func__,
					cmd->result & 0xff));

			if (driver_byte(cmd->result) & DRIVER_SENSE) {
				QL4PRINT(QLP13,
					 printk("REQUEST_SENSE data:  "
						"(MAX 0x20 bytes displayed)\n"));

				qla4xxx_dump_bytes(QLP13, cmd->sense_buffer,
						   MIN(0x20, CMD_SNSLEN(cmd)));
			}
			break;
		case INQUIRY:
			QL4PRINT(QLP13, printk("scsi%d:%d:%d:%d: %s: "
					       "INQUIRY data: "
					       "(MAX 0x30 bytes displayed)\n",
					       ha->host_no,
					       cmd->channel, cmd->target,
					       cmd->lun, __func__));

			qla4xxx_dump_bytes(QLP13, cmd->request_buffer,
					   MIN(0x30, cmd->request_bufflen));

			if (strncmp(cmd->request_buffer,
				    "\7f\00\00\00\7f\00\00\00", 8) == 0) {
				QL4PRINT(QLP2,
					 printk("scsi%d:%d:%d:%d: %s: "
						"Device not present.  "
						"Possible connection "
						"problem with iSCSI router\n",
						ha->host_no,
						cmd->channel, cmd->target,
						cmd->lun, __func__));
			}
			break;
		case REQUEST_SENSE:
			QL4PRINT(QLP13,
				 printk("scsi%d:%d:%d:%d: %s: REQUEST_SENSE "
					"data:  (MAX 0x20 bytes displayed)\n",
					ha->host_no, cmd->channel,
					cmd->target, cmd->lun, __func__));

			qla4xxx_dump_bytes(QLP13, cmd->request_buffer,
					   MIN(0x20, cmd->request_bufflen));
			break;
		case REPORT_LUNS:
			QL4PRINT(QLP13,
				 printk("scsi%d:%d:%d:%d: %s: "
					"REPORT_LUNS data: "
					"(MAX 0x40 bytes displayed)\n",
					ha->host_no, cmd->channel,
					cmd->target, cmd->lun, __func__));

			qla4xxx_dump_bytes(QLP13, cmd->request_buffer,
					   MIN(0x40, cmd->request_bufflen));
			break;
		}

	}
#endif

	/*
	 * WORKAROUND
	 * A backdoor device-reset (via eh_resets) requires different
	 * error handling.  This code differentiates between normal
	 * error handling and the backdoor method
	 */
	if (host_byte(CMD_RESULT(cmd)) == DID_RESET) {
		#define EH_ACTIVE 1
		if (ha->host->eh_active != EH_ACTIVE)
			CMD_RESULT(srb->cmd) = DID_BUS_BUSY << 16;
	}

#ifdef DEBUG
	if (CMD_RESULT(cmd) & 0xff) {
		QL4PRINT(QLP13,
			 printk("REQUEST_SENSE data:  "
				"(MAX 0x20 bytes displayed)\n"));

		qla4xxx_dump_bytes(QLP13, cmd->sense_buffer,
				   MIN(0x20, CMD_SNSLEN(cmd)));
	}

	if (host_byte(CMD_RESULT(cmd)) == DID_RESET ||
	    host_byte(CMD_RESULT(cmd)) == DID_BUS_BUSY ||
	    host_byte(CMD_RESULT(cmd)) == DID_ABORT ||
	    host_byte(CMD_RESULT(cmd)) == DID_ERROR) {
		QL4PRINT(QLP2,
			 printk("scsi%d:%d:%d:%d: %s: cmd=%p sp=%p "
				"didstat=%X comp=%x scsistat=(%x), "
				"host_failed=%d, to=%ds\n",
				ha->host_no, cmd->channel, cmd->target,
				cmd->lun, __func__, cmd, srb,
				host_byte(CMD_RESULT(cmd)), srb->cc_stat,
				CMD_SCSI_STATUS(cmd) & 0xff,
				cmd->host->host_failed, CMD_TIMEOUT(cmd)/HZ));
	}
#endif

	/* Call the mid-level driver interrupt handler */
	srb->cmd = NULL;
	add_to_free_srb_q(ha, srb);

	DBG(QL4PRINT(QLP5, printk("scsi%d: %s: request completed w/ result "
				  "hoststat=%X scsistat=(%x)\n",
				  ha->host_no, __func__,
				  host_byte(CMD_RESULT(cmd)),
				  CMD_SCSI_STATUS(cmd) & 0xff));)

	HOST_SPIN_LOCK_IRQSAVE(ha, flags);

	CMD_SP(cmd) = NULL;
	(*(cmd)->scsi_done)(cmd);

	HOST_SPIN_UNLOCK_IRQRESTORE(ha, flags);

	exit_complete_request:
	//LEAVE("qla4xxx_complete_request");

	return(status);
}

/**************************************************************************
 * qla4xxx_queuecommand
 *	This routine is invoked by Linux to send a SCSI command to the driver.
 *
 * Input:
 *	cmd - Pointer to Linux's SCSI command structure
 *	done_fn - Function that the driver calls to notify the SCSI mid-layer
 *		  that the command has been processed.
 *
 * Remarks:
 *    The mid-level driver tries to ensure that queuecommand never gets
 *    invoked concurrently with itself or the interrupt handler (although
 *    the interrupt handler may call this routine as part of request-
 *    completion handling).   Unfortunely, it sometimes calls the scheduler
 *    in interrupt context which is a big NO! NO!.
 *
 * Returns:
 *	None
 *
 * Context:
 *	Kernel context.
 **************************************************************************/
int
qla4xxx_queuecommand(Scsi_Cmnd *cmd, void (*done_fn)(Scsi_Cmnd *))
{
	scsi_qla_host_t *ha;
	ddb_entry_t     *ddb_entry;
	lun_entry_t     *lun_entry;
	uint32_t        b, t, l;
	int             return_status = 0;
	srb_t           *srb;

	b = cmd->channel;
	t = cmd->target;
	l = cmd->lun;
	ha = (scsi_qla_host_t *) cmd->host->hostdata;

	HOST_SPIN_UNLOCK(ha);

	/*
	 * Retrieve srb from pool.  If no srb available,
	 * Notify the OS to queue commands in the OS.
	 * The OS will not attempt to queue more commands
	 * until a command is returned to the OS.
	 */
	srb = del_from_free_srb_q_head(ha);
	if (srb == NULL) {
		QL4PRINT(QLP2,
			 printk("scsi%d: %s: srb not available, retry later\n",
				ha->host_no, __func__));

		return_status = 1;

		HOST_SPIN_LOCK_IRQ(ha);

		return(return_status);
	}

	/* Link the srb with cmd */
	CMD_SP(cmd) = (char *)srb;
	cmd->scsi_done = done_fn;
	srb->cmd = cmd;
	srb->r_start = jiffies;	/*Time we recieved the I/O*/
	srb->flags |= SRB_KERNEL_CMD;

	if ((CMD_TIMEOUT(cmd)/HZ) > QLA_CMD_TIMER_DELTA) {
		qla4xxx_add_timer_to_cmd(srb,
					 (CMD_TIMEOUT(cmd)/HZ) -
					 QLA_CMD_TIMER_DELTA);
	}
	else
		qla4xxx_add_timer_to_cmd(srb, (CMD_TIMEOUT(cmd)/HZ));

	#ifdef QLA4000

	/* Don't extend the timeouts on QLA4010. */

	/* Extend the command timeout value because the iscsi enviornment
	 * command needs more time to complete.  Otherwise, commands start
	 * timing out frequently causing device resets.
	 */
	qla4xxx_extend_timeout(srb->cmd, (ql4xcmdtimeout-1)*EXTEND_CMD_TOV);
	#endif

	/* retrieve device and lun handles */
	ddb_entry = qla4xxx_lookup_ddb_by_SCSIID(ha, b, t);
	if (ddb_entry == 0) {
		//QL4PRINT(QLP2,
		//	 printk("scsi%d: %s: invalid target b%d, t%d, l%d\n",
		//		ha->host_no, __func__, b, t, l));

		CMD_RESULT(cmd) = (int) (DID_NO_CONNECT << 16);
		goto exit_qc_call_done;
	}

	lun_entry = qla4xxx_lookup_lun_handle(ha, ddb_entry, l);

	if (lun_entry == 0) {
		//QL4PRINT(QLP2,
		//printk("scsi%d: %s: invalid lun b%d, t%d, l%d\n",
		//		      ha->host_no, __func__, b, t, l));

		CMD_RESULT(cmd) = (int) (DID_NO_CONNECT << 16);
		goto exit_qc_call_done;
	}

	/* Prepare the srb to be sent to the ISP */
	srb->fw_ddb_index = ddb_entry->fw_ddb_index;
	srb->lun = l;

	if (cmd->allowed < ql4xcmdretrycount)
		cmd->allowed = ql4xcmdretrycount;

	#ifdef DEBUG

	/* Print debug information */
	QL4PRINT(QLP5, printk("scsi%d: %s: b%d, t%d ,l%d, ",
			      ha->host_no, __func__, b, t, l));
	QL4PRINT(QLP5, printk("cdb = "));
	QL4PRINT(QLP5, printk("%02X %02X %02X %02X %02X %02X %02X %02X "
			      "%02X %02X\n",
			      cmd->cmnd[0], cmd->cmnd[1], cmd->cmnd[2],
			      cmd->cmnd[3], cmd->cmnd[4], cmd->cmnd[5],
			      cmd->cmnd[6], cmd->cmnd[7], cmd->cmnd[8],
			      cmd->cmnd[9]));

	QL4PRINT(QLP5, printk("scsi%d: %s: device state = 0x%x\n",
			      ha->host_no, __func__,
			      atomic_read(&ddb_entry->state)));
	#endif

	add_to_pending_srb_q(ha, srb);
	qla4xxx_start_io(ha);

	HOST_SPIN_LOCK_IRQ(ha);
	return(return_status);

	exit_qc_call_done:
	qla4xxx_complete_request(ha, srb);

	HOST_SPIN_LOCK_IRQ(ha);
	return(return_status);
}

/**************************************************************************
 * qla4xxx_extend_timeout
 *	This routine will extend the timeout to the specified value.
 *
 * Input:
 *	cmd - Pointer to Linux's SCSI command structure
 *	timeout - Amount of time to extend the OS timeout
 *
 * Returns:
 *	None
 *
 * Context:
 *	Kernel/Interrupt context.
 **************************************************************************/
static void
qla4xxx_extend_timeout(Scsi_Cmnd *cmd, int timeout)
{
	srb_t *srb = (srb_t *) CMD_SP(cmd);
	u_long our_jiffies = (timeout * HZ) + jiffies;

	if (cmd->eh_timeout.function) {
		mod_timer(&cmd->eh_timeout, our_jiffies);
	}
	if (srb->timer.function != NULL) {
		/*
		 * Our internal timer should timeout before the midlayer has a
		 * chance begin the abort process
		 */
		mod_timer(&srb->timer,
			  our_jiffies - (QLA_CMD_TIMER_DELTA * HZ));
	}
}

/**************************************************************************
 * qla4xxx_start_io
 *	This routine retrieves and processes next request from the pending
 *	queue.
 *
 * Input:
 * 	ha - Pointer to host adapter structure.
 *
 * Returns:
 *	None
 *
 * Context:
 *	Kernel/Interrupt context.
 **************************************************************************/
void
qla4xxx_start_io(scsi_qla_host_t *ha)
{
	ddb_entry_t     *ddb_entry;
	lun_entry_t     *lun_entry;
	srb_t           *srb;
	unsigned long   flags;

	DBG(ENTER("qla4xxx_start_io");)
	spin_lock_irqsave(&ha->list_lock, flags);
	while (!list_empty(&ha->pending_srb_q)) {
		srb = list_entry(ha->pending_srb_q.next, srb_t, list_entry);

		ddb_entry =
		qla4xxx_lookup_ddb_by_fw_index(ha, srb->fw_ddb_index);
		lun_entry =
		qla4xxx_lookup_lun_handle(ha, ddb_entry, srb->lun);

		__del_from_pending_srb_q(ha, srb);

		/* If device is dead or has gone away,
		 * send request back to OS */
		if ((ddb_entry == NULL) || (lun_entry == NULL) ||
		    (atomic_read(&ddb_entry->state) == DEV_STATE_DEAD)) {
			CMD_RESULT(srb->cmd) = DID_NO_CONNECT << 16;
#ifdef __VMKERNEL_MODULE__
                        /*
                         * If the device has gone away, the vmkernel
                         * would only be sending command here to
                         * bring this pat back online.  Attempt a
                         * relogin to the target.
                         */
                        if (ddb_entry &&
                            (atomic_read(&ddb_entry->retry_relogin_timer) ==
                            INVALID_ENTRY)) {
                                qla4xxx_reset_relogin_timer(ha, ddb_entry);
                        }
#endif
			add_to_done_srb_q(ha, srb);
			continue;
		}

		/*
		 * If the device is missing or the adapter is OFFLINE,
		 * put the request on the retry queue.
		 */
		if (atomic_read(&ddb_entry->state) == DEV_STATE_MISSING ||
		    !ADAPTER_UP(ha)) {

			qla4xxx_extend_timeout(srb->cmd, EXTEND_CMD_TOV);
			__add_to_retry_srb_q(ha, srb);
			continue;
		}


		/*
		 * If this request's lun is suspended then put the request on
		 * the  scsi_retry queue.
		 */
		if (lun_entry->lun_state == LS_LUN_SUSPENDED) {
			__add_to_retry_srb_q(ha, srb);
			continue;
		}


		/* Release target queue lock */
		spin_unlock_irqrestore(&ha->list_lock, flags);
		if (qla4xxx_send_command_to_isp(ha, srb) != QLA_SUCCESS) {
			/* Unable to send command to the ISP at this time.
			 * Notify the OS to queue commands in the OS.
			 * The OS will not attempt to queue more commands
			 * until a command is returned to the OS.
			 */
			QL4PRINT(QLP2,
				 printk("scsi%d: %s: unable to send cmd to "
					"ISP, retry later\n",
					ha->host_no, __func__));

			spin_lock_irqsave(&ha->list_lock, flags);
			__add_to_pending_srb_q_head(ha, srb);
			break;
		}
		spin_lock_irqsave(&ha->list_lock, flags);

	}
	spin_unlock_irqrestore(&ha->list_lock, flags);

	/* any commands in done queue */
	if (!list_empty(&ha->done_srb_q)) {
		while ((srb = del_from_done_srb_q_head(ha)) != NULL)
			qla4xxx_complete_request(ha, srb);
	}
	DBG(LEAVE("qla4xxx_start_io");)
}

/**************************************************************************
 * qla4xxx_os_cmd_timeout
 *	This routine handles the command if it times out in any state.
 *
 * Input:
 *	srb - Pointer to SCSI Request Block
 *
 * Returns:
 *	None
 **************************************************************************/
void
qla4xxx_os_cmd_timeout(srb_t *srb)
{
	Scsi_Cmnd *cmd;
	scsi_qla_host_t *ha;
	ddb_entry_t *ddb_entry;
	lun_entry_t *lun_entry;
	unsigned long flags , cpu_flags;

	ENTER("qla4xxx_os_cmd_timeout");

	if (srb == NULL) {
		QL4PRINT(QLP2, printk("%s: NULL srb\n", __func__));
		LEAVE("qla4xxx_os_cmd_timeout");
		return;
	}

	qla4xxx_print_srb_info(QLP7, srb);
	QL4PRINT(QLP7, printk("scsi %s: SCSI cmd originated from ",
			      __func__));
	if ((srb->flags & SRB_INTERNAL_CMD) != 0) {
		QL4PRINT(QLP7, printk("INTERNAL\n"));
	}
	else if ((srb->flags & SRB_IOCTL_CMD) != 0) {
		QL4PRINT(QLP7, printk("IOCTL\n"));
	}
	else if ((srb->flags & SRB_KERNEL_CMD) != 0) {
		QL4PRINT(QLP7, printk("KERNEL\n"));
	}
	else {
		QL4PRINT(QLP7, printk("UNKNOWN\n"));
	}

	if (srb->active_array_index >= MAX_SRBS) {
		QL4PRINT(QLP2, printk("%s: invalid active_array_index 0x%x\n",
				      __func__, srb->active_array_index));
		LEAVE("qla4xxx_os_cmd_timeout");
		return;
	}

	cmd = srb->cmd;
	if (cmd == NULL) {
		QL4PRINT(QLP2, printk("%s: NULL cmd in srb %p\n",
				      __func__, srb));
		LEAVE("qla4xxx_os_cmd_timeout");
		return;
	}

	qla4xxx_print_scsi_cmd(QLP7, cmd);

	ha = (scsi_qla_host_t *) cmd->host->hostdata;

	ddb_entry = qla4xxx_lookup_ddb_by_fw_index(ha, srb->fw_ddb_index);
	if (ddb_entry == NULL) {
		QL4PRINT(QLP2,
			 printk("scsi%d: %s: invalid index [%d]\n",
				ha->host_no, __func__, srb->fw_ddb_index));
		LEAVE("qla4xxx_os_cmd_timeout");
		return;
	}

	lun_entry = qla4xxx_lookup_lun_handle(ha, ddb_entry, srb->lun);
	if (lun_entry == NULL) {
		QL4PRINT(QLP2,
			 printk("scsi%d:%d:%d: %s: lun %d not present\n",
				ha->host_no, ddb_entry->bus,
				ddb_entry->target, __func__, srb->lun));
		LEAVE("qla4xxx_os_cmd_timeout");
		return;
	}

	spin_lock_irqsave(&ha->list_lock, flags);

	QL4PRINT(QLP2,
		 printk("scsi%d:%d:%d:%d: %s: srb=%p osTOV=%d, iocbTOV=%d, "
			"SS=%d, DS=%d, LS=%d\n",
			ha->host_no, ddb_entry->bus, ddb_entry->target,
			srb->lun, __func__, srb, srb->os_tov, srb->iocb_tov,
			srb->state, atomic_read(&ddb_entry->state),
			lun_entry->lun_state));

	/*
	 * If IO is found in retry Queue
	 * Return this IO back to host
	 */

	switch (srb->state) {
	case SRB_PENDING_STATE:
		__del_from_pending_srb_q(ha, srb);
		break;

	case SRB_ACTIVE_STATE:
		ha->io_timeout_count++;

		spin_unlock_irqrestore(&ha->list_lock, flags);
		spin_lock_irqsave(&ha->hardware_lock, cpu_flags);

		/* The command could have completed between the time
		 * we checked the state above and now, so check to see if
		 * the command already completed.  To do this, check to see if
		 * the cmd pointer in the srb is NULL.
		 */
		if (srb->cmd) {
			if (srb == ha->active_srb_array
			    [(unsigned long)CMD_HANDLE(srb->cmd)]) {
				if (srb->flags & SRB_IOCTL_CMD) {
					ha->ioctl_err_cmd = srb->cmd;
					set_bit(DPC_IOCTL_ERROR_RECOVERY,
						&ha->dpc_flags);
					if (ha->dpc_wait && !ha->dpc_active)
						up(ha->dpc_wait);
				}

				srb->state = SRB_ACTIVE_TIMEOUT_STATE;
			}
			else {
				QL4PRINT(QLP2,
					 printk(KERN_INFO "scsi%d: %s: "
						"State indicates srb %p "
						"cmd %p is with ISP, But not "
						"in active array\n",
						ha->host_no, __func__,
						srb, cmd));
			}
		}

		spin_unlock_irqrestore(&ha->hardware_lock, cpu_flags);
		LEAVE("qla4xxx_os_cmd_timeout");
		return;

	case SRB_RETRY_STATE:
		__del_from_retry_srb_q(ha, srb);
		break;


	case SRB_DONE_STATE:
		/* Do nothing, it is on its way back to the OS */
		spin_unlock_irqrestore(&ha->list_lock, flags);
		LEAVE("qla4xxx_os_cmd_timeout");
		return;

	default:
		/* EMPTY */
		QL4PRINT(QLP2,
			 printk("scsi%d: %s: LOST command state = 0x%x, "
				"srb=%p\n",
				ha->host_no, __func__, srb->state, srb));
		break;
	}

	/*
	 * If state is marked as dead return the cmd with
	 * DID_NO_CONNECT status.  Otherwise set the host_byte to
	 * DID_BUS_BUSY to let the OS retry this cmd.
	 */
	if (atomic_read(&ddb_entry->state) == DEV_STATE_DEAD)
		cmd->result = DID_NO_CONNECT << 16;
	else
		cmd->result = DID_BUS_BUSY << 16;

	spin_unlock_irqrestore(&ha->list_lock, flags);
	qla4xxx_complete_request(ha, srb);
	LEAVE("qla4xxx_os_cmd_timeout");
}

/**************************************************************************
 * qla4xxx_add_timer_to_cmd
 *	This routine creates a timer for the specified command. The timeout
 *	is usually the command time from kernel minus 2 secs.
 *
 * Input:
 *	srb - Pointer to SCSI Request Block
 *	timeout - Number of seconds to extend command timeout.
 *
 * Returns:
 *	None
 *
 * Context:
 *	Kernel context.
 **************************************************************************/
void
qla4xxx_add_timer_to_cmd(srb_t *srb, int timeout)
{
	init_timer(&srb->timer);
	srb->timer.expires = jiffies + timeout * HZ;
	srb->timer.data = (unsigned long) srb;
	srb->timer.function = (void (*) (unsigned long))qla4xxx_os_cmd_timeout;

#ifndef __VMKERNEL_MODULE__
	add_timer(&srb->timer);
#else
        if (timeout) {
           add_timer(&srb->timer);
        }
        else {
           srb->timer.function = NULL;
        }
#endif
	QL4PRINT(QLP12, printk("%s: srb %p, timeout %d\n",
			       __func__, srb, timeout));
}

/**************************************************************************
 * qla4xxx_delete_timer_from_cmd
 *	This routine deletes the timer for the specified command.
 *
 * Input:
 *	srb - Pointer to SCSI Request Block
 *
 * Returns:
 *	None
 *
 * Context:
 *	Kernel/Interrupt context.
 **************************************************************************/
void
qla4xxx_delete_timer_from_cmd(srb_t *srb )
{
	if (srb->timer.function != NULL) {
		del_timer(&srb->timer);
		srb->timer.function =  NULL;
		srb->timer.data = (unsigned long) NULL;
		QL4PRINT(QLP12, printk("%s: srb %p\n", __func__, srb));
	}
}


/****************************************************************************/
/*                        Interrupt Service Routine.                        */
/****************************************************************************/

/**************************************************************************
 * qla4xxx_suspend_lun
 *	This routine suspends the lun queue for the specified lun and places
 *	all requests for this lun onto the retry queue for a specified
 *	amount of time.
 *
 * Input:
 * 	ha - Pointer to host adapter structure.
 *	srb - Pointer to SCSI Request Block
 *	time - Number of seconds to suspend queue
 *	retries - Max retry count for this lun
 *
 * Remarks:
 *	The suspend queue algorithm is provided as a method to keep commands
 *	within the driver while a device is attempting to recover from certain
 *	failures.  By keeping the commands within the driver, it prevents the
 *	kernel's retries from being exhausted so quickly and minimizes failures
 *	at the application level.
 *
 * Returns:
 *	None
 *
 * Context:
 *	Interrupt context.
 **************************************************************************/
STATIC void
qla4xxx_suspend_lun(scsi_qla_host_t *ha,
		    srb_t *srb,
		    lun_entry_t *lun_entry,
		    uint16_t time,
		    uint16_t retries)
{
	unsigned long flags;
	uint8_t status = 0 ;
	struct list_head *list, *temp; /* used for traversing lists */
	srb_t *sp;

	ENTER("qla4xxx_suspend_lun");

	if (lun_entry == NULL)
		return;

	spin_lock_irqsave(&lun_entry->lun_lock, flags);

	if (lun_entry->lun_state == LS_LUN_READY ||
	    lun_entry->lun_state == LS_LUN_RETRY) {
		if (lun_entry->lun_state == LS_LUN_READY) {
			lun_entry->max_retry_count = retries;
			lun_entry->retry_count = 0;
		}

		/* put lun on suspend queue */
		add_to_suspended_lun_q(ha, lun_entry);

		/* Set the suspend time */
		atomic_set(&lun_entry->suspend_timer, time);
		QL4PRINT(QLP3,
			 printk("scsi%d: %s lun %d retry count = %d\n",
				ha->host_no, __func__, lun_entry->lun,
				lun_entry->retry_count));

		/* now suspend the lun */
		lun_entry->lun_state = LS_LUN_SUSPENDED;
		status = 1;

	}
	spin_unlock_irqrestore(&lun_entry->lun_lock, flags);

	/*
	 * Remove all pending commands from request queue
	 * and put them in the retry queue.
	 */
	if (status) {
		spin_lock_irqsave(&ha->list_lock, flags);
		list_for_each_safe(list, temp, &ha->pending_srb_q)
		{
			ddb_entry_t *ddb;
			lun_entry_t *lun;

			sp = list_entry(list, srb_t, list_entry);
			ddb =
			qla4xxx_lookup_ddb_by_fw_index(ha, sp->fw_ddb_index);
			lun = qla4xxx_lookup_lun_handle(ha, ddb, sp->lun);

			if (lun != lun_entry)
				continue;

			__del_from_pending_srb_q(ha, sp);

			if (retries > sp->cmd->allowed)
				sp->cmd->allowed = retries;
			__add_to_retry_srb_q(ha,sp);

		} /* list_for_each_safe */
		spin_unlock_irqrestore(&ha->list_lock, flags);
	}

	add_to_retry_srb_q(ha,srb);

	LEAVE("qla4xxx_suspend_lun");
}

/**************************************************************************
 * qla4xxx_check_sense
 *	This routine processes Status IOCBs
 *
 * Input:
 * 	ha - Pointer to host adapter structure.
 *      sts_entry - Pointer to status entry structure
 *	srb - Pointer to internal SCSI request block structure.
 *
 * Returns:
 *	QLA_SUCCESS - We want the caller to complete the command
 *	QLA_ERROR - We do not want the caller to complete the request
 *
 * Context:
 *	Interrupt context.
 **************************************************************************/
STATIC uint8_t
qla4xxx_check_sense(scsi_qla_host_t *ha, STATUS_ENTRY *sts_entry, srb_t *srb)
{
	Scsi_Cmnd *cmd = srb->cmd;
	uint8_t   lscsi_status = sts_entry->scsiStatus;

	/* Check for errors */
	if (lscsi_status != 0) {
		QL4PRINT(QLP2,
			 printk("scsi%d:%d:%d:%d: %s: host status (%s), "
				"compl=%02x, scsi=%02x, state=%02x, "
				"iFlags=%02x, iResp=%02x\n",
				ha->host_no, cmd->channel,
				cmd->target, cmd->lun, __func__,
				host_sts_msg[host_byte(CMD_RESULT(cmd))],
				sts_entry->completionStatus,
				lscsi_status,
				sts_entry->state_flags,
				sts_entry->iscsiFlags,
				sts_entry->iscsiResponse));

		if (lscsi_status == SCSISTAT_BUSY) {
			CMD_RESULT(cmd) = DID_BUS_BUSY << 16 | lscsi_status;
		}
		else if (lscsi_status == SCSISTAT_CHECK_CONDITION) {
			CMD_RESULT(cmd) |= lscsi_status;

			/* Save Sense info, if available */
			if (__le16_to_cpu(sts_entry->senseDataByteCnt) &&
			    sts_entry->senseData[0] & 0x70 &&
			    sts_entry->senseData[2] & 0x0f) {
				ddb_entry_t *ddb_entry =
				qla4xxx_lookup_ddb_by_fw_index(ha, srb->fw_ddb_index);
				lun_entry_t *lun_entry =
				qla4xxx_lookup_lun_handle(ha, ddb_entry,
							  srb->lun);

				QL4PRINT(QLP2,
					 printk("scsi%d:%d:%d:%d: %s: "
						"sense key = "
						"%x, ASC/ASCQ = %02x/%02x\n",
						ha->host_no, cmd->channel,
						cmd->target, cmd->lun, __func__,
						sts_entry->senseData[2] & 0x0f,
						sts_entry->senseData[12],
						sts_entry->senseData[13]));

				CMD_ACTUAL_SNSLEN(cmd) =
				__le16_to_cpu(sts_entry->senseDataByteCnt);

				memcpy(CMD_SNSP(cmd),
				       sts_entry->senseData,
				       MIN(CMD_ACTUAL_SNSLEN(cmd),
					   CMD_SNSLEN(cmd)));

				qla4xxx_dump_bytes(QLP10, CMD_SNSP(cmd),
						   MIN(CMD_ACTUAL_SNSLEN(cmd),
						       CMD_SNSLEN(cmd)));

				srb->flags |= SRB_GOT_SENSE;

				if (SENSE_KEY(sts_entry->senseData) ==
				    RECOVERED_ERROR) {
					CMD_RESULT(cmd) = DID_OK << 16;
					cmd->sense_buffer[0] = 0;
				}
				/*
				 * Suspend the lun (except during initialization)
				 */
				else if (test_bit(AF_INIT_DONE, &ha->flags) &&
					 lun_entry != NULL &&
					 lun_entry->lun_state != LS_LUN_TIMEOUT) {
					/*
					 * if target is "in process of being
					 * ready then suspend lun for 6 secs and
					 * retry all the commands.
					 */
					if ((SENSE_KEY(sts_entry->senseData) == NOT_READY) &&
					    (ASC(sts_entry->senseData) == 4) &&
					    (ASCQ(sts_entry->senseData) == 1)) {
						/* To give the lun more time to become ready,
						 * suspend lun then retry command */
						qla4xxx_suspend_lun(ha, srb, lun_entry,
								    SUSPEND_SECONDS,
								    SUSPEND_RETRIES);
						return(QLA_ERROR);
					}
					else if ((SENSE_KEY(sts_entry->senseData) == HARDWARE_ERROR) &&
						 (ASC(sts_entry->senseData) == 8) &&
						 (ASCQ(sts_entry->senseData) == 0)) {
						/* To give the lun more time to become ready,
						 * suspend lun then retry command */
						qla4xxx_suspend_lun(ha, srb, lun_entry,
								    SUSPEND_SECONDS,
								    (ha->port_down_retry_count /
								     SUSPEND_SECONDS)) ;
						return(QLA_ERROR);
					}
				}
			} /* sense info */
		} /* check condition */
		else {
			CMD_RESULT(cmd) |= lscsi_status;
		}
	} /* error occurred */
	else {
		//QL4PRINT(QLP2,
		//	 printk("scsi%d:%d:%d:%d: %s: host status (%s), "
		//		"compl=%02x, scsi=%02x, state=%02x, "
		//		"iFlags=%02x, iResp=%02x\n",
		//		ha->host_no, cmd->channel,
		//		cmd->target, cmd->lun, __func__,
		//		host_sts_msg[host_byte(CMD_RESULT(cmd))],
		//		sts_entry->completionStatus,
		//		lscsi_status,
		//		sts_entry->state_flags,
		//		sts_entry->iscsiFlags,
		//		sts_entry->iscsiResponse));
	}

	return(QLA_SUCCESS);
}

/**************************************************************************
 * qla4xxx_status_entry
 *	This routine processes Status IOCBs
 *
 * Input:
 * 	ha - Pointer to host adapter structure.
 *      sts_entry - Pointer to status entry structure.
 *
 * Returns:
 *	None
 *
 * Context:
 *	Interrupt context.
 **************************************************************************/
STATIC void
qla4xxx_status_entry(scsi_qla_host_t *ha, STATUS_ENTRY *sts_entry)
{
	srb_t *srb;
	uint8_t  *strp;

	DBG(ENTER("qla4xxx_status_entry");)

	srb = del_from_active_array(ha, __le32_to_cpu(sts_entry->handle));
	if (srb) {
		Scsi_Cmnd *cmd = srb->cmd;
		uint32_t residual = __le32_to_cpu(sts_entry->residualByteCnt);
		ddb_entry_t *ddb_entry =
		qla4xxx_lookup_ddb_by_fw_index(ha, srb->fw_ddb_index);

		if (CMD_SP(cmd) == NULL)
			return;
		/*
		 * Translate ISP error to a Linux SCSI error
		 */
		switch (sts_entry->completionStatus) {
		case SCS_COMPLETE:
			if (sts_entry->scsiStatus == SCSISTAT_BUSY) {
				CMD_RESULT(cmd)= DID_BUS_BUSY << 16;
			}
			else {
				CMD_RESULT(cmd) = DID_OK << 16;

#ifndef __VMKERNEL_MODULE__
				/*
				 * Special case considertaion On an Inquiry
				 * command (0x12) for Lun 0, device responds
				 * with no devices (0x7F), then Linux will not
				 * scan further Luns. While reporting that some
				 * device exists on Lun 0 Linux will scan all
				 * devices on this target.
				 */
				if (cmd->cmnd[0] == 0x12 && cmd->lun == 0) {
					strp = (uint8_t *)cmd->request_buffer;
					if (*strp == 0x7f) {
						/* Make lun unassigned and
						 * processor type */
						*strp = 0x23;
					}
				}
#endif // __VMKERNEL_MODULE__
			}

			/* NOTE: check conditions are handled below */
			break;

		case SCS_INCOMPLETE:
			QL4PRINT(QLP2,
				 printk("%s: %d:%d:%d:%d: Incomplete, "
					"state=0x%02x\n", __func__,
					ha->host_no, cmd->channel,
					cmd->target, cmd->lun,
					sts_entry->state_flags));

			if (!(sts_entry->state_flags & STATE_FLAG_SENT_COMMAND))
				CMD_RESULT(cmd) = DID_ERROR << 16;
			else if (!(sts_entry->state_flags &
				   STATE_FLAG_TRANSFERRED_DATA))
				CMD_RESULT(cmd) = DID_ERROR << 16;
			else if (!(sts_entry->state_flags &
				   STATE_FLAG_GOT_STATUS))
				CMD_RESULT(cmd) = DID_ERROR << 16;
			break;

		case SCS_RESET_OCCURRED:
			QL4PRINT(QLP2|QLP3, printk("scsi%d:%d:%d:%d: %s: "
						   "Device RESET occurred\n",
						   ha->host_no, cmd->channel,
						   cmd->target, cmd->lun,
						   __func__));

			CMD_RESULT(cmd) = DID_RESET << 16;
			break;

		case SCS_ABORTED:
			QL4PRINT(QLP2|QLP3, printk("scsi%d:%d:%d:%d: %s: "
						   "Abort occurred\n",
						   ha->host_no, cmd->channel,
						   cmd->target, cmd->lun,
						   __func__));

			CMD_RESULT(cmd) = DID_ABORT << 16;
			ha->aborted_io_count++;
			break;

		case SCS_TIMEOUT:
			QL4PRINT(QLP2, printk(KERN_INFO "scsi%d:%d:%d:%d: "
					      "Timeout\n",
					      ha->host_no, cmd->channel,
					      cmd->target, cmd->lun));

			/* F/W logout the connection when this occurs */
			CMD_RESULT(cmd) = DID_BUS_BUSY << 16;

			/*
			 * Mark device missing so that we won't continue to send
			 * I/O to this device.  We should get a ddb state change
			 * AEN soon.
			 */
			if (ddb_entry &&
			    (atomic_read(&ddb_entry->state) == DEV_STATE_ONLINE))
				qla4xxx_mark_device_missing(ha, ddb_entry);
			break;

		case SCS_DATA_UNDERRUN:
		case SCS_DATA_OVERRUN:
			if ((sts_entry->iscsiFlags & ISCSI_FLAG_RESIDUAL_OVER) != 0) {
				QL4PRINT(QLP2,
					 printk("scsi%d:%d:%d:%d: %s: "
						"Data overrun, "
						"residual = 0x%x\n",
						ha->host_no, cmd->channel,
						cmd->target, cmd->lun,
						__func__, residual));

				QL4PRINT(QLP10,
					 printk("scsi%d: %s: "
						"response packet data\n",
						ha->host_no, __func__));
				qla4xxx_dump_bytes(QLP10, sts_entry,
						   (sizeof(*sts_entry) *
						    sts_entry->hdr.entryCount));

				CMD_RESULT(cmd) = DID_ERROR << 16;
				break;
			}

			QL4PRINT(QLP2,
				 printk("scsi%d:%d:%d:%d: %s: "
					"UNDERRUN status detected, "
					"xferlen = 0x%x, "
					"residual = 0x%x\n",
					ha->host_no, cmd->channel,
					cmd->target, cmd->lun,
					__func__, CMD_XFRLEN(cmd),
					residual));

			/*
			 * If there is scsi_status, it takes precedense over
			 * underflow condition.
			 */
			if (sts_entry->scsiStatus != 0)
				break;

			/*
			 * If RISC reports underrun and target does not
			 * report it then we must have a lost frame, so
			 * tell upper layer to retry it by reporting a
			 * bus busy.
			 */
			if ((sts_entry->iscsiFlags & ISCSI_FLAG_RESIDUAL_UNDER) == 0) {
				QL4PRINT(QLP2,
					 printk(KERN_INFO "scsi%d:%d:%d:%d: "
						"%s: Dropped frame(s) "
						"detected (%x of %x bytes)..."
						" retrying command.\n",
						ha->host_no, cmd->channel,
						cmd->target, cmd->lun, __func__,
						residual, CMD_XFRLEN(cmd)));

				CMD_RESULT(cmd)= DID_BUS_BUSY << 16;
			}
			/*
			 * Handle mid-layer underflow???
			 *
			 * For kernels less than 2.4, the driver must
			 * return an error if an underflow is detected.
			 * For kernels equal-to and above 2.4, the
			 * mid-layer will appearantly handle the
			 * underflow by detecting the residual count --
			 * unfortunately, we do not see where this is
			 * actually being done.  In the interim, we
			 * will return DID_ERROR.
			 */
			else if ((CMD_XFRLEN(cmd) - residual) < cmd->underflow) {
				QL4PRINT(QLP2,
					 printk("scsi%d:%d:%d:%d: %s: "
						"Mid-layer Data underrun, "
						"xferlen = 0x%x, "
						"residual = 0x%x\n",
						ha->host_no, cmd->channel,
						cmd->target, cmd->lun,
						__func__, CMD_XFRLEN(cmd),
						residual));

				CMD_RESULT(cmd) = DID_ERROR << 16;
				CMD_RESID_LEN(cmd) = residual;
			}
			else {
				CMD_RESULT(cmd) = DID_OK << 16;

#ifndef __VMKERNEL_MODULE__
				/*
				 * Special case considertaion on an Inquiry
				 * command (0x12) for Lun 0, device responds
				 * with no devices (0x7F), then Linux will not
				 * scan further Luns. While reporting that some
				 * device exists on Lun 0 Linux will scan all
				 * devices on this target.
				 */
				if ((sts_entry->scsiStatus == 0) &&
				    (cmd->cmnd[0] == 0x12 && cmd->lun == 0 )) {
					strp = (uint8_t *)cmd->request_buffer;
					if (*strp == 0x7f) {
						/* Make lun unassigned and
						 * processor type */
						*strp = 0x23;
					}
				}
#endif // __VMKERNEL_MODULE__
			}
			break;

		case SCS_DEVICE_LOGGED_OUT:
		case SCS_DEVICE_UNAVAILABLE:
			/*
			 * Mark device missing so that we won't continue to
			 * send I/O to this device.  We should get a ddb
			 * state change AEN soon.
			 */
			if (ddb_entry &&
			    (atomic_read(&ddb_entry->state) == DEV_STATE_ONLINE))
				qla4xxx_mark_device_missing(ha, ddb_entry);

#ifdef __VMKERNEL_MODULE__
			/* Return DID_BUS_BUSY so commands following a
			 * timed out command get retried
			 */
			CMD_RESULT(cmd) = DID_BUS_BUSY << 16;
#else
			CMD_RESULT(cmd) = DID_ERROR << 16;
#endif
			break;

		case SCS_QUEUE_FULL:
			/*
			 * SCSI Mid-Layer handles device queue full
			 */
			CMD_RESULT(cmd) = DID_OK << 16 | sts_entry->scsiStatus;
			QL4PRINT(QLP2,
				 printk("scsi%d:%d:%d: %s: QUEUE FULL detected "
					"compl=%02x, scsi=%02x, state=%02x, "
					"iFlags=%02x, iResp=%02x\n",
					ha->host_no, cmd->target, cmd->lun,
					__func__, sts_entry->completionStatus,
					sts_entry->scsiStatus,
					sts_entry->state_flags,
					sts_entry->iscsiFlags,
					sts_entry->iscsiResponse));
			break;

		case SCS_DMA_ERROR:
		case SCS_TRANSPORT_ERROR:
		case SCS_DATA_DIRECTION_ERROR:
		case SCS_DEVICE_CONFIG_CHANGED:
		default:
			CMD_RESULT(cmd) = DID_ERROR << 16;
			break;
		}

		/* Check for errors */
		if (qla4xxx_check_sense(ha, sts_entry ,srb) == QLA_ERROR) {
			QL4PRINT(QLP2,
				 printk(KERN_INFO "scsi%d: %s: "
					"check sense error\n",
					ha->host_no, __func__));
			LEAVE("qla4xxx_status_entry");
			return;	 /* DO NOT complete request */
		}

		/* fill in info for passthru command */
		CMD_SCSI_STATUS(cmd)    = sts_entry->scsiStatus;
		if ((unsigned long)CMD_PASSTHRU_TYPE(cmd) == TRUE) {
			CMD_COMPL_STATUS(cmd)   = sts_entry->completionStatus;
			CMD_ISCSI_RESPONSE(cmd) = sts_entry->iscsiResponse;
			CMD_STATE_FLAGS(cmd)    = sts_entry->state_flags;
			CMD_HOST_STATUS(cmd)    = host_byte(CMD_RESULT(cmd));
		}

		/* complete the request */
		srb->cc_stat   = sts_entry->completionStatus;
#if 1 || defined(__VMKERNEL_MODULE__)
                /*
                 * We can deadlock with the error handler if we do this,
                 * so put the requests on a queue and complete them after
                 * the hardware_lock has been released by caller.
                 */
                add_to_done_srb_q(ha, srb);
#else
		qla4xxx_complete_request(ha, srb);
#endif
	}
	else {
		QL4PRINT(QLP2,
			 printk(KERN_WARNING "scsi%d: Status Entry invalid "
				"handle 0x%x, sp=%p. "
				"This cmd may have already been completed.\n",
				ha->host_no, __le32_to_cpu(sts_entry->handle),
				srb));

		QL4PRINT(QLP2, printk("scsi%d: %s: sts_entry 0x%p\n",
				      ha->host_no, __func__, sts_entry));
		qla4xxx_dump_bytes(QLP2, sts_entry, sizeof(*sts_entry));
	}

	DBG(LEAVE("qla4xxx_status_entry");)
}

/**************************************************************************
 * qla4xxx_process_response_queue
 *	This routine handles the Response Queue Completion.
 *
 * Input:
 * 	ha - Pointer to host adapter structure.
 *
 * Output:
 *	None
 *
 * Remarks:
 *	hardware_lock locked upon entry
 *
 * Returns:
 *	QLA_SUCCESS - Successfully processed response queue
 *	QLA_ERROR   - Failed to process response queue
 *
 * Context:
 *	Interrupt context.
 **************************************************************************/
STATIC uint32_t
qla4xxx_process_response_queue(scsi_qla_host_t *ha)
{
	uint32_t count = 0;
	uint32_t i;
	srb_t    *srb = 0;
	STATUS_ENTRY *sts_entry;

	DBG(ENTER("qla4xxx_process_response_queue");)
	DBG(QL4PRINT(QLP12, printk("scsi%d: %s entered signature=0x%x\n",
				   ha->host_no, __func__,
				   ha->response_ptr->signature));)

	/* Process all responses from response queue */
	while (ha->response_ptr->signature != RESPONSE_PROCESSED) {
		/* Make sure we don't get ahead of the adapter's in pointer
		 * (causing it to be full) */
		ha->response_in = ISP_RD_RSP_Q_IN(ha);

		sts_entry = (STATUS_ENTRY *) ha->response_ptr;
		#ifdef DEBUG
		QL4PRINT(QLP5,
			 printk("scsi%d: %s: RespPtr %p, type %x\n",
				ha->host_no, __func__,
				ha->response_ptr,
				sts_entry->hdr.entryType));
		QL4PRINT(QLP5,
			 printk("scsi%d: %s: Resp out %x, in %x, Req in\n",
				ha->host_no, __func__,
				ha->response_out, ha->response_in));

		QL4PRINT(QLP10, printk("scsi%d: %s: sts_entry 0x%p\n",
				       ha->host_no, __func__, sts_entry));
		qla4xxx_dump_bytes(QLP10, sts_entry, sizeof(*sts_entry));
		#endif

		if (ha->response_out == ha->response_in) {
			DBG(QL4PRINT(QLP5,
				     printk("scsi%d: %s: Response count %x out %x "
					    "in %x, next %p:%x.  Finished!\n",
					    ha->host_no, __func__, count,
					    ha->response_out, ha->response_in,
					    ha->request_ptr,
					    ha->response_ptr->signature));)
			break;
		}

		count++;

		/* Advance pointers for next entry */
		if (ha->response_out == (RESPONSE_QUEUE_DEPTH - 1)) {
			ha->response_out = 0;
			ha->response_ptr = ha->rsp_queue_headv;
		}
		else {
			ha->response_out++;
			ha->response_ptr++;
		}

		/* process entry */
		switch (sts_entry->hdr.entryType) {
		case ET_STATUS:
			/* Common status - Single completion posted in single
			 * IOSB */
			ha->f_end = jiffies;

			qla4xxx_status_entry(ha, sts_entry);
			break;
#ifdef QLA4010

		case ET_MAILBOX_STATUS:
			if (sts_entry->hdr.systemDefined == SD_PASSTHRU_IOCB)
				qla4xxx_iocb_pass_done(ha, (PASSTHRU_STATUS_ENTRY *) sts_entry);
			break;

		case ET_PASSTHRU_STATUS:
			if (sts_entry->hdr.systemDefined == SD_PASSTHRU_IOCB)
				qla4xxx_iocb_pass_done(ha, (PASSTHRU_STATUS_ENTRY *) sts_entry);
			else
				qla4xxx_isns_process_response(ha, (PASSTHRU_STATUS_ENTRY *) sts_entry);
			break;
#endif

		case ET_PERFORMANCE_STATUS:
			{

				/* Performance Status Entry Handling -
				 * Multiple completions posted in a single IOSB.
				 * All completions are good */
				PERFORMANCE_STATUS_ENTRY *perf_sts_entry =
				(PERFORMANCE_STATUS_ENTRY *) sts_entry;

				for (i =0; i < perf_sts_entry->entryCount; i++) {
					srb = del_from_active_array(ha,
								    perf_sts_entry->
								    handleArray[i]);
					if (srb == NULL)
						goto exit_prq_invalid_handle;

					CMD_RESULT(srb->cmd) =(int)  DID_OK << 16;
#if 1 || defined(__VMKERNEL_MODULE__)
                                        /*
                                         * We can deadlock with the error handler if we do this,
                                         * so put the requests on a queue and complete them after
                                         * the hardware_lock has been released by caller.
                                         */
                                        add_to_done_srb_q(ha, srb);
#else
					qla4xxx_complete_request(ha, srb);
#endif
				}
			}
			break;

		case ET_STATUS_CONTINUATION:
			/* Just throw away the status continuation entries */
			QL4PRINT(QLP2,
				 printk("scsi%d: %s: Status Continuation entry "
					"- ignoring\n", ha->host_no, __func__));
			break;

		case ET_COMMAND:
			/* ISP device queue is full. Command not accepted by
			 * ISP.  Queue command for later */

			srb = del_from_active_array(ha, __le32_to_cpu(sts_entry->handle));
			if (srb == NULL)
				goto exit_prq_invalid_handle;

			QL4PRINT(QLP2, printk("scsi%d: %s: FW device queue full, "
					      "srb %p, put pending_srb_q\n",
					      ha->host_no, __func__, srb));
			add_to_pending_srb_q(ha, srb);
			break;

		case ET_CONTINUE:
			/* Just throw away the continuation entries */
			QL4PRINT(QLP2, printk("scsi%d: %s: Continuation entry - "
					      "ignoring\n",
					      ha->host_no, __func__));
			break;

		default:
			/* Invalid entry in response queue, reset RISC
			 * firmware */
			QL4PRINT(QLP2, printk("scsi%d: %s: Invalid entry %x "
					      "in response queue \n",
					      ha->host_no, __func__,
					      sts_entry->hdr.entryType));

			QL4PRINT(QLP10, printk("scsi%d: %s: Dumping Response Entry "
					       "%p:%x out %x in%x\n",
					       ha->host_no, __func__,
					       sts_entry,
					       __le32_to_cpu(((QUEUE_ENTRY*)sts_entry)->
							     signature),
					       ha->response_out,
					       ha->response_in));

			qla4xxx_dump_bytes(QLP10, sts_entry,
					   sizeof(*sts_entry));
			goto exit_prq_error;
		}

		/* Set queue entry processed signature */
		((QUEUE_ENTRY *)sts_entry)->signature = (RESPONSE_PROCESSED);

		wmb();
	}

	/* Done with responses, update the ISP
	 * For QLA4010, this also clears the interrupt.
	 */
	WRT_REG_DWORD(&ha->reg->responseQueueOutPointer, ha->response_out);

	DBG(LEAVE("qla4xxx_process_response_queue");)
	return(QLA_SUCCESS);

	exit_prq_invalid_handle:
	QL4PRINT(QLP2, printk("scsi%d: %s: Invalid handle(srb)=%p type=%x "
			      "IOCS=%x\n", ha->host_no, __func__,
			      srb, sts_entry->hdr.entryType,
			      sts_entry->completionStatus));

	exit_prq_error:
	WRT_REG_DWORD(&ha->reg->requestQueueOutPointer, ha->response_out);

	QL4PRINT(QLP2, printk("scsi%d: %s: schedule DPC to restart ha %d\n",
			      ha->host_no, __func__, ha->instance));

	set_bit(DPC_RESET_HA, &ha->dpc_flags);
	set_bit(AF_SCHEDULE_DPC, &ha->flags);

	LEAVE("qla4xxx_process_response_queue");
	return(QLA_ERROR);
}

/**************************************************************************
 * qla4xxx_isr_decode_mailbox
 *	This routine decodes the mailbox status during the ISR.
 *
 * Input:
 * 	ha - Pointer to host adapter structure.
 *	mailbox_status - Mailbox status.
 *
 * Remarks:
 *      hardware_lock locked upon entry
 *
 * Returns:
 *	None.
 *
 * Context:
 *	Interrupt context.
 **************************************************************************/
STATIC void
qla4xxx_isr_decode_mailbox(scsi_qla_host_t *ha, uint32_t  mbox_status)
{
	/* used for MBOX_ASTS_ISNS_UNSOLICITED_PDU_RECEIVED */
#if 1 || defined(__VMKERNEL_MODULE__)
	uint32_t   mbox_sts[MBOX_REG_COUNT];
#else
	static uint32_t   mbox_sts[MBOX_REG_COUNT];
#endif

	if ((mbox_status == MBOX_STS_BUSY) ||
	    (mbox_status == MBOX_STS_INTERMEDIATE_COMPLETION) ||
	    (mbox_status >>12 == MBOX_COMPLETION_STATUS)) {
		ha->mbox_status[       0]        =        mbox_status;

		if (test_bit(AF_MBOX_COMMAND, &ha->flags)) {
			/*
			 * Copy all mailbox registers to a temporary
			 * location and set mailbox command done flag
			 */
			uint8_t i;

			for (i=1; i<ha->mbox_status_count; i++) {
				ha->mbox_status[i] =
				RD_REG_DWORD(&ha->reg->mailboxOut[i-1]);
			}

			QL4PRINT(QLP11,
				 printk("scsi%d: %s: mailbox cmd done!\n",
					ha->host_no, __func__));

			ha->f_end = jiffies;
			set_bit(AF_MBOX_COMMAND_DONE, &ha->flags);
			wake_up(&ha->mailbox_wait_queue);
		}
		#if 0
		else {
			QL4PRINT(QLP2,
				 printk(KERN_INFO "scsi%d: ERROR: Receiving mailbox "
					"status %08X when no mailbox command "
					"active.\n",
					ha->host_no, mbox_status));

			__dump_registers(ha);
		}
		#endif
	}
	else if (mbox_status >> 12 == MBOX_ASYNC_EVENT_STATUS) {
		/* Immediately process the AENs that don't require much work.
		 * Only queue the database_changed AENs */
		switch (mbox_status) {
		case MBOX_ASTS_SYSTEM_ERROR:
			/* Log Mailbox registers */
			QL4PRINT(QLP2,
				 printk(KERN_INFO
					"scsi%d: aen %04x, System Error, "
					"Dump Mailboxes\n",
					ha->host_no, mbox_status));
			__dump_mailbox_registers(QLP2, ha);
				#ifdef QLA4010
			set_bit(AF_GET_CRASH_RECORD, &ha->flags);
				#endif
			set_bit(DPC_RESET_HA, &ha->dpc_flags);
			set_bit(AF_SCHEDULE_DPC, &ha->flags);
			break;

		case MBOX_ASTS_REQUEST_TRANSFER_ERROR:
		case MBOX_ASTS_RESPONSE_TRANSFER_ERROR:
		case MBOX_ASTS_NVRAM_INVALID:
		case MBOX_ASTS_IP_ADDRESS_CHANGED:
		case MBOX_ASTS_DHCP_LEASE_EXPIRED:
			QL4PRINT(QLP2,
				 printk(KERN_INFO "scsi%d: aen %04x, "
					"ERROR Status, Reset HA\n",
					ha->host_no, mbox_status));

			set_bit(DPC_RESET_HA, &ha->dpc_flags);
			set_bit(AF_SCHEDULE_DPC, &ha->flags);
			break;

		case MBOX_ASTS_LINK_UP:
			QL4PRINT(QLP2,
				 printk(KERN_INFO "scsi%d: aen %04x "
					"Adapter LINK UP\n",
					ha->host_no, mbox_status));
			set_bit(AF_LINK_UP, &ha->flags);

			/* Sometimes we don't get state change AENs when the
			 * link comes back up, so we have to manually trigger
			 * relogins. */
			if (test_bit(DPC_RELOGIN_DEVICE, &ha->dpc_flags))
				set_bit(AF_SCHEDULE_DPC, &ha->flags);

			break;

		case MBOX_ASTS_LINK_DOWN:
			QL4PRINT(QLP2,
				 printk(KERN_INFO "scsi%d: aen %04x "
					"Adapter LINK DOWN\n",
					ha->host_no, mbox_status));
			clear_bit(AF_LINK_UP, &ha->flags);
			break;

		case MBOX_ASTS_HEARTBEAT:
			QL4PRINT(QLP7,
				 printk(KERN_INFO "scsi%d: aen %04x "
					"HEARTBEAT\n",
					ha->host_no, mbox_status));
			ha->seconds_since_last_heartbeat = 0;
			break;

		case MBOX_ASTS_DHCP_LEASE_ACQUIRED:
			QL4PRINT(QLP2, printk(KERN_INFO "scsi%d: aen %04x DHCP LEASE ACQUIRED\n",
					      ha->host_no, mbox_status));

			/* Reset to get devices, etc. */   //FIXME: Test this!!!!
			set_bit(DPC_RESET_HA, &ha->dpc_flags);
			set_bit(AF_SCHEDULE_DPC, &ha->flags);
			break;

		case MBOX_ASTS_PROTOCOL_STATISTIC_ALARM:
		case MBOX_ASTS_SCSI_COMMAND_PDU_REJECTED:	  /* Target mode only */
		case MBOX_ASTS_UNSOLICITED_PDU_RECEIVED:	  /* connection mode only */
		case MBOX_ASTS_IPSEC_SYSTEM_FATAL_ERROR:
			/* No action */
			QL4PRINT(QLP2, printk(KERN_INFO "scsi%d: aen %04x\n",
					      ha->host_no, mbox_status));
			break;

		case MBOX_ASTS_MAC_ADDRESS_CHANGED:
		case MBOX_ASTS_DNS:
			/* No action */
			QL4PRINT(QLP2,
				 printk(KERN_INFO "scsi%d: aen %04x, "
					"mbox_sts[1]=%04x, "
					"mbox_sts[2]=%04x\n",
					ha->host_no, mbox_status,
					RD_REG_DWORD(&ha->reg->mailboxOut[0]),
					RD_REG_DWORD(&ha->reg->mailboxOut[1])));
			break;

		case MBOX_ASTS_SELF_TEST_FAILED:
		case MBOX_ASTS_LOGIN_FAILED:
			/* No action */
			QL4PRINT(QLP2,
				 printk(KERN_INFO "scsi%d: aen %04x, "
					"mbox_sts[1]=%04x, "
					"mbox_sts[2]=%04x, mbox_sts[3]=%04x\n",
					ha->host_no, mbox_status,
					RD_REG_DWORD(&ha->reg->mailboxOut[0]),
					RD_REG_DWORD(&ha->reg->mailboxOut[1]),
					RD_REG_DWORD(&ha->reg->mailboxOut[2])));
			break;

		case MBOX_ASTS_DATABASE_CHANGED:
			/* Queue AEN information and process it in the DPC
			 * routine */
			spin_lock(&ha->aen_q_spinlock);
			if (ha->aen_q_count > 0) {
				int i;

				/* advance pointer */
				if (ha->aen_in == (MAX_AEN_ENTRIES - 1))
					ha->aen_in = 0;
				else
					ha->aen_in++;

				/* decrement available counter */
				ha->aen_q_count--;

				for (i=1; i < MBOX_AEN_REG_COUNT; i++) {
					ha->aen_q[ha->aen_in].mbox_sts[i] =
					RD_REG_DWORD(&ha->reg->mailboxOut[i-1]);
				}
				ha->aen_q[ha->aen_in].mbox_sts[0] = mbox_status;

				/* print debug message */
				QL4PRINT(QLP7,
					 printk("scsi%d: aen[%d] %04x queued!\n",
						ha->host_no, ha->aen_in,
						mbox_status));

				/* The DPC routine will process the aen */
				set_bit(AF_SCHEDULE_DPC, &ha->flags);
			}
			else {
				int i;

				QL4PRINT(QLP2,
					 printk("scsi%d: %s: aen %04x, queue "
						"overflowed!  AEN LOST!!\n",
						ha->host_no, __func__,
						mbox_status));

				QL4PRINT(QLP2,
					 printk(KERN_WARNING "scsi%d: "
						"DUMP AEN QUEUE\n",
						ha->host_no));

				for (i=0; i < MAX_AEN_ENTRIES; i++) {
					QL4PRINT(QLP2,
						 printk(KERN_WARNING "aen[%d] %04x %04x %04x %04x\n",
							i,
							ha->aen_q[i].mbox_sts[0],
							ha->aen_q[i].mbox_sts[1],
							ha->aen_q[i].mbox_sts[2],
							ha->aen_q[i].mbox_sts[3]));
				}

				#if STOP_ON_LOST_AEN
				qla4xxx_panic("LOST AEN", ha);
				#endif

			}
			spin_unlock(&ha->aen_q_spinlock);
			break;

		case MBOX_ASTS_ISNS_UNSOLICITED_PDU_RECEIVED:
			memset(&mbox_sts, 0, sizeof(mbox_sts));
			mbox_sts[0] = mbox_status;
			mbox_sts[1] = RD_REG_DWORD(&ha->reg->mailboxOut[0]);
			mbox_sts[2] = RD_REG_DWORD(&ha->reg->mailboxOut[1]);
			mbox_sts[3] = RD_REG_DWORD(&ha->reg->mailboxOut[2]);
			mbox_sts[4] = RD_REG_DWORD(&ha->reg->mailboxOut[3]);
			mbox_sts[5] = RD_REG_DWORD(&ha->reg->mailboxOut[4]);

			if (mbox_sts[1] == ISNS_EVENT_DATA_RECEIVED) {
				QL4PRINT(QLP2, printk(KERN_INFO "scsi%d: aen %04x, mbox_sts[1]=%04x, "
						      "mbox_sts[2]=%04x, mbox_sts[3]=%04x, mbox_sts[4]=%04x\n",
						      ha->host_no, mbox_status, mbox_sts[1],
						      mbox_sts[2], mbox_sts[3], mbox_sts[4]));

				if (qla4xxx_isns_get_server_request(ha,
								    mbox_sts[3],
								    mbox_sts[2])
				    != QLA_SUCCESS) {
					QL4PRINT(QLP2,
						 printk("scsi%d: %s: aen %04x, "
							"isns_get_server_request FAILED!!\n",
							ha->host_no, __func__, mbox_status));
				}
			}
			else if (mbox_sts[1] == ISNS_EVENT_CONNECTION_OPENED) {
				QL4PRINT(QLP2, printk(KERN_INFO "scsi%d: aen %04x, iSNS Service "
						      "Connection Opened!\n"
						      "mbox_sts[2]=%08x, mbox_sts[3]=%08x, "
						      "mbox_sts[4]=%08x, mbox_sts[5]=%08x\n",
						      ha->host_no, mbox_status, mbox_sts[2],
						      mbox_sts[3], mbox_sts[4], mbox_sts[5]));

				qla4xxx_isns_enable_callback(ha,
							     mbox_sts[2],
							     mbox_sts[3],
							     mbox_sts[4],
							     mbox_sts[5]);
			}
			else if (mbox_sts[1] == ISNS_EVENT_CONNECTION_FAILED) {
				QL4PRINT(QLP2, printk(KERN_INFO "scsi%d: aen %04x, iSNS Service"
						      " Connection FAILED! reason %04x\n",
						      ha->host_no, mbox_status, mbox_sts[2]));
			}
			break;
		default:
			QL4PRINT(QLP2,
				 printk(KERN_INFO "scsi%d: aen %04x UNKNOWN\n",
					ha->host_no, mbox_status));
		}
	}
	else {
		QL4PRINT(QLP2,
			 printk(KERN_INFO "scsi%d: Unknown mailbox status %08X\n",
				ha->host_no, mbox_status));

		ha->mbox_status[0] = mbox_status;
		__dump_registers(QLP2, ha);
	}
}

/**************************************************************************
 * qla4xxx_interrupt_service_routine
 *	This routine services the interrupt
 *
 * Input:
 * 	ha - Pointer to host adapter structure.
 *
 * Remarks:
 *      hardware_lock locked upon entry
 *
 * Returns:
 *      QLA_SUCCESS - success, An interrupt was found and processed
 *	QLA_ERROR - failure, The adapter was not interrupting
 *
 * Context:
 *	Interrupt context.
 **************************************************************************/
STATIC void
qla4xxx_interrupt_service_routine(scsi_qla_host_t *ha, uint32_t intr_status)
{
	DBG(ENTER("qla4xxx_interrupt_service_routine");)

	DBG(QL4PRINT(QLP11, printk("scsi%d: %s: intr_status = %X\n",
				   ha->host_no, __func__, intr_status));)

	if (intr_status == INVALID_REGISTER) {
		#ifdef QLA4000

		/*
		 * In the QLA4000, any attempt to access ISP registers during
		 * a firmware reset resutls in 0xFFFFFFFF (INVALID_REGISTER).
		 * Just ignore it.
		 */
		if (!ADAPTER_UP(ha)) {
			LEAVE("qla4xxx_interrupt_service_routine");
		}
		#endif	

		QL4PRINT(QLP2,
			 printk("scsi%d: %s: Unrecognized intr_status %x.\n",
				ha->host_no, __func__, intr_status));
		LEAVE("qla4xxx_interrupt_service_routine");
	}

	ha->seconds_since_last_intr = 0;

	#ifdef QLA4000

	/*
	 * Process fast post interrupt.
	 */
	if (intr_status & ISO_FASTPOST_COMPLETE) {
		srb_t *srb;

		srb = del_from_active_array(
					   ha, RD_REG_DWORD(&ha->reg->fastPostHandle));

		if (srb) {
			/* complete request */
#if 1 || defined(__VMKERNEL_MODULE__)
                        /*
                         * We can deadlock with the error handler if we do this,
                         * so put the requests on a queue and complete them after
                         * the hardware_lock has been released by caller.
                         */
                        add_to_done_srb_q(ha, srb);
#else
			qla4xxx_complete_request(ha, srb);
#endif
		}
		else {
			/* Invalid FastPostHandle from RISC,
			 * reset RISC firmware */
			QL4PRINT(QLP2,
				 printk(KERN_INFO "scsi%d: Invalid fast_post "
					"handle=0x%x, Scheduling Adapter Reset\n",
					ha->host_no,
					RD_REG_WORD(&ha->reg->fastPostHandle)));

			QL4PRINT(QLP2,
				 printk("scsi%d: %s: schedule DPC to reset HA\n",
					ha->host_no, __func__));

			set_bit(DPC_RESET_HA, &ha->dpc_flags);
			set_bit(AF_SCHEDULE_DPC, &ha->flags);
		}

		/* Clear Fast Post Interrupt */
		WRT_REG_DWORD(&ha->reg->interruptStatusOut,
			      ISO_FASTPOST_COMPLETE);
	}
	#endif	

	/*
	 * Process response queue interrupt.
	 */
	if (intr_status & ISO_RSP_QUEUE_COMPLETE) {
		qla4xxx_process_response_queue(ha);

		#ifdef QLA4000

		/* Clear Response Queue Interrupt */
		ISP_CLEAR_RSP_Q_INTR(ha);
		#endif
	}

	/*
	 * Process mailbox/asynch event  interrupt.
	 */
	if (intr_status & ISO_MB0X_CMD_COMPLETE) {
		uint32_t mbox_status = RD_REG_DWORD(&ha->reg->mailboxOut0);
		qla4xxx_isr_decode_mailbox(ha, mbox_status);

		/* Clear Mailbox Interrupt by writing MailboxOut0 value back to
		 * MailboxOut0 register. */
		ISP_CLEAR_MBOX_INTR(ha, mbox_status);

	} /* end mailbox interrupt */

	DBG(LEAVE("qla4xxx_interrupt_service_routine");)
}

/**************************************************************************
 * qla4xxx_mailbox_command
 *	This routine sssue mailbox commands and waits for completion.
 *
 * Input:
 * 	ha - Pointer to host adapter structure.
 *	inCount	 - number of mailbox registers to load.
 *      outCount - number of mailbox registers to return.
 *      mbx_cmd  - data pointer for mailbox in registers.
 *      mbx_sts  - data pointer for mailbox out registers.
 *
 * Output:
 *      mbx_sts - returned mailbox out data.
 *
 * Remarks:
 *	If outCount is 0, this routine completes successfully WITHOUT waiting
 *	for the mailbox command to complete.
 *
 * Returns:
 *	QLA_SUCCESS - Mailbox command completed successfully
 *	QLA_ERROR   - Mailbox command competed in error.
 *
 * Context:
 *	Kernel context.
 **************************************************************************/
uint8_t
qla4xxx_mailbox_command(scsi_qla_host_t *ha,
			uint8_t inCount,
			uint8_t outCount,
			uint32_t *mbx_cmd,
			uint32_t *mbx_sts)
{
	uint8_t      status = QLA_ERROR;
	uint8_t      i;
	u_long     wait_count;
	uint32_t     intr_status;
	unsigned long flags = 0;
	DECLARE_WAITQUEUE(wait, current);


	ENTER("qla4xxx_mailbox_command");

	down(&ha->mbox_sem);


	set_bit(AF_MBOX_COMMAND, &ha->flags);


	/* Make sure that pointers are valid */
	if (!mbx_cmd || !mbx_sts) {
		QL4PRINT(QLP2,
			 printk("scsi%d: %s: Invalid mbx_cmd or mbx_sts pointer\n",
				ha->host_no, __func__));

		goto mbox_exit;
	}

	/* To prevent overwriting mailbox registers for a command that has
	 * not yet been serviced, check to see if a previously issued
	 * mailbox command is interrupting.
	 * -----------------------------------------------------------------
	 */
	spin_lock_irqsave(&ha->hardware_lock, flags);
	intr_status = ISP_RD_INTR_STATUS(ha);

	if (intr_status & ISO_MB0X_CMD_COMPLETE) {
		QL4PRINT(QLP2,
			 printk("scsi%d: %s: Trying to execute a mailbox request, "
				"while another one is interrupting\n"
				"Service existing interrupt first\n",
				ha->host_no, __func__));

		/* Service existing interrupt */
		qla4xxx_interrupt_service_routine(ha, intr_status);
	}


	/* Send the mailbox command to the firmware
	 * ----------------------------------------
	 */
	ha->f_start = jiffies;
	ha->mbox_status_count = outCount;
	for (i=0; i < outCount; i++) {
		ha->mbox_status[i] = 0;
	}

	for (i=0; i<inCount; i++) {
		QL4PRINT(QLP11, printk("scsi%d: %s: Mailbox In[%d]  0x%08X\n",
				       ha->host_no, __func__, i, mbx_cmd[i]));
	}

	/* Load all mailbox registers, except mailbox 0.*/
	for (i = 1; i < inCount; i++) {
		WRT_REG_DWORD(&ha->reg->mailboxIn[i-1], mbx_cmd[i]);
	}

	/* Write Mailbox 0 to alert the firmware that the mailbox registers
	 * contain a command to be processed.  NOTE: We could be interrupted
	 * here if system interrupts are enabled */
	WRT_REG_DWORD(&ha->reg->mailboxIn0, mbx_cmd[0]);
#ifdef QLA4010
	WRT_REG_DWORD(&ha->reg->ISPControlStatus, CSRW_INTR_RISC);
#endif

	spin_unlock_irqrestore(&ha->hardware_lock, flags);
#if 1 || defined(__VMKERNEL_MODULE__)
        /* 
         * Need to complete requests we put on the srb_done_q to fix
         * deadlock issue between error handler and irq handler...
         */
        if (!list_empty(&ha->done_srb_q)) {
                srb_t    *srb;
                while ((srb = del_from_done_srb_q_head(ha)) != NULL)
                        qla4xxx_complete_request(ha, srb);
        }
#endif

	set_current_state(TASK_UNINTERRUPTIBLE);
	add_wait_queue(&ha->mailbox_wait_queue,&wait);

	/*
	 * If we don't want status, don't wait for the mailbox command to
	 * complete.  For example, MBOX_CMD_RESET_FW doesn't return status,
	 * you must poll the inbound Interrupt Mask for completion.
	 */
	if (outCount == 0) {
		status = QLA_SUCCESS;
		set_current_state(TASK_RUNNING);
		remove_wait_queue(&ha->mailbox_wait_queue,&wait);
		ha->f_end = jiffies;
		goto mbox_exit;
	}

	/*
	 * Wait for command to complete
	 * -----------------------------
	 */
	wait_count = jiffies + MBOX_TOV * HZ;

	while (test_bit(AF_MBOX_COMMAND_DONE, &ha->flags) == 0) {
		if (wait_count <= jiffies)
			break;

		spin_lock_irqsave(&ha->hardware_lock, flags);

		intr_status = ISP_RD_INTR_STATUS(ha);

		QL4PRINT(QLP11, printk("scsi%d: %s: INTR_STATUS = 0x%X\n",
				       ha->host_no, __func__, intr_status));

		if (intr_status & INTR_PENDING) {
			/*
			 * Service the interrupt.
			 * The ISR will save the mailbox status registers
			 * to a temporary storage location in the adapter
			 * structure.
			 */
			ha->mbox_status_count = outCount;
			qla4xxx_interrupt_service_routine(ha, intr_status);
		}
		spin_unlock_irqrestore(&ha->hardware_lock, flags);
#if 1 || defined(__VMKERNEL_MODULE__)
                /* 
                 * Need to complete requests we put on the srb_done_q to fix
                 * deadlock issue between error handler and irq handler...
                 */
                if (!list_empty(&ha->done_srb_q)) {
                   srb_t    *srb;
                   while ((srb = del_from_done_srb_q_head(ha)) != NULL)
                      qla4xxx_complete_request(ha, srb);
                }
#endif

		/*
		 * Delay for 10 microseconds
		 * NOTE: Interrupt_handler may be called here,
		 *       if interrupts are enabled
		 */
		udelay(10);

	} /* wait loop */


	set_current_state(TASK_RUNNING);
	remove_wait_queue(&ha->mailbox_wait_queue,&wait);

	/*
	 * Check for mailbox timeout
	 */
	if (!test_bit(AF_MBOX_COMMAND_DONE, &ha->flags)) {
		QL4PRINT(QLP2,
			 printk("scsi%d: Mailbox Cmd 0x%08X timed out ...,"
				" Scheduling Adapter Reset\n",
				ha->host_no, mbx_cmd[0]));

		ha->mailbox_timeout_count++;
		mbx_sts[0] = (-1);

		set_bit(DPC_RESET_HA, &ha->dpc_flags);
		set_bit(AF_SCHEDULE_DPC, &ha->flags);
		goto mbox_exit;
	}

	QL4PRINT(QLP11,
		 printk("scsi%d: %s: mailbox cmd done!\n",
			ha->host_no, __func__));

	/*
	 * Copy the mailbox out registers to the caller's mailbox in/out
	 * structure.
	 */
	spin_lock_irqsave(&ha->hardware_lock, flags);
	for (i=0; i < outCount; i++) {
		mbx_sts[i] = ha->mbox_status[i];
		QL4PRINT(QLP11,
			 printk("scsi%d: %s: Mailbox Status[%d]  0x%08X\n",
				ha->host_no, __func__, i, mbx_sts[i]));
	}

	/*
	 * Set return status and error flags (if applicable)
	 */
	switch (ha->mbox_status[0]) {
	
	case MBOX_STS_COMMAND_COMPLETE:
		status = QLA_SUCCESS;
		break;

	case MBOX_STS_INTERMEDIATE_COMPLETION:
		status = QLA_SUCCESS;
		QL4PRINT(QLP5,
			 printk("scsi%d: %s: Cmd = %08X, Intermediate completion\n",
				ha->host_no, __func__, mbx_cmd[0]));
		break;

	case MBOX_STS_BUSY:
		QL4PRINT(QLP2, printk("scsi%d: %s: Cmd = %08X, ISP BUSY\n",
				      ha->host_no, __func__, mbx_cmd[0]));

		ha->mailbox_timeout_count++;
		break;

	case MBOX_STS_INVALID_COMMAND:
	case MBOX_STS_HOST_INTERFACE_ERROR:
	case MBOX_STS_TEST_FAILED:
	case MBOX_STS_COMMAND_ERROR:
	case MBOX_STS_COMMAND_PARAMETER_ERROR:
	default:
		QL4PRINT(QLP2,
			 printk("scsi%d: %s: **** FAILED, cmd = %08X, "
				"sts = %08X ****\n",
				ha->host_no, __func__, mbx_cmd[0], mbx_sts[0]));


		__dump_registers(QLP2, ha);
		break;
	} /* switch mbox status */
	spin_unlock_irqrestore(&ha->hardware_lock, flags);

	mbox_exit:
	clear_bit(AF_MBOX_COMMAND, &ha->flags);
	clear_bit(AF_MBOX_COMMAND_DONE, &ha->flags);
	LEAVE("qla4xxx_mailbox_command");
	up(&ha->mbox_sem);

	return(status);
}

/**************************************************************************
 * qla4xxx_add_device_dynamically
 *	This routine processes adds a device as a result of an 8014h AEN.
 *
 * Input:
 * 	ha - Pointer to host adapter structure.
 *      fw_ddb_index - Firmware's device database index
 *
 * Returns:
 *	None
 *
 * Context:
 *	Kernel context.
 **************************************************************************/
void
qla4xxx_add_device_dynamically(scsi_qla_host_t *ha,
			       uint32_t fw_ddb_index)
{
	ddb_entry_t *ddb_entry;
	uint16_t target;

	ENTER("qla4xxx_add_device_dynamically");

	/* First allocate a device structure */
	ddb_entry = qla4xxx_alloc_ddb(ha, fw_ddb_index);
	if (ddb_entry == NULL) {
		QL4PRINT(QLP2,
			 printk(KERN_WARNING "scsi%d: Unable to allocate "
				"memory to add fw_ddb_index %d\n",
				ha->host_no, fw_ddb_index));
	}
	else if (qla4xxx_update_ddb_entry(ha, ddb_entry) == QLA_ERROR) {
		QL4PRINT(QLP2,
			 printk(KERN_WARNING "scsi%d: failed to add new "
				"device at index [%d]\n"
				"Unable to retrieve fw ddb entry\n",
				ha->host_no, fw_ddb_index));
	}
	else {
		/* New device. Let's add it to the database */
		QL4PRINT(QLP7,
			 printk("scsi%d: %s: new device at index [%d]\n",
				ha->host_no, __func__, fw_ddb_index));

		/*
		 * First find an unused target id
		 */
#if 1 || __VMKERNEL_MODULE__
		down(&ha->targ_array_sem);
#endif
		for (target = 0; target < MAX_TARGETS; target++) {
			if (qla4xxx_lookup_ddb_by_SCSIID(ha, 0, target) == NULL)
				break;
		}

		/*
		 * Now perform a discovery of the target id and all
		 * possible luns.
		 */
		if (target < MAX_TARGETS &&
		    qla4xxx_discover_target_luns(ha, ddb_entry, target)
		    == QLA_SUCCESS) {
			QL4PRINT(QLP7,
				 printk(KERN_WARNING "scsi%d:%d:%d: "
					"added new device at index [%d]\n",
					ha->host_no, ddb_entry->bus,
					target, fw_ddb_index));
		}
		else {
			QL4PRINT(QLP2,
				 printk(KERN_WARNING "scsi%d: failed to map "
					"new device at index [%d] to a "
					" corresponding SCSI target ID\n",
					ha->host_no, fw_ddb_index));
		}
#if 1 || __VMKERNEL_MODULE__
		up(&ha->targ_array_sem);
#endif
	}

	LEAVE("qla4xxx_add_device_dynamically");
}

/**************************************************************************
 * qla4xxx_process_ddb_changed
 *	This routine processes a Decive Database Changed AEN Event.
 *
 * Input:
 * 	ha - Pointer to host adapter structure.
 *      fw_ddb_index - Firmware's device database index
 *      state - Device state
 *
 * Returns:
 *	QLA_SUCCESS - Successfully processed ddb_changed aen
 *	QLA_ERROR   - Failed to process ddb_changed aen
 *
 * Context:
 *	Kernel context.
 **************************************************************************/
uint8_t
qla4xxx_process_ddb_changed(scsi_qla_host_t *ha,
			    uint32_t fw_ddb_index,
			    uint32_t state)
{
	uint8_t     status = QLA_ERROR;
	ddb_entry_t *ddb_entry;
	uint32_t    old_fw_ddb_device_state;

	ENTER("qla4xxx_process_ddb_changed");

	/* check for out of range index */
	if (fw_ddb_index >= MAX_DDB_ENTRIES) {
		QL4PRINT(QLP2,
			 printk("scsi%d: %s: device index [%d] out of range\n",
				ha->host_no, __func__, fw_ddb_index));

		LEAVE("qla4xxx_process_ddb_changed");
		return(status);
	}

	/* Get the corresponging ddb entry */
	ddb_entry = qla4xxx_lookup_ddb_by_fw_index(ha, fw_ddb_index);

	/* Check to see if it's a device we already have in our database */
	if (ddb_entry == NULL) {  /* device does ot exist */
		if (state == DDB_DS_SESSION_ACTIVE) {
			qla4xxx_add_device_dynamically(ha, fw_ddb_index);
		}
	}
	else {	/* device already exists */
		old_fw_ddb_device_state = ddb_entry->fw_ddb_device_state;

		/* Device came back */
		if (old_fw_ddb_device_state == state) {
			/* Do nothing, state not changed */
		}
		else if ((qla4xxx_get_ddb_entry(ha, ddb_entry->fw_ddb_index, NULL, NULL,
						NULL, &ddb_entry->fw_ddb_device_state, NULL, NULL, NULL)
			  == QLA_SUCCESS) &&
			 (ddb_entry->fw_ddb_device_state == DDB_DS_SESSION_ACTIVE)) {
			unsigned long cpu_flags;
			lun_entry_t *lun_entry;
			int i;

			atomic_set(&ddb_entry->port_down_timer,
				   ha->port_down_retry_count);
			atomic_set(&ddb_entry->state, DEV_STATE_ONLINE);
			clear_bit(DF_RELOGIN, &ddb_entry->flags);
			clear_bit(DF_NO_RELOGIN, &ddb_entry->flags);

			QL4PRINT(QLP3,
				 printk(KERN_INFO "scsi%d:%d:%d: index [%d] "
					"marked ONLINE\n", ha->host_no,
					ddb_entry->bus, ddb_entry->target,
					ddb_entry->fw_ddb_index));

			/*
			 * Change the lun state to READY in case the lun
			 * TIMEOUT before the device came back.
			 */
			for (i = 0; i < MAX_LUNS; i++) {
				lun_entry = ddb_entry->lun_table[i];

				if (lun_entry) {
					spin_lock_irqsave(
							 &lun_entry->lun_lock,
							 cpu_flags);

					lun_entry->lun_state = LS_LUN_READY;

					spin_unlock_irqrestore(
							      &lun_entry->lun_lock,
							      cpu_flags);
				}
			}

			/* Send out any commands that we're holding */
			qla4xxx_start_io(ha);
		}
		else {	/* Device went away, try to relogin */
			/* Mark device missing */
			if (atomic_read(&ddb_entry->state) == DEV_STATE_ONLINE) {
				qla4xxx_mark_device_missing(ha, ddb_entry);
			}

			/* Relogin if device state changed to a not active state.
			 * However, do not relogin if this aen is a result of
			 * an IOCTL logout (DF_NO_RELOGIN) or if this is a
			 * discovered device*/
			if ((ddb_entry->fw_ddb_device_state == DDB_DS_SESSION_FAILED) &&
			    (!test_bit(DF_RELOGIN, &ddb_entry->flags)) &&
			    (!test_bit(DF_NO_RELOGIN, &ddb_entry->flags)) &&
			    (!test_bit(DF_ISNS_DISCOVERED, &ddb_entry->flags))) {
				qla4xxx_reset_relogin_timer(ha, ddb_entry);
			}
		}
		status = QLA_SUCCESS;

	} /* valid ddb_entry */

	LEAVE("qla4xxx_process_ddb_changed");
	return(status);
}

/**************************************************************************
 * qla4xxx_process_aen
 *	This routine processes Asynchronous Events received from the firmware.
 *
 * Input:
 * 	ha - Pointer to host adapter structure.
 *	flush_ddb_chg_aens - 1 = Ignore ddb changed aens
 *
 * Returns:
 *	None
 *
 * Context:
 *	Kernel context.
 **************************************************************************/
void
qla4xxx_process_aen(scsi_qla_host_t *ha, uint8_t flush_ddb_chg_aens)
{
	uint32_t mbox_sts[MBOX_AEN_REG_COUNT];
	aen_t   *aen;
	int     i;

	ENTER("qla4xxx_process_aen");

	while (ha->aen_out != ha->aen_in) {
		spin_lock(&ha->aen_q_spinlock);

		/* Advance pointers for next entry */
		if (ha->aen_out == (MAX_AEN_ENTRIES - 1))
			ha->aen_out = 0;
		else
			ha->aen_out++;

		ha->aen_q_count++;
		aen = &ha->aen_q[ha->aen_out];

		/* copy aen information to local structure */
		for (i=0; i < MBOX_AEN_REG_COUNT; i++)
			mbox_sts[i] = aen->mbox_sts[i];

		spin_unlock(&ha->aen_q_spinlock);
		switch (mbox_sts[0]) {
		case MBOX_ASTS_DATABASE_CHANGED:
			if (flush_ddb_chg_aens) {
				QL4PRINT(QLP2,
					 printk(KERN_INFO "scsi%d: aen[%d] %04x, index [%d] "
						"state=%04x IGNORED!\n", ha->host_no,
						ha->aen_out, mbox_sts[0],
						mbox_sts[2], mbox_sts[3]));
				break;
			}

			QL4PRINT(QLP2|QLP7,
				 printk(KERN_INFO "scsi%d: aen[%d] %04x, index [%d] "
					"state=%04x\n", ha->host_no, ha->aen_out,
					mbox_sts[0], mbox_sts[2], mbox_sts[3]));

			if (mbox_sts[1] == 1) {	/* applies to specific device */
				qla4xxx_process_ddb_changed(ha, mbox_sts[2], mbox_sts[3]);
			}
			else if (mbox_sts[1] == 0) { /* applies to all devices */
				//FIXME: ?
				/* We should never really get here ...*/
				QL4PRINT(QLP2|QLP7,
					 printk("scsi%d: %s: global datbase changed"
						" aen, mbox_sts[1]=%04x\n",
						ha->host_no, __func__,
						mbox_sts[1]));
			}
			break;
		}
	}

	LEAVE("qla4xxx_process_aen");
}

/**************************************************************************
 * qla4xxx_intr_handler
 *	This routine handles the H/W interrupt
 *
 * Input:
 *	irq - Unused
 *	dev_id - Pointer to host adapter structure
 *	regs - Unused
 *
 * Returns:
 *	None
 *
 * Context:
 *	Interrupt context.
 **************************************************************************/
void
qla4xxx_intr_handler(int irq, void *dev_id, struct pt_regs *regs)
{
	scsi_qla_host_t *ha;
	uint32_t intr_status;
	unsigned long flags = 0;
	uint8_t reqs_count = 0;

	DBG(ENTER("qla4xxx_intr_handler");)
	ha = (scsi_qla_host_t *) dev_id;
	if (!ha) {
		QL4PRINT(QLP2,
			 printk(KERN_INFO "qla4xxx: Interrupt with NULL host ptr\n"));
		return;
	}

	ha->isr_count++;

	/*
	 * Check for pending interrupts
	 */
	spin_lock_irqsave(&ha->hardware_lock, flags);

	/*
	 * Repeatedly service interrupts up to a maximum of
	 * MAX_REQS_SERVICED_PER_INTR
	 */
	while (1) {
		/*
		 * Read interrupt status
		 */
		#ifdef QLA4000
		intr_status = RD_REG_DWORD(&ha->reg->interruptStatusOut);
		#endif

		#ifdef QLA4010
		if (RD_DMA_DWORD(ha, &ha->dma_regsv->responseQueueInPointer)
		    != ha->response_out) {
			DBG(QL4PRINT(QLP9, printk("scsi%d: %s response_out = %08xh\n",
						  ha->host_no, __func__,
						  ha->response_out));)
			intr_status = ISO_RSP_QUEUE_COMPLETE;
		}
		else {
			intr_status = RD_REG_DWORD(&ha->reg->ISPControlStatus);
		}

		/*
		 * Service interrupt
		 */
		if (intr_status & CSR_SCSI_RESET_INTR) {
			QL4PRINT(QLP3,
				 printk(KERN_INFO "scsi%d: Soft Reset requested by "
					"Network function or RISC\n", ha->host_no));

			clear_bit(AF_ONLINE, &ha->flags);
			__qla4xxx_disable_intrs(ha);

			QL4PRINT(QLP3,
				 printk(KERN_INFO "scsi%d: Clear SCSI Reset Interrupt\n",
					ha->host_no));
			WRT_REG_DWORD(&ha->reg->ISPControlStatus, CSRW_SCSI_RESET_INTR);

			set_bit(DPC_RESET_HA_INTR, &ha->dpc_flags);
			set_bit(AF_SCHEDULE_DPC, &ha->flags);

			spin_unlock_irqrestore(&ha->hardware_lock, flags);
			goto exit_intr_hndlr_call_dpc;
		}
		else if (intr_status & CSR_FATAL_ERROR) {
			QL4PRINT(QLP2,
				 printk(KERN_INFO "scsi%d: Fatal Error, "
					"Status 0x%04x\n", ha->host_no,
					RD_REG_DWORD(&ha->reg->portFatalErrorStatus)));


			#if STOP_ON_FALTAL_ERROR
			qla4xxx_panic("FATAL ERROR", ha);
			#endif

			QL4PRINT(QLP2,
				 printk(KERN_INFO "scsi%d: Dump Registers:\n", ha->host_no));
			__dump_registers(QLP2, ha);

			/* Issue Soft Reset to clear this error condition.
			 * This will prevent the RISC from repeatedly
			 * interrupting the driver; thus, allowing the DPC to
			 * get scheduled to continue error recovery.
			 * NOTE: Disabling RISC interrupts does not work in
			 * this case, as CSR_FATAL_ERROR overrides
			 * CSR_SCSI_INTR_ENABLE */
			if ((RD_REG_DWORD(&ha->reg->ISPControlStatus) & CSR_SCSI_RESET_INTR) == 0) {
				QL4PRINT(QLP2,
					 printk(KERN_INFO "scsi%d: Issue soft reset\n",
						ha->host_no));
				WRT_REG_DWORD(&ha->reg->ISPControlStatus, CSRW_SOFT_RESET);
			}

			QL4PRINT(QLP2,
				 printk(KERN_INFO "scsi%d: Acknowledge fatal error\n",
					ha->host_no));
			WRT_REG_DWORD(&ha->reg->ISPControlStatus, CSRW_FATAL_ERROR);

			__qla4xxx_disable_intrs(ha);

			set_bit(DPC_RESET_HA, &ha->dpc_flags);
			set_bit(AF_SCHEDULE_DPC, &ha->flags);

			spin_unlock_irqrestore(&ha->hardware_lock, flags);
			goto exit_intr_hndlr_call_dpc;
		}
		else
		#endif	/* #ifdef QLA4010 */
			if (intr_status & INTR_PENDING) {
			qla4xxx_interrupt_service_routine(ha, intr_status);
			ha->total_io_count++;
			if (++reqs_count == MAX_REQS_SERVICED_PER_INTR) {
				QL4PRINT(QLP11,
					 printk("scsi%d: %s: exiting, %d "
						"requests serviced\n",
						ha->host_no, __func__,
						reqs_count));
				break;
			}
			intr_status = 0;
		}
		else {
			if (reqs_count == 0)
				ha->spurious_int_count++;
			break;
		}
	}

	/*
	 * WORKAROUND: Memory-mapped register Writes sometimes get cached
	 * in system memory, so do a register Read to flush out the Write.
	 */
	#ifdef MEMORY_MAPPED_IO
	if (ha->spurious_int_count) {
		#ifdef QLA4000
		RD_REG_DWORD(&ha->reg->interruptStatusOut);
		#endif
		#ifdef QLA4010
		RD_REG_DWORD(&ha->reg->ISPControlStatus);
		#endif

	}
	#endif

	spin_unlock_irqrestore(&ha->hardware_lock, flags);
#if 1 || defined(__VMKERNEL_MODULE__)
        /* 
         * Need to complete requests we put on the srb_done_q to fix
         * deadlock issue between error handler and irq handler...
         */
        if (!list_empty(&ha->done_srb_q)) {
                srb_t    *srb;
                while ((srb = del_from_done_srb_q_head(ha)) != NULL)
                        qla4xxx_complete_request(ha, srb);
        }
#endif
	qla4xxx_start_io(ha);

#ifdef QLA4010
	exit_intr_hndlr_call_dpc:
#endif

	QL4PRINT(QLP11, printk("scsi%d: %s: ha->flags = 0x%08x\n",
			       ha->host_no, __func__, ha->flags));

	if (test_bit(AF_SCHEDULE_DPC, &ha->flags)  &&
	    !test_bit(AF_DPC_SCHEDULED, &ha->flags)  &&
	    !ha->dpc_active && ha->dpc_wait) {
		QL4PRINT(QLP2, printk("scsi%d: %s: scheduling dpc routine\n",
				      ha->host_no, __func__));
		set_bit(AF_DPC_SCHEDULED, &ha->flags);
		up(ha->dpc_wait);
	}

	LEAVE("qla4xxx_intr_handler");
}

inline int
qla4xxx_ok2relogin(scsi_qla_host_t *ha, ddb_entry_t *ddb_entry)
{
#if 1
	if (atomic_read(&ddb_entry->retry_relogin_timer) == INVALID_ENTRY) {
		return(FALSE);
	}
	else if (atomic_read(&ddb_entry->retry_relogin_timer) == 0) {
		atomic_set(&ddb_entry->retry_relogin_timer, INVALID_ENTRY);
		return(TRUE);
	}
	else {
		atomic_dec(&ddb_entry->retry_relogin_timer);
		return(FALSE);
	}
#else
	/*
	 * This method uses the default_time2wait value from the get_ddb
	 * mailbox command to determine if we can relogin.
	 * (We experienced some problems in DVT with this new method.
	 *  Disable it until we can do more testing.)
	 */
	qla4xxx_get_ddb_entry(ha, ddb_entry->fw_ddb_index, NULL, NULL, NULL,
			      &ddb_entry->fw_ddb_device_state,
			      &ddb_entry->default_time2wait, NULL, NULL);
	if (ddb_entry->default_time2wait == 0) {
		QL4PRINT(QLP7, printk("scsi%d: %s: index [%d] state %04x\n",
				      ha->host_no, __func__,
				      ddb_entry->fw_ddb_index,
				      ddb_entry->fw_ddb_device_state));

		return(TRUE);
	}
	else
		return(FALSE);
#endif
}

/**************************************************************************
 * qla4xxx_timer
 *	This routine is scheduled to be invoked every second to search for
 *	work to do.
 *
 * Input:
 * 	p - Pointer to host adapter structure.
 *
 * Returns:
 *	None
 *
 * Context:
 *	Interrupt context.
 **************************************************************************/
void
qla4xxx_timer(unsigned long p)
{
	scsi_qla_host_t *ha = (scsi_qla_host_t *) p;
	struct list_head *ptr, *next; /* used for traversing lists */
	ddb_entry_t *ddb_entry;
	unsigned  long flags = 0;

#if ISP_RESET_TEST
	if (ha->isp_reset_timer++ == (60 *3)) {
		printk("scsi%d: %s going to schedule BIG HAMMER\n",
		       ha->host_no, __func__);

		set_bit(DPC_RESET_HA, &ha->dpc_flags);
		set_bit(AF_SCHEDULE_DPC, &ha->flags);
		ha->isp_reset_timer = 0;
	}
#endif

	QL4PRINT(QLP6,
		 printk("scsi%d: %s: A%d=%d/%d [%x,%x,%x] <%d,%d> {%d,%d,%d,%d} %d\n",
			ha->host_no, __func__,
			ha->instance, ha->spurious_int_count, (uint32_t)ha->isr_count,
			ha->flags, ha->dpc_flags, ha->isns_flags,
			ha->aborted_io_count, ha->mailbox_timeout_count,
			MAX_AEN_ENTRIES-ha->aen_q_count, ha->pending_srb_q_count,
			ha->retry_srb_q_count, ha->active_srb_count,
			ha->seconds_since_last_intr));

	/* Do we need to process the retry queue? */
	if (!list_empty(&ha->retry_srb_q)) {
		set_bit(AF_SCHEDULE_DPC, &ha->flags);
	}

	/* LUN suspension */
	if (!list_empty(&ha->suspended_lun_q)) {
		unsigned long cpu_flags;
		QL4PRINT(QLP2, printk("scsi%d: %s: Going to process suspend_lun_q\n",
				      ha->host_no, __func__));

		spin_lock_irqsave(&ha->adapter_lock, flags);
		list_for_each_safe(ptr, next, &ha->suspended_lun_q)
		{
			lun_entry_t *lun_entry;

			lun_entry = list_entry(ptr, lun_entry_t, list_entry);
			spin_lock_irqsave(&lun_entry->lun_lock, cpu_flags);

			if (lun_entry->lun_state == LS_LUN_SUSPENDED &&
			    atomic_read(&lun_entry->suspend_timer) != 0) {
				QL4PRINT(QLP2,
					 printk("scsi%d: %s: processing suspend_lun_q "
						"lun=%d suspend_timer=%d "
						"retry_count=%d\n",
						ha->host_no, __func__,
						lun_entry->lun,
						atomic_read(&lun_entry->suspend_timer),
						lun_entry->retry_count));

				if (atomic_dec_and_test(&lun_entry->suspend_timer) != 0) {
					lun_entry->retry_count++;
					if (lun_entry->retry_count ==
					    lun_entry->max_retry_count) {
						QL4PRINT(QLP2,
							 printk("scsi%d: %s: LUN %d "
								"TIMEOUT RETRY_CNT:%d\n",
								ha->host_no,
								__func__,
								lun_entry->lun,
								lun_entry->retry_count));

						lun_entry->lun_state = LS_LUN_TIMEOUT;
						__del_from_suspended_lun_q(ha, lun_entry);
					}
					else {
						QL4PRINT(QLP2,
							 printk("scsi%d: %s: LUN %d "
								"RETRY\n",
								ha->host_no,
								__func__,
								lun_entry->lun));

						lun_entry->lun_state = LS_LUN_RETRY;
						__del_from_suspended_lun_q(ha, lun_entry);
					}
				}
			}
			spin_unlock_irqrestore(&lun_entry->lun_lock, cpu_flags);
		}
		spin_unlock_irqrestore(&ha->adapter_lock, flags);
	}

	/*
	 * Search for relogin's to time-out and port down retry.
	 */
#if 1 || __VMKERNEL_MODULE
	spin_lock_irqsave(&ha->ddb_list_spinlock, flags);
#endif
	list_for_each_safe(ptr, next, &ha->ddb_list)
	{
		uint16_t dev_state;
		ddb_entry = list_entry(ptr, ddb_entry_t, list_entry);

		dev_state = (uint16_t) atomic_read(&ddb_entry->state);

		/* First check to see if the device has exhausted the
		 * port down retry count */
		if (dev_state == DEV_STATE_MISSING) {
			if (atomic_read(&ddb_entry->port_down_timer) == 0)
				continue;

			if (atomic_dec_and_test(&ddb_entry->port_down_timer) != 0) {
				QL4PRINT(QLP2,
					 printk("scsi%d: %s: index [%d] port down "
						"retry count of (%d) secs "
						"exhausted, marking device "
						"DEAD.\n", ha->host_no, __func__,
						ddb_entry->fw_ddb_index,
						ha->port_down_retry_count));

				atomic_set(&ddb_entry->state, DEV_STATE_DEAD);

				QL4PRINT(QLP2,
					 printk(KERN_INFO "scsi%d:%d:%d: %s: index [%d] "
						"marked DEAD\n", ha->host_no,
						ddb_entry->bus, ddb_entry->target, __func__,
						ddb_entry->fw_ddb_index));
			}
		}


		/* Count down time between sending relogins */
		if (!test_bit(DF_RELOGIN, &ddb_entry->flags) &&
		    (atomic_read(&ddb_entry->state) != DEV_STATE_ONLINE) &&
		    qla4xxx_ok2relogin(ha, ddb_entry)) {
			#ifdef DEBUG
			if (ADAPTER_UP(ha)) {
				QL4PRINT(QLP2,
					 printk("scsi%d:%d:%d: index[%d] "
						"schedule relogin\n",
						ha->host_no, ddb_entry->bus,
						ddb_entry->target,
						ddb_entry->fw_ddb_index));
			}
			else {
				QL4PRINT(QLP2,
					 printk("scsi%d:%d:%d: index[%d] "
						"relogin will occur after adapter is online.\n",
						ha->host_no, ddb_entry->bus,
						ddb_entry->target,
						ddb_entry->fw_ddb_index));
			}
			#endif
#if 1 || __VMKERNEL_MODULE__
                        /*
                         * Make sure ddb entries are set first, in
                         * case DPC is already running.
                         */
                        set_bit(DF_RELOGIN, &ddb_entry->flags);
                        set_bit(DPC_RELOGIN_DEVICE, &ha->dpc_flags);
#else
			set_bit(DPC_RELOGIN_DEVICE, &ha->dpc_flags);
			set_bit(DF_RELOGIN, &ddb_entry->flags);
#endif
			set_bit(AF_SCHEDULE_DPC, &ha->flags);
		}

		/* Wait for relogin to timeout */
		if (atomic_read(&ddb_entry->relogin_timer)  &&
		    (atomic_dec_and_test(&ddb_entry->relogin_timer) != 0)) {
			/*
			 * If the relogin times out and the device is
			 * still MISSING then try and relogin again.
			 */
			if (atomic_read(&ddb_entry->state) != DEV_STATE_ONLINE &&
			    ddb_entry->fw_ddb_device_state == DDB_DS_SESSION_FAILED) {
				QL4PRINT(QLP2, printk(KERN_INFO "scsi%d:%d:%d: index[%d] "
						      "relogin timed out-"
						      "retrying relogin (%d)\n",
						      ha->host_no, ddb_entry->bus,
						      ddb_entry->target,
						      ddb_entry->fw_ddb_index,
						      atomic_read(&ddb_entry->relogin_retry_count)));

				set_bit(AF_SCHEDULE_DPC, &ha->flags);
				qla4xxx_reset_relogin_timer(ha, ddb_entry);
			}
		}
	}
#if 1 || __VMKERNEL_MODULE
	spin_unlock_irqrestore(&ha->ddb_list_spinlock, flags);
#endif

#if EH_WAKEUP_WORKAROUND
	/*
	 * Check for kernel wakeup error
	 */
	if (ha->host->in_recovery &&
	    (HOST_BUSY(ha) == ha->host->host_failed) &&
	    !ha->host->eh_active) {
		if ((ha->eh_start++) == 60) {
			if (ha->host->eh_wait)
				up(ha->host->eh_wait);
			ha->eh_start=0;

			QL4PRINT(QLP2, printk(KERN_INFO "scsi%d: !!! Waking up error "
					      "handler for scsi layer\n", ha->host_no));
		}
	}
#endif /* EH_WAKEUP_WORKAROUND */

	/*
	 * Check for heartbeat interval
	 */
	if ((ha->firmware_options & FWOPT_HEARTBEAT_ENABLE) &&
	    (ha->heartbeat_interval != 0)) {
		ha->seconds_since_last_heartbeat ++;

		if (ha->seconds_since_last_heartbeat > ha->heartbeat_interval+2) {
			QL4PRINT(QLP2,
				 printk(KERN_INFO "scsi%d: Heartbeat not "
					"received for %d seconds.  "
					"HeartbeatInterval = %d seconds. "
					"Scheduling SOFT RESET.\n",
					ha->host_no,
					ha->seconds_since_last_heartbeat,
					ha->heartbeat_interval));

			set_bit(DPC_RESET_HA, &ha->dpc_flags);
			set_bit(AF_SCHEDULE_DPC, &ha->flags);
		}
	}

#ifdef QLA4010
	/*
	 * Check for iSNS actions
	 */
	if (test_bit(ISNS_FLAG_RESTART_SERVICE, &ha->isns_flags)) {
		if (atomic_read(&ha->isns_restart_timer)) {
			if ((atomic_dec_and_test(&ha->isns_restart_timer) == 0) &&
			    test_bit(ISNS_FLAG_ISNS_SRV_ENABLED, &ha->isns_flags) &&
			    !IPAddrIsZero(ha->isns_ip_address) &&
			    ha->isns_server_port_number) {
				clear_bit(ISNS_FLAG_RESTART_SERVICE, &ha->isns_flags);

				set_bit(DPC_ISNS_RESTART_COMPLETION, &ha->dpc_flags);
				set_bit(AF_SCHEDULE_DPC, &ha->flags);
			}
		}
		else {
			clear_bit(ISNS_FLAG_RESTART_SERVICE, &ha->isns_flags);
		}
	}
#endif

	/* Wakeup the dpc routine for this adapter, if needed */
	if (test_bit(AF_SCHEDULE_DPC, &ha->flags)  &&
	    !test_bit(AF_DPC_SCHEDULED, &ha->flags) &&
	    !ha->dpc_active && ha->dpc_wait) {
		QL4PRINT(QLP2, printk("scsi%d: %s: scheduling dpc routine\n",
				      ha->host_no, __func__));
		set_bit(AF_DPC_SCHEDULED, &ha->flags);
		up(ha->dpc_wait);
	}

	/* Reschedule timer thread to call us back in one second */
	ha->timer.expires  = jiffies + HZ;
	add_timer(&ha->timer);

	ha->seconds_since_last_intr ++;
}

/**************************************************************************
 * qla4xxx_do_dpc
 *	This routine is a task that is schedule by the interrupt handler
 *	to perform the background processing for interrupts.  We put it
 *      on a task queue that is consumed whenever the scheduler runs; that's
 *      so you can do anything (i.e. put the process to sleep etc).  In fact,
 *      the mid-level tries to sleep when it reaches the driver threshold
 *      "host->can_queue". This can cause a panic if we were in our interrupt
 *      code.
 *
 * Input:
 * 	p - Pointer to host adapter structure.
 *
 * Returns:
 *	None
 *
 * Context:
 *	Kernel context.
 **************************************************************************/
void
qla4xxx_do_dpc(unsigned long p)
{
	DECLARE_MUTEX_LOCKED(sem);
	scsi_qla_host_t *ha = (scsi_qla_host_t *) p;
#if 1 || __VMKERNEL_MODULE__
	struct list_head *ptr, *next;
#else
	static struct list_head *ptr, *next;
#endif

#if 1 || __VMKERNEL_MODULE__
	/*
         * The same variable name is declared twice using different
	 * scopes in this function.  VMware fixes to the driver have
	 * eliminated the need for this declaration, so it's removed
	 * to avoid future confusion.
	 */
#else
	static ddb_entry_t *ddb_entry;
#endif

	ENTER("qla4xxx_do_dpc");

#ifdef MODULE
	siginitsetinv(&current->blocked, SHUTDOWN_SIGS);
#else
	siginitsetinv(&current->blocked, 0);
#endif

	lock_kernel();

	/* Flush resources */
	daemonize();

#if LINUX_VERSION_CODE >= KERNEL_VERSION(2,4,9)

	/* As mentioned in kernel/sched.c(RA).....
	 * Reparent the calling kernel thread to the init task.
	 *
	 * If a kernel thread is launched as a result of a system call,
	 * or if it ever exists,it should generally reparent itself to init
	 * so that it is correctly cleaned up on exit.
	 *
	 * The various task state such as scheduling policy and priority
	 * may have been inherited from a user process, so we reset them
	 * to sane values here.
	 *
	 * NOTE that reparent_to_init() gives the caller full capabilities.
	 *
	 */
	reparent_to_init();
#endif

	/*
	 * Set the name of this process.
	 */
	sprintf(current->comm, "scsi%d: %s:", ha->host_no, __func__);
	ha->dpc_wait = &sem;
	ha->dpc_handler = current;

	unlock_kernel();

	/*
	 * Wake up the thread that created us.
	 */
	QL4PRINT(QLP7, printk("scsi%d: %s: Wake up parent %d\n",
			      ha->host_no, __func__,
			      ha->dpc_notify->count.counter));
	up(ha->dpc_notify);

	while (1) {
		QL4PRINT(QLP2|QLP12, printk("scsi%d: %s: DPC handler "
					    "sleeping *****************\n",
					    ha->host_no, __func__));
		down_interruptible(&sem);

		/*
		 * If we get a signal, it means we are supposed to go
		 * away and die.  This typically happens if the user is
		 * trying to unload a module.
		 */

#ifdef __VMKERNEL_MODULE__
                // we still need this flag until signal_pending() 
                // is emulated
		if (ha->should_die == TRUE)
			break;	/* get out */
#else
		if (signal_pending(current))
			break;	 /* get out */
#endif //__VMKERNEL_MODULE__

		QL4PRINT(QLP2|QLP12, printk("scsi%d: %s: DPC handler "
					    "waking up ****************\n",
					    ha->host_no, __func__));

		QL4PRINT(QLP2, printk("scsi%d: %s: ha->flags = 0x%08x\n",
				      ha->host_no, __func__, ha->flags));

		QL4PRINT(QLP2, printk("scsi%d: %s: ha->dpc_flags = 0x%08x\n",
				      ha->host_no, __func__, ha->dpc_flags));

		if (ha->dpc_active)
			continue;

		ha->dpc_active = 1;
		clear_bit(AF_DPC_SCHEDULED, &ha->flags);
		clear_bit(AF_SCHEDULE_DPC, &ha->flags);

		/*
		 * Determine what action is necessary
		 */

		/* ---- recover adapter? --- */
		QL4PRINT(QLP3, printk("scsi%d: %s: Do we need to "
				      "recover the adapter?\n",
				      ha->host_no, __func__));

		if (ADAPTER_UP(ha) ||
		    (test_bit(DPC_RETRY_RESET_HA, &ha->dpc_flags) &&
		     test_bit(DPC_RESET_HA, &ha->dpc_flags)) ||
		    test_bit(DPC_RESET_HA_INTR, &ha->dpc_flags) ||
		    test_bit(DPC_RESET_HA_DESTROY_DDB_LIST, &ha->dpc_flags)) {
			if (test_bit(DPC_RESET_HA_DESTROY_DDB_LIST, &ha->dpc_flags))
				qla4xxx_recover_adapter(ha, REBUILD_DDB_LIST);
			if (test_bit(DPC_RESET_HA, &ha->dpc_flags))
				qla4xxx_recover_adapter(ha, PRESERVE_DDB_LIST);
#ifdef QLA4010
			if (test_bit(DPC_RESET_HA_INTR, &ha->dpc_flags)) {
				uint8_t wait_time = RESET_INTR_TOV;

				qla4xxx_flush_active_srbs(ha);
				while ((RD_REG_DWORD(&ha->reg->portStatus) &
					PSR_INIT_COMPLETE) == 0) {
					if (wait_time-- == 0)
						break;

					set_current_state(TASK_INTERRUPTIBLE);
					schedule_timeout(1 * HZ);
				}

				if (wait_time == 0)
					QL4PRINT(QLP2,
						 printk("scsi%d: %s: IC bit not set\n",
							ha->host_no, __func__));

				qla4xxx_initialize_adapter(
							  ha,
							  PRESERVE_DDB_LIST);
				clear_bit(DPC_RESET_HA_INTR, &ha->dpc_flags);
			}
#endif

		}

		/* ---- process AEN? --- */
		QL4PRINT(QLP3, printk("scsi%d: %s: Do we need to "
				      "process an AEN?\n",
				      ha->host_no, __func__));

		if (ha->aen_in != ha->aen_out)
			qla4xxx_process_aen(ha, 0);

		/* ---- relogin device? --- */
		QL4PRINT(QLP3, printk("scsi%d: %s: Do we need to "
				      "relogin a device?\n",
				      ha->host_no, __func__));

		if (ADAPTER_UP(ha) &&
		    test_and_clear_bit(DPC_RELOGIN_DEVICE, &ha->dpc_flags)) {
#if 1 || __VMKERNEL_MODULE__
                        qla4xxx_relogin_ddb_list(ha);
#else
			list_for_each_safe(ptr, next, &ha->ddb_list)
			{
				ddb_entry = list_entry(ptr,
						       ddb_entry_t,
						       list_entry);

				if (test_and_clear_bit(DF_RELOGIN,
						       &ddb_entry->flags)) {
					qla4xxx_relogin_device(ha, ddb_entry);
				}
			}
#endif
		}

		#ifdef QLA4010
		/* ---- restart iSNS server? --- */
		QL4PRINT(QLP3, printk("scsi%d: %s: Do we need to "
				      "restart the iSNS server?\n",
				      ha->host_no, __func__));

		if (ADAPTER_UP(ha) &&
		    test_and_clear_bit(DPC_ISNS_RESTART, &ha->dpc_flags)) {
			qla4xxx_isns_restart_service(ha);
		}

		if (ADAPTER_UP(ha) &&
		    test_and_clear_bit(DPC_ISNS_RESTART_COMPLETION, &ha->dpc_flags)) {
			uint32_t ip_addr = 0;
			IPAddr2Uint32(ha->isns_ip_address, &ip_addr);

			if (qla4xxx_isns_restart_service_completion(ha,
								    ip_addr,
								    ha->isns_server_port_number)
			    != QLA_SUCCESS) {
				QL4PRINT(QLP2|QLP6,
					 printk(KERN_WARNING "scsi%d: %s: "
						"restart service failed\n",
						ha->host_no, __func__));
			}
		}
		#endif

		/* ---- return cmds on retry_q? --- */
		QL4PRINT(QLP3, printk("scsi%d: %s: Do we need to "
				      "return cmds on the retry_q?\n",
				      ha->host_no, __func__));

		if (!list_empty(&ha->retry_srb_q)) {
			srb_t    *srb;
			unsigned long flags;

			spin_lock_irqsave(&ha->list_lock, flags);

			QL4PRINT(QLP2,
				 printk("scsi%d: %s: found %d srbs "
					"in retry_srb_q \n",
					ha->host_no, __func__,
					ha->retry_srb_q_count));

			list_for_each_safe(ptr, next, &ha->retry_srb_q)
			{
				ddb_entry_t *ddb_entry;
				lun_entry_t *lun_entry;

				srb = list_entry(ptr, srb_t, list_entry);
				ddb_entry = qla4xxx_lookup_ddb_by_fw_index(ha, srb->fw_ddb_index);
#if 1 || __VMKERNEL_MODULE__
				/*
				 * Check to make sure the ddb_entry lookup
				 * above was successful before using
				 * the ddb_entry.
				 */
				if (!(ddb_entry))
					continue;
#endif
				lun_entry = qla4xxx_lookup_lun_handle(ha, ddb_entry, srb->lun);

				if ((lun_entry) &&
				    (lun_entry->lun_state == LS_LUN_SUSPENDED))
					continue;

				if ((ddb_entry) &&
				    (atomic_read(&ddb_entry->state) ==
				     DEV_STATE_DEAD)) {
					QL4PRINT(QLP2,
						 printk("scsi%d: %s: found srb %p "
							"in retry_srb_q, "
							"Device DEAD, returning\n",
							ha->host_no, __func__,
							srb));

					__del_from_retry_srb_q(ha, srb);
					CMD_RESULT(srb->cmd) =
					(int) (DID_NO_CONNECT << 16);
					add_to_done_srb_q(ha,srb);
				}

				/*
				 * Send requests to OS when device goes ONLINE
				 * so that the OS will retry them via I/O thread.
				 * We don't want to issue I/O via recovery thread.
				 */
				if (ADAPTER_UP(ha) &&
				    (atomic_read(&ddb_entry->state)
				     == DEV_STATE_ONLINE)) {
					QL4PRINT(QLP2,
						 printk("scsi%d: %s: found srb %p "
							"in retry_srb_q, "
							"Device ONLINE, returning\n",
							ha->host_no, __func__,
							srb));

					__del_from_retry_srb_q(ha, srb);
					CMD_RESULT(srb->cmd) =
					(int) (DID_BUS_BUSY << 16);
					add_to_done_srb_q(ha,srb);
				}
			}
			spin_unlock_irqrestore(&ha->list_lock, flags);

			while ((srb = del_from_done_srb_q_head(ha)) != NULL)
				qla4xxx_complete_request(ha, srb);
		}

		if (test_and_clear_bit(DPC_IOCTL_ERROR_RECOVERY, &ha->dpc_flags)) {
			qla4xxx_ioctl_error_recovery(ha);
		}

		ha->dpc_active = 0;
	}

	QL4PRINT(QLP7, printk("scsi%d: %s: Wake up, we're leaving\n",
			      ha->host_no, __func__));

	/*
	* Make sure that nobody tries to wake us up again.
	*/
	ha->dpc_wait = NULL;
	ha->dpc_handler = NULL;
	ha->dpc_active = 0;

	/*
	* If anyone is waiting for us to exit (i.e. someone trying to unload
	* a driver), then wake up that process to let them know we are on
	* the way out the door.  This may be overkill - I *think* that we
	* could probably just unload the driver and send the signal, and when
	* the error handling thread wakes up that it would just exit without
	* needing to touch any memory associated with the driver itself.
	*/
	if (ha->dpc_notify != NULL)
		up(ha->dpc_notify);

	LEAVE("qla4xxx_do_dpc");
}

#if 1 || __VMKERNEL_MODULE__
/**************************************************************************
 *
 * qla4xxx_relogin_ddb_list
 *     Relogin to any devices in the ddb that don't have current
 *     sessions
 *
 * Input:
 *      ha - Pointer to the host adapter struct
 *
 * Returns:
 *      None
 *
 * Context:
 *      Kerenl context.
 *
 *************************************************************************/
static void
qla4xxx_relogin_ddb_list(scsi_qla_host_t *ha)
{
	ddb_entry_t ddbLocEnt, *ddb_entry = NULL;
	int i;
	unsigned long flags;
	struct list_head *ptr, *next;
	uint16_t relogin_timer;

	/*
	 * If the ddb_list is empty, there's a chance that the link
	 * was dead when the adapter was initialized or reset.  Try
	 * initializing the ddb_list again in this case.
	 */

	if (list_empty(&ha->ddb_list)) {
		qla4xxx_wake_the_dead(ha);
		clear_bit(DPC_RELOGIN_DEVICE, &ha->dpc_flags);
		return;
	}

	spin_lock_irqsave(&ha->ddb_list_spinlock, flags);

	/*
         * Mark each ddb entry as no relogin attempted
         */
	list_for_each_safe(ptr, next, &ha->ddb_list) {
		ddb_entry = list_entry(ptr, ddb_entry_t, list_entry);
                ddb_entry->relogin_done = 0;
	}
        spin_unlock_irqrestore(&ha->ddb_list_spinlock, flags);

        /* Process at most MAX_TARGETS entries to avoid an infinite
         * loop. In reality we will never pass this, but even if we
         * do the worst case is that we will process any remaining
         * targets next time the DPC runs.
         */
        for (i=0; i < MAX_TARGETS; i++) {
		ddb_entry = NULL;
		spin_lock_irqsave(&ha->ddb_list_spinlock, flags);
                list_for_each_safe(ptr, next, &ha->ddb_list) {
			ddb_entry = list_entry(ptr, ddb_entry_t, list_entry);
                        if (test_and_clear_bit(DF_RELOGIN, &ddb_entry->flags)
				&& !ddb_entry->relogin_done) {
                                    /* Deal with this entry outside loop */
                                    ddb_entry->relogin_done = 1;
				    /*
				     * set relogin_timer here since we
				     * pass a copy of this struct to 
				     * qla4xxx_relogin_device()
				     */
				    relogin_timer =
					MAX(ddb_entry->default_relogin_timeout,
					RELOGIN_TOV);
				    atomic_set(&ddb_entry->relogin_timer,
				    	relogin_timer);
				    ddbLocEnt = *ddb_entry;
                                    break;
                         }
                         ddb_entry = NULL; /* Use it as a flag */
		}
                spin_unlock_irqrestore(&ha->ddb_list_spinlock, flags);
                if (ddb_entry) {
			/* relogin fails if the device is already logged in */
			qla4xxx_relogin_device(ha, &ddbLocEnt);
                } else {
			break; /* Did not find anything, so we are done */
                }
	}
}

/**************************************************************************
 *
 * qla4xxx_wake_the_dead
 *     Relogin to any firmware entries that don't have active sessions
 *     (Useful bits of code were ripped from the qla4xxx_build_ddb_list().)
 *     We aren't interested in creating new entries for existing devices,
 *     and qla4xxx_add_devices_dynamically() takes care of any
 *     interesting setup that has to happen if we acutally find a
 *     target at a previously dead entry when the AEN comes in.
 *
 * Input:
 *      ha - Pointer to the host adapter struct
 *
 * Returns:
 *      None
 *
 * Context:
 *      Kernel context.
 *
 *************************************************************************/
static void
qla4xxx_wake_the_dead(scsi_qla_host_t *ha)
{
	uint32_t        fw_ddb_index = 0;
	uint32_t        next_fw_ddb_index = 0;
	uint32_t        ddb_state;
	ddb_entry_t     *ddb_entry;

	for (fw_ddb_index = 0;
	    fw_ddb_index < MAX_DDB_ENTRIES;
	    fw_ddb_index = next_fw_ddb_index) {
		if (qla4xxx_get_ddb_entry(ha, fw_ddb_index, NULL, NULL,
					  &next_fw_ddb_index, &ddb_state,
					  NULL, NULL, NULL)
		    == QLA_ERROR) {
			QL4PRINT(QLP2,
				 printk("scsi%d: %s: get_ddb_entry, "
					"fw_ddb_index %d failed",
					ha->host_no, __func__, fw_ddb_index));
			goto exit_wake_the_dead;
		}

		if (ddb_state == DDB_DS_SESSION_FAILED || 
			ddb_state == DDB_DS_NO_CONNECTION_ACTIVE || 
                         ddb_state == DDB_DS_NO_SESSION_ACTIVE) {
		    ddb_entry =
		    	qla4xxx_lookup_ddb_by_fw_index(ha, fw_ddb_index);

		    /* Check to see if the device is already in our database */
		    if (ddb_entry != NULL) {  /* device exists */
		        /*
		         * This is a target that used to be alive but isn't any
		         * more.  We should remove it during a rescan,
			 * but this introcduces target renumbering
			 * problems where a new target will potentially have
			 * the same target number as an existing path. 
                         * Do nothing for now.
			 *
                         * qla4xxx_free_ddb(ha, ddb_entry);
			 *
		         */

		    }
		    QL4PRINT(QLP7,
				 printk("scsi%d: %s: Attempt to login index [%d]\n",
					ha->host_no, __func__,
					fw_ddb_index));
		    qla4xxx_set_ddb_entry(ha, fw_ddb_index, NULL);
		}

		/* We know we've reached the last device when
		 * next_fw_ddb_index is 0
		 */
		if (next_fw_ddb_index == 0)
			break;
	}
    exit_wake_the_dead:
	return;
}
#endif

#if STOP_ON_ERROR
/**************************************************************************
 * qla4xxx_panic
 *	This routine halts the system after a specific event for debug
 *	purposes.
 *
 * Input:
 *	cp - Pointer to string containing name of routine that called us.
 * 	ha - Pointer to host adapter structure.
 *
 * Returns:
 *	None
 *
 * Context:
 *	Kernel context.
 **************************************************************************/
static void
qla4xxx_panic(char *cp, scsi_qla_host_t *ha)
{
	QL4PRINT(QLP1, printk("scsi(%d) - PANIC:  %s\n", ha->host_no, cp));
	QL4PRINT(QLP1, printk("Current time=0x%lx\n", jiffies));
	QL4PRINT(QLP1, printk("Number of pending commands =0x%x\n",
			      ha->pending_srb_q_count));
	QL4PRINT(QLP1, printk("Number of retry commands =0x%x\n",
			      ha->retry_srb_q_count));
	QL4PRINT(QLP1, printk("Number of active commands =0x%x\n",
			      ha->active_srb_count));
	QL4PRINT(QLP1, printk("Number of free entries = (%d)\n",
			      ha->req_q_count));
	QL4PRINT(QLP1, printk("Request Queue @ %08x, Response Queue @ %08xn",
			      ha->req_queue_headp, ha->rsp_queue_headp));
	QL4PRINT(QLP1, printk("Request In Ptr %d\n", ha->request_in));
	QL4PRINT(QLP1, printk("HA flags =0x%x\n", ha->flags));
	QL4PRINT(QLP1, printk("ISP Registers: \n"));
	__dump_registers(QLP1, ha);

	cli();
	while (1) {
		udelay(2); barrier();
	}
	sti();
}
#endif

/**************************************************************************
 * qla4xxx_eh_wait_on_command
 *	This routine waits for the command to be returned by the Firmware
 *	for some max time.
 *
 * Input:
 *    ha = actual ha whose done queue will contain the command
 *	      returned by firmware.
 *    cmd = Scsi Command to wait on.
 *
 * Returns:
 *    Not Found : 0
 *    Found : 1
 *
 * Context:
 *	Kernel context.
 **************************************************************************/
STATIC int
qla4xxx_eh_wait_on_command(scsi_qla_host_t *ha, Scsi_Cmnd *cmd)
{

	int             done = 0;
	srb_t           *rp;
	uint32_t        max_wait_time = EH_WAIT_CMD_TOV;

	ENTER("qla4xxx_eh_wait_on_command");

	do {
		/*Checking to see if its returned to OS */
		rp = (srb_t *) CMD_SP(cmd);
		if (rp == NULL) {
			done++;
			break;
		}

		HOST_SPIN_UNLOCK_IRQ(ha);

		set_current_state(TASK_INTERRUPTIBLE);
		schedule_timeout(2*HZ);

		while ((rp = del_from_done_srb_q_head(ha)) != NULL)
			qla4xxx_complete_request(ha, rp);

		HOST_SPIN_LOCK_IRQ(ha);


	} while ((max_wait_time--));

	if (done)
		QL4PRINT(QLP2, printk("scsi%d: %s: found cmd=%p.\n",
				      ha->host_no, __func__, cmd));

	LEAVE("qla4xxx_eh_wait_on_command");

	return(done);
}

/**************************************************************************
 * WAIT_FOR_ADAPTER_UP
 *	This routine
 *
 * Input:
 *	ha - Pointer to host adapter structure
 *
 * Remarks:
 *
 * Returns:
 *	SUCCESS - Adapter is ONLINE
 *	FAILED  - Adapter is DEAD
 *
 * Context:
 *	Kernel context.	 Assume io_request_lock LOCKED upon entry
 **************************************************************************/
inline uint8_t
WAIT_FOR_ADAPTER_UP(scsi_qla_host_t *ha)
{

#ifndef __VMKERNEL_MODULE__
	while (1) {
		if (ADAPTER_UP(ha))
			return(QLA_SUCCESS);

		if (!ADAPTER_UP(ha) &&
		    (ha->retry_reset_ha_cnt == 0)) {
			QL4PRINT(QLP2, printk("scsi%d: %s: adapter down, retry_reset_ha_cnt = %d\n",
					      ha->host_no, __func__, ha->retry_reset_ha_cnt));
			return(QLA_ERROR);
		}

		HOST_SPIN_UNLOCK_IRQ(ha);

		QL4PRINT(QLP3, printk("scsi%d: %s: adapter down, retry_reset_ha_cnt = %d, delay 2 sec.\n",
				      ha->host_no, __func__, ha->retry_reset_ha_cnt));
		set_current_state(TASK_INTERRUPTIBLE);
		schedule_timeout(2*HZ);

		HOST_SPIN_LOCK_IRQ(ha);
	}
   
#else
	// Always return good status, we do not want to wait
	return(QLA_SUCCESS); 
#endif  // __VMKERNEL_MODULE__
 
}

/**************************************************************************
 * qla4xxx_eh_abort
 *	This routine aborts commands that currently held in the adapter's
 *	internal queues.  Commands that are active are NOT aborted.
 *
 * Input:
 *	cmd - Pointer to Linux's SCSI command structure
 *
 * Remarks:
 * 	Aborts get translated to "device resets" by the scsi switch
 * 	which will return a RESET status and not ABORT. Since the
 * 	mid-level is expecting an ABORT status during an abort(),
 * 	we always elevate to device reset.
 *
 * Returns:
 *	SUCCESS - Successfully aborted non-active command
 *	FAILED  - Command not found, or command currently active
 *
 * Context:
 *	Kernel context.	 io_request_lock LOCKED
 **************************************************************************/
int
qla4xxx_eh_abort(Scsi_Cmnd *cmd)
{
	int return_status = FAILED;
	scsi_qla_host_t *ha;
	srb_t *srb;
	struct list_head *ptr, *next;
	unsigned int flags;

	ENTER("qla4xxx_eh_abort");

	ha = (scsi_qla_host_t *) cmd->host->hostdata;
	srb = (srb_t *) CMD_SP(cmd);

	if (!srb) {
		/* Srb invalid, for some reason, possibly already returned
		 * to the OS */
		QL4PRINT(QLP2, printk("scsi%d:%d:%d: %s: NULL srb %p, "
				      "cmd %p may have already been returned to OS.\n",
				      ha->host_no, cmd->channel,
				      cmd->target, __func__, srb, cmd));

#if 1 || __VMKERNEL_MODULE__
		/*
		 * Saying this function succeeded when no command
		 * could be found is wrong.  It will also produce a
		 * warning in the vmkernel.  Leave the result alone
		 * and return FAILED.
		 */
		return_status = FAILED;
#else
		CMD_RESULT(cmd) = (int) (DID_ABORT << 16);
		return_status = SUCCESS;
#endif
		return(return_status);
	}

	ha->aborted_io_count++;

	/* Print statements
	 * ---------------- */
	QL4PRINT(QLP2,
		 printk(KERN_INFO "scsi%d:%d:%d:%d: abort srb=%p, cmd=%p, state=%s, "
			"r_start=%ld , u_start=%ld\n",
			ha->host_no, cmd->channel, cmd->target, cmd->lun,
			srb, cmd, srb_state_msg[srb->state],srb->r_start,srb->u_start));

	qla4xxx_dump_dwords(QLP10, srb, sizeof(*srb));

	/* If srb found in pending_srb_q, return  the cmd with ABORTED status */
	spin_lock_irqsave(&ha->list_lock, flags);
	list_for_each_safe(ptr, next,  &ha->pending_srb_q)
	{
		srb = list_entry(ptr, srb_t, list_entry);

		if (srb->cmd != cmd)
			continue;

		QL4PRINT(QLP2,
			 printk("scsi%d: %s: srb %p found on pending queue\n",
				ha->host_no, __func__, srb));

		__del_from_pending_srb_q(ha, srb);
		CMD_RESULT(cmd) = DID_ABORT << 16;

		spin_unlock_irqrestore(&ha->list_lock, flags);

		HOST_SPIN_UNLOCK_IRQ(ha);
		qla4xxx_complete_request(ha, srb);
		HOST_SPIN_LOCK_IRQ(ha);

		return_status = SUCCESS;
		goto exit_abort;
	}
	spin_unlock_irqrestore(&ha->list_lock, flags);

	/* If srb found in retry_srb_q, return it with ABORTED status */
	spin_lock_irqsave(&ha->list_lock, flags);
	list_for_each_safe(ptr, next, &ha->retry_srb_q)
	{
		srb = list_entry(ptr, srb_t, list_entry);

		if (srb->cmd != cmd)
			continue;

		QL4PRINT(QLP2,
			 printk("scsi%d: %s: srb %p found on retry queue\n",
				ha->host_no, __func__, srb));

		__del_from_retry_srb_q(ha, srb);
		CMD_RESULT(cmd) = DID_ABORT << 16;

		spin_unlock_irqrestore(&ha->list_lock, flags);
		HOST_SPIN_UNLOCK_IRQ(ha);
		qla4xxx_complete_request(ha, srb);
		HOST_SPIN_LOCK_IRQ(ha);
		return_status = SUCCESS;
		goto exit_abort;
	}
	spin_unlock_irqrestore(&ha->list_lock, flags);

	/* If srb found in done_q, return the cmd with ABORTED status */
	spin_lock_irqsave(&ha->adapter_lock, flags);
	list_for_each_safe(ptr, next, &ha->done_srb_q)
	{
		srb = list_entry(ptr, srb_t, list_entry);

		if (srb->cmd != cmd)
			continue;

		QL4PRINT(QLP2,
			 printk("scsi%d: %s: srb %p found on done queue\n",
				ha->host_no, __func__, srb));

		CMD_RESULT(cmd) = DID_ABORT << 16;
		spin_unlock_irqrestore(&ha->adapter_lock, flags);

		HOST_SPIN_UNLOCK_IRQ(ha);
		while ((srb = del_from_done_srb_q_head(ha)) != NULL)
			qla4xxx_complete_request(ha, srb);
		HOST_SPIN_LOCK_IRQ(ha);

		return_status = SUCCESS;
		goto exit_abort;
	}
	spin_unlock_irqrestore(&ha->adapter_lock, flags);

	/*
	 * Aborts get translated to "device resets" by the scsi switch
	 * which will return a RESET status and not ABORT. Since the
	 * mid-level is expecting an ABORT status during an abort(),
	 * we always elevate to device reset.
	 */
	return_status = FAILED;

	exit_abort:
	QL4PRINT(QLP2, printk("scsi%d: %s: return with status = %x\n",
			      ha->host_no, __func__, return_status));
	LEAVE("qla4xxx_eh_abort");

#if STOP_ON_ABORT
	qla4xxx_panic("qla4xxx_eh_abort", ha);
#endif

	return(return_status);
}

#ifdef QLA4000
/**************************************************************************
 * qla4000_hard_reset
 *	This routine resets the HBA via a write to the HBA Bridge's config
 *      register space (HARD RESET).
 *
 * Input:
 * 	ha - Pointer to host adapter structure.
 *
 * Returns:
 *	QLA_SUCCESS - Successfully reset the adapter
 *	QLA_ERROR   - Failed to reset the adapter
 *
 * Context:
 *	Kernel context.
 **************************************************************************/
uint8_t
qla4000_hard_reset(scsi_qla_host_t *ha)
{
	uint8_t status = QLA_ERROR;

	ENTER("qla4000_hard_reset");
	QL4PRINT(QLP2,
		 printk(KERN_INFO "scsi%d: HARD RESET\n", ha->host_no));

#define ECBR_ADDR		0x40
#define ECBR_RESET_INTERNAL_BUS	0x0020

	if (ha->pci_brgdev) {
		uint16_t ecbr;

		if (pci_read_config_word(ha->pci_brgdev,
					 ECBR_ADDR, &ecbr) == 0) {
			qla4xxx_disable_intrs(ha);

			/* return all active commands to the mid-layer */

			ecbr |= ECBR_RESET_INTERNAL_BUS;
			if (pci_write_config_word(ha->pci_brgdev,
						  ECBR_ADDR, ecbr) == 0) {
				QL4PRINT(QLP2,
					 printk(KERN_INFO "scsi%d: DELAY for 30 "
						"seconds\n", ha->host_no));

				set_current_state(TASK_INTERRUPTIBLE);
				schedule_timeout(30*HZ);
				qla4xxx_flush_active_srbs(ha);

				QL4PRINT(QLP2, printk(KERN_INFO
						      "scsi%d: HARD RESET SUCCESS\n",
						      ha->host_no));
				status = QLA_SUCCESS;
			}
		}
		else {
			QL4PRINT(QLP2,
				 printk(KERN_INFO "scsi%d: HARD RESET failed!\n",
					ha->host_no));
		}
	}

	LEAVE("qla4000_hard_reset");
#if STOP_ON_RESET_ADAPTER
	qla4xxx_panic("qla4000_hard_reset", ha);
#endif
	return(status);
}

/**************************************************************************
 * qla4000_reset_firmware
 *	This routine performs a Firmware (SOFT) RESET.
 *
 * Input:
 * 	ha - Pointer to host adapter structure.
 *
 * Returns:
 *	QLA_SUCCESS - Successfully reset the firmware
 *	QLA_ERROR   - Failed to reset the firmware
 *
 * Context:
 *	Kernel context.
 **************************************************************************/
uint8_t
qla4000_reset_firmware(scsi_qla_host_t *ha)
{
	uint8_t   status = QLA_ERROR;
	uint32_t  mbox_cmd[MBOX_REG_COUNT];
	uint32_t  mbox_sts[MBOX_REG_COUNT];
	uint32_t  i;
	uint32_t  intr_mask;
	unsigned long flags = 0;

	ENTER("qla4000_reset_firmware");
	QL4PRINT(QLP2,
		 printk(KERN_INFO "scsi%d: RESET firmware\n", ha->host_no));

	/* Issue mailbox command, do not wait for completion */
	memset(&mbox_cmd, 0, sizeof(mbox_cmd));
	memset(&mbox_sts, 0, sizeof(mbox_sts));
	mbox_cmd[0] = MBOX_CMD_RESET_FW;
	qla4xxx_mailbox_command(ha, 1, 0, &mbox_cmd[0], &mbox_sts[0]);

	/* Poll IntrMaskIn register for completion */
	for (i = RESET_FIRMWARE_TOV; i!= 0; i--) {
		spin_lock_irqsave(&ha->hardware_lock, flags);

		intr_mask = RD_REG_DWORD(&ha->reg->interruptMaskIn);
		spin_unlock_irqrestore(&ha->hardware_lock, flags);

		if (intr_mask == IMI_FW_RESET_COMPLETE) {
			status = QLA_SUCCESS;
			break;
		}

		QL4PRINT(QLP2, printk("scsi%d: %s: Delay 1 sec, "
				      "Time Remaining = %d\n",
				      ha->host_no, __func__, i-1));

		set_current_state(TASK_INTERRUPTIBLE);
		schedule_timeout(1 * HZ);
	}

	LEAVE("qla4000_reset_firmware");

#if STOP_ON_RESET_FIRMWARE
	qla4xxx_panic("qla4000_reset_firmware", ha);
#endif

	return(status);
}
#endif	/* QLA4000 */

#ifdef QLA4010
/**************************************************************************
 * qla4010_soft_reset
 *	This routine performs a SOFT RESET.
 *
 * Input:
 * 	ha - Pointer to host adapter structure.
 *
 * Returns:
 *	QLA_SUCCESS - Successfully reset the firmware
 *	QLA_ERROR   - Failed to reset the firmware
 *
 * Context:
 *	Kernel context.
 **************************************************************************/
uint8_t
qla4010_soft_reset(scsi_qla_host_t *ha)
{
	uint32_t  max_wait_time;
	unsigned long flags = 0;
	uint8_t status = QLA_ERROR;
	uint16_t ctrl_status;

	ENTER("qla4010_soft_reset");

	spin_lock_irqsave(&ha->hardware_lock, flags);

	/*
	 * If the SCSI Reset Interrupt bit is set, clear it.
	 * Otherwise, the Soft Reset won't work.
	 */
	ctrl_status = RD_REG_DWORD(&ha->reg->ISPControlStatus);
	if ((ctrl_status & CSR_SCSI_RESET_INTR) != 0)
		WRT_REG_DWORD(&ha->reg->ISPControlStatus, CSRW_SCSI_RESET_INTR);

	/* Issue Soft Reset */
	WRT_REG_DWORD(&ha->reg->ISPControlStatus, CSRW_SOFT_RESET);

	spin_unlock_irqrestore(&ha->hardware_lock, flags);

	/* Wait until the Network Reset Intr bit is cleared */
	max_wait_time = RESET_INTR_TOV;
	do {
		spin_lock_irqsave(&ha->hardware_lock, flags);
		ctrl_status = RD_REG_WORD(&ha->reg->ISPControlStatus);
		spin_unlock_irqrestore(&ha->hardware_lock, flags);

		if ((ctrl_status & CSR_NET_RESET_INTR) == 0)
			break;

		set_current_state(TASK_INTERRUPTIBLE);
		schedule_timeout(1 * HZ);
	} while ((max_wait_time--));

	if ((ctrl_status & CSR_NET_RESET_INTR) != 0) {
		QL4PRINT(QLP2,
			 printk(KERN_WARNING "scsi%d: Network Reset Intr not cleared "
				"by Network function, clearing it now!\n", ha->host_no));
		WRT_REG_DWORD(&ha->reg->ISPControlStatus, CSRW_NET_RESET_INTR);
	}


	/* Wait until the firmware tells us the Soft Reset is done */
	max_wait_time = SOFT_RESET_TOV;
	do {
		spin_lock_irqsave(&ha->hardware_lock, flags);
		ctrl_status = RD_REG_WORD(&ha->reg->ISPControlStatus);
		spin_unlock_irqrestore(&ha->hardware_lock, flags);

		if ((ctrl_status & CSR_SOFT_RESET) == 0) {
			status = QLA_SUCCESS;
			break;
		}

		set_current_state(TASK_INTERRUPTIBLE);
		schedule_timeout(1 * HZ);
	} while ((max_wait_time--));

	/*
	 * Also, make sure that the SCSI Reset Interrupt bit has been cleared
	 * after the soft reset has taken place.
	 */
	ctrl_status = RD_REG_DWORD(&ha->reg->ISPControlStatus);
	if ((ctrl_status & CSR_SCSI_RESET_INTR) != 0)
		WRT_REG_DWORD(&ha->reg->ISPControlStatus, CSRW_SCSI_RESET_INTR);

	LEAVE("qla4010_soft_reset");
	return(status);
}

#if QLA4040_SUPPORT_ENABLED
/**************************************************************************
 * qla4010_topcat_reset
 *	This routine performs a HARD RESET of the TopCat chip.
 *
 * Input:
 * 	ha - Pointer to host adapter structure.
 *
 * Returns:
 *	QLA_SUCCESS - Successfully reset the firmware
 *	QLA_ERROR   - Failed to reset the firmware
 *
 * Context:
 *	Kernel context.
 **************************************************************************/
uint8_t
qla4010_topcat_reset(scsi_qla_host_t *ha)
{
	unsigned int flags;

	QL4PRINT(QLP2, printk(KERN_WARNING "scsi%d: %s: TopCat chip reset!\n",
			      ha->host_no, __func__));

	if (qla4xxx_take_semaphore(ha, SEM_NVRAM, TIMED_WAIT) != QLA_SUCCESS) {
		QL4PRINT(QLP2, printk(KERN_WARNING "scsi%d: %s: Unable to take SEM_NVRAM "
				      "semaphore\n", ha->host_no, __func__));
		return(QLA_ERROR);
	}

	spin_lock_irqsave(&ha->hardware_lock, flags);

	WRT_REG_DWORD(&ha->reg->generalPurposeOutput, GPORW_TOPCAT_RESET);
	TOPCAT_RESET_DELAY();
	WRT_REG_DWORD(&ha->reg->generalPurposeOutput, GPORW_TOPCAT_OUT_OF_RESET);
	TOPCAT_POST_RESET_DELAY();

	qla4xxx_clear_semaphore(ha, SEM_NVRAM);
	spin_unlock_irqrestore(&ha->hardware_lock, flags);
	return(QLA_SUCCESS);
}

#endif /* QLA4010 */
#endif

/**************************************************************************
 * qla4xxx_soft_reset
 *	This routine performs a SOFT RESET.
 *
 * Input:
 * 	ha - Pointer to host adapter structure.
 *
 * Returns:
 *	QLA_SUCCESS - Successfully reset the firmware
 *	QLA_ERROR   - Failed to reset the firmware
 *
 * Context:
 *	Kernel context.
 **************************************************************************/
inline uint8_t
qla4xxx_soft_reset(scsi_qla_host_t *ha)
{
#ifdef QLA4010
#if QLA4040_SUPPORT_ENABLED
	if (test_bit(AF_TOPCAT_CHIP_PRESENT, &ha->flags)) {
		if (qla4010_soft_reset(ha)  == QLA_SUCCESS &&
		    qla4010_topcat_reset(ha)  == QLA_SUCCESS &&
		    qla4010_soft_reset(ha) == QLA_SUCCESS) {
			return(QLA_SUCCESS);
		}
		else
			return(QLA_ERROR);
	}
	else
#endif
	return(qla4010_soft_reset(ha));
#else
	return(qla4000_reset_firmware(ha));
#endif
}

/**************************************************************************
 * qla4xxx_hard_reset
 *	This routine performs a HARD RESET.
 *
 * Input:
 * 	ha - Pointer to host adapter structure.
 *
 * Returns:
 *	QLA_SUCCESS - Successfully reset the firmware
 *	QLA_ERROR   - Failed to reset the firmware
 *
 * Context:
 *	Kernel context.
 **************************************************************************/
inline uint8_t
qla4xxx_hard_reset(scsi_qla_host_t *ha)
{
#ifdef QLA4010

	/* The QLA4010 really doesn't have an equivalent to a hard reset */
	qla4xxx_flush_active_srbs(ha);
	#if QLA4040_SUPPORT_ENABLED
	if (test_bit(AF_TOPCAT_CHIP_PRESENT, &ha->flags)) {
		if (qla4010_soft_reset(ha)  == QLA_SUCCESS &&
		    qla4010_topcat_reset(ha)  == QLA_SUCCESS &&
		    qla4010_soft_reset(ha) == QLA_SUCCESS) {
			return(QLA_SUCCESS);
		}
		else
			return(QLA_ERROR);
	}
	else
#endif
		return(qla4010_soft_reset(ha));
#else
	return(qla4000_hard_reset(ha));
#endif
}

/**************************************************************************
 * qla4xxx_cmd_wait
 *	This routine stalls the driver until all outstanding commands are
 *	returned.
 *
 * Input:
 * 	ha - Pointer to host adapter structure.
 *
 * Remarks:
 *       Caller must release the Hardware Lock prior to calling this routine.
 *
 * Returns:
 *	QLA_SUCCESS - All outstanding commands completed
 *	QLA_ERROR   - All outstanding commands did not complete
 *
 * Context:
 *	Kernel context.
 **************************************************************************/
STATIC uint8_t
qla4xxx_cmd_wait(scsi_qla_host_t *ha)
{
	uint32_t index = 0;
	uint8_t stat = QLA_SUCCESS;
	int wait_cnt = WAIT_CMD_TOV; /* Initialized for 30 seconds as we expect all
			      commands to retuned ASAP.*/
	unsigned long flags;

	ENTER("qla4xxx_cmd_wait: started\n");

	while (wait_cnt) {
		spin_lock_irqsave(&ha->hardware_lock, flags);
		/* Find a command that hasn't completed. */
		for (index = 1; index < MAX_SRBS; index++) {
			if (ha->active_srb_array[index] != NULL)
				break;
		}
		spin_unlock_irqrestore(&ha->hardware_lock, flags);

		/* If No Commands are pending, wait is complete */
		if (index == MAX_SRBS) {
			break;
		}

		/* If we timed out on waiting for commands to come back
		 * return ERROR.
		 */
		wait_cnt--;
		if (wait_cnt == 0)
			stat =  QLA_ERROR;
		else {
			/* sleep a second */
			set_current_state(TASK_INTERRUPTIBLE);
			schedule_timeout(1 * HZ);
		}
	} /* End of While (wait_cnt) */

	QL4PRINT(QLP2,printk("(%d): %s: Done waiting on commands - array_index=%d\n",
			     (int)ha->host_no, __func__, index));

	LEAVE("qla4xxx_cmd_wait");

	return(stat);
}

/**************************************************************************
 * qla4xxx_recover_adapter
 *	This routine recovers that adapter from a fatal state.
 *
 * Input:
 * 	ha - Pointer to host adapter structure.
 *	renew_ddb_list - Indicates what to do with the adapter's ddb list
 *			after adapter recovery has completed.
 *			0=preserve ddb list, 1=destroy and rebuild ddb list
 *
 * Returns:
 *	QLA_SUCCESS - Successfully recovered adapter
 *	QLA_ERROR   - Failed to recover adapter
 *
 * Context:
 *	Kernel context.
 **************************************************************************/
uint8_t
qla4xxx_recover_adapter(scsi_qla_host_t *ha, uint8_t renew_ddb_list)
{
	uint8_t status = QLA_SUCCESS;

	ENTER("qla4xxx_recover_adapter");

	QL4PRINT(QLP2,
		 printk(KERN_INFO "scsi%d: recover adapter (begin)\n",
			(int)ha->host_no));

	clear_bit(AF_ONLINE, &ha->flags);
	QL4PRINT(QLP2, printk("scsi%d: %s calling qla4xxx_cmd_wait\n",
			      ha->host_no, __func__));

	/* Wait for outstanding commands to complete.
	 * Stalls the driver for max 30 secs
	 */
	status = qla4xxx_cmd_wait(ha);

	qla4xxx_disable_intrs(ha);


	if (renew_ddb_list == 1) {
		qla4xxx_process_aen(ha, 1);
	}

	/* Reset the firmware.  If successful, function
	 * returns with ISP interrupts enabled.
	 */
	if (status == QLA_SUCCESS) {
		QL4PRINT(QLP2,
			 printk(KERN_INFO "scsi%d: %s - Performing soft reset..\n",
				(int)ha->host_no,__func__));
		status = qla4xxx_soft_reset(ha);
	}

	/* If firmware (SOFT) reset failed, or if all outstanding
	 * commands have not returned, then do a HARD reset.
	 */
	if (status == QLA_ERROR) {
		QL4PRINT(QLP2,
			 printk(KERN_INFO "scsi%d: %s - Performing hard reset..\n",
				(int)ha->host_no,__func__));
		status = qla4xxx_hard_reset(ha);
	}

	if (renew_ddb_list == 1) {
		qla4xxx_process_aen(ha, 1);
	}

	/* Re-initialize firmware. If successful, function returns
	 * with ISP interrupts enabled */
	if (status == QLA_SUCCESS) {
		QL4PRINT(QLP2,
			 printk(KERN_INFO "scsi%d: %s - Initializing adapter..\n",
				(int)ha->host_no,__func__));

		/* If successful, AF_ONLINE flag set in
		 * qla4xxx_initialize_adapter */
		status = qla4xxx_initialize_adapter(ha, renew_ddb_list);
	}

	/* Failed adapter initialization?
	 * Retry reset_ha only if invoked via DPC (DPC_RESET_HA) */
	if ((test_bit(AF_ONLINE, &ha->flags) == 0) &&
	    (test_bit(DPC_RESET_HA, &ha->dpc_flags))) {
		/* Adapter initialization failed, see if we can retry
		 * resetting the ha */
		if (!test_bit(DPC_RETRY_RESET_HA, &ha->dpc_flags)) {
			ha->retry_reset_ha_cnt = MAX_RESET_HA_RETRIES;
			QL4PRINT(QLP2,
				 printk(KERN_INFO "scsi%d: recover adapter - "
					"retrying (%d) more times\n",
					ha->host_no, ha->retry_reset_ha_cnt));
			set_bit(DPC_RETRY_RESET_HA, &ha->dpc_flags);
			set_bit(AF_SCHEDULE_DPC, &ha->flags);
			status = QLA_ERROR;
		}
		else {
			if (ha->retry_reset_ha_cnt > 0) {
				/* Schedule another Reset HA */
				ha->retry_reset_ha_cnt--;
				QL4PRINT(QLP2,
					 printk(KERN_INFO
						"scsi%d: recover adapter - "
						"retry remaining %d\n", ha->host_no,
						ha->retry_reset_ha_cnt));
				set_bit(AF_SCHEDULE_DPC, &ha->flags);
				status = QLA_ERROR;
			}

			if (ha->retry_reset_ha_cnt == 0) {
				/* Recover adapter retries have been exhausted.
				 * Adapter DEAD */
				QL4PRINT(QLP2,
					 printk(KERN_INFO
						"scsi%d: recover adapter failed - "
						"board disabled\n", ha->host_no));
				qla4xxx_flush_active_srbs(ha);
				clear_bit(DPC_RETRY_RESET_HA, &ha->dpc_flags);
				clear_bit(DPC_RESET_HA, &ha->dpc_flags);
				clear_bit(DPC_RESET_HA_DESTROY_DDB_LIST, &ha->dpc_flags);
				status = QLA_SUCCESS;
			}
		}
	}
	else {
		clear_bit(DPC_RESET_HA, &ha->dpc_flags);
		clear_bit(DPC_RESET_HA_DESTROY_DDB_LIST, &ha->dpc_flags);
		clear_bit(DPC_RETRY_RESET_HA, &ha->dpc_flags);
	}

	ha->adapter_error_count++;
	QL4PRINT(QLP2,
		 printk(KERN_INFO "scsi%d: recover adapter (end)\n", (int)ha->host_no));
	LEAVE("qla4xxx_recover_adapter");
	return(status);
}

/**************************************************************************
 * qla4xxx_eh_wait_for_active_target_commands
 *	This routine
 *
 * Input:
 * 	ha - Pointer to host adapter structure.
 *	target - SCSI target ID
 *
 * Returns:
 *	0 - All pending commands returned
 *	non-zero - All pending commands did not return
 *
 * Context:
 *	Kernel context.
 **************************************************************************/
int
qla4xxx_eh_wait_for_active_target_commands(scsi_qla_host_t *ha, int target, int lun)
{
	int     cnt;
	int     status;
	unsigned long   flags;
	srb_t           *sp;
	Scsi_Cmnd       *cmd;

	status = 0;
	ENTER("qla4xxx_eh_wait_for_active_target_commands");
	/*
	 * Waiting for all commands for the designated target in the active
	 * array
	 */
	for (cnt = 1; cnt < MAX_SRBS; cnt++) {
		spin_lock_irqsave(&ha->hardware_lock, flags);
		sp = ha->active_srb_array[cnt];
		if (sp) {
			cmd = sp->cmd;
			spin_unlock_irqrestore(&ha->hardware_lock, flags);
			if ((cmd->target == target) && (cmd->lun == lun)) {

				if (!qla4xxx_eh_wait_on_command(ha, cmd)) {
					status++;
					break;
				}
			}
		}
		else {
			spin_unlock_irqrestore(&ha->hardware_lock, flags);
		}
	}
	QL4PRINT(QLP2,
		 printk(KERN_INFO "scsi%d: %s return_status=%d\n", ha->host_no, __func__, status));
	LEAVE("qla4xxx_eh_wait_for_active_target_commands");
	return(status);
}

/**************************************************************************
 * qla4xxx_reset_lun
 *	This routine performs a LUN RESET on the specified target/lun.
 *
 * Input:
 * 	ha - Pointer to host adapter structure.
 *	ddb_entry - Pointer to device database entry
 *	lun_entry - Pointer to lun entry structure
 *
 * Remarks:
 *	The caller must ensure that the ddb_entry and lun_entry pointers
 *	are valid before calling this routine.
 *
 * Returns:
 *	QLA_SUCCESS - lun reset completed successfully
 *	QLA_ERROR   - lun reset failed
 *
 * Context:
 *	Kernel context.
 **************************************************************************/
uint8_t
qla4xxx_reset_lun(scsi_qla_host_t *ha,
		  ddb_entry_t *ddb_entry,
		  lun_entry_t *lun_entry)
{
	uint32_t mbox_cmd[MBOX_REG_COUNT];
	uint32_t mbox_sts[MBOX_REG_COUNT];
	uint8_t target = ddb_entry->target;
	uint8_t lun = lun_entry->lun;
	uint8_t status = QLA_SUCCESS;

	ENTER("qla4xxx_reset_lun");

	HOST_SPIN_UNLOCK_IRQ(ha);

	QL4PRINT(QLP2,
		 printk(KERN_INFO "scsi%d:%d:%d:%d: lun reset issued\n",
			ha->host_no, ddb_entry->bus, target, lun));

	/*
	 * Send lun reset command to ISP, so that the ISP
	 * will return all outstanding requests with RESET status
	 */
	memset(&mbox_cmd, 0, sizeof(mbox_cmd));
	memset(&mbox_sts, 0, sizeof(mbox_sts));
	mbox_cmd[0] = MBOX_CMD_LUN_RESET;
	mbox_cmd[1] = ddb_entry->fw_ddb_index;
	mbox_cmd[2] = lun << 8;
	mbox_cmd[5] = 0x01; /* Immediate Command Enable */

	qla4xxx_mailbox_command(ha, 6, 1, &mbox_cmd[0], &mbox_sts[0]);
	if ((mbox_sts[0] == MBOX_STS_COMMAND_COMPLETE) ||
	    (mbox_sts[0] == MBOX_STS_COMMAND_ERROR)) {
		QL4PRINT(QLP2,
			 printk(KERN_INFO "scsi%d:%d:%d:%d: lun reset SUCCEEDED\n",
				ha->host_no, ddb_entry->bus, target, lun));

		/* Send marker to unlock queues on next command */
		ha->marker_needed = 1;

		/* Waiting for all active commands to complete for the device */
		HOST_SPIN_LOCK_IRQ(ha);
		if (qla4xxx_eh_wait_for_active_target_commands(ha, target, lun) != 0)
			status = QLA_ERROR;
		HOST_SPIN_UNLOCK_IRQ(ha);
	}
	else {
		QL4PRINT(QLP2,
			 printk(KERN_INFO "scsi%d:%d:%d:%d: lun reset FAILED "
				"w/ status %04x\n", ha->host_no, ddb_entry->bus,
				target, lun, mbox_sts[0]));

		status = QLA_ERROR;
	}

	HOST_SPIN_LOCK_IRQ(ha);
	LEAVE("qla4xxx_reset_lun");
	return(status);
}

/**************************************************************************
 * qla4xxx_eh_device_reset
 *	This routine is called by the Linux OS to reset a specific target/lun.
 *
 * Input:
 *	cmd - Pointer to Linux's SCSI command structure
 *
 * Output:
 *	None
 *
 * Remarks:
 *	None
 *
 * Returns:
 *	SUCCESS - Successfully reset target/lun
 *	FAILED  - Failed to reset target/lun
 *
 * Context:
 *	Kernel context.	 io_request_lock LOCKED
 **************************************************************************/
int
qla4xxx_eh_device_reset(Scsi_Cmnd *cmd)
{
	int ret_status = FAILED;
	scsi_qla_host_t *ha;
	ddb_entry_t     *ddb_entry;
	lun_entry_t     *lun_entry;

	ENTER("qla4xxx_eh_device_reset");

	ha = (scsi_qla_host_t *) cmd->host->hostdata;

	QL4PRINT(QLP2,
		 printk(KERN_INFO "scsi%d:%d:%d:%d: device reset\n",
			ha->host_no, cmd->channel, cmd->target, cmd->lun));

	ddb_entry = qla4xxx_lookup_ddb_by_SCSIID(ha, cmd->channel, cmd->target);
	if (ddb_entry) {
		lun_entry = qla4xxx_lookup_lun_handle(ha,ddb_entry,cmd->lun);
		if (lun_entry) {
			if (qla4xxx_reset_lun(ha, ddb_entry, lun_entry)
			    == QLA_SUCCESS) {
				ret_status = SUCCESS;
			}
		}
	}

	QL4PRINT(QLP2, printk("scsi%d: %s: return with status = %x\n",
			      ha->host_no, __func__, ret_status));
	LEAVE("qla4xxx_eh_device_reset");

#if STOP_ON_RESET_DEVICE
	qla4xxx_panic("qla4xxx_eh_device_reset", ha);
#endif
	return(ret_status);
}

/**************************************************************************
 * qla4xxx_eh_bus_reset
 *	This routine is called by the Linux OS to reset the specified
 *      adapter/bus.
 *
 * Input:
 *	cmd - Pointer to Linux's SCSI command structure
 *
 * Returns:
 *	SUCCESS - Successfully reset adapter/bus
 *	FAILED  - Failed to reset adapter/bus
 *
 * Context:
 *	Kernel context.	 io_request_lock LOCKED
 **************************************************************************/
int
qla4xxx_eh_bus_reset(Scsi_Cmnd *cmd)
{
	uint8_t status = QLA_SUCCESS;
	int     ret_status;
	struct list_head *ptr, *next; /* used for traversing ddb_list */
	scsi_qla_host_t *ha;
	ddb_entry_t *ddb_entry;
#if 1 || __VMKERNEL_MODULE__
	ddb_entry_t *ddbEnts[MAX_TARGETS]; /* Entries we have reset */
        ddb_entry_t ddbLocEnt;
	unsigned long flags;
        int n;
#endif

	ENTER("qla4xxx_eh_bus_reset");

	ha = (scsi_qla_host_t *) cmd->host->hostdata;
	QL4PRINT(QLP2,
		 printk(KERN_INFO "scsi%d:%d: bus reset\n", ha->host_no, cmd->channel));

	/* Attempt to reset all valid targets with outstanding commands */
#if 1 || __VMKERNEL_MODULE__
        ret_status = SUCCESS; /* Assume success for now */
        memset(ddbEnts, 0, sizeof(ddbEnts));
        /* At most process MAX_TARGET targets. This is not what will normally
         * terminate this loop, it is only an extra safety measure since we
         * cannot guarantee that targets are not continuously removed and added
         * while we are resetting, though it is mostly a theoretical problem.
         */
        for (n=0; n < MAX_TARGETS; n++) {
           int i = 0;

           ddb_entry = NULL; /* We use this as a flag as well */
           spin_lock_irqsave(&ha->ddb_list_spinlock, flags);
           list_for_each_safe(ptr, next, &ha->ddb_list) {
              ddb_entry = list_entry(ptr, ddb_entry_t, list_entry);
              if (ddb_entry) {
                 /* See if we have dealt with this entry */
                 for (i=0; i < MAX_TARGETS && ddbEnts[i]; i++) {
                    if (ddbEnts[i] == ddb_entry) {
                       break;
                    }
                 }
                 if (ddbEnts[i] == ddb_entry) {
                    ddb_entry = NULL;
                    continue; /* Already processed this target */
                 }
                 ddbEnts[i] = ddb_entry;
                 ddbLocEnt = *ddb_entry;
                 break; /* process this entry outside the loop */
              }
           }
           spin_unlock_irqrestore(&ha->ddb_list_spinlock, flags);
           if (ddb_entry == NULL) {
              break; /* We are done*/
           }
           if (WAIT_FOR_ADAPTER_UP(ha) != QLA_SUCCESS) {
              QL4PRINT(QLP2, printk("scsi%d:%d:%d: %s: Unable to reset "
                                    "bus.  Adapter DEAD.\n",
                                    ha->host_no, cmd->channel, ddbLocEnt.target,
                                    __func__));
              ret_status = FAILED;
              goto exit_bus_rst;
           } else {
              status = qla4xxx_reset_target(ha, &ddbLocEnt);
              if (status == QLA_SUCCESS) {
                 QL4PRINT(QLP2,
                          printk(KERN_INFO "scsi%d:%d:%d: bus reset SUCCEEDED\n",
                                 ha->host_no, cmd->channel, ddbLocEnt.target));
              } else {
                 QL4PRINT(QLP2,
                          printk(KERN_INFO "scsi%d:%d:%d: bus reset FAILED\n",
                                 ha->host_no, cmd->channel, ddbLocEnt.target));
                 ret_status = FAILED;
              }
           }
        }
#else
	list_for_each_safe(ptr, next, &ha->ddb_list)
	{
		if (WAIT_FOR_ADAPTER_UP(ha) != QLA_SUCCESS) {
			QL4PRINT(QLP2, printk("scsi%d:%d: %s: Unable to reset "
					      "bus.  Adapter DEAD.\n",
					      ha->host_no, cmd->channel,
					      __func__));

			ret_status = FAILED;
			goto exit_bus_rst;
		}

		ddb_entry = list_entry(ptr, ddb_entry_t, list_entry);
		if (ddb_entry) {
			QL4PRINT(QLP5, printk("scsi%d: %s: reset target b%d, t%x, "
					      "index [%d]\n",
					      ha->host_no, __func__,
					      ddb_entry->bus,
					      ddb_entry->target,
					      ddb_entry->fw_ddb_index));

			/* Issue a reset */
			status |= qla4xxx_reset_target(ha, ddb_entry);
		}
	}

	/*
	 * Status is QLA_SUCCESS if target resets for ALL devices completed
	 * successfully.  Otherwise the status is QLA_ERROR.
	 */
	if (status == QLA_SUCCESS) {
		QL4PRINT(QLP2,
			 printk(KERN_INFO "scsi%d:%d: bus reset SUCCEEDED\n",
				ha->host_no, cmd->channel));

		/* restore debug level */
		ret_status = SUCCESS;
	}
	else {
		QL4PRINT(QLP2,
			 printk(KERN_INFO "scsi%d:%d: bus reset FAILED\n",
				ha->host_no, cmd->channel));
		ret_status = FAILED;
	}
#endif

	exit_bus_rst:
	QL4PRINT(QLP2, printk("scsi%d: %s: return with status = %x\n",
			      ha->host_no, __func__, ret_status));
	LEAVE("qla4xxx_eh_bus_reset");

#if STOP_ON_RESET_BUS
	qla4xxx_panic("qla4xxx_eh_bus_reset", ha);
#endif

	return(ret_status);
}

/**************************************************************************
 * qla4xxx_reset_target
 *	This routine issues either a warm or cold target reset to the
 *	specified device.
 *
 * Input:
 * 	ha - Pointer to host adapter structure.
 *	ddb_entry - Pointer to device database entry
 *
 * Remarks:
 *	The caller must ensure that the ddb_entry pointer is valid before
 *	calling this routine.
 *
 * Returns:
 *	QLA_SUCCESS - Successfully reset target
 *	QLA_ERROR   - Failed to reset target
 *
 * Context:
 *	Kernel context.
 **************************************************************************/
uint8_t qla4xxx_reset_target(scsi_qla_host_t *ha,
			     ddb_entry_t *ddb_entry)
{
	uint8_t status = QLA_SUCCESS;
	uint8_t target = ddb_entry->target;
	lun_entry_t *lun_entry;
	int i;

	/* Reset all LUNs on this target */
	for (i=0; i<MAX_LUNS; i++) {
		lun_entry = ddb_entry->lun_table[i];
		if (lun_entry) {
			status |= qla4xxx_reset_lun(ha, ddb_entry, lun_entry);
		}
	}

	if (status == QLA_SUCCESS) {
		QL4PRINT(QLP2,
			 printk(KERN_INFO "scsi%d:%d:%d: device reset SUCCEEDED\n",
				ha->host_no, ddb_entry->bus, target));
	}
	else {
		QL4PRINT(QLP2,
			 printk(KERN_INFO "scsi%d:%d:%d: device reset FAILED\n",
				ha->host_no, ddb_entry->bus, target));
		status = QLA_ERROR;
	}

	return(status);
}

/**************************************************************************
 * qla4xxx_flush_active_srbs
 *	This routine is called just prior to a HARD RESET to return all
 *      outstanding commands back to the Operating System.
 *
 * Input:
 * 	ha - Pointer to host adapter structure.
 *
 * Remarks:
 *	Caller should make sure that the following locks are released
 *	before this calling routine:
 *    		Hardware lock, io_request_lock, adapter_lock, and lun_lock.
 *
 * Returns:
 *	None
 *
 * Context:
 *	Kernel context.
 **************************************************************************/
STATIC void
qla4xxx_flush_active_srbs(scsi_qla_host_t *ha)
{
	srb_t    *srb;
	int      i;
	unsigned long flags;

	ENTER("qla4xxx_flush_active_srbs");

	spin_lock_irqsave(&ha->hardware_lock, flags);
	for (i = 1; i < MAX_SRBS; i++) {
		if ((srb = ha->active_srb_array[i]) != NULL) {
			QL4PRINT(QLP5,
				 printk("scsi%d: %s: found srb %p in active array, "
					"returning\n", ha->host_no, __func__, srb));
			del_from_active_array(ha, i);
			CMD_RESULT(srb->cmd) = (int) (DID_RESET << 16);
			add_to_done_srb_q(ha,srb);
		}
	}
	spin_unlock_irqrestore(&ha->hardware_lock, flags);

	if (!list_empty(&ha->done_srb_q)) {
		while ((srb = del_from_done_srb_q_head(ha)) != NULL)
			qla4xxx_complete_request(ha, srb);
	}

	LEAVE("qla4xxx_flush_active_srbs");
}

/**************************************************************************
 * qla4xxx_eh_host_reset
 *	This routine is invoked by the Linux kernel to perform fatal error
 *	recovery on the specified adapter.
 *
 * Input:
 *	cmd - Pointer to Linux's SCSI command structure
 *
 * Returns:
 *	SUCCESS - Successfully recovered host adapter
 *	FAILED  - Failed to recover host adapter
 *
 * Context:
 *	Kernel context.	 io_request_lock LOCKED
 **************************************************************************/
int
qla4xxx_eh_host_reset(Scsi_Cmnd *cmd)
{
	int ret_status;
	scsi_qla_host_t *ha;
	ENTER("qla4xxx_eh_host_reset");


	ha = (scsi_qla_host_t *) cmd->host->hostdata;
	QL4PRINT(QLP2,
		 printk(KERN_INFO "scsi%d: - host reset\n", (int)ha->host_no));

	if (WAIT_FOR_ADAPTER_UP(ha) != QLA_SUCCESS) {
		QL4PRINT(QLP2, printk("scsi%d: %s: Unable to reset host.  "
				      "Adapter DEAD.\n",
				      ha->host_no, __func__));

		ret_status = FAILED;
		goto exit_host_rst;
	}

	HOST_SPIN_UNLOCK_IRQ(ha);
	if (qla4xxx_recover_adapter(ha, PRESERVE_DDB_LIST)
	    == QLA_SUCCESS) {
		ret_status = SUCCESS;
	}
	else {
		ret_status = FAILED;
	}
	HOST_SPIN_LOCK_IRQ(ha);

	exit_host_rst:
	QL4PRINT(QLP2, printk("scsi%d: %s: return with status = %x\n",
			      ha->host_no, __func__, ret_status));

	LEAVE("qla4xxx_eh_host_reset");

	return(ret_status);
}

void
qla4xxx_ioctl_error_recovery(scsi_qla_host_t *ha)
{
	int return_status;
	unsigned long flags;

	QL4PRINT(QLP2, printk(KERN_INFO
			      "scsi%d: %s: issuing device reset\n", ha->host_no, __func__));
	if (!ha->ioctl_err_cmd) {
		QL4PRINT(QLP2,
			 printk("scsi%d: %s: should not occur\n", ha->host_no, __func__));
		return;
	}

	HOST_SPIN_LOCK_IRQSAVE(ha, flags);
	return_status = qla4xxx_eh_device_reset(ha->ioctl_err_cmd);
	if (return_status != QLA_SUCCESS) {
		QL4PRINT(QLP2,
			 printk("scsi%d: %s: elevation to host_reset\n",
				ha->host_no, __func__));
		return_status = qla4xxx_eh_host_reset(ha->ioctl_err_cmd);
		QL4PRINT(QLP2,
			 printk("scsi%d: %s: return_status=%x\n", ha->host_no, __func__,
				return_status));
	}
	ha->ioctl_err_cmd = NULL ;
	HOST_SPIN_UNLOCK_IRQRESTORE(ha, flags);
}


/****************************************************************************/
/*                QLogic QLA4000 Hardware Support Functions.                */
/****************************************************************************/

/**************************************************************************
 * qla4xxx_enable_intrs
 *	This routine enables the PCI interrupt request by clearing the
 *	appropriate bit.
 *
 * qla4xxx_disable_intrs
 *	This routine disables the PCI interrupt request by setting the
 *	appropriate bit.
 *
 * Remarks:
 *	The hardware_lock must be unlocked upon entry.
 *
 * Input:
 * 	ha - Pointer to host adapter structure.
 *
 * Returns:
 *	None
 *
 * Context:
 *	Kernel/Interrupt context.
 **************************************************************************/
static inline void __qla4xxx_enable_intrs(scsi_qla_host_t *ha)
{
	ENTER("qla4xxx_enable_intrs");
	set_bit(AF_INTERRUPTS_ON, &ha->flags);

	ISP_ENABLE_INTRS(ha);
	QL4PRINT(QLP7, printk("scsi%d: %s: intrMask = %08x\n",
			      ha->host_no, __func__,
			      ISP_RD_INTR_MASK(ha)));
	LEAVE("qla4xxx_enable_intrs");
}

static inline void __qla4xxx_disable_intrs(scsi_qla_host_t *ha)
{

	ENTER("qla4xxx_disable_intrs");
	clear_bit(AF_INTERRUPTS_ON, &ha->flags);
	ISP_DISABLE_INTRS(ha);
	QL4PRINT(QLP7, printk("scsi%d: %s: intrMask = %08x\n",
			      ha->host_no, __func__,
			      ISP_RD_INTR_MASK(ha)));
	LEAVE("qla4xxx_disable_intrs");
}
static inline void qla4xxx_enable_intrs(scsi_qla_host_t *ha)
{
	unsigned long flags = 0;

	spin_lock_irqsave(&ha->hardware_lock, flags);
	__qla4xxx_enable_intrs(ha);
	spin_unlock_irqrestore(&ha->hardware_lock, flags);
}

static inline void qla4xxx_disable_intrs(scsi_qla_host_t *ha)
{
	unsigned long flags = 0;

	spin_lock_irqsave(&ha->hardware_lock, flags);
	__qla4xxx_disable_intrs(ha);
	spin_unlock_irqrestore(&ha->hardware_lock, flags);
}

/**************************************************************************
 * qla4xxx_get_adapter_handle
 *	This routine returns the adapter handle that corresponds to the
 *	specified instance number.
 *
 * Input:
 * 	instance - Instance number of the desired host adapter.
 *
 * Returns:
 *	Pointer to host adapter structure
 *
 * Context:
 *	Kernel context.
 **************************************************************************/
scsi_qla_host_t *
qla4xxx_get_adapter_handle(uint16_t instance)
{
	scsi_qla_host_t *ha;

	for (ha = qla4xxx_hostlist; ha != NULL; ha = ha->next) {
		if (ha->instance == instance) {
			QL4PRINT(QLP3,
				 printk("scsi%d: %s: handle (%p) for instance %d\n",
					ha->host_no, __func__, ha, instance));
			return(ha);
		}
	}

	QL4PRINT(QLP2, printk("scsi%d: %s: not found\n", ha->host_no, __func__));
	return(NULL);
}

/**************************************************************************
 * qla4xxx_get_hba_count
 *	This routine returns the number of host adapters present.
 *
 * Input:
 *	None
 *
 * Returns:
 *    qla4xxx_hba_count	- Number of host adapters present.
 *
 * Context:
 *	Kernel context.
 **************************************************************************/
inline uint32_t
qla4xxx_get_hba_count(void)
{
	return(qla4xxx_hba_count);
}

/*
 * Declarations for load module
 *
 * We initialize the driver via the linux kernel loadable module interface
 * by defining and initializing the variable 'driver_template' and including
 * the file scsi_module.c. All routines get called automatically.
 */
Scsi_Host_Template driver_template = QLA4XXX_LINUX_TEMPLATE;
#ifdef __VMKERNEL_MODULE__
#include <scsi_module.c>
#else
#include <../drivers/scsi/scsi_module.c>
#endif

#include "ql4print.c"
#include "ql4ioctl.c"

/*
 * Overrides for Emacs so that we almost follow Linus's tabbing style.
 * Emacs will notice this stuff at the end of the file and automatically
 * adjust the settings for this buffer only.  This must remain at the end
 * of the file.
 * ---------------------------------------------------------------------------
 * Local variables:
 * c-indent-level: 2
 * c-brace-imaginary-offset: 0
 * c-brace-offset: -2
 * c-argdecl-indent: 2
 * c-label-offset: -2
 * c-continued-statement-offset: 2
 * c-continued-brace-offset: 0
 * indent-tabs-mode: nil
 * tab-width: 8
 * End:
 */


