/*****************************************************************************
*                  QLOGIC LINUX SOFTWARE
*
* QLogic  QLA4000 iSCSI driver
* Copyright (C) 2004 Qlogic Corporation
* (www.qlogic.com)
*
* This program is free software; you can redistribute it and/or modify it
* under the terms of the GNU General Public License as published by the
* Free Software Foundation; either version 2, or (at your option) any
* later version.
*
* This program is distributed in the hope that it will be useful, but
* WITHOUT ANY WARRANTY; without even the implied warranty of
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
* General Public License for more details.
*
******************************************************************************
*             Please see release.txt for revision history.                   *
*                                                                            *
******************************************************************************
* Function Table of Contents:
*	  qla4extioctl_query_hba_iscsi_node
*	  qla4extioctl_query_hba_iscsi_portal
*	  qla4extioctl_query_disc_iscsi_node
*	  qla4extioctl_query_disc_iscsi_portal
*	  qla4extioctl_query_driver
*	  qla4extioctl_query_fw
*	  qla4extioctl_query_chip
*	  qla4extioctl_query
*	  qla4extioctl_reg_aen
*	  qla4extioctl_get_aen
*	  qla4extioctl_get_statistics_gen
*	  qla4extioctl_get_statistics_iscsi
*	  qla4extioctl_get_device_entry_iscsi
*	  qla4extioctl_get_init_fw_iscsi
*	  qla4extioctl_get_isns_server
*	  qla4extioctl_get_isns_disc_targets
*	  qla4extioctl_get_data
*	  qla4extioctl_rst_statistics_gen
*	  qla4extioctl_rst_statistics_iscsi
*	  qla4extioctl_set_device_entry_iscsi
*	  qla4extioctl_set_init_fw_iscsi
*	  qla4extioctl_set_data
*	  qla4xxx_ioctl_sleep_done
*	  qla4xxx_ioctl_sem_init
*	  qla4xxx_scsi_pass_done
*	  qla4extioctl_scsi_passthru
*	  qla4extioctl_iscsi_passthru
*	  qla4extioctl_get_hbacnt
*	  qla4intioctl_logout_iscsi
*	  qla4intioctl_ping
*	  qla4intioctl_get_flash
*	  qla4intioctl_get_driver_debug_level
*	  qla4intioctl_get_host_no
*	  qla4intioctl_get_data
*	  qla4intioctl_set_flash
*	  qla4intioctl_set_driver_debug_level
*	  qla4intioctl_set_data
*	  qla4intioctl_hba_reset
*	  qla4intioctl_copy_fw_flash
*	  qla4xxx_ioctl
*        apidev_open
*        apidev_close
*        apidev_ioctl
*        apidev_init
*        apidev_cleanup
*	
****************************************************************************/

#include <linux/config.h>
#include <linux/module.h>
#include <stdarg.h>
#include <asm/io.h>
#include <asm/irq.h>
#include <asm/segment.h>
#include <asm/byteorder.h>
#include <linux/spinlock.h>
#include <linux/version.h>
#include <linux/types.h>
#include <linux/string.h>
#include <linux/errno.h>
#include <linux/kernel.h>
#include <linux/delay.h>
#include <linux/timer.h>
#include <linux/sched.h>
#include <linux/pci.h>
#include <linux/proc_fs.h>
#include <linux/blk.h>
#include <linux/tqueue.h>
#include <asm/uaccess.h>

#ifdef __VMKERNEL_MODULE__
#include <scsi.h>
#include <hosts.h>
#else
#include <../drivers/scsi/scsi.h>
#include <../drivers/scsi/hosts.h>
#endif

#include "qlisioct.h"
#include "qlinioct.h"
#include "qla4x_fw.h"
#include "qla4xxx.h"
#include "ql4print.h"

// KRH: (BEGIN) Define these locally, for now
/*
 * Sub codes for Get Data.
 * Use in combination with INT_GET_DATA as the ioctl code
 */
#define INT_SC_GET_DRIVER_DEBUG_LEVEL   2
#define INT_SC_GET_HOST_NO 		3

/*
 * Sub codes for Set Data.
 * Use in combination with INT_SET_DATA as the ioctl code
 */
#define INT_SC_SET_DRIVER_DEBUG_LEVEL	2

/*
 * Sub codes for Reset
 * Use in combination with INT_CC_HBA_RESET as the ioctl code
 */
#define INT_SC_HBA_RESET			0
#define INT_SC_FIRMWARE_RESET			1
#define INT_SC_TARGET_WARM_RESET		2
#define INT_SC_LUN_RESET			3
//KRH: (END)

/* Defines for Passthru */
#define IOCTL_INVALID_STATUS			0xffff
#define IOCTL_PASSTHRU_TOV			30

/* Defines for byte-order translation */
#define GET_DATA	0
#define SET_DATA	1

/**************************
 * Global Data
 **************************/

/*
 * Create character driver "HbaApiDev" w dynamically allocated major number
 * and create "/proc/scsi/<QLA4XXX_PROC_NAME>/HbaApiNode" as the device
 * node associated with the major number.
 */
#define APIDEV_NODE  "HbaApiNode"
#define APIDEV_NAME  "HbaApiDev"

STATIC int apidev_major = 0;
STATIC struct Scsi_Host *apidev_host = 0;


/**************************
* Functions
**************************/

/**************************************************************************
 * qla4extioctl_query_hba_iscsi_node
 *	This routine retrieves the HBA node properties
 *
 * Input:
 *	ha    = adapter structure pointer.
 *	ioctl = IOCTL structure pointer.
 *
 * Output:
 *	None
 *
 * Returns:
 *	QLA_SUCCESS = success
 *	QLA_ERROR   = error
 *
 * Context:
 *	Kernel context.
 **************************************************************************/
static int
qla4extioctl_query_hba_iscsi_node(scsi_qla_host_t *ha, EXT_IOCTL_ISCSI *ioctl)
{
	int                status = QLA_ERROR;
	uint32_t    mbox_cmd[MBOX_REG_COUNT];
	uint32_t    mbox_sts[MBOX_REG_COUNT];
	EXT_HBA_ISCSI_NODE *hba_node;
	INIT_FW_CTRL_BLK   *init_fw_cb;

	ENTER("qla4extioctl_query_hba_iscsi_node");
#if 0 && __VMKERNEL_MODULE__
	spin_lock(&ha->dma_buf_spinlock);
#else
	down(&ha->dma_buf_sem);
#endif

	hba_node = kmalloc(sizeof(*hba_node), GFP_ATOMIC);
	if (!hba_node
	    || !ha->dma_buf.virt_addr
	    || !ha->dma_buf.phys_addr
	    || !ioctl->ResponseAdr) {
		QL4PRINT(QLP2,
			 printk("scsi%d: %s: memory allocation problem\n",
				ha->host_no, __func__));

		ioctl->Status = EXT_STATUS_NO_MEMORY;
		goto exit_query_hba_node;
	}

	if (ioctl->ResponseLen < sizeof(*hba_node) ||
	    ha->dma_buf.buf_len < sizeof(*init_fw_cb)) {
		QL4PRINT(QLP2,
			 printk("scsi%d: %s: memory area too small\n",
				ha->host_no, __func__));

		ioctl->Status = EXT_STATUS_BUFFER_TOO_SMALL;
		goto exit_query_hba_node;
	}

	/*
	 * Send mailbox command
	 */
	memset(&mbox_cmd, 0, sizeof(mbox_cmd));
	memset(&mbox_sts, 0, sizeof(mbox_sts));
	mbox_cmd[0] = MBOX_CMD_GET_INIT_FW_CTRL_BLOCK;
	mbox_cmd[2] = LSDW(ha->dma_buf.phys_addr);
	mbox_cmd[3] = MSDW(ha->dma_buf.phys_addr);

	if (qla4xxx_mailbox_command(ha, 4, 1, &mbox_cmd[0], &mbox_sts[0])
	    == QLA_ERROR) {
		QL4PRINT(QLP2, printk("scsi%d: %s: command failed \n",
				      ha->host_no, __func__));

		ioctl->Status = EXT_STATUS_ERR;
		ioctl->DetailStatus = mbox_sts[0];
		goto exit_query_hba_node;
	}

	/*
	 * Transfer data from Fw's DEV_DB_ENTRY buffer to IOCTL's
	 * EXT_HBA_ISCSI_NODE buffer
	 */
	init_fw_cb = (INIT_FW_CTRL_BLK *) ha->dma_buf.virt_addr;

	memset(hba_node, 0, sizeof(*hba_node));
	hba_node->PortNumber = __le16_to_cpu(init_fw_cb->PortNumber);
	hba_node->NodeInfo.PortalCount = 1;

	memcpy(hba_node->NodeInfo.IPAddr.IPAddress,
	       init_fw_cb->IPAddr,
	       sizeof(hba_node->NodeInfo.IPAddr.IPAddress));
	memcpy(hba_node->NodeInfo.iSCSIName,
	       init_fw_cb->iSCSINameString,
	       sizeof(hba_node->NodeInfo.iSCSIName));
	memcpy(hba_node->NodeInfo.Alias,
	       init_fw_cb->Alias,
	       sizeof(hba_node->NodeInfo.Alias));

	sprintf(hba_node->DeviceName, "/proc/scsi/%s/HbaApiNode",
		QLA4XXX_PROC_NAME);


	/*
	 * Copy the IOCTL EXT_HBA_ISCSI_NODE buffer to the user's data space
	 */
	if (copy_to_user((void *)_64BIT_TO_ULONG(ioctl->ResponseAdr),
			 hba_node,
			 ioctl->ResponseLen) != 0) {
		QL4PRINT(QLP2, printk("scsi%d: %s: copy failed\n",
				      ha->host_no, __func__));

		ioctl->Status = EXT_STATUS_COPY_ERR;
		goto exit_query_hba_node;
	}

	ioctl->Status = EXT_STATUS_OK;
	status = QLA_SUCCESS;

	exit_query_hba_node:
	if (hba_node) kfree(hba_node);
#if 0 && __VMKERNEL_MODULE__
	spin_unlock(&ha->dma_buf_spinlock);
#else
	up(&ha->dma_buf_sem);
#endif
	LEAVE("qla4extioctl_query_hba_iscsi_node");
	return(status);
}

/**************************************************************************
 * qla4extioctl_query_hba_iscsi_portal
 *	This routine retrieves the HBA iSCSI portal properties
 *
 * Input:
 *	ha    = adapter structure pointer.
 *	ioctl = IOCTL structure pointer.
 *
 * Output:
 *	None
 *
 * Returns:
 *	QLA_SUCCESS = success
 *	QLA_ERROR   = error
 *
 * Context:
 *	Kernel context.
 **************************************************************************/
static int
qla4extioctl_query_hba_iscsi_portal(scsi_qla_host_t *ha, EXT_IOCTL_ISCSI *ioctl)
{
	int status = QLA_ERROR;
	uint32_t mbox_cmd[MBOX_REG_COUNT];
	uint32_t mbox_sts[MBOX_REG_COUNT];
	EXT_HBA_ISCSI_PORTAL *hba_portal;
	FLASH_SYS_INFO *sys_info;
	uint32_t num_valid_ddb_entries;

	ENTER("qla4extioctl_query_hba_iscsi_portal");

	hba_portal = kmalloc(sizeof(*hba_portal), GFP_ATOMIC);
	if (!hba_portal || !ioctl->ResponseAdr) {
		QL4PRINT(QLP2,
			 printk("scsi%d: %s: memory allocation problem\n",
				ha->host_no, __func__));

		ioctl->Status = EXT_STATUS_NO_MEMORY;
		goto exit_query_hba_portal;
	}

	if (ioctl->ResponseLen < sizeof(*hba_portal)) {
		QL4PRINT(QLP2, printk("scsi%d: %s: memory area too small\n",
				      ha->host_no, __func__));

		ioctl->Status = EXT_STATUS_BUFFER_TOO_SMALL;
		goto exit_query_hba_portal;
	}

	/*
	 * Fill in EXT_HBA_ISCSI_PORTAL buffer
	 */
	memset(hba_portal, 0, sizeof(*hba_portal));

	strcpy(hba_portal->DriverVersion, QLA4XXX_DRIVER_VERSION);
	sprintf(hba_portal->FWVersion, "%02d.%02d Patch %02d Build %02d",
		ha->firmware_version[0], ha->firmware_version[1],
		ha->patch_number, ha->build_number);

	/* ----- Get firmware state information ---- */
	memset(&mbox_cmd, 0, sizeof(mbox_cmd));
	memset(&mbox_sts, 0, sizeof(mbox_sts));
	mbox_cmd[0] = MBOX_CMD_GET_FW_STATE;
	if (qla4xxx_mailbox_command(ha, 1, 4, &mbox_cmd[0], &mbox_sts[0])
	    != QLA_SUCCESS) {
		QL4PRINT(QLP2, printk("scsi%d: %s: MBOX_CMD_GET_FW_STATE "
				      "failed w/ status %04X\n",
				      ha->host_no, __func__, mbox_sts[0]));

		goto exit_query_hba_portal;
	}

	switch (mbox_sts[1]) {
	case FW_STATE_READY:
		hba_portal->State = EXT_DEF_CARD_STATE_READY;
		break;
	case FW_STATE_CONFIG_WAIT:
		hba_portal->State = EXT_DEF_CARD_STATE_CONFIG_WAIT;
		break;
	case FW_STATE_WAIT_LOGIN:
		hba_portal->State = EXT_DEF_CARD_STATE_LOGIN;
		break;
	case FW_STATE_ERROR:
		hba_portal->State = EXT_DEF_CARD_STATE_ERROR;
		break;
	}

	switch (mbox_sts[3] & 0x0001) {
	case FW_ADDSTATE_COPPER_MEDIA:
		hba_portal->Type = EXT_DEF_TYPE_COPPER;
		break;
	case FW_ADDSTATE_OPTICAL_MEDIA:
		hba_portal->Type = EXT_DEF_TYPE_OPTICAL;
		break;
	}

	/* ----- Get ddb entry information ---- */
	if (qla4xxx_get_ddb_entry(ha, 0, NULL, &num_valid_ddb_entries,
				  NULL, NULL, NULL, NULL, NULL) == QLA_ERROR) {
		QL4PRINT(QLP2, printk("scsi%d: %s: qla4xxx_get_ddb_entry failed!\n",
				      ha->host_no, __func__));

		goto exit_query_hba_portal;
	}
	hba_portal->DiscTargetCount = (uint16_t) num_valid_ddb_entries;

	/* ----- Get flash sys info information ---- */
#if 0 && __VMKERNEL_MODULE__
	spin_lock(&ha->dma_buf_spinlock);
#else
	down(&ha->dma_buf_sem);
#endif
	sys_info = (FLASH_SYS_INFO *) ha->dma_buf.virt_addr;

	memset(&mbox_cmd, 0, sizeof(mbox_cmd));
	memset(&mbox_sts, 0, sizeof(mbox_sts));
	mbox_cmd[0] = MBOX_CMD_READ_FLASH;
	mbox_cmd[1] = LSDW(ha->dma_buf.phys_addr);
	mbox_cmd[2] = MSDW(ha->dma_buf.phys_addr);
	mbox_cmd[3] = INT_ISCSI_SYSINFO_FLASH_OFFSET;
	mbox_cmd[4] = sizeof(*sys_info);

	if (qla4xxx_mailbox_command(ha, 5, 2, &mbox_cmd[0], &mbox_sts[0])
	    != QLA_SUCCESS) {
		QL4PRINT(QLP2,
			 printk("scsi%d: %s: MBOX_CMD_READ_FLASH failed w/ "
				"status %04X\n",
				ha->host_no, __func__, mbox_sts[0]));

#if 0 && __VMKERNEL_MODULE__
		spin_unlock(&ha->dma_buf_spinlock);
#else
		up(&ha->dma_buf_sem);
#endif
		goto exit_query_hba_portal;
	}

	hba_portal->SerialNum = __le32_to_cpu(sys_info->serialNumber);
	memcpy(hba_portal->IPAddr.IPAddress,
	       ha->ip_address,
	       MIN(sizeof(hba_portal->IPAddr.IPAddress),
		   sizeof(ha->ip_address)));
	memcpy(hba_portal->MacAddr,
	       sys_info->physAddr[0].address,
	       sizeof(hba_portal->MacAddr));
	memcpy(hba_portal->Manufacturer,
	       sys_info->vendorId,
	       sizeof(hba_portal->Manufacturer));
	memcpy(hba_portal->Model,
	       sys_info->productId,
	       sizeof(hba_portal->Model));
#if 0 && __VMKERNEL_MODULE__
	spin_unlock(&ha->dma_buf_spinlock);
#else
	up(&ha->dma_buf_sem);
#endif

	/*memcpy(hba_portal->OptRomVersion, ?,
		sizeof(hba_portal->OptRomVersion)); */

	/*
	 * Copy the IOCTL EXT_HBA_ISCSI_PORTAL buffer to the user's data space
	 */
	if (copy_to_user((void *)_64BIT_TO_ULONG(ioctl->ResponseAdr),
			 hba_portal, ioctl->ResponseLen) != 0) {
		QL4PRINT(QLP2, printk("scsi%d: %s: memory area too small\n",
				      ha->host_no, __func__));

		ioctl->Status = EXT_STATUS_COPY_ERR;
		goto exit_query_hba_portal;
	}

	ioctl->Status = EXT_STATUS_OK;
	status = QLA_SUCCESS;

	exit_query_hba_portal:
	if (hba_portal)	kfree(hba_portal);
	LEAVE("qla4extioctl_query_hba_iscsi_portal");
	return(status);
}

/**************************************************************************
 * qla4extioctl_query_disc_iscsi_node
 *	This routine retrieves the properties of the attached devices
 *	registered as iSCSI nodes discovered by the HBA driver.
 *
 * Input:
 *	ha    = adapter structure pointer.
 *	ioctl = IOCTL structure pointer.
 *
 * Output:
 *	None
 *
 * Returns:
 *	QLA_SUCCESS = success
 *	QLA_ERROR   = error
 *
 * Context:
 *	Kernel context.
 **************************************************************************/
static int
qla4extioctl_query_disc_iscsi_node(scsi_qla_host_t *ha, EXT_IOCTL_ISCSI *ioctl)
{
	int status = QLA_ERROR;
	DEV_DB_ENTRY *fw_ddb_entry = (DEV_DB_ENTRY *) ha->dma_buf.virt_addr;
#if 1 || __VMKERNEL_MODULE__
	EXT_DISC_ISCSI_NODE *disc_node;
#else
	static EXT_DISC_ISCSI_NODE disc_node;
#endif
	ddb_entry_t *ddb_entry;

	ENTER("qla4extioctl_query_disc_iscsi_node");

#if 0 && __VMKERNEL_MODULE__
	spin_lock(&ha->dma_buf_spinlock);
#else
	down(&ha->dma_buf_sem);

	if ( !(disc_node = kmalloc (sizeof(*disc_node), GFP_ATOMIC)) ) {
		QL4PRINT(QLP2,
			 printk("scsi%d: %s: unable to allocate memory.\n",
				ha->host_no, __func__));

		ioctl->Status = EXT_STATUS_NO_MEMORY;
		goto exit_disc_node;
	}

#	define disc_node (*disc_node)
#endif

	if (ioctl->ResponseLen < sizeof(disc_node)) {
		QL4PRINT(QLP2,
			 printk("scsi%d: %s: response buffer too small\n",
				ha->host_no, __func__));

		ioctl->Status = EXT_STATUS_BUFFER_TOO_SMALL;
		goto exit_disc_node;
	}

	if (ha->dma_buf.buf_len < sizeof(DEV_DB_ENTRY)) {
		if (qla4xxx_resize_dma_buf(ha, &ha->dma_buf,
					   sizeof(DEV_DB_ENTRY)) != QLA_SUCCESS) {
			QL4PRINT(QLP2,
				 printk("scsi%d: %s: unable to allocate memory "
					"for dma buffer.\n",
					ha->host_no, __func__));
			ioctl->Status = EXT_STATUS_NO_MEMORY;
			goto exit_disc_node;
		}
	}

	/* ----- get device database entry info from firmware ---- */
	if (qla4xxx_get_ddb_entry(ha, ioctl->Instance, fw_ddb_entry,
				  NULL, NULL, NULL, NULL, NULL, NULL) != QLA_SUCCESS) {
		QL4PRINT(QLP2, printk("scsi%d: %s: failed to get DEV_DB_ENTRY "
				      "info.\n", ha->host_no, __func__));

		ioctl->Status = EXT_STATUS_ERR;
		ioctl->RequestLen = 0;
		ioctl->DetailStatus = ioctl->Instance;
		goto exit_disc_node;
	}

	/* --- Transfer data from Fw's DEV_DB_ENTRY buffer to
	*      IOCTL's EXT_DISC_ISCSI_PORTAL buffer --- */
	memset(&disc_node, 0, sizeof(disc_node));
	disc_node.NodeInfo.PortalCount = 1;
	disc_node.NodeInfo.IPAddr.Type = EXT_DEF_TYPE_ISCSI_IP;
	memcpy(disc_node.NodeInfo.IPAddr.IPAddress, fw_ddb_entry->ipAddr,
	       MIN(sizeof(disc_node.NodeInfo.IPAddr.IPAddress),
		   sizeof(fw_ddb_entry->ipAddr)));
	strncpy(disc_node.NodeInfo.Alias, fw_ddb_entry->iSCSIAlias,
		MIN(sizeof(disc_node.NodeInfo.Alias),
		    sizeof(fw_ddb_entry->iSCSIAlias)));
	strncpy(disc_node.NodeInfo.iSCSIName, fw_ddb_entry->iscsiName,
		MIN(sizeof(disc_node.NodeInfo.iSCSIName),
		    sizeof(fw_ddb_entry->iscsiName)));

	if ((ddb_entry = qla4xxx_lookup_ddb_by_fw_index(ha, ioctl->Instance))
	    == NULL) {
		QL4PRINT(QLP2,
			 printk("scsi%d: %s: device index [%d] not logged in. "
				"Dummy target info returned.\n",
				ha->host_no, __func__, ioctl->Instance));

		disc_node.SessionID       = 0xDEAD;
		disc_node.ConnectionID    = 0xDEAD;
		disc_node.PortalGroupID   = 0xDEAD;
		disc_node.ScsiAddr.Bus    = 0xFF;
		disc_node.ScsiAddr.Target = 0xFF;
		disc_node.ScsiAddr.Lun    = 0xFF;
	}
	else {
		disc_node.SessionID       = ddb_entry->target_session_id;
		disc_node.ConnectionID    = ddb_entry->connection_id;
		disc_node.PortalGroupID   = 0;
		disc_node.ScsiAddr.Bus    = ddb_entry->bus;
		disc_node.ScsiAddr.Target = ddb_entry->target;
		disc_node.ScsiAddr.Lun    = 0;
	}

	/* --- Copy Results to user space --- */
	if (copy_to_user((void*)_64BIT_TO_ULONG(ioctl->ResponseAdr),
			 &disc_node,
			 sizeof(disc_node)) != 0) {
		QL4PRINT(QLP2,
			 printk("scsi%d: %s: copy error to user space.\n",
				ha->host_no, __func__));

		ioctl->Status = EXT_STATUS_COPY_ERR;
		goto exit_disc_node;
	}

	ioctl->Status = EXT_STATUS_OK;
	status = QLA_SUCCESS;

	exit_disc_node:
#if 0 && __VMKERNEL_MODULE__
	spin_unlock(&ha->dma_buf_spinlock);
#else
#	undef disc_node
	if (disc_node) {
		kfree (disc_node);
	}
	up(&ha->dma_buf_sem);
#endif
	LEAVE("qla4extioctl_query_disc_iscsi_node");
	return(status);
}

/**************************************************************************
 * qla4extioctl_query_disc_iscsi_portal
 *	This routine retrieves the properties of the iSCSI portal
 *	discovered by the HBA driver.
 *
 * Input:
 *	ha    = adapter structure pointer.
 *	ioctl = IOCTL structure pointer.
 *
 * Output:
 *	None
 *
 * Returns:
 *	QLA_SUCCESS = success
 *	QLA_ERROR   = error
 *
 * Context:
 *	Kernel context.
 **************************************************************************/
static int
qla4extioctl_query_disc_iscsi_portal(
				    scsi_qla_host_t *ha, EXT_IOCTL_ISCSI *ioctl)
{
	int status = QLA_ERROR;
	DEV_DB_ENTRY *fw_ddb_entry = (DEV_DB_ENTRY *) ha->dma_buf.virt_addr;
	EXT_DISC_ISCSI_PORTAL *disc_portal = kmalloc(sizeof(*disc_portal), GFP_ATOMIC);

	ENTER("qla4extioctl_query_disc_iscsi_portal");

#if 0 && __VMKERNEL_MODULE__
	spin_lock(&ha->dma_buf_spinlock);
#else
	down(&ha->dma_buf_sem);
#endif

	if (!disc_portal) {
		QL4PRINT(QLP2,
			 printk("scsi%d: %s: failed to allocate memory\n",
				ha->host_no, __func__));

		ioctl->Status = EXT_STATUS_NO_MEMORY;
		goto exit_disc_portal;
	}

	if (ioctl->ResponseLen < sizeof(*disc_portal)) {
		QL4PRINT(QLP2,
			 printk("scsi%d: %s: response buffer too small\n",
				ha->host_no, __func__));

		ioctl->Status = EXT_STATUS_BUFFER_TOO_SMALL;
		goto exit_disc_portal;
	}

	if (ha->dma_buf.buf_len < sizeof(DEV_DB_ENTRY)) {
		if (qla4xxx_resize_dma_buf(ha, &ha->dma_buf,
					   sizeof(DEV_DB_ENTRY)) != QLA_SUCCESS) {
			QL4PRINT(QLP2,
				 printk("scsi%d: %s: unable to allocate memory "
					"for dma buffer.\n",
					ha->host_no, __func__));

			ioctl->Status = EXT_STATUS_NO_MEMORY;
			goto exit_disc_portal;
		}
	}

	/* ----- get device database entry info from firmware ---- */
	if (qla4xxx_get_ddb_entry(ha, ioctl->Instance, fw_ddb_entry, NULL,
				  NULL, NULL, NULL, NULL, NULL) != QLA_SUCCESS) {
		QL4PRINT(QLP2,
			 printk("scsi%d: %s: failed to get DEV_DB_ENTRY info.\n",
				ha->host_no, __func__));

		ioctl->Status = EXT_STATUS_ERR;
		ioctl->RequestLen = 0;
		ioctl->DetailStatus = ioctl->Instance;
		goto exit_disc_portal;
	}

	/* --- Transfer data from Fw's DEV_DB_ENTRY buffer to IOCTL's
	*      EXT_DISC_ISCSI_PORTAL buffer --- */
	memset(disc_portal, 0, sizeof(*disc_portal));
	memcpy(disc_portal->IPAddr.IPAddress, fw_ddb_entry->ipAddr,
	       MIN(sizeof(disc_portal->IPAddr.IPAddress),
		   sizeof(fw_ddb_entry->ipAddr)));

	disc_portal->PortNumber = __le16_to_cpu(fw_ddb_entry->portNumber);
	disc_portal->IPAddr.Type = EXT_DEF_TYPE_ISCSI_IP;
	disc_portal->NodeCount = 0;

	strncpy(disc_portal->HostName, fw_ddb_entry->iscsiName,
		MIN(sizeof(disc_portal->HostName),
		    sizeof(fw_ddb_entry->iscsiName)));

	/* --- Copy Results to user space --- */
	if (copy_to_user((void*)_64BIT_TO_ULONG(ioctl->ResponseAdr),
			 disc_portal, sizeof(*disc_portal)) != 0) {
		QL4PRINT(QLP2,
			 printk("scsi%d: %s: copy error to user space.\n",
				ha->host_no, __func__));

		ioctl->Status = EXT_STATUS_COPY_ERR;
		goto exit_disc_portal;
	}

	ioctl->Status = EXT_STATUS_OK;
	status = QLA_SUCCESS;

	exit_disc_portal:
	if (disc_portal) kfree(disc_portal);
#if 0 && __VMKERNEL_MODULE__
	spin_unlock(&ha->dma_buf_spinlock);
#else
	up(&ha->dma_buf_sem);
#endif
	LEAVE("qla4extioctl_query_disc_iscsi_portal");
	return(status);
}

/**************************************************************************
 * qla4extioctl_query_driver
 *	This routine retrieves the driver properties.
 *
 * Input:
 *	ha    = adapter structure pointer.
 *	ioctl = IOCTL structure pointer.
 *
 * Output:
 *	None
 *
 * Returns:
 *	QLA_SUCCESS = success
 *	QLA_ERROR   = error
 *
 * Context:
 *	Kernel context.
 **************************************************************************/
static int
qla4extioctl_query_driver(scsi_qla_host_t *ha, EXT_IOCTL_ISCSI *ioctl)
{
	EXT_DRIVER_INFO *dinfo = kmalloc(sizeof(*dinfo), GFP_ATOMIC);
	int status = QLA_ERROR;

	QL4PRINT(QLP3, printk("scsi%d: %s:\n", ha->host_no, __func__));

	if (!dinfo) {
		QL4PRINT(QLP2,
			 printk("scsi%d: %s: failed to allocate memory\n",
				ha->host_no, __func__));

		ioctl->Status = EXT_STATUS_NO_MEMORY;
		goto exit_query_driver;
	}

	if (ioctl->ResponseLen < sizeof(*dinfo)) {
		QL4PRINT(QLP2,
			 printk("scsi%d: %s: response buffer too small\n",
				ha->host_no, __func__));

		ioctl->Status = EXT_STATUS_BUFFER_TOO_SMALL;
		goto exit_query_driver;
	}

	memset(dinfo, 0, sizeof(*dinfo));
	memcpy(dinfo->Version,
	       QLA4XXX_DRIVER_VERSION,
	       sizeof(QLA4XXX_DRIVER_VERSION));

	dinfo->NumOfBus        = EXT_DEF_MAX_HBA;
	dinfo->TargetsPerBus   = EXT_DEF_MAX_TARGET;
	dinfo->LunPerTarget    = EXT_DEF_MAX_LUN;
	dinfo->LunPerTargetOS  = EXT_DEF_MAX_BUS;

	if (sizeof(dma_addr_t) > 4)
		dinfo->DmaBitAddresses = 1;  /* 64-bit */
	else
		dinfo->DmaBitAddresses = 0;  /* 32-bit */

	#if MEMORY_MAPPED_IO
	dinfo->IoMapType       = 1;  /* Memory Mapped I/O */
	#else
	dinfo->IoMapType       = 0;  /* I/O Mapped I/O */
	#endif

	//FIXME: Incomplete
	//dinfo->MaxTransferLen  = ?;
	//dinfo->MaxDataSegments = ?;
	//dinfo->Attrib          = ?;
	//dinfo->InternalFlags   = ?;

	if (copy_to_user((void*)_64BIT_TO_ULONG(ioctl->ResponseAdr),
			 dinfo,
			 sizeof(*dinfo)) != 0) {
		ioctl->Status = EXT_STATUS_COPY_ERR;
		goto exit_query_driver;
	}

	ioctl->Status = EXT_STATUS_OK;
	status = QLA_SUCCESS;

	exit_query_driver:
	if (dinfo) kfree(dinfo);
	return(status);
}

/**************************************************************************
 * qla4extioctl_query_fw
 *	This routine retrieves the firmware properties.
 *
 * Input:
 *	ha    = adapter structure pointer.
 *	ioctl = IOCTL structure pointer.
 *
 * Output:
 *	None
 *
 * Returns:
 *	QLA_SUCCESS = success
 *	QLA_ERROR   = error
 *
 * Context:
 *	Kernel context.
 **************************************************************************/
static int
qla4extioctl_query_fw(scsi_qla_host_t *ha, EXT_IOCTL_ISCSI *ioctl)
{
	EXT_FW_INFO *fw_info = kmalloc(sizeof(*fw_info), GFP_ATOMIC);
	uint32_t mbox_cmd[MBOX_REG_COUNT];
	uint32_t mbox_sts[MBOX_REG_COUNT];
	int status = QLA_ERROR;

	QL4PRINT(QLP3, printk("scsi%d: %s:\n", ha->host_no, __func__));

	if (!fw_info) {
		QL4PRINT(QLP2,
			 printk("scsi%d: %s: failed to allocate memory\n",
				ha->host_no, __func__));

		ioctl->Status = EXT_STATUS_NO_MEMORY;
		goto exit_query_fw;
	}

	if (ioctl->ResponseLen < sizeof(*fw_info)) {
		QL4PRINT(QLP2,
			 printk("scsi%d: %s: response buffer too small\n",
				ha->host_no, __func__));

		ioctl->Status = EXT_STATUS_BUFFER_TOO_SMALL;
		goto exit_query_fw;
	}

	/* Fill in structure */
	memset(fw_info, 0, sizeof(*fw_info));

	/* ----- Get firmware version information ---- */
	memset(&mbox_cmd, 0, sizeof(mbox_cmd));
	memset(&mbox_sts, 0, sizeof(mbox_sts));
	mbox_cmd[0] = MBOX_CMD_ABOUT_FW;

	/*
	 * NOTE: In QLA4010, mailboxes 2 & 3 may hold an address for data.
	 * Make sure that we write 0 to those mailboxes, if unused.
	 */
	if (qla4xxx_mailbox_command(ha, 4, 5, &mbox_cmd[0], &mbox_sts[0])
	    != QLA_SUCCESS) {
		QL4PRINT(QLP2,
			 printk("scsi%d: %s: MBOX_CMD_ABOUT_FW failed w/ "
				"status %04X\n",
				ha->host_no, __func__, mbox_sts[0]));
		goto exit_query_fw;
	}

	sprintf(fw_info->Version,
		"FW Version %d.%d Patch %d Build %d",
		mbox_sts[1], mbox_sts[2],
		mbox_sts[3], mbox_sts[4]);

	/* Copy info to caller */
	if (copy_to_user((void*)_64BIT_TO_ULONG(ioctl->ResponseAdr),
			 fw_info,
			 sizeof(*fw_info)) != 0) {
		ioctl->Status = EXT_STATUS_COPY_ERR;
		goto exit_query_fw;
	}

	ioctl->Status = EXT_STATUS_OK;
	status = QLA_SUCCESS;

	exit_query_fw:
	return(status);
}

/**************************************************************************
 * qla4extioctl_query_chip
 *	This routine retrieves the chip properties.
 *
 * Input:
 *	ha    = adapter structure pointer.
 *	ioctl = IOCTL structure pointer.
 *
 * Output:
 *	None
 *
 * Returns:
 *	QLA_SUCCESS = success
 *	QLA_ERROR   = error
 *
 * Context:
 *	Kernel context.
 **************************************************************************/
static int
qla4extioctl_query_chip(scsi_qla_host_t *ha, EXT_IOCTL_ISCSI *ioctl)
{
#if 0 && __VMKERNEL_MODULE__
	static uint32_t mbox_cmd[MBOX_REG_COUNT];
	static uint32_t mbox_sts[MBOX_REG_COUNT];
#else
	uint32_t mbox_cmd[MBOX_REG_COUNT];
	uint32_t mbox_sts[MBOX_REG_COUNT];
#endif
	EXT_CHIP_INFO chip_info;
	FLASH_SYS_INFO *sys_info;

	QL4PRINT(QLP3, printk("scsi%d: %s:\n", ha->host_no, __func__));

	if (!ioctl->ResponseAdr || ioctl->ResponseLen < sizeof(chip_info)) {
		QL4PRINT(QLP2,
			 printk("scsi%d: %s: response buffer too small\n",
				ha->host_no, __func__));

		ioctl->Status = EXT_STATUS_BUFFER_TOO_SMALL;
		return(QLA_ERROR);
	}

	/* Fill in structure */
	memset(&chip_info, 0, sizeof(chip_info));

	/* ----- Get flash sys info information ---- */
#if 0 && __VMKERNEL_MODULE__
	spin_lock(&ha->dma_buf_spinlock);
#else
	down(&ha->dma_buf_sem);
#endif
	sys_info = (FLASH_SYS_INFO *) ha->dma_buf.virt_addr;

	memset(&mbox_cmd, 0, sizeof(mbox_cmd));
	memset(&mbox_sts, 0, sizeof(mbox_sts));
	mbox_cmd[0] = MBOX_CMD_READ_FLASH;
	mbox_cmd[1] = LSDW(ha->dma_buf.phys_addr);
	mbox_cmd[2] = MSDW(ha->dma_buf.phys_addr);
	mbox_cmd[3] = INT_ISCSI_SYSINFO_FLASH_OFFSET;
	mbox_cmd[4] = sizeof(*sys_info);

	if (qla4xxx_mailbox_command(ha, 5, 2, &mbox_cmd[0], &mbox_sts[0])
	    != QLA_SUCCESS) {
		QL4PRINT(QLP2,
			 printk("scsi%d: %s: MBOX_CMD_READ_FLASH failed "
				"w/ status %04X\n",
				ha->host_no, __func__, mbox_sts[0]));

#if 0 && __VMKERNEL_MODULE__
		spin_unlock(&ha->dma_buf_spinlock);
#else
		up(&ha->dma_buf_sem);
#endif
		return(QLA_ERROR);
	}
	chip_info.VendorId    = __le32_to_cpu(sys_info->pciDeviceVendor);
	chip_info.DeviceId    = __le32_to_cpu(sys_info->pciDeviceId);
	chip_info.SubVendorId = __le32_to_cpu(sys_info->pciSubsysVendor);
	chip_info.SubSystemId = __le32_to_cpu(sys_info->pciSubsysId);
#if 0 && __VMKERNEL_MODULE__
	spin_unlock(&ha->dma_buf_spinlock);
#else
	up(&ha->dma_buf_sem);
#endif

	/* ----- Get firmware state information ---- */
	memset(&mbox_cmd, 0, sizeof(mbox_cmd));
	memset(&mbox_sts, 0, sizeof(mbox_sts));
	mbox_cmd[0] = MBOX_CMD_GET_FW_STATE;
	if (qla4xxx_mailbox_command(ha, 1, 4, &mbox_cmd[0], &mbox_sts[0])
	    != QLA_SUCCESS) {
		QL4PRINT(QLP2,
			 printk("scsi%d: %s: MBOX_CMD_GET_FW_STATE failed "
				"w/ status %04X\n",
				ha->host_no, __func__, mbox_sts[0]));

		return(QLA_ERROR);
	}

	chip_info.BoardID     = mbox_sts[2];

	/* Copy info to caller */
	if (copy_to_user((void*)_64BIT_TO_ULONG(ioctl->ResponseAdr),
			 &chip_info,
			 sizeof(chip_info)) != 0) {
		ioctl->Status = EXT_STATUS_COPY_ERR;
		return(QLA_ERROR);
	}

	ioctl->Status = EXT_STATUS_OK;
	return(QLA_SUCCESS);
}

/**************************************************************************
 * qla4extioctl_query
 *	This routine calls query IOCTLs based on the IOCTL Sub Code.
 *
 * Input:
 *	ha    = adapter structure pointer.
 *	ioctl = IOCTL structure pointer.
 *
 * Output:
 *	None
 *
 * Returns:
 *	QLA_SUCCESS = success
 *	QLA_ERROR   = error
 *    	-EINVAL     = if the command is invalid
 *
 * Context:
 *	Kernel context.
 **************************************************************************/
static int
qla4extioctl_query(scsi_qla_host_t *ha, EXT_IOCTL_ISCSI *ioctl)
{
	switch (ioctl->SubCode) {
	case EXT_SC_QUERY_HBA_ISCSI_NODE:
		return(qla4extioctl_query_hba_iscsi_node(ha, ioctl));

	case EXT_SC_QUERY_HBA_ISCSI_PORTAL:
		return(qla4extioctl_query_hba_iscsi_portal(ha, ioctl));

	case EXT_SC_QUERY_DISC_ISCSI_NODE:
		return(qla4extioctl_query_disc_iscsi_node(ha, ioctl));

	case EXT_SC_QUERY_DISC_ISCSI_PORTAL:
		return(qla4extioctl_query_disc_iscsi_portal(ha, ioctl));

	case EXT_SC_QUERY_DRIVER:
		return(qla4extioctl_query_driver(ha, ioctl));

	case EXT_SC_QUERY_FW:
		return(qla4extioctl_query_fw(ha, ioctl));

	case EXT_SC_QUERY_CHIP:
		return(qla4extioctl_query_chip(ha, ioctl));

	default:
		QL4PRINT(QLP2,
			 printk("scsi%d: %s: unsupported query sub-command "
				"code (%X)\n",
				ha->host_no, __func__, ioctl->SubCode));

		ioctl->Status = EXT_STATUS_UNSUPPORTED_SUBCODE;
		return(-EINVAL);
	}
}

/**************************************************************************
 * qla4extioctl_reg_aen
 *	This routine enables/disables storing of asynchronous events
 *	from the ISP into the driver's internal buffer.
 *
 * Input:
 *	ha    = adapter structure pointer.
 *	ioctl = IOCTL structure pointer.
 *
 * Output:
 *	None
 *
 * Returns:
 *	QLA_SUCCESS = success
 *	QLA_ERROR   = error
 *
 * Context:
 *	Kernel context.
 **************************************************************************/
static int
qla4extioctl_reg_aen(scsi_qla_host_t *ha, EXT_IOCTL_ISCSI *ioctl)
{
	int status = QLA_ERROR;
	ioctl->Status = EXT_STATUS_UNSUPPORTED_SUBCODE;
	ENTER("qla4extioctl_reg_aen");
	QL4PRINT(QLP4, printk("scsi%d: %s: UNSUPPORTED\n", ha->host_no, __func__));
	LEAVE("qla4extioctl_reg_aen");
	return(status);
}

/**************************************************************************
 * qla4extioctl_get_aen
 *	This routine retrieves the contents of the driver's internal
 *	asynchronous event tracking queue.
 *
 * Input:
 *	ha    = adapter structure pointer.
 *	ioctl = IOCTL structure pointer.
 *
 * Output:
 *	None
 *
 * Returns:
 *	QLA_SUCCESS = success
 *	QLA_ERROR   = error
 *
 * Context:
 *	Kernel context.
 **************************************************************************/
static int
qla4extioctl_get_aen(scsi_qla_host_t *ha, EXT_IOCTL_ISCSI *ioctl)
{
	int status = QLA_ERROR;
	ioctl->Status = EXT_STATUS_UNSUPPORTED_SUBCODE;
	ENTER("qla4extioctl_get_aen");
	QL4PRINT(QLP4, printk("scsi%d: %s: UNSUPPORTED\n", ha->host_no, __func__));
	LEAVE("qla4extioctl_get_aen");
	return(status);
}

/**************************************************************************
 * qla4extioctl_get_statistics_gen
 *	This routine retrieves the HBA general statistical information.
 *
 * Input:
 *	ha    = adapter structure pointer.
 *	ioctl = IOCTL structure pointer.
 *
 * Output:
 *	None
 *
 * Returns:
 *	QLA_SUCCESS = success
 *	QLA_ERROR   = error
 *
 * Context:
 *	Kernel context.
 **************************************************************************/
static int
qla4extioctl_get_statistics_gen(scsi_qla_host_t *ha, EXT_IOCTL_ISCSI *ioctl)
{
	int status = QLA_ERROR;
	EXT_HBA_PORT_STAT_GEN   *stat_gen;

	ENTER("qla4extioctl_get_statistics_gen");
	QL4PRINT(QLP4, printk("scsi%d: %s: index [%d]\n",
			      ha->host_no, __func__, ioctl->Instance));

	stat_gen = kmalloc(sizeof(*stat_gen), GFP_ATOMIC);
	if (!stat_gen || !ioctl->ResponseAdr) {
		QL4PRINT(QLP2,
			 printk("scsi%d: %s: memory allocation problem\n",
				ha->host_no, __func__));

		ioctl->Status = EXT_STATUS_NO_MEMORY;
		goto exit_get_stat_gen;
	}

	if (ioctl->ResponseLen < sizeof(*stat_gen)) {
		QL4PRINT(QLP2, printk("scsi%d: %s: memory area too small\n",
				      ha->host_no, __func__));

		ioctl->Status = EXT_STATUS_BUFFER_TOO_SMALL;
		goto exit_get_stat_gen;
	}

	/*
	 * Fill in the data
	 */
	memset(stat_gen, 0, sizeof(*stat_gen));
	stat_gen->HBAPortErrorCount     = ha->adapter_error_count;
	stat_gen->DevicePortErrorCount  = ha->device_error_count;
	stat_gen->IoCount               = ha->total_io_count;
	stat_gen->MBytesCount           = ha->total_mbytes_xferred;
	stat_gen->InterruptCount        = ha->isr_count;
	stat_gen->LinkFailureCount      = ha->link_failure_count;
	stat_gen->InvalidCrcCount       = ha->invalid_crc_count;

	/*
	 * Copy the IOCTL EXT_HBA_PORT_STAT_GEN buffer to the user's data space
	 */
	if (copy_to_user((void *)_64BIT_TO_ULONG(ioctl->ResponseAdr),
			 stat_gen,
			 ioctl->ResponseLen) != 0) {
		QL4PRINT(QLP2, printk("scsi%d: %s: memory area too small\n",
				      ha->host_no, __func__));

		ioctl->Status = EXT_STATUS_COPY_ERR;
		goto exit_get_stat_gen;
	}

	ioctl->Status = EXT_STATUS_OK;
	status = QLA_SUCCESS;

	exit_get_stat_gen:
	if (stat_gen) kfree(stat_gen);
	LEAVE("qla4extioctl_get_statistics_gen");
	return(status);
}

/**************************************************************************
 * qla4extioctl_get_statistics_iscsi
 *	This routine retrieves the HBA iSCSI statistical information.
 *
 * Input:
 *	ha    = adapter structure pointer.
 *	ioctl = IOCTL structure pointer.
 *
 * Output:
 *	None
 *
 * Returns:
 *	QLA_SUCCESS = success
 *	QLA_ERROR   = error
 *
 * Context:
 *	Kernel context.
 **************************************************************************/
static int
qla4extioctl_get_statistics_iscsi(scsi_qla_host_t *ha, EXT_IOCTL_ISCSI *ioctl)
{
	int status = QLA_ERROR;
	uint32_t mbox_cmd[MBOX_REG_COUNT];
	uint32_t mbox_sts[MBOX_REG_COUNT];
	EXT_HBA_PORT_STAT_ISCSI* stat_local;
	EXT_HBA_PORT_STAT_ISCSI* stat_user;

	ENTER("qla4extioctl_get_statistics_iscsi");
	QL4PRINT(QLP4, printk("scsi%d: %s: index [%d]\n",
			      ha->host_no, __func__, ioctl->Instance));

#if 0 && __VMKERNEL_MODULE__
	spin_lock(&ha->dma_buf_spinlock);
#else
	down(&ha->dma_buf_sem);
#endif

	stat_user = kmalloc(sizeof(EXT_HBA_PORT_STAT_ISCSI), GFP_ATOMIC);

	if (!stat_user) {
		QL4PRINT(QLP2, printk("scsi%d: %s: unable to allocate memory.\n",
				      ha->host_no, __func__));

		ioctl->Status = EXT_STATUS_NO_MEMORY;
	}

	if (!ioctl->ResponseAdr || !ioctl->ResponseLen) {
		QL4PRINT(QLP2, printk("scsi%d: %s: invalid parameter\n",
				      ha->host_no, __func__));

		ioctl->Status = EXT_STATUS_INVALID_PARAM;
		goto exit_get_stats_iscsi;
	}

	if (ioctl->ResponseLen < sizeof(EXT_HBA_PORT_STAT_ISCSI)) {
		QL4PRINT(QLP2, printk("scsi%d: %s: RespLen too small (0x%x),  "
				      "need (0x%x).\n",
				      ha->host_no, __func__,
				      ioctl->ResponseLen,
				      (unsigned int) sizeof(EXT_HBA_PORT_STAT_ISCSI)));

		ioctl->Status = EXT_STATUS_BUFFER_TOO_SMALL;
		goto exit_get_stats_iscsi;
	}

	if ((ha->dma_buf.buf_len < sizeof(EXT_HBA_PORT_STAT_ISCSI)) &&
	    (qla4xxx_resize_dma_buf(ha, &ha->dma_buf, sizeof(EXT_HBA_PORT_STAT_ISCSI))
	     != QLA_SUCCESS)) {
		QL4PRINT(QLP2, printk("scsi%d: %s: unable to allocate memory "
				      "for dma buffer.\n",
				      ha->host_no, __func__));

		ioctl->Status = EXT_STATUS_NO_MEMORY;
		goto exit_get_stats_iscsi;
	}

	/*
	 * Make the mailbox call
	 */
	memset(&mbox_cmd, 0, sizeof(mbox_cmd));
	memset(&mbox_sts, 0, sizeof(mbox_sts));
	mbox_cmd[0] = MBOX_CMD_GET_MANAGEMENT_DATA;
	mbox_cmd[1] = ioctl->Instance;
	mbox_cmd[2] = LSDW(ha->dma_buf.phys_addr);
	mbox_cmd[3] = MSDW(ha->dma_buf.phys_addr);

	if (qla4xxx_mailbox_command(ha, 4, 1, &mbox_cmd[0], &mbox_sts[0])
	    != QLA_SUCCESS) {
		QL4PRINT(QLP2,
			 printk("scsi%d: %s: get mngmt data for index [%d] failed "
				"w/ mailbox ststus 0x%x\n",
				ha->host_no, __func__, ioctl->Instance, mbox_sts[0]));

		ioctl->Status = EXT_STATUS_MAILBOX;
		ioctl->DetailStatus = mbox_sts[0];
		goto exit_get_stats_iscsi;
	}

	stat_local = (EXT_HBA_PORT_STAT_ISCSI *) ha->dma_buf.virt_addr;
	memset(stat_user, 0, sizeof(*stat_user));
	stat_user->MACTxFramesCount          = __le64_to_cpu(stat_local->MACTxFramesCount);
	stat_user->MACTxBytesCount           = __le64_to_cpu(stat_local->MACTxBytesCount);
	stat_user->MACRxFramesCount          = __le64_to_cpu(stat_local->MACRxFramesCount);
	stat_user->MACRxBytesCount           = __le64_to_cpu(stat_local->MACRxBytesCount);
	stat_user->MACCRCErrorCount          = __le64_to_cpu(stat_local->MACCRCErrorCount);
	stat_user->MACEncodingErrorCount     = __le64_to_cpu(stat_local->MACEncodingErrorCount);
	stat_user->IPTxPacketsCount          = __le64_to_cpu(stat_local->IPTxPacketsCount);
	stat_user->IPTxBytesCount            = __le64_to_cpu(stat_local->IPTxBytesCount);
	stat_user->IPTxFragmentsCount        = __le64_to_cpu(stat_local->IPTxFragmentsCount);
	stat_user->IPRxPacketsCount          = __le64_to_cpu(stat_local->IPRxPacketsCount);
	stat_user->IPRxBytesCount            = __le64_to_cpu(stat_local->IPRxBytesCount);
	stat_user->IPRxFragmentsCount        = __le64_to_cpu(stat_local->IPRxFragmentsCount);
	stat_user->IPDatagramReassemblyCount = __le64_to_cpu(stat_local->IPDatagramReassemblyCount);
	stat_user->IPv6RxPacketsCount        = __le64_to_cpu(stat_local->IPv6RxPacketsCount);
	stat_user->IPRxPacketErrorCount      = __le64_to_cpu(stat_local->IPRxPacketErrorCount);
	stat_user->IPReassemblyErrorCount    = __le64_to_cpu(stat_local->IPReassemblyErrorCount);
	stat_user->TCPTxSegmentsCount        = __le64_to_cpu(stat_local->TCPTxSegmentsCount);
	stat_user->TCPTxBytesCount           = __le64_to_cpu(stat_local->TCPTxBytesCount);
	stat_user->TCPRxSegmentsCount        = __le64_to_cpu(stat_local->TCPRxSegmentsCount);
	stat_user->TCPRxBytesCount           = __le64_to_cpu(stat_local->TCPRxBytesCount);
	stat_user->TCPTimerExpiredCount      = __le64_to_cpu(stat_local->TCPTimerExpiredCount);
	stat_user->TCPRxACKCount             = __le64_to_cpu(stat_local->TCPRxACKCount);
	stat_user->TCPTxACKCount             = __le64_to_cpu(stat_local->TCPTxACKCount);
	stat_user->TCPRxErrorSegmentCount    = __le64_to_cpu(stat_local->TCPRxErrorSegmentCount);
	stat_user->TCPWindowProbeUpdateCount = __le64_to_cpu(stat_local->TCPWindowProbeUpdateCount);
	stat_user->iSCSITxPDUCount           = __le64_to_cpu(stat_local->iSCSITxPDUCount);
	stat_user->iSCSITxBytesCount         = __le64_to_cpu(stat_local->iSCSITxBytesCount);
	stat_user->iSCSIRxPDUCount           = __le64_to_cpu(stat_local->iSCSIRxPDUCount);
	stat_user->iSCSIRxBytesCount         = __le64_to_cpu(stat_local->iSCSIRxBytesCount);
	stat_user->iSCSICompleteIOsCount     = __le64_to_cpu(stat_local->iSCSICompleteIOsCount);
	stat_user->iSCSIUnexpectedIORxCount  = __le64_to_cpu(stat_local->iSCSIUnexpectedIORxCount);
	stat_user->iSCSIFormatErrorCount     = __le64_to_cpu(stat_local->iSCSIFormatErrorCount);
	stat_user->iSCSIHeaderDigestCount    = __le64_to_cpu(stat_local->iSCSIHeaderDigestCount);
	stat_user->iSCSIDataDigestErrorCount = __le64_to_cpu(stat_local->iSCSIDataDigestErrorCount);
	stat_user->iSCSISeqErrorCount        = __le64_to_cpu(stat_local->iSCSISeqErrorCount);


	/*
	 * Copy the data from the dma buffer to the user's data space
	 */
	if (copy_to_user((void *)_64BIT_TO_ULONG(ioctl->ResponseAdr),
			 stat_user,
			 ioctl->ResponseLen) != 0) {
		QL4PRINT(QLP2, printk("scsi%d: %s: memory area too small\n",
				      ha->host_no, __func__));

		ioctl->Status = EXT_STATUS_COPY_ERR;
		goto exit_get_stats_iscsi;
	}

	ioctl->Status = EXT_STATUS_OK;
	status = QLA_SUCCESS;

	exit_get_stats_iscsi:
	if (stat_user) kfree(stat_user);
#if 0 && __VMKERNEL_MODULE__
	spin_unlock(&ha->dma_buf_spinlock);
#else
	up(&ha->dma_buf_sem);
#endif
	LEAVE("qla4extioctl_get_statistics_iscsi");
	return(status);
}

/**************************************************************************
 * qla4extioctl_get_device_entry_iscsi
 *	This routine retrieves the database entry for the specified device.
 *
 * Input:
 *	ha    = adapter structure pointer.
 *	ioctl = IOCTL structure pointer.
 *
 * Output:
 *	None
 *
 * Returns:
 *	QLA_SUCCESS = success
 *	QLA_ERROR   = error
 *
 * Context:
 *	Kernel context.
 **************************************************************************/
static int
qla4extioctl_get_device_entry_iscsi(scsi_qla_host_t *ha, EXT_IOCTL_ISCSI *ioctl)
{
	int                     status = QLA_ERROR;
	uint32_t         mbox_cmd[MBOX_REG_COUNT];
	uint32_t         mbox_sts[MBOX_REG_COUNT];
	DEV_DB_ENTRY             *fw_ddb_entry;
	EXT_DEVICE_ENTRY_ISCSI  *dev_entry;

	ENTER("qla4extioctl_get_device_entry_iscsi");
	QL4PRINT(QLP4, printk("scsi%d: %s: index [%d]\n",
			      ha->host_no, __func__, ioctl->Instance));

#if 0 && __VMKERNEL_MODULE__
	spin_lock(&ha->dma_buf_spinlock);
#else
	down(&ha->dma_buf_sem);
#endif
	dev_entry = kmalloc(sizeof(*dev_entry), GFP_ATOMIC);

#ifdef __VMKERNEL_MODULE__
/*
 * The stock driver doesn't resize the DMA buffer in this case.  This
 * was causing IMA to return OS Failures when the buffer wasn't large
 * enough.  We'll resize the buffer in the vmkernel.
 */
	if (!ha->dma_buf.virt_addr
	    || !ha->dma_buf.phys_addr
	    || (ha->dma_buf.buf_len < sizeof(*fw_ddb_entry))) {
		if (qla4xxx_resize_dma_buf(ha, &ha->dma_buf,
		   sizeof(DEV_DB_ENTRY)) != QLA_SUCCESS) {
			QL4PRINT(QLP2,
				 printk("scsi%d: %s: unable to allocate memory "
					"for dma buffer.\n",
					ha->host_no, __func__));

			ioctl->Status = EXT_STATUS_NO_MEMORY;
			ioctl->ResponseLen = 0;
			goto exit_get_dev_entry;
		}
	}

	if (!dev_entry || !ioctl->ResponseAdr) {
		QL4PRINT(QLP2,
			 printk("scsi%d: %s: memory allocation problem\n",
				ha->host_no, __func__));
		ioctl->Status = EXT_STATUS_NO_MEMORY;
		ioctl->ResponseLen = 0;
		goto exit_get_dev_entry;
	}

	if (ioctl->ResponseLen < sizeof(*dev_entry)) {
		QL4PRINT(QLP2, printk("scsi%d: %s: memory area too small\n",
				      ha->host_no, __func__));

		ioctl->Status = EXT_STATUS_BUFFER_TOO_SMALL;
		goto exit_get_dev_entry;
	}

#else
	if (!dev_entry
	    || !ha->dma_buf.virt_addr
	    || !ha->dma_buf.phys_addr
	    || !ioctl->ResponseAdr) {
		QL4PRINT(QLP2,
			 printk("scsi%d: %s: memory allocation problem\n",
				ha->host_no, __func__));

		ioctl->Status = EXT_STATUS_NO_MEMORY;
		goto exit_get_dev_entry;
	}

	if (ioctl->ResponseLen < sizeof(*dev_entry) ||
	    ha->dma_buf.buf_len < sizeof(*fw_ddb_entry)) {
		QL4PRINT(QLP2, printk("scsi%d: %s: memory area too small\n",
				      ha->host_no, __func__));

		ioctl->Status = EXT_STATUS_BUFFER_TOO_SMALL;
		goto exit_get_dev_entry;
	}
#endif // __VMKERNEL_MODULE__

	/*
	 * Make the mailbox call
	 */
	memset(&mbox_cmd, 0, sizeof(mbox_cmd));
	memset(&mbox_sts, 0, sizeof(mbox_sts));
	memset(dev_entry, 0, sizeof(*dev_entry));
	if (ioctl->SubCode == EXT_SC_GET_DEVICE_ENTRY_ISCSI)
		mbox_cmd[0] = MBOX_CMD_GET_DATABASE_ENTRY;
	else
		mbox_cmd[0] = MBOX_CMD_GET_DATABASE_ENTRY_DEFAULTS;
	mbox_cmd[1] = ioctl->Instance;
	mbox_cmd[2] = LSDW(ha->dma_buf.phys_addr);
	mbox_cmd[3] = MSDW(ha->dma_buf.phys_addr);

	if (qla4xxx_mailbox_command(ha, 4, 5, &mbox_cmd[0], &mbox_sts[0]) != QLA_SUCCESS) {
		QL4PRINT(QLP2, printk("scsi%d: %s: get ddb entry for index [%d] failed "
				      "w/ mailbox ststus 0x%x\n",
				      ha->host_no, __func__,
				      ioctl->Instance, mbox_sts[0]));

		ioctl->Status = EXT_STATUS_MAILBOX;
		ioctl->DetailStatus = mbox_sts[0];
		goto exit_get_dev_entry;
	}

	/*
	 * Transfer data from Fw's DEV_DB_ENTRY buffer to IOCTL's
	 * EXT_DEVICE_ENTRY_ISCSI buffer
	 */
	fw_ddb_entry = ha->dma_buf.virt_addr;

	dev_entry->NumValid                     = mbox_sts[2];
	dev_entry->NextValid                    = mbox_sts[3];
	dev_entry->DeviceState                  = mbox_sts[4];
	dev_entry->Options                      = fw_ddb_entry->options;
	dev_entry->Control                      = fw_ddb_entry->control;
	dev_entry->TargetSessID                 = __le16_to_cpu(fw_ddb_entry->TSID);
	memcpy(dev_entry->InitiatorSessID,
	       fw_ddb_entry->ISID,
	       sizeof(fw_ddb_entry->ISID));

	dev_entry->DeviceInfo.DeviceType        = __le16_to_cpu(EXT_DEF_ISCSI_REMOTE);
	dev_entry->DeviceInfo.ExeThrottle       = __le16_to_cpu(fw_ddb_entry->exeThrottle);
	dev_entry->DeviceInfo.InitMarkerlessInt = __le16_to_cpu(fw_ddb_entry->iSCSIMaxSndDataSegLen);
	dev_entry->DeviceInfo.RetryCount        = fw_ddb_entry->retryCount;
	dev_entry->DeviceInfo.RetryDelay        = fw_ddb_entry->retryDelay;
	dev_entry->DeviceInfo.iSCSIOptions      = __le16_to_cpu(fw_ddb_entry->iSCSIOptions);
	dev_entry->DeviceInfo.TCPOptions        = __le16_to_cpu(fw_ddb_entry->TCPOptions);
	dev_entry->DeviceInfo.IPOptions         = __le16_to_cpu(fw_ddb_entry->IPOptions);
	dev_entry->DeviceInfo.MaxPDUSize        = __le16_to_cpu(fw_ddb_entry->maxPDUSize);
	dev_entry->DeviceInfo.FirstBurstSize    = __le16_to_cpu(fw_ddb_entry->firstBurstSize);
	dev_entry->DeviceInfo.LogoutMinTime     = __le16_to_cpu(fw_ddb_entry->minTime2Wait);
	dev_entry->DeviceInfo.LogoutMaxTime     = __le16_to_cpu(fw_ddb_entry->maxTime2Retain);
	dev_entry->DeviceInfo.MaxOutstandingR2T = __le16_to_cpu(fw_ddb_entry->maxOutstndngR2T);
	dev_entry->DeviceInfo.KeepAliveTimeout  = __le16_to_cpu(fw_ddb_entry->keepAliveTimeout);
	dev_entry->DeviceInfo.PortNumber        = __le16_to_cpu(fw_ddb_entry->portNumber);
	dev_entry->DeviceInfo.MaxBurstSize      = __le16_to_cpu(fw_ddb_entry->maxBurstSize);
	dev_entry->DeviceInfo.TaskMgmtTimeout   = __le16_to_cpu(fw_ddb_entry->taskMngmntTimeout);
	dev_entry->EntryInfo.PortalCount        = mbox_sts[2];
	dev_entry->ExeCount                     = __le16_to_cpu(fw_ddb_entry->exeCount);
	dev_entry->DDBLink                      = __le16_to_cpu(fw_ddb_entry->ddbLink);

	memcpy(dev_entry->UserID,
	       fw_ddb_entry->userID,
	       sizeof(dev_entry->UserID));
	memcpy(dev_entry->Password,
	       fw_ddb_entry->password,
	       sizeof(dev_entry->Password));

	memcpy(dev_entry->DeviceInfo.TargetAddr,
	       fw_ddb_entry->targetAddr,
	       sizeof(dev_entry->DeviceInfo.TargetAddr));
	memcpy(dev_entry->EntryInfo.IPAddr.IPAddress,
	       fw_ddb_entry->ipAddr,
	       sizeof(dev_entry->EntryInfo.IPAddr.IPAddress));
	memcpy(dev_entry->EntryInfo.iSCSIName,
	       fw_ddb_entry->iscsiName,
	       sizeof(dev_entry->EntryInfo.iSCSIName));
	memcpy(dev_entry->EntryInfo.Alias,
	       fw_ddb_entry->iSCSIAlias,
	       sizeof(dev_entry->EntryInfo.Alias));

	QL4PRINT(QLP10, printk("scsi%d: DEV_DB_ENTRY structure:\n", ha->host_no));
	qla4xxx_dump_bytes(QLP10, fw_ddb_entry, sizeof(*fw_ddb_entry));
	QL4PRINT(QLP10, printk("scsi%d: EXT_DEVICE_ENTRY_ISCSI structure:\n", ha->host_no));
	qla4xxx_dump_bytes(QLP10, dev_entry, sizeof(*dev_entry));

	/*
	 * Copy the IOCTL EXT_DEVICE_ENTRY_ISCSI buffer to the user's data space
	 */
	if (copy_to_user((void *)_64BIT_TO_ULONG(ioctl->ResponseAdr),
			 dev_entry, ioctl->ResponseLen) != 0) {
		QL4PRINT(QLP2, printk("scsi%d: %s: memory area too small\n",
				      ha->host_no, __func__));

		ioctl->Status = EXT_STATUS_COPY_ERR;
		goto exit_get_dev_entry;
	}

	ioctl->Status = EXT_STATUS_OK;
	status = QLA_SUCCESS;

	exit_get_dev_entry:
#if 0 && __VMKERNEL_MODULE__
	spin_unlock(&ha->dma_buf_spinlock);
#else
	up(&ha->dma_buf_sem);
#endif
	if (dev_entry) kfree(dev_entry);
	LEAVE("qla4extioctl_get_device_entry_iscsi");
	return(status);
}


/**************************************************************************
 * qla4extioctl_get_init_fw_iscsi
 *	This routine retrieves the initialize firmware control block for
 *	the specified HBA.
 *
 * Input:
 *	ha    = adapter structure pointer.
 *	ioctl = IOCTL structure pointer.
 *
 * Output:
 *	None
 *
 * Returns:
 *	QLA_SUCCESS = success
 *	QLA_ERROR   = error
 *
 * Context:
 *	Kernel context.
 **************************************************************************/
static int
qla4extioctl_get_init_fw_iscsi(scsi_qla_host_t *ha, EXT_IOCTL_ISCSI *ioctl)
{
	int               status   = QLA_ERROR;
	uint32_t   mbox_cmd[MBOX_REG_COUNT];
	uint32_t   mbox_sts[MBOX_REG_COUNT];
	EXT_INIT_FW_ISCSI *init_fw;
	INIT_FW_CTRL_BLK  *init_fw_cb;

	ENTER("qla4extioctl_get_init_fw_iscsi");
	QL4PRINT(QLP3, printk("scsi%d: %s:\n", ha->host_no, __func__));

#if 0 && __VMKERNEL_MODULE__
	spin_lock(&ha->dma_buf_spinlock);
#else
	down(&ha->dma_buf_sem);
#endif

	init_fw = kmalloc(sizeof(*init_fw), GFP_ATOMIC);
	if (!ha->dma_buf.virt_addr
	    || !ha->dma_buf.phys_addr
	    || !init_fw
	    || (ha->dma_buf.buf_len < sizeof(INIT_FW_CTRL_BLK))
	    || (ioctl->ResponseLen < sizeof(EXT_INIT_FW_ISCSI))
	   ) {
		QL4PRINT(QLP2,
			 printk("scsi%d: %s: response buffer too small\n",
				ha->host_no, __func__));

		ioctl->Status = EXT_STATUS_BUFFER_TOO_SMALL;
		goto exit_get_init_fw;
	}

	/*
	 * Send mailbox command
	 */
	memset(&mbox_cmd, 0, sizeof(mbox_cmd));
	memset(&mbox_sts, 0, sizeof(mbox_sts));
	switch (ioctl->SubCode) {
	case EXT_SC_GET_INIT_FW_ISCSI:
		mbox_cmd[0] = MBOX_CMD_GET_INIT_FW_CTRL_BLOCK;
		break;
	case EXT_SC_GET_INIT_FW_DEFAULTS_ISCSI:
		mbox_cmd[0] = MBOX_CMD_GET_INIT_FW_CTRL_BLOCK_DEFAULTS;
		break;
	default:
		QL4PRINT(QLP2,
			 printk("scsi%d: %s: invalid subcode (0x%04X) speficied\n",
				ha->host_no, __func__, ioctl->SubCode));

		ioctl->Status = EXT_STATUS_INVALID_PARAM;
		goto exit_get_init_fw;
	}

	mbox_cmd[1] = 0;
	mbox_cmd[2] = LSDW(ha->dma_buf.phys_addr);
	mbox_cmd[3] = MSDW(ha->dma_buf.phys_addr);

	if (qla4xxx_mailbox_command(ha, 4, 1, &mbox_cmd[0], &mbox_sts[0])
	    == QLA_ERROR) {
		QL4PRINT(QLP2, printk("scsi%d: %s: command failed \n",
				      ha->host_no, __func__));

		ioctl->Status = EXT_STATUS_ERR;
		ioctl->DetailStatus = mbox_sts[0];
		goto exit_get_init_fw;
	}

	/*
	 * Transfer Data from DMA buffer to Local buffer
	 */
	init_fw_cb = (INIT_FW_CTRL_BLK *)ha->dma_buf.virt_addr;
	memset(init_fw, 0, sizeof(*init_fw));

	init_fw->Version         = init_fw_cb->Version;
	init_fw->FWOptions       = __le16_to_cpu(init_fw_cb->FwOptions);
	init_fw->AddFWOptions    = __le16_to_cpu(init_fw_cb->AddFwOptions);
	//FIXME: init_fw->WakeupThreshold = __le16_to_cpu(init_fw_cb->WakeupThreshold);
	memcpy(&init_fw->IPAddr.IPAddress,
	       &init_fw_cb->IPAddr,
	       MIN(sizeof(init_fw->IPAddr.IPAddress),
		   sizeof(init_fw_cb->IPAddr)));
	memcpy(&init_fw->SubnetMask.IPAddress,
	       &init_fw_cb->SubnetMask,
	       MIN(sizeof(init_fw->SubnetMask.IPAddress),
		   sizeof(init_fw_cb->SubnetMask)));
	memcpy(&init_fw->Gateway.IPAddress,
	       &init_fw_cb->GatewayIPAddr,
	       MIN(sizeof(init_fw->Gateway.IPAddress),
		   sizeof(init_fw_cb->GatewayIPAddr)));
	memcpy(&init_fw->DNSConfig.IPAddr.IPAddress,
	       &init_fw_cb->PriDNSIPAddr,
	       MIN(sizeof(init_fw->DNSConfig.IPAddr.IPAddress),
		   sizeof(init_fw_cb->PriDNSIPAddr)));
	memcpy(&init_fw->Alias,
	       &init_fw_cb->Alias,
	       MIN(sizeof(init_fw->Alias),
		   sizeof(init_fw_cb->Alias)));
	memcpy(&init_fw->iSCSIName,
	       &init_fw_cb->iSCSINameString,
	       MIN(sizeof(init_fw->iSCSIName),
		   sizeof(init_fw_cb->iSCSINameString)));

	init_fw->DeviceInfo.DeviceType        = __le16_to_cpu(EXT_DEF_ISCSI_LOCAL);
	init_fw->DeviceInfo.ExeThrottle       = __le16_to_cpu(init_fw_cb->ExecThrottle);
	init_fw->DeviceInfo.InitMarkerlessInt = __le16_to_cpu(init_fw_cb->InitMarkerlessInt);
	init_fw->DeviceInfo.RetryCount        = init_fw_cb->RetryCount;
	init_fw->DeviceInfo.RetryDelay        = init_fw_cb->RetryDelay;
	init_fw->DeviceInfo.iSCSIOptions      = __le16_to_cpu(init_fw_cb->iSCSIOptions);
	init_fw->DeviceInfo.TCPOptions        = __le16_to_cpu(init_fw_cb->TCPOptions);
	init_fw->DeviceInfo.IPOptions         = __le16_to_cpu(init_fw_cb->IPOptions);
	init_fw->DeviceInfo.MaxPDUSize        = __le16_to_cpu(init_fw_cb->MaxPDUSize);
	init_fw->DeviceInfo.FirstBurstSize    = __le16_to_cpu(init_fw_cb->FirstBurstSize);
	init_fw->DeviceInfo.LogoutMinTime     = __le16_to_cpu(init_fw_cb->DefaultTime2Wait);
	init_fw->DeviceInfo.LogoutMaxTime     = __le16_to_cpu(init_fw_cb->DefaultTime2Retain);
	init_fw->DeviceInfo.LogoutMaxTime     = __le16_to_cpu(init_fw_cb->DefaultTime2Retain);
	init_fw->DeviceInfo.MaxOutstandingR2T = __le16_to_cpu(init_fw_cb->MaxOutStndngR2T);
	init_fw->DeviceInfo.KeepAliveTimeout  = __le16_to_cpu(init_fw_cb->KeepAliveTimeout);
	init_fw->DeviceInfo.PortNumber        = __le16_to_cpu(init_fw_cb->PortNumber);
	init_fw->DeviceInfo.MaxBurstSize      = __le16_to_cpu(init_fw_cb->MaxBurstSize);
	//init_fw->DeviceInfo.TaskMgmtTimeout   = init_fw_cb->T;
	memcpy(&init_fw->DeviceInfo.TargetAddr,
	       &init_fw_cb->TargAddr,
	       EXT_DEF_ISCSI_TADDR_SIZE);

	/*
	 * Copy the local data to the user's buffer
	 */
	if (copy_to_user((void*)_64BIT_TO_ULONG(ioctl->ResponseAdr),
			 init_fw,
			 sizeof(*init_fw)) != 0) {
		QL4PRINT(QLP2,
			 printk("scsi%d: %s: unable to copy data to user's "
				"memory area\n", ha->host_no, __func__));

		ioctl->Status = EXT_STATUS_COPY_ERR;
		goto exit_get_init_fw;
	}

	ioctl->Status = EXT_STATUS_OK;
	status = QLA_SUCCESS;

	QL4PRINT(QLP10, printk("scsi%d: EXT_INIT_FW_ISCSI structure:\n", ha->host_no));
	qla4xxx_dump_bytes(QLP10, init_fw, sizeof(*init_fw));

	exit_get_init_fw:
	if (init_fw) kfree(init_fw);
#if 0 && __VMKERNEL_MODULE__
	spin_unlock(&ha->dma_buf_spinlock);
#else
	up(&ha->dma_buf_sem);
#endif
	LEAVE("qla4extioctl_get_init_fw_iscsi");
	return(status);
}

#ifndef QLA4000
/**************************************************************************
 * qla4extioctl_get_isns_server
 *	This routine retrieves the iSNS server information.
 *
 * Input:
 *	ha    = adapter structure pointer.
 *	ioctl = IOCTL structure pointer.
 *
 * Output:
 *	None
 *
 * Returns:
 *	QLA_SUCCESS = success
 *	QLA_ERROR   = error
 *
 * Context:
 *	Kernel context.
 **************************************************************************/
static int
qla4extioctl_get_isns_server(scsi_qla_host_t *ha, EXT_IOCTL_ISCSI *ioctl)
{
	int               status   = QLA_ERROR;
	uint32_t   mbox_cmd[MBOX_REG_COUNT];
	uint32_t   mbox_sts[MBOX_REG_COUNT];
	EXT_ISNS_SERVER *isns_server = kmalloc(sizeof(*isns_server), GFP_ATOMIC);
	FLASH_INIT_FW_CTRL_BLK *flash_init_fw_cb = NULL;

	ENTER("qla4extioctl_get_isns_server");

#if 0 && __VMKERNEL_MODULE__
	spin_lock(&ha->dma_buf_spinlock);
#else
	down(&ha->dma_buf_sem);
#endif

	if (!isns_server) {
		QL4PRINT(QLP2,
			 printk("scsi%d: %s: failed to allocate memory\n",
				ha->host_no, __func__));

		ioctl->Status = EXT_STATUS_NO_MEMORY;
		ioctl->ResponseLen = 0;
		goto exit_get_isns_server;
	}

	if (ioctl->ResponseLen < sizeof(*isns_server)) {
		QL4PRINT(QLP2,
			 printk("scsi%d: %s: response buffer too small\n",
				ha->host_no, __func__));

		ioctl->Status = EXT_STATUS_BUFFER_TOO_SMALL;
		ioctl->ResponseLen = 0;
		goto exit_get_isns_server;
	}

	if (!ha->dma_buf.virt_addr
	    || !ha->dma_buf.phys_addr
	    || (ha->dma_buf.buf_len < sizeof(*flash_init_fw_cb))) {
		if (qla4xxx_resize_dma_buf(ha, &ha->dma_buf,
					   sizeof(DEV_DB_ENTRY)) != QLA_SUCCESS) {
			QL4PRINT(QLP2,
				 printk("scsi%d: %s: unable to allocate memory "
					"for dma buffer.\n",
					ha->host_no, __func__));

			ioctl->Status = EXT_STATUS_NO_MEMORY;
			ioctl->ResponseLen = 0;
			goto exit_get_isns_server;
		}
	}

	/*
	 * First get Flash Initialize Firmware Control Block, so as not to
	 * destroy unaffected data
	 *----------------------------------------------------------------*/
	flash_init_fw_cb = (FLASH_INIT_FW_CTRL_BLK *)ha->dma_buf.virt_addr;

	memset(&mbox_cmd, 0, sizeof(mbox_cmd));
	memset(&mbox_sts, 0, sizeof(mbox_sts));
	mbox_cmd[0] = MBOX_CMD_READ_FLASH;
	mbox_cmd[1] = LSDW(ha->dma_buf.phys_addr);
	mbox_cmd[2] = MSDW(ha->dma_buf.phys_addr);
	mbox_cmd[3] = INT_ISCSI_INITFW_FLASH_OFFSET;
	mbox_cmd[4] = sizeof(*flash_init_fw_cb);

	if (qla4xxx_mailbox_command(ha, 5, 2, &mbox_cmd[0], &mbox_sts[0])
	    == QLA_ERROR) {
		QL4PRINT(QLP2, printk("scsi%d: %s: READ_FLASH command failed \n",
				      ha->host_no, __func__));

		ioctl->Status = EXT_STATUS_ERR;
		ioctl->DetailStatus = mbox_sts[0];
		ioctl->ResponseLen = 0;
		goto exit_get_isns_server;
	}

	QL4PRINT(QLP4, printk("scsi%d: %s: READ_FLASH command successful \n",
			      ha->host_no, __func__));

	/*
	 * Copy iSNS Server info to the isns_server structure
	 *---------------------------------------------------*/
	memset(isns_server, 0, sizeof(*isns_server));
	isns_server->PerformiSNSDiscovery =
	(flash_init_fw_cb->init_fw_cb.TCPOptions & TOPT_ISNS_ENABLE) ?1:0;

	isns_server->AutomaticiSNSDiscovery =
	(flash_init_fw_cb->init_fw_cb.TCPOptions & TOPT_LEARN_ISNS_IP_ADDR_ENABLE) ? 1 : 0;

	isns_server->PortNumber = flash_init_fw_cb->init_fw_cb.iSNSServerPortNumber;
	isns_server->IPAddr.Type = EXT_DEF_TYPE_ISCSI_IP;
	memcpy(isns_server->IPAddr.IPAddress,
	       flash_init_fw_cb->init_fw_cb.iSNSIPAddr,
	       MIN(sizeof(isns_server->IPAddr.IPAddress),
		   sizeof(flash_init_fw_cb->init_fw_cb.iSNSIPAddr)));
	memcpy(isns_server->InitiatorName,
	       flash_init_fw_cb->init_fw_cb.iSCSINameString,
	       MIN(sizeof(isns_server->InitiatorName),
		   sizeof(flash_init_fw_cb->init_fw_cb.iSCSINameString)));

	#if 1
	memset(&mbox_cmd, 0, sizeof(mbox_cmd));
	memset(&mbox_sts, 0, sizeof(mbox_sts));
	mbox_cmd[0] = MBOX_CMD_SET_ISNS_SERVICE;
	mbox_cmd[1] = ISNS_STATUS;
	if (qla4xxx_mailbox_command(ha, 2, 2, &mbox_cmd[0], &mbox_sts[0])
	    != QLA_SUCCESS) {
		QL4PRINT(QLP2, printk("scsi%d: %s: GET ISNS SERVICE STATUS command failed \n",
				      ha->host_no, __func__));
		ioctl->Status = EXT_STATUS_ERR;
		ioctl->DetailStatus = mbox_sts[0];
		ioctl->ResponseLen = 0;
		goto exit_get_isns_server;
	}
	QL4PRINT(QLP7|QLP20, printk("scsi%d: %s: GET ISNS SERVICE STATUS = 0x%04x \"%s\"\n",
				    ha->host_no, __func__, mbox_sts[1],
				    ((mbox_sts[1] & 1) == 0) ? "DISABLED" : "ENABLED"));
	#endif

	/*
	 * Copy the local data to the user's buffer
	 *-----------------------------------------*/
	if (copy_to_user((void*)_64BIT_TO_ULONG(ioctl->ResponseAdr),
			 isns_server,
			 sizeof(*isns_server)) != 0) {
		QL4PRINT(QLP2,
			 printk("scsi%d: %s: unable to copy data to user's "
				"memory area\n", ha->host_no, __func__));

		ioctl->Status = EXT_STATUS_COPY_ERR;
		goto exit_get_isns_server;
	}

	ioctl->Status = EXT_STATUS_OK;
	ioctl->ResponseLen = sizeof(*isns_server);
	ioctl->DetailStatus = 0;
	status = QLA_SUCCESS;

	QL4PRINT(QLP10, printk("scsi%d: EXT_ISNS_SERVER structure:\n", ha->host_no));
	qla4xxx_dump_bytes(QLP10, isns_server, sizeof(*isns_server));

	exit_get_isns_server:
	if (isns_server) kfree(isns_server);
#if 0 && __VMKERNEL_MODULE__
	spin_unlock(&ha->dma_buf_spinlock);
#else
	up(&ha->dma_buf_sem);
#endif
	LEAVE("qla4extioctl_get_isns_server");
	return(status);
}

/**************************************************************************
 * qla4extioctl_get_isns_disc_targets
 *	This routine retrieves the targets discovered via iSNS.
 *
 * Input:
 *	ha    = adapter structure pointer.
 *	ioctl = IOCTL structure pointer.
 *
 * Output:
 *	None
 *
 * Returns:
 *	QLA_SUCCESS = success
 *	QLA_ERROR   = error
 *
 * Context:
 *	Kernel context.
 **************************************************************************/
static int
qla4extioctl_get_isns_disc_targets(scsi_qla_host_t *ha, EXT_IOCTL_ISCSI *ioctl)
{
	int       status   = QLA_ERROR;
	EXT_ISNS_DISCOVERED_TARGETS *isns_disc_tgts = NULL;
	uint32_t  isns_disc_tgt_index_start, i, j;

	ENTER("qla4extioctl_get_isns_disc_targets");
#if 0 && __VMKERNEL_MODULE__
	spin_lock(&ha->dma_buf_spinlock);
#else
	down(&ha->dma_buf_sem);
#endif

	if (ioctl->ResponseLen < sizeof(EXT_ISNS_DISCOVERED_TARGETS)) {
		QL4PRINT(QLP2, printk("scsi%d: %s: response buffer "
				      "too small.  RspLen=0x%x, need 0x%x\n",
				      ha->host_no, __func__,
				      ioctl->ResponseLen,
				      (unsigned int) sizeof(EXT_ISNS_DISCOVERED_TARGETS)));
		ioctl->ResponseLen = 0;
		ioctl->Status = EXT_STATUS_BUFFER_TOO_SMALL;
		ioctl->DetailStatus = sizeof(EXT_ISNS_DISCOVERED_TARGETS);
		goto exit_get_isns_disc_tgts;
	}

	if (!ha->dma_buf.virt_addr ||
	    ((ioctl->ResponseLen > ha->dma_buf.buf_len) &&
	     (qla4xxx_resize_dma_buf(ha, &ha->dma_buf, sizeof(EXT_ISNS_DISCOVERED_TARGETS))
	      != QLA_SUCCESS))) {
		QL4PRINT(QLP2, printk("scsi%d: %s: unable to allocate memory "
				      "for dma buffer.\n",
				      ha->host_no, __func__));
		ioctl->ResponseLen = 0;
		ioctl->Status = EXT_STATUS_NO_MEMORY;
		goto exit_get_isns_disc_tgts;
	}

	/*
	 * Copy the IOCTL EXT_ISNS_DISCOVERED_TARGETS buffer from the user's
	 * data space
	 */
	isns_disc_tgts = (EXT_ISNS_DISCOVERED_TARGETS *) ha->dma_buf.virt_addr;
	if (copy_from_user((uint8_t *)isns_disc_tgts,
			   (void *)_64BIT_TO_ULONG(ioctl->RequestAdr),
			   ioctl->RequestLen) != 0) {
		QL4PRINT(QLP2,
			 printk("scsi%d: %s: unable to copy data from user's "
				"memory area\n", ha->host_no, __func__));

		ioctl->ResponseLen = 0;
		ioctl->Status = EXT_STATUS_COPY_ERR;
		goto exit_get_isns_disc_tgts;
	}

	isns_disc_tgt_index_start = isns_disc_tgts->iSNSDiscoveredTargetIndexStart;
	memset(isns_disc_tgts, 0, sizeof(*isns_disc_tgts));
	isns_disc_tgts->iSNSDiscoveredTargetIndexStart = isns_disc_tgt_index_start;

	/*
	 * Transfer Data from Local buffer to DMA buffer
	 */
	if (isns_disc_tgt_index_start < ha->isns_num_discovered_targets) {
		EXT_ISNS_DISCOVERED_TARGET *isns_disc_tgt;
		ISNS_DISCOVERED_TARGET *isns_local_disc_target;

		for (i = isns_disc_tgt_index_start;
		    i < ha->isns_num_discovered_targets &&
		    isns_disc_tgts->NumiSNSDiscoveredTargets < EXT_DEF_NUM_ISNS_DISCOVERED_TARGETS;
		    i++) {
			isns_disc_tgt = (EXT_ISNS_DISCOVERED_TARGET *) &isns_disc_tgts->iSNSDiscoveredTargets[isns_disc_tgts->NumiSNSDiscoveredTargets];
			isns_local_disc_target = (ISNS_DISCOVERED_TARGET *) &ha->dma_mem_blkv->isns_discovered_target_database[i];

			isns_disc_tgt->NumPortals = isns_local_disc_target->NumPortals;
			for (j = 0; j < isns_disc_tgt->NumPortals; j++) {
				memcpy(isns_disc_tgt->Portal[j].IPAddr.IPAddress,
				       isns_local_disc_target->Portal[j].IPAddr,
				       MIN(sizeof(isns_disc_tgt->Portal[j].IPAddr.IPAddress),
					   sizeof(isns_local_disc_target->Portal[j].IPAddr)));
				isns_disc_tgt->Portal[j].IPAddr.Type = EXT_DEF_TYPE_ISCSI_IP;
				isns_disc_tgt->Portal[j].PortNumber = isns_local_disc_target->Portal[j].PortNumber;
			}

			isns_disc_tgt->DDID = isns_local_disc_target->DDID;

			memcpy(isns_disc_tgt->NameString,
			       isns_local_disc_target->NameString,
			       MIN(sizeof(isns_disc_tgt->NameString),
				   sizeof(isns_local_disc_target->NameString)));
			memcpy(isns_disc_tgt->Alias,
			       isns_local_disc_target->Alias,
			       MIN(sizeof(isns_disc_tgt->Alias),
				   sizeof(isns_local_disc_target->Alias)));

			isns_disc_tgts->NumiSNSDiscoveredTargets++;
		}
	}

	/*
	 * Copy the data to the user's buffer
	 */
	if (copy_to_user((void*)_64BIT_TO_ULONG(ioctl->ResponseAdr),
			 isns_disc_tgts,
			 sizeof(*isns_disc_tgts)) != 0) {
		QL4PRINT(QLP2,
			 printk("scsi%d: %s: unable to copy data to user's "
				"memory area\n", ha->host_no, __func__));

		ioctl->Status = EXT_STATUS_COPY_ERR;
		goto exit_get_isns_disc_tgts;
	}

	ioctl->Status = EXT_STATUS_OK;
	status = QLA_SUCCESS;

	QL4PRINT(QLP10, printk("scsi%d: EXT_INIT_FW_ISCSI structure:\n", ha->host_no));
	qla4xxx_dump_bytes(QLP10, isns_disc_tgts, sizeof(*isns_disc_tgts));

	exit_get_isns_disc_tgts:
#if 0 && __VMKERNEL_MODULE__
	spin_unlock(&ha->dma_buf_spinlock);
#else
	up(&ha->dma_buf_sem);
#endif
	LEAVE("qla4extioctl_get_isns_disc_targets");
	return(status);
}
#endif

/**************************************************************************
 * qla4extioctl_get_data
 *	This routine calls get data IOCTLs based on the IOCTL Sub Code.
 *
 * Input:
 *	ha    = adapter structure pointer.
 *	ioctl = IOCTL structure pointer.
 *
 * Output:
 *	None
 *
 * Returns:
 *	QLA_SUCCESS = success
 *	QLA_ERROR   = error
 *    	-EINVAL     = if the command is invalid
 *
 * Context:
 *	Kernel context.
 **************************************************************************/
static int
qla4extioctl_get_data(scsi_qla_host_t *ha, EXT_IOCTL_ISCSI *ioctl)
{
	switch (ioctl->SubCode) {
	case EXT_SC_GET_STATISTICS_GEN:
		return(qla4extioctl_get_statistics_gen(ha, ioctl));

	case EXT_SC_GET_STATISTICS_ISCSI:
		return(qla4extioctl_get_statistics_iscsi(ha, ioctl));

	case EXT_SC_GET_DEVICE_ENTRY_ISCSI:
	case EXT_SC_GET_DEVICE_ENTRY_DEFAULTS_ISCSI:
		return(qla4extioctl_get_device_entry_iscsi(ha, ioctl));

	case EXT_SC_GET_INIT_FW_ISCSI:
	case EXT_SC_GET_INIT_FW_DEFAULTS_ISCSI:
		return(qla4extioctl_get_init_fw_iscsi(ha, ioctl));

#ifndef QLA4000
	case EXT_SC_GET_ISNS_SERVER:
		return(qla4extioctl_get_isns_server(ha, ioctl));

	case EXT_SC_GET_ISNS_DISCOVERED_TARGETS:
		return(qla4extioctl_get_isns_disc_targets(ha, ioctl));
#endif

	default:
		QL4PRINT(QLP2,
			 printk("scsi%d: %s: unsupported external get "
				"data sub-command code (%X)\n",
				ha->host_no, __func__, ioctl->SubCode));

		ioctl->Status = EXT_STATUS_UNSUPPORTED_SUBCODE;
		return(-EINVAL);
	}
}

/**************************************************************************
 * qla4extioctl_rst_statistics_gen
 *	This routine clears the HBA general statistical information.
 *
 * Input:
 *	ha    = adapter structure pointer.
 *	ioctl = IOCTL structure pointer.
 *
 * Output:
 *	None
 *
 * Returns:
 *	QLA_SUCCESS = success
 *	QLA_ERROR   = error
 *
 * Context:
 *	Kernel context.
 **************************************************************************/
static int
qla4extioctl_rst_statistics_gen(scsi_qla_host_t *ha, EXT_IOCTL_ISCSI *ioctl)
{
	QL4PRINT(QLP3, printk("scsi%d: %s:\n", ha->host_no, __func__));

	/*
	 * Reset the general statistics fields
	 */
	ha->adapter_error_count = 0;
	ha->device_error_count = 0;
	ha->total_io_count = 0;
	ha->total_mbytes_xferred = 0;
	ha->isr_count = 0;
	ha->link_failure_count = 0;
	ha->invalid_crc_count = 0;

	ioctl->Status = EXT_STATUS_OK;
	return(QLA_SUCCESS);
}

/**************************************************************************
 * qla4extioctl_rst_statistics_iscsi
 *	This routine clears the HBA iSCSI statistical information.
 *
 * Input:
 *	ha    = adapter structure pointer.
 *	ioctl = IOCTL structure pointer.
 *
 * Output:
 *	None
 *
 * Returns:
 *	QLA_SUCCESS = success
 *	QLA_ERROR   = error
 *
 * Context:
 *	Kernel context.
 **************************************************************************/
static int
qla4extioctl_rst_statistics_iscsi(scsi_qla_host_t *ha, EXT_IOCTL_ISCSI *ioctl)
{
	uint32_t mbox_cmd[MBOX_REG_COUNT];
	uint32_t mbox_sts[MBOX_REG_COUNT];

	QL4PRINT(QLP4, printk("scsi%d: %s: index [%d]\n",
			      ha->host_no, __func__, ioctl->Instance));

	/*
	 * Make the mailbox call
	 */
	memset(&mbox_cmd, 0, sizeof(mbox_cmd));
	memset(&mbox_sts, 0, sizeof(mbox_sts));
	mbox_cmd[0] = MBOX_CMD_GET_MANAGEMENT_DATA;
	mbox_cmd[1] = ioctl->Instance;
	mbox_cmd[2] = 0;
	mbox_cmd[3] = 0;

	if (qla4xxx_mailbox_command(ha, 4, 1, &mbox_cmd[0], &mbox_sts[0])
	    != QLA_SUCCESS) {
		QL4PRINT(QLP2,
			 printk("scsi%d: %s: get mngmt data for index [%d] failed! "
				"w/ mailbox ststus 0x%x\n",
				ha->host_no, __func__,
				ioctl->Instance, mbox_sts[0]));

		ioctl->Status = EXT_STATUS_MAILBOX;
		ioctl->DetailStatus = mbox_sts[0];
		return(QLA_ERROR);
	}

	ioctl->Status = EXT_STATUS_OK;
	return(QLA_SUCCESS);
}

/**************************************************************************
 * qla4extioctl_set_device_entry_iscsi
 *	This routine configures a device with specific database entry data.
 *
 * Input:
 *	ha    = adapter structure pointer.
 *	ioctl = IOCTL structure pointer.
 *
 * Output:
 *	None
 *
 * Returns:
 *	QLA_SUCCESS = success
 *	QLA_ERROR   = error
 *
 * Context:
 *	Kernel context.
 **************************************************************************/
static int
qla4extioctl_set_device_entry_iscsi(scsi_qla_host_t *ha, EXT_IOCTL_ISCSI *ioctl)
{
	int         status = QLA_ERROR;
	DEV_DB_ENTRY *fw_ddb_entry;
	EXT_DEVICE_ENTRY_ISCSI *dev_entry;

	ENTER("qla4extioctl_set_device_entry_iscsi");
	QL4PRINT(QLP4, printk("scsi%d: %s: index [%d]\n",
			      ha->host_no, __func__, ioctl->Instance));

#if 0 && __VMKERNEL_MODULE__
	spin_lock(&ha->dma_buf_spinlock);
#else
	down(&ha->dma_buf_sem);
#endif
	dev_entry = kmalloc(sizeof(*dev_entry), GFP_ATOMIC);
	if (!dev_entry
	    || !ha->dma_buf.virt_addr
	    || !ha->dma_buf.phys_addr
	    || !ioctl->RequestAdr) {
		QL4PRINT(QLP2,
			 printk("scsi%d: %s: memory allocation problem\n",
				ha->host_no, __func__));

		ioctl->Status = EXT_STATUS_NO_MEMORY;
		goto exit_set_dev_entry;
	}

	if (ioctl->RequestLen < sizeof(*dev_entry) ||
	    ha->dma_buf.buf_len < sizeof(*fw_ddb_entry)) {
		QL4PRINT(QLP2, printk("scsi%d: %s: memory area too small\n",
				      ha->host_no, __func__));

		ioctl->Status = EXT_STATUS_BUFFER_TOO_SMALL;
		goto exit_set_dev_entry;
	}

	/*
	 * Copy the IOCTL EXT_DEVICE_ENTRY_ISCSI buffer from the user's
	 * data space
	 */
	if (copy_from_user((uint8_t *)dev_entry,
			   (void *)_64BIT_TO_ULONG(ioctl->RequestAdr),
			   ioctl->RequestLen) != 0) {
		QL4PRINT(QLP2,
			 printk("scsi%d: %s: unable to copy data from user's "
				"memory area\n", ha->host_no, __func__));

		ioctl->Status = EXT_STATUS_COPY_ERR;
		goto exit_set_dev_entry;
	}

	/*
	 * Transfer data from IOCTL's EXT_DEVICE_ENTRY_ISCSI buffer to
	 * Fw's DEV_DB_ENTRY buffer
	 */
	fw_ddb_entry = ha->dma_buf.virt_addr;
	memset(fw_ddb_entry, 0, sizeof(*fw_ddb_entry));

	fw_ddb_entry->options          = dev_entry->Options;
	fw_ddb_entry->control          = dev_entry->Control;
	fw_ddb_entry->TSID             = __cpu_to_le16(dev_entry->TargetSessID);
	fw_ddb_entry->exeCount         = __cpu_to_le16(dev_entry->ExeCount);
	fw_ddb_entry->ddbLink          = __cpu_to_le16(dev_entry->DDBLink);
	memcpy(fw_ddb_entry->ISID,
	       dev_entry->InitiatorSessID,
	       sizeof(dev_entry->InitiatorSessID));
	memcpy(fw_ddb_entry->userID,
	       dev_entry->UserID,
	       sizeof(dev_entry->UserID));
	memcpy(fw_ddb_entry->password,
	       dev_entry->Password,
	       sizeof(dev_entry->Password));

	fw_ddb_entry->exeThrottle      = __cpu_to_le16(dev_entry->DeviceInfo.ExeThrottle);
	fw_ddb_entry->iSCSIMaxSndDataSegLen= __cpu_to_le16(dev_entry->DeviceInfo.InitMarkerlessInt);
	fw_ddb_entry->retryCount       = dev_entry->DeviceInfo.RetryCount;
	fw_ddb_entry->retryDelay       = dev_entry->DeviceInfo.RetryDelay;
	fw_ddb_entry->iSCSIOptions     = __cpu_to_le16(dev_entry->DeviceInfo.iSCSIOptions);
	fw_ddb_entry->TCPOptions       = __cpu_to_le16(dev_entry->DeviceInfo.TCPOptions);
	fw_ddb_entry->IPOptions        = __cpu_to_le16(dev_entry->DeviceInfo.IPOptions);
	fw_ddb_entry->maxPDUSize       = __cpu_to_le16(dev_entry->DeviceInfo.MaxPDUSize);
	fw_ddb_entry->firstBurstSize   = __cpu_to_le16(dev_entry->DeviceInfo.FirstBurstSize);
	fw_ddb_entry->minTime2Wait     = __cpu_to_le16(dev_entry->DeviceInfo.LogoutMinTime);
	fw_ddb_entry->maxTime2Retain   = __cpu_to_le16(dev_entry->DeviceInfo.LogoutMaxTime);
	fw_ddb_entry->maxOutstndngR2T  = __cpu_to_le16(dev_entry->DeviceInfo.MaxOutstandingR2T);
	fw_ddb_entry->keepAliveTimeout = __cpu_to_le16(dev_entry->DeviceInfo.KeepAliveTimeout);
	fw_ddb_entry->portNumber       = __cpu_to_le16(dev_entry->DeviceInfo.PortNumber);
	fw_ddb_entry->maxBurstSize     = __cpu_to_le16(dev_entry->DeviceInfo.MaxBurstSize);
	fw_ddb_entry->taskMngmntTimeout= __cpu_to_le16(dev_entry->DeviceInfo.TaskMgmtTimeout);
	memcpy(fw_ddb_entry->targetAddr,
	       dev_entry->DeviceInfo.TargetAddr,
	       sizeof(dev_entry->DeviceInfo.TargetAddr));

	memcpy(fw_ddb_entry->ipAddr,
	       dev_entry->EntryInfo.IPAddr.IPAddress,
	       sizeof(dev_entry->EntryInfo.IPAddr.IPAddress));
	memcpy(fw_ddb_entry->iscsiName,
	       dev_entry->EntryInfo.iSCSIName,
	       sizeof(dev_entry->EntryInfo.iSCSIName));
	memcpy(fw_ddb_entry->iSCSIAlias,
	       dev_entry->EntryInfo.Alias,
	       sizeof(dev_entry->EntryInfo.Alias));

	/*
	 * Make the IOCTL call
	 */
	if (qla4xxx_set_ddb_entry(ha, ioctl->Instance, fw_ddb_entry)
	    != QLA_SUCCESS) {
		QL4PRINT(QLP2, printk("scsi%d: %s: SET DDB Entry failed\n",
				      ha->host_no, __func__));

		ioctl->Status = EXT_STATUS_ERR;
		goto exit_set_dev_entry;
	}

	ioctl->Status = EXT_STATUS_OK;
	status = QLA_SUCCESS;

	exit_set_dev_entry:
#if 0 && __VMKERNEL_MODULE__
	spin_unlock(&ha->dma_buf_spinlock);
#else
	up(&ha->dma_buf_sem);
#endif
	if (dev_entry) kfree(dev_entry);
	LEAVE("qla4extioctl_set_device_entry_iscsi");
	return(status);
}

/**************************************************************************
 * qla4extioctl_set_init_fw_iscsi
 *	This routine configures a device with specific data entry data.
 *
 * Input:
 *	ha    = adapter structure pointer.
 *	ioctl = IOCTL structure pointer.
 *
 * Output:
 *	None
 *
 * Returns:
 *	QLA_SUCCESS = success
 *	QLA_ERROR   = error
 *
 * Context:
 *	Kernel context.
 **************************************************************************/
static int
qla4extioctl_set_init_fw_iscsi(scsi_qla_host_t *ha, EXT_IOCTL_ISCSI *ioctl)
{
	int status = QLA_ERROR;
	EXT_INIT_FW_ISCSI *init_fw = kmalloc(sizeof(*init_fw), GFP_ATOMIC);
	INIT_FW_CTRL_BLK  *init_fw_cb;
	uint32_t mbox_cmd[MBOX_REG_COUNT];
	uint32_t mbox_sts[MBOX_REG_COUNT];

	ENTER("qla4extioctl_set_init_fw_iscsi");
	QL4PRINT(QLP3, printk("scsi%d: %s:\n", ha->host_no, __func__));

#if 0 && __VMKERNEL_MODULE__
	spin_lock(&ha->dma_buf_spinlock);
#else
	down(&ha->dma_buf_sem);
#endif

	if (!ha->dma_buf.virt_addr
	    || !ha->dma_buf.phys_addr
	    || !init_fw
	    || (ha->dma_buf.buf_len < sizeof(INIT_FW_CTRL_BLK))
	    || (ioctl->RequestLen < sizeof(EXT_INIT_FW_ISCSI))
	   ) {
		QL4PRINT(QLP2,
			 printk("scsi%d: %s: requst buffer too small\n",
				ha->host_no, __func__));

		ioctl->Status = EXT_STATUS_BUFFER_TOO_SMALL;
		goto exit_set_init_fw;
	}

	/*
	 * Copy the data from the user's buffer
	 */
	if (copy_from_user((uint8_t *)init_fw,
			   (void*)_64BIT_TO_ULONG(ioctl->RequestAdr),
			   sizeof(*init_fw)) != 0) {
		QL4PRINT(QLP2,
			 printk("scsi%d: %s: unable to copy data to user's "
				"memory area\n", ha->host_no, __func__));

		ioctl->Status = EXT_STATUS_COPY_ERR;
		goto exit_set_init_fw;
	}

	/*
	 * First get Initialize Firmware Control Block, so as not to
	 * destroy unaffected data
	 */
	memset(&mbox_cmd, 0, sizeof(mbox_cmd));
	memset(&mbox_sts, 0, sizeof(mbox_sts));
	mbox_cmd[0] = MBOX_CMD_GET_INIT_FW_CTRL_BLOCK;
	mbox_cmd[2] = LSDW(ha->dma_buf.phys_addr);
	mbox_cmd[3] = MSDW(ha->dma_buf.phys_addr);

	if (qla4xxx_mailbox_command(ha, 4, 1, &mbox_cmd[0], &mbox_sts[0])
	    == QLA_ERROR) {
		QL4PRINT(QLP2, printk("scsi%d: %s: command failed \n",
				      ha->host_no, __func__));

		ioctl->Status = EXT_STATUS_ERR;
		ioctl->DetailStatus = mbox_sts[0];
		goto exit_set_init_fw;
	}

	/*
	 * Transfer Data from Local buffer to DMA buffer
	 */
	init_fw_cb = (INIT_FW_CTRL_BLK *)ha->dma_buf.virt_addr;

	init_fw_cb->Version         = init_fw->Version;
	init_fw_cb->FwOptions       = __cpu_to_le16(init_fw->FWOptions);
	init_fw_cb->AddFwOptions    = __cpu_to_le16(init_fw->AddFWOptions);
	//FIXME: init_fw_cb->WakeupThreshold = __cpu_to_le16(init_fw->WakeupThreshold);
	memcpy(init_fw_cb->IPAddr,
	       init_fw->IPAddr.IPAddress,
	       MIN(sizeof(init_fw_cb->IPAddr),
		   sizeof(init_fw->IPAddr.IPAddress)));
	memcpy(init_fw_cb->SubnetMask,
	       init_fw->SubnetMask.IPAddress,
	       MIN(sizeof(init_fw_cb->SubnetMask),
		   sizeof(init_fw->SubnetMask.IPAddress)));
	memcpy(init_fw_cb->GatewayIPAddr,
	       init_fw->Gateway.IPAddress,
	       MIN(sizeof(init_fw_cb->GatewayIPAddr),
		   sizeof(init_fw->Gateway.IPAddress)));
	memcpy(init_fw_cb->PriDNSIPAddr,
	       init_fw->DNSConfig.IPAddr.IPAddress,
	       MIN(sizeof(init_fw_cb->PriDNSIPAddr),
		   sizeof(init_fw->DNSConfig.IPAddr.IPAddress)));
	memcpy(init_fw_cb->Alias,
	       init_fw->Alias,
	       MIN(sizeof(init_fw_cb->Alias),
		   sizeof(init_fw->Alias)));
	memcpy(init_fw_cb->iSCSINameString,
	       init_fw->iSCSIName,
	       MIN(sizeof(init_fw_cb->iSCSINameString),
		   sizeof(init_fw->iSCSIName)));

	init_fw_cb->ExecThrottle       = __cpu_to_le16(init_fw->DeviceInfo.ExeThrottle);
	init_fw_cb->InitMarkerlessInt  = __cpu_to_le16(init_fw->DeviceInfo.InitMarkerlessInt);
	init_fw_cb->RetryCount         = init_fw->DeviceInfo.RetryCount;
	init_fw_cb->RetryDelay         = init_fw->DeviceInfo.RetryDelay;
	init_fw_cb->iSCSIOptions       = __cpu_to_le16(init_fw->DeviceInfo.iSCSIOptions);
	init_fw_cb->TCPOptions         = __cpu_to_le16(init_fw->DeviceInfo.TCPOptions);
	init_fw_cb->IPOptions          = __cpu_to_le16(init_fw->DeviceInfo.IPOptions);
	init_fw_cb->MaxPDUSize         = __cpu_to_le16(init_fw->DeviceInfo.MaxPDUSize);
	init_fw_cb->FirstBurstSize     = __cpu_to_le16(init_fw->DeviceInfo.FirstBurstSize);
	init_fw_cb->DefaultTime2Wait   = __cpu_to_le16(init_fw->DeviceInfo.LogoutMinTime);
	init_fw_cb->DefaultTime2Retain = __cpu_to_le16(init_fw->DeviceInfo.LogoutMaxTime);
	init_fw_cb->MaxOutStndngR2T    = __cpu_to_le16(init_fw->DeviceInfo.MaxOutstandingR2T);
	init_fw_cb->KeepAliveTimeout   = __cpu_to_le16(init_fw->DeviceInfo.KeepAliveTimeout);
	init_fw_cb->PortNumber         = __cpu_to_le16(init_fw->DeviceInfo.PortNumber);
	init_fw_cb->MaxBurstSize       = __cpu_to_le16(init_fw->DeviceInfo.MaxBurstSize);
	//init_fw_cb->?                = init_fw->DeviceInfo.TaskMgmtTimeout;
	memcpy(init_fw_cb->TargAddr,
	       init_fw->DeviceInfo.TargetAddr,
	       EXT_DEF_ISCSI_TADDR_SIZE);

	/*
	 * Send mailbox command
	 */

	memset(&mbox_cmd, 0, sizeof(mbox_cmd));
	memset(&mbox_sts, 0, sizeof(mbox_sts));
	mbox_cmd[0] = MBOX_CMD_INITIALIZE_FIRMWARE;
	mbox_cmd[2] = LSDW(ha->dma_buf.phys_addr);
	mbox_cmd[3] = MSDW(ha->dma_buf.phys_addr);

	if (qla4xxx_mailbox_command(ha, 4, 1, &mbox_cmd[0], &mbox_sts[0])
	    == QLA_ERROR) {
		QL4PRINT(QLP2, printk("scsi%d: %s: command failed \n",
				      ha->host_no, __func__));

		ioctl->Status = EXT_STATUS_ERR;
		ioctl->DetailStatus = mbox_sts[0];
		goto exit_set_init_fw;
	}

	ioctl->Status = EXT_STATUS_OK;
	status = QLA_SUCCESS;

	QL4PRINT(QLP10, printk("scsi%d: EXT_INIT_FW_ISCSI structure:\n", ha->host_no));
	qla4xxx_dump_bytes(QLP10, init_fw, sizeof(*init_fw));

	exit_set_init_fw:
	if (init_fw) kfree(init_fw);
#if 0 && __VMKERNEL_MODULE__
	spin_unlock(&ha->dma_buf_spinlock);
#else
	up(&ha->dma_buf_sem);
#endif
	LEAVE("qla4extioctl_set_init_fw_iscsi");
	return(status);
}

#ifndef QLA4000
/**************************************************************************
 * qla4extioctl_set_isns_server
 *	This routine retrieves the targets discovered via iSNS.
 *
 * Input:
 *	ha    = adapter structure pointer.
 *	ioctl = IOCTL structure pointer.
 *
 * Output:
 *	None
 *
 * Returns:
 *	QLA_SUCCESS = success
 *	QLA_ERROR   = error
 *
 * Context:
 *	Kernel context.
 **************************************************************************/
static int
qla4extioctl_set_isns_server(scsi_qla_host_t *ha, EXT_IOCTL_ISCSI *ioctl)
{
	int               status   = QLA_ERROR;
	uint32_t   mbox_cmd[MBOX_REG_COUNT];
	uint32_t   mbox_sts[MBOX_REG_COUNT];
	EXT_ISNS_SERVER   *isns_server = kmalloc(sizeof(*isns_server), GFP_ATOMIC);
	FLASH_INIT_FW_CTRL_BLK *flash_init_fw_cb = NULL;

	ENTER("qla4extioctl_set_isns_server");

#if 0 && __VMKERNEL_MODULE__
	spin_lock(&ha->dma_buf_spinlock);
#else
	down(&ha->dma_buf_sem);
#endif

	if (!isns_server) {
		QL4PRINT(QLP2,
			 printk("scsi%d: %s: failed to allocate memory\n",
				ha->host_no, __func__));

		ioctl->Status = EXT_STATUS_NO_MEMORY;
		ioctl->ResponseLen = 0;
		goto exit_set_isns_svr;
	}

	if (ioctl->RequestLen < sizeof(*isns_server)) {
		QL4PRINT(QLP2,
			 printk("scsi%d: %s: requst buffer too small (%d/%xh)\n",
				ha->host_no, __func__, ioctl->RequestLen, ioctl->RequestLen));

		ioctl->Status = EXT_STATUS_BUFFER_TOO_SMALL;
		ioctl->ResponseLen = 0;
		goto exit_set_isns_svr;
	}

	if (!ha->dma_buf.virt_addr
	    || !ha->dma_buf.phys_addr
	    || (ha->dma_buf.buf_len < sizeof(*flash_init_fw_cb))) {
		if (qla4xxx_resize_dma_buf(ha, &ha->dma_buf,
					   sizeof(DEV_DB_ENTRY)) != QLA_SUCCESS) {
			QL4PRINT(QLP2,
				 printk("scsi%d: %s: unable to allocate memory "
					"for dma buffer.\n",
					ha->host_no, __func__));

			ioctl->Status = EXT_STATUS_NO_MEMORY;
			ioctl->ResponseLen = 0;
			goto exit_set_isns_svr;
		}
	}

	/*
	 * Copy iSNS Server info from the user's buffer
	 *---------------------------------------------*/
	if (copy_from_user((uint8_t *)isns_server,
			   (void*)_64BIT_TO_ULONG(ioctl->RequestAdr),
			   sizeof(*isns_server)) != 0) {
		QL4PRINT(QLP2,
			 printk("scsi%d: %s: unable to copy data to user's "
				"memory area\n", ha->host_no, __func__));

		ioctl->Status = EXT_STATUS_COPY_ERR;
		ioctl->ResponseLen = 0;
		goto exit_set_isns_svr;
	}

	QL4PRINT(QLP10, printk("scsi%d: EXT_ISNS_SERVER structure:\n", ha->host_no));
	qla4xxx_dump_bytes(QLP10, isns_server, sizeof(*isns_server));

	/*
	 * First get Flash Initialize Firmware Control Block, so as not to
	 * destroy unaffected data
	 *----------------------------------------------------------------*/
	memset(&mbox_cmd, 0, sizeof(mbox_cmd));
	memset(&mbox_sts, 0, sizeof(mbox_sts));
	mbox_cmd[0] = MBOX_CMD_READ_FLASH;
	mbox_cmd[1] = LSDW(ha->dma_buf.phys_addr);
	mbox_cmd[2] = MSDW(ha->dma_buf.phys_addr);
	mbox_cmd[3] = INT_ISCSI_INITFW_FLASH_OFFSET;
	mbox_cmd[4] = sizeof(*flash_init_fw_cb);

	if (qla4xxx_mailbox_command(ha, 5, 2, &mbox_cmd[0], &mbox_sts[0])
	    == QLA_ERROR) {
		QL4PRINT(QLP2, printk("scsi%d: %s: READ_FLASH command failed \n",
				      ha->host_no, __func__));

		ioctl->Status = EXT_STATUS_ERR;
		ioctl->DetailStatus = mbox_sts[0];
		ioctl->ResponseLen = 0;
		goto exit_set_isns_svr;
	}

	QL4PRINT(QLP4, printk("scsi%d: %s: READ_FLASH command successful \n",
			      ha->host_no, __func__));

	/*
	 * Copy iSNS Server info to the flash_init_fw_cb
	 *----------------------------------------------*/
	flash_init_fw_cb = (FLASH_INIT_FW_CTRL_BLK *)ha->dma_buf.virt_addr;

	if (isns_server->PerformiSNSDiscovery) {
		#if 0
		if (isns_server->AutomaticiSNSDiscovery) {
			flash_init_fw_cb->init_fw_cb.TCPOptions |= TOPT_LEARN_ISNS_IP_ADDR_ENABLE;
			memset(flash_init_fw_cb->init_fw_cb.iSNSIPAddr, 0,
			       sizeof(flash_init_fw_cb->init_fw_cb.iSNSIPAddr));
		}
		else
		#endif
		{
			flash_init_fw_cb->init_fw_cb.TCPOptions &= ~TOPT_LEARN_ISNS_IP_ADDR_ENABLE;
			memcpy(flash_init_fw_cb->init_fw_cb.iSNSIPAddr,
			       isns_server->IPAddr.IPAddress,
			       MIN(sizeof(flash_init_fw_cb->init_fw_cb.iSNSIPAddr),
				   sizeof(isns_server->IPAddr.IPAddress)));
		}
		flash_init_fw_cb->init_fw_cb.iSNSServerPortNumber =
		(isns_server->PortNumber) ?
		isns_server->PortNumber :
		EXT_DEF_ISNS_WELL_KNOWN_PORT;
		flash_init_fw_cb->init_fw_cb.TCPOptions |= TOPT_ISNS_ENABLE;
	}
	else {
		flash_init_fw_cb->init_fw_cb.TCPOptions &= ~TOPT_ISNS_ENABLE;
		memset(flash_init_fw_cb->init_fw_cb.iSNSIPAddr, 0,
		       sizeof(flash_init_fw_cb->init_fw_cb.iSNSIPAddr));
		flash_init_fw_cb->init_fw_cb.iSNSServerPortNumber = 0;
	}
	QL4PRINT(QLP4, printk("scsi%d: %s: IPAddr %d.%d.%d.%d Port# %04d\n",
			      ha->host_no, __func__,
			      flash_init_fw_cb->init_fw_cb.iSNSIPAddr[0],
			      flash_init_fw_cb->init_fw_cb.iSNSIPAddr[1],
			      flash_init_fw_cb->init_fw_cb.iSNSIPAddr[2],
			      flash_init_fw_cb->init_fw_cb.iSNSIPAddr[3],
			      flash_init_fw_cb->init_fw_cb.iSNSServerPortNumber));

	/*
	 * If the internal iSNS info is different from the flash_init_fw_cb,
	 * flash it now.
	 *------------------------------------------------------------------*/
	if (((ha->tcp_options & TOPT_LEARN_ISNS_IP_ADDR_ENABLE) !=
	     (flash_init_fw_cb->init_fw_cb.TCPOptions & TOPT_LEARN_ISNS_IP_ADDR_ENABLE)) ||
	    (!IPAddrIsEqual(ha->isns_ip_address, flash_init_fw_cb->init_fw_cb.iSNSIPAddr)) ||
	    (ha->isns_server_port_number !=
	     flash_init_fw_cb->init_fw_cb.iSNSServerPortNumber)) {
		memset(&mbox_cmd, 0, sizeof(mbox_cmd));
		memset(&mbox_sts, 0, sizeof(mbox_sts));
		mbox_cmd[0] = MBOX_CMD_WRITE_FLASH;
		mbox_cmd[1] = LSDW(ha->dma_buf.phys_addr);
		mbox_cmd[2] = MSDW(ha->dma_buf.phys_addr);
		mbox_cmd[3] = INT_ISCSI_INITFW_FLASH_OFFSET;
		mbox_cmd[4] = sizeof(*flash_init_fw_cb);
		mbox_cmd[5] = WRITE_FLASH_OPTION_COMMIT_DATA;

		if (qla4xxx_mailbox_command(ha, 6, 2, &mbox_cmd[0], &mbox_sts[0])
		    == QLA_ERROR) {
			QL4PRINT(QLP2,
				 printk("scsi%d: %s: WRITE_FLASH command failed \n",
					ha->host_no, __func__));

			ioctl->Status = EXT_STATUS_ERR;
			ioctl->DetailStatus = mbox_sts[0];
			ioctl->ResponseLen = 0;
			goto exit_set_isns_svr;
		}

		QL4PRINT(QLP4,
			 printk("scsi%d: %s: WRITE_FLASH command successful \n",
				ha->host_no, __func__));
		QL4PRINT(QLP4, printk("scsi%d: Init Fw Ctrl Blk\n", ha->host_no));
		qla4xxx_dump_bytes(QLP4, flash_init_fw_cb,
				   sizeof(*flash_init_fw_cb));

		/*
		 * Update internal iSNS info
		 */
		if (isns_server->AutomaticiSNSDiscovery)
			ha->tcp_options |= TOPT_LEARN_ISNS_IP_ADDR_ENABLE;
		else
			ha->tcp_options	&= ~TOPT_LEARN_ISNS_IP_ADDR_ENABLE;

		memcpy(ha->isns_ip_address,
		       flash_init_fw_cb->init_fw_cb.iSNSIPAddr,
		       MIN(sizeof(ha->isns_ip_address),
			   sizeof(flash_init_fw_cb->init_fw_cb.iSNSIPAddr)));
		ha->isns_server_port_number =
		flash_init_fw_cb->init_fw_cb.iSNSServerPortNumber;
	}

	/*
	 * Start or Stop iSNS Service accordingly, if needed.
	 *---------------------------------------------------*/
	//FIXME:
	if (test_bit(ISNS_FLAG_ISNS_ENABLED_IN_ISP, &ha->isns_flags)) {
		if (!IPAddrIsZero(ha->isns_ip_address) &&
		    ha->isns_server_port_number &&
		    (ha->tcp_options & TOPT_LEARN_ISNS_IP_ADDR_ENABLE) == 0) {
			uint32_t ip_addr;
			IPAddr2Uint32(ha->isns_ip_address, &ip_addr);

			status =
			qla4xxx_isns_reenable(ha, ip_addr,
					      ha->isns_server_port_number);

			if (status == QLA_ERROR) {
				QL4PRINT(QLP4, printk("scsi%d: qla4xxx_isns_reenable failed!\n", ha->host_no));
				ioctl->Status = EXT_STATUS_ERR;
				ioctl->DetailStatus = 0;
				ioctl->ResponseLen = 0;
				goto exit_set_isns_svr;
			}
		}
		else if (test_bit(ISNS_FLAG_ISNS_SRV_ENABLED, &ha->isns_flags) &&
			 IPAddrIsZero(ha->isns_ip_address)) {
			qla4xxx_isns_disable(ha);
		}
	}

	/*
	 * Complete IOCTL successfully
	 *----------------------------*/
	ioctl->Status = EXT_STATUS_OK;
	ioctl->DetailStatus = 0;
	ioctl->ResponseLen = 0;
	status = QLA_SUCCESS;

	exit_set_isns_svr:
	if (isns_server) kfree(isns_server);
#if 0 && __VMKERNEL_MODULE__
	spin_unlock(&ha->dma_buf_spinlock);
#else
	up(&ha->dma_buf_sem);
#endif
	LEAVE("qla4extioctl_set_isns_server");
	return(status);
}
#endif

/**************************************************************************
 * qla4extioctl_set_data
 *	This routine calls set data IOCTLs based on the IOCTL Sub Code.
 *
 * Input:
 *	ha    = adapter structure pointer.
 *	ioctl = IOCTL structure pointer.
 *
 * Output:
 *	None
 *
 * Returns:
 *	QLA_SUCCESS = success
 *	QLA_ERROR   = error
 *    	-EINVAL     = if the command is invalid
 *
 * Context:
 *	Kernel context.
 **************************************************************************/
static int qla4extioctl_set_data(scsi_qla_host_t *ha, EXT_IOCTL_ISCSI *ioctl)
{
	switch (ioctl->SubCode) {
	case EXT_SC_RST_STATISTICS_GEN:
		return(qla4extioctl_rst_statistics_gen(ha, ioctl));

	case EXT_SC_RST_STATISTICS_ISCSI:
		return(qla4extioctl_rst_statistics_iscsi(ha, ioctl));

	case EXT_SC_SET_DEVICE_ENTRY_ISCSI:
		return(qla4extioctl_set_device_entry_iscsi(ha, ioctl));

	case EXT_SC_SET_INIT_FW_ISCSI:
		return(qla4extioctl_set_init_fw_iscsi(ha, ioctl));

#ifndef QLA4000
	case EXT_SC_SET_ISNS_SERVER:
		return(qla4extioctl_set_isns_server(ha, ioctl));
#endif

	default:
		QL4PRINT(QLP2,
			 printk("scsi%d: %s: unsupported set data sub-command "
				"code (%X)\n",
				ha->host_no, __func__, ioctl->SubCode));

		ioctl->Status = EXT_STATUS_UNSUPPORTED_SUBCODE;
		return(-EINVAL);
	}
}

/**************************************************************************
 * qla4xxx_ioctl_sleep_done
 *	This routine is the callback function to wakeup ioctl completion
 *	semaphore for the ioctl request that is waiting.
 *
 * Input:
 *   	sem - pointer to the ioctl completion semaphore.
 *
 * Output:
 *	None
 *
 * Returns:
 *	None
 *
 * Context:
 *	Kernel context.
 **************************************************************************/
STATIC void
qla4xxx_ioctl_sleep_done (struct semaphore * sem)
{
	ENTER("qla4xxx_ioctl_sleep_done");

	if (sem != NULL) {
		QL4PRINT(QLP4, printk("%s: wake up sem.\n", __func__));
		QL4PRINT(QLP10, printk("%s: UP count=%d\n", __func__,
				       atomic_read(&sem->count)));
		up(sem);
	}

	LEAVE("qla4xxx_ioctl_sleep_done");
}

/**************************************************************************
 * qla4xxx_ioctl_sem_init
 *	This routine initializes the ioctl timer and semaphore used to wait
 *	for passthru completion.
 *
 * Input:
 *	ha    = adapter structure pointer.
 *
 * Output:
 *	None
 *
 * Returns:
 *	None
 *
 * Context:
 *	Kernel context.
 **************************************************************************/
void
qla4xxx_ioctl_sem_init (scsi_qla_host_t *ha)
{
	init_timer(&(ha->ioctl_cmpl_timer));
	ha->ioctl_cmpl_timer.data = (unsigned long)&ha->ioctl_cmpl_sem;
	ha->ioctl_cmpl_timer.function =
	(void (*)(unsigned long))qla4xxx_ioctl_sleep_done;
}

/**************************************************************************
 * qla4xxx_scsi_pass_done
 *	This routine resets the ioctl progress flag and wakes up the ioctl
 * 	completion semaphore.
 *
 * Input:
 *   	cmd - pointer to the passthru Scsi cmd structure which has completed.
 *
 * Output:
 *	None
 *
 * Returns:
 *	None
 *
 * Context:
 *	Interrupt context.
 **************************************************************************/
void
qla4xxx_scsi_pass_done(Scsi_Cmnd *cmd)
{
	scsi_qla_host_t *ha = (scsi_qla_host_t *) cmd->host->hostdata;

	ENTER("qla4xxx_scsi_pass_done");

	/* First check to see if the command has previously timed-out
	 * because we don't want to get the up/down semaphore counters off.
	 */
	if (ha->ioctl_scsi_pass_in_progress == TRUE) {
		ha->ioctl_scsi_pass_in_progress = FALSE;
		ha->ioctl_tov = 0;
		ha->ioctl_err_cmd = NULL;

		up(&ha->ioctl_cmpl_sem);
	}

	LEAVE("qla4xxx_scsi_pass_done");
	return;
}

/**************************************************************************
 * qla4extioctl_scsi_passthru
 *	This routine
 *
 * Input:
 *	ha    = adapter structure pointer.
 *	ioctl = IOCTL structure pointer.
 *
 * Output:
 *	None
 *
 * Map of DMA Buffer:
 *    +-------------------------+
 *    | EXT_SCSI_PASSTHRU_ISCSI |
 *    +-------------------------+
 *    | [SCSI READ|WRITE data]  |
 *    +-------------------------+
 *
 * Returns:
 *	QLA_SUCCESS = success
 *	QLA_ERROR   = error
 *
 * Context:
 *	Kernel context.
 **************************************************************************/
static int
qla4extioctl_scsi_passthru(scsi_qla_host_t *ha, EXT_IOCTL_ISCSI *ioctl)
{
	int                     status = QLA_ERROR;
	ddb_entry_t             *ddb_entry;
	int                     i;
#if 1 || __VMKERNEL_MODULE__
	EXT_SCSI_PASSTHRU_ISCSI *scsi_pass = NULL;
	Scsi_Device      *scsi_device = NULL;
	Scsi_Cmnd        *cmd = NULL;
#else
	static EXT_SCSI_PASSTHRU_ISCSI scsi_pass;
	static Scsi_Device      scsi_device;
	static Scsi_Cmnd        cmd;
#endif
	srb_t       *srb;
	uint32_t dma_buf_len;

	ENTER("qla4extioctl_scsi_passthru");

	if (!ADAPTER_UP(ha)) {
		QL4PRINT(QLP2,
			 printk("scsi%d: %s: command not pocessed, "
				"adapter link down.\n",
				ha->host_no, __func__));
		ioctl->Status = EXT_STATUS_HBA_NOT_READY;
		return(QLA_ERROR);
	}
#if 1 || __VMKERNEL_MODULE__
	if ( !(scsi_pass = kmalloc(sizeof(*scsi_pass), GFP_ATOMIC)) ||
		!(scsi_device = kmalloc(sizeof(*scsi_device), GFP_ATOMIC)) ||
		!(cmd = kmalloc(sizeof(*cmd), GFP_ATOMIC)) ) {

		if(scsi_pass) {
			kfree(scsi_pass);
			if(scsi_device) {
				kfree(scsi_device);
				if(cmd) {
					kfree(cmd);
				}
			}
		}
		QL4PRINT(QLP2,
			printk("scsi%d: %s: could not allocate memory.\n",
				ha->host_no, __func__));
		ioctl->Status = EXT_STATUS_NO_MEMORY;
		return(QLA_ERROR);
	}
#	define scsi_pass (*scsi_pass)
#	define scsi_device (*scsi_device)
#	define cmd (*cmd)
#endif

#if 0 && __VMKERNEL_MODULE__
	spin_lock(&ha->dma_buf_spinlock);
#else
	down(&ha->dma_buf_sem);
#endif

	memset(&scsi_device, 0, sizeof(scsi_device));
	memset(&scsi_pass, 0, sizeof(scsi_pass));
	memset(&cmd, 0, sizeof(cmd));

	/* ---- Get passthru structure from user space ---- */
	if (copy_from_user((uint8_t *)&scsi_pass,
			   (void *)_64BIT_TO_ULONG(ioctl->RequestAdr),
			   sizeof(scsi_pass)) != 0) {
		QL4PRINT(QLP2,
			 printk("scsi%d: %s: unable to copy passthru struct "
				"from user's memory area.\n",
				ha->host_no, __func__));

		ioctl->Status = EXT_STATUS_COPY_ERR;
		goto error_exit_scsi_pass;
	}

	QL4PRINT(QLP10, printk("scsi%d: %s: incoming  EXT_SCSI_PASSTHRU_ISCSI "
			       "structure: \n", ha->host_no, __func__));
	qla4xxx_dump_bytes(QLP10, &scsi_pass, sizeof(scsi_pass));

	/* ---- Make sure device exists ---- */
	ddb_entry = qla4xxx_lookup_ddb_by_SCSIID(ha,
						 scsi_pass.Addr.Bus,
						 scsi_pass.Addr.Target);
	if (ddb_entry == NULL) {
		QL4PRINT(QLP2,
			 printk("scsi%d: %s: invalid device (b%d,t%d) specified.\n",
				ha->host_no, __func__,
				scsi_pass.Addr.Bus, scsi_pass.Addr.Target));

		ioctl->Status = EXT_STATUS_DEV_NOT_FOUND;
		goto error_exit_scsi_pass;
	}

	/* ---- Make sure device is in an active state ---- */
	if (ddb_entry->fw_ddb_device_state != DDB_DS_SESSION_ACTIVE) {
		QL4PRINT(QLP2,
			 printk("scsi%d: %s: device (b%d,t%d) not in active state.\n",
				ha->host_no, __func__,
				scsi_pass.Addr.Bus, scsi_pass.Addr.Target));

		ioctl->Status = EXT_STATUS_DEVICE_NOT_READY;
		goto error_exit_scsi_pass;
	}

	/* ---- Retrieve srb from pool ---- */
	srb = del_from_free_srb_q_head(ha);
	if (srb == NULL) {
		QL4PRINT(QLP2, printk("scsi%d: %s: srb not available\n",
				      ha->host_no, __func__));
		goto error_exit_scsi_pass;
	}


	/* ---- Allocate larger DMA buffer, if neccessary ---- */
	dma_buf_len = MAX(ioctl->ResponseLen - sizeof(scsi_pass),
			  ioctl->RequestLen - sizeof(scsi_pass));

	if (ha->dma_buf.buf_len < dma_buf_len
	    && qla4xxx_resize_dma_buf(ha, &ha->dma_buf, dma_buf_len)
	    != QLA_SUCCESS) {
		QL4PRINT(QLP2,
			 printk("scsi%d: %s: ERROR cannot allocate requested "
				"DMA buffer size 0x%x.\n",
				ha->host_no, __func__, dma_buf_len));

		ioctl->Status = EXT_STATUS_NO_MEMORY;
		goto error_exit_scsi_pass;
	}

	memset(ha->dma_buf.virt_addr, 0, ha->dma_buf.buf_len);

	/* ---- Fill in the SCSI command structure ---- */
	cmd.channel         = scsi_pass.Addr.Bus;
	cmd.target          = scsi_pass.Addr.Target;
	cmd.lun             = scsi_pass.Addr.Lun;
	cmd.request_buffer  = ha->dma_buf.virt_addr;
#ifdef __VMKERNEL_MODULE__
	cmd.request_bufferMA = (dma_addr_t)virt_to_phys(cmd.request_buffer);
#endif //__VMKERNEL_MODULE__
	cmd.device          = &scsi_device;
	cmd.host            = ha->host;
	cmd.scsi_done       = qla4xxx_scsi_pass_done;
	CMD_TIMEOUT(&cmd)   = IOCTL_PASSTHRU_TOV * HZ;

	CMD_SP(&cmd) = (char *) srb;
#if 1 || __VMKERNEL_MODULE__
#	undef cmd
	srb->cmd = cmd;
#	define cmd (*cmd)
#else
	srb->cmd = &cmd;
#endif
	srb->fw_ddb_index = ddb_entry->fw_ddb_index;
	srb->lun = cmd.lun;
	srb->flags |= SRB_IOCTL_CMD;

	if (scsi_pass.CdbLength == 6 ||
	    scsi_pass.CdbLength == 10 ||
	    scsi_pass.CdbLength == 12 ||
	    scsi_pass.CdbLength == 16) {
		cmd.cmd_len = scsi_pass.CdbLength;
	}
	else {
		QL4PRINT(QLP2,
			 printk("scsi%d: %s: Unsupported CDB length 0x%x \n",
				ha->host_no, __func__, cmd.cmd_len));

		ioctl->Status = EXT_STATUS_INVALID_PARAM;
		goto error_exit_scsi_pass;
	}

	if (scsi_pass.Direction == EXT_DEF_SCSI_PASSTHRU_DATA_IN) {
		cmd.sc_data_direction = SCSI_DATA_READ;
		cmd.request_bufflen = ioctl->ResponseLen - sizeof(scsi_pass);

	}
	else if (scsi_pass.Direction ==  EXT_DEF_SCSI_PASSTHRU_DATA_OUT) {
		cmd.sc_data_direction = SCSI_DATA_WRITE;
		cmd.request_bufflen = ioctl->RequestLen - sizeof(scsi_pass);

		/* Sending user data from ioctl->ResponseAddr to SCSI
		 * command buffer */
		if (copy_from_user((uint8_t *)cmd.request_buffer,
				   (void*)_64BIT_TO_ULONG(ioctl->RequestAdr) +
				   sizeof(scsi_pass), cmd.request_bufflen) != 0) {
			QL4PRINT(QLP2,
				 printk("scsi%d: %s: unable to copy write buffer "
					"from user's memory area.\n",
					ha->host_no, __func__));

			ioctl->Status = EXT_STATUS_COPY_ERR;
			goto error_exit_scsi_pass;
		}
	}
	else {
		cmd.sc_data_direction = SCSI_DATA_NONE;
		cmd.request_buffer  = 0;
		cmd.request_bufflen = 0;
	}


	memcpy(cmd.cmnd,      scsi_pass.Cdb, cmd.cmd_len);
	memcpy(cmd.data_cmnd, scsi_pass.Cdb, cmd.cmd_len);

	QL4PRINT(QLP4, printk("scsi%d:%d:%d:%d: %s: CDB = ",
			      ha->host_no, cmd.channel, cmd.target, cmd.lun, __func__));
	for (i = 0; i < cmd.cmd_len; i++)
		QL4PRINT(QLP4, printk("%02X ", cmd.cmnd[i]));
	QL4PRINT(QLP4, printk("\n"));


	/* ---- prepare for receiving completion ---- */
	ha->ioctl_scsi_pass_in_progress = TRUE;
	ha->ioctl_tov = CMD_TIMEOUT(&cmd);

	qla4xxx_ioctl_sem_init(ha);
	CMD_COMPL_STATUS(&cmd)  = (int) IOCTL_INVALID_STATUS;
	CMD_PASSTHRU_TYPE(&cmd) = (unsigned long)TRUE;

	/* ---- send command to adapter ---- */
	QL4PRINT(QLP4, printk("scsi%d:%d:%d:%d: %s: sending command.\n",
			      ha->host_no, cmd.channel, cmd.target, cmd.lun, __func__));

	ha->ioctl_cmpl_timer.expires = jiffies + ha->ioctl_tov;
	add_timer(&ha->ioctl_cmpl_timer);

	if (qla4xxx_send_command_to_isp(ha, srb) != QLA_SUCCESS) {
		add_to_free_srb_q(ha, srb);
		QL4PRINT(QLP2,
			 printk("scsi%d: %s: error sending cmd to isp\n",
				ha->host_no, __func__));
		del_timer(&ha->ioctl_cmpl_timer);
		ioctl->Status = EXT_STATUS_DEV_NOT_FOUND;
		goto error_exit_scsi_pass;
	}


	down(&ha->ioctl_cmpl_sem);

	/*******************************************************
	 *						       *
	 *             Passthru Completion                     *
	 *						       *
	 *******************************************************/
	del_timer(&ha->ioctl_cmpl_timer);

	/* ---- check for timeout --- */
	if (ha->ioctl_scsi_pass_in_progress == TRUE) {
		QL4PRINT(QLP2,
			 printk("scsi%d: %s: ERROR = command timeout.\n",
				ha->host_no, __func__));

		ioctl->Status = EXT_STATUS_ERR;

		if ((srb != NULL) && (srb->active_array_index < MAX_SRBS)) {
			u_long wait_cnt = WAIT_CMD_TOV;

			if ((srb->flags & SRB_FREE_STATE) == 0)
				qla4xxx_delete_timer_from_cmd(srb);

			/* Wait for command to get out of active state */
			wait_cnt = jiffies + WAIT_CMD_TOV * HZ;
			QLA4XXX_WAIT(wait_cnt, srb->flags != SRB_ACTIVE_STATE);
		}
		ha->ioctl_scsi_pass_in_progress = FALSE;
		goto error_exit_scsi_pass;
	}

	/* --- Return info from status entry --- */
	ioctl->DetailStatus = CMD_SCSI_STATUS(&cmd);
	scsi_pass.Reserved[0] = (uint8_t) CMD_SCSI_STATUS(&cmd);
	scsi_pass.Reserved[1] = (uint8_t) CMD_COMPL_STATUS(&cmd);
	scsi_pass.Reserved[2] = (uint8_t) CMD_ACTUAL_SNSLEN(&cmd);
	scsi_pass.Reserved[3] = (uint8_t) CMD_HOST_STATUS(&cmd);
	scsi_pass.Reserved[6] = (uint8_t) CMD_ISCSI_RESPONSE(&cmd);
	scsi_pass.Reserved[7] = (uint8_t) CMD_STATE_FLAGS(&cmd);

	if (CMD_ACTUAL_SNSLEN(&cmd)) {
		memcpy(scsi_pass.SenseData, CMD_SNSP(&cmd),
		       MIN(CMD_ACTUAL_SNSLEN(&cmd),
			   sizeof(scsi_pass.SenseData)));
	}


	/* ---- check for command completion --- */
	if (CMD_COMPL_STATUS(&cmd) == IOCTL_INVALID_STATUS) {
		QL4PRINT(QLP2,
			 printk("scsi%d:%d:%d:%d: %s: ERROR = "
				"command not completed.\n",
				ha->host_no, cmd.channel, cmd.target, cmd.lun, __func__));

		ioctl->Status = EXT_STATUS_ERR;
		goto error_exit_scsi_pass;
	}
	else if (CMD_HOST_STATUS(&cmd) == DID_OK) {
		ioctl->Status = EXT_STATUS_OK;
	}
	else if (CMD_COMPL_STATUS(&cmd) == SCS_DATA_UNDERRUN) {
		QL4PRINT(QLP2,
			 printk("scsi%d: %s: Data underrun.  Resid = 0x%x\n",
				ha->host_no, __func__, CMD_RESID_LEN(&cmd)));

		ioctl->Status = EXT_STATUS_DATA_UNDERRUN;
		scsi_pass.Reserved[4] = MSB(CMD_RESID_LEN(&cmd));
		scsi_pass.Reserved[5] = LSB(CMD_RESID_LEN(&cmd));
	}
	else if (CMD_COMPL_STATUS(&cmd) == SCS_DATA_OVERRUN) {
		QL4PRINT(QLP2,
			 printk("scsi%d: %s: Data overrun.  Resid = 0x%x\n",
				ha->host_no, __func__, CMD_RESID_LEN(&cmd)));

		ioctl->Status = EXT_STATUS_DATA_OVERRUN;
		scsi_pass.Reserved[4] = MSB(CMD_RESID_LEN(&cmd));
		scsi_pass.Reserved[5] = LSB(CMD_RESID_LEN(&cmd));
	}
	else {
		QL4PRINT(QLP2,
			 printk("scsi%d: %s: Command completed in ERROR. "
				"cs=%04x, ss=%-4x\n", ha->host_no, __func__,
				CMD_COMPL_STATUS(&cmd), CMD_SCSI_STATUS(&cmd)));

		if (CMD_SCSI_STATUS(&cmd) != SCSI_GOOD)
			ioctl->Status = EXT_STATUS_SCSI_STATUS;
		else
			ioctl->Status = EXT_STATUS_ERR;
	}

	/* ---- Copy SCSI Passthru structure with updated sense buffer
	 *      to user space ---- */
	if (copy_to_user((void *)_64BIT_TO_ULONG(ioctl->ResponseAdr),
			 &scsi_pass,
			 sizeof(EXT_SCSI_PASSTHRU_ISCSI)) != 0) {
		QL4PRINT(QLP2,
			 printk("scsi%d: %s: unable to copy passthru struct "
				"to user's memory area.\n",
				ha->host_no, __func__));

		ioctl->Status = EXT_STATUS_COPY_ERR;
		goto error_exit_scsi_pass;
	}

	QL4PRINT(QLP10,
		 printk("scsi%d: %s: outgoing EXT_SCSI_PASSTHRU_ISCSI structure:\n",
			ha->host_no, __func__));
	qla4xxx_dump_bytes(QLP10,
			   (void *)_64BIT_TO_ULONG(ioctl->ResponseAdr),
			   sizeof(EXT_SCSI_PASSTHRU_ISCSI));

	/* ---- Copy SCSI READ data from SCSI command buffer
	*       to user space ---- */
	if (scsi_pass.Direction == EXT_DEF_SCSI_PASSTHRU_DATA_IN) {
		void    *xfer_ptr = (void *)_64BIT_TO_ULONG(ioctl->ResponseAdr) +
				    sizeof(EXT_SCSI_PASSTHRU_ISCSI);
		uint32_t xfer_len = ioctl->ResponseLen -
				    sizeof(EXT_SCSI_PASSTHRU_ISCSI);


		/* Update ResponseLen if a data underrun occurred */
		if (CMD_COMPL_STATUS(&cmd) == SCS_DATA_UNDERRUN &&
		    CMD_RESID_LEN(&cmd)) {
			xfer_len -= CMD_RESID_LEN(&cmd);
		}

		if (copy_to_user(xfer_ptr, cmd.request_buffer, xfer_len) != 0) {
			QL4PRINT(QLP2,
				 printk("scsi%d: %s: unable to copy READ data "
					"to user's memory area.\n",
					ha->host_no, __func__));

			ioctl->Status = EXT_STATUS_COPY_ERR;
			goto error_exit_scsi_pass;
		}

		QL4PRINT(QLP10,
			 printk("scsi%d: %s: outgoing READ data:  (0x%p)\n",
				ha->host_no, __func__, xfer_ptr));

		qla4xxx_dump_bytes(QLP10, xfer_ptr, xfer_len);
	}

	goto exit_scsi_pass;

	error_exit_scsi_pass:
	ioctl->ResponseLen = 0;

	exit_scsi_pass:
#if 0 && __VMKERNEL_MODULE__
	spin_unlock(&ha->dma_buf_spinlock);
#else
	up(&ha->dma_buf_sem);

#	undef scsi_pass
#	undef scsi_device
#	undef cmd
        kfree (scsi_pass);
        kfree (scsi_device);
        kfree (cmd);
#endif
	status = ioctl->Status;
	LEAVE("qla4extioctl_scsi_passthru");
	return(status);
}

/**************************************************************************
 * qla4extioctl_iscsi_passthru
 *	This routine sends iSCSI pass-through to destination.
 *
 * Input:
 *	ha    = adapter structure pointer.
 *	ioctl = IOCTL structure pointer.
 *
 * Output:
 *	None
 *
 * Returns:
 *	QLA_SUCCESS = success
 *	QLA_ERROR   = error
 *
 * Context:
 *	Kernel context.
 **************************************************************************/
static int
qla4extioctl_iscsi_passthru(scsi_qla_host_t *ha, EXT_IOCTL_ISCSI *ioctl)
{
	int status = QLA_ERROR;
	ioctl->Status = EXT_STATUS_UNSUPPORTED_SUBCODE;
	ENTER("qla4extioctl_iscsi_passthru");
	QL4PRINT(QLP4, printk("scsi%d: %s: UNSUPPORTED\n", ha->host_no, __func__));
	LEAVE("qla4extioctl_iscsi_passthru");
	return(status);
}

/**************************************************************************
 * qla4extioctl_get_hbacnt
 *	This routine retrieves the number of supported HBAs found.
 *
 * Input:
 *	ha    = adapter structure pointer.
 *	ioctl = IOCTL structure pointer.
 *
 * Output:
 *	None
 *
 * Returns:
 *	QLA_SUCCESS = success
 *	QLA_ERROR   = error
 *
 * Context:
 *	Kernel context.
 **************************************************************************/
static int
qla4extioctl_get_hbacnt(scsi_qla_host_t *ha, EXT_IOCTL_ISCSI *ioctl)
{
	int status = QLA_ERROR;
	uint32_t hba_count;

	ENTER("qla4extioctl_get_hbacnt");

	hba_count = qla4xxx_get_hba_count();
	if (put_user(hba_count, (uint16_t *) _64BIT_TO_ULONG(ioctl->ResponseAdr)) != 0) {
		QL4PRINT(QLP2, printk("scsi%d: %s: failed to copy data\n",
				      ha->host_no, __func__));

		ioctl->Status = EXT_STATUS_COPY_ERR;
		goto exit_get_hbacnt;
	}

	QL4PRINT(QLP4, printk("scsi%d: %s: hbacnt is %d\n",
			      ha->host_no, __func__, hba_count));
	status = QLA_SUCCESS;
	ioctl->Status = EXT_STATUS_OK;

	exit_get_hbacnt:

	LEAVE("qla4extioctl_get_hbacnt");
	return(status);
}

/**************************************************************************
 * qla4intioctl_logout_iscsi
 *	This routine requests that the specified device either login or
 *	logout, depending on the option specified.
 *
 * Input:
 *	ha    = adapter structure pointer.
 *	ioctl = IOCTL structure pointer.
 *
 * Output:
 *	None
 *
 * Returns:
 *	QLA_SUCCESS = success
 *	QLA_ERROR   = error
 *
 * Context:
 *	Kernel context.
 **************************************************************************/
static int
qla4intioctl_logout_iscsi(scsi_qla_host_t *ha, EXT_IOCTL_ISCSI *ioctl)
{
	int status = QLA_ERROR;
	INT_LOGOUT_ISCSI logout;

	ENTER("qla4intioctl_logout_iscsi");

	if (ioctl->RequestLen > sizeof(INT_LOGOUT_ISCSI)) {
		QL4PRINT(QLP2,
			 printk("scsi%d: %s: memory area too small\n",
				ha->host_no, __func__));

		ioctl->Status = EXT_STATUS_BUFFER_TOO_SMALL;
		goto exit_logout;
	}

	/* --- Copy logout structure from user space --- */
	if (copy_from_user((uint8_t *)&logout,
			   (void*)_64BIT_TO_ULONG(ioctl->RequestAdr),
			   sizeof(logout)) != 0) {
		QL4PRINT(QLP2,
			 printk("scsi%d: %s: unable to copy data from "
				"user's memory area\n", ha->host_no, __func__));

		ioctl->Status = EXT_STATUS_COPY_ERR;
		goto exit_logout;
	}

	/* --- Execute command --- */
	if (logout.Options == INT_DEF_CLOSE_SESSION) {
		if ((status = qla4xxx_logout_device(ha,
						    logout.TargetID,
						    logout.ConnectionID))
		    == QLA_SUCCESS) {
			QL4PRINT(QLP4,
				 printk("scsi%d: %s: CLOSE_SESSION SUCCEEDED!, "
					"target %d\n", ha->host_no, __func__,
					logout.TargetID));

			ioctl->Status = EXT_STATUS_OK;
		}
		else {
			QL4PRINT(QLP2,
				 printk("scsi%d: %s: CLOSE_SESSION FAILED!, "
					"target %d\n", ha->host_no, __func__,
					logout.TargetID));

			ioctl->Status = EXT_STATUS_ERR;
		}
	}
	else if (logout.Options == INT_DEF_RELOGIN_CONNECTION) {
		if ((status = qla4xxx_login_device(ha,
						   logout.TargetID,
						   logout.ConnectionID))
		    == QLA_SUCCESS) {
			QL4PRINT(QLP4,
				 printk("scsi%d: %s: RELOGIN_CONNECTION "
					"SUCCEEDED!, target %d\n",
					ha->host_no, __func__,
					logout.TargetID));

			ioctl->Status = EXT_STATUS_OK;
		}
		else {
			QL4PRINT(QLP2,
				 printk("scsi%d: %s: RELOGIN_CONNECTION "
					"FAILED!, target %d\n",
					ha->host_no, __func__,
					logout.TargetID));

			ioctl->Status = EXT_STATUS_ERR;
		}
	}
	else if (logout.Options == INT_DEF_DELETE_DDB) {
		if ((status = qla4xxx_delete_device(ha,
						    logout.TargetID,
						    logout.ConnectionID))
		    == QLA_SUCCESS) {
			QL4PRINT(QLP4,
				 printk("scsi%d: %s: DELETE_DDB "
					"SUCCEEDED!, target %d\n",
					ha->host_no, __func__,
					logout.TargetID));

			ioctl->Status = EXT_STATUS_OK;
		}
		else {
			QL4PRINT(QLP2,
				 printk("scsi%d: %s: DELETE_DDB FAILED!, "
					"target %d\n", ha->host_no,
					__func__, logout.TargetID));

			ioctl->Status = EXT_STATUS_ERR;
		}
	}

	exit_logout:
	LEAVE("qla4intioctl_logout_iscsi");
	return(status);
}

/**************************************************************************
 * qla4intioctl_ping
 *	This routine requests that the HBA PING the specified IP Address.
 *
 * Input:
 *	ha    = adapter structure pointer.
 *	ioctl = IOCTL structure pointer.
 *
 * Output:
 *	None
 *
 * Returns:
 *	QLA_SUCCESS = success
 *	QLA_ERROR   = error
 *
 * Context:
 *	Kernel context.
 **************************************************************************/
static int
qla4intioctl_ping(scsi_qla_host_t *ha, EXT_IOCTL_ISCSI *ioctl)
{
	INT_PING        ping;
	uint32_t mbox_cmd[MBOX_REG_COUNT];
	uint32_t mbox_sts[MBOX_REG_COUNT];

	/*
	 * Copy user's data to local buffer
	 */
	if (copy_from_user((uint8_t *)&ping,
			   (void*)_64BIT_TO_ULONG(ioctl->RequestAdr),
			   sizeof(ping)) != 0) {
		QL4PRINT(QLP2,
			 printk("scsi%d: %s: unable to copy data from "
				"user's memory area\n", ha->host_no, __func__));
		ioctl->Status = EXT_STATUS_COPY_ERR;
		return(QLA_ERROR);
	}

	/*
	 * Debug Print Statement
	 */
	if (ping.IPAddr.Type == EXT_DEF_TYPE_ISCSI_IP) {
		QL4PRINT(QLP4,
			 printk("scsi%d: %s: %d.%d.%d.%d\n",
				ha->host_no, __func__,
				ping.IPAddr.IPAddress[0],
				ping.IPAddr.IPAddress[1],
				ping.IPAddr.IPAddress[2],
				ping.IPAddr.IPAddress[3]));
	}
	else {
		QL4PRINT(QLP4,
			 printk("scsi%d: %s: %d.%d.%d.%d. %d.%d.%d.%d. "
				"%d.%d.%d.%d. %d.%d.%d.%d\n",
				ha->host_no, __func__,
				ping.IPAddr.IPAddress[0],
				ping.IPAddr.IPAddress[1],
				ping.IPAddr.IPAddress[2],
				ping.IPAddr.IPAddress[3],
				ping.IPAddr.IPAddress[4],
				ping.IPAddr.IPAddress[5],
				ping.IPAddr.IPAddress[6],
				ping.IPAddr.IPAddress[7],
				ping.IPAddr.IPAddress[8],
				ping.IPAddr.IPAddress[9],
				ping.IPAddr.IPAddress[10],
				ping.IPAddr.IPAddress[11],
				ping.IPAddr.IPAddress[12],
				ping.IPAddr.IPAddress[13],
				ping.IPAddr.IPAddress[14],
				ping.IPAddr.IPAddress[15]));
	}

	/*
	 * Issue Mailbox Command
	 */
	memset(&mbox_cmd, 0, sizeof(mbox_cmd));
	memset(&mbox_sts, 0, sizeof(mbox_sts));
	mbox_cmd[0] = MBOX_CMD_PING;
	mbox_cmd[1] = ping.PacketCount;
	memcpy(&mbox_cmd[2], &ping.IPAddr.IPAddress, EXT_DEF_IP_ADDR_SIZE);

	if (qla4xxx_mailbox_command(ha, 6, 1, &mbox_cmd[0], &mbox_sts[0])
	    == QLA_ERROR) {
		QL4PRINT(QLP2, printk("scsi%d: %s: command failed \n",
				      ha->host_no, __func__));

		ioctl->Status = EXT_STATUS_ERR;
		ioctl->DetailStatus = mbox_sts[0];
		return(QLA_ERROR);
	}

	ioctl->Status = EXT_STATUS_OK;
	return(QLA_SUCCESS);
}

#if BYTE_ORDER_SUPPORT_ENABLED
void __xlate_sys_info(FLASH_SYS_INFO *from,
		      FLASH_SYS_INFO *to,
		      uint8_t direction,
		      uint32_t length)
{
	switch (direction) {
	case GET_DATA:
		from->cookie            = __le32_to_cpu(to->cookie);
		from->physAddrCount     = __le32_to_cpu(to->physAddrCount);
		memcpy(from->physAddr,  to->physAddr, sizeof(from->physAddr));
		memcpy(from->vendorId,  to->vendorId, sizeof(from->vendorId));
		memcpy(from->productId, to->productId, sizeof(from->productId));
		from->serialNumber      = __le32_to_cpu(to->serialNumber);
		from->pciDeviceVendor   = __le32_to_cpu(to->pciDeviceVendor);
		from->pciDeviceId       = __le32_to_cpu(to->pciDeviceId);
		from->pciSubsysVendor   = __le32_to_cpu(to->pciSubsysVendor);
		from->pciSubsysId       = __le32_to_cpu(to->pciSubsysId);
		from->crumbs            = __le32_to_cpu(to->crumbs);
		from->enterpriseNumber  = __le32_to_cpu(to->enterpriseNumber);
		from->mtu               = __le32_to_cpu(to->mtu);
		from->reserved0         = __le32_to_cpu(to->reserved0);
		from->crumbs2           = __le32_to_cpu(to->crumbs2);
		memcpy(from->acSerialNumber, to->acSerialNumber, sizeof(from->acSerialNumber));
		from->crumbs3           = __le32_to_cpu(to->crumbs3);
		memcpy(from->reserved1, to->reserved1, sizeof(from->reserved1));
		break;

	case SET_DATA:
		from->cookie            = __cpu_to_le32(to->cookie);
		from->physAddrCount     = __cpu_to_le32(to->physAddrCount);
		memcpy(from->physAddr,  to->physAddr, sizeof(from->physAddr));
		memcpy(from->vendorId,  to->vendorId, sizeof(from->vendorId));
		memcpy(from->productId, to->productId, sizeof(from->productId));
		from->serialNumber      = __cpu_to_le32(to->serialNumber);
		from->pciDeviceVendor   = __cpu_to_le32(to->pciDeviceVendor);
		from->pciDeviceId       = __cpu_to_le32(to->pciDeviceId);
		from->pciSubsysVendor   = __cpu_to_le32(to->pciSubsysVendor);
		from->pciSubsysId       = __cpu_to_le32(to->pciSubsysId);
		from->crumbs            = __cpu_to_le32(to->crumbs);
		from->enterpriseNumber  = __cpu_to_le32(to->enterpriseNumber);
		from->mtu               = __cpu_to_le32(to->mtu);
		from->reserved0         = __cpu_to_le32(to->reserved0);
		from->crumbs2           = __cpu_to_le32(to->crumbs2);
		memcpy(from->acSerialNumber, to->acSerialNumber, sizeof(from->acSerialNumber));
		from->crumbs3           = __cpu_to_le32(to->crumbs3);
		memcpy(from->reserved1, to->reserved1, sizeof(from->reserved1));
		break;
	}
}

void __xlate_driver_info(INT_FLASH_DRIVER_PARAM *from,
			 INT_FLASH_DRIVER_PARAM *to,
			 uint8_t direction,
			 uint32_t length)
{
	switch (direction) {
	case GET_DATA:
		from->DiscoveryTimeOut = __le16_to_cpu(to->DiscoveryTimeOut);
		from->PortDownTimeout = __le16_to_cpu(to->PortDownTimeout);
		memcpy(from->Reserved, to->Reserved, sizeof(from->Reserved));
		break;

	case SET_DATA:
		from->DiscoveryTimeOut = __cpu_to_le32(to->DiscoveryTimeOut);
		from->PortDownTimeout = __cpu_to_le32(to->PortDownTimeout);
		memcpy(from->Reserved, to->Reserved, sizeof(from->Reserved));
		break;
	}
}

void __xlate_init_fw_ctrl_blk(INIT_FW_CTRL_BLK *from,
			      INIT_FW_CTRL_BLK *to,
			      uint8_t direction,
			      uint32_t length)
{
	switch (direction) {
	case GET_DATA:
		from->Version           = to->Version;
		from->Control           = to->Control;
		from->FwOptions         = __le16_to_cpu(to->FwOptions);
		from->ExecThrottle      = __le16_to_cpu(to->ExecThrottle);
		from->RetryCount        = to->RetryCount;
		from->RetryDelay        = to->RetryDelay;
		from->MaxEthFrPayloadSize = __le16_to_cpu(to->MaxEthFrPayloadSize);
		from->AddFwOptions      = __le16_to_cpu(to->AddFwOptions);
		from->HeartbeatInterval = to->HeartbeatInterval;
		from->InstanceNumber    = to->InstanceNumber;
		from->RES2              = __le16_to_cpu(to->RES2);
		from->ReqQConsumerIndex = __le16_to_cpu(to->ReqQConsumerIndex);
		from->ComplQProducerIndex = __le16_to_cpu(to->ComplQProducerIndex);
		from->ReqQLen           = __le16_to_cpu(to->ReqQLen);
		from->ComplQLen         = __le16_to_cpu(to->ComplQLen);
		from->ReqQAddrLo        = __le32_to_cpu(to->ReqQAddrLo);
		from->ReqQAddrHi        = __le32_to_cpu(to->ReqQAddrHi);
		from->ComplQAddrLo      = __le32_to_cpu(to->ComplQAddrLo);
		from->ComplQAddrHi      = __le32_to_cpu(to->ComplQAddrHi);
		from->ShadowRegBufAddrLo= __le32_to_cpu(to->ShadowRegBufAddrLo);
		from->ShadowRegBufAddrHi= __le32_to_cpu(to->ShadowRegBufAddrHi);
		from->iSCSIOptions      = __le16_to_cpu(to->iSCSIOptions);
		from->TCPOptions        = __le16_to_cpu(to->TCPOptions);
		from->IPOptions         = __le16_to_cpu(to->IPOptions);
		from->MaxPDUSize        = __le16_to_cpu(to->MaxPDUSize);
		from->RcvMarkerInt      = __le16_to_cpu(to->RcvMarkerInt);
		from->SndMarkerInt      = __le16_to_cpu(to->SndMarkerInt);
		from->InitMarkerlessInt = __le16_to_cpu(to->InitMarkerlessInt);
		from->FirstBurstSize    = __le16_to_cpu(to->FirstBurstSize);
		from->DefaultTime2Wait  = __le16_to_cpu(to->DefaultTime2Wait);
		from->DefaultTime2Retain= __le16_to_cpu(to->DefaultTime2Retain);
		from->MaxOutStndngR2T   = __le16_to_cpu(to->MaxOutStndngR2T);
		from->KeepAliveTimeout  = __le16_to_cpu(to->KeepAliveTimeout);
		from->PortNumber        = __le16_to_cpu(to->PortNumber);
		from->MaxBurstSize      = __le16_to_cpu(to->MaxBurstSize);
		from->RES4              = __le32_to_cpu(to->RES4);
		memcpy(from->IPAddr, to->IPAddr, sizeof(from->IPAddr));
		memcpy(from->RES5, to->RES5, sizeof(from->RES5));
		memcpy(from->SubnetMask, to->SubnetMask, sizeof(from->SubnetMask));
		memcpy(from->RES6, to->RES6, sizeof(from->RES6));
		memcpy(from->GatewayIPAddr, to->GatewayIPAddr, sizeof(from->GatewayIPAddr));
		memcpy(from->RES7, to->RES7, sizeof(from->RES7));
		memcpy(from->PriDNSIPAddr, to->PriDNSIPAddr, sizeof(from->PriDNSIPAddr));
		memcpy(from->SecDNSIPAddr, to->SecDNSIPAddr, sizeof(from->SecDNSIPAddr));
		memcpy(from->RES8, to->RES8, sizeof(from->RES8));
		memcpy(from->Alias, to->Alias, sizeof(from->Alias));
		memcpy(from->TargAddr, to->TargAddr, sizeof(from->TargAddr));
		memcpy(from->CHAPNameSecretsTable, to->CHAPNameSecretsTable, sizeof(from->CHAPNameSecretsTable));
		memcpy(from->EthernetMACAddr, to->EthernetMACAddr, sizeof(from->EthernetMACAddr));
		from->TargetPortalGroup = __le16_to_cpu(to->TargetPortalGroup);
		from->SendScale         = to->SendScale;
		from->RecvScale         = to->RecvScale;
		from->TypeOfService     = to->TypeOfService;
		from->Time2Live         = to->Time2Live;
		from->VLANPriority      = __le16_to_cpu(to->VLANPriority);
		from->Reserved8         = __le16_to_cpu(to->Reserved8);
		memcpy(from->SecIPAddr, to->SecIPAddr, sizeof(from->SecIPAddr));
		memcpy(from->Reserved9, to->Reserved9, sizeof(from->Reserved9));
		memcpy(from->iSNSIPAddr, to->iSNSIPAddr, sizeof(from->iSNSIPAddr));
		memcpy(from->Reserved10, to->Reserved10, sizeof(from->Reserved10));
		from->iSNSServerPortNumber      = __le16_to_cpu(to->iSNSServerPortNumber);
		memcpy(from->SLPDAIPAddr, to->SLPDAIPAddr, sizeof(from->SLPDAIPAddr));
		memcpy(from->Reserved11, to->Reserved11, sizeof(from->Reserved11));
		memcpy(from->iSCSINameString, to->iSCSINameString, sizeof(from->iSCSINameString));
		break;

	case SET_DATA:
		from->Version           = to->Version;
		from->Control           = to->Control;
		from->FwOptions         = __cpu_to_le16(to->FwOptions);
		from->ExecThrottle      = __cpu_to_le16(to->ExecThrottle);
		from->RetryCount        = to->RetryCount;
		from->RetryDelay        = to->RetryDelay;
		from->MaxEthFrPayloadSize = __cpu_to_le16(to->MaxEthFrPayloadSize);
		from->AddFwOptions      = __cpu_to_le16(to->AddFwOptions);
		from->HeartbeatInterval = to->HeartbeatInterval;
		from->InstanceNumber    = to->InstanceNumber;
		from->RES2              = __cpu_to_le16(to->RES2);
		from->ReqQConsumerIndex = __cpu_to_le16(to->ReqQConsumerIndex);
		from->ComplQProducerIndex = __cpu_to_le16(to->ComplQProducerIndex);
		from->ReqQLen           = __cpu_to_le16(to->ReqQLen);
		from->ComplQLen         = __cpu_to_le16(to->ComplQLen);
		from->ReqQAddrLo        = __cpu_to_le32(to->ReqQAddrLo);
		from->ReqQAddrHi        = __cpu_to_le32(to->ReqQAddrHi);
		from->ComplQAddrLo      = __cpu_to_le32(to->ComplQAddrLo);
		from->ComplQAddrHi      = __cpu_to_le32(to->ComplQAddrHi);
		from->ShadowRegBufAddrLo= __cpu_to_le32(to->ShadowRegBufAddrLo);
		from->ShadowRegBufAddrHi= __cpu_to_le32(to->ShadowRegBufAddrHi);
		from->iSCSIOptions      = __cpu_to_le16(to->iSCSIOptions);
		from->TCPOptions        = __cpu_to_le16(to->TCPOptions);
		from->IPOptions         = __cpu_to_le16(to->IPOptions);
		from->MaxPDUSize        = __cpu_to_le16(to->MaxPDUSize);
		from->RcvMarkerInt      = __cpu_to_le16(to->RcvMarkerInt);
		from->SndMarkerInt      = __cpu_to_le16(to->SndMarkerInt);
		from->InitMarkerlessInt = __cpu_to_le16(to->InitMarkerlessInt);
		from->FirstBurstSize    = __cpu_to_le16(to->FirstBurstSize);
		from->DefaultTime2Wait  = __cpu_to_le16(to->DefaultTime2Wait);
		from->DefaultTime2Retain= __cpu_to_le16(to->DefaultTime2Retain);
		from->MaxOutStndngR2T   = __cpu_to_le16(to->MaxOutStndngR2T);
		from->KeepAliveTimeout  = __cpu_to_le16(to->KeepAliveTimeout);
		from->PortNumber        = __cpu_to_le16(to->PortNumber);
		from->MaxBurstSize      = __cpu_to_le16(to->MaxBurstSize);
		from->RES4              = __cpu_to_le32(to->RES4);
		memcpy(from->IPAddr, to->IPAddr, sizeof(from->IPAddr));
		memcpy(from->RES5, to->RES5, sizeof(from->RES5));
		memcpy(from->SubnetMask, to->SubnetMask, sizeof(from->SubnetMask));
		memcpy(from->RES6, to->RES6, sizeof(from->RES6));
		memcpy(from->GatewayIPAddr, to->GatewayIPAddr, sizeof(from->GatewayIPAddr));
		memcpy(from->RES7, to->RES7, sizeof(from->RES7));
		memcpy(from->PriDNSIPAddr, to->PriDNSIPAddr, sizeof(from->PriDNSIPAddr));
		memcpy(from->SecDNSIPAddr, to->SecDNSIPAddr, sizeof(from->SecDNSIPAddr));
		memcpy(from->RES8, to->RES8, sizeof(from->RES8));
		memcpy(from->Alias, to->Alias, sizeof(from->Alias));
		memcpy(from->TargAddr, to->TargAddr, sizeof(from->TargAddr));
		memcpy(from->CHAPNameSecretsTable, to->CHAPNameSecretsTable, sizeof(from->CHAPNameSecretsTable));
		memcpy(from->EthernetMACAddr, to->EthernetMACAddr, sizeof(from->EthernetMACAddr));
		from->TargetPortalGroup = __cpu_to_le16(to->TargetPortalGroup);
		from->SendScale         = to->SendScale;
		from->RecvScale         = to->RecvScale;
		from->TypeOfService     = to->TypeOfService;
		from->Time2Live         = to->Time2Live;
		from->VLANPriority      = __cpu_to_le16(to->VLANPriority);
		from->Reserved8         = __cpu_to_le16(to->Reserved8);
		memcpy(from->SecIPAddr, to->SecIPAddr, sizeof(from->SecIPAddr));
		memcpy(from->Reserved9, to->Reserved9, sizeof(from->Reserved9));
		memcpy(from->iSNSIPAddr, to->iSNSIPAddr, sizeof(from->iSNSIPAddr));
		memcpy(from->Reserved10, to->Reserved10, sizeof(from->Reserved10));
		from->iSNSServerPortNumber      = __cpu_to_le16(to->iSNSServerPortNumber);
		memcpy(from->SLPDAIPAddr, to->SLPDAIPAddr, sizeof(from->SLPDAIPAddr));
		memcpy(from->Reserved11, to->Reserved11, sizeof(from->Reserved11));
		memcpy(from->iSCSINameString, to->iSCSINameString, sizeof(from->iSCSINameString));
		break;
	}
}
void __xlate_dev_db(DEV_DB_ENTRY *from,
		    DEV_DB_ENTRY *to,
		    uint8_t direction,
		    uint32_t length)
{
	switch (direction) {
	case GET_DATA:
		from->options           = to->options;
		from->control           = to->control;
		from->exeThrottle       = __le16_to_cpu(to->exeThrottle);
		from->exeCount          = __le16_to_cpu(to->exeCount);
		from->retryCount        = to->retryCount;
		from->retryDelay        = to->retryDelay;
		from->iSCSIOptions      = __le16_to_cpu(to->iSCSIOptions);
		from->TCPOptions        = __le16_to_cpu(to->TCPOptions);
		from->IPOptions         = __le16_to_cpu(to->IPOptions);
		from->maxPDUSize        = __le16_to_cpu(to->maxPDUSize);
		from->rcvMarkerInt      = __le16_to_cpu(to->rcvMarkerInt);
		from->sndMarkerInt      = __le16_to_cpu(to->sndMarkerInt);
		from->iSCSIMaxSndDataSegLen     = __le16_to_cpu(to->iSCSIMaxSndDataSegLen);
		from->firstBurstSize    = __le16_to_cpu(to->firstBurstSize);
		from->minTime2Wait      = __le16_to_cpu(to->minTime2Wait);
		from->maxTime2Retain    = __le16_to_cpu(to->maxTime2Retain);
		from->maxOutstndngR2T   = __le16_to_cpu(to->maxOutstndngR2T);
		from->keepAliveTimeout  = __le16_to_cpu(to->keepAliveTimeout);
		memcpy(from->ISID, to->ISID, sizeof(from->ISID));
		from->TSID              = __le16_to_cpu(to->TSID);
		from->portNumber        = __le16_to_cpu(to->portNumber);
		from->maxBurstSize      = __le16_to_cpu(to->maxBurstSize);
		from->taskMngmntTimeout = __le16_to_cpu(to->taskMngmntTimeout);
		from->reserved1         = __le16_to_cpu(to->reserved1);
		memcpy(from->ipAddr, to->ipAddr, sizeof(from->ipAddr));
		memcpy(from->iSCSIAlias, to->iSCSIAlias, sizeof(from->iSCSIAlias));
		memcpy(from->targetAddr, to->targetAddr, sizeof(from->targetAddr));
		memcpy(from->userID, to->userID, sizeof(from->userID));
		memcpy(from->password, to->password, sizeof(from->password));
		memcpy(from->iscsiName, to->iscsiName, sizeof(from->iscsiName));
		from->ddbLink           = __le16_to_cpu(to->ddbLink);
		from->CHAPTableIndex    = __le16_to_cpu(to->CHAPTableIndex);
		memcpy(from->reserved2, to->reserved2, sizeof(from->reserved2));
		from->Cookie            = __le16_to_cpu(to->Cookie);
		break;

	case SET_DATA:
		from->options           = to->options;
		from->control           = to->control;
		from->exeThrottle       = __cpu_to_le16(to->exeThrottle);
		from->exeCount          = __cpu_to_le16(to->exeCount);
		from->retryCount        = to->retryCount;
		from->retryDelay        = to->retryDelay;
		from->iSCSIOptions      = __cpu_to_le16(to->iSCSIOptions);
		from->TCPOptions        = __cpu_to_le16(to->TCPOptions);
		from->IPOptions         = __cpu_to_le16(to->IPOptions);
		from->maxPDUSize        = __cpu_to_le16(to->maxPDUSize);
		from->rcvMarkerInt      = __cpu_to_le16(to->rcvMarkerInt);
		from->sndMarkerInt      = __cpu_to_le16(to->sndMarkerInt);
		from->iSCSIMaxSndDataSegLen     = __cpu_to_le16(to->iSCSIMaxSndDataSegLen);
		from->firstBurstSize    = __cpu_to_le16(to->firstBurstSize);
		from->minTime2Wait      = __cpu_to_le16(to->minTime2Wait);
		from->maxTime2Retain    = __cpu_to_le16(to->maxTime2Retain);
		from->maxOutstndngR2T   = __cpu_to_le16(to->maxOutstndngR2T);
		from->keepAliveTimeout  = __cpu_to_le16(to->keepAliveTimeout);
		memcpy(from->ISID, to->ISID, sizeof(from->ISID));
		from->TSID              = __cpu_to_le16(to->TSID);
		from->portNumber        = __cpu_to_le16(to->portNumber);
		from->maxBurstSize      = __cpu_to_le16(to->maxBurstSize);
		from->taskMngmntTimeout = __cpu_to_le16(to->taskMngmntTimeout);
		from->reserved1         = __cpu_to_le16(to->reserved1);
		memcpy(from->ipAddr, to->ipAddr, sizeof(from->ipAddr));
		memcpy(from->iSCSIAlias, to->iSCSIAlias, sizeof(from->iSCSIAlias));
		memcpy(from->targetAddr, to->targetAddr, sizeof(from->targetAddr));
		memcpy(from->userID, to->userID, sizeof(from->userID));
		memcpy(from->password, to->password, sizeof(from->password));
		memcpy(from->iscsiName, to->iscsiName, sizeof(from->iscsiName));
		from->ddbLink           = __cpu_to_le16(to->ddbLink);
		from->CHAPTableIndex    = __cpu_to_le16(to->CHAPTableIndex);
		memcpy(from->reserved2, to->reserved2, sizeof(from->reserved2));
		from->Cookie            = __cpu_to_le16(to->Cookie);
		break;
	}
}

void __xlate_chap(CHAP_ENTRY *from,
		  CHAP_ENTRY *to,
		  uint8_t direction,
		  uint32_t length)
{
	switch (direction) {
	case GET_DATA:
		from->link              = __le16_to_cpu(to->link);
		from->flags             = to->flags;
		from->secretLength      = to->secretLength;
		memcpy(from->secret, to->secret, sizeof(from->secret));
		memcpy(from->user_name, to->user_name, sizeof(from->user_name));
		from->reserved          = __le16_to_cpu(to->reserved);
		from->cookie            = __le16_to_cpu(to->cookie);
		break;

	case SET_DATA:
		from->link              = __cpu_to_le16(to->link);
		from->flags             = to->flags;
		from->secretLength      = to->secretLength;
		memcpy(from->secret, to->secret, sizeof(from->secret));
		memcpy(from->user_name, to->user_name, sizeof(from->user_name));
		from->reserved          = __cpu_to_le16(to->reserved);
		from->cookie            = __cpu_to_le16(to->cookie);
		break;
	}
}
#endif

/**************************************************************************
 * qla4intioctl_get_flash
 *	This routine reads the requested area of FLASH.
 *
 * Input:
 *	ha    = adapter structure pointer.
 *	ioctl = IOCTL structure pointer.
 *
 * Output:
 *	None
 *
 * Returns:
 *	QLA_SUCCESS = success
 *	QLA_ERROR   = error
 *
 * Context:
 *	Kernel context.
 **************************************************************************/
static int
qla4intioctl_get_flash(scsi_qla_host_t *ha, EXT_IOCTL_ISCSI *ioctl)
{
	int              status = QLA_ERROR;
	dma_buf_t        local_dma_buf;
	INT_ACCESS_FLASH *access_flash;
	uint32_t  mbox_cmd[MBOX_REG_COUNT];
	uint32_t  mbox_sts[MBOX_REG_COUNT];

	ENTER("qla4intioctl_get_flash");
	QL4PRINT(QLP3, printk("scsi%d: %s:\n", ha->host_no, __func__));

	/*
	 * Allocate local flash buffer
	 */
	access_flash = kmalloc(sizeof(*access_flash), GFP_ATOMIC);
	if (access_flash == NULL) {
		QL4PRINT(QLP2,
			 printk("scsi%d: %s: unable to allocate local memory\n",
				ha->host_no, __func__));

		ioctl->Status = EXT_STATUS_NO_MEMORY;
		goto exit_get_flash;
	}

	/*
	 * Copy user's data to local flash buffer
	 */
	if (copy_from_user((uint8_t *)access_flash,
			   (void*)_64BIT_TO_ULONG(ioctl->RequestAdr),
			   sizeof(*access_flash)) != 0) {
		QL4PRINT(QLP2,
			 printk("scsi%d: %s: unable to copy data from user's "
				"memory area\n", ha->host_no, __func__));

		ioctl->Status = EXT_STATUS_COPY_ERR;
		goto exit_get_flash;
	}

	/*
	 * Allocate DMA memory
	 */
	if (qla4xxx_alloc_dma_memory(ha->pdev,
				     &local_dma_buf,
				     access_flash->DataLen) == QLA_ERROR) {
		QL4PRINT(QLP2,
			 printk("scsi%d: %s: unable to allocate dma memory\n",
				ha->host_no, __func__));

		ioctl->Status = EXT_STATUS_NO_MEMORY;
		goto exit_get_flash;
	}

	QL4PRINT(QLP4, printk("scsi%d: %s: offset=%08x, len=%08x\n",
			      ha->host_no, __func__,
			      access_flash->DataOffset, access_flash->DataLen));

	/*
	 * Issue Mailbox Command
	 */
	memset(&mbox_cmd, 0, sizeof(mbox_cmd));
	memset(&mbox_sts, 0, sizeof(mbox_sts));
	mbox_cmd[0] = MBOX_CMD_READ_FLASH;
	mbox_cmd[1] = LSDW(local_dma_buf.phys_addr);
	mbox_cmd[2] = MSDW(local_dma_buf.phys_addr);
	mbox_cmd[3] = access_flash->DataOffset;
	mbox_cmd[4] = access_flash->DataLen;

	if (qla4xxx_mailbox_command(ha, 5, 2, &mbox_cmd[0], &mbox_sts[0])
	    == QLA_ERROR) {
		QL4PRINT(QLP2, printk("scsi%d: %s: command failed \n",
				      ha->host_no, __func__));

		ioctl->Status = EXT_STATUS_ERR;
		ioctl->DetailStatus = mbox_sts[0];
		goto exit_get_flash;
	}

	//FIXME: For byte-order support, this entire structure must be translated
	#if BYTE_ORDER_SUPPORT_ENABLED
	/*
	 * Copy data from DMA buffer into access_flash->FlashData buffer
	 * (in the process, translating for byte-order support, if necessary)
	 */
	switch (access_flash->DataOffset & INT_ISCSI_PAGE_MASK) {
	case INT_ISCSI_SYSINFO_FLASH_OFFSET:
		__xlate_sys_info((FLASH_SYS_INFO *) local_dma_buf.virt_addr,
				 (FLASH_SYS_INFO *) &access_flash->FlashData[0],
				 GET_DATA, access_flash->DataLen);
		break;
	case INT_ISCSI_DRIVER_FLASH_OFFSET:
		__xlate_driver_info((INT_FLASH_DRIVER_PARAM *) local_dma_buf.virt_addr,
				    (INT_FLASH_DRIVER_PARAM *) &access_flash->FlashData[0],
				    GET_DATA, access_flash->DataLen);
		break;
	case INT_ISCSI_INITFW_FLASH_OFFSET:
		__xlate_init_fw_ctrl_blk((INIT_FW_CTRL_BLK *) local_dma_buf.virt_addr,
					 (INIT_FW_CTRL_BLK *) &access_flash->FlashData[0],
					 GET_DATA, access_flash->DataLen);
		break;
	case INT_ISCSI_DDB_FLASH_OFFSET:
		__xlate_dev_db((DEV_DB_ENTRY *)local_dma_buf.virt_addr,
			       (DEV_DB_ENTRY *) &access_flash->FlashData[0],
			       GET_DATA, access_flash->DataLen);
		break;
	case INT_ISCSI_CHAP_FLASH_OFFSET:
		__xlate_chap((CHAP_ENTRY *) local_dma_buf.virt_addr,
			     (CHAP_ENTRY *) &access_flash->FlashData[0],
			     GET_DATA, access_flash->DataLen);
		break;
	case INT_ISCSI_FW_IMAGE2_FLASH_OFFSET:
	case INT_ISCSI_FW_IMAGE1_FLASH_OFFSET:
	default:
		memcpy(&access_flash->FlashData[0], local_dma_buf.virt_addr,
		       MIN(local_dma_buf.buf_len, sizeof(access_flash->FlashData)));
		break;
	}
	#else
	memcpy(&access_flash->FlashData[0], local_dma_buf.virt_addr,
	       MIN(local_dma_buf.buf_len, sizeof(access_flash->FlashData)));

	#endif

	/*
	 * Copy local DMA buffer to user's response data area
	 */
	if (copy_to_user((void*)_64BIT_TO_ULONG(ioctl->ResponseAdr),
			 access_flash, sizeof(*access_flash)) != 0) {
		QL4PRINT(QLP2,
			 printk("scsi%d: %s: unable to copy data to user's "
				"memory area\n", ha->host_no, __func__));

		ioctl->Status = EXT_STATUS_COPY_ERR;
		goto exit_get_flash;
	}

	ioctl->Status = EXT_STATUS_OK;
	ioctl->ResponseLen = access_flash->DataLen;
	QL4PRINT(QLP10,
		 printk("INT_ACCESS_FLASH buffer (1st 60h bytes only):\n"));
	qla4xxx_dump_bytes(QLP10, access_flash, 0x60);
	status = QLA_SUCCESS;

	exit_get_flash:
	/*
	 * Free Memory
	 */

	if (local_dma_buf.virt_addr)
		qla4xxx_free_dma_memory(ha->pdev, &local_dma_buf);

	if (access_flash)
		kfree(access_flash);

	LEAVE("qla4intioctl_get_flash");
	return(status);
}

/**************************************************************************
 * qla4intioctl_get_driver_debug_level
 *	This routine retrieves the driver's debug print level.
 *
 * Input:
 *	ha    = adapter structure pointer.
 *	ioctl = IOCTL structure pointer.
 *
 * Output:
 *	None
 *
 * Returns:
 *	QLA_SUCCESS = success
 *	QLA_ERROR   = error
 *
 * Context:
 *	Kernel context.
 **************************************************************************/
static int
qla4intioctl_get_driver_debug_level(scsi_qla_host_t *ha, EXT_IOCTL_ISCSI *ioctl)
{
	int status = QLA_ERROR;
	uint32_t dbg_level;

	ENTER("qla4intioctl_get_driver_debug_level");

	if (qla4xxx_get_debug_level(&dbg_level) == QLA_ERROR) {
		QL4PRINT(QLP2,
			 printk("scsi%d: %s: unable to set debug level, "
				"debug driver not loaded!\n",
				ha->host_no, __func__));

		ioctl->Status = EXT_STATUS_ERR;
		goto exit_get_driver_debug_level;
	}

	if (put_user(dbg_level, (uint32_t *) _64BIT_TO_ULONG(ioctl->ResponseAdr)) != 0) {
		QL4PRINT(QLP2,
			 printk("scsi%d: %s: failed to copy data\n",
				ha->host_no, __func__));

		ioctl->Status = EXT_STATUS_COPY_ERR;
		goto exit_get_driver_debug_level;
	}

	QL4PRINT(QLP2|QLP4, printk("scsi%d: %s: debug level is %04x\n",
				   ha->host_no, __func__, dbg_level));

	status = QLA_SUCCESS;
	ioctl->Status = EXT_STATUS_OK;

	exit_get_driver_debug_level:

	LEAVE("qla4intioctl_get_driver_debug_level");
	return(status);
}

/**************************************************************************
 * qla4intioctl_get_host_no
 *	This routine retrieves the host number for the specified adapter
 *	instance.
 *
 * Input:
 *	ha    = adapter structure pointer.
 *	ioctl = IOCTL structure pointer.
 *
 * Output:
 *	None
 *
 * Returns:
 *	QLA_SUCCESS = success
 *	QLA_ERROR   = error
 *
 * Context:
 *	Kernel context.
 **************************************************************************/
static int
qla4intioctl_get_host_no(scsi_qla_host_t *ha, EXT_IOCTL_ISCSI *ioctl)
{
	int status = QLA_ERROR;

	ENTER("qla4intioctl_get_host_no");

	QL4PRINT(QLP4, printk("scsi%d: %s: for instance %d\n",
			      ha->host_no, __func__, ha->instance));

	if (put_user((uint32_t) ha->host_no, (uint32_t *) _64BIT_TO_ULONG(ioctl->ResponseAdr)) != 0) {
		QL4PRINT(QLP2,
			 printk("scsi%d: %s: failed to copy data\n",
				ha->host_no,        __func__));

		ioctl->Status = EXT_STATUS_COPY_ERR;
	}
	else {
		status = QLA_SUCCESS;
		ioctl->Status = EXT_STATUS_OK;
	}

	LEAVE("qla4intioctl_get_host_no");
	return(status);
}

/**************************************************************************
 * qla4intioctl_get_data
 *	This routine calls get data IOCTLs based on the IOCTL Sub Code.
 *
 * Input:
 *	ha    = adapter structure pointer.
 *	ioctl = IOCTL structure pointer.
 *
 * Output:
 *	None
 *
 * Returns:
 *	QLA_SUCCESS = success
 *	QLA_ERROR   = error
 *    	-EINVAL     = if the command is invalid
 *
 * Context:
 *	Kernel context.
 **************************************************************************/
static int
qla4intioctl_get_data(scsi_qla_host_t *ha, EXT_IOCTL_ISCSI *ioctl)
{
	switch (ioctl->SubCode) {
	case INT_SC_GET_FLASH:
		return(qla4intioctl_get_flash(ha, ioctl));

	case INT_SC_GET_DRIVER_DEBUG_LEVEL:
		return(qla4intioctl_get_driver_debug_level(ha, ioctl));

	case INT_SC_GET_HOST_NO:
		return(qla4intioctl_get_host_no(ha, ioctl));


	default:
		QL4PRINT(QLP2,
			 printk("scsi%d: %s: unsupported internal get data "
				"sub-command code (%X)\n",
				ha->host_no, __func__, ioctl->SubCode));

		ioctl->Status = EXT_STATUS_UNSUPPORTED_SUBCODE;
		return(-EINVAL);
	}
}

/**************************************************************************
 * qla4intioctl_set_flash
 *	This routine writes the requested area of FLASH.
 *
 * Input:
 *	ha    = adapter structure pointer.
 *	ioctl = IOCTL structure pointer.
 *
 * Output:
 *	None
 *
 * Returns:
 *	QLA_SUCCESS = success
 *	QLA_ERROR   = error
 *
 * Context:
 *	Kernel context.
 **************************************************************************/
static int
qla4intioctl_set_flash(scsi_qla_host_t *ha, EXT_IOCTL_ISCSI *ioctl)
{
	int status = QLA_ERROR;
	dma_buf_t        local_dma_buf;
	INT_ACCESS_FLASH *access_flash;
	uint32_t         mbox_cmd[MBOX_REG_COUNT];
	uint32_t         mbox_sts[MBOX_REG_COUNT];

	ENTER("qla4intioctl_set_flash");
	QL4PRINT(QLP3, printk("scsi%d: %s:\n", ha->host_no, __func__));

	/*
	 * Allocate local flash buffer
	 */
	access_flash = kmalloc(sizeof(*access_flash), GFP_ATOMIC);
	if (access_flash == NULL) {
		QL4PRINT(QLP2,
			 printk("scsi%d: %s: unable to allocate local memory\n",
				ha->host_no, __func__));

		ioctl->Status = EXT_STATUS_NO_MEMORY;
		goto exit_set_flash;
	}

	/*
	 * Copy user's data to local DMA buffer
	 */
	if (copy_from_user((uint8_t *)access_flash,
			   (void*)_64BIT_TO_ULONG(ioctl->RequestAdr),
			   sizeof(*access_flash)) != 0) {
		QL4PRINT(QLP2,
			 printk("scsi%d: %s: unable to copy data from user's "
				"memory area\n", ha->host_no, __func__));

		ioctl->Status = EXT_STATUS_COPY_ERR;
		goto exit_set_flash;
	}

	/*
	 * Allocate DMA memory
	 */
	if (qla4xxx_alloc_dma_memory(ha->pdev,
				     &local_dma_buf,
				     access_flash->DataLen) == QLA_ERROR) {
		QL4PRINT(QLP2,
			 printk("scsi%d: %s: unable to allocate dma memory\n",
				ha->host_no, __func__));

		ioctl->Status = EXT_STATUS_NO_MEMORY;
		goto exit_set_flash;
	}

	//FIXME: For byte-order support, this entire structure must be translated
	#if BYTE_ORDER_SUPPORT_ENABLED
	/*
	 * Copy data from DMA buffer into access_flash->FlashData buffer
	 * (in the process, translating for byte-order support, if necessary)
	 */
	switch (access_flash->DataOffset & INT_ISCSI_PAGE_MASK) {
	case INT_ISCSI_SYSINFO_FLASH_OFFSET:
		__xlate_sys_info((FLASH_SYS_INFO *) &access_flash->FlashData[0],
				 (FLASH_SYS_INFO *) local_dma_buf.virt_addr,
				 SET_DATA, access_flash->DataLen);
		break;
	case INT_ISCSI_DRIVER_FLASH_OFFSET:
		__xlate_driver_info((INT_FLASH_DRIVER_PARAM *) &access_flash->FlashData[0],
				    (INT_FLASH_DRIVER_PARAM *) local_dma_buf.virt_addr,
				    SET_DATA, access_flash->DataLen);
		break;
	case INT_ISCSI_INITFW_FLASH_OFFSET:
		__xlate_init_fw_ctrl_blk((INIT_FW_CTRL_BLK *) &access_flash->FlashData[0],
					 (INIT_FW_CTRL_BLK *) local_dma_buf.virt_addr,
					 SET_DATA, access_flash->DataLen);
		break;
	case INT_ISCSI_DDB_FLASH_OFFSET:
		__xlate_dev_db((DEV_DB_ENTRY *) &access_flash->FlashData[0],
			       (DEV_DB_ENTRY *) local_dma_buf.virt_addr,
			       SET_DATA, access_flash->DataLen);
		break;
	case INT_ISCSI_CHAP_FLASH_OFFSET:
		__xlate_chap((CHAP_ENTRY *) &access_flash->FlashData[0],
			     (CHAP_ENTRY *) local_dma_buf.virt_addr,
			     SET_DATA, access_flash->DataLen);
		break;
	case INT_ISCSI_FW_IMAGE2_FLASH_OFFSET:
	case INT_ISCSI_FW_IMAGE1_FLASH_OFFSET:
	default:
		memcpy(local_dma_buf.virt_addr, &access_flash->FlashData[0],
		       MIN(local_dma_buf.buf_len, sizeof(access_flash->FlashData)));
		break;
	}
	#else
	memcpy(local_dma_buf.virt_addr, &access_flash->FlashData[0],
	       MIN(local_dma_buf.buf_len, sizeof(access_flash->FlashData)));
	#endif

	/*
	 * Issue Mailbox Command
	 */
	memset(&mbox_cmd, 0, sizeof(mbox_cmd));
	memset(&mbox_sts, 0, sizeof(mbox_sts));
	mbox_cmd[0] = MBOX_CMD_WRITE_FLASH;
	mbox_cmd[1] = LSDW(local_dma_buf.phys_addr);
	mbox_cmd[2] = MSDW(local_dma_buf.phys_addr);
	mbox_cmd[3] = access_flash->DataOffset;
	mbox_cmd[4] = access_flash->DataLen;
	mbox_cmd[5] = access_flash->Options;

	if (qla4xxx_mailbox_command(ha, 6, 2, &mbox_cmd[0], &mbox_sts[0])
	    == QLA_ERROR) {
		QL4PRINT(QLP2, printk("scsi%d: %s: command failed \n",
				      ha->host_no, __func__));

		ioctl->Status = EXT_STATUS_ERR;
		ioctl->DetailStatus = mbox_sts[0];
		ioctl->VendorSpecificStatus[0] = mbox_sts[1];
		goto exit_set_flash;
	}

	ioctl->Status = EXT_STATUS_OK;
	ioctl->ResponseLen = access_flash->DataLen;
	QL4PRINT(QLP10,
		 printk("scsi%d): INT_ACCESS_FLASH buffer (1st 60h bytes only:\n",
			ha->host_no));
	qla4xxx_dump_bytes(QLP10, local_dma_buf.virt_addr, 0x60);
	status = QLA_SUCCESS;

	exit_set_flash:
	/*
	 * Free Memory
	 */
	if (local_dma_buf.virt_addr)
		qla4xxx_free_dma_memory(ha->pdev, &local_dma_buf);

	if (access_flash)
		kfree(access_flash);

	LEAVE("qla4intioctl_set_flash");
	return(status);
}

/**************************************************************************
 * qla4intioctl_set_driver_debug_level
 *	This routine sets the driver's debug print level.
 *
 * Input:
 *	ha    = adapter structure pointer.
 *	ioctl = IOCTL structure pointer.
 *
 * Output:
 *	None
 *
 * Returns:
 *	QLA_SUCCESS = success
 *	QLA_ERROR   = error
 *
 * Context:
 *	Kernel context.
 **************************************************************************/
static int
qla4intioctl_set_driver_debug_level(scsi_qla_host_t *ha, EXT_IOCTL_ISCSI *ioctl)
{
	int status = QLA_ERROR;
	uint32_t dbg_level;

	ENTER("qla4intioctl_set_driver_debug_level");

	if (get_user(dbg_level, (uint32_t *) _64BIT_TO_ULONG(ioctl->RequestAdr)) != 0) {
		QL4PRINT(QLP2, printk("scsi%d: %s: failed to copy data\n",
				      ha->host_no, __func__));

		ioctl->Status = EXT_STATUS_COPY_ERR;
		goto exit_set_driver_debug_level;
	}

	if (qla4xxx_set_debug_level(dbg_level) == QLA_ERROR) {
		QL4PRINT(QLP2,
			 printk("scsi%d: %s: unable to set debug level, "
				"debug driver not loaded!\n",
				ha->host_no, __func__));

		ioctl->Status = EXT_STATUS_ERR;
		goto exit_set_driver_debug_level;
	}

	QL4PRINT(QLP2|QLP4,
		 printk("scsi%d: %s: debug level set to 0x%04X\n",
			ha->host_no, __func__, dbg_level));

	status = QLA_SUCCESS;
	ioctl->Status = EXT_STATUS_OK;

	exit_set_driver_debug_level:

	LEAVE("qla4intioctl_set_driver_debug_level");
	return(status);
}

/**************************************************************************
 * qla4intioctl_set_data
 *	This routine calls set data IOCTLs based on the IOCTL Sub Code.
 *
 * Input:
 *	ha    = adapter structure pointer.
 *	ioctl = IOCTL structure pointer.
 *
 * Output:
 *	None
 *
 * Returns:
 *	QLA_SUCCESS = success
 *	QLA_ERROR   = error
 *    	-EINVAL     = if the command is invalid
 *
 * Context:
 *	Kernel context.
 **************************************************************************/
static int
qla4intioctl_set_data(scsi_qla_host_t *ha, EXT_IOCTL_ISCSI *ioctl)
{
	switch (ioctl->SubCode) {
	case INT_SC_SET_FLASH:
		return(qla4intioctl_set_flash(ha, ioctl));

	case INT_SC_SET_DRIVER_DEBUG_LEVEL:
		return(qla4intioctl_set_driver_debug_level(ha, ioctl));

	default:
		QL4PRINT(QLP2,
			 printk("scsi%d: %s: unsupported internal set data "
				"sub-command code (%X)\n",
				ha->host_no, __func__, ioctl->SubCode));

		ioctl->Status = EXT_STATUS_UNSUPPORTED_SUBCODE;
		return(-EINVAL);
	}
}

/**************************************************************************
 * qla4intioctl_hba_reset
 *	This routine resets the specified HBA.
 *
 * Input:
 *	ha    = adapter structure pointer.
 *	ioctl = IOCTL structure pointer.
 *
 * Output:
 *	None
 *
 * Returns:
 *	QLA_SUCCESS = success
 *	QLA_ERROR   = error
 *
 * Context:
 *	Kernel context.
 **************************************************************************/
static int
qla4intioctl_hba_reset(scsi_qla_host_t *ha, EXT_IOCTL_ISCSI *ioctl)
{
	int status = QLA_ERROR;
	u_long     wait_count;

	ENTER("qla4intioctl_hba_reset");
	QL4PRINT(QLP4, printk("scsi%d: %s:\n", ha->host_no, __func__));

	switch (ioctl->SubCode) {
	case INT_SC_HBA_RESET:
	case INT_SC_FIRMWARE_RESET:
		set_bit(DPC_RESET_HA_DESTROY_DDB_LIST, &ha->dpc_flags);
		set_bit(AF_SCHEDULE_DPC, &ha->flags);

		/* Wait a fixed amount of time for reset to complete */
		wait_count = jiffies + ADAPTER_RESET_TOV * HZ;
		while (test_bit(DPC_RESET_HA_DESTROY_DDB_LIST, &ha->dpc_flags) != 0) {
			if (wait_count <= jiffies)
				break;

			/* wait for 1 second */
			set_current_state(TASK_INTERRUPTIBLE);
			schedule_timeout(1*HZ);
		}

		if (test_bit(AF_ONLINE, &ha->flags))
			status = QLA_SUCCESS;
		else
			status = QLA_ERROR;
		break;

	case INT_SC_TARGET_WARM_RESET:
	case INT_SC_LUN_RESET:
		break;
	}

	if (status == QLA_SUCCESS) {
		QL4PRINT(QLP4, printk("scsi%d: %s: Succeeded\n",
				      ha->host_no, __func__));

		status = QLA_SUCCESS;
		ioctl->Status = EXT_STATUS_OK;
	}
	else {
		QL4PRINT(QLP4, printk("scsi%d: %s: FAILED\n",
				      ha->host_no, __func__));

		ioctl->Status = EXT_STATUS_ERR;
	}

	LEAVE("qla4intioctl_hba_reset");
	return(status);
}

/**************************************************************************
 * qla4intioctl_copy_fw_flash
 *	This routine requests copying the FW image in FLASH from primary-to-
 *	secondary or secondary-to-primary.
 *
 * Input:
 *	ha    = adapter structure pointer.
 *	ioctl = IOCTL structure pointer.
 *
 * Output:
 *	None
 *
 * Returns:
 *	QLA_SUCCESS = success
 *	QLA_ERROR   = error
 *
 * Context:
 *	Kernel context.
 **************************************************************************/
static int
qla4intioctl_copy_fw_flash(scsi_qla_host_t *ha, EXT_IOCTL_ISCSI *ioctl)
{
	int status = QLA_ERROR;
	INT_COPY_FW_FLASH copy_flash;
	uint32_t mbox_cmd[MBOX_REG_COUNT];
	uint32_t mbox_sts[MBOX_REG_COUNT];

	ENTER("qla4intioctl_copy_fw_flash");
	QL4PRINT(QLP3, printk("scsi%d: %s:\n", ha->host_no, __func__));

	if (copy_from_user((uint8_t *)&copy_flash,
			   (void *)_64BIT_TO_ULONG(ioctl->RequestAdr),
			   ioctl->RequestLen) != 0) {
		QL4PRINT(QLP2,
			 printk("scsi%d: %s: unable to copy data from user's "
				"memory area\n", ha->host_no, __func__));

		ioctl->Status = EXT_STATUS_COPY_ERR;
		goto exit_copy_flash;
	}

	memset(&mbox_cmd, 0, sizeof(mbox_cmd));
	memset(&mbox_sts, 0, sizeof(mbox_sts));
	mbox_cmd[0] = MBOX_CMD_COPY_FLASH;
	mbox_cmd[1] = copy_flash.Options;

	if (qla4xxx_mailbox_command(ha, 2, 2, &mbox_cmd[0], &mbox_sts[0])
	    == QLA_SUCCESS) {
		QL4PRINT(QLP10, printk("scsi%d: %s: Succeeded\n",
				       ha->host_no, __func__));
		status = QLA_SUCCESS;
		ioctl->Status = EXT_STATUS_OK;
	}
	else {
		QL4PRINT(QLP10, printk("scsi%d: %s: FAILED\n",
				       ha->host_no, __func__));

		ioctl->Status = EXT_STATUS_ERR;
		ioctl->DetailStatus = mbox_sts[0];
		ioctl->VendorSpecificStatus[0] = mbox_sts[1];
	}

	exit_copy_flash:
	LEAVE("qla4intioctl_copy_fw_flash");
	return(status);
}

/**************************************************************************
 * qla4xxx_iocb_pass_done
 *	This routine resets the ioctl progress flag and wakes up the ioctl
 * 	completion semaphore.
 *
 * Input:
 *	ha    = adapter structure pointer.
 *   	sts_entry - pointer to passthru status buffer.
 *
 * Output:
 *	None
 *
 * Returns:
 *	None
 *
 * Context:
 *	Interrupt context.
 **************************************************************************/
void
qla4xxx_iocb_pass_done(scsi_qla_host_t *ha, PASSTHRU_STATUS_ENTRY *sts_entry)
{
	INT_IOCB_PASSTHRU *iocb;

	ENTER("qla4xxx_iocb_pass_done");

	/* --- Copy passthru status buffer to iocb passthru buffer ---*/
	iocb = (INT_IOCB_PASSTHRU *) (unsigned long) sts_entry->handle;
	memcpy(iocb->IOCBStatusBuffer, sts_entry,
	       MIN(sizeof(iocb->IOCBStatusBuffer),
		   sizeof(*sts_entry)));

	/* --- Reset IOCTL flags and wakeup semaphore.
	 *     But first check to see if IOCTL has already
	 *     timed out because we don't want to get the
	 *     up/down semaphore counters off.             --- */
	if (ha->ioctl_iocb_pass_in_progress == TRUE) {
		ha->ioctl_iocb_pass_in_progress = FALSE;
		ha->ioctl_tov = 0;

		QL4PRINT(QLP10, printk("%s: UP count=%d\n", __func__,
				       atomic_read(&ha->ioctl_cmpl_sem.count)));
		up(&ha->ioctl_cmpl_sem);
	}

	LEAVE("qla4xxx_iocb_pass_done");
	return;
}


#ifndef QLA4000
/**************************************************************************
 * qla4intioctl_iocb_passthru
 *	This routine
 *	
 *
 * Input:
 *	ha    = adapter structure pointer.
 *	ioctl = IOCTL structure pointer.
 *
 * Output:
 *	None
 *
 * Returns:
 *	QLA_SUCCESS = success
 *	QLA_ERROR   = error
 *
 * Context:
 *	Kernel context.
 **************************************************************************/
static int
qla4intioctl_iocb_passthru(scsi_qla_host_t *ha, EXT_IOCTL_ISCSI *ioctl)
{
	int status = QLA_ERROR;
	INT_IOCB_PASSTHRU *iocb;
	INT_IOCB_PASSTHRU *iocb_phys;
	PASSTHRU0_ENTRY *passthru_entry;
	unsigned long flags;
	DATA_SEG_A64 *data_seg;


	ENTER("qla4intioctl_iocb_passthru");
	QL4PRINT(QLP3, printk("scsi%d: %s:\n", ha->host_no, __func__));

	/* --- Use internal DMA buffer for iocb structure --- */
#if 0 && __VMKERNEL_MODULE__
	spin_lock(&ha->dma_buf_spinlock);
#else
	down(&ha->dma_buf_sem);
#endif

	if (ha->dma_buf.buf_len < sizeof(*iocb))
		qla4xxx_resize_dma_buf(ha, &ha->dma_buf, sizeof(*iocb));
	if (!ha->dma_buf.virt_addr || ha->dma_buf.buf_len < sizeof(*iocb)) {
		QL4PRINT(QLP2, printk("scsi%d: %s: dma buffer inaccessible.\n",
				      ha->host_no, __func__));
		ioctl->Status = EXT_STATUS_BUFFER_TOO_SMALL;
		goto exit_iocb_passthru;
	}

	iocb = (INT_IOCB_PASSTHRU *) ha->dma_buf.virt_addr;
	iocb_phys = (INT_IOCB_PASSTHRU *)(unsigned long)ha->dma_buf.phys_addr;

	/* --- Copy IOCB_PASSTHRU structure from user space --- */
	if (copy_from_user((uint8_t *)iocb,
			   (void *)_64BIT_TO_ULONG(ioctl->RequestAdr),
			   ioctl->RequestLen) != 0) {
		QL4PRINT(QLP2,
			 printk("scsi%d: %s: unable to copy data from user's "
				"memory area\n", ha->host_no, __func__));

		ioctl->Status = EXT_STATUS_COPY_ERR;
		goto exit_iocb_passthru;
	}


	/* --- Get pointer to the passthru queue entry --- */
	spin_lock_irqsave(&ha->hardware_lock, flags);
	if (qla4xxx_get_req_pkt(ha, (QUEUE_ENTRY **) &passthru_entry)
	    != QLA_SUCCESS) {
		QL4PRINT(QLP2,
			 printk("scsi%d: %s: request queue full, try again later\n",
				ha->host_no, __func__));

		ioctl->Status = EXT_STATUS_HBA_QUEUE_FULL;
		spin_unlock_irqrestore(&ha->hardware_lock, flags);
		goto exit_iocb_passthru;
	}

	/* --- Fill in passthru queue entry --- */
	if (iocb->SendDMAOffset) {
		data_seg = (DATA_SEG_A64 *) (iocb->IOCBCmdBuffer + iocb->SendDMAOffset);
		data_seg->base.addrHigh = __cpu_to_le32(MSDW(iocb_phys->SendData));
		data_seg->base.addrLow  = __cpu_to_le32(LSDW(iocb_phys->SendData));
	}

	if (iocb->RspDMAOffset) {
		data_seg = (DATA_SEG_A64 *) (iocb->IOCBCmdBuffer + iocb->RspDMAOffset);
		data_seg->base.addrHigh = __cpu_to_le32(MSDW(iocb_phys->RspData));
		data_seg->base.addrLow  = __cpu_to_le32(LSDW(iocb_phys->RspData));
	}

	memcpy(passthru_entry, iocb->IOCBCmdBuffer,
	       MIN(sizeof(*passthru_entry), sizeof(iocb->IOCBCmdBuffer)));
	passthru_entry->handle = (uint32_t) (unsigned long) iocb;
	passthru_entry->hdr.systemDefined = SD_PASSTHRU_IOCB;

	if (passthru_entry->hdr.entryType != ET_PASSTHRU0)
		passthru_entry->timeout = MBOX_TOV;

	QL4PRINT(QLP10, printk(KERN_INFO "scsi%d: Passthru0 IOCB type %x count %x In (%x) %p\n",
			       ha->host_no, passthru_entry->hdr.entryType,
			       passthru_entry->hdr.entryCount, ha->request_in, passthru_entry));

	QL4PRINT(QLP10, printk(KERN_INFO "scsi%d: Dump Passthru entry %p: \n",
			       ha->host_no, passthru_entry));
	qla4xxx_dump_bytes(QLP10, passthru_entry, sizeof(*passthru_entry));

	/* ---- Prepare for receiving completion ---- */
	ha->ioctl_iocb_pass_in_progress = TRUE;
	ha->ioctl_tov = passthru_entry->timeout * HZ;
	qla4xxx_ioctl_sem_init(ha);

	/* ---- Send command to adapter ---- */
	ha->ioctl_cmpl_timer.expires = jiffies + ha->ioctl_tov;
	add_timer(&ha->ioctl_cmpl_timer);

	WRT_REG_DWORD(&ha->reg->requestQueueInPointer, ha->request_in);
	spin_unlock_irqrestore(&ha->hardware_lock, flags);

	QL4PRINT(QLP10, printk("%s: DOWN count=%d\n", __func__, atomic_read(&ha->ioctl_cmpl_sem.count)));
	down(&ha->ioctl_cmpl_sem);

	/*******************************************************
	 *						       *
	 *             Passthru Completion                     *
	 *						       *
	 *******************************************************/
	del_timer(&ha->ioctl_cmpl_timer);

	/* ---- Check for timeout --- */
	if (ha->ioctl_iocb_pass_in_progress == TRUE) {
		QL4PRINT(QLP2,
			 printk("scsi%d: %s: ERROR = command timeout.\n",
				ha->host_no, __func__));

		ha->ioctl_iocb_pass_in_progress = FALSE;
		ioctl->Status = EXT_STATUS_ERR;
		goto exit_iocb_passthru;
	}

	/* ---- Copy IOCB Passthru structure with updated status buffer
	 *      to user space ---- */
	if (copy_to_user((void *)_64BIT_TO_ULONG(ioctl->ResponseAdr),
			 iocb, sizeof(INT_IOCB_PASSTHRU)) != 0) {
		QL4PRINT(QLP2,
			 printk("scsi%d: %s: unable to copy passthru struct "
				"to user's memory area.\n",
				ha->host_no, __func__));

		ioctl->Status = EXT_STATUS_COPY_ERR;
		goto exit_iocb_passthru;
	}

	QL4PRINT(QLP10, printk("Dump iocb structure (OUT)\n"));
	qla4xxx_print_iocb_passthru(QLP10, ha, iocb);

	QL4PRINT(QLP4, printk("scsi%d: %s: Succeeded\n",
			      ha->host_no, __func__));
	status = QLA_SUCCESS;
	ioctl->Status = EXT_STATUS_OK;

	exit_iocb_passthru:
#if 0 && __VMKERNEL_MODULE__
	spin_unlock(&ha->dma_buf_spinlock);
#else
	up(&ha->dma_buf_sem);
#endif
	LEAVE("qla4intioctl_iocb_passthru");
	return(status);
}
#endif

typedef struct {
	int cmd;
	char *s;
} ioctl_tbl_row_t;

ioctl_tbl_row_t IOCTL_CMD_TBL[] =
{
	{EXT_CC_QUERY, "EXT_CC_QUERY"},
	{EXT_CC_REG_AEN, "EXT_CC_REG_AEN"},
	{EXT_CC_GET_AEN, "EXT_CC_GET_AEN"},
	{EXT_CC_GET_DATA, "EXT_CC_GET_DATA"},
	{EXT_CC_SET_DATA, "EXT_CC_SET_DATA"},
	{EXT_CC_SEND_SCSI_PASSTHRU, "EXT_CC_SEND_SCSI_PASSTHRU"},
	{EXT_CC_SEND_ISCSI_PASSTHRU, "EXT_CC_SEND_ISCSI_PASSTHRU"},
	{INT_CC_LOGOUT_ISCSI, "INT_CC_LOGOUT_ISCSI"},
	{EXT_CC_GET_HBACNT, "EXT_CC_GET_HBACNT"},
	{INT_CC_DIAG_PING, "INT_CC_DIAG_PING"},
	{INT_CC_GET_DATA, "INT_CC_GET_DATA"},
	{INT_CC_SET_DATA, "INT_CC_SET_DATA"},
	{INT_CC_HBA_RESET, "INT_CC_HBA_RESET"},
	{INT_CC_COPY_FW_FLASH, "INT_CC_COPY_FW_FLASH"},
	{INT_CC_IOCB_PASSTHRU, "INT_CC_IOCB_PASSTHRU"},
	{0, "UNKNOWN"}
};

ioctl_tbl_row_t IOCTL_SCMD_QUERY_TBL[] =
{
	{EXT_SC_QUERY_HBA_ISCSI_NODE, "EXT_SC_QUERY_HBA_ISCSI_NODE"},
	{EXT_SC_QUERY_HBA_ISCSI_PORTAL, "EXT_SC_QUERY_HBA_ISCSI_PORTAL"},
	{EXT_SC_QUERY_DISC_ISCSI_NODE, "EXT_SC_QUERY_DISC_ISCSI_NODE"},
	{EXT_SC_QUERY_DISC_ISCSI_PORTAL, "EXT_SC_QUERY_DISC_ISCSI_PORTAL"},
	{EXT_SC_QUERY_DRIVER, "EXT_SC_QUERY_DRIVER"},
	{EXT_SC_QUERY_FW, "EXT_SC_QUERY_FW"},
	{EXT_SC_QUERY_CHIP, "EXT_SC_QUERY_CHIP"},
	{0, "UNKNOWN"}
};

ioctl_tbl_row_t IOCTL_SCMD_EGET_DATA_TBL[] =
{
	{EXT_SC_GET_STATISTICS_ISCSI, "EXT_SC_GET_STATISTICS_ISCSI"},
	{EXT_SC_GET_DEVICE_ENTRY_ISCSI, "EXT_SC_GET_DEVICE_ENTRY_ISCSI"},
	{EXT_SC_GET_DEVICE_ENTRY_DEFAULTS_ISCSI, "EXT_SC_GET_DEVICE_ENTRY_DEFAULTS_ISCSI"},
	{EXT_SC_GET_INIT_FW_ISCSI, "EXT_SC_GET_INIT_FW_ISCSI"},
	{EXT_SC_GET_INIT_FW_DEFAULTS_ISCSI, "EXT_SC_GET_INIT_FW_DEFAULTS_ISCSI"},
	{EXT_SC_GET_ISNS_SERVER, "EXT_SC_GET_ISNS_SERVER"},
	{EXT_SC_GET_ISNS_DISCOVERED_TARGETS, "EXT_SC_GET_ISNS_DISCOVERED_TARGETS"},
	{0, "UNKNOWN"}
};

ioctl_tbl_row_t IOCTL_SCMD_ESET_DATA_TBL[] =
{
	{EXT_SC_RST_STATISTICS_GEN, "EXT_SC_RST_STATISTICS_GEN"},
	{EXT_SC_RST_STATISTICS_ISCSI, "EXT_SC_RST_STATISTICS_ISCSI"},
	{EXT_SC_SET_DEVICE_ENTRY_ISCSI, "EXT_SC_SET_DEVICE_ENTRY_ISCSI"},
	{EXT_SC_SET_INIT_FW_ISCSI, "EXT_SC_SET_INIT_FW_ISCSI"},
	{EXT_SC_SET_ISNS_SERVER, "EXT_SC_SET_ISNS_SERVER"},
	{0, "UNKNOWN"}
};

ioctl_tbl_row_t IOCTL_SCMD_IGET_DATA_TBL[] =
{
	{INT_SC_GET_FLASH, "INT_SC_GET_FLASH"},
	{INT_SC_GET_DRIVER_DEBUG_LEVEL, "INT_SC_GET_DRIVER_DEBUG_LEVEL"},
	{INT_SC_GET_HOST_NO, "INT_SC_GET_HOST_NO"},
	{0, "UNKNOWN"}
};

ioctl_tbl_row_t IOCTL_SCMD_ISET_DATA_TBL[] =
{
	{INT_SC_SET_FLASH, "INT_SC_SET_FLASH"},
	{INT_SC_SET_DRIVER_DEBUG_LEVEL, "INT_SC_SET_DRIVER_DEBUG_LEVEL"},
	{0, "UNKNOWN"}
};

char *IOCTL_TBL_STR(int cc, int sc)
{
	ioctl_tbl_row_t *r;
	int cmd;

	switch (cc) {
	case EXT_CC_QUERY:
		r = IOCTL_SCMD_QUERY_TBL;
		cmd = sc;
		break;
	case EXT_CC_GET_DATA:
		r = IOCTL_SCMD_EGET_DATA_TBL;
		cmd = sc;
		break;
	case EXT_CC_SET_DATA:
		r = IOCTL_SCMD_ESET_DATA_TBL;
		cmd = sc;
		break;
	case INT_CC_GET_DATA:
		r = IOCTL_SCMD_IGET_DATA_TBL;
		cmd = sc;
		break;
	case INT_CC_SET_DATA:
		r = IOCTL_SCMD_ISET_DATA_TBL;
		cmd = sc;
		break;

	default:
		r = IOCTL_CMD_TBL;
		cmd = cc;
		break;
	}

	while (r->cmd != 0) {
		if (r->cmd == cmd) break;
		r++;
	}
	return(r->s);

}

/**************************************************************************
 * qla4xxx_ioctl
 * 	This the main entry point for all ioctl requests
 *
 * Input:
 *    	dev - pointer to SCSI device structure
 *	cmd - internal or external ioctl command code
 *	arg - pointer to ioctl structure
 *
 *    	Instance field in ioctl structure - to determine which adapter to
 *    	perform ioctl
 *
 * Output:
 *	Status field in ioctl structure, if returns QLA_SUCCESS,
 *      QLA_ERROR or -EINVAL
 *
 * Returns:
 *	 QLA_SUCCESS
 *	 QLA_ERROR
 *    	-EFAULT      - if the arg pointer is NULL
 *    	-EINVAL      - if the command is invalid
 *
 * Context:
 *	Kernel context.
 **************************************************************************/
int qla4xxx_ioctl(Scsi_Device *dev, int cmd, void *arg)
{
#if 1 || __VMKERNEL_MODULE__
	EXT_IOCTL_ISCSI *ioctl = NULL;
#else
	static EXT_IOCTL_ISCSI qla4ioctl;
	EXT_IOCTL_ISCSI *ioctl = &qla4ioctl;
#endif
	scsi_qla_host_t *ha;
	int status = QLA_SUCCESS;

	ENTER("qla4xxx_ioctl");

	/*
	 * Check to see if we can access the ioctl command structure
	 */
#ifndef __VMKERNEL_MODULE__
	/* 
         * vmkernel doesn't implement access_ok()
	 */
	if (!access_ok(VERIFY_WRITE, arg, sizeof(EXT_IOCTL_ISCSI))) {
		QL4PRINT(QLP2,
			 printk("%s: NULL EXT_IOCTL_ISCSI buffer\n",
				__func__));

		status = (-EFAULT);
		goto exit_qla4xxx_ioctl;
	}
#else
	if ( !(ioctl = kmalloc (sizeof(*ioctl), GFP_ATOMIC)) ) {
		status = (-ENOMEM);
		goto exit_qla4xxx_ioctl;
	}
#endif

	/*
	 * Copy the ioctl command structure from user space to the local structure
	 */
#ifndef __VMKERNEL_MODULE__
	copy_from_user((uint8_t *)ioctl, arg, sizeof(EXT_IOCTL_ISCSI));
#else
	if (copy_from_user((uint8_t *)ioctl, arg, sizeof(EXT_IOCTL_ISCSI))) {
	        printk("qla4010: copy from user failed\n");
		QL4PRINT(QLP2,
			 printk("%s: Bad EXT_IOCTL_ISCSI buffer\n",
				__func__));

		status = (-EFAULT);
		goto exit_qla4xxx_ioctl;
	}
#endif // __VMKERNEL_MODULE__

	QL4PRINT(QLP10, printk("EXT_IOCTL_ISCSI structure: \n"));
	qla4xxx_dump_dwords(QLP10, ioctl, sizeof(*ioctl));

	/*
	 * Get the adapter handle for the corresponding adapter instance
	 */
	ha = qla4xxx_get_adapter_handle(ioctl->HbaSelect);
	if (ha == NULL) {
		QL4PRINT(QLP2,
			 printk("%s: NULL EXT_IOCTL_ISCSI buffer\n",
				__func__));

		status = QLA_ERROR;
		ioctl->Status = EXT_STATUS_DEV_NOT_FOUND;
		goto exit_qla4xxx_ioctl;
	}

	QL4PRINT(QLP4, printk("scsi%d: ioctl+ (%s)\n", ha->host_no, IOCTL_TBL_STR(cmd, ioctl->SubCode)));
	ha->i_start = jiffies;
	ha->i_end = 0;
	ha->f_start = 0;
	ha->f_end = 0;

	/*
	 * If the DPC is active, wait for it to complete before proceeding
	 */
	while (ha->dpc_active) {
		set_current_state(TASK_INTERRUPTIBLE);
		schedule_timeout(1*HZ);
	}

	down(&ha->ioctl_sem);

	/*
	 * Issue the ioctl command
	 */
	switch (cmd) {
	case EXT_CC_QUERY:
		status = qla4extioctl_query(ha, ioctl);
		break;

	case EXT_CC_REG_AEN:
		status = qla4extioctl_reg_aen(ha, ioctl);
		break;

	case EXT_CC_GET_AEN:
		status = qla4extioctl_get_aen(ha, ioctl);
		break;

	case EXT_CC_GET_DATA:
		status = qla4extioctl_get_data(ha, ioctl);
		break;

	case EXT_CC_SET_DATA:
		status = qla4extioctl_set_data(ha, ioctl);
		break;

	case EXT_CC_SEND_SCSI_PASSTHRU:
		status = qla4extioctl_scsi_passthru(ha, ioctl);
		break;

	case EXT_CC_SEND_ISCSI_PASSTHRU:
		status = qla4extioctl_iscsi_passthru(ha, ioctl);
		break;

	case INT_CC_LOGOUT_ISCSI:
		status = qla4intioctl_logout_iscsi(ha, ioctl);
		break;

	case EXT_CC_GET_HBACNT:
		status = qla4extioctl_get_hbacnt(ha, ioctl);
		break;

	case INT_CC_DIAG_PING:
		status = qla4intioctl_ping(ha, ioctl);
		break;

	case INT_CC_GET_DATA:
		status = qla4intioctl_get_data(ha, ioctl);
		break;

	case INT_CC_SET_DATA:
		status = qla4intioctl_set_data(ha, ioctl);
		break;

	case INT_CC_HBA_RESET:
		status = qla4intioctl_hba_reset(ha, ioctl);
		break;

	case INT_CC_COPY_FW_FLASH:
		status = qla4intioctl_copy_fw_flash(ha, ioctl);
		break;

#ifndef QLA4000
	case INT_CC_IOCB_PASSTHRU:
		status = qla4intioctl_iocb_passthru(ha, ioctl);
		break;
#endif

	default:
		QL4PRINT(QLP2,
			 printk("scsi%d: %s: unsupported command code (%X)\n",
				ha->host_no, __func__, cmd));

		ioctl->Status = EXT_STATUS_UNSUPPORTED_SUBCODE;
		status = (-EINVAL);
		up(&ha->ioctl_sem);
		goto exit_qla4xxx_ioctl;
	}

	/*
	 * Copy the updated ioctl structure back to the user
	 */
#ifndef __VMKERNEL_MODULE__
	copy_to_user(arg, (void *)ioctl, sizeof(EXT_IOCTL_ISCSI));
#else
	if (copy_to_user(arg, (void *)ioctl, sizeof(EXT_IOCTL_ISCSI))) {
		QL4PRINT(QLP2,
			 printk("%s: Bad EXT_IOCTL_ISCSI buffer\n",
				__func__));

		status = (-EFAULT);
	}
#endif // __VMKERNEL_MODULE__

	up(&ha->ioctl_sem);

	ha->i_end = jiffies;
	QL4PRINT(QLP15, printk("scsi%d: ioctl- (%s) "
			       "i_start=%lx, f_start=%lx, f_end=%lx, i_end=%lx\n",
			       ha->host_no, IOCTL_TBL_STR(cmd, ioctl->SubCode), ha->i_start, ha->f_start, ha->f_end, ha->i_end));

	exit_qla4xxx_ioctl:

#if 1 || __VMKERNEL_MODULE__
	if (ioctl) {
		kfree (ioctl);
	}
#endif

	LEAVE("qla4xxx_ioctl");
	return(status);
}

/**************************************************************************
 * apidev_open
 *	This routine is invoked just prior to every IOCTL call.  We only
 *	display debug information.
 *
 * Input:
 *	Unused
 *
 * Returns:
 *	0 - Always returns successful
 *
 * Context:
 *	Kernel context.
 **************************************************************************/
STATIC int
apidev_open(struct inode *inode, struct file *file)
{
	QL4PRINT(QLP4, printk("scsi: apidev_open MAJOR number = %d, "
			      "MINOR number = %d\n",
			      MAJOR (inode->i_rdev),
			      MINOR (inode->i_rdev)));
	return(0);
}

/**************************************************************************
 * apidev_close
 *	This routine is invoked just after every IOCTL call.  We only
 *	display debug information.
 *
 * Input:
 *	Unused
 *
 * Returns:
 *	0 - Always returns successful
 *
 * Context:
 *	Kernel context.
 **************************************************************************/
STATIC int
apidev_close(struct inode *inode, struct file *file)
{
	QL4PRINT(QLP4, printk("scsi: apidev_close\n"));
	return(0);
}

/**************************************************************************
 * apidev_ioctl
 *	This routine is invoked whenever an ioctl call is made.  It in turn
 *	calls the IOCTL function for this driver.
 *
 * Input:
 *	inode - unused
 *	fp    - unused
 *	cmd - internal or external ioctl command code
 *	arg - pointer to ioctl structure
 *
 * Output:
 *	None
 *
 * Returns:
 *	 QLA_SUCCESS - IOCTL completed successfully
 *	 QLA_ERROR   - IOCTL completed in error
 *    	-EFAULT      - if the arg pointer is NULL
 *    	-EINVAL      - if the command is invalid
 *
 * Context:
 *	Kernel context.
 **************************************************************************/
STATIC int
apidev_ioctl(struct inode *inode, struct file *fp,
	     unsigned int cmd, unsigned long arg)
{
	static Scsi_Device fake_scsi_device;
	fake_scsi_device.host = apidev_host;
	return(qla4xxx_ioctl(&fake_scsi_device, (int)cmd, (void*)arg));
}

static struct file_operations
apidev_fops = {
	ioctl:    apidev_ioctl,
	open:     apidev_open,
	release:  apidev_close
};

/**************************************************************************
 * apidev_init
 *	This routine creates a proc file for IOCTL interface.
 *
 * Input:
 *	None
 *
 * Output:
 *	apidev_host - Updated with desired host number.
 *	apidev_major - Registered.
 *
 * Remarks:
 *	Create character driver "HbaApiDev" w dynamically allocated major
 *	number and create "/proc/scsi/<QLA4XXX_PROC_NAME>/HbaApiNode" as
 *	the device node associated with the major number.
 *
 * Returns:
 *	0 - Always returns successful
 *
 * Context:
 *	Kernel context.
 **************************************************************************/
int
apidev_init(struct Scsi_Host *host)
{
	if (apidev_host) return(0);

	if (0 > (apidev_major = register_chrdev(0, APIDEV_NAME, &apidev_fops))) {
		QL4PRINT(QLP4|QLP7,
			 printk("scsi: apidev_init: rc=%d\n",
				apidev_major));
		return(apidev_major);
	}

	apidev_host = host;
	QL4PRINT(QLP4|QLP7,
		 printk("scsi: Created /proc/scsi/%s/%s major=%d\n",
			QLA4XXX_PROC_NAME, APIDEV_NODE, apidev_major));

	proc_mknod(APIDEV_NODE,
		   0600+S_IFCHR,
		   host->hostt->proc_dir,
		   (kdev_t)MKDEV(apidev_major,0));
	return(0);
}

/**************************************************************************
 * apidev_cleanup
 *	This routine removes the proc file for the IOCTL interface
 *
 * Input:
 *	None
 *
 * Output:
 *	apidev_host - Cleared.
 *	apidev_major - Unregistered.
 *
 * Returns:
 *	0 - Always returns successful
 *
 * Context:
 *	Kernel context.
 **************************************************************************/
int
apidev_cleanup(void)
{
	if (!apidev_host) return(0);
	unregister_chrdev(apidev_major,APIDEV_NAME);

	QL4PRINT(QLP4|QLP7, printk("scsi: apidev_cleanup\n"));
	remove_proc_entry(APIDEV_NODE,apidev_host->hostt->proc_dir);
	apidev_host = 0;
	return(0);
}



/*
 * Overrides for Emacs so that we almost follow Linus's tabbing style.
 * Emacs will notice this stuff at the end of the file and automatically
 * adjust the settings for this buffer only.  This must remain at the end
 * of the file.
 * ---------------------------------------------------------------------------
 * Local variables:
 * c-indent-level: 2
 * c-brace-imaginary-offset: 0
 * c-brace-offset: -2
 * c-argdecl-indent: 2
 * c-label-offset: -2
 * c-continued-statement-offset: 2
 * c-continued-brace-offset: 0
 * indent-tabs-mode: nil
 * tab-width: 8
 * End:
 */
