#include <linux/string.h>
const char *isns_error_code_msg[] = ISNS_ERROR_CODE_TBL();

void
qla4xxx_strtolower(uint8_t *str)
{
	uint8_t *tmp;
	for (tmp = str; *tmp != '\0'; tmp++) {
		if (*tmp >= 'A' && *tmp <= 'Z')
			*tmp += 'a' - 'A';
	}
}

void
qla4xxx_isns_build_entity_id(scsi_qla_host_t *ha)
{
	sprintf(ha->isns_entity_id, "eid:qlogic:qla4010-%s", ha->serial_number);
	qla4xxx_strtolower(ha->isns_entity_id);
}

uint8_t
qla4xxx_isns_enable(scsi_qla_host_t *ha,
		    uint32_t isns_ip_addr,
		    uint16_t isns_server_port_num)
{
	uint32_t mbox_cmd[MBOX_REG_COUNT];
	uint32_t mbox_sts[MBOX_REG_COUNT];

	QL4PRINT(QLP20, printk("scsi%d: %s: isns_ip_addr %08x\n",
			       ha->host_no, __func__, isns_ip_addr));

	qla4xxx_isns_build_entity_id(ha);

	memset(&mbox_cmd, 0, sizeof(mbox_cmd));
	memset(&mbox_sts, 0, sizeof(mbox_sts));
	mbox_cmd[0] = MBOX_CMD_SET_ISNS_SERVICE;
	mbox_cmd[1] = ISNS_ENABLE;
	mbox_cmd[2] = isns_ip_addr;
	mbox_cmd[3] = isns_server_port_num;

	if (qla4xxx_mailbox_command(ha, 4, 6, &mbox_cmd[0], &mbox_sts[0])
	    != QLA_SUCCESS) {
		QL4PRINT(QLP2,
			 printk("scsi%d: %s: MBOX_CMD_SET_ISNS_SERVICE failed "
				"w/ status %04X %04X\n",
				ha->host_no, __func__, mbox_sts[0], mbox_sts[1]));
		return(QLA_ERROR);
	}

	QL4PRINT(QLP7|QLP20, printk(KERN_INFO "scsi%d: Start iSNS Service "
				    "%d.%d.%d.%d Port %04d . . .\n", ha->host_no,
				    (isns_ip_addr & 0x000000FF),
				    (isns_ip_addr & 0x0000FF00) >> 8,
				    (isns_ip_addr & 0x00FF0000) >> 16,
				    (isns_ip_addr & 0xFF000000) >> 24,
				    isns_server_port_num));

	return(QLA_SUCCESS);
}

uint8_t
qla4xxx_isns_reenable(scsi_qla_host_t *ha,
		      uint32_t isns_ip_addr,
		      uint16_t isns_server_port_num)
{
	set_bit(ISNS_FLAG_REREGISTER, &ha->isns_flags);
	ISNS_CLEAR_FLAGS(ha);

	if (qla4xxx_isns_enable(ha, isns_ip_addr, isns_server_port_num)
	    != QLA_SUCCESS) {
		QL4PRINT(QLP2, printk("scsi%d: %s: Failed!\n",
				      ha->host_no, __func__));
		return(QLA_ERROR);
	}

	return(QLA_SUCCESS);
}

/* interrupt context, hardware lock set */
void
qla4xxx_isns_enable_callback(scsi_qla_host_t *ha,
			     uint32_t svr,
			     uint32_t scn,
			     uint32_t esi,
			     uint32_t nsh)
{
	ha->isns_connection_id   = (uint16_t) svr & 0x0000FFFF;
	ha->isns_scn_conn_id     = (uint16_t) scn & 0x0000FFFF;
	ha->isns_esi_conn_id     = (uint16_t) esi & 0x0000FFFF;
	ha->isns_nsh_conn_id     = (uint16_t) nsh & 0x0000FFFF;

	ha->isns_remote_port_num = (uint16_t) (svr >> 16);
	ha->isns_scn_port_num    = (uint16_t) (scn >> 16);
	ha->isns_esi_port_num    = (uint16_t) (esi >> 16);
	ha->isns_nsh_port_num    = (uint16_t) (nsh >> 16);

	QL4PRINT(QLP20,
		 printk("scsi%d: %s: iSNS Server TCP Connect succeeded %d\n",
			ha->host_no, __func__, svr));
	QL4PRINT(QLP20,
		 printk("scsi%d: %s: Remote iSNS Server %d ConnID %x\n",
			ha->host_no, __func__,
			ha->isns_remote_port_num,
			ha->isns_connection_id));
	QL4PRINT(QLP20,
		 printk("scsi%d: %s: Local  SCN  Listen %d ConnID %x\n",
			ha->host_no, __func__,
			ha->isns_scn_port_num,
			ha->isns_scn_conn_id));
	QL4PRINT(QLP20,
		 printk("scsi%d: %s: Local  ESI  Listen %d ConnID %x\n",
			ha->host_no, __func__,
			ha->isns_esi_port_num,
			ha->isns_esi_conn_id));
	QL4PRINT(QLP20,
		 printk("scsi%d: %s: Local  HSN  Listen %d ConnID %x\n",
			ha->host_no, __func__,
			ha->isns_nsh_port_num,
			ha->isns_nsh_conn_id));

	if (ha->isns_connection_id == (uint16_t)-1) {
		QL4PRINT(QLP2,
			 printk("scsi%d: %s: iSNS server refused connection\n",
				ha->host_no, __func__));

		qla4xxx_isns_restart_service(ha);
		return;
	}

	set_bit(ISNS_FLAG_ISNS_SRV_ENABLED, &ha->isns_flags);

	if (test_bit(ISNS_FLAG_REREGISTER, &ha->isns_flags)) {
		if (qla4xxx_isns_scn_dereg(ha) != QLA_SUCCESS) {
			QL4PRINT(QLP2,
				 printk("scsi%d: %s: qla4xxx_isns_scn_dereg failed!\n",
					ha->host_no, __func__));
			return;
		}
	}
	else {
		if (qla4xxx_isns_dev_attr_reg(ha) != QLA_SUCCESS) {
			QL4PRINT(QLP2,
				 printk("scsi%d: %s: qla4xxx_isns_dev_attr_reg failed!\n",
					ha->host_no, __func__));
			return;
		}
	}
}


uint8_t
qla4xxx_isns_restart_service(scsi_qla_host_t *ha)
{
	qla4xxx_isns_disable(ha);
	set_bit(ISNS_FLAG_RESTART_SERVICE, &ha->isns_flags);
	ISNS_CLEAR_FLAGS(ha);

	/* Set timer for restart to complete */
	atomic_set(&ha->isns_restart_timer, ISNS_RESTART_TOV);
	return(QLA_SUCCESS);
}

uint8_t
qla4xxx_isns_restart_service_completion(scsi_qla_host_t *ha,
					uint32_t isns_ip_addr,
					uint16_t isns_server_port_num)
{
	QL4PRINT(QLP20, printk("scsi%d: %s: isns_ip_addr %08x\n",
			       ha->host_no, __func__, isns_ip_addr));

	if (qla4xxx_isns_enable(ha, isns_ip_addr, isns_server_port_num)
	    != QLA_SUCCESS) {
		QL4PRINT(QLP2, printk("scsi%d: %s: failed!\n",
				      ha->host_no, __func__));
		return(QLA_ERROR);
	}
	else {
		set_bit(ISNS_FLAG_REREGISTER, &ha->isns_flags);
		ISNS_CLEAR_FLAGS(ha);
		return(QLA_SUCCESS);
	}
}

uint8_t
qla4xxx_isns_status(scsi_qla_host_t *ha)
{
	uint32_t mbox_cmd[MBOX_REG_COUNT];
	uint32_t mbox_sts[MBOX_REG_COUNT];

	memset(&mbox_cmd, 0, sizeof(mbox_cmd));
	memset(&mbox_sts, 0, sizeof(mbox_sts));
	mbox_cmd[0] = MBOX_CMD_SET_ISNS_SERVICE;
	mbox_cmd[1] = ISNS_STATUS;

	if (qla4xxx_mailbox_command(ha, 2, 2, &mbox_cmd[0], &mbox_sts[0])
	    != QLA_SUCCESS) {
		QL4PRINT(QLP2,
			 printk("scsi%d: %s: MBOX_CMD_SET_ISNS_SERVICE failed "
				"w/ status %04X %04X\n",
				ha->host_no, __func__, mbox_sts[0], mbox_sts[1]));
		return(QLA_ERROR);
	}

	QL4PRINT(QLP20, printk("scsi%d: %s: = %s\n",
			       ha->host_no, __func__,
			       ((mbox_sts[1] & 1) == 0) ? "DISABLED" : "ENABLED"));
	return(QLA_SUCCESS);
}

uint8_t
qla4xxx_isns_disable(scsi_qla_host_t *ha)
{
	uint32_t mbox_cmd[MBOX_REG_COUNT];
	uint32_t mbox_sts[MBOX_REG_COUNT];

	if (test_bit(ISNS_FLAG_ISNS_SRV_ENABLED, &ha->isns_flags)) {
		memset(&mbox_cmd, 0, sizeof(mbox_cmd));
		memset(&mbox_sts, 0, sizeof(mbox_sts));
		mbox_cmd[0] = MBOX_CMD_SET_ISNS_SERVICE;
		mbox_cmd[1] = ISNS_DISABLE;

		if (qla4xxx_mailbox_command(ha, 2, 2, &mbox_cmd[0], &mbox_sts[0])
		    != QLA_SUCCESS) {
			QL4PRINT(QLP2,
				 printk("scsi%d: %s: MBOX_CMD_SET_ISNS_SERVICE failed "
					"w/ status %04X %04X\n",
					ha->host_no, __func__, mbox_sts[0], mbox_sts[1]));
			return(QLA_ERROR);
		}
	}

	clear_bit(ISNS_FLAG_ISNS_SRV_ENABLED, &ha->isns_flags);
	ISNS_CLEAR_FLAGS(ha);

	ha->isns_connection_id   = 0;
	//ha->isns_scn_conn_id     = 0;
	//ha->isns_esi_conn_id     = 0;
	//ha->isns_nsh_conn_id     = 0;

	ha->isns_remote_port_num = 0;
	ha->isns_scn_port_num    = 0;
	ha->isns_esi_port_num    = 0;
	ha->isns_nsh_port_num    = 0;

	ha->isns_num_discovered_targets = 0;
	memset(ha->isns_entity_id, 0, sizeof(ha->isns_entity_id));
	return(QLA_SUCCESS);
}

void
qla4xxx_isns_init_isns_reg_attr_list(scsi_qla_host_t *ha)
{
	ATTRIBUTE_LIST isns_reg_attr_list[] = {
		// Source attribute
		{ ISNS_ATTR_TAG_ISCSI_NAME,        ISNS_ATTR_TYPE_STRING,  (unsigned long) ha->name_string},
		{ ISNS_ATTR_TAG_ENTITY_IDENTIFIER, ISNS_ATTR_TYPE_STRING,  -1},
		// Entity ID.
		{ ISNS_ATTR_TAG_DELIMITER,         ISNS_ATTR_TYPE_EMPTY,   0},
		// Operating attributes to register
		{ ISNS_ATTR_TAG_ENTITY_IDENTIFIER, ISNS_ATTR_TYPE_STRING,  -1},
		{ ISNS_ATTR_TAG_ENTITY_PROTOCOL,   ISNS_ATTR_TYPE_ULONG,   DWSWAP(ENTITY_PROTOCOL_ISCSI)},
		{ ISNS_ATTR_TAG_PORTAL_IP_ADDRESS, ISNS_ATTR_TYPE_ADDRESS, -1},
		{ ISNS_ATTR_TAG_PORTAL_PORT,       ISNS_ATTR_TYPE_ULONG,   -1},
		{ ISNS_ATTR_TAG_SCN_PORT,          ISNS_ATTR_TYPE_ULONG,   -1},
		{ ISNS_ATTR_TAG_ESI_PORT,          ISNS_ATTR_TYPE_ULONG,   -1},
		{ ISNS_ATTR_TAG_ISCSI_NAME,        ISNS_ATTR_TYPE_STRING,  (unsigned long) ha->name_string},
		{ ISNS_ATTR_TAG_ISCSI_NODE_TYPE,   ISNS_ATTR_TYPE_ULONG,   DWSWAP(ISCSI_NODE_TYPE_INITIATOR)},
		{ ISNS_ATTR_TAG_ISCSI_ALIAS,       ISNS_ATTR_TYPE_STRING,  -1},		// Friendly machine name?

		{ 0, 0, 0}		// Terminating NULL entry
	};

	memcpy(ha->isns_reg_attr_list,          isns_reg_attr_list,          sizeof(isns_reg_attr_list));
}

void
qla4xxx_isns_init_isns_dereg_attr_list(scsi_qla_host_t *ha)
{
	ATTRIBUTE_LIST isns_dereg_attr_list[] = {
		// Source attribute
		{ ISNS_ATTR_TAG_ISCSI_NAME,        ISNS_ATTR_TYPE_STRING,  (unsigned long) ha->name_string},
		// No key attribute for DevDereg
		{ ISNS_ATTR_TAG_DELIMITER,         ISNS_ATTR_TYPE_EMPTY,    0},
		// Operating attributes
		{ ISNS_ATTR_TAG_ENTITY_IDENTIFIER, ISNS_ATTR_TYPE_STRING,  -1},		// FQDN
#if 0
		{ ISNS_ATTR_TAG_PORTAL_IP_ADDRESS, ISNS_ATTR_TYPE_ADDRESS, -1},
		{ ISNS_ATTR_TAG_PORTAL_PORT,       ISNS_ATTR_TYPE_ULONG,   -1},
		{ ISNS_ATTR_TAG_ISCSI_NAME,        ISNS_ATTR_TYPE_STRING,  (unsigned long) ha->name_string},
#endif

		{ 0, 0, 0}		// Terminating NULL entry
	};

	memcpy(ha->isns_dereg_attr_list,        isns_dereg_attr_list,        sizeof(isns_dereg_attr_list));
}

void
qla4xxx_isns_init_isns_scn_reg_attr_list(scsi_qla_host_t *ha)
{
	ATTRIBUTE_LIST isns_scn_reg_attr_list[] = {
		// Source attribute
		{ ISNS_ATTR_TAG_ISCSI_NAME,        ISNS_ATTR_TYPE_STRING,  (unsigned long) ha->name_string},
		// Key attributes
		{ ISNS_ATTR_TAG_ISCSI_NAME,        ISNS_ATTR_TYPE_STRING,  (unsigned long) ha->name_string},
		// Required delimiter to indicate division between key and operating attrs.
		{ ISNS_ATTR_TAG_DELIMITER,         ISNS_ATTR_TYPE_EMPTY,   0},
		// Operating attributes
		{ ISNS_ATTR_TAG_ISCSI_SCN_BITMAP,  ISNS_ATTR_TYPE_ULONG,   DWSWAP(ISCSI_SCN_OBJECT_UPDATED |
										  ISCSI_SCN_OBJECT_ADDED |
										  ISCSI_SCN_OBJECT_REMOVED |
										  ISCSI_SCN_TARGET_AND_SELF_INFO_ONLY)},

		{ 0, 0, 0}		// Terminating NULL entry
	};

	memcpy(ha->isns_scn_reg_attr_list,      isns_scn_reg_attr_list,      sizeof(isns_scn_reg_attr_list));
}

void
qla4xxx_isns_init_isns_scn_dereg_attr_list(scsi_qla_host_t *ha)
{
	ATTRIBUTE_LIST isns_scn_dereg_attr_list[] = {
		// Source attribute
		{ ISNS_ATTR_TAG_ISCSI_NAME,        ISNS_ATTR_TYPE_STRING, (unsigned long) ha->name_string},
		// Key attributes
		{ ISNS_ATTR_TAG_ISCSI_NAME,        ISNS_ATTR_TYPE_STRING, (unsigned long) ha->name_string},

		{ 0, 0, 0}		// Terminating NULL entry
	};

	memcpy(ha->isns_scn_dereg_attr_list,    isns_scn_dereg_attr_list,    sizeof(isns_scn_dereg_attr_list));
}

void
qla4xxx_isns_init_isns_dev_get_next_attr_list(scsi_qla_host_t *ha)
{
	ATTRIBUTE_LIST isns_dev_get_next_attr_list[] = {
		// Source attribute
		{ ISNS_ATTR_TAG_ISCSI_NAME,        ISNS_ATTR_TYPE_STRING, (unsigned long) ha->name_string},
		// Key attributes
		{ ISNS_ATTR_TAG_ISCSI_NAME,        ISNS_ATTR_TYPE_STRING, -1},
		// Required delimiter to indicate division between key and operating attrs.
		{ ISNS_ATTR_TAG_DELIMITER,         ISNS_ATTR_TYPE_EMPTY,  0},
		// Operating attributes (attributes of object matching key attribute to return)
		{ ISNS_ATTR_TAG_ISCSI_NODE_TYPE,   ISNS_ATTR_TYPE_ULONG,  DWSWAP(ISCSI_NODE_TYPE_TARGET)},

		{ 0, 0, 0}		// Terminating NULL entry
	};

	memcpy(ha->isns_dev_get_next_attr_list, isns_dev_get_next_attr_list, sizeof(isns_dev_get_next_attr_list));
}

void
qla4xxx_isns_init_isns_dev_attr_qry_attr_list(scsi_qla_host_t *ha)
{
	ATTRIBUTE_LIST isns_dev_attr_qry_attr_list[] = {
		// Source attribute
		{ ISNS_ATTR_TAG_ISCSI_NAME,        ISNS_ATTR_TYPE_STRING, (unsigned long) ha->name_string},
		// Key attributes
		{ ISNS_ATTR_TAG_ISCSI_NAME,        ISNS_ATTR_TYPE_STRING, -1},
		// Required delimiter to indicate division between key and operating attrs.
		{ ISNS_ATTR_TAG_DELIMITER,         ISNS_ATTR_TYPE_EMPTY,  0},
		// Operating attributes (attributes of objects matching key attributes to return)
		{ ISNS_ATTR_TAG_ENTITY_PROTOCOL,   ISNS_ATTR_TYPE_EMPTY,  0},
		{ ISNS_ATTR_TAG_ISCSI_NAME,        ISNS_ATTR_TYPE_EMPTY,  0},
		{ ISNS_ATTR_TAG_ISCSI_NODE_TYPE,   ISNS_ATTR_TYPE_EMPTY,  0},
		{ ISNS_ATTR_TAG_ISCSI_ALIAS,       ISNS_ATTR_TYPE_EMPTY,  0},	// Friendly name
		{ ISNS_ATTR_TAG_PORTAL_SYMBOLIC_NAME, ISNS_ATTR_TYPE_EMPTY, 0},
		{ ISNS_ATTR_TAG_PORTAL_IP_ADDRESS, ISNS_ATTR_TYPE_EMPTY,  0},
		{ ISNS_ATTR_TAG_PORTAL_PORT,       ISNS_ATTR_TYPE_EMPTY,  0},
		{ ISNS_ATTR_TAG_PORTAL_SECURITY_BITMAP, ISNS_ATTR_TYPE_EMPTY, 0},
		{ ISNS_ATTR_TAG_DD_ID,             ISNS_ATTR_TYPE_EMPTY,  0},

		{ 0, 0, 0}		// Terminating NULL entry
	};

	memcpy(ha->isns_dev_attr_qry_attr_list, isns_dev_attr_qry_attr_list, sizeof(isns_dev_attr_qry_attr_list));
}

uint8_t
qla4xxx_isns_init_attributes (scsi_qla_host_t *ha)
{
	/* Separate these calls to minimize stack usage */

	qla4xxx_isns_init_isns_reg_attr_list(ha);
	qla4xxx_isns_init_isns_dereg_attr_list(ha);
	qla4xxx_isns_init_isns_scn_reg_attr_list(ha);
	qla4xxx_isns_init_isns_scn_dereg_attr_list(ha);
	qla4xxx_isns_init_isns_dev_get_next_attr_list(ha);
	qla4xxx_isns_init_isns_dev_attr_qry_attr_list(ha);

#if 0
	{
		ATTRIBUTE_LIST asRegUpdateAddObjectsAttrList[] = {
			// Source attribute
			{ ISNS_ATTR_TAG_ISCSI_NAME,        ISNS_ATTR_TYPE_STRING, (unsigned long) ha->name_string},
			// We are adding objects to an Entity so specify the Entity as the Key
			{ ISNS_ATTR_TAG_ENTITY_IDENTIFIER, ISNS_ATTR_TYPE_STRING, -1},	// FQDN
			{ ISNS_ATTR_TAG_DELIMITER,         ISNS_ATTR_TYPE_EMPTY,  0},
			// Operating attributes to register
			{ ISNS_ATTR_TAG_ISCSI_NAME,        ISNS_ATTR_TYPE_STRING, (unsigned long) ha->name_string},
			{ ISNS_ATTR_TAG_ISCSI_NODE_TYPE,   ISNS_ATTR_TYPE_ULONG,  DWSWAP(ISCSI_NODE_TYPE_INITIATOR)},
			{ ISNS_ATTR_TAG_ISCSI_ALIAS,       ISNS_ATTR_TYPE_STRING, -1},	    // Friendly machine name?

			{ 0, 0, 0}	// Terminating NULL entry
		};

		ATTRIBUTE_LIST asRegUpdateNodeAttrList[] = {
			// Source attribute
			{ ISNS_ATTR_TAG_ISCSI_NAME,        ISNS_ATTR_TYPE_STRING, (unsigned long) ha->name_string},
			// We updating attributes of a Node so specify the Node as the Key
			{ ISNS_ATTR_TAG_ISCSI_NAME,        ISNS_ATTR_TYPE_STRING, (unsigned long) ha->name_string},
			{ ISNS_ATTR_TAG_DELIMITER,         ISNS_ATTR_TYPE_EMPTY,  0},
			// Operating attributes to update
			{ ISNS_ATTR_TAG_ISCSI_NAME,        ISNS_ATTR_TYPE_STRING, (unsigned long) ha->name_string},
			{ ISNS_ATTR_TAG_ISCSI_ALIAS,       ISNS_ATTR_TYPE_STRING, -1},	    // Friendly machine name?

			{ 0, 0, 0}	// Terminating NULL entry
		};

		ATTRIBUTE_LIST asRegReplaceNodeAttrList[] = {
			// Source attribute
			{ ISNS_ATTR_TAG_ISCSI_NAME,        ISNS_ATTR_TYPE_STRING, (unsigned long) ha->name_string},
			// We updating attributes of a Node so specify the Node as the Key
			{ ISNS_ATTR_TAG_ISCSI_NAME,        ISNS_ATTR_TYPE_STRING, (unsigned long) ha->name_string},
			{ ISNS_ATTR_TAG_DELIMITER,         ISNS_ATTR_TYPE_EMPTY,  0},
			// Operating attributes to update
			{ ISNS_ATTR_TAG_ISCSI_NAME,        ISNS_ATTR_TYPE_STRING, (unsigned long) ha->name_string},
			{ ISNS_ATTR_TAG_ISCSI_NODE_TYPE,   ISNS_ATTR_TYPE_ULONG,  DWSWAP(ISCSI_NODE_TYPE_INITIATOR)},
			{ ISNS_ATTR_TAG_ISCSI_ALIAS,       ISNS_ATTR_TYPE_STRING, -1},	    // Friendly machine name?

			{ 0, 0, 0}	// Terminating NULL entry
		};

		ATTRIBUTE_LIST asRegUpdateEntityAttrList[] = {
			// Source attribute
			{ ISNS_ATTR_TAG_ISCSI_NAME,        ISNS_ATTR_TYPE_STRING,  (unsigned long) ha->name_string},
			// We updating attributes of an Entity so specify the Entity as the Key
			{ ISNS_ATTR_TAG_ENTITY_IDENTIFIER, ISNS_ATTR_TYPE_STRING,  -1},	 // FQDN
			{ ISNS_ATTR_TAG_DELIMITER,         ISNS_ATTR_TYPE_EMPTY,   0},
			// Operating attributes to update
			{ ISNS_ATTR_TAG_ENTITY_IDENTIFIER, ISNS_ATTR_TYPE_STRING,  -1},	 // FQDN
			{ ISNS_ATTR_TAG_MGMT_IP_ADDRESS,   ISNS_ATTR_TYPE_ADDRESS, -1},

			{ 0, 0, 0}	// Terminating NULL entry
		};

		memcpy(ha->asRegUpdateAddObjectsAttrList, asRegUpdateAddObjectsAttrList, sizeof(asRegUpdateAddObjectsAttrList));
		memcpy(ha->asRegUpdateNodeAttrList,       asRegUpdateNodeAttrList,       sizeof(asRegUpdateNodeAttrList));
		memcpy(ha->asRegReplaceNodeAttrList,      asRegReplaceNodeAttrList,      sizeof(asRegReplaceNodeAttrList));
		memcpy(ha->asRegUpdateEntityAttrList,     asRegUpdateEntityAttrList,     sizeof(asRegUpdateEntityAttrList));
	}
#endif

	return(QLA_SUCCESS);
}

uint8_t
qla4xxx_isns_append_attribute(scsi_qla_host_t *ha,
			      uint8_t **buffer,
			      uint8_t *buffer_end,
			      ATTRIBUTE_LIST *attribute)
{

	ISNS_ATTRIBUTE *isns_attr;
	uint32_t data_len;
	uint8_t *local;

	isns_attr =  (ISNS_ATTRIBUTE *) *buffer;

	switch (attribute->type) {
	case ISNS_ATTR_TYPE_EMPTY:
		data_len = 0;
		if ((isns_attr->value + data_len) > buffer_end) {
			return(QLA_ERROR);
		}
		isns_attr->tag = attribute->isns_tag;
		isns_attr->length = data_len;
		break;

	case ISNS_ATTR_TYPE_STRING:
		/*
		 * Length must include NULL terminator.
		 * Note also that all iSNS strings must be UTF-8 encoded.
		 * You should encode your strings for UTF-8 before registering
		 * them with the iSNS server.
		 */
		data_len = strlen ((uint8_t *) (unsigned long) attribute->data) + sizeof(uint8_t);
		if (data_len % 4) {
			data_len += (4 - (data_len % 4)); // Pad to 4 byte boundary.
		}
		if ((isns_attr->value + data_len) > buffer_end) {
			return(QLA_ERROR);
		}
		isns_attr->tag = attribute->isns_tag;
		isns_attr->length = data_len;
		memset(isns_attr->value, 0, data_len);
		strcpy (&isns_attr->value[0], (uint8_t *) (unsigned long) attribute->data);
		break;

	case ISNS_ATTR_TYPE_ULONG:
		data_len = sizeof(uint32_t);
		if ((isns_attr->value + data_len) > buffer_end) {
			return(QLA_ERROR);
		}
		isns_attr->tag = attribute->isns_tag;
		isns_attr->length = data_len;
		*(uint32_t *) isns_attr->value = (uint32_t) attribute->data;
		break;

	case ISNS_ATTR_TYPE_ADDRESS:
		local = (uint8_t *) (unsigned long) attribute->data;
		data_len = 16;	     // Size of an IPv6 address
		if ((isns_attr->value + data_len) > buffer_end) {
			return(QLA_ERROR);
		}
		isns_attr->tag = attribute->isns_tag;
		isns_attr->length = data_len;
		// WARNING: This doesn't handle IPv6 addresses.
		memset(isns_attr->value, 0, 16);
		isns_attr->value[12] = local[0];
		isns_attr->value[13] = local[1];
		isns_attr->value[14] = local[2];
		isns_attr->value[15] = local[3];
		break;

	default:
		return(QLA_ERROR);

	}

	*buffer = isns_attr->value + isns_attr->length;

	// Swap Tag and Length values.

	isns_attr->tag = DWSWAP(isns_attr->tag);
	isns_attr->length = DWSWAP(isns_attr->length);

	return(QLA_SUCCESS);
}


uint32_t
qla4xxx_isns_build_iocb_handle(scsi_qla_host_t *ha,
			       uint32_t type,
			       PDU_ENTRY *pdu_entry)
{
	uint32_t handle;

	handle = (IOCB_ISNS_PT_PDU_TYPE(type) |
		  (((uint8_t *)pdu_entry - (uint8_t *)ha->pdu_queue)
		   / sizeof(PDU_ENTRY)));

	QL4PRINT(QLP20, printk("scsi%d: %s: type %x PDU %p = handle %x\n",
			       ha->host_no, __func__,
			       type, pdu_entry, handle));
	return(handle);
}

uint8_t
qla4xxx_isns_get_server_request(scsi_qla_host_t *ha,
				uint32_t pdu_buff_len,
				uint16_t connection_id)
{
	PDU_ENTRY *pdu_entry;

	pdu_entry = qla4xxx_get_pdu(ha, MAX(pdu_buff_len, PAGE_SIZE));
	if (pdu_entry == NULL) {
		QL4PRINT(QLP2, printk("scsi%d: %s: get_pdu failed\n",
				      ha->host_no, __func__));
		return(QLA_ERROR);
	}

	pdu_entry->SendBuffLen = 0;
	pdu_entry->RecvBuffLen = pdu_entry->BuffLen;

	if (qla4xxx_send_passthru0_iocb(ha, ISNS_DEVICE_INDEX, connection_id,
					pdu_entry->Buff, pdu_entry->SendBuffLen,
					pdu_entry->RecvBuffLen,
					PT_FLAG_ISNS_PDU | PT_FLAG_WAIT_4_RESPONSE,
					qla4xxx_isns_build_iocb_handle(ha, /*ISNS_REQ_RSP_PDU*/ISNS_ASYNCH_REQ_PDU, pdu_entry))
	    != QLA_SUCCESS) {
		QL4PRINT(QLP2, printk("scsi%d: %s: send_passthru_iocb failed\n",
				      ha->host_no, __func__));
		qla4xxx_free_pdu(ha, pdu_entry);
		return(QLA_ERROR);
	}

	return(QLA_SUCCESS);
}

uint8_t
qla4xxx_isns_build_scn_registration_packet(scsi_qla_host_t *ha,
					   uint8_t *buffer,
					   uint32_t buffer_size,
					   uint32_t *packet_size)
{
	/*
	 * Fill in all of the run time requested data in the attribute array
	 * then call iSNSBuildRequestPacket to do the actual work.
	 */

	return(qla4xxx_isns_build_request_packet(ha, buffer, buffer_size,
						 ISNS_FCID_SCNReg,
						 ha->isns_transaction_id,
						 FALSE,
						 ha->isns_scn_reg_attr_list,
						 packet_size));
}


uint8_t
qla4xxx_isns_build_scn_deregistration_packet(scsi_qla_host_t *ha,
					     uint8_t *buffer,
					     uint32_t buffer_size,
					     uint32_t *packet_size)
{
	/*
	 * Fill in all of the run time requested data in the attribute array
	 * then call iSNSBuildRequestPacket to do the actual work.
	 */

	return(qla4xxx_isns_build_request_packet(ha, buffer, buffer_size,
						 ISNS_FCID_SCNDereg,
						 ha->isns_transaction_id,
						 FALSE,
						 ha->isns_scn_dereg_attr_list,
						 packet_size));
}

uint8_t
qla4xxx_isns_build_registration_packet(scsi_qla_host_t *ha,
				       uint8_t *buff,
				       uint32_t buff_size,
				       uint8_t *isns_entity_id,
				       uint8_t *ip_addr,
				       uint32_t port_number,
				       uint32_t scn_port,
				       uint32_t esi_port,
				       uint8_t *local_alias,
				       uint32_t *packet_size)
{
	/*
	 * Fill in all of the run time requested data in the attribute array,
	 * then call build_request_packet to do the actual work.
	 */
	ha->isns_reg_attr_list[1].data = (unsigned long) isns_entity_id;
	ha->isns_reg_attr_list[3].data = (unsigned long) isns_entity_id;
	ha->isns_reg_attr_list[5].data = (unsigned long) ip_addr;
	ha->isns_reg_attr_list[6].data = DWSWAP(port_number);
	ha->isns_reg_attr_list[7].data = DWSWAP(scn_port);
	ha->isns_reg_attr_list[8].data = DWSWAP(esi_port);
	if (local_alias && local_alias[0]) {
		ha->isns_reg_attr_list[11].data = (uint32_t) (unsigned long) local_alias;
	}
	else {
		ha->isns_reg_attr_list[11].data = (uint32_t) "<No alias specified>";
	}

	return(qla4xxx_isns_build_request_packet(ha, buff, buff_size,
						 ISNS_FCID_DevAttrReg,
						 ha->isns_transaction_id,
						 FALSE,
						 ha->isns_reg_attr_list,
						 packet_size));
}

uint8_t
qla4xxx_isns_build_deregistration_packet(scsi_qla_host_t *ha,
					 uint8_t *buff,
					 uint32_t buff_size,
					 uint8_t *isns_entity_id,
					 uint8_t *ip_addr,
					 uint32_t port_number,
					 uint32_t *packet_size)
{
	/*
	 * Fill in all of the run time requested data in the attribute array,
	 * then call build_request_packet to do the actual work.
	 */
	ha->isns_dereg_attr_list[2].data = (uint32_t) (unsigned long) isns_entity_id;
	#if 0
	ha->isns_dereg_attr_list[3].data = (uint32_t) ip_addr;
	ha->isns_dereg_attr_list[4].data = (uint32_t) DWSWAP(WSWAP(port_number));
	#endif

	return(qla4xxx_isns_build_request_packet(ha, buff, buff_size,
						 ISNS_FCID_DevDereg,
						 ha->isns_transaction_id,
						 FALSE,
						 ha->isns_dereg_attr_list,
						 packet_size));
}

uint8_t
qla4xxx_isns_build_request_packet(scsi_qla_host_t *ha,
				  uint8_t *buffer,
				  uint32_t buffer_size,
				  uint16_t function_id,
				  uint16_t tx_id,
				  uint8_t  use_replace_flag,
				  ATTRIBUTE_LIST *attr_list,
				  uint32_t *packet_size)
{
	ISNSP_MESSAGE_HEADER *isns_message;
	uint8_t  *ptr;
	uint8_t  *buffer_end;
	uint8_t  *payload_start;
	uint32_t i;
	uint8_t  success;

	/*
	 * Ensure that the buffer size is at a minimum sufficient to hold the
	 * message header plus at least one attribute.
	 */
	if (buffer_size < (sizeof(*isns_message) + sizeof(*attr_list))) {
		QL4PRINT(QLP12, printk("scsi%d: %s: Insufficient buffer size %d, need %d\n",
				       ha->host_no, __func__, buffer_size,
				       (unsigned int) (sizeof(*isns_message) + sizeof(*attr_list))));

		return(QLA_ERROR);
	}

	isns_message = (ISNSP_MESSAGE_HEADER *) buffer;
	buffer_end = (uint8_t *) ((unsigned long) buffer + buffer_size);

	/* Initialize message header contents */
	isns_message->isnsp_version = WSWAP(ISNSP_VERSION);
	isns_message->function_id   = WSWAP(function_id);
	isns_message->flags         = WSWAP(ISNSP_CLIENT_SENDER |
					    ISNSP_FIRST_PDU |
					    ISNSP_LAST_PDU);
	if (use_replace_flag)
		isns_message->flags |= WSWAP(ISNSP_REPLACE_FLAG);
	isns_message->transaction_id = WSWAP(tx_id);
	isns_message->sequence_id    = 0; // First and only packet in this message

	ptr = payload_start = &isns_message->payload[0];

	/*
	 * Now that most of the message header has been initialized (we'll fill in
	 * the size when we're finished), let's append the desired attributes
	 * to the request packet.
	 */
	success = TRUE;
	for (i = 0; attr_list[i].type && success; i++) {
		success = (qla4xxx_isns_append_attribute (ha, &ptr, buffer_end,
							  &attr_list[i])
			   == QLA_SUCCESS);
	}

	if (!success) {
		QL4PRINT(QLP2, printk("scsi%d: %s: Ran out of buffer space\n",
				      ha->host_no, __func__));
		return(QLA_ERROR);
	}

	/*
	 * We've successfully finished building the request packet.
	 * Set the size field.
	 */
	isns_message->pdu_length = WSWAP((uint32_t) (unsigned long) ptr - (uint32_t) (unsigned long) payload_start);

	*packet_size = (uint32_t) ((uint32_t) (unsigned long) ptr - (uint32_t) (unsigned long) buffer);

	return(QLA_SUCCESS);
}

uint8_t
qla4xxx_isns_build_server_request_response_packet(scsi_qla_host_t *ha,
						  uint8_t * buffer,
						  uint32_t buffer_size,
						  uint16_t function_id,
						  uint32_t error_code,
						  uint16_t transaction_id,
						  uint32_t *packet_size)
{
	ISNSP_MESSAGE_HEADER * isns_message;
	ISNSP_RESPONSE_HEADER * isns_response;
	uint8_t *ptr;
	uint8_t *buffer_end;
	uint8_t *payload_start;

	// Ensure that the buffer size is at a minimum sufficient to hold the
	// message headers.

	if (buffer_size < (sizeof(ISNSP_MESSAGE_HEADER) + sizeof(ISNSP_RESPONSE_HEADER))) {
		QL4PRINT(QLP2, printk("scsi%d: %s: Insufficient buffer size %x\n",
				      ha->host_no, __func__, buffer_size));
		return(QLA_ERROR);
	}

	isns_message = (ISNSP_MESSAGE_HEADER *) buffer;
	isns_response = (ISNSP_RESPONSE_HEADER *) isns_message->payload;
	payload_start = ( uint8_t *) isns_response;
	buffer_end = ( uint8_t *) (buffer + buffer_size);

	// Initialize message header contents.

	isns_message->isnsp_version = WSWAP(ISNSP_VERSION);
	isns_message->function_id = function_id;
	isns_message->flags = WSWAP(ISNSP_CLIENT_SENDER |
				    ISNSP_FIRST_PDU |
				    ISNSP_LAST_PDU);
	isns_message->transaction_id = transaction_id;
	isns_message->sequence_id = 0;	 // First and only packet in this message

	isns_response->error_code = DWSWAP(error_code);

	ptr = isns_response->attributes;

	// We've successfully finished building the request packet.
	// Set the size field.

	//QLASSERT (!((ptr - payload_start) % 4));

	isns_message->pdu_length = WSWAP((uint32_t) (unsigned long) ptr - (uint32_t) (unsigned long) payload_start);

	*packet_size = (uint32_t) (unsigned long) ptr - (uint32_t) (unsigned long) buffer;

	return(QLA_SUCCESS);
}

uint8_t
qla4xxx_isns_build_dev_get_next_packet (scsi_qla_host_t *ha,
					uint8_t * buffer,
					uint32_t buffer_size,
					uint8_t * last_iscsi_name,
					uint32_t *packet_size)
{
	// Fill in all of the run time requested data in the attribute array
	// then call qla4xxx_isns_build_request_packet to do the actual work.

	if (last_iscsi_name && last_iscsi_name[0]) {
		ha->isns_dev_get_next_attr_list[1].type = ISNS_ATTR_TYPE_STRING;
		ha->isns_dev_get_next_attr_list[1].data = (uint32_t) (unsigned long) last_iscsi_name;
	}
	else {
		ha->isns_dev_get_next_attr_list[1].type = ISNS_ATTR_TYPE_EMPTY;
		ha->isns_dev_get_next_attr_list[1].data = 0;
	}

	return(qla4xxx_isns_build_request_packet(ha, buffer, buffer_size,
						 ISNS_FCID_DevGetNext,
						 ha->isns_transaction_id,
						 FALSE,
						 ha->isns_dev_get_next_attr_list,
						 packet_size));
}

uint8_t
qla4xxx_isns_build_dev_attr_qry_packet (scsi_qla_host_t *ha,
					uint8_t *buffer,
					uint32_t buffer_size,
					uint8_t *object_iscsi_name,
					uint32_t *packet_size)
{
	// Fill in all of the run time requested data in the attribute array
	// then call qla4xxx_isns_build_request_packet to do the actual work.

	ha->isns_dev_attr_qry_attr_list[1].data = (uint32_t) (unsigned long) object_iscsi_name;

	return(qla4xxx_isns_build_request_packet(ha, buffer, buffer_size,
						 ISNS_FCID_DevAttrQry,
						 ha->isns_transaction_id, FALSE,
						 ha->isns_dev_attr_qry_attr_list,
						 packet_size));
}

uint8_t
qla4xxx_isns_parse_get_next_response(scsi_qla_host_t *ha,
				     uint8_t *buffer,
				     uint32_t buffer_size,
				     uint32_t *isns_error,
				     uint8_t *last_iscsi_name,
				     uint32_t last_iscsi_name_size,
				     uint8_t *IsTarget)
{
	ISNSP_MESSAGE_HEADER *isns_message;
	ISNSP_RESPONSE_HEADER *isns_response;
	ISNS_ATTRIBUTE *isns_attr;
	uint8_t *buffer_end;

	*IsTarget = FALSE;

	isns_message = (ISNSP_MESSAGE_HEADER *) buffer;
	buffer_end = ( uint8_t *) (( uint8_t *) isns_message->payload +
				   WSWAP(isns_message->pdu_length));

	// Validate pdu_length specified in the iSNS message header.

	if (((uint32_t) (unsigned long) buffer_end - (uint32_t) (unsigned long) buffer) > buffer_size) {
		QL4PRINT(QLP2, printk("scsi%d: %s: Invalid length field in iSNS response from iSNS server\n",
				      ha->host_no, __func__));
		return(QLA_ERROR);
	}

	// It is safe to assume from this point on that the pdu_length value
	// (and thus our idea about the end of the buffer) is valid.

	// Ensure that we have the correct function_id.

	if (isns_message->function_id != WSWAP(ISNS_FCID_DevGetNextRsp)) {
		QL4PRINT(QLP2, printk("scsi%d: %s: Invalid Function ID (0x%04x) "
				      "in iSNS response from iSNS server\n",
				      ha->host_no, __func__,
				      WSWAP(isns_message->function_id)));
		return(QLA_ERROR);
	}

	isns_response = (ISNSP_RESPONSE_HEADER *) isns_message->payload;

	*isns_error = DWSWAP(isns_response->error_code);
	if (*isns_error) {
		QL4PRINT(QLP2, printk("scsi%d: %s: iSNS Error code: %d\n",
				      ha->host_no, __func__, *isns_error));

		if (*isns_error == ISNS_ERR_NO_SUCH_ENTRY) {
			QL4PRINT(QLP2, printk("scsi%d: %s: No more targets.\n",
					      ha->host_no, __func__));
		}
		else {
			QL4PRINT(QLP2, printk("scsi%d: %s: Get Next failed. Error code %x\n",
					      ha->host_no, __func__, *isns_error));
		}
		return(QLA_ERROR);
	}

	isns_attr = (ISNS_ATTRIBUTE *) isns_response->attributes;

	// Save the returned key attribute for the next DevGetNext request.

	if (VALIDATE_ATTR(isns_attr, buffer_end) && isns_attr->tag ==
	    DWSWAP(ISNS_ATTR_TAG_ISCSI_NAME)) {
		strncpy(last_iscsi_name, isns_attr->value, last_iscsi_name_size);
	}
	else {
		QL4PRINT(QLP2, printk("scsi%d: %s: Bad Key attribute in DevGetNextRsp\n",
				      ha->host_no, __func__));
		return(QLA_ERROR);
	}

	// Point to next attribute.

	isns_attr = NEXT_ATTR(isns_attr);

	if (VALIDATE_ATTR(isns_attr, buffer_end) && isns_attr->tag
	    == DWSWAP(ISNS_ATTR_TAG_DELIMITER)) {
		;	// Do nothing.
	}
	else {
		QL4PRINT(QLP2, printk("scsi%d: %s: No delimiter in DevGetNextRsp\n",
				      ha->host_no, __func__));
		return(QLA_ERROR);
	}

	*IsTarget = TRUE;  //FIXME

	// Point to next attribute.

	isns_attr = NEXT_ATTR(isns_attr);

	if (VALIDATE_ATTR(isns_attr, buffer_end) && isns_attr->tag ==
	    DWSWAP(ISNS_ATTR_TAG_ISCSI_NODE_TYPE)) {
		if (DWSWAP(*(uint32_t *) isns_attr->value) & ISCSI_NODE_TYPE_TARGET) {
			*IsTarget = TRUE;
		}
	}
	#if 0
	else {
		QL4PRINT(QLP2, printk("scsi%d: %s: Bad operating attr in DevGetNextRsp\n",
				      ha->host_no, __func__));
		return(QLA_ERROR);
	}
	#endif

	return(QLA_SUCCESS);
}

uint8_t
qla4xxx_isns_parse_query_response (scsi_qla_host_t *ha,
				   uint8_t *buffer,
				   uint32_t buffer_size,
				   uint32_t *isns_error,
				   ISNS_DISCOVERED_TARGET *isns_discovered_target,
				   uint8_t *IsTarget,
				   uint8_t *last_iscsi_name)
{
	ISNSP_MESSAGE_HEADER *isns_message;
	ISNSP_RESPONSE_HEADER *isns_response;
	ISNS_ATTRIBUTE *isns_attr;
	uint8_t *buffer_end;
	uint8_t *tmpptr;
	uint16_t wTmp;
	uint32_t ulTmp;
	uint32_t i;

	isns_message = (ISNSP_MESSAGE_HEADER *) buffer;
	buffer_end = isns_message->payload + WSWAP(isns_message->pdu_length);

	// Validate pdu_length specified in the iSNS message header.

	if (((uint32_t) (unsigned long) buffer_end - (uint32_t) (unsigned long) buffer) > buffer_size) {
		QL4PRINT(QLP2, printk("scsi%d: %s: Invalid length field in iSNS response from iSNS server\n", ha->host_no, __func__));
		return(QLA_ERROR);
	}

	// It is safe to assume from this point on that the pdu_length value
	// (and thus our idea about the end of the buffer) is valid.

	// Ensure that we have the correct function_id.

	if (isns_message->function_id != WSWAP(ISNS_FCID_DevAttrQryRsp)) {
		QL4PRINT(QLP2, printk("scsi%d: %s: Invalid Function ID %04x in iSNS response\n",
				      ha->host_no, __func__,
				      WSWAP(isns_message->function_id)));
		return(QLA_ERROR);
	}

	isns_response = (ISNSP_RESPONSE_HEADER *) isns_message->payload;

	QL4PRINT(QLP20, printk("-----------------------------\n"));
	QL4PRINT(QLP20, printk("scsi%d: %s: DevAttrQry response from iSNS server:\n",
			       ha->host_no, __func__));

	*isns_error = DWSWAP(isns_response->error_code);
	if (isns_response->error_code) {
		QL4PRINT(QLP2, printk("scsi%d: %s: iSNS Query failed.  error_code %x.\n",
				      ha->host_no, __func__, *isns_error));
		return(QLA_ERROR);
	}

	QL4PRINT(QLP20, printk("scsi%d: %s: Attributes:\n", ha->host_no, __func__));

	isns_attr = (ISNS_ATTRIBUTE *) isns_response->attributes;

	// Skip key and delimiter attributes.

	while (VALIDATE_ATTR(isns_attr, buffer_end) && isns_attr->tag !=
	       DWSWAP(ISNS_ATTR_TAG_DELIMITER)) {
		// Point to next attribute.
		if (DWSWAP(isns_attr->tag) == ISNS_ATTR_TAG_ISCSI_NAME) {
			// Note that this string is in UTF-8 format.  In production code,
			// it would be necessary to convert from UTF-8 before using the
			// string.
			QL4PRINT(QLP20, printk("scsi%d: %s: MsgTag iSCSI Name: \"%s\"\n",
					       ha->host_no, __func__, isns_attr->value));
			if (strlen (isns_attr->value) > 256)
				return(QLA_ERROR);
			strcpy (last_iscsi_name, (uint8_t *) isns_attr->value);
		}
		isns_attr = NEXT_ATTR(isns_attr);
	}

	if (!VALIDATE_ATTR(isns_attr, buffer_end) || isns_attr->tag !=
	    DWSWAP(ISNS_ATTR_TAG_DELIMITER)) {
		// There was no delimiter attribute in the response.
		return(QLA_ERROR);
	}

	// Skip delimiter attribute.
	isns_attr = NEXT_ATTR(isns_attr);

	while (VALIDATE_ATTR(isns_attr, buffer_end)) {
		// We only need to parse for the operating attributes that we
		// requested in the DevAttrQuery.

		switch (DWSWAP(isns_attr->tag)) {
		case ISNS_ATTR_TAG_ENTITY_PROTOCOL:
			if (DWSWAP(*(uint32_t *) isns_attr->value) != ENTITY_PROTOCOL_ISCSI) {
				QL4PRINT(QLP2, printk("scsi%d: %s: Entity does not support iSCSI protocol\n", ha->host_no, __func__));
			}
			break;

		case ISNS_ATTR_TAG_ISCSI_NODE_TYPE:
			switch (DWSWAP(*(uint32_t *) isns_attr->value)) {
			case ISCSI_NODE_TYPE_TARGET:
				QL4PRINT(QLP20, printk("scsi%d: %s: iSCSI node type Target\n", ha->host_no, __func__));
				*IsTarget = TRUE;
				break;
			case ISCSI_NODE_TYPE_INITIATOR:
				QL4PRINT(QLP20, printk("scsi%d: %s: iSCSI node type Initiator\n", ha->host_no, __func__));
				*IsTarget = FALSE;
				break;
			case ISCSI_NODE_TYPE_CONTROL:
				QL4PRINT(QLP20, printk("scsi%d: %s: iSCSI node type Control\n", ha->host_no, __func__));
				*IsTarget = FALSE;
				break;
			default:
				QL4PRINT(QLP20, printk("scsi%d: %s: iSCSI node type unknown\n", ha->host_no, __func__));
				*IsTarget = FALSE;
				break;
			}
			break;

		case ISNS_ATTR_TAG_MGMT_IP_ADDRESS:
			// WARNING: This doesn't handle IPv6 addresses.
			tmpptr = isns_attr->value;
			for (i = 0; i < 8; i++) {
				if (tmpptr[i])
					return(QLA_ERROR);
			}

			for (i = 8; i < 12; i++) {
				if (tmpptr[i] != 0 && tmpptr[i] != 0xFF)
					return(QLA_ERROR);
			}

			QL4PRINT(QLP20, printk("scsi%d: %s: Management IP address: %u.%u.%u.%u\n",
					       ha->host_no, __func__, tmpptr[12],
					       tmpptr[13], tmpptr[14], tmpptr[15]));
			break;

		case ISNS_ATTR_TAG_PORTAL_IP_ADDRESS:
			// WARNING: This doesn't handle IPv6 addresses.
			tmpptr = isns_attr->value;
			for (i = 0; i < 8; i++) {
				if (tmpptr[i])
					return(QLA_ERROR);
			}

			for (i = 8; i < 12; i++) {
				if (tmpptr[i] != 0 && tmpptr[i] != 0xFF)
					return(QLA_ERROR);
			}

			QL4PRINT(QLP20, printk("scsi%d: %s: Portal IP address: %u.%u.%u.%u\n",
					       ha->host_no, __func__, tmpptr[12],
					       tmpptr[13], tmpptr[14], tmpptr[15]));

			if (isns_discovered_target->NumPortals >= ISNS_MAX_PORTALS)
				break;
			memcpy(isns_discovered_target->Portal[isns_discovered_target->NumPortals].IPAddr,
			       &tmpptr[12], 4);
			break;

		case ISNS_ATTR_TAG_PORTAL_PORT:
			wTmp = (uint16_t) (DWSWAP(*(uint32_t *) isns_attr->value));
			QL4PRINT(QLP20, printk("scsi%d: %s: Portal port: %u\n",
					       ha->host_no, __func__, DWSWAP(*(uint32_t *) isns_attr->value)));
			if (isns_discovered_target->NumPortals >= ISNS_MAX_PORTALS)
				break;
			isns_discovered_target->Portal[isns_discovered_target->NumPortals].PortNumber = wTmp;
			isns_discovered_target->NumPortals++;
			break;

		case ISNS_ATTR_TAG_PORTAL_SYMBOLIC_NAME:
			// Note that this string is in UTF-8 format.  In production code,
			// it would be necessary to convert from UTF-8 before using the
			// string.
			QL4PRINT(QLP20, printk("scsi%d: %s: Portal Symbolic Name: \"%s\"\n",
					       ha->host_no, __func__, isns_attr->value));
#if 0
			if (isns_discovered_target->NumPortals >= ISNS_MAX_PORTALS)
				break;
			qlstrncpy(isns_discovered_target->Portal[isns_discovered_target->NumPortals].SymbolicName,
				  (uint8_t *) isns_attr->value, 32);
			isns_discovered_target->Portal[isns_discovered_target->NumPortals].SymbolicName[31] = 0;
#endif
			break;

		case ISNS_ATTR_TAG_SCN_PORT:
			QL4PRINT(QLP20, printk("scsi%d: %s: SCN port: %u\n",
					       ha->host_no, __func__,
					       DWSWAP(*(uint32_t *) isns_attr->value)));
			break;

		case ISNS_ATTR_TAG_ESI_PORT:
			QL4PRINT(QLP20, printk("scsi%d: %s: ESI port: %u\n",
					       ha->host_no, __func__,
					       DWSWAP(*(uint32_t *) isns_attr->value)));
			break;

		case ISNS_ATTR_TAG_ESI_INTERVAL:
			QL4PRINT(QLP20, printk("scsi%d: %s: ESI Interval: %u\n",
					       ha->host_no, __func__,
					       DWSWAP(*(uint32_t *) isns_attr->value)));
			break;

		case ISNS_ATTR_TAG_REGISTRATION_PERIOD:
			QL4PRINT(QLP20, printk("scsi%d: %s: Entity Registration Period: %u\n",
					       ha->host_no, __func__,
					       DWSWAP(*(uint32_t *) isns_attr->value)));
			break;

		case ISNS_ATTR_TAG_PORTAL_SECURITY_BITMAP:
			ulTmp = DWSWAP(*(uint32_t *) isns_attr->value);

			QL4PRINT(QLP20, printk("scsi%d: %s: Portal Security Bitmap:\n", ha->host_no, __func__));
			if (ulTmp & ISNS_SECURITY_BITMAP_VALID) {
				QL4PRINT(QLP20, printk("scsi%d: %s:\tISNS_SECURITY_BITMAP_VALID\n", ha->host_no, __func__));
			}
			if (ulTmp & ISNS_SECURITY_IKE_IPSEC_ENABLED) {
				QL4PRINT(QLP20, printk("scsi%d: %s:\tISNS_SECURITY_IKE_IPSEC_ENABLED\n", ha->host_no, __func__));
			}
			if (ulTmp & ISNS_SECURITY_MAIN_MODE_ENABLED) {
				QL4PRINT(QLP20, printk("scsi%d: %s:\tISNS_SECURITY_MAIN_MODE_ENABLED\n", ha->host_no, __func__));
			}
			if (ulTmp & ISNS_SECURITY_AGGRESSIVE_MODE_ENABLED) {
				QL4PRINT(QLP20, printk("scsi%d: %s:\tISNS_SECURITY_AGGRESSIVE_MODE_ENABLED\n", ha->host_no, __func__));
			}
			if (ulTmp & ISNS_SECURITY_PFS_ENABLED) {
				QL4PRINT(QLP20, printk("scsi%d: %s:\tISNS_SECURITY_PFS_ENABLED\n", ha->host_no, __func__));
			}
			if (ulTmp & ISNS_SECURITY_TRANSPORT_MODE_PREFERRED) {
				QL4PRINT(QLP20, printk("scsi%d: %s:\tISNS_SECURITY_TRANSPORT_MODE_PREFERRED\n", ha->host_no, __func__));
			}
			if (ulTmp & ISNS_SECURITY_TUNNEL_MODE_PREFERRED) {
				QL4PRINT(QLP20, printk("scsi%d: %s:\tISNS_SECURITY_TUNNEL_MODE_PREFERRED\n", ha->host_no, __func__));
			}
			// isns_discovered_target->SecurityBitmap = ulTmp;
			break;

		case ISNS_ATTR_TAG_ENTITY_IDENTIFIER:
			// Note that this string is in UTF-8 format.  In production code,
			// it would be necessary to convert from UTF-8 before using the
			// string.
			QL4PRINT(QLP20, printk("scsi%d: %s: Entity Identifier: \"%s\"\n",
					       ha->host_no, __func__, isns_attr->value));
			break;

		case ISNS_ATTR_TAG_ISCSI_NAME:
			// Note that this string is in UTF-8 format.  In production code,
			// it would be necessary to convert from UTF-8 before using the
			// string.
			QL4PRINT(QLP20, printk("scsi%d: %s: iSCSI Name: \"%s\"\n",
					       ha->host_no, __func__, isns_attr->value));
			if (strlen (isns_attr->value) > 256)
				return(QLA_ERROR);
			strcpy (isns_discovered_target->NameString, ( uint8_t *) isns_attr->value);
			break;

		case ISNS_ATTR_TAG_ISCSI_ALIAS:
			// Note that this string is in UTF-8 format.  In production code,
			// it would be necessary to convert from UTF-8 before using the
			// string.
			QL4PRINT(QLP20, printk("scsi%d: %s: Alias: \"%s\"\n",
					       ha->host_no, __func__, isns_attr->value));
			if (strlen (isns_attr->value) <= 32)
				strcpy (isns_discovered_target->Alias, ( uint8_t *) isns_attr->value);
			break;

		case ISNS_ATTR_TAG_DD_ID:
			ulTmp = DWSWAP(*(uint32_t *) isns_attr->value);
			QL4PRINT(QLP20, printk("scsi%d: %s: DD ID: %u\n",
					       ha->host_no, __func__,
					       DWSWAP(*(uint32_t *) isns_attr->value)));
			isns_discovered_target->DDID = ulTmp;
			break;

		default:
			//QLASSERT (FALSE);
			break;
		}

		// Point to next attribute.

		isns_attr = NEXT_ATTR(isns_attr);
	}

	return(QLA_SUCCESS);
}

STATIC uint8_t
qla4xxx_isns_process_response(scsi_qla_host_t *ha, PASSTHRU_STATUS_ENTRY *sts_entry)
{
	PDU_ENTRY *pdu_entry = (PDU_ENTRY *) &ha->pdu_queue[IOCB_ISNS_PT_PDU_INDEX(sts_entry->handle)];
	uint32_t pdu_type = IOCB_ISNS_PT_PDU_TYPE(sts_entry->handle);
	uint8_t status = QLA_SUCCESS;

	ENTER("qla4xxx_passthru_status_entry");

	QL4PRINT(QLP20,
		 printk("scsi%d: %s isns_flags 0x%x to=0x%x "
			"IOCS=0x%02x OutResidual/Len=0x%x/0x%x "
			"InResidual/Len=0x%x/0x%x\n",
			ha->host_no, __func__, sts_entry->timeout,
			ha->isns_flags,
			sts_entry->completionStatus,
			sts_entry->outResidual,
			pdu_entry->SendBuffLen,
			sts_entry->inResidual,
			pdu_entry->RecvBuffLen));

	if (pdu_entry->RecvBuffLen - sts_entry->inResidual) {
		QL4PRINT(QLP19, printk("PDU (0x%p) <-\n", pdu_entry->Buff));
		qla4xxx_dump_bytes(QLP19, pdu_entry->Buff, (pdu_entry->RecvBuffLen - sts_entry->inResidual));
	}


	if (sts_entry->completionStatus != PASSTHRU_STATUS_COMPLETE) {

		qla4xxx_free_pdu(ha, pdu_entry);
		set_bit(DPC_ISNS_RESTART, &ha->dpc_flags);
		set_bit(AF_SCHEDULE_DPC, &ha->flags);
		goto exit_pt_sts;
	}

	switch (pdu_type) {
	case ISNS_ASYNCH_RSP_PDU:
		qla4xxx_free_pdu(ha, pdu_entry);
		break;

	case ISNS_ASYNCH_REQ_PDU:
		pdu_entry->RecvBuffLen -= sts_entry->inResidual;

		QL4PRINT(QLP19, printk("scsi%d: %s ISNS_ASYNCH_REQ_PDU  PDU Buff=%p, PDU RecvLen=0x%X\n",
				       ha->host_no, __func__, pdu_entry->Buff, pdu_entry->RecvBuffLen));

		if (qla4xxx_isns_reassemble_pdu(ha, pdu_entry->Buff,
						&pdu_entry->RecvBuffLen)
		    != QLA_SUCCESS) {
			QL4PRINT(QLP2,
				 printk("scsi%d: %s ISNS_ASYNCH_REQ_PDU "
					"reassemble_pdu failed!\n",
					ha->host_no, __func__));
			goto exit_pt_sts;
		}

		if (qla4xxx_isns_parse_and_dispatch_server_request(ha,
								   pdu_entry->Buff,
								   pdu_entry->RecvBuffLen,
								   sts_entry->connectionID)
		    != QLA_SUCCESS) {
			QL4PRINT(QLP2,
				 printk("scsi%d: %s ISNS_ASYNCH_REQ_PDU "
					"parse_and_dispatch_server_request failed!\n",
					ha->host_no, __func__));
		}
		qla4xxx_free_pdu(ha, pdu_entry);
		break;

	case ISNS_REQ_RSP_PDU:
		pdu_entry->RecvBuffLen -= sts_entry->inResidual;

		QL4PRINT(QLP19, printk("scsi%d: %s ISNS_REQ_RSP_PDU  PDU Buff=%p, PDU RecvLen=0x%X\n",
				       ha->host_no, __func__, pdu_entry->Buff, pdu_entry->RecvBuffLen));


		if (qla4xxx_isns_reassemble_pdu(ha, pdu_entry->Buff,
						&pdu_entry->RecvBuffLen)
		    != QLA_SUCCESS) {
			QL4PRINT(QLP2,
				 printk("scsi%d: %s ISNS_REQ_RSP_PDU "
					"reassemble_pdu failed!\n",
					ha->host_no, __func__));
			goto exit_pt_sts;
		}

		if (qla4xxx_isns_parse_and_dispatch_server_response(ha,
								    pdu_entry->Buff,
								    pdu_entry->RecvBuffLen)
		    != QLA_SUCCESS) {
			QL4PRINT(QLP2,
				 printk("scsi%d: %s ISNS_REQ_RSP_PDU "
					"parse_and_dispatch_server_response failed!\n",
					ha->host_no, __func__));
		}
		qla4xxx_free_pdu(ha, pdu_entry);
		break;
	default:
		QL4PRINT(QLP2,
			 printk("scsi%d: %s iSNS handle 0x%x invalid\n",
				ha->host_no, __func__, sts_entry->handle));
		status = QLA_ERROR;
		break;
	}

	exit_pt_sts:
	LEAVE("qla4xxx_passthru_status_entry");
	return(status);
}

uint8_t
qla4xxx_isns_reassemble_pdu(scsi_qla_host_t *ha, uint8_t *buffer, uint32_t *buffer_size)
{
	uint16_t copy_size = 0;
	uint32_t new_pdu_length = 0;
	uint32_t bytes_remaining;
	uint32_t pdu_size;
	uint8_t *dest_ptr = NULL;
	uint8_t *src_ptr = NULL;
	ISNSP_MESSAGE_HEADER *isns_message;
	uint32_t i;

	// We have read all the PDU's for this message.  Now reassemble them
	// into a single PDU.
	if (buffer == NULL || buffer_size == 0) {
		return(QLA_ERROR);
	}

	if (*buffer_size == 0) {
		QL4PRINT(QLP2,
			 printk(KERN_WARNING "scsi%d: %s: Length 0.  "
				"Nothing to reassemble\n",
				ha->host_no, __func__));
		return(QLA_ERROR);
	}

	new_pdu_length = 0;
	bytes_remaining = *buffer_size;
	isns_message = (ISNSP_MESSAGE_HEADER *) buffer;


	// First, calculate the size of the payload for the collapsed PDU
	do {
		if (bytes_remaining < sizeof(ISNSP_MESSAGE_HEADER)) {
			QL4PRINT(QLP2,
				 printk(KERN_WARNING "scsi%d: %s: Length 0.  "
					"bytes_remaining < "
					"sizeof(ISNSP_MESSAGE_HEADER).  "
					"BytesRemaining %x, discard PDU\n",
					ha->host_no, __func__,
					bytes_remaining));
			*buffer_size = 0;
			return(QLA_ERROR);
		}
		else if (isns_message->isnsp_version != WSWAP(ISNSP_VERSION)) {
			QL4PRINT(QLP2,
				 printk(KERN_WARNING "scsi%d: %s: Bad Version "
					"number in iSNS Message Header "
					"(%04x, expecting %04x), discard PDU\n",
					ha->host_no, __func__,
					isns_message->isnsp_version,
					WSWAP(ISNSP_VERSION)));
			*buffer_size = 0;
			return(QLA_ERROR);
		}
		else if (bytes_remaining < sizeof(ISNSP_MESSAGE_HEADER) + WSWAP(isns_message->pdu_length)) {
			QL4PRINT(QLP2,
				 printk(KERN_WARNING "scsi%d: %s: Short PDU "
					"in sequence. BytesRemaining %x, "
					"discard PDU\n",
					ha->host_no, __func__,
					bytes_remaining));
			*buffer_size = 0;
			return(QLA_ERROR);
		}

		if (bytes_remaining == sizeof(ISNSP_MESSAGE_HEADER) + WSWAP(isns_message->pdu_length)) {
			if (!(WSWAP(isns_message->flags) & ISNSP_LAST_PDU)) {
				QL4PRINT(QLP2,
					 printk(KERN_WARNING "scsi%d: %s: "
						"Last PDU Flag not set at end "
						"of sequence. discard PDU\n",
						ha->host_no, __func__));
				*buffer_size = 0;
				return(QLA_ERROR);
			}
		}

		new_pdu_length += WSWAP(isns_message->pdu_length);
		pdu_size = sizeof(ISNSP_MESSAGE_HEADER) + WSWAP(isns_message->pdu_length);
		isns_message = (ISNSP_MESSAGE_HEADER *) ((uint8_t *) isns_message + pdu_size);
		bytes_remaining = bytes_remaining > pdu_size ? bytes_remaining - pdu_size : 0;
	}
	while (bytes_remaining);

	dest_ptr = buffer;
	bytes_remaining = *buffer_size;
	isns_message = (ISNSP_MESSAGE_HEADER *) buffer;
	i = 0;
	QL4PRINT(QLP19, printk("scsi%d: %s: PDU%d=%p payloadLength=%04x\n",
			       ha->host_no, __func__, i, dest_ptr,
			       WSWAP(isns_message->pdu_length)));

	while (bytes_remaining) {
		// If this is the first PDU perform no copy,
		// otherwise copy just the payload.

		if (dest_ptr != buffer) {
			i++;
			copy_size = WSWAP(isns_message->pdu_length);
			src_ptr = (uint8_t *) isns_message->payload;
			QL4PRINT(QLP19,
				 printk("scsi%d: %s: PDU%d %p <= %p (%04x)\n",
					ha->host_no, __func__, i, dest_ptr,
					src_ptr, copy_size));
			memcpy(dest_ptr, src_ptr, copy_size);
			dest_ptr += copy_size;
		}
		pdu_size = sizeof(ISNSP_MESSAGE_HEADER) + WSWAP(isns_message->pdu_length);
		isns_message = (ISNSP_MESSAGE_HEADER *) ((uint8_t *) isns_message + pdu_size);
		bytes_remaining = bytes_remaining > pdu_size ? bytes_remaining - pdu_size : 0;
	}

	isns_message = (ISNSP_MESSAGE_HEADER *) buffer;

	// Update pdu_length field in reassembled PDU to reflect actual
	// combined PDU payload length.
	isns_message->pdu_length = WSWAP(new_pdu_length);

	// Also set LAST_PDU flag in reassembled PDU
	isns_message->flags |= WSWAP(ISNSP_LAST_PDU);

	// Return number of bytes in buffer to caller.
	*buffer_size = new_pdu_length + sizeof(ISNSP_MESSAGE_HEADER);
	return(QLA_SUCCESS);
}

uint8_t
qla4xxx_isns_scn (scsi_qla_host_t *ha,
		  uint8_t * req_buffer,
		  uint32_t req_buffer_size,
		  uint16_t ConnectionId)
{
	ISNSP_MESSAGE_HEADER * isns_req_message;
	ISNSP_MESSAGE_HEADER * isns_rsp_message;
	ISNSP_RESPONSE_HEADER * isns_response;
	PDU_ENTRY * pdu_entry;
	ISNS_ATTRIBUTE * attr;
	uint8_t * req_buffer_end;
	uint8_t * rsp_buffer_end;
	uint8_t * payload_start;
	uint8_t * ptr;
	uint32_t packet_size;
	uint32_t copy_size;

	isns_req_message = (ISNSP_MESSAGE_HEADER *) req_buffer;

	if ((pdu_entry = qla4xxx_get_pdu (ha, PAGE_SIZE)) == NULL) {
		QL4PRINT(QLP2, printk("scsi%d: %s: qla4xxx_get_pdu failed\n",
				      ha->host_no, __func__));
		return(QLA_ERROR);
	}

	// First, setup the response packet.
	if (qla4xxx_isns_build_server_request_response_packet(ha,
							      pdu_entry->Buff,
							      pdu_entry->BuffLen,
							      isns_req_message->function_id | WSWAP(0x8000),
							      ISNS_ERR_SUCCESS,
							      isns_req_message->transaction_id,
							      &packet_size)
	    != QLA_SUCCESS) {
		QL4PRINT(QLP2,
			 printk("scsi%d: %s: qla4xxx_isns_build_server_request_response_packet failed\n",
				ha->host_no, __func__));
		qla4xxx_free_pdu (ha, pdu_entry);
		return(QLA_ERROR);
	}
	isns_rsp_message = (ISNSP_MESSAGE_HEADER *) pdu_entry->Buff;
	isns_response = (ISNSP_RESPONSE_HEADER *) isns_rsp_message->payload;
	payload_start = (uint8_t *) isns_response;
	rsp_buffer_end = (uint8_t *) (pdu_entry->Buff + pdu_entry->BuffLen);

	ptr = isns_response->attributes;

	req_buffer_end = (uint8_t *) ((uint8_t *) isns_req_message->payload +
				      WSWAP(isns_req_message->pdu_length));

	// Point to the source attribute in the request.  We need to return only
	// this attribute in the SCN Response.
	attr = (ISNS_ATTRIBUTE *) isns_req_message->payload;
	if (!VALIDATE_ATTR(attr, req_buffer_end)) {
		isns_response->error_code = DWSWAP(ISNS_ERR_MSG_FORMAT);
		QL4PRINT(QLP2, printk("scsi%d: %s: Malformed packet\n",
				      ha->host_no, __func__));
	}

	// Validate that this is an iSCSI Name attribute.
	if (attr->tag != DWSWAP(ISNS_ATTR_TAG_ISCSI_NAME)) {
		QL4PRINT(QLP2, printk("scsi%d: %s: Did not find iSCSN Name attribute\n",
				      ha->host_no, __func__));
	}

	// Copy source attribute to return buffer.
	copy_size = sizeof(ISNS_ATTRIBUTE) + DWSWAP(attr->length);

	if (ptr + copy_size < rsp_buffer_end) {
		// Attribute will fit in the response buffer.  Go ahead
		// and copy it.
		memcpy(ptr, attr, copy_size);
		ptr += copy_size;
	}
	else {
		QL4PRINT(QLP2, printk("scsi%d: %s: Insufficient buffer size\n",
				      ha->host_no, __func__));
	}

	// We've successfully finished building the response packet.
	// Set the size field.

	//QLASSERT (!((ptr - payload_start) % 4));

	isns_rsp_message->pdu_length = WSWAP((uint32_t) (unsigned long) ptr - (uint32_t) (unsigned long) payload_start);

	packet_size = (uint32_t) (unsigned long) ptr - (uint32_t) (unsigned long) pdu_entry->Buff;

	pdu_entry->SendBuffLen = packet_size;
	pdu_entry->RecvBuffLen = 0;

	QL4PRINT(QLP10, printk("scsi%d: %s: Dump Send Buff %p %x\n",
			       ha->host_no, __func__, pdu_entry->Buff,
			       pdu_entry->SendBuffLen));
	qla4xxx_dump_bytes (QLP10, pdu_entry->Buff, pdu_entry->SendBuffLen);
	QL4PRINT(QLP20, printk("---------------------------\n"));
	QL4PRINT(QLP20, printk("scsi%d: %s:                            sending  %d SCNRsp\n",
			       ha->host_no, __func__,
			       WSWAP(isns_rsp_message->transaction_id)));

	if (qla4xxx_send_passthru0_iocb (ha, ISNS_DEVICE_INDEX, ConnectionId,
					 pdu_entry->Buff,
					 pdu_entry->SendBuffLen,
					 pdu_entry->RecvBuffLen,
					 PT_FLAG_ISNS_PDU,
					 qla4xxx_isns_build_iocb_handle(ha, ISNS_ASYNCH_RSP_PDU, pdu_entry))
	    != QLA_SUCCESS) {
		QL4PRINT(QLP2, printk("scsi%d: %s:  qla4xxx_send_passthru0_iocb failed\n",
				      ha->host_no, __func__));
		qla4xxx_free_pdu (ha, pdu_entry);
		return(QLA_ERROR);
	}

	if (test_bit(ISNS_FLAG_SCN_IN_PROGRESS, &ha->isns_flags)) {
		set_bit(ISNS_FLAG_SCN_RESTART, &ha->isns_flags);
	}
	else {
		set_bit(ISNS_FLAG_SCN_IN_PROGRESS, &ha->isns_flags);
		clear_bit(ISNS_FLAG_SCN_RESTART, &ha->isns_flags);
		ha->isns_num_discovered_targets = 0;
		if (qla4xxx_isns_dev_get_next (ha, NULL) != QLA_SUCCESS) {
			QL4PRINT(QLP2, printk("scsi%d: %s: qla4xxx_isns_dev_get_next failed\n",
					      ha->host_no, __func__));
			ISNS_CLEAR_FLAGS(ha);
		}
	}

	return(QLA_SUCCESS);
}

uint8_t
qla4xxx_isns_esi (scsi_qla_host_t *ha,
		  uint8_t *req_buffer,
		  uint32_t req_buffer_size,
		  uint16_t ConnectionId)
{
	ISNSP_MESSAGE_HEADER *isns_req_message;
	ISNSP_MESSAGE_HEADER *isns_rsp_message;
	ISNSP_RESPONSE_HEADER *isns_response;
	PDU_ENTRY * pdu_entry;
	ISNS_ATTRIBUTE *attr;
	uint8_t * req_buffer_end;
	uint8_t * rsp_buffer_end;
	uint8_t * payload_start;
	uint8_t * ptr;
	uint32_t packet_size;
	uint32_t copy_size;

	isns_req_message = (ISNSP_MESSAGE_HEADER *) req_buffer;

	if ((pdu_entry = qla4xxx_get_pdu (ha, req_buffer_size + sizeof(uint32_t))) == NULL) {
		QL4PRINT(QLP2, printk("scsi%d: %s:  qla4xxx_get_pdu failed\n",
				      ha->host_no, __func__));
		return(QLA_ERROR);
	}

	// First, setup the response packet.
	if (qla4xxx_isns_build_server_request_response_packet(ha,
							      pdu_entry->Buff,
							      pdu_entry->BuffLen,
							      isns_req_message->function_id | WSWAP(0x8000),
							      ISNS_ERR_SUCCESS,
							      isns_req_message->transaction_id,
							      &packet_size)
	    != QLA_SUCCESS) {
		QL4PRINT(QLP2, printk("scsi%d: %s:  qla4xxx_isns_build_server_request_response_packet failed\n",
				      ha->host_no, __func__));
		qla4xxx_free_pdu (ha, pdu_entry);
		return(QLA_ERROR);
	}
	isns_rsp_message = (ISNSP_MESSAGE_HEADER *) pdu_entry->Buff;
	isns_response = (ISNSP_RESPONSE_HEADER *) isns_rsp_message->payload;
	payload_start = ( uint8_t *) isns_response;
	rsp_buffer_end = ( uint8_t *) (pdu_entry->Buff + pdu_entry->BuffLen);

	ptr = isns_response->attributes;

	req_buffer_end = ( uint8_t *) (( uint8_t *) isns_req_message->payload +
				       WSWAP(isns_req_message->pdu_length));

	// Point to the source attribute in the request.  We need to return
	// all attributes in the ESI Response.
	attr = (ISNS_ATTRIBUTE *) isns_req_message->payload;

	// Copy source attributes to return buffer.
	copy_size = req_buffer_end - ( uint8_t *) attr;

	if (ptr + copy_size < rsp_buffer_end) {
		// Attributes will fit in the response buffer.  Go ahead
		// and copy them.
		memcpy(ptr, attr, copy_size);
		ptr += copy_size;
	}
	else {
		QL4PRINT(QLP2,
			 printk("scsi%d: %s: Insufficient buffer size\n",
				ha->host_no, __func__));
	}

	// We've successfully finished building the response packet.
	// Set the size field.

	//QLASSERT (!((ptr - payload_start) % 4));

	isns_rsp_message->pdu_length = WSWAP((uint32_t) (unsigned long) ptr - (uint32_t) (unsigned long) payload_start);

	packet_size = (uint32_t) (unsigned long) ptr - (uint32_t) (unsigned long) pdu_entry->Buff;

	pdu_entry->SendBuffLen = packet_size;
	pdu_entry->RecvBuffLen = 0;

	QL4PRINT(QLP10, printk("scsi%d: %s: Dump Send Buff %p %x\n",
			       ha->host_no, __func__, pdu_entry->Buff,
			       pdu_entry->SendBuffLen));
	qla4xxx_dump_bytes (QLP10, pdu_entry->Buff, pdu_entry->SendBuffLen);
	QL4PRINT(QLP20, printk("---------------------------\n"));
	QL4PRINT(QLP20,
		 printk("scsi%d: %s:                            sending  %d ESIRsp\n",
			ha->host_no, __func__,
			WSWAP(isns_rsp_message->transaction_id)));

	if (qla4xxx_send_passthru0_iocb(ha, ISNS_DEVICE_INDEX,
					ConnectionId,
					pdu_entry->Buff,
					pdu_entry->SendBuffLen,
					pdu_entry->RecvBuffLen,
					PT_FLAG_ISNS_PDU,
					qla4xxx_isns_build_iocb_handle (ha, ISNS_ASYNCH_RSP_PDU, pdu_entry))
	    != QLA_SUCCESS) {
		QL4PRINT(QLP2, printk("scsi%d: %s:  qla4xxx_send_passthru0_iocb failed\n",
				      ha->host_no, __func__));
		qla4xxx_free_pdu (ha, pdu_entry);
		return(QLA_ERROR);
	}

	return(QLA_SUCCESS);
}


uint8_t
qla4xxx_isns_server_request_error(scsi_qla_host_t *ha,
				  uint8_t *buffer,
				  uint32_t buffer_size,
				  uint16_t connection_id,
				  uint32_t error_code)
{
	PDU_ENTRY *pdu_entry;
	ISNSP_MESSAGE_HEADER *isns_message;
	uint16_t function_id;
	uint32_t packet_size;

	isns_message = (ISNSP_MESSAGE_HEADER *) buffer;
	function_id = WSWAP(isns_message->function_id);

	// Return "Message Format Error"
	if ((pdu_entry = qla4xxx_get_pdu(ha, sizeof(ISNSP_MESSAGE_HEADER) +
					 sizeof(uint32_t))) == NULL) {
		QL4PRINT(QLP2, printk("scsi%d: %s:  qla4xxx_get_pdu failed\n",
				      ha->host_no, __func__));
		return(QLA_ERROR);
	}

	if (qla4xxx_isns_build_server_request_response_packet(ha, pdu_entry->Buff, pdu_entry->BuffLen,
							      isns_message->function_id | WSWAP(0x8000),
							      error_code, isns_message->transaction_id,
							      &packet_size)
	    != QLA_SUCCESS) {
		QL4PRINT(QLP2, printk("scsi%d: %s:  qla4xxx_isns_build_server_request_response_packet failed\n",
				      ha->host_no, __func__));
		qla4xxx_free_pdu(ha, pdu_entry);
		return(QLA_ERROR);
	}

	pdu_entry->SendBuffLen = packet_size;
	pdu_entry->RecvBuffLen = 0;
	QL4PRINT(QLP10, printk("scsi%d: %s: Dump Send Buff %p %x\n",
			       ha->host_no, __func__, pdu_entry->Buff,
			       pdu_entry->SendBuffLen));
	qla4xxx_dump_bytes(QLP10, pdu_entry->Buff, pdu_entry->SendBuffLen);
	if (qla4xxx_send_passthru0_iocb(ha, ISNS_DEVICE_INDEX, connection_id,
					pdu_entry->Buff, pdu_entry->SendBuffLen,
					pdu_entry->RecvBuffLen, PT_FLAG_ISNS_PDU,
					qla4xxx_isns_build_iocb_handle(ha, ISNS_ASYNCH_RSP_PDU, pdu_entry))
	    != QLA_SUCCESS) {
		QL4PRINT(QLP2, printk("scsi%d: %s:  qla4xxx_send_passthru0_iocb failed\n, ha->host_no, __func__",
				      ha->host_no, __func__));
		qla4xxx_free_pdu(ha, pdu_entry);
		return(QLA_ERROR);
	}
	return(QLA_SUCCESS);
}


uint8_t
qla4xxx_isns_parse_and_dispatch_server_request(scsi_qla_host_t *ha,
					       uint8_t *buffer,
					       uint32_t buffer_size,
					       uint16_t connection_id)
{
	ISNSP_MESSAGE_HEADER *isns_message;
	uint16_t function_id;

	isns_message = (ISNSP_MESSAGE_HEADER *) buffer;
	function_id = WSWAP(isns_message->function_id);

	// Validate pdu_length specified in the iSNS message header.
	if ((uint32_t) (offsetof (ISNSP_MESSAGE_HEADER, payload) +
			WSWAP(isns_message->pdu_length)) > buffer_size) {
		QL4PRINT(QLP2, printk("scsi%d: %s: Invalid message size %u %u\n",
				      ha->host_no, __func__,
				      (uint32_t) (offsetof(ISNSP_MESSAGE_HEADER, payload) +
						  WSWAP(isns_message->pdu_length)),
				      buffer_size));

		if (function_id <= ISNS_FCID_ESI) {
			return(qla4xxx_isns_server_request_error(ha, buffer, buffer_size, connection_id, ISNS_ERR_MSG_FORMAT));
		}
		return(QLA_ERROR);
	}

	// It is safe to assume from this point on that the pdu_length value
	// (and thus our idea about the end of the buffer) is valid.

	switch (function_id) {
	case ISNS_FCID_SCN:
		QL4PRINT(QLP2, printk("scsi%d: %s:  received %d SCN\n",
				      ha->host_no, __func__,
				      WSWAP(isns_message->transaction_id)));
		return(qla4xxx_isns_scn(ha, buffer, buffer_size, connection_id));
		break;

	case ISNS_FCID_ESI:
		QL4PRINT(QLP2, printk("scsi%d: %s:  received %d ESI\n",
				      ha->host_no, __func__,
				      WSWAP(isns_message->transaction_id)));
		return(qla4xxx_isns_esi(ha, buffer, buffer_size, connection_id));
		break;

	default:
		QL4PRINT(QLP2, printk("scsi%d: %s:  received %d Unknown iSNS ServerRequest %x\n",
				      ha->host_no, __func__,
				      WSWAP(isns_message->transaction_id), function_id));
		if (function_id <= ISNS_FCID_ESI) {
			// Return "Message Not Supported"
			return(qla4xxx_isns_server_request_error (ha,
								  buffer,
								  buffer_size,
								  connection_id,
								  ISNS_ERR_MSG_NOT_SUPPORTED));
		}
		return(QLA_ERROR);
		break;
	}
	return(QLA_SUCCESS);


}

uint8_t
qla4xxx_isns_parse_and_dispatch_server_response(scsi_qla_host_t *ha,
						uint8_t *buffer,
						uint32_t buffer_size)
{
	ISNSP_MESSAGE_HEADER *isns_message;
	ISNSP_RESPONSE_HEADER *isns_response;
	ISNS_ATTRIBUTE *isns_attr;
	uint16_t function_id;
	uint8_t *buffer_end;

	isns_message = (ISNSP_MESSAGE_HEADER *) buffer;
	buffer_end = (uint8_t *) ((uint8_t *) isns_message->payload + WSWAP(isns_message->pdu_length));

	isns_attr = (ISNS_ATTRIBUTE *) isns_message->payload;

	/* Validate pdu_length specified in the iSNS message header. */
	if (((uint32_t *) buffer_end - (uint32_t *) buffer) > buffer_size) {
		QL4PRINT(QLP2,
			 printk("scsi%d: %s: Invalid message size %u %u\n",
				ha->host_no, __func__,
				(unsigned int) ((uint32_t *) buffer_end - (uint32_t *) buffer),
				buffer_size));
		return(QLA_ERROR);
	}

	/*
	 * It is safe to assume from this point on that the pdu_length value
	 * (and thus our idea about the end of the buffer) is valid.
	 */
	if (WSWAP(isns_message->transaction_id) > ha->isns_transaction_id) {
		QL4PRINT(QLP2, printk("scsi%d: %s: Invalid message transaction ID recv %x exp %x\n",
				      ha->host_no, __func__,
				      WSWAP(isns_message->transaction_id),
				      ha->isns_transaction_id));
		qla4xxx_dump_bytes(QLP2, buffer, buffer_size);
		set_bit(DPC_ISNS_RESTART, &ha->dpc_flags);
		set_bit(AF_SCHEDULE_DPC, &ha->flags);
		return(QLA_ERROR);
	}

	function_id = WSWAP(isns_message->function_id);

	isns_response = (ISNSP_RESPONSE_HEADER *) isns_message->payload;

	//QL4PRINT(QLP20, printk("---------------------------\n"));
	//QL4PRINT(QLP20, printk("scsi%d: %s: received function_id %x\n",
	//		       ha->host_no, __func__, function_id));

	switch (function_id) {
	case ISNS_FCID_DevAttrRegRsp:
		QL4PRINT(QLP20, printk("scsi%d: %s: received %d DevAttrRegRsp\n",
				       ha->host_no, __func__,
				       WSWAP(isns_message->transaction_id)));
		return(qla4xxx_isns_dev_attr_reg_rsp(ha, buffer, buffer_size));

	case ISNS_FCID_DevAttrQryRsp:
		QL4PRINT(QLP20, printk("scsi%d: %s: received %d DevAttrQryRsp\n",
				       ha->host_no, __func__,
				       WSWAP(isns_message->transaction_id)));
		return(qla4xxx_isns_dev_attr_qry_rsp(ha, buffer, buffer_size));

	case ISNS_FCID_DevGetNextRsp:
		QL4PRINT(QLP20, printk("scsi%d: %s: received %d DevGetNextRsp\n",
				       ha->host_no, __func__,
				       WSWAP(isns_message->transaction_id)));
		return(qla4xxx_isns_dev_get_next_rsp(ha, buffer, buffer_size));

	case ISNS_FCID_DevDeregRsp:
		QL4PRINT(QLP20, printk("scsi%d: %s: received %d DevDeregRsp\n",
				       ha->host_no, __func__,
				       WSWAP(isns_message->transaction_id)));
		return(qla4xxx_isns_dev_dereg_rsp(ha, buffer, buffer_size));

	case ISNS_FCID_SCNRegRsp:
		QL4PRINT(QLP20, printk("scsi%d: %s: received %d SCNRegRsp\n",
				       ha->host_no, __func__,
				       WSWAP(isns_message->transaction_id)));
		return(qla4xxx_isns_scn_reg_rsp(ha, buffer, buffer_size));

	case ISNS_FCID_SCNDeregRsp:
		QL4PRINT(QLP20, printk("scsi%d: %s: received %d SCNDeregRsp\n",
				       ha->host_no, __func__,
				       WSWAP(isns_message->transaction_id)));
		return(qla4xxx_isns_scn_dereg_rsp(ha, buffer, buffer_size));

	default:
		QL4PRINT(QLP2, printk("scsi%d: %s: Received %d Unknown iSNS function_id %x\n",
				      ha->host_no, __func__,
				      WSWAP(isns_message->transaction_id), function_id));
		break;
	}
	return(QLA_SUCCESS);
}

uint8_t
qla4xxx_isns_dev_attr_reg(scsi_qla_host_t *ha)
{
	PDU_ENTRY *pdu_entry;
	uint32_t  packet_size;

	pdu_entry = qla4xxx_get_pdu(ha, PAGE_SIZE);
	if (pdu_entry == NULL) {
		QL4PRINT(QLP2, printk("scsi%d: %s: get pdu failed\n",
				      ha->host_no, __func__));
		return(QLA_ERROR);
	}

	if (qla4xxx_isns_build_registration_packet(ha, pdu_entry->Buff,
						   pdu_entry->BuffLen,
						   ha->isns_entity_id,
						   ha->ip_address,
						   ha->isns_remote_port_num,
						   ha->isns_scn_port_num,
						   ha->isns_esi_port_num,
						   ha->alias, &packet_size)
	    != QLA_SUCCESS) {
		QL4PRINT(QLP2,
			 printk("scsi%d: %s: "
				"qla4xxx_isns_build_registration_packet failed\n",
				ha->host_no, __func__));
		qla4xxx_free_pdu(ha, pdu_entry);
		return(QLA_ERROR);
	}

	pdu_entry->SendBuffLen = packet_size;
	pdu_entry->RecvBuffLen = pdu_entry->BuffLen;

	QL4PRINT(QLP19, printk("scsi%d: Dump Send Buff 0x%p 0x%x\n",
			       ha->host_no, pdu_entry->Buff, pdu_entry->SendBuffLen));
	qla4xxx_dump_bytes(QLP19, pdu_entry->Buff, pdu_entry->SendBuffLen);
	QL4PRINT(QLP20, printk("---------------------------\n"));
	QL4PRINT(QLP20, printk("scsi%d: %s:                    sending %d DevAttrReg\n",
			       ha->host_no, __func__, ha->isns_transaction_id));

	QL4PRINT(QLP20, printk("scsi%d: %s: Registering iSNS . . .\n", ha->host_no, __func__));

	if (qla4xxx_send_passthru0_iocb(ha, ISNS_DEVICE_INDEX,
					ISNS_DEFAULT_SERVER_CONN_ID,
					pdu_entry->Buff,
					pdu_entry->SendBuffLen,
					pdu_entry->RecvBuffLen,
					PT_FLAG_ISNS_PDU|PT_FLAG_WAIT_4_RESPONSE,
					qla4xxx_isns_build_iocb_handle(ha, ISNS_REQ_RSP_PDU, pdu_entry))
	    != QLA_SUCCESS) {
		QL4PRINT(QLP2,
			 printk("scsi%d: %s: "
				"qla4xxx_send_passthru0_iocb failed\n",
				ha->host_no, __func__));
		qla4xxx_free_pdu(ha, pdu_entry);
		return(QLA_ERROR);
	}

	ha->isns_transaction_id++;
	return(QLA_SUCCESS);
}

uint8_t
qla4xxx_isns_dev_attr_reg_rsp(scsi_qla_host_t *ha,
			      uint8_t *buffer,
			      uint32_t buffer_size)
{
	ISNSP_MESSAGE_HEADER *isns_message;
	ISNSP_RESPONSE_HEADER *isns_response;

	isns_message = (ISNSP_MESSAGE_HEADER *) buffer;
	isns_response = (ISNSP_RESPONSE_HEADER *) isns_message->payload;

	if (isns_response->error_code) {
		QL4PRINT(QLP2, printk("scsi%d: %s: iSNS DevAttrReg failed, "
				      "error code (%x) \"%s\"\n",
				      ha->host_no, __func__,
				      DWSWAP(isns_response->error_code),
				      isns_error_code_msg[DWSWAP(isns_response->error_code)]));
		clear_bit(ISNS_FLAG_ISNS_SRV_REGISTERED, &ha->isns_flags);
		return(QLA_ERROR);
	}

	set_bit(ISNS_FLAG_ISNS_SRV_REGISTERED, &ha->isns_flags);
	if (qla4xxx_isns_scn_reg(ha) != QLA_SUCCESS) {
		QL4PRINT(QLP2, printk("scsi%d: %s: qla4xxx_isns_scn_reg failed\n",
				      ha->host_no, __func__));
		return(QLA_ERROR);
	}
	return(QLA_SUCCESS);
}

uint8_t
qla4xxx_isns_scn_reg(scsi_qla_host_t *ha)
{
	PDU_ENTRY *isns_pdu_entry;
	uint32_t packet_size;

	if ((isns_pdu_entry = qla4xxx_get_pdu (ha, PAGE_SIZE)) == NULL) {
		QL4PRINT(QLP2, printk("scsi%d: %s: qla4xxx_get_pdu failed\n",
				      ha->host_no, __func__));
		return(QLA_ERROR);
	}

	if (qla4xxx_isns_build_scn_registration_packet(ha, isns_pdu_entry->Buff,
						       isns_pdu_entry->BuffLen,
						       &packet_size) != QLA_SUCCESS) {
		QL4PRINT(QLP2, printk("scsi%d: %s: qla4xx_isns_build_scn_registration_packet failed\n",
				      ha->host_no, __func__));
		qla4xxx_free_pdu(ha, isns_pdu_entry);
		return(QLA_ERROR);
	}

	isns_pdu_entry->SendBuffLen = packet_size;
	isns_pdu_entry->RecvBuffLen = isns_pdu_entry->BuffLen;
	QL4PRINT(QLP10, printk("scsi%d: %s: Dump Send Buff %p %x\n",
			       ha->host_no, __func__,
			       isns_pdu_entry->Buff, isns_pdu_entry->SendBuffLen));
	qla4xxx_dump_bytes(QLP10, isns_pdu_entry->Buff, isns_pdu_entry->SendBuffLen);
	QL4PRINT(QLP20, printk("---------------------------\n"));
	QL4PRINT(QLP20, printk("scsi%d :%s:                        sending  %d SCNReg\n",
			       ha->host_no, __func__, ha->isns_transaction_id));

	if (qla4xxx_send_passthru0_iocb(ha, ISNS_DEVICE_INDEX,
					ISNS_DEFAULT_SERVER_CONN_ID,
					isns_pdu_entry->Buff,
					isns_pdu_entry->SendBuffLen,
					isns_pdu_entry->RecvBuffLen,
					PT_FLAG_ISNS_PDU | PT_FLAG_WAIT_4_RESPONSE,
					qla4xxx_isns_build_iocb_handle(ha, ISNS_REQ_RSP_PDU, isns_pdu_entry))
	    != QLA_SUCCESS) {
		QL4PRINT(QLP2, printk("scsi%d: %s: qla4xxx_send_passthru0_iocb failed\n",
				      ha->host_no, __func__));
		qla4xxx_free_pdu(ha, isns_pdu_entry);
		return(QLA_ERROR);
	}
	ha->isns_transaction_id++;
	return(QLA_SUCCESS);
}

uint8_t
qla4xxx_isns_scn_reg_rsp(scsi_qla_host_t *ha,
			 uint8_t *buffer,
			 uint32_t buffer_size)
{
	ISNSP_MESSAGE_HEADER *isns_message;
	ISNSP_RESPONSE_HEADER *isns_response;

	isns_message = (ISNSP_MESSAGE_HEADER *) buffer;
	isns_response = (ISNSP_RESPONSE_HEADER *) isns_message->payload;

	if (isns_response->error_code) {
		QL4PRINT(QLP2,
			 printk("scsi%d: %s: iSNS SCNReg failed, error code %x\n",
				ha->host_no, __func__,
				DWSWAP(isns_response->error_code)));
		clear_bit(ISNS_FLAG_ISNS_SCN_REGISTERED, &ha->isns_flags);
		return(QLA_ERROR);
	}

	set_bit(ISNS_FLAG_ISNS_SCN_REGISTERED, &ha->isns_flags);

	ha->isns_num_discovered_targets = 0;
	if (qla4xxx_isns_dev_get_next(ha, NULL) != QLA_SUCCESS) {
		QL4PRINT(QLP2, printk("scsi%d: %s: qla4xxx_isns_dev_get_next failed\n",
				      ha->host_no, __func__));
	}

	return(QLA_SUCCESS);
}

uint8_t
qla4xxx_isns_dev_attr_qry(scsi_qla_host_t *ha,
			  uint8_t *last_iscsi_name)
{
	PDU_ENTRY *pdu_entry;
	uint32_t packet_size;

	if ((pdu_entry = qla4xxx_get_pdu(ha, PAGE_SIZE)) == NULL) {
		QL4PRINT(QLP2, printk("scsi%d: %s:  qla4xxx_get_pdu failed\n",
				      ha->host_no, __func__));
		return(QLA_ERROR);
	}

	if (qla4xxx_isns_build_dev_attr_qry_packet(ha, pdu_entry->Buff,
						   pdu_entry->BuffLen,
						   last_iscsi_name,
						   &packet_size)
	    != QLA_SUCCESS) {
		QL4PRINT(QLP2,
			 printk("scsi%d: %s:  qla4xxx_isns_build_dev_attr_qry_packet failed\n",
				ha->host_no, __func__));
		qla4xxx_free_pdu (ha, pdu_entry);
		return(QLA_ERROR);
	}

	pdu_entry->SendBuffLen = packet_size;
	pdu_entry->RecvBuffLen = pdu_entry->BuffLen;

	QL4PRINT(QLP10, printk("scsi%d: %s: Dump Send Buff %p %x\n",
			       ha->host_no, __func__, pdu_entry->Buff,
			       pdu_entry->SendBuffLen));
	qla4xxx_dump_bytes (QLP10, pdu_entry->Buff, pdu_entry->SendBuffLen);
	QL4PRINT(QLP20, printk("---------------------------\n"));
	QL4PRINT(QLP20,
		 printk("scsi%d: %s:                     sending  %d DevAttrQry\n",
			ha->host_no, __func__, ha->isns_transaction_id));

	if (qla4xxx_send_passthru0_iocb(ha, ISNS_DEVICE_INDEX,
					ISNS_DEFAULT_SERVER_CONN_ID,
					pdu_entry->Buff,
					pdu_entry->SendBuffLen,
					pdu_entry->RecvBuffLen,
					PT_FLAG_ISNS_PDU | PT_FLAG_WAIT_4_RESPONSE,
					qla4xxx_isns_build_iocb_handle (ha, ISNS_REQ_RSP_PDU, pdu_entry))
	    != QLA_SUCCESS) {
		QL4PRINT(QLP2, printk("scsi%d: %s:  qla4xxx_send_passthru0_iocb failed\n",
				      ha->host_no, __func__));
		qla4xxx_free_pdu (ha, pdu_entry);
		return(QLA_ERROR);
	}
	ha->isns_transaction_id++;
	return(QLA_SUCCESS);
}

uint8_t
qla4xxx_isns_dev_attr_qry_rsp(scsi_qla_host_t *ha,
			      uint8_t *buffer,
			      uint32_t buffer_size)
{
	uint8_t *last_iscsi_name = NULL;
	ISNS_DISCOVERED_TARGET *discovered_target = NULL;
	uint32_t isns_error;
	int i;
	BOOLEAN bIsTarget = TRUE;
	BOOLEAN bFound = FALSE;
	uint8_t status = QLA_SUCCESS;

	if (test_bit(ISNS_FLAG_SCN_RESTART, &ha->isns_flags)) {
		clear_bit(ISNS_FLAG_SCN_RESTART, &ha->isns_flags);
		ha->isns_num_discovered_targets = 0;
		if (qla4xxx_isns_dev_get_next(ha, NULL) != QLA_SUCCESS) {
			QL4PRINT(QLP2, printk("scsi%d: %s: qla4xxx_isns_dev_get_next failed\n",
					      ha->host_no, __func__));
			goto exit_qry_rsp_clear_flags;
		}
		goto exit_qry_rsp;
	}

	last_iscsi_name = kmalloc(256, GFP_ATOMIC);
	discovered_target = kmalloc(sizeof(*discovered_target), GFP_ATOMIC);
	if (!last_iscsi_name || !discovered_target) {
		QL4PRINT(QLP2, printk("scsi%d: %s: failed to allocate memory\n",
				      ha->host_no, __func__));
		status = QLA_ERROR;
		goto exit_qry_rsp;
	}

	memset(last_iscsi_name, 0, 256);
	memset(discovered_target, 0, sizeof(ISNS_DISCOVERED_TARGET));
	if (qla4xxx_isns_parse_query_response(ha, buffer, buffer_size,
					      &isns_error,
					      discovered_target,
					      &bIsTarget,
					      last_iscsi_name)
	    == QLA_SUCCESS) {
		if (bIsTarget &&
		    discovered_target->NameString[0] &&
		    discovered_target->NumPortals) {
			for (i = 0; i < ha->isns_num_discovered_targets; i++) {
				if (!strcmp(discovered_target->NameString,
					    ha->dma_mem_blkv->isns_discovered_target_database[i].NameString)) {
					QL4PRINT(QLP2, printk("scsi%d: %s: found at index %x\n",
							      ha->host_no, __func__, i));
					memcpy(&ha->dma_mem_blkv->isns_discovered_target_database[i],
					       discovered_target,
					       sizeof(ISNS_DISCOVERED_TARGET));
					ha->dma_mem_blkv->isns_discovered_target_database[i] = *discovered_target;
					bFound = TRUE;
					break;
				}
			}
			if (!bFound && i < MAX_ISNS_DISCOVERED_TARGETS) {
				QL4PRINT(QLP20,
					 printk("scsi%d: %s: not already present, "
						"put in index %x\n",
						ha->host_no, __func__, i));
				memcpy(&ha->dma_mem_blkv->isns_discovered_target_database[i],
				       discovered_target,
				       sizeof(ISNS_DISCOVERED_TARGET));
				ha->isns_num_discovered_targets++;
			}
		}
	}

	if (test_bit(ISNS_FLAG_QUERY_SINGLE_OBJECT, &ha->isns_flags)) {
		goto exit_qry_rsp_clear_flags;
	}
	else if (last_iscsi_name[0] == 0) {
		goto exit_qry_rsp_clear_flags;
	}
	else {
		if (qla4xxx_isns_dev_get_next (ha, last_iscsi_name) != QLA_SUCCESS) {
			QL4PRINT(QLP2,
				 printk("scsi%d: %s: "
					"qla4xxx_isns_dev_get_next failed\n",
					ha->host_no, __func__));
			goto exit_qry_rsp_clear_flags;
		}
	}

	goto exit_qry_rsp;

	exit_qry_rsp_clear_flags:
	ISNS_CLEAR_FLAGS(ha);

	exit_qry_rsp:
	if (last_iscsi_name) kfree(last_iscsi_name);
	if (discovered_target) kfree (discovered_target);
	return(status);
}

uint8_t
qla4xxx_isns_dev_get_next(scsi_qla_host_t *ha,
			  uint8_t *last_iscsi_name)
{
	PDU_ENTRY *pdu_entry;
	uint32_t packet_size;

	if ((pdu_entry = qla4xxx_get_pdu(ha, PAGE_SIZE)) == NULL) {
		QL4PRINT(QLP2, printk("scsi%d: %s:  qla4xxx_get_pdu failed\n",
				      ha->host_no, __func__));
		return(QLA_ERROR);
	}

	if (qla4xxx_isns_build_dev_get_next_packet (ha, pdu_entry->Buff,
						    pdu_entry->BuffLen,
						    last_iscsi_name,
						    &packet_size)
	    != QLA_SUCCESS) {
		QL4PRINT(QLP2, printk("scsi%d: %s:  qla4xxx_isns_build_dev_get_next_packet failed\n",
				      ha->host_no, __func__));
		qla4xxx_free_pdu (ha, pdu_entry);
		return(QLA_ERROR);
	}

	pdu_entry->SendBuffLen = packet_size;
	pdu_entry->RecvBuffLen = pdu_entry->BuffLen;

	QL4PRINT(QLP10, printk("scsi%d: %s: Dump Send Buff %p %x\n",
			       ha->host_no, __func__, pdu_entry->Buff,
			       pdu_entry->SendBuffLen));
	qla4xxx_dump_bytes (QLP10, pdu_entry->Buff, pdu_entry->SendBuffLen);
	QL4PRINT(QLP20, printk("---------------------------\n"));
	QL4PRINT(QLP20, printk("scsi%d: %s:                     sending  %d DevGetNext\n",
			       ha->host_no, __func__, ha->isns_transaction_id));

	if (qla4xxx_send_passthru0_iocb(ha, ISNS_DEVICE_INDEX,
					ISNS_DEFAULT_SERVER_CONN_ID,
					pdu_entry->Buff,
					pdu_entry->SendBuffLen,
					pdu_entry->RecvBuffLen,
					PT_FLAG_ISNS_PDU | PT_FLAG_WAIT_4_RESPONSE,
					qla4xxx_isns_build_iocb_handle(ha, ISNS_REQ_RSP_PDU, pdu_entry))
	    != QLA_SUCCESS) {
		QL4PRINT(QLP2, printk("scsi%d: %s:  qla4xxx_send_passthru0_iocb failed\n",
				      ha->host_no, __func__));
		qla4xxx_free_pdu (ha, pdu_entry);
		return(QLA_ERROR);
	}
	ha->isns_transaction_id++;
	return(QLA_SUCCESS);
}

uint8_t
qla4xxx_isns_dev_get_next_rsp(scsi_qla_host_t *ha,
			      uint8_t *buffer,
			      uint32_t buffer_size)
{
	uint32_t isns_error = 0;
	uint8_t bIsTarget;
	static uint8_t last_iscsi_name[256];

	if (test_bit(ISNS_FLAG_SCN_RESTART, &ha->isns_flags)) {
		clear_bit(ISNS_FLAG_SCN_RESTART, &ha->isns_flags);
		ha->isns_num_discovered_targets = 0;
		if (qla4xxx_isns_dev_get_next(ha, NULL) != QLA_SUCCESS) {
			QL4PRINT(QLP2, printk("scsi%d: %s: qla4xxx_isns_dev_get_next failed\n",
					      ha->host_no, __func__));
			goto exit_get_next_rsp;
		}
		return(QLA_SUCCESS);
	}

	if (qla4xxx_isns_parse_get_next_response(ha, buffer, buffer_size,
						 &isns_error, &last_iscsi_name[0],
						 sizeof(last_iscsi_name) - 1,
						 &bIsTarget)
	    != QLA_SUCCESS) {
		if (isns_error != ISNS_ERR_NO_SUCH_ENTRY) {
			QL4PRINT(QLP2, printk("scsi%d: %s: qla4xxx_isns_parse_get_next_response failed\n",
					      ha->host_no, __func__));
		}
		goto exit_get_next_rsp;
	}

    #if 1
	if (bIsTarget) {
		if (qla4xxx_isns_dev_attr_qry(ha, &last_iscsi_name[0]) != QLA_SUCCESS) {
			QL4PRINT(QLP2, printk("scsi%d: %s: qla4xxx_isns_dev_attr_qry failed\n",
					      ha->host_no, __func__));
			goto exit_get_next_rsp;
		}
	}
	else {
		if (qla4xxx_isns_dev_get_next(ha, &last_iscsi_name[0]) != QLA_SUCCESS) {
			QL4PRINT(QLP2, printk("scsi%d: %s: qla4xxx_isns_dev_get_next failed\n",
					      ha->host_no, __func__));
			goto exit_get_next_rsp;
		}
	}
    #else
	if (qla4xxx_isns_dev_attr_qry(ha, &last_iscsi_name[0]) != QLA_SUCCESS) {
		QL4PRINT(QLP2, printk("scsi%d: %s: qla4xxx_isns_dev_attr_qry failed\n",
				      ha->host_no, __func__));
		goto exit_get_next_rsp;
	}
    #endif

	return(QLA_SUCCESS);

	exit_get_next_rsp:
	clear_bit(ISNS_FLAG_SCN_IN_PROGRESS, &ha->isns_flags);
	clear_bit(ISNS_FLAG_SCN_RESTART, &ha->isns_flags);
	return(QLA_SUCCESS);
}

uint8_t
qla4xxx_isns_dev_dereg(scsi_qla_host_t *ha)
{
	PDU_ENTRY *pdu_entry;
	uint32_t packet_size;

	if ((pdu_entry = qla4xxx_get_pdu (ha, PAGE_SIZE)) == NULL) {
		QL4PRINT(QLP2, printk("scsi%d: %s:  qla4xxx_get_pdu failed\n",
				      ha->host_no, __func__));
		return(QLA_ERROR);
	}

	if (qla4xxx_isns_build_deregistration_packet(ha, pdu_entry->Buff,
						     pdu_entry->BuffLen,
						     ha->isns_entity_id,
						     ha->isns_ip_address,
						     ha->isns_server_port_number,
						     &packet_size)
	    != QLA_SUCCESS) {
		QL4PRINT(QLP2,
			 printk("scsi%d: %s:  QLiSNSBuildDeregistrationPacket failed\n",
				ha->host_no, __func__));
		qla4xxx_free_pdu (ha, pdu_entry);
		return(QLA_ERROR);
	}

	pdu_entry->SendBuffLen = packet_size;
	pdu_entry->RecvBuffLen = pdu_entry->BuffLen;

	QL4PRINT(QLP10, printk("scsi%d: %s: Dump Send Buff %p %x\n",
			       ha->host_no, __func__, pdu_entry->Buff,
			       pdu_entry->SendBuffLen));
	qla4xxx_dump_bytes(QLP10, pdu_entry->Buff, pdu_entry->SendBuffLen);
	QL4PRINT(QLP20, printk("---------------------------\n"));
	QL4PRINT(QLP20,
		 printk("scsi%d: %s:                       sending  %d DevDereg\n",
			ha->host_no, __func__, ha->isns_transaction_id));

	if (qla4xxx_send_passthru0_iocb(ha, ISNS_DEVICE_INDEX,
					ISNS_DEFAULT_SERVER_CONN_ID,
					pdu_entry->Buff,
					pdu_entry->SendBuffLen,
					pdu_entry->RecvBuffLen,
					PT_FLAG_ISNS_PDU | PT_FLAG_WAIT_4_RESPONSE,
					qla4xxx_isns_build_iocb_handle(ha, ISNS_REQ_RSP_PDU, pdu_entry))
	    != QLA_SUCCESS) {
		QL4PRINT(QLP2, printk("scsi%d: %s:  qla4xxx_send_passthru0_iocb failed\n",
				      ha->host_no, __func__));
		qla4xxx_free_pdu(ha, pdu_entry);
		return(QLA_ERROR);
	}
	ha->isns_transaction_id++;
	return(QLA_SUCCESS);
}


uint8_t
qla4xxx_isns_dev_dereg_rsp(scsi_qla_host_t *ha,
			   uint8_t *buffer,
			   uint32_t buffer_size)
{
	ISNSP_MESSAGE_HEADER * isns_message;
	ISNSP_RESPONSE_HEADER * isns_response;

	isns_message = (ISNSP_MESSAGE_HEADER *) buffer;
	isns_response = (ISNSP_RESPONSE_HEADER *) isns_message->payload;

	clear_bit(ISNS_FLAG_ISNS_SRV_REGISTERED, &ha->isns_flags);

	if (isns_response->error_code) {
		QL4PRINT(QLP10, printk("scsi%d: %s: iSNS SCNDereg rsp code %x\n",
				       ha->host_no, __func__,
				       DWSWAP(isns_response->error_code)));
	}

	if (test_bit(ISNS_FLAG_REREGISTER, &ha->isns_flags)) {
		clear_bit(ISNS_FLAG_REREGISTER, &ha->isns_flags);

		if (qla4xxx_isns_dev_attr_reg(ha) != QLA_SUCCESS) {
			QL4PRINT(QLP2, printk("scsi%d: %s: qla4xxx_isns_dev_attr_reg failed\n",
					      ha->host_no, __func__));
			return(QLA_ERROR);
		}
	}
	return(QLA_SUCCESS);
}

uint8_t
qla4xxx_isns_scn_dereg(scsi_qla_host_t *ha)
{
	PDU_ENTRY *pdu_entry;
	uint32_t packet_size;

	if ((pdu_entry = qla4xxx_get_pdu(ha, PAGE_SIZE)) == NULL) {
		QL4PRINT(QLP2, printk("scsi%d: %s:  qla4xxx_get_pdu failed\n", ha->host_no, __func__));
		return(QLA_ERROR);
	}

	if (qla4xxx_isns_build_scn_deregistration_packet(ha, pdu_entry->Buff,
							 pdu_entry->BuffLen,
							 &packet_size)
	    != QLA_SUCCESS) {
		QL4PRINT(QLP2, printk("scsi%d: %s:  QLiSNSBuildSCNDeregistrationPacket failed\n", ha->host_no, __func__));
		qla4xxx_free_pdu (ha, pdu_entry);
		return(QLA_ERROR);
	}

	pdu_entry->SendBuffLen = packet_size;
	pdu_entry->RecvBuffLen = pdu_entry->BuffLen;
	QL4PRINT(QLP10, printk("scsi%d: %s: Dump Send Buff %p %x\n",
			       ha->host_no, __func__, pdu_entry->Buff,
			       pdu_entry->SendBuffLen));
	qla4xxx_dump_bytes (QLP10, pdu_entry->Buff, pdu_entry->SendBuffLen);
	QL4PRINT(QLP20, printk("---------------------------\n"));
	QL4PRINT(QLP20, printk("scsi%d: %s:                       sending  %d SCNDereg\n", ha->host_no, __func__, ha->isns_transaction_id));
	if (qla4xxx_send_passthru0_iocb (ha, ISNS_DEVICE_INDEX,
					 ISNS_DEFAULT_SERVER_CONN_ID,
					 pdu_entry->Buff,
					 pdu_entry->SendBuffLen,
					 pdu_entry->RecvBuffLen,
					 PT_FLAG_ISNS_PDU | PT_FLAG_WAIT_4_RESPONSE,
					 qla4xxx_isns_build_iocb_handle (ha, ISNS_REQ_RSP_PDU, pdu_entry))
	    != QLA_SUCCESS) {
		QL4PRINT(QLP2, printk("scsi%d: %s:  qla4xxx_send_passthru0_iocb failed\n", ha->host_no, __func__));
		qla4xxx_free_pdu (ha, pdu_entry);
		return(QLA_ERROR);
	}
	ha->isns_transaction_id++;
	return(QLA_SUCCESS);
}

uint8_t
qla4xxx_isns_scn_dereg_rsp(scsi_qla_host_t *ha,
			   uint8_t *buffer,
			   uint32_t buffer_size)
{
	ISNSP_MESSAGE_HEADER *isns_message;
	ISNSP_RESPONSE_HEADER *isns_response;

	isns_message = (ISNSP_MESSAGE_HEADER *) buffer;
	isns_response = (ISNSP_RESPONSE_HEADER *) isns_message->payload;

	clear_bit(ISNS_FLAG_ISNS_SCN_REGISTERED, &ha->isns_flags);

	if (isns_response->error_code) {
		QL4PRINT(QLP10, printk("scsi%d: %s: iSNS SCNDereg rsp code %x\n",
				       ha->host_no, __func__,
				       DWSWAP(isns_response->error_code)));
	}

	if (test_bit(ISNS_FLAG_REREGISTER, &ha->isns_flags)) {
		if (qla4xxx_isns_dev_dereg(ha) != QLA_SUCCESS) {
			QL4PRINT(QLP2, printk("scsi%d: %s: QLiSNSDevDereg failed\n",
					      ha->host_no, __func__));
			return(QLA_ERROR);
		}
	}
	return(QLA_SUCCESS);
}




