/*****************************************************************************
*                  QLOGIC LINUX SOFTWARE
*
* QLogic  QLA4000 iSCSI driver
* Copyright (C) 2004 Qlogic Corporation
* (www.qlogic.com)
*
* This program is free software; you can redistribute it and/or modify it
* under the terms of the GNU General Public License as published by the
* Free Software Foundation; either version 2, or (at your option) any
* later version.
*
* This program is distributed in the hope that it will be useful, but
* WITHOUT ANY WARRANTY; without even the implied warranty of
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
* General Public License for more details.
*
******************************************************************************
*             Please see release.txt for revision history.                   *
*                                                                            *
******************************************************************************
* Function Table of Contents:
*	  qla4xxx_get_debug_level
*	  qla4xxx_set_debug_level
*	  qla4xxx_dump_bytes
*	  qla4xxx_dump_words
*	  qla4xxx_dump_dwords
*	  qla4xxx_dump_registers
*	  qla4xxx_print_scsi_cmd
*	
****************************************************************************/

#include <linux/config.h>
#include <linux/module.h>
#include <stdarg.h>
#include <asm/io.h>
#include <asm/irq.h>
#include <asm/segment.h>
#include <asm/byteorder.h>
#include <linux/spinlock.h>
#include <linux/version.h>
#include <linux/types.h>
#include <linux/string.h>
#include <linux/errno.h>
#include <linux/kernel.h>
#include <linux/delay.h>
#include <linux/timer.h>
#include <linux/sched.h>
#include <linux/pci.h>
#include <linux/proc_fs.h>
#include <linux/blk.h>
#include <linux/tqueue.h>

#ifdef __VMKERNEL_MODULE__
#include <scsi.h>
#include <hosts.h>
#else
#include <../drivers/scsi/scsi.h>
#include <../drivers/scsi/hosts.h>
#endif

#include "qla4x_fw.h"
#include "qla4xxx.h"
#include "ql4print.h"

extern Scsi_Host_Template driver_template;

//#define QLP1	0x00000002  // Unrecoverable error messages
//#define QLP2	0x00000004  // Unexpected completion path error messages
//#define QLP3	0x00000008  // Function trace messages
//#define QLP4	0x00000010  // IOCTL trace messages
//#define QLP5	0x00000020  // I/O & Request/Response queue trace messages
//#define QLP6	0x00000040  // Watchdog messages (current state)
//#define QLP7	0x00000080  // Initialization
//#define QLP8	0x00000100  // Internal command queue traces
//#define QLP9	0x00000200  // Register accesses
//#define QLP10	0x00000400  // Extra Debug messages (dump buffers)
//#define QLP11	0x00000800  // Mailbox & ISR Details
//#define QLP12	0x00001000  // Enter/Leave routine messages
//#define QLP13 0x00002000  // Display data for Inquiry, TUR, ReqSense, RptLuns
//#define QLP14 0x00004000
//#define QLP15 0x00008000  // Display jiffies for IOCTL calls
//#define QLP16 0x00010000  // Extended proc print statements (srb info)
//#define QLP17 0x00020000  // Display NVRAM Accesses
//#define QLP18 0x00040000  // unused
//#define QLP19	0x00080000  // PDU info
//#define QLP20 0x00100000  // iSNS info
//#define QLP24 0x01000000  // Scatter/Gather info

uint32_t ql_dbg_level = QLP1;

/**************************************************************************
 * qla4xxx_get_debug_level
 *	This routine retrieves the driver's debug print level.
 *
 * Input:
 *	dbg_level - driver's debug print level
 *
 * Output:
 *	None
 *
 * Returns:
 *	QLA_SUCCESS - if DEBUG driver is loaded
 *	QLA_ERROR   - if non-DEBUG driver is loaded (see ql4print.h)
 **************************************************************************/
inline uint8_t
qla4xxx_get_debug_level(uint32_t *dbg_level)
{
	*dbg_level = ql_dbg_level;
	barrier();
	return(QLA_SUCCESS);
}

/**************************************************************************
 * qla4xxx_set_debug_level
 *	This routine sets the driver's debug print level.
 *
 * Input:
 *	dbg_level - driver's debug print level
 *
 * Output:
 *	None
 *
 * Returns:
 *	QLA_SUCCESS - if DEBUG driver is loaded
 *	QLA_ERROR   - if non-DEBUG driver is loaded (see ql4print.h)
 **************************************************************************/
inline uint8_t
qla4xxx_set_debug_level(uint32_t dbg_level)
{
	ql_dbg_level = dbg_level;
	barrier();
	return(QLA_SUCCESS);
}

/****************************************************************************/
/*                      Debug Print Routines                          	    */
/****************************************************************************/

void printchar(char ch)
{
	if (ch>=32)
		printk("%c", ch);
	else
		printk(".");
}


/**************************************************************************
 * qla4xxx_dump_bytes
 *	This routine displays bytes in hex format
 *
 * Input:
 *	dbg_mask - this call's debug print mask
 *	buffer   - data buffer to display
 *	size     - number of bytes to display
 *
 * Output:
 *	None
 *
 * Returns:
 *	None
 **************************************************************************/
void
qla4xxx_dump_bytes(uint32_t dbg_mask, void *buffer, uint32_t size)
{
	uint32_t i;
	uint8_t  *data = (uint8_t *)buffer;

	if ((ql_dbg_level & dbg_mask) != 0) {
		//printk("        0  1  2  3  4  5  6  7 -  8  9  A  B  C  D  E  F\n");
		//printk("---------------------------------------------------------\n");

		for (i = 0; i < size; i++, data++) {
			if (i % 0x10 == 0) {
				printk("%04X:  %02X", i, *data);
			}
			else if (i % 0x10 == 0x08) {
				printk(" - %02X", *data);
			}
			else if (i % 0x10 == 0xF) {
				printk(" %02X:  ", *data);
				printchar(*(data-15));
				printchar(*(data-14));
				printchar(*(data-13));
				printchar(*(data-12));
				printchar(*(data-11));
				printchar(*(data-10));
				printchar(*(data-9));
				printchar(*(data-8));
				printchar(*(data-7));
				printchar(*(data-6));
				printchar(*(data-5));
				printchar(*(data-4));
				printchar(*(data-3));
				printchar(*(data-2));
				printchar(*(data-1));
				printchar(*data);
				printk("\n");
			}
			else {
				printk(" %02X", *data);
			}
		}

		if ((i != 0) && (i % 0x10)) {
			printk("\n");
		}
		printk("\n");
	}
}

/**************************************************************************
 * qla4xxx_dump_words
 *	This routine displays words in hex format
 *
 * Input:
 *	dbg_mask - this call's debug print mask
 *	buffer   - data buffer to display
 *	size     - number of bytes to display
 *
 * Output:
 *	None
 *
 * Returns:
 *	None
 **************************************************************************/
void
qla4xxx_dump_words(uint32_t dbg_mask, void *buffer, uint32_t size)
{
	if ((ql_dbg_level & dbg_mask) != 0)
		__dump_words(buffer, size);
}

/**************************************************************************
 * qla4xxx_dump_dwords
 *	This routine displays double words in hex format
 *
 * Input:
 *	dbg_mask - this call's debug print mask
 *	buffer   - data buffer to display
 *	size     - number of bytes to display
 *
 * Output:
 *	None
 *
 * Returns:
 *	None
 **************************************************************************/
void
qla4xxx_dump_dwords(uint32_t dbg_mask, void *buffer, uint32_t size)
{
	if ((ql_dbg_level & dbg_mask) != 0)
		__dump_dwords(buffer, size);
}

/**************************************************************************
 * qla4xxx_print_scsi_cmd
 *	This routine displays the SCSI command
 *
 * Input:
 *	dbg_mask - this call's debug print mask
 *	cmd      - pointer to Linux kernel command structure
 *
 * Output:
 *	None
 *
 * Returns:
 *	None
 **************************************************************************/
void
qla4xxx_print_scsi_cmd(uint32_t dbg_mask, Scsi_Cmnd *cmd)
{
	if ((ql_dbg_level & dbg_mask) != 0) {
		int   i;

		printk("SCSI Command = 0x%p, Handle=0x%p\n",
		       cmd, CMD_HANDLE(cmd));

		printk("  b=%d, t=%02xh, l=%02xh, cmd_len = %02xh\n",
		       cmd->channel, cmd->target, cmd->lun, cmd->cmd_len);

		printk("  CDB = ");
		for (i = 0; i < cmd->cmd_len; i++)
			printk("%02x ", cmd->cmnd[i]);

		printk("  seg_cnt = %d\n",cmd->use_sg);
		printk("  request buffer = 0x%p, request buffer len = 0x%x\n",
		       cmd->request_buffer,cmd->request_bufflen);

		if (cmd->use_sg) {
			struct scatterlist *sg;
			sg = (struct scatterlist *) cmd->request_buffer;
			printk("  SG buffer: \n");
			qla4xxx_dump_bytes(dbg_mask, (caddr_t)sg,
					   (cmd->use_sg *
					    sizeof(struct scatterlist)));
		}

		printk("  tag = %d, flags = 0x%x, transfersize = 0x%x \n",
		       cmd->tag, cmd->flags,cmd->transfersize );

		printk("  Pid = %d, SP = 0x%p\n", (int)cmd->pid, CMD_SP(cmd));
		printk("  underflow size = 0x%x, direction=0x%x, req.cmd=0x%x\n",
		       cmd->underflow, cmd->sc_data_direction, cmd->request.cmd);

		if (driver_template.use_new_eh_code) {
			printk("  Current time (jiffies) = 0x%lx, "
			       "timeout expires = 0x%lx\n",
			       jiffies, cmd->eh_timeout.expires);
		}
	}
}

/**************************************************************************
 * qla4xxx_print_srb_info
 *	This routine displays the srb structure
 *
 * Input:
 *	dbg_mask - this call's debug print mask
 *	srb      - pointer to srb structure
 *
 * Output:
 *	None
 *
 * Returns:
 *	None
 **************************************************************************/
void
qla4xxx_print_srb_info(uint32_t dbg_mask, srb_t *srb)
{
	if ((ql_dbg_level & dbg_mask) != 0) {
		printk("%s: srb = 0x%p, flags=0x%02x\n",
		       __func__, srb, srb->flags);
		printk("%s: entry_count = 0x%02x, active_array_index=0x%04x\n",
		       __func__, srb->entry_count, srb->active_array_index);
		printk("%s: cmd = 0x%p, saved_dma_handle = 0x%x\n",
		       __func__, srb->cmd, (uint32_t) srb->saved_dma_handle);
		printk("%s: fw_ddb_index = %d, lun = %d\n",
		       __func__, srb->fw_ddb_index, srb->lun);
		printk("%s: os_tov = %d, iocb_tov = %d\n",
		       __func__, srb->os_tov, srb->iocb_tov);
		printk("%s: cc_stat = 0x%x, r_start = 0x%lx, u_start = 0x%lx\n\n",
		       __func__, srb->cc_stat, srb->r_start, srb->u_start);
	}
}

void
__dump_dwords(void *buffer, uint32_t size)
{
	uint32_t *data = (uint32_t *)buffer;
	uint32_t i;

	for (i = 0; i < size; i+=4, data++) {
		if (i % 0x10 == 0) {
			printk("%04X:  %08X", i, *data);
		}
		else if (i % 0x10 == 0x08) {
			printk(" - %08X", *data);
		}
		else if (i % 0x10 == 0x0C) {
			printk(" %08X\n", *data);
		}
		else {
			printk(" %08X", *data);
		}
	}
	if ((i != 0) && (i % 0x10 != 0)) {
		printk("\n");
	}
}

void
__dump_words(void *buffer, uint32_t size)
{
	uint16_t *data = (uint16_t *)buffer;
	uint32_t i;

	for (i = 0; i < size; i+=2, data++) {
		if (i % 0x10 == 0) {
			printk(KERN_INFO "%04X:  %04X", i, *data);
		}
		else if (i % 0x10 == 0x08) {
			printk(KERN_INFO " - %04X", *data);
		}
		else if (i % 0x10 == 0x0E) {
			uint8_t *bdata = (uint8_t *) data;
			printk(KERN_INFO " %04X:  ", *data);
			printchar(*(bdata-13));
			printchar(*(bdata-14));
			printchar(*(bdata-11));
			printchar(*(bdata-12));
			printchar(*(bdata-9));
			printchar(*(bdata-10));
			printchar(*(bdata-7));
			printchar(*(bdata-8));
			printchar(*(bdata-5));
			printchar(*(bdata-6));
			printchar(*(bdata-3));
			printchar(*(bdata-4));
			printchar(*(bdata-1));
			printchar(*(bdata-2));
			printchar(*(bdata+1));
			printchar(*(bdata));
			printk("\n");
		}
		else {
			printk(KERN_INFO " %04X", *data);
		}
	}
	if ((i != 0) && (i % 0x10 != 0)) {
		printk(KERN_INFO "\n");
	}
}

#ifdef QLA4000
static void
__qla4000_dump_registers(scsi_qla_host_t *ha)
{
	uint8_t  i;

	printk(KERN_INFO "0x%02X RequestQueueIn  = 0x%08X\n",
	       (uint8_t) offsetof(ISP_REG_T, requestQueueInPointer),
	       RD_REG_DWORD(&ha->reg->requestQueueInPointer));
	printk(KERN_INFO "0x%02X ResponseQueueIn = 0x%08X\n",
	       (uint8_t) offsetof(ISP_REG_T, responseQueueInPointer),
	       RD_REG_DWORD(&ha->reg->responseQueueInPointer));
	printk(KERN_INFO "0x%02X FastPostHandle  = 0x%08X\n",
	       (uint8_t) offsetof(ISP_REG_T, fastPostHandle),
	       RD_REG_DWORD(&ha->reg->fastPostHandle));
	printk(KERN_INFO "0x%02X MailBoxIn0      = 0x%08X\n",
	       (uint8_t) offsetof(ISP_REG_T, mailboxIn0),
	       RD_REG_DWORD(&ha->reg->mailboxIn0));
	printk(KERN_INFO "0x%02X IntrStatusIn    = 0x%08X\n",
	       (uint8_t) offsetof(ISP_REG_T, interruptStatusIn),
	       RD_REG_DWORD(&ha->reg->interruptStatusIn));
	printk(KERN_INFO "0x%02X IntrMaskIn      = 0x%08X\n",
	       (uint8_t) offsetof(ISP_REG_T, interruptMaskIn),
	       RD_REG_DWORD(&ha->reg->interruptMaskIn));
	printk(KERN_INFO "0x%02X MailBoxOut0     = 0x%08X\n",
	       (uint8_t) offsetof(ISP_REG_T, mailboxOut0),
	       RD_REG_DWORD(&ha->reg->mailboxOut0));
	printk(KERN_INFO "0x%02X IntrStatusOut   = 0x%08X\n",
	       (uint8_t) offsetof(ISP_REG_T, interruptStatusOut),
	       RD_REG_DWORD(&ha->reg->interruptStatusOut));
	printk(KERN_INFO "0x%02X IntrMaskOut     = 0x%08X\n",
	       (uint8_t) offsetof(ISP_REG_T, interruptMaskOut),
	       RD_REG_DWORD(&ha->reg->interruptMaskOut));
	for (i=0; i<MBOX_REG_COUNT/2; i++) {
		printk(KERN_INFO "0x%02X MailBoxIn[%d]    = 0x%08X\n",
		       (uint8_t) offsetof(ISP_REG_T, mailboxIn[i]), i+1,
		       RD_REG_DWORD(&ha->reg->mailboxIn[i]));
	}
	for (i=0; i<MBOX_REG_COUNT/2; i++) {
		printk(KERN_INFO "0x%02X MailBoxOut[%d]   = 0x%08X\n",
		       (uint8_t) offsetof(ISP_REG_T, mailboxOut[i]), i+1,
		       RD_REG_DWORD(&ha->reg->mailboxOut[i]));
	}
	printk(KERN_INFO "0x%02X RequestQueueOut = 0x%08X\n",
	       (uint8_t) offsetof(ISP_REG_T, requestQueueOutPointer),
	       RD_REG_DWORD(&ha->reg->requestQueueOutPointer));
	printk(KERN_INFO "0x%02X ResponseQueueOut= 0x%08X\n",
	       (uint8_t) offsetof(ISP_REG_T, responseQueueOutPointer),
	       RD_REG_DWORD(&ha->reg->responseQueueOutPointer));
}
#endif

#ifdef QLA4010
static void
__qla4010_dump_registers(scsi_qla_host_t *ha)
{
	uint8_t  i;

	printk(KERN_INFO "0x%02X Mailbox[0]    = 0x%08X\n",
	       (uint8_t) offsetof(ISP_REG_T, mailboxIn0),
	       RD_REG_DWORD(&ha->reg->mailboxIn0));

	for (i=0; i<MBOX_REG_COUNT-1; i++) {
		printk(KERN_INFO "0x%02X MailBox[%d]    = 0x%08X\n",
		       (uint8_t) offsetof(ISP_REG_T, mailboxIn[i]), i+1,
		       RD_REG_DWORD(&ha->reg->mailboxIn[i]));
	}
	printk(KERN_INFO "0x%02X ExtFlashBiosAccessAddr       = 0x%08X\n",
	       (uint8_t) offsetof(ISP_REG_T, extFlashBiosAccessAddress),
	       RD_REG_DWORD(&ha->reg->extFlashBiosAccessAddress));

	printk(KERN_INFO "0x%02X ExtFlashBiosAccessData       = 0x%08X\n",
	       (uint8_t) offsetof(ISP_REG_T, extFlashBiosAccessData),
	       RD_REG_DWORD(&ha->reg->extFlashBiosAccessData));

	printk(KERN_INFO "0x%02X ISPControlStatus             = 0x%08X\n",
	       (uint8_t) offsetof(ISP_REG_T, ISPControlStatus),
	       RD_REG_DWORD(&ha->reg->ISPControlStatus));

	printk(KERN_INFO "0x%02X ISP_NVRAM                    = 0x%08X\n",
	       (uint8_t) offsetof(ISP_REG_T, NVRAM),
	       RD_REG_DWORD(&ha->reg->NVRAM));

	printk(KERN_INFO "0x%02X RequestQueueProducerIndex    = 0x%08X\n",
	       (uint8_t) offsetof(ISP_REG_T, requestQueueInPointer),
	       RD_REG_DWORD(&ha->reg->requestQueueInPointer));

	printk(KERN_INFO "0x%02X CompletionQueueConsumerIndex = 0x%08X\n",
	       (uint8_t) offsetof(ISP_REG_T, responseQueueOutPointer),
	       RD_REG_DWORD(&ha->reg->responseQueueOutPointer));

	printk(KERN_INFO "0x%02X ExtHardwareConfiguration     = 0x%08X\n",
	       (uint8_t) offsetof(ISP_REG_T, extHardwareConfiguration),
	       RD_REG_DWORD(&ha->reg->extHardwareConfiguration));

	printk(KERN_INFO "0x%02X PortStatus                   = 0x%08X\n",
	       (uint8_t) offsetof(ISP_REG_T, portStatus),
	       RD_REG_DWORD(&ha->reg->portStatus));

	printk(KERN_INFO "0x%02X RequestQueueOut              = 0x%08X\n",
	       (uint8_t) offsetof(ISP_REG_T, requestQueueOutPointer),
	       RD_REG_DWORD(&ha->reg->requestQueueOutPointer));

	printk(KERN_INFO "0x%02X GeneralPurposeOutput         = 0x%08X\n",
	       (uint8_t) offsetof(ISP_REG_T, generalPurposeOutput),
	       RD_REG_DWORD(&ha->reg->generalPurposeOutput));

	printk(KERN_INFO "0x%02X PortFatalErrorStatus         = 0x%08X\n",
	       (uint8_t) offsetof(ISP_REG_T, portFatalErrorStatus),
	       RD_REG_DWORD(&ha->reg->portFatalErrorStatus));
}
#endif


void
__dump_registers(uint32_t dbg_mask, scsi_qla_host_t *ha)
{
	if ((ql_dbg_level & dbg_mask) == 0)
		return;
#ifdef QLA4000
	__qla4000_dump_registers(ha);
#endif
#ifdef QLA4010
	__qla4010_dump_registers(ha);
#endif
}

/**************************************************************************
 * qla4xxx_dump_registers
 *	This routine displays ISP registers
 *
 * Input:
 *	dbg_mask - this call's debug print mask
 *	ha       - adapter structure pointer
 *
 * Output:
 *	None
 *
 * Returns:
 *	None
 **************************************************************************/
void
qla4xxx_dump_registers(uint32_t dbg_mask, scsi_qla_host_t *ha)
{
	unsigned long flags = 0;

	spin_lock_irqsave(&ha->hardware_lock, flags);
	__dump_registers(dbg_mask, ha);
	spin_unlock_irqrestore(&ha->hardware_lock, flags);
}

void
__dump_mailbox_registers(uint32_t dbg_mask, scsi_qla_host_t *ha)
{
	int i =  0;

	if ((ql_dbg_level & dbg_mask) == 0)
		return;

#ifdef QLA4000
	printk(KERN_INFO "  Mailbox In[%d] = %04x, Mailbox Out[%d] = 08%x\n",
	       i, ha->reg->mailboxIn0, i, RD_REG_DWORD(&ha->reg->mailboxOut0));

	for (i = 1; i < MBOX_REG_COUNT-1; i++)
		printk(KERN_INFO "  Mailbox In[%d] = %04x, Mailbox Out[%d] = 08%x\n",
		       i, ha->reg->mailboxIn[i], i, RD_REG_DWORD(&ha->reg->mailboxOut[i-1]));
#endif

#ifdef QLA4010
	printk(KERN_INFO "  Mailbox[%d] = %08x\n", i, RD_REG_DWORD(&ha->reg->mailboxIn0));

	for (i = 1; i < MBOX_REG_COUNT-1; i++)
		printk(KERN_INFO "  Mailbox[%d] = %08x\n", i, RD_REG_DWORD(&ha->reg->mailboxIn[i-1]));
#endif
}

void
qla4xxx_print_iocb_passthru(uint32_t dbg_mask, scsi_qla_host_t *ha, INT_IOCB_PASSTHRU *iocb)
{
	if ((ql_dbg_level & dbg_mask) != 0) {
		printk("SendDMAOffset=0x%x, RspDMAOffset=0x%x\n",
		       iocb->SendDMAOffset, iocb->RspDMAOffset);
		printk("IOCBCmdBuffer:\n");
		qla4xxx_dump_bytes(dbg_mask, iocb->IOCBCmdBuffer, sizeof(iocb->IOCBCmdBuffer));
		printk("IOCBStatusBuffer:\n");
		qla4xxx_dump_bytes(dbg_mask, iocb->IOCBStatusBuffer, sizeof(iocb->IOCBStatusBuffer));
		printk("SendData:  (SendData %p, Len=%d)\n", iocb->SendData, iocb->SendDataLen);
		qla4xxx_dump_bytes(dbg_mask, iocb->SendData, iocb->SendDataLen);
		printk("RspData:  (RspData %p, Len=%d)\n", iocb->RspData, iocb->RspDataLen);
		qla4xxx_dump_bytes(dbg_mask, iocb->RspData, iocb->RspDataLen);
	}
}


