/******************************************************************************
 *     Copyright (C)  2003 -2005 QLogic Corporation
 * QLogic ISP4xxx Device Driver
 *
 * This program includes a device driver for Linux 2.4.x that may be
 * distributed with QLogic hardware specific firmware binary file.
 * You may modify and redistribute the device driver code under the
 * GNU General Public License as published by the Free Software Foundation
 * (version 2 or a later version) and/or under the following terms,
 * as applicable:
 *
 * 	1. Redistribution of source code must retain the above copyright
 *         notice, this list of conditions and the following disclaimer.
 * 	2. Redistribution in binary form must reproduce the above copyright
 *         notice, this list of conditions and the following disclaimer in
 *         the documentation and/or other materials provided with the
 *         distribution.
 * 	3. The name of QLogic Corporation may not be used to endorse or
 *         promote products derived from this software without specific
 *         prior written permission
 * 	
 * You may redistribute the hardware specific firmware binary file under
 * the following terms:
 * 	1. Redistribution of source code (only if applicable), must
 *         retain the above copyright notice, this list of conditions and
 *         the following disclaimer.
 * 	2. Redistribution in binary form must reproduce the above
 *         copyright notice, this list of conditions and the following
 *         disclaimer in the documentation and/or other materials provided
 *         with the distribution.
 * 	3. The name of QLogic Corporation may not be used to endorse or
 *         promote products derived from this software without specific
 *         prior written permission
 *
 * REGARDLESS OF WHAT LICENSING MECHANISM IS USED OR APPLICABLE,
 * THIS PROGRAM IS PROVIDED BY QLOGIC CORPORATION "AS IS'' AND ANY
 * EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR
 * PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR
 * ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT,
 * STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING
 * IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGE.
 *
 * USER ACKNOWLEDGES AND AGREES THAT USE OF THIS PROGRAM WILL NOT CREATE
 * OR GIVE GROUNDS FOR A LICENSE BY IMPLICATION, ESTOPPEL, OR OTHERWISE
 * IN ANY INTELLECTUAL PROPERTY RIGHTS (PATENT, COPYRIGHT, TRADE SECRET,
 * MASK WORK, OR OTHER PROPRIETARY RIGHT) EMBODIED IN ANY OTHER QLOGIC
 * HARDWARE OR SOFTWARE EITHER SOLELY OR IN COMBINATION WITH THIS PROGRAM
 *
 ******************************************************************************
 *             Please see release.txt for revision history.                   *
 *                                                                            *
 ******************************************************************************
 * Function Table of Contents:
 *	  qla4extioctl_query_hba_iscsi_node
 *	  qla4extioctl_query_hba_iscsi_portal
 *	  qla4extioctl_query_disc_iscsi_node
 *	  qla4extioctl_query_disc_iscsi_portal
 *	  qla4extioctl_query_driver
 *	  qla4extioctl_query_fw
 *	  qla4extioctl_query_chip
 *	  qla4extioctl_query
 *	  qla4extioctl_reg_aen
 *	  qla4extioctl_get_aen
 *	  qla4extioctl_get_statistics_gen
 *	  qla4extioctl_get_statistics_iscsi
 *	  qla4extioctl_get_device_entry_iscsi
 *	  qla4extioctl_get_init_fw_iscsi
 *	  qla4extioctl_get_isns_server
 *	  qla4extioctl_get_isns_disc_targets
 *	  qla4extioctl_get_data
 *	  qla4extioctl_rst_statistics_gen
 *	  qla4extioctl_rst_statistics_iscsi
 *	  qla4extioctl_set_device_entry_iscsi
 *	  qla4extioctl_set_init_fw_iscsi
 *	  qla4extioctl_set_data
 *	  qla4xxx_ioctl_sleep_done
 *	  qla4xxx_ioctl_sem_init
 *	  qla4xxx_scsi_pass_done
 *	  qla4extioctl_scsi_passthru
 *	  qla4extioctl_iscsi_passthru
 *	  qla4extioctl_get_hbacnt
 *	  qla4intioctl_logout_iscsi
 *	  qla4intioctl_ping
 *	  qla4intioctl_get_flash
 *	  qla4intioctl_get_driver_debug_level
 *	  qla4intioctl_get_host_no
 *	  qla4intioctl_get_data
 *	  qla4intioctl_set_flash
 *	  qla4intioctl_set_driver_debug_level
 *	  qla4intioctl_set_data
 *	  qla4intioctl_hba_reset
 *	  qla4intioctl_copy_fw_flash
 *	  qla4xxx_ioctl
 *        apidev_open
 *        apidev_close
 *        apidev_ioctl
 *        apidev_init
 *        apidev_cleanup
 *	
 ****************************************************************************/

#include <linux/config.h>
#include <linux/module.h>
#include <stdarg.h>
#include <asm/io.h>
#include <asm/irq.h>
#include <asm/segment.h>
#include <asm/byteorder.h>
#include <linux/spinlock.h>
#include <linux/version.h>
#include <linux/types.h>
#include <linux/string.h>
#include <linux/errno.h>
#include <linux/kernel.h>
#include <linux/delay.h>
#include <linux/timer.h>
#include <linux/sched.h>
#include <linux/pci.h>
#include <linux/proc_fs.h>
#include <linux/blk.h>
#include <linux/tqueue.h>
#include <asm/uaccess.h>

#ifdef __VMKERNEL_MODULE__
#include <scsi.h>
#include <hosts.h>
#else
#include <../drivers/scsi/scsi.h>
#include <../drivers/scsi/hosts.h>
#endif

#include "qlisioct.h"
#include "qlinioct.h"
#include "qla4x_fw.h"
#include "qla4xxx.h"
#include "ql4print.h"


// KRH: (BEGIN) Define these locally, for now
/*
 * Sub codes for Get Data.
 * Use in combination with INT_GET_DATA as the ioctl code
 */
#define INT_SC_GET_DRIVER_DEBUG_LEVEL   2
#define INT_SC_GET_HOST_NO 		3

/*
 * Sub codes for Set Data.
 * Use in combination with INT_SET_DATA as the ioctl code
 */
#define INT_SC_SET_DRIVER_DEBUG_LEVEL	2

/*
 * Sub codes for Reset
 * Use in combination with INT_CC_HBA_RESET as the ioctl code
 */
#define INT_SC_HBA_RESET			0
#define INT_SC_FIRMWARE_RESET			1
#define INT_SC_TARGET_WARM_RESET		2
#define INT_SC_LUN_RESET			3
//KRH: (END)

/* Defines for Passthru */
#define IOCTL_INVALID_STATUS			0xffff

/* Defines for byte-order translation */
#define GET_DATA	0
#define SET_DATA	1

/**************************
 * Global Data
 **************************/

#define QLA_IOCTL_SCRAP_SIZE		16400 /* scrap memory for local use. */

/*
 * Create character driver "HbaApiDev" w dynamically allocated major number
 * and create "/proc/scsi/<QLA4XXX_PROC_NAME>/HbaApiNode" as the device
 * node associated with the major number.
 */
#define APIDEV_NODE  "HbaApiNode"
#ifdef __VMKERNEL_MODULE__
#define APIDEV_NAME  "HbaApiDev-"QLA4XXX_PROC_NAME
#else
#define APIDEV_NAME  "HbaApiDev"
#endif

STATIC int apidev_major = 0;
STATIC struct Scsi_Host *apidev_host = 0;


/**************************
 * Helper Functions
 **************************/
void *
Q64BIT_TO_PTR(uint64_t buf_addr)
{
#if defined(QLA_CONFIG_COMPAT) || !defined(CONFIG_64BIT)
	union ql_doublelong {
		struct {
			uint32_t        lsl;
			uint32_t        msl;
		} longs;
		uint64_t        dl;
	};

	union ql_doublelong tmpval;

	tmpval.dl = buf_addr;

#if defined(QLA_CONFIG_COMPAT)
	/* return lower 32bit in 64bit pointer */
	return((void *)(uint64_t)(tmpval.longs.lsl));
#else
	/* return lower 32bit in 32bit pointer */
	return((void *)(tmpval.longs.lsl));
#endif
#else
	/* return the full 64bit as is */
	return((void *)buf_addr);
#endif
}

inline void *
ql4_kmem_zalloc(int siz)
{
	void	*bp;

	if ((bp = kmalloc(siz, GFP_ATOMIC)) != NULL) {
		memset(bp, 0, siz);
	}

	return (bp);
}

/*
 * qla4xxx_alloc_ioctl_mem
 *	Allocates memory needed by IOCTL code.
 *
 * Input:
 *	ha = adapter state pointer.
 *
 * Returns:
 *	qla4xxx local function return status code.
 *
 * Context:
 *	Kernel context.
 */
static int
qla4xxx_alloc_ioctl_mem(scsi_qla_host_t *ha)
{
	ENTER(__func__);
	//QL4PRINT(QLP4,
	//    printk("scsi%d: %s: inst %d entered.\n",
	//    ha->host_no, __func__, ha->instance));

	/* Pick the largest size we'll need per ha of all ioctl cmds.
	 * Use this size when freeing.
	 */
	ha->ioctl_scrap_mem = ql4_kmem_zalloc(QLA_IOCTL_SCRAP_SIZE);
	if (ha->ioctl_scrap_mem == NULL) {
		printk(KERN_WARNING
		    "scsi%d: ERROR in ioctl scrap_mem allocation.\n",
		    ha->host_no);
		return QLA_ERROR;
	}
	ha->ioctl_scrap_mem_size = QLA_IOCTL_SCRAP_SIZE;
	ha->ioctl_scrap_mem_used = 0;

	QL4PRINT(QLP4|QLP7,
	    printk("scsi(%d): %s: scrap_mem_size=%d.\n",
	    ha->host_no, __func__, ha->ioctl_scrap_mem_size));

	//QL4PRINT(QLP4,
	//    printk("scsi(%d): %s: inst %d exiting.\n",
	//    ha->host_no, __func__, ha->instance));

	LEAVE(__func__);
	return QLA_SUCCESS;
}

/*
 * qla4xxx_free_ioctl_mem
 *	Frees memory used by IOCTL code for the specified ha.
 *
 * Input:
 *	ha = adapter state pointer.
 *
 * Context:
 *	Kernel context.
 */
static void
qla4xxx_free_ioctl_mem(scsi_qla_host_t *ha)
{
	ENTER(__func__);
	//QL4PRINT(QLP4,
	//    printk("scsi%d: %s: inst %d entered.\n",
	//    ha->host_no, __func__, ha->instance));

	if (ha->ioctl_scrap_mem != NULL) {
		/* The size here must match up to what we
		 * allocated before.
		 */
		kfree(ha->ioctl_scrap_mem);
		ha->ioctl_scrap_mem = NULL;
		ha->ioctl_scrap_mem_size = 0;
	}

	//QL4PRINT(QLP4,
	//    printk("scsi%d: %s: inst %d exiting.\n",
	//    ha->host_no, __func__, ha->instance));
	LEAVE(__func__);
}

/*
 * qla4xxx_get_ioctl_scrap_mem
 *	Returns pointer to memory of the specified size from the scrap buffer.
 *	This can be called multiple times before the free call as long
 *	as the memory is to be used by the same ioctl command and
 *	there's still memory left in the scrap buffer.
 *
 * Input:
 *	ha = adapter state pointer.
 *	ppmem = pointer to return a buffer pointer.
 *	size = size of buffer to return.
 *
 * Returns:
 *	qla4xxx local function return status code.
 *
 * Context:
 *	Kernel context.
 */
int
qla4xxx_get_ioctl_scrap_mem(scsi_qla_host_t *ha, void **ppmem, uint32_t size)
{
	int		ret = QLA_SUCCESS;
	uint32_t	free_mem;

	ENTER(__func__);
	//QL4PRINT(QLP4,
	//    printk("scsi%d: %s: inst %d entered.\n",
	//    ha->host_no, __func__, ha->instance));

	free_mem = ha->ioctl_scrap_mem_size - ha->ioctl_scrap_mem_used;

	/* Round memory allocation up to a long aligned size, if needed */
	if (size % sizeof(long)) {
		size = ((size / sizeof(long)) + 1) * sizeof(long);
	}

	if (free_mem >= size) {
		*ppmem = ha->ioctl_scrap_mem + ha->ioctl_scrap_mem_used;
		ha->ioctl_scrap_mem_used += size;
	} else {
		QL4PRINT(QLP2|QLP4,
		    printk("scsi(%d): %s: no more scrap memory.\n",
		    ha->host_no, __func__));

		ret = QLA_ERROR;
	}

	//QL4PRINT(QLP4,
	//    printk("scsi%d: %s: inst %d exiting.\n",
	//    ha->host_no, __func__, ha->instance));
	LEAVE(__func__);

	return (ret);
}

/*
 * qla4xxx_free_ioctl_scrap_mem
 *	Makes the entire scrap buffer free for use.
 *
 * Input:
 *	ha = adapter state pointer.
 *
 * Returns:
 *	qla4xxx local function return status code.
 *
 */
void
qla4xxx_free_ioctl_scrap_mem(scsi_qla_host_t *ha)
{
	ENTER(__func__);
	//QL4PRINT(QLP4,
	//    printk("scsi%d: %s: inst %d entered.\n",
	//    ha->host_no, __func__, ha->instance));

	memset(ha->ioctl_scrap_mem, 0, ha->ioctl_scrap_mem_size);
	ha->ioctl_scrap_mem_used = 0;

	//QL4PRINT(QLP4,
	//    printk("scsi%d: %s: inst %d exiting.\n",
	//    ha->host_no, __func__, ha->instance));
	LEAVE(__func__);
}


/**************************
* Functions
**************************/

/**************************************************************************
 * qla4extioctl_query_hba_iscsi_node
 *	This routine retrieves the HBA node properties
 *
 * Input:
 *	ha    = adapter structure pointer.
 *	ioctl = IOCTL structure pointer.
 *
 * Output:
 *	None
 *
 * Returns:
 *	QLA_SUCCESS = success
 *	QLA_ERROR   = error
 *
 * Context:
 *	Kernel context.
 **************************************************************************/
static int
qla4extioctl_query_hba_iscsi_node(scsi_qla_host_t *ha, EXT_IOCTL_ISCSI *ioctl)
{
	int			status = 0;
	uint32_t		mbox_cmd[MBOX_REG_COUNT];
	uint32_t		mbox_sts[MBOX_REG_COUNT];
	EXT_HBA_ISCSI_NODE	*phba_node = NULL;
	INIT_FW_CTRL_BLK	*init_fw_cb;


	ENTER("qla4extioctl_query_hba_iscsi_node");
#if 0 && defined(__VMKERNEL_MODULE__)
	spin_lock(&ha->dma_buf_spinlock);
#else
	down(&ha->dma_buf_sem);
#endif

	if (qla4xxx_get_ioctl_scrap_mem(ha, (void **)&phba_node,
	    sizeof(EXT_HBA_ISCSI_NODE))) {
		/* not enough memory */
		ioctl->Status = EXT_STATUS_NO_MEMORY;
		QL4PRINT(QLP2|QLP4,
		    printk("%s(%d): inst=%d scrap not big enough. "
		    "size requested=%ld.\n",
		    __func__, ha->host_no, ha->instance,
		    (ulong)sizeof(EXT_HBA_ISCSI_NODE)));
		goto exit_query_hba_node;
	}

	if (!ha->dma_buf.virt_addr || !ha->dma_buf.phys_addr ||
	    !ioctl->ResponseAdr) {
		QL4PRINT(QLP2,
			 printk("scsi%d: %s: memory allocation problem\n",
				ha->host_no, __func__));

		ioctl->Status = EXT_STATUS_NO_MEMORY;
		goto exit_query_hba_node;
	}

	if (ioctl->ResponseLen < sizeof(*phba_node) ||
	    ha->dma_buf.buf_len < sizeof(*init_fw_cb)) {
		QL4PRINT(QLP2,
			 printk("scsi%d: %s: memory area too small\n",
				ha->host_no, __func__));

		ioctl->Status = EXT_STATUS_BUFFER_TOO_SMALL;
		goto exit_query_hba_node;
	}

	/*
	 * Send mailbox command
	 */
	memset(mbox_cmd, 0, sizeof(mbox_cmd));
	memset(mbox_sts, 0, sizeof(mbox_sts));
	mbox_cmd[0] = MBOX_CMD_GET_INIT_FW_CTRL_BLOCK;
	mbox_cmd[2] = LSDW(ha->dma_buf.phys_addr);
	mbox_cmd[3] = MSDW(ha->dma_buf.phys_addr);

	if (qla4xxx_mailbox_command(ha, 4, 1, &mbox_cmd[0], &mbox_sts[0]) ==
	    QLA_ERROR) {
		QL4PRINT(QLP2, printk("scsi%d: %s: command failed \n",
				      ha->host_no, __func__));

		ioctl->Status = EXT_STATUS_MAILBOX;
		ioctl->DetailStatus = mbox_sts[0];

		goto exit_query_hba_node;
	}

	/*
	 * Transfer data from Fw's DEV_DB_ENTRY buffer to IOCTL's
	 * EXT_HBA_ISCSI_NODE buffer
	 */
	init_fw_cb = (INIT_FW_CTRL_BLK *) ha->dma_buf.virt_addr;

	memset(phba_node, 0, sizeof(*phba_node));
	phba_node->PortNumber = le16_to_cpu(init_fw_cb->PortNumber);
	phba_node->NodeInfo.PortalCount = 1;

	memcpy(phba_node->NodeInfo.IPAddr.IPAddress,
	    init_fw_cb->IPAddr,
	    sizeof(phba_node->NodeInfo.IPAddr.IPAddress));
	memcpy(phba_node->NodeInfo.iSCSIName,
	    init_fw_cb->iSCSINameString,
	    sizeof(phba_node->NodeInfo.iSCSIName));
	memcpy(phba_node->NodeInfo.Alias,
	    init_fw_cb->Alias,
	    sizeof(phba_node->NodeInfo.Alias));

	sprintf(phba_node->DeviceName, "/proc/scsi/%s/HbaApiNode",
	    QLA4XXX_PROC_NAME);

	/*
	 * Copy the IOCTL EXT_HBA_ISCSI_NODE buffer to the user's data space
	 */
	if (copy_to_user(Q64BIT_TO_PTR(ioctl->ResponseAdr),
	    phba_node, ioctl->ResponseLen) != 0) {
		QL4PRINT(QLP2, printk("scsi%d: %s: copy failed\n",
				      ha->host_no, __func__));

		status = (-EFAULT);
		ioctl->Status = EXT_STATUS_COPY_ERR;
		goto exit_query_hba_node;
	}

	ioctl->Status = EXT_STATUS_OK;

	exit_query_hba_node:
	qla4xxx_free_ioctl_scrap_mem(ha);
#if 0 && defined(__VMKERNEL_MODULE__)
	spin_unlock(&ha->dma_buf_spinlock);
#else
	up(&ha->dma_buf_sem);
#endif
	LEAVE("qla4extioctl_query_hba_iscsi_node");
	return(status);
}

/**************************************************************************
 * qla4extioctl_query_hba_iscsi_portal
 *	This routine retrieves the HBA iSCSI portal properties
 *
 * Input:
 *	ha    = adapter structure pointer.
 *	ioctl = IOCTL structure pointer.
 *
 * Output:
 *	None
 *
 * Returns:
 *	QLA_SUCCESS = success
 *	QLA_ERROR   = error
 *
 * Context:
 *	Kernel context.
 **************************************************************************/
static int
qla4extioctl_query_hba_iscsi_portal(scsi_qla_host_t *ha, EXT_IOCTL_ISCSI *ioctl)
{
	int			status = 0;
	uint32_t		mbox_cmd[MBOX_REG_COUNT];
	uint32_t		mbox_sts[MBOX_REG_COUNT];
	EXT_HBA_ISCSI_PORTAL	*phba_portal;
	FLASH_SYS_INFO *sys_info;
	uint32_t num_valid_ddb_entries;


	ENTER(__func__);

	if (!ioctl->ResponseAdr) {
		QL4PRINT(QLP2,
			 printk("scsi%d: %s: memory allocation problem\n",
				ha->host_no, __func__));

		ioctl->Status = EXT_STATUS_NO_MEMORY;
		goto exit_query_hba_portal;
	}

	if (qla4xxx_get_ioctl_scrap_mem(ha, (void **)&phba_portal,
	    sizeof(EXT_HBA_ISCSI_PORTAL))) {
		/* not enough memory */
		QL4PRINT(QLP2|QLP4,
		    printk("%s(%d): inst=%d scrap not big enough. "
		    "size requested=%ld.\n",
		    __func__, ha->host_no, ha->instance,
		    (ulong)sizeof(EXT_HBA_ISCSI_PORTAL)));

		ioctl->Status = EXT_STATUS_NO_MEMORY;
		goto exit_query_hba_portal;
	}

	if (ioctl->ResponseLen < sizeof(*phba_portal)) {
		QL4PRINT(QLP2, printk("scsi%d: %s: memory area too small\n",
				      ha->host_no, __func__));

		ioctl->Status = EXT_STATUS_BUFFER_TOO_SMALL;
		goto exit_query_hba_portal;
	}

	/*
	 * Fill in EXT_HBA_ISCSI_PORTAL buffer
	 */
	memset(phba_portal, 0, sizeof(*phba_portal));

	strcpy(phba_portal->DriverVersion, QLA4XXX_DRIVER_VERSION);
	sprintf(phba_portal->FWVersion, "%02d.%02d Patch %02d Build %02d",
	    ha->firmware_version[0], ha->firmware_version[1],
	    ha->patch_number, ha->build_number);

	/* ----- Get firmware state information ---- */
	memset(mbox_cmd, 0, sizeof(mbox_cmd));
	memset(mbox_sts, 0, sizeof(mbox_sts));
	mbox_cmd[0] = MBOX_CMD_GET_FW_STATE;
	if (qla4xxx_mailbox_command(ha, 1, 4, &mbox_cmd[0], &mbox_sts[0])
	    != QLA_SUCCESS) {
		QL4PRINT(QLP2, printk("scsi%d: %s: MBOX_CMD_GET_FW_STATE "
		    "failed w/ status %04X\n",
		    ha->host_no, __func__, mbox_sts[0]));

		ioctl->Status = EXT_STATUS_MAILBOX;
		ioctl->DetailStatus = mbox_sts[0];

		goto exit_query_hba_portal;
	}

	switch (mbox_sts[1]) {
	case FW_STATE_READY:
		phba_portal->State = EXT_DEF_CARD_STATE_READY;
		break;
	case FW_STATE_CONFIG_WAIT:
		phba_portal->State = EXT_DEF_CARD_STATE_CONFIG_WAIT;
		break;
	case FW_STATE_WAIT_LOGIN:
		phba_portal->State = EXT_DEF_CARD_STATE_LOGIN;
		break;
	case FW_STATE_ERROR:
		phba_portal->State = EXT_DEF_CARD_STATE_ERROR;
		break;
	}

	switch (mbox_sts[3] & 0x0001) {
	case FW_ADDSTATE_COPPER_MEDIA:
		phba_portal->Type = EXT_DEF_TYPE_COPPER;
		break;
	case FW_ADDSTATE_OPTICAL_MEDIA:
		phba_portal->Type = EXT_DEF_TYPE_OPTICAL;
		break;
	}

	/* ----- Get ddb entry information ---- */
	if (qla4xxx_get_ddb_entry(ha, 0, NULL, 0, &num_valid_ddb_entries,
				  NULL, NULL, NULL, NULL, NULL) == QLA_ERROR) {
		QL4PRINT(QLP2,
		    printk("scsi%d: %s: qla4xxx_get_ddb_entry failed!\n",
				      ha->host_no, __func__));

		goto exit_query_hba_portal;
	}
	phba_portal->DiscTargetCount = (uint16_t) num_valid_ddb_entries;

	/* ----- Get flash sys info information ---- */
#if 0 && defined(__VMKERNEL_MODULE__)
	spin_lock(&ha->dma_buf_spinlock);
#else
	down(&ha->dma_buf_sem);
#endif
	sys_info = (FLASH_SYS_INFO *) ha->dma_buf.virt_addr;

	memset(mbox_cmd, 0, sizeof(mbox_cmd));
	memset(mbox_sts, 0, sizeof(mbox_sts));
	mbox_cmd[0] = MBOX_CMD_READ_FLASH;
	mbox_cmd[1] = LSDW(ha->dma_buf.phys_addr);
	mbox_cmd[2] = MSDW(ha->dma_buf.phys_addr);
	mbox_cmd[3] = INT_ISCSI_SYSINFO_FLASH_OFFSET;
	mbox_cmd[4] = sizeof(*sys_info);

	if (qla4xxx_mailbox_command(ha, 5, 2, &mbox_cmd[0], &mbox_sts[0])
	    != QLA_SUCCESS) {
		QL4PRINT(QLP2,
			 printk("scsi%d: %s: MBOX_CMD_READ_FLASH failed w/ "
				"status %04X\n",
				ha->host_no, __func__, mbox_sts[0]));

#if 0 && defined(__VMKERNEL_MODULE__)
		spin_unlock(&ha->dma_buf_spinlock);
#else
		up(&ha->dma_buf_sem);
#endif
		ioctl->Status = EXT_STATUS_MAILBOX;
		ioctl->DetailStatus = mbox_sts[0];

		goto exit_query_hba_portal;
	}

	phba_portal->SerialNum = le32_to_cpu(sys_info->serialNumber);
	memcpy(phba_portal->IPAddr.IPAddress,
	    ha->ip_address,
	    MIN(sizeof(phba_portal->IPAddr.IPAddress),
	    sizeof(ha->ip_address)));
	memcpy(phba_portal->MacAddr,
	    sys_info->physAddr[0].address,
	    sizeof(phba_portal->MacAddr));
	memcpy(phba_portal->Manufacturer,
	    sys_info->vendorId,
	    sizeof(phba_portal->Manufacturer));
	memcpy(phba_portal->Model,
	    sys_info->productId,
	    sizeof(phba_portal->Model));
#if 0 && defined(__VMKERNEL_MODULE__)
	spin_unlock(&ha->dma_buf_spinlock);
#else
	up(&ha->dma_buf_sem);
#endif

	/*memcpy(phba_portal->OptRomVersion, ?,
		sizeof(phba_portal->OptRomVersion)); */

	/*
	 * Copy the IOCTL EXT_HBA_ISCSI_PORTAL buffer to the user's data space
	 */
	if (copy_to_user(Q64BIT_TO_PTR(ioctl->ResponseAdr),
	    phba_portal, ioctl->ResponseLen) != 0) {
		QL4PRINT(QLP2, printk("scsi%d: %s: memory area too small\n",
				      ha->host_no, __func__));

		status = (-EFAULT);
		ioctl->Status = EXT_STATUS_COPY_ERR;

		goto exit_query_hba_portal;
	}

	ioctl->Status = EXT_STATUS_OK;

	exit_query_hba_portal:
	qla4xxx_free_ioctl_scrap_mem(ha);

	LEAVE(__func__);
	return(status);
}

/**************************************************************************
 * qla4extioctl_query_disc_iscsi_node
 *	This routine retrieves the properties of the attached devices
 *	registered as iSCSI nodes discovered by the HBA driver.
 *
 * Input:
 *	ha    = adapter structure pointer.
 *	ioctl = IOCTL structure pointer.
 *
 * Output:
 *	None
 *
 * Returns:
 *	QLA_SUCCESS = success
 *	QLA_ERROR   = error
 *
 * Context:
 *	Kernel context.
 **************************************************************************/
static int
qla4extioctl_query_disc_iscsi_node(scsi_qla_host_t *ha, EXT_IOCTL_ISCSI *ioctl)
{
	int		status = 0;
	DEV_DB_ENTRY *fw_ddb_entry = (DEV_DB_ENTRY *) ha->dma_buf.virt_addr;
	dma_addr_t      fw_ddb_entry_dma = ha->dma_buf.phys_addr;
	EXT_DISC_ISCSI_NODE *pdisc_node = NULL;
	ddb_entry_t *ddb_entry;


	ENTER(__func__);

#if 0 && defined(__VMKERNEL_MODULE__)
	spin_lock(&ha->dma_buf_spinlock);
#else
	down(&ha->dma_buf_sem);
#endif

	if (ioctl->ResponseLen < sizeof(*pdisc_node)) {
		QL4PRINT(QLP2,
			 printk("scsi%d: %s: response buffer too small\n",
				ha->host_no, __func__));

		ioctl->Status = EXT_STATUS_BUFFER_TOO_SMALL;
		goto exit_disc_node;
	}

	if (ha->dma_buf.buf_len < sizeof(DEV_DB_ENTRY)) {
		if (qla4xxx_resize_dma_buf(ha, &ha->dma_buf,
					   sizeof(DEV_DB_ENTRY)) != QLA_SUCCESS) {
			QL4PRINT(QLP2,
				 printk("scsi%d: %s: unable to allocate memory "
					"for dma buffer.\n",
					ha->host_no, __func__));
			ioctl->Status = EXT_STATUS_NO_MEMORY;
			goto exit_disc_node;
		}
	}

	if (qla4xxx_get_ioctl_scrap_mem(ha, (void **)&pdisc_node,
	    sizeof(EXT_DISC_ISCSI_NODE))) {
		/* not enough memory */
		ioctl->Status = EXT_STATUS_NO_MEMORY;
		QL4PRINT(QLP2|QLP4,
		    printk("%s(%d): inst=%d scrap not big enough. "
		    "size requested=%ld.\n",
		    __func__, ha->host_no, ha->instance,
		    (ulong)sizeof(EXT_DISC_ISCSI_NODE)));
		goto exit_disc_node;
	}

	/* ----- get device database entry info from firmware ---- */
	if (qla4xxx_get_ddb_entry(ha, ioctl->Instance, fw_ddb_entry,
				  fw_ddb_entry_dma, NULL, NULL, NULL, NULL,
				  NULL, NULL) != QLA_SUCCESS) {
		QL4PRINT(QLP2, printk("scsi%d: %s: failed to get DEV_DB_ENTRY "
				      "info.\n", ha->host_no, __func__));

		ioctl->Status = EXT_STATUS_ERR;
		ioctl->RequestLen = 0;
		ioctl->DetailStatus = ioctl->Instance;
		goto exit_disc_node;
	}

	/* --- Transfer data from Fw's DEV_DB_ENTRY buffer to
	*      IOCTL's EXT_DISC_ISCSI_PORTAL buffer --- */
	memset(pdisc_node, 0, sizeof(*pdisc_node));
	pdisc_node->NodeInfo.PortalCount = 1;
	pdisc_node->NodeInfo.IPAddr.Type = EXT_DEF_TYPE_ISCSI_IP;
	memcpy(pdisc_node->NodeInfo.IPAddr.IPAddress, fw_ddb_entry->ipAddr,
	    MIN(sizeof(pdisc_node->NodeInfo.IPAddr.IPAddress),
	    sizeof(fw_ddb_entry->ipAddr)));
	strncpy(pdisc_node->NodeInfo.Alias, fw_ddb_entry->iSCSIAlias,
	    MIN(sizeof(pdisc_node->NodeInfo.Alias),
	    sizeof(fw_ddb_entry->iSCSIAlias)));
	strncpy(pdisc_node->NodeInfo.iSCSIName, fw_ddb_entry->iscsiName,
	    MIN(sizeof(pdisc_node->NodeInfo.iSCSIName),
	    sizeof(fw_ddb_entry->iscsiName)));

	if ((ddb_entry = qla4xxx_lookup_ddb_by_fw_index(ha, ioctl->Instance)) ==
	    NULL) {
		QL4PRINT(QLP2,
			 printk("scsi%d: %s: device index [%d] not logged in. "
				"Dummy target info returned.\n",
				ha->host_no, __func__, ioctl->Instance));

		pdisc_node->SessionID       = 0xDEAD;
		pdisc_node->ConnectionID    = 0xDEAD;
		pdisc_node->PortalGroupID   = 0xDEAD;
		pdisc_node->ScsiAddr.Bus    = 0xFF;
		pdisc_node->ScsiAddr.Target = 0xFF;
		pdisc_node->ScsiAddr.Lun    = 0xFF;
	}
	else {
		pdisc_node->SessionID       = ddb_entry->target_session_id;
		pdisc_node->ConnectionID    = ddb_entry->connection_id;
		pdisc_node->PortalGroupID   = 0;
		pdisc_node->ScsiAddr.Bus    = ddb_entry->bus;
		pdisc_node->ScsiAddr.Target = ddb_entry->target;
		pdisc_node->ScsiAddr.Lun    = 0;
	}

	/* --- Copy Results to user space --- */
	if (copy_to_user(Q64BIT_TO_PTR(ioctl->ResponseAdr),
	    pdisc_node, sizeof(*pdisc_node)) != 0) {
		QL4PRINT(QLP2,
			 printk("scsi%d: %s: copy error to user space.\n",
				ha->host_no, __func__));

		status = (-EFAULT);
		ioctl->Status = EXT_STATUS_COPY_ERR;
		goto exit_disc_node;
	}

	ioctl->Status = EXT_STATUS_OK;

	exit_disc_node:
	qla4xxx_free_ioctl_scrap_mem(ha);
#if 0 && defined(__VMKERNEL_MODULE__)
	spin_unlock(&ha->dma_buf_spinlock);
#else
	up(&ha->dma_buf_sem);
#endif
	LEAVE("qla4extioctl_query_disc_iscsi_node");
	return(status);
}

/**************************************************************************
 * qla4extioctl_query_disc_iscsi_portal
 *	This routine retrieves the properties of the iSCSI portal
 *	discovered by the HBA driver.
 *
 * Input:
 *	ha    = adapter structure pointer.
 *	ioctl = IOCTL structure pointer.
 *
 * Output:
 *	None
 *
 * Returns:
 *	QLA_SUCCESS = success
 *	QLA_ERROR   = error
 *
 * Context:
 *	Kernel context.
 **************************************************************************/
static int
qla4extioctl_query_disc_iscsi_portal(scsi_qla_host_t *ha,
    EXT_IOCTL_ISCSI *ioctl)
{
	int		status = 0;
	DEV_DB_ENTRY *fw_ddb_entry = (DEV_DB_ENTRY *) ha->dma_buf.virt_addr;
	dma_addr_t    fw_ddb_entry_dma = ha->dma_buf.phys_addr;
	EXT_DISC_ISCSI_PORTAL *pdisc_portal = NULL;


	ENTER(__func__);

#if 0 && defined(__VMKERNEL_MODULE__)
	spin_lock(&ha->dma_buf_spinlock);
#else
	down(&ha->dma_buf_sem);
#endif

	if (qla4xxx_get_ioctl_scrap_mem(ha, (void **)&pdisc_portal,
	    sizeof(EXT_DISC_ISCSI_PORTAL))) {
		/* not enough memory */
		ioctl->Status = EXT_STATUS_NO_MEMORY;
		QL4PRINT(QLP2|QLP4,
		    printk("%s(%d): inst=%d scrap not big enough. "
		    "size requested=%ld.\n",
		    __func__, ha->host_no, ha->instance,
		    (ulong)sizeof(EXT_DISC_ISCSI_PORTAL)));
		goto exit_disc_portal;
	}

	if (ioctl->ResponseLen < sizeof(*pdisc_portal)) {
		QL4PRINT(QLP2,
			 printk("scsi%d: %s: response buffer too small\n",
				ha->host_no, __func__));

		ioctl->Status = EXT_STATUS_BUFFER_TOO_SMALL;
		goto exit_disc_portal;
	}

	if (ha->dma_buf.buf_len < sizeof(DEV_DB_ENTRY)) {
		if (qla4xxx_resize_dma_buf(ha, &ha->dma_buf,
					   sizeof(DEV_DB_ENTRY)) != QLA_SUCCESS) {
			QL4PRINT(QLP2,
				 printk("scsi%d: %s: unable to allocate memory "
					"for dma buffer.\n",
					ha->host_no, __func__));

			ioctl->Status = EXT_STATUS_NO_MEMORY;
			goto exit_disc_portal;
		}
	}

	/* ----- get device database entry info from firmware ---- */
	if (qla4xxx_get_ddb_entry(ha, ioctl->Instance, fw_ddb_entry,
				  fw_ddb_entry_dma, NULL, NULL, NULL, NULL,
				  NULL, NULL) != QLA_SUCCESS) {
		QL4PRINT(QLP2,
			 printk("scsi%d: %s: failed to get DEV_DB_ENTRY info.\n",
				ha->host_no, __func__));

		ioctl->Status = EXT_STATUS_ERR;
		ioctl->RequestLen = 0;
		ioctl->DetailStatus = ioctl->Instance;
		goto exit_disc_portal;
	}

	/* --- Transfer data from Fw's DEV_DB_ENTRY buffer to IOCTL's
	*      EXT_DISC_ISCSI_PORTAL buffer --- */
	memset(pdisc_portal, 0, sizeof(*pdisc_portal));
	memcpy(pdisc_portal->IPAddr.IPAddress, fw_ddb_entry->ipAddr,
	    MIN(sizeof(pdisc_portal->IPAddr.IPAddress),
	    sizeof(fw_ddb_entry->ipAddr)));

	pdisc_portal->PortNumber = le16_to_cpu(fw_ddb_entry->portNumber);
	pdisc_portal->IPAddr.Type = EXT_DEF_TYPE_ISCSI_IP;
	pdisc_portal->NodeCount = 0;

	strncpy(pdisc_portal->HostName, fw_ddb_entry->iscsiName,
	    MIN(sizeof(pdisc_portal->HostName),
		    sizeof(fw_ddb_entry->iscsiName)));

	/* --- Copy Results to user space --- */
	if (copy_to_user(Q64BIT_TO_PTR(ioctl->ResponseAdr),
	    pdisc_portal, sizeof(*pdisc_portal)) != 0) {
		QL4PRINT(QLP2,
			 printk("scsi%d: %s: copy error to user space.\n",
				ha->host_no, __func__));

		status = (-EFAULT);
		ioctl->Status = EXT_STATUS_COPY_ERR;
		goto exit_disc_portal;
	}

	ioctl->Status = EXT_STATUS_OK;

	exit_disc_portal:
	qla4xxx_free_ioctl_scrap_mem(ha);
#if 0 && defined(__VMKERNEL_MODULE__)
	spin_unlock(&ha->dma_buf_spinlock);
#else
	up(&ha->dma_buf_sem);
#endif
	LEAVE(__func__);
	return(status);
}

/**************************************************************************
 * qla4extioctl_query_driver
 *	This routine retrieves the driver properties.
 *
 * Input:
 *	ha    = adapter structure pointer.
 *	ioctl = IOCTL structure pointer.
 *
 * Output:
 *	None
 *
 * Returns:
 *	QLA_SUCCESS = success
 *	QLA_ERROR   = error
 *
 * Context:
 *	Kernel context.
 **************************************************************************/
static int
qla4extioctl_query_driver(scsi_qla_host_t *ha, EXT_IOCTL_ISCSI *ioctl)
{
	EXT_DRIVER_INFO	*pdinfo = NULL;
	int		status = 0;


	ENTER(__func__);
	QL4PRINT(QLP3, printk("scsi%d: %s:\n", ha->host_no, __func__));

	if (qla4xxx_get_ioctl_scrap_mem(ha, (void **)&pdinfo,
	    sizeof(EXT_DRIVER_INFO))) {
		/* not enough memory */
		ioctl->Status = EXT_STATUS_NO_MEMORY;
		QL4PRINT(QLP2|QLP4,
		    printk("%s(%d): inst=%d scrap not big enough. "
		    "size requested=%ld.\n",
		    __func__, ha->host_no, ha->instance,
		    (ulong)sizeof(EXT_DRIVER_INFO)));
		goto exit_query_driver;
	}

	if (ioctl->ResponseLen < sizeof(*pdinfo)) {
		QL4PRINT(QLP2,
			 printk("scsi%d: %s: response buffer too small\n",
				ha->host_no, __func__));

		ioctl->Status = EXT_STATUS_BUFFER_TOO_SMALL;
		goto exit_query_driver;
	}

	memset(pdinfo, 0, sizeof(*pdinfo));
	memcpy(pdinfo->Version, QLA4XXX_DRIVER_VERSION,
	    sizeof(QLA4XXX_DRIVER_VERSION));

	pdinfo->NumOfBus        = EXT_DEF_MAX_HBA;
	pdinfo->TargetsPerBus   = EXT_DEF_MAX_TARGET;
	pdinfo->LunPerTarget    = EXT_DEF_MAX_LUN;
	pdinfo->LunPerTargetOS  = EXT_DEF_MAX_BUS;

	if (sizeof(dma_addr_t) > 4)
		pdinfo->DmaBitAddresses = 1;  /* 64-bit */
	else
		pdinfo->DmaBitAddresses = 0;  /* 32-bit */

	#if MEMORY_MAPPED_IO
	pdinfo->IoMapType       = 1;  /* Memory Mapped I/O */
	#else
	pdinfo->IoMapType       = 0;  /* I/O Mapped I/O */
	#endif

	//FIXME: Incomplete
	//dinfo->MaxTransferLen  = ?;
	//dinfo->MaxDataSegments = ?;
	//dinfo->Attrib          = ?;
	//dinfo->InternalFlags   = ?;

	if (copy_to_user(Q64BIT_TO_PTR(ioctl->ResponseAdr), pdinfo,
	    sizeof(*pdinfo)) != 0) {
		status = (-EFAULT);
		ioctl->Status = EXT_STATUS_COPY_ERR;
		goto exit_query_driver;
	}

	ioctl->Status = EXT_STATUS_OK;

	exit_query_driver:
	qla4xxx_free_ioctl_scrap_mem(ha);

	LEAVE(__func__);

	return(status);
}

/**************************************************************************
 * qla4extioctl_query_fw
 *	This routine retrieves the firmware properties.
 *
 * Input:
 *	ha    = adapter structure pointer.
 *	ioctl = IOCTL structure pointer.
 *
 * Output:
 *	None
 *
 * Returns:
 *	QLA_SUCCESS = success
 *	QLA_ERROR   = error
 *
 * Context:
 *	Kernel context.
 **************************************************************************/
static int
qla4extioctl_query_fw(scsi_qla_host_t *ha, EXT_IOCTL_ISCSI *ioctl)
{
	EXT_FW_INFO	*pfw_info = NULL;
	uint32_t	mbox_cmd[MBOX_REG_COUNT];
	uint32_t	mbox_sts[MBOX_REG_COUNT];
	int		status = 0;


	ENTER(__func__);
	QL4PRINT(QLP3, printk("scsi%d: %s:\n", ha->host_no, __func__));

	if (qla4xxx_get_ioctl_scrap_mem(ha, (void **)&pfw_info,
	    sizeof(EXT_FW_INFO))) {
		/* not enough memory */
		ioctl->Status = EXT_STATUS_NO_MEMORY;
		QL4PRINT(QLP2|QLP4,
		    printk("%s(%d): inst=%d scrap not big enough. "
		    "size requested=%ld.\n",
		    __func__, ha->host_no, ha->instance,
		    (ulong)sizeof(EXT_FW_INFO)));
		goto exit_query_fw;
	}

	if (ioctl->ResponseLen < sizeof(*pfw_info)) {
		QL4PRINT(QLP2,
			 printk("scsi%d: %s: response buffer too small\n",
				ha->host_no, __func__));

		ioctl->Status = EXT_STATUS_BUFFER_TOO_SMALL;
		goto exit_query_fw;
	}

	/* Fill in structure */
	memset(pfw_info, 0, sizeof(*pfw_info));

	/* ----- Get firmware version information ---- */
	memset(mbox_cmd, 0, sizeof(mbox_cmd));
	memset(mbox_sts, 0, sizeof(mbox_sts));
	mbox_cmd[0] = MBOX_CMD_ABOUT_FW;

	/*
	 * NOTE: In QLA4010, mailboxes 2 & 3 may hold an address for data.
	 * Make sure that we write 0 to those mailboxes, if unused.
	 */
	if (qla4xxx_mailbox_command(ha, 4, 5, &mbox_cmd[0], &mbox_sts[0]) !=
	    QLA_SUCCESS) {
		QL4PRINT(QLP2,
			 printk("scsi%d: %s: MBOX_CMD_ABOUT_FW failed w/ "
				"status %04X\n",
				ha->host_no, __func__, mbox_sts[0]));
		ioctl->Status = EXT_STATUS_MAILBOX;
		ioctl->DetailStatus = mbox_sts[0];
		goto exit_query_fw;
	}

	sprintf(pfw_info->Version, "FW Version %d.%d Patch %d Build %d",
	    mbox_sts[1], mbox_sts[2], mbox_sts[3], mbox_sts[4]);

	/* Copy info to caller */
	if (copy_to_user(Q64BIT_TO_PTR(ioctl->ResponseAdr), pfw_info,
	    sizeof(*pfw_info)) != 0) {
		QL4PRINT(QLP2|QLP4,
		    printk("scsi(%d): %s: response copy error.\n",
		    ha->host_no, __func__));
		status = (-EFAULT);
		ioctl->Status = EXT_STATUS_COPY_ERR;
		goto exit_query_fw;
	}

	ioctl->Status = EXT_STATUS_OK;

	exit_query_fw:
	qla4xxx_free_ioctl_scrap_mem(ha);

	LEAVE(__func__);
	return(status);
}

/**************************************************************************
 * qla4extioctl_query_chip
 *	This routine retrieves the chip properties.
 *
 * Input:
 *	ha    = adapter structure pointer.
 *	ioctl = IOCTL structure pointer.
 *
 * Output:
 *	None
 *
 * Returns:
 *	QLA_SUCCESS = success
 *	QLA_ERROR   = error
 *
 * Context:
 *	Kernel context.
 **************************************************************************/
static int
qla4extioctl_query_chip(scsi_qla_host_t *ha, EXT_IOCTL_ISCSI *ioctl)
{
	int		status = 0;
	uint32_t	mbox_cmd[MBOX_REG_COUNT];
	uint32_t	mbox_sts[MBOX_REG_COUNT];
	EXT_CHIP_INFO	*pchip_info = NULL;
	FLASH_SYS_INFO *sys_info;


	ENTER(__func__);
	QL4PRINT(QLP3, printk("scsi%d: %s:\n", ha->host_no, __func__));
#if 1 || defined(__VMKERNEL_MODULE__)
	down(&ha->dma_buf_sem);
#endif

	if (qla4xxx_get_ioctl_scrap_mem(ha, (void **)&pchip_info,
	    sizeof(EXT_CHIP_INFO))) {
		/* not enough memory */
		ioctl->Status = EXT_STATUS_NO_MEMORY;
		QL4PRINT(QLP2|QLP4,
		    printk("%s(%d): inst=%d scrap not big enough. "
		    "size requested=%ld.\n",
		    __func__, ha->host_no, ha->instance,
		    (ulong)sizeof(EXT_CHIP_INFO)));
		goto exit_query_chip;
	}

	if (!ioctl->ResponseAdr || ioctl->ResponseLen < sizeof(*pchip_info)) {
		QL4PRINT(QLP2,
			 printk("scsi%d: %s: response buffer too small\n",
				ha->host_no, __func__));

		ioctl->Status = EXT_STATUS_BUFFER_TOO_SMALL;
		goto exit_query_chip;
	}

	/* Fill in structure */
	memset(pchip_info, 0, sizeof(*pchip_info));

	/* ----- Get flash sys info information ---- */
#if 0 && defined(__VMKERNEL_MODULE__)
	spin_lock(&ha->dma_buf_spinlock);
#endif
	sys_info = (FLASH_SYS_INFO *) ha->dma_buf.virt_addr;

	memset(mbox_cmd, 0, sizeof(mbox_cmd));
	memset(mbox_sts, 0, sizeof(mbox_sts));
	mbox_cmd[0] = MBOX_CMD_READ_FLASH;
	mbox_cmd[1] = LSDW(ha->dma_buf.phys_addr);
	mbox_cmd[2] = MSDW(ha->dma_buf.phys_addr);
	mbox_cmd[3] = INT_ISCSI_SYSINFO_FLASH_OFFSET;
	mbox_cmd[4] = sizeof(*sys_info);

	if (qla4xxx_mailbox_command(ha, 5, 2, &mbox_cmd[0], &mbox_sts[0]) !=
	    QLA_SUCCESS) {
		QL4PRINT(QLP2,
			 printk("scsi%d: %s: MBOX_CMD_READ_FLASH failed "
				"w/ status %04X\n",
				ha->host_no, __func__, mbox_sts[0]));

#if 0 && defined(__VMKERNEL_MODULE__)
		spin_unlock(&ha->dma_buf_spinlock);
#endif
		ioctl->Status = EXT_STATUS_MAILBOX;
		ioctl->DetailStatus = mbox_sts[0];
		goto exit_query_chip;
	}

	pchip_info->VendorId    = le32_to_cpu(sys_info->pciDeviceVendor);
	pchip_info->DeviceId    = le32_to_cpu(sys_info->pciDeviceId);
	pchip_info->SubVendorId = le32_to_cpu(sys_info->pciSubsysVendor);
	pchip_info->SubSystemId = le32_to_cpu(sys_info->pciSubsysId);
#if 0 && defined(__VMKERNEL_MODULE__)
	spin_unlock(&ha->dma_buf_spinlock);
#endif

	/* ----- Get firmware state information ---- */
	memset(mbox_cmd, 0, sizeof(mbox_cmd));
	memset(mbox_sts, 0, sizeof(mbox_sts));
	mbox_cmd[0] = MBOX_CMD_GET_FW_STATE;
	if (qla4xxx_mailbox_command(ha, 1, 4, &mbox_cmd[0], &mbox_sts[0]) !=
	    QLA_SUCCESS) {
		QL4PRINT(QLP2,
			 printk("scsi%d: %s: MBOX_CMD_GET_FW_STATE failed "
				"w/ status %04X\n",
				ha->host_no, __func__, mbox_sts[0]));

		ioctl->Status = EXT_STATUS_MAILBOX;
		ioctl->DetailStatus = mbox_sts[0];
		goto exit_query_chip;
	}

	pchip_info->BoardID     = mbox_sts[2];

	/* Copy info to caller */
	if (copy_to_user(Q64BIT_TO_PTR(ioctl->ResponseAdr),
	    pchip_info, sizeof(*pchip_info)) != 0) {
		status = (-EFAULT);
		ioctl->Status = EXT_STATUS_COPY_ERR;
		goto exit_query_chip;
	}

	ioctl->Status = EXT_STATUS_OK;

exit_query_chip:
	qla4xxx_free_ioctl_scrap_mem(ha);

#if 1 || defined(__VMKERNEL_MODULE__)
	up(&ha->dma_buf_sem);
#endif
	LEAVE(__func__);

	return(status);
}

/**************************************************************************
 * qla4extioctl_query
 *	This routine calls query IOCTLs based on the IOCTL Sub Code.
 *
 * Input:
 *	ha    = adapter structure pointer.
 *	ioctl = IOCTL structure pointer.
 *
 * Output:
 *	None
 *
 * Returns:
 *	QLA_SUCCESS = success
 *	QLA_ERROR   = error
 *    	-EINVAL     = if the command is invalid
 *
 * Context:
 *	Kernel context.
 **************************************************************************/
static int
qla4extioctl_query(scsi_qla_host_t *ha, EXT_IOCTL_ISCSI *ioctl)
{
	switch (ioctl->SubCode) {
	case EXT_SC_QUERY_HBA_ISCSI_NODE:
		return(qla4extioctl_query_hba_iscsi_node(ha, ioctl));

	case EXT_SC_QUERY_HBA_ISCSI_PORTAL:
		return(qla4extioctl_query_hba_iscsi_portal(ha, ioctl));

	case EXT_SC_QUERY_DISC_ISCSI_NODE:
		return(qla4extioctl_query_disc_iscsi_node(ha, ioctl));

	case EXT_SC_QUERY_DISC_ISCSI_PORTAL:
		return(qla4extioctl_query_disc_iscsi_portal(ha, ioctl));

	case EXT_SC_QUERY_DRIVER:
		return(qla4extioctl_query_driver(ha, ioctl));

	case EXT_SC_QUERY_FW:
		return(qla4extioctl_query_fw(ha, ioctl));

	case EXT_SC_QUERY_CHIP:
		return(qla4extioctl_query_chip(ha, ioctl));

	default:
		QL4PRINT(QLP2,
			 printk("scsi%d: %s: unsupported query sub-command "
				"code (%X)\n",
				ha->host_no, __func__, ioctl->SubCode));

		ioctl->Status = EXT_STATUS_UNSUPPORTED_SUBCODE;
		break;
	}
	return(0);
}

/**************************************************************************
 * qla4extioctl_reg_aen
 *	This routine enables/disables storing of asynchronous events
 *	from the ISP into the driver's internal buffer.
 *
 * Input:
 *	ha    = adapter structure pointer.
 *	ioctl = IOCTL structure pointer.
 *
 * Output:
 *	None
 *
 * Returns:
 *	QLA_SUCCESS = success
 *	QLA_ERROR   = error
 *
 * Context:
 *	Kernel context.
 **************************************************************************/
static int
qla4extioctl_reg_aen(scsi_qla_host_t *ha, EXT_IOCTL_ISCSI *ioctl)
{
	int	status = 0;

	ENTER(__func__);

	ioctl->Status = EXT_STATUS_UNSUPPORTED_SUBCODE;
	QL4PRINT(QLP4,
	    printk("scsi%d: %s: UNSUPPORTED\n", ha->host_no, __func__));

	LEAVE(__func__);

	return(status);
}

/**************************************************************************
 * qla4extioctl_get_aen
 *	This routine retrieves the contents of the driver's internal
 *	asynchronous event tracking queue.
 *
 * Input:
 *	ha    = adapter structure pointer.
 *	ioctl = IOCTL structure pointer.
 *
 * Output:
 *	None
 *
 * Returns:
 *	QLA_SUCCESS = success
 *	QLA_ERROR   = error
 *
 * Context:
 *	Kernel context.
 **************************************************************************/
static int
qla4extioctl_get_aen(scsi_qla_host_t *ha, EXT_IOCTL_ISCSI *ioctl)
{
	int	status = 0;

	ENTER(__func__);

	ioctl->Status = EXT_STATUS_UNSUPPORTED_SUBCODE;
	QL4PRINT(QLP4,
	    printk("scsi%d: %s: UNSUPPORTED\n", ha->host_no, __func__));

	LEAVE(__func__);

	return(status);
}

/**************************************************************************
 * qla4extioctl_get_statistics_gen
 *	This routine retrieves the HBA general statistical information.
 *
 * Input:
 *	ha    = adapter structure pointer.
 *	ioctl = IOCTL structure pointer.
 *
 * Output:
 *	None
 *
 * Returns:
 *	QLA_SUCCESS = success
 *	QLA_ERROR   = error
 *
 * Context:
 *	Kernel context.
 **************************************************************************/
static int
qla4extioctl_get_statistics_gen(scsi_qla_host_t *ha, EXT_IOCTL_ISCSI *ioctl)
{
	int			status = 0;
	EXT_HBA_PORT_STAT_GEN	*pstat_gen = NULL;

	ENTER("qla4extioctl_get_statistics_gen");
	QL4PRINT(QLP4, printk("scsi%d: %s: index [%d]\n",
			      ha->host_no, __func__, ioctl->Instance));

	if (qla4xxx_get_ioctl_scrap_mem(ha, (void **)&pstat_gen,
	    sizeof(EXT_HBA_PORT_STAT_GEN))) {
		/* not enough memory */
		ioctl->Status = EXT_STATUS_NO_MEMORY;
		QL4PRINT(QLP2|QLP4,
		    printk("%s(%d): inst=%d scrap not big enough. "
		    "size requested=%ld.\n",
		    __func__, ha->host_no, ha->instance,
		    (ulong)sizeof(EXT_HBA_PORT_STAT_GEN)));
		goto exit_get_stat_gen;
	}

	if (ioctl->ResponseLen < sizeof(*pstat_gen)) {
		QL4PRINT(QLP2, printk("scsi%d: %s: memory area too small\n",
				      ha->host_no, __func__));

		ioctl->Status = EXT_STATUS_BUFFER_TOO_SMALL;
		goto exit_get_stat_gen;
	}

	/*
	 * Fill in the data
	 */
	memset(pstat_gen, 0, sizeof(*pstat_gen));
	pstat_gen->HBAPortErrorCount     = ha->adapter_error_count;
	pstat_gen->DevicePortErrorCount  = ha->device_error_count;
	pstat_gen->IoCount               = ha->total_io_count;
	pstat_gen->MBytesCount           = ha->total_mbytes_xferred;
	pstat_gen->InterruptCount        = ha->isr_count;
	pstat_gen->LinkFailureCount      = ha->link_failure_count;
	pstat_gen->InvalidCrcCount       = ha->invalid_crc_count;

	/*
	 * Copy the IOCTL EXT_HBA_PORT_STAT_GEN buffer to the user's data space
	 */
	if (copy_to_user(Q64BIT_TO_PTR(ioctl->ResponseAdr),
	    pstat_gen, ioctl->ResponseLen) != 0) {
		QL4PRINT(QLP2, printk("scsi%d: %s: memory area too small\n",
				      ha->host_no, __func__));

		status = (-EFAULT);
		ioctl->Status = EXT_STATUS_COPY_ERR;
		goto exit_get_stat_gen;
	}

	ioctl->Status = EXT_STATUS_OK;

	exit_get_stat_gen:
	qla4xxx_free_ioctl_scrap_mem(ha);

	LEAVE(__func__);

	return(status);
}

/**************************************************************************
 * qla4extioctl_get_statistics_iscsi
 *	This routine retrieves the HBA iSCSI statistical information.
 *
 * Input:
 *	ha    = adapter structure pointer.
 *	ioctl = IOCTL structure pointer.
 *
 * Output:
 *	None
 *
 * Returns:
 *	QLA_SUCCESS = success
 *	QLA_ERROR   = error
 *
 * Context:
 *	Kernel context.
 **************************************************************************/
static int
qla4extioctl_get_statistics_iscsi(scsi_qla_host_t *ha, EXT_IOCTL_ISCSI *ioctl)
{
	int		status = 0;
	uint32_t	mbox_cmd[MBOX_REG_COUNT];
	uint32_t	mbox_sts[MBOX_REG_COUNT];
	EXT_HBA_PORT_STAT_ISCSI* pstat_local;
	EXT_HBA_PORT_STAT_ISCSI* pstat_user;


	ENTER("qla4extioctl_get_statistics_iscsi");
	QL4PRINT(QLP4, printk("scsi%d: %s: index [%d]\n",
	    ha->host_no, __func__, ioctl->Instance));

#if 0 && defined(__VMKERNEL_MODULE__)
	spin_lock(&ha->dma_buf_spinlock);
#else
	down(&ha->dma_buf_sem);
#endif

	if (qla4xxx_get_ioctl_scrap_mem(ha, (void **)&pstat_user,
	    sizeof(EXT_HBA_PORT_STAT_ISCSI))) {
		/* not enough memory */
		ioctl->Status = EXT_STATUS_NO_MEMORY;
		QL4PRINT(QLP2|QLP4,
		    printk("%s(%d): inst=%d scrap not big enough. "
		    "size requested=%ld.\n",
		    __func__, ha->host_no, ha->instance,
		    (ulong)sizeof(EXT_HBA_PORT_STAT_ISCSI)));
		goto exit_get_stats_iscsi;
	}

	if (!ioctl->ResponseAdr || !ioctl->ResponseLen) {
		QL4PRINT(QLP2, printk("scsi%d: %s: invalid parameter\n",
		    ha->host_no, __func__));

		ioctl->Status = EXT_STATUS_INVALID_PARAM;
		goto exit_get_stats_iscsi;
	}

	if (ioctl->ResponseLen < sizeof(EXT_HBA_PORT_STAT_ISCSI)) {
		QL4PRINT(QLP2, printk("scsi%d: %s: RespLen too small (0x%x),  "
		    "need (0x%x).\n",
		    ha->host_no, __func__, ioctl->ResponseLen,
		    (unsigned int) sizeof(EXT_HBA_PORT_STAT_ISCSI)));

		ioctl->Status = EXT_STATUS_BUFFER_TOO_SMALL;
		goto exit_get_stats_iscsi;
	}

	if ((ha->dma_buf.buf_len < sizeof(EXT_HBA_PORT_STAT_ISCSI)) &&
	    (qla4xxx_resize_dma_buf(ha, &ha->dma_buf,
	    sizeof(EXT_HBA_PORT_STAT_ISCSI)) != QLA_SUCCESS)) {
		QL4PRINT(QLP2, printk("scsi%d: %s: unable to allocate memory "
		    "for dma buffer.\n",
		    ha->host_no, __func__));

		ioctl->Status = EXT_STATUS_NO_MEMORY;
		goto exit_get_stats_iscsi;
	}

	/*
	 * Make the mailbox call
	 */
	memset(mbox_cmd, 0, sizeof(mbox_cmd));
	memset(mbox_sts, 0, sizeof(mbox_sts));
	mbox_cmd[0] = MBOX_CMD_GET_MANAGEMENT_DATA;
	mbox_cmd[1] = ioctl->Instance;
	mbox_cmd[2] = LSDW(ha->dma_buf.phys_addr);
	mbox_cmd[3] = MSDW(ha->dma_buf.phys_addr);

	if (qla4xxx_mailbox_command(ha, 4, 1, &mbox_cmd[0], &mbox_sts[0]) !=
	    QLA_SUCCESS) {
		QL4PRINT(QLP2,
		    printk("scsi%d: %s: get mngmt data for index [%d] failed "
		    "w/ mailbox ststus 0x%x\n",
		    ha->host_no, __func__, ioctl->Instance, mbox_sts[0]));

		ioctl->Status = EXT_STATUS_MAILBOX;
		ioctl->DetailStatus = mbox_sts[0];
		goto exit_get_stats_iscsi;
	}

	pstat_local = (EXT_HBA_PORT_STAT_ISCSI *) ha->dma_buf.virt_addr;
	memset(pstat_user, 0, sizeof(*pstat_user));
	pstat_user->MACTxFramesCount          =
	    le64_to_cpu(pstat_local->MACTxFramesCount);
	pstat_user->MACTxBytesCount           =
	    le64_to_cpu(pstat_local->MACTxBytesCount);
	pstat_user->MACRxFramesCount          =
	    le64_to_cpu(pstat_local->MACRxFramesCount);
	pstat_user->MACRxBytesCount           =
	    le64_to_cpu(pstat_local->MACRxBytesCount);
	pstat_user->MACCRCErrorCount          =
	    le64_to_cpu(pstat_local->MACCRCErrorCount);
	pstat_user->MACEncodingErrorCount     =
	    le64_to_cpu(pstat_local->MACEncodingErrorCount);
	pstat_user->IPTxPacketsCount          =
	    le64_to_cpu(pstat_local->IPTxPacketsCount);
	pstat_user->IPTxBytesCount            =
	    le64_to_cpu(pstat_local->IPTxBytesCount);
	pstat_user->IPTxFragmentsCount        =
	    le64_to_cpu(pstat_local->IPTxFragmentsCount);
	pstat_user->IPRxPacketsCount          =
	    le64_to_cpu(pstat_local->IPRxPacketsCount);
	pstat_user->IPRxBytesCount            =
	    le64_to_cpu(pstat_local->IPRxBytesCount);
	pstat_user->IPRxFragmentsCount        =
	    le64_to_cpu(pstat_local->IPRxFragmentsCount);
	pstat_user->IPDatagramReassemblyCount =
	    le64_to_cpu(pstat_local->IPDatagramReassemblyCount);
	pstat_user->IPv6RxPacketsCount        =
	    le64_to_cpu(pstat_local->IPv6RxPacketsCount);
	pstat_user->IPRxPacketErrorCount      =
	    le64_to_cpu(pstat_local->IPRxPacketErrorCount);
	pstat_user->IPReassemblyErrorCount    =
	    le64_to_cpu(pstat_local->IPReassemblyErrorCount);
	pstat_user->TCPTxSegmentsCount        =
	    le64_to_cpu(pstat_local->TCPTxSegmentsCount);
	pstat_user->TCPTxBytesCount           =
	    le64_to_cpu(pstat_local->TCPTxBytesCount);
	pstat_user->TCPRxSegmentsCount        =
	    le64_to_cpu(pstat_local->TCPRxSegmentsCount);
	pstat_user->TCPRxBytesCount           =
	    le64_to_cpu(pstat_local->TCPRxBytesCount);
	pstat_user->TCPTimerExpiredCount      =
	    le64_to_cpu(pstat_local->TCPTimerExpiredCount);
	pstat_user->TCPRxACKCount             =
	    le64_to_cpu(pstat_local->TCPRxACKCount);
	pstat_user->TCPTxACKCount             =
	    le64_to_cpu(pstat_local->TCPTxACKCount);
	pstat_user->TCPRxErrorSegmentCount    =
	    le64_to_cpu(pstat_local->TCPRxErrorSegmentCount);
	pstat_user->TCPWindowProbeUpdateCount =
	    le64_to_cpu(pstat_local->TCPWindowProbeUpdateCount);
	pstat_user->iSCSITxPDUCount           =
	    le64_to_cpu(pstat_local->iSCSITxPDUCount);
	pstat_user->iSCSITxBytesCount         =
	    le64_to_cpu(pstat_local->iSCSITxBytesCount);
	pstat_user->iSCSIRxPDUCount           =
	    le64_to_cpu(pstat_local->iSCSIRxPDUCount);
	pstat_user->iSCSIRxBytesCount         =
	    le64_to_cpu(pstat_local->iSCSIRxBytesCount);
	pstat_user->iSCSICompleteIOsCount     =
	    le64_to_cpu(pstat_local->iSCSICompleteIOsCount);
	pstat_user->iSCSIUnexpectedIORxCount  =
	    le64_to_cpu(pstat_local->iSCSIUnexpectedIORxCount);
	pstat_user->iSCSIFormatErrorCount     =
	    le64_to_cpu(pstat_local->iSCSIFormatErrorCount);
	pstat_user->iSCSIHeaderDigestCount    =
	    le64_to_cpu(pstat_local->iSCSIHeaderDigestCount);
	pstat_user->iSCSIDataDigestErrorCount =
	    le64_to_cpu(pstat_local->iSCSIDataDigestErrorCount);
	pstat_user->iSCSISeqErrorCount        =
	    le64_to_cpu(pstat_local->iSCSISeqErrorCount);

	/*
	 * Copy the data from the dma buffer to the user's data space
	 */
	if (copy_to_user(Q64BIT_TO_PTR(ioctl->ResponseAdr),
	    pstat_user, ioctl->ResponseLen) != 0) {
		QL4PRINT(QLP2, printk("scsi%d: %s: memory area too small\n",
		    ha->host_no, __func__));

		status = (-EFAULT);
		ioctl->Status = EXT_STATUS_COPY_ERR;
		goto exit_get_stats_iscsi;
	}

	ioctl->Status = EXT_STATUS_OK;

	exit_get_stats_iscsi:
	qla4xxx_free_ioctl_scrap_mem(ha);
#if 0 && defined(__VMKERNEL_MODULE__)
	spin_unlock(&ha->dma_buf_spinlock);
#else
	up(&ha->dma_buf_sem);
#endif

	LEAVE(__func__);
	return(status);
}

/**************************************************************************
 * qla4extioctl_get_device_entry_iscsi
 *	This routine retrieves the database entry for the specified device.
 *
 * Input:
 *	ha    = adapter structure pointer.
 *	ioctl = IOCTL structure pointer.
 *
 * Output:
 *	None
 *
 * Returns:
 *	QLA_SUCCESS = success
 *	QLA_ERROR   = error
 *
 * Context:
 *	Kernel context.
 **************************************************************************/
static int
qla4extioctl_get_device_entry_iscsi(scsi_qla_host_t *ha, EXT_IOCTL_ISCSI *ioctl)
{
	int                     status = 0;
	uint32_t		mbox_cmd[MBOX_REG_COUNT];
	uint32_t		mbox_sts[MBOX_REG_COUNT];
	DEV_DB_ENTRY		*pfw_ddb_entry;
	EXT_DEVICE_ENTRY_ISCSI	*pdev_entry;


	ENTER(__func__);
	QL4PRINT(QLP4, printk("scsi%d: %s: index [%d]\n",
			      ha->host_no, __func__, ioctl->Instance));

#if 0 && defined(__VMKERNEL_MODULE__)
	spin_lock(&ha->dma_buf_spinlock);
#else
	down(&ha->dma_buf_sem);
#endif

	if (qla4xxx_get_ioctl_scrap_mem(ha, (void **)&pdev_entry,
	    sizeof(EXT_DEVICE_ENTRY_ISCSI))) {
		/* not enough memory */
		ioctl->Status = EXT_STATUS_NO_MEMORY;
		QL4PRINT(QLP2|QLP4,
		    printk("%s(%d): inst=%d scrap not big enough. "
		    "size requested=%ld.\n",
		    __func__, ha->host_no, ha->instance,
		    (ulong)sizeof(EXT_DEVICE_ENTRY_ISCSI)));
		goto exit_get_dev_entry;
	}

#if 0 && defined(__VMKERNEL_MODULE__)
	if (!ha->dma_buf.virt_addr || !ha->dma_buf.phys_addr ||
	    !ioctl->ResponseAdr) {
		QL4PRINT(QLP2,
			 printk("scsi%d: %s: memory allocation problem\n",
				ha->host_no, __func__));

		ioctl->Status = EXT_STATUS_NO_MEMORY;
		goto exit_get_dev_entry;
	}
#else
        /*
         * We check the phys and virt addresses below before resizing
         * our buffer.  We shouldn't bail out becasue they're unset
         * here.
         */
	if (!ioctl->ResponseAdr) {
		QL4PRINT(QLP2,
			 printk("scsi%d: %s: memory allocation problem\n",
				ha->host_no, __func__));
		ioctl->Status = EXT_STATUS_NO_MEMORY;
		ioctl->ResponseLen = 0;
		goto exit_get_dev_entry;
	}
#endif

	#if 0
	if (ioctl->ResponseLen < sizeof(*pdev_entry) ||
	    ha->dma_buf.buf_len < sizeof(*pfw_ddb_entry)) {
		QL4PRINT(QLP2, printk("scsi%d: %s: memory area too small\n",
				      ha->host_no, __func__));

		ioctl->Status = EXT_STATUS_BUFFER_TOO_SMALL;
		goto exit_get_dev_entry;
	}
	#endif
	if (!ha->dma_buf.virt_addr || !ha->dma_buf.phys_addr ||
	    (ha->dma_buf.buf_len < sizeof(*pfw_ddb_entry))) {
		if (qla4xxx_resize_dma_buf(ha, &ha->dma_buf,
		    sizeof(*pfw_ddb_entry)) != QLA_SUCCESS) {
			QL4PRINT(QLP2,
				 printk("scsi%d: %s: unable to allocate memory "
					"for dma buffer.\n",
					ha->host_no, __func__));

			ioctl->Status = EXT_STATUS_NO_MEMORY;
			ioctl->ResponseLen = 0;
			goto exit_get_dev_entry;
		}
	}


	/*
	 * Make the mailbox call
	 */
	memset(mbox_cmd, 0, sizeof(mbox_cmd));
	memset(mbox_sts, 0, sizeof(mbox_sts));
	memset(pdev_entry, 0, sizeof(*pdev_entry));
	if (ioctl->SubCode == EXT_SC_GET_DEVICE_ENTRY_ISCSI)
		mbox_cmd[0] = MBOX_CMD_GET_DATABASE_ENTRY;
	else
		mbox_cmd[0] = MBOX_CMD_GET_DATABASE_ENTRY_DEFAULTS;
	mbox_cmd[1] = ioctl->Instance;
	mbox_cmd[2] = LSDW(ha->dma_buf.phys_addr);
	mbox_cmd[3] = MSDW(ha->dma_buf.phys_addr);

	if (qla4xxx_mailbox_command(ha, 4, 5, &mbox_cmd[0], &mbox_sts[0]) !=
	    QLA_SUCCESS) {
		QL4PRINT(QLP2,
		    printk("scsi%d: %s: get ddb entry for index [%d] failed "
				      "w/ mailbox ststus 0x%x\n",
		    ha->host_no, __func__, ioctl->Instance, mbox_sts[0]));

		ioctl->Status = EXT_STATUS_MAILBOX;
		ioctl->DetailStatus = mbox_sts[0];
		goto exit_get_dev_entry;
	}

	/*
	 * Transfer data from Fw's DEV_DB_ENTRY buffer to IOCTL's
	 * EXT_DEVICE_ENTRY_ISCSI buffer
	 */
	pfw_ddb_entry = ha->dma_buf.virt_addr;

	pdev_entry->NumValid                     = mbox_sts[2];
	pdev_entry->NextValid                    = mbox_sts[3];
	pdev_entry->DeviceState                  = mbox_sts[4];
	pdev_entry->Options                      = pfw_ddb_entry->options;
	pdev_entry->Control                      = pfw_ddb_entry->control;
	pdev_entry->TargetSessID                 =
	    le16_to_cpu(pfw_ddb_entry->TSID);
	memcpy(pdev_entry->InitiatorSessID, pfw_ddb_entry->ISID,
	       sizeof(pfw_ddb_entry->ISID));

	pdev_entry->DeviceInfo.DeviceType        =
	    le16_to_cpu(EXT_DEF_ISCSI_REMOTE);
	pdev_entry->DeviceInfo.ExeThrottle       =
	    le16_to_cpu(pfw_ddb_entry->exeThrottle);
	pdev_entry->DeviceInfo.InitMarkerlessInt =
	    le16_to_cpu(pfw_ddb_entry->iSCSIMaxSndDataSegLen);
	pdev_entry->DeviceInfo.RetryCount        = pfw_ddb_entry->retryCount;
	pdev_entry->DeviceInfo.RetryDelay        = pfw_ddb_entry->retryDelay;
	pdev_entry->DeviceInfo.iSCSIOptions      =
	    le16_to_cpu(pfw_ddb_entry->iSCSIOptions);
	pdev_entry->DeviceInfo.TCPOptions        =
	    le16_to_cpu(pfw_ddb_entry->TCPOptions);
	pdev_entry->DeviceInfo.IPOptions         =
	    le16_to_cpu(pfw_ddb_entry->IPOptions);
	pdev_entry->DeviceInfo.MaxPDUSize        =
	    le16_to_cpu(pfw_ddb_entry->maxPDUSize);
	pdev_entry->DeviceInfo.FirstBurstSize    =
	    le16_to_cpu(pfw_ddb_entry->firstBurstSize);
	pdev_entry->DeviceInfo.LogoutMinTime     =
	    le16_to_cpu(pfw_ddb_entry->minTime2Wait);
	pdev_entry->DeviceInfo.LogoutMaxTime     =
	    le16_to_cpu(pfw_ddb_entry->maxTime2Retain);
	pdev_entry->DeviceInfo.MaxOutstandingR2T =
	    le16_to_cpu(pfw_ddb_entry->maxOutstndngR2T);
	pdev_entry->DeviceInfo.KeepAliveTimeout  =
	    le16_to_cpu(pfw_ddb_entry->keepAliveTimeout);
	pdev_entry->DeviceInfo.PortNumber        =
	    le16_to_cpu(pfw_ddb_entry->portNumber);
	pdev_entry->DeviceInfo.MaxBurstSize      =
	    le16_to_cpu(pfw_ddb_entry->maxBurstSize);
	pdev_entry->DeviceInfo.TaskMgmtTimeout   =
	    le16_to_cpu(pfw_ddb_entry->taskMngmntTimeout);
	pdev_entry->EntryInfo.PortalCount        = mbox_sts[2];
	pdev_entry->ExeCount                     =
	    le16_to_cpu(pfw_ddb_entry->exeCount);
	pdev_entry->DDBLink                      =
	    le16_to_cpu(pfw_ddb_entry->ddbLink);

	memcpy(pdev_entry->UserID, pfw_ddb_entry->userID,
	       sizeof(pdev_entry->UserID));
	memcpy(pdev_entry->Password, pfw_ddb_entry->password,
	       sizeof(pdev_entry->Password));

	memcpy(pdev_entry->DeviceInfo.TargetAddr, pfw_ddb_entry->targetAddr,
	       sizeof(pdev_entry->DeviceInfo.TargetAddr));
	memcpy(pdev_entry->EntryInfo.IPAddr.IPAddress, pfw_ddb_entry->ipAddr,
	       sizeof(pdev_entry->EntryInfo.IPAddr.IPAddress));
	memcpy(pdev_entry->EntryInfo.iSCSIName, pfw_ddb_entry->iscsiName,
	       sizeof(pdev_entry->EntryInfo.iSCSIName));
	memcpy(pdev_entry->EntryInfo.Alias, pfw_ddb_entry->iSCSIAlias,
	       sizeof(pdev_entry->EntryInfo.Alias));

	QL4PRINT(QLP10,
	    printk("scsi%d: DEV_DB_ENTRY structure:\n", ha->host_no));
	qla4xxx_dump_bytes(QLP10, pfw_ddb_entry, sizeof(*pfw_ddb_entry));
	QL4PRINT(QLP10,
	    printk("scsi%d: EXT_DEVICE_ENTRY_ISCSI structure:\n", ha->host_no));
	qla4xxx_dump_bytes(QLP10, pdev_entry, sizeof(*pdev_entry));

	/*
	 * Copy the IOCTL EXT_DEVICE_ENTRY_ISCSI buffer to the user's data space
	 */
	if (copy_to_user(Q64BIT_TO_PTR(ioctl->ResponseAdr),
	    pdev_entry, ioctl->ResponseLen) != 0) {
		QL4PRINT(QLP2, printk("scsi%d: %s: memory area too small\n",
				      ha->host_no, __func__));

		status = (-EFAULT);
		ioctl->Status = EXT_STATUS_COPY_ERR;
		goto exit_get_dev_entry;
	}

	ioctl->Status = EXT_STATUS_OK;

	exit_get_dev_entry:
	qla4xxx_free_ioctl_scrap_mem(ha);
#if 0 && defined(__VMKERNEL_MODULE__)
	spin_unlock(&ha->dma_buf_spinlock);
#else
	up(&ha->dma_buf_sem);
#endif

	LEAVE(__func__);
	return(status);
}


/**************************************************************************
 * qla4extioctl_get_init_fw_iscsi
 *	This routine retrieves the initialize firmware control block for
 *	the specified HBA.
 *
 * Input:
 *	ha    = adapter structure pointer.
 *	ioctl = IOCTL structure pointer.
 *
 * Output:
 *	None
 *
 * Returns:
 *	QLA_SUCCESS = success
 *	QLA_ERROR   = error
 *
 * Context:
 *	Kernel context.
 **************************************************************************/
static int
qla4extioctl_get_init_fw_iscsi(scsi_qla_host_t *ha, EXT_IOCTL_ISCSI *ioctl)
{
	int			status = 0;
	uint32_t		mbox_cmd[MBOX_REG_COUNT];
	uint32_t		mbox_sts[MBOX_REG_COUNT];
	EXT_INIT_FW_ISCSI	*pinit_fw = NULL;
	INIT_FW_CTRL_BLK	*pinit_fw_cb;


	ENTER(__func__);
	QL4PRINT(QLP3, printk("scsi%d: %s:\n", ha->host_no, __func__));

#if 0 && defined(__VMKERNEL_MODULE__)
	spin_lock(&ha->dma_buf_spinlock);
#else
	down(&ha->dma_buf_sem);
#endif

	if (qla4xxx_get_ioctl_scrap_mem(ha, (void **)&pinit_fw,
	    sizeof(EXT_INIT_FW_ISCSI))) {
		/* not enough memory */
		ioctl->Status = EXT_STATUS_NO_MEMORY;
		QL4PRINT(QLP2|QLP4,
		    printk("%s(%d): inst=%d scrap not big enough. "
		    "size requested=%ld.\n",
		    __func__, ha->host_no, ha->instance,
		    (ulong)sizeof(EXT_INIT_FW_ISCSI)));
		goto exit_get_init_fw;
	}

	if (!ha->dma_buf.virt_addr || !ha->dma_buf.phys_addr ||
	    (ha->dma_buf.buf_len < sizeof(INIT_FW_CTRL_BLK)) ||
	    (ioctl->ResponseLen < sizeof(EXT_INIT_FW_ISCSI))) {
		QL4PRINT(QLP2,
			 printk("scsi%d: %s: response buffer too small\n",
				ha->host_no, __func__));

		ioctl->Status = EXT_STATUS_BUFFER_TOO_SMALL;
		goto exit_get_init_fw;
	}

	/*
	 * Send mailbox command
	 */
	memset(mbox_cmd, 0, sizeof(mbox_cmd));
	memset(mbox_sts, 0, sizeof(mbox_sts));
	switch (ioctl->SubCode) {
	case EXT_SC_GET_INIT_FW_ISCSI:
		mbox_cmd[0] = MBOX_CMD_GET_INIT_FW_CTRL_BLOCK;
		break;
	case EXT_SC_GET_INIT_FW_DEFAULTS_ISCSI:
		mbox_cmd[0] = MBOX_CMD_GET_INIT_FW_CTRL_BLOCK_DEFAULTS;
		break;
	default:
		QL4PRINT(QLP2,
		    printk("scsi%d: %s: invalid subcode (0x%04X) speficied\n",
		    ha->host_no, __func__, ioctl->SubCode));

		ioctl->Status = EXT_STATUS_INVALID_PARAM;
		goto exit_get_init_fw;
	}

	mbox_cmd[1] = 0;
	mbox_cmd[2] = LSDW(ha->dma_buf.phys_addr);
	mbox_cmd[3] = MSDW(ha->dma_buf.phys_addr);

	if (qla4xxx_mailbox_command(ha, 4, 1, &mbox_cmd[0], &mbox_sts[0]) ==
	    QLA_ERROR) {
		QL4PRINT(QLP2, printk("scsi%d: %s: command failed \n",
		    ha->host_no, __func__));

		ioctl->Status = EXT_STATUS_MAILBOX;
		ioctl->DetailStatus = mbox_sts[0];
		goto exit_get_init_fw;
	}

	/*
	 * Transfer Data from DMA buffer to Local buffer
	 */
	pinit_fw_cb = (INIT_FW_CTRL_BLK *)ha->dma_buf.virt_addr;
	memset(pinit_fw, 0, sizeof(*pinit_fw));

	pinit_fw->Version         = pinit_fw_cb->Version;
	pinit_fw->FWOptions       = le16_to_cpu(pinit_fw_cb->FwOptions);
	pinit_fw->AddFWOptions    = le16_to_cpu(pinit_fw_cb->AddFwOptions);
	//FIXME: pinit_fw->WakeupThreshold = le16_to_cpu(pinit_fw_cb->WakeupThreshold);
	memcpy(pinit_fw->IPAddr.IPAddress, pinit_fw_cb->IPAddr,
	    MIN(sizeof(pinit_fw->IPAddr.IPAddress),
	    sizeof(pinit_fw_cb->IPAddr)));
	memcpy(pinit_fw->SubnetMask.IPAddress, pinit_fw_cb->SubnetMask,
	    MIN(sizeof(pinit_fw->SubnetMask.IPAddress),
	    sizeof(pinit_fw_cb->SubnetMask)));
	memcpy(pinit_fw->Gateway.IPAddress, pinit_fw_cb->GatewayIPAddr,
	    MIN(sizeof(pinit_fw->Gateway.IPAddress),
	    sizeof(pinit_fw_cb->GatewayIPAddr)));
	memcpy(pinit_fw->DNSConfig.IPAddr.IPAddress,
	    pinit_fw_cb->PriDNSIPAddr,
	    MIN(sizeof(pinit_fw->DNSConfig.IPAddr.IPAddress),
	    sizeof(pinit_fw_cb->PriDNSIPAddr)));
	memcpy(pinit_fw->Alias, pinit_fw_cb->Alias,
	    MIN(sizeof(pinit_fw->Alias),
	    sizeof(pinit_fw_cb->Alias)));
	memcpy(pinit_fw->iSCSIName, pinit_fw_cb->iSCSINameString,
	    MIN(sizeof(pinit_fw->iSCSIName),
	    sizeof(pinit_fw_cb->iSCSINameString)));

	pinit_fw->DeviceInfo.DeviceType        =
	    le16_to_cpu(EXT_DEF_ISCSI_LOCAL);
	pinit_fw->DeviceInfo.ExeThrottle       =
	    le16_to_cpu(pinit_fw_cb->ExecThrottle);
	pinit_fw->DeviceInfo.InitMarkerlessInt =
	    le16_to_cpu(pinit_fw_cb->InitMarkerlessInt);
	pinit_fw->DeviceInfo.RetryCount        = pinit_fw_cb->RetryCount;
	pinit_fw->DeviceInfo.RetryDelay        =
	    pinit_fw_cb->RetryDelay;
	pinit_fw->DeviceInfo.iSCSIOptions      =
	    le16_to_cpu(pinit_fw_cb->iSCSIOptions);
	pinit_fw->DeviceInfo.TCPOptions        =
	    le16_to_cpu(pinit_fw_cb->TCPOptions);
	pinit_fw->DeviceInfo.IPOptions         =
	    le16_to_cpu(pinit_fw_cb->IPOptions);
	pinit_fw->DeviceInfo.MaxPDUSize        =
	    le16_to_cpu(pinit_fw_cb->MaxPDUSize);
	pinit_fw->DeviceInfo.FirstBurstSize    =
	    le16_to_cpu(pinit_fw_cb->FirstBurstSize);
	pinit_fw->DeviceInfo.LogoutMinTime     =
	    le16_to_cpu(pinit_fw_cb->DefaultTime2Wait);
	pinit_fw->DeviceInfo.LogoutMaxTime     =
	    le16_to_cpu(pinit_fw_cb->DefaultTime2Retain);
	pinit_fw->DeviceInfo.LogoutMaxTime     =
	    le16_to_cpu(pinit_fw_cb->DefaultTime2Retain);
	pinit_fw->DeviceInfo.MaxOutstandingR2T =
	    le16_to_cpu(pinit_fw_cb->MaxOutStndngR2T);
	pinit_fw->DeviceInfo.KeepAliveTimeout  =
	    le16_to_cpu(pinit_fw_cb->KeepAliveTimeout);
	pinit_fw->DeviceInfo.PortNumber        =
	    le16_to_cpu(pinit_fw_cb->PortNumber);
	pinit_fw->DeviceInfo.MaxBurstSize      =
	    le16_to_cpu(pinit_fw_cb->MaxBurstSize);
	//pinit_fw->DeviceInfo.TaskMgmtTimeout   = pinit_fw_cb->T;
	memcpy(pinit_fw->DeviceInfo.TargetAddr, pinit_fw_cb->TargAddr,
	    EXT_DEF_ISCSI_TADDR_SIZE);

	/*
	 * Copy the local data to the user's buffer
	 */
	if (copy_to_user(Q64BIT_TO_PTR(ioctl->ResponseAdr),
	    pinit_fw, sizeof(*pinit_fw)) != 0) {
		QL4PRINT(QLP2,
		    printk("scsi%d: %s: unable to copy data to user's "
		    "memory area\n", ha->host_no, __func__));

		status = (-EFAULT);
		ioctl->Status = EXT_STATUS_COPY_ERR;
		goto exit_get_init_fw;
	}

	ioctl->Status = EXT_STATUS_OK;

	QL4PRINT(QLP10,
	    printk("scsi%d: EXT_INIT_FW_ISCSI structure:\n", ha->host_no));
	qla4xxx_dump_bytes(QLP10, pinit_fw, sizeof(*pinit_fw));

	exit_get_init_fw:
	qla4xxx_free_ioctl_scrap_mem(ha);
#if 0 && defined(__VMKERNEL_MODULE__)
	spin_unlock(&ha->dma_buf_spinlock);
#else
	up(&ha->dma_buf_sem);
#endif

	LEAVE(__func__);
	return(status);
}

#ifndef QLA4000
/**************************************************************************
 * qla4extioctl_get_isns_server
 *	This routine retrieves the iSNS server information.
 *
 * Input:
 *	ha    = adapter structure pointer.
 *	ioctl = IOCTL structure pointer.
 *
 * Output:
 *	None
 *
 * Returns:
 *	QLA_SUCCESS = success
 *	QLA_ERROR   = error
 *
 * Context:
 *	Kernel context.
 **************************************************************************/
static int
qla4extioctl_get_isns_server(scsi_qla_host_t *ha, EXT_IOCTL_ISCSI *ioctl)
{
	int			status = 0;
	uint32_t		mbox_cmd[MBOX_REG_COUNT];
	uint32_t		mbox_sts[MBOX_REG_COUNT];
	EXT_ISNS_SERVER		*pisns_server = NULL;
	FLASH_INIT_FW_CTRL_BLK	*pflash_init_fw_cb = NULL;


	ENTER(__func__);
#if 0 && defined(__VMKERNEL_MODULE__)
	spin_lock(&ha->dma_buf_spinlock);
#else
	down(&ha->dma_buf_sem);
#endif

	if (qla4xxx_get_ioctl_scrap_mem(ha, (void **)&pisns_server,
	    sizeof(EXT_ISNS_SERVER))) {
		/* not enough memory */
		ioctl->Status = EXT_STATUS_NO_MEMORY;
		QL4PRINT(QLP2|QLP4,
		    printk("%s(%d): inst=%d scrap not big enough. "
		    "size requested=%ld.\n",
		    __func__, ha->host_no, ha->instance,
		    (ulong)sizeof(EXT_ISNS_SERVER)));
		goto exit_get_isns_server;
	}

	if (ioctl->ResponseLen < sizeof(EXT_ISNS_SERVER)) {
		QL4PRINT(QLP2,
			 printk("scsi%d: %s: response buffer too small\n",
				ha->host_no, __func__));

		ioctl->Status = EXT_STATUS_BUFFER_TOO_SMALL;
		ioctl->ResponseLen = 0;
		goto exit_get_isns_server;
	}

	if (!ha->dma_buf.virt_addr || !ha->dma_buf.phys_addr ||
	    (ha->dma_buf.buf_len < sizeof(*pflash_init_fw_cb))) {
		if (qla4xxx_resize_dma_buf(ha, &ha->dma_buf,
		    sizeof(DEV_DB_ENTRY)) != QLA_SUCCESS) {
			QL4PRINT(QLP2,
			    printk("scsi%d: %s: unable to allocate memory "
			    "for dma buffer.\n",
			    ha->host_no, __func__));

			ioctl->Status = EXT_STATUS_NO_MEMORY;
			ioctl->ResponseLen = 0;
			goto exit_get_isns_server;
		}
	}

	/*
	 * First get Flash Initialize Firmware Control Block, so as not to
	 * destroy unaffected data
	 *----------------------------------------------------------------*/
	pflash_init_fw_cb = (FLASH_INIT_FW_CTRL_BLK *)ha->dma_buf.virt_addr;

	memset(mbox_cmd, 0, sizeof(mbox_cmd));
	memset(mbox_sts, 0, sizeof(mbox_sts));
	mbox_cmd[0] = MBOX_CMD_READ_FLASH;
	mbox_cmd[1] = LSDW(ha->dma_buf.phys_addr);
	mbox_cmd[2] = MSDW(ha->dma_buf.phys_addr);
	mbox_cmd[3] = INT_ISCSI_INITFW_FLASH_OFFSET;
	mbox_cmd[4] = sizeof(*pflash_init_fw_cb);

	if (qla4xxx_mailbox_command(ha, 5, 2, &mbox_cmd[0], &mbox_sts[0]) ==
	    QLA_ERROR) {
		QL4PRINT(QLP2,
		    printk("scsi%d: %s: READ_FLASH command failed \n",
		    ha->host_no, __func__));

		ioctl->Status = EXT_STATUS_MAILBOX;
		ioctl->DetailStatus = mbox_sts[0];
		ioctl->ResponseLen = 0;
		goto exit_get_isns_server;
	}

	QL4PRINT(QLP4, printk("scsi%d: %s: READ_FLASH command successful \n",
	    ha->host_no, __func__));

	/*
	 * Copy iSNS Server info to the isns_server structure
	 *---------------------------------------------------*/
	memset(pisns_server, 0, sizeof(*pisns_server));
	pisns_server->PerformiSNSDiscovery =
	    (cpu_to_le16(pflash_init_fw_cb->init_fw_cb.TCPOptions) & TOPT_ISNS_ENABLE) ?1:0;

	pisns_server->AutomaticiSNSDiscovery =
	    (cpu_to_le16(pflash_init_fw_cb->init_fw_cb.TCPOptions) &
	    TOPT_LEARN_ISNS_IP_ADDR_ENABLE) ? 1 : 0;
	pisns_server->PortNumber =
		cpu_to_le16(pflash_init_fw_cb->init_fw_cb.iSNSServerPortNumber);
	pisns_server->IPAddr.Type = EXT_DEF_TYPE_ISCSI_IP;
	memcpy(pisns_server->IPAddr.IPAddress,
	    pflash_init_fw_cb->init_fw_cb.iSNSIPAddr,
	    MIN(sizeof(pisns_server->IPAddr.IPAddress),
	    sizeof(pflash_init_fw_cb->init_fw_cb.iSNSIPAddr)));
	memcpy(pisns_server->InitiatorName,
	    pflash_init_fw_cb->init_fw_cb.iSCSINameString,
	    MIN(sizeof(pisns_server->InitiatorName),
	    sizeof(pflash_init_fw_cb->init_fw_cb.iSCSINameString)));

#if 1
	memset(mbox_cmd, 0, sizeof(mbox_cmd));
	memset(mbox_sts, 0, sizeof(mbox_sts));
	mbox_cmd[0] = MBOX_CMD_SET_ISNS_SERVICE;
	mbox_cmd[1] = ISNS_STATUS;

	if (qla4xxx_mailbox_command(ha, 2, 2, &mbox_cmd[0], &mbox_sts[0]) !=
	    QLA_SUCCESS) {
		QL4PRINT(QLP2,
		    printk("scsi%d: %s: GET ISNS SERVICE STATUS command "
		    "failed \n",
				      ha->host_no, __func__));
		ioctl->Status = EXT_STATUS_MAILBOX;
		ioctl->DetailStatus = mbox_sts[0];
		ioctl->ResponseLen = 0;
		goto exit_get_isns_server;
	}

	QL4PRINT(QLP7|QLP20,
	    printk("scsi%d: %s: GET ISNS SERVICE STATUS = 0x%04x \"%s\"\n",
	    ha->host_no, __func__, mbox_sts[1],
	    ((mbox_sts[1] & 1) == 0) ? "DISABLED" : "ENABLED"));
#endif

	/*
	 * Copy the local data to the user's buffer
	 *-----------------------------------------*/
	if (copy_to_user(Q64BIT_TO_PTR(ioctl->ResponseAdr),
	    pisns_server, sizeof(*pisns_server)) != 0) {
		QL4PRINT(QLP2,
			 printk("scsi%d: %s: unable to copy data to user's "
				"memory area\n", ha->host_no, __func__));

		status = (-EFAULT);
		ioctl->Status = EXT_STATUS_COPY_ERR;
		goto exit_get_isns_server;
	}

	ioctl->Status = EXT_STATUS_OK;
	ioctl->ResponseLen = sizeof(*pisns_server);
	ioctl->DetailStatus = 0;

	QL4PRINT(QLP10,
	    printk("scsi%d: EXT_ISNS_SERVER structure:\n", ha->host_no));
	qla4xxx_dump_bytes(QLP10, pisns_server, sizeof(*pisns_server));

	exit_get_isns_server:
	qla4xxx_free_ioctl_scrap_mem(ha);
#if 0 && defined(__VMKERNEL_MODULE__)
	spin_unlock(&ha->dma_buf_spinlock);
#else
	up(&ha->dma_buf_sem);
#endif

	LEAVE(__func__);
	return(status);
}

/**************************************************************************
 * qla4extioctl_get_isns_disc_targets
 *	This routine retrieves the targets discovered via iSNS.
 *
 * Input:
 *	ha    = adapter structure pointer.
 *	ioctl = IOCTL structure pointer.
 *
 * Output:
 *	None
 *
 * Returns:
 *	QLA_SUCCESS = success
 *	QLA_ERROR   = error
 *
 * Context:
 *	Kernel context.
 **************************************************************************/
static int
qla4extioctl_get_isns_disc_targets(scsi_qla_host_t *ha, EXT_IOCTL_ISCSI *ioctl)
{
	int			status = 0;
	EXT_ISNS_DISCOVERED_TARGETS *pisns_disc_tgts = NULL;
	uint32_t		isns_disc_tgt_index_start;
	uint32_t		i, j;


	ENTER(__func__);
#if 0 && defined(__VMKERNEL_MODULE__)
	spin_lock(&ha->dma_buf_spinlock);
#else
	down(&ha->dma_buf_sem);
#endif

	if (ioctl->ResponseLen < sizeof(EXT_ISNS_DISCOVERED_TARGETS)) {
		QL4PRINT(QLP2, printk("scsi%d: %s: response buffer "
				      "too small.  RspLen=0x%x, need 0x%x\n",
		    ha->host_no, __func__, ioctl->ResponseLen,
				      (unsigned int) sizeof(EXT_ISNS_DISCOVERED_TARGETS)));
		ioctl->ResponseLen = 0;
		ioctl->Status = EXT_STATUS_BUFFER_TOO_SMALL;
		ioctl->DetailStatus = sizeof(EXT_ISNS_DISCOVERED_TARGETS);
		goto exit_get_isns_disc_tgts;
	}

	if (!ha->dma_buf.virt_addr ||
	    ((ioctl->ResponseLen > ha->dma_buf.buf_len) &&
	    (qla4xxx_resize_dma_buf(ha, &ha->dma_buf,
	    sizeof(EXT_ISNS_DISCOVERED_TARGETS)) != QLA_SUCCESS))) {
		QL4PRINT(QLP2, printk("scsi%d: %s: unable to allocate memory "
				      "for dma buffer.\n",
				      ha->host_no, __func__));
		ioctl->ResponseLen = 0;
		ioctl->Status = EXT_STATUS_NO_MEMORY;
		goto exit_get_isns_disc_tgts;
	}

	/*
	 * Copy the IOCTL EXT_ISNS_DISCOVERED_TARGETS buffer from the user's
	 * data space
	 */
	pisns_disc_tgts = (EXT_ISNS_DISCOVERED_TARGETS *) ha->dma_buf.virt_addr;
	if (copy_from_user((uint8_t *)pisns_disc_tgts,
	    Q64BIT_TO_PTR(ioctl->RequestAdr), ioctl->RequestLen) != 0) {
		QL4PRINT(QLP2,
			 printk("scsi%d: %s: unable to copy data from user's "
				"memory area\n", ha->host_no, __func__));

		status = (-EFAULT);
		ioctl->ResponseLen = 0;
		ioctl->Status = EXT_STATUS_COPY_ERR;
		goto exit_get_isns_disc_tgts;
	}

	isns_disc_tgt_index_start =
	    pisns_disc_tgts->iSNSDiscoveredTargetIndexStart;
	memset(pisns_disc_tgts, 0, sizeof(*pisns_disc_tgts));
	pisns_disc_tgts->iSNSDiscoveredTargetIndexStart =
	    isns_disc_tgt_index_start;

	/*
	 * Transfer Data from Local buffer to DMA buffer
	 */
	if (isns_disc_tgt_index_start < ha->isns_num_discovered_targets) {
		EXT_ISNS_DISCOVERED_TARGET *isns_disc_tgt;
		ISNS_DISCOVERED_TARGET *isns_local_disc_target;

		for (i = isns_disc_tgt_index_start;
		    i < ha->isns_num_discovered_targets &&
		    pisns_disc_tgts->NumiSNSDiscoveredTargets <
		    EXT_DEF_NUM_ISNS_DISCOVERED_TARGETS;
		    i++) {

			isns_disc_tgt = (EXT_ISNS_DISCOVERED_TARGET *)
				&pisns_disc_tgts->iSNSDiscoveredTargets
				[pisns_disc_tgts->NumiSNSDiscoveredTargets];
			isns_local_disc_target = (ISNS_DISCOVERED_TARGET *)
				&ha->dma_mem_blkv->
				isns_discovered_target_database[i];

			isns_disc_tgt->NumPortals =
			    isns_local_disc_target->NumPortals;

			for (j = 0; j < isns_disc_tgt->NumPortals; j++) {
				memcpy(isns_disc_tgt->Portal[j].IPAddr.
				    IPAddress,
				       isns_local_disc_target->Portal[j].IPAddr,
				    MIN(sizeof(isns_disc_tgt->Portal[j].IPAddr.
				    IPAddress),
				    sizeof(isns_local_disc_target->Portal[j].
				    IPAddr)));
				isns_disc_tgt->Portal[j].IPAddr.Type =
				    EXT_DEF_TYPE_ISCSI_IP;
				isns_disc_tgt->Portal[j].PortNumber =
				    isns_local_disc_target->Portal[j].
				    PortNumber;
			}

			isns_disc_tgt->DDID = isns_local_disc_target->DDID;

			memcpy(isns_disc_tgt->NameString,
			       isns_local_disc_target->NameString,
			       MIN(sizeof(isns_disc_tgt->NameString),
				   sizeof(isns_local_disc_target->NameString)));
			memcpy(isns_disc_tgt->Alias,
			       isns_local_disc_target->Alias,
			       MIN(sizeof(isns_disc_tgt->Alias),
				   sizeof(isns_local_disc_target->Alias)));

			pisns_disc_tgts->NumiSNSDiscoveredTargets++;
		}
	}

	/*
	 * Copy the data to the user's buffer
	 */
	if (copy_to_user(Q64BIT_TO_PTR(ioctl->ResponseAdr),
	    pisns_disc_tgts, sizeof(*pisns_disc_tgts)) != 0) {
		QL4PRINT(QLP2,
			 printk("scsi%d: %s: unable to copy data to user's "
				"memory area\n", ha->host_no, __func__));

		status = (-EFAULT);
		ioctl->Status = EXT_STATUS_COPY_ERR;
		goto exit_get_isns_disc_tgts;
	}

	ioctl->Status = EXT_STATUS_OK;

	QL4PRINT(QLP10,
	    printk("scsi%d: EXT_INIT_FW_ISCSI structure:\n", ha->host_no));
	qla4xxx_dump_bytes(QLP10, pisns_disc_tgts, sizeof(*pisns_disc_tgts));

	exit_get_isns_disc_tgts:
	qla4xxx_free_ioctl_scrap_mem(ha);
#if 0 && defined(__VMKERNEL_MODULE__)
	spin_unlock(&ha->dma_buf_spinlock);
#else
	up(&ha->dma_buf_sem);
#endif

	LEAVE(__func__);
	return(status);
}
#endif

/**************************************************************************
 * qla4extioctl_get_data
 *	This routine calls get data IOCTLs based on the IOCTL Sub Code.
 *
 * Input:
 *	ha    = adapter structure pointer.
 *	ioctl = IOCTL structure pointer.
 *
 * Output:
 *	None
 *
 * Returns:
 *	QLA_SUCCESS = success
 *	QLA_ERROR   = error
 *    	-EINVAL     = if the command is invalid
 *
 * Context:
 *	Kernel context.
 **************************************************************************/
static int
qla4extioctl_get_data(scsi_qla_host_t *ha, EXT_IOCTL_ISCSI *ioctl)
{
	switch (ioctl->SubCode) {
	case EXT_SC_GET_STATISTICS_GEN:
		return(qla4extioctl_get_statistics_gen(ha, ioctl));

	case EXT_SC_GET_STATISTICS_ISCSI:
		return(qla4extioctl_get_statistics_iscsi(ha, ioctl));

	case EXT_SC_GET_DEVICE_ENTRY_ISCSI:
	case EXT_SC_GET_DEVICE_ENTRY_DEFAULTS_ISCSI:
		return(qla4extioctl_get_device_entry_iscsi(ha, ioctl));

	case EXT_SC_GET_INIT_FW_ISCSI:
	case EXT_SC_GET_INIT_FW_DEFAULTS_ISCSI:
		return(qla4extioctl_get_init_fw_iscsi(ha, ioctl));

#ifndef QLA4000
	case EXT_SC_GET_ISNS_SERVER:
		return(qla4extioctl_get_isns_server(ha, ioctl));

	case EXT_SC_GET_ISNS_DISCOVERED_TARGETS:
		return(qla4extioctl_get_isns_disc_targets(ha, ioctl));
#endif

	default:
		QL4PRINT(QLP2,
			 printk("scsi%d: %s: unsupported external get "
				"data sub-command code (%X)\n",
				ha->host_no, __func__, ioctl->SubCode));

		ioctl->Status = EXT_STATUS_UNSUPPORTED_SUBCODE;
		break;
	}
	return(0);
}

/**************************************************************************
 * qla4extioctl_rst_statistics_gen
 *	This routine clears the HBA general statistical information.
 *
 * Input:
 *	ha    = adapter structure pointer.
 *	ioctl = IOCTL structure pointer.
 *
 * Output:
 *	None
 *
 * Returns:
 *	QLA_SUCCESS = success
 *	QLA_ERROR   = error
 *
 * Context:
 *	Kernel context.
 **************************************************************************/
static int
qla4extioctl_rst_statistics_gen(scsi_qla_host_t *ha, EXT_IOCTL_ISCSI *ioctl)
{
	ENTER(__func__);
	QL4PRINT(QLP3, printk("scsi%d: %s:\n", ha->host_no, __func__));

	/*
	 * Reset the general statistics fields
	 */
	ha->adapter_error_count = 0;
	ha->device_error_count = 0;
	ha->total_io_count = 0;
	ha->total_mbytes_xferred = 0;
	ha->isr_count = 0;
	ha->link_failure_count = 0;
	ha->invalid_crc_count = 0;

	ioctl->Status = EXT_STATUS_OK;

	LEAVE(__func__);
	return(QLA_SUCCESS);
}

/**************************************************************************
 * qla4extioctl_rst_statistics_iscsi
 *	This routine clears the HBA iSCSI statistical information.
 *
 * Input:
 *	ha    = adapter structure pointer.
 *	ioctl = IOCTL structure pointer.
 *
 * Output:
 *	None
 *
 * Returns:
 *	QLA_SUCCESS = success
 *	QLA_ERROR   = error
 *
 * Context:
 *	Kernel context.
 **************************************************************************/
static int
qla4extioctl_rst_statistics_iscsi(scsi_qla_host_t *ha, EXT_IOCTL_ISCSI *ioctl)
{
	uint32_t	mbox_cmd[MBOX_REG_COUNT];
	uint32_t	mbox_sts[MBOX_REG_COUNT];


	ENTER(__func__);
	QL4PRINT(QLP4, printk("scsi%d: %s: index [%d]\n",
			      ha->host_no, __func__, ioctl->Instance));

	/*
	 * Make the mailbox call
	 */
	memset(mbox_cmd, 0, sizeof(mbox_cmd));
	memset(mbox_sts, 0, sizeof(mbox_sts));
	mbox_cmd[0] = MBOX_CMD_GET_MANAGEMENT_DATA;
	mbox_cmd[1] = ioctl->Instance;
	mbox_cmd[2] = 0;
	mbox_cmd[3] = 0;

	if (qla4xxx_mailbox_command(ha, 4, 1, &mbox_cmd[0], &mbox_sts[0])
	    != QLA_SUCCESS) {
		QL4PRINT(QLP2,
		    printk("scsi%d: %s: get mngmt data for index [%d] failed! "
		    "w/ mailbox ststus 0x%x\n",
		    ha->host_no, __func__,
		    ioctl->Instance, mbox_sts[0]));

		ioctl->Status = EXT_STATUS_MAILBOX;
		ioctl->DetailStatus = mbox_sts[0];
		LEAVE(__func__);
		return(0);
	}

	ioctl->Status = EXT_STATUS_OK;

	LEAVE(__func__);
	return(0);
}

/**************************************************************************
 * qla4extioctl_set_device_entry_iscsi
 *	This routine configures a device with specific database entry data.
 *
 * Input:
 *	ha    = adapter structure pointer.
 *	ioctl = IOCTL structure pointer.
 *
 * Output:
 *	None
 *
 * Returns:
 *	QLA_SUCCESS = success
 *	QLA_ERROR   = error
 *
 * Context:
 *	Kernel context.
 **************************************************************************/
static int
qla4extioctl_set_device_entry_iscsi(scsi_qla_host_t *ha, EXT_IOCTL_ISCSI *ioctl)
{
	int		status = 0;
	DEV_DB_ENTRY	*pfw_ddb_entry;
	dma_addr_t	fw_ddb_entry_dma;
	EXT_DEVICE_ENTRY_ISCSI *pdev_entry;


	ENTER(__func__);
	QL4PRINT(QLP4, printk("scsi%d: %s: index [%d]\n",
			      ha->host_no, __func__, ioctl->Instance));

#if 0 && defined(__VMKERNEL_MODULE__)
	spin_lock(&ha->dma_buf_spinlock);
#else
	down(&ha->dma_buf_sem);
#endif
	if (qla4xxx_get_ioctl_scrap_mem(ha, (void **)&pdev_entry,
	    sizeof(EXT_DEVICE_ENTRY_ISCSI))) {
		/* not enough memory */
		ioctl->Status = EXT_STATUS_NO_MEMORY;
		QL4PRINT(QLP2|QLP4,
		    printk("%s(%d): inst=%d scrap not big enough. "
		    "size requested=%ld.\n",
		    __func__, ha->host_no, ha->instance,
		    (ulong)sizeof(EXT_DEVICE_ENTRY_ISCSI)));
		goto exit_set_dev_entry;
	}

	if (!ha->dma_buf.virt_addr || !ha->dma_buf.phys_addr ||
	    !ioctl->RequestAdr) {
		QL4PRINT(QLP2,
			 printk("scsi%d: %s: memory allocation problem\n",
				ha->host_no, __func__));

		ioctl->Status = EXT_STATUS_NO_MEMORY;
		goto exit_set_dev_entry;
	}

	if (ioctl->RequestLen < sizeof(*pdev_entry) ||
	    ha->dma_buf.buf_len < sizeof(*pfw_ddb_entry)) {
		QL4PRINT(QLP2, printk("scsi%d: %s: memory area too small\n",
				      ha->host_no, __func__));

		ioctl->Status = EXT_STATUS_BUFFER_TOO_SMALL;
		goto exit_set_dev_entry;
	}

	/*
	 * Copy the IOCTL EXT_DEVICE_ENTRY_ISCSI buffer from the user's
	 * data space
	 */
	if (copy_from_user((uint8_t *)pdev_entry,
	    Q64BIT_TO_PTR(ioctl->RequestAdr), ioctl->RequestLen) != 0) {
		QL4PRINT(QLP2,
			 printk("scsi%d: %s: unable to copy data from user's "
				"memory area\n", ha->host_no, __func__));

		status = (-EFAULT);
		ioctl->Status = EXT_STATUS_COPY_ERR;
		goto exit_set_dev_entry;
	}

	/*
	 * Transfer data from IOCTL's EXT_DEVICE_ENTRY_ISCSI buffer to
	 * Fw's DEV_DB_ENTRY buffer
	 */
	pfw_ddb_entry = ha->dma_buf.virt_addr;
	fw_ddb_entry_dma = ha->dma_buf.phys_addr;
	memset(pfw_ddb_entry, 0, sizeof(*pfw_ddb_entry));

	pfw_ddb_entry->options          = pdev_entry->Options;
	pfw_ddb_entry->control          = pdev_entry->Control;
	pfw_ddb_entry->TSID             =
	    __cpu_to_le16(pdev_entry->TargetSessID);
	pfw_ddb_entry->exeCount         = __cpu_to_le16(pdev_entry->ExeCount);
	pfw_ddb_entry->ddbLink          = __cpu_to_le16(pdev_entry->DDBLink);
	memcpy(pfw_ddb_entry->ISID, pdev_entry->InitiatorSessID,
	    sizeof(pdev_entry->InitiatorSessID));
	memcpy(pfw_ddb_entry->userID, pdev_entry->UserID,
	    sizeof(pdev_entry->UserID));
	memcpy(pfw_ddb_entry->password, pdev_entry->Password,
	    sizeof(pdev_entry->Password));

	pfw_ddb_entry->exeThrottle      =
	    __cpu_to_le16(pdev_entry->DeviceInfo.ExeThrottle);
	pfw_ddb_entry->iSCSIMaxSndDataSegLen =
	    __cpu_to_le16(pdev_entry->DeviceInfo.InitMarkerlessInt);
	pfw_ddb_entry->retryCount       = pdev_entry->DeviceInfo.RetryCount;
	pfw_ddb_entry->retryDelay       = pdev_entry->DeviceInfo.RetryDelay;
	pfw_ddb_entry->iSCSIOptions     =
	    __cpu_to_le16(pdev_entry->DeviceInfo.iSCSIOptions);
	pfw_ddb_entry->TCPOptions       =
	    __cpu_to_le16(pdev_entry->DeviceInfo.TCPOptions);
	pfw_ddb_entry->IPOptions        =
	    __cpu_to_le16(pdev_entry->DeviceInfo.IPOptions);
	pfw_ddb_entry->maxPDUSize       =
	    __cpu_to_le16(pdev_entry->DeviceInfo.MaxPDUSize);
	pfw_ddb_entry->firstBurstSize   =
	    __cpu_to_le16(pdev_entry->DeviceInfo.FirstBurstSize);
	pfw_ddb_entry->minTime2Wait     =
	    __cpu_to_le16(pdev_entry->DeviceInfo.LogoutMinTime);
	pfw_ddb_entry->maxTime2Retain   =
	    __cpu_to_le16(pdev_entry->DeviceInfo.LogoutMaxTime);
	pfw_ddb_entry->maxOutstndngR2T  =
	    __cpu_to_le16(pdev_entry->DeviceInfo.MaxOutstandingR2T);
	pfw_ddb_entry->keepAliveTimeout =
	    __cpu_to_le16(pdev_entry->DeviceInfo.KeepAliveTimeout);
	pfw_ddb_entry->portNumber       =
	    __cpu_to_le16(pdev_entry->DeviceInfo.PortNumber);
	pfw_ddb_entry->maxBurstSize     =
	    __cpu_to_le16(pdev_entry->DeviceInfo.MaxBurstSize);
	pfw_ddb_entry->taskMngmntTimeout=
	    __cpu_to_le16(pdev_entry->DeviceInfo.TaskMgmtTimeout);
	memcpy(pfw_ddb_entry->targetAddr,
	    pdev_entry->DeviceInfo.TargetAddr,
	    sizeof(pdev_entry->DeviceInfo.TargetAddr));

	memcpy(pfw_ddb_entry->ipAddr,
	    pdev_entry->EntryInfo.IPAddr.IPAddress,
	    sizeof(pdev_entry->EntryInfo.IPAddr.IPAddress));
	memcpy(pfw_ddb_entry->iscsiName,
	    pdev_entry->EntryInfo.iSCSIName,
	    sizeof(pdev_entry->EntryInfo.iSCSIName));
	memcpy(pfw_ddb_entry->iSCSIAlias,
	    pdev_entry->EntryInfo.Alias,
	    sizeof(pdev_entry->EntryInfo.Alias));

	/*
	 * Make the IOCTL call
	 */
	if (qla4xxx_set_ddb_entry(ha, ioctl->Instance, pfw_ddb_entry, fw_ddb_entry_dma) !=
	    QLA_SUCCESS) {
		QL4PRINT(QLP2, printk("scsi%d: %s: SET DDB Entry failed\n",
				      ha->host_no, __func__));

		ioctl->Status = EXT_STATUS_ERR;
		goto exit_set_dev_entry;
	}

	ioctl->Status = EXT_STATUS_OK;

	exit_set_dev_entry:
	qla4xxx_free_ioctl_scrap_mem(ha);
#if 0 && defined(__VMKERNEL_MODULE__)
	spin_unlock(&ha->dma_buf_spinlock);
#else
	up(&ha->dma_buf_sem);
#endif

	LEAVE(__func__);
	return(status);
}

/**************************************************************************
 * qla4extioctl_set_init_fw_iscsi
 *	This routine configures a device with specific data entry data.
 *
 * Input:
 *	ha    = adapter structure pointer.
 *	ioctl = IOCTL structure pointer.
 *
 * Output:
 *	None
 *
 * Returns:
 *	QLA_SUCCESS = success
 *	QLA_ERROR   = error
 *
 * Context:
 *	Kernel context.
 **************************************************************************/
static int
qla4extioctl_set_init_fw_iscsi(scsi_qla_host_t *ha, EXT_IOCTL_ISCSI *ioctl)
{
	int			status = 0;
	EXT_INIT_FW_ISCSI	*pinit_fw = NULL;
	INIT_FW_CTRL_BLK	*pinit_fw_cb;
	uint32_t		mbox_cmd[MBOX_REG_COUNT];
	uint32_t		mbox_sts[MBOX_REG_COUNT];


	ENTER(__func__);
	QL4PRINT(QLP3, printk("scsi%d: %s:\n", ha->host_no, __func__));

#if 0 && defined(__VMKERNEL_MODULE__)
	spin_lock(&ha->dma_buf_spinlock);
#else
	down(&ha->dma_buf_sem);
#endif

	if (qla4xxx_get_ioctl_scrap_mem(ha, (void **)&pinit_fw,
	    sizeof(EXT_INIT_FW_ISCSI))) {
		/* not enough memory */
		ioctl->Status = EXT_STATUS_NO_MEMORY;
		QL4PRINT(QLP2|QLP4,
		    printk("%s(%d): inst=%d scrap not big enough. "
		    "size requested=%ld.\n",
		    __func__, ha->host_no, ha->instance,
		    (ulong)sizeof(EXT_INIT_FW_ISCSI)));
		goto exit_set_init_fw;
	}

	if (!ha->dma_buf.virt_addr || !ha->dma_buf.phys_addr ||
	    (ha->dma_buf.buf_len < sizeof(INIT_FW_CTRL_BLK)) ||
	    (ioctl->RequestLen < sizeof(EXT_INIT_FW_ISCSI))) {
		QL4PRINT(QLP2,
		    printk("scsi%d: %s: requst buffer too small\n",
		    ha->host_no, __func__));

		ioctl->Status = EXT_STATUS_BUFFER_TOO_SMALL;
		goto exit_set_init_fw;
	}

	/*
	 * Copy the data from the user's buffer
	 */
	if (copy_from_user((uint8_t *)pinit_fw,
	    Q64BIT_TO_PTR(ioctl->RequestAdr), sizeof(*pinit_fw)) != 0) {
		QL4PRINT(QLP2,
		    printk("scsi%d: %s: unable to copy data to user's "
		    "memory area\n", ha->host_no, __func__));

		status = (-EFAULT);
		ioctl->Status = EXT_STATUS_COPY_ERR;
		goto exit_set_init_fw;
	}

	/*
	 * First get Initialize Firmware Control Block, so as not to
	 * destroy unaffected data
	 */
	memset(mbox_cmd, 0, sizeof(mbox_cmd));
	memset(mbox_sts, 0, sizeof(mbox_sts));
	mbox_cmd[0] = MBOX_CMD_GET_INIT_FW_CTRL_BLOCK;
	mbox_cmd[2] = LSDW(ha->dma_buf.phys_addr);
	mbox_cmd[3] = MSDW(ha->dma_buf.phys_addr);

	if (qla4xxx_mailbox_command(ha, 4, 1, &mbox_cmd[0], &mbox_sts[0]) ==
	    QLA_ERROR) {
		QL4PRINT(QLP2, printk("scsi%d: %s: command failed \n",
		    ha->host_no, __func__));

		ioctl->Status = EXT_STATUS_MAILBOX;
		ioctl->DetailStatus = mbox_sts[0];
		goto exit_set_init_fw;
	}

	/*
	 * Transfer Data from Local buffer to DMA buffer
	 */
	pinit_fw_cb = (INIT_FW_CTRL_BLK *)ha->dma_buf.virt_addr;

	pinit_fw_cb->Version         = pinit_fw->Version;
	pinit_fw_cb->FwOptions       = __cpu_to_le16(pinit_fw->FWOptions);
	pinit_fw_cb->AddFwOptions    = __cpu_to_le16(pinit_fw->AddFWOptions);
	//FIXME: pinit_fw_cb->WakeupThreshold = __cpu_to_le16(pinit_fw->WakeupThreshold);
	memcpy(pinit_fw_cb->IPAddr, pinit_fw->IPAddr.IPAddress,
	    MIN(sizeof(pinit_fw_cb->IPAddr),
	    sizeof(pinit_fw->IPAddr.IPAddress)));
	memcpy(pinit_fw_cb->SubnetMask, pinit_fw->SubnetMask.IPAddress,
	    MIN(sizeof(pinit_fw_cb->SubnetMask),
	    sizeof(pinit_fw->SubnetMask.IPAddress)));
	memcpy(pinit_fw_cb->GatewayIPAddr, pinit_fw->Gateway.IPAddress,
	    MIN(sizeof(pinit_fw_cb->GatewayIPAddr),
	    sizeof(pinit_fw->Gateway.IPAddress)));
	memcpy(pinit_fw_cb->PriDNSIPAddr, pinit_fw->DNSConfig.IPAddr.IPAddress,
	    MIN(sizeof(pinit_fw_cb->PriDNSIPAddr),
	    sizeof(pinit_fw->DNSConfig.IPAddr.IPAddress)));
	memcpy(pinit_fw_cb->Alias, pinit_fw->Alias,
	    MIN(sizeof(pinit_fw_cb->Alias),
	    sizeof(pinit_fw->Alias)));
	memcpy(pinit_fw_cb->iSCSINameString, pinit_fw->iSCSIName,
	    MIN(sizeof(pinit_fw_cb->iSCSINameString),
	    sizeof(pinit_fw->iSCSIName)));

	pinit_fw_cb->ExecThrottle       =
	    __cpu_to_le16(pinit_fw->DeviceInfo.ExeThrottle);
	pinit_fw_cb->InitMarkerlessInt  =
	    __cpu_to_le16(pinit_fw->DeviceInfo.InitMarkerlessInt);
	pinit_fw_cb->RetryCount         = pinit_fw->DeviceInfo.RetryCount;
	pinit_fw_cb->RetryDelay         = pinit_fw->DeviceInfo.RetryDelay;
	pinit_fw_cb->iSCSIOptions       =
	    __cpu_to_le16(pinit_fw->DeviceInfo.iSCSIOptions);
	pinit_fw_cb->TCPOptions         =
	    __cpu_to_le16(pinit_fw->DeviceInfo.TCPOptions);
	pinit_fw_cb->IPOptions          =
	    __cpu_to_le16(pinit_fw->DeviceInfo.IPOptions);
	pinit_fw_cb->MaxPDUSize         =
	    __cpu_to_le16(pinit_fw->DeviceInfo.MaxPDUSize);
	pinit_fw_cb->FirstBurstSize     =
	    __cpu_to_le16(pinit_fw->DeviceInfo.FirstBurstSize);
	pinit_fw_cb->DefaultTime2Wait   =
	    __cpu_to_le16(pinit_fw->DeviceInfo.LogoutMinTime);
	pinit_fw_cb->DefaultTime2Retain =
	    __cpu_to_le16(pinit_fw->DeviceInfo.LogoutMaxTime);
	pinit_fw_cb->MaxOutStndngR2T    =
	    __cpu_to_le16(pinit_fw->DeviceInfo.MaxOutstandingR2T);
	pinit_fw_cb->KeepAliveTimeout   =
	    __cpu_to_le16(pinit_fw->DeviceInfo.KeepAliveTimeout);
	pinit_fw_cb->PortNumber         =
	    __cpu_to_le16(pinit_fw->DeviceInfo.PortNumber);
	pinit_fw_cb->MaxBurstSize       =
	    __cpu_to_le16(pinit_fw->DeviceInfo.MaxBurstSize);
	//pinit_fw_cb->?                = pinit_fw->DeviceInfo.TaskMgmtTimeout;
	memcpy(pinit_fw_cb->TargAddr, pinit_fw->DeviceInfo.TargetAddr,
	       EXT_DEF_ISCSI_TADDR_SIZE);

	/*
	 * Send mailbox command
	 */

	memset(mbox_cmd, 0, sizeof(mbox_cmd));
	memset(mbox_sts, 0, sizeof(mbox_sts));
	mbox_cmd[0] = MBOX_CMD_INITIALIZE_FIRMWARE;
	mbox_cmd[2] = LSDW(ha->dma_buf.phys_addr);
	mbox_cmd[3] = MSDW(ha->dma_buf.phys_addr);

	if (qla4xxx_mailbox_command(ha, 4, 1, &mbox_cmd[0], &mbox_sts[0]) ==
	    QLA_ERROR) {
		QL4PRINT(QLP2, printk("scsi%d: %s: command failed \n",
				      ha->host_no, __func__));

		ioctl->Status = EXT_STATUS_MAILBOX;
		ioctl->DetailStatus = mbox_sts[0];
		goto exit_set_init_fw;
	}

	ioctl->Status = EXT_STATUS_OK;

	QL4PRINT(QLP10,
	    printk("scsi%d: EXT_INIT_FW_ISCSI structure:\n", ha->host_no));
	qla4xxx_dump_bytes(QLP10, pinit_fw, sizeof(*pinit_fw));

	exit_set_init_fw:
	qla4xxx_free_ioctl_scrap_mem(ha);
#if 0 && defined(__VMKERNEL_MODULE__)
	spin_unlock(&ha->dma_buf_spinlock);
#else
	up(&ha->dma_buf_sem);
#endif

	LEAVE(__func__);
	return(status);
}

#ifndef QLA4000
/**************************************************************************
 * qla4extioctl_set_isns_server
 *	This routine retrieves the targets discovered via iSNS.
 *
 * Input:
 *	ha    = adapter structure pointer.
 *	ioctl = IOCTL structure pointer.
 *
 * Output:
 *	None
 *
 * Returns:
 *	QLA_SUCCESS = success
 *	QLA_ERROR   = error
 *
 * Context:
 *	Kernel context.
 **************************************************************************/
static int
qla4extioctl_set_isns_server(scsi_qla_host_t *ha, EXT_IOCTL_ISCSI *ioctl)
{
	int			status = 0;
	uint32_t		mbox_cmd[MBOX_REG_COUNT];
	uint32_t		mbox_sts[MBOX_REG_COUNT];
	EXT_ISNS_SERVER		*pisns_server = NULL;
	FLASH_INIT_FW_CTRL_BLK	*pflash_init_fw_cb;
	uint16_t	tcp_options;
	uint16_t	port_number;


	ENTER(__func__);
#if 0 && defined(__VMKERNEL_MODULE__)
	spin_lock(&ha->dma_buf_spinlock);
#else
	down(&ha->dma_buf_sem);
#endif

	if (qla4xxx_get_ioctl_scrap_mem(ha, (void **)&pisns_server,
	    sizeof(EXT_ISNS_SERVER))) {
		/* not enough memory */
		ioctl->Status = EXT_STATUS_NO_MEMORY;
		QL4PRINT(QLP2|QLP4,
		    printk("%s(%d): inst=%d scrap not big enough. "
		    "size requested=%ld.\n",
		    __func__, ha->host_no, ha->instance,
		    (ulong)sizeof(EXT_ISNS_SERVER)));
		goto exit_set_isns_svr;
	}

	if (ioctl->RequestLen < sizeof(*pisns_server)) {
		QL4PRINT(QLP2,
		    printk("scsi%d: %s: requst buffer too small (%d/%xh)\n",
		    ha->host_no, __func__, ioctl->RequestLen,
		    ioctl->RequestLen));

		ioctl->Status = EXT_STATUS_BUFFER_TOO_SMALL;
		ioctl->ResponseLen = 0;
		goto exit_set_isns_svr;
	}

	if (!ha->dma_buf.virt_addr || !ha->dma_buf.phys_addr ||
	    (ha->dma_buf.buf_len < sizeof(*pflash_init_fw_cb))) {
		if (qla4xxx_resize_dma_buf(ha, &ha->dma_buf,
		    sizeof(DEV_DB_ENTRY)) != QLA_SUCCESS) {
			QL4PRINT(QLP2,
			    printk("scsi%d: %s: unable to allocate memory "
			    "for dma buffer.\n",
			    ha->host_no, __func__));

			ioctl->Status = EXT_STATUS_NO_MEMORY;
			ioctl->ResponseLen = 0;
			goto exit_set_isns_svr;
		}
	}

	/*
	 * Copy iSNS Server info from the user's buffer
	 *---------------------------------------------*/
	if (copy_from_user((uint8_t *)pisns_server,
	    Q64BIT_TO_PTR(ioctl->RequestAdr), sizeof(*pisns_server)) != 0) {
		QL4PRINT(QLP2,
		    printk("scsi%d: %s: unable to copy data to user's "
		    "memory area\n", ha->host_no, __func__));

		status = (-EFAULT);
		ioctl->Status = EXT_STATUS_COPY_ERR;
		ioctl->ResponseLen = 0;
		goto exit_set_isns_svr;
	}

	QL4PRINT(QLP10,
	    printk("scsi%d: EXT_ISNS_SERVER structure:\n", ha->host_no));
	qla4xxx_dump_bytes(QLP10, pisns_server, sizeof(*pisns_server));

	/*
	 * First get Flash Initialize Firmware Control Block, so as not to
	 * destroy unaffected data
	 *----------------------------------------------------------------*/
	memset(mbox_cmd, 0, sizeof(mbox_cmd));
	memset(mbox_sts, 0, sizeof(mbox_sts));
	mbox_cmd[0] = MBOX_CMD_READ_FLASH;
	mbox_cmd[1] = LSDW(ha->dma_buf.phys_addr);
	mbox_cmd[2] = MSDW(ha->dma_buf.phys_addr);
	mbox_cmd[3] = INT_ISCSI_INITFW_FLASH_OFFSET;
	mbox_cmd[4] = sizeof(*pflash_init_fw_cb);

	if (qla4xxx_mailbox_command(ha, 5, 2, &mbox_cmd[0], &mbox_sts[0]) ==
	    QLA_ERROR) {
		QL4PRINT(QLP2,
		    printk("scsi%d: %s: READ_FLASH command failed \n",
		    ha->host_no, __func__));

		ioctl->Status = EXT_STATUS_MAILBOX;
		ioctl->DetailStatus = mbox_sts[0];
		ioctl->ResponseLen = 0;
		goto exit_set_isns_svr;
	}

	QL4PRINT(QLP4, printk("scsi%d: %s: READ_FLASH command successful \n",
	    ha->host_no, __func__));

	/*
	 * Copy iSNS Server info to the flash_init_fw_cb
	 *----------------------------------------------*/
	pflash_init_fw_cb = (FLASH_INIT_FW_CTRL_BLK *)ha->dma_buf.virt_addr;

	// convert a couple of variables used for comparisons
	tcp_options = le16_to_cpu(pflash_init_fw_cb->init_fw_cb.TCPOptions);
	port_number = le16_to_cpu(pflash_init_fw_cb->init_fw_cb.iSNSServerPortNumber);

	if (pisns_server->PerformiSNSDiscovery) {
		if (pisns_server->AutomaticiSNSDiscovery) {
			tcp_options |= TOPT_LEARN_ISNS_IP_ADDR_ENABLE;
			memset(pflash_init_fw_cb->init_fw_cb.iSNSIPAddr, 0,
			       sizeof(pflash_init_fw_cb->init_fw_cb.iSNSIPAddr));
		}
		else
		{
			tcp_options &= ~TOPT_LEARN_ISNS_IP_ADDR_ENABLE;
			memcpy(pflash_init_fw_cb->init_fw_cb.iSNSIPAddr,
			    pisns_server->IPAddr.IPAddress,
			    MIN(sizeof(pflash_init_fw_cb->init_fw_cb.iSNSIPAddr),
			    sizeof(pisns_server->IPAddr.IPAddress)));
		}

		port_number = EXT_DEF_ISNS_WELL_KNOWN_PORT;
		tcp_options |= TOPT_ISNS_ENABLE;
	} else {
		tcp_options &= ~TOPT_ISNS_ENABLE;
		memset(pflash_init_fw_cb->init_fw_cb.iSNSIPAddr, 0,
		    sizeof(pflash_init_fw_cb->init_fw_cb.iSNSIPAddr));
		port_number = 0;
	}

	QL4PRINT(QLP4, printk("scsi%d: %s: IPAddr %d.%d.%d.%d Port# %04d\n",
	    ha->host_no, __func__,
	    pflash_init_fw_cb->init_fw_cb.iSNSIPAddr[0],
	    pflash_init_fw_cb->init_fw_cb.iSNSIPAddr[1],
	    pflash_init_fw_cb->init_fw_cb.iSNSIPAddr[2],
	    pflash_init_fw_cb->init_fw_cb.iSNSIPAddr[3],
	    port_number));

	/*
	 * If the internal iSNS info is different from the pflash_init_fw_cb,
	 * flash it now.
	 *------------------------------------------------------------------*/
	if (((ha->tcp_options & TOPT_LEARN_ISNS_IP_ADDR_ENABLE) !=
	    (tcp_options & TOPT_LEARN_ISNS_IP_ADDR_ENABLE)) ||
	    (!IPAddrIsEqual(ha->isns_ip_address,
	    pflash_init_fw_cb->init_fw_cb.iSNSIPAddr)) ||
	    (ha->isns_server_port_number != port_number)) {

		pflash_init_fw_cb->init_fw_cb.TCPOptions = cpu_to_le16(tcp_options);
		pflash_init_fw_cb->init_fw_cb.iSNSServerPortNumber = cpu_to_le16(port_number);


		memset(mbox_cmd, 0, sizeof(mbox_cmd));
		memset(mbox_sts, 0, sizeof(mbox_sts));
		mbox_cmd[0] = MBOX_CMD_WRITE_FLASH;
		mbox_cmd[1] = LSDW(ha->dma_buf.phys_addr);
		mbox_cmd[2] = MSDW(ha->dma_buf.phys_addr);
		mbox_cmd[3] = INT_ISCSI_INITFW_FLASH_OFFSET;
		mbox_cmd[4] = sizeof(*pflash_init_fw_cb);
		mbox_cmd[5] = WRITE_FLASH_OPTION_COMMIT_DATA;

		if (qla4xxx_mailbox_command(ha, 6, 2, &mbox_cmd[0],
		    &mbox_sts[0]) == QLA_ERROR) {
			QL4PRINT(QLP2,
			    printk("scsi%d: %s: WRITE_FLASH command failed \n",
			    ha->host_no, __func__));

			ioctl->Status = EXT_STATUS_MAILBOX;
			ioctl->DetailStatus = mbox_sts[0];
			ioctl->ResponseLen = 0;
			goto exit_set_isns_svr;
		}

		QL4PRINT(QLP4,
			 printk("scsi%d: %s: WRITE_FLASH command successful \n",
				ha->host_no, __func__));
		QL4PRINT(QLP4,
		    printk("scsi%d: Init Fw Ctrl Blk\n", ha->host_no));
		qla4xxx_dump_bytes(QLP4, pflash_init_fw_cb,
		    sizeof(*pflash_init_fw_cb));

		/*
		 * Update internal iSNS info
		 */
		if (pisns_server->AutomaticiSNSDiscovery)
			ha->tcp_options |= TOPT_LEARN_ISNS_IP_ADDR_ENABLE;
		else
			ha->tcp_options	&= ~TOPT_LEARN_ISNS_IP_ADDR_ENABLE;

		memcpy(ha->isns_ip_address,
		    pflash_init_fw_cb->init_fw_cb.iSNSIPAddr,
		    MIN(sizeof(ha->isns_ip_address),
		    sizeof(pflash_init_fw_cb->init_fw_cb.iSNSIPAddr)));
		ha->isns_server_port_number = port_number;
	}

	/*
	 * Start or Stop iSNS Service accordingly, if needed.
	 *---------------------------------------------------*/
	//FIXME:
	if (test_bit(ISNS_FLAG_ISNS_ENABLED_IN_ISP, &ha->isns_flags)) {

		if (!IPAddrIsZero(ha->isns_ip_address) &&
		    ha->isns_server_port_number &&
		    (ha->tcp_options & TOPT_LEARN_ISNS_IP_ADDR_ENABLE) == 0) {

			uint32_t ip_addr;
			IPAddr2Uint32(ha->isns_ip_address, &ip_addr);

			if (qla4xxx_isns_reenable(ha, ip_addr,
			    ha->isns_server_port_number)) {

				QL4PRINT(QLP4,
				    printk("scsi%d: qla4xxx_isns_reenable "
				    "failed!\n", ha->host_no));
				ioctl->Status = EXT_STATUS_ERR;
				ioctl->DetailStatus = 0;
				ioctl->ResponseLen = 0;
				goto exit_set_isns_svr;
			}
		} else if (test_bit(ISNS_FLAG_ISNS_SRV_ENABLED,
		    &ha->isns_flags) && IPAddrIsZero(ha->isns_ip_address)) {
			qla4xxx_isns_disable(ha);
		}
	}

	/*
	 * Complete IOCTL successfully
	 *----------------------------*/
	ioctl->Status = EXT_STATUS_OK;
	ioctl->DetailStatus = 0;
	ioctl->ResponseLen = 0;

	exit_set_isns_svr:
	qla4xxx_free_ioctl_scrap_mem(ha);
#if 0 && defined(__VMKERNEL_MODULE__)
	spin_unlock(&ha->dma_buf_spinlock);
#else
	up(&ha->dma_buf_sem);
#endif

	LEAVE(__func__);
	return(status);
}
#endif

/**************************************************************************
 * qla4extioctl_set_data
 *	This routine calls set data IOCTLs based on the IOCTL Sub Code.
 *
 * Input:
 *	ha    = adapter structure pointer.
 *	ioctl = IOCTL structure pointer.
 *
 * Output:
 *	None
 *
 * Returns:
 *	QLA_SUCCESS = success
 *	QLA_ERROR   = error
 *    	-EINVAL     = if the command is invalid
 *
 * Context:
 *	Kernel context.
 **************************************************************************/
static int qla4extioctl_set_data(scsi_qla_host_t *ha, EXT_IOCTL_ISCSI *ioctl)
{
	switch (ioctl->SubCode) {
	case EXT_SC_RST_STATISTICS_GEN:
		return(qla4extioctl_rst_statistics_gen(ha, ioctl));

	case EXT_SC_RST_STATISTICS_ISCSI:
		return(qla4extioctl_rst_statistics_iscsi(ha, ioctl));

	case EXT_SC_SET_DEVICE_ENTRY_ISCSI:
		return(qla4extioctl_set_device_entry_iscsi(ha, ioctl));

	case EXT_SC_SET_INIT_FW_ISCSI:
		return(qla4extioctl_set_init_fw_iscsi(ha, ioctl));

#ifndef QLA4000
	case EXT_SC_SET_ISNS_SERVER:
		return(qla4extioctl_set_isns_server(ha, ioctl));
#endif

	default:
		QL4PRINT(QLP2,
		    printk("scsi%d: %s: unsupported set data sub-command "
		    "code (%X)\n",
		    ha->host_no, __func__, ioctl->SubCode));

		ioctl->Status = EXT_STATUS_UNSUPPORTED_SUBCODE;
		break;
	}
	return (0);
}

/**************************************************************************
 * qla4xxx_ioctl_sleep_done
 *	This routine is the callback function to wakeup ioctl completion
 *	semaphore for the ioctl request that is waiting.
 *
 * Input:
 *   	sem - pointer to the ioctl completion semaphore.
 *
 * Output:
 *	None
 *
 * Returns:
 *	None
 *
 * Context:
 *	Kernel context.
 **************************************************************************/
static  void
qla4xxx_ioctl_sleep_done (struct semaphore * sem)
{
	ENTER(__func__);

	if (sem != NULL) {
		QL4PRINT(QLP4, printk("%s: wake up sem.\n", __func__));
		QL4PRINT(QLP10, printk("%s: UP count=%d\n", __func__,
		    atomic_read(&sem->count)));
		up(sem);
	}

	LEAVE(__func__);
}

/**************************************************************************
 * qla4xxx_ioctl_sem_init
 *	This routine initializes the ioctl timer and semaphore used to wait
 *	for passthru completion.
 *
 * Input:
 *	ha    = adapter structure pointer.
 *
 * Output:
 *	None
 *
 * Returns:
 *	None
 *
 * Context:
 *	Kernel context.
 **************************************************************************/
void
qla4xxx_ioctl_sem_init (scsi_qla_host_t *ha)
{
	init_timer(&(ha->ioctl_cmpl_timer));
	ha->ioctl_cmpl_timer.data = (unsigned long)&ha->ioctl_cmpl_sem;
	ha->ioctl_cmpl_timer.function =
	(void (*)(unsigned long))qla4xxx_ioctl_sleep_done;
}

/**************************************************************************
 * qla4xxx_scsi_pass_done
 *	This routine resets the ioctl progress flag and wakes up the ioctl
 * 	completion semaphore.
 *
 * Input:
 *   	cmd - pointer to the passthru Scsi cmd structure which has completed.
 *
 * Output:
 *	None
 *
 * Returns:
 *	None
 *
 * Context:
 *	Interrupt context.
 **************************************************************************/
void
qla4xxx_scsi_pass_done(Scsi_Cmnd *cmd)
{
	scsi_qla_host_t *ha = (scsi_qla_host_t *) cmd->host->hostdata;


	ENTER(__func__);

	/* First check to see if the command has previously timed-out
	 * because we don't want to get the up/down semaphore counters off.
	 */
	if (ha->ioctl_scsi_pass_in_progress == TRUE) {
		ha->ioctl_scsi_pass_in_progress = FALSE;
		ha->ioctl_tov = 0;
		ha->ioctl_err_cmd = NULL;

		up(&ha->ioctl_cmpl_sem);
	}

	LEAVE(__func__);
	return;
}

/**************************************************************************
 * qla4extioctl_scsi_passthru
 *	This routine
 *
 * Input:
 *	ha    = adapter structure pointer.
 *	ioctl = IOCTL structure pointer.
 *
 * Output:
 *	None
 *
 * Map of DMA Buffer:
 *    +-------------------------+
 *    | EXT_SCSI_PASSTHRU_ISCSI |
 *    +-------------------------+
 *    | [SCSI READ|WRITE data]  |
 *    +-------------------------+
 *
 * Returns:
 *	QLA_SUCCESS = success
 *	QLA_ERROR   = error
 *
 * Context:
 *	Kernel context.
 **************************************************************************/
static int
qla4extioctl_scsi_passthru(scsi_qla_host_t *ha, EXT_IOCTL_ISCSI *ioctl)
{
	int			status = 0;
	ddb_entry_t             *ddb_entry;
	int                     i;
	EXT_SCSI_PASSTHRU_ISCSI	*pscsi_pass = NULL;
	Scsi_Device		*pscsi_device = NULL;
	Scsi_Cmnd		*pscsi_cmd = NULL;
	srb_t       *srb;
	uint32_t dma_buf_len;


	ENTER(__func__);

	if (!ADAPTER_UP(ha)) {
		QL4PRINT(QLP2,
		    printk("scsi%d: %s: command not pocessed, "
		    "adapter link down.\n",
		    ha->host_no, __func__));
		ioctl->Status = EXT_STATUS_HBA_NOT_READY;
		return(QLA_ERROR);
	}

#if 0 && defined(__VMKERNEL_MODULE__)
	spin_lock(&ha->dma_buf_spinlock);
#else
	down(&ha->dma_buf_sem);
#endif

	if (qla4xxx_get_ioctl_scrap_mem(ha, (void **)&pscsi_cmd,
	    sizeof(Scsi_Cmnd))) {
		/* not enough memory */
		ioctl->Status = EXT_STATUS_NO_MEMORY;
		QL4PRINT(QLP2|QLP4,
		    printk("%s(%d): inst=%d scrap not big enough. "
		    "size requested=%ld.\n",
		    __func__, ha->host_no, ha->instance,
		    (ulong)sizeof(Scsi_Cmnd)));
		goto error_exit_scsi_pass;
	}

	if (qla4xxx_get_ioctl_scrap_mem(ha, (void **)&pscsi_device,
	    sizeof(Scsi_Device))) {
		/* not enough memory */
		ioctl->Status = EXT_STATUS_NO_MEMORY;
		QL4PRINT(QLP2|QLP4,
		    printk("%s(%d): inst=%d scrap not big enough. "
		    "size requested=%ld.\n",
		    __func__, ha->host_no, ha->instance,
		    (ulong)sizeof(Scsi_Device)));
		goto error_exit_scsi_pass;
	}

	if (qla4xxx_get_ioctl_scrap_mem(ha, (void **)&pscsi_pass,
	    sizeof(EXT_SCSI_PASSTHRU_ISCSI))) {
		/* not enough memory */
		ioctl->Status = EXT_STATUS_NO_MEMORY;
		QL4PRINT(QLP2|QLP4,
		    printk("%s(%d): inst=%d scrap not big enough. "
		    "size requested=%ld.\n",
		    __func__, ha->host_no, ha->instance,
		    (ulong)sizeof(EXT_SCSI_PASSTHRU_ISCSI)));
		goto error_exit_scsi_pass;
	}

	memset(pscsi_device, 0, sizeof(Scsi_Device));
	memset(pscsi_pass, 0, sizeof(EXT_SCSI_PASSTHRU_ISCSI));
	memset(pscsi_cmd, 0, sizeof(Scsi_Cmnd));
	pscsi_cmd->device = pscsi_device;
	pscsi_cmd->request.nr_hw_segments = 1;

	/* ---- Get passthru structure from user space ---- */
	if (copy_from_user((uint8_t *)pscsi_pass,
	    Q64BIT_TO_PTR(ioctl->RequestAdr), sizeof(*pscsi_pass)) != 0) {
		QL4PRINT(QLP2,
		    printk("scsi%d: %s: unable to copy passthru struct "
		    "from user's memory area.\n",
		    ha->host_no, __func__));

		status = (-EFAULT);
		ioctl->Status = EXT_STATUS_COPY_ERR;
		goto error_exit_scsi_pass;
	}

	QL4PRINT(QLP10, printk("scsi%d: %s: incoming  EXT_SCSI_PASSTHRU_ISCSI "
	    "structure: \n", ha->host_no, __func__));
	qla4xxx_dump_bytes(QLP10, pscsi_pass, sizeof(*pscsi_pass));

	/* ---- Make sure device exists ---- */
	ddb_entry = qla4xxx_lookup_ddb_by_SCSIID(ha, pscsi_pass->Addr.Bus,
	    pscsi_pass->Addr.Target);
	if (ddb_entry == NULL) {
		QL4PRINT(QLP2,
		    printk("scsi%d: %s: invalid device (b%d,t%d) specified.\n",
		    ha->host_no, __func__,
		    pscsi_pass->Addr.Bus, pscsi_pass->Addr.Target));

		ioctl->Status = EXT_STATUS_DEV_NOT_FOUND;
		goto error_exit_scsi_pass;
	}

	/* ---- Make sure device is in an active state ---- */
	if (ddb_entry->fw_ddb_device_state != DDB_DS_SESSION_ACTIVE) {
		QL4PRINT(QLP2,
		    printk("scsi%d: %s: device (b%d,t%d) not in active state\n",
		    ha->host_no, __func__,
		    pscsi_pass->Addr.Bus, pscsi_pass->Addr.Target));

		ioctl->Status = EXT_STATUS_DEVICE_NOT_READY;
		goto error_exit_scsi_pass;
	}

	/* ---- Retrieve srb from pool ---- */
	srb = del_from_free_srb_q_head(ha);
	if (srb == NULL) {
		QL4PRINT(QLP2, printk("scsi%d: %s: srb not available\n",
		    ha->host_no, __func__));
		goto error_exit_scsi_pass;
	}

	/* ---- Allocate larger DMA buffer, if neccessary ---- */
	dma_buf_len = MAX(ioctl->ResponseLen - sizeof(*pscsi_pass),
	    ioctl->RequestLen - sizeof(*pscsi_pass));

	if (ha->dma_buf.buf_len < dma_buf_len
	    && qla4xxx_resize_dma_buf(ha, &ha->dma_buf, dma_buf_len) !=
	    QLA_SUCCESS) {
		QL4PRINT(QLP2,
		    printk("scsi%d: %s: ERROR cannot allocate requested "
		    "DMA buffer size 0x%x.\n",
		    ha->host_no, __func__, dma_buf_len));

		ioctl->Status = EXT_STATUS_NO_MEMORY;
		goto error_exit_scsi_pass;
	}

	memset(ha->dma_buf.virt_addr, 0, ha->dma_buf.buf_len);

	/* ---- Fill in the SCSI command structure ---- */
	pscsi_cmd->channel         = pscsi_pass->Addr.Bus;
	pscsi_cmd->target          = pscsi_pass->Addr.Target;
	pscsi_cmd->lun             = pscsi_pass->Addr.Lun;
	pscsi_cmd->request_buffer  = ha->dma_buf.virt_addr;
#ifdef __VMKERNEL_MODULE__
	pscsi_cmd->request_bufferMA =
              (dma_addr_t)virt_to_phys(pscsi_cmd->request_buffer);
#endif //__VMKERNEL_MODULE__
	pscsi_cmd->device          = pscsi_device;
	pscsi_cmd->host            = ha->host;
	pscsi_cmd->scsi_done       = qla4xxx_scsi_pass_done;
	CMD_TIMEOUT(pscsi_cmd)     = ql4xioctltimeout * HZ;

	CMD_SP(pscsi_cmd) = (char *) srb;
	srb->cmd = pscsi_cmd;
	srb->fw_ddb_index = ddb_entry->fw_ddb_index;
	srb->lun = pscsi_cmd->lun;
	srb->flags |= SRB_IOCTL_CMD;

	if (pscsi_pass->CdbLength == 6 || pscsi_pass->CdbLength == 10 ||
	    pscsi_pass->CdbLength == 12 || pscsi_pass->CdbLength == 16) {
		pscsi_cmd->cmd_len = pscsi_pass->CdbLength;
	} else {
		QL4PRINT(QLP2,
		    printk("scsi%d: %s: Unsupported CDB length 0x%x \n",
		    ha->host_no, __func__, pscsi_cmd->cmd_len));

		ioctl->Status = EXT_STATUS_INVALID_PARAM;
		goto error_exit_scsi_pass;
	}

	if (pscsi_pass->Direction == EXT_DEF_SCSI_PASSTHRU_DATA_IN) {
		pscsi_cmd->sc_data_direction = SCSI_DATA_READ;
		pscsi_cmd->request_bufflen = ioctl->ResponseLen -
		    sizeof(*pscsi_pass);

	} else if (pscsi_pass->Direction == EXT_DEF_SCSI_PASSTHRU_DATA_OUT) {
		pscsi_cmd->sc_data_direction = SCSI_DATA_WRITE;
		pscsi_cmd->request_bufflen = ioctl->RequestLen -
		    sizeof(*pscsi_pass);

		/* Sending user data from ioctl->ResponseAddr to SCSI
		 * command buffer */
		if (copy_from_user((uint8_t *)pscsi_cmd->request_buffer,
		    Q64BIT_TO_PTR(ioctl->RequestAdr) + sizeof(*pscsi_pass),
		    pscsi_cmd->request_bufflen) != 0) {
			QL4PRINT(QLP2,
			    printk("scsi%d: %s: unable to copy write buffer "
			    "from user's memory area.\n",
			    ha->host_no, __func__));

			status = (-EFAULT);
			ioctl->Status = EXT_STATUS_COPY_ERR;
			goto error_exit_scsi_pass;
		}
	} else {
		pscsi_cmd->sc_data_direction = SCSI_DATA_NONE;
		pscsi_cmd->request_buffer  = 0;
		pscsi_cmd->request_bufflen = 0;
	}

	memcpy(pscsi_cmd->cmnd,      pscsi_pass->Cdb, pscsi_cmd->cmd_len);
	memcpy(pscsi_cmd->data_cmnd, pscsi_pass->Cdb, pscsi_cmd->cmd_len);

	QL4PRINT(QLP4, printk("scsi%d:%d:%d:%d: %s: CDB = ",
	    ha->host_no, pscsi_cmd->channel, pscsi_cmd->target,
	    pscsi_cmd->lun, __func__));

	for (i = 0; i < pscsi_cmd->cmd_len; i++)
		QL4PRINT(QLP4, printk("%02X ", pscsi_cmd->cmnd[i]));

	QL4PRINT(QLP4, printk("\n"));

	/* ---- prepare for receiving completion ---- */
	ha->ioctl_scsi_pass_in_progress = TRUE;
	ha->ioctl_tov = (ql4xioctltimeout - QLA_CMD_TIMER_DELTA) * HZ;

	qla4xxx_ioctl_sem_init(ha);
	CMD_COMPL_STATUS(pscsi_cmd)  = (int) IOCTL_INVALID_STATUS;
	CMD_PASSTHRU_TYPE(pscsi_cmd) = (unsigned long)TRUE;

	/* ---- send command to adapter ---- */
	QL4PRINT(QLP4, printk("scsi%d:%d:%d:%d: %s: sending command.\n",
	    ha->host_no, pscsi_cmd->channel, pscsi_cmd->target,
	    pscsi_cmd->lun, __func__));

	ha->ioctl_cmpl_timer.expires = jiffies + ha->ioctl_tov;
	add_timer(&ha->ioctl_cmpl_timer);

	if (qla4xxx_send_command_to_isp(ha, srb) != QLA_SUCCESS) {
		add_to_free_srb_q(ha, srb);
		QL4PRINT(QLP2,
		    printk("scsi%d: %s: error sending cmd to isp\n",
		    ha->host_no, __func__));
		del_timer(&ha->ioctl_cmpl_timer);
		ioctl->Status = EXT_STATUS_DEV_NOT_FOUND;
		goto error_exit_scsi_pass;
	}

	down(&ha->ioctl_cmpl_sem);

	/*******************************************************
	 *						       *
	 *             Passthru Completion                     *
	 *						       *
	 *******************************************************/
	del_timer(&ha->ioctl_cmpl_timer);

	/* ---- check for timeout --- */
	if (ha->ioctl_scsi_pass_in_progress == TRUE) {
		QL4PRINT(QLP2,
		    printk("scsi%d: %s: ERROR = command timeout.\n",
		    ha->host_no, __func__));

		ioctl->Status = EXT_STATUS_ERR;

		if ((srb != NULL) && (srb->active_array_index < MAX_SRBS)) {
			unsigned long wait_cnt = WAIT_CMD_TOV;

			if ((srb->flags & SRB_FREE_STATE) == 0)
				qla4xxx_delete_timer_from_cmd(srb);

			/* Wait for command to get out of active state */
			wait_cnt = jiffies + WAIT_CMD_TOV * HZ;
			while (wait_cnt > jiffies) {
				if (srb->flags != SRB_ACTIVE_STATE)
					break;

				QL4PRINT(QLP7, printk("."));

				set_current_state(TASK_UNINTERRUPTIBLE);
				schedule_timeout(1 * HZ);
			}
		}

		ha->ioctl_scsi_pass_in_progress = FALSE;
		goto error_exit_scsi_pass;
	}

	/* --- Return info from status entry --- */
	ioctl->DetailStatus = CMD_SCSI_STATUS(pscsi_cmd);
	pscsi_pass->Reserved[0] = (uint8_t) CMD_SCSI_STATUS(pscsi_cmd);
	pscsi_pass->Reserved[1] = (uint8_t) CMD_COMPL_STATUS(pscsi_cmd);
	pscsi_pass->Reserved[2] = (uint8_t) CMD_ACTUAL_SNSLEN(pscsi_cmd);
	pscsi_pass->Reserved[3] = (uint8_t) CMD_HOST_STATUS(pscsi_cmd);
	pscsi_pass->Reserved[4] = MSB(CMD_RESID_LEN(pscsi_cmd));
	pscsi_pass->Reserved[5] = LSB(CMD_RESID_LEN(pscsi_cmd));
	pscsi_pass->Reserved[6] = (uint8_t) CMD_ISCSI_RESPONSE(pscsi_cmd);
	pscsi_pass->Reserved[7] = (uint8_t) CMD_ISCSI_FLAGS(pscsi_cmd);

	if (CMD_ACTUAL_SNSLEN(pscsi_cmd)) {
		memcpy(pscsi_pass->SenseData, CMD_SNSP(pscsi_cmd),
		    MIN(CMD_ACTUAL_SNSLEN(pscsi_cmd),
		    sizeof(pscsi_pass->SenseData)));
	}

	/* ---- check for command completion --- */
	if (CMD_COMPL_STATUS(pscsi_cmd) == IOCTL_INVALID_STATUS) {
		QL4PRINT(QLP2,
		    printk("scsi%d:%d:%d:%d: %s: ERROR = "
		    "command not completed.\n",
		    ha->host_no, pscsi_cmd->channel, pscsi_cmd->target,
		    pscsi_cmd->lun, __func__));

		ioctl->Status = EXT_STATUS_ERR;
		goto error_exit_scsi_pass;
	} else if (CMD_HOST_STATUS(pscsi_cmd) == DID_OK) {
		ioctl->Status = EXT_STATUS_OK;
	} else if (CMD_COMPL_STATUS(pscsi_cmd) == SCS_DATA_UNDERRUN) {
		QL4PRINT(QLP2,
		    printk("scsi%d: %s: Data underrun.  Resid = 0x%x\n",
		    ha->host_no, __func__, CMD_RESID_LEN(pscsi_cmd)));

		ioctl->Status = EXT_STATUS_DATA_UNDERRUN;
	} else if (CMD_COMPL_STATUS(pscsi_cmd) == SCS_DATA_OVERRUN) {
		QL4PRINT(QLP2,
		    printk("scsi%d: %s: Data overrun.  Resid = 0x%x\n",
		    ha->host_no, __func__, CMD_RESID_LEN(pscsi_cmd)));

		ioctl->Status = EXT_STATUS_DATA_OVERRUN;
	} else {
		QL4PRINT(QLP2,
		    printk("scsi%d: %s: Command completed in ERROR. "
		    "cs=%04x, ss=%-4x\n", ha->host_no, __func__,
		    CMD_COMPL_STATUS(pscsi_cmd), CMD_SCSI_STATUS(pscsi_cmd)));

		if (CMD_SCSI_STATUS(pscsi_cmd) != SCSI_GOOD)
			ioctl->Status = EXT_STATUS_SCSI_STATUS;
		else
			ioctl->Status = EXT_STATUS_ERR;
	}

	/* ---- Copy SCSI Passthru structure with updated sense buffer
	 *      to user space ---- */
	if (copy_to_user(Q64BIT_TO_PTR(ioctl->ResponseAdr),
	    pscsi_pass, sizeof(EXT_SCSI_PASSTHRU_ISCSI)) != 0) {
		QL4PRINT(QLP2,
		    printk("scsi%d: %s: unable to copy passthru struct "
		    "to user's memory area.\n",
		    ha->host_no, __func__));

		status = (-EFAULT);
		ioctl->Status = EXT_STATUS_COPY_ERR;
		goto error_exit_scsi_pass;
	}

	QL4PRINT(QLP10,
	    printk("scsi%d: %s: outgoing EXT_SCSI_PASSTHRU_ISCSI structure:\n",
	    ha->host_no, __func__));
	qla4xxx_dump_bytes(QLP10,
	    Q64BIT_TO_PTR(ioctl->ResponseAdr), sizeof(EXT_SCSI_PASSTHRU_ISCSI));

	/* ---- Copy SCSI READ data from SCSI command buffer
	*       to user space ---- */
	if (pscsi_pass->Direction == EXT_DEF_SCSI_PASSTHRU_DATA_IN) {
		void   *xfer_ptr = Q64BIT_TO_PTR(ioctl->ResponseAdr) +
				    sizeof(EXT_SCSI_PASSTHRU_ISCSI);
		uint32_t xfer_len = ioctl->ResponseLen -
				    sizeof(EXT_SCSI_PASSTHRU_ISCSI);


		/* Update ResponseLen if a data underrun occurred */
		if (CMD_COMPL_STATUS(pscsi_cmd) == SCS_DATA_UNDERRUN &&
		    CMD_RESID_LEN(pscsi_cmd)) {
			xfer_len -= CMD_RESID_LEN(pscsi_cmd);
		}

		if (copy_to_user(xfer_ptr,
		    pscsi_cmd->request_buffer, xfer_len) != 0) {
			QL4PRINT(QLP2,
			    printk("scsi%d: %s: unable to copy READ data "
			    "to user's memory area.\n",
			    ha->host_no, __func__));

			status = (-EFAULT);
			ioctl->Status = EXT_STATUS_COPY_ERR;
			goto error_exit_scsi_pass;
		}

		QL4PRINT(QLP10,
		    printk("scsi%d: %s: outgoing READ data:  (0x%p)\n",
		    ha->host_no, __func__, xfer_ptr));

		qla4xxx_dump_bytes(QLP10, xfer_ptr, xfer_len);
	}

	goto exit_scsi_pass;

	error_exit_scsi_pass:
	ioctl->ResponseLen = 0;

	exit_scsi_pass:
	qla4xxx_free_ioctl_scrap_mem(ha);
#if 0 && defined(__VMKERNEL_MODULE__)
	spin_unlock(&ha->dma_buf_spinlock);
#else
	up(&ha->dma_buf_sem);
#endif

	LEAVE(__func__);
	return(status);
}

/**************************************************************************
 * qla4extioctl_iscsi_passthru
 *	This routine sends iSCSI pass-through to destination.
 *
 * Input:
 *	ha    = adapter structure pointer.
 *	ioctl = IOCTL structure pointer.
 *
 * Output:
 *	None
 *
 * Returns:
 *	QLA_SUCCESS = success
 *	QLA_ERROR   = error
 *
 * Context:
 *	Kernel context.
 **************************************************************************/
static int
qla4extioctl_iscsi_passthru(scsi_qla_host_t *ha, EXT_IOCTL_ISCSI *ioctl)
{
	int	status = 0;

	ENTER(__func__);

	ioctl->Status = EXT_STATUS_UNSUPPORTED_SUBCODE;
	QL4PRINT(QLP4,
	    printk("scsi%d: %s: UNSUPPORTED\n", ha->host_no, __func__));

	LEAVE(__func__);
	return(status);
}

/**************************************************************************
 * qla4extioctl_get_hbacnt
 *	This routine retrieves the number of supported HBAs found.
 *
 * Input:
 *	ha    = adapter structure pointer.
 *	ioctl = IOCTL structure pointer.
 *
 * Output:
 *	None
 *
 * Returns:
 *	QLA_SUCCESS = success
 *	QLA_ERROR   = error
 *
 * Context:
 *	Kernel context.
 **************************************************************************/
static int
qla4extioctl_get_hbacnt(scsi_qla_host_t *ha, EXT_IOCTL_ISCSI *ioctl)
{
	int status = QLA_ERROR;
	EXT_HBA_COUNT	hba_count;


	ENTER(__func__);

	hba_count.HbaCnt = qla4xxx_get_hba_count();
	if ((status = copy_to_user(Q64BIT_TO_PTR(ioctl->ResponseAdr),
	    &hba_count, sizeof(hba_count))) != 0) {
		QL4PRINT(QLP2|QLP4,
		    printk("scsi%d: %s: failed to copy data. struct size=%d.\n",
		    ha->host_no, __func__, (uint32_t)sizeof(EXT_IOCTL_ISCSI)));

		status = (-EFAULT);
		ioctl->Status = EXT_STATUS_COPY_ERR;
		goto exit_get_hbacnt;
	}

	QL4PRINT(QLP4, printk("scsi%d: %s: hbacnt is %d\n",
	    ha->host_no, __func__, hba_count.HbaCnt));

	ioctl->Status = EXT_STATUS_OK;

	exit_get_hbacnt:

	LEAVE(__func__);
	return(status);
}

/**************************************************************************
 * qla4intioctl_logout_iscsi
 *	This routine requests that the specified device either login or
 *	logout, depending on the option specified.
 *
 * Input:
 *	ha    = adapter structure pointer.
 *	ioctl = IOCTL structure pointer.
 *
 * Output:
 *	None
 *
 * Returns:
 *	QLA_SUCCESS = success
 *	QLA_ERROR   = error
 *
 * Context:
 *	Kernel context.
 **************************************************************************/
static int
qla4intioctl_logout_iscsi(scsi_qla_host_t *ha, EXT_IOCTL_ISCSI *ioctl)
{
	int			status = 0;
	INT_LOGOUT_ISCSI	logout;


	ENTER(__func__);

	if (ioctl->RequestLen > sizeof(INT_LOGOUT_ISCSI)) {
		QL4PRINT(QLP2,
		    printk("scsi%d: %s: memory area too small\n",
		    ha->host_no, __func__));

		ioctl->Status = EXT_STATUS_BUFFER_TOO_SMALL;
		goto exit_logout;
	}

	/* --- Copy logout structure from user space --- */
	if (copy_from_user((uint8_t *)&logout,
	    Q64BIT_TO_PTR(ioctl->RequestAdr), sizeof(logout)) != 0) {
		QL4PRINT(QLP2,
		    printk("scsi%d: %s: unable to copy data from "
		    "user's memory area\n", ha->host_no, __func__));

		status = (-EFAULT);
		ioctl->Status = EXT_STATUS_COPY_ERR;
		goto exit_logout;
	}

	/* --- Execute command --- */
	if (logout.Options == INT_DEF_CLOSE_SESSION) {
		if ((status = qla4xxx_logout_device(ha, logout.TargetID,
		    logout.ConnectionID)) == QLA_SUCCESS) {
			QL4PRINT(QLP4,
			    printk("scsi%d: %s: CLOSE_SESSION SUCCEEDED!, "
			    "target %d\n", ha->host_no, __func__,
			    logout.TargetID));

			ioctl->Status = EXT_STATUS_OK;
		} else {
			QL4PRINT(QLP2,
			    printk("scsi%d: %s: CLOSE_SESSION FAILED!, "
			    "target %d\n", ha->host_no, __func__,
			    logout.TargetID));

			ioctl->Status = EXT_STATUS_ERR;
		}

	} else if (logout.Options == INT_DEF_RELOGIN_CONNECTION) {

		if ((status = qla4xxx_login_device(ha, logout.TargetID,
		    logout.ConnectionID)) == QLA_SUCCESS) {
			QL4PRINT(QLP4,
			    printk("scsi%d: %s: RELOGIN_CONNECTION "
			    "SUCCEEDED!, target %d\n",
			    ha->host_no, __func__, logout.TargetID));

			ioctl->Status = EXT_STATUS_OK;
		} else {
			QL4PRINT(QLP2,
			    printk("scsi%d: %s: RELOGIN_CONNECTION "
			    "FAILED!, target %d\n",
			    ha->host_no, __func__, logout.TargetID));

			ioctl->Status = EXT_STATUS_ERR;
		}

	} else if (logout.Options == INT_DEF_DELETE_DDB) {

		if ((status = qla4xxx_delete_device(ha, logout.TargetID,
		    logout.ConnectionID)) == QLA_SUCCESS) {
			QL4PRINT(QLP4,
			    printk("scsi%d: %s: DELETE_DDB "
			    "SUCCEEDED!, target %d\n",
			    ha->host_no, __func__, logout.TargetID));

			ioctl->Status = EXT_STATUS_OK;
		} else {
			QL4PRINT(QLP2,
			    printk("scsi%d: %s: DELETE_DDB FAILED!, "
			    "target %d\n", ha->host_no,
			    __func__, logout.TargetID));

			ioctl->Status = EXT_STATUS_ERR;
		}
	}

	exit_logout:

	LEAVE(__func__);
	return(status);
}

/**************************************************************************
 * qla4intioctl_ping
 *	This routine requests that the HBA PING the specified IP Address.
 *
 * Input:
 *	ha    = adapter structure pointer.
 *	ioctl = IOCTL structure pointer.
 *
 * Output:
 *	None
 *
 * Returns:
 *	QLA_SUCCESS = success
 *	QLA_ERROR   = error
 *
 * Context:
 *	Kernel context.
 **************************************************************************/
static int
qla4intioctl_ping(scsi_qla_host_t *ha, EXT_IOCTL_ISCSI *ioctl)
{
	int		status = 0;
	INT_PING	ping;
	uint32_t	mbox_cmd[MBOX_REG_COUNT];
	uint32_t	mbox_sts[MBOX_REG_COUNT];


	ENTER(__func__);
	/*
	 * Copy user's data to local buffer
	 */
	if (copy_from_user((uint8_t *)&ping, Q64BIT_TO_PTR(ioctl->RequestAdr),
	    sizeof(ping)) != 0) {
		QL4PRINT(QLP2,
		    printk("scsi%d: %s: unable to copy data from "
		    "user's memory area\n", ha->host_no, __func__));
		status = (-EFAULT);
		ioctl->Status = EXT_STATUS_COPY_ERR;
		goto exit_ping;
	}

	/*
	 * Debug Print Statement
	 */
	if (ping.IPAddr.Type == EXT_DEF_TYPE_ISCSI_IP) {
		QL4PRINT(QLP4,
		    printk("scsi%d: %s: %d.%d.%d.%d\n",
		    ha->host_no, __func__,
		    ping.IPAddr.IPAddress[0],
		    ping.IPAddr.IPAddress[1],
		    ping.IPAddr.IPAddress[2],
		    ping.IPAddr.IPAddress[3]));
	}
	else {
		QL4PRINT(QLP4,
		    printk("scsi%d: %s: %d.%d.%d.%d. %d.%d.%d.%d. "
		    "%d.%d.%d.%d. %d.%d.%d.%d\n",
		    ha->host_no, __func__,
		    ping.IPAddr.IPAddress[0], ping.IPAddr.IPAddress[1],
		    ping.IPAddr.IPAddress[2], ping.IPAddr.IPAddress[3],
		    ping.IPAddr.IPAddress[4], ping.IPAddr.IPAddress[5],
		    ping.IPAddr.IPAddress[6], ping.IPAddr.IPAddress[7],
		    ping.IPAddr.IPAddress[8], ping.IPAddr.IPAddress[9],
		    ping.IPAddr.IPAddress[10], ping.IPAddr.IPAddress[11],
		    ping.IPAddr.IPAddress[12], ping.IPAddr.IPAddress[13],
		    ping.IPAddr.IPAddress[14], ping.IPAddr.IPAddress[15]));
	}

	/*
	 * Issue Mailbox Command
	 */
	memset(mbox_cmd, 0, sizeof(mbox_cmd));
	memset(mbox_sts, 0, sizeof(mbox_sts));
	mbox_cmd[0] = MBOX_CMD_PING;
	mbox_cmd[1] = ping.PacketCount;
	memcpy(&mbox_cmd[2], ping.IPAddr.IPAddress, EXT_DEF_IP_ADDR_SIZE);

	if (qla4xxx_mailbox_command(ha, 6, 1, &mbox_cmd[0], &mbox_sts[0]) ==
	    QLA_ERROR) {
		QL4PRINT(QLP2, printk("scsi%d: %s: command failed \n",
		    ha->host_no, __func__));

		ioctl->Status = EXT_STATUS_MAILBOX;
		ioctl->DetailStatus = mbox_sts[0];
		goto exit_ping;
	}

	ioctl->Status = EXT_STATUS_OK;

exit_ping:
	LEAVE(__func__);

	return(status);
}

#if BYTE_ORDER_SUPPORT_ENABLED
static void
__xlate_sys_info(FLASH_SYS_INFO *from, FLASH_SYS_INFO *to,
    uint8_t direction, uint32_t length)
{
	switch (direction) {
	case GET_DATA:
		from->cookie            = le32_to_cpu(to->cookie);
		from->physAddrCount     = le32_to_cpu(to->physAddrCount);
		memcpy(from->physAddr,  to->physAddr, sizeof(from->physAddr));
		memcpy(from->vendorId,  to->vendorId, sizeof(from->vendorId));
		memcpy(from->productId, to->productId, sizeof(from->productId));
		from->serialNumber      = le32_to_cpu(to->serialNumber);
		from->pciDeviceVendor   = le32_to_cpu(to->pciDeviceVendor);
		from->pciDeviceId       = le32_to_cpu(to->pciDeviceId);
		from->pciSubsysVendor   = le32_to_cpu(to->pciSubsysVendor);
		from->pciSubsysId       = le32_to_cpu(to->pciSubsysId);
		from->crumbs            = le32_to_cpu(to->crumbs);
		from->enterpriseNumber  = le32_to_cpu(to->enterpriseNumber);
		from->mtu               = le32_to_cpu(to->mtu);
		from->reserved0         = le32_to_cpu(to->reserved0);
		from->crumbs2           = le32_to_cpu(to->crumbs2);
		memcpy(from->acSerialNumber, to->acSerialNumber,
		    sizeof(from->acSerialNumber));
		from->crumbs3           = le32_to_cpu(to->crumbs3);
		memcpy(from->reserved1, to->reserved1, sizeof(from->reserved1));
		break;

	case SET_DATA:
		from->cookie            = __cpu_to_le32(to->cookie);
		from->physAddrCount     = __cpu_to_le32(to->physAddrCount);
		memcpy(from->physAddr,  to->physAddr, sizeof(from->physAddr));
		memcpy(from->vendorId,  to->vendorId, sizeof(from->vendorId));
		memcpy(from->productId, to->productId, sizeof(from->productId));
		from->serialNumber      = __cpu_to_le32(to->serialNumber);
		from->pciDeviceVendor   = __cpu_to_le32(to->pciDeviceVendor);
		from->pciDeviceId       = __cpu_to_le32(to->pciDeviceId);
		from->pciSubsysVendor   = __cpu_to_le32(to->pciSubsysVendor);
		from->pciSubsysId       = __cpu_to_le32(to->pciSubsysId);
		from->crumbs            = __cpu_to_le32(to->crumbs);
		from->enterpriseNumber  = __cpu_to_le32(to->enterpriseNumber);
		from->mtu               = __cpu_to_le32(to->mtu);
		from->reserved0         = __cpu_to_le32(to->reserved0);
		from->crumbs2           = __cpu_to_le32(to->crumbs2);
		memcpy(from->acSerialNumber, to->acSerialNumber,
		    sizeof(from->acSerialNumber));
		from->crumbs3           = __cpu_to_le32(to->crumbs3);
		memcpy(from->reserved1, to->reserved1, sizeof(from->reserved1));
		break;
	}
}

static void
__xlate_driver_info(INT_FLASH_DRIVER_PARAM *from, INT_FLASH_DRIVER_PARAM *to,
    uint8_t direction, uint32_t length)
{
	switch (direction) {
	case GET_DATA:
		from->DiscoveryTimeOut = le16_to_cpu(to->DiscoveryTimeOut);
		from->PortDownTimeout = le16_to_cpu(to->PortDownTimeout);
		memcpy(from->Reserved, to->Reserved, sizeof(from->Reserved));
		break;

	case SET_DATA:
		from->DiscoveryTimeOut = __cpu_to_le32(to->DiscoveryTimeOut);
		from->PortDownTimeout = __cpu_to_le32(to->PortDownTimeout);
		memcpy(from->Reserved, to->Reserved, sizeof(from->Reserved));
		break;
	}
}

static void
__xlate_init_fw_ctrl_blk(INIT_FW_CTRL_BLK *from, INIT_FW_CTRL_BLK *to,
    uint8_t direction, uint32_t length)
{
	switch (direction) {
	case GET_DATA:
		from->Version           = to->Version;
		from->Control           = to->Control;
		from->FwOptions         = le16_to_cpu(to->FwOptions);
		from->ExecThrottle      = le16_to_cpu(to->ExecThrottle);
		from->RetryCount        = to->RetryCount;
		from->RetryDelay        = to->RetryDelay;
		from->MaxEthFrPayloadSize = le16_to_cpu(to->MaxEthFrPayloadSize);
		from->AddFwOptions      = le16_to_cpu(to->AddFwOptions);
		from->HeartbeatInterval = to->HeartbeatInterval;
		from->InstanceNumber    = to->InstanceNumber;
		from->RES2              = le16_to_cpu(to->RES2);
		from->ReqQConsumerIndex = le16_to_cpu(to->ReqQConsumerIndex);
		from->ComplQProducerIndex = le16_to_cpu(to->ComplQProducerIndex);
		from->ReqQLen           = le16_to_cpu(to->ReqQLen);
		from->ComplQLen         = le16_to_cpu(to->ComplQLen);
		from->ReqQAddrLo        = le32_to_cpu(to->ReqQAddrLo);
		from->ReqQAddrHi        = le32_to_cpu(to->ReqQAddrHi);
		from->ComplQAddrLo      = le32_to_cpu(to->ComplQAddrLo);
		from->ComplQAddrHi      = le32_to_cpu(to->ComplQAddrHi);
		from->ShadowRegBufAddrLo= le32_to_cpu(to->ShadowRegBufAddrLo);
		from->ShadowRegBufAddrHi= le32_to_cpu(to->ShadowRegBufAddrHi);
		from->iSCSIOptions      = le16_to_cpu(to->iSCSIOptions);
		from->TCPOptions        = le16_to_cpu(to->TCPOptions);
		from->IPOptions         = le16_to_cpu(to->IPOptions);
		from->MaxPDUSize        = le16_to_cpu(to->MaxPDUSize);
		from->RcvMarkerInt      = le16_to_cpu(to->RcvMarkerInt);
		from->SndMarkerInt      = le16_to_cpu(to->SndMarkerInt);
		from->InitMarkerlessInt = le16_to_cpu(to->InitMarkerlessInt);
		from->FirstBurstSize    = le16_to_cpu(to->FirstBurstSize);
		from->DefaultTime2Wait  = le16_to_cpu(to->DefaultTime2Wait);
		from->DefaultTime2Retain= le16_to_cpu(to->DefaultTime2Retain);
		from->MaxOutStndngR2T   = le16_to_cpu(to->MaxOutStndngR2T);
		from->KeepAliveTimeout  = le16_to_cpu(to->KeepAliveTimeout);
		from->PortNumber        = le16_to_cpu(to->PortNumber);
		from->MaxBurstSize      = le16_to_cpu(to->MaxBurstSize);
		from->RES4              = le32_to_cpu(to->RES4);
		memcpy(from->IPAddr, to->IPAddr, sizeof(from->IPAddr));
		memcpy(from->RES5, to->RES5, sizeof(from->RES5));
		memcpy(from->SubnetMask, to->SubnetMask,
		    sizeof(from->SubnetMask));
		memcpy(from->RES6, to->RES6, sizeof(from->RES6));
		memcpy(from->GatewayIPAddr, to->GatewayIPAddr,
		    sizeof(from->GatewayIPAddr));
		memcpy(from->RES7, to->RES7, sizeof(from->RES7));
		memcpy(from->PriDNSIPAddr, to->PriDNSIPAddr,
		    sizeof(from->PriDNSIPAddr));
		memcpy(from->SecDNSIPAddr, to->SecDNSIPAddr,
		    sizeof(from->SecDNSIPAddr));
		memcpy(from->RES8, to->RES8, sizeof(from->RES8));
		memcpy(from->Alias, to->Alias, sizeof(from->Alias));
		memcpy(from->TargAddr, to->TargAddr, sizeof(from->TargAddr));
		memcpy(from->CHAPNameSecretsTable, to->CHAPNameSecretsTable,
		    sizeof(from->CHAPNameSecretsTable));
		memcpy(from->EthernetMACAddr, to->EthernetMACAddr,
		    sizeof(from->EthernetMACAddr));
		from->TargetPortalGroup = le16_to_cpu(to->TargetPortalGroup);
		from->SendScale         = to->SendScale;
		from->RecvScale         = to->RecvScale;
		from->TypeOfService     = to->TypeOfService;
		from->Time2Live         = to->Time2Live;
		from->VLANPriority      = le16_to_cpu(to->VLANPriority);
		from->Reserved8         = le16_to_cpu(to->Reserved8);
		memcpy(from->SecIPAddr, to->SecIPAddr, sizeof(from->SecIPAddr));
		memcpy(from->Reserved9, to->Reserved9, sizeof(from->Reserved9));
		memcpy(from->iSNSIPAddr, to->iSNSIPAddr,
		    sizeof(from->iSNSIPAddr));
		memcpy(from->Reserved10, to->Reserved10,
		    sizeof(from->Reserved10));
		from->iSNSServerPortNumber =
		    le16_to_cpu(to->iSNSServerPortNumber);
		memcpy(from->SLPDAIPAddr, to->SLPDAIPAddr,
		    sizeof(from->SLPDAIPAddr));
		memcpy(from->Reserved11, to->Reserved11,
		    sizeof(from->Reserved11));
		memcpy(from->iSCSINameString, to->iSCSINameString,
		    sizeof(from->iSCSINameString));
		break;

	case SET_DATA:
		from->Version           = to->Version;
		from->Control           = to->Control;
		from->FwOptions         = __cpu_to_le16(to->FwOptions);
		from->ExecThrottle      = __cpu_to_le16(to->ExecThrottle);
		from->RetryCount        = to->RetryCount;
		from->RetryDelay        = to->RetryDelay;
		from->MaxEthFrPayloadSize =
		    __cpu_to_le16(to->MaxEthFrPayloadSize);
		from->AddFwOptions      = __cpu_to_le16(to->AddFwOptions);
		from->HeartbeatInterval = to->HeartbeatInterval;
		from->InstanceNumber    = to->InstanceNumber;
		from->RES2              = __cpu_to_le16(to->RES2);
		from->ReqQConsumerIndex = __cpu_to_le16(to->ReqQConsumerIndex);
		from->ComplQProducerIndex =
		    __cpu_to_le16(to->ComplQProducerIndex);
		from->ReqQLen           = __cpu_to_le16(to->ReqQLen);
		from->ComplQLen         = __cpu_to_le16(to->ComplQLen);
		from->ReqQAddrLo        = __cpu_to_le32(to->ReqQAddrLo);
		from->ReqQAddrHi        = __cpu_to_le32(to->ReqQAddrHi);
		from->ComplQAddrLo      = __cpu_to_le32(to->ComplQAddrLo);
		from->ComplQAddrHi      = __cpu_to_le32(to->ComplQAddrHi);
		from->ShadowRegBufAddrLo= __cpu_to_le32(to->ShadowRegBufAddrLo);
		from->ShadowRegBufAddrHi= __cpu_to_le32(to->ShadowRegBufAddrHi);
		from->iSCSIOptions      = __cpu_to_le16(to->iSCSIOptions);
		from->TCPOptions        = __cpu_to_le16(to->TCPOptions);
		from->IPOptions         = __cpu_to_le16(to->IPOptions);
		from->MaxPDUSize        = __cpu_to_le16(to->MaxPDUSize);
		from->RcvMarkerInt      = __cpu_to_le16(to->RcvMarkerInt);
		from->SndMarkerInt      = __cpu_to_le16(to->SndMarkerInt);
		from->InitMarkerlessInt = __cpu_to_le16(to->InitMarkerlessInt);
		from->FirstBurstSize    = __cpu_to_le16(to->FirstBurstSize);
		from->DefaultTime2Wait  = __cpu_to_le16(to->DefaultTime2Wait);
		from->DefaultTime2Retain= __cpu_to_le16(to->DefaultTime2Retain);
		from->MaxOutStndngR2T   = __cpu_to_le16(to->MaxOutStndngR2T);
		from->KeepAliveTimeout  = __cpu_to_le16(to->KeepAliveTimeout);
		from->PortNumber        = __cpu_to_le16(to->PortNumber);
		from->MaxBurstSize      = __cpu_to_le16(to->MaxBurstSize);
		from->RES4              = __cpu_to_le32(to->RES4);
		memcpy(from->IPAddr, to->IPAddr, sizeof(from->IPAddr));
		memcpy(from->RES5, to->RES5, sizeof(from->RES5));
		memcpy(from->SubnetMask, to->SubnetMask,
		    sizeof(from->SubnetMask));
		memcpy(from->RES6, to->RES6, sizeof(from->RES6));
		memcpy(from->GatewayIPAddr, to->GatewayIPAddr,
		    sizeof(from->GatewayIPAddr));
		memcpy(from->RES7, to->RES7, sizeof(from->RES7));
		memcpy(from->PriDNSIPAddr, to->PriDNSIPAddr,
		    sizeof(from->PriDNSIPAddr));
		memcpy(from->SecDNSIPAddr, to->SecDNSIPAddr,
		    sizeof(from->SecDNSIPAddr));
		memcpy(from->RES8, to->RES8, sizeof(from->RES8));
		memcpy(from->Alias, to->Alias, sizeof(from->Alias));
		memcpy(from->TargAddr, to->TargAddr, sizeof(from->TargAddr));
		memcpy(from->CHAPNameSecretsTable, to->CHAPNameSecretsTable,
		    sizeof(from->CHAPNameSecretsTable));
		memcpy(from->EthernetMACAddr, to->EthernetMACAddr,
		    sizeof(from->EthernetMACAddr));
		from->TargetPortalGroup = __cpu_to_le16(to->TargetPortalGroup);
		from->SendScale         = to->SendScale;
		from->RecvScale         = to->RecvScale;
		from->TypeOfService     = to->TypeOfService;
		from->Time2Live         = to->Time2Live;
		from->VLANPriority      = __cpu_to_le16(to->VLANPriority);
		from->Reserved8         = __cpu_to_le16(to->Reserved8);
		memcpy(from->SecIPAddr, to->SecIPAddr, sizeof(from->SecIPAddr));
		memcpy(from->Reserved9, to->Reserved9, sizeof(from->Reserved9));
		memcpy(from->iSNSIPAddr, to->iSNSIPAddr,
		    sizeof(from->iSNSIPAddr));
		memcpy(from->Reserved10, to->Reserved10,
		    sizeof(from->Reserved10));
		from->iSNSServerPortNumber =
		    __cpu_to_le16(to->iSNSServerPortNumber);
		memcpy(from->SLPDAIPAddr, to->SLPDAIPAddr,
		    sizeof(from->SLPDAIPAddr));
		memcpy(from->Reserved11, to->Reserved11,
		    sizeof(from->Reserved11));
		memcpy(from->iSCSINameString, to->iSCSINameString,
		    sizeof(from->iSCSINameString));
		break;
	}
}

static void
__xlate_dev_db(DEV_DB_ENTRY *from, DEV_DB_ENTRY *to,
    uint8_t direction, uint32_t length)
{
	switch (direction) {
	case GET_DATA:
		from->options           = to->options;
		from->control           = to->control;
		from->exeThrottle       = le16_to_cpu(to->exeThrottle);
		from->exeCount          = le16_to_cpu(to->exeCount);
		from->retryCount        = to->retryCount;
		from->retryDelay        = to->retryDelay;
		from->iSCSIOptions      = le16_to_cpu(to->iSCSIOptions);
		from->TCPOptions        = le16_to_cpu(to->TCPOptions);
		from->IPOptions         = le16_to_cpu(to->IPOptions);
		from->maxPDUSize        = le16_to_cpu(to->maxPDUSize);
		from->rcvMarkerInt      = le16_to_cpu(to->rcvMarkerInt);
		from->sndMarkerInt      = le16_to_cpu(to->sndMarkerInt);
		from->iSCSIMaxSndDataSegLen =
		    le16_to_cpu(to->iSCSIMaxSndDataSegLen);
		from->firstBurstSize    = le16_to_cpu(to->firstBurstSize);
		from->minTime2Wait      = le16_to_cpu(to->minTime2Wait);
		from->maxTime2Retain    = le16_to_cpu(to->maxTime2Retain);
		from->maxOutstndngR2T   = le16_to_cpu(to->maxOutstndngR2T);
		from->keepAliveTimeout  = le16_to_cpu(to->keepAliveTimeout);
		memcpy(from->ISID, to->ISID, sizeof(from->ISID));
		from->TSID              = le16_to_cpu(to->TSID);
		from->portNumber        = le16_to_cpu(to->portNumber);
		from->maxBurstSize      = le16_to_cpu(to->maxBurstSize);
		from->taskMngmntTimeout = le16_to_cpu(to->taskMngmntTimeout);
		from->reserved1         = le16_to_cpu(to->reserved1);
		memcpy(from->ipAddr, to->ipAddr, sizeof(from->ipAddr));
		memcpy(from->iSCSIAlias, to->iSCSIAlias,
		    sizeof(from->iSCSIAlias));
		memcpy(from->targetAddr, to->targetAddr,
		    sizeof(from->targetAddr));
		memcpy(from->userID, to->userID, sizeof(from->userID));
		memcpy(from->password, to->password, sizeof(from->password));
		memcpy(from->iscsiName, to->iscsiName, sizeof(from->iscsiName));
		from->ddbLink           = le16_to_cpu(to->ddbLink);
		from->CHAPTableIndex    = le16_to_cpu(to->CHAPTableIndex);
		memcpy(from->reserved2, to->reserved2, sizeof(from->reserved2));
		from->Cookie            = le16_to_cpu(to->Cookie);
		break;

	case SET_DATA:
		from->options           = to->options;
		from->control           = to->control;
		from->exeThrottle       = __cpu_to_le16(to->exeThrottle);
		from->exeCount          = __cpu_to_le16(to->exeCount);
		from->retryCount        = to->retryCount;
		from->retryDelay        = to->retryDelay;
		from->iSCSIOptions      = __cpu_to_le16(to->iSCSIOptions);
		from->TCPOptions        = __cpu_to_le16(to->TCPOptions);
		from->IPOptions         = __cpu_to_le16(to->IPOptions);
		from->maxPDUSize        = __cpu_to_le16(to->maxPDUSize);
		from->rcvMarkerInt      = __cpu_to_le16(to->rcvMarkerInt);
		from->sndMarkerInt      = __cpu_to_le16(to->sndMarkerInt);
		from->iSCSIMaxSndDataSegLen =
		    __cpu_to_le16(to->iSCSIMaxSndDataSegLen);
		from->firstBurstSize    = __cpu_to_le16(to->firstBurstSize);
		from->minTime2Wait      = __cpu_to_le16(to->minTime2Wait);
		from->maxTime2Retain    = __cpu_to_le16(to->maxTime2Retain);
		from->maxOutstndngR2T   = __cpu_to_le16(to->maxOutstndngR2T);
		from->keepAliveTimeout  = __cpu_to_le16(to->keepAliveTimeout);
		memcpy(from->ISID, to->ISID, sizeof(from->ISID));
		from->TSID              = __cpu_to_le16(to->TSID);
		from->portNumber        = __cpu_to_le16(to->portNumber);
		from->maxBurstSize      = __cpu_to_le16(to->maxBurstSize);
		from->taskMngmntTimeout = __cpu_to_le16(to->taskMngmntTimeout);
		from->reserved1         = __cpu_to_le16(to->reserved1);
		memcpy(from->ipAddr, to->ipAddr, sizeof(from->ipAddr));
		memcpy(from->iSCSIAlias, to->iSCSIAlias,
		    sizeof(from->iSCSIAlias));
		memcpy(from->targetAddr, to->targetAddr,
		    sizeof(from->targetAddr));
		memcpy(from->userID, to->userID, sizeof(from->userID));
		memcpy(from->password, to->password, sizeof(from->password));
		memcpy(from->iscsiName, to->iscsiName, sizeof(from->iscsiName));
		from->ddbLink           = __cpu_to_le16(to->ddbLink);
		from->CHAPTableIndex    = __cpu_to_le16(to->CHAPTableIndex);
		memcpy(from->reserved2, to->reserved2, sizeof(from->reserved2));
		from->Cookie            = __cpu_to_le16(to->Cookie);
		break;
	}
}

static void
__xlate_chap(CHAP_ENTRY *from, CHAP_ENTRY *to,
    uint8_t direction, uint32_t length)
{
	switch (direction) {
	case GET_DATA:
		from->link              = le16_to_cpu(to->link);
		from->flags             = to->flags;
		from->secretLength      = to->secretLength;
		memcpy(from->secret, to->secret, sizeof(from->secret));
		memcpy(from->user_name, to->user_name, sizeof(from->user_name));
		from->reserved          = le16_to_cpu(to->reserved);
		from->cookie            = le16_to_cpu(to->cookie);
		break;

	case SET_DATA:
		from->link              = __cpu_to_le16(to->link);
		from->flags             = to->flags;
		from->secretLength      = to->secretLength;
		memcpy(from->secret, to->secret, sizeof(from->secret));
		memcpy(from->user_name, to->user_name, sizeof(from->user_name));
		from->reserved          = __cpu_to_le16(to->reserved);
		from->cookie            = __cpu_to_le16(to->cookie);
		break;
	}
}
#endif

/**************************************************************************
 * qla4intioctl_get_flash
 *	This routine reads the requested area of FLASH.
 *
 * Input:
 *	ha    = adapter structure pointer.
 *	ioctl = IOCTL structure pointer.
 *
 * Output:
 *	None
 *
 * Returns:
 *	QLA_SUCCESS = success
 *	QLA_ERROR   = error
 *
 * Context:
 *	Kernel context.
 **************************************************************************/
static int
qla4intioctl_get_flash(scsi_qla_host_t *ha, EXT_IOCTL_ISCSI *ioctl)
{
	int		status = 0;
	dma_buf_t	local_dma_buf;
	INT_ACCESS_FLASH *paccess_flash = NULL;
	uint32_t	mbox_cmd[MBOX_REG_COUNT];
	uint32_t	mbox_sts[MBOX_REG_COUNT];


	ENTER(__func__);
	QL4PRINT(QLP3, printk("scsi%d: %s:\n", ha->host_no, __func__));

	/*
	 * Allocate local flash buffer
	 */
	if (qla4xxx_get_ioctl_scrap_mem(ha, (void **)&paccess_flash,
	    sizeof(INT_ACCESS_FLASH))) {
		/* not enough memory */
		ioctl->Status = EXT_STATUS_NO_MEMORY;
		QL4PRINT(QLP2|QLP4,
		    printk("%s(%d): inst=%d scrap not big enough. "
		    "size requested=%ld.\n",
		    __func__, ha->host_no, ha->instance,
		    (ulong)sizeof(INT_ACCESS_FLASH)));
		goto exit_get_flash;
	}

	/*
	 * Copy user's data to local flash buffer
	 */
	if (copy_from_user((uint8_t *)paccess_flash, Q64BIT_TO_PTR(ioctl->
	    RequestAdr), sizeof(INT_ACCESS_FLASH)) != 0) {
		QL4PRINT(QLP2|QLP4,
		    printk("scsi%d: %s: unable to copy data from user's "
		    "memory area\n", ha->host_no, __func__));

		status = (-EFAULT);
		ioctl->Status = EXT_STATUS_COPY_ERR;
		goto exit_get_flash;
	}

	/*
	 * Allocate DMA memory
	 */
	if (qla4xxx_alloc_dma_memory(ha->pdev, &local_dma_buf,
	    paccess_flash->DataLen) == QLA_ERROR) {
		QL4PRINT(QLP2,
		    printk("scsi%d: %s: unable to allocate dma memory\n",
		    ha->host_no, __func__));

		status = (-ENOMEM);
		ioctl->Status = EXT_STATUS_NO_MEMORY;
		goto exit_get_flash;
	}

	QL4PRINT(QLP4, printk("scsi%d: %s: offset=%08x, len=%08x\n",
	    ha->host_no, __func__, paccess_flash->DataOffset,
	    paccess_flash->DataLen));

	/*
	 * Issue Mailbox Command
	 */
	memset(mbox_cmd, 0, sizeof(mbox_cmd));
	memset(mbox_sts, 0, sizeof(mbox_sts));
	mbox_cmd[0] = MBOX_CMD_READ_FLASH;
	mbox_cmd[1] = LSDW(local_dma_buf.phys_addr);
	mbox_cmd[2] = MSDW(local_dma_buf.phys_addr);
	mbox_cmd[3] = paccess_flash->DataOffset;
	mbox_cmd[4] = paccess_flash->DataLen;

	if (qla4xxx_mailbox_command(ha, 5, 2, &mbox_cmd[0], &mbox_sts[0])
	    == QLA_ERROR) {
		QL4PRINT(QLP2, printk("scsi%d: %s: command failed \n",
		    ha->host_no, __func__));

		ioctl->Status = EXT_STATUS_MAILBOX;
		ioctl->DetailStatus = mbox_sts[0];
		goto exit_get_flash;
	}

	//FIXME: For byte-order support, this entire structure must be translated
#if BYTE_ORDER_SUPPORT_ENABLED
	/*
	 * Copy data from DMA buffer into access_flash->FlashData buffer
	 * (in the process, translating for byte-order support, if necessary)
	 */
	switch (paccess_flash->DataOffset & INT_ISCSI_PAGE_MASK) {
	case INT_ISCSI_SYSINFO_FLASH_OFFSET:
		__xlate_sys_info((FLASH_SYS_INFO *) local_dma_buf.virt_addr,
		    (FLASH_SYS_INFO *) &paccess_flash->FlashData[0],
		    GET_DATA, paccess_flash->DataLen);
		break;
	case INT_ISCSI_DRIVER_FLASH_OFFSET:
		__xlate_driver_info(
		    (INT_FLASH_DRIVER_PARAM *)local_dma_buf.virt_addr,
		    (INT_FLASH_DRIVER_PARAM *)&paccess_flash->FlashData[0],
		    GET_DATA, paccess_flash->DataLen);
		break;
	case INT_ISCSI_INITFW_FLASH_OFFSET:
		__xlate_init_fw_ctrl_blk(
		    (INIT_FW_CTRL_BLK *)local_dma_buf.virt_addr,
		    (INIT_FW_CTRL_BLK *) &paccess_flash->FlashData[0],
		    GET_DATA, paccess_flash->DataLen);
		break;
	case INT_ISCSI_DDB_FLASH_OFFSET:
		__xlate_dev_db((DEV_DB_ENTRY *)local_dma_buf.virt_addr,
		    (DEV_DB_ENTRY *) &paccess_flash->FlashData[0],
		    GET_DATA, paccess_flash->DataLen);
		break;
	case INT_ISCSI_CHAP_FLASH_OFFSET:
		__xlate_chap((CHAP_ENTRY *) local_dma_buf.virt_addr,
		    (CHAP_ENTRY *) &paccess_flash->FlashData[0],
		    GET_DATA, paccess_flash->DataLen);
		break;
	case INT_ISCSI_FW_IMAGE2_FLASH_OFFSET:
	case INT_ISCSI_FW_IMAGE1_FLASH_OFFSET:
	default:
		memcpy(&paccess_flash->FlashData[0], local_dma_buf.virt_addr,
		    MIN(local_dma_buf.buf_len,
		    sizeof(paccess_flash->FlashData)));
		break;
	}
#else
	memcpy(&paccess_flash->FlashData[0], local_dma_buf.virt_addr,
	    MIN(local_dma_buf.buf_len, sizeof(paccess_flash->FlashData)));

#endif

	/*
	 * Copy local DMA buffer to user's response data area
	 */
	if (copy_to_user(Q64BIT_TO_PTR(ioctl->ResponseAdr),
	    paccess_flash, sizeof(*paccess_flash)) != 0) {
		QL4PRINT(QLP2,
		    printk("scsi%d: %s: unable to copy data to user's "
		    "memory area\n", ha->host_no, __func__));

		status = (-EFAULT);
		ioctl->Status = EXT_STATUS_COPY_ERR;
		goto exit_get_flash;
	}

	ioctl->Status = EXT_STATUS_OK;
	ioctl->ResponseLen = paccess_flash->DataLen;

	QL4PRINT(QLP10,
	    printk("INT_ACCESS_FLASH buffer (1st 60h bytes only):\n"));
	qla4xxx_dump_bytes(QLP10, paccess_flash, 0x60);

	exit_get_flash:
	/*
	 * Free Memory
	 */

	if (local_dma_buf.virt_addr)
		qla4xxx_free_dma_memory(ha->pdev, &local_dma_buf);

	qla4xxx_free_ioctl_scrap_mem(ha);

	LEAVE(__func__);
	return(status);
}

/**************************************************************************
 * qla4intioctl_get_driver_debug_level
 *	This routine retrieves the driver's debug print level.
 *
 * Input:
 *	ha    = adapter structure pointer.
 *	ioctl = IOCTL structure pointer.
 *
 * Output:
 *	None
 *
 * Returns:
 *	QLA_SUCCESS = success
 *	QLA_ERROR   = error
 *
 * Context:
 *	Kernel context.
 **************************************************************************/
static int
qla4intioctl_get_driver_debug_level(scsi_qla_host_t *ha, EXT_IOCTL_ISCSI *ioctl)
{
	int		status = 0;
	uint32_t	dbg_level;


	ENTER(__func__);

	if (qla4xxx_get_debug_level(&dbg_level) == QLA_ERROR) {
		QL4PRINT(QLP2,
		    printk("scsi%d: %s: unable to set debug level, "
		    "debug driver not loaded!\n",
		    ha->host_no, __func__));

		ioctl->Status = EXT_STATUS_ERR;
		goto exit_get_driver_debug_level;
	}

	if (copy_to_user(Q64BIT_TO_PTR(ioctl->ResponseAdr), &dbg_level,
	    sizeof(dbg_level)) != 0) {
		QL4PRINT(QLP2|QLP4,
		    printk("scsi%d: %s: failed to copy data\n",
		    ha->host_no, __func__));

		status = (-EFAULT);
		ioctl->Status = EXT_STATUS_COPY_ERR;
		goto exit_get_driver_debug_level;
	}

	QL4PRINT(QLP2|QLP4, printk("scsi%d: %s: debug level is %04x\n",
	    ha->host_no, __func__, dbg_level));

	ioctl->Status = EXT_STATUS_OK;

	exit_get_driver_debug_level:

	LEAVE(__func__);
	return(status);
}

/**************************************************************************
 * qla4intioctl_get_host_no
 *	This routine retrieves the host number for the specified adapter
 *	instance.
 *
 * Input:
 *	ha    = adapter structure pointer.
 *	ioctl = IOCTL structure pointer.
 *
 * Output:
 *	None
 *
 * Returns:
 *	QLA_SUCCESS = success
 *	QLA_ERROR   = error
 *
 * Context:
 *	Kernel context.
 **************************************************************************/
static int
qla4intioctl_get_host_no(scsi_qla_host_t *ha, EXT_IOCTL_ISCSI *ioctl)
{
	int	status = 0;


	ENTER(__func__);

	QL4PRINT(QLP4, printk("scsi%d: %s: for instance %d\n",
	    ha->host_no, __func__, ha->instance));

	if (copy_to_user(Q64BIT_TO_PTR(ioctl->ResponseAdr), &(ha->host_no),
	    sizeof(ha->host_no)) != 0) {
		QL4PRINT(QLP2|QLP4,
		    printk("scsi%d: %s: failed to copy data\n",
		    ha->host_no,        __func__));

		status = (-EFAULT);
		ioctl->Status = EXT_STATUS_COPY_ERR;
	} else {
		ioctl->Status = EXT_STATUS_OK;
	}

	LEAVE(__func__);
	return(status);
}

/**************************************************************************
 * qla4intioctl_get_data
 *	This routine calls get data IOCTLs based on the IOCTL Sub Code.
 *
 * Input:
 *	ha    = adapter structure pointer.
 *	ioctl = IOCTL structure pointer.
 *
 * Output:
 *	None
 *
 * Returns:
 *	QLA_SUCCESS = success
 *	QLA_ERROR   = error
 *    	-EINVAL     = if the command is invalid
 *
 * Context:
 *	Kernel context.
 **************************************************************************/
static int
qla4intioctl_get_data(scsi_qla_host_t *ha, EXT_IOCTL_ISCSI *ioctl)
{
	int	status = 0;

	switch (ioctl->SubCode) {
	case INT_SC_GET_FLASH:
		return(qla4intioctl_get_flash(ha, ioctl));

	case INT_SC_GET_DRIVER_DEBUG_LEVEL:
		return(qla4intioctl_get_driver_debug_level(ha, ioctl));

	case INT_SC_GET_HOST_NO:
		return(qla4intioctl_get_host_no(ha, ioctl));

	default:
		QL4PRINT(QLP2,
		    printk("scsi%d: %s: unsupported internal get data "
		    "sub-command code (%X)\n",
		    ha->host_no, __func__, ioctl->SubCode));

		ioctl->Status = EXT_STATUS_UNSUPPORTED_SUBCODE;
		break;
	}
	return status;
}

/**************************************************************************
 * qla4intioctl_set_flash
 *	This routine writes the requested area of FLASH.
 *
 * Input:
 *	ha    = adapter structure pointer.
 *	ioctl = IOCTL structure pointer.
 *
 * Output:
 *	None
 *
 * Returns:
 *	QLA_SUCCESS = success
 *	QLA_ERROR   = error
 *
 * Context:
 *	Kernel context.
 **************************************************************************/
static int
qla4intioctl_set_flash(scsi_qla_host_t *ha, EXT_IOCTL_ISCSI *ioctl)
{
	int			status = 0;
	dma_buf_t		local_dma_buf = { 0 };
	INT_ACCESS_FLASH	*paccess_flash = NULL;
	uint32_t		mbox_cmd[MBOX_REG_COUNT];
	uint32_t		mbox_sts[MBOX_REG_COUNT];


	ENTER(__func__);
	QL4PRINT(QLP3, printk("scsi%d: %s:\n", ha->host_no, __func__));

	/*
	 * Allocate local flash buffer
	 */
	if (qla4xxx_get_ioctl_scrap_mem(ha, (void **)&paccess_flash,
	    sizeof(INT_ACCESS_FLASH))) {
		/* not enough memory */
		ioctl->Status = EXT_STATUS_NO_MEMORY;
		QL4PRINT(QLP2|QLP4,
		    printk("%s(%d): inst=%d scrap not big enough. "
		    "size requested=%ld.\n",
		    __func__, ha->host_no, ha->instance,
		    (ulong)sizeof(INT_ACCESS_FLASH)));
		goto exit_set_flash;
	}

	/*
	 * Copy user's data to local DMA buffer
	 */
	if (copy_from_user((uint8_t *)paccess_flash,
	    Q64BIT_TO_PTR(ioctl->RequestAdr), sizeof(*paccess_flash)) != 0) {
		QL4PRINT(QLP2,
		    printk("scsi%d: %s: unable to copy data from user's "
		    "memory area\n", ha->host_no, __func__));

		status = (-EFAULT);
		ioctl->Status = EXT_STATUS_COPY_ERR;
		goto exit_set_flash;
	}

	/*
	 * Allocate DMA memory
	 */
	if (qla4xxx_alloc_dma_memory(ha->pdev, &local_dma_buf,
	    paccess_flash->DataLen) == QLA_ERROR) {
		QL4PRINT(QLP2,
		    printk("scsi%d: %s: unable to allocate dma memory\n",
		    ha->host_no, __func__));

		status = (-ENOMEM);
		ioctl->Status = EXT_STATUS_NO_MEMORY;
		goto exit_set_flash;
	}

	//FIXME: For byte-order support, this entire structure must be translated
#if BYTE_ORDER_SUPPORT_ENABLED
	/*
	 * Copy data from DMA buffer into access_flash->FlashData buffer
	 * (in the process, translating for byte-order support, if necessary)
	 */
	switch (access_flash->DataOffset & INT_ISCSI_PAGE_MASK) {
	case INT_ISCSI_SYSINFO_FLASH_OFFSET:
		__xlate_sys_info((FLASH_SYS_INFO *) &access_flash->FlashData[0],
		    (FLASH_SYS_INFO *) local_dma_buf.virt_addr,
		    SET_DATA, access_flash->DataLen);
		break;

	case INT_ISCSI_DRIVER_FLASH_OFFSET:
		__xlate_driver_info(
		    (INT_FLASH_DRIVER_PARAM *)&access_flash->FlashData[0],
		    (INT_FLASH_DRIVER_PARAM *) local_dma_buf.virt_addr,
		    SET_DATA, access_flash->DataLen);
		break;

	case INT_ISCSI_INITFW_FLASH_OFFSET:
		__xlate_init_fw_ctrl_blk(
		    (INIT_FW_CTRL_BLK *)&access_flash->FlashData[0],
		    (INIT_FW_CTRL_BLK *) local_dma_buf.virt_addr,
		    SET_DATA, access_flash->DataLen);
		break;

	case INT_ISCSI_DDB_FLASH_OFFSET:
		__xlate_dev_db((DEV_DB_ENTRY *) &access_flash->FlashData[0],
		    (DEV_DB_ENTRY *) local_dma_buf.virt_addr,
		    SET_DATA, access_flash->DataLen);
		break;

	case INT_ISCSI_CHAP_FLASH_OFFSET:
		__xlate_chap((CHAP_ENTRY *) &access_flash->FlashData[0],
		    (CHAP_ENTRY *) local_dma_buf.virt_addr,
		    SET_DATA, access_flash->DataLen);
		break;

	case INT_ISCSI_FW_IMAGE2_FLASH_OFFSET:
	case INT_ISCSI_FW_IMAGE1_FLASH_OFFSET:
	default:
		memcpy(local_dma_buf.virt_addr, &access_flash->FlashData[0],
		    MIN(local_dma_buf.buf_len,
		    sizeof(access_flash->FlashData)));
		break;
	}
#else
	memcpy(local_dma_buf.virt_addr, &paccess_flash->FlashData[0],
	    MIN(local_dma_buf.buf_len, sizeof(paccess_flash->FlashData)));
#endif

	/*
	 * Issue Mailbox Command
	 */
	memset(mbox_cmd, 0, sizeof(mbox_cmd));
	memset(mbox_sts, 0, sizeof(mbox_sts));
	mbox_cmd[0] = MBOX_CMD_WRITE_FLASH;
	mbox_cmd[1] = LSDW(local_dma_buf.phys_addr);
	mbox_cmd[2] = MSDW(local_dma_buf.phys_addr);
	mbox_cmd[3] = paccess_flash->DataOffset;
	mbox_cmd[4] = paccess_flash->DataLen;
	mbox_cmd[5] = paccess_flash->Options;

	if (qla4xxx_mailbox_command(ha, 6, 2, &mbox_cmd[0], &mbox_sts[0])
	    == QLA_ERROR) {
		QL4PRINT(QLP2, printk("scsi%d: %s: command failed \n",
		    ha->host_no, __func__));

		ioctl->Status = EXT_STATUS_MAILBOX;
		ioctl->DetailStatus = mbox_sts[0];
		ioctl->VendorSpecificStatus[0] = mbox_sts[1];
		goto exit_set_flash;
	}

	ioctl->Status = EXT_STATUS_OK;
	ioctl->ResponseLen = paccess_flash->DataLen;

	QL4PRINT(QLP10,
	    printk("scsi%d): INT_ACCESS_FLASH buffer (1st 60h bytes only:\n",
	   ha->host_no));
	qla4xxx_dump_bytes(QLP10, local_dma_buf.virt_addr, 0x60);

	exit_set_flash:
	/*
	 * Free Memory
	 */
	if (local_dma_buf.virt_addr)
		qla4xxx_free_dma_memory(ha->pdev, &local_dma_buf);

	qla4xxx_free_ioctl_scrap_mem(ha);

	LEAVE(__func__);
	return(status);
}

/**************************************************************************
 * qla4intioctl_set_driver_debug_level
 *	This routine sets the driver's debug print level.
 *
 * Input:
 *	ha    = adapter structure pointer.
 *	ioctl = IOCTL structure pointer.
 *
 * Output:
 *	None
 *
 * Returns:
 *	QLA_SUCCESS = success
 *	QLA_ERROR   = error
 *
 * Context:
 *	Kernel context.
 **************************************************************************/
static int
qla4intioctl_set_driver_debug_level(scsi_qla_host_t *ha, EXT_IOCTL_ISCSI *ioctl)
{
	int		status = 0;
	uint32_t	dbg_level;


	ENTER(__func__);

	if (copy_from_user(&dbg_level, Q64BIT_TO_PTR(ioctl->RequestAdr),
	    sizeof(dbg_level)) != 0) {
		QL4PRINT(QLP2|QLP4,
		    printk("scsi%d: %s: failed to copy data\n",
		    ha->host_no, __func__));

		status = (-EFAULT);
		ioctl->Status = EXT_STATUS_COPY_ERR;
		goto exit_set_driver_debug_level;
	}

	if (qla4xxx_set_debug_level(dbg_level) == QLA_ERROR) {
		QL4PRINT(QLP2,
		    printk("scsi%d: %s: unable to set debug level, "
		    "debug driver not loaded!\n",
		    ha->host_no, __func__));

		ioctl->Status = EXT_STATUS_ERR;
		goto exit_set_driver_debug_level;
	}

	QL4PRINT(QLP2|QLP4,
	    printk("scsi%d: %s: debug level set to 0x%04X\n",
	    ha->host_no, __func__, dbg_level));

	ioctl->Status = EXT_STATUS_OK;

	exit_set_driver_debug_level:

	LEAVE(__func__);
	return(status);
}

/**************************************************************************
 * qla4intioctl_set_data
 *	This routine calls set data IOCTLs based on the IOCTL Sub Code.
 *
 * Input:
 *	ha    = adapter structure pointer.
 *	ioctl = IOCTL structure pointer.
 *
 * Output:
 *	None
 *
 * Returns:
 *	QLA_SUCCESS = success
 *	QLA_ERROR   = error
 *    	-EINVAL     = if the command is invalid
 *
 * Context:
 *	Kernel context.
 **************************************************************************/
static int
qla4intioctl_set_data(scsi_qla_host_t *ha, EXT_IOCTL_ISCSI *ioctl)
{
	int	status = 0;

	switch (ioctl->SubCode) {
	case INT_SC_SET_FLASH:
		return(qla4intioctl_set_flash(ha, ioctl));

	case INT_SC_SET_DRIVER_DEBUG_LEVEL:
		return(qla4intioctl_set_driver_debug_level(ha, ioctl));

	default:
		QL4PRINT(QLP2,
		    printk("scsi%d: %s: unsupported internal set data "
		    "sub-command code (%X)\n",
		    ha->host_no, __func__, ioctl->SubCode));

		ioctl->Status = EXT_STATUS_UNSUPPORTED_SUBCODE;
		break;
	}
	return (status);
}

/**************************************************************************
 * qla4intioctl_hba_reset
 *	This routine resets the specified HBA.
 *
 * Input:
 *	ha    = adapter structure pointer.
 *	ioctl = IOCTL structure pointer.
 *
 * Output:
 *	None
 *
 * Returns:
 *	QLA_SUCCESS = success
 *	QLA_ERROR   = error
 *
 * Context:
 *	Kernel context.
 **************************************************************************/
static int
qla4intioctl_hba_reset(scsi_qla_host_t *ha, EXT_IOCTL_ISCSI *ioctl)
{
	int	status = 0;
	u_long	wait_count;


	ENTER(__func__);
	QL4PRINT(QLP4, printk("scsi%d: %s:\n", ha->host_no, __func__));

	switch (ioctl->SubCode) {
	case INT_SC_HBA_RESET:
	case INT_SC_FIRMWARE_RESET:
		set_bit(DPC_RESET_HA_DESTROY_DDB_LIST, &ha->dpc_flags);

		/* Wait a fixed amount of time for reset to complete */
		wait_count = jiffies + ADAPTER_RESET_TOV * HZ;
		while (test_bit(DPC_RESET_HA_DESTROY_DDB_LIST,
		    &ha->dpc_flags) != 0) {
			if (wait_count <= jiffies)
				break;

			/* wait for 1 second */
			set_current_state(TASK_UNINTERRUPTIBLE);
			schedule_timeout(1*HZ);
		}

		if (test_bit(AF_ONLINE, &ha->flags))
			status = QLA_SUCCESS;
		else
			status = QLA_ERROR;

		break;

	case INT_SC_TARGET_WARM_RESET:
	case INT_SC_LUN_RESET:
		break;
	}

	if (status == QLA_SUCCESS) {
		QL4PRINT(QLP4, printk("scsi%d: %s: Succeeded\n",
		    ha->host_no, __func__));

		status = QLA_SUCCESS;
		ioctl->Status = EXT_STATUS_OK;
	} else {
		QL4PRINT(QLP4, printk("scsi%d: %s: FAILED\n",
		    ha->host_no, __func__));

		ioctl->Status = EXT_STATUS_ERR;
	}

	LEAVE(__func__);
	return(status);
}

/**************************************************************************
 * qla4intioctl_copy_fw_flash
 *	This routine requests copying the FW image in FLASH from primary-to-
 *	secondary or secondary-to-primary.
 *
 * Input:
 *	ha    = adapter structure pointer.
 *	ioctl = IOCTL structure pointer.
 *
 * Output:
 *	None
 *
 * Returns:
 *	QLA_SUCCESS = success
 *	QLA_ERROR   = error
 *
 * Context:
 *	Kernel context.
 **************************************************************************/
static int
qla4intioctl_copy_fw_flash(scsi_qla_host_t *ha, EXT_IOCTL_ISCSI *ioctl)
{
	int			status = 0;
	INT_COPY_FW_FLASH	copy_flash;
	uint32_t		mbox_cmd[MBOX_REG_COUNT];
	uint32_t		mbox_sts[MBOX_REG_COUNT];


	ENTER(__func__);
	QL4PRINT(QLP3, printk("scsi%d: %s:\n", ha->host_no, __func__));

	if (copy_from_user((uint8_t *)&copy_flash,
	    Q64BIT_TO_PTR(ioctl->RequestAdr), ioctl->RequestLen) != 0) {
		QL4PRINT(QLP2,
		    printk("scsi%d: %s: unable to copy data from user's "
		    "memory area\n", ha->host_no, __func__));

		status = (-EFAULT);
		ioctl->Status = EXT_STATUS_COPY_ERR;
		goto exit_copy_flash;
	}

	memset(mbox_cmd, 0, sizeof(mbox_cmd));
	memset(mbox_sts, 0, sizeof(mbox_sts));
	mbox_cmd[0] = MBOX_CMD_COPY_FLASH;
	mbox_cmd[1] = copy_flash.Options;

	if (qla4xxx_mailbox_command(ha, 2, 2, &mbox_cmd[0], &mbox_sts[0])
	    == QLA_SUCCESS) {
		QL4PRINT(QLP10, printk("scsi%d: %s: Succeeded\n",
		    ha->host_no, __func__));
		ioctl->Status = EXT_STATUS_OK;
	} else {
		QL4PRINT(QLP10, printk("scsi%d: %s: FAILED\n",
		    ha->host_no, __func__));

		ioctl->Status = EXT_STATUS_MAILBOX;
		ioctl->DetailStatus = mbox_sts[0];
		ioctl->VendorSpecificStatus[0] = mbox_sts[1];
	}

	exit_copy_flash:

	LEAVE(__func__);
	return(status);
}

/**************************************************************************
 * qla4xxx_iocb_pass_done
 *	This routine resets the ioctl progress flag and wakes up the ioctl
 * 	completion semaphore.
 *
 * Input:
 *	ha    = adapter structure pointer.
 *   	sts_entry - pointer to passthru status buffer.
 *
 * Output:
 *	None
 *
 * Returns:
 *	None
 *
 * Context:
 *	Interrupt context.
 **************************************************************************/
static void
qla4xxx_iocb_pass_done(scsi_qla_host_t *ha, PASSTHRU_STATUS_ENTRY *sts_entry)
{
	INT_IOCB_PASSTHRU *iocb;


	ENTER(__func__);

	/* --- Copy passthru status buffer to iocb passthru buffer ---*/
	iocb = (INT_IOCB_PASSTHRU *)(ulong)le32_to_cpu(sts_entry->handle);
	memcpy(iocb->IOCBStatusBuffer, sts_entry,
	    MIN(sizeof(iocb->IOCBStatusBuffer), sizeof(*sts_entry)));

	/* --- Reset IOCTL flags and wakeup semaphore.
	 *     But first check to see if IOCTL has already
	 *     timed out because we don't want to get the
	 *     up/down semaphore counters off.             --- */
	if (ha->ioctl_iocb_pass_in_progress == TRUE) {
		ha->ioctl_iocb_pass_in_progress = FALSE;
		ha->ioctl_tov = 0;

		QL4PRINT(QLP10, printk("%s: UP count=%d\n",
		    __func__, atomic_read(&ha->ioctl_cmpl_sem.count)));
		up(&ha->ioctl_cmpl_sem);
	}

	LEAVE(__func__);
	return;
}


#ifndef QLA4000
/**************************************************************************
 * qla4intioctl_iocb_passthru
 *	This routine
 *	
 *
 * Input:
 *	ha    = adapter structure pointer.
 *	ioctl = IOCTL structure pointer.
 *
 * Output:
 *	None
 *
 * Returns:
 *	QLA_SUCCESS = success
 *	QLA_ERROR   = error
 *
 * Context:
 *	Kernel context.
 **************************************************************************/
static int
qla4intioctl_iocb_passthru(scsi_qla_host_t *ha, EXT_IOCTL_ISCSI *ioctl)
{
	int			status = 0;
	INT_IOCB_PASSTHRU	*iocb;
	INT_IOCB_PASSTHRU	*iocb_phys;
	PASSTHRU0_ENTRY		*passthru_entry;
	unsigned long		flags;
	DATA_SEG_A64		*data_seg;


	ENTER(__func__);
	QL4PRINT(QLP3, printk("scsi%d: %s:\n", ha->host_no, __func__));

	/* --- Use internal DMA buffer for iocb structure --- */
#if 0 && defined(__VMKERNEL_MODULE__)
	spin_lock(&ha->dma_buf_spinlock);
#else
	down(&ha->dma_buf_sem);
#endif

	if (ha->dma_buf.buf_len < sizeof(*iocb))
		qla4xxx_resize_dma_buf(ha, &ha->dma_buf, sizeof(*iocb));

	if (!ha->dma_buf.virt_addr || ha->dma_buf.buf_len < sizeof(*iocb)) {
		QL4PRINT(QLP2,
		    printk("scsi%d: %s: dma buffer inaccessible.\n",
				      ha->host_no, __func__));
		ioctl->Status = EXT_STATUS_BUFFER_TOO_SMALL;
		goto exit_iocb_passthru;
	}

	iocb = (INT_IOCB_PASSTHRU *) ha->dma_buf.virt_addr;
	iocb_phys = (INT_IOCB_PASSTHRU *)(unsigned long)ha->dma_buf.phys_addr;

	/* --- Copy IOCB_PASSTHRU structure from user space --- */
	if (copy_from_user((uint8_t *)iocb, Q64BIT_TO_PTR(ioctl->RequestAdr),
	    ioctl->RequestLen) != 0) {
		QL4PRINT(QLP2,
		    printk("scsi%d: %s: unable to copy data from user's "
		    "memory area\n", ha->host_no, __func__));

		status = (-EFAULT);
		ioctl->Status = EXT_STATUS_COPY_ERR;
		goto exit_iocb_passthru;
	}

	if ((iocb->IOCBCmdBuffer[0x00] == 0x3A) &&
	    (iocb->IOCBCmdBuffer[0x0A] == 0x10) &&
	    (iocb->IOCBCmdBuffer[0x0B] == 0x80) &&
	    (iocb->SendData[0x0C] == 0x81) &&
	    (iocb->SendData[0x0D] == 0x4F)) {
		// ok to process command, proceed ...
	} else {
		QL4PRINT(QLP2, printk("scsi%d: %s: unable to process command. "
                                      "Did not pass secure I/O boundary check.\n",
                                      ha->host_no, __func__));
		status = (-EFAULT);
		ioctl->Status = EXT_STATUS_INVALID_PARAM;
		goto exit_iocb_passthru;
	}

	/* --- Get pointer to the passthru queue entry --- */
	spin_lock_irqsave(&ha->hardware_lock, flags);
	if (qla4xxx_get_req_pkt(ha, (QUEUE_ENTRY **) &passthru_entry)
	    != QLA_SUCCESS) {
		QL4PRINT(QLP2,
		    printk("scsi%d: %s: request queue full, try again later\n",
		    ha->host_no, __func__));

		ioctl->Status = EXT_STATUS_HBA_QUEUE_FULL;
		spin_unlock_irqrestore(&ha->hardware_lock, flags);
		goto exit_iocb_passthru;
	}

	/* --- Fill in passthru queue entry --- */
	if (iocb->SendDMAOffset) {
		data_seg = (DATA_SEG_A64 *)(iocb->IOCBCmdBuffer +
		    iocb->SendDMAOffset);
		data_seg->base.addrHigh =
		    __cpu_to_le32(MSDW(iocb_phys->SendData));
		data_seg->base.addrLow  =
		    __cpu_to_le32(LSDW(iocb_phys->SendData));
	}

	if (iocb->RspDMAOffset) {
		data_seg = (DATA_SEG_A64 *)(iocb->IOCBCmdBuffer +
		    iocb->RspDMAOffset);
		data_seg->base.addrHigh =
		    __cpu_to_le32(MSDW(iocb_phys->RspData));
		data_seg->base.addrLow  =
		    __cpu_to_le32(LSDW(iocb_phys->RspData));
	}

	memcpy(passthru_entry, iocb->IOCBCmdBuffer,
	    MIN(sizeof(*passthru_entry), sizeof(iocb->IOCBCmdBuffer)));
	passthru_entry->handle = (uint32_t) (unsigned long) iocb;
	passthru_entry->hdr.systemDefined = SD_PASSTHRU_IOCB;

	if (passthru_entry->hdr.entryType != ET_PASSTHRU0)
		passthru_entry->timeout = MBOX_TOV;

	QL4PRINT(QLP10,
	    printk(KERN_INFO
	    "scsi%d: Passthru0 IOCB type %x count %x In (%x) %p\n",
	    ha->host_no, passthru_entry->hdr.entryType,
	    passthru_entry->hdr.entryCount, ha->request_in, passthru_entry));

	QL4PRINT(QLP10,
	    printk(KERN_INFO "scsi%d: Dump Passthru entry %p: \n",
	    ha->host_no, passthru_entry));
	qla4xxx_dump_bytes(QLP10, passthru_entry, sizeof(*passthru_entry));

	/* ---- Prepare for receiving completion ---- */
	ha->ioctl_iocb_pass_in_progress = TRUE;
	ha->ioctl_tov = passthru_entry->timeout * HZ;
	qla4xxx_ioctl_sem_init(ha);

	/* ---- Send command to adapter ---- */
	ha->ioctl_cmpl_timer.expires = jiffies + ha->ioctl_tov;
	add_timer(&ha->ioctl_cmpl_timer);

	WRT_REG_DWORD(&ha->reg->requestQueueInPointer, ha->request_in);
	PCI_POSTING(&ha->reg->requestQueueInPointer);
	spin_unlock_irqrestore(&ha->hardware_lock, flags);

	QL4PRINT(QLP10,
	    printk("%s: DOWN count=%d\n",
	    __func__, atomic_read(&ha->ioctl_cmpl_sem.count)));
	down(&ha->ioctl_cmpl_sem);

	/*******************************************************
	 *						       *
	 *             Passthru Completion                     *
	 *						       *
	 *******************************************************/
	del_timer(&ha->ioctl_cmpl_timer);

	/* ---- Check for timeout --- */
	if (ha->ioctl_iocb_pass_in_progress == TRUE) {
		QL4PRINT(QLP2,
		    printk("scsi%d: %s: ERROR = command timeout.\n",
		    ha->host_no, __func__));

		ha->ioctl_iocb_pass_in_progress = FALSE;
		ioctl->Status = EXT_STATUS_ERR;
		goto exit_iocb_passthru;
	}

	/* ---- Copy IOCB Passthru structure with updated status buffer
	 *      to user space ---- */

	iocb->RspDataLen -= passthru_entry->residual;

	if (copy_to_user(Q64BIT_TO_PTR(ioctl->ResponseAdr),
	    iocb, sizeof(INT_IOCB_PASSTHRU)) != 0) {
		QL4PRINT(QLP2,
		    printk("scsi%d: %s: unable to copy passthru struct "
		    "to user's memory area.\n",
		    ha->host_no, __func__));

		status = (-EFAULT);
		ioctl->Status = EXT_STATUS_COPY_ERR;
		goto exit_iocb_passthru;
	}

	QL4PRINT(QLP10, printk("Dump iocb structure (OUT)\n"));
	qla4xxx_print_iocb_passthru(QLP10, ha, iocb);

	QL4PRINT(QLP4, printk("scsi%d: %s: Succeeded\n",
	    ha->host_no, __func__));

	ioctl->Status = EXT_STATUS_OK;

	exit_iocb_passthru:
#if 0 && defined(__VMKERNEL_MODULE__)
	spin_unlock(&ha->dma_buf_spinlock);
#else
	up(&ha->dma_buf_sem);
#endif

	LEAVE(__func__);
	return(status);
}
#endif


 /**************************************************************************
 * qla4intioctl_restore_factory_defaults
 *	This routine restores factory defaults of the adapter.
 *
 * Input:
 *	ha    = adapter structure pointer.
 *	ioctl = IOCTL structure pointer.
 *
 * Output:
 *	None
 *
 * Returns:
 *	QLA_SUCCESS = success
 *	QLA_ERROR   = error
 *
 * Context:
 *	Kernel context.
 **************************************************************************/
static int
qla4intioctl_restore_factory_defaults(scsi_qla_host_t *ha, EXT_IOCTL_ISCSI *ioctl)
{
	int status = 0;
	INT_RESTORE_FACTORY_DEFAULTS defaults;

	ENTER(__func__);
	QL4PRINT(QLP4,
	    printk("scsi%d: %s: inst %d entered.\n",
	    ha->host_no, __func__, ha->instance));

	if (ioctl->RequestLen > sizeof(INT_RESTORE_FACTORY_DEFAULTS)) {
		QL4PRINT(QLP2|QLP4,
		    printk("scsi%d: %s: memory area too small\n",
		    ha->host_no, __func__));

		ioctl->Status = EXT_STATUS_BUFFER_TOO_SMALL;
		goto exit_defaults;
	}

	/* --- Copy logout structure from user space --- */
	if ((status = copy_from_user((void *)&defaults,
	    Q64BIT_TO_PTR(ioctl->RequestAdr), sizeof(INT_RESTORE_FACTORY_DEFAULTS))) != 0) {
		QL4PRINT(QLP2|QLP4,
		    printk("scsi%d: %s: unable to copy data from "
		    "user's memory area\n", ha->host_no, __func__));

		ioctl->Status = EXT_STATUS_COPY_ERR;
		goto exit_defaults;
	}

	/* --- Execute command --- */
	if (qla4xxx_restore_factory_defaults(ha, defaults.BlockMask,
	    defaults.IFCBMask) == QLA_SUCCESS) {
		QL4PRINT(QLP4,
		    printk("scsi%d: %s: RESTORE_FACTORY_DEFAULTS SUCCEEDED!\n",
			   ha->host_no, __func__));

		ioctl->Status = EXT_STATUS_OK;
	} else {
		QL4PRINT(QLP2|QLP4,
		    printk("scsi%d: %s: RESTORE_FACTORY_DEFAULTS FAILED\n",
		    ha->host_no, __func__));

		ioctl->Status = EXT_STATUS_ERR;
	}

exit_defaults:
	QL4PRINT(QLP4,
	    printk("scsi%d: %s: inst %d exiting.\n",
	    ha->host_no, __func__, ha->instance));
	LEAVE(__func__);

	return(status);
}

typedef struct {
	int cmd;
	char *s;
} ioctl_tbl_row_t;

ioctl_tbl_row_t IOCTL_CMD_TBL[] =
{
	{EXT_CC_QUERY, "EXT_CC_QUERY"},
	{EXT_CC_REG_AEN, "EXT_CC_REG_AEN"},
	{EXT_CC_GET_AEN, "EXT_CC_GET_AEN"},
	{EXT_CC_GET_DATA, "EXT_CC_GET_DATA"},
	{EXT_CC_SET_DATA, "EXT_CC_SET_DATA"},
	{EXT_CC_SEND_SCSI_PASSTHRU, "EXT_CC_SEND_SCSI_PASSTHRU"},
	{EXT_CC_SEND_ISCSI_PASSTHRU, "EXT_CC_SEND_ISCSI_PASSTHRU"},
	{INT_CC_LOGOUT_ISCSI, "INT_CC_LOGOUT_ISCSI"},
	{EXT_CC_GET_HBACNT, "EXT_CC_GET_HBACNT"},
	{INT_CC_DIAG_PING, "INT_CC_DIAG_PING"},
	{INT_CC_GET_DATA, "INT_CC_GET_DATA"},
	{INT_CC_SET_DATA, "INT_CC_SET_DATA"},
	{INT_CC_HBA_RESET, "INT_CC_HBA_RESET"},
	{INT_CC_COPY_FW_FLASH, "INT_CC_COPY_FW_FLASH"},
	{INT_CC_IOCB_PASSTHRU, "INT_CC_IOCB_PASSTHRU"},
	{INT_CC_RESTORE_FACTORY_DEFAULTS, "INT_CC_RESTORE_FACTORY_DEFAULTS"},
	{0, "UNKNOWN"}
};

ioctl_tbl_row_t IOCTL_SCMD_QUERY_TBL[] =
{
	{EXT_SC_QUERY_HBA_ISCSI_NODE, "EXT_SC_QUERY_HBA_ISCSI_NODE"},
	{EXT_SC_QUERY_HBA_ISCSI_PORTAL, "EXT_SC_QUERY_HBA_ISCSI_PORTAL"},
	{EXT_SC_QUERY_DISC_ISCSI_NODE, "EXT_SC_QUERY_DISC_ISCSI_NODE"},
	{EXT_SC_QUERY_DISC_ISCSI_PORTAL, "EXT_SC_QUERY_DISC_ISCSI_PORTAL"},
	{EXT_SC_QUERY_DRIVER, "EXT_SC_QUERY_DRIVER"},
	{EXT_SC_QUERY_FW, "EXT_SC_QUERY_FW"},
	{EXT_SC_QUERY_CHIP, "EXT_SC_QUERY_CHIP"},
	{0, "UNKNOWN"}
};

ioctl_tbl_row_t IOCTL_SCMD_EGET_DATA_TBL[] =
{
	{EXT_SC_GET_STATISTICS_ISCSI, "EXT_SC_GET_STATISTICS_ISCSI"},
	{EXT_SC_GET_DEVICE_ENTRY_ISCSI, "EXT_SC_GET_DEVICE_ENTRY_ISCSI"},
	{EXT_SC_GET_DEVICE_ENTRY_DEFAULTS_ISCSI, "EXT_SC_GET_DEVICE_ENTRY_DEFAULTS_ISCSI"},
	{EXT_SC_GET_INIT_FW_ISCSI, "EXT_SC_GET_INIT_FW_ISCSI"},
	{EXT_SC_GET_INIT_FW_DEFAULTS_ISCSI, "EXT_SC_GET_INIT_FW_DEFAULTS_ISCSI"},
	{EXT_SC_GET_ISNS_SERVER, "EXT_SC_GET_ISNS_SERVER"},
	{EXT_SC_GET_ISNS_DISCOVERED_TARGETS, "EXT_SC_GET_ISNS_DISCOVERED_TARGETS"},
	{0, "UNKNOWN"}
};

ioctl_tbl_row_t IOCTL_SCMD_ESET_DATA_TBL[] =
{
	{EXT_SC_RST_STATISTICS_GEN, "EXT_SC_RST_STATISTICS_GEN"},
	{EXT_SC_RST_STATISTICS_ISCSI, "EXT_SC_RST_STATISTICS_ISCSI"},
	{EXT_SC_SET_DEVICE_ENTRY_ISCSI, "EXT_SC_SET_DEVICE_ENTRY_ISCSI"},
	{EXT_SC_SET_INIT_FW_ISCSI, "EXT_SC_SET_INIT_FW_ISCSI"},
	{EXT_SC_SET_ISNS_SERVER, "EXT_SC_SET_ISNS_SERVER"},
	{0, "UNKNOWN"}
};

ioctl_tbl_row_t IOCTL_SCMD_IGET_DATA_TBL[] =
{
	{INT_SC_GET_FLASH, "INT_SC_GET_FLASH"},
	{INT_SC_GET_DRIVER_DEBUG_LEVEL, "INT_SC_GET_DRIVER_DEBUG_LEVEL"},
	{INT_SC_GET_HOST_NO, "INT_SC_GET_HOST_NO"},
	{0, "UNKNOWN"}
};

ioctl_tbl_row_t IOCTL_SCMD_ISET_DATA_TBL[] =
{
	{INT_SC_SET_FLASH, "INT_SC_SET_FLASH"},
	{INT_SC_SET_DRIVER_DEBUG_LEVEL, "INT_SC_SET_DRIVER_DEBUG_LEVEL"},
	{0, "UNKNOWN"}
};

static char *IOCTL_TBL_STR(int cc, int sc)
{
	ioctl_tbl_row_t *r;
	int cmd;

	switch (cc) {
	case EXT_CC_QUERY:
		r = IOCTL_SCMD_QUERY_TBL;
		cmd = sc;
		break;
	case EXT_CC_GET_DATA:
		r = IOCTL_SCMD_EGET_DATA_TBL;
		cmd = sc;
		break;
	case EXT_CC_SET_DATA:
		r = IOCTL_SCMD_ESET_DATA_TBL;
		cmd = sc;
		break;
	case INT_CC_GET_DATA:
		r = IOCTL_SCMD_IGET_DATA_TBL;
		cmd = sc;
		break;
	case INT_CC_SET_DATA:
		r = IOCTL_SCMD_ISET_DATA_TBL;
		cmd = sc;
		break;

	default:
		r = IOCTL_CMD_TBL;
		cmd = cc;
		break;
	}

	while (r->cmd != 0) {
		if (r->cmd == cmd) break;
		r++;
	}
	return(r->s);

}

/**************************************************************************
 * qla4xxx_ioctl
 * 	This the main entry point for all ioctl requests
 *
 * Input:
 *    	dev - pointer to SCSI device structure
 *	cmd - internal or external ioctl command code
 *	arg - pointer to ioctl structure
 *
 *    	Instance field in ioctl structure - to determine which adapter to
 *    	perform ioctl
 *
 * Output:
 *	Status field in ioctl structure, if returns QLA_SUCCESS,
 *      QLA_ERROR or -EINVAL
 *
 * Returns:
 *	 QLA_SUCCESS
 *	 QLA_ERROR
 *    	-EFAULT      - if the arg pointer is NULL
 *    	-EINVAL      - if the command is invalid
 *
 * Context:
 *	Kernel context.
 **************************************************************************/
int qla4xxx_ioctl(Scsi_Device *dev, int cmd, void *arg)
{
	EXT_IOCTL_ISCSI	*ioctl = NULL;
	scsi_qla_host_t	*ha = NULL;
	int		status = 0;


	ENTER(__func__);

	/*
	 * Since internal IOCTLs are improperly defined in the 3-series IOCTLs
	 * (i.e. no magic number), check only the Signature for internal IOCTLs
	 */
	switch (cmd) {
	case INT_CC_LOGOUT_ISCSI:
	case INT_CC_DIAG_PING:
	case INT_CC_GET_DATA:
	case INT_CC_SET_DATA:
	case INT_CC_HBA_RESET:
	case INT_CC_COPY_FW_FLASH:
	case INT_CC_IOCB_PASSTHRU:
	case INT_CC_RESTORE_FACTORY_DEFAULTS:
		/* proceed to check signature */
		break;
	default:
		/* Catch any non-QLOGIC ioctls */
		if (_IOC_TYPE(cmd) != QLMULTIPATH_MAGIC) {
			status = (-EINVAL);
			goto exit_qla4xxx_ioctl;
		}
		break;
	}

	/* Allocate ioctl structure buffer to support multiple concurrent
	 * entries. NO static structures allowed.
	 */
	ioctl = ql4_kmem_zalloc(sizeof(EXT_IOCTL_ISCSI));
	if (ioctl == NULL) {
		/* error */
		printk(KERN_WARNING
		    "qla4xxx: ERROR in main ioctl buffer allocation.\n");
		status = (-ENOMEM);
		goto exit_qla4xxx_ioctl;
	}

	/*
	 * Check to see if we can access the ioctl command structure
	 */
#ifndef __VMKERNEL_MODULE__
	/* 
         * vmkernel doesn't implement access_ok()
	 */
	if (!access_ok(VERIFY_WRITE, arg, sizeof(EXT_IOCTL_ISCSI))) {
		QL4PRINT(QLP2,
		    printk("%s: EXT_IOCTL_ISCSI verify failed.\n",
		    __func__));

		status = (-EFAULT);
		goto exit_qla4xxx_ioctl;
	}
#endif

	/*
	 * Copy the ioctl command structure from user space
	 */
	if (copy_from_user((uint8_t *)ioctl, arg, sizeof(EXT_IOCTL_ISCSI))) {
		QL4PRINT(QLP2|QLP4,
		    printk("%s: EXT_IOCTL_ISCSI copy read error.\n",
		    __func__));
		status = (-EFAULT);
		goto exit_qla4xxx_ioctl;
	}

	QL4PRINT(QLP10|QLP4, printk("EXT_IOCTL_ISCSI structure: \n"));
	qla4xxx_dump_dwords(QLP10|QLP4, ioctl, sizeof(*ioctl));

	/* Check signature */
	if (strncmp((char *) &ioctl->Signature, "QLOGIC",
	    sizeof(ioctl->Signature)) != QLA_SUCCESS) {
		status = (-EINVAL);
		goto exit_qla4xxx_ioctl;
	}

	/*
	 * Get the adapter handle for the corresponding adapter instance
	 */
	ha = qla4xxx_get_adapter_handle(ioctl->HbaSelect);
	if (ha == NULL) {
		QL4PRINT(QLP2,
		    printk("%s: ha not found for ha instance %d.\n",
		    __func__, ioctl->HbaSelect));

		ioctl->Status = EXT_STATUS_DEV_NOT_FOUND;

		if (copy_to_user(arg, (void *)ioctl, sizeof(EXT_IOCTL_ISCSI))) {
			QL4PRINT(QLP2|QLP4,
			    printk("%s: EXT_IOCTL_ISCSI copy write error.\n",
			    __func__));
			status = (-EFAULT);
		}

		goto exit_qla4xxx_ioctl;
	}

	QL4PRINT(QLP4,
	    printk("scsi%d: ioctl+ (%s)\n",
	    ha->host_no, IOCTL_TBL_STR(cmd, ioctl->SubCode)));

	ha->i_start = jiffies;
	ha->i_end = 0;
	ha->f_start = 0;
	ha->f_end = 0;

	down(&ha->ioctl_sem);

	/*
	 * If the DPC is active, wait for it to complete before proceeding
	 */
	while (ha->dpc_active) {
		set_current_state(TASK_UNINTERRUPTIBLE);
		schedule_timeout(1*HZ);
	}

	/*
	 * Issue the ioctl command
	 */
	switch (cmd) {
	case EXT_CC_QUERY:
		status = qla4extioctl_query(ha, ioctl);
		break;

	case EXT_CC_REG_AEN:
		status = qla4extioctl_reg_aen(ha, ioctl);
		break;

	case EXT_CC_GET_AEN:
		status = qla4extioctl_get_aen(ha, ioctl);
		break;

	case EXT_CC_GET_DATA:
		status = qla4extioctl_get_data(ha, ioctl);
		break;

	case EXT_CC_SET_DATA:
		status = qla4extioctl_set_data(ha, ioctl);
		break;

	case EXT_CC_SEND_SCSI_PASSTHRU:
		status = qla4extioctl_scsi_passthru(ha, ioctl);
		break;

	case EXT_CC_SEND_ISCSI_PASSTHRU:
		status = qla4extioctl_iscsi_passthru(ha, ioctl);
		break;

	case INT_CC_LOGOUT_ISCSI:
		status = qla4intioctl_logout_iscsi(ha, ioctl);
		break;

	case EXT_CC_GET_HBACNT:
		status = qla4extioctl_get_hbacnt(ha, ioctl);
		break;

	case INT_CC_DIAG_PING:
		status = qla4intioctl_ping(ha, ioctl);
		break;

	case INT_CC_GET_DATA:
		status = qla4intioctl_get_data(ha, ioctl);
		break;

	case INT_CC_SET_DATA:
		status = qla4intioctl_set_data(ha, ioctl);
		break;

	case INT_CC_HBA_RESET:
		status = qla4intioctl_hba_reset(ha, ioctl);
		break;

	case INT_CC_COPY_FW_FLASH:
		status = qla4intioctl_copy_fw_flash(ha, ioctl);
		break;

	case INT_CC_RESTORE_FACTORY_DEFAULTS:
		status = qla4intioctl_restore_factory_defaults(ha, ioctl);
		break;

#ifndef QLA4000
	case INT_CC_IOCB_PASSTHRU:
		status = qla4intioctl_iocb_passthru(ha, ioctl);
		break;
#endif

	default:
		QL4PRINT(QLP2,
		    printk("scsi%d: %s: unsupported command code (%X)\n",
		    ha->host_no, __func__, cmd));

		ioctl->Status = EXT_STATUS_UNSUPPORTED_SUBCODE;
	}

	/*
	 * Copy the updated ioctl structure back to the user
	 */
	if (copy_to_user(arg, (void *)ioctl, sizeof(EXT_IOCTL_ISCSI))) {
		QL4PRINT(QLP2,
		    printk("scsi%d: %s: EXT_IOCTL_ISCSI copy write error.\n",
		    ha->host_no, __func__));

		if (status == 0)
			status = (-EFAULT);
	}

	up(&ha->ioctl_sem);

	ha->i_end = jiffies;

	QL4PRINT(QLP15, printk("scsi%d: ioctl- (%s) "
	    "i_start=%lx, f_start=%lx, f_end=%lx, i_end=%lx\n",
	    ha->host_no, IOCTL_TBL_STR(cmd, ioctl->SubCode), ha->i_start,
	    ha->f_start, ha->f_end, ha->i_end));

	exit_qla4xxx_ioctl:

	if (ioctl)
		kfree(ioctl);

	LEAVE(__func__);
	return(status);
}

/**************************************************************************
 * apidev_open
 *	This routine is invoked just prior to every IOCTL call.  We only
 *	display debug information.
 *
 * Input:
 *	Unused
 *
 * Returns:
 *	0 - Always returns successful
 *
 * Context:
 *	Kernel context.
 **************************************************************************/
static  int
apidev_open(struct inode *inode, struct file *file)
{
	QL4PRINT(QLP4, printk("scsi: apidev_open MAJOR number = %d, "
	    "MINOR number = %d\n",
	    MAJOR (inode->i_rdev),
	    MINOR (inode->i_rdev)));
	return(0);
}

/**************************************************************************
 * apidev_close
 *	This routine is invoked just after every IOCTL call.  We only
 *	display debug information.
 *
 * Input:
 *	Unused
 *
 * Returns:
 *	0 - Always returns successful
 *
 * Context:
 *	Kernel context.
 **************************************************************************/
static  int
apidev_close(struct inode *inode, struct file *file)
{
	QL4PRINT(QLP4, printk("scsi: apidev_close\n"));
	return(0);
}

/**************************************************************************
 * apidev_ioctl
 *	This routine is invoked whenever an ioctl call is made.  It in turn
 *	calls the IOCTL function for this driver.
 *
 * Input:
 *	inode - unused
 *	fp    - unused
 *	cmd - internal or external ioctl command code
 *	arg - pointer to ioctl structure
 *
 * Output:
 *	None
 *
 * Returns:
 *	 QLA_SUCCESS - IOCTL completed successfully
 *	 QLA_ERROR   - IOCTL completed in error
 *    	-EFAULT      - if the arg pointer is NULL
 *    	-EINVAL      - if the command is invalid
 *
 * Context:
 *	Kernel context.
 **************************************************************************/
static  int
apidev_ioctl(struct inode *inode, struct file *fp, unsigned int cmd,
    unsigned long arg)
{
	static Scsi_Device fake_scsi_device;
	fake_scsi_device.host = apidev_host;
	return(qla4xxx_ioctl(&fake_scsi_device, (int)cmd, (void*)arg));
}

static struct file_operations
apidev_fops = {
	ioctl:    apidev_ioctl,
	open:     apidev_open,
	release:  apidev_close
};

/**************************************************************************
 * apidev_init
 *	This routine creates a proc file for IOCTL interface.
 *
 * Input:
 *	None
 *
 * Output:
 *	apidev_host - Updated with desired host number.
 *	apidev_major - Registered.
 *
 * Remarks:
 *	Create character driver "HbaApiDev" w dynamically allocated major
 *	number and create "/proc/scsi/<QLA4XXX_PROC_NAME>/HbaApiNode" as
 *	the device node associated with the major number.
 *
 * Returns:
 *	0 - Always returns successful
 *
 * Context:
 *	Kernel context.
 **************************************************************************/
static int
apidev_init(struct Scsi_Host *host)
{
	if (apidev_host)
		return(0);

	if (0 > (apidev_major = register_chrdev(0, APIDEV_NAME,
	    &apidev_fops))) {
		QL4PRINT(QLP4|QLP7,
		printk("scsi: apidev_init: rc=%d\n", apidev_major));
		return(apidev_major);
	}

	apidev_host = host;
	QL4PRINT(QLP4|QLP7,
	    printk("scsi: Created /proc/scsi/%s/%s major=%d\n",
	    QLA4XXX_PROC_NAME, APIDEV_NODE, apidev_major));

	proc_mknod(APIDEV_NODE, 0600+S_IFCHR,
	    host->hostt->proc_dir, (kdev_t)MKDEV(apidev_major,0));

#if defined (QLA_CONFIG_COMPAT)
	ql4_apidev_init_32ioctl();
#endif

	return(0);
}

/**************************************************************************
 * apidev_cleanup
 *	This routine removes the proc file for the IOCTL interface
 *
 * Input:
 *	None
 *
 * Output:
 *	apidev_host - Cleared.
 *	apidev_major - Unregistered.
 *
 * Returns:
 *	0 - Always returns successful
 *
 * Context:
 *	Kernel context.
 **************************************************************************/
static int
apidev_cleanup(void)
{
	if (!apidev_host)
		return(0);

#if defined (QLA_CONFIG_COMPAT)
	ql4_apidev_cleanup_32ioctl();
#endif

	unregister_chrdev(apidev_major,APIDEV_NAME);

	QL4PRINT(QLP4|QLP7, printk("scsi: apidev_cleanup\n"));

	remove_proc_entry(APIDEV_NODE,apidev_host->hostt->proc_dir);
	apidev_host = 0;

	return(0);
}



/*
 * Overrides for Emacs so that we almost follow Linus's tabbing style.
 * Emacs will notice this stuff at the end of the file and automatically
 * adjust the settings for this buffer only.  This must remain at the end
 * of the file.
 * ---------------------------------------------------------------------------
 * Local variables:
 * c-indent-level: 2
 * c-brace-imaginary-offset: 0
 * c-brace-offset: -2
 * c-argdecl-indent: 2
 * c-label-offset: -2
 * c-continued-statement-offset: 2
 * c-continued-brace-offset: 0
 * indent-tabs-mode: nil
 * tab-width: 8
 * End:
 */
