/*
 * QLogic iSCSI HBA Driver
 * Copyright (c)  2003-2007 QLogic Corporation
 *
 * See LICENSE.qla4xxx for copyright and licensing details.
 */
#include "ql4_dump.h"

extern uint8_t qla4xxx_capture_dump_image(scsi_qla_host_t *ha, void *dump_image, uint32_t size);
extern void qla4xxx_build_dump_image_header(scsi_qla_host_t *ha, void *dump_image_header);

static void qla4xxx_retrieve_probe_dump(scsi_qla_host_t *ha, probe_dump_t *probe_dump);
static void qla4xxx_retrieve_core_dump(scsi_qla_host_t *ha, core_dump_t *core_dump);

uint8_t
qla4xxx_capture_dump_image(scsi_qla_host_t *ha,
			   void *dump_image,
			   uint32_t size)
{
	dump_image_t *image = (dump_image_t *)dump_image;

	ENTER(__func__);

	if (dump_image == NULL) {
		QL4PRINT(QLP2, printk("scsi%d: %s: NULL dump image address\n",
				      ha->host_no, __func__));
		LEAVE(__func__);
		return QLA_ERROR;
	}

	if (size >= DUMP_IMAGE_SIZE) {
		qla4xxx_build_dump_image_header(ha, &image->dump_header);
		qla4xxx_retrieve_core_dump(ha, &image->core_dump);
		qla4xxx_retrieve_probe_dump(ha, &image->probe_dump);
		LEAVE(__func__);
		return QLA_SUCCESS;
	}

	LEAVE(__func__);
	return QLA_ERROR;

}
//
// qla4xxx_build_dump_image_header
//

void
qla4xxx_build_dump_image_header(scsi_qla_host_t *ha, void *dump_image_header)
{
	dump_image_header_t *header = (dump_image_header_t *) dump_image_header;

	ENTER(__func__);
	memset(header, 0, sizeof(*header));

	header->cookie = QLGC_COOKIE;
	sprintf((char *)&header->dump_id_string,"%4x Dump", ha->pdev->device);
	header->time_stamp = jiffies / HZ;
	header->total_image_size  = DUMP_IMAGE_SIZE; // image size excluding header
	header->core_dump_offset  = CORE_DUMP_OFFSET;        //
	header->probe_dump_offset = PROBE_DUMP_OFFSET;
	sprintf((char *)&header->driver,"%s-%d v%s",
		    DRIVER_NAME, ha->instance, QLA4XXX_DRIVER_VERSION);

	QL4PRINT(QLP25, printk("scsi%d: %s: Dump Image Header:\n",
			ha->host_no, __func__));
	qla4xxx_dump_bytes(QLP25, (uint8_t *)header, sizeof(*header));
	LEAVE(__func__);
}

/************************************************************
 *
 *                  Core Dump Routines
 *
 ************************************************************/

/*
 * Perform a Write operation via the MADI registers
 */
static uint8_t qla4_write_MADI(scsi_qla_host_t *ha, int addr, int data)
{
   int done = 0;
   int count = 0;
   unsigned long flags;

   spin_lock_irqsave(&ha->hardware_lock, flags);
   if (((RD_REG_DWORD(&ha->reg->arc_madi_cmd) & MADI_STAT_MASK) >> 27) == MADI_STAT_COMMAND_BUSY)
   {
      QL4PRINT(QLP2, printk("scsi%d: %s: MADI not ready for write operation\n",
			    ha->host_no, __func__));
      spin_unlock_irqrestore(&ha->hardware_lock, flags);
      return QLA_ERROR;
   }

   WRT_REG_DWORD(&ha->reg->arc_madi_cmd, addr);
   WRT_REG_DWORD(&ha->reg->arc_madi_data, data);

   while (!done)
   {
      switch ((RD_REG_DWORD(&ha->reg->arc_madi_cmd) & MADI_STAT_MASK) >> 27)
      {
         case MADI_STAT_DATA_VALID:
            done = 1;
            break;

         case MADI_STAT_DATA_INVALID:
	    WRT_REG_DWORD(&ha->reg->arc_madi_cmd, addr);
	    WRT_REG_DWORD(&ha->reg->arc_madi_data, data);
            break;

         case MADI_STAT_COMMAND_BUSY:
         default:
            if (count++ > 10000)
            {
               QL4PRINT(QLP2, printk("scsi%d: %s: "
			"Waiting too long on write (Address 0x%08x)\n",
			ha->host_no, __func__, addr<<2));
	       spin_unlock_irqrestore(&ha->hardware_lock, flags);
               return QLA_ERROR;
            }
            break;
      }
   }

   spin_unlock_irqrestore(&ha->hardware_lock, flags);
   return QLA_SUCCESS;
}

/*
 * Perform a Read operation via the MADI registers
 */
static uint8_t qla4_read_MADI(scsi_qla_host_t *ha, int addr, uint32_t *data)
{
   int done = 0;
   int count = 0;
   unsigned long flags;

   spin_lock_irqsave(&ha->hardware_lock, flags);
   WRT_REG_DWORD(&ha->reg->arc_madi_cmd, MADI_READ_CMD | addr);
   while (!done)
   {
      switch ((RD_REG_DWORD(&ha->reg->arc_madi_cmd) & MADI_STAT_MASK) >> 27)
      {
         case MADI_STAT_DATA_VALID:
            done = 1;
            break;

         case MADI_STAT_DATA_INVALID:
            //badRead++;
	    WRT_REG_DWORD(&ha->reg->arc_madi_cmd, MADI_READ_CMD | addr);
            break;

         case MADI_STAT_COMMAND_BUSY:
         default:
            if (count++ > 10000)
            {
               QL4PRINT(QLP2, printk("scsi%d: %s: "
			"Waiting too long on read (Address 0x%08x)\n",
			ha->host_no, __func__, addr<<2));
	       spin_unlock_irqrestore(&ha->hardware_lock, flags);
               return QLA_ERROR;
            }
            break;

      }
   }

   *data = RD_REG_DWORD(&ha->reg->arc_madi_data);
   spin_unlock_irqrestore(&ha->hardware_lock, flags);
   return QLA_SUCCESS;
}

void
qla4xxx_retrieve_core_dump(scsi_qla_host_t *ha, core_dump_t *core_dump)
{
	uint32_t             addr, data, rval;
	volatile uint32_t   *reg_ptr;
	unsigned long        flags;
	uint8_t		     page, num_pci_pages;

	QL4PRINT(QLP12, printk("Entering %s: core_dump = %p\n", __func__, core_dump));

	if (core_dump == NULL) {
		QL4PRINT(QLP2, printk("scsi%d: %s: No core dump buffer available\n",
			ha->host_no, __func__));
		goto core_dump_exit;
	}

	/* 1 - Select OAP Processor */
	if (qla4_write_MADI(ha, MADI_DEST_ARC_DEBUG, 0x00000000) != QLA_SUCCESS) {
		QL4PRINT(QLP2, printk("scsi%d: %s: 1 - Select OAP Processor Failed1\n",
			ha->host_no, __func__));
		goto core_dump_exit;
	}

	/* 2 - Halt the OAP Processor */
	if (qla4_write_MADI(ha, MADI_DEST_AUX_REG | 0x00000005, 0x00000002) != QLA_SUCCESS) {
		QL4PRINT(QLP2, printk("scsi%d: %s: 2 - Halt OAP Processor Failed\n",
			ha->host_no, __func__));
		goto core_dump_exit;
	}

	/* 3 - Disable SRAM Parity */
	if (qla4_write_MADI(ha, MADI_DEST_AUX_REG | 0x00000020, 0x00000001) != QLA_SUCCESS) {
		QL4PRINT(QLP2, printk("scsi%d: %s: 3 - Disable SRAM Parity Failed\n",
			ha->host_no, __func__));
		goto core_dump_exit;
	}

	/* 4 - Select IAP Processor */
	if (qla4_write_MADI(ha, MADI_DEST_ARC_DEBUG, 0x00000001) != QLA_SUCCESS) {
		QL4PRINT(QLP2, printk("scsi%d: %s: 4 - Select IAP Processor Failed\n",
			ha->host_no, __func__));
		goto core_dump_exit;
	}

	/* 5 - Halt IAP Processor */
	if (qla4_write_MADI(ha, MADI_DEST_AUX_REG | 0x00000005, 0x00000002) != QLA_SUCCESS) {
		QL4PRINT(QLP2, printk("scsi%d: %s: 5 - Halt IAP Processor Failed\n",
			ha->host_no, __func__));
		goto core_dump_exit;
	}

	/* 6 - Select OAP Processor */
	if (qla4_write_MADI(ha, MADI_DEST_ARC_DEBUG, 0x00000000) != QLA_SUCCESS) {
		QL4PRINT(QLP2, printk("scsi%d: %s: 6 - Select OAP Processor Failed\n",
			ha->host_no, __func__));
		goto core_dump_exit;
	}

	/* 7 - PCI Registers from Processor's Perspective */
	for (addr = (PCI_START >> 2); (addr <= (PCI_END >> 2)) ; addr++) {
		rval = qla4_read_MADI(ha, (addr | MADI_READ_CMD) , &data);
		if (rval != QLA_SUCCESS) {
			QL4PRINT(QLP2, printk("scsi%d: %s: 7 - PCI Registers "
				"from Processor's Perspective Failed\n",
				ha->host_no, __func__));
			QL4PRINT(QLP2, printk("scsi%d: %s: PCIReg 0 addr = 0x%08x Failed0\n",
			ha->host_no, __func__, (addr << 2)));
			goto core_dump_exit;
		}
		core_dump->PCIRegProc[(addr & (0xFFC >> 2))] = data;
	}
	QL4PRINT(QLP25, printk("scsi%d: %s: 7 - PCI Registers from Processor's Perspective:\n",
			ha->host_no, __func__));
	qla4xxx_dump_bytes(QLP25, (uint8_t *)core_dump->PCIRegProc, 32);


	/* 8 - SRAM Content */
	for (addr = (RAM_START >> 2); (addr <= (RAM_END >> 2)) ; addr++) {
		rval = qla4_read_MADI(ha, (addr | MADI_READ_CMD) , &data);
		if (rval != QLA_SUCCESS) {
			QL4PRINT(QLP2, printk("scsi%d: %s: 8 - SRAM Content Failed,  addr = 0x%08x\n",
				ha->host_no, __func__, (addr << 2)));
			goto core_dump_exit;
		}
		core_dump->SRAM[addr] = data;
	}
	QL4PRINT(QLP25, printk("scsi%d: %s: 8 - SRAM Content:\n", ha->host_no, __func__));
	qla4xxx_dump_bytes(QLP25, (uint8_t *)core_dump->SRAM, 32);


	/* 9 - OAP Core Registers */
	for (addr = CORE_START; (addr <= CORE_END) ; addr++) {
		rval = qla4_read_MADI(ha, (addr | MADI_READ_CMD | MADI_DEST_CORE_REG) , &data);
		if (rval != QLA_SUCCESS) {
			QL4PRINT(QLP2, printk("scsi%d: %s: 9 - OAP Core Reg Failed, addr = 0x%08x\n",
                                ha->host_no, __func__, (addr)));
			goto core_dump_exit;
		}
		core_dump->OAPCoreReg[addr] = data;
	}
	QL4PRINT(QLP25, printk("scsi%d: %s: 9 - OAP Core Register:\n", ha->host_no, __func__));
	qla4xxx_dump_bytes(QLP25, (uint8_t *)core_dump->OAPCoreReg, 32);


	/* 10 - OAP Auxiliary Registers */
	for (addr = 0; (addr <= 0x309) ; addr++) {
		rval = qla4_read_MADI(ha, (addr | MADI_READ_CMD | MADI_DEST_AUX_REG) , &data);
		if (rval != QLA_SUCCESS) {
			QL4PRINT(QLP2, printk("scsi%d: %s: 10 - OAP Aux Reg Failed, "
				"addr = 0x%08x\n", ha->host_no, __func__, (addr)));
			goto core_dump_exit;
		}
		core_dump->OAPAuxReg[addr] = data;
	}
	QL4PRINT(QLP25, printk("scsi%d: %s: 10 - OAP Auxiliary Registers:\n",
		ha->host_no, __func__));
	qla4xxx_dump_bytes(QLP25, (uint8_t *)core_dump->OAPAuxReg, 32);


	/* 11 - Select IAP Processor */
	if (qla4_write_MADI(ha, MADI_DEST_ARC_DEBUG, 0x00000001) != QLA_SUCCESS) {
		QL4PRINT(QLP2, printk("scsi%d: %s: 11 - Select IAP Processor Failed\n",
			ha->host_no, __func__));
		goto core_dump_exit;
	}

	/* 12 - IAP Core Registers */
	for (addr = 0; (addr <= 0x3F) ; addr++) {
		rval = qla4_read_MADI(ha, (addr | MADI_READ_CMD | MADI_DEST_CORE_REG), &data);
		if (rval != QLA_SUCCESS) {
			QL4PRINT(QLP2, printk("scsi%d: %s: 12 - IAP Core Reg Failed, "
				"addr = 0x%08x \n", ha->host_no, __func__, addr));
			goto core_dump_exit;
		}
		core_dump->IAPCoreReg[addr] = data;
	}
	QL4PRINT(QLP25, printk("scsi%d: %s: 12 - IAP Core Registers:\n",
		ha->host_no, __func__));
	qla4xxx_dump_bytes(QLP25, (uint8_t *)core_dump->IAPCoreReg, 32);


	/* 13 - IAP Auxiliary Registers */
	for (addr = 0; (addr <= 0x309) ; addr++) {
		rval = qla4_read_MADI(ha, (addr | MADI_READ_CMD | MADI_DEST_AUX_REG), &data);
		if (rval != QLA_SUCCESS) {
			QL4PRINT(QLP2, printk("scsi%d: %s: 13 - IAP Aux Reg Failed, "
				"addr = 0x%08x\n", ha->host_no, __func__, (addr)));
			goto core_dump_exit;
		}
		core_dump->IAPAuxReg[addr] = data;
	}
	QL4PRINT(QLP25, printk("scsi%d: %s: 13 - IAP Auxiliary Registers:\n",
		ha->host_no, __func__));
	qla4xxx_dump_bytes(QLP25, (uint8_t *)core_dump->IAPAuxReg, 32);


	/* 14 - Save IAP load/store RAM */
	for (addr = (LDST_START >> 2); (addr <= (LDST_END >> 2)) ; addr++) {
		rval = qla4_read_MADI(ha, (addr | MADI_READ_CMD), &data);
		if (rval != QLA_SUCCESS) {
			QL4PRINT(QLP2, printk("scsi%d: %s: 14 - IAP SRAM Content Failed, "
				"addr = 0x%08x\n", ha->host_no, __func__, (addr << 2)));
			goto core_dump_exit;
		}
		core_dump->IAPSRAM[(addr & (0x1FFC >> 2))] = data;
	}
	QL4PRINT(QLP25, printk("scsi%d: %s: 14 - IAP Load/Store RAM\n",
		ha->host_no, __func__));
	qla4xxx_dump_bytes(QLP25, (uint8_t *)core_dump->IAPSRAM, 64);


	/* 15 - Select OAP Processor */
	if (qla4_write_MADI(ha, MADI_DEST_ARC_DEBUG, 0x00000000) != QLA_SUCCESS) {
		QL4PRINT(QLP2, printk("scsi%d: %s: 15 - Select OAP Processor Failed3\n",
			ha->host_no, __func__));
		goto core_dump_exit;
	}

	/* 16 - Save Host PCI Registers */
	if (IS_QLA4010(ha))
                num_pci_pages = 4;
	else
                num_pci_pages = 3;

	for (page = 0; page < num_pci_pages; page++) {
		spin_lock_irqsave(&ha->hardware_lock, flags);
		WRT_REG_DWORD(&ha->reg->ISPControlStatus, CLR_RMASK(CSR_SCSI_PAGE_SELECT) | page);
		reg_ptr = &ha->reg->mailboxIn0;
		for (addr = 0; addr < 64; addr++) {
			data = RD_REG_DWORD(reg_ptr);
			core_dump->HostPCIRegPage[page][addr] = data;
			reg_ptr++;
		}
		spin_unlock_irqrestore(&ha->hardware_lock, flags);
		QL4PRINT(QLP25, printk("scsi%d: %s: 16 - Host PCI Registers Page %d:\n",
			ha->host_no, __func__, page));
		qla4xxx_dump_bytes(QLP25, (uint8_t *)core_dump->HostPCIRegPage[page], 64);
	}

	/* 17 - Save statistics registers */
	if (IS_QLA4010(ha)) {
		/* Statistics registers were saved from page 3 registers above */
	}
	else {
		spin_lock_irqsave(&ha->hardware_lock, flags);
		WRT_REG_DWORD(&ha->reg->ISPControlStatus, CLR_RMASK(CSR_SCSI_PAGE_SELECT));
#if (defined(QLA4022) && defined(__VMKERNEL_MODULE__))
		WRT_REG_DWORD(&ha->reg->pcs_regs.stats_index, 0);
#else
		WRT_REG_DWORD(&ha->reg->stats_index, 0);
#endif
		for (addr = 0; addr < 64; addr++) {
#if (defined(QLA4022) && defined(__VMKERNEL_MODULE__))
			data = RD_REG_DWORD(&ha->reg->pcs_regs.stats_read_data_inc);
#else
			data = RD_REG_DWORD(&ha->reg->stats_read_data_inc);
#endif
			core_dump->HostPCIRegPage[PROT_STAT_PAGE][addr] = data;
		}
		QL4PRINT(QLP25, printk("scsi%d: %s: 17 - Statistics Registers:\n",
			ha->host_no, __func__));
		qla4xxx_dump_bytes(QLP25, (uint8_t *)core_dump->HostPCIRegPage[page], 64);
		spin_unlock_irqrestore(&ha->hardware_lock, flags);
	}

	/* Select OAP Processor */
	if (qla4_write_MADI(ha, MADI_DEST_ARC_DEBUG, 0x00000000) != QLA_SUCCESS) {
		QL4PRINT(QLP2, printk("scsi%d: %s: 18 - Select OAP Processor Failed\n",
			ha->host_no, __func__));
		goto core_dump_exit;
	}
	/* Enable SRAM Parity */
	if (qla4_write_MADI(ha, MADI_DEST_AUX_REG | 0x00000020, 0x00000000) != QLA_SUCCESS) {
		QL4PRINT(QLP2, printk("scsi%d: %s: 19 - Enable SRAM Parity Failed\n",
			ha->host_no, __func__));
		goto core_dump_exit;
	}

core_dump_exit:
	LEAVE(__func__);
}


/************************************************************
 *
 *                  Probe Dump Routines
 *
 ************************************************************/
static uint32_t
probeRead(scsi_qla_host_t *ha, uint32_t probeAddr, probeData_t *probeData)
{
	uint32_t   oldPage;
	unsigned long	flags;
	
	spin_lock_irqsave(&ha->hardware_lock, flags);
	oldPage = RD_REG_DWORD(&ha->reg->ISPControlStatus) & 0x0003;
	WRT_REG_DWORD(&ha->reg->ISPControlStatus, 0x00030000);  // Set to page 0
	
	WRT_REG_DWORD(ISP_PROBE_MUX_ADDR(ha), (u_long)(probeAddr | PROBE_RE | PROBE_UP));
	probeData->high = (RD_REG_DWORD(ISP_PROBE_MUX_DATA(ha)) >> 24) & 0xff;
	WRT_REG_DWORD(ISP_PROBE_MUX_ADDR(ha), (u_long)(probeAddr | PROBE_RE | PROBE_LO));
	probeData->low = RD_REG_DWORD(ISP_PROBE_MUX_DATA(ha));
	
	WRT_REG_DWORD(&ha->reg->ISPControlStatus, (u_long)(0x00030000 | oldPage)); // Reset page
	spin_unlock_irqrestore(&ha->hardware_lock, flags);
	return probeData->low;
}
//
// 4010 ProbeMux table
//
PROBEMUX_INFO  probeModuleInfo4010[] = {
     {"0"        , CLK_BIT(SYSCLK)                      , MUX_SELECT_MAX}
   , {"DA"       , CLK_BIT(SYSCLK) | CLK_BIT(PCICLK)    , MUX_SELECT_MAX}
   , {"NRM"      , CLK_BIT(SYSCLK)                      , MUX_SELECT_MAX}
   , {"ODE"      , CLK_BIT(SYSCLK)                      , MUX_SELECT_MAX}
   , {"SRM"      , CLK_BIT(SYSCLK)                      , MUX_SELECT_MAX}
   , {"SCM"      , CLK_BIT(SYSCLK)                      , MUX_SELECT_MAX}
   , {"NCM"      , CLK_BIT(SYSCLK)                      , MUX_SELECT_MAX}
   , {"PRD"      , CLK_BIT(SYSCLK)                      , MUX_SELECT_MAX}
   , {"SDE"      , CLK_BIT(SYSCLK)                      , MUX_SELECT_MAX}
   , {"RBM"      , CLK_BIT(SYSCLK)                      , MUX_SELECT_MAX}
   , {"IDE"      , CLK_BIT(SYSCLK)                      , MUX_SELECT_MAX}
   , {"TDE"      , CLK_BIT(SYSCLK)                      , MUX_SELECT_MAX}
   , {"RA"       , CLK_BIT(SYSCLK)                      , MUX_SELECT_MAX}
   , {"ERM"      , CLK_BIT(SYSCLK) | CLK_BIT(CPUCLK)    , MUX_SELECT_MAX}
   , {"RMI"      , CLK_BIT(SYSCLK) | CLK_BIT(CPUCLK)    , MUX_SELECT_MAX}
   , {"OAP"      ,                   CLK_BIT(CPUCLK)    , MUX_SELECT_MAX}
   , {"ECM"      , CLK_BIT(SYSCLK) | CLK_BIT(CPUCLK)    , MUX_SELECT_MAX}
   , {"NPF"      , CLK_BIT(SYSCLK) | CLK_BIT(CPUCLK)    , MUX_SELECT_MAX}
   , {"IAP"      , CLK_BIT(SYSCLK) | CLK_BIT(CPUCLK)    , MUX_SELECT_MAX}
   , {"OTP"      , CLK_BIT(SYSCLK)                      , MUX_SELECT_MAX}
   , {"TTM"      , CLK_BIT(SYSCLK)                      , MUX_SELECT_MAX}
   , {"ITP"      , CLK_BIT(SYSCLK)                      , MUX_SELECT_MAX}
   , {"MAM"      , CLK_BIT(SYSCLK) | CLK_BIT(CPUCLK)    , MUX_SELECT_MAX}
   , {"BLM"      , CLK_BIT(SYSCLK)                      , MUX_SELECT_MAX}
   , {"ILM"      , CLK_BIT(SYSCLK)                      , MUX_SELECT_MAX}
   , {"IFP"      , CLK_BIT(SYSCLK)                      , MUX_SELECT_MAX}
   , {"IPV"      , CLK_BIT(SYSCLK)                      , MUX_SELECT_MAX}
   , {"OIP"      , CLK_BIT(SYSCLK)                      , MUX_SELECT_MAX}
   , {"OFB"      , CLK_BIT(SYSCLK) | CLK_BIT(NRXCLK)    , MUX_SELECT_MAX}
   , {"MAC"      , CLK_BIT(SYSCLK)                      , MUX_SELECT_MAX}
   , {"IFB"      , CLK_BIT(SYSCLK) | CLK_BIT(NRXCLK)    , MUX_SELECT_MAX}
   , {"PCORE"    , CLK_BIT(PCICLK)                      , MUX_SELECT_MAX}
   , {"20"       , CLK_BIT(SYSCLK)                      , MUX_SELECT_MAX}
   , {"21"       , CLK_BIT(SYSCLK)                      , MUX_SELECT_MAX}
   , {"22"       , CLK_BIT(SYSCLK)                      , MUX_SELECT_MAX}
   , {"23"       , CLK_BIT(SYSCLK)                      , MUX_SELECT_MAX}
   , {"24"       , CLK_BIT(SYSCLK)                      , MUX_SELECT_MAX}
   , {"25"       , CLK_BIT(SYSCLK)                      , MUX_SELECT_MAX}
   , {"26"       , CLK_BIT(SYSCLK)                      , MUX_SELECT_MAX}
   , {"27"       , CLK_BIT(SYSCLK)                      , MUX_SELECT_MAX}
   , {"28"       , CLK_BIT(SYSCLK)                      , MUX_SELECT_MAX}
   , {"29"       , CLK_BIT(SYSCLK)                      , MUX_SELECT_MAX}
   , {"2A"       , CLK_BIT(SYSCLK)                      , MUX_SELECT_MAX}
   , {"2B"       , CLK_BIT(SYSCLK)                      , MUX_SELECT_MAX}
   , {"2C"       , CLK_BIT(SYSCLK)                      , MUX_SELECT_MAX}
   , {"2D"       , CLK_BIT(SYSCLK)                      , MUX_SELECT_MAX}
   , {"2E"       , CLK_BIT(SYSCLK)                      , MUX_SELECT_MAX}
   , {"2F"       , CLK_BIT(SYSCLK)                      , MUX_SELECT_MAX}
   , {"30"       , CLK_BIT(SYSCLK)                      , MUX_SELECT_MAX}
   , {"31"       , CLK_BIT(SYSCLK)                      , MUX_SELECT_MAX}
   , {"32"       , CLK_BIT(SYSCLK)                      , MUX_SELECT_MAX}
   , {"33"       , CLK_BIT(SYSCLK)                      , MUX_SELECT_MAX}
   , {"34"       , CLK_BIT(SYSCLK)                      , MUX_SELECT_MAX}
   , {"35"       , CLK_BIT(SYSCLK)                      , MUX_SELECT_MAX}
   , {"36"       , CLK_BIT(SYSCLK)                      , MUX_SELECT_MAX}
   , {"37"       , CLK_BIT(SYSCLK)                      , MUX_SELECT_MAX}
   , {"38"       , CLK_BIT(SYSCLK)                      , MUX_SELECT_MAX}
   , {"39"       , CLK_BIT(SYSCLK)                      , MUX_SELECT_MAX}
   , {"3a"       , CLK_BIT(SYSCLK)                      , MUX_SELECT_MAX}
   , {"3b"       , CLK_BIT(SYSCLK)                      , MUX_SELECT_MAX}
   , {"3c"       , CLK_BIT(SYSCLK)                      , MUX_SELECT_MAX}
   , {"3d"       , CLK_BIT(SYSCLK)                      , MUX_SELECT_MAX}
   , {"3e"       , CLK_BIT(SYSCLK)                      , MUX_SELECT_MAX}
   , {"3f"       , CLK_BIT(SYSCLK)                      , MUX_SELECT_MAX}
};

//
// 4022/4032 ProbeMux table
//

PROBEMUX_INFO  probeModuleInfo4022[] = {
     {"0"        , CLK_BIT(SYSCLK)                      , MUX_SELECT_MAX}
   , {"DA"       , CLK_BIT(SYSCLK) | CLK_BIT(PCICLK)    , MUX_SELECT_MAX}
   , {"BPM"      , CLK_BIT(SYSCLK)                      , MUX_SELECT_MAX}
   , {"ODE"      , CLK_BIT(SYSCLK)                      , MUX_SELECT_MAX}
   , {"SRM0"     , CLK_BIT(SYSCLK)                      , MUX_SELECT_MAX}
   , {"SRM1"     , CLK_BIT(SYSCLK)                      , MUX_SELECT_MAX}
   , {"PMD"      , CLK_BIT(SYSCLK)                      , MUX_SELECT_MAX}
   , {"PRD"      , CLK_BIT(SYSCLK)                      , MUX_SELECT_MAX}
   , {"SDE"      , CLK_BIT(SYSCLK)                      , MUX_SELECT_MAX}
   , {"RMD"      , CLK_BIT(SYSCLK)                      , MUX_SELECT_MAX}
   , {"IDE"      , CLK_BIT(SYSCLK)                      , MUX_SELECT_MAX}
   , {"TDE"      , CLK_BIT(SYSCLK)                      , MUX_SELECT_MAX}
   , {"RA"       , CLK_BIT(SYSCLK)                      , MUX_SELECT_MAX}
   , {"REG"      , CLK_BIT(SYSCLK)                      , MUX_SELECT_MAX}
   , {"RMI"      , CLK_BIT(SYSCLK) | CLK_BIT(CPUCLK)    , MUX_SELECT_MAX}
   , {"OAP"      ,                   CLK_BIT(CPUCLK)    , MUX_SELECT_MAX}
   , {"ECM"      , CLK_BIT(SYSCLK) | CLK_BIT(CPUCLK)    , MUX_SELECT_MAX}
   , {"NPF"      , CLK_BIT(SYSCLK) | CLK_BIT(CPUCLK)    , MUX_SELECT_MAX}
   , {"IAP"      , CLK_BIT(SYSCLK) | CLK_BIT(CPUCLK)    , MUX_SELECT_MAX}
   , {"OTP"      , CLK_BIT(SYSCLK)                      , MUX_SELECT_MAX}
   , {"TTM"      , CLK_BIT(SYSCLK)                      , MUX_SELECT_MAX}
   , {"ITP"      , CLK_BIT(SYSCLK)                      , MUX_SELECT_MAX}
   , {"MAM"      , CLK_BIT(SYSCLK) | CLK_BIT(CPUCLK)    , MUX_SELECT_MAX}
   , {"BLM"      , CLK_BIT(SYSCLK)                      , MUX_SELECT_MAX}
   , {"ILM"      , CLK_BIT(SYSCLK)                      , MUX_SELECT_MAX}
   , {"IFP"      , CLK_BIT(SYSCLK)                      , MUX_SELECT_MAX}
   , {"IPV"      , CLK_BIT(SYSCLK)                      , MUX_SELECT_MAX}
   , {"OIP"      , CLK_BIT(SYSCLK)                      , MUX_SELECT_MAX}
   , {"OFB"      , CLK_BIT(SYSCLK) | CLK_BIT(NRXCLK)    , MUX_SELECT_MAX}
   , {"MAC"      , CLK_BIT(SYSCLK)                      , MUX_SELECT_MAX}
   , {"IFB"      , CLK_BIT(SYSCLK) | CLK_BIT(NRXCLK)    , MUX_SELECT_MAX}
   , {"PCORE"    , CLK_BIT(PCICLK)                      , MUX_SELECT_MAX}
   , {"NRM0"     , CLK_BIT(SYSCLK)                      , MUX_SELECT_MAX}
   , {"NRM1"     , CLK_BIT(SYSCLK)                      , MUX_SELECT_MAX}
   , {"SCM0"     , CLK_BIT(SYSCLK)                      , MUX_SELECT_MAX}
   , {"SCM1"     , CLK_BIT(SYSCLK)                      , MUX_SELECT_MAX}
   , {"NCM0"     , CLK_BIT(SYSCLK)                      , MUX_SELECT_MAX}
   , {"NCM1"     , CLK_BIT(SYSCLK)                      , MUX_SELECT_MAX}
   , {"RBM0"     , CLK_BIT(SYSCLK)                      , MUX_SELECT_MAX}
   , {"RBM1"     , CLK_BIT(SYSCLK)                      , MUX_SELECT_MAX}
   , {"RBM2"     , CLK_BIT(SYSCLK)                      , MUX_SELECT_MAX}
   , {"RBM3"     , CLK_BIT(SYSCLK)                      , MUX_SELECT_MAX}
   , {"ERM0"     , CLK_BIT(SYSCLK) | CLK_BIT(CPUCLK)    , MUX_SELECT_MAX}
   , {"ERM1"     , CLK_BIT(SYSCLK) | CLK_BIT(CPUCLK)    , MUX_SELECT_MAX}
   , {"PERF0"    , CLK_BIT(SYSCLK)                      , MUX_SELECT_MAX}
   , {"PERF1"    , CLK_BIT(SYSCLK)                      , MUX_SELECT_MAX}
   , {"2E"       , CLK_BIT(SYSCLK)                      , MUX_SELECT_MAX}
   , {"2F"       , CLK_BIT(SYSCLK)                      , MUX_SELECT_MAX}
   , {"30"       , CLK_BIT(SYSCLK)                      , MUX_SELECT_MAX}
   , {"31"       , CLK_BIT(SYSCLK)                      , MUX_SELECT_MAX}
   , {"32"       , CLK_BIT(SYSCLK)                      , MUX_SELECT_MAX}
   , {"33"       , CLK_BIT(SYSCLK)                      , MUX_SELECT_MAX}
   , {"34"       , CLK_BIT(SYSCLK)                      , MUX_SELECT_MAX}
   , {"35"       , CLK_BIT(SYSCLK)                      , MUX_SELECT_MAX}
   , {"36"       , CLK_BIT(SYSCLK)                      , MUX_SELECT_MAX}
   , {"37"       , CLK_BIT(SYSCLK)                      , MUX_SELECT_MAX}
   , {"38"       , CLK_BIT(SYSCLK)                      , MUX_SELECT_MAX}
   , {"39"       , CLK_BIT(SYSCLK)                      , MUX_SELECT_MAX}
   , {"3a"       , CLK_BIT(SYSCLK)                      , MUX_SELECT_MAX}
   , {"3b"       , CLK_BIT(SYSCLK)                      , MUX_SELECT_MAX}
   , {"3c"       , CLK_BIT(SYSCLK)                      , MUX_SELECT_MAX}
   , {"3d"       , CLK_BIT(SYSCLK)                      , MUX_SELECT_MAX}
   , {"3e"       , CLK_BIT(SYSCLK)                      , MUX_SELECT_MAX}
   , {"3f"       , CLK_BIT(SYSCLK)                      , MUX_SELECT_MAX}
};

char  *probeClockNames[] = {
     "SYS"
   , "PCI"
   , "NRX"
   , "CPU"
};

static void
qla4xxx_retrieve_probe_dump(scsi_qla_host_t *ha, probe_dump_t *probe_dump)
{
	uint32_t   firstModule = probe_DA;
	uint32_t   lastModule  = probe_PERF1;
	uint32_t   probeModule;
	
	uint32_t   firstClock = 0;
	uint32_t   lastClock  = 4;
	uint32_t   probeClock;

	//uint32_t   forceClock = 0;
	//uint32_t   forceLast = 0;
	
	uint32_t   lastMux;
	uint32_t   muxSelect;
	
	uint32_t   probeAddr;
	probeData_t probeData[MUX_SELECT_MAX];

	PROBEMUX_INFO  *probeModuleInfo;
	
	
	QL4PRINT(QLP12, printk("qla4xxx: Entering %s(): probe_dump = %p\n",
			       __func__, probe_dump));

	probeModuleInfo = (IS_QLA4010(ha)) ? probeModuleInfo4010 : probeModuleInfo4022;
	
	for (probeModule = firstModule; probeModule <= lastModule; probeModule++) {
		for (probeClock = firstClock; probeClock < lastClock; probeClock++) {
			if (/* forceClock || */(probeModuleInfo[probeModule].clocks & (1 << probeClock))) {
				//if (!forceLast) {
					lastMux = probeModuleInfo[probeModule].maxSelect;
				//}
				probeAddr = (probeModule << 8) | (probeClock << 6);
				QL4PRINT(QLP25, printk("%5s %3s:  ModuleSelect:0x%02x   Clock:0x%x  StartingProbeAddr:0x%04x\n",
					probeModuleInfo[probeModule].moduleName,
					probeClockNames[probeClock],
					probeModule, probeClock,
					probeAddr));
				for (muxSelect = 0; muxSelect < lastMux; muxSelect++) {
					probeRead(ha, (uint32_t)(probeAddr | muxSelect), &probeData[muxSelect]);
				}
				
				if (ql_dbg_level & QLP25){
					for (muxSelect = 0; muxSelect < lastMux; muxSelect+=4) {
					QL4PRINT(QLP25, printk("%5s %3s 0x%04x:  %02x_%08x  %02x_%08x  %02x_%08x  %02x_%08x\n",
						probeModuleInfo[probeModule].moduleName,
						probeClockNames[probeClock],
						probeAddr | muxSelect,
						probeData[muxSelect].high,   probeData[muxSelect].low,
						probeData[muxSelect+1].high, probeData[muxSelect+1].low,
						probeData[muxSelect+2].high, probeData[muxSelect+2].low,
						probeData[muxSelect+3].high, probeData[muxSelect+3].low));
					}
					QL4PRINT(QLP25, printk("\n"));
				}
			}
		}
	}
	LEAVE(__func__);
}

