/*
 * QLogic iSCSI HBA Driver
 * Copyright (c)  2003-2006 QLogic Corporation
 *
 * See LICENSE.qla4xxx for copyright and licensing details.
 */

/******************************************************************************
 *             Please see release.txt for revision history.                   *
 *                                                                            *
 ******************************************************************************
 * Function Table of Contents:
 *	  qla4xxx_alloc_ioctl_mem
 *	  qla4xxx_free_ioctl_mem
 *	  qla4xxx_get_ioctl_scrap_mem
 *	  qla4xxx_free_ioctl_scrap_mem
 *	  qla4extioctl_query_hba_iscsi_node
 *	  qla4extioctl_query_hba_iscsi_portal
 *	  qla4extioctl_query_disc_iscsi_node
 *	  qla4extioctl_query_disc_iscsi_portal
 *	  qla4extioctl_query_driver
 *	  qla4extioctl_query_fw
 *	  qla4extioctl_query_chip
 *	  qla4extioctl_query_ip_state
 *	  qla4extioctl_query_device_current_ip
 *	  qla4extioctl_query
 *	  qla4extioctl_reg_aen
 *	  qla4extioctl_get_aen
 *	  qla4extioctl_get_statistics_gen
 *	  qla4extioctl_get_statistics_iscsi
 *	  qla4extioctl_get_device_entry_iscsi
 *	  qla4extioctl_get_init_fw_iscsi
 *	  qla4extioctl_get_isns_server
 *	  qla4extioctl_get_isns_disc_targets
 *	  qla4extioctl_get_acb
 *	  qla4extioctl_get_neighbor_cache
 *	  qla4extioctl_get_destination_cache
 *	  qla4extioctl_get_default_router_list
 *	  qla4extioctl_get_local_prefix_list
 *	  qla4extioctl_get_data
 *	  qla4extioctl_rst_statistics_gen
 *	  qla4extioctl_rst_statistics_iscsi
 *	  qla4extioctl_set_device_entry_iscsi
 *	  qla4extioctl_set_init_fw_iscsi
 *	  qla4extioctl_set_isns_server
 *	  qla4extioctl_set_acb
 *	  qla4extioctl_set_data
 *	  qla4xxx_ioctl_sleep_done
 *	  qla4xxx_ioctl_sem_init
 *	  qla4xxx_scsi_pass_done
 *	  qla4extioctl_scsi_passthru
 *	  qla4extioctl_iscsi_passthru
 *	  qla4extioctl_disable_acb
 *	  qla4extioctl_send_router_sol
 *	  qla4extioctl_get_hbacnt
 *	  qla4intioctl_logout_iscsi
 *	  qla4intioctl_ping
 *	  qla4intioctl_get_flash
 *	  qla4intioctl_get_core_dump
 *	  qla4intioctl_get_driver_debug_level
 *	  qla4intioctl_get_host_no
 *	  qla4intioctl_get_data
 *	  qla4intioctl_set_flash
 *	  qla4intioctl_set_driver_debug_level
 *	  qla4intioctl_set_data
 *	  qla4intioctl_hba_reset
 *	  qla4intioctl_copy_fw_flash
 *	  qla4xxx_iocb_pass_done
 *	  qla4intioctl_restore_factory_defaults
 *	  qla4xxx_ioctl
 *        apidev_open
 *        apidev_close
 *        apidev_ioctl
 *        apidev_init
 *        apidev_cleanup
 *	
 ****************************************************************************/

#include <linux/config.h>
#include <linux/module.h>
#include <stdarg.h>
#include <asm/io.h>
#include <asm/irq.h>
#include <asm/segment.h>
#include <asm/byteorder.h>
#include <linux/spinlock.h>
#include <linux/version.h>
#include <linux/types.h>
#include <linux/string.h>
#include <linux/errno.h>
#include <linux/kernel.h>
#include <linux/delay.h>
#include <linux/timer.h>
#include <linux/sched.h>
#include <linux/pci.h>
#include <linux/proc_fs.h>
#include <linux/blk.h>
#include <linux/tqueue.h>
#include <asm/uaccess.h>

#ifdef __VMKERNEL_MODULE__
#include <scsi.h>
#include <hosts.h>
#else
#include <../drivers/scsi/scsi.h>
#include <../drivers/scsi/hosts.h>
#endif

#include "qlisioct.h"
#include "qlinioct.h"
#include "qla4x_fw.h"
#include "qla4xxx.h"
#include "ql4print.h"

// KRH: (BEGIN) Define these locally, for now
/*
 * Sub codes for Get Data.
 * Use in combination with INT_GET_DATA as the ioctl code
 */
#define INT_SC_GET_DRIVER_DEBUG_LEVEL   5
#define INT_SC_GET_HOST_NO 		6

/*
 * Sub codes for Set Data.
 * Use in combination with INT_SET_DATA as the ioctl code
 */
#define INT_SC_SET_DRIVER_DEBUG_LEVEL	5

/*
 * Sub codes for Reset
 * Use in combination with INT_CC_HBA_RESET as the ioctl code
 */
#define INT_SC_HBA_RESET			0
#define INT_SC_FIRMWARE_RESET			1
#define INT_SC_TARGET_WARM_RESET		2
#define INT_SC_LUN_RESET			3
//KRH: (END)

/* Defines for Passthru */
#define IOCTL_INVALID_STATUS			0xffff

/* Defines for byte-order translation */
#define GET_DATA	0
#define SET_DATA	1

/**************************
 * Global Data
 **************************/

/* scrap memory for local use. */
#define QLA_IOCTL_SCRAP_SIZE		17000
#define QLA_IOCTL_DUMP_IMAGE_SIZE	DUMP_IMAGE_SIZE + (0x4000 - (DUMP_IMAGE_SIZE % 0x4000))
					/* Round up to 16K */

/*
 * Create character driver "HbaApiDev" w dynamically allocated major number
 * and create "/proc/scsi/<QLA4XXX_PROC_NAME>/HbaApiNode" as the device
 * node associated with the major number.
 */
#define APIDEV_NODE  "HbaApiNode"
#ifdef __VMKERNEL_MODULE__
#define APIDEV_NAME  "HbaApiDev-"QLA4XXX_PROC_NAME
#else
#define APIDEV_NAME  "HbaApiDev"
#endif

STATIC int apidev_major = 0;
STATIC struct Scsi_Host *apidev_host = 0;


/**************************
 * Helper Functions
 **************************/
void *
Q64BIT_TO_PTR(uint64_t buf_addr)
{
#if defined(QLA_CONFIG_COMPAT) || !defined(CONFIG_64BIT)
	union ql_doublelong {
		struct {
			uint32_t        lsl;
			uint32_t        msl;
		} longs;
		uint64_t        dl;
	};

	union ql_doublelong tmpval;

	tmpval.dl = buf_addr;

#if defined(QLA_CONFIG_COMPAT)
	/* return lower 32bit in 64bit pointer */
	return((void *)(uint64_t)(tmpval.longs.lsl));
#else
	/* return lower 32bit in 32bit pointer */
	return((void *)(tmpval.longs.lsl));
#endif
#else
	/* return the full 64bit as is */
	return((void *)buf_addr);
#endif
}

inline void *
ql4_kmem_zalloc(int siz)
{
	void	*bp;

	if ((bp = kmalloc(siz, GFP_ATOMIC)) != NULL) {
		memset(bp, 0, siz);
	}

	return (bp);
}

/*
 * qla4xxx_alloc_ioctl_mem
 *	Allocates memory needed by IOCTL code.
 *
 * Input:
 *	ha = adapter state pointer.
 *
 * Returns:
 *	qla4xxx local function return status code.
 *
 * Context:
 *	Kernel context.
 */
static int
qla4xxx_alloc_ioctl_mem(scsi_qla_host_t *ha)
{
	ENTER(__func__);

	/* Pick the largest size we'll need per ha of all ioctl cmds.
	 * Use this size when freeing.
	 */
	ha->ioctl_scrap_mem = ql4_kmem_zalloc(QLA_IOCTL_SCRAP_SIZE);
	if (ha->ioctl_scrap_mem == NULL) {
		printk(KERN_WARNING
		    "scsi%d: ERROR in ioctl scrap_mem allocation.\n",
		    ha->host_no);
		return QLA_ERROR;
	}
	ha->ioctl_scrap_mem_size = QLA_IOCTL_SCRAP_SIZE;
	ha->ioctl_scrap_mem_used = 0;

	QL4PRINT(QLP4|QLP7,
	    printk("scsi%d: %s: scrap_mem_size=%d.\n",
	    ha->host_no, __func__, ha->ioctl_scrap_mem_size));

	/*
	 * Allocate core dump buffer
	 */
	ha->core_dump_mem = vmalloc(QLA_IOCTL_DUMP_IMAGE_SIZE);
	if (ha->core_dump_mem == NULL) {
		printk(KERN_WARNING
			"scsi%d: ERROR in ioctl core dump memory allocation.\n",
			ha->host_no);
		return QLA_ERROR;
	}

	QL4PRINT(QLP4|QLP7,
	    printk("scsi%d: %s: core_dump_mem_size=%d, addr=%p.\n",
	    ha->host_no, __func__, QLA_IOCTL_DUMP_IMAGE_SIZE, ha->core_dump_mem));


	LEAVE(__func__);
	return QLA_SUCCESS;
}

/*
 * qla4xxx_free_ioctl_mem
 *	Frees memory used by IOCTL code for the specified ha.
 *
 * Input:
 *	ha = adapter state pointer.
 *
 * Context:
 *	Kernel context.
 */
static void
qla4xxx_free_ioctl_mem(scsi_qla_host_t *ha)
{
	ENTER(__func__);

	if (ha->ioctl_scrap_mem != NULL) {
		/* The size here must match up to what we
		 * allocated before.
		 */
		QL4PRINT(QLP4|QLP7,
		    printk("scsi%d: %s: scrap_mem_size=%d.\n",
		    ha->host_no, __func__, ha->ioctl_scrap_mem_size));

		kfree(ha->ioctl_scrap_mem);
		ha->ioctl_scrap_mem = NULL;
		ha->ioctl_scrap_mem_size = 0;
	}

	if (ha->core_dump_mem) {
		QL4PRINT(QLP4|QLP7,
		    printk("scsi%d: %s: freeing IOCTL Core Dump Buffer %p\n",
		    ha->host_no, __func__, ha->core_dump_mem));

		vfree(ha->core_dump_mem);
		ha->core_dump_mem = NULL;	
	}

	LEAVE(__func__);
}

/*
 * qla4xxx_get_ioctl_scrap_mem
 *	Returns pointer to memory of the specified size from the scrap buffer.
 *	This can be called multiple times before the free call as long
 *	as the memory is to be used by the same ioctl command and
 *	there's still memory left in the scrap buffer.
 *
 * Input:
 *	ha = adapter state pointer.
 *	ppmem = pointer to return a buffer pointer.
 *	size = size of buffer to return.
 *
 * Returns:
 *	qla4xxx local function return status code.
 *
 * Context:
 *	Kernel context.
 */
int
qla4xxx_get_ioctl_scrap_mem(scsi_qla_host_t *ha, void **ppmem, uint32_t size)
{
	int		ret = QLA_SUCCESS;
	uint32_t	free_mem;

	ENTER(__func__);

	free_mem = ha->ioctl_scrap_mem_size - ha->ioctl_scrap_mem_used;

	/* Round memory allocation up to a long aligned size, if needed */
	if (size % sizeof(long)) {
		size = ((size / sizeof(long)) + 1) * sizeof(long);
	}

	if (free_mem >= size) {
		*ppmem = ha->ioctl_scrap_mem + ha->ioctl_scrap_mem_used;
		ha->ioctl_scrap_mem_used += size;
	} else {
		QL4PRINT(QLP2,
		    printk("scsi%d: %s: no more scrap memory.\n",
		    ha->host_no, __func__));

		ret = QLA_ERROR;
	}

	LEAVE(__func__);

	return (ret);
}

/*
 * qla4xxx_free_ioctl_scrap_mem
 *	Makes the entire scrap buffer free for use.
 *
 * Input:
 *	ha = adapter state pointer.
 *
 * Returns:
 *	qla4xxx local function return status code.
 *
 */
void
qla4xxx_free_ioctl_scrap_mem(scsi_qla_host_t *ha)
{
	ENTER(__func__);

	memset(ha->ioctl_scrap_mem, 0, ha->ioctl_scrap_mem_size);
	ha->ioctl_scrap_mem_used = 0;

	LEAVE(__func__);
}


/**************************
* Functions
**************************/

/**************************************************************************
 * qla4extioctl_query_hba_iscsi_node
 *	This routine retrieves the HBA node properties
 *
 * Input:
 *	ha    = adapter structure pointer.
 *	ioctl = IOCTL structure pointer.
 *
 * Output:
 *	None
 *
 * Returns:
 *	QLA_SUCCESS = success
 *	QLA_ERROR   = error
 *
 * Context:
 *	Kernel context.
 **************************************************************************/
static int
qla4extioctl_query_hba_iscsi_node(scsi_qla_host_t *ha, EXT_IOCTL_ISCSI *ioctl)
{
	int			status = 0;
	uint32_t		mbox_cmd[MBOX_REG_COUNT];
	uint32_t		mbox_sts[MBOX_REG_COUNT];
	EXT_HBA_ISCSI_NODE	*phba_node = NULL;
	ADDRESS_CTRL_BLK	*init_fw_cb;


	ENTER(__func__);

#if 0 && defined(__VMKERNEL_MODULE__)
	spin_lock(&ha->dma_buf_spinlock);
#else
	down(&ha->dma_buf_sem);
#endif

	if (!ha->dma_buf.virt_addr || !ha->dma_buf.phys_addr ||
	    !ioctl->ResponseAdr) {
		QL4PRINT(QLP2,
			 printk("scsi%d: %s: memory allocation problem\n",
				ha->host_no, __func__));

		status = (-ENOMEM);
		ioctl->Status = EXT_STATUS_NO_MEMORY;
		ioctl->ResponseLen = 0;
		goto exit_query_hba_node;
	}

	if (ha->dma_buf.buf_len < sizeof(*init_fw_cb)) {
		if (qla4xxx_resize_dma_buf(ha, &ha->dma_buf, sizeof(*init_fw_cb)) !=
		    QLA_SUCCESS) {
			QL4PRINT(QLP2,
			    printk("scsi%d: %s: unable to allocate memory "
			    "for dma buffer.\n",
			    ha->host_no, __func__));

			status = (-ENOMEM);
			ioctl->Status = EXT_STATUS_BUFFER_TOO_SMALL;
			ioctl->ResponseLen = 0;
			goto exit_query_hba_node;
		}
	}

	if (qla4xxx_get_ioctl_scrap_mem(ha, (void **)&phba_node,
	    sizeof(EXT_HBA_ISCSI_NODE))) {
		/* not enough memory */
		QL4PRINT(QLP2,
		    printk("scsi%d: %s: inst=%d scrap not big enough. "
		    "size requested=%ld.\n",
		    ha->host_no, __func__, ha->instance,
                    (ulong)sizeof(EXT_HBA_ISCSI_NODE)));

		status = (-ENOMEM);
		ioctl->Status = EXT_STATUS_NO_MEMORY;
		ioctl->ResponseLen = 0;
		goto exit_query_hba_node;
	}

	/*
	 * Send mailbox command
	 */
	if (qla4xxx_get_ifcb(ha, &mbox_cmd[0], &mbox_sts[0], ha->dma_buf.phys_addr)
	    == QLA_ERROR) {
		QL4PRINT(QLP2, printk("scsi%d: %s: command failed \n",
				      ha->host_no, __func__));

		ioctl->Status = EXT_STATUS_MAILBOX;
		ioctl->VendorSpecificStatus[0] = mbox_sts[0];
		ioctl->ResponseLen = 0;

		goto exit_query_hba_node;
	}

	/*
	 * Transfer data from Fw's DEV_DB_ENTRY buffer to IOCTL's
	 * EXT_HBA_ISCSI_NODE buffer
	 */
	init_fw_cb = (ADDRESS_CTRL_BLK *) ha->dma_buf.virt_addr;

	memset(phba_node, 0, sizeof(EXT_HBA_ISCSI_NODE));
	phba_node->PortNumber = le16_to_cpu(init_fw_cb->PortNumber);
	phba_node->NodeInfo.PortalCount = 1;

	memcpy(phba_node->NodeInfo.IPAddr.IPAddress, init_fw_cb->IPAddr,
	    min(sizeof(phba_node->NodeInfo.IPAddr.IPAddress),
		sizeof(init_fw_cb->IPAddr)));
	memcpy(phba_node->NodeInfo.iSCSIName, init_fw_cb->iSCSINameString,
	    min(sizeof(phba_node->NodeInfo.iSCSIName),
		sizeof(init_fw_cb->iSCSINameString)));
	memcpy(phba_node->NodeInfo.Alias, init_fw_cb->iSCSIAlias,
	    min(sizeof(phba_node->NodeInfo.Alias),
		sizeof(init_fw_cb->iSCSIAlias)));

	sprintf(phba_node->DeviceName, "/proc/scsi/%s/HbaApiNode",
	    QLA4XXX_PROC_NAME);

	/*
	 * Copy the IOCTL EXT_HBA_ISCSI_NODE buffer to the user's data space
	 */
	if (copy_to_user(Q64BIT_TO_PTR(ioctl->ResponseAdr),
	    phba_node, ioctl->ResponseLen) != 0) {
		QL4PRINT(QLP2, printk("scsi%d: %s: copy failed\n",
				      ha->host_no, __func__));

		status = (-EFAULT);
		ioctl->Status = EXT_STATUS_COPY_ERR;
		ioctl->ResponseLen = 0;
		goto exit_query_hba_node;
	}

	ioctl->Status = EXT_STATUS_OK;

exit_query_hba_node:
	qla4xxx_free_ioctl_scrap_mem(ha);

#if 0 && defined(__VMKERNEL_MODULE__)
	spin_unlock(&ha->dma_buf_spinlock);

#else
	up(&ha->dma_buf_sem);
#endif
	LEAVE(__func__);
	return(status);
}

/**************************************************************************
 * qla4extioctl_query_hba_iscsi_portal
 *	This routine retrieves the HBA iSCSI portal properties
 *
 * Input:
 *	ha    = adapter structure pointer.
 *	ioctl = IOCTL structure pointer.
 *
 * Output:
 *	None
 *
 * Returns:
 *	QLA_SUCCESS = success
 *	QLA_ERROR   = error
 *
 * Context:
 *	Kernel context.
 **************************************************************************/
static int
qla4extioctl_query_hba_iscsi_portal(scsi_qla_host_t *ha, EXT_IOCTL_ISCSI *ioctl)
{
	int			status = 0;
	uint32_t		mbox_cmd[MBOX_REG_COUNT];
	uint32_t		mbox_sts[MBOX_REG_COUNT];
	EXT_HBA_ISCSI_PORTAL	*phba_portal;
	FLASH_SYS_INFO 		*sys_info;
	uint32_t 		num_valid_ddb_entries;


	ENTER(__func__);

	if (!ioctl->ResponseAdr) {
		QL4PRINT(QLP2,
		    printk("scsi%d: %s: no response buffer found.\n",
		    ha->host_no, __func__));

		status = (-ENOMEM);
		ioctl->Status = EXT_STATUS_NO_MEMORY;
		ioctl->ResponseLen = 0;
		goto exit_query_hba_portal;
	}

	if (qla4xxx_get_ioctl_scrap_mem(ha, (void **)&phba_portal,
	    sizeof(EXT_HBA_ISCSI_PORTAL))) {
		/* not enough memory */
		QL4PRINT(QLP2,
		    printk("scsi%d: %s: inst=%d scrap not big enough. "
		    "size requested=%ld.\n",
		    ha->host_no, __func__, ha->instance,
		    (ulong)sizeof(EXT_HBA_ISCSI_PORTAL)));

		status = (-ENOMEM);
		ioctl->Status = EXT_STATUS_NO_MEMORY;
		ioctl->ResponseLen = 0;
		goto exit_query_hba_portal;
	}

	if (ioctl->ResponseLen < sizeof(*phba_portal)) {
		QL4PRINT(QLP2, printk("scsi%d: %s: memory area too small\n",
				      ha->host_no, __func__));

		status = (-ENOMEM);
		ioctl->Status = EXT_STATUS_BUFFER_TOO_SMALL;
		ioctl->ResponseLen = 0;
		goto exit_query_hba_portal;
	}

	/*
	 * Fill in EXT_HBA_ISCSI_PORTAL buffer
	 */
	memset(phba_portal, 0, sizeof(EXT_HBA_ISCSI_PORTAL));

	strcpy(phba_portal->DriverVersion, QLA4XXX_DRIVER_VERSION);
	sprintf(phba_portal->FWVersion, "%02d.%02d Patch %02d Build %02d",
	    ha->firmware_version[0], ha->firmware_version[1],
	    ha->patch_number, ha->build_number);

	/* ----- Get firmware state information ---- */
	memset(mbox_cmd, 0, sizeof(mbox_cmd));
	memset(mbox_sts, 0, sizeof(mbox_sts));
	mbox_cmd[0] = MBOX_CMD_GET_FW_STATE;
	if (qla4xxx_mailbox_command(ha, 1, 4, &mbox_cmd[0], &mbox_sts[0])
	    != QLA_SUCCESS) {
		QL4PRINT(QLP2, printk("scsi%d: %s: MBOX_CMD_GET_FW_STATE "
		    "failed w/ status %04X\n",
		    ha->host_no, __func__, mbox_sts[0]));

		ioctl->Status = EXT_STATUS_MAILBOX;
		ioctl->VendorSpecificStatus[0] = mbox_sts[0];
		ioctl->VendorSpecificStatus[1] = mbox_sts[1];
		ioctl->VendorSpecificStatus[2] = mbox_sts[2];
		ioctl->VendorSpecificStatus[3] = mbox_sts[3];
		ioctl->ResponseLen = 0;
		goto exit_query_hba_portal;
	}

	switch (mbox_sts[1]) {
	case FW_STATE_READY:
		phba_portal->State = EXT_DEF_CARD_STATE_READY;
		break;
	case FW_STATE_CONFIG_WAIT:
		phba_portal->State = EXT_DEF_CARD_STATE_CONFIG_WAIT;
		break;
	case FW_STATE_WAIT_AUTOCONNECT:
		phba_portal->State = EXT_DEF_CARD_STATE_LOGIN;
		break;
	case FW_STATE_ERROR:
		phba_portal->State = EXT_DEF_CARD_STATE_ERROR;
		break;
        case FW_STATE_CONFIGURING_IP:
                phba_portal->State = EXT_DEF_CARD_STATE_CONFIGURING_IP;
                break;
	}

	switch (mbox_sts[3] & 0x0001) {
	case FW_ADDSTATE_COPPER_MEDIA:
		phba_portal->Type = EXT_DEF_TYPE_COPPER;
		break;
	case FW_ADDSTATE_OPTICAL_MEDIA:
		phba_portal->Type = EXT_DEF_TYPE_OPTICAL;
		break;
	}

	/* ----- Get ddb entry information ---- */
	if (qla4xxx_get_ddb_entry(ha, 0, NULL, 0, &num_valid_ddb_entries,
				  NULL, NULL, NULL, NULL, NULL) == QLA_ERROR) {
		QL4PRINT(QLP2,
		    printk("scsi%d: %s: qla4xxx_get_ddb_entry failed!\n",
				      ha->host_no, __func__));

		status = (-EFAULT);
		ioctl->Status = EXT_STATUS_ERR;
		ioctl->RequestLen = 0;

		goto exit_query_hba_portal;
	}

	phba_portal->DiscTargetCount = (uint16_t) num_valid_ddb_entries;

	/* ----- Get flash sys info information ---- */
#if 0 && defined(__VMKERNEL_MODULE__)
	spin_lock(&ha->dma_buf_spinlock);
#else
	down(&ha->dma_buf_sem);
#endif
	sys_info = (FLASH_SYS_INFO *) ha->dma_buf.virt_addr;

	memset(mbox_cmd, 0, sizeof(mbox_cmd));
	memset(mbox_sts, 0, sizeof(mbox_sts));
	mbox_cmd[0] = MBOX_CMD_READ_FLASH;
	mbox_cmd[1] = LSDW(ha->dma_buf.phys_addr);
	mbox_cmd[2] = MSDW(ha->dma_buf.phys_addr);
	mbox_cmd[3] = INT_ISCSI_SYSINFO_FLASH_OFFSET;
	mbox_cmd[4] = sizeof(*sys_info);

	if (qla4xxx_mailbox_command(ha, 5, 2, &mbox_cmd[0], &mbox_sts[0])
	    != QLA_SUCCESS) {
		QL4PRINT(QLP2,
			 printk("scsi%d: %s: MBOX_CMD_READ_FLASH failed w/ "
				"status %04X\n",
				ha->host_no, __func__, mbox_sts[0]));

#if 0 && defined(__VMKERNEL_MODULE__)
		spin_unlock(&ha->dma_buf_spinlock);

#else
		up(&ha->dma_buf_sem);
#endif

		ioctl->Status = EXT_STATUS_MAILBOX;
		ioctl->VendorSpecificStatus[0] = mbox_sts[0];
		ioctl->VendorSpecificStatus[1] = mbox_sts[1];
		ioctl->ResponseLen = 0;
		goto exit_query_hba_portal;
	}

	phba_portal->SerialNum = le32_to_cpu(sys_info->serialNumber);
	memcpy(phba_portal->IPAddr.IPAddress, ha->ip_address,
	    MIN(sizeof(phba_portal->IPAddr.IPAddress), sizeof(ha->ip_address)));
	memcpy(phba_portal->MacAddr, sys_info->physAddr[0].address,
	    sizeof(phba_portal->MacAddr));
	memcpy(phba_portal->Manufacturer, sys_info->vendorId,
	    sizeof(phba_portal->Manufacturer));
	memcpy(phba_portal->Model, sys_info->productId,
	    sizeof(phba_portal->Model));
#if 0 && defined(__VMKERNEL_MODULE__)
	spin_unlock(&ha->dma_buf_spinlock);
#else
	up(&ha->dma_buf_sem);
#endif

	/*memcpy(phba_portal->OptRomVersion, ?,
		sizeof(phba_portal->OptRomVersion)); */

#ifdef __VMWARE__
        /*
	 * Return the PCI info so we can associate an instance to a
	 * card
	 */
	phba_portal->Bus = ha->pdev->bus->number;
	phba_portal->DevFn = ha->pdev->devfn;
#endif

	/*
	 * Copy the IOCTL EXT_HBA_ISCSI_PORTAL buffer to the user's data space
	 */
	if (copy_to_user(Q64BIT_TO_PTR(ioctl->ResponseAdr),
	    phba_portal, ioctl->ResponseLen) != 0) {
		QL4PRINT(QLP2, printk("scsi%d: %s: memory area too small\n",
				      ha->host_no, __func__));

		status = (-EFAULT);
		ioctl->Status = EXT_STATUS_COPY_ERR;
		ioctl->ResponseLen = 0;
		goto exit_query_hba_portal;
	}

	ioctl->Status = EXT_STATUS_OK;

exit_query_hba_portal:
	qla4xxx_free_ioctl_scrap_mem(ha);
	LEAVE(__func__);
	return(status);
}

/**************************************************************************
 * qla4extioctl_query_disc_iscsi_node
 *	This routine retrieves the properties of the attached devices
 *	registered as iSCSI nodes discovered by the HBA driver.
 *
 * Input:
 *	ha    = adapter structure pointer.
 *	ioctl = IOCTL structure pointer.
 *
 * Output:
 *	None
 *
 * Returns:
 *	QLA_SUCCESS = success
 *	QLA_ERROR   = error
 *
 * Context:
 *	Kernel context.
 **************************************************************************/
static int
qla4extioctl_query_disc_iscsi_node(scsi_qla_host_t *ha, EXT_IOCTL_ISCSI *ioctl)
{
	int		status = 0;
	DEV_DB_ENTRY 	*fw_ddb_entry = (DEV_DB_ENTRY *) ha->dma_buf.virt_addr;
	dma_addr_t      fw_ddb_entry_dma = ha->dma_buf.phys_addr;
	EXT_DISC_ISCSI_NODE *pdisc_node = NULL;
	ddb_entry_t 	*ddb_entry;


	ENTER(__func__);
#if 0 && defined(__VMKERNEL_MODULE__)
	spin_lock(&ha->dma_buf_spinlock);
#else
	down(&ha->dma_buf_sem);
#endif

	if (ioctl->ResponseLen < sizeof(EXT_DISC_ISCSI_NODE)) {
		QL4PRINT(QLP2,
			 printk("scsi%d: %s: response buffer too small\n",
				ha->host_no, __func__));

		status = (-ENOMEM);
		ioctl->Status = EXT_STATUS_BUFFER_TOO_SMALL;
		ioctl->ResponseLen = 0;
		goto exit_disc_node;
	}

	if (ha->dma_buf.buf_len < sizeof(DEV_DB_ENTRY)) {
		if (qla4xxx_resize_dma_buf(ha, &ha->dma_buf,
					   sizeof(DEV_DB_ENTRY)) != QLA_SUCCESS) {
			QL4PRINT(QLP2,
				 printk("scsi%d: %s: unable to allocate memory "
					"for dma buffer.\n",
					ha->host_no, __func__));

			status = (-ENOMEM);
			ioctl->Status = EXT_STATUS_NO_MEMORY;
			ioctl->ResponseLen = 0;
			goto exit_disc_node;
		}
	}

	if (qla4xxx_get_ioctl_scrap_mem(ha, (void **)&pdisc_node,
	    sizeof(EXT_DISC_ISCSI_NODE))) {
		/* not enough memory */
		QL4PRINT(QLP2,
		    printk("scsi%d: %s: inst=%d scrap not big enough. "
		    "size requested=%ld.\n",
		    ha->host_no, __func__, ha->instance,
		    (ulong)sizeof(EXT_DISC_ISCSI_NODE)));

		status = (-ENOMEM);
		ioctl->Status = EXT_STATUS_NO_MEMORY;
		ioctl->ResponseLen = 0;
		goto exit_disc_node;
	}

	/* ----- get device database entry info from firmware ---- */
	if (qla4xxx_get_ddb_entry(ha, ioctl->Instance, fw_ddb_entry,
				  fw_ddb_entry_dma, NULL, NULL, NULL, NULL,
				  NULL, NULL) != QLA_SUCCESS) {
		QL4PRINT(QLP2, printk("scsi%d: %s: failed to get DEV_DB_ENTRY "
				      "info.\n", ha->host_no, __func__));

		status = (-EFAULT);
		ioctl->Status = EXT_STATUS_ERR;
		ioctl->RequestLen = 0;
		goto exit_disc_node;
	}

	/* --- Transfer data from Fw's DEV_DB_ENTRY buffer to
	*      IOCTL's EXT_DISC_ISCSI_PORTAL buffer --- */
	memset(pdisc_node, 0, sizeof(EXT_DISC_ISCSI_NODE));
	pdisc_node->NodeInfo.PortalCount = 1;
	pdisc_node->NodeInfo.IPAddr.Type = EXT_DEF_TYPE_ISCSI_IP;
	memcpy(pdisc_node->NodeInfo.IPAddr.IPAddress, fw_ddb_entry->RemoteIPAddr,
	    MIN(sizeof(pdisc_node->NodeInfo.IPAddr.IPAddress),
	    sizeof(fw_ddb_entry->RemoteIPAddr)));
	strncpy(pdisc_node->NodeInfo.Alias, fw_ddb_entry->iSCSIAlias,
	    MIN(sizeof(pdisc_node->NodeInfo.Alias),
	    sizeof(fw_ddb_entry->iSCSIAlias)));
	strncpy(pdisc_node->NodeInfo.iSCSIName, fw_ddb_entry->iscsiName,
	    MIN(sizeof(pdisc_node->NodeInfo.iSCSIName),
	    sizeof(fw_ddb_entry->iscsiName)));

	if ((ddb_entry = qla4xxx_lookup_ddb_by_fw_index(ha, ioctl->Instance)) ==
	    NULL) {
		QL4PRINT(QLP2,
			 printk("scsi%d: %s: device index [%d] not logged in. "
				"Dummy target info returned.\n",
				ha->host_no, __func__, ioctl->Instance));

		pdisc_node->SessionID       = 0xDEAD;
		pdisc_node->ConnectionID    = 0xDEAD;
		pdisc_node->PortalGroupID   = 0xDEAD;
		pdisc_node->ScsiAddr.Bus    = 0xFF;
		pdisc_node->ScsiAddr.Target = 0xFF;
		pdisc_node->ScsiAddr.Lun    = 0xFF;
	}
	else {
		pdisc_node->SessionID       = ddb_entry->target_session_id;
		pdisc_node->ConnectionID    = ddb_entry->connection_id;
		pdisc_node->PortalGroupID   = 0;
		pdisc_node->ScsiAddr.Bus    = ddb_entry->bus;
		pdisc_node->ScsiAddr.Target = ddb_entry->target;
		pdisc_node->ScsiAddr.Lun    = 0;
	}

	/* --- Copy Results to user space --- */
	if (copy_to_user(Q64BIT_TO_PTR(ioctl->ResponseAdr),
                         pdisc_node, sizeof(EXT_DISC_ISCSI_NODE)) != 0) {
		QL4PRINT(QLP2,
			 printk("scsi%d: %s: copy error to user space.\n",
				ha->host_no, __func__));

		status = (-EFAULT);
		ioctl->Status = EXT_STATUS_COPY_ERR;
		ioctl->ResponseLen = 0;
		goto exit_disc_node;
	}

	ioctl->Status = EXT_STATUS_OK;

exit_disc_node:
	qla4xxx_free_ioctl_scrap_mem(ha);

#if 0 && defined(__VMKERNEL_MODULE__)
	spin_unlock(&ha->dma_buf_spinlock);
#else
	up(&ha->dma_buf_sem);
#endif
	LEAVE(__func__);
	return(status);
}

/**************************************************************************
 * qla4extioctl_query_disc_iscsi_portal
 *	This routine retrieves the properties of the iSCSI portal
 *	discovered by the HBA driver.
 *
 * Input:
 *	ha    = adapter structure pointer.
 *	ioctl = IOCTL structure pointer.
 *
 * Output:
 *	None
 *
 * Returns:
 *	QLA_SUCCESS = success
 *	QLA_ERROR   = error
 *
 * Context:
 *	Kernel context.
 **************************************************************************/
static int
qla4extioctl_query_disc_iscsi_portal(scsi_qla_host_t *ha,
    EXT_IOCTL_ISCSI *ioctl)
{
	int		status = 0;
	DEV_DB_ENTRY 	*fw_ddb_entry = (DEV_DB_ENTRY *) ha->dma_buf.virt_addr;
	dma_addr_t    	fw_ddb_entry_dma = ha->dma_buf.phys_addr;
	EXT_DISC_ISCSI_PORTAL *pdisc_portal = NULL;


	ENTER(__func__);

#if 0 && defined(__VMKERNEL_MODULE__)
	spin_lock(&ha->dma_buf_spinlock);
#else
	down(&ha->dma_buf_sem);
#endif

	if (qla4xxx_get_ioctl_scrap_mem(ha, (void **)&pdisc_portal,
	    sizeof(EXT_DISC_ISCSI_PORTAL))) {
		/* not enough memory */
		QL4PRINT(QLP2,
			 printk("scsi%d: %s: inst=%d scrap not big enough. "
                                "size requested=%ld.\n",
                                ha->host_no, __func__, ha->instance,
                                (ulong)sizeof(EXT_DISC_ISCSI_PORTAL)));

		status = (-ENOMEM);
		ioctl->Status = EXT_STATUS_NO_MEMORY;
		ioctl->ResponseLen = 0;
		goto exit_disc_portal;
	}

	if (ioctl->ResponseLen < sizeof(EXT_DISC_ISCSI_PORTAL)) {
		QL4PRINT(QLP2,
			 printk("scsi%d: %s: response buffer too small\n",
				ha->host_no, __func__));

		status = (-ENOMEM);
		ioctl->Status = EXT_STATUS_BUFFER_TOO_SMALL;
		ioctl->ResponseLen = 0;
		goto exit_disc_portal;
	}

	if (ha->dma_buf.buf_len < sizeof(DEV_DB_ENTRY)) {
		if (qla4xxx_resize_dma_buf(ha, &ha->dma_buf,
					   sizeof(DEV_DB_ENTRY)) != QLA_SUCCESS) {
			QL4PRINT(QLP2,
				 printk("scsi%d: %s: unable to allocate memory "
					"for dma buffer.\n",
					ha->host_no, __func__));

			status = (-ENOMEM);
			ioctl->Status = EXT_STATUS_NO_MEMORY;
			ioctl->ResponseLen = 0;
			goto exit_disc_portal;
		}
	}

	/* ----- get device database entry info from firmware ---- */
	if (qla4xxx_get_ddb_entry(ha, ioctl->Instance, fw_ddb_entry,
				  fw_ddb_entry_dma, NULL, NULL, NULL, NULL,
				  NULL, NULL) != QLA_SUCCESS) {
		QL4PRINT(QLP2,
			 printk("scsi%d: %s: failed to get DEV_DB_ENTRY info "
				"for index[%d].\n",
				ha->host_no, __func__, ioctl->Instance));

		status = (-EFAULT);
		ioctl->Status = EXT_STATUS_ERR;
		ioctl->RequestLen = 0;
		goto exit_disc_portal;
	}

	/* --- Transfer data from Fw's DEV_DB_ENTRY buffer to IOCTL's
	*      EXT_DISC_ISCSI_PORTAL buffer --- */
	memset(pdisc_portal, 0, sizeof(EXT_DISC_ISCSI_PORTAL));
	memcpy(pdisc_portal->IPAddr.IPAddress, fw_ddb_entry->RemoteIPAddr,
	    MIN(sizeof(pdisc_portal->IPAddr.IPAddress),
	    sizeof(fw_ddb_entry->RemoteIPAddr)));

	pdisc_portal->PortNumber = le16_to_cpu(fw_ddb_entry->RemoteTCPPortNumber);
	pdisc_portal->IPAddr.Type = EXT_DEF_TYPE_ISCSI_IP;
	pdisc_portal->NodeCount = 0;

	strncpy(pdisc_portal->HostName, fw_ddb_entry->iscsiName,
	    MIN(sizeof(pdisc_portal->HostName),
                sizeof(fw_ddb_entry->iscsiName)));

	/* --- Copy Results to user space --- */
	if (copy_to_user(Q64BIT_TO_PTR(ioctl->ResponseAdr),
	     pdisc_portal, sizeof(EXT_DISC_ISCSI_PORTAL)) != 0) {
		QL4PRINT(QLP2,
			 printk("scsi%d: %s: copy error to user space.\n",
				ha->host_no, __func__));

		status = (-EFAULT);
		ioctl->Status = EXT_STATUS_COPY_ERR;
		ioctl->ResponseLen = 0;
		goto exit_disc_portal;
	}

	ioctl->Status = EXT_STATUS_OK;

exit_disc_portal:
	qla4xxx_free_ioctl_scrap_mem(ha);
#if 0 && defined(__VMKERNEL_MODULE__)
	spin_unlock(&ha->dma_buf_spinlock);

#else
	up(&ha->dma_buf_sem);
#endif
	LEAVE(__func__);
	return(status);
}

/**************************************************************************
 * qla4extioctl_query_driver
 *	This routine retrieves the driver properties.
 *
 * Input:
 *	ha    = adapter structure pointer.
 *	ioctl = IOCTL structure pointer.
 *
 * Output:
 *	None
 *
 * Returns:
 *	QLA_SUCCESS = success
 *	QLA_ERROR   = error
 *
 * Context:
 *	Kernel context.
 **************************************************************************/
static int
qla4extioctl_query_driver(scsi_qla_host_t *ha, EXT_IOCTL_ISCSI *ioctl)
{
	EXT_DRIVER_INFO	*pdinfo = NULL;
	int		status = 0;

	ENTER(__func__);

	if (qla4xxx_get_ioctl_scrap_mem(ha, (void **)&pdinfo,
	    sizeof(EXT_DRIVER_INFO))) {
		/* not enough memory */
		QL4PRINT(QLP2,
		    printk("scsi%d: %s: inst=%d scrap not big enough. "
		    "size requested=%ld.\n",
		    ha->host_no, __func__, ha->instance,
	            (ulong)sizeof(EXT_DRIVER_INFO)));

		status = (-ENOMEM);
		ioctl->Status = EXT_STATUS_NO_MEMORY;
		ioctl->ResponseLen = 0;
		goto exit_query_driver;
	}

	if (ioctl->ResponseLen < sizeof(EXT_DRIVER_INFO)) {
		QL4PRINT(QLP2,
			 printk("scsi%d: %s: response buffer too small\n",
				ha->host_no, __func__));

		status = (-ENOMEM);
		ioctl->Status = EXT_STATUS_BUFFER_TOO_SMALL;
		ioctl->ResponseLen = 0;
		goto exit_query_driver;
	}

	memset(pdinfo, 0, sizeof(EXT_DRIVER_INFO));
	memcpy(pdinfo->Version, QLA4XXX_DRIVER_VERSION,
	       sizeof(QLA4XXX_DRIVER_VERSION));

	pdinfo->NumOfBus        = EXT_DEF_MAX_HBA;
	pdinfo->TargetsPerBus   = EXT_DEF_MAX_TARGET;
	pdinfo->LunPerTarget    = EXT_DEF_MAX_LUN;
	pdinfo->LunPerTargetOS  = EXT_DEF_MAX_BUS;

	if (sizeof(dma_addr_t) > 4)
		pdinfo->DmaBitAddresses = 1;  /* 64-bit */
	else
		pdinfo->DmaBitAddresses = 0;  /* 32-bit */

	#if MEMORY_MAPPED_IO
	pdinfo->IoMapType       = 1;  /* Memory Mapped I/O */
	#else
	pdinfo->IoMapType       = 0;  /* I/O Mapped I/O */
	#endif

	if (copy_to_user(Q64BIT_TO_PTR(ioctl->ResponseAdr), pdinfo,
	    sizeof(EXT_DRIVER_INFO)) != 0) {
		status = (-EFAULT);
		QL4PRINT(QLP2,
		    printk("scsi%d: %s: error copy to response buffer.\n",
		    ha->host_no, __func__));

		status = (-EFAULT);
		ioctl->Status = EXT_STATUS_COPY_ERR;
		ioctl->ResponseLen = 0;
		goto exit_query_driver;
	}

	ioctl->Status = EXT_STATUS_OK;

exit_query_driver:
	qla4xxx_free_ioctl_scrap_mem(ha);
	LEAVE(__func__);
	return(status);
}

/**************************************************************************
 * qla4extioctl_query_fw
 *	This routine retrieves the firmware properties.
 *
 * Input:
 *	ha    = adapter structure pointer.
 *	ioctl = IOCTL structure pointer.
 *
 * Output:
 *	None
 *
 * Returns:
 *	QLA_SUCCESS = success
 *	QLA_ERROR   = error
 *
 * Context:
 *	Kernel context.
 **************************************************************************/
static int
qla4extioctl_query_fw(scsi_qla_host_t *ha, EXT_IOCTL_ISCSI *ioctl)
{
	EXT_FW_INFO	*pfw_info = NULL;
	uint32_t	mbox_cmd[MBOX_REG_COUNT];
	uint32_t	mbox_sts[MBOX_REG_COUNT];
	int		status = 0;

	ENTER(__func__);

	if (qla4xxx_get_ioctl_scrap_mem(ha, (void **)&pfw_info,
	    sizeof(EXT_FW_INFO))) {
		/* not enough memory */
		QL4PRINT(QLP2,
		    printk("scsi%d: %s: inst=%d scrap not big enough. "
		    "size requested=%ld.\n",
		    ha->host_no, __func__, ha->instance,
		    (ulong)sizeof(EXT_FW_INFO)));

		status = (-ENOMEM);
		ioctl->Status = EXT_STATUS_NO_MEMORY;
		ioctl->ResponseLen = 0;
		goto exit_query_fw;
	}

	if (ioctl->ResponseLen < sizeof(EXT_FW_INFO)) {
		QL4PRINT(QLP2,
			 printk("scsi%d: %s: response buffer too small\n",
				ha->host_no, __func__));

		status = (-ENOMEM);
		ioctl->Status = EXT_STATUS_BUFFER_TOO_SMALL;
		ioctl->ResponseLen = 0;
		goto exit_query_fw;
	}

	/* Fill in structure */
	memset(pfw_info, 0, sizeof(EXT_FW_INFO));

	/* ----- Get firmware version information ---- */
	memset(mbox_cmd, 0, sizeof(mbox_cmd));
	memset(mbox_sts, 0, sizeof(mbox_sts));
	mbox_cmd[0] = MBOX_CMD_ABOUT_FW;

	/*
	 * NOTE: In QLA4010, mailboxes 2 & 3 may hold an address for data.
	 * Make sure that we write 0 to those mailboxes, if unused.
	 */
	if (qla4xxx_mailbox_command(ha, 4, 5, &mbox_cmd[0], &mbox_sts[0]) !=
	    QLA_SUCCESS) {
		QL4PRINT(QLP2,
			 printk("scsi%d: %s: MBOX_CMD_ABOUT_FW failed w/ "
				"status %04X\n",
				ha->host_no, __func__, mbox_sts[0]));

		ioctl->Status = EXT_STATUS_MAILBOX;
		ioctl->VendorSpecificStatus[0] = mbox_sts[0];
		ioctl->VendorSpecificStatus[1] = mbox_sts[1];
		ioctl->VendorSpecificStatus[2] = mbox_sts[2];
		ioctl->VendorSpecificStatus[3] = mbox_sts[3];
		ioctl->VendorSpecificStatus[4] = mbox_sts[4];
		ioctl->ResponseLen = 0;
		goto exit_query_fw;
	}

	sprintf(pfw_info->Version, "FW Version %d.%d Patch %d Build %d",
	    mbox_sts[1], mbox_sts[2], mbox_sts[3], mbox_sts[4]);

	/* Copy info to caller */
	if (copy_to_user(Q64BIT_TO_PTR(ioctl->ResponseAdr), pfw_info,
	    sizeof(EXT_FW_INFO)) != 0) {
		QL4PRINT(QLP2,
		    printk("scsi%d: %s: response copy error.\n",
		    ha->host_no, __func__));

		status = (-EFAULT);
		ioctl->Status = EXT_STATUS_COPY_ERR;
		ioctl->ResponseLen = 0;
		goto exit_query_fw;
	}

	ioctl->Status = EXT_STATUS_OK;

exit_query_fw:
	qla4xxx_free_ioctl_scrap_mem(ha);
	LEAVE(__func__);
	return(status);
}

/**************************************************************************
 * qla4extioctl_query_chip
 *	This routine retrieves the chip properties.
 *
 * Input:
 *	ha    = adapter structure pointer.
 *	ioctl = IOCTL structure pointer.
 *
 * Output:
 *	None
 *
 * Returns:
 *	QLA_SUCCESS = success
 *	QLA_ERROR   = error
 *
 * Context:
 *	Kernel context.
 **************************************************************************/
static int
qla4extioctl_query_chip(scsi_qla_host_t *ha, EXT_IOCTL_ISCSI *ioctl)
{
	int		status = 0;
	uint32_t	mbox_cmd[MBOX_REG_COUNT];
	uint32_t	mbox_sts[MBOX_REG_COUNT];
	EXT_CHIP_INFO	*pchip_info = NULL;


	ENTER(__func__);

#if 1 || defined(__VMKERNEL_MODULE__)
	down(&ha->dma_buf_sem);
#endif
	if (qla4xxx_get_ioctl_scrap_mem(ha, (void **)&pchip_info,
	    sizeof(EXT_CHIP_INFO))) {
		/* not enough memory */
		QL4PRINT(QLP2,
		    printk("scsi%d: %s: inst=%d scrap not big enough. "
		    "size requested=%ld.\n",
		    ha->host_no, __func__, ha->instance,
		    (ulong)sizeof(EXT_CHIP_INFO)));

		status = (-ENOMEM);
		ioctl->Status = EXT_STATUS_NO_MEMORY;
		ioctl->ResponseLen = 0;
		goto exit_query_chip;
	}

	if (!ioctl->ResponseAdr || ioctl->ResponseLen < sizeof(EXT_CHIP_INFO)) {
		QL4PRINT(QLP2,
			 printk("scsi%d: %s: response buffer too small\n",
				ha->host_no, __func__));

		status = (-ENOMEM);
		ioctl->Status = EXT_STATUS_BUFFER_TOO_SMALL;
		ioctl->ResponseLen = 0;
		goto exit_query_chip;
	}

	/* Fill in structure */
	memset(pchip_info, 0, sizeof(EXT_CHIP_INFO));

	pchip_info->VendorId    = ha->pdev->vendor;
	pchip_info->DeviceId    = ha->pdev->device;
	pchip_info->SubVendorId = ha->pdev->subsystem_vendor;
	pchip_info->SubSystemId = ha->pdev->subsystem_device;

	/* ----- Get firmware state information ---- */
	memset(mbox_cmd, 0, sizeof(mbox_cmd));
	memset(mbox_sts, 0, sizeof(mbox_sts));
	mbox_cmd[0] = MBOX_CMD_GET_FW_STATE;
	if (qla4xxx_mailbox_command(ha, 1, 4, &mbox_cmd[0], &mbox_sts[0]) !=
	    QLA_SUCCESS) {
		QL4PRINT(QLP2,
			 printk("scsi%d: %s: MBOX_CMD_GET_FW_STATE failed "
				"w/ status %04X\n",
				ha->host_no, __func__, mbox_sts[0]));

		ioctl->Status = EXT_STATUS_MAILBOX;
		ioctl->VendorSpecificStatus[0] = mbox_sts[0];
		ioctl->VendorSpecificStatus[1] = mbox_sts[1];
		ioctl->VendorSpecificStatus[2] = mbox_sts[2];
		ioctl->VendorSpecificStatus[3] = mbox_sts[3];
		ioctl->ResponseLen = 0;
		goto exit_query_chip;
	}

	pchip_info->BoardID     = mbox_sts[2];

	/* Copy info to caller */
	if (copy_to_user(Q64BIT_TO_PTR(ioctl->ResponseAdr),
	    pchip_info, sizeof(EXT_CHIP_INFO)) != 0) {
		QL4PRINT(QLP2,
		    printk("scsi%d: %s: response copy error.\n",
		    ha->host_no, __func__));

		status = (-EFAULT);
		ioctl->Status = EXT_STATUS_COPY_ERR;
		ioctl->ResponseLen = 0;
		goto exit_query_chip;
	}

	ioctl->Status = EXT_STATUS_OK;

exit_query_chip:
	qla4xxx_free_ioctl_scrap_mem(ha);
#if 1 || defined(__VMKERNEL_MODULE__)
	up(&ha->dma_buf_sem);
#endif
	LEAVE(__func__);
	return(status);
}

#ifndef QLA4000
/**************************************************************************
 * qla4extioctl_query_ip_state
 *	This routine retrieves operational state of the address control
 *	block and IPv6.
 *
 * Input:
 *	ha    = adapter structure pointer.
 *	ioctl = IOCTL structure pointer.
 *
 * Output:
 *	None
 *
 * Returns:
 *	QLA_SUCCESS = success
 *	QLA_ERROR   = error
 *
 * Context:
 *	Kernel context.
 **************************************************************************/
static int
qla4extioctl_query_ip_state(scsi_qla_host_t *ha, EXT_IOCTL_ISCSI *ioctl)
{
	int	 status = 0;
	EXT_QUERY_IP_STATE *pquery_ip_state = NULL;
	uint32_t mbox_cmd[MBOX_REG_COUNT];
	uint32_t mbox_sts[MBOX_REG_COUNT];

	ENTER(__func__);

	if (qla4xxx_get_ioctl_scrap_mem(ha, (void **)&pquery_ip_state,
	    sizeof(EXT_QUERY_IP_STATE))) {
		/* not enough memory */
		QL4PRINT(QLP2,
		    printk("scsi%d: %s: inst=%d scrap not big enough. "
		    "size requested=%ld.\n",
		    ha->host_no, __func__, ha->instance,
                    (ulong)sizeof(EXT_QUERY_IP_STATE)));

		status = (-ENOMEM);
		ioctl->Status = EXT_STATUS_NO_MEMORY;
		ioctl->ResponseLen = 0;
		goto exit_query_ip_state;
	}

	memset(mbox_cmd, 0, sizeof(mbox_cmd));
	memset(mbox_sts, 0, sizeof(mbox_sts));
	mbox_cmd[0] = MBOX_CMD_GET_IP_ADDR_STATE;
	mbox_cmd[1] = ioctl->Instance;
	mbox_cmd[2] = ioctl->Reserved1;

	if (qla4xxx_mailbox_command(ha, 3, 8, &mbox_cmd[0], &mbox_sts[0])
	    != QLA_SUCCESS) {
		QL4PRINT(QLP2, printk("scsi%d: %s: MBOX_CMD_GET_IP_ADDR_STATE "
				       "FAILED w/ status %04X\n",
		    ha->host_no, __func__, mbox_sts[0]));

		ioctl->Status = EXT_STATUS_MAILBOX;
		ioctl->VendorSpecificStatus[0] = mbox_sts[0];
		ioctl->VendorSpecificStatus[1] = mbox_sts[1];
		ioctl->VendorSpecificStatus[2] = mbox_sts[2];
		ioctl->VendorSpecificStatus[3] = mbox_sts[3];
		ioctl->VendorSpecificStatus[4] = mbox_sts[4];
		ioctl->VendorSpecificStatus[5] = mbox_sts[5];
		ioctl->VendorSpecificStatus[6] = mbox_sts[6];
		ioctl->VendorSpecificStatus[7] = mbox_sts[7];
		ioctl->ResponseLen = 0;
		goto exit_query_ip_state;
	}

	QL4PRINT(QLP4, printk("scsi%d: %s: Succeeded\n",
	    ha->host_no, __func__));

	memcpy(pquery_ip_state->IP_ACBState, &mbox_sts[1],
	       MIN(sizeof(pquery_ip_state->IP_ACBState), sizeof(mbox_sts[1])));
	pquery_ip_state->ValidLifetime     = le32_to_cpu(mbox_sts[2]);
	pquery_ip_state->PreferredLifetime = le32_to_cpu(mbox_sts[3]);
	memcpy(&pquery_ip_state->IPAddressInfo1, &mbox_sts[4],
	       MIN(sizeof(pquery_ip_state->IPAddressInfo2), sizeof(mbox_sts[4])));
	memcpy(&pquery_ip_state->IPAddressInfo2, &mbox_sts[5],
	       MIN(sizeof(pquery_ip_state->IPAddressInfo3), sizeof(mbox_sts[5])));
	memcpy(&pquery_ip_state->IPAddressInfo3, &mbox_sts[6],
	       MIN(sizeof(pquery_ip_state->IPAddressInfo4), sizeof(mbox_sts[6])));
	memcpy(&pquery_ip_state->IPAddressInfo4, &mbox_sts[7],
	       MIN(sizeof(pquery_ip_state->IPAddressInfo1), sizeof(mbox_sts[7])));

	/*
	 * Copy the IOCTL EXT_QUERY_IP_STATE buffer to the user's data space
	 */
	if (copy_to_user(Q64BIT_TO_PTR(ioctl->ResponseAdr),
	    pquery_ip_state, ioctl->ResponseLen) != 0) {
		QL4PRINT(QLP2, printk("scsi%d: %s: copy failed\n",
				      ha->host_no, __func__));

		status = (-EFAULT);
		ioctl->Status = EXT_STATUS_COPY_ERR;
		ioctl->ResponseLen = 0;
		goto exit_query_ip_state;
	}

	ioctl->Status = EXT_STATUS_OK;

exit_query_ip_state:
	qla4xxx_free_ioctl_scrap_mem(ha);
	LEAVE(__func__);
	return(status);
}

/**************************************************************************
 * qla4extioctl_query_device_current_ip
 *	This routine retrieves the current IP Address, TCP port nember
 * 	and other information for the specified DDB.
 *
 * Input:
 *	ha    = adapter structure pointer.
 *	ioctl = IOCTL structure pointer.
 *
 * Output:
 *	None
 *
 * Returns:
 *	QLA_SUCCESS = success
 *	QLA_ERROR   = error
 *
 * Context:
 *	Kernel context.
 **************************************************************************/
static int
qla4extioctl_query_device_current_ip(scsi_qla_host_t *ha, EXT_IOCTL_ISCSI *ioctl)
{
	int	 status = 0;
	EXT_QUERY_DEVICE_CURRENT_IP *pquery_ddb_curr_ip = NULL;
	uint32_t mbox_cmd[MBOX_REG_COUNT];
	uint32_t mbox_sts[MBOX_REG_COUNT];

	ENTER(__func__);

	if (qla4xxx_get_ioctl_scrap_mem(ha, (void **)&pquery_ddb_curr_ip,
	    sizeof(EXT_QUERY_DEVICE_CURRENT_IP))) {
		/* not enough memory */
		QL4PRINT(QLP2,
		    printk("scsi%d: %s: inst=%d scrap not big enough. "
		    "size requested=%ld.\n",
		    ha->host_no, __func__, ha->instance,
		    (ulong)sizeof(EXT_QUERY_DEVICE_CURRENT_IP)));

		status = (-ENOMEM);
		ioctl->Status = EXT_STATUS_NO_MEMORY;
		ioctl->ResponseLen = 0;
		goto exit_query_ddb_curr_ip;
	}

	memset(mbox_cmd, 0, sizeof(mbox_cmd));
	memset(mbox_sts, 0, sizeof(mbox_sts));
	mbox_cmd[0] = MBOX_CMD_GET_DATABASE_ENTRY_CURRENT_IP_ADDR;
	mbox_cmd[1] = ioctl->Instance;

	if (qla4xxx_mailbox_command(ha, 2, 7, &mbox_cmd[0], &mbox_sts[0])
	    != QLA_SUCCESS) {
		QL4PRINT(QLP2, printk("scsi%d: %s: "
			"MBOX_CMD_GET_DATABASE_ENTRY_CURRENT_IP_ADDR "
			"FAILED w/ status %04X\n",
		        ha->host_no, __func__, mbox_sts[0]));

		ioctl->Status = EXT_STATUS_MAILBOX;
		ioctl->VendorSpecificStatus[0] = mbox_sts[0];
		ioctl->VendorSpecificStatus[1] = mbox_sts[1];
		ioctl->VendorSpecificStatus[2] = mbox_sts[2];
		ioctl->VendorSpecificStatus[3] = mbox_sts[3];
		ioctl->VendorSpecificStatus[4] = mbox_sts[4];
		ioctl->VendorSpecificStatus[5] = mbox_sts[5];
		ioctl->VendorSpecificStatus[6] = mbox_sts[6];
		ioctl->ResponseLen = 0;
		goto exit_query_ddb_curr_ip;
	}

	QL4PRINT(QLP4, printk("scsi%d: %s: Succeeded\n",
	    ha->host_no, __func__));


	pquery_ddb_curr_ip->DeviceState = (uint32_t) __le16_to_cpu(MSW(mbox_sts[1]));
	pquery_ddb_curr_ip->TCPPort = __le16_to_cpu(MSW(mbox_sts[2]));
	memcpy(&pquery_ddb_curr_ip->Flags[0], &mbox_sts[2],
	       sizeof(pquery_ddb_curr_ip->Flags));
	memcpy(&pquery_ddb_curr_ip->Addr.IPAddress[0], &mbox_sts[3], 4);
	memcpy(&pquery_ddb_curr_ip->Addr.IPAddress[4], &mbox_sts[4], 4);
	memcpy(&pquery_ddb_curr_ip->Addr.IPAddress[8], &mbox_sts[5], 4);
	memcpy(&pquery_ddb_curr_ip->Addr.IPAddress[12], &mbox_sts[6], 4);
	if (IS_IPv6_ENABLED(ha))
		pquery_ddb_curr_ip->Addr.Type = EXT_DEF_TYPE_ISCSI_IPV6;

	/*
	 * Copy the IOCTL EXT_QUERY_IP_STATE buffer to the user's data space
	 */
	if (copy_to_user(Q64BIT_TO_PTR(ioctl->ResponseAdr),
	    pquery_ddb_curr_ip, ioctl->ResponseLen) != 0) {
		QL4PRINT(QLP2, printk("scsi%d: %s: copy failed\n",
				      ha->host_no, __func__));

		status = (-EFAULT);
		ioctl->Status = EXT_STATUS_COPY_ERR;
		ioctl->ResponseLen = 0;
		goto exit_query_ddb_curr_ip;
	}

	ioctl->Status = EXT_STATUS_OK;

exit_query_ddb_curr_ip:
	qla4xxx_free_ioctl_scrap_mem(ha);
	LEAVE(__func__);
	return(status);
}

#endif

/**************************************************************************
 * qla4extioctl_query
 *	This routine calls query IOCTLs based on the IOCTL Sub Code.
 *
 * Input:
 *	ha    = adapter structure pointer.
 *	ioctl = IOCTL structure pointer.
 *
 * Output:
 *	None
 *
 * Returns:
 *	QLA_SUCCESS = success
 *	QLA_ERROR   = error
 *    	-EINVAL     = if the command is invalid
 *
 * Context:
 *	Kernel context.
 **************************************************************************/
static int
qla4extioctl_query(scsi_qla_host_t *ha, EXT_IOCTL_ISCSI *ioctl)
{
	switch (ioctl->SubCode) {
	case EXT_SC_QUERY_HBA_ISCSI_NODE:
		return(qla4extioctl_query_hba_iscsi_node(ha, ioctl));

	case EXT_SC_QUERY_HBA_ISCSI_PORTAL:
		return(qla4extioctl_query_hba_iscsi_portal(ha, ioctl));

	case EXT_SC_QUERY_DISC_ISCSI_NODE:
		return(qla4extioctl_query_disc_iscsi_node(ha, ioctl));

	case EXT_SC_QUERY_DISC_ISCSI_PORTAL:
		return(qla4extioctl_query_disc_iscsi_portal(ha, ioctl));

	case EXT_SC_QUERY_DRIVER:
		return(qla4extioctl_query_driver(ha, ioctl));

	case EXT_SC_QUERY_FW:
		return(qla4extioctl_query_fw(ha, ioctl));

	case EXT_SC_QUERY_CHIP:
		return(qla4extioctl_query_chip(ha, ioctl));

#ifndef QLA4000
	case EXT_SC_QUERY_IP_STATE:
		return(qla4extioctl_query_ip_state(ha, ioctl));

	case EXT_SC_QUERY_DEVICE_CURRENT_IP:
		return(qla4extioctl_query_device_current_ip(ha, ioctl));
#endif

	default:
		QL4PRINT(QLP2,
			 printk("scsi%d: %s: unsupported query sub-command "
				"code (%X)\n",
				ha->host_no, __func__, ioctl->SubCode));

		ioctl->Status = EXT_STATUS_UNSUPPORTED_SUBCODE;
		return(-EINVAL);
	}
}

/**************************************************************************
 * qla4extioctl_reg_aen
 *	This routine enables/disables storing of asynchronous events
 *	from the ISP into the driver's internal buffer.
 *
 * Input:
 *	ha    = adapter structure pointer.
 *	ioctl = IOCTL structure pointer.
 *
 * Output:
 *	None
 *
 * Returns:
 *	QLA_SUCCESS = success
 *	QLA_ERROR   = error
 *
 * Context:
 *	Kernel context.
 **************************************************************************/
static int
qla4extioctl_reg_aen(scsi_qla_host_t *ha, EXT_IOCTL_ISCSI *ioctl)
{
	EXT_REG_AEN_ISCSI reg_aen;
	int	status = 0;

	ENTER(__func__);

	if (ioctl->RequestLen > sizeof(EXT_REG_AEN_ISCSI)) {
		QL4PRINT(QLP2,
		    printk("scsi%d: %s: memory area too small\n",
		    ha->host_no, __func__));

		status = (-ENOMEM);
		ioctl->Status = EXT_STATUS_BUFFER_TOO_SMALL;
		ioctl->ResponseLen = 0;
		goto exit_reg_aen;
	}

	/* --- Copy input structure from user space --- */
	if ((status = copy_from_user((void *)&reg_aen,
	    Q64BIT_TO_PTR(ioctl->RequestAdr), sizeof(EXT_REG_AEN_ISCSI))) != 0) {
		QL4PRINT(QLP2,
		    printk("scsi%d: %s: unable to copy data from "
		    "user's memory area\n", ha->host_no, __func__));

		status = (-EFAULT);
		ioctl->Status = EXT_STATUS_COPY_ERR;
		ioctl->ResponseLen = 0;
		goto exit_reg_aen;
	}

	ha->aen_reg_mask = reg_aen.Enable;
	ioctl->Status = EXT_STATUS_OK;

exit_reg_aen:
	LEAVE(__func__);

	return(status);
}

/**************************************************************************
 * qla4extioctl_get_aen
 *	This routine retrieves the contents of the driver's internal
 *	asynchronous event tracking queue.
 *
 * Input:
 *	ha    = adapter structure pointer.
 *	ioctl = IOCTL structure pointer.
 *
 * Output:
 *	None
 *
 * Returns:
 *	QLA_SUCCESS = success
 *	QLA_ERROR   = error
 *
 * Context:
 *	Kernel context.
 **************************************************************************/
static int
qla4extioctl_get_aen(scsi_qla_host_t *ha, EXT_IOCTL_ISCSI *ioctl)
{
	EXT_ASYNC_EVENT *async_event;
	uint32_t 	num_aens_returned;
	uint32_t 	drvr_aen_index;
	int 		i, j;
	int		status = 0;

	ENTER(__func__);

	if (ha->aen_reg_mask == EXT_DEF_ENABLE_NO_AENS) {
		QL4PRINT(QLP2, printk("scsi%d: %s: AEN mask not enabled\n",
		    ha->host_no, __func__));
		ioctl->Status = EXT_STATUS_OK;
		return 0;
	}

	if (qla4xxx_get_ioctl_scrap_mem(ha, (void **)&async_event,
	    sizeof(EXT_ASYNC_EVENT) * EXT_DEF_MAX_AEN_QUEUE)) {
		/* not enough memory */
		QL4PRINT(QLP2,
		    printk("scsi%d: %s: inst=%d scrap not big enough. "
		    "size requested=%ld.\n",
		    ha->host_no, __func__, ha->instance,
		    (ulong)(sizeof(EXT_ASYNC_EVENT) * EXT_DEF_MAX_AEN_QUEUE)));

		status = (-ENOMEM);
		ioctl->Status = EXT_STATUS_NO_MEMORY;
		ioctl->ResponseLen = 0;
		goto exit_get_gen;
	}

	if (ioctl->ResponseLen < sizeof(EXT_ASYNC_EVENT) * EXT_DEF_MAX_AEN_QUEUE) {
		QL4PRINT(QLP2, printk("scsi%d: %s: memory area too small\n",
		    ha->host_no, __func__));

		status = (-ENOMEM);
		ioctl->Status = EXT_STATUS_BUFFER_TOO_SMALL;
		ioctl->ResponseLen = 0;
		goto exit_get_gen;
	}

	memset(async_event, 0, sizeof(EXT_ASYNC_EVENT) * EXT_DEF_MAX_AEN_QUEUE);

	if (ha->aen_count - ha->aen_report < MAX_AEN_ENTRIES)
		num_aens_returned = ha->aen_count - ha->aen_report;
	else
		num_aens_returned = MAX_AEN_ENTRIES;

	drvr_aen_index = ha->aen_report % MAX_AEN_ENTRIES;
	ha->aen_report += num_aens_returned;

	for (i=0; i<num_aens_returned; i++) {
		async_event[i].AsyncEventCode = ha->aen_q[drvr_aen_index].mbox_sts[0];

		for (j=0; j<EXT_DEF_MAX_AEN_PAYLOAD; j++) {
			async_event[i].Payload[j] = ha->aen_q[drvr_aen_index].mbox_sts[j+1];
		}

		if (drvr_aen_index < MAX_AEN_ENTRIES-1)
			drvr_aen_index++;
		else
			drvr_aen_index = 0;
	}

	ioctl->ResponseLen = sizeof(EXT_ASYNC_EVENT) * num_aens_returned;
	ioctl->Status = EXT_STATUS_OK;

	/*
	 * Copy the IOCTL EXT_ASYNC_EVENT buffer to the user's data space
	 */
	if ((status = copy_to_user(Q64BIT_TO_PTR(ioctl->ResponseAdr), async_event,
	    ioctl->ResponseLen)) != 0) {
		QL4PRINT(QLP2, printk("scsi%d: %s: memory area too small\n",
		    ha->host_no, __func__));

		status = (-EFAULT);
		ioctl->Status = EXT_STATUS_COPY_ERR;
		ioctl->ResponseLen = 0;
	}

exit_get_gen:
        qla4xxx_free_ioctl_scrap_mem(ha);

	LEAVE(__func__);
	return(status);
}

/**************************************************************************
 * qla4extioctl_get_statistics_gen
 *	This routine retrieves the HBA general statistical information.
 *
 * Input:
 *	ha    = adapter structure pointer.
 *	ioctl = IOCTL structure pointer.
 *
 * Output:
 *	None
 *
 * Returns:
 *	QLA_SUCCESS = success
 *	QLA_ERROR   = error
 *
 * Context:
 *	Kernel context.
 **************************************************************************/
static int
qla4extioctl_get_statistics_gen(scsi_qla_host_t *ha, EXT_IOCTL_ISCSI *ioctl)
{
	int			status = 0;
	EXT_HBA_PORT_STAT_GEN	*pstat_gen = NULL;

	QL4PRINT(QLP4, printk("scsi%d: %s: index [%d]\n",
			      ha->host_no, __func__, ioctl->Instance));

	if (qla4xxx_get_ioctl_scrap_mem(ha, (void **)&pstat_gen,
	    sizeof(EXT_HBA_PORT_STAT_GEN))) {
		/* not enough memory */
		QL4PRINT(QLP2,
		    printk("scsi%d: %s: inst=%d scrap not big enough. "
		    "size requested=%ld.\n",
		    ha->host_no, __func__, ha->instance,
		    (ulong)sizeof(EXT_HBA_PORT_STAT_GEN)));

		status = (-ENOMEM);
		ioctl->Status = EXT_STATUS_NO_MEMORY;
		ioctl->ResponseLen = 0;
		goto exit_get_stat_gen;
	}

	if (ioctl->ResponseLen < sizeof(EXT_HBA_PORT_STAT_GEN)) {
		QL4PRINT(QLP2, printk("scsi%d: %s: memory area too small\n",
				      ha->host_no, __func__));

		status = (-ENOMEM);
		ioctl->Status = EXT_STATUS_BUFFER_TOO_SMALL;
		ioctl->ResponseLen = 0;
		goto exit_get_stat_gen;
	}

	/*
	 * Fill in the data
	 */
	memset(pstat_gen, 0, sizeof(EXT_HBA_PORT_STAT_GEN));
	pstat_gen->HBAPortErrorCount     = ha->adapter_error_count;
	pstat_gen->DevicePortErrorCount  = ha->device_error_count;
	pstat_gen->IoCount               = ha->total_io_count;
	pstat_gen->MBytesCount           = ha->total_mbytes_xferred;
	pstat_gen->InterruptCount        = ha->isr_count;
	pstat_gen->LinkFailureCount      = ha->link_failure_count;
	pstat_gen->InvalidCrcCount       = ha->invalid_crc_count;

	/*
	 * Copy the IOCTL EXT_HBA_PORT_STAT_GEN buffer to the user's data space
	 */
	if (copy_to_user(Q64BIT_TO_PTR(ioctl->ResponseAdr),
	    pstat_gen, ioctl->ResponseLen) != 0) {
		QL4PRINT(QLP2, printk("scsi%d: %s: memory area too small\n",
				      ha->host_no, __func__));

		status = (-EFAULT);
		ioctl->Status = EXT_STATUS_COPY_ERR;
		ioctl->ResponseLen = 0;
		goto exit_get_stat_gen;
	}

	ioctl->Status = EXT_STATUS_OK;

exit_get_stat_gen:
	qla4xxx_free_ioctl_scrap_mem(ha);
	return(status);
}

/**************************************************************************
 * qla4extioctl_get_statistics_iscsi
 *	This routine retrieves the HBA iSCSI statistical information.
 *
 * Input:
 *	ha    = adapter structure pointer.
 *	ioctl = IOCTL structure pointer.
 *
 * Output:
 *	None
 *
 * Returns:
 *	QLA_SUCCESS = success
 *	QLA_ERROR   = error
 *
 * Context:
 *	Kernel context.
 **************************************************************************/
static int
qla4extioctl_get_statistics_iscsi(scsi_qla_host_t *ha, EXT_IOCTL_ISCSI *ioctl)
{
	int		status = 0;
	uint32_t	mbox_cmd[MBOX_REG_COUNT];
	uint32_t	mbox_sts[MBOX_REG_COUNT];
	EXT_HBA_PORT_STAT_ISCSI* pstat_local;
	EXT_HBA_PORT_STAT_ISCSI* pstat_user;

	ENTER(__func__);
	QL4PRINT(QLP4, printk("scsi%d: %s: index [%d]\n",
			      ha->host_no, __func__, ioctl->Instance));

#if 0 && defined(__VMKERNEL_MODULE__)
	spin_lock(&ha->dma_buf_spinlock);
#else
	down(&ha->dma_buf_sem);
#endif

	if (qla4xxx_get_ioctl_scrap_mem(ha, (void **)&pstat_user,
	    sizeof(EXT_HBA_PORT_STAT_ISCSI))) {
		/* not enough memory */
		QL4PRINT(QLP2,
		    printk("scsi%d: %s: inst=%d scrap not big enough. "
		    "size requested=%ld.\n",
		    ha->host_no, __func__, ha->instance,
		    (ulong)sizeof(EXT_HBA_PORT_STAT_ISCSI)));

		status = (-ENOMEM);
		ioctl->Status = EXT_STATUS_NO_MEMORY;
		ioctl->ResponseLen = 0;
		goto exit_get_stats_iscsi;
	}

	if (!ioctl->ResponseAdr || !ioctl->ResponseLen) {
		QL4PRINT(QLP2, printk("scsi%d: %s: invalid parameter\n",
		    ha->host_no, __func__));

		status = (-EINVAL);
		ioctl->Status = EXT_STATUS_INVALID_PARAM;
		ioctl->ResponseLen = 0;
		goto exit_get_stats_iscsi;
	}

	if (ioctl->ResponseLen < sizeof(EXT_HBA_PORT_STAT_ISCSI)) {
		QL4PRINT(QLP2, printk("scsi%d: %s: RespLen too small (0x%x),  "
		    "need (0x%x).\n",
		    ha->host_no, __func__, ioctl->ResponseLen,
		    (unsigned int) sizeof(EXT_HBA_PORT_STAT_ISCSI)));

		status = (-ENOMEM);
		ioctl->Status = EXT_STATUS_BUFFER_TOO_SMALL;
		ioctl->ResponseLen = 0;
		goto exit_get_stats_iscsi;
	}

	if ((ha->dma_buf.buf_len < sizeof(EXT_HBA_PORT_STAT_ISCSI)) &&
	    (qla4xxx_resize_dma_buf(ha, &ha->dma_buf,
	    sizeof(EXT_HBA_PORT_STAT_ISCSI)) != QLA_SUCCESS)) {
		QL4PRINT(QLP2, printk("scsi%d: %s: unable to allocate memory "
		    "for dma buffer.\n",
		    ha->host_no, __func__));

		status = (-ENOMEM);
		ioctl->Status = EXT_STATUS_NO_MEMORY;
		ioctl->ResponseLen = 0;
		goto exit_get_stats_iscsi;
	}

	/*
	 * Make the mailbox call
	 */
	memset(mbox_cmd, 0, sizeof(mbox_cmd));
	memset(mbox_sts, 0, sizeof(mbox_sts));
	mbox_cmd[0] = MBOX_CMD_GET_MANAGEMENT_DATA;
	mbox_cmd[1] = ioctl->Instance;
	mbox_cmd[2] = LSDW(ha->dma_buf.phys_addr);
	mbox_cmd[3] = MSDW(ha->dma_buf.phys_addr);

	if (qla4xxx_mailbox_command(ha, 4, 1, &mbox_cmd[0], &mbox_sts[0]) !=
	    QLA_SUCCESS) {
		QL4PRINT(QLP2,
		    printk("scsi%d: %s: get mngmt data for index [%d] failed "
		    "w/ mailbox ststus 0x%x\n",
		    ha->host_no, __func__, ioctl->Instance, mbox_sts[0]));

		ioctl->Status = EXT_STATUS_MAILBOX;
		ioctl->VendorSpecificStatus[0] = mbox_sts[0];
		ioctl->ResponseLen = 0;
		goto exit_get_stats_iscsi;
	}

	pstat_local = (EXT_HBA_PORT_STAT_ISCSI *) ha->dma_buf.virt_addr;
	memset(pstat_user, 0, sizeof(EXT_HBA_PORT_STAT_ISCSI));
	pstat_user->MACTxFramesCount          =
	    le64_to_cpu(pstat_local->MACTxFramesCount);
	pstat_user->MACTxBytesCount           =
	    le64_to_cpu(pstat_local->MACTxBytesCount);
	pstat_user->MACRxFramesCount          =
	    le64_to_cpu(pstat_local->MACRxFramesCount);
	pstat_user->MACRxBytesCount           =
	    le64_to_cpu(pstat_local->MACRxBytesCount);
	pstat_user->MACCRCErrorCount          =
	    le64_to_cpu(pstat_local->MACCRCErrorCount);
	pstat_user->MACEncodingErrorCount     =
	    le64_to_cpu(pstat_local->MACEncodingErrorCount);
	pstat_user->IPTxPacketsCount          =
	    le64_to_cpu(pstat_local->IPTxPacketsCount);
	pstat_user->IPTxBytesCount            =
	    le64_to_cpu(pstat_local->IPTxBytesCount);
	pstat_user->IPTxFragmentsCount        =
	    le64_to_cpu(pstat_local->IPTxFragmentsCount);
	pstat_user->IPRxPacketsCount          =
	    le64_to_cpu(pstat_local->IPRxPacketsCount);
	pstat_user->IPRxBytesCount            =
	    le64_to_cpu(pstat_local->IPRxBytesCount);
	pstat_user->IPRxFragmentsCount        =
	    le64_to_cpu(pstat_local->IPRxFragmentsCount);
	pstat_user->IPDatagramReassemblyCount =
	    le64_to_cpu(pstat_local->IPDatagramReassemblyCount);
	pstat_user->IPv6RxPacketsCount        =
	    le64_to_cpu(pstat_local->IPv6RxPacketsCount);
	pstat_user->IPRxPacketErrorCount      =
	    le64_to_cpu(pstat_local->IPRxPacketErrorCount);
	pstat_user->IPReassemblyErrorCount    =
	    le64_to_cpu(pstat_local->IPReassemblyErrorCount);
	pstat_user->TCPTxSegmentsCount        =
	    le64_to_cpu(pstat_local->TCPTxSegmentsCount);
	pstat_user->TCPTxBytesCount           =
	    le64_to_cpu(pstat_local->TCPTxBytesCount);
	pstat_user->TCPRxSegmentsCount        =
	    le64_to_cpu(pstat_local->TCPRxSegmentsCount);
	pstat_user->TCPRxBytesCount           =
	    le64_to_cpu(pstat_local->TCPRxBytesCount);
	pstat_user->TCPTimerExpiredCount      =
	    le64_to_cpu(pstat_local->TCPTimerExpiredCount);
	pstat_user->TCPRxACKCount             =
	    le64_to_cpu(pstat_local->TCPRxACKCount);
	pstat_user->TCPTxACKCount             =
	    le64_to_cpu(pstat_local->TCPTxACKCount);
	pstat_user->TCPRxErrorSegmentCount    =
	    le64_to_cpu(pstat_local->TCPRxErrorSegmentCount);
	pstat_user->TCPWindowProbeUpdateCount =
	    le64_to_cpu(pstat_local->TCPWindowProbeUpdateCount);
	pstat_user->iSCSITxPDUCount           =
	    le64_to_cpu(pstat_local->iSCSITxPDUCount);
	pstat_user->iSCSITxBytesCount         =
	    le64_to_cpu(pstat_local->iSCSITxBytesCount);
	pstat_user->iSCSIRxPDUCount           =
	    le64_to_cpu(pstat_local->iSCSIRxPDUCount);
	pstat_user->iSCSIRxBytesCount         =
	    le64_to_cpu(pstat_local->iSCSIRxBytesCount);
	pstat_user->iSCSICompleteIOsCount     =
	    le64_to_cpu(pstat_local->iSCSICompleteIOsCount);
	pstat_user->iSCSIUnexpectedIORxCount  =
	    le64_to_cpu(pstat_local->iSCSIUnexpectedIORxCount);
	pstat_user->iSCSIFormatErrorCount     =
	    le64_to_cpu(pstat_local->iSCSIFormatErrorCount);
	pstat_user->iSCSIHeaderDigestCount    =
	    le64_to_cpu(pstat_local->iSCSIHeaderDigestCount);
	pstat_user->iSCSIDataDigestErrorCount =
	    le64_to_cpu(pstat_local->iSCSIDataDigestErrorCount);
	pstat_user->iSCSISeqErrorCount        =
	    le64_to_cpu(pstat_local->iSCSISeqErrorCount);

	/*
	 * Copy the data from the dma buffer to the user's data space
	 */
	if (copy_to_user(Q64BIT_TO_PTR(ioctl->ResponseAdr),
	    pstat_user, ioctl->ResponseLen) != 0) {
		QL4PRINT(QLP2, printk("scsi%d: %s: memory area too small\n",
		    ha->host_no, __func__));

		status = (-EFAULT);
		ioctl->Status = EXT_STATUS_COPY_ERR;
		ioctl->ResponseLen = 0;
		goto exit_get_stats_iscsi;
	}

	ioctl->Status = EXT_STATUS_OK;

exit_get_stats_iscsi:
	qla4xxx_free_ioctl_scrap_mem(ha);
#if 0 && defined(__VMKERNEL_MODULE__)
	spin_unlock(&ha->dma_buf_spinlock);
#else
	up(&ha->dma_buf_sem);
#endif
	LEAVE(__func__);
	return(status);
}

/**************************************************************************
 * qla4extioctl_get_device_entry_iscsi
 *	This routine retrieves the database entry for the specified device.
 *
 * Input:
 *	ha    = adapter structure pointer.
 *	ioctl = IOCTL structure pointer.
 *
 * Output:
 *	None
 *
 * Returns:
 *	QLA_SUCCESS = success
 *	QLA_ERROR   = error
 *
 * Context:
 *	Kernel context.
 **************************************************************************/
static int
qla4extioctl_get_device_entry_iscsi(scsi_qla_host_t *ha, EXT_IOCTL_ISCSI *ioctl)
{
	int                     status = 0;
	uint32_t		mbox_cmd[MBOX_REG_COUNT];
	uint32_t		mbox_sts[MBOX_REG_COUNT];
	DEV_DB_ENTRY		*pfw_ddb_entry;
	EXT_DEVICE_ENTRY_ISCSI	*pdev_entry;

	ENTER(__func__);
	QL4PRINT(QLP4, printk("scsi%d: %s: index [%d]\n",
			      ha->host_no, __func__, ioctl->Instance));

#if 0 && defined(__VMKERNEL_MODULE__)
	spin_lock(&ha->dma_buf_spinlock);
#else
	down(&ha->dma_buf_sem);
#endif

	if (qla4xxx_get_ioctl_scrap_mem(ha, (void **)&pdev_entry,
	    sizeof(EXT_DEVICE_ENTRY_ISCSI))) {
		/* not enough memory */
		QL4PRINT(QLP2,
		    printk("scsi%d: %s: inst=%d scrap not big enough. "
		    "size requested=%ld.\n",
		    ha->host_no, __func__, ha->instance,
		    (ulong)sizeof(EXT_DEVICE_ENTRY_ISCSI)));

		status = (-ENOMEM);
		ioctl->Status = EXT_STATUS_NO_MEMORY;
		ioctl->ResponseLen = 0;
		goto exit_get_dev_entry;
	}

#if 0 && defined(__VMKERNEL_MODULE__)
	if (!ha->dma_buf.virt_addr || !ha->dma_buf.phys_addr ||
	    !ioctl->ResponseAdr) {
		QL4PRINT(QLP2,
			 printk("scsi%d: %s: memory allocation problem\n",
				ha->host_no, __func__));

		status = (-ENOMEM);
		ioctl->Status = EXT_STATUS_NO_MEMORY;
		ioctl->ResponseLen = 0;
		goto exit_get_dev_entry;
	}
#else
        /*
         * We check the phys and virt addresses below before resizing
         * our buffer.  We shouldn't bail out becasue they're unset
         * here.
         */
	if (!ioctl->ResponseAdr) {
		QL4PRINT(QLP2,
			 printk("scsi%d: %s: memory allocation problem\n",
				ha->host_no, __func__));
		status = (-ENOMEM);
		ioctl->Status = EXT_STATUS_NO_MEMORY;
		ioctl->ResponseLen = 0;
		goto exit_get_dev_entry;
	}
#endif

	#if 0
	if (ioctl->ResponseLen < sizeof(*pdev_entry) ||
	    ha->dma_buf.buf_len < sizeof(DEV_DB_ENTRY)) {
		QL4PRINT(QLP2, printk("scsi%d: %s: memory area too small\n",
				      ha->host_no, __func__));

		status = (-ENOMEM);
		ioctl->Status = EXT_STATUS_BUFFER_TOO_SMALL;
		ioctl->ResponseLen = 0;
		goto exit_get_dev_entry;
	}
	#endif
	if (!ha->dma_buf.virt_addr || !ha->dma_buf.phys_addr ||
	    (ha->dma_buf.buf_len < sizeof(*pfw_ddb_entry))) {
		if (qla4xxx_resize_dma_buf(ha, &ha->dma_buf,
		    sizeof(DEV_DB_ENTRY)) != QLA_SUCCESS) {
			QL4PRINT(QLP2,
				 printk("scsi%d: %s: unable to allocate memory "
					"for dma buffer.\n",
					ha->host_no, __func__));

			status = (-ENOMEM);
			ioctl->Status = EXT_STATUS_NO_MEMORY;
			ioctl->ResponseLen = 0;
			goto exit_get_dev_entry;
		}
	}


	/*
	 * Make the mailbox call
	 */
	memset(mbox_cmd, 0, sizeof(mbox_cmd));
	memset(mbox_sts, 0, sizeof(mbox_sts));
	memset(pdev_entry, 0, sizeof(EXT_DEVICE_ENTRY_ISCSI));

	if (ioctl->SubCode == EXT_SC_GET_DEVICE_ENTRY_ISCSI)
		mbox_cmd[0] = MBOX_CMD_GET_DATABASE_ENTRY;
	else
		mbox_cmd[0] = MBOX_CMD_GET_DATABASE_ENTRY_DEFAULTS;

	mbox_cmd[1] = ioctl->Instance;
	mbox_cmd[2] = LSDW(ha->dma_buf.phys_addr);
	mbox_cmd[3] = MSDW(ha->dma_buf.phys_addr);

	if (qla4xxx_mailbox_command(ha, 4, 5, &mbox_cmd[0], &mbox_sts[0]) !=
	    QLA_SUCCESS) {
		QL4PRINT(QLP2,
		    printk("scsi%d: %s: get ddb entry for index [%d] failed "
				      "w/ mailbox ststus 0x%x\n",
		    ha->host_no, __func__, ioctl->Instance, mbox_sts[0]));

		ioctl->Status = EXT_STATUS_MAILBOX;
		ioctl->VendorSpecificStatus[0] = mbox_sts[0];
		ioctl->VendorSpecificStatus[1] = mbox_sts[1];
		ioctl->VendorSpecificStatus[2] = mbox_sts[2];
		ioctl->VendorSpecificStatus[3] = mbox_sts[3];
		ioctl->VendorSpecificStatus[4] = mbox_sts[4];
		ioctl->ResponseLen = 0;
		goto exit_get_dev_entry;
	}

	/*
	 * Transfer data from Fw's DEV_DB_ENTRY buffer to IOCTL's
	 * EXT_DEVICE_ENTRY_ISCSI buffer
	 */
	pfw_ddb_entry = ha->dma_buf.virt_addr;

	pdev_entry->NumValid     = mbox_sts[2];
	pdev_entry->NextValid    = mbox_sts[3];
	pdev_entry->DeviceState  = mbox_sts[4];
	pdev_entry->Options      = pfw_ddb_entry->options;
	pdev_entry->TargetSessID = le16_to_cpu(pfw_ddb_entry->TSID);
	memcpy(pdev_entry->InitiatorSessID, pfw_ddb_entry->ISID,
	       sizeof(pfw_ddb_entry->ISID));

	pdev_entry->DeviceInfo.DeviceType = le16_to_cpu(EXT_DEF_ISCSI_REMOTE);
	pdev_entry->DeviceInfo.ExeThrottle =
	    le16_to_cpu(pfw_ddb_entry->exeThrottle);
	pdev_entry->DeviceInfo.InitMarkerlessInt =
	    le16_to_cpu(pfw_ddb_entry->iSCSIMaxSndDataSegLen);
	pdev_entry->DeviceInfo.RetryCount = pfw_ddb_entry->retryCount;
	pdev_entry->DeviceInfo.RetryDelay = pfw_ddb_entry->retryDelay;
	pdev_entry->DeviceInfo.iSCSIOptions =
	    le16_to_cpu(pfw_ddb_entry->iSCSIOptions);
	pdev_entry->DeviceInfo.TCPOptions =
	    le16_to_cpu(pfw_ddb_entry->TCPOptions);
	pdev_entry->DeviceInfo.IPOptions =
	    le16_to_cpu(pfw_ddb_entry->IPOptions);
	pdev_entry->DeviceInfo.MaxPDUSize =
	    le16_to_cpu(pfw_ddb_entry->maxPDUSize);
	pdev_entry->DeviceInfo.FirstBurstSize =
	    le16_to_cpu(pfw_ddb_entry->firstBurstSize);
	pdev_entry->DeviceInfo.LogoutMinTime =
	    le16_to_cpu(pfw_ddb_entry->DefaultTime2Wait);
	pdev_entry->DeviceInfo.LogoutMaxTime =
	    le16_to_cpu(pfw_ddb_entry->DefaultTime2Retain);
	pdev_entry->DeviceInfo.MaxOutstandingR2T =
	    le16_to_cpu(pfw_ddb_entry->maxOutstndngR2T);
	pdev_entry->DeviceInfo.KeepAliveTimeout =
	    le16_to_cpu(pfw_ddb_entry->keepAliveTimeout);
	pdev_entry->DeviceInfo.PortNumber =
	    le16_to_cpu(pfw_ddb_entry->RemoteTCPPortNumber);
	pdev_entry->DeviceInfo.MaxBurstSize =
	    le16_to_cpu(pfw_ddb_entry->maxBurstSize);
	pdev_entry->DeviceInfo.TaskMgmtTimeout =
	    le16_to_cpu(pfw_ddb_entry->taskMngmntTimeout);
	pdev_entry->EntryInfo.PortalCount = mbox_sts[2];
	pdev_entry->ExeCount = le16_to_cpu(pfw_ddb_entry->exeCount);
	pdev_entry->DDBLink = le16_to_cpu(pfw_ddb_entry->ddbLink);

	memcpy(pdev_entry->DeviceInfo.TargetAddr, pfw_ddb_entry->targetAddr,
	    sizeof(pdev_entry->DeviceInfo.TargetAddr));
	memcpy(pdev_entry->EntryInfo.IPAddr.IPAddress, pfw_ddb_entry->RemoteIPAddr,
	    sizeof(pdev_entry->EntryInfo.IPAddr.IPAddress));
	memcpy(pdev_entry->EntryInfo.iSCSIName, pfw_ddb_entry->iscsiName,
	    sizeof(pdev_entry->EntryInfo.iSCSIName));
	memcpy(pdev_entry->EntryInfo.Alias, pfw_ddb_entry->iSCSIAlias,
	    sizeof(pdev_entry->EntryInfo.Alias));

	QL4PRINT(QLP10,
	    printk("scsi%d: DEV_DB_ENTRY structure:\n", ha->host_no));
	qla4xxx_dump_bytes(QLP10,
	    pfw_ddb_entry, sizeof(DEV_DB_ENTRY));
	QL4PRINT(QLP10,
	    printk("scsi%d: EXT_DEVICE_ENTRY_ISCSI structure:\n", ha->host_no));
	qla4xxx_dump_bytes(QLP10, pdev_entry, sizeof(EXT_DEVICE_ENTRY_ISCSI));

	/*
	 * Copy the IOCTL EXT_DEVICE_ENTRY_ISCSI buffer to the user's data space
	 */
	if (copy_to_user(Q64BIT_TO_PTR(ioctl->ResponseAdr),
	    pdev_entry, ioctl->ResponseLen) != 0) {
		QL4PRINT(QLP2, printk("scsi%d: %s: memory area too small\n",
				      ha->host_no, __func__));

		status = (-EFAULT);
		ioctl->Status = EXT_STATUS_COPY_ERR;
		ioctl->ResponseLen = 0;
		goto exit_get_dev_entry;
	}

	ioctl->Status = EXT_STATUS_OK;

exit_get_dev_entry:
	qla4xxx_free_ioctl_scrap_mem(ha);
#if 0 && defined(__VMKERNEL_MODULE__)
	spin_unlock(&ha->dma_buf_spinlock);
#else
	up(&ha->dma_buf_sem);
#endif
	LEAVE(__func__);
	return(status);
}


/**************************************************************************
 * qla4extioctl_get_init_fw_iscsi
 *	This routine retrieves the initialize firmware control block for
 *	the specified HBA.
 *
 * Input:
 *	ha    = adapter structure pointer.
 *	ioctl = IOCTL structure pointer.
 *
 * Output:
 *	None
 *
 * Returns:
 *	QLA_SUCCESS = success
 *	QLA_ERROR   = error
 *
 * Context:
 *	Kernel context.
 **************************************************************************/
static int
qla4extioctl_get_init_fw_iscsi(scsi_qla_host_t *ha, EXT_IOCTL_ISCSI *ioctl)
{
	int			status = 0;
	uint32_t		mbox_cmd[MBOX_REG_COUNT];
	uint32_t		mbox_sts[MBOX_REG_COUNT];
	EXT_INIT_FW_ISCSI	*pinit_fw = NULL;
	ADDRESS_CTRL_BLK	*pinit_fw_cb;

	ENTER(__func__);
#if 0 && defined(__VMKERNEL_MODULE__)
	spin_lock(&ha->dma_buf_spinlock);
#else
	down(&ha->dma_buf_sem);
#endif

	if (qla4xxx_get_ioctl_scrap_mem(ha, (void **)&pinit_fw,
	    sizeof(EXT_INIT_FW_ISCSI))) {
		/* not enough memory */
		QL4PRINT(QLP2,
		    printk("scsi%d: %s: inst=%d scrap not big enough. "
		    "size requested=%ld.\n",
		    ha->host_no, __func__, ha->instance,
		    (ulong)sizeof(EXT_INIT_FW_ISCSI)));

		status = (-ENOMEM);
		ioctl->Status = EXT_STATUS_NO_MEMORY;
		ioctl->ResponseLen = 0;
		goto exit_get_init_fw;
	}

	if (!ha->dma_buf.virt_addr || !ha->dma_buf.phys_addr ||
	    (ha->dma_buf.buf_len < ha->ifcb_size) ||
	    (ioctl->ResponseLen < sizeof(EXT_INIT_FW_ISCSI))) {
		QL4PRINT(QLP2,
			 printk("scsi%d: %s: response buffer too small\n",
				ha->host_no, __func__));

		status = (-ENOMEM);
		ioctl->Status = EXT_STATUS_BUFFER_TOO_SMALL;
		ioctl->ResponseLen = 0;
		goto exit_get_init_fw;
	}

	/*
	 * Send mailbox command
	 */
	memset(mbox_cmd, 0, sizeof(mbox_cmd));
	memset(mbox_sts, 0, sizeof(mbox_sts));
	switch (ioctl->SubCode) {
	case EXT_SC_GET_INIT_FW_ISCSI:
		mbox_cmd[0] = MBOX_CMD_GET_INIT_FW_CTRL_BLOCK;
		break;
	case EXT_SC_GET_INIT_FW_DEFAULTS_ISCSI:
		mbox_cmd[0] = MBOX_CMD_GET_INIT_FW_CTRL_BLOCK_DEFAULTS;
		break;
	default:
		QL4PRINT(QLP2,
		    printk("scsi%d: %s: invalid subcode (0x%04X) speficied\n",
		    ha->host_no, __func__, ioctl->SubCode));

		status = (-EINVAL);
		ioctl->Status = EXT_STATUS_INVALID_PARAM;
		ioctl->ResponseLen = 0;
		goto exit_get_init_fw;
	}

	mbox_cmd[1] = 0;
	mbox_cmd[2] = LSDW(ha->dma_buf.phys_addr);
	mbox_cmd[3] = MSDW(ha->dma_buf.phys_addr);
	if (ha->acb_version == ACB_SUPPORTED) {
		mbox_cmd[4] = ha->ifcb_size;
	}

	if (qla4xxx_mailbox_command(ha, 5, 1, &mbox_cmd[0], &mbox_sts[0]) ==
	    QLA_ERROR) {
		QL4PRINT(QLP2, printk("scsi%d: %s: command failed \n",
		    ha->host_no, __func__));

		ioctl->Status = EXT_STATUS_MAILBOX;
		ioctl->VendorSpecificStatus[0] = mbox_sts[0];
		goto exit_get_init_fw;
	}

	/*
	 * Transfer Data from DMA buffer to Local buffer
	 */
	pinit_fw_cb = (ADDRESS_CTRL_BLK *)ha->dma_buf.virt_addr;
	memset(pinit_fw, 0, sizeof(EXT_INIT_FW_ISCSI));

	pinit_fw->Version         = pinit_fw_cb->Version;
	pinit_fw->FWOptions       = le16_to_cpu(pinit_fw_cb->FwOptions);
	pinit_fw->AddFWOptions    = le16_to_cpu(pinit_fw_cb->AddFwOptions);
	memcpy(pinit_fw->IPAddr.IPAddress, pinit_fw_cb->IPAddr,
	    MIN(sizeof(pinit_fw->IPAddr.IPAddress),
	    sizeof(pinit_fw_cb->IPAddr)));
	memcpy(pinit_fw->SubnetMask.IPAddress, pinit_fw_cb->SubnetMask,
	    MIN(sizeof(pinit_fw->SubnetMask.IPAddress),
	    sizeof(pinit_fw_cb->SubnetMask)));
	memcpy(pinit_fw->Gateway.IPAddress, pinit_fw_cb->GatewayIPAddr,
	    MIN(sizeof(pinit_fw->Gateway.IPAddress),
	    sizeof(pinit_fw_cb->GatewayIPAddr)));
	memcpy(pinit_fw->DNSConfig.IPAddr.IPAddress,
	    pinit_fw_cb->PriDNSIPAddr,
	    MIN(sizeof(pinit_fw->DNSConfig.IPAddr.IPAddress),
	    sizeof(pinit_fw_cb->PriDNSIPAddr)));
	memcpy(pinit_fw->Alias, pinit_fw_cb->iSCSIAlias,
	    MIN(sizeof(pinit_fw->Alias), sizeof(pinit_fw_cb->iSCSIAlias)));
	memcpy(pinit_fw->iSCSIName, pinit_fw_cb->iSCSINameString,
	    MIN(sizeof(pinit_fw->iSCSIName),
	    sizeof(pinit_fw_cb->iSCSINameString)));

	pinit_fw->DeviceInfo.DeviceType = le16_to_cpu(EXT_DEF_ISCSI_LOCAL);
	pinit_fw->DeviceInfo.ExeThrottle =
	    le16_to_cpu(pinit_fw_cb->ExecThrottle);
	pinit_fw->DeviceInfo.RetryCount = pinit_fw_cb->ZIOCount;
	pinit_fw->DeviceInfo.iSCSIOptions =
	    le16_to_cpu(pinit_fw_cb->iSCSIOptions);
	pinit_fw->DeviceInfo.TCPOptions = le16_to_cpu(pinit_fw_cb->TCPOptions);
	pinit_fw->DeviceInfo.IPOptions = le16_to_cpu(pinit_fw_cb->IPOptions);
	pinit_fw->DeviceInfo.MaxPDUSize = le16_to_cpu(pinit_fw_cb->MaxPDUSize);
	pinit_fw->DeviceInfo.FirstBurstSize =
	    le16_to_cpu(pinit_fw_cb->FirstBurstSize);
	pinit_fw->DeviceInfo.LogoutMinTime =
	    le16_to_cpu(pinit_fw_cb->DefaultTime2Wait);
	pinit_fw->DeviceInfo.LogoutMaxTime =
	    le16_to_cpu(pinit_fw_cb->DefaultTime2Retain);
	pinit_fw->DeviceInfo.LogoutMaxTime =
	    le16_to_cpu(pinit_fw_cb->DefaultTime2Retain);
	pinit_fw->DeviceInfo.MaxOutstandingR2T =
	    le16_to_cpu(pinit_fw_cb->MaxOutStndngR2T);
	pinit_fw->DeviceInfo.KeepAliveTimeout =
	    le16_to_cpu(pinit_fw_cb->KeepAliveTimeout);
	pinit_fw->DeviceInfo.PortNumber = le16_to_cpu(pinit_fw_cb->PortNumber);
	pinit_fw->DeviceInfo.MaxBurstSize =
	    le16_to_cpu(pinit_fw_cb->MaxBurstSize);

	/*
	 * Copy the local data to the user's buffer
	 */
	if (copy_to_user(Q64BIT_TO_PTR(ioctl->ResponseAdr),
	    pinit_fw, sizeof(EXT_INIT_FW_ISCSI)) != 0) {
		QL4PRINT(QLP2,
		    printk("scsi%d: %s: unable to copy data to user's "
		    "memory area\n", ha->host_no, __func__));

		status = (-EFAULT);
		ioctl->Status = EXT_STATUS_COPY_ERR;
		ioctl->ResponseLen = 0;
		goto exit_get_init_fw;
	}

	ioctl->Status = EXT_STATUS_OK;

	QL4PRINT(QLP10,
	    printk("scsi%d: EXT_INIT_FW_ISCSI structure:\n", ha->host_no));
	qla4xxx_dump_bytes(QLP10, pinit_fw, sizeof(EXT_INIT_FW_ISCSI));

exit_get_init_fw:
	qla4xxx_free_ioctl_scrap_mem(ha);
#if 0 && defined(__VMKERNEL_MODULE__)
	spin_unlock(&ha->dma_buf_spinlock);
#else
	up(&ha->dma_buf_sem);
#endif
	LEAVE(__func__);
	return(status);
}

#ifndef QLA4000
/**************************************************************************
 * qla4extioctl_get_isns_server
 *	This routine retrieves the iSNS server information.
 *
 * Input:
 *	ha    = adapter structure pointer.
 *	ioctl = IOCTL structure pointer.
 *
 * Output:
 *	None
 *
 * Returns:
 *	QLA_SUCCESS = success
 *	QLA_ERROR   = error
 *
 * Context:
 *	Kernel context.
 **************************************************************************/
static int
qla4extioctl_get_isns_server(scsi_qla_host_t *ha, EXT_IOCTL_ISCSI *ioctl)
{
	int			status = 0;
	uint32_t		mbox_cmd[MBOX_REG_COUNT];
	uint32_t		mbox_sts[MBOX_REG_COUNT];
	EXT_ISNS_SERVER		*pisns_server = NULL;
	FLASH_INIT_FW_CTRL_BLK	*pflash_init_fw_cb = NULL;

	ENTER(__func__);
#if 0 && defined(__VMKERNEL_MODULE__)
	spin_lock(&ha->dma_buf_spinlock);
#else
	down(&ha->dma_buf_sem);
#endif

	if (qla4xxx_get_ioctl_scrap_mem(ha, (void **)&pisns_server,
	    sizeof(EXT_ISNS_SERVER))) {
		/* not enough memory */
		QL4PRINT(QLP2,
		    printk("scsi%d: %s: inst=%d scrap not big enough. "
		    "size requested=%ld.\n",
		    ha->host_no, __func__, ha->instance,
		    (ulong)sizeof(EXT_ISNS_SERVER)));

		status = (-ENOMEM);
		ioctl->Status = EXT_STATUS_NO_MEMORY;
		ioctl->ResponseLen = 0;
		goto exit_get_isns_server;
	}

	if (ioctl->ResponseLen < sizeof(EXT_ISNS_SERVER)) {
		QL4PRINT(QLP2,
			 printk("scsi%d: %s: response buffer too small\n",
				ha->host_no, __func__));

		status = (-ENOMEM);
		ioctl->Status = EXT_STATUS_BUFFER_TOO_SMALL;
		ioctl->ResponseLen = 0;
		goto exit_get_isns_server;
	}

	if (!ha->dma_buf.virt_addr || !ha->dma_buf.phys_addr ||
	    (ha->dma_buf.buf_len < sizeof(FLASH_INIT_FW_CTRL_BLK))) {
		if (qla4xxx_resize_dma_buf(ha, &ha->dma_buf,
		    sizeof(FLASH_INIT_FW_CTRL_BLK)) != QLA_SUCCESS) {
			QL4PRINT(QLP2,
			    printk("scsi%d: %s: unable to allocate memory "
			    "for dma buffer.\n",
			    ha->host_no, __func__));

			status = (-ENOMEM);
			ioctl->Status = EXT_STATUS_NO_MEMORY;
			ioctl->ResponseLen = 0;
			goto exit_get_isns_server;
		}
	}

	/*
	 * First get Flash Initialize Firmware Control Block, so as not to
	 * destroy unaffected data
	 *----------------------------------------------------------------*/
	pflash_init_fw_cb = (FLASH_INIT_FW_CTRL_BLK *)ha->dma_buf.virt_addr;

	memset(mbox_cmd, 0, sizeof(mbox_cmd));
	memset(mbox_sts, 0, sizeof(mbox_sts));
	mbox_cmd[0] = MBOX_CMD_READ_FLASH;
	mbox_cmd[1] = LSDW(ha->dma_buf.phys_addr);
	mbox_cmd[2] = MSDW(ha->dma_buf.phys_addr);
	mbox_cmd[3] = INT_ISCSI_INITFW_FLASH_OFFSET;
	mbox_cmd[4] = sizeof(FLASH_INIT_FW_CTRL_BLK);

	if (qla4xxx_mailbox_command(ha, 5, 2, &mbox_cmd[0], &mbox_sts[0]) ==
	    QLA_ERROR) {
		QL4PRINT(QLP2,
		    printk("scsi%d: %s: READ_FLASH command failed \n",
		    ha->host_no, __func__));

		ioctl->Status = EXT_STATUS_MAILBOX;
		ioctl->VendorSpecificStatus[0] = mbox_sts[0];
		ioctl->VendorSpecificStatus[1] = mbox_sts[1];
		ioctl->ResponseLen = 0;
		goto exit_get_isns_server;
	}

	QL4PRINT(QLP4, printk("scsi%d: %s: READ_FLASH command successful \n",
	    ha->host_no, __func__));

	/*
	 * Copy iSNS Server info to the isns_server structure
	 *---------------------------------------------------*/
	memset(pisns_server, 0, sizeof(EXT_ISNS_SERVER));
	if (pisns_server->IPAddr.Type == EXT_DEF_TYPE_ISCSI_IPV6) {
		pisns_server->PerformiSNSDiscovery =
			(cpu_to_le16(pflash_init_fw_cb->init_fw_cb.IPv6TCPOptions)
			& IPV6_TCPOPT_ISNSv6_ENABLE) ? 1:0;

	}
	else {
		pisns_server->PerformiSNSDiscovery =
			(cpu_to_le16(pflash_init_fw_cb->init_fw_cb.TCPOptions)
			& TOPT_ISNSv4_ENABLE) ? 1:0;
	}
	pisns_server->AutomaticiSNSDiscovery = 0;
	pisns_server->PortNumber =
	    cpu_to_le16(pflash_init_fw_cb->init_fw_cb.iSNSServerPortNumber);
	pisns_server->IPAddr.Type = EXT_DEF_TYPE_ISCSI_IP;
	memcpy(pisns_server->IPAddr.IPAddress,
	       pflash_init_fw_cb->init_fw_cb.iSNSIPAddr,
	       MIN(sizeof(pisns_server->IPAddr.IPAddress),
		   sizeof(pflash_init_fw_cb->init_fw_cb.iSNSIPAddr)));
	memcpy(pisns_server->InitiatorName,
	       pflash_init_fw_cb->init_fw_cb.iSCSINameString,
	       MIN(sizeof(pisns_server->InitiatorName),
		   sizeof(pflash_init_fw_cb->init_fw_cb.iSCSINameString)));

	/*
	 * Display the iSNS Server State
	 *-----------------------------------------------*/
	if ((QLP4|QLP7|QLP20) & ql_dbg_level) {
		memset(mbox_cmd, 0, sizeof(mbox_cmd));
		memset(mbox_sts, 0, sizeof(mbox_sts));
		mbox_cmd[0] = MBOX_CMD_GET_FW_STATE;

		if (qla4xxx_mailbox_command(ha, 1, 4, &mbox_cmd[0], &mbox_sts[0]) ==
		    QLA_SUCCESS) {
			QL4PRINT(QLP4|QLP7|QLP20,
			    printk("scsi%d: %s: GET ISNS SERVICE STATUS = \"%s\"\n",
			    ha->host_no, __func__,
			    ((mbox_sts[3] & FW_ADDSTATE_ISNSv4_SVC_ENABLED) != 0)
			    ? "ENABLED" : "DISABLED"));
		}
		else {
			QL4PRINT(QLP2,
			    printk("scsi%d: %s: MBOX_CMD_GET_FW_STATE command "
			    "failed \n", ha->host_no, __func__));
		}
	}


	/*
	 * Copy the local data to the user's buffer
	 *-----------------------------------------*/
	if (copy_to_user(Q64BIT_TO_PTR(ioctl->ResponseAdr),
	    pisns_server, sizeof(EXT_ISNS_SERVER)) != 0) {
		QL4PRINT(QLP2,
			 printk("scsi%d: %s: unable to copy data to user's "
				"memory area\n", ha->host_no, __func__));

		status = (-EFAULT);
		ioctl->Status = EXT_STATUS_COPY_ERR;
		ioctl->ResponseLen = 0;
		goto exit_get_isns_server;
	}

	ioctl->Status = EXT_STATUS_OK;
	ioctl->ResponseLen = sizeof(EXT_ISNS_SERVER);
	ioctl->ResponseLen = 0;

	QL4PRINT(QLP10,
	    printk("scsi%d: EXT_ISNS_SERVER structure:\n", ha->host_no));
	qla4xxx_dump_bytes(QLP10, pisns_server, sizeof(EXT_ISNS_SERVER));

exit_get_isns_server:
	qla4xxx_free_ioctl_scrap_mem(ha);
#if 0 && defined(__VMKERNEL_MODULE__)
	spin_unlock(&ha->dma_buf_spinlock);
#else
	up(&ha->dma_buf_sem);
#endif
	LEAVE(__func__);
	return(status);
}

/**************************************************************************
 * qla4extioctl_get_isns_disc_targets
 *	This routine retrieves the targets discovered via iSNS.
 *
 * Input:
 *	ha    = adapter structure pointer.
 *	ioctl = IOCTL structure pointer.
 *
 * Output:
 *	None
 *
 * Returns:
 *	QLA_SUCCESS = success
 *	QLA_ERROR   = error
 *
 * Context:
 *	Kernel context.
 **************************************************************************/
static int
qla4extioctl_get_isns_disc_targets(scsi_qla_host_t *ha, EXT_IOCTL_ISCSI *ioctl)
{
	int			status = 0;
	uint32_t		isns_disc_tgt_index_start;
	uint32_t		i, j;
	EXT_ISNS_DISCOVERED_TARGETS *pisns_disc_tgts = NULL;

	ENTER(__func__);
#if 0 && defined(__VMKERNEL_MODULE__)
	spin_lock(&ha->dma_buf_spinlock);
#else
	down(&ha->dma_buf_sem);
#endif

	if (ioctl->ResponseLen < sizeof(EXT_ISNS_DISCOVERED_TARGETS)) {
		QL4PRINT(QLP2, printk("scsi%d: %s: response buffer "
				      "too small.  RspLen=0x%x, need 0x%x\n",
				      ha->host_no, __func__, ioctl->ResponseLen,
				      (unsigned int) sizeof(EXT_ISNS_DISCOVERED_TARGETS)));
		status = (-ENOMEM);
		ioctl->Status = EXT_STATUS_BUFFER_TOO_SMALL;
		ioctl->ResponseLen = 0;
		goto exit_get_isns_disc_tgts;
	}

	if (!ha->dma_buf.virt_addr ||
	    ((ioctl->ResponseLen > ha->dma_buf.buf_len) &&
	    (qla4xxx_resize_dma_buf(ha, &ha->dma_buf,
	    sizeof(EXT_ISNS_DISCOVERED_TARGETS)) != QLA_SUCCESS))) {
		QL4PRINT(QLP2, printk("scsi%d: %s: unable to allocate memory "
				      "for dma buffer.\n",
				      ha->host_no, __func__));

		status = (-ENOMEM);
		ioctl->Status = EXT_STATUS_NO_MEMORY;
		ioctl->ResponseLen = 0;
		goto exit_get_isns_disc_tgts;
	}

	/*
	 * Copy the IOCTL EXT_ISNS_DISCOVERED_TARGETS buffer from the user's
	 * data space
	 */
	pisns_disc_tgts = (EXT_ISNS_DISCOVERED_TARGETS *) ha->dma_buf.virt_addr;
	if (copy_from_user((uint8_t *)pisns_disc_tgts,
	    Q64BIT_TO_PTR(ioctl->RequestAdr), ioctl->RequestLen) != 0) {
		QL4PRINT(QLP2,
			 printk("scsi%d: %s: unable to copy data from user's "
				"memory area\n", ha->host_no, __func__));

		status = (-EFAULT);
		ioctl->Status = EXT_STATUS_COPY_ERR;
		ioctl->ResponseLen = 0;
		goto exit_get_isns_disc_tgts;
	}

	isns_disc_tgt_index_start =
	    pisns_disc_tgts->iSNSDiscoveredTargetIndexStart;
	memset(pisns_disc_tgts, 0, sizeof(EXT_ISNS_DISCOVERED_TARGETS));
	pisns_disc_tgts->iSNSDiscoveredTargetIndexStart =
	    isns_disc_tgt_index_start;

	/*
	 * Transfer Data from Local buffer to DMA buffer
	 */
	if (isns_disc_tgt_index_start < ha->isns_num_discovered_targets) {
		EXT_ISNS_DISCOVERED_TARGET *isns_disc_tgt;
		ISNS_DISCOVERED_TARGET *isns_local_disc_target;

		for (i = isns_disc_tgt_index_start;
		    i < ha->isns_num_discovered_targets &&
		    pisns_disc_tgts->NumiSNSDiscoveredTargets <
		    EXT_DEF_NUM_ISNS_DISCOVERED_TARGETS;
		    i++) {

			isns_disc_tgt = (EXT_ISNS_DISCOVERED_TARGET *)
				&pisns_disc_tgts->iSNSDiscoveredTargets
				[pisns_disc_tgts->NumiSNSDiscoveredTargets];
			isns_local_disc_target = (ISNS_DISCOVERED_TARGET *)
				&ha->dma_mem_blkv->
				isns_discovered_target_database[i];

			isns_disc_tgt->NumPortals =
			    isns_local_disc_target->NumPortals;

			for (j = 0; j < isns_disc_tgt->NumPortals; j++) {
				memcpy(isns_disc_tgt->Portal[j].IPAddr.
				    IPAddress,
				    isns_local_disc_target->Portal[j].IPAddr,
				    MIN(sizeof(isns_disc_tgt->Portal[j].IPAddr.
				    IPAddress),
				    sizeof(isns_local_disc_target->Portal[j].
				    IPAddr)));
				isns_disc_tgt->Portal[j].IPAddr.Type =
				    EXT_DEF_TYPE_ISCSI_IP;
				isns_disc_tgt->Portal[j].PortNumber =
				    isns_local_disc_target->Portal[j].
				    PortNumber;
			}

			isns_disc_tgt->DDID = isns_local_disc_target->DDID;

			memcpy(isns_disc_tgt->NameString,
			       isns_local_disc_target->NameString,
			       MIN(sizeof(isns_disc_tgt->NameString),
				   sizeof(isns_local_disc_target->NameString)));
			memcpy(isns_disc_tgt->Alias,
			       isns_local_disc_target->Alias,
			       MIN(sizeof(isns_disc_tgt->Alias),
				   sizeof(isns_local_disc_target->Alias)));

			pisns_disc_tgts->NumiSNSDiscoveredTargets++;
		}
	}

	/*
	 * Copy the data to the user's buffer
	 */
	if (copy_to_user(Q64BIT_TO_PTR(ioctl->ResponseAdr),
	    pisns_disc_tgts, sizeof(EXT_ISNS_DISCOVERED_TARGETS)) != 0) {
		QL4PRINT(QLP2,
			 printk("scsi%d: %s: unable to copy data to user's "
				"memory area\n", ha->host_no, __func__));

		status = (-EFAULT);
		ioctl->Status = EXT_STATUS_COPY_ERR;
		ioctl->ResponseLen = 0;
		goto exit_get_isns_disc_tgts;
	}

	ioctl->Status = EXT_STATUS_OK;

	QL4PRINT(QLP10,
	    printk("scsi%d: EXT_INIT_FW_ISCSI structure:\n", ha->host_no));
	qla4xxx_dump_bytes(QLP10,
			   pisns_disc_tgts, sizeof(EXT_ISNS_DISCOVERED_TARGETS));

exit_get_isns_disc_tgts:
#if 0 && defined(__VMKERNEL_MODULE__)
	spin_unlock(&ha->dma_buf_spinlock);
#else
	up(&ha->dma_buf_sem);
#endif
	LEAVE(__func__);
	return(status);
}

/**************************************************************************
 * qla4extioctl_get_acb
 *	This routine retrieves the address control block.
 *
 * Input:
 *	ha    = adapter structure pointer.
 *	ioctl = IOCTL structure pointer.
 *
 * Output:
 *	None
 *
 * Returns:
 *	QLA_SUCCESS = success
 *	QLA_ERROR   = error
 *
 * Context:
 *	Kernel context.
 **************************************************************************/
static int
qla4extioctl_get_acb(scsi_qla_host_t *ha, EXT_IOCTL_ISCSI *ioctl)
{
	int	status = 0;
	uint32_t mbox_cmd[MBOX_REG_COUNT];
	uint32_t mbox_sts[MBOX_REG_COUNT];

	ENTER(__func__);
#if 0 && defined(__VMKERNEL_MODULE__)
	spin_lock(&ha->dma_buf_spinlock);
#else
	down(&ha->dma_buf_sem);
#endif

	if (ioctl->ResponseLen < EXT_DEF_ACB_SIZE) {
		QL4PRINT(QLP2,
		    printk("scsi%d: %s: response buffer too small (%d/%xh)\n",
		    ha->host_no, __func__, ioctl->ResponseLen,
		    ioctl->ResponseLen));

		status = (-ENOMEM);
		ioctl->Status = EXT_STATUS_BUFFER_TOO_SMALL;
		ioctl->ResponseLen = 0;
		goto exit_get_acb;
	}

	if (!ha->dma_buf.virt_addr || !ha->dma_buf.phys_addr ||
	    (ha->dma_buf.buf_len < EXT_DEF_ACB_SIZE)) {
		if (qla4xxx_resize_dma_buf(ha, &ha->dma_buf,
		    EXT_DEF_ACB_SIZE) != QLA_SUCCESS) {
			QL4PRINT(QLP2,
			    printk("scsi%d: %s: unable to allocate memory "
			    "for dma buffer.\n",
			    ha->host_no, __func__));

			status = (-ENOMEM);
			ioctl->Status = EXT_STATUS_NO_MEMORY;
			ioctl->ResponseLen = 0;
			goto exit_get_acb;
		}
	}

	memset(mbox_cmd, 0, sizeof(mbox_cmd));
	memset(mbox_sts, 0, sizeof(mbox_sts));
	mbox_cmd[0] = MBOX_CMD_GET_ACB;
	mbox_cmd[1] = ioctl->Instance;  /* Primary/Secondary */
	mbox_cmd[2] = LSDW(ha->dma_buf.phys_addr);
	mbox_cmd[3] = MSDW(ha->dma_buf.phys_addr);
	mbox_cmd[4] = ioctl->ResponseLen;

	if (qla4xxx_mailbox_command(ha, 5, 5, &mbox_cmd[0], &mbox_sts[0])
	    == QLA_ERROR) {
		QL4PRINT(QLP2, printk("scsi%d: %s: MBOX_CMD_GET_ACB FAILED "
				       " w/ status %04X\n",
				       ha->host_no, __func__, mbox_sts[0]));
		QL4PRINT(QLP2, printk("mbox[1] = 0x%04x\n"
					   "mbox[2] = 0x%04x\n"
					   "mbox[3] = 0x%04x\n"
					   "mbox[4] = 0x%04x\n",
                                           mbox_sts[1], mbox_sts[2],
					   mbox_sts[3], mbox_sts[4]));

		ioctl->Status = EXT_STATUS_MAILBOX;
		ioctl->VendorSpecificStatus[0] = mbox_sts[0];
		ioctl->VendorSpecificStatus[1] = mbox_sts[1];
		ioctl->VendorSpecificStatus[2] = mbox_sts[2];
		ioctl->VendorSpecificStatus[3] = mbox_sts[3];
		ioctl->VendorSpecificStatus[4] = mbox_sts[4];
		ioctl->ResponseLen = 0;
		goto exit_get_acb;
	}

	/*
	 * Transfer Data from DMA buffer to the user's buffer
	 */
	if (copy_to_user(Q64BIT_TO_PTR(ioctl->ResponseAdr),
	    ha->dma_buf.virt_addr, EXT_DEF_ACB_SIZE) != 0) {
		QL4PRINT(QLP2,
		    printk("scsi%d: %s: unable to copy data to user's "
		    "memory area\n", ha->host_no, __func__));

		status = (-EFAULT);
		ioctl->Status = EXT_STATUS_COPY_ERR;
		ioctl->ResponseLen = 0;
		goto exit_get_acb;
	}

	ioctl->Status = EXT_STATUS_OK;
	ioctl->ResponseLen = mbox_sts[4];

	QL4PRINT(QLP4, printk("scsi%d: %s: Succeeded\n",
	    ha->host_no, __func__));
	QL4PRINT(QLP10, printk("scsi%d: ACB dump:\n", ha->host_no));
	qla4xxx_dump_bytes(QLP10, ha->dma_buf.virt_addr, EXT_DEF_ACB_SIZE);

exit_get_acb:
#if 0 && defined(__VMKERNEL_MODULE__)
	spin_unlock(&ha->dma_buf_spinlock);
#else
	up(&ha->dma_buf_sem);
#endif
	LEAVE(__func__);
	return(status);
}
#endif


/**************************************************************************
 * qla4extioctl_get_neighbor_cache
 *	This routine retrieves the neighbor cache or neighbor cache size.
 *
 * Input:
 *	ha    = adapter structure pointer.
 *	ioctl = IOCTL structure pointer.
 *
 * Output:
 *	None
 *
 * Returns:
 *	QLA_SUCCESS = success
 *	QLA_ERROR   = error
 *
 * Context:
 *	Kernel context.
 **************************************************************************/
static int
qla4extioctl_get_neighbor_cache(scsi_qla_host_t *ha, EXT_IOCTL_ISCSI *ioctl)
{
	int	status = 0;
	EXT_NEIGHBOR_CACHE *pncache = NULL;
	uint32_t mbox_cmd[MBOX_REG_COUNT];
	uint32_t mbox_sts[MBOX_REG_COUNT];

	ENTER(__func__);
#if 0 && defined(__VMKERNEL_MODULE__)
	spin_lock(&ha->dma_buf_spinlock);
#else
	down(&ha->dma_buf_sem);
#endif

	/*
	 * Allocate local EXT_NEIGHBOR_CACHE buffer
	 */
	 if (ioctl->ResponseLen >= sizeof(EXT_NEIGHBOR_CACHE) &&
	     qla4xxx_get_ioctl_scrap_mem(ha, (void **)&pncache,
	     ioctl->ResponseLen)) {
		 /* not enough memory */
		 QL4PRINT(QLP2,
		     printk("scsi%d: %s: inst=%d scrap not big enough. "
		     "size requested=%ld.\n",
		     ha->host_no, __func__, ha->instance,
		     (ulong)ioctl->ResponseLen));

		 status = (-ENOMEM);
		 ioctl->Status = EXT_STATUS_NO_MEMORY;
		 ioctl->ResponseLen = 0;
		 goto exit_get_ncache;
	 }

	 /*
	  * Copy user's data to local buffer
	  */
	 if ((status = copy_from_user((uint8_t *)pncache,
	     Q64BIT_TO_PTR(ioctl->RequestAdr), ioctl->RequestLen)) != 0) {
		 QL4PRINT(QLP2,
		     printk("scsi%d: %s: unable to copy data from user's "
		     "memory area\n", ha->host_no, __func__));

		 status = (-EFAULT);
		 ioctl->Status = EXT_STATUS_COPY_ERR;
		 ioctl->ResponseLen = 0;
		 goto exit_get_ncache;
	 }

	 if (pncache->CacheBufferSize) {
		 /*
		  * Resize pre-allocated IOCTL DMA memory, if necesary
		  */
		if (!ha->dma_buf.virt_addr || !ha->dma_buf.phys_addr ||
		    (ha->dma_buf.buf_len < pncache->CacheBufferSize)) {
			if (qla4xxx_resize_dma_buf(ha, &ha->dma_buf,
			    pncache->CacheBufferSize) != QLA_SUCCESS) {
				QL4PRINT(QLP2,
				    printk("scsi%d: %s: unable to allocate memory "
				    "for dma buffer.\n",
				    ha->host_no, __func__));

				status = (-ENOMEM);
				ioctl->Status = EXT_STATUS_NO_MEMORY;
				ioctl->ResponseLen = 0;
				goto exit_get_ncache;
			}
		}
	 }

	memset(mbox_cmd, 0, sizeof(mbox_cmd));
	memset(mbox_sts, 0, sizeof(mbox_sts));
	mbox_cmd[0] = MBOX_CMD_GET_IPV6_NEIGHBOR_CACHE;
	mbox_cmd[1] = ioctl->Instance;  /* Primary/Secondary */
	mbox_cmd[2] = LSDW(ha->dma_buf.phys_addr);
	mbox_cmd[3] = MSDW(ha->dma_buf.phys_addr);
	mbox_cmd[4] = ioctl->Reserved1;
	mbox_cmd[5] = pncache->CacheBufferSize;

	status = qla4xxx_mailbox_command(ha, 6, 7, &mbox_cmd[0], &mbox_sts[0]);

	ioctl->VendorSpecificStatus[0] = mbox_sts[0];
	ioctl->VendorSpecificStatus[1] = mbox_sts[1];
	ioctl->VendorSpecificStatus[2] = mbox_sts[2];
	ioctl->VendorSpecificStatus[3] = mbox_sts[3];
	ioctl->VendorSpecificStatus[4] = mbox_sts[4];
	ioctl->VendorSpecificStatus[5] = mbox_sts[5];
	ioctl->VendorSpecificStatus[6] = mbox_sts[6];

	if (status == QLA_ERROR) {
		QL4PRINT(QLP2, printk("scsi%d: %s: "
					   "MBOX_CMD_GET_IPV6_NEIGHBOR_CACHE "
					   "FAILED w/ status %04X\n",
                                           ha->host_no, __func__, mbox_sts[0]));
		QL4PRINT(QLP2, printk("mbox[1] = 0x%04x\n"
					   "mbox[2] = 0x%04x\n"
					   "mbox[3] = 0x%04x\n"
					   "mbox[4] = 0x%04x\n"
					   "mbox[5] = 0x%04x\n"
					   "mbox[6] = 0x%04x\n",
                                           mbox_sts[1], mbox_sts[2],
                                           mbox_sts[3], mbox_sts[4],
					   mbox_sts[5], mbox_sts[6]));

		ioctl->Status = EXT_STATUS_MAILBOX;
		ioctl->ResponseLen = 0;
		goto exit_get_ncache;
	}

	if (pncache->CacheBufferSize) {
		/*
		 * Copy DMA data to local IOCTL buffer
		 */
		memcpy(&pncache->Buffer[0], ha->dma_buf.virt_addr,
		    MIN(pncache->CacheBufferSize,
			ioctl->ResponseLen - offsetof(EXT_NEIGHBOR_CACHE, Buffer)));
	}
	else {
		pncache->CacheBufferSize = mbox_sts[5];
	}

	/*
	 * Transfer Data from local IOCTL buffer to the user's buffer
	 */
	if (copy_to_user(Q64BIT_TO_PTR(ioctl->ResponseAdr),
	    pncache, ioctl->ResponseLen) != 0) {
		QL4PRINT(QLP2,
		    printk("scsi%d: %s: unable to copy data to user's "
		    "memory area\n", ha->host_no, __func__));

		status = (-EFAULT);
		ioctl->Status = EXT_STATUS_COPY_ERR;
		ioctl->ResponseLen = 0;
		goto exit_get_ncache;
	}

	ioctl->Status = EXT_STATUS_OK;

	QL4PRINT(QLP4, printk("scsi%d: %s: Succeeded\n",
	    ha->host_no, __func__));
	QL4PRINT(QLP10, printk("scsi%d: NEIGHBOR CACHE dump:\n",
	    ha->host_no));
	qla4xxx_dump_bytes(QLP10, pncache, ioctl->ResponseLen);

exit_get_ncache:
	qla4xxx_free_ioctl_scrap_mem(ha);
#if 0 && defined(__VMKERNEL_MODULE__)
	spin_unlock(&ha->dma_buf_spinlock);
#else
	up(&ha->dma_buf_sem);
#endif
	LEAVE(__func__);
	return(status);
}

/**************************************************************************
 * qla4extioctl_get_destination_cache
 *	This routine retrieves the destination cache or destination cache size
 *
 * Input:
 *	ha    = adapter structure pointer.
 *	ioctl = IOCTL structure pointer.
 *
 * Output:
 *	None
 *
 * Returns:
 *	QLA_SUCCESS = success
 *	QLA_ERROR   = error
 *
 * Context:
 *	Kernel context.
 **************************************************************************/
static int
qla4extioctl_get_destination_cache(scsi_qla_host_t *ha, EXT_IOCTL_ISCSI *ioctl)
{
	int	status = 0;
	EXT_DESTINATION_CACHE *pdcache = NULL;
	uint32_t mbox_cmd[MBOX_REG_COUNT];
	uint32_t mbox_sts[MBOX_REG_COUNT];

	ENTER(__func__);
#if 0 && defined(__VMKERNEL_MODULE__)
	spin_lock(&ha->dma_buf_spinlock);
#else
	down(&ha->dma_buf_sem);
#endif

	/*
	 * Allocate local EXT_DESTINATION_CACHE buffer
	 */
	 if (ioctl->ResponseLen >= sizeof(EXT_DESTINATION_CACHE) &&
	     qla4xxx_get_ioctl_scrap_mem(ha, (void **)&pdcache,
	     ioctl->ResponseLen)) {
		 /* not enough memory */
		 QL4PRINT(QLP2,
		     printk("scsi%d: %s: inst=%d scrap not big enough. "
		     "size requested=%ld.\n",
		     ha->host_no, __func__, ha->instance,
		     (ulong)ioctl->ResponseLen));

		 status = (-ENOMEM);
		 ioctl->Status = EXT_STATUS_NO_MEMORY;
		 ioctl->ResponseLen = 0;
		 goto exit_get_dcache;
	 }

	 /*
	  * Copy user's data to local buffer
	  */
	 if ((status = copy_from_user((uint8_t *)pdcache,
	     Q64BIT_TO_PTR(ioctl->RequestAdr), ioctl->RequestLen)) != 0) {
		 QL4PRINT(QLP2,
		     printk("scsi%d: %s: unable to copy data from user's "
		     "memory area\n", ha->host_no, __func__));

		 status = (-EFAULT);
		 ioctl->Status = EXT_STATUS_COPY_ERR;
		 ioctl->ResponseLen = 0;
		 goto exit_get_dcache;
	 }

	 if (pdcache->CacheBufferSize) {
		 /*
		  * Resize pre-allocated IOCTL DMA memory, if necesary
		  */
		if (!ha->dma_buf.virt_addr || !ha->dma_buf.phys_addr ||
		    (ha->dma_buf.buf_len < pdcache->CacheBufferSize)) {
			if (qla4xxx_resize_dma_buf(ha, &ha->dma_buf,
			    pdcache->CacheBufferSize) != QLA_SUCCESS) {
				QL4PRINT(QLP2,
				    printk("scsi%d: %s: unable to allocate memory "
				    "for dma buffer.\n",
				    ha->host_no, __func__));

				status = (-ENOMEM);
				ioctl->Status = EXT_STATUS_NO_MEMORY;
				ioctl->ResponseLen = 0;
				goto exit_get_dcache;
			}
		}
	 }

	memset(mbox_cmd, 0, sizeof(mbox_cmd));
	memset(mbox_sts, 0, sizeof(mbox_sts));
	mbox_cmd[0] = MBOX_CMD_GET_IPV6_DEST_CACHE;
	mbox_cmd[1] = ioctl->Instance;  /* Primary/Secondary */
	mbox_cmd[2] = LSDW(ha->dma_buf.phys_addr);
	mbox_cmd[3] = MSDW(ha->dma_buf.phys_addr);
	mbox_cmd[4] = ioctl->Reserved1;
	mbox_cmd[5] = pdcache->CacheBufferSize;

	status = qla4xxx_mailbox_command(ha, 6, 7, &mbox_cmd[0], &mbox_sts[0]);

	ioctl->VendorSpecificStatus[0] = mbox_sts[0];
	ioctl->VendorSpecificStatus[1] = mbox_sts[1];
	ioctl->VendorSpecificStatus[2] = mbox_sts[2];
	ioctl->VendorSpecificStatus[3] = mbox_sts[3];
	ioctl->VendorSpecificStatus[4] = mbox_sts[4];
	ioctl->VendorSpecificStatus[5] = mbox_sts[5];
	ioctl->VendorSpecificStatus[6] = mbox_sts[6];

	if (status == QLA_ERROR) {
		QL4PRINT(QLP2, printk("scsi%d: %s: MBOX_CMD_GET_IPV6_DEST_CACHE "
				           "FAILED w/ status %04X\n",
                                           ha->host_no, __func__, mbox_sts[0]));
		QL4PRINT(QLP2, printk("mbox[1] = 0x%04x\n"
					   "mbox[2] = 0x%04x\n"
					   "mbox[3] = 0x%04x\n"
					   "mbox[4] = 0x%04x\n"
					   "mbox[5] = 0x%04x\n"
					   "mbox[6] = 0x%04x\n",
                                           mbox_sts[1], mbox_sts[2],
                                           mbox_sts[3], mbox_sts[4],
					   mbox_sts[5], mbox_sts[6]));

		ioctl->Status = EXT_STATUS_MAILBOX;
		ioctl->ResponseLen = 0;
		goto exit_get_dcache;
	}

	if (pdcache->CacheBufferSize) {
		/*
		 * Copy DMA data to local IOCTL buffer
		 */
		memcpy(&pdcache->Buffer[0], ha->dma_buf.virt_addr,
		    MIN(pdcache->CacheBufferSize,
			ioctl->ResponseLen - offsetof(EXT_DESTINATION_CACHE, Buffer)));
	}
	else {
		pdcache->CacheBufferSize = mbox_sts[5];
	}

	/*
	 * Transfer Data from local IOCTL buffer to the user's buffer
	 */
	if (copy_to_user(Q64BIT_TO_PTR(ioctl->ResponseAdr),
	    pdcache, ioctl->ResponseLen) != 0) {
		QL4PRINT(QLP2,
		    printk("scsi%d: %s: unable to copy data to user's "
		    "memory area\n", ha->host_no, __func__));

		status = (-EFAULT);
		ioctl->Status = EXT_STATUS_COPY_ERR;
		ioctl->ResponseLen = 0;
		goto exit_get_dcache;
	}

	ioctl->Status = EXT_STATUS_OK;

	QL4PRINT(QLP4, printk("scsi%d: %s: Succeeded\n",
	    ha->host_no, __func__));
	QL4PRINT(QLP10, printk("scsi%d: DESTINATION CACHE dump:\n",
	    ha->host_no));
	qla4xxx_dump_bytes(QLP10, pdcache, ioctl->ResponseLen);

exit_get_dcache:
	qla4xxx_free_ioctl_scrap_mem(ha);
#if 0 && defined(__VMKERNEL_MODULE__)
	spin_unlock(&ha->dma_buf_spinlock);
#else
	up(&ha->dma_buf_sem);
#endif
	LEAVE(__func__);
	return(status);
}

/**************************************************************************
 * qla4extioctl_get_default_router_list
 *	This routine retrieves the IPv6 default router list of size of the
 *      list.
 *
 * Input:
 *	ha    = adapter structure pointer.
 *	ioctl = IOCTL structure pointer.
 *
 * Output:
 *	None
 *
 * Returns:
 *	QLA_SUCCESS = success
 *	QLA_ERROR   = error
 *
 * Context:
 *	Kernel context.
 **************************************************************************/
static int
qla4extioctl_get_default_router_list(scsi_qla_host_t *ha, EXT_IOCTL_ISCSI *ioctl)
{
	int	status = 0;
	EXT_ROUTER_LIST *pdrlist = NULL;
	uint32_t mbox_cmd[MBOX_REG_COUNT];
	uint32_t mbox_sts[MBOX_REG_COUNT];

	ENTER(__func__);

#if 0 && defined(__VMKERNEL_MODULE__)
	spin_lock(&ha->dma_buf_spinlock);
#else
	down(&ha->dma_buf_sem);
#endif
	/*
	 * Allocate local EXT_ROUTER_LIST buffer
	 */
	 if (ioctl->ResponseLen >= sizeof(EXT_ROUTER_LIST) &&
	     qla4xxx_get_ioctl_scrap_mem(ha, (void **)&pdrlist,
	     ioctl->ResponseLen)) {
		 /* not enough memory */
		 QL4PRINT(QLP2,
		     printk("scsi%d: %s: inst=%d scrap not big enough. "
		     "size requested=%ld.\n",
		     ha->host_no, __func__, ha->instance,
		     (ulong)ioctl->ResponseLen));

		 status = (-ENOMEM);
		 ioctl->Status = EXT_STATUS_NO_MEMORY;
		 ioctl->ResponseLen = 0;
		 goto exit_get_drlist;
	 }

	 /*
	  * Copy user's data to local buffer
	  */
	 if ((status = copy_from_user((uint8_t *)pdrlist,
	     Q64BIT_TO_PTR(ioctl->RequestAdr), ioctl->RequestLen)) != 0) {
		 QL4PRINT(QLP2,
		     printk("scsi%d: %s: unable to copy data from user's "
		     "memory area\n", ha->host_no, __func__));

		 status = (-EFAULT);
		 ioctl->Status = EXT_STATUS_COPY_ERR;
		 ioctl->ResponseLen = 0;
		 goto exit_get_drlist;
	 }

	 if (pdrlist->CacheBufferSize) {
		 /*
		  * Resize pre-allocated IOCTL DMA memory, if necesary
		  */
		if (!ha->dma_buf.virt_addr || !ha->dma_buf.phys_addr ||
		    (ha->dma_buf.buf_len < pdrlist->CacheBufferSize)) {
			if (qla4xxx_resize_dma_buf(ha, &ha->dma_buf,
			    pdrlist->CacheBufferSize) != QLA_SUCCESS) {
				QL4PRINT(QLP2,
				    printk("scsi%d: %s: unable to allocate memory "
				    "for dma buffer.\n",
				    ha->host_no, __func__));

				status = (-ENOMEM);
				ioctl->Status = EXT_STATUS_NO_MEMORY;
				ioctl->ResponseLen = 0;
				goto exit_get_drlist;
			}
		}
	 }

	memset(mbox_cmd, 0, sizeof(mbox_cmd));
	memset(mbox_sts, 0, sizeof(mbox_sts));
	mbox_cmd[0] = MBOX_CMD_GET_IPV6_DEF_ROUTER_LIST;
	mbox_cmd[1] = ioctl->Instance;  /* Primary/Secondary */
	mbox_cmd[2] = LSDW(ha->dma_buf.phys_addr);
	mbox_cmd[3] = MSDW(ha->dma_buf.phys_addr);
	mbox_cmd[4] = ioctl->Reserved1;
	mbox_cmd[5] = pdrlist->CacheBufferSize;

	status = qla4xxx_mailbox_command(ha, 6, 6, &mbox_cmd[0], &mbox_sts[0]);

	ioctl->VendorSpecificStatus[0] = mbox_sts[0];
	ioctl->VendorSpecificStatus[1] = mbox_sts[1];
	ioctl->VendorSpecificStatus[2] = mbox_sts[2];
	ioctl->VendorSpecificStatus[3] = mbox_sts[3];
	ioctl->VendorSpecificStatus[4] = mbox_sts[4];
	ioctl->VendorSpecificStatus[5] = mbox_sts[5];

	if (status == QLA_ERROR) {
		QL4PRINT(QLP2, printk("scsi%d: %s: MBOX_CMD_GET_IPV6_DEF_ROUTER_LIST "
				           "FAILED w/ status %04X\n",
                                           ha->host_no, __func__, mbox_sts[0]));
		QL4PRINT(QLP2, printk("mbox[1] = 0x%04x\n"
					   "mbox[2] = 0x%04x\n"
					   "mbox[3] = 0x%04x\n"
					   "mbox[4] = 0x%04x\n"
					   "mbox[5] = 0x%04x\n",
                                           mbox_sts[1], mbox_sts[2],
                                           mbox_sts[3], mbox_sts[4],
					   mbox_sts[5]));

		ioctl->Status = EXT_STATUS_MAILBOX;
		ioctl->ResponseLen = 0;
		goto exit_get_drlist;
	}

	if (pdrlist->CacheBufferSize) {
		/*
		 * Copy DMA data to local IOCTL buffer
		 */
		memcpy(&pdrlist->Buffer[0], ha->dma_buf.virt_addr,
		    MIN(pdrlist->CacheBufferSize,
			ioctl->ResponseLen - offsetof(EXT_ROUTER_LIST, Buffer)));
	}
	else {
		pdrlist->CacheBufferSize = mbox_sts[5];
	}

	/*
	 * Transfer Data from local IOCTL buffer to the user's buffer
	 */
	if (copy_to_user(Q64BIT_TO_PTR(ioctl->ResponseAdr),
	    pdrlist, ioctl->ResponseLen) != 0) {
		QL4PRINT(QLP2,
		    printk("scsi%d: %s: unable to copy data to user's "
		    "memory area\n", ha->host_no, __func__));

		status = (-EFAULT);
		ioctl->Status = EXT_STATUS_COPY_ERR;
		ioctl->ResponseLen = 0;
		goto exit_get_drlist;
	}

	ioctl->Status = EXT_STATUS_OK;

	QL4PRINT(QLP4, printk("scsi%d: %s: Succeeded\n",
	    ha->host_no, __func__));
	QL4PRINT(QLP10, printk("scsi%d: DEFAULT_ROUTER_LIST dump:\n",
	    ha->host_no));
	qla4xxx_dump_bytes(QLP10, ha->dma_buf.virt_addr, ioctl->ResponseLen);

exit_get_drlist:
	qla4xxx_free_ioctl_scrap_mem(ha);
#if 0 && defined(__VMKERNEL_MODULE__)
	spin_unlock(&ha->dma_buf_spinlock);
#else
	up(&ha->dma_buf_sem);
#endif
	LEAVE(__func__);
	return(status);
}

/**************************************************************************
 * qla4extioctl_get_local_prefix_list
 *	This routine retrieves the IPv6 local prefix list or the size of
 *	the list.
 *
 * Input:
 *	ha    = adapter structure pointer.
 *	ioctl = IOCTL structure pointer.
 *
 * Output:
 *	None
 *
 * Returns:
 *	QLA_SUCCESS = success
 *	QLA_ERROR   = error
 *
 * Context:
 *	Kernel context.
 **************************************************************************/
static int
qla4extioctl_get_local_prefix_list(scsi_qla_host_t *ha, EXT_IOCTL_ISCSI *ioctl)
{
	int	status = 0;
	EXT_PREFIX_LIST *plplist = NULL;
	uint32_t mbox_cmd[MBOX_REG_COUNT];
	uint32_t mbox_sts[MBOX_REG_COUNT];

	ENTER(__func__);

#if 0 && defined(__VMKERNEL_MODULE__)
	spin_lock(&ha->dma_buf_spinlock);
#else
	down(&ha->dma_buf_sem);
#endif
	/*
	 * Allocate local EXT_PREFIX_LIST buffer
	 */
	 if (ioctl->ResponseLen >= sizeof(EXT_PREFIX_LIST) &&
	     qla4xxx_get_ioctl_scrap_mem(ha, (void **)&plplist,
	     ioctl->ResponseLen)) {
		 /* not enough memory */
		 QL4PRINT(QLP2,
		     printk("scsi%d: %s: inst=%d scrap not big enough. "
		     "size requested=%ld.\n",
		     ha->host_no, __func__, ha->instance,
		     (ulong)ioctl->ResponseLen));

		 status = (-ENOMEM);
		 ioctl->Status = EXT_STATUS_NO_MEMORY;
		 ioctl->ResponseLen = 0;
		 goto exit_get_lplist;
	 }

	 /*
	  * Copy user's data to local buffer
	  */
	 if ((status = copy_from_user((uint8_t *)plplist,
	     Q64BIT_TO_PTR(ioctl->RequestAdr), ioctl->RequestLen)) != 0) {
		 QL4PRINT(QLP2,
		     printk("scsi%d: %s: unable to copy data from user's "
		     "memory area\n", ha->host_no, __func__));

		 status = (-EFAULT);
		 ioctl->Status = EXT_STATUS_COPY_ERR;
		 ioctl->ResponseLen = 0;
		 goto exit_get_lplist;
	 }

	 if (plplist->CacheBufferSize) {
		 /*
		  * Resize pre-allocated IOCTL DMA memory, if necesary
		  */
		if (!ha->dma_buf.virt_addr || !ha->dma_buf.phys_addr ||
		    (ha->dma_buf.buf_len < plplist->CacheBufferSize)) {
			if (qla4xxx_resize_dma_buf(ha,  &ha->dma_buf,
			    plplist->CacheBufferSize) != QLA_SUCCESS) {
				QL4PRINT(QLP2,
				    printk("scsi%d: %s: unable to allocate memory "
				    "for dma buffer.\n",
				    ha->host_no, __func__));

				status = (-ENOMEM);
				ioctl->Status = EXT_STATUS_NO_MEMORY;
				ioctl->ResponseLen = 0;
				goto exit_get_lplist;
			}
		}
	 }

	memset(mbox_cmd, 0, sizeof(mbox_cmd));
	memset(mbox_sts, 0, sizeof(mbox_sts));
	mbox_cmd[0] = MBOX_CMD_GET_IPV6_LCL_PREFIX_LIST;
	mbox_cmd[1] = ioctl->Instance;  /* Primary/Secondary */
	mbox_cmd[2] = LSDW(ha->dma_buf.phys_addr);
	mbox_cmd[3] = MSDW(ha->dma_buf.phys_addr);
	mbox_cmd[4] = ioctl->Reserved1;
	mbox_cmd[5] = plplist->CacheBufferSize;

	status = qla4xxx_mailbox_command(ha, 6, 6, &mbox_cmd[0], &mbox_sts[0]);

	ioctl->VendorSpecificStatus[0] = mbox_sts[0];
	ioctl->VendorSpecificStatus[1] = mbox_sts[1];
	ioctl->VendorSpecificStatus[2] = mbox_sts[2];
	ioctl->VendorSpecificStatus[3] = mbox_sts[3];
	ioctl->VendorSpecificStatus[4] = mbox_sts[4];
	ioctl->VendorSpecificStatus[5] = mbox_sts[5];

	if (status == QLA_ERROR) {
		QL4PRINT(QLP2, printk("scsi%d: %s: MBOX_CMD_GET_IPV6_LCL_PREFIX_LIST "
				           "FAILED w/ status %04X\n",
                                           ha->host_no, __func__, mbox_sts[0]));
		QL4PRINT(QLP2, printk("mbox[1] = 0x%04x\n"
					   "mbox[2] = 0x%04x\n"
					   "mbox[3] = 0x%04x\n"
					   "mbox[4] = 0x%04x\n"
					   "mbox[5] = 0x%04x\n",
                                           mbox_sts[1], mbox_sts[2],
                                           mbox_sts[3], mbox_sts[4],
					   mbox_sts[5]));

		ioctl->Status = EXT_STATUS_MAILBOX;
		ioctl->ResponseLen = 0;
		goto exit_get_lplist;
	}

	if (plplist->CacheBufferSize) {
		/*
		 * Copy DMA data to local IOCTL buffer
		 */
		memcpy(&plplist->Buffer[0], ha->dma_buf.virt_addr,
		    MIN(plplist->CacheBufferSize,
			ioctl->ResponseLen - offsetof(EXT_PREFIX_LIST, Buffer)));
	}
	else {
		plplist->CacheBufferSize = mbox_sts[5];
	}

	/*
	 * Transfer Data from local IOCTL buffer to the user's buffer
	 */
	if (copy_to_user(Q64BIT_TO_PTR(ioctl->ResponseAdr),
	    plplist, ioctl->ResponseLen) != 0) {
		QL4PRINT(QLP2,
		    printk("scsi%d: %s: unable to copy data to user's "
		    "memory area\n", ha->host_no, __func__));

		status = (-EFAULT);
		ioctl->Status = EXT_STATUS_COPY_ERR;
		ioctl->ResponseLen = 0;
		goto exit_get_lplist;
	}

	ioctl->Status = EXT_STATUS_OK;

	QL4PRINT(QLP4, printk("scsi%d: %s: Succeeded\n",
	    ha->host_no, __func__));
	QL4PRINT(QLP10, printk("scsi%d: LOCAL_PREFIX_LIST dump:\n",
	    ha->host_no));
	qla4xxx_dump_bytes(QLP10, ha->dma_buf.virt_addr, ioctl->ResponseLen);

exit_get_lplist:
	qla4xxx_free_ioctl_scrap_mem(ha);
	LEAVE(__func__);
#if 0 && defined(__VMKERNEL_MODULE__)
	spin_unlock(&ha->dma_buf_spinlock);
#else
	up(&ha->dma_buf_sem);
#endif
	return(status);
}

/**************************************************************************
 * qla4extioctl_get_statistics_iscsi_block
 *	This routine retrieves the HBA iSCSI statistical information.
 *
 * Input:
 *	ha    = adapter structure pointer.
 *	ioctl = IOCTL structure pointer.
 *
 * Output:
 *	None
 *
 * Returns:
 *	QLA_SUCCESS = success
 *	QLA_ERROR   = error
 *
 * Context:
 *	Kernel context.
 **************************************************************************/
static int
qla4extioctl_get_statistics_iscsi_block(scsi_qla_host_t *ha, EXT_IOCTL_ISCSI *ioctl)
{
	int		status = 0;
	uint32_t	mbox_cmd[MBOX_REG_COUNT];
	uint32_t	mbox_sts[MBOX_REG_COUNT];

	ENTER(__func__);
	QL4PRINT(QLP4, printk("scsi%d: %s: index [%d]\n",
			      ha->host_no, __func__, ioctl->Instance));

#if 0 && defined(__VMKERNEL_MODULE__)
	spin_lock(&ha->dma_buf_spinlock);
#else
	down(&ha->dma_buf_sem);
#endif

	if (!ioctl->ResponseAdr || !ioctl->ResponseLen) {
		QL4PRINT(QLP2, printk("scsi%d: %s: invalid parameter\n",
		    ha->host_no, __func__));

		status = (-EINVAL);
		ioctl->Status = EXT_STATUS_INVALID_PARAM;
		ioctl->ResponseLen = 0;
		goto exit_get_stats_iscsi_block;
	}

	if (ioctl->ResponseLen < sizeof(EXT_HBA_PORT_STAT_ISCSI_BLOCK)) {
		QL4PRINT(QLP2, printk("scsi%d: %s: RespLen too small (0x%x),  "
		    "need (0x%x).\n",
		    ha->host_no, __func__, ioctl->ResponseLen,
		    (unsigned int) sizeof(EXT_HBA_PORT_STAT_ISCSI_BLOCK)));

		status = (-ENOMEM);
		ioctl->Status = EXT_STATUS_BUFFER_TOO_SMALL;
		ioctl->ResponseLen = 0;
		goto exit_get_stats_iscsi_block;
	}

	if ((ha->dma_buf.buf_len < sizeof(EXT_HBA_PORT_STAT_ISCSI_BLOCK)) &&
	    (qla4xxx_resize_dma_buf(ha, &ha->dma_buf,
		sizeof(EXT_HBA_PORT_STAT_ISCSI_BLOCK)) != QLA_SUCCESS)) {
		QL4PRINT(QLP2, printk("scsi%d: %s: unable to allocate memory "
		    "for dma buffer.\n",
		    ha->host_no, __func__));

		status = (-ENOMEM);
		ioctl->Status = EXT_STATUS_NO_MEMORY;
		ioctl->ResponseLen = 0;
		goto exit_get_stats_iscsi_block;
	}

	/*
	 * Make the mailbox call
	 */
	memset(mbox_cmd, 0, sizeof(mbox_cmd));
	memset(mbox_sts, 0, sizeof(mbox_sts));
	mbox_cmd[0] = MBOX_CMD_GET_MANAGEMENT_DATA;
	mbox_cmd[1] = ioctl->Instance;
	mbox_cmd[2] = LSDW(ha->dma_buf.phys_addr);
	mbox_cmd[3] = MSDW(ha->dma_buf.phys_addr);

	if (qla4xxx_mailbox_command(ha, 4, 5, &mbox_cmd[0], &mbox_sts[0]) !=
	    QLA_SUCCESS) {
		QL4PRINT(QLP2,
		    printk("scsi%d: %s: get mngmt data for index [%d] failed "
		    "w/ mailbox ststus 0x%x\n",
		    ha->host_no, __func__, ioctl->Instance, mbox_sts[0]));

		ioctl->Status = EXT_STATUS_MAILBOX;
		ioctl->VendorSpecificStatus[0] = mbox_sts[0];
		ioctl->ResponseLen = 0;
		goto exit_get_stats_iscsi_block;
	}

	/*
	 * Copy the data from the dma buffer to the user's data space
	 */
	if (copy_to_user(Q64BIT_TO_PTR(ioctl->ResponseAdr),
	    ha->dma_buf.virt_addr, ioctl->ResponseLen) != 0) {
		QL4PRINT(QLP2, printk("scsi%d: %s: memory area too small\n",
		    ha->host_no, __func__));

		status = (-EFAULT);
		ioctl->Status = EXT_STATUS_COPY_ERR;
		ioctl->ResponseLen = 0;
		goto exit_get_stats_iscsi_block;
	}

	ioctl->Status = EXT_STATUS_OK;
	ioctl->ResponseLen = mbox_sts[4];

exit_get_stats_iscsi_block:
	qla4xxx_free_ioctl_scrap_mem(ha);
#if 0 && defined(__VMKERNEL_MODULE__)
	spin_unlock(&ha->dma_buf_spinlock);
#else
	up(&ha->dma_buf_sem);
#endif
	LEAVE(__func__);
	return(status);
}

/**************************************************************************
 * qla4extioctl_get_data
 *	This routine calls get data IOCTLs based on the IOCTL Sub Code.
 *
 * Input:
 *	ha    = adapter structure pointer.
 *	ioctl = IOCTL structure pointer.
 *
 * Output:
 *	None
 *
 * Returns:
 *	QLA_SUCCESS = success
 *	QLA_ERROR   = error
 *    	-EINVAL     = if the command is invalid
 *
 * Context:
 *	Kernel context.
 **************************************************************************/
static int
qla4extioctl_get_data(scsi_qla_host_t *ha, EXT_IOCTL_ISCSI *ioctl)
{
	switch (ioctl->SubCode) {
	case EXT_SC_GET_STATISTICS_GEN:
		return(qla4extioctl_get_statistics_gen(ha, ioctl));

	case EXT_SC_GET_STATISTICS_ISCSI:
		return(qla4extioctl_get_statistics_iscsi(ha, ioctl));

	case EXT_SC_GET_DEVICE_ENTRY_ISCSI:
	case EXT_SC_GET_DEVICE_ENTRY_DEFAULTS_ISCSI:
		return(qla4extioctl_get_device_entry_iscsi(ha, ioctl));

	case EXT_SC_GET_INIT_FW_ISCSI:
	case EXT_SC_GET_INIT_FW_DEFAULTS_ISCSI:
		return(qla4extioctl_get_init_fw_iscsi(ha, ioctl));

#ifndef QLA4000
	case EXT_SC_GET_ISNS_SERVER:
		return(qla4extioctl_get_isns_server(ha, ioctl));

	case EXT_SC_GET_ISNS_DISCOVERED_TARGETS:
		return(qla4extioctl_get_isns_disc_targets(ha, ioctl));

	case EXT_SC_GET_ACB:
		return(qla4extioctl_get_acb(ha, ioctl));

	case EXT_SC_GET_NEIGHBOR_CACHE:
		return(qla4extioctl_get_neighbor_cache(ha, ioctl));

	case EXT_SC_GET_DESTINATION_CACHE:
		return(qla4extioctl_get_destination_cache(ha, ioctl));

	case EXT_SC_GET_DEFAULT_ROUTER_LIST:
		return(qla4extioctl_get_default_router_list(ha, ioctl));

	case EXT_SC_GET_LOCAL_PREFIX_LIST:
		return(qla4extioctl_get_local_prefix_list(ha, ioctl));

	case EXT_SC_GET_STATISTICS_ISCSI_BLOCK:
		return(qla4extioctl_get_statistics_iscsi_block(ha, ioctl));
#endif

	default:
		QL4PRINT(QLP2,
			 printk("scsi%d: %s: unsupported external get "
				"data sub-command code (%X)\n",
				ha->host_no, __func__, ioctl->SubCode));

		ioctl->Status = EXT_STATUS_UNSUPPORTED_SUBCODE;
		return(-EINVAL);
	}
}

/**************************************************************************
 * qla4extioctl_rst_statistics_gen
 *	This routine clears the HBA general statistical information.
 *
 * Input:
 *	ha    = adapter structure pointer.
 *	ioctl = IOCTL structure pointer.
 *
 * Output:
 *	None
 *
 * Returns:
 *	QLA_SUCCESS = success
 *	QLA_ERROR   = error
 *
 * Context:
 *	Kernel context.
 **************************************************************************/
static int
qla4extioctl_rst_statistics_gen(scsi_qla_host_t *ha, EXT_IOCTL_ISCSI *ioctl)
{
	ENTER(__func__);

	/*
	 * Reset the general statistics fields
	 */
	ha->adapter_error_count = 0;
	ha->device_error_count = 0;
	ha->total_io_count = 0;
	ha->total_mbytes_xferred = 0;
	ha->isr_count = 0;
	ha->link_failure_count = 0;
	ha->invalid_crc_count = 0;

	ioctl->Status = EXT_STATUS_OK;

	LEAVE(__func__);
	return(QLA_SUCCESS);
}

/**************************************************************************
 * qla4extioctl_rst_statistics_iscsi
 *	This routine clears the HBA iSCSI statistical information.
 *
 * Input:
 *	ha    = adapter structure pointer.
 *	ioctl = IOCTL structure pointer.
 *
 * Output:
 *	None
 *
 * Returns:
 *	QLA_SUCCESS = success
 *	QLA_ERROR   = error
 *
 * Context:
 *	Kernel context.
 **************************************************************************/
static int
qla4extioctl_rst_statistics_iscsi(scsi_qla_host_t *ha, EXT_IOCTL_ISCSI *ioctl)
{
	uint32_t mbox_cmd[MBOX_REG_COUNT];
	uint32_t mbox_sts[MBOX_REG_COUNT];

	ENTER(__func__);
	QL4PRINT(QLP4, printk("scsi%d: %s: index [%d]\n",
			      ha->host_no, __func__, ioctl->Instance));

	/*
	 * Make the mailbox call
	 */
	memset(mbox_cmd, 0, sizeof(mbox_cmd));
	memset(mbox_sts, 0, sizeof(mbox_sts));
	mbox_cmd[0] = MBOX_CMD_GET_MANAGEMENT_DATA;
	mbox_cmd[1] = ioctl->Instance;
	mbox_cmd[2] = 0;
	mbox_cmd[3] = 0;

	if (qla4xxx_mailbox_command(ha, 4, 1, &mbox_cmd[0], &mbox_sts[0])
	    != QLA_SUCCESS) {
		QL4PRINT(QLP2,
		    printk("scsi%d: %s: get mngmt data for index [%d] failed! "
		    "w/ mailbox ststus 0x%x\n",
		    ha->host_no, __func__, ioctl->Instance, mbox_sts[0]));

		ioctl->Status = EXT_STATUS_MAILBOX;
		ioctl->VendorSpecificStatus[0] = mbox_sts[0];
		ioctl->ResponseLen = 0;
		LEAVE(__func__);
		return(0);
	}

	ioctl->Status = EXT_STATUS_OK;

	LEAVE(__func__);
	return(0);
}

/**************************************************************************
 * qla4extioctl_set_device_entry_iscsi
 *	This routine configures a device with specific database entry data.
 *
 * Input:
 *	ha    = adapter structure pointer.
 *	ioctl = IOCTL structure pointer.
 *
 * Output:
 *	None
 *
 * Returns:
 *	QLA_SUCCESS = success
 *	QLA_ERROR   = error
 *
 * Context:
 *	Kernel context.
 **************************************************************************/
static int
qla4extioctl_set_device_entry_iscsi(scsi_qla_host_t *ha, EXT_IOCTL_ISCSI *ioctl)
{
	int		status = 0;
	DEV_DB_ENTRY	*pfw_ddb_entry;
	dma_addr_t	fw_ddb_entry_dma;
	EXT_DEVICE_ENTRY_ISCSI *pdev_entry;


	ENTER(__func__);
	QL4PRINT(QLP4, printk("scsi%d: %s: index [%d]\n",
			      ha->host_no, __func__, ioctl->Instance));

#if 0 && defined(__VMKERNEL_MODULE__)
	spin_lock(&ha->dma_buf_spinlock);
#else
	down(&ha->dma_buf_sem);
#endif

	if (qla4xxx_get_ioctl_scrap_mem(ha, (void **)&pdev_entry,
	    sizeof(EXT_DEVICE_ENTRY_ISCSI))) {
		/* not enough memory */
		QL4PRINT(QLP2,
		    printk("scsi%d: %s: inst=%d scrap not big enough. "
		    "size requested=%ld.\n",
		    ha->host_no, __func__, ha->instance,
		    (ulong)sizeof(EXT_DEVICE_ENTRY_ISCSI)));

		status = (-ENOMEM);
		ioctl->Status = EXT_STATUS_NO_MEMORY;
		ioctl->ResponseLen = 0;
		goto exit_set_dev_entry;
	}

	if (!ha->dma_buf.virt_addr || !ha->dma_buf.phys_addr ||
	    !ioctl->RequestAdr) {
		QL4PRINT(QLP2,
			 printk("scsi%d: %s: memory allocation problem\n",
				ha->host_no, __func__));

		status = (-ENOMEM);
		ioctl->Status = EXT_STATUS_NO_MEMORY;
		ioctl->ResponseLen = 0;
		goto exit_set_dev_entry;
	}

	if (ioctl->RequestLen < sizeof(DEV_DB_ENTRY) ||
	    ha->dma_buf.buf_len < sizeof(DEV_DB_ENTRY)) {
		QL4PRINT(QLP2, printk("scsi%d: %s: memory area too small\n",
				      ha->host_no, __func__));

		status = (-ENOMEM);
		ioctl->Status = EXT_STATUS_BUFFER_TOO_SMALL;
		ioctl->ResponseLen = 0;
		goto exit_set_dev_entry;
	}

	/*
	 * Copy the IOCTL EXT_DEVICE_ENTRY_ISCSI buffer from the user's
	 * data space
	 */
	if (copy_from_user((uint8_t *)pdev_entry,
	    Q64BIT_TO_PTR(ioctl->RequestAdr), ioctl->RequestLen) != 0) {
		QL4PRINT(QLP2,
			 printk("scsi%d: %s: unable to copy data from user's "
				"memory area\n", ha->host_no, __func__));

		status = (-EFAULT);
		ioctl->Status = EXT_STATUS_COPY_ERR;
		ioctl->ResponseLen = 0;
		goto exit_set_dev_entry;
	}

	/*
	 * Transfer data from IOCTL's EXT_DEVICE_ENTRY_ISCSI buffer to
	 * Fw's DEV_DB_ENTRY buffer
	 */
	pfw_ddb_entry = ha->dma_buf.virt_addr;
	fw_ddb_entry_dma = ha->dma_buf.phys_addr;
	memset(pfw_ddb_entry, 0, sizeof(DEV_DB_ENTRY));

	pfw_ddb_entry->options          = pdev_entry->Options;
	pfw_ddb_entry->TSID             = cpu_to_le16(pdev_entry->TargetSessID);
	pfw_ddb_entry->exeCount         = cpu_to_le16(pdev_entry->ExeCount);
	pfw_ddb_entry->ddbLink          = cpu_to_le16(pdev_entry->DDBLink);
	memcpy(pfw_ddb_entry->ISID, pdev_entry->InitiatorSessID,
	    sizeof(pdev_entry->InitiatorSessID));
	pfw_ddb_entry->exeThrottle =
	    cpu_to_le16(pdev_entry->DeviceInfo.ExeThrottle);
	pfw_ddb_entry->iSCSIMaxSndDataSegLen =
	    cpu_to_le16(pdev_entry->DeviceInfo.InitMarkerlessInt);
	pfw_ddb_entry->retryCount       = pdev_entry->DeviceInfo.RetryCount;
	pfw_ddb_entry->retryDelay       = pdev_entry->DeviceInfo.RetryDelay;
	pfw_ddb_entry->iSCSIOptions =
	    cpu_to_le16(pdev_entry->DeviceInfo.iSCSIOptions);
	pfw_ddb_entry->TCPOptions =
	    cpu_to_le16(pdev_entry->DeviceInfo.TCPOptions);
	pfw_ddb_entry->IPOptions =
	    cpu_to_le16(pdev_entry->DeviceInfo.IPOptions);
	pfw_ddb_entry->maxPDUSize =
	    cpu_to_le16(pdev_entry->DeviceInfo.MaxPDUSize);
	pfw_ddb_entry->firstBurstSize =
	    cpu_to_le16(pdev_entry->DeviceInfo.FirstBurstSize);
	pfw_ddb_entry->DefaultTime2Wait =
	    cpu_to_le16(pdev_entry->DeviceInfo.LogoutMinTime);
	pfw_ddb_entry->DefaultTime2Retain =
	    cpu_to_le16(pdev_entry->DeviceInfo.LogoutMaxTime);
	pfw_ddb_entry->maxOutstndngR2T =
	    cpu_to_le16(pdev_entry->DeviceInfo.MaxOutstandingR2T);
	pfw_ddb_entry->keepAliveTimeout =
	    cpu_to_le16(pdev_entry->DeviceInfo.KeepAliveTimeout);
	pfw_ddb_entry->RemoteTCPPortNumber =
	    cpu_to_le16(pdev_entry->DeviceInfo.PortNumber);
	pfw_ddb_entry->maxBurstSize =
	    cpu_to_le16(pdev_entry->DeviceInfo.MaxBurstSize);
	pfw_ddb_entry->taskMngmntTimeout =
	    cpu_to_le16(pdev_entry->DeviceInfo.TaskMgmtTimeout);
	memcpy(pfw_ddb_entry->targetAddr,
	    pdev_entry->DeviceInfo.TargetAddr,
	    sizeof(pdev_entry->DeviceInfo.TargetAddr));
	memcpy(pfw_ddb_entry->RemoteIPAddr,
	    pdev_entry->EntryInfo.IPAddr.IPAddress,
	    sizeof(pdev_entry->EntryInfo.IPAddr.IPAddress));
	memcpy(pfw_ddb_entry->iscsiName,
	    pdev_entry->EntryInfo.iSCSIName,
	    sizeof(pdev_entry->EntryInfo.iSCSIName));
	memcpy(pfw_ddb_entry->iSCSIAlias,
	    pdev_entry->EntryInfo.Alias,
	    sizeof(pdev_entry->EntryInfo.Alias));

	/*
	 * Make the IOCTL call
	 */
	if (qla4xxx_set_ddb_entry(ha, ioctl->Instance, pfw_ddb_entry,
				  fw_ddb_entry_dma) != QLA_SUCCESS) {
		QL4PRINT(QLP2, printk("scsi%d: %s: SET DDB Entry failed\n",
				      ha->host_no, __func__));

		status = (-EFAULT);
		ioctl->Status = EXT_STATUS_ERR;
		ioctl->ResponseLen = 0;
		goto exit_set_dev_entry;
	}

	ioctl->Status = EXT_STATUS_OK;

exit_set_dev_entry:
	qla4xxx_free_ioctl_scrap_mem(ha);
#if 0 && defined(__VMKERNEL_MODULE__)
	spin_unlock(&ha->dma_buf_spinlock);
#else
	up(&ha->dma_buf_sem);
#endif
	LEAVE(__func__);
	return(status);
}

/**************************************************************************
 * qla4extioctl_set_init_fw_iscsi
 *	This routine configures a device with specific data entry data.
 *
 * Input:
 *	ha    = adapter structure pointer.
 *	ioctl = IOCTL structure pointer.
 *
 * Output:
 *	None
 *
 * Returns:
 *	QLA_SUCCESS = success
 *	QLA_ERROR   = error
 *
 * Context:
 *	Kernel context.
 **************************************************************************/
static int
qla4extioctl_set_init_fw_iscsi(scsi_qla_host_t *ha, EXT_IOCTL_ISCSI *ioctl)
{
	int			status = 0;
	EXT_INIT_FW_ISCSI	*pinit_fw = NULL;
	ADDRESS_CTRL_BLK	*pinit_fw_cb;
	uint32_t		mbox_cmd[MBOX_REG_COUNT];
	uint32_t		mbox_sts[MBOX_REG_COUNT];


	ENTER(__func__);
#if 0 && defined(__VMKERNEL_MODULE__)
	spin_lock(&ha->dma_buf_spinlock);
#else
	down(&ha->dma_buf_sem);
#endif

	if (qla4xxx_get_ioctl_scrap_mem(ha, (void **)&pinit_fw,
	    sizeof(EXT_INIT_FW_ISCSI))) {
		/* not enough memory */
		QL4PRINT(QLP2,
		    printk("scsi%d: %s: inst=%d scrap not big enough. "
		    "size requested=%ld.\n",
		    ha->host_no, __func__, ha->instance,
		    (ulong)sizeof(EXT_INIT_FW_ISCSI)));

		status = (-ENOMEM);
		ioctl->Status = EXT_STATUS_NO_MEMORY;
		ioctl->ResponseLen = 0;
		goto exit_set_init_fw;
	}

	if (!ha->dma_buf.virt_addr || !ha->dma_buf.phys_addr ||
	    (ha->dma_buf.buf_len < sizeof(INIT_FW_CTRL_BLK)) ||
	    (ioctl->RequestLen < sizeof(EXT_INIT_FW_ISCSI))) {
		QL4PRINT(QLP2,
		    printk("scsi%d: %s: requst buffer too small\n",
		    ha->host_no, __func__));

		status = (-ENOMEM);
		ioctl->Status = EXT_STATUS_BUFFER_TOO_SMALL;
		ioctl->ResponseLen = 0;
		goto exit_set_init_fw;
	}

	/*
	 * Copy the data from the user's buffer
	 */
	if (copy_from_user((uint8_t *)pinit_fw,
	    Q64BIT_TO_PTR(ioctl->RequestAdr), sizeof(EXT_INIT_FW_ISCSI)) != 0) {
		QL4PRINT(QLP2,
		    printk("scsi%d: %s: unable to copy data to user's "
		    "memory area\n", ha->host_no, __func__));

		status = (-EFAULT);
		ioctl->Status = EXT_STATUS_COPY_ERR;
		ioctl->ResponseLen = 0;
		goto exit_set_init_fw;
	}

	/*
	 * First get Initialize Firmware Control Block, so as not to
	 * destroy unaffected data
	 */
	if (qla4xxx_get_ifcb(ha, &mbox_cmd[0], &mbox_sts[0], ha->dma_buf.phys_addr)
	    == QLA_ERROR) {
		QL4PRINT(QLP2, printk("scsi%d: %s: command failed \n",
		    ha->host_no, __func__));

		ioctl->Status = EXT_STATUS_MAILBOX;
		ioctl->VendorSpecificStatus[0] = mbox_sts[0];
		ioctl->ResponseLen = 0;
		goto exit_set_init_fw;
	}

	/*
	 * Transfer Data from Local buffer to DMA buffer
	 */
	pinit_fw_cb = (ADDRESS_CTRL_BLK *)ha->dma_buf.virt_addr;

	pinit_fw_cb->Version         = pinit_fw->Version;
	pinit_fw_cb->FwOptions       = cpu_to_le16(pinit_fw->FWOptions);
	pinit_fw_cb->AddFwOptions    = cpu_to_le16(pinit_fw->AddFWOptions);
	memcpy(pinit_fw_cb->IPAddr, pinit_fw->IPAddr.IPAddress,
	    MIN(sizeof(pinit_fw_cb->IPAddr),
	    sizeof(pinit_fw->IPAddr.IPAddress)));
	memcpy(pinit_fw_cb->SubnetMask, pinit_fw->SubnetMask.IPAddress,
	    MIN(sizeof(pinit_fw_cb->SubnetMask),
	    sizeof(pinit_fw->SubnetMask.IPAddress)));
	memcpy(pinit_fw_cb->GatewayIPAddr, pinit_fw->Gateway.IPAddress,
	    MIN(sizeof(pinit_fw_cb->GatewayIPAddr),
	    sizeof(pinit_fw->Gateway.IPAddress)));
	memcpy(pinit_fw_cb->PriDNSIPAddr, pinit_fw->DNSConfig.IPAddr.IPAddress,
	    MIN(sizeof(pinit_fw_cb->PriDNSIPAddr),
	    sizeof(pinit_fw->DNSConfig.IPAddr.IPAddress)));
	memcpy(pinit_fw_cb->iSCSIAlias, pinit_fw->Alias,
	    MIN(sizeof(pinit_fw_cb->iSCSIAlias),
	    sizeof(pinit_fw->Alias)));
	memcpy(pinit_fw_cb->iSCSINameString, pinit_fw->iSCSIName,
	    MIN(sizeof(pinit_fw_cb->iSCSINameString),
	    sizeof(pinit_fw->iSCSIName)));

	pinit_fw_cb->ExecThrottle =
	    cpu_to_le16(pinit_fw->DeviceInfo.ExeThrottle);
	pinit_fw_cb->ZIOCount         = pinit_fw->DeviceInfo.RetryCount;
	pinit_fw_cb->iSCSIOptions =
	    cpu_to_le16(pinit_fw->DeviceInfo.iSCSIOptions);
	pinit_fw_cb->TCPOptions = cpu_to_le16(pinit_fw->DeviceInfo.TCPOptions);
	pinit_fw_cb->IPOptions = cpu_to_le16(pinit_fw->DeviceInfo.IPOptions);
	pinit_fw_cb->MaxPDUSize = cpu_to_le16(pinit_fw->DeviceInfo.MaxPDUSize);
	pinit_fw_cb->FirstBurstSize =
	    cpu_to_le16(pinit_fw->DeviceInfo.FirstBurstSize);
	pinit_fw_cb->DefaultTime2Wait =
	    cpu_to_le16(pinit_fw->DeviceInfo.LogoutMinTime);
	pinit_fw_cb->DefaultTime2Retain =
	    cpu_to_le16(pinit_fw->DeviceInfo.LogoutMaxTime);
	pinit_fw_cb->MaxOutStndngR2T =
	    cpu_to_le16(pinit_fw->DeviceInfo.MaxOutstandingR2T);
	pinit_fw_cb->KeepAliveTimeout =
	    cpu_to_le16(pinit_fw->DeviceInfo.KeepAliveTimeout);
	pinit_fw_cb->PortNumber =
	    cpu_to_le16(pinit_fw->DeviceInfo.PortNumber);
	pinit_fw_cb->MaxBurstSize =
	    cpu_to_le16(pinit_fw->DeviceInfo.MaxBurstSize);

	/*
	 * Send mailbox command
	 */
	if (qla4xxx_set_ifcb(ha, &mbox_cmd[0], &mbox_sts[0], ha->dma_buf.phys_addr)
	    == QLA_ERROR) {
		QL4PRINT(QLP2, printk("scsi%d: %s: command failed \n",
				      ha->host_no, __func__));

		ioctl->Status = EXT_STATUS_MAILBOX;
		ioctl->VendorSpecificStatus[0] = mbox_sts[0];
		ioctl->ResponseLen = 0;
		goto exit_set_init_fw;
	}

	ioctl->Status = EXT_STATUS_OK;

	QL4PRINT(QLP10,
	    printk("scsi%d: EXT_INIT_FW_ISCSI structure:\n", ha->host_no));
	qla4xxx_dump_bytes(QLP10, pinit_fw, sizeof(EXT_INIT_FW_ISCSI));

exit_set_init_fw:
	qla4xxx_free_ioctl_scrap_mem(ha);
#if 0 && defined(__VMKERNEL_MODULE__)
	spin_unlock(&ha->dma_buf_spinlock);
#else
	up(&ha->dma_buf_sem);
#endif
	LEAVE(__func__);
	return(status);
}

#ifndef QLA4000
/**************************************************************************
 * qla4extioctl_set_isns_server
 *	This routine retrieves the targets discovered via iSNS.
 *
 * Input:
 *	ha    = adapter structure pointer.
 *	ioctl = IOCTL structure pointer.
 *
 * Output:
 *	None
 *
 * Returns:
 *	QLA_SUCCESS = success
 *	QLA_ERROR   = error
 *
 * Context:
 *	Kernel context.
 **************************************************************************/
static int
qla4extioctl_set_isns_server(scsi_qla_host_t *ha, EXT_IOCTL_ISCSI *ioctl)
{
	int			status = 0;
	uint32_t		mbox_cmd[MBOX_REG_COUNT];
	uint32_t		mbox_sts[MBOX_REG_COUNT];
	EXT_ISNS_SERVER		*pisns_server = NULL;
	FLASH_INIT_FW_CTRL_BLK *pflash_init_fw_cb = NULL;
	uint16_t		tcp_options;
	uint16_t        	ipv6_tcp_options = 0;
	uint16_t		port_number;
	static uint8_t    	ip_addr_str[40];
	uint8_t			isns_ip_addr_is_equal;
	uint8_t			isns_ip_addr_is_zero;
	uint8_t			tcp_options_changed = 0;


	ENTER(__func__);
#if 0 && defined(__VMKERNEL_MODULE__)
	spin_lock(&ha->dma_buf_spinlock);
#else
	down(&ha->dma_buf_sem);
#endif

	if (qla4xxx_get_ioctl_scrap_mem(ha, (void **)&pisns_server,
	    sizeof(EXT_ISNS_SERVER))) {
		/* not enough memory */
		QL4PRINT(QLP2,
		    printk("scsi%d: %s: inst=%d scrap not big enough. "
		    "size requested=%ld.\n",
		    ha->host_no, __func__, ha->instance,
		    (ulong)sizeof(EXT_ISNS_SERVER)));

		status = (-ENOMEM);
		ioctl->Status = EXT_STATUS_NO_MEMORY;
		ioctl->ResponseLen = 0;
		goto exit_set_isns_svr;
	}

	if (ioctl->RequestLen < sizeof(*pisns_server)) {
		QL4PRINT(QLP2,
		    printk("scsi%d: %s: requst buffer too small (%d/%xh)\n",
		    ha->host_no, __func__, ioctl->RequestLen,
		    ioctl->RequestLen));

		status = (-ENOMEM);
		ioctl->Status = EXT_STATUS_BUFFER_TOO_SMALL;
		ioctl->ResponseLen = 0;
		goto exit_set_isns_svr;
	}

	if (!ha->dma_buf.virt_addr || !ha->dma_buf.phys_addr ||
	    (ha->dma_buf.buf_len < sizeof(*pflash_init_fw_cb))) {
		if (qla4xxx_resize_dma_buf(ha, &ha->dma_buf,
		    sizeof(DEV_DB_ENTRY)) != QLA_SUCCESS) {
			QL4PRINT(QLP2,
			    printk("scsi%d: %s: unable to allocate memory "
			    "for dma buffer.\n",
			    ha->host_no, __func__));

			status = (-ENOMEM);
			ioctl->Status = EXT_STATUS_NO_MEMORY;
			ioctl->ResponseLen = 0;
			goto exit_set_isns_svr;
		}
	}

	/*
	 * Copy iSNS Server info from the user's buffer
	 *---------------------------------------------*/
	if (copy_from_user((uint8_t *)pisns_server,
	    Q64BIT_TO_PTR(ioctl->RequestAdr), sizeof(EXT_ISNS_SERVER)) != 0) {
		QL4PRINT(QLP2,
		    printk("scsi%d: %s: unable to copy data to user's "
		    "memory area\n", ha->host_no, __func__));

		status = (-EFAULT);
		ioctl->Status = EXT_STATUS_COPY_ERR;
		ioctl->ResponseLen = 0;
		goto exit_set_isns_svr;
	}

	/* If IPv6 is not supported and an IPv6 address is specified -- ERROR */
	if (!IS_IPv6_ENABLED(ha) &&
		pisns_server->IPAddr.Type == EXT_DEF_TYPE_ISCSI_IPV6) {
		QL4PRINT(QLP2|QLP4,
			printk("scsi%d: %s: iSNS IPv6 address specified, "
			"but IPv6 not supported\n", ha->host_no, __func__));

		ioctl->Status = EXT_STATUS_INVALID_PARAM;
		ioctl->ResponseLen = 0;
		goto exit_set_isns_svr;
	}

	QL4PRINT(QLP10,
	    printk("scsi%d: EXT_ISNS_SERVER structure:\n", ha->host_no));
	qla4xxx_dump_bytes(QLP10, pisns_server, sizeof(EXT_ISNS_SERVER));

	/*
	 * First get Flash Initialize Firmware Control Block, so as not to
	 * destroy unaffected data
	 *----------------------------------------------------------------*/
	memset(mbox_cmd, 0, sizeof(mbox_cmd));
	memset(mbox_sts, 0, sizeof(mbox_sts));
	mbox_cmd[0] = MBOX_CMD_READ_FLASH;
	mbox_cmd[1] = LSDW(ha->dma_buf.phys_addr);
	mbox_cmd[2] = MSDW(ha->dma_buf.phys_addr);
	mbox_cmd[3] = INT_ISCSI_INITFW_FLASH_OFFSET;
	mbox_cmd[4] = sizeof(FLASH_INIT_FW_CTRL_BLK);

	if (qla4xxx_mailbox_command(ha, 5, 2, &mbox_cmd[0], &mbox_sts[0]) ==
	    QLA_ERROR) {
		QL4PRINT(QLP2,
		    printk("scsi%d: %s: READ_FLASH command failed \n",
		    ha->host_no, __func__));

		ioctl->Status = EXT_STATUS_MAILBOX;
		ioctl->VendorSpecificStatus[0] = mbox_sts[0];
		ioctl->VendorSpecificStatus[1] = mbox_sts[1];
		ioctl->ResponseLen = 0;
		goto exit_set_isns_svr;
	}

	QL4PRINT(QLP4, printk("scsi%d: %s: READ_FLASH command successful \n",
	    ha->host_no, __func__));

	/*
	 * Copy iSNS Server info to the flash_init_fw_cb
	 *----------------------------------------------*/
	pflash_init_fw_cb = (FLASH_INIT_FW_CTRL_BLK *)ha->dma_buf.virt_addr;

	// convert a couple of variables used for comparisons
	if (pisns_server->IPAddr.Type == EXT_DEF_TYPE_ISCSI_IPV6)
		ipv6_tcp_options = le16_to_cpu(pflash_init_fw_cb->init_fw_cb.IPv6TCPOptions);
	tcp_options = le16_to_cpu(pflash_init_fw_cb->init_fw_cb.TCPOptions);
	port_number = le16_to_cpu(pflash_init_fw_cb->init_fw_cb.iSNSServerPortNumber);

	if (pisns_server->PerformiSNSDiscovery) {
		memcpy(pflash_init_fw_cb->init_fw_cb.iSNSIPAddr,
			pisns_server->IPAddr.IPAddress,
			MIN(sizeof(pflash_init_fw_cb->init_fw_cb.iSNSIPAddr),
			sizeof(pisns_server->IPAddr.IPAddress)));

		port_number = EXT_DEF_ISNS_WELL_KNOWN_PORT;
		if (pisns_server->IPAddr.Type == EXT_DEF_TYPE_ISCSI_IPV6) {
			ipv6_tcp_options |= IPV6_TCPOPT_ISNSv6_ENABLE;
			tcp_options_changed = 1;
		}
		else {
			tcp_options |= TOPT_ISNSv4_ENABLE;
			tcp_options_changed = 1;
		}
	}
	else {
		tcp_options_changed = 1;
		tcp_options &= ~TOPT_ISNSv4_ENABLE;
		if (pisns_server->IPAddr.Type == EXT_DEF_TYPE_ISCSI_IPV6)
			ipv6_tcp_options &= ~IPV6_TCPOPT_ISNSv6_ENABLE;

		memset(pflash_init_fw_cb->init_fw_cb.iSNSIPAddr, 0,
			sizeof(pflash_init_fw_cb->init_fw_cb.iSNSIPAddr));
		port_number = 0;
	}

	if (pisns_server->IPAddr.Type == EXT_DEF_TYPE_ISCSI_IPV6) {
		IPv6Addr2Str(pisns_server->IPAddr.IPAddress, (uint8_t *) &ip_addr_str);
	}
	else {
		IPv4Addr2Str(pisns_server->IPAddr.IPAddress, (uint8_t *) &ip_addr_str);
	}
	QL4PRINT(QLP4, printk("scsi%d: %s: IPAddr %s Port# %04d\n",
	    ha->host_no, __func__, ip_addr_str, port_number));

	/*
	 * If the internal iSNS info is different from the pflash_init_fw_cb,
	 * flash it now.
	 *------------------------------------------------------------------*/
	if (pisns_server->IPAddr.Type == EXT_DEF_TYPE_ISCSI_IPV6) {
		isns_ip_addr_is_equal = IPv6AddrIsEqual(ha->isns_server_ip_addr,
					  pflash_init_fw_cb->init_fw_cb.iSNSIPAddr);
	} else {
		isns_ip_addr_is_equal = IPv4AddrIsEqual(ha->isns_server_ip_addr,
					  pflash_init_fw_cb->init_fw_cb.iSNSIPAddr);
	}

	if (tcp_options_changed || !isns_ip_addr_is_equal ||
             (ha->isns_server_port_number != port_number)) {

		if (pisns_server->IPAddr.Type == EXT_DEF_TYPE_ISCSI_IPV6)
			pflash_init_fw_cb->init_fw_cb.IPv6TCPOptions = cpu_to_le16(ipv6_tcp_options);
		pflash_init_fw_cb->init_fw_cb.TCPOptions = cpu_to_le16(tcp_options);
		pflash_init_fw_cb->init_fw_cb.iSNSServerPortNumber = cpu_to_le16(port_number);

		memset(mbox_cmd, 0, sizeof(mbox_cmd));
		memset(mbox_sts, 0, sizeof(mbox_sts));
		mbox_cmd[0] = MBOX_CMD_WRITE_FLASH;
		mbox_cmd[1] = LSDW(ha->dma_buf.phys_addr);
		mbox_cmd[2] = MSDW(ha->dma_buf.phys_addr);
		mbox_cmd[3] = INT_ISCSI_INITFW_FLASH_OFFSET;
		mbox_cmd[4] = sizeof(*pflash_init_fw_cb);
		mbox_cmd[5] = WRITE_FLASH_OPTION_COMMIT_DATA;

		if (qla4xxx_mailbox_command(ha, 6, 2, &mbox_cmd[0],
		    &mbox_sts[0]) == QLA_ERROR) {
			QL4PRINT(QLP2,
			    printk("scsi%d: %s: WRITE_FLASH command failed \n",
			    ha->host_no, __func__));

			ioctl->Status = EXT_STATUS_MAILBOX;
			ioctl->VendorSpecificStatus[0] = mbox_sts[0];
			ioctl->VendorSpecificStatus[1] = mbox_sts[1];
			ioctl->ResponseLen = 0;
			goto exit_set_isns_svr;
		}

		QL4PRINT(QLP4,
			 printk("scsi%d: %s: WRITE_FLASH command successful \n",
				ha->host_no, __func__));
		QL4PRINT(QLP10,
		    printk("scsi%d: Init Fw Ctrl Blk\n", ha->host_no));
		qla4xxx_dump_bytes(QLP10, pflash_init_fw_cb,
		    sizeof(FLASH_INIT_FW_CTRL_BLK));

		/*
		 * Update internal iSNS info
		 */
		memcpy(ha->isns_server_ip_addr,
			pflash_init_fw_cb->init_fw_cb.iSNSIPAddr,
			MIN(sizeof(ha->isns_server_ip_addr),
			sizeof(pflash_init_fw_cb->init_fw_cb.iSNSIPAddr)));

		ha->isns_server_port_number = port_number;

		if (pisns_server->IPAddr.Type == EXT_DEF_TYPE_ISCSI_IPV6) {
			ha->ipv6_tcp_options = ipv6_tcp_options;

			if (ipv6_tcp_options & IPV6_TCPOPT_ISNSv6_ENABLE)
				set_bit(ISNS_FLAG_ISNS_ENABLED_IN_ISP, &ha->isns_flags);
			else
				clear_bit(ISNS_FLAG_ISNS_ENABLED_IN_ISP, &ha->isns_flags);
		}
		else {
			ha->tcp_options = tcp_options;

			if (tcp_options & TOPT_ISNSv4_ENABLE)
				set_bit(ISNS_FLAG_ISNS_ENABLED_IN_ISP, &ha->isns_flags);
			else
				clear_bit(ISNS_FLAG_ISNS_ENABLED_IN_ISP, &ha->isns_flags);

		}
	}
        else {
		QL4PRINT(QLP2, printk("scsi%d: %s: Don't write flash\n",
			ha->host_no, __func__));
		QL4PRINT(QLP2, printk("scsi%d: %s: performDiscovery = %d\n",
			ha->host_no, __func__, pisns_server->PerformiSNSDiscovery));
		QL4PRINT(QLP2, printk("scsi%d: %s: isns_ip_addr_is_equal = %d\n",
			ha->host_no, __func__, isns_ip_addr_is_equal));
		QL4PRINT(QLP2, printk("scsi%d: %s: ha->isns_server_port_number = %d\n",
			ha->host_no, __func__, ha->isns_server_port_number));
		QL4PRINT(QLP2, printk("scsi%d: %s: tcp_options_changed = %d\n",
			ha->host_no, __func__, tcp_options_changed));
		QL4PRINT(QLP2, printk("scsi%d: %s: ha->isns_flags = %ld\n",
				      ha->host_no, __func__, ha->isns_flags));
        }

	/*
	 * Start or Stop iSNS Service accordingly, if needed.
	 *---------------------------------------------------*/
	if (test_bit(ISNS_FLAG_ISNS_ENABLED_IN_ISP, &ha->isns_flags)) {

		isns_ip_addr_is_zero =
			(pisns_server->IPAddr.Type == EXT_DEF_TYPE_ISCSI_IPV6)
			? IPv6AddrIsZero(ha->isns_server_ip_addr)
			: iSNSIPv4AddrIsZero(ha->isns_server_ip_addr);

		if (!isns_ip_addr_is_zero && ha->isns_server_port_number) {
			status = qla4xxx_isns_reenable(ha, ha->isns_server_ip_addr,
					 ha->isns_server_port_number);

			if (status == QLA_ERROR) {
				QL4PRINT(QLP2, printk(
				    "scsi%d: qla4xxx_isns_reenable failed!\n",
				    ha->host_no));
				ioctl->Status = EXT_STATUS_ERR;
				ioctl->ResponseLen = 0;
				goto exit_set_isns_svr;
			}
		}
		else if (test_bit(ISNS_FLAG_ISNS_SRV_ENABLED,
			&ha->isns_flags) && isns_ip_addr_is_zero) {
			qla4xxx_isns_disable(ha);
		}
	}

	/*
	 * Complete IOCTL successfully
	 *----------------------------*/
	ioctl->Status = EXT_STATUS_OK;
	ioctl->ResponseLen = 0;

exit_set_isns_svr:
	qla4xxx_free_ioctl_scrap_mem(ha);
#if 0 && defined(__VMKERNEL_MODULE__)
	spin_unlock(&ha->dma_buf_spinlock);
#else
	up(&ha->dma_buf_sem);
#endif
	LEAVE(__func__);
	return(status);
}

/**************************************************************************
 * qla4extioctl_set_acb
 *	This routine modifies the address control block.
 *
 * Input:
 *	ha    = adapter structure pointer.
 *	ioctl = IOCTL structure pointer.
 *
 * Output:
 *	None
 *
 * Returns:
 *	QLA_SUCCESS = success
 *	QLA_ERROR   = error
 *
 * Context:
 *	Kernel context.
 **************************************************************************/
static int
qla4extioctl_set_acb(scsi_qla_host_t *ha, EXT_IOCTL_ISCSI *ioctl)
{
	int	status = 0;
	uint32_t mbox_cmd[MBOX_REG_COUNT];
	uint32_t mbox_sts[MBOX_REG_COUNT];

	ENTER(__func__);
#if 0 && defined(__VMKERNEL_MODULE__)
	spin_lock(&ha->dma_buf_spinlock);
#else
	down(&ha->dma_buf_sem);
#endif

	if (ioctl->RequestLen < EXT_DEF_ACB_SIZE) {
		QL4PRINT(QLP2,
		    printk("scsi%d: %s: requst buffer too small (%d/%xh)\n",
		    ha->host_no, __func__, ioctl->RequestLen,
		    ioctl->RequestLen));

		status = (-ENOMEM);
		ioctl->Status = EXT_STATUS_BUFFER_TOO_SMALL;
		ioctl->ResponseLen = 0;
		goto exit_set_acb;
	}

	if (!ha->dma_buf.virt_addr || !ha->dma_buf.phys_addr ||
	    (ha->dma_buf.buf_len < EXT_DEF_ACB_SIZE)) {
		if (qla4xxx_resize_dma_buf(ha, &ha->dma_buf,
		    EXT_DEF_ACB_SIZE) != QLA_SUCCESS) {
			QL4PRINT(QLP2,
			    printk("scsi%d: %s: unable to allocate memory "
			    "for dma buffer.\n",
			    ha->host_no, __func__));

			status = (-ENOMEM);
			ioctl->Status = EXT_STATUS_NO_MEMORY;
			ioctl->ResponseLen = 0;
			goto exit_set_acb;
		}
	}

	/*
	 * Transfer Data from the user's buffer to DMA buffer
	 */
	if (copy_from_user(ha->dma_buf.virt_addr,
		Q64BIT_TO_PTR(ioctl->RequestAdr), EXT_DEF_ACB_SIZE) != 0) {
		QL4PRINT(QLP2,
		    printk("scsi%d: %s: unable to copy data to user's "
		    "memory area\n", ha->host_no, __func__));

		status = (-EFAULT);
		ioctl->Status = EXT_STATUS_COPY_ERR;
		ioctl->ResponseLen = 0;
		goto exit_set_acb;
	}

	memset(mbox_cmd, 0, sizeof(mbox_cmd));
	memset(mbox_sts, 0, sizeof(mbox_sts));
	mbox_cmd[0] = MBOX_CMD_SET_ACB;
	mbox_cmd[1] = ioctl->Instance;  /* Primary/Secondary */
	mbox_cmd[2] = LSDW(ha->dma_buf.phys_addr);
	mbox_cmd[3] = MSDW(ha->dma_buf.phys_addr);
	mbox_cmd[4] = ioctl->RequestLen;

	if (qla4xxx_mailbox_command(ha, 5, 5, &mbox_cmd[0], &mbox_sts[0])
	    == QLA_ERROR) {
		QL4PRINT(QLP2, printk("scsi%d: %s: MBOX_CMD_SET_ACB "
				           "FAILED w/ status %04X\n",
                                           ha->host_no, __func__, mbox_sts[0]));
		QL4PRINT(QLP2, printk("mbox[1] = 0x%04x\n"
					   "mbox[2] = 0x%04x\n"
					   "mbox[3] = 0x%04x\n"
					   "mbox[4] = 0x%04x\n",
                                           mbox_sts[1], mbox_sts[2],
                                           mbox_sts[3], mbox_sts[4]));

		ioctl->Status = EXT_STATUS_MAILBOX;
		ioctl->VendorSpecificStatus[0] = mbox_sts[0];
		ioctl->VendorSpecificStatus[1] = mbox_sts[1];
		ioctl->VendorSpecificStatus[2] = mbox_sts[2];
		ioctl->VendorSpecificStatus[3] = mbox_sts[3];
		ioctl->VendorSpecificStatus[4] = mbox_sts[4];
		ioctl->ResponseLen = 0;
		goto exit_set_acb;
	}

	ioctl->Status = EXT_STATUS_OK;
	ioctl->ResponseLen = mbox_sts[4];

	QL4PRINT(QLP4, printk("scsi%d: %s: Succeeded\n",
		ha->host_no, __func__));
	QL4PRINT(QLP10, printk("scsi%d: ACB structure dump:\n", ha->host_no));
	qla4xxx_dump_bytes(QLP10, ha->dma_buf.virt_addr, EXT_DEF_ACB_SIZE);

exit_set_acb:
#if 0 && defined(__VMKERNEL_MODULE__)
	spin_unlock(&ha->dma_buf_spinlock);
#else
	up(&ha->dma_buf_sem);
#endif

	LEAVE(__func__);
	return(status);
}
#endif

/**************************************************************************
 * qla4extioctl_set_data
 *	This routine calls set data IOCTLs based on the IOCTL Sub Code.
 *
 * Input:
 *	ha    = adapter structure pointer.
 *	ioctl = IOCTL structure pointer.
 *
 * Output:
 *	None
 *
 * Returns:
 *	QLA_SUCCESS = success
 *	QLA_ERROR   = error
 *    	-EINVAL     = if the command is invalid
 *
 * Context:
 *	Kernel context.
 **************************************************************************/
static int qla4extioctl_set_data(scsi_qla_host_t *ha, EXT_IOCTL_ISCSI *ioctl)
{
	switch (ioctl->SubCode) {
	case EXT_SC_RST_STATISTICS_GEN:
		return(qla4extioctl_rst_statistics_gen(ha, ioctl));

	case EXT_SC_RST_STATISTICS_ISCSI:
		return(qla4extioctl_rst_statistics_iscsi(ha, ioctl));

	case EXT_SC_SET_DEVICE_ENTRY_ISCSI:
		return(qla4extioctl_set_device_entry_iscsi(ha, ioctl));

	case EXT_SC_SET_INIT_FW_ISCSI:
		return(qla4extioctl_set_init_fw_iscsi(ha, ioctl));

#ifndef QLA4000
	case EXT_SC_SET_ISNS_SERVER:
		return(qla4extioctl_set_isns_server(ha, ioctl));

	case EXT_SC_SET_ACB:
		return(qla4extioctl_set_acb(ha, ioctl));
#endif

	default:
		QL4PRINT(QLP2,
		    printk("scsi%d: %s: unsupported set data sub-command "
		    "code (%X)\n",
		    ha->host_no, __func__, ioctl->SubCode));

		ioctl->Status = EXT_STATUS_UNSUPPORTED_SUBCODE;
		ioctl->ResponseLen = 0;
		return(-EINVAL);
	}
	return(0);
}

/**************************************************************************
 * qla4xxx_ioctl_sleep_done
 *	This routine is the callback function to wakeup ioctl completion
 *	semaphore for the ioctl request that is waiting.
 *
 * Input:
 *   	sem - pointer to the ioctl completion semaphore.
 *
 * Output:
 *	None
 *
 * Returns:
 *	None
 *
 * Context:
 *	Kernel context.
 **************************************************************************/
static  void
qla4xxx_ioctl_sleep_done (struct semaphore * sem)
{
	ENTER(__func__);

	if (sem != NULL) {
		QL4PRINT(QLP4, printk("%s: wake up sem.\n", __func__));
		QL4PRINT(QLP10, printk("%s: UP count=%d\n", __func__,
		    atomic_read(&sem->count)));
		up(sem);
	}

	LEAVE(__func__);
}

/**************************************************************************
 * qla4xxx_ioctl_sem_init
 *	This routine initializes the ioctl timer and semaphore used to wait
 *	for passthru completion.
 *
 * Input:
 *	ha    = adapter structure pointer.
 *
 * Output:
 *	None
 *
 * Returns:
 *	None
 *
 * Context:
 *	Kernel context.
 **************************************************************************/
void
qla4xxx_ioctl_sem_init (scsi_qla_host_t *ha)
{
	init_timer(&(ha->ioctl_cmpl_timer));
	ha->ioctl_cmpl_timer.data = (unsigned long)&ha->ioctl_cmpl_sem;
	ha->ioctl_cmpl_timer.function =
	(void (*)(unsigned long))qla4xxx_ioctl_sleep_done;
}

/**************************************************************************
 * qla4xxx_scsi_pass_done
 *	This routine resets the ioctl progress flag and wakes up the ioctl
 * 	completion semaphore.
 *
 * Input:
 *   	cmd - pointer to the passthru Scsi cmd structure which has completed.
 *
 * Output:
 *	None
 *
 * Returns:
 *	None
 *
 * Context:
 *	Interrupt context.
 **************************************************************************/
void
qla4xxx_scsi_pass_done(Scsi_Cmnd *cmd)
{
	scsi_qla_host_t *ha = (scsi_qla_host_t *) cmd->host->hostdata;


	ENTER(__func__);

	/* First check to see if the command has previously timed-out
	 * because we don't want to get the up/down semaphore counters off.
	 */
	if (ha->ioctl_scsi_pass_in_progress == 1) {
		ha->ioctl_scsi_pass_in_progress = 0;
		ha->ioctl_tov = 0;
		ha->ioctl_err_cmd = NULL;

		up(&ha->ioctl_cmpl_sem);
	}

	LEAVE(__func__);
	return;
}

/**************************************************************************
 * qla4extioctl_scsi_passthru
 *	This routine
 *
 * Input:
 *	ha    = adapter structure pointer.
 *	ioctl = IOCTL structure pointer.
 *
 * Output:
 *	None
 *
 * Map of DMA Buffer:
 *    +-------------------------+
 *    | EXT_SCSI_PASSTHRU_ISCSI |
 *    +-------------------------+
 *    | [SCSI READ|WRITE data]  |
 *    +-------------------------+
 *
 * Returns:
 *	QLA_SUCCESS = success
 *	QLA_ERROR   = error
 *
 * Context:
 *	Kernel context.
 **************************************************************************/
static int
qla4extioctl_scsi_passthru(scsi_qla_host_t *ha, EXT_IOCTL_ISCSI *ioctl)
{
	int			status = 0;
	ddb_entry_t             *ddb_entry;
	int                     i;
	EXT_SCSI_PASSTHRU_ISCSI	*pscsi_pass = NULL;
	Scsi_Device		*pscsi_device = NULL;
	Scsi_Cmnd		*pscsi_cmd = NULL;
	srb_t			*srb;
	uint32_t		dma_buf_len;
#ifdef __VMKERNEL_MODULE__
	uint8_t			ispStatus;
#endif

	ENTER(__func__);

	if (!ADAPTER_UP(ha)) {
		QL4PRINT(QLP2,
		    printk("scsi%d: %s: command not pocessed, "
		    "adapter link down.\n",
		    ha->host_no, __func__));
		ioctl->Status = EXT_STATUS_HBA_NOT_READY;
		ioctl->ResponseLen = 0;
		return(QLA_ERROR);
	}

#if 0 && defined(__VMKERNEL_MODULE__)
	spin_lock(&ha->dma_buf_spinlock);
#else
	down(&ha->dma_buf_sem);
#endif

	if (qla4xxx_get_ioctl_scrap_mem(ha, (void **)&pscsi_cmd,
	    sizeof(Scsi_Cmnd))) {
		/* not enough memory */
		QL4PRINT(QLP2,
		    printk("scsi%d: %s: inst=%d scrap not big enough. "
		    "size requested=%ld.\n",
		    ha->host_no, __func__, ha->instance,
		    (ulong)sizeof(Scsi_Cmnd)));

		status = (-ENOMEM);
		ioctl->Status = EXT_STATUS_NO_MEMORY;
		ioctl->ResponseLen = 0;
		goto error_exit_scsi_pass;
	}

	if (qla4xxx_get_ioctl_scrap_mem(ha, (void **)&pscsi_device,
	    sizeof(Scsi_Device))) {
		/* not enough memory */
		QL4PRINT(QLP2,
		    printk("scsi%d: %s: inst=%d scrap not big enough. "
		    "size requested=%ld.\n",
		    ha->host_no, __func__, ha->instance,
		    (ulong)sizeof(Scsi_Device)));

		status = (-ENOMEM);
		ioctl->Status = EXT_STATUS_NO_MEMORY;
		ioctl->ResponseLen = 0;
		goto error_exit_scsi_pass;
	}

	if (qla4xxx_get_ioctl_scrap_mem(ha, (void **)&pscsi_pass,
	    sizeof(EXT_SCSI_PASSTHRU_ISCSI))) {
		/* not enough memory */
		QL4PRINT(QLP2,
		    printk("scsi%d: %s: inst=%d scrap not big enough. "
		    "size requested=%ld.\n",
		    ha->host_no, __func__, ha->instance,
		    (ulong)sizeof(EXT_SCSI_PASSTHRU_ISCSI)));

		status = (-ENOMEM);
		ioctl->Status = EXT_STATUS_NO_MEMORY;
		ioctl->ResponseLen = 0;
		goto error_exit_scsi_pass;
	}

	memset(pscsi_device, 0, sizeof(Scsi_Device));
	memset(pscsi_pass, 0, sizeof(EXT_SCSI_PASSTHRU_ISCSI));
	memset(pscsi_cmd, 0, sizeof(Scsi_Cmnd));
	pscsi_cmd->device = pscsi_device;
	pscsi_cmd->request.nr_hw_segments = 1;

	/* ---- Get passthru structure from user space ---- */
	if (copy_from_user((uint8_t *)pscsi_pass,
	    Q64BIT_TO_PTR(ioctl->RequestAdr),
			   sizeof(EXT_SCSI_PASSTHRU_ISCSI)) != 0) {
		QL4PRINT(QLP2,
		    printk("scsi%d: %s: unable to copy passthru struct "
		    "from user's memory area.\n",
		    ha->host_no, __func__));

		status = (-EFAULT);
		ioctl->Status = EXT_STATUS_COPY_ERR;
		ioctl->ResponseLen = 0;
		goto error_exit_scsi_pass;
	}

	QL4PRINT(QLP10, printk("scsi%d: %s: incoming  EXT_SCSI_PASSTHRU_ISCSI "
	    "structure: \n", ha->host_no, __func__));
	qla4xxx_dump_bytes(QLP10, pscsi_pass, sizeof(EXT_SCSI_PASSTHRU_ISCSI));

	/* ---- Make sure device exists ---- */
	ddb_entry = qla4xxx_lookup_ddb_by_SCSIID(ha, pscsi_pass->Addr.Bus,
	    pscsi_pass->Addr.Target);
	if (ddb_entry == NULL) {
		QL4PRINT(QLP2,
		    printk("scsi%d: %s: invalid device (b%d,t%d) specified.\n",
		    ha->host_no, __func__,
		    pscsi_pass->Addr.Bus, pscsi_pass->Addr.Target));

		status = (-EINVAL);
		ioctl->Status = EXT_STATUS_DEV_NOT_FOUND;
		ioctl->ResponseLen = 0;
		goto error_exit_scsi_pass;
	}

	/* ---- Make sure device is in an active state ---- */
	if (ddb_entry->fw_ddb_device_state != DDB_DS_SESSION_ACTIVE) {
		QL4PRINT(QLP2,
		    printk("scsi%d: %s: device (b%d,t%d) not in active state\n",
		    ha->host_no, __func__,
		    pscsi_pass->Addr.Bus, pscsi_pass->Addr.Target));

		ioctl->Status = EXT_STATUS_DEVICE_NOT_READY;
		ioctl->ResponseLen = 0;
		goto error_exit_scsi_pass;
	}

	/* ---- Retrieve srb from pool ---- */
	srb = del_from_free_srb_q_head(ha);
	if (srb == NULL) {
		QL4PRINT(QLP2, printk("scsi%d: %s: srb not available\n",
		    ha->host_no, __func__));
		goto error_exit_scsi_pass;
	}

	/* ---- Allocate larger DMA buffer, if neccessary ---- */
	dma_buf_len = MAX(ioctl->ResponseLen - sizeof(EXT_SCSI_PASSTHRU_ISCSI),
	    ioctl->RequestLen - sizeof(EXT_SCSI_PASSTHRU_ISCSI));

	if (ha->dma_buf.buf_len < dma_buf_len
	    && qla4xxx_resize_dma_buf(ha, &ha->dma_buf, dma_buf_len) !=
	    QLA_SUCCESS) {
		QL4PRINT(QLP2,
		    printk("scsi%d: %s: ERROR cannot allocate requested "
		    "DMA buffer size 0x%x.\n",
		    ha->host_no, __func__, dma_buf_len));

		status = (-ENOMEM);
		ioctl->Status = EXT_STATUS_NO_MEMORY;
		ioctl->ResponseLen = 0;
		goto error_exit_scsi_pass;
	}

	memset(ha->dma_buf.virt_addr, 0, ha->dma_buf.buf_len);

	/* ---- Fill in the SCSI command structure ---- */
	pscsi_cmd->channel         = pscsi_pass->Addr.Bus;
	pscsi_cmd->target          = pscsi_pass->Addr.Target;
	pscsi_cmd->lun             = pscsi_pass->Addr.Lun;
	pscsi_cmd->request_buffer  = ha->dma_buf.virt_addr;
#ifdef __VMKERNEL_MODULE__
	pscsi_cmd->request_bufferMA =
              (dma_addr_t)virt_to_phys(pscsi_cmd->request_buffer);
#endif //__VMKERNEL_MODULE__
	pscsi_cmd->device          = pscsi_device;
	pscsi_cmd->host            = ha->host;
	pscsi_cmd->scsi_done       = qla4xxx_scsi_pass_done;
	CMD_TIMEOUT(pscsi_cmd)     = ql4xioctltimeout * HZ;

	CMD_SP(pscsi_cmd) = (char *) srb;
	srb->cmd = pscsi_cmd;
	srb->fw_ddb_index = ddb_entry->fw_ddb_index;
	srb->lun = pscsi_cmd->lun;
	srb->flags |= SRB_IOCTL_CMD;

	if (pscsi_pass->CdbLength == 6 || pscsi_pass->CdbLength == 10 ||
	    pscsi_pass->CdbLength == 12 || pscsi_pass->CdbLength == 16) {
		pscsi_cmd->cmd_len = pscsi_pass->CdbLength;
	} else {
		QL4PRINT(QLP2,
		    printk("scsi%d: %s: Unsupported CDB length 0x%x \n",
		    ha->host_no, __func__, pscsi_cmd->cmd_len));

		status = (-EINVAL);
		ioctl->Status = EXT_STATUS_INVALID_PARAM;
		ioctl->ResponseLen = 0;
		goto error_exit_scsi_pass;
	}

	if (pscsi_pass->Direction == EXT_DEF_SCSI_PASSTHRU_DATA_IN) {
		pscsi_cmd->sc_data_direction = SCSI_DATA_READ;
		pscsi_cmd->request_bufflen = ioctl->ResponseLen -
		    sizeof(*pscsi_pass);

	} else if (pscsi_pass->Direction == EXT_DEF_SCSI_PASSTHRU_DATA_OUT) {
		pscsi_cmd->sc_data_direction = SCSI_DATA_WRITE;
		pscsi_cmd->request_bufflen = ioctl->RequestLen -
		    sizeof(*pscsi_pass);

		/* Sending user data from ioctl->ResponseAddr to SCSI
		 * command buffer */
		if (copy_from_user((uint8_t *)pscsi_cmd->request_buffer,
		    Q64BIT_TO_PTR(ioctl->RequestAdr) + sizeof(*pscsi_pass),
		    pscsi_cmd->request_bufflen) != 0) {
			QL4PRINT(QLP2,
			    printk("scsi%d: %s: unable to copy write buffer "
			    "from user's memory area.\n",
			    ha->host_no, __func__));

			status = (-EFAULT);
			ioctl->Status = EXT_STATUS_COPY_ERR;
			goto error_exit_scsi_pass;
		}
	} else {
		pscsi_cmd->sc_data_direction = SCSI_DATA_NONE;
		pscsi_cmd->request_buffer  = 0;
		pscsi_cmd->request_bufflen = 0;
	}

	memcpy(pscsi_cmd->cmnd,      pscsi_pass->Cdb, pscsi_cmd->cmd_len);
	memcpy(pscsi_cmd->data_cmnd, pscsi_pass->Cdb, pscsi_cmd->cmd_len);

	QL4PRINT(QLP4, printk("scsi%d:%d:%d:%d: %s: CDB = ",
	    ha->host_no, pscsi_cmd->channel, pscsi_cmd->target,
	    pscsi_cmd->lun, __func__));

	for (i = 0; i < pscsi_cmd->cmd_len; i++)
		QL4PRINT(QLP4, printk("%02X ", pscsi_cmd->cmnd[i]));

	QL4PRINT(QLP4, printk("\n"));

	/* ---- prepare for receiving completion ---- */
	ha->ioctl_scsi_pass_in_progress = 1;
	ha->ioctl_tov = (ql4xioctltimeout - QLA_CMD_TIMER_DELTA) * HZ;

	qla4xxx_ioctl_sem_init(ha);
	CMD_COMPL_STATUS(pscsi_cmd)  = IOCTL_INVALID_STATUS;
	CMD_PASSTHRU_TYPE(pscsi_cmd) = (void *) 1;

	/* ---- send command to adapter ---- */
	QL4PRINT(QLP4, printk("scsi%d:%d:%d:%d: %s: sending command.\n",
	    ha->host_no, pscsi_cmd->channel, pscsi_cmd->target,
	    pscsi_cmd->lun, __func__));

	ha->ioctl_cmpl_timer.expires = jiffies + ha->ioctl_tov;
	add_timer(&ha->ioctl_cmpl_timer);

#ifdef __VMKERNEL_MODULE__
	ispStatus = qla4xxx_send_command_to_isp(ha, srb);
	if( ispStatus == QLA_ERROR ){
#else
	if (qla4xxx_send_command_to_isp(ha, srb) != QLA_SUCCESS) {
#endif
		add_to_free_srb_q(ha, srb);
		QL4PRINT(QLP2,
		    printk("scsi%d: %s: error sending cmd to isp\n",
		    ha->host_no, __func__));
		del_timer(&ha->ioctl_cmpl_timer);

		status = (-EINVAL);
		ioctl->Status = EXT_STATUS_DEV_NOT_FOUND;
		ioctl->ResponseLen = 0;
		goto error_exit_scsi_pass;
	}
#ifdef __VMKERNEL_MODULE__
	else if( ispStatus == QLA_BUSY ){
		QL4PRINT(QLP2,
		    printk("scsi%d: %s: isp busy while sending cmd (added to pending q)\n",
                    ha->host_no, __func__));
		add_to_pending_srb_q_head(ha, srb);
	}
#endif
	down(&ha->ioctl_cmpl_sem);

	/*******************************************************
	 *						       *
	 *             Passthru Completion                     *
	 *						       *
	 *******************************************************/
	del_timer(&ha->ioctl_cmpl_timer);

	/* ---- check for timeout --- */
	if (ha->ioctl_scsi_pass_in_progress == 1) {
		QL4PRINT(QLP2,
		    printk("scsi%d: %s: ERROR = command timeout.\n",
		    ha->host_no, __func__));

		ioctl->Status = EXT_STATUS_ERR;

		if ((srb != NULL) && (srb->active_array_index < MAX_SRBS)) {
			unsigned long wait_cnt = WAIT_CMD_TOV;

#ifdef __VMWARE__
			if (srb->state != SRB_FREE_STATE)
#else
			if ((srb->flags & SRB_FREE_STATE) == 0)
#endif
				qla4xxx_delete_timer_from_cmd(srb);

			/* Wait for command to get out of active state */
			wait_cnt = jiffies + WAIT_CMD_TOV * HZ;
			while (!time_after_eq(jiffies, wait_cnt)) {
#ifdef __VMWARE__
				if (srb->state != SRB_ACTIVE_STATE)
#else
				if (srb->flags != SRB_ACTIVE_STATE)
#endif
					break;

				QL4PRINT(QLP7, printk("."));
				set_current_state(TASK_UNINTERRUPTIBLE);
				schedule_timeout(1 * HZ);
			}
		}
#ifdef __VMWARE__
		switch (srb->state) {
			/* If it's still in the active state, then clean it up here. */
			case SRB_ACTIVE_STATE:
			case SRB_ACTIVE_TIMEOUT_STATE:
				/* add it to free_srb_q only of delete was successful */
				if (del_from_active_array(ha, (uint32_t) srb->active_array_index)) {
					add_to_free_srb_q(ha, srb);
				}
				break;
			case SRB_PENDING_STATE:
				del_from_pending_srb_q(ha, srb);
				add_to_free_srb_q(ha, srb);
				break;
			case SRB_RETRY_STATE:
				del_from_retry_srb_q(ha, srb);
				add_to_free_srb_q(ha, srb);
				break;
			case SRB_DONE_STATE:
				del_from_done_srb_q(ha, srb);
				add_to_free_srb_q(ha, srb);
				break;
		}
#endif
		ha->ioctl_scsi_pass_in_progress = 0;
		goto error_exit_scsi_pass;
	}

	/* --- Return info from status entry --- */
	ioctl->DetailStatus = CMD_SCSI_STATUS(pscsi_cmd);
	pscsi_pass->Reserved[0] = (uint8_t) CMD_SCSI_STATUS(pscsi_cmd);
	pscsi_pass->Reserved[1] = (uint8_t) CMD_COMPL_STATUS(pscsi_cmd);
	pscsi_pass->Reserved[2] = (uint8_t) CMD_ACTUAL_SNSLEN(pscsi_cmd);
	pscsi_pass->Reserved[3] = (uint8_t) CMD_HOST_STATUS(pscsi_cmd);
	pscsi_pass->Reserved[4] = MSB(CMD_RESID_LEN(pscsi_cmd));
	pscsi_pass->Reserved[5] = LSB(CMD_RESID_LEN(pscsi_cmd));
	pscsi_pass->Reserved[6] = (uint8_t) CMD_ISCSI_RESPONSE(pscsi_cmd);
	pscsi_pass->Reserved[7] = (uint8_t) CMD_ISCSI_FLAGS(pscsi_cmd);

	if (CMD_ACTUAL_SNSLEN(pscsi_cmd)) {
		memcpy(pscsi_pass->SenseData, CMD_SNSP(pscsi_cmd),
		    MIN(CMD_ACTUAL_SNSLEN(pscsi_cmd),
		    sizeof(pscsi_pass->SenseData)));

		QL4PRINT(QLP2|QLP10,
		    printk("scsi%d: %s: sense data dump:\n",
		    ha->host_no, __func__));
		qla4xxx_dump_bytes(QLP2|QLP10,
		    pscsi_pass->SenseData, sizeof(pscsi_pass->SenseData));
	}

	/* ---- check for command completion --- */
	if (CMD_COMPL_STATUS(pscsi_cmd) == IOCTL_INVALID_STATUS) {
		QL4PRINT(QLP2,
		    printk("scsi%d:%d:%d:%d: %s: ERROR = "
		    "command not completed.\n",
		    ha->host_no, pscsi_cmd->channel, pscsi_cmd->target,
		    pscsi_cmd->lun, __func__));

		ioctl->Status = EXT_STATUS_ERR;
		goto error_exit_scsi_pass;

	} else if (CMD_HOST_STATUS(pscsi_cmd) == DID_OK) {

		ioctl->Status = EXT_STATUS_OK;

	} else if (CMD_COMPL_STATUS(pscsi_cmd) == SCS_DATA_UNDERRUN) {
		QL4PRINT(QLP2,
		    printk("scsi%d: %s: Data underrun.  Resid = 0x%x\n",
		    ha->host_no, __func__, CMD_RESID_LEN(pscsi_cmd)));

		ioctl->Status = EXT_STATUS_DATA_UNDERRUN;
		pscsi_pass->Reserved[4] = MSB(CMD_RESID_LEN(pscsi_cmd));
		pscsi_pass->Reserved[5] = LSB(CMD_RESID_LEN(pscsi_cmd));

	} else if (CMD_COMPL_STATUS(pscsi_cmd) == SCS_DATA_OVERRUN) {
		QL4PRINT(QLP2,
		    printk("scsi%d: %s: Data overrun.  Resid = 0x%x\n",
		    ha->host_no, __func__, CMD_RESID_LEN(pscsi_cmd)));

		ioctl->Status = EXT_STATUS_DATA_OVERRUN;
		pscsi_pass->Reserved[4] = MSB(CMD_RESID_LEN(pscsi_cmd));
		pscsi_pass->Reserved[5] = LSB(CMD_RESID_LEN(pscsi_cmd));

	} else {
		QL4PRINT(QLP2,
		    printk("scsi%d: %s: Command completed in ERROR. "
		    "cs=%04x, ss=%-4x\n", ha->host_no, __func__,
		    CMD_COMPL_STATUS(pscsi_cmd), CMD_SCSI_STATUS(pscsi_cmd)));

		if (CMD_SCSI_STATUS(pscsi_cmd) != SCSI_GOOD) {
			ioctl->Status = EXT_STATUS_SCSI_STATUS;
		} else {
			ioctl->Status = EXT_STATUS_ERR;
		}
	}

	/* ---- Copy SCSI Passthru structure with updated sense buffer
	 *      to user space ---- */
	if (copy_to_user(Q64BIT_TO_PTR(ioctl->ResponseAdr), pscsi_pass,
			 sizeof(EXT_SCSI_PASSTHRU_ISCSI)) != 0) {
		QL4PRINT(QLP2,
		    printk("scsi%d: %s: unable to copy passthru struct "
		    "to user's memory area.\n",
		    ha->host_no, __func__));

		status = (-EFAULT);
		ioctl->Status = EXT_STATUS_COPY_ERR;
		goto error_exit_scsi_pass;
	}

	QL4PRINT(QLP10,
	    printk("scsi%d: %s: outgoing EXT_SCSI_PASSTHRU_ISCSI structure:\n",
	    ha->host_no, __func__));
	qla4xxx_dump_bytes(QLP10,
	    Q64BIT_TO_PTR(ioctl->ResponseAdr),
			   sizeof(EXT_SCSI_PASSTHRU_ISCSI));

	/* ---- Copy SCSI READ data from SCSI command buffer
	*       to user space ---- */
	if (pscsi_pass->Direction == EXT_DEF_SCSI_PASSTHRU_DATA_IN) {
		void   *xfer_ptr = Q64BIT_TO_PTR(ioctl->ResponseAdr) +
				    sizeof(EXT_SCSI_PASSTHRU_ISCSI);
		uint32_t xfer_len = ioctl->ResponseLen -
				    sizeof(EXT_SCSI_PASSTHRU_ISCSI);


		/* Update ResponseLen if a data underrun occurred */
		if (CMD_COMPL_STATUS(pscsi_cmd) == SCS_DATA_UNDERRUN &&
		    CMD_RESID_LEN(pscsi_cmd)) {
			xfer_len -= CMD_RESID_LEN(pscsi_cmd);
		}

		if (copy_to_user(xfer_ptr, pscsi_cmd->request_buffer,
				 xfer_len) != 0) {
			QL4PRINT(QLP2,
			    printk("scsi%d: %s: unable to copy READ data "
			    "to user's memory area.\n",
			    ha->host_no, __func__));

			status = (-EFAULT);
			ioctl->Status = EXT_STATUS_COPY_ERR;
			goto error_exit_scsi_pass;
		}

		QL4PRINT(QLP10,
		    printk("scsi%d: %s: outgoing READ data:  (0x%p)\n",
		    ha->host_no, __func__, xfer_ptr));
		qla4xxx_dump_bytes(QLP10, xfer_ptr, xfer_len);
	}

	goto exit_scsi_pass;

error_exit_scsi_pass:
	ioctl->ResponseLen = 0;

exit_scsi_pass:
	qla4xxx_free_ioctl_scrap_mem(ha);
#if 0 && defined(__VMKERNEL_MODULE__)
	spin_unlock(&ha->dma_buf_spinlock);
#else
	up(&ha->dma_buf_sem);
#endif

	QL4PRINT(QLP12,
	    printk("scsi%d: %s: inst %d exiting.\n",
	    ha->host_no, __func__, ha->instance));
	LEAVE(__func__);
	return(status);
}

/**************************************************************************
 * qla4extioctl_iscsi_passthru
 *	This routine sends iSCSI pass-through to destination.
 *
 * Input:
 *	ha    = adapter structure pointer.
 *	ioctl = IOCTL structure pointer.
 *
 * Output:
 *	None
 *
 * Returns:
 *	QLA_SUCCESS = success
 *	QLA_ERROR   = error
 *
 * Context:
 *	Kernel context.
 **************************************************************************/
static int
qla4extioctl_iscsi_passthru(scsi_qla_host_t *ha, EXT_IOCTL_ISCSI *ioctl)
{
	int	status = 0;

	ENTER(__func__);
	QL4PRINT(QLP2|QLP4, printk("scsi%d: %s: UNSUPPORTED\n",
	    ha->host_no, __func__));

	status = (-EINVAL);
	ioctl->Status = EXT_STATUS_UNSUPPORTED_SUBCODE;
	ioctl->ResponseLen = 0;
	LEAVE(__func__);
	return(status);
}

/**************************************************************************
 * qla4extioctl_disable_acb
 *	This routine disables the access control block
 *
 * Input:
 *	ha    = adapter structure pointer.
 *	ioctl = IOCTL structure pointer.
 *
 * Output:
 *	None
 *
 * Returns:
 *	QLA_SUCCESS = success
 *	QLA_ERROR   = error
 *
 * Context:
 *	Kernel context.
 **************************************************************************/
static int
qla4extioctl_disable_acb(scsi_qla_host_t *ha, EXT_IOCTL_ISCSI *ioctl)
{
	int	status = 0;
	uint32_t mbox_cmd[MBOX_REG_COUNT];
	uint32_t mbox_sts[MBOX_REG_COUNT];

	ENTER(__func__);

	memset(mbox_cmd, 0, sizeof(mbox_cmd));
	memset(mbox_sts, 0, sizeof(mbox_sts));
	mbox_cmd[0] = MBOX_CMD_DISABLE_ACB;
	mbox_cmd[1] = ioctl->Instance;  /* Primary/Secondary */
	mbox_cmd[2] = ioctl->Reserved1;	/* Command Option */

	if (qla4xxx_mailbox_command(ha, 3, 1, &mbox_cmd[0], &mbox_sts[0])
	    == QLA_SUCCESS) {
		QL4PRINT(QLP4, printk("scsi%d: %s: Succeeded\n",
			ha->host_no, __func__));

		ioctl->Status = EXT_STATUS_OK;
	} else {
		if (mbox_sts[0] == MBOX_STS_INTERMEDIATE_COMPLETION) {
			unsigned long  wait_cnt;

			/* Poll AF_DISABLE_ACB_COMPLETE bit for
			 * AEN 8027 completion */
			wait_cnt = jiffies + WAIT_CMD_TOV * HZ;
			while (!time_after_eq(jiffies, wait_cnt)) {

				if (test_bit(AF_DISABLE_ACB_COMPLETE,
					&ha->flags)) {
					break;
				}

				QL4PRINT(QLP2, printk("."));
				set_current_state(TASK_UNINTERRUPTIBLE);
				schedule_timeout(1 * HZ);
			}

			if (test_and_clear_bit(AF_DISABLE_ACB_COMPLETE,
					&ha->flags)) {
				QL4PRINT(QLP4, printk("scsi%d: %s: Succeeded\n",
				    ha->host_no, __func__));
				ioctl->Status = EXT_STATUS_OK;
				ioctl->ResponseLen = 0;
				goto exit_disable_acb;
			} else {

				QL4PRINT(QLP2, printk("scsi%d: %s: "
				    "Intermediate Timed Out\n",
				    ha->host_no, __func__));

				ioctl->Status = EXT_STATUS_PENDING;
				ioctl->ResponseLen = 0;
				goto exit_disable_acb;
			}
		} else {
			QL4PRINT(QLP2, printk("scsi%d: %s: FAILED w/ Status %04X\n",
			    ha->host_no, __func__, mbox_sts[0]));

			ioctl->Status = EXT_STATUS_MAILBOX;
			ioctl->VendorSpecificStatus[0] = mbox_sts[0];
			ioctl->ResponseLen = 0;
		}
	}

exit_disable_acb:
	LEAVE(__func__);
	return(status);
}

/**************************************************************************
 * qla4extioctl_send_router_sol
 *	This routine requests the driver to send a router solicitation.
 *
 * Input:
 *	ha    = adapter structure pointer.
 *	ioctl = IOCTL structure pointer.
 *
 * Output:
 *	None
 *
 * Returns:
 *	QLA_SUCCESS = success
 *	QLA_ERROR   = error
 *
 * Context:
 *	Kernel context.
 **************************************************************************/
static int
qla4extioctl_send_router_sol(scsi_qla_host_t *ha, EXT_IOCTL_ISCSI *ioctl)
{
	int	status = 0;
	EXT_SEND_ROUTER_SOL *psend_router_sol = NULL;
	uint32_t mbox_cmd[MBOX_REG_COUNT];
	uint32_t mbox_sts[MBOX_REG_COUNT];

	ENTER(__func__);

	if (ioctl->RequestLen < sizeof(EXT_SEND_ROUTER_SOL)) {
		QL4PRINT(QLP2,
		    printk("scsi%d: %s: requst buffer too small (%d/%xh)\n",
		    ha->host_no, __func__, ioctl->RequestLen,
		    ioctl->RequestLen));

		status = (-ENOMEM);
		ioctl->Status = EXT_STATUS_BUFFER_TOO_SMALL;
		ioctl->ResponseLen = 0;
		goto exit_send_ipv6_router_sol;
	}

	if (qla4xxx_get_ioctl_scrap_mem(ha, (void **)&psend_router_sol,
					sizeof(*psend_router_sol))) {
		/* not enough memory */
		QL4PRINT(QLP2,
		    printk("scsi%d: %s: inst=%d scrap not big enough. "
		    "size requested=%ld.\n",
		    ha->host_no, __func__, ha->instance,
		    (ulong)sizeof(*psend_router_sol)));

		status = (-ENOMEM);
		ioctl->Status = EXT_STATUS_NO_MEMORY;
		ioctl->ResponseLen = 0;
		goto exit_send_ipv6_router_sol;
	}

	/*
	 * Transfer Data from the user's buffer to DMA buffer
	 */
	if (copy_from_user(psend_router_sol,
                           Q64BIT_TO_PTR(ioctl->RequestAdr),
			   sizeof(EXT_SEND_ROUTER_SOL)) != 0) {
		QL4PRINT(QLP2,
		    printk("scsi%d: %s: unable to copy data to user's "
		    "memory area\n", ha->host_no, __func__));

		status = (-EFAULT);
		ioctl->Status = EXT_STATUS_COPY_ERR;
		ioctl->ResponseLen = 0;
		goto exit_send_ipv6_router_sol;
	}

	memset(mbox_cmd, 0, sizeof(mbox_cmd));
	memset(mbox_sts, 0, sizeof(mbox_sts));
	mbox_cmd[0] = MBOX_CMD_SEND_IPV6_ROUTER_SOL;
	mbox_cmd[1] = ioctl->Instance;  /* Primary/Secondary */
	mbox_cmd[2] = psend_router_sol->Flags;
	memcpy(&mbox_cmd[3], &psend_router_sol->Addr.IPAddress[0], 4);
	memcpy(&mbox_cmd[4], &psend_router_sol->Addr.IPAddress[4], 4);
	memcpy(&mbox_cmd[5], &psend_router_sol->Addr.IPAddress[8], 4);
	memcpy(&mbox_cmd[6], &psend_router_sol->Addr.IPAddress[12], 4);


	status = qla4xxx_mailbox_command(ha, 7, 2, &mbox_cmd[0], &mbox_sts[0]);

	ioctl->VendorSpecificStatus[0] = mbox_sts[0];
	ioctl->VendorSpecificStatus[1] = mbox_sts[1];
	
	if (status == QLA_ERROR) {
		QL4PRINT(QLP2, printk("scsi%d: %s: "
					   "MBOX_CMD_SEND_IPV6_ROUTER_SOL "
				           "FAILED w/ status %04X\n",
                                           ha->host_no, __func__, mbox_sts[0]));
		QL4PRINT(QLP2, printk("mbox[1] = 0x%04x\n", mbox_sts[1]));

		ioctl->Status = EXT_STATUS_MAILBOX;
		ioctl->ResponseLen = 0;
		goto exit_send_ipv6_router_sol;
	}

	ioctl->Status = EXT_STATUS_OK;
	ioctl->ResponseLen = mbox_sts[4];

	QL4PRINT(QLP4, printk("scsi%d: %s: Succeeded\n",
		ha->host_no, __func__));

exit_send_ipv6_router_sol:
	qla4xxx_free_ioctl_scrap_mem(ha);
	LEAVE(__func__);
	return(status);
}

/**************************************************************************
 * qla4extioctl_get_hbacnt
 *	This routine retrieves the number of supported HBAs found.
 *
 * Input:
 *	ha    = adapter structure pointer.
 *	ioctl = IOCTL structure pointer.
 *
 * Output:
 *	None
 *
 * Returns:
 *	QLA_SUCCESS = success
 *	QLA_ERROR   = error
 *
 * Context:
 *	Kernel context.
 **************************************************************************/
static int
qla4extioctl_get_hbacnt(EXT_IOCTL_ISCSI *ioctl)
{
	int status = QLA_ERROR;
	EXT_HBA_COUNT	hba_count;


	ENTER(__func__);

	qla4xxx_get_hba_count((uint16_t *) &hba_count.HbaCnt);
	if ((status = copy_to_user(Q64BIT_TO_PTR(ioctl->ResponseAdr),
	    &hba_count, sizeof(hba_count))) != 0) {
		QL4PRINT(QLP2|QLP4,
		    printk("scsi: %s: failed to copy data. struct size=%d.\n",
		    __func__, (uint32_t)sizeof(EXT_IOCTL_ISCSI)));

		status = (-EFAULT);
		ioctl->Status = EXT_STATUS_COPY_ERR;
		goto exit_get_hbacnt;
	}

	QL4PRINT(QLP4, printk("scsi: %s: hbacnt is %d\n",
	    __func__, hba_count.HbaCnt));

	ioctl->Status = EXT_STATUS_OK;

	exit_get_hbacnt:

	LEAVE(__func__);
	return(status);
}

/**************************************************************************
 * qla4intioctl_logout_iscsi
 *	This routine requests that the specified device either login or
 *	logout, depending on the option specified.
 *
 * Input:
 *	ha    = adapter structure pointer.
 *	ioctl = IOCTL structure pointer.
 *
 * Output:
 *	None
 *
 * Returns:
 *	QLA_SUCCESS = success
 *	QLA_ERROR   = error
 *
 * Context:
 *	Kernel context.
 **************************************************************************/
static int
qla4intioctl_logout_iscsi(scsi_qla_host_t *ha, EXT_IOCTL_ISCSI *ioctl)
{
	int			status = 0;
	INT_LOGOUT_ISCSI	logout;

	ENTER(__func__);

	if (ioctl->RequestLen > sizeof(INT_LOGOUT_ISCSI)) {
		QL4PRINT(QLP2,
		    printk("scsi%d: %s: memory area too small\n",
		    ha->host_no, __func__));

		ioctl->Status = EXT_STATUS_BUFFER_TOO_SMALL;
		goto exit_logout;
	}

	/* --- Copy logout structure from user space --- */
	if (copy_from_user((uint8_t *)&logout,
	    Q64BIT_TO_PTR(ioctl->RequestAdr), sizeof(INT_LOGOUT_ISCSI)) != 0) {
		QL4PRINT(QLP2,
		    printk("scsi%d: %s: unable to copy data from "
		    "user's memory area\n", ha->host_no, __func__));

		status = (-EFAULT);
		ioctl->Status = EXT_STATUS_COPY_ERR;
		goto exit_logout;
	}

	/* --- Execute command --- */
	if (logout.Options == INT_DEF_CLOSE_SESSION) {
		if (qla4xxx_logout_device(ha, logout.TargetID,
		    logout.ConnectionID) == QLA_SUCCESS) {
			QL4PRINT(QLP4,
			    printk("scsi%d: %s: CLOSE_SESSION SUCCEEDED!, "
			    "target %d\n", ha->host_no, __func__,
			    logout.TargetID));

			ioctl->Status = EXT_STATUS_OK;
		} else {
			QL4PRINT(QLP2,
			    printk("scsi%d: %s: CLOSE_SESSION FAILED!, "
			    "target %d\n", ha->host_no, __func__,
			    logout.TargetID));

			ioctl->Status = EXT_STATUS_ERR;
		}

	} else if (logout.Options == INT_DEF_RELOGIN_CONNECTION) {
		if (qla4xxx_login_device(ha, logout.TargetID,
		    logout.ConnectionID) == QLA_SUCCESS) {
			QL4PRINT(QLP4,
			    printk("scsi%d: %s: RELOGIN_CONNECTION "
			    "SUCCEEDED!, target %d\n",
			    ha->host_no, __func__, logout.TargetID));

			ioctl->Status = EXT_STATUS_OK;
		} else {
			QL4PRINT(QLP2,
			    printk("scsi%d: %s: RELOGIN_CONNECTION "
			    "FAILED!, target %d\n",
			    ha->host_no, __func__, logout.TargetID));

			ioctl->Status = EXT_STATUS_ERR;
		}

	} else if (logout.Options == INT_DEF_DELETE_DDB) {
		if (qla4xxx_delete_device(ha, logout.TargetID,
		    logout.ConnectionID) == QLA_SUCCESS) {
			QL4PRINT(QLP4,
			    printk("scsi%d: %s: DELETE_DDB "
			    "SUCCEEDED!, target %d\n",
			    ha->host_no, __func__, logout.TargetID));

			ioctl->Status = EXT_STATUS_OK;
		} else {
			QL4PRINT(QLP2,
			    printk("scsi%d: %s: DELETE_DDB FAILED!, "
			    "target %d\n",
			    ha->host_no, __func__, logout.TargetID));

			ioctl->Status = EXT_STATUS_ERR;
		}
	}

exit_logout:
	LEAVE(__func__);
	return(status);
}

/**************************************************************************
 * qla4intioctl_ping
 *	This routine requests that the HBA PING the specified IP Address.
 *
 * Input:
 *	ha    = adapter structure pointer.
 *	ioctl = IOCTL structure pointer.
 *
 * Output:
 *	None
 *
 * Returns:
 *	QLA_SUCCESS = success
 *	QLA_ERROR   = error
 *
 * Context:
 *	Kernel context.
 **************************************************************************/
static int
qla4intioctl_ping(scsi_qla_host_t *ha, EXT_IOCTL_ISCSI *ioctl)
{
	int		status = 0;
	INT_PING	ping;
	uint32_t	mbox_cmd[MBOX_REG_COUNT];
	uint32_t	mbox_sts[MBOX_REG_COUNT];
	static uint8_t  ip_addr_str[40];

	ENTER(__func__);

	/*
	 * Copy user's data to local buffer
	 */
	if (copy_from_user((uint8_t *)&ping, Q64BIT_TO_PTR(ioctl->RequestAdr),
	    sizeof(ping)) != 0) {
		QL4PRINT(QLP2,
		    printk("scsi%d: %s: unable to copy data from "
		    "user's memory area\n", ha->host_no, __func__));
		status = (-EFAULT);
		ioctl->Status = EXT_STATUS_COPY_ERR;
		goto exit_ping;
	}

	/*
	 * Debug Print Statement
	 */
	IPAddr2Str(ha, ping.IPAddr.IPAddress, (uint8_t *) &ip_addr_str);
	QL4PRINT(QLP4,
	    printk("scsi%d: %s: %s\n",
	    ha->host_no, __func__, ip_addr_str));

	/*
	 * Issue Mailbox Command
	 */
	memset(mbox_cmd, 0, sizeof(mbox_cmd));
	memset(mbox_sts, 0, sizeof(mbox_sts));
	mbox_cmd[0] = MBOX_CMD_PING;
	mbox_cmd[1] = ping.Reserved;  /* bitmap - no endian conversion */
	memcpy(&mbox_cmd[2], &ping.IPAddr.IPAddress[0], 4);
	memcpy(&mbox_cmd[3], &ping.IPAddr.IPAddress[4], 4);
	memcpy(&mbox_cmd[4], &ping.IPAddr.IPAddress[8], 4);
	memcpy(&mbox_cmd[5], &ping.IPAddr.IPAddress[12], 4);
	mbox_cmd[6] = (uint32_t) cpu_to_le16(ping.PacketSize);

	if (qla4xxx_mailbox_command(ha, 7, 7, &mbox_cmd[0], &mbox_sts[0]) ==
	    QLA_ERROR) {
		QL4PRINT(QLP2, printk("scsi%d: %s: command failed \n",
		    ha->host_no, __func__));

		ioctl->Status = EXT_STATUS_MAILBOX;
		ioctl->VendorSpecificStatus[0] = mbox_sts[0];
		ioctl->VendorSpecificStatus[1] = mbox_sts[1];
		ioctl->VendorSpecificStatus[2] = mbox_sts[2];
		ioctl->VendorSpecificStatus[3] = mbox_sts[3];
		ioctl->VendorSpecificStatus[4] = mbox_sts[4];
		ioctl->VendorSpecificStatus[5] = mbox_sts[5];
		ioctl->VendorSpecificStatus[6] = mbox_sts[6];
		goto exit_ping;
	}

	ioctl->Status = EXT_STATUS_OK;

exit_ping:
	LEAVE(__func__);
	return(status);
}

/**************************************************************************
 * qla4intioctl_get_flash
 *	This routine reads the requested area of FLASH.
 *
 * Input:
 *	ha    = adapter structure pointer.
 *	ioctl = IOCTL structure pointer.
 *
 * Output:
 *	None
 *
 * Returns:
 *	QLA_SUCCESS = success
 *	QLA_ERROR   = error
 *
 * Context:
 *	Kernel context.
 **************************************************************************/
static int
qla4intioctl_get_flash(scsi_qla_host_t *ha, EXT_IOCTL_ISCSI *ioctl)
{
	int		status = 0;
	dma_buf_t	local_dma_buf;
	INT_ACCESS_FLASH *paccess_flash = NULL;
	uint32_t	mbox_cmd[MBOX_REG_COUNT];
	uint32_t	mbox_sts[MBOX_REG_COUNT];


	ENTER(__func__);

	/*
	 * Allocate local flash buffer
	 */
	if (qla4xxx_get_ioctl_scrap_mem(ha, (void **)&paccess_flash,
	    sizeof(INT_ACCESS_FLASH))) {
		/* not enough memory */
		QL4PRINT(QLP2,
		    printk("scsi%d: %s: inst=%d scrap not big enough. "
		    "size requested=%ld.\n",
		    ha->host_no, __func__, ha->instance,
		    (ulong)sizeof(INT_ACCESS_FLASH)));

		status = (-ENOMEM);
		ioctl->Status = EXT_STATUS_NO_MEMORY;
		ioctl->ResponseLen = 0;
		goto exit_get_flash;
	}

	/*
	 * Copy user's data to local flash buffer
	 */
	if (copy_from_user((uint8_t *)paccess_flash,
	    Q64BIT_TO_PTR(ioctl->RequestAdr), sizeof(INT_ACCESS_FLASH)) != 0) {
		QL4PRINT(QLP2,
		    printk("scsi%d: %s: unable to copy data from user's "
		    "memory area\n", ha->host_no, __func__));

		status = (-EFAULT);
		ioctl->Status = EXT_STATUS_COPY_ERR;
		goto exit_get_flash;
	}

	/*
	 * Allocate DMA memory
	 */
	if (qla4xxx_alloc_dma_memory(ha->pdev, &local_dma_buf,
	    paccess_flash->DataLen) == QLA_ERROR) {
		QL4PRINT(QLP2,
		    printk("scsi%d: %s: unable to allocate dma memory\n",
		    ha->host_no, __func__));

		status = (-ENOMEM);
		ioctl->Status = EXT_STATUS_NO_MEMORY;
		ioctl->ResponseLen = 0;
		goto exit_get_flash;
	}

	QL4PRINT(QLP4, printk("scsi%d: %s: offset=%08x, len=%08x\n",
	    ha->host_no, __func__, paccess_flash->DataOffset,
	    paccess_flash->DataLen));

	/*
	 * Issue Mailbox Command
	 */
	memset(mbox_cmd, 0, sizeof(mbox_cmd));
	memset(mbox_sts, 0, sizeof(mbox_sts));
	mbox_cmd[0] = MBOX_CMD_READ_FLASH;
	mbox_cmd[1] = LSDW(local_dma_buf.phys_addr);
	mbox_cmd[2] = MSDW(local_dma_buf.phys_addr);
	mbox_cmd[3] = paccess_flash->DataOffset;
	mbox_cmd[4] = paccess_flash->DataLen;

	if (qla4xxx_mailbox_command(ha, 5, 2, &mbox_cmd[0], &mbox_sts[0])
	    == QLA_ERROR) {
		QL4PRINT(QLP2, printk("scsi%d: %s: command failed \n",
		    ha->host_no, __func__));

		ioctl->Status = EXT_STATUS_MAILBOX;
		ioctl->VendorSpecificStatus[0] = mbox_sts[0];
		ioctl->VendorSpecificStatus[1] = mbox_sts[1];
		goto exit_get_flash;
	}

	memcpy(&paccess_flash->FlashData[0], local_dma_buf.virt_addr,
	    MIN(local_dma_buf.buf_len, sizeof(paccess_flash->FlashData)));

	/*
	 * Copy local DMA buffer to user's response data area
	 */
	if (copy_to_user(Q64BIT_TO_PTR(ioctl->ResponseAdr),
	    paccess_flash, sizeof(*paccess_flash)) != 0) {
		QL4PRINT(QLP2,
		    printk("scsi%d: %s: unable to copy data to user's "
		    "memory area\n", ha->host_no, __func__));

		status = (-EFAULT);
		ioctl->Status = EXT_STATUS_COPY_ERR;
		goto exit_get_flash;
	}

	ioctl->Status = EXT_STATUS_OK;
	ioctl->ResponseLen = paccess_flash->DataLen;

	QL4PRINT(QLP10,
	    printk("INT_ACCESS_FLASH buffer (1st 60h bytes only):\n"));
	qla4xxx_dump_bytes(QLP10, paccess_flash, 0x60);

exit_get_flash:
	/*
	 * Free Memory
	 */

	if (local_dma_buf.virt_addr)
		qla4xxx_free_dma_memory(ha->pdev, &local_dma_buf);

	qla4xxx_free_ioctl_scrap_mem(ha);
        LEAVE(__func__);
	return(status);
}

/**************************************************************************
 * qla4intioctl_get_core_dump
 *	This routine reads a block of data for a core dump.
 *
 * Input:
 *	ha    = adapter structure pointer.
 *	ioctl = IOCTL structure pointer.
 *
 * Output:
 *	None
 *
 * Returns:
 *	QLA_SUCCESS = success
 *	QLA_ERROR   = error
 *
 * Context:
 *	Kernel context.
 **************************************************************************/
static int
qla4intioctl_get_core_dump(scsi_qla_host_t *ha, EXT_IOCTL_ISCSI *ioctl)
{
	int		status = 0;
	INT_ACCESS_CORE_DUMP *pcore_dump = NULL;
	uint8_t		*dump_image = ha->core_dump_mem;

	ENTER(__func__);

	/*
	 * Allocate local core dump buffer
	 */
	if (qla4xxx_get_ioctl_scrap_mem(ha, (void **)&pcore_dump,
	    ioctl->RequestLen)) {
		/* not enough memory */
		QL4PRINT(QLP2,
		    printk("scsi%d: %s: inst=%d scrap not big enough. "
		    "size requested=%d.\n",
		    ha->host_no, __func__, ha->instance,
		    ioctl->RequestLen));

		status = (-ENOMEM);
		ioctl->Status = EXT_STATUS_NO_MEMORY;
		ioctl->ResponseLen = 0;
		goto exit_get_core_dump;
	}

	/*
	 * Copy user's data to local core dump buffer
	 */
	if (copy_from_user((uint8_t *)pcore_dump,
	    Q64BIT_TO_PTR(ioctl->RequestAdr), ioctl->RequestLen) != 0) {
		QL4PRINT(QLP2,
		    printk("scsi%d: %s: unable to copy data from user's "
		    "memory area\n", ha->host_no, __func__));

		status = (-EFAULT);
		ioctl->Status = EXT_STATUS_COPY_ERR;
		goto exit_get_core_dump;
	}

	QL4PRINT(QLP25|QLP4, printk("scsi%d: %s: offset=%08x, len=%08x, "
				    "max=%08lx/%08lx\n", ha->host_no, __func__,
				    pcore_dump->Offset, pcore_dump->DataLen,
				    (u_long)DUMP_IMAGE_SIZE, (u_long)QLA_IOCTL_DUMP_IMAGE_SIZE));

	if (pcore_dump->Offset + pcore_dump->DataLen > DUMP_IMAGE_SIZE) {
		QL4PRINT(QLP25|QLP4, printk("scsi%d: %s: LastBlock\n",
		    ha->host_no, __func__));
		pcore_dump->LastBlockFlag = 1;
	}
	if (pcore_dump->Offset + pcore_dump->DataLen > QLA_IOCTL_DUMP_IMAGE_SIZE) {
		QL4PRINT(QLP2,
		    printk("scsi%d: %s: data requested beyond buffer "
		    "memory area\n", ha->host_no, __func__));

		status = (-EFAULT);
		ioctl->Status = EXT_STATUS_DATA_OVERRUN;
		goto exit_get_core_dump;
	}

	/*
	 * Retrieve Core Dump Data
	 */

	if (pcore_dump->Offset == 0) {
		if (pcore_dump->DataLen == DUMP_IMAGE_HEADER_SIZE) {
			QL4PRINT(QLP25|QLP4, printk("scsi%d: %s: calling qla4xxx_build_dump_image_header\n",
			    ha->host_no, __func__));
			qla4xxx_build_dump_image_header(ha, &pcore_dump->Data[0]);
		}

		if (pcore_dump->DataLen > DUMP_IMAGE_HEADER_SIZE) {
			/*
			 * Retrieve Core Dump Data
			 */
			if (dump_image == NULL) {
				// should never get here, allocated at init  ...
				QL4PRINT(QLP2,
				    printk("scsi%d: %s: unable to allocate "
				    "dump image memory area\n",
				    ha->host_no, __func__));

				status = (-EFAULT);
				ioctl->Status = EXT_STATUS_ERR;
				goto exit_get_core_dump;
			}
			memset (dump_image, 0, QLA_IOCTL_DUMP_IMAGE_SIZE);

			QL4PRINT(QLP25|QLP4, printk("scsi%d: %s: calling "
				"qla4xxx_capture_dump_image\n",
				ha->host_no, __func__));
			if (qla4xxx_capture_dump_image(ha, dump_image,
				QLA_IOCTL_DUMP_IMAGE_SIZE) == QLA_ERROR) {
				QL4PRINT(QLP2,
				    printk("scsi%d: %s: command failed \n",
				    ha->host_no, __func__));

				ioctl->Status = EXT_STATUS_ERR;
				goto exit_get_core_dump;
			}
		}
	}

	if (pcore_dump->DataLen > DUMP_IMAGE_HEADER_SIZE && dump_image) {
		memcpy(&pcore_dump->Data[0],
		       dump_image + pcore_dump->Offset,
		       pcore_dump->DataLen);
	}

	/*
	 * Copy local DMA buffer to user's response data area
	 */
	if (copy_to_user(Q64BIT_TO_PTR(ioctl->ResponseAdr),
	    pcore_dump, ioctl->ResponseLen) != 0) {
		QL4PRINT(QLP2,
		    printk("scsi%d: %s: unable to copy data to user's "
		    "memory area\n", ha->host_no, __func__));

		status = (-EFAULT);
		ioctl->Status = EXT_STATUS_COPY_ERR;
		goto exit_get_core_dump;
	}

	ioctl->Status = EXT_STATUS_OK;
	ioctl->ResponseLen = pcore_dump->DataLen;

//	QL4PRINT(QLP25|QLP10, printk("INT_ACCESS_CORE_DUMP buffer: MAX 60h\n"));
//	qla4xxx_dump_bytes(QLP25|QLP10, &pcore_dump->Data[0],
//		(uint32_t) min((uint32_t)0x00000060, (uint32_t)pcore_dump->DataLen));

exit_get_core_dump:
	qla4xxx_free_ioctl_scrap_mem(ha);
	LEAVE(__func__);
	return(status);
}


/**************************************************************************
 * qla4intioctl_get_driver_debug_level
 *	This routine retrieves the driver's debug print level.
 *
 * Input:
 *	ha    = adapter structure pointer.
 *	ioctl = IOCTL structure pointer.
 *
 * Output:
 *	None
 *
 * Returns:
 *	QLA_SUCCESS = success
 *	QLA_ERROR   = error
 *
 * Context:
 *	Kernel context.
 **************************************************************************/
static int
qla4intioctl_get_driver_debug_level(scsi_qla_host_t *ha, EXT_IOCTL_ISCSI *ioctl)
{
	int		status = 0;
	uint32_t	dbg_level;


	ENTER(__func__);

	if (qla4xxx_get_debug_level(&dbg_level) == QLA_ERROR) {
		QL4PRINT(QLP2,
		    printk("scsi%d: %s: unable to set debug level, "
		    "debug driver not loaded!\n",
		    ha->host_no, __func__));

		ioctl->Status = EXT_STATUS_ERR;
		goto exit_get_driver_debug_level;
	}

	if (copy_to_user(Q64BIT_TO_PTR(ioctl->ResponseAdr), &dbg_level,
	    sizeof(dbg_level)) != 0) {
		QL4PRINT(QLP2|QLP4,
		    printk("scsi%d: %s: failed to copy data\n",
		    ha->host_no, __func__));

		status = (-EFAULT);
		ioctl->Status = EXT_STATUS_COPY_ERR;
		goto exit_get_driver_debug_level;
	}

	QL4PRINT(QLP4,
	    printk("scsi%d: %s: debug level is %04x\n",
	    ha->host_no, __func__, dbg_level));

	ioctl->Status = EXT_STATUS_OK;

	exit_get_driver_debug_level:
	LEAVE(__func__);
	return(status);
}

/**************************************************************************
 * qla4intioctl_get_host_no
 *	This routine retrieves the host number for the specified adapter
 *	instance.
 *
 * Input:
 *	ha    = adapter structure pointer.
 *	ioctl = IOCTL structure pointer.
 *
 * Output:
 *	None
 *
 * Returns:
 *	QLA_SUCCESS = success
 *	QLA_ERROR   = error
 *
 * Context:
 *	Kernel context.
 **************************************************************************/
static int
qla4intioctl_get_host_no(scsi_qla_host_t *ha, EXT_IOCTL_ISCSI *ioctl)
{
	int	status = 0;

	ENTER(__func__);

	QL4PRINT(QLP4, printk("scsi%d: %s: for instance %d\n",
	    ha->host_no, __func__, ha->instance));

	if (copy_to_user(Q64BIT_TO_PTR(ioctl->ResponseAdr), &(ha->host_no),
	    sizeof(ha->host_no)) != 0) {
		QL4PRINT(QLP2,
		    printk("scsi%d: %s: failed to copy data\n",
		    ha->host_no, __func__));

		status = (-EFAULT);
		ioctl->Status = EXT_STATUS_COPY_ERR;
	} else {
		ioctl->Status = EXT_STATUS_OK;
	}

	LEAVE(__func__);
	return(status);
}

/**************************************************************************
 * qla4intioctl_get_data
 *	This routine calls get data IOCTLs based on the IOCTL Sub Code.
 *
 * Input:
 *	ha    = adapter structure pointer.
 *	ioctl = IOCTL structure pointer.
 *
 * Output:
 *	None
 *
 * Returns:
 *	QLA_SUCCESS = success
 *	QLA_ERROR   = error
 *    	-EINVAL     = if the command is invalid
 *
 * Context:
 *	Kernel context.
 **************************************************************************/
static int
qla4intioctl_get_data(scsi_qla_host_t *ha, EXT_IOCTL_ISCSI *ioctl)
{
	switch (ioctl->SubCode) {
	case INT_SC_GET_FLASH:
		return(qla4intioctl_get_flash(ha, ioctl));

	case INT_SC_GET_CORE_DUMP:
		return(qla4intioctl_get_core_dump(ha, ioctl));

	case INT_SC_GET_DRIVER_DEBUG_LEVEL:
		return(qla4intioctl_get_driver_debug_level(ha, ioctl));

	case INT_SC_GET_HOST_NO:
		return(qla4intioctl_get_host_no(ha, ioctl));

	default:
		QL4PRINT(QLP2,
		    printk("scsi%d: %s: unsupported internal get data "
		    "sub-command code (%X)\n",
		    ha->host_no, __func__, ioctl->SubCode));

		ioctl->Status = EXT_STATUS_UNSUPPORTED_SUBCODE;
		return (-EINVAL);
	}
}

/**************************************************************************
 * qla4intioctl_set_flash
 *	This routine writes the requested area of FLASH.
 *
 * Input:
 *	ha    = adapter structure pointer.
 *	ioctl = IOCTL structure pointer.
 *
 * Output:
 *	None
 *
 * Returns:
 *	QLA_SUCCESS = success
 *	QLA_ERROR   = error
 *
 * Context:
 *	Kernel context.
 **************************************************************************/
static int
qla4intioctl_set_flash(scsi_qla_host_t *ha, EXT_IOCTL_ISCSI *ioctl)
{
	int			status = 0;
	dma_buf_t		local_dma_buf = { 0 };
	INT_ACCESS_FLASH	*paccess_flash = NULL;
	uint32_t		mbox_cmd[MBOX_REG_COUNT];
	uint32_t		mbox_sts[MBOX_REG_COUNT];


	ENTER(__func__);

	/*
	 * Allocate local flash buffer
	 */
	if (qla4xxx_get_ioctl_scrap_mem(ha, (void **)&paccess_flash,
	    sizeof(INT_ACCESS_FLASH))) {
		/* not enough memory */
		QL4PRINT(QLP2,
		    printk("scsi%d: %s: inst=%d scrap not big enough. "
		    "size requested=%ld.\n",
		    ha->host_no, __func__, ha->instance,
		    (ulong)sizeof(INT_ACCESS_FLASH)));

		status = (-ENOMEM);
		ioctl->Status = EXT_STATUS_NO_MEMORY;
		ioctl->ResponseLen = 0;
		goto exit_set_flash;
	}

	/*
	 * Copy user's data to local DMA buffer
	 */
	if (copy_from_user((uint8_t *)paccess_flash,
	    Q64BIT_TO_PTR(ioctl->RequestAdr), sizeof(INT_ACCESS_FLASH)) != 0) {
		QL4PRINT(QLP2,
		    printk("scsi%d: %s: unable to copy data from user's "
		    "memory area\n", ha->host_no, __func__));

		status = (-EFAULT);
		ioctl->Status = EXT_STATUS_COPY_ERR;
		goto exit_set_flash;
	}

	/*
	 * Allocate DMA memory
	 */
	if (qla4xxx_alloc_dma_memory(ha->pdev, &local_dma_buf,
	    paccess_flash->DataLen) == QLA_ERROR) {
		QL4PRINT(QLP2,
		    printk("scsi%d: %s: unable to allocate dma memory\n",
		    ha->host_no, __func__));

		status = (-ENOMEM);
		ioctl->Status = EXT_STATUS_NO_MEMORY;
		ioctl->ResponseLen = 0;
		goto exit_set_flash;
	}

	memcpy(local_dma_buf.virt_addr, &paccess_flash->FlashData[0],
	    MIN(local_dma_buf.buf_len, sizeof(paccess_flash->FlashData)));

	QL4PRINT(QLP4,
	    printk("scsi%d: %s: offset=%08x, len=%08x\n",
	    ha->host_no, __func__,
	    paccess_flash->DataOffset, paccess_flash->DataLen));

	/*
	 * Issue Mailbox Command
	 */
	memset(mbox_cmd, 0, sizeof(mbox_cmd));
	memset(mbox_sts, 0, sizeof(mbox_sts));
	mbox_cmd[0] = MBOX_CMD_WRITE_FLASH;
	mbox_cmd[1] = LSDW(local_dma_buf.phys_addr);
	mbox_cmd[2] = MSDW(local_dma_buf.phys_addr);
	mbox_cmd[3] = paccess_flash->DataOffset;
	mbox_cmd[4] = paccess_flash->DataLen;
	mbox_cmd[5] = paccess_flash->Options;

	if (qla4xxx_mailbox_command(ha, 6, 2, &mbox_cmd[0], &mbox_sts[0])
	    == QLA_ERROR) {
		QL4PRINT(QLP2, printk("scsi%d: %s: command failed \n",
		    ha->host_no, __func__));

		ioctl->Status = EXT_STATUS_MAILBOX;
		ioctl->VendorSpecificStatus[0] = mbox_sts[0];
		ioctl->VendorSpecificStatus[1] = mbox_sts[1];
		goto exit_set_flash;
	}

	ioctl->Status = EXT_STATUS_OK;
	ioctl->ResponseLen = paccess_flash->DataLen;

	QL4PRINT(QLP10,
	    printk("scsi%d): INT_ACCESS_FLASH buffer (1st 60h bytes only:\n",
	   ha->host_no));
	qla4xxx_dump_bytes(QLP10, local_dma_buf.virt_addr, 0x60);

exit_set_flash:
	/*
	 * Free Memory
	 */
	if (local_dma_buf.virt_addr)
		qla4xxx_free_dma_memory(ha->pdev, &local_dma_buf);

	qla4xxx_free_ioctl_scrap_mem(ha);
	LEAVE(__func__);
	return(status);
}

/**************************************************************************
 * qla4intioctl_set_driver_debug_level
 *	This routine sets the driver's debug print level.
 *
 * Input:
 *	ha    = adapter structure pointer.
 *	ioctl = IOCTL structure pointer.
 *
 * Output:
 *	None
 *
 * Returns:
 *	QLA_SUCCESS = success
 *	QLA_ERROR   = error
 *
 * Context:
 *	Kernel context.
 **************************************************************************/
static int
qla4intioctl_set_driver_debug_level(scsi_qla_host_t *ha, EXT_IOCTL_ISCSI *ioctl)
{
	int		status = 0;
	uint32_t	dbg_level;

	ENTER(__func__);

	if (copy_from_user(&dbg_level, Q64BIT_TO_PTR(ioctl->RequestAdr),
	    sizeof(dbg_level)) != 0) {
		QL4PRINT(QLP2,
		    printk("scsi%d: %s: failed to copy data\n",
		    ha->host_no, __func__));

		status = (-EFAULT);
		ioctl->Status = EXT_STATUS_COPY_ERR;
		goto exit_set_driver_debug_level;
	}

	if (qla4xxx_set_debug_level(dbg_level) == QLA_ERROR) {
		QL4PRINT(QLP2,
		    printk("scsi%d: %s: unable to set debug level, "
		    "debug driver not loaded!\n",
		    ha->host_no, __func__));

		ioctl->Status = EXT_STATUS_ERR;
		goto exit_set_driver_debug_level;
	}

	QL4PRINT(QLP4,
	    printk("scsi%d: %s: debug level set to 0x%04X\n",
	    ha->host_no, __func__, dbg_level));

	ioctl->Status = EXT_STATUS_OK;

exit_set_driver_debug_level:
	LEAVE(__func__);
	return(status);
}

/**************************************************************************
 * qla4intioctl_set_data
 *	This routine calls set data IOCTLs based on the IOCTL Sub Code.
 *
 * Input:
 *	ha    = adapter structure pointer.
 *	ioctl = IOCTL structure pointer.
 *
 * Output:
 *	None
 *
 * Returns:
 *	QLA_SUCCESS = success
 *	QLA_ERROR   = error
 *    	-EINVAL     = if the command is invalid
 *
 * Context:
 *	Kernel context.
 **************************************************************************/
static int
qla4intioctl_set_data(scsi_qla_host_t *ha, EXT_IOCTL_ISCSI *ioctl)
{
	switch (ioctl->SubCode) {
	case INT_SC_SET_FLASH:
		return(qla4intioctl_set_flash(ha, ioctl));

	case INT_SC_SET_DRIVER_DEBUG_LEVEL:
		return(qla4intioctl_set_driver_debug_level(ha, ioctl));

	default:
		QL4PRINT(QLP2,
		    printk("scsi%d: %s: unsupported internal set data "
		    "sub-command code (%X)\n",
		    ha->host_no, __func__, ioctl->SubCode));

		ioctl->Status = EXT_STATUS_UNSUPPORTED_SUBCODE;
		return (-EINVAL);
	}
}

/**************************************************************************
 * qla4intioctl_hba_reset
 *	This routine resets the specified HBA.
 *
 * Input:
 *	ha    = adapter structure pointer.
 *	ioctl = IOCTL structure pointer.
 *
 * Output:
 *	None
 *
 * Returns:
 *	QLA_SUCCESS = success
 *	QLA_ERROR   = error
 *
 * Context:
 *	Kernel context.
 **************************************************************************/
static int
qla4intioctl_hba_reset(scsi_qla_host_t *ha, EXT_IOCTL_ISCSI *ioctl)
{
	int	status = 0;
	u_long	wait_count;

	ENTER(__func__);

	switch (ioctl->SubCode) {
	case INT_SC_HBA_RESET:
	case INT_SC_FIRMWARE_RESET:
		set_bit(DPC_RESET_HA_DESTROY_DDB_LIST, &ha->dpc_flags);

		/* Wait a fixed amount of time for reset to complete */
		wait_count = jiffies + ADAPTER_RESET_TOV * HZ;
		while (test_bit(DPC_RESET_HA_DESTROY_DDB_LIST,
		    &ha->dpc_flags) != 0) {
			if (time_after_eq(jiffies, wait_count))
				break;

			/* wait for 1 second */
			set_current_state(TASK_UNINTERRUPTIBLE);
			schedule_timeout(1*HZ);
		}

		if (test_bit(AF_ONLINE, &ha->flags)) {
			QL4PRINT(QLP4, printk("scsi%d: %s: Succeeded\n",
			    ha->host_no, __func__));
			ioctl->Status = EXT_STATUS_OK;
		} else {
			QL4PRINT(QLP2, printk("scsi%d: %s: FAILED\n",
			    ha->host_no, __func__));
			ioctl->Status = EXT_STATUS_ERR;
			status = (-EFAULT);
		}

		break;

	case INT_SC_TARGET_WARM_RESET:
	case INT_SC_LUN_RESET:
	default:
		QL4PRINT(QLP2, printk("scsi%d: %s: not supported.\n",
		    ha->host_no, __func__));
		ioctl->Status = EXT_STATUS_UNSUPPORTED_SUBCODE;
		break;
	}

	LEAVE(__func__);
	return(status);
}

/**************************************************************************
 * qla4intioctl_copy_fw_flash
 *	This routine requests copying the FW image in FLASH from primary-to-
 *	secondary or secondary-to-primary.
 *
 * Input:
 *	ha    = adapter structure pointer.
 *	ioctl = IOCTL structure pointer.
 *
 * Output:
 *	None
 *
 * Returns:
 *	QLA_SUCCESS = success
 *	QLA_ERROR   = error
 *
 * Context:
 *	Kernel context.
 **************************************************************************/
static int
qla4intioctl_copy_fw_flash(scsi_qla_host_t *ha, EXT_IOCTL_ISCSI *ioctl)
{
	int			status = 0;
	INT_COPY_FW_FLASH	copy_flash;
	uint32_t		mbox_cmd[MBOX_REG_COUNT];
	uint32_t		mbox_sts[MBOX_REG_COUNT];


	ENTER(__func__);
	QL4PRINT(QLP3, printk("scsi%d: %s:\n", ha->host_no, __func__));

	if (copy_from_user((uint8_t *)&copy_flash,
	    Q64BIT_TO_PTR(ioctl->RequestAdr), ioctl->RequestLen) != 0) {
		QL4PRINT(QLP2,
		    printk("scsi%d: %s: unable to copy data from user's "
		    "memory area\n", ha->host_no, __func__));

		status = (-EFAULT);
		ioctl->Status = EXT_STATUS_COPY_ERR;
		goto exit_copy_flash;
	}

	memset(mbox_cmd, 0, sizeof(mbox_cmd));
	memset(mbox_sts, 0, sizeof(mbox_sts));
	mbox_cmd[0] = MBOX_CMD_COPY_FLASH;
	mbox_cmd[1] = copy_flash.Options;

	if (qla4xxx_mailbox_command(ha, 2, 2, &mbox_cmd[0], &mbox_sts[0])
	    == QLA_SUCCESS) {
		QL4PRINT(QLP4, printk("scsi%d: %s: Succeeded\n",
		    ha->host_no, __func__));

		ioctl->Status = EXT_STATUS_OK;
	} else {
		QL4PRINT(QLP4, printk("scsi%d: %s: FAILED\n",
		    ha->host_no, __func__));

		ioctl->Status = EXT_STATUS_MAILBOX;
		ioctl->VendorSpecificStatus[0] = mbox_sts[0];
		ioctl->VendorSpecificStatus[1] = mbox_sts[1];
	}

exit_copy_flash:
	LEAVE(__func__);
	return(status);
}

/**************************************************************************
 * qla4xxx_iocb_pass_done
 *	This routine resets the ioctl progress flag and wakes up the ioctl
 * 	completion semaphore.
 *
 * Input:
 *	ha    = adapter structure pointer.
 *   	sts_entry - pointer to passthru status buffer.
 *
 * Output:
 *	None
 *
 * Returns:
 *	None
 *
 * Context:
 *	Interrupt context.
 **************************************************************************/
void
qla4xxx_iocb_pass_done(scsi_qla_host_t *ha, PASSTHRU_STATUS_ENTRY *sts_entry)
{
	INT_IOCB_PASSTHRU *iocb;

	ENTER(__func__);

	/* --- Copy passthru status buffer to iocb passthru buffer ---*/
	iocb = (INT_IOCB_PASSTHRU *)(ulong)le32_to_cpu(sts_entry->handle);
	memcpy(iocb->IOCBStatusBuffer, sts_entry,
	    MIN(sizeof(iocb->IOCBStatusBuffer), sizeof(*sts_entry)));

	/* --- Reset IOCTL flags and wakeup semaphore.
	 *     But first check to see if IOCTL has already
	 *     timed out because we don't want to get the
	 *     up/down semaphore counters off.             --- */
	if (ha->ioctl_iocb_pass_in_progress == 1) {
		ha->ioctl_iocb_pass_in_progress = 0;
		ha->ioctl_tov = 0;

		QL4PRINT(QLP10, printk("%s: UP count=%d\n",
		    __func__, atomic_read(&ha->ioctl_cmpl_sem.count)));
		up(&ha->ioctl_cmpl_sem);
	}

	LEAVE(__func__);
	return;
}


#ifndef QLA4000
/**************************************************************************
 * qla4intioctl_iocb_passthru
 *	This routine
 *	
 *
 * Input:
 *	ha    = adapter structure pointer.
 *	ioctl = IOCTL structure pointer.
 *
 * Output:
 *	None
 *
 * Returns:
 *	QLA_SUCCESS = success
 *	QLA_ERROR   = error
 *
 * Context:
 *	Kernel context.
 **************************************************************************/
static int
qla4intioctl_iocb_passthru(scsi_qla_host_t *ha, EXT_IOCTL_ISCSI *ioctl)
{
	int			status = 0;
	INT_IOCB_PASSTHRU	*iocb;
	INT_IOCB_PASSTHRU	*iocb_phys;
	PASSTHRU0_ENTRY		*passthru_entry;
	unsigned long		flags;
	DATA_SEG_A64		*data_seg;


	ENTER(__func__);
	QL4PRINT(QLP3, printk("scsi%d: %s:\n", ha->host_no, __func__));

	/* --- Use internal DMA buffer for iocb structure --- */
#if 0 && defined(__VMKERNEL_MODULE__)
	spin_lock(&ha->dma_buf_spinlock);
#else
	down(&ha->dma_buf_sem);
#endif

	if (ha->dma_buf.buf_len < sizeof(*iocb))
		qla4xxx_resize_dma_buf(ha, &ha->dma_buf, sizeof(*iocb));

	if (!ha->dma_buf.virt_addr || ha->dma_buf.buf_len < sizeof(*iocb)) {
		QL4PRINT(QLP2,
		    printk("scsi%d: %s: dma buffer inaccessible.\n",
				      ha->host_no, __func__));
		ioctl->Status = EXT_STATUS_BUFFER_TOO_SMALL;
		goto exit_iocb_passthru;
	}

	iocb = (INT_IOCB_PASSTHRU *) ha->dma_buf.virt_addr;
	iocb_phys = (INT_IOCB_PASSTHRU *)(unsigned long)ha->dma_buf.phys_addr;

	/* --- Copy IOCB_PASSTHRU structure from user space --- */
	if (copy_from_user((uint8_t *)iocb, Q64BIT_TO_PTR(ioctl->RequestAdr),
	    ioctl->RequestLen) != 0) {
		QL4PRINT(QLP2,
		    printk("scsi%d: %s: unable to copy data from user's "
		    "memory area\n", ha->host_no, __func__));

		status = (-EFAULT);
		ioctl->Status = EXT_STATUS_COPY_ERR;
		goto exit_iocb_passthru;
	}

	if ((iocb->IOCBCmdBuffer[0x00] == 0x3A) &&
	    (iocb->IOCBCmdBuffer[0x0A] == 0x10) &&
	    (iocb->IOCBCmdBuffer[0x0B] == 0x80) &&
	    (iocb->SendData[0x0C] == 0x81) &&
	    (iocb->SendData[0x0D] == 0x4F)) {
		// ok to process command, proceed ...
	} else {
		QL4PRINT(QLP2, printk("scsi%d: %s: unable to process command. "
                                      "Did not pass secure I/O boundary check.\n",
                                      ha->host_no, __func__));
		QL4PRINT(QLP2, printk("IOCBCmdBuffer[0x00] = 0x%x, expecting 0x3A\n", iocb->IOCBCmdBuffer[0x00]));
		QL4PRINT(QLP2, printk("IOCBCmdBuffer[0x0A] = 0x%x, expecting 0x10\n", iocb->IOCBCmdBuffer[0x0A]));
		QL4PRINT(QLP2, printk("IOCBCmdBuffer[0x0B] = 0x%x, expecting 0x80\n", iocb->IOCBCmdBuffer[0x0B]));
		QL4PRINT(QLP2, printk("SendData[0x0C] = 0x%x, expecting 0x81\n", iocb->SendData[0x0C]));
		QL4PRINT(QLP2, printk("SendData[0x0D] = 0x%x, expecting 0x4F\n", iocb->SendData[0x0D]));
		status = (-EFAULT);
		ioctl->Status = EXT_STATUS_INVALID_PARAM;
		goto exit_iocb_passthru;
	}

	/* --- Get pointer to the passthru queue entry --- */
	spin_lock_irqsave(&ha->hardware_lock, flags);
	if (qla4xxx_get_req_pkt(ha, (QUEUE_ENTRY **) &passthru_entry)
	    != QLA_SUCCESS) {
		QL4PRINT(QLP2,
		    printk("scsi%d: %s: request queue full, try again later\n",
		    ha->host_no, __func__));

		ioctl->Status = EXT_STATUS_HBA_QUEUE_FULL;
		spin_unlock_irqrestore(&ha->hardware_lock, flags);
		goto exit_iocb_passthru;
	}

	/* --- Fill in passthru queue entry --- */
	if (iocb->SendDMAOffset) {
		data_seg = (DATA_SEG_A64 *)(iocb->IOCBCmdBuffer +
		    iocb->SendDMAOffset);
		data_seg->base.addrHigh =
		    cpu_to_le32(MSDW(iocb_phys->SendData));
		data_seg->base.addrLow  =
		    cpu_to_le32(LSDW(iocb_phys->SendData));
	}

	if (iocb->RspDMAOffset) {
		data_seg = (DATA_SEG_A64 *)(iocb->IOCBCmdBuffer +
		    iocb->RspDMAOffset);
		data_seg->base.addrHigh =
		    cpu_to_le32(MSDW(iocb_phys->RspData));
		data_seg->base.addrLow  =
		    cpu_to_le32(LSDW(iocb_phys->RspData));
	}

	memcpy(passthru_entry, iocb->IOCBCmdBuffer,
	    MIN(sizeof(*passthru_entry), sizeof(iocb->IOCBCmdBuffer)));
	passthru_entry->handle = (uint32_t) (unsigned long) iocb;
	passthru_entry->hdr.systemDefined = SD_PASSTHRU_IOCB;

	if (passthru_entry->hdr.entryType != ET_PASSTHRU0)
		passthru_entry->timeout = MBOX_TOV;

	QL4PRINT(QLP10,
	    printk(KERN_INFO
	    "scsi%d: Passthru0 IOCB type %x count %x In (%x) %p\n",
	    ha->host_no, passthru_entry->hdr.entryType,
	    passthru_entry->hdr.entryCount, ha->request_in, passthru_entry));

	QL4PRINT(QLP10,
	    printk(KERN_INFO "scsi%d: Dump Passthru entry %p: \n",
	    ha->host_no, passthru_entry));
	qla4xxx_dump_bytes(QLP10, passthru_entry, sizeof(*passthru_entry));

	/* ---- Prepare for receiving completion ---- */
	ha->ioctl_iocb_pass_in_progress = 1;
	ha->ioctl_tov = passthru_entry->timeout * HZ;
	qla4xxx_ioctl_sem_init(ha);

	/* ---- Send command to adapter ---- */
	ha->ioctl_cmpl_timer.expires = jiffies + ha->ioctl_tov;
	add_timer(&ha->ioctl_cmpl_timer);

	WRT_REG_DWORD(&ha->reg->requestQueueInPointer, ha->request_in);
	PCI_POSTING(&ha->reg->requestQueueInPointer);
	spin_unlock_irqrestore(&ha->hardware_lock, flags);

	QL4PRINT(QLP10,
	    printk("%s: DOWN count=%d\n",
	    __func__, atomic_read(&ha->ioctl_cmpl_sem.count)));
	down(&ha->ioctl_cmpl_sem);

	/*******************************************************
	 *						       *
	 *             Passthru Completion                     *
	 *						       *
	 *******************************************************/
	del_timer(&ha->ioctl_cmpl_timer);

	/* ---- Check for timeout --- */
	if (ha->ioctl_iocb_pass_in_progress == 1) {
		QL4PRINT(QLP2,
		    printk("scsi%d: %s: ERROR = command timeout.\n",
		    ha->host_no, __func__));

		ha->ioctl_iocb_pass_in_progress = 0;
		ioctl->Status = EXT_STATUS_ERR;
		goto exit_iocb_passthru;
	}

	/* ---- Copy IOCB Passthru structure with updated status buffer
	 *      to user space ---- */

	iocb->RspDataLen -= passthru_entry->residual;

	if (copy_to_user(Q64BIT_TO_PTR(ioctl->ResponseAdr),
	    iocb, sizeof(INT_IOCB_PASSTHRU)) != 0) {
		QL4PRINT(QLP2,
		    printk("scsi%d: %s: unable to copy passthru struct "
		    "to user's memory area.\n",
		    ha->host_no, __func__));

		status = (-EFAULT);
		ioctl->Status = EXT_STATUS_COPY_ERR;
		goto exit_iocb_passthru;
	}

	QL4PRINT(QLP10, printk("Dump iocb structure (OUT)\n"));
	qla4xxx_print_iocb_passthru(QLP10, ha, iocb);

	QL4PRINT(QLP4, printk("scsi%d: %s: Succeeded\n",
	    ha->host_no, __func__));

	ioctl->Status = EXT_STATUS_OK;

exit_iocb_passthru:
#if 0 && defined(__VMKERNEL_MODULE__)
	spin_unlock(&ha->dma_buf_spinlock);
#else
	up(&ha->dma_buf_sem);
#endif

	LEAVE(__func__);
	return(status);
}
#endif


 /**************************************************************************
 * qla4intioctl_restore_factory_defaults
 *	This routine restores factory defaults of the adapter.
 *
 * Input:
 *	ha    = adapter structure pointer.
 *	ioctl = IOCTL structure pointer.
 *
 * Output:
 *	None
 *
 * Returns:
 *	QLA_SUCCESS = success
 *	QLA_ERROR   = error
 *
 * Context:
 *	Kernel context.
 **************************************************************************/
static int
qla4intioctl_restore_factory_defaults(scsi_qla_host_t *ha, EXT_IOCTL_ISCSI *ioctl)
{
	int status = 0;
	INT_RESTORE_FACTORY_DEFAULTS defaults;

	ENTER(__func__);
	QL4PRINT(QLP4,
	    printk("scsi%d: %s: inst %d entered.\n",
	    ha->host_no, __func__, ha->instance));

	if (ioctl->RequestLen > sizeof(INT_RESTORE_FACTORY_DEFAULTS)) {
		QL4PRINT(QLP2|QLP4,
		    printk("scsi%d: %s: memory area too small\n",
		    ha->host_no, __func__));

		ioctl->Status = EXT_STATUS_BUFFER_TOO_SMALL;
		goto exit_defaults;
	}

	/* --- Copy logout structure from user space --- */
	if ((status = copy_from_user((void *)&defaults,
	    Q64BIT_TO_PTR(ioctl->RequestAdr), sizeof(INT_RESTORE_FACTORY_DEFAULTS))) != 0) {
		QL4PRINT(QLP2|QLP4,
		    printk("scsi%d: %s: unable to copy data from "
		    "user's memory area\n", ha->host_no, __func__));

		ioctl->Status = EXT_STATUS_COPY_ERR;
		goto exit_defaults;
	}

	/* --- Execute command --- */
	if (qla4xxx_restore_factory_defaults(ha, defaults.BlockMask,
	    defaults.IFCBMask1, defaults.IFCBMask2) == QLA_SUCCESS) {
		QL4PRINT(QLP4,
		    printk("scsi%d: %s: RESTORE_FACTORY_DEFAULTS SUCCEEDED!\n",
			   ha->host_no, __func__));

		ioctl->Status = EXT_STATUS_OK;
	} else {
		QL4PRINT(QLP2|QLP4,
		    printk("scsi%d: %s: RESTORE_FACTORY_DEFAULTS FAILED\n",
		    ha->host_no, __func__));

		ioctl->Status = EXT_STATUS_ERR;
	}

exit_defaults:
	QL4PRINT(QLP12,
	    printk("scsi%d: %s: inst %d exiting.\n",
	    ha->host_no, __func__, ha->instance));
	LEAVE(__func__);

	return(status);
}

typedef struct {
	int cmd;
	char *s;
} ioctl_tbl_row_t;

ioctl_tbl_row_t IOCTL_CMD_TBL[] =
{
	{EXT_CC_QUERY, "EXT_CC_QUERY"},
	{EXT_CC_REG_AEN, "EXT_CC_REG_AEN"},
	{EXT_CC_GET_AEN, "EXT_CC_GET_AEN"},
	{EXT_CC_GET_DATA, "EXT_CC_GET_DATA"},
	{EXT_CC_SET_DATA, "EXT_CC_SET_DATA"},
	{EXT_CC_SEND_SCSI_PASSTHRU, "EXT_CC_SEND_SCSI_PASSTHRU"},
	{EXT_CC_SEND_ISCSI_PASSTHRU, "EXT_CC_SEND_ISCSI_PASSTHRU"},
	{INT_CC_LOGOUT_ISCSI, "INT_CC_LOGOUT_ISCSI"},
	{EXT_CC_DISABLE_ACB, "EXT_CC_DISABLE_ACB"},
	{EXT_CC_SEND_ROUTER_SOL, "EXT_CC_SEND_ROUTER_SOL"},
	{EXT_CC_GET_HBACNT, "EXT_CC_GET_HBACNT"},
	{INT_CC_LOGOUT_ISCSI, "INT_CC_LOGOUT_ISCSI"},
	{INT_CC_DIAG_PING, "INT_CC_DIAG_PING"},
	{INT_CC_GET_DATA, "INT_CC_GET_DATA"},
	{INT_CC_SET_DATA, "INT_CC_SET_DATA"},
	{INT_CC_HBA_RESET, "INT_CC_HBA_RESET"},
	{INT_CC_COPY_FW_FLASH, "INT_CC_COPY_FW_FLASH"},
	{INT_CC_IOCB_PASSTHRU, "INT_CC_IOCB_PASSTHRU"},
	{INT_CC_RESTORE_FACTORY_DEFAULTS, "INT_CC_RESTORE_FACTORY_DEFAULTS"},
	{0, "UNKNOWN"}
};

ioctl_tbl_row_t IOCTL_SCMD_QUERY_TBL[] =
{
	{EXT_SC_QUERY_HBA_ISCSI_NODE, "EXT_SC_QUERY_HBA_ISCSI_NODE"},
	{EXT_SC_QUERY_HBA_ISCSI_PORTAL, "EXT_SC_QUERY_HBA_ISCSI_PORTAL"},
	{EXT_SC_QUERY_DISC_ISCSI_NODE, "EXT_SC_QUERY_DISC_ISCSI_NODE"},
	{EXT_SC_QUERY_DISC_ISCSI_PORTAL, "EXT_SC_QUERY_DISC_ISCSI_PORTAL"},
	{EXT_SC_QUERY_DRIVER, "EXT_SC_QUERY_DRIVER"},
	{EXT_SC_QUERY_FW, "EXT_SC_QUERY_FW"},
	{EXT_SC_QUERY_CHIP, "EXT_SC_QUERY_CHIP"},
	{EXT_SC_QUERY_IP_STATE, "EXT_SC_QUERY_IP_STATE"},
	{EXT_SC_QUERY_DEVICE_CURRENT_IP, "EXT_SC_QUERY_DEVICE_CURRENT_IP"},
	{0, "UNKNOWN"}
};

ioctl_tbl_row_t IOCTL_SCMD_EGET_DATA_TBL[] =
{
	{EXT_SC_GET_STATISTICS_GEN, "EXT_SC_GET_STATISTICS_GEN"},
	{EXT_SC_GET_STATISTICS_ISCSI, "EXT_SC_GET_STATISTICS_ISCSI"},
	{EXT_SC_GET_DEVICE_ENTRY_ISCSI, "EXT_SC_GET_DEVICE_ENTRY_ISCSI"},
	{EXT_SC_GET_DEVICE_ENTRY_DEFAULTS_ISCSI, "EXT_SC_GET_DEVICE_ENTRY_DEFAULTS_ISCSI"},
	{EXT_SC_GET_INIT_FW_ISCSI, "EXT_SC_GET_INIT_FW_ISCSI"},
	{EXT_SC_GET_INIT_FW_DEFAULTS_ISCSI, "EXT_SC_GET_INIT_FW_DEFAULTS_ISCSI"},
	{EXT_SC_GET_ISNS_SERVER, "EXT_SC_GET_ISNS_SERVER"},
	{EXT_SC_GET_ISNS_DISCOVERED_TARGETS, "EXT_SC_GET_ISNS_DISCOVERED_TARGETS"},
	{EXT_SC_GET_ACB, "EXT_SC_GET_ACB"},
	{EXT_SC_GET_NEIGHBOR_CACHE, "EXT_SC_GET_NEIGHBOR_CACHE"},
	{EXT_SC_GET_DESTINATION_CACHE, "EXT_SC_GET_DESTINATION_CACHE"},
	{EXT_SC_GET_DEFAULT_ROUTER_LIST, "EXT_SC_GET_DEFAULT_ROUTER_LIST"},
	{EXT_SC_GET_LOCAL_PREFIX_LIST, "EXT_SC_GET_LOCAL_PREFIX_LIST"},
	{EXT_SC_GET_STATISTICS_ISCSI_BLOCK, "EXT_SC_GET_STATISTICS_ISCSI_BLOCK"},
	{0, "UNKNOWN"}
};

ioctl_tbl_row_t IOCTL_SCMD_ESET_DATA_TBL[] =
{
	{EXT_SC_RST_STATISTICS_GEN, "EXT_SC_RST_STATISTICS_GEN"},
	{EXT_SC_RST_STATISTICS_ISCSI, "EXT_SC_RST_STATISTICS_ISCSI"},
	{EXT_SC_SET_DEVICE_ENTRY_ISCSI, "EXT_SC_SET_DEVICE_ENTRY_ISCSI"},
	{EXT_SC_SET_INIT_FW_ISCSI, "EXT_SC_SET_INIT_FW_ISCSI"},
	{EXT_SC_SET_ISNS_SERVER, "EXT_SC_SET_ISNS_SERVER"},
	{EXT_SC_SET_ACB, "EXT_SC_SET_ACB"},
	{0, "UNKNOWN"}
};

ioctl_tbl_row_t IOCTL_SCMD_IGET_DATA_TBL[] =
{
	{INT_SC_GET_FLASH, "INT_SC_GET_FLASH"},
	{INT_SC_GET_CORE_DUMP, "INT_SC_GET_CORE_DUMP"},
	{0, "UNKNOWN"},
	{0, "UNKNOWN"},
	{INT_SC_GET_DRIVER_DEBUG_LEVEL, "INT_SC_GET_DRIVER_DEBUG_LEVEL"},
	{INT_SC_GET_HOST_NO, "INT_SC_GET_HOST_NO"},
	{0, "UNKNOWN"}
};

ioctl_tbl_row_t IOCTL_SCMD_ISET_DATA_TBL[] =
{
	{INT_SC_SET_FLASH, "INT_SC_SET_FLASH"},
	{0, "UNKNOWN"},
	{0, "UNKNOWN"},
	{0, "UNKNOWN"},
	{INT_SC_SET_DRIVER_DEBUG_LEVEL, "INT_SC_SET_DRIVER_DEBUG_LEVEL"},
	{0, "UNKNOWN"}
};

static char *IOCTL_TBL_STR(int cc, int sc)
{
	ioctl_tbl_row_t *r;
	int cmd;

	switch (cc) {
	case EXT_CC_QUERY:
		r = IOCTL_SCMD_QUERY_TBL;
		cmd = sc;
		break;
	case EXT_CC_GET_DATA:
		r = IOCTL_SCMD_EGET_DATA_TBL;
		cmd = sc;
		break;
	case EXT_CC_SET_DATA:
		r = IOCTL_SCMD_ESET_DATA_TBL;
		cmd = sc;
		break;
	case INT_CC_GET_DATA:
		r = IOCTL_SCMD_IGET_DATA_TBL;
		cmd = sc;
		break;
	case INT_CC_SET_DATA:
		r = IOCTL_SCMD_ISET_DATA_TBL;
		cmd = sc;
		break;

	default:
		r = IOCTL_CMD_TBL;
		cmd = cc;
		break;
	}

	while (r->cmd != 0) {
		if (r->cmd == cmd) break;
		r++;
	}
	return(r->s);

}

/**************************************************************************
 * qla4xxx_ioctl
 * 	This the main entry point for all ioctl requests
 *
 * Input:
 *    	dev - pointer to SCSI device structure
 *	cmd - internal or external ioctl command code
 *	arg - pointer to the main ioctl structure
 *
 *	Instance field in ioctl structure - to determine which device to
 *	perform ioctl
 *	HbaSelect field in ioctl structure - to determine which adapter to
 *    	perform ioctl
 *
 * Output:
 *	The resulting data/status is returned via the main ioctl structure.
 *
 *	When Status field in ioctl structure is valid for normal command errors
 * 	this function returns 0 (QLA_SUCCESS).
 *
 *      All other return values indicate ioctl/system specific error which
 *	prevented the actual ioctl command from completing.
 *
 * Returns:
 *	 QLA_SUCCESS - command completed successfully, either with or without
 *			errors in the Status field of the main ioctl structure
 *    	-EFAULT      - arg pointer is NULL or memory access error
 *    	-EINVAL      - command is invalid
 *    	-ENOMEM      - memory allocation failed
 *
 * Context:
 *	Kernel context.
 **************************************************************************/
int qla4xxx_ioctl(Scsi_Device *dev, int cmd, void *arg)
{
	EXT_IOCTL_ISCSI	*ioctl = NULL;
	scsi_qla_host_t	*ha = NULL;
	int		status = 0;


	ENTER(__func__);

	/*
	 * Since internal IOCTLs are improperly defined in the 3-series IOCTLs
	 * (i.e. no magic number), check only the Signature for internal IOCTLs
	 */
	switch (cmd) {
	case INT_CC_LOGOUT_ISCSI:
	case INT_CC_DIAG_PING:
	case INT_CC_GET_DATA:
	case INT_CC_SET_DATA:
	case INT_CC_HBA_RESET:
	case INT_CC_COPY_FW_FLASH:
	case INT_CC_IOCB_PASSTHRU:
	case INT_CC_RESTORE_FACTORY_DEFAULTS:
		/* proceed to check signature */
		break;
	default:
		/* Catch any non-QLOGIC ioctls */
		if (_IOC_TYPE(cmd) != QLMULTIPATH_MAGIC) {
			status = (-EINVAL);
			goto exit_qla4xxx_ioctl;
		}
		break;
	}

	/* Allocate ioctl structure buffer to support multiple concurrent
	 * entries. NO static structures allowed.
	 */
	ioctl = ql4_kmem_zalloc(sizeof(EXT_IOCTL_ISCSI));
	if (ioctl == NULL) {
		/* error */
		printk(KERN_WARNING
		    "qla4xxx: ERROR in main ioctl buffer allocation.\n");
		status = (-ENOMEM);
		goto exit_qla4xxx_ioctl;
	}

	/*
	 * Check to see if we can access the ioctl command structure
	 */
#ifndef __VMKERNEL_MODULE__
	/*
         * vmkernel doesn't implement access_ok()
	 */
	if (!access_ok(VERIFY_WRITE, arg, sizeof(EXT_IOCTL_ISCSI))) {
		QL4PRINT(QLP2,
		    printk("%s: EXT_IOCTL_ISCSI verify failed.\n",
		    __func__));

		status = (-EFAULT);
		goto exit_qla4xxx_ioctl;
	}
#endif
	/*
	 * Copy the ioctl command structure from user space
	 */
	if (copy_from_user((uint8_t *)ioctl, arg, sizeof(EXT_IOCTL_ISCSI))) {
		QL4PRINT(QLP2|QLP4,
		    printk("%s: EXT_IOCTL_ISCSI copy read error.\n",
		    __func__));
		status = (-EFAULT);
		goto exit_qla4xxx_ioctl;
	}

	QL4PRINT(QLP10, printk("EXT_IOCTL_ISCSI structure: \n"));
	qla4xxx_dump_dwords(QLP10, ioctl, sizeof(*ioctl));

	/* Check signature */
	if (strncmp((char *) &ioctl->Signature, "QLOGIC",
	    sizeof(ioctl->Signature)) != QLA_SUCCESS) {
		status = (-EINVAL);
		goto exit_qla4xxx_ioctl;
	}

 	/*
	 * Retrieving HBA Count does not require dependance on an adapter instance.
	 * Apps need the ability to retrieve hba count without assuming that
	 * driver instance 0 is present. In a hot swap environment, instance 0
	 * may not be present.
	 */
	if (cmd == (int)EXT_CC_GET_HBACNT) {
		status = qla4extioctl_get_hbacnt(ioctl);
		/*
		 * Copy the updated ioctl structure back to the user
		 */
		if (copy_to_user(arg, (void *)ioctl, sizeof(EXT_IOCTL_ISCSI))) {
			QL4PRINT(QLP2,
			    printk("scsi%d: %s: EXT_IOCTL_ISCSI copy write error.\n",
			    ha->host_no, __func__));

			if (status == 0)
				status = (-EFAULT);
		}
		goto exit_qla4xxx_ioctl;
	}

	/*
	 * Get the adapter handle for the corresponding adapter instance
	 */
	ha = qla4xxx_get_adapter_handle(ioctl->HbaSelect);
	if (ha == NULL) {
		QL4PRINT(QLP2,
		    printk("%s: ha not found for ha instance %d.\n",
		    __func__, ioctl->HbaSelect));

		ioctl->Status = EXT_STATUS_DEV_NOT_FOUND;

		if (copy_to_user(arg, (void *)ioctl, sizeof(EXT_IOCTL_ISCSI))) {
			QL4PRINT(QLP2|QLP4,
			    printk("%s: EXT_IOCTL_ISCSI copy write error.\n",
			    __func__));
			status = (-EFAULT);
		}

		goto exit_qla4xxx_ioctl;
	}

       	if (!test_bit(AF_ONLINE, &ha->flags)) {
		if (cmd == (int)INT_CC_GET_DATA &&
		    ioctl->SubCode == INT_SC_GET_CORE_DUMP) {
			/* Allow Core Dump IOCTLs if adapter is offline  */
		}
		else if (qla4xxx_wait_for_hba_online(ha) == QLA_ERROR) {
			QL4PRINT(QLP2, printk("scsi%d: %s: IOCTL (%s) not pocessed, "
				"adapter offline.\n",
				ha->host_no, __func__, IOCTL_TBL_STR(cmd, ioctl->SubCode)));
			ioctl->Status = EXT_STATUS_HBA_NOT_READY;
			goto exit_qla4xxx_ioctl;
		}
	}

	QL4PRINT(QLP4,
	    printk("scsi%d: ioctl+ (%s)\n",
	    ha->host_no, IOCTL_TBL_STR(cmd, ioctl->SubCode)));

	ha->i_start = jiffies;
	ha->i_end = 0;
	ha->f_start = 0;
	ha->f_end = 0;

	down(&ha->ioctl_sem);

	/*
	 * If the DPC is active, wait for it to complete before proceeding
	 */
	while (ha->dpc_active) {
		set_current_state(TASK_UNINTERRUPTIBLE);
		schedule_timeout(1*HZ);
	}

	/*
	 * Issue the ioctl command
	 */
	switch (cmd) {
	case EXT_CC_QUERY:
		status = qla4extioctl_query(ha, ioctl);
		break;

	case EXT_CC_REG_AEN:
		status = qla4extioctl_reg_aen(ha, ioctl);
		break;

	case EXT_CC_GET_AEN:
		status = qla4extioctl_get_aen(ha, ioctl);
		break;

	case EXT_CC_GET_DATA:
		status = qla4extioctl_get_data(ha, ioctl);
		break;

	case EXT_CC_SET_DATA:
		status = qla4extioctl_set_data(ha, ioctl);
		break;

	case EXT_CC_SEND_SCSI_PASSTHRU:
		status = qla4extioctl_scsi_passthru(ha, ioctl);
		break;

	case EXT_CC_SEND_ISCSI_PASSTHRU:
		status = qla4extioctl_iscsi_passthru(ha, ioctl);
		break;

	case EXT_CC_DISABLE_ACB:
		status = qla4extioctl_disable_acb(ha, ioctl);
		break;

	case EXT_CC_SEND_ROUTER_SOL:
		status = qla4extioctl_send_router_sol(ha, ioctl);
		break;

	case INT_CC_LOGOUT_ISCSI:
		status = qla4intioctl_logout_iscsi(ha, ioctl);
		break;

	case INT_CC_DIAG_PING:
		status = qla4intioctl_ping(ha, ioctl);
		break;

	case INT_CC_GET_DATA:
		status = qla4intioctl_get_data(ha, ioctl);
		break;

	case INT_CC_SET_DATA:
		status = qla4intioctl_set_data(ha, ioctl);
		break;

	case INT_CC_HBA_RESET:
		status = qla4intioctl_hba_reset(ha, ioctl);
		break;

	case INT_CC_COPY_FW_FLASH:
		status = qla4intioctl_copy_fw_flash(ha, ioctl);
		break;

	case INT_CC_RESTORE_FACTORY_DEFAULTS:
		status = qla4intioctl_restore_factory_defaults(ha, ioctl);
		break;

#ifndef QLA4000
	case INT_CC_IOCB_PASSTHRU:
		status = qla4intioctl_iocb_passthru(ha, ioctl);
		break;
#endif

	default:
		QL4PRINT(QLP2,
		    printk("scsi%d: %s: unsupported command code (%X)\n",
		    ha->host_no, __func__, cmd));

		ioctl->Status = EXT_STATUS_UNSUPPORTED_SUBCODE;
	}

	/*
	 * Copy the updated ioctl structure back to the user
	 */
	if (copy_to_user(arg, (void *)ioctl, sizeof(EXT_IOCTL_ISCSI))) {
		QL4PRINT(QLP2,
		    printk("scsi%d: %s: EXT_IOCTL_ISCSI copy write error.\n",
		    ha->host_no, __func__));

		if (status == 0)
			status = (-EFAULT);
	}

	up(&ha->ioctl_sem);

	ha->i_end = jiffies;

	QL4PRINT(QLP15, printk("scsi%d: ioctl- (%s) "
	    "i_start=%lx, f_start=%lx, f_end=%lx, i_end=%lx\n",
	    ha->host_no, IOCTL_TBL_STR(cmd, ioctl->SubCode), ha->i_start,
	    ha->f_start, ha->f_end, ha->i_end));

	exit_qla4xxx_ioctl:

	if (ioctl)
		kfree(ioctl);

	LEAVE(__func__);
	return(status);
}

/**************************************************************************
 * apidev_open
 *	This routine is invoked just prior to every IOCTL call.  We only
 *	display debug information.
 *
 * Input:
 *	Unused
 *
 * Returns:
 *	0 - Always returns successful
 *
 * Context:
 *	Kernel context.
 **************************************************************************/
static  int
apidev_open(struct inode *inode, struct file *file)
{
	QL4PRINT(QLP4, printk("scsi: apidev_open MAJOR number = %d, "
	    "MINOR number = %d\n",
	    MAJOR (inode->i_rdev),
	    MINOR (inode->i_rdev)));
	return(0);
}

/**************************************************************************
 * apidev_close
 *	This routine is invoked just after every IOCTL call.  We only
 *	display debug information.
 *
 * Input:
 *	Unused
 *
 * Returns:
 *	0 - Always returns successful
 *
 * Context:
 *	Kernel context.
 **************************************************************************/
static  int
apidev_close(struct inode *inode, struct file *file)
{
	QL4PRINT(QLP4, printk("scsi: apidev_close\n"));
	return(0);
}

/**************************************************************************
 * apidev_ioctl
 *	This routine is invoked whenever an ioctl call is made.  It in turn
 *	calls the IOCTL function for this driver.
 *
 * Input:
 *	inode - unused
 *	fp    - unused
 *	cmd - internal or external ioctl command code
 *	arg - pointer to ioctl structure
 *
 * Output:
 *	None
 *
 * Returns:
 *	 QLA_SUCCESS - IOCTL completed successfully
 *	 QLA_ERROR   - IOCTL completed in error
 *    	-EFAULT      - if the arg pointer is NULL
 *    	-EINVAL      - if the command is invalid
 *
 * Context:
 *	Kernel context.
 **************************************************************************/
static  int
apidev_ioctl(struct inode *inode, struct file *fp, unsigned int cmd,
    unsigned long arg)
{
	static Scsi_Device fake_scsi_device;
	fake_scsi_device.host = apidev_host;
	return(qla4xxx_ioctl(&fake_scsi_device, (int)cmd, (void*)arg));
}

static struct file_operations
apidev_fops = {
	ioctl:    apidev_ioctl,
	open:     apidev_open,
	release:  apidev_close
};

/**************************************************************************
 * apidev_init
 *	This routine creates a proc file for IOCTL interface.
 *
 * Input:
 *	None
 *
 * Output:
 *	apidev_host - Updated with desired host number.
 *	apidev_major - Registered.
 *
 * Remarks:
 *	Create character driver "HbaApiDev" w dynamically allocated major
 *	number and create "/proc/scsi/<QLA4XXX_PROC_NAME>/HbaApiNode" as
 *	the device node associated with the major number.
 *
 * Returns:
 *	0 - Always returns successful
 *
 * Context:
 *	Kernel context.
 **************************************************************************/
int
apidev_init(struct Scsi_Host *host)
{
	if (apidev_host)
		return(0);

	if (0 > (apidev_major = register_chrdev(0, APIDEV_NAME,
	    &apidev_fops))) {
		QL4PRINT(QLP4|QLP7,
		printk("scsi: apidev_init: rc=%d\n", apidev_major));
		return(apidev_major);
	}

	apidev_host = host;
	QL4PRINT(QLP4|QLP7,
	    printk("scsi: Created /proc/scsi/%s/%s major=%d\n",
	    QLA4XXX_PROC_NAME, APIDEV_NODE, apidev_major));

	proc_mknod(APIDEV_NODE, 0600+S_IFCHR,
	    host->hostt->proc_dir, (kdev_t)MKDEV(apidev_major,0));

#if defined (QLA_CONFIG_COMPAT)
	ql4_apidev_init_32ioctl();
#endif

	return(0);
}

/**************************************************************************
 * apidev_cleanup
 *	This routine removes the proc file for the IOCTL interface
 *
 * Input:
 *	None
 *
 * Output:
 *	apidev_host - Cleared.
 *	apidev_major - Unregistered.
 *
 * Returns:
 *	0 - Always returns successful
 *
 * Context:
 *	Kernel context.
 **************************************************************************/
int
apidev_cleanup(void)
{
	if (!apidev_host)
		return(0);

#if defined (QLA_CONFIG_COMPAT)
	ql4_apidev_cleanup_32ioctl();
#endif

	unregister_chrdev(apidev_major,APIDEV_NAME);

	QL4PRINT(QLP4|QLP7, printk("scsi: apidev_cleanup\n"));

	remove_proc_entry(APIDEV_NODE,apidev_host->hostt->proc_dir);
	apidev_host = 0;

	return(0);
}



/*
 * Overrides for Emacs so that we almost follow Linus's tabbing style.
 * Emacs will notice this stuff at the end of the file and automatically
 * adjust the settings for this buffer only.  This must remain at the end
 * of the file.
 * ---------------------------------------------------------------------------
 * Local variables:
 * c-indent-level: 2
 * c-brace-imaginary-offset: 0
 * c-brace-offset: -2
 * c-argdecl-indent: 2
 * c-label-offset: -2
 * c-continued-statement-offset: 2
 * c-continued-brace-offset: 0
 * indent-tabs-mode: nil
 * tab-width: 8
 * End:
 */
