/*
 * QLogic iSCSI HBA Driver
 * Copyright (c)  2003-2006 QLogic Corporation
 *
 * See LICENSE.qla4xxx for copyright and licensing details.
 */


/* Management functions for various lists */

/*
 * Found in kernel 2.4.9 and higher in include/linux/lists.h
 *
 * Iterate over a list safe against removal of list.
 *
 */
#if !defined(list_for_each_safe)
	#define list_for_each_safe(pos, n, head) \
	for( pos= (head)->next, n = pos->next; pos != (head); \
		pos = n, n = pos->next )
#endif

#if defined(DEBUG) && defined(__VMKERNEL_MODULE__)
#define QL4ASSERT_SRB_STATE(srb, st) do { \
		if ((srb)->state != st) { \
			panic("%s:%d: SRB state does not match: srb=%p exp:%d got:%d\n", \
				__FILE__, __LINE__, srb, st, (srb)->state); \
		} \
	} while (0);
#else
#define QL4ASSERT_SRB_STATE(srb, st)
#endif

/*************************************/

static inline void
__add_to_pending_srb_q(scsi_qla_host_t *ha, srb_t *srb)
{
	DBG(QL4PRINT(QLP8, printk("scsi%d: %s: ha %d, srb = %p\n",
			      ha->host_no, __func__, ha->instance, srb));)
	list_add_tail(&srb->list_entry, &ha->pending_srb_q);
#ifdef __VMWARE__
	QL4ASSERT_SRB_STATE(srb, SRB_NO_QUEUE_STATE);
#endif
	srb->state = SRB_PENDING_STATE;
	ha->pending_srb_q_count++;
}

static inline void
__add_to_pending_srb_q_head(scsi_qla_host_t *ha, srb_t *srb)
{
	DBG(QL4PRINT(QLP8, printk("scsi%d: %s: ha %d, srb = %p\n",
			      ha->host_no, __func__, ha->instance, srb));)
	list_add(&srb->list_entry, &ha->pending_srb_q);
#ifdef __VMWARE__
	QL4ASSERT_SRB_STATE(srb, SRB_NO_QUEUE_STATE);
#endif
	srb->state = SRB_PENDING_STATE;
	ha->pending_srb_q_count++;
}

static inline void
__del_from_pending_srb_q(scsi_qla_host_t *ha, srb_t *srb)
{
	DBG(QL4PRINT(QLP8, printk("scsi%d: %s: ha %d, srb = %p\n",
			      ha->host_no, __func__, ha->instance, srb));)
	list_del_init(&srb->list_entry);
#ifdef __VMWARE__
	QL4ASSERT_SRB_STATE(srb, SRB_PENDING_STATE);
#endif
	srb->state = SRB_NO_QUEUE_STATE;
	ha->pending_srb_q_count--;
}

/*************************************/

static inline void
__add_to_retry_srb_q(scsi_qla_host_t *ha, srb_t *srb)
{
	QL4PRINT(QLP8, printk("scsi%d: %s: ha %d, srb = %p\n",
			      ha->host_no, __func__, ha->instance, srb));
	list_add_tail(&srb->list_entry, &ha->retry_srb_q);
#ifdef __VMWARE__
	QL4ASSERT_SRB_STATE(srb, SRB_NO_QUEUE_STATE);
#endif
	srb->state = SRB_RETRY_STATE;
	ha->retry_srb_q_count++;
}

static inline void
__del_from_retry_srb_q(scsi_qla_host_t *ha, srb_t *srb)
{
	QL4PRINT(QLP8, printk("scsi%d: %s: ha %d, srb = %p\n",
			      ha->host_no, __func__, ha->instance, srb));
	list_del_init(&srb->list_entry);
#ifdef __VMWARE__
	QL4ASSERT_SRB_STATE(srb, SRB_RETRY_STATE);
#endif
	srb->state = SRB_NO_QUEUE_STATE;
	ha->retry_srb_q_count--;
}

/*************************************/

static inline void
__add_to_done_srb_q(scsi_qla_host_t *ha, srb_t *srb)
{
	DBG(QL4PRINT(QLP8, printk("scsi%d: %s: ha %d, srb = %p\n",
			      ha->host_no, __func__, ha->instance, srb));)
	list_add_tail(&srb->list_entry, &ha->done_srb_q);
#ifdef __VMWARE__
	QL4ASSERT_SRB_STATE(srb, SRB_NO_QUEUE_STATE);
#endif
	srb->state = SRB_DONE_STATE;
	ha->done_srb_q_count++;
}

static inline void
__del_from_done_srb_q(scsi_qla_host_t *ha, srb_t *srb)
{
	DBG(QL4PRINT(QLP8, printk("scsi%d: %s: ha %d, srb = %p\n",
			      ha->host_no, __func__, ha->instance, srb));)
	list_del_init(&srb->list_entry);
#ifdef __VMWARE__
	QL4ASSERT_SRB_STATE(srb, SRB_DONE_STATE);
#endif
	srb->state = SRB_NO_QUEUE_STATE;
	ha->done_srb_q_count--;
}

static inline srb_t *__del_from_done_srb_q_head(scsi_qla_host_t *ha)
{
	struct list_head *ptr;
	srb_t *srb = NULL;

	if (!list_empty(&ha->done_srb_q))
	{
		/* Remove list entry from head of queue */
		ptr = ha->done_srb_q.next;
		list_del_init(ptr);

		/* Return pointer to srb structure */
		srb = list_entry(ptr, srb_t, list_entry);
#ifdef __VMWARE__
		QL4ASSERT_SRB_STATE(srb, SRB_DONE_STATE);
#endif
		srb->state = SRB_NO_QUEUE_STATE;
		ha->done_srb_q_count--;
	}
	DBG(QL4PRINT(QLP8, printk("scsi%d: %s: ha %d, srb = %p\n",
			      ha->host_no, __func__, ha->instance, srb));)

	return(srb);
}

/*************************************/

static inline void
__add_to_free_srb_q(scsi_qla_host_t *ha, srb_t *srb)
{
	DBG(QL4PRINT(QLP8, printk("scsi%d: %s: ha %d, srb = %p\n",
			      ha->host_no, __func__, ha->instance, srb));)

	//memset(srb, 0, sizeof(srb_t));
	list_add_tail(&srb->list_entry, &ha->free_srb_q);
#ifdef __VMWARE__
	QL4ASSERT_SRB_STATE(srb, SRB_NO_QUEUE_STATE);
#endif
	ha->free_srb_q_count++;
	srb->state = SRB_FREE_STATE;
}

static inline void __del_from_free_srb_q(scsi_qla_host_t *ha, srb_t *srb)
{

	DBG(QL4PRINT(QLP8, printk("scsi%d: %s: ha %d, srb = %p\n",
			      ha->host_no, __func__, ha->instance, srb));)
	list_del_init(&srb->list_entry);
#ifdef __VMWARE__
	QL4ASSERT_SRB_STATE(srb, SRB_FREE_STATE);
#endif
	srb->state = SRB_NO_QUEUE_STATE;
	ha->free_srb_q_count--;
}

static inline srb_t *__del_from_free_srb_q_head(scsi_qla_host_t *ha)
{
	struct list_head *ptr;
	srb_t *srb = NULL;

	if (!list_empty(&ha->free_srb_q))
	{
		/* Remove list entry from head of queue */
		ptr = ha->free_srb_q.next;
		list_del_init(ptr);

		/* Return pointer to srb structure */
		srb = list_entry(ptr, srb_t, list_entry);
#ifdef __VMWARE__
		QL4ASSERT_SRB_STATE(srb, SRB_FREE_STATE);
#endif
		memset(srb, 0, sizeof(*srb));
		srb->state = SRB_NO_QUEUE_STATE;
		ha->free_srb_q_count--;
	}
	DBG(QL4PRINT(QLP8, printk("scsi%d: %s: ha %d, srb = %p\n",
			      ha->host_no, __func__, ha->instance, srb));)

	return(srb);
}
/******************************************************************************/
/******************************************************************************/

static inline void
add_to_pending_srb_q(scsi_qla_host_t *ha, srb_t *srb)
{
	unsigned long flags;

	spin_lock_irqsave(&ha->list_lock, flags);
	__add_to_pending_srb_q(ha ,srb);
	spin_unlock_irqrestore(&ha->list_lock, flags);
}

static inline void
add_to_pending_srb_q_head(scsi_qla_host_t *ha, srb_t *srb)
{
	unsigned long flags;

	spin_lock_irqsave(&ha->list_lock, flags);
	__add_to_pending_srb_q_head(ha ,srb);
	spin_unlock_irqrestore(&ha->list_lock, flags);
}

static inline void
del_from_pending_srb_q(scsi_qla_host_t *ha, srb_t *srb)
{
	unsigned long flags;

	spin_lock_irqsave(&ha->list_lock, flags);
	__del_from_pending_srb_q(ha ,srb);
	spin_unlock_irqrestore(&ha->list_lock, flags);
}

/*************************************/

static inline void
add_to_retry_srb_q(scsi_qla_host_t *ha, srb_t *srb)
{
	unsigned long flags;

	spin_lock_irqsave(&ha->list_lock, flags);
	__add_to_retry_srb_q(ha ,srb);
	spin_unlock_irqrestore(&ha->list_lock, flags);
}

static inline void
del_from_retry_srb_q(scsi_qla_host_t *ha, srb_t *srb)
{
	unsigned long flags;

	spin_lock_irqsave(&ha->list_lock, flags);
	__del_from_retry_srb_q(ha ,srb);
	spin_unlock_irqrestore(&ha->list_lock, flags);
}

/*************************************/

static inline void
add_to_done_srb_q(scsi_qla_host_t *ha, srb_t *srb)
{
	unsigned long flags;

	spin_lock_irqsave(&ha->list_lock, flags);
	__add_to_done_srb_q(ha ,srb);
	spin_unlock_irqrestore(&ha->list_lock, flags);
}

static inline void
del_from_done_srb_q(scsi_qla_host_t *ha, srb_t *srb)
{
	unsigned long flags;

	spin_lock_irqsave(&ha->list_lock, flags);
	__del_from_done_srb_q(ha ,srb);
	spin_unlock_irqrestore(&ha->list_lock, flags);
}

static inline srb_t *
del_from_done_srb_q_head(scsi_qla_host_t *ha)
{
	unsigned long flags;
	srb_t *srb;

	spin_lock_irqsave(&ha->list_lock, flags);
	srb = __del_from_done_srb_q_head(ha);
	spin_unlock_irqrestore(&ha->list_lock, flags);
	return(srb);
}

/*************************************/

static inline void
add_to_free_srb_q(scsi_qla_host_t *ha, srb_t *srb)
{
	unsigned long flags;

	spin_lock_irqsave(&ha->list_lock, flags);
#ifdef __VMWARE__
	QL4ASSERT_SRB_STATE(srb, SRB_NO_QUEUE_STATE);
#endif
	memset(srb, 0, sizeof(*srb));
#ifdef __VMWARE__
	srb->state = SRB_NO_QUEUE_STATE;
#endif
	__add_to_free_srb_q(ha ,srb);
	spin_unlock_irqrestore(&ha->list_lock, flags);
}

static inline srb_t *
del_from_free_srb_q_head(scsi_qla_host_t *ha)
{
	unsigned long flags;
	srb_t *srb;

	spin_lock_irqsave(&ha->list_lock, flags);
	srb = __del_from_free_srb_q_head(ha);
	spin_unlock_irqrestore(&ha->list_lock, flags);

	if (srb)
	{
		#ifdef DEBUG
		if ((int) atomic_read(&srb->ref_count) != 0)
		{
			QL4PRINT(QLP2, printk("scsi%d: %s: WARNING: "
					      "ref_count not zero.\n",
                                          ha->host_no, __func__));
		}
		#endif
		
		atomic_set(&srb->ref_count, 1);
	}
	return(srb);
}
