/* ****************************************************************
 * Portions Copyright 2004 VMware, Inc.
 * 
 *
 * ****************************************************************/
/*
 * NOTE: This code is shared with the VMKernel and should not be modified
 *       without consent from the storage team. Another copy of the same 
 *       file is at bora/lib/diskId/scsi_dev_id.c
 *       if you make any changes to this file you should also make the 
 *	 same changes there.
 */

#if defined(VMKERNEL)

#include "scsi_vmware.h"
#include "vm_libc.h"
#define printk _Log
#define memcmp __builtin_memcmp

#elif defined(__KERNEL__) || defined(VMNIX)

#include <linux/config.h>
#include <linux/module.h>

#include <linux/sched.h>
#include <linux/timer.h>
#include <linux/string.h> 
#include <linux/slab.h>
#include <linux/kernel.h>
#include <linux/scsi_vmware.h>

#else

#include <string.h>
#include "scsi_vmware.h"
#include "vmware.h"
#define printk Log
#ifdef _MSC_VER
#define inline
#endif

#endif

static char *idPrefix = "VMWARE SCSI Id"; 

#define MAX_SYMM_GATEKEEPER_LUN_SIZE   ((unsigned long long)(50*1024*1024))
#define MODE_SENSE_WRITE_PROTECTED	0x08
#define SCSI_READ_CAPACITY_MAX_LBN 	0xffffffff

/*
 * Function: scsi_vmware_swap_bytes
 *
 * Purpose: re-arrange bytes into an unsigned integer
 *
 * Arguments: inBuf 	(input)  - array of 4 chars
 *
 * Returns: unsigned integer
 *
 * Notes:   none
 */
static inline unsigned int
scsi_vmware_swap_bytes(unsigned char *inBuf)
{
   return ( (inBuf[0] << 24) | (inBuf[1] << 16) | (inBuf[2] << 8) | (inBuf[3]) );
}

/*
 * Function:	scsi_vmware_disk_is_pseudo_device
 *
 * Purpose:	Determine if the given disk is a gatekeeper/pseudo device
 *             
 * 
 * Arguments:	name               (input)   - a text name for the SCSI disk
 *              lun                (input)   - logical unit number
 *              inquiryBuffer      (input)   - buffer containing results from
 *                                             a SCSI INQUIRY cmd of page 0
 *              modeSenseBuffer    (input)   - buffer containing results from 
 *                                             a SCSI MODE SENSE cmd
 *              readCapacityBuffer (input)   - buffer containing results from
 *                                             a SCSI READ CAPACITY cmd
 *
 *		The device is considered a gatekeeper if:
 *		1) it is write protected
 *		2) it has a capacity of 0
 *		3) it has a LUN id of 0 on a COMPAQ HSV100 (EVA) device
 *		4) it has a Vendor of IBM and a model of "Universal Xport"
 *		5) it has a LUN id of 0 with a model of "LUNZ" on a DGC/EMC Clariion device
 *		6) it is a EMC SYMMETRIX device and has a capacity of < 50 MBs
 *
 * Returns:
 *		1 if this is a gatekeeper
 *		0 otherwise
 *
 * Notes:	This is used by the both the COS and the vmkernel.
 */ 
int
scsi_vmware_disk_is_pseudo_device(
   const char *name,
   int  lun,
   unsigned char *inquiryBuffer,
   unsigned char *modeSenseBuffer,
   unsigned char *readCapacityBuffer)
{
   int isPseudoDevice = 0;
   char vendor[SCSI_VENDOR_LENGTH + 1];
   char model[SCSI_MODEL_LENGTH + 1];
   unsigned long blockSize, numBlocks;
   unsigned long long capacity;

   memset(vendor, 0,  SCSI_VENDOR_LENGTH + 1);
   memset(model, 0,  SCSI_MODEL_LENGTH + 1);
   memcpy(vendor, inquiryBuffer + SCSI_VENDOR_OFFSET , SCSI_VENDOR_LENGTH );
   memcpy(model, inquiryBuffer + SCSI_MODEL_OFFSET , SCSI_MODEL_LENGTH );

   blockSize = scsi_vmware_swap_bytes(&readCapacityBuffer[4]);
   if ((scsi_vmware_swap_bytes(&readCapacityBuffer[0]) == SCSI_READ_CAPACITY_MAX_LBN) && (blockSize > 0)) {
      /*
       * This LUN has more than 2 TB of capacity. It is not
       * necessary to determine its true capacity. It is too big
       * to be a pseudo device.
       */
      numBlocks = (MAX_SYMM_GATEKEEPER_LUN_SIZE/blockSize) + 1;
   } else {
      numBlocks = scsi_vmware_swap_bytes(&readCapacityBuffer[0]) + 1;
   }

   capacity = ((unsigned long long)blockSize) * ((unsigned long long)numBlocks);
   if (capacity == (unsigned long long)0) {
      printk("Capacity is 0 for Disk %s, lun %d : 0x%x 0x%x 0x%x 0x%x 0x%x 0x%x 0x%x 0x%x\n", 
        name, lun, readCapacityBuffer[0],
        readCapacityBuffer[1], readCapacityBuffer[2], readCapacityBuffer[3],
        readCapacityBuffer[4], readCapacityBuffer[5], readCapacityBuffer[6],
        readCapacityBuffer[7]);
   }
   
   if (modeSenseBuffer[2] & MODE_SENSE_WRITE_PROTECTED) {
      isPseudoDevice = 1;
   } else if (capacity == (unsigned long long)0) {
      isPseudoDevice = 1;
   } else if ((lun == 0) &&
              ((strncmp(vendor, "COMPAQ", 6) == 0) ||
               (strncmp(vendor, "HP", 2) == 0)) && 
	      (strncmp(model, "HSV", 3) == 0)) {
      isPseudoDevice = 1;
   } else if ((lun == 0) && (strncmp(vendor, "DGC", 3) == 0) && 
	      (strncmp(model, "LUNZ", 4) == 0)) {
      isPseudoDevice = 1; 
   } else if ((lun == 0) && (strncmp(vendor, "EMC", 3) == 0) &&
              (strncmp(model, "LUNZ", 4) == 0)) {
      isPseudoDevice = 1;
   } else if ((strncmp(vendor, "EMC", 3) == 0) &&
	      (strncmp(model, "SYMMETRIX", 9) == 0) &&
              (capacity < MAX_SYMM_GATEKEEPER_LUN_SIZE)) {
      isPseudoDevice = 1;
   } else if ((strncmp(vendor, "IBM", 3) == 0) &&
	      (strncmp(model, "Universal Xport", 15) == 0)) {
      isPseudoDevice = 1;
   }

   if (isPseudoDevice) {
      printk("Disk %s " 
	     "is a pseudo device. lid = %d, ro = %d, cap: (%lu * %lu) = %llu\n",
	     name, lun, modeSenseBuffer[2] & 0x08, blockSize, numBlocks, capacity);
   }
   return isPseudoDevice;
}

/*
 * Function:	scsi_vmware_get_disk_id
 *
 * Purpose:	Extract a unique identifier for a SCSI disk that can be used
 *              to identify those disks connected via multiple data paths.
 * 
 * Arguments:	name           (input)   - a text name for the SCSI disk
 *              inquiryBuffer  (input)   - buffer containing results from
 *                                         a SCSI INQUIRY cmd of page 0
 *              evpdBuffer     (input)   - buffer containing results from
 *                                         a SCSI INQUIRY cmd of page 1 
 *		deviceIdBuffer (input)   - buffer containing results from
 *                                         a SCSI INQUIRY cmd of page 0x83
 *	        serialIdBuffer (input)   - buffer containing results from
 *                                         a SCSI INQUIRY cmd of page 0x80
 *		diskId	       (output)  - 44 byte buffer that will contain VMware disk id
 *		diskIdLen      (output)  - length of the diskId string
 *
 * Returns:	VMWARE_SCSI_ID_* value      
 *		*diskIdLen contains the length of the string in diskId
 *	        *diskId contains the disk id and has a maximum length of SCSI_DISK_ID_LEN
 *
 *		if the returned type is VMWARE_SCSI_ID_SYMM
 *		    diskId[0 - 4] = 5 byte Symm Id
 *		    diskId[5 - 8] = 4 byte device Id
 *		    diskId[9 - 14] = 1st 6 bytes of Disk Model
 * 
 *		if the returned type is VMWARE_SCSI_ID_NAA
 *		    diskId[0 - N] = NAA identifier
 *		    diskId[N - (N+5)] = 1st 6 bytes of Model 
 *	            N can be any value between 0 and SCSI_DISK_ID_LEN - 6.
 *
 *		if the returned type is VMWARE_SCSI_ID_SERIAL_NUM
 *		    diskId[0 - N] = SERIAL number
 *		    diskId[N - (N+5)] = 1st 6 bytes of Model		     
 *                  N can be any value between 0 and SCSI_DISK_ID_LEN - 6.
 *
 *		if the returned type is VMWARE_SCSI_ID_UNIQUE
 *		    diskId[0] = 0
 *
 *		A disk id with a 1st character of NULL is considered a 
 *		VMWARE_SCSI_ID_UNIQUE id. If an actual disk id has a NULL as
 *		a 1st character it is converted to a SPACE.
 *
 *
 * Notes:
 *              For some strange reason, there are two other copies of this file.  See
 *              bug 126108 for details.   The other files are:
 *
 *                 console-os/drivers/scsi/vmware/scsi_dev_id.c
 *                 bora/vmkernel/storage/scsi_legacy_diskid.c
 *
 *              These files must be kept in sync, or the world will end (a harbinger
 *              of the apocalypse?).
 *
 *		Be extremely careful when modifying the SCSI_DiskId code. These
 *		ids are written to disk for LV volumes and RDMs. SCSI_DiskIds
 *		must be backward compatible.
 */ 
int
scsi_vmware_get_disk_id(
   const char *name,
   unsigned char *inquiryBuffer,
   unsigned char *evpdBuffer,
   unsigned char *deviceIdBuffer,
   unsigned char *serialIdBuffer,
   char  	 *diskId,
   int  	 *diskIdLen)
{
   int i;
   char vendor[SCSI_VENDOR_LENGTH + 1];
   char model[SCSI_MODEL_LENGTH + 1];
   int hasSerialId, hasDeviceId;
   int diskIdType = VMWARE_SCSI_ID_UNIQUE;

   hasSerialId = hasDeviceId = 0;
   memset(diskId, 0, SCSI_DISK_ID_LEN);
   memset(vendor, 0,  SCSI_VENDOR_LENGTH + 1);
   memset(model, 0,  SCSI_MODEL_LENGTH + 1);

   memcpy(vendor, inquiryBuffer + SCSI_VENDOR_OFFSET , SCSI_VENDOR_LENGTH );
   memcpy(model, inquiryBuffer + SCSI_MODEL_OFFSET , SCSI_MODEL_LENGTH );

   *diskIdLen = 0;

   if ((strncmp(vendor, "EMC     ", SCSI_VENDOR_LENGTH) == 0 &&
        strncmp(model, "SYMMETRIX       ", SCSI_MODEL_LENGTH) == 0)) {
      if (inquiryBuffer[0x2] == 0x2) {
         /*
          * This is an old style Symm (Symm6) that has a non-standard 
          * page 0x83 format (SCSI-2)
          */
         diskIdType = VMWARE_SCSI_ID_SYMM; 
         memcpy(diskId, inquiryBuffer+124, 5);
         memcpy(diskId+5, inquiryBuffer+132, 4);
         *diskIdLen = 9;
         printk("%s:  Symmetrix device with a Symm6 device id\n", idPrefix);
         goto gotdiskid;
      } else if (inquiryBuffer[0x2] == 0x4) {
         /*
          * The new style Symm (Symm7), with a version number of 0x4, has a standard  
          * page 0x83 format (SPC-2) and an NAA identifier.
          * Fallthru to NAA code.
          */
         printk("%s:  Symmetrix device with a Symm7 device id\n", idPrefix);
      } else {
         printk("%s:  Symmetrix device with an unknown version of 0x%x\n", idPrefix, inquiryBuffer[0x2]);
      }
   }

   printk("%s: Supported VPD pages for %s : ", idPrefix, name);
   for (i = 0; i < evpdBuffer[3]; i++) {
      printk("0x%x ", evpdBuffer[4+i]);
   }
   printk("\n");

   /*
    * Determine which INQUIRY pages are supported by this device.
    */
   for (i = 0; i < evpdBuffer[3]; i++) {
      if (evpdBuffer[4+i] == SCSI_VPROD_UNIT_SERIAL_NUMBER) {
	 hasSerialId = 1;
      }
      else if (evpdBuffer[4+i] == SCSI_VPROD_DEVICE_ID) {
	 hasDeviceId = 1;
      }
   }

   if (hasDeviceId) {
      if (deviceIdBuffer[1] == SCSI_VPROD_DEVICE_ID) {
	 printk("%s: Device id info for %s: ", idPrefix, name);
	 for (i = 0; i < deviceIdBuffer[3]; i++) {
	    printk("0x%x ", deviceIdBuffer[4+i]);
	 }
	 printk("\n");
	 for (i = 4; i < deviceIdBuffer[3]+4; ) {
	    /* 
             * Look for an NAA identifier that is for the whole device, not
	     * the individual port. i is the index of the current descriptor.
	     * The descriptor list starts at index 4. 
             */
	    if (deviceIdBuffer[i+1] == SCSI_NAA_IDENTIFIER) {
	       diskIdType = VMWARE_SCSI_ID_NAA;
	       *diskIdLen = deviceIdBuffer[i+3];
	       if ((*diskIdLen) > SCSI_DISK_ID_LEN) {
		  *diskIdLen = SCSI_DISK_ID_LEN;
	       }
	       memcpy(diskId, deviceIdBuffer+i+4, *diskIdLen);
	       goto gotdiskid;
	    }
	    i += 4+deviceIdBuffer[i+3];
	 }
      }
   }
   if (!hasSerialId) {
      goto nodiskid;
   }

   if (serialIdBuffer[1] != SCSI_VPROD_UNIT_SERIAL_NUMBER) {
      goto nodiskid;
   }

   diskIdType = VMWARE_SCSI_ID_SERIAL_NUM;
   *diskIdLen = serialIdBuffer[3];
   if ((*diskIdLen) > SCSI_DISK_ID_LEN) {
      *diskIdLen = SCSI_DISK_ID_LEN;
   }


   if (!((*diskIdLen) < SCSI_VPROD_PAGE_LENGTH-3)) {
      printk("%s:  ERROR ID LEN OF %d  TOO LONG FOR  %s\n:", idPrefix, *diskIdLen, name); 
   }
   memcpy(diskId, serialIdBuffer+4, *diskIdLen);
  

gotdiskid: 
   /* 
    * Append first 6 chars of model, since these ids may not be unique
    * across vendors or models.
    */
   if ((*diskIdLen) + SCSI_DISK_ID_MODEL_LEN > SCSI_DISK_ID_LEN) {
      *diskIdLen = SCSI_DISK_ID_LEN - SCSI_DISK_ID_MODEL_LEN;
   }
   memcpy(diskId + (*diskIdLen), model, SCSI_DISK_ID_MODEL_LEN);
   (*diskIdLen) += SCSI_DISK_ID_MODEL_LEN;

   printk("%s: Id for %s ", idPrefix, name);
   scsi_vmware_log_disk_id(diskId, *diskIdLen); 
   printk("\n");

   /*
    * Convert leading NULL to a space.
    */
   if ((*diskIdLen > 0) && (diskId[0] == 0)) {
      diskId[0] = ' ';
   }
 
   return( diskIdType );

nodiskid: 
   printk("%s: Could not get disk id for %s\n", idPrefix, name);
   diskId[0] = 0;
   *diskIdLen = 1;
   diskIdType = VMWARE_SCSI_ID_UNIQUE;

   return( diskIdType );
}



/*
 * Function:	scsi_vmware_disk_id_equal
 *
 * Purpose:	Compare the LU numbers and Device ids of of two disks
 *              to determine equality.
 * 
 * Arguments:	id1           (input)   - identifier of 1st disk
 *              idLun1        (input)   - LUN of 1st disk
 *              id2           (input)   - identifier of 2nd disk
 *		idLun2        (input)   - LUN of 2nd disk
 *
 * Returns:	1 if the disks are equal, 0 otherwise
 *
 * Notes:	This is used by the both the COS and the vmkernel.
 */
int
scsi_vmware_disk_id_equal(unsigned char *id1, int idLun1, unsigned char *id2, int idLun2)
{
   if (((*id1) == 0) || ((*id2) == 0)) {
      /*
       * At least one of the disks has an id of VMWARE_SCSI_ID_UNIQUE.
       */
      return( 0 );
   }

   if (idLun1 == idLun2) {
      if ( memcmp(id1, id2, SCSI_DISK_ID_LEN) == 0) { 
         return( 1 );
      }
   }
   return( 0 );
}

/*
 * Function:	scsi_vmware_log_disk_id
 *
 * Purpose:	Write disk id to log file
 *             
 * 
 * Arguments:	diskId           (input)   - identifier of disk
 *              len		 (input)   - line of identifier
 *
 * Returns:	None
 *
 * Notes:	None
 */
void
scsi_vmware_log_disk_id(unsigned char *diskId, int len)
{
   int i;

   for(i = 0; i < len; i++) {
#ifdef VMKERNEL
      if ( ((i+1) % (VMK_LOG_ENTRY_SIZE / 5)) == 0 ) { // 5 == strlen('0xNN ')
         printk("\n");
      }
#endif
         printk("0x%02x ", 0xFF & ((unsigned char)(diskId[i])));
   }
}
/*
 * Function:	scsi_vmware_filter_lun
 *
 * Purpose:	Check for devices that have good status but the LUN does
 *              not really exist.
 *             
 * 
 * Arguments:	inqStr           (input)   - pointer to a SCSI inquiry string
 *
 * Returns:	NZ		 Device should be ignored
 *              Z                Real device, don't ignore
 *
 * Notes:       None
 */
int
scsi_vmware_filter_lun(char *inqStr)
{
	/* check for no LUN at this location */
	if ((inqStr[0] & 0x7F) == 0x7F)
		return 1;
	if ((strncmp(&inqStr[8], "DGC", 3) == 0) &&
			(strncmp(&inqStr[16], "    ", 4) == 0) &&
			inqStr[1] == 0)
		return 1;

        /* Reject DELL MD3000i pseudo lun */
        if ((strncmp(&inqStr[8], "DELL", 4) == 0) &&
            (strncmp(&inqStr[16], "Universal Xport", 15) == 0)) {
                return 1;
        }

	/* don't filter */
	return 0;
}

