/*
 *  Copyright 2003 VMware, Inc.  All rights reserved.
 */

/*
 * Functions used to parse the cpci line to find which devices the console
 * can see.
 *
 * cpci line := { bus ':' slots ';' }
 * bus := number
 * slots := nothing | '*' | slot [ { ',' slot } ]
 * slot := number [ '.' number ] 
 *
 * If a device has its bus listed, it must be mentionned to be assigned to
 * the console. But if a device does not have its bus listed, then it is
 * also assigned to the console.
 */

/*
 * Parse 'bus' or 'slot'
 */
static int
vmnix_get_num(char **p, int *pfcn) {
   int dev = 0;
   int fcn = -1;

   // get a number
   while (**p >= '0' && **p <= '9') {
      dev = (dev * 10) + (**p - '0'); 
      (*p)++;
   }
   // if followed by a '.', get a second number
   if (**p == '.') {
      fcn = 0;
      (*p)++;
      while (**p >= '0' && **p <= '9') {
	 fcn = (fcn * 10) + (**p - '0'); 
	 (*p)++;
      }
   }

   // skip the character that possibly follows a 'bus' or a 'slot'
   if (**p == ':') {
      (*p)++;
   }
   if (**p == ',') {
      (*p)++;
   }

   *pfcn = fcn;

   return dev;
}

/*
 * Parse the cpci line for a specific device
 */
static int
vmnix_is_visible(char *cpciLine, int bus, int devFunction)
{
   int slotFound = 0;
   int curBus, fcn, dev;
   char *p = cpciLine;
   int slot = PCI_SLOT(devFunction);
   int function = PCI_FUNC(devFunction);

   while (*p != 0) {

      // get 'bus' and skip ':'
      curBus = vmnix_get_num(&p, &fcn);

      // if followed by '*', all slots valid
      if (*p == '*') {
	 slotFound = 1;
	 p++; // skip the '*'
      } else {
	 while (*p != 0 && *p != ';') {
	    // get 'slot' ans skip ','
            dev = vmnix_get_num(&p, &fcn);
	    if ((dev == slot) && ((fcn == function) || (fcn == -1))) {
               slotFound = 1;
	    }             
	 }
      }
      // if it's the right bus, the result is whether we found the correct slot
      if (bus == curBus) {
	 return slotFound;
      }
      slotFound = 0;
      if (*p != 0) {
	 p++; // skip the ';'
      }
   }

   // the right bus was not listed, the device is visible by default
   return  1;
}
