##########################################################################
# Copyright 2004 VMware, Inc.  All rights reserved. -- VMware Confidential
##########################################################################

#
# BootInfo.pm --
#
#       Procedural module collecting functions related to boot configuration.
#       These functions do the bulk of the work for the esxcfg-boot script.
#

package VMware::Boot::BootInfo;

use VMware::Log qw(:log :manip :count);
use VMware::Config qw(:config);
use VMware::Config::VmacoreConfigObj;
use VMware::Boot::ChecksumManager;
use VMware::PCI::HostInterface qw(MakeCPCILine);
use VMware::PCI::PCIInfo qw(GetPCIDeviceManagerFromConfig);
use VMware::FileSys::StandardLocator qw(StdFile);

require Exporter;
use strict;

@VMware::Boot::BootInfo::ISA = qw(Exporter);
@VMware::Boot::BootInfo::EXPORT_OK =
   qw(LoadChecksumManager UpdateBootConfig CommitConfig CommitInitrd);

########################################################################
#
# LoadChecksumConfig --
#
#       Loads the checksum config object. If esx_checksum.conf exists,
#       loads it into the object, otherwise returns a blank config.
#
# Results:
#       A VmacoreConfigObj holding either the old checksum data,
#       or nothing.
#
# Side effects:
#       Loads and locks esx_checksum.conf.
# 
########################################################################

sub LoadChecksumConfig
{
   my $file = StdFile('esx_checksum_config', 'esx');
   my $config = new VMware::Config::VmacoreConfigObj();

   my $success = 0;
   if (-r $file) {
      $success = $config->LockAndReadFile($file);
   } else {
      $success = $config->LockFile($file);
   }

   unless ($success) {
      LogError("Could not load esx checksum config.");
      return undef;
   }

   return $config;
}


########################################################################
#
# LoadChecksumManager --
#
#       Loads the checksum manager object.
#
# Results:
#       The checksum manager.
#
# Side effects:
#       See LoadChecksumConfig.
# 
########################################################################

sub LoadChecksumManager
{
   my $checksumCfg = LoadChecksumConfig();
   unless (defined $checksumCfg) {
      return undef;
   }

   my $checksumTree = $checksumCfg->GetTree(['checksum']);
   $checksumCfg->UnlockFile();

   return new VMware::Boot::ChecksumManager($checksumTree);
}


########################################################################
#
# GetCPCILineFromConfig --
#
#       Creates a boot CPCI line based on the provided config tree.
#
# Results:
#       The CPCI line on success, undef otherwise.
#
# Side effects:
#       None.
# 
########################################################################

sub GetCPCILineFromConfig
{
   my $config = shift;  # IN: The esx configuration object.

   my $devMgr = GetPCIDeviceManagerFromConfig($config);
   unless (defined $devMgr) {
      LogError("Could not get PCI device configuration for CPCI line.");
      return undef;
   }
   
   return MakeCPCILine($devMgr);
}


########################################################################
#
# UpdateBootConfig --
#
#       Update the boot configuration.
#       Update the config object with the results.
#
# Results:
#       True on success, false on failure.
#
# Side effects:
#       Updates contents of the configuration object.
#
########################################################################

sub UpdateBootConfig
{
   my $config = shift;     # IN/OUT: The esx configuration object.
   my $bootMgr = shift;    # IN: The boot manager.
   my $rootDev = shift;    # IN: The new root device.
   my $kernelAppend = shift; # IN: The new kernel append.
   my $newrd = shift;      # IN: New-style boot/rd. 
   my $listCpci= shift;    # IN: Explicit cpci. 
   
   LogInfo("Updating boot configuration...");

   my $cpci = undef; 
   if ($listCpci) {
      $cpci = GetCPCILineFromConfig($config);
   }
   
   unless ($bootMgr->Update($cpci, $rootDev, $kernelAppend, $newrd)) {
      LogError("Could not update the boot configuration.");
      return 0;
   }

   # Update the config tree to the new settings.
   $config->SetTree(['boot'], $bootMgr->GetConfigData());
   return 1;
}


########################################################################
#
# CommitConfig --
#
#       Saves the system configuration to disk.
#
# Results:
#       True on success, false on failure.
#
# Side effects:
#       Writes esx.conf and grub.conf. Leaves esx.conf locked.
#
########################################################################

sub CommitConfig
{
   my $config = shift;     # IN: The esx configuration object.
   my $bootMgr = shift;    # IN: The boot manager.

   LogInfo("Writing esx.conf...");
   unless ($config->WriteFile()) {
      LogError("Could not commit changes back to esx.conf.");
      return 0;
   }

   LogInfo("Writing grub.conf...");
   unless ($bootMgr->CommitBootloader()) {
      LogError("Could not commit changes back to grub.conf.");
      return 0;
   }

   return 1;
}


########################################################################
#
# CommitInitrd --
#
#       Saves the initrd and checksum information to disk.
#
# Results:
#       True on success, false on failure.
#
# Side effects:
#       Writes initrds for all images in the boot configuration.
#       Writes esx_checksum.conf.
#
########################################################################

sub CommitInitrd
{
   my $config = shift;        # IN: The esx configuration object.
   my $bootMgr = shift;       # IN: The boot manager.
   my $checksumMgr = shift;   # IN: The checksum manager.
   my $initrdCmd = shift;     # IN: esxcfg-divvy uses this.
   my $fullrebuild = shift;   # IN: build whole initrd
   my $newrd = shift;         # IN: create new style initrd

   LogInfo("Recreating initrds...");
   unless ($bootMgr->CommitInitrd($initrdCmd, $fullrebuild, $newrd)) {
      LogError("Could not write out new initrds.");
      return 0;
   }

   LogInfo("Writing esx_checksum.conf...");
   unless (CommitChecksum($checksumMgr)) {
      LogError("Could not write out new checksum.");
      return 0;
   }

   return 1;
}


########################################################################
#
# CommitChecksum --
#
#       Saves the checksum information to disk.
#
# Results:
#       True on success, false on failure.
#
# Side effects:
#       Writes esx_checksum.conf.
#
########################################################################

sub CommitChecksum 
{
   my $checksumMgr = shift;   # IN: The checksum manager.

   my $checksumCfg = LoadChecksumConfig();
   unless (defined $checksumCfg) {
      return 0;
   }

   $checksumMgr->Update();
   $checksumCfg->SetTree(['checksum'], $checksumMgr->GetConfigData());

   #
   # If this is the first time creating the checksum file,
   # we won't have called ReadFile and set the filename. So pass
   # in the filename to save to explicitly.
   #

   my $file = StdFile('esx_checksum_config', 'esx');
   unless ($checksumCfg->WriteAndUnlockFile($file)) {
      LogError("Could not commit changes back to esx_checksum.conf.");
      return 0;
   }

   return 1;
}

1;
