##########################################################################
# Copyright 2005 VMware, Inc.  All rights reserved. -- VMware Confidential
##########################################################################

#
# BootManager.pm --
#
#       This class manages the boot configuration tasks, including:
#       1) Updating the boot config tree
#       2) Updating the bootloader configuration
#       3) Generating the initial ram disks.
#

package VMware::Boot::BootManager;

use File::Copy qw(copy);
use Math::BigInt;

use VMware::Log qw(:log);
use VMware::Boot::GRUBConfig;
use VMware::Boot::GRUBDeviceMap;
use VMware::FileSys::Lock;
use VMware::FileSys::MountInfo qw(LookupMountDevice LookupMountUuid
                                  LookupMountSize LookupMount);
use VMware::FileSys::StandardLocator qw(StdCommand StdDir);
use VMware::FileSys::Move qw(MoveFileKeepDestMode);
use VMware::FileSys::Tmp qw(:tmp);
use VMware::System qw(LogCommand);
use strict;

use constant AUTOGENERATED_ESX => "esx";
use constant DEFAULT_MEM_SIZE => 512;
use constant VMNIX_SUFFIX => 'vmnix';
use constant VMNIX_VERSION_STR => '2.4.21';

#
# The bootloader config stores a version number which allows
# us to detect old-style GRUB configurations. From 3.0 onwards
# this version is explicitly listed. If there is no version
# listed it is implicitly version 0.
#

use constant CONFIG_VERSION => 1;

use constant BOOT_CHOICES => (
   {
      label => "ESX",
      title => "VMware ESX Server",
      append => "",
      kernelSuffix => VMNIX_SUFFIX(),
      initrdSuffix => "",
      debug => 0,
      default => 1,
      regenerateInitrd => 1,
      mkinitrdCmd => 'vmware_mkinitrd',
      mkinitrdRefreshCmd => 'vmware_mkinitrd_refresh',
   },
   {
      label => "ESX-debug",
      title => "VMware ESX Server (debug mode)",
      append => "console=ttyS0,115200 console=tty0 debug",
      kernelSuffix => VMNIX_SUFFIX(),
      initrdSuffix => "-dbg",
      debug => 1,
      default => 0,
      regenerateInitrd => 1,
      mkinitrdCmd => 'vmware_mkinitrd',
      mkinitrdRefreshCmd => 'vmware_mkinitrd_refresh',
   },
   {
      label => "ESX-sc",
      title => "Service Console only (troubleshooting mode)",
      append => "tblsht",
      kernelSuffix => VMNIX_SUFFIX(),
      initrdSuffix => "-sc",
      debug => 0,
      default => 0,
      regenerateInitrd => 0, # We use this also to distinguish ESX vs. SC-only.
      mkinitrdCmd => 'mkinitrd',
      mkinitrdRefreshCmd => 'mkinitrd',
   },
);


########################################################################
#
# BootManager::new --
#
#       Constructor.  Optionally takes the configuration tree
#       representing all boot info from the main config file and builds
#       its internal set of boot objects from the tree and from
#       the bootloader configuration file.
#
# Results:
#       The new instance, or undef on error.
#
# Side effects:
#       Reads /boot/grub/grub.conf.
#
########################################################################

sub new
{
   my $class = shift;            # IN: Invoking class name.
   my $bootcfg = (shift || {});  # IN: Config tree, defaults to empty.

   my $self = {
      cfgTree => undef,
      updated => 0,
      bootloaderConfig => undef,
   };
   bless $self => $class;

   $self->{cfgTree} = $bootcfg;
   $self->{bootloaderConfig} = $self->LoadBootloaderConfig();
   return undef unless (defined $self->{bootloaderConfig});
   $self->{includeDebug} = $self->HasRoomForDebugInitrd();

   return $self;
}


########################################################################
#
# BootManager::LoadBootloaderConfig --
#
#       Private method used by the constructor. Loads the bootloader
#       configuration.
#
# Results:
#       The config object on success, undef otherwise.
#
# Side effects:
#       Reads /boot/grub/grub.conf.
#
########################################################################

sub LoadBootloaderConfig
{
   my $self = shift;    # IN: Invoking instance.

   my $config = new VMware::Boot::GRUBConfig();
   unless ($config->LoadFromSystem()) {
      LogError("Could not load bootloader configuration.");
      return undef;
   }

   return $config;
}


########################################################################
#
# BootManager::Update --
#
#       Updates boot configuration from system information and from
#       defaults.
#
# Results:
#       1 on success, 0 otherwise.
#
# Side effects:
#       None.
#
########################################################################

sub Update
{
   my $self = shift;             # IN: Invoking instance.
   my $cpci = shift;             # IN: The CPCI line to boot with.
   my $rootDev = shift;          # IN: (Optional) The UUID of the root
                                 #     partition in the form UUID=<uuid>
   my $kernelAppend = shift;     # IN: (Optional) Extra kernel
                                 #     arguments, as a string.
   my $newrd = shift;            # IN: New-style boot/rd. 

   #
   # Determine the kernel version and root device
   #

   my $kernelVersion = $self->LoadKernelVersion();
   return 0 unless (defined $kernelVersion);

   if (defined $rootDev) {
      # If the user specified a rootDev, use it
   } elsif (defined $self->{cfgTree}->{rootDev}) {
      # Otherwise use the preconfigured setting 
      $rootDev = $self->{cfgTree}->{rootDev};
   } else {
      # Finally, if the rootDev is not set, determine it automatically
      if ($newrd) {
         $rootDev = "LABEL=horde";
      } else {
         $rootDev = $self->LoadRootDev();
      }
      return 0 unless (defined $rootDev);
   }

   unless (defined $kernelAppend) {
      if (defined $self->{cfgTree}->{kernelAppend}) {
         # Fall back to preconfigured kernel append.
         $kernelAppend = $self->{cfgTree}->{kernelAppend};
      } else {
         $kernelAppend = "";
      }
   }

   # If memSize is not set, set it to the COS default
   my $memSize = $self->{cfgTree}->{memSize};
   unless (defined $memSize) {
      $memSize = DEFAULT_MEM_SIZE;
   }

   #
   # Update the config tree and bootloader
   #

   unless ($self->UpdateBootloader($kernelVersion,
                                   $cpci, $rootDev, $memSize,
                                   $kernelAppend, $newrd)) {
      return 0;
   }

   $self->{cfgTree}->{kernelVersion} = $kernelVersion;
   $self->{cfgTree}->{kernelAppend} = $kernelAppend;
   $self->{cfgTree}->{rootDev} = $rootDev;
   $self->{cfgTree}->{memSize} = $memSize;

   $self->{updated} = 1;
   return 1;
}


########################################################################
#
# BootManager::UpdateBootloader --
#
#       Private method used by Update. Updates the bootloader
#       configuration.
#
# Results:
#       1 on success, 0 otherwise.
#
# Side effects:
#       None.
#
########################################################################

sub UpdateBootloader
{
   my $self = shift;             # IN: Invoking instance.
   my $kernelVersion = shift;    # IN: The new kernel version.
   my $cpci = shift;             # IN: The new CPCI line.
   my $rootDev = shift;          # IN: The new root device.
   my $memSize = shift;          # IN: The new COS memory size.
   my $kernelAppend = shift;     # IN: The new kernel append.
   my $newrd = shift;            # IN: New-style boot/rd. 

   my $grubRoot = $self->LoadGRUBRoot();
   unless (defined $grubRoot) {
      LogError("Could not determine GRUB root");
      return 0;
   }

   # 
   # Retrieve custom images and reset boot menu.
   #

   my @customImages;
   my $customDefault = 0;
   if ($self->{bootloaderConfig}->GetVersion() == 0) {
      #
      # Version 0 (3.0 beta 1) configurations lack a way
      # of tracking autogenerated entries. In this case,
      # nuke everything; at least that way we don't end up
      # with duplicates.
      #
      
      @customImages = ();
   } else {
      @customImages = grep { $_->GetAutoGenerated() ne AUTOGENERATED_ESX }
                           @{$self->{bootloaderConfig}->GetImages()};
      $customDefault = grep { $_->GetDefault() } @customImages;
   }

   $self->{bootloaderConfig}->SetImages([]);
   $self->{bootloaderConfig}->SetVersion(CONFIG_VERSION);

   my $bootPath = $self->MakeBootPath();

   foreach my $choice (BOOT_CHOICES) {

      # ServiceConsole boot target won't work with newrd, so don't bother.
      if ($newrd && !$choice->{regenerateInitrd}) { 
         next;
      }

      if ($choice->{debug} and not $self->{includeDebug}) {
         next;
      }

      my $kernelImage = $self->MakeKernelName($kernelVersion,
                                              $choice->{kernelSuffix});
      my $kernelInitrd = $self->MakeInitrdName($kernelVersion,
                                               $choice->{kernelSuffix},
                                               $choice->{initrdSuffix});

      my $kernelTitle = $choice->{title};

      my $img = new VMware::Boot::GRUBBootImage();
      $img->SetTitle($kernelTitle);
      $img->SetGRUBRoot($grubRoot);
      $img->SetReadOnly(1);
      $img->SetLabel($choice->{label});
      $img->SetImage("$bootPath/$kernelImage");
      $img->SetInitrd("$bootPath/$kernelInitrd");
      $img->SetRootDev($rootDev);

      my $append = $choice->{append};
      if ($kernelAppend) {
         $append .= " $kernelAppend";
      }
      $img->SetOtherAppend($append);

      $img->SetMemSize($memSize);
      if ($choice->{regenerateInitrd} && defined $cpci) {
         $img->SetCPCI($cpci);
      }

      #
      # Mark this as auto-generated so that we can replace it
      # on later invocations.
      #

      $img->SetAutoGenerated(AUTOGENERATED_ESX);

      #
      # Add image. Set the new default if necessary.
      #

      my $default = ($customDefault) ? 0 : $choice->{default};
      $img->SetDefault($default);

      $self->{bootloaderConfig}->AddImage($img);
   }

   foreach my $img (@customImages) {
      $self->{bootloaderConfig}->AddImage($img);
   }

   return 1;
}


########################################################################
#
# BootManager::LoadGRUBRoot --
#
#       Private method. Determines the GRUB root.
#
# Results:
#       The GRUB root on success, undef otherwise.
#
# Side effects:
#       See MountInfo::LookupMountDevice and
#       GRUBDeviceMap::LoadFromSystem.
#
########################################################################

sub LoadGRUBRoot
{
   my $self = shift;    # IN: Invoking instance.

   my $bootDir = StdDir('boot', 'all');
   my ($devName, $partNum) = LookupMountDevice($bootDir, 1);
   unless ($devName) {
      LogError("Could not determine device for /boot");
      return undef;
   }

   my $deviceMap = new VMware::Boot::GRUBDeviceMap();
   unless ($deviceMap->LoadFromSystem()) {
      LogError("Could not load bootloader device map.");
      return undef;
   }

   my $key = $deviceMap->GetKeyForDevice($devName);
   unless ($key) {
      # XXX: SAN/SCSI reordering can cause the entries in device.map
      # to become invalid. Until we have a better solution, default
      # to hd0 as the installer will always try to put the boot
      # drive there.
      LogWarn("Could not find device map key for $devName, " .
              "defaulting to hd0");
      $key = "hd0";
   }

   #
   # GRUB partition numbers are indexed from 0, not 1.
   #

   my $keyNum = $partNum - 1;
   return "($key,$keyNum)";
}


########################################################################
#
# BootManager::LoadRootDev --
#
#       Private method. Determines the root device
#       in UUID spec form.
#
# Results:
#       The root dev on success, undef otherwise.
#
# Side effects:
#       See MountInfo::LookupMountUuid.
#
########################################################################

sub LoadRootDev
{
   my $self = shift;    # IN: Invoking instance.

   my $uuid = LookupMountUuid("/");
   unless ($uuid) {
      LogError("Could not determine the root device UUID");
      return undef;
   }
   
   return "UUID=$uuid";
}


########################################################################
#
# BootManager::ParseVersion --
#
#       Private function. Used to parse a version string into its
#       numeric components.
#
# Results:
#       A list of numeric components, or an empty list if none found.
#
# Side effects:
#       None.
#
########################################################################

sub ParseVersion($)
{
   my $str = shift; # IN: The version string to parse.

   unless ($str =~ /(                     # Capture whole version string
                     (\d+)\.(\d+)\.(\d+)  # Version (X.Y.Z)
                     -\d+(\.\d+)*         # Release (variable-length)
                    )/x) {
      return ();
   }

   my $version = $1;
   my @components = split(/[\.\-]/, $version);
   return @components;
}


########################################################################
#
# BootManager::ExtendVersion --
#
#       Private function. Used to pad a version to a minimum length.
#
# Results:
#       A zero-padded list of version components.
#
# Side effects:
#       None.
#
########################################################################

sub ExtendVersion($$)
{
   my $components = shift;    # IN: The list of version components.
   my $length = shift;        # IN: The length to pad to.

   my @ret = @$components;
   if (scalar(@ret) < $length) {
      my $extend = $length - scalar(@ret);
      push @ret, (0) x $extend;
   }

   return @ret;
}


########################################################################
#
# BootManager::VersionCmp --
#
#       Private function. Used by sort to return highest version.
#
# Results:
#       The <=> result between the two kernel versions.
#
# Side effects:
#       None.
#
########################################################################

sub VersionCmp($$)
{
   my $a = shift;    # IN: The first version to compare.
   my $b = shift;    # IN: The second version to compare.

   my @aVer = ParseVersion($a);
   my @bVer = ParseVersion($b);

   # Ensure aVer and bVer are the same length.
   @aVer = ExtendVersion(\@aVer, scalar(@bVer));
   @bVer = ExtendVersion(\@bVer, scalar(@aVer));

   my $len = scalar(@aVer);
   for (my $i = 0; $i < $len; $i++) {
      my $result = $bVer[$i] <=> $aVer[$i];
      return $result if ($result != 0);
   }

   return 0;
}


########################################################################
#
# BootManager::LoadKernelVersion --
#
#       Private method. Determines the highest kernel verson on the
#       system.
#
# Results:
#       The kernel version on success, undef otherwise.
#
# Side effects:
#       Searches /boot.
#
########################################################################

sub LoadKernelVersion
{
   my $self = shift;    # IN: Invoking instance.
 
   #
   # Look for the VMnix kernel as the definitive version
   #

   my $suffix = VMNIX_SUFFIX();
   my $ver = VMNIX_VERSION_STR();

   my $bootDir = StdDir('boot', 'all');
   if (! -d $bootDir) {
      LogError("Boot directory '$bootDir' is not a directory.");
      return undef;
   }

   my $dirh;
   if (! opendir($dirh, $bootDir)) {
      LogError("Could not open directory '$bootDir': $!.");
      return undef;
   }

   my @versions = ();
   foreach my $filename (readdir($dirh)) {
      if ($filename =~ /^vmlinuz-(\Q$ver\E.*)\Q$suffix\E$/ &&
          -f "$bootDir/$filename") {
         push(@versions, $1);
      }
   }
   
   closedir($dirh);

   unless (@versions) {
      LogError("No kernel images found with " . 
               "version '$ver' and suffix '$suffix'.");
      return undef;
   }

   @versions = sort VersionCmp  @versions;
   return $versions[0];
}


########################################################################
#
# BootManager::HasRoomForDebugInitrd --
#
#       Private method used by the constructor.. Determines whether or
#       not there is enough space for a debug initrd.
#
# Results:
#       1 if there is enough room, 0 otherwise.
#
# Side effects:
#       See LookupMountSize.
#
########################################################################

sub HasRoomForDebugInitrd
{
   my $self = shift;    # IN: Invoking instance.

   my $bootDir = StdDir('boot', 'all');
   my $sizeStr = LookupMountSize($bootDir);
   my $size = Math::BigInt->new($sizeStr);
   
   return ($size > 90);
}


########################################################################
#
# BootManager::MakeKernelName --
#
#       Construct the name of the kernel file for the given version.
#
# Results:
#       The kernel filename on success, undef on failure. Note that
#       this is simply the filename and not a full path.
#
# Side effects:
#       None.
#
########################################################################

sub MakeKernelName
{
   my $self = shift;             # IN: Invoking instance.
   my $kernelVersion = shift;    # IN: The new kernel version.
   my $kernelSuffix = shift;     # IN: The new kernel suffix.

   return "vmlinuz-$kernelVersion$kernelSuffix";
}


########################################################################
#
# BootManager::MakeInitrdName --
#
#       Construct the name of the initrd file for the given version.
#
# Results:
#       The initrd filename on success, undef on failure. Note that
#       this is simply the filename and not a full path.
#
# Side effects:
#       None.
#
########################################################################

sub MakeInitrdName
{
   my $self = shift;             # IN: Invoking instance.
   my $kernelVersion = shift;    # IN: The new kernel version.
   my $kernelSuffix = shift;     # IN: The new kernel suffix.
   my $initrdSuffix = shift;     # IN: The new initrd suffix.

   return "initrd-$kernelVersion$kernelSuffix.img$initrdSuffix";
}


########################################################################
#
# BootManager::MakeBootPath --
#
#       Construct the boot path as used in grub.conf.
#       This path is relative to the mount point of the partition
#       we are booting off of - the empty string if booting from /boot,
#       and /boot if booting from /.
#
# Results:
#       The boot path, or undef on failure.
#
# Side effects:
#       See LookupMount.
#
########################################################################

sub MakeBootPath
{
   my $self = shift;             # IN: Invoking instance.

   my $bootDir = StdDir('boot', 'all');
   my ($dev, $type, $mnt) = LookupMount($bootDir);
   unless ($dev) {
      LogError("Cannot find partition containing /boot.");
      return undef;
   }

   if ($mnt eq "/boot") {
      #
      # If the system has a /boot partition, compute grub paths relative
      # to it.
      # 
      return "";
   } else {
      #
      # Otherwise, the system is booting from / and we should use
      # a path relative to / (that is, an absolute path).
      #
      return $bootDir;
   }
}


########################################################################
#
# BootManager::CommitBootloader --
#
#       Commits boot configuration to disk.
#
# Results:
#       1 on success, 0 otherwise.
#
# Side effects:
#       None.
#
########################################################################

sub CommitBootloader
{
   my $self = shift;             # IN: Invoking instance.

   return $self->{bootloaderConfig}->Commit();
}


########################################################################
#
# BootManager::CommitInitrd --
#
#       Regenerates the initial ramdisks. Note that if Update() has
#       previously been called on the boot manager, the result of
#       GetConfigData() must be saved to esx.conf before this method
#       can be called.
#
# Results:
#       1 on success, 0 otherwise.
#
# Side effects:
#       None.
#
########################################################################

sub CommitInitrd
{
   my $self = shift;          # IN: Invoking instance.
   my $initrdCmd = shift;     # IN: esxcfg-divvy uses this.
   my $fullrebuild = shift;   # IN: rebuild initrd
   my $newrd = shift;         # IN: create new style initrd

   #
   # Regenerate initial ramdisks.
   #

   my $kernelVersion = $self->{cfgTree}->{kernelVersion};

   foreach my $choice (BOOT_CHOICES) {
      unless ($choice->{regenerateInitrd}) {
         next;
      }

      if ($choice->{debug} and not $self->{includeDebug}) {
         next;
      }

      if (defined ($initrdCmd)) {
         LogInfo("Using initrd helper $initrdCmd.");
      } elsif ($self->{updated} || $fullrebuild) {
         $initrdCmd = $choice->{mkinitrdCmd};
      } else {
         $initrdCmd = $choice->{mkinitrdRefreshCmd};
      }

      my $cmdList = StdCommand($initrdCmd, 'esx');
      unless (defined $cmdList) {
         LogError("Could not get initrd helper $initrdCmd.");
         return 0;
      }

      my $kernelFullVersion = $kernelVersion . $choice->{kernelSuffix};
      my $kernelInitrd = $self->MakeInitrdName($kernelVersion,
                                               $choice->{kernelSuffix},
                                               $choice->{initrdSuffix});
      my $bootDir = StdDir('boot', 'all');
      my $initrdFile = "$bootDir/$kernelInitrd";

      #
      # Create a temp file to run mkinitrd on.
      # If we are refreshing, this file should be a copy of
      # the existing initrd. Otherwise, it can be blank.
      # Once complete, move the temp file to the real initrd.
      #

      my $filename = TmpFile(1);
      unless ($self->{updated} || copy($initrdFile, $filename)) {
         LogError("Could not copy '$initrdFile' to '$filename': $!");
         return 0;
      }
   
      #
      # vmware-mkinitrd needs environment variables to tell
      # it to build debug or new-style rds. Set them if necessary.
      #

      my $oldDebugESX;
      my $oldnewrd;
      if ($choice->{debug}) {
         $oldDebugESX = $ENV{DEBUG_ESX};
         $ENV{DEBUG_ESX} = 1;
      }
      if ($newrd) {
         $oldnewrd = $ENV{NEWRD};
         $ENV{NEWRD} = 1;
      }

      my $cmd = join(' ', @$cmdList, $filename, "'$kernelFullVersion'");
      unless (LogCommand($cmd)) {
         LogError("Execution of '$cmd' failed.  See log messages above.");
         return 0;
      }

      #
      # Reset debug environment variable.
      #

      if ($choice->{debug}) {
         if (defined $oldDebugESX) {
            $ENV{DEBUG_ESX} = $oldDebugESX;
         } else {
            delete $ENV{DEBUG_ESX};
         }
      }

      if ($newrd) {
         if (defined $oldnewrd) {
            $ENV{NEWRD} = $oldnewrd;
         } else {
            delete $ENV{NEWRD};
         }
      }

      #
      # Rewrite file from temp file.
      #

      my $lock = new VMware::FileSys::Lock();
      unless ($lock->LockFile($initrdFile)) {
         return 0;
      }

      unless (MoveFileKeepDestMode($filename, $initrdFile)) {
         $lock->UnlockFile();
         return 0;
      }
      RemoveFileFromTmpCleanup($filename);
      $lock->UnlockFile();
   }

   return 1;
}


########################################################################
#
# BootManager::GetConfigData --
#
#       Return the config file structure representing the boot settings.
#
# Results:
#       The structure as a hash reference.
#
# Side effects:
#       None.
#
########################################################################

sub GetConfigData
{
   my $self = shift;  # IN: Invoking instance.

   return $self->{cfgTree};
}


########################################################################
#
# QueryBootConfig --
#
#       Queries the boot configuration and returns it as a list
#       reference.
#
# Results:
#       The configuration on success, undef otherwise.
#
# Side effects:
#       None.
#
########################################################################

sub QueryBootConfig
{
   my $self = shift;             # IN: Invoking instance.

   #
   # Look for a VMnix config.
   #

   my $vmnixSuffix = VMNIX_SUFFIX();
   my @imgs = grep($_->GetImage() =~ /$vmnixSuffix/,
                   @{$self->{bootloaderConfig}->GetImages()});
   if ( $#imgs > -1 ) {
      # 
      # Pick first such config.
      #

      my $img = $imgs[0];
      my $mem = $img->GetMemSize() || 0;
      my $cpci = $img->GetCPCI() || "0:*;";
      return [$mem,
              $cpci,
              $img->GetRootDev(),
              $img->GetImage(),
              $img->GetInitrd()];
   }

   return undef;
}


########################################################################
#
# BootManager::InstallBootloader --
#
#       Installs GRUB as the bootloader on the boot device specified
#       in grub.conf.
#
# Results:
#       1 on success, 0 otherwise.
#
# Side effects:
#       Calls grub-install.
#
########################################################################

sub InstallBootloader
{
   my $self = shift;             # IN: Invoking instance.

   my $bootDevice = $self->{cfgTree}->{legacyBootDevice};
   unless ($bootDevice) {
      LogWarn("No boot drive specified. Defaulting to hd0");
      $bootDevice = "(hd0)";
   }

   my $cmdList = StdCommand('grub_install', 'all');
   unless (defined $cmdList) {
      LogError("Could not get grub-install helper.");
      return 0;
   }

   my $cmd = join(' ', @$cmdList, $bootDevice);
   unless (LogCommand($cmd)) {
      LogError("Execution of '$cmd' failed.  See log messages above.");
      return 0;
   }

   return 1;
}

1;
