##########################################################################
# Copyright 2005 VMware, Inc.  All rights reserved. -- VMware Confidential
##########################################################################

#
# ChecksumManager.pm --
#
#       This class manages tracking the checksum of the esx.conf file.
#

package VMware::Boot::ChecksumManager;

use VMware::Log qw(:log);
use VMware::FileSys::StandardLocator qw(StdCommand StdFile);
use VMware::System qw(LogCommandGetOutput);
use strict;


########################################################################
#
# ChecksumManager::new --
#
#       Constructor.  Optionally takes the configuration tree
#       representing all checksum info from the main config file.
#
# Results:
#       The new instance, or undef on error.
#
# Side effects:
#       Reads the main config file.
#
########################################################################

sub new
{
   my $class = shift;                # IN: Invoking class name.
   my $checksumcfg = (shift || {});  # IN: Checksum config tree,
                                     #     defaults to empty.
   my $configFile = shift;    # IN: (Optional) The StdFile name of
                              #     the main config file to checksum. Defaults
                              #     to esx_config.
   my $product = shift;       # IN: (Optional) The product of the
                              #     main config file to checksum. Defaults
                              #     to esx.

   my $self = {
      cfgTree => undef,
      cfgFile => undef,
   };
   bless $self => $class;

   $configFile = "esx_config" unless defined $configFile;
   $product = "esx" unless defined $product;
   my $file = StdFile($configFile, $product);
   unless (-r $file) {
      LogError("Configuration file '$file' is not readable or does not exist.");
      return undef;
   }

   $self->{cfgTree} = $checksumcfg;
   $self->{cfgFile} = $file;

   return $self;
}


##############################################################################
#
# Calculate  --
#
#       Get md5sum of current esx.conf.
#
# Results:
#       The md5sum on success, undef otherwise.
#
# Side effects:
#       None. 
#
##############################################################################

sub Calculate
{
   my $self = shift;    # IN: Invoking instance.

   my $cmdList = StdCommand('md5sum', 'all');
   unless (defined $cmdList) {
      LogError("Couldn't find md5sum command.");
      return undef;
   }

   my $cmd = join(' ', @$cmdList, $self->{cfgFile}, ' 2>/dev/null');
   my $output = LogCommandGetOutput($cmd);
   my ($md5Sum, $rest) = split(' ', $output, 2);
   return $md5Sum;
}


##############################################################################
#
# Compare  --
#
#       Compare current esx.conf with saved checksum.
#
# Results:
#       0 if checksums match; 1 if not. 
#
# Side effects:
#       None. 
#
##############################################################################

sub Compare
{
   my $self = shift;    # IN: Invoking instance.

   my $oldSum = $self->{cfgTree}->{confMD5Sum};
   if (not defined ($oldSum)) { 
      # Not an error; should proceed as if checksums don't match.
      return 1;
   }

   my $newSum = $self->Calculate();
   if (not defined ($newSum)) {
      LogWarn("Could not calculate checksum.");
      return 1;
   }

   if ($oldSum ne $newSum) {
      return 1;
   } else {
      return 0;
   }
}


##############################################################################
#
# Update  --
#
#       Save checksum of current esx.conf.  
#
# Results:
#       1 - success; 0 - failure.
#
# Side effects:
#       None. 
#
##############################################################################

sub Update 
{
   my $self = shift;    # IN: Invoking instance.

   my $newSum = $self->Calculate();
   if (not defined ($newSum)) {
      LogError("Couldn't calc checksum.");
      return 0;
   }

   my $date = scalar(localtime());

   $self->{cfgTree}->{confMD5Sum} = $newSum;
   $self->{cfgTree}->{lastRDRefresh} = $date;

   return 1; 
}


########################################################################
#
# ChecksumManager::GetConfigData --
#
#       Return the config file structure representing the
#       checksum settings.
#
# Results:
#       The structure as a hash reference.
#
# Side effects:
#       None.
#
########################################################################

sub GetConfigData
{
   my $self = shift;  # IN/OUT: Invoking instance.

   return $self->{cfgTree};
}

1;
