#!/usr/bin/perl -w

#############################################################
# Copyright 2004 VMware, Inc.  All rights reserved. -- VMware Confidential
#############################################################
#
# GRUBBootImage.pm
# 
# A basic class that contains information representing the grub.conf
# configuration file.  Information includes a series of options, both
# global and local.
#
# How a GRUB boot image corresponds to the various fields:
# title TITLE
#         root GRUBROOT      e.g. (hd0,0)
# PRE_CMDS                   e.g. (dhcp, etc)
#         kernel [--no_mem_options] IMAGE [ro if READONLY]
#                root=RDEV [mem=MEMSIZE] [cpci=CPCI]
#                [OTHER_APPEND]
#         initrd INITRD
# POST_CMDS
#
# In case of Windows or non-Linux OS'es that use chain-loading,
# there won't be kernel or initrd.  In that case we see
# title TITLE
# PRE_CMDS
#
# This is also used for menu titles for misc options that GRUB
# allows, such as reboot/restart, install GRUB, etc.

package VMware::Boot::GRUBBootImage;

use strict;


########################################################################
#
# GRUBBootImage::new --
#
#       Constructor. Creates a new grub boot image object.
#
# Results:
#       The new instance, or undef on error.
#
# Side effects:
#       None.
#
########################################################################

sub new
{
   my $class = shift;      # IN: Invoking class name.

   my $self = {
      image => "",
      label => "",
      title => "",
      initrd => "",
      memSize => "",
      rootDev => "",
      CPCI => "",
      otherAppend => "",
      readOnly => 0,
      GRUBRoot => "",
      preCmds => [],
      postCmds => [],
      autoGenerated => "",
      default => 0,
   };

   bless $self => $class;
   return $self;

}

#
# Trivial Accessor Functions
#

sub GetImage
{
   my $self = shift;   
   return $self->{image};
}

sub SetImage
{
   my $self = shift;
   $self->{image} = shift;
}

sub GetLabel
{
   my $self = shift;
   return $self->{label};
}

sub SetLabel
{
   my $self = shift;
   $self->{label} = shift;
}

sub GetTitle
{
   my $self = shift;
   return $self->{title};
}

sub SetTitle
{
   my $self = shift;
   $self->{title} = shift;
}

sub GetInitrd
{
   my $self = shift;   
   return $self->{initrd};
}

sub SetInitrd
{
   my $self = shift;
   $self->{initrd} = shift;
}

sub GetMemSize
{
   my $self = shift;
   return $self->{memSize};
}

sub SetMemSize
{
   my $self = shift;
   $self->{memSize} = shift;
}

sub GetRootDev
{
   my $self = shift;
   return $self->{rootDev};
}

sub SetRootDev
{
   my $self = shift;
   $self->{rootDev} = shift;
}

sub GetCPCI
{
   my $self = shift;
   return $self->{CPCI};
}

sub SetCPCI
{
   my $self = shift;
   $self->{CPCI} = shift;
}

sub GetOtherAppend
{
   my $self = shift;
   return $self->{otherAppend};
}

sub SetOtherAppend
{
   my $self = shift;
   $self->{otherAppend} = shift;
}

sub GetReadOnly
{
   my $self = shift;
   return $self->{readOnly};
}

sub SetReadOnly
{
   my $self = shift;
   $self->{readOnly} = shift;
}

sub GetGRUBRoot
{
   my $self = shift;
   return $self->{GRUBRoot};
}

sub SetGRUBRoot
{
   my $self = shift;
   $self->{GRUBRoot} = shift;
}

sub GetAutoGenerated
{
   my $self = shift;
   return $self->{autoGenerated};
}

sub SetAutoGenerated
{
   my $self = shift;
   $self->{autoGenerated} = shift;
}

sub GetDefault
{
   my $self = shift;
   return $self->{default};
}

sub SetDefault
{
   my $self = shift;
   $self->{default} = shift;
}

# Array ref of text of any commands executed before "kernel".

sub GetPreCmds
{
   my $self = shift;
   return $self->{preCmds};
}

sub SetPreCmds
{
   my $self = shift;
   $self->{preCmds} = shift;
}

# Array ref of text of any commands executed after "initrd".

sub GetPostCmds
{
   my $self = shift;
   return $self->{postCmds};
}

sub SetPostCmds
{
   my $self = shift;
   $self->{postCmds} = shift;
}


########################################################################
#
# GRUBBootImage::ToString --
#
#       Writes a new grub.conf image entry into a string.
#
# Results:
#       The new string on success, undef otherwise.
#
# Side effects:
#       None.
#
########################################################################

sub ToString
{
   my $self = shift;       # IN: Invoking instance.
   
   my $s = "";

   # Add the title
   $s .= "title " . $self->GetTitle() . "\n";

   if ($self->GetAutoGenerated()) {
      $s .= "\t#vmware:autogenerated " . $self->GetAutoGenerated() . "\n";
   }

   # Add the GRUB root, ie the partition containing /boot
   if ($self->GetGRUBRoot()) {
      $s .= "\troot " . $self->GetGRUBRoot() . "\n";
   }

   # Add any commands before kernel
   foreach my $line (@{$self->GetPreCmds()}) {
      $s .= "\t$line\n";
   }

   # Specify kernel image and options
   # No boot image means no kernel line... non-UNIX menu option.
   if ($self->GetImage() ne "") {

      # uppermem tells GRUB the upper memory size in KBytes, where
      # "upper" is memory above the first MB.  See
      # http://mail.gnu.org/archive/html/bug-grub/2002-02/msg00064.html
      if ($self->GetMemSize()) {
         $s .= "\tuppermem " . ($self->GetMemSize() - 1) * 1024 . "\n";
      }

      $s .= "\tkernel ";

      # --no_mem_option prevents GRUB from inserting its own mem= thing
      if ($self->GetMemSize()) {
         $s .= "--no-mem-option ";
      }

      $s .= $self->GetImage() . " ";

      if ($self->GetReadOnly()) {
         $s .= "ro ";
      }

      if ($self->GetRootDev()) {
         $s .= "root=" . $self->GetRootDev() . " ";
      }

      if ($self->GetMemSize()) {
         $s .= "mem=" . $self->GetMemSize() . "M ";
      }
      if ($self->GetCPCI() ne "") {
         $s .= "cpci=" . $self->GetCPCI() . " ";
      }

      if ($self->GetOtherAppend()) {
         $s .= $self->GetOtherAppend();
      }

      $s .= "\n";
   }

   # Add initrd
   if ($self->GetInitrd()) {
      $s .= "\tinitrd " . $self->GetInitrd() . "\n";
   }

   # Add any commands after initrd
   foreach my $line (@{$self->GetPostCmds()}) {
      $s .= "\t$line\n";
   }

   return($s);
}

1;
