#############################################################
# Copyright 1998 VMware, Inc.  All rights reserved. -- VMware Confidential
#############################################################

#
# GRUBConfig.pm -- 
# 
# A class representing the grub.conf configuration file.
#

package VMware::Boot::GRUBConfig;

use VMware::Boot::GRUBBootImage;
use VMware::Log qw(:log);
use VMware::FileManip qw(FileSlurpArray);
use VMware::FileSys::Tmp qw(:tmp);
use VMware::FileSys::Move qw(MoveFileKeepDestMode);
use VMware::FileSys::StandardLocator qw(StdFile StdCommand);
use VMware::Tokenizer;
use strict;


########################################################################
#
# GRUBConfig::new --
#
#       Constructor. Creates a new grub config object.
#
# Results:
#       The new instance, or undef on error.
#
# Side effects:
#       None.
#
########################################################################

sub new
{
   my $class = shift;      # IN: Invoking class name.

   my $self = {
      globals => "",       # String of config data for global options
      images => [],        # Ref to an array of GRUBBootImage objects
      version => 0,        # Config format version
   };

   bless $self => $class;
   return $self;
}


#
# Trivial Accessor Functions
#

sub GetGlobals
{
   my $self = shift;
   return $self->{globals};
}

sub SetGlobals
{
   my $self = shift;
   my $value = shift;
   return $self->{globals} = $value;
}

sub GetImages
{
   my $self = shift;
   return $self->{images};
}

sub SetImages
{
   my $self = shift;
   my $value = shift;
   return $self->{images} = $value;
}

sub GetVersion
{
   my $self = shift;
   return $self->{version};
}

sub SetVersion
{
   my $self = shift;
   my $value = shift;
   return $self->{version} = $value;
}


########################################################################
#
# GRUBConfig::AddImage --
#
#       Adds an image to the boot menu.
#
# Results:
#       1 on success, 0 otherwise.
#
# Side effects:
#       Updates the images list.
#
########################################################################

sub AddImage
{
   my $self = shift;       # IN/OUT: Invoking instance.
   my $image = shift;      # IN: The GRUBBootImage to add.

   push @{$self->{images}}, $image;

   return 1;
}


########################################################################
#
# GRUBConfig::ToString --
#
#       Writes a new grub.conf into a string.
#
# Results:
#       The new string on success, undef otherwise.
#
# Side effects:
#       None.
#
########################################################################

sub ToString
{
   my $self = shift;       # IN: Invoking instance.

   #
   # Get the default image. This is the first entry in the
   # image list which is set to default.
   #

   my $defaultIndex = 0;
   my $imageIndex = 0;
   foreach my $image (@{$self->GetImages()}) {
      if ($image->GetDefault()) {
         $defaultIndex = $imageIndex;
      }

      $imageIndex++;
   }

   my $str = "";
   $str .= "#vmware:configversion " . $self->GetVersion() . "\n";
   $str .= $self->GetGlobals();
   $str .= "default=$defaultIndex\n";
   foreach my $image (@{$self->GetImages()}) {
      $str .= $image->ToString();
   }
  
   return $str;
}


########################################################################
#
# GRUBConfig::Commit --
#       Commit the grub configuration to disk.
#
# Results:
#       On success, writes out grub.conf and returns 1.
#       Returns 0 otherwise.
#
# Side effects:
#       Writes to /boot/grub/grub.conf
#
########################################################################

sub Commit
{
   my $self = shift;       # IN: Invoking instance.

   # Write to temporary grub.conf
   my ($tmpfile, $tmpfh) = TmpFile();
   if (not (defined($tmpfile) && defined($tmpfh))) {
      return 0;
   }

   print $tmpfh $self->ToString();

   unless (close($tmpfh)) {
      LogWarn("Could not close $tmpfile. Reason $!");
   }

   # Change grub.conf
   unless (MoveFileKeepDestMode($tmpfile, StdFile('grub_conf', 'all'))) {
      # Error
      LogError("Unable to create new grub.conf file.  Reason $!");
      return 0;
   }

   RemoveFileFromTmpCleanup($tmpfile);
   return 1;
}


########################################################################
#
# GRUBConfig::LoadFromSystem --
#
#       Load configuration data from the system grub.conf.
#
# Results:
#       1 on success, 0 otherwise.
#
# Side effects:
#       See GRUBConfig::LoadFromFile.
#
########################################################################

sub LoadFromSystem
{
   my $self = shift;       # IN/OUT: Invoking instance.

   my $filename = StdFile('grub_conf', 'all');
   return $self->LoadFromFile($filename);
}


########################################################################
#
# GRUBConfig::LoadFromFile --
#
#       Load configuration data from the specified file.
#
# Results:
#       1 on success, 0 otherwise.
#
# Side effects:
#       Reads the specified file and initializes the internal
#       configuration.
#
########################################################################

sub LoadFromFile
{
   my $self = shift;       # IN/OUT: Invoking instance.
   my $filename = shift;   # IN: The filename to load from.

   my $global = 1;  # State flag indicating that parser is looking for globals
   my $post = 0;    # State flag indicating we have passed "kernel" line in locals
   my $curlocal = undef;  # Local object

   my $lines = FileSlurpArray($filename);
   unless (defined $lines) {
      LogError("Could not read file '$filename'");
      return undef;
   }

   #
   # Set up the tokenizer to understand the grub.conf syntax
   #
   my $tokenizer = new VMware::Tokenizer();
   $tokenizer->setEscapeChar("\\");
   $tokenizer->setEscapeMapping({
      "\\\\" => "\\",
      '\n' => "\n",
      '\t' => "\t",
      '\"' => "\"",
   });
   $tokenizer->setDelimiters("=");

   #
   # First process the global options
   # Basic approach:
   #
   # For the global options, we are just going to copy everything
   # verbatim from grub.conf with the exception of the default
   # tag.  We are going to keep the default tag in case we need to
   # change it at the request of the user.
   # 
   # For the local options, we are going to save information regarding
   # the menu title, image and initrd files, as well as other kernel
   # parameters which we may wish to modify later, including memory size
   # and CPCI device allocation info.  Lines we don't care about are also
   # saved, but comments are dumped.
   #

   my $defaultIndex = 0;

   foreach my $line (@$lines) {
      chomp($line);

      #
      # Prepare to tokenize by reseting the count to zero before tokenizing
      #
      $tokenizer->resetCount();    
      my ($rc, $toks) = $tokenizer->tokenize($line);

      my @tokens = @$toks;

      unless ($rc) {
         #
         # Error in parsing
         #
         LogWarn("Could not parse line in grub.conf: $line");
      }

      #
      # Read global options until we get to the per-image options.
      #
      if ($global) {
         if (not @tokens) {
            #
            # Skip white space
            #
            next;
         } elsif ($tokens[0]->token() eq "title") {
            #
            # We have reached the start of the per-image options.
            #
            my $value = "";
            $global = 0;

            #
            # Everything after "title" should be taken verbatim
            #
            if ($line =~ /^\s*(\w+)\s+(.+)/) {
               $value = $2;
            }
            
            #
            # Make a new boot image
            #
            $curlocal = new VMware::Boot::GRUBBootImage();
            $curlocal->SetTitle($value);
            $self->AddImage($curlocal);
            next;
         } elsif ($#tokens == 2 && $tokens[1]->token() eq '=') {
            my $key = $tokens[0]->token();
            my $value = $tokens[2]->token();
            
            if ($key eq "default") {
               # We always take the last default that is defined in the file
               # in the event that there are multiple default tags.
               $defaultIndex = $value;
               next;
            }
         } elsif ($#tokens == 1 &&
                  $tokens[0]->token() eq '#vmware:configversion' &&
                  $tokens[1]->token() =~ /^\d+$/) {
            my $version = $tokens[1]->token();
            $self->SetVersion($version);
            next;
         }
         
         #
         # Uninteresting global option, save it and go on to the next line
         #
         $self->{globals} .= $line . "\n";
      } elsif ($#tokens > -1) {
         #
         # Global options are done. Handle per-image options.
         #
         my $key = $tokens[0]->token();

         #
         # Is this the next image?
         #
         if ($key eq "title") {
            #
            # Everything after "title" should be taken verbatim
            #
            my $value = "";
            if ($line =~ /^\s*(\w+)\s+(.+)\s*$/) {
               $value = $2;
            }
            
            #
            # Make a new boot image
            #
            $curlocal = new VMware::Boot::GRUBBootImage();
            $curlocal->SetTitle($value);
            $self->AddImage($curlocal);
            $post = 0;
            next;
         } elsif ($key eq "kernel") {
            #
            # parse kernel line with all its options
            #
            my $i = 1;
            $post = 1;
            my $append = "";

            #
            # Skip -- options right after kernel
            #
            while ($i <= $#tokens &&
                   $tokens[$i]->token() =~ /^--/ ) {
               $i++;
            }

            #
            # next is the kernel image
            #
            if ($i <= $#tokens) {
               my $image = $tokens[$i]->token();
               $curlocal->SetImage($image);
               $i++;
            }

            #
            # Loop through all remaining options
            #
            while ($i <= $#tokens) {
               #
               # Go through key=value pairs
               #
               if (($i+2) <= $#tokens &&
                   $tokens[$i+1]->token() eq '=') {
                  my $key = $tokens[$i]->token();
                  my $value = $tokens[$i+2]->token();

                  if ($key eq "root") {
                     # check for root=LABEL=/1, constructs, which create
                     # 5 tokens!  Also for UUID=<uuid> constructs.
                     if (($value eq "LABEL" or $value eq "UUID") &&
                         ($i+4) <= $#tokens &&
                         $tokens[$i+3]->token() eq '=') {
                        $curlocal->SetRootDev($value . '=' . $tokens[$i+4]->token());
                        $i += 2;
                     } else {
                        $curlocal->SetRootDev($value);
                     }
                  } elsif ($key eq "mem" && $value =~ /(\d+)M/ ) {
                     $curlocal->SetMemSize($1);
                  } elsif ($key eq "cpci" && $value =~ /[0-9:;,-]*/ ) {
                     $curlocal->SetCPCI($value);
                  } elsif ($key eq "BOOT_IMAGE") {
                     $curlocal->SetLabel($value);
                  } else {
                     $append .= "$key=$value ";
                  }

                  $i += 3;
                  next;
               } elsif ($tokens[$i]->token() eq "ro") {
                  $curlocal->SetReadOnly(1);
               } else {
                  #
                  # Other unrecognized option, add to append...
                  #
                  $append .= $tokens[$i]->token() . " ";
               }

               $i++;
            }

            #
            # Add any extra options
            #
            $curlocal->SetOtherAppend($append);

            next;
         } elsif ($#tokens == 1) {
            my $value = $tokens[1]->token();

            #
            # Is this the initrd?
            #
            if ($key eq "initrd") {
               my $initrd = $value;
               $curlocal->SetInitrd($initrd);
               next;
            }

            #
            # Is this the GRUB Root?
            #
            if ($key eq "root") {
               $curlocal->SetGRUBRoot($value);
               next;
            }

            #
            # ignore uppermem line, use mem= value on kernel line
            #
            if ($key eq "uppermem") {
               next;
            }

            #
            # directive to tell vmkbootcfg whether to replace/update entry
            #
            if ($key eq "#vmware:autogenerated") {
               $curlocal->SetAutoGenerated($value);
               next;
            }
         }

         #
         # Uninteresting image option, save it and go on to the next line
         #
         my $cmd = $line;
         $cmd =~ s/^\s+//;
         if ($post) {
            push(@{$curlocal->GetPostCmds()}, $cmd);
         } else {
            push(@{$curlocal->GetPreCmds()}, $cmd);
         }
      }
   }

   #
   # Set default image
   #

   my $imageIndex = 0;
   foreach my $image (@{$self->GetImages()}) {
      if ($imageIndex == $defaultIndex) {
         $image->SetDefault(1);
         last;
      }

      $imageIndex++;
   }

   return 1;
}

1;
