##########################################################################
# Copyright 2005 VMware, Inc.  All rights reserved. -- VMware Confidential
##########################################################################

#
# GRUBDeviceMap.pm --
#
#       A basic class that contains information representing a GRUB
#       device.map file.
#

package VMware::Boot::GRUBDeviceMap;

use VMware::FileManip qw(FileSlurpArray);
use VMware::FileSys::StandardLocator qw(StdFile);
use VMware::Log qw(:log);
use strict;


########################################################################
#
# GRUBDeviceMap::new --
#
#       Constructor for a VMware::Net::GRUBDeviceMap object.
#
# Results:
#       The new object, or undef on error.
#
# Side effects:
#       None.
#
########################################################################

sub new
{
   my $class = shift;      # IN: Invoking class.

   my $self = {
      mappings => {},
   };
   bless $self => $class;

   return $self;
}


########################################################################
#
# GRUBDeviceMap::AddMapping --
#
#       Add a new mapping GRUB device key to a device filename.
#       If the GRUB device key is already mapped, updates the old
#       mapping. If the GRUB device key is already mapped, and the
#       device filename is undefined, deletes the old mapping.
#
# Results:
#       None.
#
# Side effects:
#       None.
#
########################################################################

sub AddMapping 
{
   my $self = shift;    # IN/OUT: Invoking instance.
   my $key = shift;     # IN: The key to map.
   my $value = shift;   # IN: The device to map.

   if (defined $value) {
      $self->{mappings}->{$key} = $value;
   } else {
      delete $self->{mappings}->{$key};
   }
}


########################################################################
#
# GRUBDeviceMap::GetKeyForDevice --
#
#       Lookup the GRUB device key for the given device.
#       If more than one device key is mapped to a given device,
#       returns the first one in alphabetical order.
#
# Results:
#       The key on success, undef otherwise.
#
# Side effects:
#       None.
#
########################################################################

sub GetKeyForDevice
{
   my $self = shift;    # IN/OUT: Invoking instance.
   my $device = shift;  # IN: The device to lookup.

   foreach my $key (sort(keys(%{$self->{mappings}}))) {
      if ($self->{mappings}->{$key} eq $device) {
         return $key;
      }
   }

   return undef;
}


########################################################################
#
# GRUBDeviceMap::Commit --
#       Commit the grub device mapping to disk.
#
# Results:
#       On success, writes out the device map file and returns 1.
#       Returns 0 otherwise.
#
# Side effects:
#       Writes to /boot/grub/devices.map.
#
########################################################################

sub Commit
{
   my $self = shift;       # IN: Invoking instance.

   my $filename = StdFile('device_map', 'all');

   my $fh;
   unless (open($fh, ">$filename")) {
      LogError("Could not open file $filename: $!");
      return 0;
   }

   print $fh "# Automatically generated by $0\n";
   foreach my $key (sort keys %{$self->{mappings}}) {
      my $value = $self->{mappings}->{$key};
      print $fh "($key)     $value\n";
   }

   unless (close($fh)) {
      LogError("Could not close file $filename: $!");
      return 0;
   }

   return 1;
}


########################################################################
#
# GRUBDeviceMap::LoadFromSystem --
#
#       Load the grub device map from the system device.map.
#
# Results:
#       1 on success, 0 otherwise.
#
# Side effects:
#       See GRUBDeviceMap::LoadFromFile.
#
########################################################################

sub LoadFromSystem
{
   my $self = shift;       # IN/OUT: Invoking instance.

   my $filename = StdFile('device_map', 'all');
   return $self->LoadFromFile($filename);
}


########################################################################
#
# GRUBDeviceMap::LoadFromFile --
#
#       Load the device map from the specified file.
#
# Results:
#       1 on success, 0 otherwise.
#
# Side effects:
#       Reads the specified file and initializes the internal
#       mapping list.
#
########################################################################

sub LoadFromFile
{
   my $self = shift;       # IN/OUT: Invoking instance.
   my $filename = shift;   # IN: The filename to load from.

   my $lines = FileSlurpArray($filename);
   unless (defined $lines) {
      LogError("Could not read file '$filename'");
      return undef;
   }

   foreach my $line (@$lines) {
      #
      # Remove comments
      #

      $line =~ s/#.*//;

      #
      # Parse entries of the form:
      # (key)     device
      #

      if ($line =~ /^\(([^)]+)\)\s+(\S+)/) {
         my ($key, $device) = ($1, $2);
         $self->AddMapping($key, $device);
      }
   }

   return 1;
}

1;
