##########################################################################
# Copyright 2005 VMware, Inc.  All rights reserved. -- VMware Confidential
##########################################################################

#
# CmdTool --
#
#       Contains convenience functions for writing a
#       command line tool that relies on the common libraries.
#

package VMware::CmdTool;

use Getopt::Long;

use VMware::Log qw(:log :manip :count);
use VMware::Config qw(:config);
use VMware::Init qw(:init);
use VMware::Panic qw(Panic RegisterPanicCallback);
use VMware::FileSys::StandardLocator qw(StdFile StdDir);
use VMware::FileSys::Tmp qw(CleanTmpFiles);
use VMware::Config::VmacoreConfigObj;
use strict;

# The StdFile names of the log file and product. Used by Init and Finish.
my $toolLogFile = undef;
my $toolLogProduct = undef;


########################################################################
#
# InitFramework --
#
#       Initialize infrastructure like config and logging.
#
# Results:
#       True on success, false on failure.
#
# Side effects:
#       Log and configuration setup.
#
########################################################################

sub InitFramework
{
   my $cmdHash = shift;  # IN: The command-line options.
   my $cfgFile = shift;  # IN: The StdFile name of the config file.
   my $logFile = shift;  # IN: The StdFile name of the log file.
   my $product = shift;  # IN: The product to refer StdFile to.

   unless (InitConfig([StdFile($cfgFile, $product)],
                      undef,
                      $cmdHash->{cmdCfgFile},
                      $cmdHash->{cmdCfgOpts},
                      $cmdHash->{verbosity})) {
      LogError("Could not initialize script configuration.");
      return 0;
   }

   #
   # TODO: Can we rely on /var/log/vmware being present?
   # TODO: Need to make errors (and warnings?) visible in
   #       a user-friendly way.  For now dup to stderr.
   #
   my $logfn = ConfigGet("log.file", StdFile($logFile, $product));
   unless (InitLog($logfn, 1)) {
      LogWarn("Could not initialize script logfile, logging to stdout.");
   }

   my $log = LogPeek();
   $log->AddDestination(\*STDERR, "STDERR", VMware::Log::ID_ERROR);

   return 1;
}


########################################################################
#
# ParseOptions --
#
#       Parse program arguments, adding in standard system options.
#
# Results:
#       A reference to a hash of standard options.
#
# Side effects:
#       Configures getopt for bundling.
#
########################################################################

sub ParseOptions
{
   my $options = shift;          # IN: A hash reference mapping getopt
                                 #     specifications to value references.

   #
   # Verbosity defaults to undef so it is properly handled
   # in InitConfig(), which would treat a 0 as an override from
   # the --verbosity command line option.  undef is interpreted
   # to mean "default to the config file value".
   #
   my $cmdHash = {
      help => 0,           # Help flag.
      verbosity => undef,  # Verbosity of output
      cmdCfgFile => undef, # Override config file location.
      cmdCfgOpts => {},    # Override individual config options.
   };

   my $fullOptions = {
      # Tool-specific options
      %$options,

      # Note that $cmdHash->{cmdCfgOpts} is already a reference,
      # so don't take an additional reference of it (no \).
      "config-opt|C=s%" => $cmdHash->{cmdCfgOpts},
      "help|h" => \$cmdHash->{help},
      "verbosity=i" => \$cmdHash->{verbosity},
      "verbose|v+" => \$cmdHash->{verbosity},
      "config-file|c=s" => \$cmdHash->{cmdCfgFile},
   };

   Getopt::Long::Configure("bundling");
   unless (GetOptions(%$fullOptions)) {
      return undef;
   }

   return $cmdHash;
}

########################################################################
#
# InitTool --
#
#       Initialize the command line tool, including handling all
#       command-line processing.
#
# Results:
#       1 for success, 0 otherwise.
#       Dies with the usage statement on error.
#
# Side effects:
#       Log and configuration setup.
#       Dies with the usage statement on error.
# 
########################################################################

sub InitTool
{
   my $cfgFile = shift;          # IN: The StdFile name of the config file.
   my $logFile = shift;          # IN: The StdFile name of the log file.
   my $options = shift;          # IN: A hash reference mapping getopt
                                 #     specifications to value references.
   my $usage = shift;            # IN: A usage statement
   my $product = shift;          # IN: (Optional) The product to refer
                                 #     StdFile to. Defaults to esx.

   $product = "esx" unless defined $product;

   my $cmdHash = ParseOptions($options);
   unless ($cmdHash) {
      die $usage;
   }

   if ($cmdHash->{help}) {
      print $usage;
      exit(0);
   }

   unless (InitFramework($cmdHash, $cfgFile, $logFile, $product)) {
      LogError("Could not initialize tool framework");
      return 0;
   }

   # Set for later use in Finish
   $toolLogFile = $logFile;
   $toolLogProduct = $product;

   return 1;
}


########################################################################
#
# Finish --
#
#       Exit the program indicating success or failure, printing out
#       an appropriate success or failure message and exiting with the
#       correct error code.
#
#       Note that even if called with $success set
#       to true, it may still be determined that the program failed
#       if LogGetErrorCount returns a non-zero value. In this case
#       the result will be the same as if called with $success set to
#       false.
#
# Results:
#       Calls exit() with either 0 indicating success or -1 indicating
#       failure.
#
# Side effects:
#       Log and configuration setup.
#       Dies with the usage statement on error.
# 
########################################################################

sub Finish
{
   my $success = shift; # IN: Whether or not the program succeeded.
   my $quiet = shift;   # IN: Don't print to stdout if this is true.

   CleanTmpFiles();
   if ($success and LogGetErrorCount() == 0) {
      LogInfo("$0 completed successfully.");
      print "Success!\n" unless $quiet;
      exit 0;
   } else {
      my $logfile = ConfigGet("log.file",
                              StdFile($toolLogFile, $toolLogProduct));
      LogError("$0 failed, examine $logfile for details.");
      exit -1;
   }
}


########################################################################
#
# LockConfigFile --
#
#       Check that the config file can be written and then lock
#       and read it.
#
# Results:
#       True on success, false on failure.
#
# Side effects:
#       Filesystem activity (create lock file, read config).
#
########################################################################

sub LockConfigFile
{
   my $config = shift;     # IN/OUT: Config object to hold the data.
   my $file = shift;       # IN: Path of the config file.
   my $loadFile = shift;   # IN: (Optional) Whether or not to also load
                           #     the file. Defaults to true.

   unless (defined $loadFile) {
      $loadFile = 1;
   }

   #
   # We've been asked to perform a read-write operation.
   # Lock the configuration and read it in.
   #

   unless (-w $file || (!$loadFile && ! -e $file)) {
      LogError("Configuration file '$file' is not writable.");
   }
   my $locked = 0;
   my $numRetries = ConfigGet("boot.lockRetries", 1);
   for (1 .. $numRetries) {
      if ($loadFile) {
         $locked = $config->LockAndReadFile($file);
      } else {
         $locked = $config->LockFile($file);
      }

      if ($locked) {
         last;
      } else {
         sleep 1;
      }
   }

   unless ($locked) {
      LogError("Could not lock $file.  Please try again later.");
      return 0;
   }

   #
   # Try to ensure that the config file is unlocked in the event
   # a library function feel compelled to Panic().
   #

   my $panicUnlockSub = sub { $config->UnlockFile(); };
   RegisterPanicCallback($panicUnlockSub, [], "VMware::CmdTool");
   return 1;
}


########################################################################
#
# LoadConfig --
#
#       Load the main configuration tree and optionally
#       lock it for writing.
#
# Results:
#       The config tree on success, false on failure.
#
# Side effects:
#       Filesystem activity (create lock file, read config).
#
########################################################################

sub LoadConfig
{
   my $lockFile = shift;   # IN: (Optional) Lock the config file.
                           #     Defaults to false.
   my $configFile = shift; # IN: (Optional) The StdFile name of the 
                           #     config file to load. Defaults to
                           #     esx_config.
   my $product = shift;    # IN: (Optional) The product of the config file
                           #     to load. Defaults to esx.

   $configFile = "esx_config" unless defined $configFile;
   $product = "esx" unless defined $product;
   my $config = new VMware::Config::VmacoreConfigObj();
   my $file = StdFile($configFile, $product);
   unless (-r $file) {
      Panic("Configuration file '$file' is not readable or does not exist.");
   }

   my $success = undef;
   if ($lockFile) {
      $success = LockConfigFile($config, $file);
   } else {
      $success = $config->ReadFile($file);
   }

   unless ($success) {
      # XXX: We should add an option to not panic here.
      Panic("Could not load configuration file '$file' for " .
            " exclusive access.");
      return undef;
   }

   return $config;
}

1;
