##########################################################################
# Copyright 2003 VMware, Inc.  All rights reserved. -- VMware Confidential
##########################################################################

#
# Config.pm --
#
#	Module defining a "global" configuration space, through a
#	private instance of ConfigObj.
#

package VMware::Config;

use strict;

require Exporter;

@VMware::Config::ISA = qw(Exporter);

#
# Allow export of commonly used subroutines, and define a 'config' tag
# for importing the most common ones.
#

@VMware::Config::EXPORT_OK =
   qw(ConfigGet ConfigGetBoolean ConfigGetArray ConfigGetHash
      ConfigSet ConfigSetValidator);
%VMware::Config::EXPORT_TAGS =
   (config => [qw(ConfigGet ConfigGetBoolean ConfigGetArray ConfigGetHash)]);

use VMware::Config::ConfigObj;

########################################################################
# Define the instance and declare the validation instance.
########################################################################

my $config = VMware::Config::ConfigObj->new();


########################################################################
#
# Config::AUTOLOAD --
#
#       We cannot directly use or require any VMware::Log modules,
#       because they depend on us.  This is only really a problem
#       when we use Log while perl is evaluating Log using us.
#       So once we're done with that, we can load log functions
#       on demmand.  That way we can still use logging here.
#
#       All this means is that Config and ConfigObj cannot try
#       to make Logging calls in the compilation phase (like
#       during BEGIN blocks), nor can it use modules that do so.
#
# Results:
#       The results of the newly loaded routine.
#
# Side effects:
#       Loads the Log module.
#       Process death if the routine is still not defined after
#       the Log module has been loaded.
#
########################################################################

sub AUTOLOAD
{
   return if $VMware::Config::AUTOLOAD =~ m/::DESTROY$/;
   eval "use VMware::Log qw(:log :manip)";
   if ($@) {
      #
      # Its reasonable to die here, because that's what would
      # happen if a totally undefined subroutine call was
      # made anyway.
      #
      die $@ . "\n";
   }
   if (defined &$VMware::Config::AUTOLOAD) {
      goto &$VMware::Config::AUTOLOAD;

   } else {
      my $trace = "";
      my $i = undef;

      while (my ($package, $filename, $line, $subroutine) = caller($i++)) {
         $trace .= "$package $subroutine in $filename line $line\n";
      }
      print STDERR $trace;

      die "Undefined subroutine in VMware::Config: $VMware::Config::AUTOLOAD\n";
   }
}


########################################################################
#
# Config::LoadFromFile --
#
#	Loads a config file into the config instance.
#
# Results:
#       See ConfigObj::LoadFromFile.
#
# Side effects:
#       See ConfigObj::LoadFromFile.
#
########################################################################

sub LoadFromFile
{
   return $config->LoadFromFile(@_);
}


########################################################################
#
# Config::LoadFromHash --
#
#	Adds config values from a hash reference.
#
# Results:
#       See ConfigObj::LoadFromHash.
#
# Side effects:
#       See ConfigObj::LoadFromHash.
#
########################################################################

sub LoadFromHash
{
   return $config->LoadFromHash(@_);
}


########################################################################
#
# Config::ToHash --
#
# 	Returns a reference to a hash representing raw config values.
#
# Results:
#       See ConfigObj::ToHash.
#
# Side effects:
#       See ConfigObj::ToHash.
#
########################################################################

sub ToHash
{
   return $config->ToHash(@_);
}


########################################################################
#
# Config::Clear --
#
#	Clears the config space by removing all configuration keys.
#
# Results:
#       See ConfigObj::Clear.
#
# Side effects:
#       See ConfigObj::Clear.
#
########################################################################

sub Clear
{
   return $config->Clear(@_);
}


########################################################################
#
# Config::ConfigGet --
#
#	Public subroutine for retreiving configuration data.
#
# Results:
#       See ConfigObj::Get.
#
# Side effects:
#       See ConfigObj::Get.
#
########################################################################

sub ConfigGet
{
   return $config->Get(@_);
}


########################################################################
#
# Config::ConfigGetBoolean --
#
#	Public subroutine for retreiving configuration data as a
#	boolean value.
#
# Results:
#       See ConfigObj::GetBoolean.
#
# Side effects:
#       See ConfigObj::GetBoolean.
#
########################################################################

sub ConfigGetBoolean
{
   return $config->GetBoolean(@_);
}


########################################################################
#
# Config::ConfigGetArray --
#
#	Public subroutine for retreiving configuration data as an
#	array value.
#
# Results:
#       See ConfigObj::GetArray.
#
# Side effects:
#       See ConfigObj::GetArray.
#
########################################################################

sub ConfigGetArray
{
   return $config->GetArray(@_);
}


########################################################################
#
# Config::ConfigGetHash --
#
#	Public subroutine for retreiving configuration data as an
#	hash value.
#
# Results:
#       See ConfigObj::GetHash.
#
# Side effects:
#       See ConfigObj::GetHash.
#
########################################################################

sub ConfigGetHash
{
   return $config->GetHash(@_);
}


########################################################################
#
# Config::ConfigSet --
#
#	Public subroutine for setting configuration data.
#
# Results:
#       See ConfigObj::Set.
#
# Side effects:
#       See ConfigObj::Set.
#
########################################################################

sub ConfigSet
{
   return $config->Set(@_);
}


########################################################################
#
# Config::ConfigSetObj --
#
#       Blow away the entire config space and replace it with a new one.
#       Not usually done- Config::LoadFromHash is more often what you
#       want.
#
# Results:
#       The old ConfigObj.
#
# Side effects:
#       All current configuration is permanently erased, including
#       whatever validator was currently set.
#
########################################################################

sub ConfigSetObj
{
   my $old = $config;
   $config = shift;   # IN: The new ConfigObj instance.
   return $old;
}


########################################################################
#
# Config::ConfigSetValidator --
#
#       Set a given ConfigObj instance as the validator for the
#       global config.  For all subsequent loads, sets or gets
#       (until another call to this method), the name of the config
#       variable will be checked against the names in this config
#       instance.
#
#       To disable validation, pass undef as the validator.
#
# Results:
#       The old validator, if any.
#
# Side effects:
#       Sets a new validator.  Enables validation if no previous
#       validator was present.  Disables validation if the existing
#       validator is set to undef.
#
########################################################################

sub ConfigSetValidator
{
   return $config->SetValidator(@_);
}


1;

