##########################################################################
# Copyright 2003 VMware, Inc.  All rights reserved. -- VMware Confidential
##########################################################################

#
# ConfigObj.pm --
#
#	Object providing access to a configuration space.
#
#	The config space can be populated in two ways:
#
#	By providing an existing HASH reference; all keys are copied
#	from it into the object.
#
#	From a file of the following format:
#		One name = value pair on each line, where whitespace
#		on either side of the equals sign is trimmed.
#		Double-quotes can be used to include a literal equals,
#		hash mark "#", backslash "\", or leading or trailing whitespace
#		in a value.  Two adjacent double-quotes produce a single
#		literal double-quote.  Backslashes do not perform escapes.
#
#		Valid characters for config names are A-Z,a-z,0-9,[-_.:]
#
#		Empty config values are allowed.
#
#               If the final character on a line is a "\", then the
#               following line is assumed to be a continuation of the
#               value.  The continuation may not be within quotes, (quote
#               each line of the value individually), but it may be within
#               a comment as described below.
#               
#		Any text after a hash mark "#" (including the hash mark)
#		to the end of the line, and any blank lines, will be
#		treated as a comment, with 2 exceptions:
#
#		The first exception is a #! line on the first line of the file.
#		This will set two special "config.binary" and "config.opts"
#		names interpreted in the bourne shell style.
#           
#               The other exception is a final "\" character, which still
#               functions as a line continuation as described above.
#

package VMware::Config::ConfigObj;

use VMware::FileSys::Lock;
use strict;


########################################################################
#
# ConfigObj::AUTOLOAD --
#
#       We cannot directly use or require any VMware::Log modules,
#       because they depend on us.  This is only really a problem
#       when we use Log while perl is evaluating Log using us.
#       So once we're done with that, we can load log functions
#       on demmand.  That way we can still use logging here.
#
#       All this means is that Config and ConfigObj cannot try
#       to make Logging calls in the compilation phase (like
#       during BEGIN blocks), nor can it use modules that do so.
#
# Results:
#       The results of the triggering method;
#
# Side effects:
#       An unexpeced routine (one not defined in Log :log :manip
#       or :count) will cause process death.
#
########################################################################

{
   my $justOnce = 1;
   sub AUTOLOAD
   {
      return if $VMware::Config::ConfigObj::AUTOLOAD =~ m/::DESTROY$/;
      if ($justOnce) {
         $justOnce = 0;
         eval "use VMware::Log qw(:log :manip :count)";
         if ($@) {
            #
            # Its reasonable to die here, because that's what would
            # happen if a totally undefined subroutine call was
            # made anyway.
            #
            die $@ . "\n";
         }
         goto &$VMware::Config::ConfigObj::AUTOLOAD;
      } else {
         die "Attempted to call the undefined subroutine " .
             $VMware::Config::ConfigObj::AUTOLOAD . "\n";
      }
   }
}


########################################################################
#
# ConfigObj::new --
#
#	Creates a new, empty instance of configuration space. Two parameters
#       are taken. The first determines the case-sensitivity of the config
#       space. If true (the default) is passed in, then the configuration
#       space is case sensitive. If false is passed in, all values are
#       stored and retrieved using all lowercase names. The second parameter
#       is another ConfigObj with which validation may be performed. See
#       the comments for SetValidator for more information.
#
# Results:
#       ConfigObj instance.
#
# Side effects:
#       None.
#
########################################################################

sub new
{
   my $class  = shift; # IN: The class to which this instance will belong.
   my $strict = shift; # IN: The case sensitivity of the config object.

   if (!defined($strict)) {
      $strict = 1;
   }

   my $self = {
   	config => {},
        strict => $strict,
        validator => shift, # IN: Optional ConfigObj instance against
                            #     which names will be validated
                            #     upon loads, sets and gets.
   };

   return bless $self, $class;
}


########################################################################
#
# ConfigObj::LoadFromFile --
#
#	Loads a file into the configuration namespace, using the
#	format described above.
#
#	Existing configuration values will remain unless overwritten
#	by new values from the file with the same name.
#
#	The $ignore argument specifies whether to abort and cancel
#	on finding an invalid line.  A warning is logged either way.
#
#       Note that all validation (if enabled) is handled in the
#       call to LoadFromHash at the end of the method.
#
# Results:
#	True on success, undef on failure.  On failure, the namespace
#       will remain unchanged.
#
# Side effects:
#       On success, adds the contents of the file to the ConfigObj's
#       namespace.
#
########################################################################

sub LoadFromFile
{
   my $self = shift;            # IN/OUT: Invoking instance
   my $file = shift;		# IN: File to load from
   my $ignore = shift;		# IN: Set to true to ignore invalid lines
   my $validator = shift;       # IN: Optional ConfigObj for name validation.
   my $keepQuotes = shift;      # IN: Whether to preserve double quotes
                                # surrounding values
   my $prefix = shift;          # IN: prefix which must be matched for a
                                #     value to be loaded.
   return undef unless ($file);
   
   
   unless (open(CONFIG, "<$file")) {
      LogWarn("Config: Can't open $file: $!");
      return undef;
   }

   my $tempConfig = {};

   my $line;
   my $isFirst = 1;
   my $savedName = undef;
   my $savedValue = undef;
   my $dequote = not $keepQuotes;

   while ($line = <CONFIG>) {

      chomp($line);

      #
      # Handle bourne-style first lines
      #
      if ($isFirst) {
         if ($line =~ /^#!(\S+)\s*(.*)\s*$/) {	# Match #! followed by
	 					# non-whitespace characters,
						# optional whitespace, and
						# everything except trailing
                                                # whitespace after that.
	    my ($binary, $opts) = ($1, $2);
	    $tempConfig->{"config.binary"} = $binary;
	    $tempConfig->{"config.opts"} = $opts;
	 }
	 $isFirst = 0;
      }

      #
      # Skip blank lines, lines containing only a comment, or lines that 
      # don't match the prefix we've been passed.
      #
      next if ($line =~ /^\s*\#/ || $line =~ /^\s*$/);
      if (defined($prefix)) {
         next if ($line !~ /^\s*$prefix/);
      }

      #
      # See header for description of the config file line format.  This
      # matches the identifier, equals sign, and all text after it, trimming
      # the whitespace.  The value may be blank.
      #
      my $keyRegexp = $self->GetValidKeyRegexp();
      if ($line =~ /^\s*($keyRegexp)\s*=\s*(\S.*|)\s*$/) {
         my ($name, $value) = ($1, $2);

         if ($value =~ /\\$/) {
            chop($value);
            $savedName = $name;
            $savedValue =$self->TrimValue($value, $dequote);

         } else {
            $savedValue = $savedName = undef;
	    $tempConfig->{$name} = $self->TrimValue($value, $dequote);
         }

      } elsif ($savedValue && ($line =~ /^\s*(\S.*)\s*/)) {
         #
         # This is a continuation line for a value.
         #

         my $value = $1;
         if ($value =~ /\\$/) {
           chop($value);
           $savedValue .= $self->TrimValue($value, $dequote);

         } else {
           $tempConfig->{$savedName} = $savedValue .
                                       $self->TrimValue($value, $dequote);
         }

      } else {
	 LogWarn("Config: Unrecognized line in $file: $line");
         unless ($ignore) {
	    unless (close(CONFIG)) {
	       LogWarn("Config: Can't close $file: $!");
	    }
	    return undef;
	 }
      }
   }
   unless (close(CONFIG)) {
      LogWarn("Config: Can't close $file: $!");
      return undef;
   }

   return $self->LoadFromHash($tempConfig);
}


########################################################################
#
# ConfigObj::LoadFromHash --
#
#	Loads values from a pre-populated hash object into the
#	configuration namespace.  This can be used in conjunction
#	with Getopt::Long to include config values from the command
#	line, by passing a hash reference to GetOptions.
#
#	Existing configuration values will remain unless overwritten
#	by new values from the hash with the same name.
#
#       Optionally the names being loaded can be validated against
#       the names defined in another ConfigObj instance, with a
#       mismatch causing a failure.
#
# Results:
#	True on success, undef on failure.
#
# Side effects:
#       On success, adds the contents of the file to the ConfigObj's
#       namespace.
#
########################################################################

sub LoadFromHash
{
   my $self = shift;            # IN/OUT: Invoking instance
   my $hash = shift;		# IN: Hash ref to load from

   return undef unless (ref($hash) eq "HASH");

   my %origConfig = %{$self->{config}};

   foreach my $name (keys %$hash) {
      #
      # Calling Set() enforces rules on names and values.
      #

      unless ($self->Set($name, $hash->{$name})) {
         $self->{config} = \%origConfig;
	 return undef;
      }
   }

   return 1;
}


########################################################################
#
# ConfigObj::Get --
#
#	Returns a value from a configuration space by name.  A default
#	may be passed as a fallback when the name has no value.
#
# Results:
#	The value of the named config value if validation passed,
#       the default value if the name is not defined and validation
#       passed, and undef otherwise.
#
# Side effects:
#       Increments the warn log count if validation fails.  This is
#       the *only* way to distinguish between failed validation
#       and an undefined value if no default was supplied.
#
########################################################################

sub Get
{
   my $self = shift;            # IN: Invoking instance
   my $name = shift;		# IN: Name of config value to get
   my $default = shift;         # IN: Optional default value

   unless ($self->{strict}) {
      $name = lc($name);
   }

   if (defined $self->{validator}) {
      #
      # There is a loophole for log.*verbosity config options,
      # or else you can end up in an infite recursion of failing
      # config checks while trying to log that you're failing
      # a config check... etc.  In other words, verbosity is
      # always a valid config option.
      #
      # Don't call Get on the validator, because we might be
      # our own validator so we may recurse.
      #

      if ((not ($name =~ /^log\..*verbosity/)) &&
          (not exists $self->{validator}->{config}->{$name})) {
         LogWarn("Attempt to get invalid config variable $name.");
         return undef;
      }
   }

   my $value = $self->{config}->{$name};
   if ((not defined $value) and (defined $default)) {
      return $default;
   }
   return $value;
}


########################################################################
#
# ConfigObj::GetBoolean --
#
#	Looks up a configuration value and interprets it as a boolean
#	value.
#
# Results:
#	1 if the value is "true" ( == /true/i or "1")
#	defined 0 if the value is false,
#	undef if the value is not defined.
#
# Side effects:
#       None.
#
########################################################################

sub GetBoolean
{
   my $self = shift;     # IN: Invoking instance.
   my $name = shift;     # IN: Config variable name.
   my $default = shift;  # IN: Default value if undefined.

   #
   # Only reference Log functions if a validator is defined
   # to avoid compile-time circular references. See AUTOLOAD.
   #

   my $warnings = 0;
   if (defined $self->{validator}) {
      $warnings = LogGetWarnCount();
   }

   my $value = $self->Get($name);

   unless (defined($value)) {
      if (defined($self->{validator}) && ($warnings < LogGetWarnCount())) {
         return undef;
      }
      $value = $default;
   }

   if (defined($value) && ($value =~ /^true$/i || $value eq "1")) {
      return 1;
   } else {
      return 0;
   }
}


########################################################################
#
# ConfigObj::GetArray --
#
#	Looks up a configuration value and interprets it as a comma
#	separated list of values.  A default array reference may
#	be passed in, in case no value is found.
#
# Results:
#	An array of the values, or a reference to them if (!wantarray).
#
# Side effects:
#       None.
#
########################################################################

sub GetArray
{
   my $self = shift;            # IN: Invoking instance
   my $name = shift;            # IN: Parameter name
   my $default;

   if (@_ == 1 && ref($_[0]) eq "ARRAY") {
      $default = shift;         # IN: Default array reference
   } else {
      $default = \@_;           # IN: Default array values
   }

   #
   # Only reference Log functions if a validator is defined
   # to avoid compile-time circular references. See AUTOLOAD.
   #

   my $warnings = 0;
   if (defined $self->{validator}) {
      $warnings = LogGetWarnCount();
   }

   my $value = $self->Get($name);

   if (not defined $value) {
      if (defined($self->{validator}) && ($warnings < LogGetWarnCount())) {
         return wantarray ? () : undef;
      }

      if (defined($default)) {
         return wantarray ? @$default : $default;
      } else {
         return wantarray ? () : undef;
      }
   }

   my @values = split(/\s*\,\s*/, $value);
   return wantarray ? @values : \@values;
}


########################################################################
#
# ConfigObj::GetHash --
#
#	Looks up a configuration value and interprets it as a comma
#	separated list of name=value pairs.  If a string between commas
#	has no '=' sign, the string itself is the name and gets a value
#	of 1.
#
# Results:
#	A reference to a hash of the values, or undef if not set.
#
# Side effects:
#       None.
#
########################################################################

sub GetHash
{
   my $self = shift;            # IN: Invoking instance
   my $name = shift;            # IN: Parameter name
   my $defaultRef = shift;      # IN: Optional ref of hash for default value

   my %default = ref($defaultRef) eq "HASH" ? %$defaultRef : ();

   #
   # Only reference Log functions if a validator is defined
   # to avoid compile-time circular references. See AUTOLOAD.
   #

   my $warnings = 0;
   if (defined $self->{validator}) {
      $warnings = LogGetWarnCount();
   }

   my $value = $self->Get($name);

   if (not defined $value) {
      if (defined($self->{validator}) && ($warnings < LogGetWarnCount())) {
         return undef;
      }
        
      if (scalar(%default)) {
         return \%default;
      } else {
         return undef;
      }
   }

   my %result;
   foreach my $entry (split(/\s*\,\s*/, $value)) {
      if ($entry =~ /^(\w+)=(.*)$/) {
         $result{$1} = $2;
      } else {
         $result{$entry} = 1;
      }
   }

   return \%result;
}


########################################################################
#
# ConfigObj::GetValidKeyRegexp --
#
# Results:
#       A compiled regexp string that will match valid keys for the
#       config file format.
#
# Side effects:
#       None.
#
########################################################################

sub GetValidKeyRegexp
{
   return qr/[A-Za-z0-9_\-.:]+/;
}


########################################################################
#
# ConfigObj::Set --
#
#	Sets a value in a configuration space by name.  It refuses to set
#	any value which is not a plain scalar, or any name which does
#	not match the naming rules for configuration names.
#
# Results:
#	True on success, undef on failure.
#
# Side effects:
#       Alters the config variable's value.
#
########################################################################

sub Set
{
   my $self = shift;            # IN/OUT: Invoking instance
   my $name = shift;		# IN: Parameter name
   my $value = shift;		# IN: Value to set

   return undef if (ref($value));
   my $re = $self->GetValidKeyRegexp();
   return undef unless ($name =~ /^($re)$/);

   unless ($self->{strict}) {
      $name = lc($name);
   }

   if (defined $self->{validator}) {
      unless (exists $self->{validator}->{config}->{$name}) {
         LogWarn("Attempt to set invalid config variable $name.");
         return undef;
      }
   }

   $self->{config}->{$name} = $value;

   return 1;
}


########################################################################
#
# ConfigObj::UnSet --
#
#	Removes a value from the configuration space by name. 
#
# Results:
#	The value that was removed, undef if no value was removed.
#
# Side effects:
#       None.
#
########################################################################

sub UnSet
{
   my $self = shift;            # IN/OUT: Invoking instance
   my $name = shift;		# IN: Parameter name

   return undef if $self->{keyLocked};

   my $re = $self->GetValidKeyRegexp();
   return undef unless ($name =~ /^($re)$/);

   unless ($self->{strict}) {
      $name = lc($name);
   }

   my $ret = $self->{config}->{$name};
   delete $self->{config}->{$name} if (defined($ret));

   return $ret;
}


########################################################################
#
# ConfigObj::SetValidator --
#
#       Set a given ConfigObj instance as the validator for this
#       instance.  For all subsequent loads, sets or gets (until
#       another call to this method), the name of the config variable
#       will be checked against the names in this config instance.
#
#       To disable validation, pass undef as the validator.
#
# Results:
#       The old validator, if any.
#
# Side effects:
#       Sets a new validator.  Enables validation if no previous
#       validator was present.  Disables validation if the existing
#       validator is set to undef.
#
########################################################################

sub SetValidator
{
   my $self = shift;   # IN/OUT: Invoking instance.
   my $newVal = shift; # IN: The new validator.

   my $oldVal = $self->{validator};
   $self->{validator} = $newVal;
   return $oldVal;
}


########################################################################
#
# ConfigObj::Clear --
#
#	Clears all names and values from the configuration space.
#
# Results:
#       Always returns true.
#
# Side effects:
#       Clears all names and values from the configuration space.
#
########################################################################

sub Clear
{
   my $self = shift;            # IN/OUT: Invoking instance

   $self->{config} = {};

   return 1;
}


########################################################################
#
# ConfigObj::ToString --
#
#	Returns a string describing the configuration space,
#	suitable for display or writing to a file.
#
# Results:
#       A pretty string.
#
# Side effects:
#       None.
#
########################################################################

sub ToString
{
   my $self = shift;            # IN: Invoking instance
   my $quotes = shift;          # IN: Whether to surround values in double
                                # quotes.  Defaults to true.

   my $str;

   if ($self->{config}->{"config.binary"}) {
      $str .= "#!".$self->{config}->{"config.binary"};

      if ($self->{config}->{"config.opts"}) {
	 $str .= " ".$self->{config}->{"config.opts"};
      }
      $str .= "\n";
   }

   my $quote = (!defined($quotes) || $quotes) ? "\"" : "";

   foreach my $name (sort keys %{$self->{config}}) {
      next if ($name =~ /^(?:config\.binary|config\.opts)$/);
      my $value = defined($self->{config}->{$name}) ?
                  $self->{config}->{$name} : "";
      $str .= $name."=$quote".$value."$quote\n";
   }

   return $str;
}


########################################################################
#
# ConfigObj::ToHash --
#
#	Returns a hash ref that is a copy of the contents of the object.
#
# Results:
#       A ref to a hash copy of the ConfigObj contents.
#
# Side effects:
#       Makes a copy.
#
########################################################################

sub ToHash
{
   my $self = shift;            # IN: Invoking instance

   my %copy = %{$self->{config}};

   return \%copy;
}


########################################################################
#
# ConfigObj::SaveToFile --
#
#       Saves the contents of the config object to a file that can be
#       read back again, using ToString.
#
# Results:
#       True on success, undef on failure.
#
# Side effects:
#       Writes a file.
#
########################################################################

sub SaveToFile
{
   my $self = shift;            # IN: Invoking instance
   my $fileName = shift;        # IN: File to save to
   my $quotes = shift;          # IN: Whether to quote values

   unless (open(CONFIG, ">$fileName")) {
      LogWarn("Failed to open $fileName: $!");
      return undef;
   }
   
   #
   # Make sure there is something and if so write it out
   # 
   if (scalar (keys %{$self->ToHash()}) >= 1) { 
      unless (print CONFIG $self->ToString($quotes)) {
         LogWarn("Failed to write config to file '$fileName': $!");
         return;
      }
   }

   unless (close(CONFIG)) {
      LogWarn("Failed to close $fileName: $!");
      return undef;
   }

   return 1;
}


########################################################################
#
# ConfigObj::TrimValue --
#
#       Trims comments and trailing whitespace from values read out
#       of a file, and de-quotes them as needed.
#
# Results:
#       The trimmed value, sans comments, quotes and unquoted whitespace.
#
# Side effects:
#       None.
#
########################################################################

sub TrimValue
{
   my $self = shift;            # IN: Invoking instance
   my $value = shift;           # IN: The value to trim
   my $dequote = shift;         # IN: Whether to trim quotes

   $dequote = 1 unless (defined $dequote);

   #
   # Trim the comment and any trailing whitespace.
   #
   $value =~ s/\#[^\"]*$//;
   $value =~ s/\s*$//;

   if ($dequote && $value =~ /^\"(.*)\"$/) {
      $value = $1;
      $value =~ s/\"\"/\"/g;
   }
   return $value;
}


########################################################################
#
# ConfigObj::LockKeys --
#
#       Prevents further modification to the set of keys in this
#       ConfigObj by making it its own validator.  Any previous
#       validator is saved to be restored on unlock.  Subsequent
#       calls to LockKeys are ignored until after unlock.
#
#       This is a little different than just doing this yourself
#       with SetValidator in that there is a flag that is set
#       which primarily prevents deletion of current keys.  Otherwise,
#       its perfectly OK to delete keys because valid keys are not
#       required to exist.  You can still set the key to "undef",
#       you just can't delete it with ConfigObj->UnSet().
#
#       NOTE:  This has *no* effect on the values in the config space,
#              nor does it have anything to do with the backing file.
#              See LockFile and UnlockFile to obtain a lock on the
#              backing file while making changes.
#
# Results:
#       No return value.  The config key space is locked.
#
# Side effects:
#       The old validator is saved and the keyLocked member flag is set.
#
########################################################################

sub LockKeys
{
   my $self = shift;    # IN/OUT: Invoking instance.

   return if $self->{keyLocked};
   $self->{savedValidator} = $self->SetValidator($self);
   $self->{keyLocked} = 1;
}


########################################################################
#
# ConfigObj::UnlockKeys --
#
#       Allow keys to be changed (as allowed by whatever validator
#       was previously present, if any).
#
#       NOTE:  This has *no* effect on the values in the config space,
#              nor does it have anything to do with the backing file.
#              See LockFile and UnlockFile to obtain a lock on the
#              backing file while making changes.
#
# Results:
#       No return value.  The config key space is unlocked.
#
# Side effects:
#       Previous validation settings are restored.
#
########################################################################

sub UnlockKeys
{
   my $self = shift;    # IN/OUT: Invoking instance.

   return unless $self->{keyLocked};
   $self->SetValidator($self->{savedValidator});
   undef($self->{savedValidator});
   undef($self->{keyLocked});
}


########################################################################
#
# ConfigObj::ReadFile --
#
#       Use LoadFromFile to read the given file, and replace the
#       current contents of this ConfigObj with its contents.
#
#       If you wish to merge a file into the current contents,
#       call LoadFromFile directly.
#
# Results:
#       True on success, false otherwise.
#
# Side effects:
#       The contents of the ConfigObj are replaced with the file's
#       contents on success, and left unchanged otherwise.
#
########################################################################

sub ReadFile
{
   my $self = shift;  # IN/OUT: Invoking instance.
   my $file = shift;  # IN: The file from which to read.
   my $ignore = shift;          # IN: Set to true to ignore invalid lines
   my $validator = shift;       # IN: Optional ConfigObj for name validation.
   my $keepQuotes = shift;      # IN: Whether to preserve double quotes
                                #     surrounding values
   my $prefix = shift;          # IN: prefix which must be matched for a
                                #     value to be loaded.
   if ($self->{keyLocked}) {
      #
      # Cannot nuke config space and read new one with locked keys.
      #
      LogWarn("Attempted to read a config file with keys locked.");
      return undef;
   }

   my $old = $self->ToHash();
   $self->Clear();
   unless ($self->LoadFromFile($file, $ignore, $validator,
                               $keepQuotes, $prefix)) {
      #
      # Oops.  Back out.
      #

      LogWarn("Could not read config file '$file', restoring previous " .
               "contents.");
      $self->LoadFromHash($old);
      return undef;
   }
   $self->{file} = $file;
   return 1;
}


########################################################################
#
# ConfigObj::WriteFile --
#
#       Uses SaveToFile to write back to whatever file was
#       last read (or a new file if that is overidden).
#
# Results:
#       True on success, false otherwise.
#
# Side effects:
#       Writes out the config file, replacing its previous contents.
#
########################################################################

sub WriteFile
{
   my $self = shift;  # IN: Invoking instance.
   my $file = shift;  # IN: Optional dest file, defaults to whatever
                      #     was last given to ReadFile.
   my $quotes = shift; # IN: See SaveToFile's $quotes parameter.

   $file = $self->{file} unless defined $file;

   return $self->SaveToFile($file, $quotes);
}


########################################################################
#
# ConfigObj::LockFile --
#
#       Lock the config file.
#
# Results:
#       True on success, False but defined (0) if the lock could not
#       be acquired, undef on other failure.
# 
# Side effects:
#       See FileSys::Lock::LockFile.
#
########################################################################

sub LockFile
{
   my $self = shift;    # IN/OUT: Invoking instance.
   my $file = shift;    # IN: The backing file to lock.

   if (defined $self->{lockFile}) {
      if ($self->{lockFile}->GetFile() eq $file) {
         LogInfo("Redundant lock request, ignoring.");
         return 1;
      } else {
         LogWarn("Cannot lock two config files at once.");
         return undef;
      }
   }

   my $lockFile = new VMware::FileSys::Lock();
   my $result = $lockFile->LockFile($file);
   if ($result) {
      $self->{lockFile} = $lockFile;
   }

   return $result;
}


########################################################################
#
# ConfigObj::UnlockFile --
#
#       Unlock the config file.
#
# Results:
#       True on success, undef on failure.
#
# Side effects:
#       See FileSys::Lock::UnlockFile.
#
########################################################################

sub UnlockFile
{
   my $self = shift;    # IN/OUT: Invoking instance.

   if (not defined($self->{lockFile})) {
      LogInfo("Asked to unlock file, but no file is locked. Ignoring.");
      return 1;
   }

   my $result = $self->{lockFile}->UnlockFile();
   if ($result) {
      $self->{lockFile} = undef;
   }

   return $result;
}


########################################################################
#
# ConfigObj::LockAndReadFile --
#
#       Convenience function that just calls LockFile and ReadFile.
#       If the read fails, we release the lock on the file.
#
# Results:
#       True if successful, false but defined (0) if the lock could not
#       be acquired because someone else holds it, or undef on failure.
#
# Side effects:
#       The file is locked as with LockFile and the contents of that
#       file then replace the current contents of this file as with
#       ReadFile.  The lock is released if the file cannot be read.
#
########################################################################

sub LockAndReadFile
{
   my $self = shift;    # IN/OUT: Invoking instance.
   my $file = shift;    # IN: The backing file to read and lock.
   my $ignore = shift;          # IN: Set to true to ignore invalid lines
   my $validator = shift;       # IN: Optional ConfigObj for name validation.
   my $keepQuotes = shift;      # IN: Whether to preserve double quotes
                                #     surrounding values
   my $prefix = shift;          # IN: prefix which must be matched for a
                                #     value to be loaded.

   my $rc = $self->LockFile($file);
   return $rc unless $rc;

   unless ($self->ReadFile($file, $ignore, $validator, $keepQuotes, $prefix)) {
      #
      # UnlockFile will log its own warnings, so no need to recheck since
      # we are already returning undef if we get here at all.
      #

      LogWarn("Releasing lock as file '$file' could not be read.");
      $self->UnlockFile();
      return undef;
   }
   return 1;
}


########################################################################
#
# ConfigObj::WriteAndUnlockFile --
#
#       Convenience method calling WriteFile() and UnlockFile().
#
# Results:
#       True on success, false on failure.
#
# Side effects:
#       File is written and lock is released (lock file removed).
#
########################################################################

sub WriteAndUnlockFile
{
   my $self = shift;   # IN/OUT: Invoking instance.
   my $file = shift;   # IN: Optional dest file, defaults to whatever
                       #     was last given to ReadFile.
   my $quotes = shift; # IN: See SaveToFile's $quotes parameter.

   unless ($self->WriteFile($file, $quotes)) {
      LogWarn("Could not write to config file '$self->{file}', " .
               "retaining lock.");
      return undef;
   }
   unless ($self->UnlockFile()) {
      return undef;
   }
}


########################################################################
#
# ConfigObj::Filter --
#
#       Produce a hash of config key=>value pairs where the key
#       matches the supplied regular expression.  This allows
#       the caller to work on a subset of the config and then
#       submit the changes in bulk using either Merge() or
#       LoadFromHash().
#
# Results:
#       A hash ref containing the key=>value pairs for all matching keys.
#       Note that this is not a true deep copy, so if your values are
#       references (not typical with config) then they will refer to
#       the same targets.
#
# Side effects:
#       None.
#
########################################################################

sub Filter
{
   my $self = shift;   # IN: Invoking instance.
   my $regexp = shift; # IN: Regular expression to filter keys.
   my $delete = shift; # IN: If true, remove matching keys from $self.

   my @newKeys = grep { $_ =~ /$regexp/; } keys %{$self->{config}};
   my $newHash = {};
   foreach my $k (@newKeys) {
      $newHash->{$k} = $self->{config}->{$k};
      delete $self->{config}->{$k} if $delete;
   }
   return $newHash;
}


########################################################################
#
# ConfigObj::Merge --
#
#       Merge a hash of config key=>value paris with this ConfigObj.
#       This differs from LoadFromHash in two important ways:
#
#       1.  You can use a regexp to limit the set of keys affected.
#       2.  Keys in the current ConfigObj that match the regexp but
#           do not exist in the new ConfigObj are deleted.
#
#       The regexp is applied to both the ConfigObj and the hash
#       that is passed in.
#
#       Calling this with the universal regexp is equivalent to
#       but more expensive than clearing the config and doing
#       a LoadFromHash().
#
#       Note:  If you have deleted or added keys in the hash you
#              are loading, you cannot Merge() it back in if you
#              have locked the keys of this ConfigObj.
#
# Results:
#       True on success, false on failure (due to locked or
#       invalid keys).
#
# Side effects:
#       The subset of keys matching the regexp may be changed, added
#       or deleted.
#
########################################################################

sub Merge
{
   my $self = shift;         # IN/OUT: Invoking instance.
   my $regexp = shift;       # IN: Regular expression limiting the set
                             #     of keys to merge.  Defaults to .* (all keys).
   my $hash = shift;         # IN: Hash to merge.  The same regexp is
                             #     applied to this hash.
   my $exceptRegexp = shift; # IN: An optional regexp used to limit the
                             #     merge.  Keys matching this regexp will
                             #     be left undisturbed in the config.

   my @hashKeys = grep { /$regexp/; } keys(%$hash);
   my @cfgKeys = grep { /$regexp/; } keys(%{$self->{config}});

   if (defined $exceptRegexp) {
      @hashKeys = grep { not /$exceptRegexp/; } @hashKeys;
      @cfgKeys = grep { not /$exceptRegexp/; } @cfgKeys;
   }

   my %cfgKeysSeen = map { ($_ => 0); } @cfgKeys;

   #
   # Go through Set() and UnSet() to enforce validator/keylock semantics.
   #

   my %origConfig = %{$self->{config}};
   foreach my $hk (@hashKeys) {
      $cfgKeysSeen{$hk} = 1 if exists $cfgKeysSeen{$hk};
      unless ($self->Set($hk, $hash->{$hk})) {
         LogInfo("Config merge failed: could not set key '$hk'.");
         %{$self->{config}} = %origConfig;
         return undef;
      }
   }

   foreach my $ck (@cfgKeys) {
      unless ($cfgKeysSeen{$ck}) {
         #
         # Technically, this will fail in the pathological
         # case of unsetting a key whose value is undef (as the
         # value is the return fromUnSet), but its rather hard to
         # get the system into that state so this should do.
         #

         unless (defined $self->UnSet($ck)) {
            LogInfo("Config merge failed: could not delete key '$ck'.");
            %{$self->{config}} = %origConfig;
            return undef;
         }
      }
   }
   return 1;
}


1;

