##########################################################################
# Copyright 2005 VMware, Inc.  All rights reserved. -- VMware Confidential
##########################################################################

#
# VMKConfigObj.pm --
#
#       VMware::Config::ConfigObj subclass that uses a very slightly
#       different format than ConfigObj:  keys may contain "/".
#       This variation is used by the ESX 2.x /etc/vmware/vmkconfig
#       file.  It seemed easier to do this than to change the
#       expected behavior of the well-established ConfigObj class.
#
#       All other behavior is inherited directly from ConfigObj.
#                  

package VMware::Config::VMKConfigObj;

use strict;

use VMware::Config::ConfigObj;

@VMware::Config::VMKConfigObj::ISA = qw(VMware::Config::ConfigObj);


########################################################################
#
# VMKConfigObj::AUTOLOAD --
#
#       We cannot directly use or require any VMware::Log modules,
#       because they depend on us.  This is only really a problem
#       when we use Log while perl is evaluating Log using us.
#       So once we're done with that, we can load log functions
#       on demmand.  That way we can still use logging here.
#
#       All this means is that Config and VMKConfigObj cannot try
#       to make Logging calls in the compilation phase (like
#       during BEGIN blocks), nor can it use modules that do so.
#
# Results:
#       The results of the triggering method;
#
# Side effects:
#       An unexpeced routine (one not defined in Log :log :manip
#       or :count) will cause process death.
#
########################################################################

{
   my $justOnce = 1;
   sub AUTOLOAD
   {
      return if $VMware::Config::VMKConfigObj::AUTOLOAD =~ m/::DESTROY$/;
      if ($justOnce) {
         $justOnce = 0;
         eval "use VMware::Log qw(:log :manip :count)";
         if ($@) {
            #
            # Its reasonable to die here, because that's what would
            # happen if a totally undefined subroutine call was
            # made anyway.
            #
            die $@ . "\n";
         }
         goto &$VMware::Config::VMKConfigObj::AUTOLOAD;
      } else {
         die "Attempted to call the undefined subroutine " .
             $VMware::Config::VMKConfigObj::AUTOLOAD . "\n";
      }
   }
}


########################################################################
#
# VMKConfigObj::GetValidKeyRegexp --
#
# Results:
#       A compiled regexp string that will match valid keys for the
#       config file format.
#
# Side effects:
#       None.
#
########################################################################

sub GetValidKeyRegexp
{
   my $invocant = shift;  # IN: Invoking class or instance.

   my $superRegexp = $invocant->SUPER::GetValidKeyRegexp();

   #
   # We want the initial character to be valid from the base
   # class, followed by any number of base class valid characters
   # plus '/' characters.  Doing the regexp this way introduces some
   # extra bits but is correct and will be robust to changes in
   # the parent class.
   #

   return qr/$superRegexp(?:\/|$superRegexp)+/;
}


1;
