##########################################################################
# Copyright 2005 VMware, Inc.  All rights reserved. -- VMware Confidential
##########################################################################

#
# VmacoreConfigObj.pm --
#
#       VMware::Config::ConfigObj subclass that uses the file format
#       that the VIM team's vmacore library uses for flat (non-XML)
#       config files.  This format is also used by VmkCtl and various
#       scripts running in the ESX Console OS.
#
#       See bora/vim/lib/public/vmacore/config.h and
#       bora/vim/lib/vmacore/main/config.cpp for format details.
#       The current interpretation here is:
#          Keys are like filesystem paths: /node/node/node etc.
#          A node may have an id indexing it: /node/node[id]/node
#          (This code only enforces the initial /.)
#          Whitespace around the = is permitted.
#          Values must be quoted in double quotes.
#          No escape sequences (embedded quotes are OK as literals).
#       The vmacore library does not currently explicitly support
#       comments, but we support shell/perl-style '#' comments,
#       which may be at the end of a data line or their own line.
#       See PR 68884 which tracks support for this on the C++ side.
#
#       Keys in the config space may or may not be case sensitive
#       depending on constructor arguments (set FMT_CASE_INSENSITIVE).
#       This is not an option in vmacore, but is inherited from ConfigObj.
#
#       Also inherited is the ability to use a validator on the config
#       namespace.
#                  

package VMware::Config::VmacoreConfigObj;

use strict;

use VMware::Config::ConfigObj;

@VMware::Config::VmacoreConfigObj::ISA = qw(VMware::Config::ConfigObj);


########################################################################
#
# VmacoreConfigObj::AUTOLOAD --
#
#       We cannot directly use or require any VMware::Log modules,
#       because they depend on us.  This is only really a problem
#       when we use Log while perl is evaluating Log using us.
#       So once we're done with that, we can load log functions
#       on demmand.  That way we can still use logging here.
#
#       All this means is that Config and VmacoreConfigObj cannot try
#       to make Logging calls in the compilation phase (like
#       during BEGIN blocks), nor can it use modules that do so.
#
# Results:
#       The results of the triggering method;
#
# Side effects:
#       An unexpeced routine (one not defined in Log :log :manip
#       or :count) will cause process death.
#
########################################################################

{
   my $justOnce = 1;
   sub AUTOLOAD
   {
      return if $VMware::Config::VmacoreConfigObj::AUTOLOAD =~ m/::DESTROY$/;
      if ($justOnce) {
         $justOnce = 0;
         eval "use VMware::Log qw(:log :manip :count)";
         if ($@) {
            #
            # Its reasonable to die here, because that's what would
            # happen if a totally undefined subroutine call was
            # made anyway.
            #
            die $@ . "\n";
         }
         goto &$VMware::Config::VmacoreConfigObj::AUTOLOAD;
      } else {
         die "Attempted to call the undefined subroutine " .
             $VMware::Config::VmacoreConfigObj::AUTOLOAD . "\n";
      }
   }
}


########################################################################
#
# VmacoreConfigObj::LoadFromFile
#
#       Reads config from a vmacore flat config file format.
#       See the large comment at the top of this file for the format
#       description.
#
#       This subclass ignores several parameters respected by the
#       superclass as they are not part of the vmacore format.
#       See the parameter docs for details.
#       TODO: Make this nicer somehow.
#
# Results:
#       True on success, undef on failure.  On failure, the namespace
#       will remain unchanged.
#
# Side effects:
#       On success, adds the contents of the file to the VmacoreConfigObj's
#       namespace.
#
########################################################################

sub LoadFromFile
{
   my $self = shift;       # IN/OUT: Invoking instance
   my $file = shift;       # IN: File to load from
   my $ignore = shift;     # IN: Suppress errors from malformed lines.
                           #     NOTE: Currently, this subclass always
                           #           acts as if this is true, as does vmacore.
                           #     TODO: Decide if we should bring this back.
   my $keepQuotes = shift; # IN: Keep quotes in values.
                           #     NOTE: Currently, this subclass ignores
                           #           this option (not a vmacore option).
                           #     TODO: Decide if we should bring this back.
   my $validator = shift;  # IN: Optional ConfigObj for name validation.
   my $prefix = shift;     # IN: prefix which must be matched for a
                           #     value to be loaded.
                           #     NOTE: Currently ignored (not a vmacore option).
                           #     TODO: Decide if we should bring this back.

   local *CONFIG;
   unless (open(CONFIG, "<$file")) {
      LogError("Config: Can't open $file: $!");
      return undef;
   }

   my $tempConfig = {};
   while (my $line = <CONFIG>) {
      #
      # Trim newlines and skip comment lines.
      #

      chomp($line);
      next if ($line =~ /^\s*#/);

      my $keyRegexp = $self->GetValidKeyRegexp();
      my ($name, $value) = ($line =~ /^($keyRegexp)\s*=\s*"(.*)"\s*$/);

      if (not defined($name) or not defined($value)) {
         #
         # TODO: What now?  vmacore has a XXX: comment here
         # as well.  For now, we will LogWarn.
         #

         LogWarn("Malformed config file line in $file:\n$line");
      } else {
         $tempConfig->{$name} = $value;
      }
   }

   unless (close(CONFIG)) {
      LogError("Config: Can't close $file: $!");
      return undef;
   }

   return $self->LoadFromHash($tempConfig);
}


########################################################################
#
# VmacoreConfigObj::GetValidKeyRegexp --
#
# Results:
#       A compiled regexp string that will match valid keys for the
#       config file format.
#
# Side effects:
#       None.
#
########################################################################

sub GetValidKeyRegexp
{
   return qr/\/.*\S/;
}


########################################################################
#
# VmacoreConfigObj::ToString --
#
#       Returns a string describing the configuration space,
#       suitable for display or writing to a file.
#
# Results:
#       A pretty string.
#
# Side effects:
#       None.
#
########################################################################

sub ToString
{
   my $self = shift;   # IN: Invoking instance

   # NOTE: The parent class has a 2nd parameter about adding
   # or not adding quotes.  We ignore that parameter as vmacore
   # config files must have quotes.  Our LoadFromFile also enforces
   # this on the other end.  TODO: Clean up this interface?

   my $str = "";

   foreach my $name (sort keys %{$self->{config}}) {
      $str .= $name . " = \"" . $self->{config}->{$name} . "\"\n";
   }

   return $str;
}


########################################################################
#
# VmacoreConfigObj::GetTree --
#
#       Retrieve the information in a config file in a tree
#       form, given a list of initial path components to the
#       root of the desired tree.  The "tree" is a structure of
#       hash and array refs based on the config file as follows:
#
#       Vmacore config keys are '/' separated, so each component
#       represents a level of hash keys where the values are either
#       another hash keyed by the next level down, an array indexed
#       by the index numbers when the key is of the form foo[0001]
#       (i.e. uses a *numeric* index), or the actual string value of
#       the config line if we've reached a leaf.
#
#       That's confusing.  As an example, if we give ["foo", "fun"] as
#       our list of path components to request from the following config data:
#
#       /foo/fun/aaa = "1"
#       /foo/fun/bbb = "2"
#       /foo/fun/ccc/yyy = "happy"
#       /foo/fun/ccc/zzz = "sleepy"
#       /foo/fun/bar[0000]/id = "20"
#       /foo/fun/bar[0000]/data = "qwerty"
#       /foo/fun/bar[0001]/id = "42"
#       /foo/fun/bar[0001]/data = "zaphod"
#       /foo/fun/zip[abc] = "zorak"
#       /foo/fun/zip[def] = "brak"
#       /foo/funlist/intersection = "flooded"
#       /other/stuff = "don't care"
#       /other/foo/fun/ddd = "nothing"
#
#       We would get the following perl structure:
#
#       {
#          aaa => 1,
#          bbb => 2,
#          ccc => {
#             yyy => "happy",
#             zzz => "sleepy",
#          },
#          bar => [
#             {
#                id  => 20,
#                data => "qwerty",
#             },
#             {
#                id => 42,
#                data => "zaphod",
#             },
#          ],
#          'zip[abc]' => "zorak",
#          'zip[def]' => "brak",
#       };
#
#       Note that "foo" and "fun" don't appear here.  Presumably, since you
#       filtered for them, you can associate them with the tree yourself
#       if you want to.
#
#       Note also the difference between the /foo/fun/bar and /foo/fun/zip
#       entries- the bar entries get special treatment due to their
#       numeric indexes, while the zip entries do not.  An indexing scheme
#       that mixes numeric and non-numeric will probably cause a failure,
#       which is arguably a bug but one that is not currently a problem.
#       Use traditional ConfigObj access to work around this.
#
#       If a numerically-indexed array is a component, specify the name
#       with empty brackets
#       to get all array entries.
#       So searching for ["foo", "fun", "bar[]"] would produce
#
#       [
#          {
#             id => 20,
#             data => "qwerty",
#          },
#          {
#             id => 42,
#             data => "zaphod",
#          }
#       ]
#
#       Note that in this example, the reference returned is an array
#       ref instead of a hash ref.
#
#       You can also get just one array entry the normal way, by
#       specifying the whole key, i.e. ["foo", "fun", "bar[0000]"] will
#       just get the 0th entry {id => 20, data => "querty"}.
#
#       Since you must always filter starting at the root element,
#       you don't get anything from the /other/foo/fun/ddd = "nothing" line.
#       And since only whole path components are matched, the
#       /foo/funlist/ line is filtered out as well.
#
#       If you pass an empty list, or no list at all, the whole config
#       space will be converted to a tree.
#
# Results:
#       The tree as described above.
#
# Side effects:
#       None.
#
########################################################################

sub GetTree
{
   my $self = shift;     # IN: Invoking instance.
   my $complist = shift; # IN: Initial path of components for which to filter.

   $complist = [] if not defined $complist;
   my $prefixSize = @$complist;
   my $arrayRoot = @$complist && $complist->[-1] =~ /\[\]$/;

   my $subcfg = $self->Filter($self->BuildPathRegexp($complist));
   return ($arrayRoot ? [] : {}) unless (scalar(keys(%$subcfg)));

   my $tree = {};
   my $curRef = \$tree;
   foreach my $key (keys %$subcfg) {
      #
      # Strip off the empty component you get because of
      # the leading '/', plus the components that were part
      # of our query.  That way we construct a hash holding only
      # things *below* our queried components.
      #
      # Don't take off the last key if we're requesting an array,
      # because it includes the array indices, and we obviously need those.
      # In that case, we build the "tree" *including* the name of the
      # last queried component, and then strip off that name and return
      # the array to which it points at the end.  This simplifies the code.
      #

      my @comps = split('/', $key);
      shift(@comps) if $comps[0] eq "";
      splice(@comps, 0, ($arrayRoot ? $prefixSize - 1 : $prefixSize));
      foreach my $c (@comps) {
         if ($c =~ /(.+)\[(\d+)\]$/) {
            my $name = $1;
            my $index = $2;
            if (!exists($$curRef->{$name})) {
               $$curRef->{$name} = [];
            }
            if (!exists($$curRef->{$name}->[$index])) {
               $$curRef->{$name}->[$index] = {};
            }
            $curRef = \$$curRef->{$name}->[$index];
         } else {
            if (!exists($$curRef->{$c})) {
               $$curRef->{$c} = {};
            }
            $curRef = \$$curRef->{$c};
         }
      }
      $$curRef = $subcfg->{$key};
      $curRef = \$tree;
   }

   if ($arrayRoot) {
      #
      # If we built something rooted at an array, then we will only have
      # one named key, which is the subscripted name.  It will point
      # to the array ref holding the requested data.
      #

      my ($key) = keys(%$tree);
      $tree = $tree->{$key};
   }

   return $tree;
}


########################################################################
#
# VmacoreConfigObj::SetTree --
#
#       The inverse of GetTree.  Given the same sort of path
#       (i.e. a list of path components), and a tree of values,
#       root the tree at that path and collapse the result into
#       the proper config format.  Then Merge() the resulting
#       config back in.  Because of how Merge() works, this has
#       the effect of replacing the entire old tree with the new one,
#       including removing old keys that are no longer used.
#
#       In addition, this can take an array of exception arrays.
#       Each exception array is an array of the same form as the
#       component list, except that paths matching both the component
#       list and any exception list are left alone in the config
#       file.  Empty list elements denote a wildcard for that component
#       level.  So:
#
#       $cfg->SetTree(["device"], $newData, [["device", "", "owner"],
#                                            ["device", "", "options"]]);
#
#       will set the tree rooted at /device/, but will not touch
#       entries such as /device/foo/owner or /device/bar/options/baz.
#
#       SetTree understands the same "foo[]" working with numeric
#       array-style keys as GetTree when it comes to the final path
#       component and the exception lists.
#
#       However, "foo[]" form components cannot be passed as non-final
#       path components because there is no way to tell what keys should
#       be set in that case.
#
#       If the tree reference is an array, it will silently assume
#       that the tree it is setting should be array-indexed even if
#       the named component does not include the [].
#
# Results:
#       True on success, false on failure (due to locked or invalid
#       keys or improper use of array wildcards).
#
# Side effects:
#       Changes the key/value pairs in the config corresponding
#       to the tree.
#
########################################################################

sub SetTree
{
   my $self = shift;     # IN/OUT: Invoking instance.
   my $complist = shift; # IN: Array of path components to set.
   my $tree = shift;     # IN: The new tree values.
   my $exceptions = (shift || []);  # IN: Array of arrays of path components to
                                    #     exempt from overwriting.

   for (my $i = 0; $i < @$complist; ++$i) {
      if (($i != $#$complist) && $complist->[$i] =~ /\[\]$/) {
         LogError("Attempt to set tree with undefined array path.");
         return undef;
      }
   }

   $complist = [] if not defined $complist;
   my $path = '';
   if (ref($tree) eq "ARRAY") {
      if (@$complist > 1) {
         $path .= '/' . join('/', @$complist[0 .. ($#$complist - 1)]);
      }
      my $arrayName = $complist->[-1];
      $arrayName =~ s/\[\]$//;
      $tree = { $arrayName => $tree };
   } else {
      $path .= '/' . join('/', @$complist) if @$complist;
   }

   if (not ref($tree)) {
      #
      # We're just doing a fancy set without having to know what the
      # separator format is.  So tree is a simple value, so set it.
      #

      return $self->Set($path, $tree);
   }

   my $subcfg = {};

   my $exceptRe = undef;
   if (@$exceptions) {
      my @exceptReList;
      foreach my $exc (@$exceptions) {
         #
         # Build up a regexp to match the particular exception.
         # Specified components are escaped with \Q \E for regexp use.
         # Array-wildcard components need digit matching inside the [].
         # Wildcard components get the non-greedy .*? to match them.
         #

         for (my $i = 0; $i < @$exc; ++$i) {
            if ($exc->[$i]) {
               my $comp = $exc->[$i];
               if ($comp =~ /(.*\[)(\])$/) {
                  $exc->[$i] = qr/\Q$1\E\d+\Q$2\E/;
               } else {
                  $exc->[$i] = qr/\Q$comp\E/;
               }
            } else {
               $exc->[$i] = qr/.*?/;
            }
         }
         my $str = '/' . join('/', @$exc);
         push(@exceptReList, qr/$str/);
      }

      #
      # Build the master exception regexp, which is an OR of all
      # of the individual exceptions.
      #

      my $exceptStr = join('|', @exceptReList);
      $exceptRe = qr/^(?:$exceptStr)/;
   }

   #
   # The top level must be a hash ref, because the name of
   # an array always goes into a hash, with the array below it.
   # You cannot have an array standing on its own.
   #

   foreach my $key (keys(%$tree)) {
      $self->ProcessHash($key, $path, $tree, $subcfg);
   }
   return $self->Merge($self->BuildPathRegexp($complist), $subcfg, $exceptRe);
}


########################################################################
#
# VmacoreConfigObj::ProcessHash --
#
#       Helper routine for SetTree for handling a hash ref node in
#       the tree.
#
# Results:
#       No return code.
#       The new config hash we are building will gain entries.
#
# Side effects:
#       None.
#
########################################################################

sub ProcessHash
{
   my $self = shift;       # IN: Invoking instance
   my $curKey = shift;     # IN: Current key from $curRef hash.
   my $keyStr = shift;     # IN: Key we are building.
   my $tree = shift;       # IN: Current tree position.
   my $subCfg = shift;     # IN/OUT: Config hash we are building.

   foreach my $key (keys(%$tree)) {
      my $newKeyStr .= "$keyStr/$curKey";

      if (ref $tree->{$curKey} eq "HASH") {
         foreach my $newKey (keys(%{$tree->{$curKey}})) {
            $self->ProcessHash($newKey, $newKeyStr, $tree->{$curKey}, $subCfg);
         }

      } elsif (ref $tree->{$curKey} eq "ARRAY") {
         $self->ProcessArray($newKeyStr, $tree->{$curKey}, $subCfg);
      } elsif (defined $tree->{$curKey}) {
         $subCfg->{$newKeyStr} = $tree->{$curKey};
      }
   }
}


########################################################################
#
# VmacoreConfigObj::ProcessArray --
#
#       Helper routine for SetTree for handling an array ref node in
#       the tree.
#
# Results:
#       No return code.
#       The new config hash we are building will gain entries.
#
# Side effects:
#       None.
#
########################################################################

sub ProcessArray
{
   my $self = shift;       # IN: Invoking instance
   my $keyStr = shift;     # IN: Key we are building.
   my $array = shift;      # IN: Current array in the tree.
   my $subCfg = shift;     # IN/OUT: Config hash we are building.

   for (my $i = 0; $i < @$array; ++$i) {
      next if not exists $array->[$i];
      my $newKeyStr = $keyStr . sprintf('[%04d]', $i);

      if (ref($array->[$i]) eq "HASH") {
         foreach my $newKey (keys(%{$array->[$i]})) {
            $self->ProcessHash($newKey,
                               $newKeyStr,
                               $array->[$i],
                               $subCfg);
         }
      } elsif (defined $array->[$i]) {
         $subCfg->{$newKeyStr} = $array->[$i];
      }
   }
}

   
########################################################################
#
# VmacoreConfigObj::BuildPathRegexp --
#
#       Helper method for GetTree and SetTree both of which need
#       to build a regular expression representing the keys they
#       need to match out of a list of components.  See GetTree
#       for a description of the path component format rules.
#
# Results;
#       The regular expression with the path components properly
#       quoted and adjusted.
#
# Side effects:
#       None.
#
########################################################################

sub BuildPathRegexp
{
   my $self = shift;     # IN: Invoking instance (unused);
   my $complist = shift; # IN: Array of path components.

   #
   # Quote components individually.  Components indicating array
   # format with the [] suffix need a regexp bit inserted to match properly.
   # Everything else needs to be quoted as non-regexp characters.
   #

   my @quotedComplist = ();
   foreach my $comp (@$complist) {
      if ($comp =~ /(.*\[)(\])$/) {
         push(@quotedComplist, qr/\Q$1\E\d+\Q$2\E/);
      } else {
         push(@quotedComplist, qr/\Q$comp\E/);
      }
   }

   my $path = '^/';
   $path .= join('/', @quotedComplist) if @quotedComplist;
   return $path;
}


1;
