use strict;

use File::Basename;
use File::Spec;
use Test;

BEGIN {
   plan tests => 29;

   my $path = File::Spec->catfile(dirname($0), "..", "..", "..");
   push @INC, $path;
   push @INC, ".";
}

use VMware::Config::VmacoreConfigObj;

my $file = <<EOD;
/boot/resMem = "700"
/enabled = "true"
/net/routes/console/route[0000]/gateway = "10.17.237.253"
/net/routes/console/route[0000]/network = "default"
/net/routes/kernel/route[0000]/gateway = "10.17.237.253"
/net/routes/kernel/route[0000]/network = "default"
/net/routes/test = "false"
/net/vmkernelnic/child[0002]/dhcp = "false"
/net/vmkernelnic/child[0002]/ipv4address = "10.17.237.104"
/net/vmkernelnic/child[0002]/ipv4broadcast = "10.17.237.255"
/net/vmkernelnic/child[0002]/ipv4netmask = "255.255.255.0"
/net/vmkernelnic/child[0002]/mac = "00:05:69:11:ed:68"
/net/vmkernelnic/child[0002]/mtu = "1514"
/net/vmkernelnic/child[0002]/portgroup = "default"
/net/vmkernelnic/child[0002]/sharedIface = ""
/net/vswifs/vswif0 = "/etc/sysconfig/network-scripts/ifcfg-vswif0"
/net/vswifs/vswif1 = "/etc/sysconfig/network-scripts/ifcfg-vswif1"
/net/vswitch/child[0000]/name = "kmk"
/net/vswitch/child[0000]/portgroup/child[0000]/name = "kmk"
/net/vswitch/child[0000]/portgroup/child[0000]/vlanId = "1"
/net/vswitch/child[0001]/name = "default"
/net/vswitch/child[0001]/portgroup/child[0000]/name = "default"
/net/vswitch/child[0001]/portgroup/child[0000]/vlanId = "0"
/test/node with space/foo = "name with"embedded""quotes"
EOD

my $testFile = File::Spec->catfile(dirname($0), "vmacore.conf");
open(FILE, ">$testFile") or die "Could not open $testFile: $!\n";
print FILE ($file) or die "Could not print to $testFile: $!\n";
close(FILE) or die "Could not close $testFile: $!\n";

my $cfg = new VMware::Config::VmacoreConfigObj::();
$cfg->LoadFromFile($testFile);
my $result = $cfg->ToString();

#
# Note that in order for this to work, the test conf file must
# have a single space to either side of the equals because that
# is how ToString formats things.
#

ok($result, $file);

ok($cfg->GetTree(['net', 'routes', 'test']), $cfg->Get("/net/routes/test"));
ok($cfg->SetTree(['net', 'routes', 'test'], "true"));
ok($cfg->Get("/net/routes/test"), "true");

#
# Set back for later test.
#
$cfg->Set("/net/routes/test", "false");

my $tree = $cfg->GetTree();
my $newCfg = new VMware::Config::VmacoreConfigObj::();
ok($newCfg->SetTree([], $tree), 1);
ok($newCfg->ToString(), $file);

$tree = $cfg->GetTree(["net", "routes"]);
ok(defined($tree));
ok($tree->{console}->{route}->[0]->{gateway}, "10.17.237.253");
ok($tree->{console}->{route}->[0]->{network}, "default");
ok($tree->{kernel}->{route}->[0]->{gateway}, "10.17.237.253");
ok($tree->{kernel}->{route}->[0]->{network}, "default");
ok($tree->{test}, "false");

$tree->{console}->{route}->[0]->{gateway} = "1.2.3.4";
$tree->{console}->{route}->[0]->{network} = "a";
$tree->{kernel}->{route}->[0]->{gateway} = "5.6.7.8";
$tree->{kernel}->{route}->[0]->{network} = "b";
undef($tree->{test});

ok($cfg->SetTree(["net", "routes"], $tree,
                 [["net", "routes", "", "", "gateway"],
                  ["net", "routes", "kernel"]]), 1);

#
# Only the console network should be changed, as the other three are
# masked out by one or both exceptions.
#

$tree = $cfg->GetTree(["net", "routes"]);
ok(defined($tree));
ok($tree->{console}->{route}->[0]->{gateway}, "10.17.237.253");
ok($tree->{console}->{route}->[0]->{network}, "a");
ok($tree->{kernel}->{route}->[0]->{gateway}, "10.17.237.253");
ok($tree->{kernel}->{route}->[0]->{network}, "default");
ok(!exists($tree->{test}));

$tree = $cfg->GetTree(["net", "vswitch", "child[]"]);
ok(ref($tree), "ARRAY");
ok(scalar(@$tree), 2);
ok($tree->[0]->{name}, "kmk");
ok($tree->[0]->{portgroup}->{child}->[0]->{vlanId}, 1);
ok($tree->[1]->{name}, "default");
ok($tree->[1]->{portgroup}->{child}->[0]->{vlanId}, 0);

delete($tree->[1]);
$tree->[0]->{name} = "brak";
$tree->[2] = {
   name => "zorak",
   portgroup => {
      child => [
         {
            name => "zorak",
            vlanId => 0,
         }
      ],
   },
};
ok($cfg->SetTree(["net", "vswitch", "child[]"], $tree));
ok(keys(%{$cfg->Filter(qr/^\/net\/vswitch\/child\[0*1\]/, $tree)}), 0);
ok($cfg->Get("/net/vswitch/child[0000]/name"), "brak");
ok($cfg->Get("/net/vswitch/child[0002]/portgroup/child[0000]/vlanId"), 0);

unlink($testFile) or die "Could not remove test file: $!\n";

my $fileWithComments = <<EOD;
# Intro comment.
/line/with/no = "comment"
   # Interior comment.
/more/stuff = "fun"
/not/a/comment = "foo#bar"
#/commented/entry = "blah"
EOD

my $fileWithNoComments = <<EOD;
/line/with/no = "comment"
/more/stuff = "fun"
/not/a/comment = "foo#bar"
EOD

open(FILE, ">$testFile") or die "Could not open $testFile: $!\n";
print FILE ($fileWithComments) or die "Could not print to $testFile: $!\n";
close(FILE) or die "Could not close $testFile: $!\n";

my $comments = new VMware::Config::VmacoreConfigObj::();
$comments->LoadFromFile($testFile);
my $commentString = $cfg->ToString();
ok($comments->Get("/not/a/comment"), "foo#bar");

unlink($testFile) or die "Could not remove test file: $!\n";

open(FILE, ">$testFile") or die "Could not open $testFile: $!\n";
print FILE ($fileWithNoComments) or die "Could not print to $testFile: $!\n";
close(FILE) or die "Could not close $testFile: $!\n";

my $noComments = new VMware::Config::VmacoreConfigObj::();
$noComments->LoadFromFile($testFile);
my $noCommentString = $cfg->ToString();

unlink($testFile) or die "Could not remove test file: $!\n";

ok(($commentString ne ""));
ok($commentString, $noCommentString);

