##########################################################################
# Copyright 2003 VMware, Inc.  All rights reserved. -- VMware Confidential
##########################################################################

#
# DataStruct.pm --
#
#       Subs for dealing with nested reference data structures.
#

package VMware::DataStruct;

use strict;

require Exporter;

@VMware::DataStruct::ISA = qw(Exporter);

@VMware::DataStruct::EXPORT_OK = qw(Clone);


###########################################################################
#
# DataStruct::Clone --
#
#       Given a reference, makes a deep-copy of it recursively.
#
#       NOTE - This is a workaround for certain limited distributions
#       of perl 5.005.  Newer Perls can use the dclone routine in the
#       Storable package.  Therefore, there are some limitations to this
#       routine:
#
#         - Objects are not handled at all.  Just SCALAR, HASH, and ARRAY
#         - Self-referencing structures will inifitely recurse.
#
# Results:
#       A deep copy of the original reference.
#
# Side effects:
#       None.
#
###########################################################################

sub Clone
{
   my $ref = shift;    # IN: A reference to a data structure to clone.
   my $norefs = shift; # IN: Boolean if true drop refs
   
   unless ($norefs) { 
      return $ref unless (ref($ref) && ref($ref) =~ /^(?:SCALAR|ARRAY|HASH)$/);
   } elsif (ref($ref)) {
      return unless (ref($ref) =~ /^(?:SCALAR|ARRAY|HASH)$/);
   } else {
      return $ref;
   }

   if (ref($ref) eq "SCALAR") {
      my $scalar = $$ref;
      return $scalar;
   } elsif (ref($ref) eq "ARRAY") {
      my @array = map { Clone($_, $norefs) } @$ref;
      return \@array;
   } else {
      my %hash = %$ref;
      foreach my $key (keys %hash) {
	 $hash{$key} = Clone($ref->{$key}, $norefs);
      }
      return \%hash;
   }
}


1;
