###########################################################################
# Copyright 2003 VMware, Inc.  All rights reserved. -- VMware Confidential
###########################################################################

#
# FileManip.pm --
#
#	Subroutines for manipulating the contents of files
#

package VMware::FileManip;

use strict;

use VMware::Log qw(:log);

use POSIX qw(strftime);

require Exporter;

@VMware::FileManip::ISA = qw(Exporter);

@VMware::FileManip::EXPORT_OK =
   qw(FileSlurp FileSlurpArray SearchAndReplace Search);

%VMware::FileManip::EXPORT_TAGS =
   (filemanip => [qw(FileSlurp FileSlurpArray SearchAndReplace Search)]);


###########################################################################
#
# FileManip::FileSlurp --
#
#	Given a filename, returns a reference to a scalar containing all
#	text from the file. 
#
# Results:
#       A reference to the contents of the file (as one big string).
#       undef on failure.
#
# Side effects:
#       File I/O.
#
###########################################################################

sub FileSlurp
{
   my $filename = shift;  # IN: The file to slurp.
   my $nonfatal = shift;  # IN: If true, return the contents even if
                          #     close fails.  The caller can determine
                          #     errors by examining the error log count.

   return FileSlurpInternal($filename, $nonfatal, 0);
}


###########################################################################
#
# FileManip::FileSlurpArray --
#
#       Given a filename, returns a reference to an array containing all of
#       the lines from the file. 
#
# Results:
#       A reference to the contents of the file (as an array of strings).
#       undef on failure.
#
# Side effects:
#       File I/O.
#
###########################################################################

sub FileSlurpArray
{
   my $filename = shift;  # IN: The file to slurp.
   my $nonfatal = shift;  # IN: If true, return the contents even if
                          #     close fails.  The caller can determine
                          #     errors by examining the error log count.

   return FileSlurpInternal($filename, $nonfatal, 1);
}


###########################################################################
#
# FileManip::FileSlurpInternal --
#
#       Implements both FileSlurp and FileSlurpArray.
#
# Results:
#       A reference to a scalar or array containing the file contents.
#       undef on failure.
#
# Side effects:
#       File I/O.
#
###########################################################################

sub FileSlurpInternal
{
   my $filename = shift;  # IN: The file to slurp.
   my $nonfatal = shift;  # IN: If true, return the contents even if
                          #     close fails.  The caller can determine
                          #     errors by examining the error log count.
   my $array = shift;     # IN: If true, return a ref to an array
                          #     containing the line-by-line content.
                          #     Else return a ref to a scalar of the whole file.

   local *FILE;
   unless (open (FILE, $filename)) {
      LogError("Can't open file $filename: $!");
      return undef;
   }

   my $ref;
   if ($array) {
      my @contents = <FILE>;
      $ref = \@contents;
   } else {
      local $/ = undef;
      my $contents = <FILE>;
      $ref = \$contents;
   }

   unless (close FILE) {
      LogError("Can't close file $filename: $!");
      return undef unless $nonfatal;
   }
   return $ref;
}


###########################################################################
#
# FileManip::SearchAndReplace --
#
#       Given a filename, searches and replaces all instances of
#       some text with some other text.  The text to match is interpreted
#       as a regular expression.
#
# Results:
#       True on success, undef on failure.
#
# Side effects:
#       Edits the file as described above.
#
###########################################################################

sub SearchAndReplace
{
   my $filename = shift;        # IN: file to search
   my $find = shift;            # IN: text or regexp to match
   my $replace = shift;         # IN: text to replace matches with

   my $found = 0;

   my $fileContentsRef = FileSlurp($filename);

   unless ($fileContentsRef) {
      return undef;
   }

   $$fileContentsRef =~ s/$find/$replace/g;

   local *FILE;
   unless (open(FILE, ">$filename")) {
      LogError("Couldn't write $filename: $!");
      return undef;
   }
   print FILE $$fileContentsRef;
   unless (close(FILE)) {
      LogError("Couldn't close $filename: $!");
      return undef;
   }

   return 1;
}


###########################################################################
#
# FileManip::Search --
#
#       Given a string return a true if the string is in the file, and 
#       return undef if it is not in the file.
#       
#
# Results:
#       True on success, undef on failure.
#
# Side effects:
#       None
#
###########################################################################

sub Search
{
   my $filename = shift;	# IN: file to search
   my $find = shift;            # IN: text or regexp to match
   my $case = shift;            # IN: true for case insensitivity

   my $fileContentsRef = FileSlurp($filename);

   unless ($fileContentsRef) {
      return undef;
   }

   if ($case) {
      if ($$fileContentsRef =~ m/$find/i) {
         return 1;
      } else {
         return undef;
      }
   }
   else {
      if ($$fileContentsRef =~ m/$find/) {
         return 1;
      } else {
         return undef;
      }
   }

}


1;
