##########################################################################
# Copyright 2005 VMware, Inc.  All rights reserved. -- VMware Confidential
##########################################################################

#
# FstabInfo.pm --
#
#       Utility functions for working with /etc/fstab.
#

package VMware::FileSys::FstabInfo;

use VMware::FileSys::Move qw(MoveFileKeepDestMode);
use VMware::FileSys::StandardLocator qw(StdFile);
use VMware::FileSys::Tmp qw(:tmp);
use VMware::FileSys::UuidCache;
use VMware::Log qw(:log);

require Exporter;

use strict;

@VMware::FileSys::FstabInfo::ISA = qw(Exporter);
@VMware::FileSys::FstabInfo::EXPORT_OK = qw(UpgradeFstab);
%VMware::FileSys::FstabInfo::EXPORT_TAGS =
   (all => [(@VMware::FileSys::FstabInfo::EXPORT_OK)]);


#######################################################################
#
# FstabInfo::UpgradeFstabEntry --
#
#       Private utility function. Converts an entry line from fstab to use
#       UUID specifications.
#
# Results:
#       Returns the new line if successful, undef otherwise.
#
# Side effects:
#       None.
#
#######################################################################

sub UpgradeFstabEntry
{
   my $line = shift;    # IN: The entry line to upgrade
   my $ucache = shift;  # IN: The uuid cache

   # Strip out comments
   $line =~ s/#.*$//;

   # Strip out leading/trailing whitespace
   $line =~ s/^\s*//;
   $line =~ s/\s*$//;

   # Skip empty lines
   unless ($line) {
      return undef;
   }

   my @fields = split(/\s+/, $line);
   unless ($#fields == 5) {
      # Not a valid record;
      return undef;
   }

   # Lookup device specification
   my $uentry = $ucache->LookupSpec($fields[0]);
   unless ($uentry) {
      # Device not listed in UUID cache, leave as is.
      return undef;
   }

   $fields[0] = $uentry->ToUuidSpec();
   return join("\t", @fields) . "\n";
}


#######################################################################
#
# FstabInfo::UpgradeFstab --
#
#       Converts all entries in /etc/fstab to use UUID specifications.
#
# Results:
#       1 for success, 0 otherwise.
#
# Side effects:
#       Reads all uuids on the system (through UuidCache)
#
#######################################################################

sub UpgradeFstab
{
   # Initialize uuid cache
   my $ucache = new VMware::FileSys::UuidCache();
   unless ($ucache) {
      LogError("Could not initialize UUID cache");
      return 0;
   }

   # Where is /etc/fstab?
   my $fstab = StdFile("fstab", "all");
   unless (defined $fstab) {
      LogError("Could not get /etc/fstab helper");
      return 0;
   }

   # Read from original fstab
   my $infh = undef;
   unless (open($infh, $fstab)) {
      LogError("Failed to open $fstab: $!");
      return 0;
   }

   # Write to temporary fstab
   my ($tmpfile, $tmpfh) = TmpFile();
   if (not (defined($tmpfile) && defined($tmpfh))) {
      return 0;
   }

   while (<$infh>) {
      my $line = $_;

      my $fsent = UpgradeFstabEntry($line, $ucache);
      if (defined $fsent) {
         print $tmpfh $fsent;
      } else {
         # Not an upgradable entry. Leave it as it is.
         print $tmpfh $line;
      }
   }

   unless (close($tmpfh)) {
      LogError("Failed to close $fstab: $!");
      return 0;
   }
   
   # Change fstab
   unless (MoveFileKeepDestMode($tmpfile, $fstab)) {
      LogError("Unable to create new fstab file.  Reason $!");
      return 0;
   }

   RemoveFileFromTmpCleanup($tmpfile);

   return 1;
}


1;
