##########################################################################
# Copyright 2005 VMware, Inc.  All rights reserved. -- VMware Confidential
##########################################################################

#
# Lock.pm --
#
#       This class represents a file lock. The lockfile format used
#       is (mostly) compatible with bora/lib/vmkctl/config
#
#       TODO: Implement locking for Win32. Implement readlocks.
#

package VMware::FileSys::Lock;
use strict;

use Fcntl;
use Errno qw(:POSIX);

########################################################################
#
# Lock::AUTOLOAD --
#
#       We cannot directly use or require any VMware::Log modules,
#       because they depend on us.  This is only really a problem
#       when we use Log while perl is evaluating Log using us.
#       So once we're done with that, we can load log functions
#       on demmand.  That way we can still use logging here.
#
#       All this means is that Lock cannot try
#       to make Logging calls in the compilation phase (like
#       during BEGIN blocks), nor can it use modules that do so.
#
# Results:
#       The results of the triggering method;
#
# Side effects:
#       An unexpeced routine (one not defined in Log :log :manip
#       or :count) will cause process death.
#
########################################################################

{
   my $justOnce = 1;
   sub AUTOLOAD
   {
      return if $VMware::FileSys::Lock::AUTOLOAD =~ m/::DESTROY$/;
      if ($justOnce) {
         $justOnce = 0;
         eval "use VMware::Log qw(:log :manip :count)";
         if ($@) {
            #
            # Its reasonable to die here, because that's what would
            # happen if a totally undefined subroutine call was
            # made anyway.
            #
            die $@ . "\n";
         }
         goto &$VMware::FileSys::Lock::AUTOLOAD;
      } else {
         die "Attempted to call the undefined subroutine " .
             $VMware::FileSys::Lock::AUTOLOAD . "\n";
      }
   }
}


########################################################################
#
# Lock::new --
#
#       Constructor for a VMware::FileSys::Lock object.
#
# Results:
#       The new object, or undef on error.
#
# Side effects:
#       None.
#
########################################################################

sub new {
   my $class = shift;   # IN: Invoking class.

   my $self = {
      file => undef,       # The filename we are locking
      lockFile => undef,   # The filename of the lock file
   };

   bless $self => $class;
   return $self;
}


########################################################################
#
# Lock::LockFile --
#
#       Lock a file by creating a lockfile that is (mostly)
#       compatible with bora/lib/vmkctl/config
#
# Results:
#       True on success, False but defined (0) if the lock could not
#       be acquired, undef on other failure.
# 
# Side effects:
#       Creates lock file, file is locked as far as any other
#       code using these locking conventions is concerned.
#
########################################################################

sub LockFile {
   my $self = shift;    # IN/OUT: Invoking instance.
   my $file = shift;    # IN: The backing file to lock.


   if (defined $self->{lockFile}) {
      if ($self->{lockFile} eq "$file.LOCK") {
         LogInfo("Redundant lock request, ignoring.");
         return 1;
      } else {
         LogWarn("Cannot lock two files at once.");
         return undef;
      }
   }


   my $lockFile = "$file.LOCK";
   LogInfo("Acquiring lock on file '$file'.");

   #
   # TODO: Yes, this is not safe on NFS.  Determining whether link(2)
   #       is supported on the filesystem is annoying and not worth it
   #       at this time.  Using a directory for locking would be safe
   #       but would not match bora/lib/file/fileLock*.c.
   #
   #       We should add an option for directory-based locking for
   #       callers who don't care about lib/file but need NFS support.
   #       File-based locking should remain the default for compatibility.
   #
   # TODO: We also do not implement the remote FS aspects of the lib/file
   #       library (namely, putting the hostname in the lockfile on nfs/smb).
   #       We should detect remote filesystems and do that.
   #

   local *LOCK;
   while (1) {
      unless (sysopen(LOCK, $lockFile, O_RDWR|O_CREAT|O_EXCL)) {
         if ($! == EEXIST) {

            #
            # Don't use FileSlurp because module dependencies in
            # ConfigObj are problematic due to its involvement in
            # startup procedures.
            #

            local *OLDLOCK;
            unless (open(OLDLOCK, "<$lockFile")) {
               LogWarn("Could not open existing lock file '$lockFile: $!'.");
               return undef;
            }
            my $lockerInfo = <OLDLOCK>;
            chomp($lockerInfo);
            unless (close(OLDLOCK)) {
               LogWarn("Could not close existing lock file '$lockFile: $!'.");
               return undef;
            }

            if (not defined($lockerInfo)) {
               LogWarn("Could not read pid out of existing lock file " .
                        "'$lockFile'.");
               return undef;
            }

            LogInfo("Old lock file contents: '$lockerInfo'.");
            $lockerInfo =~ /(\d+)/;
            my $pid = $1;
            if (!defined($pid) || $pid < 0) {
               #
               # We have a bad pid.  Delete the file and error out.
               #
               LogWarn("Found bad pid $pid in lock file '$lockFile'.");
               if (unlink($lockFile)) {
                  LogInfo("Removed bad lock file '$lockFile'.");
               } else {
                  LogWarn("Could not remove lock file '$lockFile': $!");
               }
               return undef;
            }

            if (kill(0, $pid)) {
               unless ($pid == $$) {
                  LogInfo("Could not get lock on file '$file' as the file " .
                          "is already locked by process $pid.\n");
                  return 0;
               }
            }
            LogInfo("Found stale lock file, removing.");
            if (unlink($lockFile)) {
               next;

            } else {
               LogWarn("Could not remove stale lock file '$lockFile': $!");
               return undef;
            }
         } else {
            LogWarn("Error creating lock file '$lockFile': $!");
            return undef;
         }
      }
      last;
   } # End while.

   my $pidWidth10 = sprintf("%10d\n", $$);
   my $writeError = 0;
   unless (print LOCK $pidWidth10) {
      LogWarn("Could not write pid to lockfile!");
      $writeError = 1;
   }

   unless (close(LOCK)) {
      LogWarn("Could not close lock file '$lockFile': $!");
   }

   if ($writeError) {
      return 0;
   } else {
      $self->{file} = $file;
      $self->{lockFile} = $lockFile;
      return 1;
   }
}


########################################################################
#
# Lock::UnlockFile --
#
#       Unlock a file by removing the lockfile.
#
# Results:
#       True on success, undef on failure.
#
# Side effects:
#       Lock file is removed, file unlocked.
#
########################################################################

sub UnlockFile {
   my $self = shift;    # IN/OUT: Invoking instance.

   if (not defined($self->{lockFile})) {
      LogInfo("Asked to unlock file, but no file is locked. Ignoring.");
      return 1;
   }

   my $file = $self->{lockFile};
   $file =~ s/^(.+)\.LOCK$/$1/;
   LogInfo("Releasing lock on file '$file'.");
   unless (unlink($self->{lockFile})) {
      LogWarn("Could not remove lock file '$self->{lockFile}': $!");
      return undef;
   }

   undef($self->{lockFile});
   return 1;
}


#
# Trivial Accessor Functions
#

sub GetFile
{
   my $self = shift;
   return $self->{file};
}

1;
