##########################################################################
# Copyright 2005 VMware, Inc.  All rights reserved. -- VMware Confidential
##########################################################################

#
# MountInfo.pm --
#
#       Utility functions for figuring out mount points.
#

package VMware::FileSys::MountInfo;

use VMware::Log qw(:log);
use VMware::FileSys::StandardLocator qw(StdFile StdCommand);
use VMware::FileSys::Tmp qw(:tmp);
use VMware::System qw(LogCommand);

require Exporter;

use strict;

@VMware::FileSys::MountInfo::ISA = qw(Exporter);
@VMware::FileSys::MountInfo::EXPORT_OK =
   qw(LookupMountDevice LookupMountUuid LookupMount LookupDeviceUuid 
      FindDirMapping ParseMtabFile LookupMountSize ParseDeviceFilename);

%VMware::FileSys::MountInfo::EXPORT_TAGS =
   (all => [(@VMware::FileSys::MountInfo::EXPORT_OK)]);


#######################################################################
#
# MountInfo::ParseDeviceFilename --
#
#       Parses a device filename into a device name and a partition
#       number. For example, "/dev/sda1" would be parsed into
#       ("/dev/sda", 1).
#
# Results:
#       A 2-tuple of the device name and partition number, or
#       (undef, undef) on error.
#
# Side effects:
#       None.
#
#######################################################################

sub ParseDeviceFilename
{
   my $device = shift; # IN: The device filename to parse.
   
   if ($device =~ /^(.+c\d+d\d+)p(\d+)$/ or $device =~ /^(\D+)(\d+)$/) {
      my $devName = $1;
      my $partNum = $2;
      return ($devName, $partNum);
   }
   
   return (undef, undef);
}


#######################################################################
#
# MountInfo::LookupMountDevice --
#
#       Looks for the device on which a directory is mounted.
#
# Results:
#       If the split option is specified, returns the device name
#       and partition number, or ("", "") if none found.
#       Otherwise, returns the device or "" (the empty string) if
#       none found.
#
# Side effects:
#       Reads mtab.
#
#######################################################################

sub LookupMountDevice
{
   my $dir = shift; # IN: The directory whose mount device should be found.
   my $split = shift;   # IN: (Optional) Return result as
                        #     (device name, part num).

   my ($dev) = LookupMount($dir);
   unless (defined($dev)) {
      $dev = "";
   }

   if ($split) {
      my ($devName, $partNum) = ParseDeviceFilename($dev);
      if (defined($devName)) {
         return ($devName, $partNum);
      } else {
         LogInfo("Could not parse device filename: $dev");
         return ("", "");
      }
   } else {
      return $dev;
   }
}


#######################################################################
#
# MountInfo::LookupMountUuid --
#
#       Looks for the UUID of the device on which a directory is mounted.
#
# Results:
#       The UUID or "" (the empty string) if none found.
#
# Side effects:
#       Reads mtab and invokes dumpuuid.
#
#######################################################################

sub LookupMountUuid
{
   my $dir = shift; # IN: The directory whose mount device should be found.

   my ($dev) = LookupMount($dir);
   unless (defined($dev)) {
      return "";
   }

   return LookupDeviceUuid($dev);
}


#######################################################################
#
# MountInfo::LookupMount --
#
#       Looks for the device on which a directory is mounted.
#
# Results:
#       (device, type, mnt) if found, empty list otherwise.
#
# Side effects:
#       Reads mtab
#
#######################################################################

sub LookupMount
{
   my $dir = shift;  # IN: The directory whose mount info should be found.

   my $map = FindDirMapping();

   # Sort the directories by reverse length.  First one that matches
   # completely will be the one.
   my @keys = sort { length($b) <=> length($a) } keys(%$map);

   foreach my $k (@keys) {
      if ($dir =~ /^\Q$k\E/) {
         return($map->{$k}->[0], $map->{$k}->[1], $k);
      }
   }
   return ();
}


#######################################################################
#
# MountInfo::LookupMountSize --
#
#       Looks for the size of the device on which a directory is
#       mounted.
#
# Results:
#       The size in megabytes or undef if none found.
#
# Side effects:
#       Invokes df.
#
#######################################################################

sub LookupMountSize
{
   my $dir = shift; # IN: The directory whose mount device should be found.

   # Get df helper
   my $cmdList = StdCommand('df', 'all');
   unless (defined $cmdList) {
      LogError("Could not get df helper.");
      return undef;
   }

   # Set block size to megabytes, and only check the specified dir.
   my $cmd = join(' ', @$cmdList, '-B M', $dir);
   my $result = LogCommand($cmd, 1);
   unless (defined($result)) {
      LogError("Execution of '$cmd' failed.  See log messages above.");
      return undef;
   }

   # Parse df output. The first line will be column headers, the
   # second will be our actual values. For example:
   #
   # Filesystem         1048576-blocks      Used Available Capacity Mounted on
   # /dev/hda1                   99M        9M       86M       9% /boot
   my @lines = split(/\n/, $result);
   unless($#lines == 1) {
      LogError("Execution of '$cmd' failed: Unexpected output.  See log messages above.");
      return undef;
   }

   my @columns = split(/\s+/, $lines[1]);
   # df 5.x outputs with units. df 4.x outputs without units.
   # Support both versions since one day we may upgrade to df 5.x.
   unless($#columns == 5 and $columns[1] =~ /^(\d+)M?$/) {
      LogError("Execution of '$cmd' failed: Unexpected output.  See log messages above.");
      return undef;
   }

   my $size = $1;
   return $size;
}


#######################################################################
#
# MountInfo::FindDirMapping --
#
#       Builds a directory to device mapping.
#
# Results:
#       A hash ref mapping directories to a two-element list where
#       the 1st element is the device name and the 2nd element is
#       the filesystem type.  "vmfs" is the device name for VMFS.
#
#       undef if the mtab file could not be read.
#
# Side effects:
#       Reads
#
#######################################################################

sub FindDirMapping
{
   my $dirMap = {};

   unless (ParseMtabFile($dirMap)) {
      LogError("Could not parse mtab file.");
      return undef;
   }

   # XXX Insert better method here

   # XXX Hard code /vmfs to be the path containing all VMFS file systems.
   # We really should use a more dynamic way of determining the location
   # of the VMFS.

   $dirMap->{'/vmfs'}->[0] = "vmfs";
   $dirMap->{'/vmfs'}->[1] = "vmfs";

   return $dirMap;
}


#######################################################################
#
# MountInfo::ParseMtabFile --
#
#       Reads the mtab file and builds a directory map with
#       the following structure:
#
#       $dirMap->{'mount-point'}->[0] = device name (/dev/sda1)
#       $dirMap->{'mount-point'}->[1] = type (ext2)
#
# Results:
#       True on success, false otherwise.
#
# Side effects:
#       Populates the $dirMap referece.
#       Reads the mtab file.
#
#######################################################################

sub ParseMtabFile
{
   my $dirMap = shift;  # IN/OUT: The reference to fill with mtab info.

   local *FD;
   my $mtab = StdFile('mtab', 'all');
   unless (open(FD, "<$mtab")) {
      LogError("Could not open '$mtab': $!");
      return 0;
   }

   while (my $line = <FD>) {
      if ($line =~ /^(\S+)\s+(\S+)\s+(\S+)\s+(\S+)\s+(\S+)\s+(\S+)/) {
         my ($dev, $mntpt, $type, $opt, $dmp, $fsck) =
            ($1, $2, $3, $4, $5, $6);
         #
         # Unescape spaces in the path, if any.
         #

         $mntpt =~ s/\\040/ /g;
         $dirMap->{$mntpt}->[0] = $dev;
         $dirMap->{$mntpt}->[1] = $type;
      }
   }
   unless (close(FD)) {
      LogWarn("Could not close '$mtab': $!");
   }
   return 1;
}

#######################################################################
#
# MountInfo::LookupDeviceUuid --
#
#       Looks up the UUID of a given device via dumpuuid.
#
# Results:
#       The UUID or "" (the empty string) if none found. If
#       $includeLabel is specified, returns a two-tuple of the form
#       (uuid, label).
#
# Side effects:
#       Runs the dumpuuid command.
#
#######################################################################

sub LookupDeviceUuid 
{
   my $dev = shift; # IN: The device whose UUID should be found.
   my $includeLabel = shift; # IN: Include label. (Optional)

   my $cmdList = StdCommand('dumpuuid', 'all');
   unless (defined $cmdList) {
      LogError("Could not get dumpuuid helper.");
      return undef;
   }

   my $cmd = join(' ', @$cmdList, $dev);
   my @lines = split(/\n/, LogCommand($cmd, 1));
   unless (@lines) {
      LogError("Could not run dumpuuid!");
      return undef;
   }

   my $uuid = "" ;
   my $label = "";

   foreach my $line (@lines) {
      if ($line =~ /^Filesystem UUID:\s+([0-9a-f\-]+)$/) {
         $uuid = $1;
      } elsif ($line =~ /^Filesystem volume name:\s+(.+)$/) {
         $label = $1;
      }
   }

   if ($includeLabel) {
      return ($uuid, $label);
   } else {
      return $uuid;
   }
}

1;
