##########################################################################
# Copyright 2005 VMware, Inc.  All rights reserved. -- VMware Confidential
##########################################################################

#
# Move.pm --
#
#       Module for moving filesystem objects with various conditions.
#
#

package VMware::FileSys::Move;

require Exporter;

@VMware::FileSys::Move::ISA = qw(Exporter);
@VMware::FileSys::Move::EXPORT_OK = qw(MoveFileKeepDestMode);
%VMware::FileSys::Move::EXPORT_TAGS = (move => [qw(MoveFileKeepDestMode)]);

use File::Copy qw(move);

use VMware::Log qw(:log);
use VMware::FileSys::StandardLocator qw(StdCommand);
use VMware::System qw(LogCommand);

use strict;


########################################################################
#
# Move::MoveFileKeepDestMode
#
#       Moves a file, preserving the original mode of the destination
#       if it existed prior to being overwritten.
#
# Results:
#       True on success, false otherwise.
#
# Side effects:
#       Removes the orignal file, overwrites the destination if present.
#
########################################################################

sub MoveFileKeepDestMode
{
   my $src = shift;   # IN: Path of the source file.
   my $dest = shift;  # IN: Path of the destination file.

   my (undef, undef, $mode) = stat($dest);
   if (defined($mode)) {
      $mode &= 07777;
   }

   if (defined($mode) && !chmod($mode, $src)) {
      LogError("Could not chmod '$src' to mode " .
               sprintf("%o", $mode) . ": $!");
      return undef;
   }
   
   # In limited-space situations, overwriting the file without
   # syncing the disk can result in insufficient disk space errors.
   # Truncate the file to 0 bytes and sync manually first.
   truncate($dest, 0);

   my $cmdList = StdCommand('sync', 'all');
   unless (defined $cmdList) {
      LogError("Could not get sync helper.");
      return undef;
   }
   my $cmd = join(' ', @$cmdList);
   LogCommand($cmd);

   unless (move($src, $dest)) {
      LogError("Could not move '$src' to '$dest': $!");
      return undef;
   }
   return 1;
}


1;

