##########################################################################
# Copyright 2005 VMware, Inc.  All rights reserved. -- VMware Confidential
##########################################################################

#
# StandardLocator.pm --
#
#       This module maps standard names to filesystem locations
#       (files or directories).
#

package VMware::FileSys::StandardLocator;

require Exporter;

use VMware::Config qw(:config);
use VMware::DataStruct qw(Clone);

use strict;

@VMware::FileSys::StandardLocator::ISA = qw(Exporter);
@VMware::FileSys::StandardLocator::EXPORT_OK =
   qw(StdFile StdDir StdCommand);


########################################################################
#
# StandardLocator::StdFile --
#
#       Retrieve a standard file location by ID.
#
# Results:
#       The path to the file.
#
# Side effects:
#       None.
#
########################################################################

{
   my $files = {
      esx => {
         # Master ESX Server configuration file.
         esx_config => '/etc/vmware/esx.conf',

         # Checksums of master ESX Server configuration file.
         esx_checksum_config => '/etc/vmware/esx_checksum.conf',
         rdrebuild_flag => '/etc/vmware/rdrebuild',

         # File mapping devices to VMkernel drivers (generated from pci.xml).
         pcidev_map => '/etc/vmware/vmware-devices.map',

         # Custom map of devices to VMkernel drivers.
         pcidev_map_local => '/etc/vmware/vmware-devices.map.local',

         # Config and log files for the esxcfg-boot script.
         esxcfg_boot_cfg => '/etc/vmware/esxcfg-boot.cfg',
         esxcfg_boot_log => '/var/log/vmware/esxcfg-boot.log',

         # Config and log files for the esxcfg-divvy script.
         esxcfg_divvy_cfg => '/etc/vmware/esxcfg-divvy.cfg',
         esxcfg_divvy_log => '/var/log/vmware/esxcfg-divvy.log',

         # Log file for the esxcfg-firewall script.
         esxcfg_firewall_log => '/var/log/vmware/esxcfg-firewall.log',

         # Config and log files for the esxcfg-linuxnet script.
         esxcfg_linuxnet_cfg => '/etc/vmware/esxcfg-linuxnet.cfg',
         esxcfg_linuxnet_log => '/var/log/vmware/esxcfg-linuxnet.log',

         # Config and log files for the esxcfg-upgrade script.
         esxcfg_upgrade_cfg => '/etc/vmware/esxcfg-upgrade.cfg',
         esxcfg_upgrade_log => '/var/log/vmware/esxcfg-upgrade.log',

         # Log file for customer-visible upgrade warnings.
         customer_upgrade_warnings => '/root/upgrade-customer-warnings.log',
         #
         # Config and log files for vmvisor-conf
         vmvisor_cfg => '/etc/vmware/vmvisor.cfg',
         vmvisor_log => '/var/log/vmware/vmvisor.log',
      },
      all => {
         # Proc node to read for partition information
         proc_partitions => '/proc/partitions',

         # Proc node to read for PCI information.
         proc_pci => '/proc/pci',

         # Proc node to read kernel boot command line args.
         proc_cmdline => '/proc/cmdline',

         fstab => '/etc/fstab',
         mtab => '/etc/mtab',
 
         lilo_conf => '/etc/lilo.conf',
         grub_conf => '/boot/grub/grub.conf',

         # GRUB device mappings.
         device_map => '/boot/grub/device.map',

         # Kudzu hardware configuration
         hwconf => '/etc/sysconfig/hwconf',

         # Main network config file
         network => '/etc/sysconfig/network',
      },
   };

   sub StdFile
   {
      my $fileId = shift;  # IN: The file ID to look up.
      my $product = shift; # IN: The product (optional).

      my $testVal =
         ConfigGet("test.filesys.standardlocator.files.$product.$fileId");
      unless (defined $testVal) {
         $testVal =
         ConfigGet("test.filesys.standardlocator.files.all.$fileId");
      }
      return $testVal if defined $testVal;

      my $file;
      if (defined($product)) {
         $file = $files->{$product}->{$fileId};
      }
      return $file || $files->{all}->{$fileId};
   }
}


########################################################################
#
# StandardLocator::StdDir --
#
#       Retrieve a standard dir location by ID.
#
# Results:
#       The full directory path (no trailing slash).
#
# Side effects:
#       None.
#
########################################################################

{
   my $dirs = {
      esx => {
         # VMkernel modules can be found here.
         vmkmodule => '/usr/lib/vmware/vmkmod',
      },
      all => {
         # Boot files can be found here in the boot directory.
         boot => '/boot',
         boot_grub => '/boot/grub',
         dev => '/dev',

         # ifcfg interface files
         network_scripts => '/etc/sysconfig/network-scripts',
      },
   };

   sub StdDir
   {
      my $dirId = shift;   # IN: The directory ID to look up.
      my $product = shift; # IN: The product (optional).

      my $testVal =
         ConfigGet("test.filesys.standardlocator.dirs.$product.$dirId");
      unless (defined $testVal) {
         $testVal =
         ConfigGet("test.filesys.standardlocator.dirs.all.$dirId");
      }
      return $testVal if defined $testVal;

      my $dir;
      if (defined($product)) {
         $dir = $dirs->{$product}->{$dirId};
      }
      return $dir || $dirs->{all}->{$dirId};
   }
}


########################################################################
#
# StandardLocator::StdCommand --
#
#       Retrieve a standard command location and arguments by ID.
#
# Results:
#       An array of the command and its arguments.
#
# Side effects:
#       None.
#
########################################################################

{
   my $commands = {
      esx => {
         lilo => ['/sbin/lilo'],
         mkinitrd => ['/sbin/mkinitrd', '-v', '-f'],
         vmware_mkinitrd => ['/sbin/vmware-mkinitrd', '-f', '-v'],
         vmware_mkinitrd_refresh => ['/sbin/vmware-mkinitrd', '-r', '-f', '-v'],
         mpath_bulk => ['/usr/sbin/esxcfg-mpath', '--bulk'],
         vmhbadevs => ['/usr/sbin/esxcfg-vmhbadevs', '--vmfs'],
      },
      all => {
         safe_lspci => ['/sbin/lspci', '-vn'],
         safe_lspci_names => ['/sbin/lspci', '-v'],
         probe_lspci => ['/sbin/lspci', '-H1', '-M', '-vn'],
         probe_lspci_names => ['/sbin/lspci', '-H1', '-M', '-v'],
         dumpuuid => ['/sbin/dumpuuid'],
         df => [ '/bin/df', '-P' ],
         sync => [ '/bin/sync' ],
         kudzu => [ '/usr/sbin/kudzu', '-q', '-s' ],
         md5sum => [ '/usr/bin/md5sum' ],
         egrep => [ '/bin/egrep' ],
         egrep => [ '/bin/egrep' ],
         grub_install => [ '/sbin/grub-install' ],
      },
   };

   sub StdCommand
   {
      my $cmdId = shift;   # IN: The command to look up.
      my $product = shift; # IN: The product (optional).

      my $testVal =
         ConfigGet("test.filesys.standardlocator.commands.$product.$cmdId");
      unless (defined $testVal) {
         $testVal =
         ConfigGet("test.filesys.standardlocator.commands.all.$cmdId");
      }
      return [$testVal] if defined $testVal;

      my $cmd;
      if (defined($product)) {
         $cmd = $commands->{$product}->{$cmdId};
      }
      unless (defined $cmd) {
         $cmd = $commands->{all}->{$cmdId};
      }
      return $cmd;
   }
}


1;
