##########################################################################
# Copyright 2003 VMware, Inc.  All rights reserved. -- VMware Confidential
##########################################################################

#
# Tmp.pm --
#
#       Module for messing with temporary files and directories.
#
# TODO: Find an implementation of File::Temp that works on 
#       all of our distributions and use that instead.
#

package VMware::FileSys::Tmp;

require Exporter;

@VMware::FileSys::Tmp::ISA = qw(Exporter);
@VMware::FileSys::Tmp::EXPORT_OK = qw(TmpFile TmpFileDataDump TmpDir 
                                    GetTmpDir CleanTmpFiles AddFileToTmpCleanup
                                    RemoveDirFromTmpCleanup 
                                    RemoveFileFromTmpCleanup);
%VMware::FileSys::Tmp::EXPORT_TAGS = (tmp => [qw(TmpFile TmpFileDataDump TmpDir 
                                               GetTmpDir CleanTmpFiles
                                               AddFileToTmpCleanup
                                               RemoveDirFromTmpCleanup
                                               RemoveFileFromTmpCleanup)]);

use Cwd;
use Fcntl;
use File::Spec;
use File::Path;
use Errno qw(:POSIX);

use VMware::Log qw(:log);
use VMware::Config qw(:config ConfigSet);
use strict;

#
# Use this counter to avoid really obvious collisions.
#

my $counter = 0;

#
# Use these arrays to keep track of all the tmp files
# and dirs created during a script run.
#

my @tmpFiles = ();
my @tmpDirs = ();


########################################################################
#
# Tmp::TmpFile --
#
#       Creates a unique temporary file.
#       Uses the temp.dir config option to determine where to place
#       the file.  Note that unlike the tmpfile(3) C library call,
#       the caller must unlink the file when done with it.  This does
#       mean that the file can be closed and reopened, though.
#
# Results:
#       A list consisting of the absolute path of the filename,
#       and (optionally) an open handle to the file.  The list is
#       empty on failure.
#
# Side effects:
#       Creates a file in the filesystem.
#
########################################################################

sub TmpFile
{
   my $nohandle = shift;  # IN: if true, close the handle and don't
                          #     return it- just return the file name.
   my $prefix = (shift || "vmware.");  # IN: Prefix for the filename.

   #
   # TODO: We will at some point guarantee a default here.
   #
   my $dir = GetTmpDir();

   unless (File::Spec->file_name_is_absolute($dir)) {
      $dir = File::Spec->catdir(cwd(), $dir);
   }
   my $tmpbase = File::Spec->catfile($dir, $prefix);
   my $tmpname;

   local *TMPFILE;

   #
   # TODO: The O_EXCL trick won't work reliably on NFS.
   #

   while(1) {
      #
      # Use restrictive permission to prevent other users from examining
      # the file's contents.
      #

      $tmpname = ($tmpbase . $counter++ . ".tmp");
      if (sysopen(TMPFILE, "$tmpname", (O_WRONLY | O_CREAT | O_EXCL), 00700)) {
         push(@tmpFiles, $tmpname);
         if ($nohandle) {
            if (close(TMPFILE)) {
               return $tmpname;
            } else {
               LogError("Could not close filehandle for $tmpname: $!\n");
               return ();
            }

         } else {
            return ($tmpname, *TMPFILE);
         }
      } elsif (($! != EEXIST) && (($^O ne "MSWin32") || not (-d $tmpname))) {
         LogError("Could not create temporary file $tmpname: $!\n");
         return ();
      }
   }
}


########################################################################
#
# Tmp::TmpDir --
#
#       Creates a unique temporary directory.
#       Uses the temp.dir config option to determine where to place
#       the dir.
#
# Results:
#       The absolute path of the directory, or undef on failure.
#
# Side effects:
#       Creates a directory in the filesystem.
#
########################################################################

sub TmpDir
{
   my $mask = shift;  # IN: Optional permissions mask for mkdir(2).
   my $prefix = (shift || "vmwaretmp.");  # IN: Prefix for the filename.

   unless (defined $mask) {
      $mask = 0777;
   }

   #
   # TODO: We will at some point guarantee a default here.
   #
   my $dir = GetTmpDir();
   unless (File::Spec->file_name_is_absolute($dir)) {
      $dir = File::Spec->catdir(cwd(), $dir);
   }

   #
   # Use catfile rather than catdir because we don't want a '/' or '\'
   # tacked on after the prefix, because there are more charcters coming.
   #

   my $tmpbase = File::Spec->catfile($dir, $prefix);
   my $tmpname;

   #
   # TODO: I suspect NFS could cause problems here, too.
   #

   until(mkdir(($tmpname = ($tmpbase . $counter++ )), $mask)) {
      unless (($! == EEXIST) || (($^O eq "MSWin32") && -d $tmpname)) {
         LogError("Could not create temporary directory $tmpname: $!\n");
         return undef;
      }
   }
   push(@tmpDirs, $tmpname);
   return $tmpname;
}


###########################################################################
#
# Tmp::GetTmpDir --
#
#       Returns a valid path to a temporary directory.  Guaranteed to
#       return a consistent and usable result.  May log debug messages
#
# Results:
#       The temporary directory path.
#
# Side effects:
#       Will call 'die' on complete failure.
#
###########################################################################

sub GetTmpDir
{
   my $tempDir;

   #
   # First, try the configured directory
   #
   $tempDir = ConfigGet("temp.dir");
   if (defined($tempDir) && (-d $tempDir)) {
      return $tempDir;
   }

   #
   # Next, try an environment variable
   #
   $tempDir = $ENV{TEMP};
   if (defined($tempDir) && (-d $tempDir)) {
      LogDebug("Using temp dir from TEMP environment variable: $tempDir",
                2, "tmp");
      ConfigSet("temp.dir", $tempDir);
      return $tempDir;
   }

   #
   # Try some standard locations
   #
   my @dirs;
   if ($^O eq "MSWin32") {
      @dirs = ('C:\WINNT\TEMP', 'C:\TEMP');
   } else {
      @dirs = ('/tmp');
   }
   foreach my $dir (@dirs) {
      if (-d $dir) {
         LogDebug("Using system temp dir: $dir", 2, "tmp");
         ConfigSet("temp.dir", $dir);
         return $dir;
      }
   }

   # Last-ditch effort.
   $tempDir = ($^O eq "MSWin32") ? "$ENV{SystemDrive}\\" : $ENV{HOME};
   unless (-d $tempDir) {
      LogError("$tempDir is not accessible, giving up.");
      die "\nPlease configure a writable temp directory.\n";
   } else {
      LogDebug("Using fallback temp dir: $tempDir", 2, "tmp");
   }
   ConfigSet("temp.dir", $tempDir);
   return $tempDir;
}


###########################################################################
#
# Tmp::TmpFileDataDump --
#           
#       Creates a tmp file and writes a data structure
#       out to it. This file can then be easily eval to 
#       read the data back in
#
# Results:
#       The path to that file or undef on failure
#
# Side effects:
#       uses up a little disk space
#  
###########################################################################

sub TmpFileDataDump
{
   my $data = shift; #IN: Data to dump
   my $var = shift;  #IN: variable name to assign that data to
   
   my $dumpInfo = sprintf("%s", Data::Dumper->Dump([$data], [$var]));
   my ($tmpFile, $handle) = TmpFile();
   LogDebug("Dumping to file: $tmpFile\n $dumpInfo", 3, "tmp");
   print $handle $dumpInfo or LogError("Could not dump: $var to: $tmpFile: $!")
      and return undef;
   close $handle or 
      LogError("Failed to close data dump file: $tmpFile: $!") 
         and return undef;
   return $tmpFile;
}


###########################################################################
#
# Tmp::CleanTmpFiles --
#
#       Tries Unlink every tmp file and every tmp dir created.
#
# Results:
#       Files may get removed
#
# Side effects:
#       none
#
###########################################################################

sub CleanTmpFiles 
{
   foreach my $file (@tmpFiles) { 
      if ( -e $file) { 
         unlink $file or LogDebug("Could not unlink $file: $!", 2, "tmp");
      }
   }
   foreach my $dir (@tmpDirs) { 
      if ( -d $dir ) { 
         rmtree $dir or LogDebug("Could not unlink $dir: $!", 2, "tmp");
      }
   }
}


###########################################################################
#
# Tmp::AddFileToTmpCleanup --
#
#       Adds a file to the cleanup list.
#
# Results:
#       None
#
# Side effects:
#       None
#
###########################################################################

sub AddFileToTmpCleanup 
{
   my $filename = shift; # IN: filename of file to clean up 

   push @tmpFiles, $filename;
}


###############################################################################
#
# RemoveDirFromTmpCleanup --
#
#      Removes a directory from the tmpDirs array. This is useful for scripts
#      that dont want their temporary directories to be deleted.
#
# Results:
#      None.
#
# Side effects:
#      None.
#
###############################################################################

sub RemoveDirFromTmpCleanup
{
   my $dir = shift;  # IN: directory name that doesnt have to be cleaned

   @tmpDirs = grep {$dir ne $_} @tmpDirs;
}


###############################################################################
#
# RemoveFileFromTmpCleanup --
#
#      Removes a file from the tmpFiles array. This is useful for scripts
#      that dont want their temporary files to be deleted.
#
# Results:
#      None.
#
# Side effects:
#      None.
#
###############################################################################

sub RemoveFileFromTmpCleanup
{
   my $filename = shift;  # IN: file name that doesnt have to be cleaned

   @tmpFiles = grep {$filename ne $_} @tmpFiles;
}

1;
