##########################################################################
# Copyright 2005 VMware, Inc.  All rights reserved. -- VMware Confidential
##########################################################################

#
# UuidCache.pm --
#
#       Utility class responsible for loading uuid/label information
#       for all system partitions and translating between uuids, labels,
#       and devices.
#

package VMware::FileSys::UuidCache;

use VMware::Log qw(:log);
use VMware::FileSys::MountInfo qw(LookupDeviceUuid);
use VMware::FileSys::StandardLocator qw(StdFile StdDir);
use VMware::FileSys::UuidCacheEntry;

use strict;


########################################################################
#
# UuidCache::new --
#
#       Constructor. Loads all cache data from /proc/partitions.
#
# Results:
#       The new instance, or undef on error.
#
# Side effects:
#       None.
#
########################################################################

sub new
{
   my $class = shift;   # IN: Invoking class name.

   my $self = {
      # Lookup based on uuid, label, and device. All three hashes reference
      # the same set of entries.
      devices => {},
      uuids => {},
      labels => {},
   };
   bless $self => $class;

   $self->LoadPartitions() or return undef;
   return $self;
}


########################################################################
#
# LoadPartitions --
#
#       Private method used by the constructor. Loads uuid/label
#       information for all system partitions.
#
# Results:
#       1 on success, 0 otherwise.
#
# Side effects:
#       Reads uuid/label information from all partitions listed in
#       /proc/partitions.
#
########################################################################

sub LoadPartitions
{
   my $self = shift;    # IN: Invoking instance.

   my $devices = $self->ListPartitionDevices();
   unless ($devices) {
      return 0;
   }

   foreach my $device (@$devices) {
      my ($uuid, $label) = LookupDeviceUuid($device, 1);
      unless ($uuid) {
         next;
      }

      my $entry = new VMware::FileSys::UuidCacheEntry($device, $uuid, $label);
      $self->{devices}->{$device} = $entry;
      $self->{uuids}->{$uuid} = $entry;
      $self->{labels}->{$label} = $entry;
   }

   return 1;
}


########################################################################
#
# ListPartitionDevices --
#
#       Private method used by the constructor. Lists device names
#       for all partitions on the system.
#
# Results:
#       On success returns a reference to a list of device names.
#       Otherwise returns undef.
#
# Side effects:
#       Reads /proc/partitions.
#
########################################################################

sub ListPartitionDevices
{
   my $self = shift;    # IN: Invoking instance.

   my $proc = StdFile("proc_partitions", "all");
   unless (defined $proc) {
      LogError("Could not get /proc/partitions helper");
      return undef;
   }

   my $devdir = StdDir("dev", "all");
   unless (defined $proc) {
      LogError("Could not get /dev helper");
      return undef;
   }

   my $fh = undef;
   unless (open($fh, $proc)) {
      LogError("Failed to open $proc: $!");
      return undef;
   }

   my @devices = ();
   my $header = 1;
   while (<$fh>) {
      my $line = $_;
      
      # Strip out leading and trailing whitespace including newline
      $line =~ s/^\s+//;
      $line =~ s/\s+$//;

      # Skip blank lines
      unless ($line) {
         next;
      }

      my @fields = split(/\s+/, $line);
      if ($header) {
         unless ($fields[2] eq "#blocks" and
                 $fields[3] eq "name") {
            LogError("Invalid header in $proc");
            close($fh);
            return undef;
         }

         $header = 0;
      } else {
         my ($major, $minor, $size, $name, @etc) = @fields;
        
         # Filter out disks/extended partitions. Valid partitions:
         #  1) Have names ending in digits
         #  2) Have a size of more than 1 block.
         if ($name =~ /\d$/ and $size > 1) {
            push @devices, "$devdir/$name";
         }
      }
   }

   unless (close($fh)) {
      LogError("Failed to close $proc: $!");
      return undef;
   }

   return \@devices;
}

#######################################################################
#
# LookupSpec --
#
#       Looks up a uuid cache entry by its spec, which is one of:
#        * UUID=<uuid>
#        * LABEL=<label>
#        * <device>
#
# Results:
#       The entry or undef if none found.
#
# Side effects:
#       None
#
#######################################################################

sub LookupSpec
{
   my $self = shift; # IN: Invoking instance.
   my $spec = shift; # IN: The spec to lookup the entry for.

   if ($spec =~ /^UUID=(.+)$/) {
      return $self->{uuids}->{$1};
   } elsif ($spec =~ /^LABEL=(.+)$/) {
      return $self->{labels}->{$1};
   } else {
      return $self->{devices}->{$spec};
   }
}

1;
