##########################################################################
# Copyright 2003 VMware, Inc.  All rights reserved. -- VMware Confidential
##########################################################################

#
# Init --
#
#       Contains convenience functions for bootstrapping an
#       environment that relies on the common libraries, particularly
#       logging and the config system.
#

package VMware::Init;

use File::Basename;
use File::Path;
use Getopt::Long;

use VMware::Config qw(:config ConfigSet);
use VMware::Config::ConfigObj;
use VMware::Log::DebugLevel;
use VMware::Log qw(:log :manip);

use strict;

@VMware::Init::ISA = qw(Exporter);
@VMware::Init::EXPORT_OK = qw(InitConfig InitDebugLevels InitLog);
%VMware::Init::EXPORT_TAGS = ('init' => [@VMware::Init::EXPORT_OK]);


###########################################################################
#
# Init::InitConfig --
#
#       Perform the initial set-up of the gloabl Config space.
#       Config files are initialized as described in the comment for
#       InitConfigObj.
#
#       Also handles the interaction of the command-line global verbosity
#       option with the other methods of initializing Config.
#
# Results:
#       True on success, false on failure.
#
# Side effects:
#       Populates the global configuration space.
#
###########################################################################

sub InitConfig
{
   my $defaultConfigFile = shift;  # IN: list of default config filenames.
   my $definitionFiles = shift;    # IN: list of config definition filenames.
   my $cmdConfigFile = shift;      # IN: Config file from the command line.
   my $cmdOpts = shift;            # IN: Hash of config overrides from
                                   #     command line (-C name=value).
   my $verbosity = shift;          # IN: Verbosity from the command line.
   my $noDefaults = shift;         # IN: Skip most of the search path.

   # Error checking is in InitConfigObj.
   my $conf = InitConfigObj($defaultConfigFile,
                            $definitionFiles,
                            $cmdConfigFile,
                            $cmdOpts,
                            $noDefaults);
   unless (defined $conf) {
      return 0;
   }
   VMware::Config::ConfigSetObj($conf);

   #
   # Ensure that verbosity is always defined, and 
   # is overridden properly if needed.
   #

   if (defined $verbosity) {
      ConfigSet("log.verbosity", $verbosity);
   } elsif (not defined ConfigGet("log.verbosity")) {
      ConfigSet("log.verbosity", 0);
   }

   return 1;
}


###########################################################################
#
# Init::InitDebugLevels --
#
#       Initialize the DebugLevel configuration space.
#       Typically, the command-line overrides mentioned are usually
#       supported as --debug-opt <name>=<value> or -D <name>=<value>, while
#       the debug config file is usually --debug-file <file> or -d <file>.
#
# Results:
#       True on success, false on failure.
#
# Side effects:
#       Loads information into the DebugLevel module, reading files
#       as described in InitConfigObj.
#
###########################################################################

sub InitDebugLevels
{
   my $defaultConfigFile = shift;  # IN: list of default config filenames.
   my $definitionFiles = shift;    # IN: list of config definition filenames.
   my $cmdConfigFile = shift;      # IN: Config file from the command line.
   my $cmdOpts = shift;            # IN: Hash of config overrides from
                                   #     command line (-C name=value).
   my $noDefaults = shift;         # IN: Skip most of the search path.

   my $conf = InitConfigObj($defaultConfigFile,
                            $definitionFiles,
                            $cmdConfigFile,
                            $cmdOpts,
                            $noDefaults);
   unless (defined $conf) {
      return 0;
   }
   VMware::Log::DebugLevel::SetObj($conf);
   return 1;
}


###########################################################################
#
# Init::InitConfigObj --
#
#       Perform the initial set-up of a generic config space.
#       Uses passed-in lists of config and definition files for initialization,
#       as well as an additional config file from the command line
#       (which may be treated separately).
#
#       Typically, the command-line overrides mentioned are usually
#       supported as --config-opt <name>=<value> or -C <name>=<value>, while
#       the debug config file is usually --config-file <file> or -c <file>.
#
#       If the no defaults parameter is set, then we only
#       load the file and options specified on the command line.
#
# Results:
#       The newly populated ConfigObj.
#
# Side effects:
#       Reads files.
#
###########################################################################

sub InitConfigObj
{
   my $defaultConfigFiles = shift; # IN: list of default config filenames
                                   #     to use (including full path).
   my $definitionFiles = shift;    # IN: list of config definition filenames
                                   #     to use (including full path).
   my $cmdConfigFile = shift;      # IN: Config file from the command line.
   my $cmdOpts = shift;            # IN: Hash of config overrides from
                                   #     command line (-C name=value).
   my $noDefaults = shift;         # IN: Skip default config files if true.

   my $conf = new VMware::Config::ConfigObj::();

   #
   # Set up the validator for our global config space if we're using one.
   #

   if (defined $definitionFiles) {
      my $defsObj = new VMware::Config::ConfigObj::();

      foreach my $defsFile (@$definitionFiles) {
         #
         # Defs files are considered to be required.
         #

         unless (-f $defsFile) {
            LogError("Configuration definitions file '$defsFile' " .
                     "does not exist");
            return undef;
         }
         $defsObj->LoadFromFile($defsFile) or return undef;
      }

      $conf->SetValidator($defsObj);
   }

   my @files = ();
   if (defined($defaultConfigFiles) && not $noDefaults) {
      @files = @$defaultConfigFiles;
   }
   if (defined($cmdConfigFile)) {
      push(@files, $cmdConfigFile);
   }
   foreach my $file (@files) {
      if (-f $file) {
         $conf->LoadFromFile($file) or return undef;
      }
   }

   if (defined($cmdOpts)) {
      $conf->LoadFromHash($cmdOpts) or return undef;
   }
   return $conf;
}


###########################################################################
#
# InitLog --
#
#       Initialize the global logging system.
#
# Results:
#       True on success, undef otherwise.
#
# Side effects:
#       Sets up the main log.
#
###########################################################################

sub InitLog
{
   my $logfile = shift;         # IN: Log file specified by caller
   my $append = shift;          # IN: Whether to append to log file or not
   my $autoflush = shift;       # IN: Whether this log should be autoflushed
   unless ($append) {
      $append = ConfigGetBoolean("log.append", 0);
   }
   my $mode = $append ? ">>" : ">";
   
   my $logfh;
   if (defined $logfile) {
      unless (open(CMDLOG, "$mode$logfile")) {
         LogError("Could not open $logfile with mode $mode: $!");
         return undef;
      }
      if (defined($autoflush)) { 
         select CMDLOG;
         $| = 1;
      }
      $logfh = \*CMDLOG;
   } else {
      print "Using STDERR\n";
      $logfh = \*STDERR;
      $logfile = "STDERR";
      LogError("No log filename specified: using STDERR");
   }

   my @policies = ConfigGetArray("log.errorPolicies", "stderr", "log");
   my $errorPolicy = 0;
   foreach my $policy (@policies) {
      if ($policy =~ /stderr/i) {
         $errorPolicy |= VMware::Log::LogObj::ON_ERROR_STDERR;
      } elsif ($policy =~ /log/i) {
         $errorPolicy |= VMware::Log::LogObj::ON_ERROR_LOG;
      } elsif ($policy =~ /fatal/i) {
         $errorPolicy |= VMware::Log::LogObj::ON_ERROR_FATAL;
      } else {
         LogWarn("Unknown logging error policy selected: $policy");
      }
   }
   
   my $datestamp = ConfigGetBoolean("log.datestamp", 1);
   my $timestamp = ConfigGetBoolean("log.timestamp", 1);
   my $codestamp = ConfigGetBoolean("log.codestamp", undef);
   my $pidstamp = ConfigGetBoolean("log.pidstamp", 1);

   if ($codestamp) {
      $codestamp += VMware::Log::LogObj::STACK_DEPTH_OFFSET;
   } else {
      undef $codestamp;
   }

   my $fmt = new VMware::Log::Format::($datestamp,
                                       $timestamp,
                                       $codestamp,
                                       $pidstamp);

   my $log = new VMware::Log::LogObj::($logfh, $logfile, $fmt, $errorPolicy, 1);

   if (ConfigGetBoolean("log.stdout", 0)) {
      $log->AddDestination(*STDOUT, "STDOUT");
   }

   LogSet($log);
   return 1;
}


1;
