##########################################################################
# Copyright 2003 VMware, Inc.  All rights reserved. -- VMware Confidential
##########################################################################

#
# Log.pm --
#
#	Module defining a "global" log through a private instance of LogObj.
#


package VMware::Log;

use strict;

require Exporter;

@VMware::Log::ISA = qw(Exporter);

#
# Allow export of commonly used subroutines, and define a 'log' tag
# for importing the most common ones.
#

@VMware::Log::EXPORT_OK =
   qw(LogInfo LogWarn LogError LogDebug Log
      LogPush LogPop LogPeek LogSet LogMute LogUnmute
      LogGetInfoCount LogGetWarnCount LogGetErrorCount
      LogGetDebugCount LogGetCount);

%VMware::Log::EXPORT_TAGS = (
   'log' => [qw(LogInfo LogWarn LogError LogDebug Log)],
   manip => [qw(LogPush LogPop LogPeek LogSet LogMute LogUnmute)],
   count => [qw(LogGetInfoCount LogGetWarnCount LogGetErrorCount
                LogGetDebugCount LogGetCount)],
);

use VMware::Log::LogObj;

########################################################################
# Log.pm implements a stack of logs so that a new log can be used
# for a time, and the prior log easily restored.
#
########################################################################

#
# Constants for the IDs of the standard channels, allowing users
# to apply the generic functions such as GetCount and AddDestination
# to them.  ID_INTERNAL is for the logging system's internal channel,
# and should probably usually be left alone.  But perhaps not always.
#

use constant ID_INFO => VMware::Log::LogObj::ID_INFO;
use constant ID_WARN => VMware::Log::LogObj::ID_WARN;
use constant ID_ERROR => VMware::Log::LogObj::ID_ERROR;
use constant ID_DEBUG => VMware::Log::LogObj::ID_DEBUG;
use constant ID_INTERNAL => VMware::Log::LogObj::ID_INTERNAL;

#
# Do NOT initialize these here, as they are set in 
# the BEGIN block, and any "initialization" here actually
# happens afterwards.  However, they cannot be declared
# in the BEGIN block or they will be restricted to that
# lexical scope.
#

my @logs;
my $top;

#
# Initialize to STDOUT with default settings, except that
# printing of format lines is suppressed.
#
# In many cases, users will set a new log almost immediately,
# but this will make log calls valid in the meantime.
#

BEGIN {
   my $offset = VMware::Log::LogObj::STACK_DEPTH_OFFSET + 1;
   my $fmt = VMware::Log::Format->new(1, 1, $offset);
   push @logs, VMware::Log::LogObj->new(\*STDOUT,
                                        "Default Log",
                                        $fmt,
                                        undef,
                                        1);
   $top = $logs[$#logs];
}


########################################################################
#
# Log::LogInfo --
#
#       Routine for logging informational methods that should
#       always be written to the log.  See VMware::Log::LogObj::Info.
#
# Results:
#       See VMware::Log::LogObj::Info.
#
# Side effects:
#       See VMware::Log::LogObj::Info.
#
########################################################################

sub LogInfo
{
   return $top->Info(@_);
}


########################################################################
#
# Log::LogWarn --
#
#       Routine for logging warnings that are always printed,
#       and should be messages that may or may not be problems but
#       always warrant manual inspection.  See VMware::Log:LogObj::Warn
#
# Results:
#       See VMware::Log::LogObj::Warn.
#
# Side effects:
#       See VMware::Log::LogObj::Warn.
#
########################################################################

sub LogWarn
{
   return $top->Warn(@_);
}


########################################################################
#
# Log::LogError --
#
#       Routine for logging error messages.  These are always 
#       printed.  See VMware::Log::LogObj::Error
#
# Results:
#       See VMware::Log::LogObj::Error.
#
# Side effects:
#       See VMware::Log::LogObj::Error.
#
########################################################################

sub LogError
{
   return $top->Error(@_);
}


########################################################################
#
# Log::LogDebug --
#
#       Routine for logging debugging messages.  Takes an additional
#       parameter to match against a verbosity setting, which controls
#       whether the message is actually printed.
#       See VMware::Log::LogObj::Debug
#
# Results:
#       See VMware::Log::LogObj::Debug
#
# Side effects:
#       See VMware::Log::LogObj::Debug
#
########################################################################

sub LogDebug
{
   return $top->Debug(@_);
}


########################################################################
#
# Log::Log --
#
#       Routine for logging to a non-standard channel, presumably
#       added with LogCreateChannel.  See VMware::Log::LogObj::Log
#
# Results:
#       See VMware::Log::LogObj::Log
#
# Side effects:
#       See VMware::Log::LogObj::Log
#
########################################################################

sub Log
{
   return $top->Log(@_);
}


########################################################################
#
# Log::LogPush --
#
#       Change the current log, saving previous logs.
#
# Results:
#       The previous log.
#
# Side effects:
#       The new log becomes active.  Previous logs remain open,
#       but inaccessible.
#
########################################################################

sub LogPush
{
   my $new = shift;  # IN: The new log.

   my $ret = $logs[$#logs];
   push @logs, $new;

   #
   # Only adjust top if it is pointing at the stack.  Otherwise
   # it is muted so leave it alone.
   #

   if ($top == $ret) {
      $top = $logs[$#logs];
   }
   return $ret;
}


########################################################################
#
# Log::LogPop --
#
#       Restore the previous log.
#
# Results:
#       The current log (i.e., the one just popped).
#
# Side effects:
#       Re-activates the previous log.
#
########################################################################

sub LogPop
{
   my $old = pop @logs;

   #
   # Only adjust top if it is pointing at the stack.  Otherwise
   # it is muted so leave it alone.
   #

   if ($top == $old) {
      $top = $logs[$#logs];
   }
   return $old;
}


########################################################################
#
# Log::LogPeek --
#
#       Return a reference to the current log.  This is primarily
#       intended so that more advanced log manipulation functions
#       may be accessed.
#
#       In array context, returns the entire log stack.
#
#       If the logs are muted, will return only the mute log (in a
#       list of one element if called in list context).
#
# Results:
#       A reference to the current log object.
#
# Side effects:
#       None.
#
########################################################################

sub LogPeek
{
   wantarray ? return ($top == $logs[$#logs] ? @logs : ($top)) :
               return $top;
}


########################################################################
#
# Log::LogSet --
#
#       Set a new log, discarding all prior logs.
#
# Results:
#       The list of all prior logs.  The most recent log is at the
#       end of the list.
#
# Side effects:
#       The new log becomes active, and prior logs can no longer
#       be accessed by Pop()'ing the current log.
#
########################################################################

sub LogSet
{
   my $new = shift;  # IN: The new log.

   my @ret = @logs;
   @logs = ($new);

   #
   # Only adjust top if it is pointing at the stack.  Otherwise
   # it is muted so leave it alone.
   #

   if ($top == $ret[$#ret]) {
      $top = $logs[$#logs];
   }
   wantarray ? return @ret : return \@ret;
}


########################################################################
#
# Log::LogMute --
#
#       Redirect $top to a special log object that does not
#       have any destinations.
#       Primarily for use in test code that wants to suppress output
#       witout having to save the state of the log stack manually.
#
# Results:
#       None.
#
# Side effects:
#       Log messages are lost until Log::LogUnmute is called.
#
########################################################################

sub LogMute
{
   my $muteFmt = new VMware::Log::Format::();
   my $muteLog = new VMware::Log::LogObj::(\*STDOUT, "STDOUT", $muteFmt,
                                           VMware::Log::LogObj::ON_ERROR_LOG,
                                           1);
   $muteLog->ClearDestinations();
   $top = $muteLog;
}


########################################################################
#
# Log::LogUnmute --
#
#       Unmute the log if it has been muted by LogMute.  Technically,
#       this just ensure that $log is really pointing to the top
#       of the stack, so it could undo #something other than a call
#       to LogMute if some other code is written that redirects $top.
#
# Results:
#       None.
#
# Side effects:
#       Log messages will once again go to the top log on the stack.
#
########################################################################

sub LogUnmute
{
   $top = $logs[$#logs];
}


########################################################################
#
# Log::LogGetInfoCount --
#
#       Get the number of times the Info channel has been accessed.
#
# Results:
#       The number of times the Info channel has been accessed.
#
# Side effects:
#       None.
#
########################################################################

sub LogGetInfoCount
{
   return $top->GetCount(ID_INFO);
}


########################################################################
#
# Log::LogGetWarnCount --
#
#       Get the number of times the Warn channel has been accessed.
#
# Results:
#       The number of times the Warn channel has been accessed.
#
# Side effects:
#       None.
#
########################################################################

sub LogGetWarnCount
{
   return $top->GetCount(ID_WARN);
}


########################################################################
#
# Log::LogGetErrorCount --
#
#       Get the number of times the Error channel has been accessed.
#
# Results:
#       The number of times the Error channel has been accessed.
#
# Side effects:
#       None.
#
########################################################################

sub LogGetErrorCount
{
   return $top->GetCount(ID_ERROR);
}


########################################################################
#
# Log::LogGetDebugCount --
#
#       Get the number of times the Debug channel has been accessed.
#       Note that this is an access count, rather than a "number of
#       lines printed" count.  Accesses that do not result in any
#       printed lines because the level was too low still count as well.
#
# Results:
#       The number of times the Debug channel has been accessed.
#
# Side effects:
#       None.
#
########################################################################

sub LogGetDebugCount
{
   return $top->GetCount(ID_DEBUG);
}


########################################################################
#
# Log::LogGetCount --
#
#       Get the number of times a generic channel has been accessed.
#       Note that this is an access count, rather than a "number of
#       lines printed" count.  Accesses that do not result in any
#       printed lines because the level was too low still count as well.
#
# Results:
#       The number of times the specified channel has been accessed.
#
# Side effects:
#       None.
#
########################################################################

sub LogGetCount
{
   return $top->GetCount(@_);
}

1;

