##########################################################################
# Copyright 2003 VMware, Inc.  All rights reserved. -- VMware Confidential
##########################################################################


#
# Channel.pm --
#
#       Helper package for the VMware::Log[Obj] package.  This class
#       implements the channel mechanism for logs.  It is not really
#       meant to be used directly outside of VMware::LogObj
#
#       If a channel's destination is a filehandle reference, then
#       it never opens or closes the filehandle- it assumes that
#       the caller supplied an open handle and will take care of
#       closing it after use as needed.
#
#       If a channel's destination is a string, it assumes that the
#       string represents something that can be opened in append mode,
#       and does so for each message written to the log, closing it
#       after each write.
#


package VMware::Log::Channel;

use VMware::Log::Format;
use VMware::Log::DebugLevel qw(DebugCheck);
use VMware::Config qw(:config);

use strict;


########################################################################
#
# Channel::new --
#
#       Constructor for the channel object.  Builds a channel with
#       an ID, format, initial set of destination filehandles, and
#       optional verbosity config variable.
#
# Results:
#       Constructor.  Produces a new Channel instance.
#
# Side effects:
#       None.
#
########################################################################

sub new
{
   my $class = shift;      # IN: Invoking class name.
   my $self = {
      fhs => shift,        # IN: Hash of destination filehandles, by string ID.
      id => shift,         # IN: Identifier for this channel.
      fmt => shift,        # IN: Ref to ref to the format object for this
                           #     channel.  *NOTE DOUBLE REFERENCE*.  This is
                           #     so format changes in the LogObj are reflected
                           #     properly in all channels.
      threshold => shift,  # IN: If defined, hardcodes the verbosity
                           #     level of this channel.  Otherwise, the
                           #     threshold is expected on each Write call.
   };

   return bless $self => $class;
}


########################################################################
#
# AddDestination --
#
#       Add another filehandle or file path to this channel,
#       effectively splitting the output to multiple locations.
#
# Results:
#       True on success, false otherwise.  The only reason to return
#       false is that a destination already exists with that ID.
#
# Side effects:
#       Additional logging overhead for each call to the channel.
#
########################################################################

sub AddDestination
{
   my $self = shift;  # IN/OUT: Invoking instance.
   my $dest = shift;  # IN: Filehandle or path string for printing messages.
   my $id = shift;    # IN: ID associated with the filehandle.

   if (defined $self->{fhs}->{$id}) {
      return 0;
   }
   $self->{fhs}->{$id} = $dest;
   return 1;
}


########################################################################
#
# ClearDestinations --
#
#       Effectively close this channel by removing all filehandles.
#
# Results:
#       The hash of removed filehandles by ID.
#
# Side effects:
#       The channel no longer performs any I/O, and is effectively
#       a bitbucket.
#
########################################################################

sub ClearDestinations
{
   my $self = shift;  # IN/OUT: Invoking instance.

   my $ret = $self->{fhs};
   $self->{fhs} = {};
   return %$ret;
}


########################################################################
#
# GetDestinations --
#
#       Returns the destinations in use by this channel.
#
# Results:
#       The hash of filehandles by ID.
#
# Side effects:
#       None.
#
########################################################################

sub GetDestinations
{
   my $self = shift;  # IN: Invoking instance.
   return %{$self->{fhs}};
}


########################################################################
#
# Channel::Write --
#
#       Write the message to the destination as long as the threshold
#       and verbosity settings allow for it.
#
#       Does nifty splitting of lines and adjustments for ommitted
#       trailing newlines before routing things to the formatter.
#
#       TODO:  Decide if nifty linesplitting belongs here, in the
#       formatter, elsewhere, and/or should be configurable in some way.
#
# Results:
#       A list of arrays, one for each error in writing.  The arrays
#       have the following format:
#          index 0:  The ID of the destination filehandle that failed.
#          index 1:  The errno string from the print call.
#          index 2:  The message string that could not be written.
#
# Side effects:
#       Performs I/O on the destination file descriptors.
#
########################################################################

sub Write
{
   my $self = shift;            # IN: Invoking instance.
   my $msgstring = shift || ""; # IN: message to log.
   my $thresharg = shift;       # IN: Threshold to use if there is none
                                #     hardcoded for this channel.
   my @tags = @_;               # IN: Classification tags for fine-grained
                                #     debug level adjustment.  If any tag
                                #     in the list maps to a level at or
                                #     above the thresharg, the line
                                #     will be logged.

   if (defined $self->{threshold}) {
      #
      # This channel has a hardwired threshold, so ignore anything passed in.
      #
      $thresharg = $self->{threshold};

   } elsif  (not defined $thresharg) {
      #
      # We have neither a hardwired nor passed threshold, so default to 1.
      #
      $thresharg = 1;
   }

   #
   # Check verbosity every time because we would like to
   # be able to change verbosity dynamically, i.e.
   # by signalling a config reload, etc.
   #

   my @errors = ();
   if (DebugCheck($thresharg, @tags)) {
      foreach my $fhID (keys %{$self->{fhs}}) {
         chomp($msgstring);
         my @msg = split("\n", $msgstring);

         my $fh;
         if (ref $self->{fhs}->{$fhID}) {
            $fh = $self->{fhs}->{$fhID};
         } else {
            unless (open($fh, ">>$self->{fhs}->{$fhID}")) {
               push(@errors, [$fhID,
                              "Could not open '$self->{fhs}->{$fhID}': $!",
                              $msgstring]);
               next;
            }
         }

         foreach my $line (@msg) {
            my $formatted = ${$self->{fmt}}->Format($self->{id}, $line);
            unless (print $fh ($formatted, "\n")) {
               push(@errors, [$fhID, "$!", $msgstring]);
               next;
            }
         }
         unless (ref $self->{fhs}->{$fhID}) {
            #
            # Flush buffers (this is the closest analogue to calling fflush).
            # Calling close() does not guarantee an immediate flush.
            #

            my $old = select($fh);
            $| = 1;
            select($old);

            #
            # It is possible for a problem with flushing buffered
            # I/O to disk to be reported through an EIO from close().
            # So treat errors like print() errors.
            #

            unless (close($fh)) {
               push(@errors, [$fhID,
                              "Could not close '$self->{fhs}->{$fhID}': $!",
                              $msgstring]);
            }
         }
      }
   }
   return @errors;
}

1;

