########################################################################
# Copyright (C) 2003 VMWare, Inc.
# All Rights Reserved
########################################################################

#
# DebugLevel.pm --
#
#       Module defining global access to the debug level configuration
#       space, which is independent of the main global config.
#

package VMware::Log::DebugLevel;

use strict;

require Exporter;

@VMware::Log::DebugLevel::ISA = qw(Exporter);

@VMware::Log::DebugLevel::EXPORT_OK = qw(DebugCheck);

use VMware::Config::ConfigObj;
use VMware::Config qw(ConfigGet);

########################################################################
# Define the instance and declare the validation instance.
########################################################################

my $config = VMware::Config::ConfigObj->new();


########################################################################
#
# DebugLevel::AUTOLOAD --
#
#       We cannot directly use or require any VMware::Log modules,
#       because they depend on us.  This is only really a problem
#       when we use Log while perl is evaluating Log using us.
#       So once we're done with that, we can load log functions
#       on demmand.  That way we can still use logging here.
#
#       All this means is that DebugLevel cannot try to make
#       Logging calls in the compilation phase (like during
#       BEGIN blocks), nor can it use modules that do so.
#
#       This restriction is due to our dependence on ConfigObj
#       rather than directly from the code in this module.
#
# Results:
#       The results of the newly loaded routine.
#
# Side effects:
#       Loads the Log module.
#       Process death if the routine is still not defined after
#       the Log module has been loaded.
#
########################################################################

sub AUTOLOAD
{
   return if $VMware::Log::DebugLevel::AUTOLOAD =~ m/::DESTROY$/;
   eval "use VMware::Log qw(:log :manip)";
   if ($@) {
      #
      # Its reasonable to die here, because that's what would
      # happen if a totally undefined subroutine call was
      # made anyway.
      #
      die $@ . "\n";
   }
   if (defined &$VMware::Log::DebugLevel::AUTOLOAD) {
      goto &$VMware::Log::DebugLevel::AUTOLOAD;
   } else {
      die "Undefined subroutine in VMware::DebugLevel: " .
          "$VMware::DebugLevel::AUTOLOAD\n";
   }
}


########################################################################
#
# DebugLevel::DebugCheck --
#
#       Public subroutine for checking debug levels for a given
#       list of tags.
#
#       This is the method used by the logging system to determine
#       verbosity, and is also appropriate to call if a large block
#       of code should be conditionally executed depending on verbosity.
#
# Results:
#       True if the supplied level passes any relevant threshold.
#       Passing in this case means that it would be logged in the
#       context of a call to LogDebug or a similar channel.
#
# Side effects:
#       None.
#
########################################################################

sub DebugCheck
{
   my $level = shift; # IN: The level to check against.  Equivalent
                      #     to the level arg to LogDebug(), etc.
   my @tags = @_;  # IN: The tags whose level we want to find.

   if (not defined $level) {
      $level = 1;
   }

   my $threshPassed = 0;
   if (@tags) {
      foreach my $t (@tags) {
         #
         # Yes, we do need this "next" statement.  You can
         # get weird undef elements in the array under some
         # circumstances.
         #

         next if not defined $t;

         if ($config->Get($t, 0) >= $level) {
            $threshPassed = 1;
            last;
         }
      }
   }

   unless ($threshPassed) {
      my $v = ConfigGet("log.verbosity", 0);
      $threshPassed ||= ($v >= $level);
   }

   return $threshPassed;
}


########################################################################
#
# DebugLevel::SetObj --
#
#       Blow away the entire config space and replace it with a new one.
#       Since the DebugLevel system is usually set up or updated all
#       at once, this is currently the only way to affect it.
#
# Results:
#       The old ConfigObj.
#
# Side effects:
#       All current configuration is permanently erased, including
#       whatever validator was currently set.
#
########################################################################

sub SetObj
{
   my $old = $config;
   $config = shift;   # IN: The new ConfigObj instance.
   return $old;
}


########################################################################
#
# DebugLevel::ToHash --
#
#       Return a hash representing the current debug config space.
#
# Results:
#       A hash ref from the current ConfigObj.
#
# Side effects:
#       None.
#
########################################################################

sub ToHash
{
   return $config->ToHash();
}

1;

