##########################################################################
# Copyright 2003 VMware, Inc.  All rights reserved. -- VMware Confidential
##########################################################################


#
# Format.pm --
#
#       This module encapsulates formatting details for log files.
#       It handles both formatting messages for logging and
#       parsing such formatted messages back into meaningful fields.
#       It relies in part on the VERSION variable to figure out
#       how to interpret old log text for parsing.
#
#       TODO: Design/Implement the format string/parsing half of this.
#

package VMware::Log::Format;

use POSIX qw(strftime);
use File::Spec;

use strict;

#
# This version number need only be updated if you break compatibility.
# i.e., if the unit tests still run, then there's no reason to change it.
#

$VMware::Log::Format::VERSION = 1;


########################################################################
#
# Format::new --
#
#       Constructor.  Builds a format with all of the fields
#       turned off by default.
#
# Results:
#       A newly initialized format object.
#
# Side effects:
#       None.
#
########################################################################

sub new
{
   my $class = shift;      # IN: Invoking class name.
   my $self = {
      datestamp => (shift || 0),  # IN: If true, print date on each line.
      timestamp => (shift || 0),  # IN: If true, print time on each line.
      codestamp => shift,  # IN: If defined, should be a non-negative
                           #       integer.  This will cause the file, line and
                           #       subroutine name that far back in the stack
                           #       to be printed on each log line.  i.e. pass
                           #       a value of 1 to get the caller printed.
      pidstamp => (shift || 0),   # IN: If true, print pid on each line.
   };
   bless $self => $class;
   $self->BuildDateTimeFormat();
   return $self;
}


########################################################################
#
# Format::BuildDateTimeFormat --
#
#       Construct a strftime-compliant date/time format string
#       based on the current date- and timestamp settings.
#
# Results:
#       None.
#
# Side effects:
#       Rebuilds dtFormat data member.
#
########################################################################

sub BuildDateTimeFormat
{
   my $self = shift;  # IN/OUT: Invocant.

   $self->{dtFormat} = "";
   if ($self->{datestamp}) {
      $self->{dtFormat} .= "%Y-%m-%d";
   }
   if (($self->{datestamp}) && ($self->{timestamp})) {
      $self->{dtFormat} .= " ";
   }
   if ($self->{timestamp}) {
      $self->{dtFormat} .= "%H:%M:%S";
   }
}


########################################################################
#
# Format::ClearCode --
#
#       Unconditionally turns off the code identifier format option.
#
# Results:
#       None.
#
# Side effects:
#       Turns off codestamp.
#
########################################################################

sub ClearCode
{
   my $self = shift;  # IN/OUT: Invocant.

   #
   # Note that 0 is actually a valid codestamp.
   #

   $self->{codestamp} = undef;
}


########################################################################
#
# Format::ClearDate --
#
#       Unconditionally turns off the datestamp in the format.
#
# Results:
#       None.
#
# Side effects:
#       Turns off datestamp.
#
########################################################################

sub ClearDate
{
   my $self = shift;  # IN/OUT: Invocant.
   $self->{datestamp} = 0;
   BuildDateTimeFormat();
}


########################################################################
#
# Format::ClearPid --
#
#       Unconditionally turns off the process ID in the format.
#
# Results:
#       None.
#
# Side effects.
#       Turns off pidstamp.
#
########################################################################

sub ClearPid
{
   my $self = shift;  # IN/OUT: Invocant.
   $self->{pidstamp} = 0;
}


########################################################################
#
# Format::ClearTime --
#
#       Unconditionally turns off the timestamp in the format.
#
# Results:
#       None.
#
# Side effects:
#       Turns off timestamp.
#
########################################################################

sub ClearTime
{
   my $self = shift;  # IN/OUT: Invocant.
   $self->{timestamp} = 0;
   BuildDateTimeFormat();
}


########################################################################
#
# Format::Description --
#
#       Produces a string identifying the version and settings of
#       this format, such that the class can read it later and
#       reparse the output of the format.
#
# Results:
#       Format descriptions string.
#
# Side effects:
#       None.
#
########################################################################

sub Description
{
   my $self = shift; # IN: Invocant.
   my $desc = "FORMAT: $VMware::Log::Format::VERSION";

   if ($self->{datestamp}) {
      $desc .= ', datestamp, [0-9]{4}(?:-[0-9]{2}){2}';
   }
   if ($self->{timestamp}) {
      $desc .= ', timestamp, [0-9]{2}(?::[0-9]{2}){2}';
   }
   if (defined $self->{codestamp}) {
      #
      # TODO: This will break if we have files with ":" in them.
      #
      $desc .= ', codestamp, [^:]*:[0-9]+';
   }
   if ($self->{pidstamp}) {
      $desc .= ', pidstamp, \([0-9]+\)';
   }
   return $desc;
}

########################################################################
#
# Format::Format --
#       Format a string into a proper log line based on the currently
#       set options.
#
# Results:
#       Formatted string containing the data, ID, and whatever else
#       is currently configured.
#
# Side effects:
#       None.
#
########################################################################

sub Format
{
   my $self = shift;       # IN: Invocant.
   my $id = shift;         # IN: Free-form ID string.  Typically the ID
                           #     of the VMware::Log::Channel writing the msg.
   my $data = shift || ""; # IN: The actual message.

   my $prefix = "";
   if (defined $self->{dtFormat}) {
      $prefix .= strftime($self->{dtFormat}, localtime());
   }
   if ((defined $self->{codestamp}) && ($self->{codestamp} >= 0)) {
      if ($prefix) {
         $prefix .= " ";
      }
      my (undef, $filename, $line) = caller($self->{codestamp});
      $filename = File::Spec->canonpath($filename);
      $prefix .= "$filename:$line";
   }
   if ($self->{pidstamp}) {
      if ($prefix) {
         $prefix .= " ";
      }
      $prefix .= "(" . $$ . ")";
   }

   if ($prefix) {
      $prefix .= " ";
   }
   $prefix .= "$id:  ";

   return "$prefix$data";
}


########################################################################
#
# Format::SetCode --
#
#       Set the code location stamp for the format.
#
# Results:
#       None.
#
# Side effects:
#       Codestamp is set to the given stack depth, which is 0 by
#       default (immediate caller).
#
########################################################################

sub SetCode
{
   my $self = shift;  # IN/OUT: Invocant.

   # If no paramter passed, default
   # trace depth is 0.
   $self->{codestamp} = shift || 0;
}


########################################################################
#
# Format::SetDate --
#
#       Unconditionally turn on datestamps.
#
# Results:
#       None.
#
# Side effects:
#       Turn on datestamp.
#
########################################################################

sub SetDate
{
   my $self = shift;  # IN/OUT: Invocant.
   $self->{datestamp} = 1;
   BuildDateTimeFormat();
}


########################################################################
#
# Format::SetPid --
#
#       Unconditionally turn on pid.
#
# Results:
#       None.
#
# Side effects:
#       Turn on pidstamp.
#
########################################################################

sub SetPid
{
   my $self = shift;  # IN/OUT: Invocant.
   $self->{pidstamp} = 1;
}


########################################################################
#
# Format::SetTime --
#
#       Unconditionally turn on timestamps.
#
# Results:
#       None.
#
# Side effects:
#       Turn on timestamp.
#
########################################################################

sub SetTime
{
   my $self = shift;  # IN/OUT: Invocant.
   $self->{timestamp} = 1;
   BuildDateTimeFormat();
}

1;

