##########################################################################
# Copyright 2003 VMware, Inc.  All rights reserved. -- VMware Confidential
##########################################################################


#
# LogObj.pm --
#
#       A log object is a collection of channels together with a format.
#
#       This class supports four special channels:
#
#           INFO : Informational messages that should always be logged.
#
#           WARN : Warnings that should *ALWAYS* be examined when present
#                  and may help diagnose errors.
#
#           ERROR: Error messages.
#
#           DEBUG: Informational messages that are conditionally
#                  logged depending on checks in VMware::Log::DebugLevel.
#                  
#       Additional channels may be added, and all channels may be set
#       to go to different/multiple destinations.
#
#

package VMware::Log::LogObj;

use VMware::Config qw(:config);
use VMware::Log::Channel;
use VMware::Log::Format;

use strict;

#
# Constants for the IDs of the standard channels, allowing users
# to apply the generic functions such as GetCount and AddDestination
# to them.  ID_INTERNAL is for the logging system's internal channel,
# and should probably usually be left alone.  But perhaps not always.
#

use constant ID_INFO => "INFO ";
use constant ID_WARN => "WARN ";
use constant ID_ERROR => "ERROR";
use constant ID_DEBUG => "DEBUG";
use constant ID_INTERNAL => "VMWARE_LOG_INTERNAL";

#
# When specifying a format that uses codestamps, this is the number
# that should be added to the stack depth for the stamp to compensate
# for calls made in this module.  Do not use the raw number directly,
# as it is subject to change if abstraction layers are added or removed.
#

use constant STACK_DEPTH_OFFSET => 2;

#
# These constants are for use in specifying error handling policy for
# a LogObj, specifically when writing to or reading from a log.  
# Other, more administrative methods communicate errors in 
# standard ways (such as returning undef).  However, checking the
# return value of logging statements is cumbersome, so this mechanism
# allows for alternatives.
#
# The logging methods always return undef on error, but these options
# allow clients to reasonably avoid checking that return call.
# Policy flags may be ANDed to request use of more than one policy.
#

# Write errors to the internal log.
use constant ON_ERROR_LOG => 1 << 0;

# Attempt to write an error message and the original
# message to STDERR, ignoring any further problems.
use constant ON_ERROR_STDERR => 1 << 1;

# Kill the process whenever there is a logging error.
use constant ON_ERROR_FATAL => 1 << 2;


########################################################################
#
# LogObj::new --
#
#       Create a log object based on a supplied format and destination.
#       Create all of the standard channels based on that destination.
#       Writes the format description to the destination.
#
# Results:
#       A fully initialized log object, or undef if the destination
#       could not be written to.
#
# Side effects:
#       None.
#
########################################################################

sub new
{
   my $class = shift;  # IN: Invoking instance or class name.
   my $dest = shift;   # IN: Destination filehandle reference.
   my $destID = shift; # IN: String describing the destination filehandle.
   my $format = shift; # IN: Format for log messages.  If codestamps
                       #     are being used, the stack depth should
                       #     be STACK_DEPTH_OFFSET greater than expected
                       #     to skip the call(s) made in this module.
   my $errorPolicy = shift; # IN: Determines how errors in writes or reads
                            #     to/from log destinations are handled.
                            #     Must be logical ands of the constants
                            #     defined in this package.
   my $noFormatLines = shift; # IN: If true, do not print the format lines.

   if (not defined $format) {
      $format = VMware::Log::Format->new();

      $format->SetDate();
      $format->SetTime();

      $format->SetCode(STACK_DEPTH_OFFSET);
   }

   if (not defined $dest) {
      #
      # TODO: Do we want to have ERROR and WARN go to STDERR by default instead?
      #

      $dest = \*STDOUT;
      if (not defined $destID) {
         $destID = "STDOUT";
      }
   }

   my %channels;
   my $self = {
      channels => \%channels,
      counters => {},
      format => $format,
      errorPolicy => defined($errorPolicy) ? $errorPolicy : 0,
      destCounts => {},
      destinations => {},
      noFormatLines => ($noFormatLines || 0)
   };

   bless $self => $class;

   $self->CreateChannel(ID_INFO, $dest, $destID, 0) || return undef;
   $self->CreateChannel(ID_WARN, $dest, $destID, 0) || return undef;
   $self->CreateChannel(ID_ERROR, $dest, $destID, 0) || return undef;
   $self->CreateChannel(ID_DEBUG, $dest, $destID) || return undef;
   $self->CreateChannel(ID_INTERNAL, $dest, $destID, 0) || return undef;

   return $self;
}


########################################################################
#
# LogObj::Resume --
#
# 	After log object has been persisted and restored, file handles
# 	will be invalid.  Resume logging as it was by reopening each
# 	channel's file handles in append mode.
#
# Results:
#       True on success, undef otherwise.
#
# Side effects:
#       Log file handles are reopened.
#
########################################################################

sub Resume
{
   my $self = shift; # IN/OUT: Invoking instance.

   my $restored = {};

   foreach my $channelName (keys %{$self->{channels}}) {
      my $channel = $self->{channels}->{$channelName};
      my %dests = $channel->GetDestinations();

      foreach my $id (keys %dests) {
	 my $fh = $dests{$id};
         next if ($fh =~ /(?:STDOUT|STDERR)/);

	 unless ($restored->{$id}) {
	    unless (open ($fh, ">>$id")) {
               my $e = "Couldn't restore log file: $id: $!\n";
               if ($self->{errorPolicy} & ON_ERROR_LOG()) {
                  eval {
                     $self->Log(ID_INTERNAL, $e);
                  };
               }
               if ($self->{errorPolicy} & ON_ERROR_STDERR()) {
                  print STDERR $e;
               }
               if ($self->{errorPolicy} & ON_ERROR_THROW()) {
                  die $e;
               }
	       return undef;
	    }
	    select((select($fh), $| = 1)[0]);
	    $restored->{$id} = $fh;
	 }
      }
   }

   return 1;
}


########################################################################
#
# LogObj::Info --
#
#       Shorthand method for the info channel.  Acts exactly like
#       calling Log(ID_INFO, $msg);
#       Informational messages are always printed, and should be
#       critical messages that aid in general log interpretation.
#
# Results:
#       True if all is well, depend on HandleError otherwise.
#
# Side effects:
#       Writes a message to the channel and increments the associated
#       counter.  Other effects depend on HandleError iff there is a problem.
#
########################################################################

sub Info
{
   my $self = shift;  # IN/OUT: Invoking instance.
   my $msg = shift;   # IN: Message to log.

   ++($self->{counters}->{ID_INFO()});
   my @errors = $self->{channels}->{ID_INFO()}->Write($msg);
   if (@errors) {
      return $self->HandleError("writing", ID_INFO, @errors);
   }
   return 1;
}


########################################################################
#
# LogObj::Warn --
#
#       Shorthand method for the warn channel.  Acts exactly like
#       calling Log(ID_WARN, $msg);
#       Warnings are always printed, and should be messages that
#       may or may not be problems but always warrant manual inspection.
#
# Results:
#       True if all is well, depend on HandleError otherwise.
#
# Side effects:
#       Writes a message to the channel and increments the associated
#       counter.  Other effects depend on HandleError iff there is a problem.
#
########################################################################

sub Warn
{
   my $self = shift;  # IN/OUT: Invoking instance.
   my $msg = shift;   # IN: Message to log.

   ++($self->{counters}->{ID_WARN()});
   my @errors = $self->{channels}->{ID_WARN()}->Write($msg);
   if (@errors) {
      return $self->HandleError("writing", ID_WARN, @errors);
   }
   return 1;
}


########################################################################
#
# LogObj::Error --
#
#       Shorthand method for the error channel.  Acts exactly like
#       calling Log(ID_ERROR, $msg);
#       Errors are always printed, and should be used to log error
#       conditions.
#
# Results:
#       True if all is well, depend on HandleError otherwise.
#
# Side effects:
#       Writes a message to the channel and increments the associated
#       counter.  Other effects depend on HandleError iff there is a problem.
#
########################################################################

sub Error
{
   my $self = shift;  # IN/OUT: Invoking instance.
   my $msg = shift;   # IN: Message to log.

   ++($self->{counters}->{ID_ERROR()});
   my @errors = $self->{channels}->{ID_ERROR()}->Write($msg);
   if (@errors) {
      return $self->HandleError("writing", ID_ERROR, @errors);
   }
   return 1;
}


########################################################################
#
# LogObj::Debug --
#
#       Shorthand method for the debug channel.  Acts exactly like
#       calling Log(ID_DEBUG, $msg, $level);
#       Debug messages are logged based on the level supplied and the
#       current verbosity settings.
#
# Results:
#       True if all is well, depend on HandleError otherwise.
#
# Side effects:
#       Writes a message to the channel and increments the associated
#       counter.  Other effects depend on HandleError iff there is a problem.
#
########################################################################

sub Debug
{
   my $self = shift;  # IN/OUT: Invoking instance.
   my $msg = shift;   # IN: Message to log.
   my $level = shift; # IN: Verbosity level.
   my @tags = @_;     # IN: Classification tag for fine-grained control.

   ++($self->{counters}->{ID_DEBUG()});
   my @errors = $self->{channels}->{ID_DEBUG()}->Write($msg, $level, @tags);
   if (@errors) {
      return $self->HandleError("writing", ID_DEBUG, @errors);
   }
   return 1;
}


########################################################################
#
# LogObj::Log --
#
#       Generic method to write a log message to any channel.
#
# Results:
#       True if all is well, depend on HandleError otherwise.
#
# Side effects:
#       Writes a message to the channel and increments the associated
#       counter.  Other effects depend on HandleError iff there is a problem.
#
########################################################################

sub Log
{
   my $self = shift;       # IN/OUT: Invoking instance.
   my $channelID = shift;  # IN: Channel to which to log.
   my $msg = shift;        # IN: Message to log.
   my $level = shift;      # IN: Verbosity level.
   my @tags = @_;          # IN: Classification tags for fine-grained control.

   my $channel = $self->{channels}->{$channelID};
   if ((not defined $channel) || ($channelID eq ID_INTERNAL())) {
      # TODO: Error handling policies.
      return;
   }
   ++($self->{counters}->{$channelID});
   my @errors = $channel->Write($msg, $level, @tags);
   if (@errors) {
      return $self->HandleError("writing", $channelID, @errors);
   }
   return 1;
}


########################################################################
#
# LogObj::GetCount --
#
#       Return the count of how often a channel has been accessed
#       by the Log method (or one of the shorthand methods for
#       the standard channels).  Note that this is the access count,
#       so calls that do not result in a print due to level and
#       verbosity settings count as much as calls that print messages.
#
# Results:
#       The access count for the channel.
#
# Side effects:
#       None.
#
########################################################################

sub GetCount
{
   my $self = shift;      # IN: Invoking instance.
   my $channelID = shift; # IN: The channel whose count we want.

   if (defined $self->{channels}->{$channelID}) {
      return $self->{counters}->{$channelID};
   } else {
      return undef;
   }
}


########################################################################
#
# LogObj::CreateChannel --
#
#       Create a new channel on this log object and do the necessary
#       accounting with respect to its destinations.
#
# Results:
#       False if a channel with that id already exists, or if
#       the system needed to write a format description to
#       the destination and could not.
#       True otherwise.
#
# Side effects:
#       Calls IncrementDestCount- see that method's comment.
#
########################################################################

sub CreateChannel
{
   my $self = shift;       # IN/OUT: Invoking instance.
   my $channelID = shift;  # IN: ID of the channel to create.
   my $dest = shift;       # IN: Filehandle.
   my $destID = shift;     # IN: String ID of the filehandle.
   my $threshold = shift;  # IN: Hardcoded threshold/level of this channel,
                           #     or undef if it should be passed for each
                           #     logging call on the channel.

   if (defined $self->{channels}->{$channelID}) {
      return 0;
   }

   $self->{channels}->{$channelID} =
      VMware::Log::Channel->new({$destID => $dest},
                                $channelID,
                                \$self->{format},
                                $threshold);
   $self->{counters}->{$channelID} = 0;
   return $self->IncrementDestCount($dest, $destID);
}


########################################################################
#
# LogObj::DuplicateChannel --
#
#       Create a new channel for this log object with the same
#       destination(s) as an existing channel.
#
# Results:
#       False if the source channel does not exist or the new
#       channel already does exist, or if the system needed to
#       write a format description to the destination(s) and
#       could not.
#       True otherwise.
#
# Side effects:
#       Calls IncrementDestCount for each destination in the
#       source channel- see that method's comment.
#
########################################################################

sub DuplicateChannel
{
   my $self = shift;       # IN/OUT: Invoking instance.
   my $srcID = shift;      # IN: ID of the duplication source.
   my $newID = shift;      # IN: ID of the new channel to create.
   my $threshold = shift;  # IN: Hardcoded threshold/level of this channel,
                           #     or undef if it should be passed for each
                           #     logging call on the channel.

   if (defined $self->{channels}->{$newID} ||
       not defined $self->{channels}->{$srcID}) {
      return 0;
   }
   my %destinations = $self->{channels}->{$srcID}->GetDestinations();
   if (not scalar(%destinations)) {
      return 1;
   }

   my ($destID, $destFH) = each %destinations;

   $self->{channels}->{$newID} =
      VMware::Log::Channel->new({$destID => $destFH},
                                $newID,
                                \$self->{format},
                                $threshold);
   $self->{counters}->{$newID} = 0;
   my $rc = $self->IncrementDestCount($destFH, $destID);

   while (($destID, $destFH) = each %destinations) {
      $rc &= $self->AddDestination($destFH, $destID, $newID);
   }
   return $rc;
}


########################################################################
#
# LogObj::AddDestination --
#
#       Invokes AddDestination on the specified channel, or on all
#       channels if none is specified, and does the necessary
#       accounting with respect to destinations.
#
#       See comments in Channel.pm about how filehandles vs.
#       file path strings are dealt with.
#
# Results:
#       True on success, false if a requested channel does not exist,
#       a destination with that ID already exists, 
#       or if the system needed to write the format description to
#       the destination and could not.
#
#
# Side effects:
#       Calls IncrementDestCount- see that method's comment.
#
########################################################################

sub AddDestination
{
   my $self = shift;       # IN/OUT: Invoking instance.
   my $dest = shift;       # IN: Open filehandle or file path string.
   my $destID = shift;     # IN: String identifier of the filehandle.
   my $channelID = shift;  # IN: Channel to which to add this dest, 
                           #     or undef to add to all existing channels.

   my $ret = 0;
   if (defined $channelID) {
      if (not defined $self->{channels}->{$channelID}) {
         return 0;
      }
      $ret = $self->{channels}->{$channelID}->AddDestination($dest, $destID);
      if ($ret) {
         $ret = $self->IncrementDestCount($dest, $destID);
      }

   } else {
      $ret = 1;
      foreach my $id (keys %{$self->{channels}}) {
         my $added = $self->{channels}->{$id}->AddDestination($dest, $destID);
         if ($added) {
            $ret &= $self->IncrementDestCount($dest, $destID);
         }
         $ret &= $added;
      }
   }
   return $ret;
}


########################################################################
#
# LogObj::ClearDestinations --
#
#       Invokes ClearDestinations on the specified channel, or on all
#       channels if none is specified, and does the necessary
#       accounting with respect to destinations.
#
# Results
#       If given a specific channel ID, returns the results of calling
#       ClearDestinations on that channel (i.e., the hash of filehandles
#       by ID that were being used as destinations).
#       If applied to all channels, returns a hash mapping channel
#       IDs to the return hash for each channel.
#
# Side Effects
#       Calls DecrementDestCount- see that method's comment.
#
########################################################################

sub ClearDestinations
{
   my $self = shift;       # IN/OUT: Invoking instance
   my $channelID = shift;  # IN: ID of the channel to clear, or undef to
                           #     clear all.

   if (defined $channelID) {
      if (defined $self->{channels}->{$channelID}) {

         my %ret = $self->{channels}->{$channelID}->ClearDestinations();
         foreach my $destID (keys %ret) {
            $self->DecrementDestCount($ret{$destID}, $destID);
         }
         return %ret;
      } else {
         return undef;
      }
   } else {
      my %retHash;
      foreach my $id (keys %{$self->{channels}}) {
         my %ret = $self->{channels}->{$id}->ClearDestinations();
         foreach my $destID (keys %ret) {
            $self->DecrementDestCount($ret{$destID}, $destID);
         }
         $retHash{$id} = \%ret;
      }
      return %retHash;
   }
}


########################################################################
#
# LogObj::GetDestinations --
#
#       Invokes GetDestinations on the specified channel, or on all
#       channels if none is specified, and hands back the results.
#
# Results:
#       If given a specific channel ID, returns the results of calling
#       GetDestinations on that channel (i.e., the hash of filehandles
#       or file path strings by ID that were being used as destinations).
#       If applied to all channels, returns a hash mapping channel
#       IDs to the return hash for each channel.
#
# Side effects:
#       None.
#
########################################################################

sub GetDestinations
{
   my $self = shift;       # IN: Invoking instance
   my $channelID = shift;  # IN: ID of the channel, or undef to get from all.

   if (defined $channelID) {
      if (defined $self->{channels}->{$channelID}) {
         return $self->{channels}->{$channelID}->GetDestinations();
      } else {
         return undef;
      }
   } else {
      my %retHash;
      foreach my $id (keys %{$self->{channels}}) {
         my %ret = $self->{channels}->{$id}->GetDestinations();
         $retHash{$id} = \%ret;
      }
      return %retHash;
   }
}


########################################################################
#
# LogObj::IncrementDestCount --
#
#       Handles the necessary accounting when a destination comes into
#       use through either channel creation or the addition of a new
#       destination to an existing channel.
#
# Results:
#       True on success, false if it needed to print the format
#       description to the destination and could not.
#
# Side effects:
#       Bumps the refcount on the destination- if it is a new 
#       destination, writes the format description to it.
#
########################################################################

sub IncrementDestCount
{
   my $self = shift;   # IN/OUT: Invoking instance.
   my $dest = shift;   # IN: Filehandle for the destination.
   my $destID = shift; # IN: ID string corresponding to the filehandle.

   unless (defined($self->{destCounts}->{$destID})) {
      $self->{destCounts}->{$destID} = 0;
   }
   unless ($self->{destCounts}->{$destID}++) {
      $self->{destinations}->{$destID} = $dest;

      return 1 if $self->{noFormatLines};
      my $ret = print $dest $self->{format}->Description(), "\n";
      unless ($ret) {
         #
         # Try printing error to the internal log.  Even if this was an error
         # on the internal log's destination, there may be other open
         # destinations.  Catch any exceptions in case we're using that policy.
         # Return 0 no matter what, since either way we've failed to write
         # the format.
         #

         eval {
            $self->Log(ID_INTERNAL,
                       "Could not write format description to $destID: $!\n");
         };
         return 0;
      }
   }
   return 1;
}


########################################################################
#
# LogObj::DecrementDestCount --
#
#       Handles the necessary accounting when a channel is no longer
#       using a destination.
#
# Results:
#       none
#
# Side effects:
#       Lowers the refcount of a destination by one, and deletes it from
#       the hash of active destinations if the result is zero.
#
########################################################################

sub DecrementDestCount
{
   my $self = shift;   # IN/OUT: Invoking instance
   my $dest = shift;   # IN: Filehandle.  Currently unused, but nicely
                       #     symmetric.
   my $destID = shift; # IN: ID string corresponding to the filehandle.

   unless (--$self->{destCounts}->{$destID}) {
      delete $self->{destinations}->{$destID};
   }
}


########################################################################
#
# LogObj::SetFormat --
#
#       Set a new format for the entire log object (i.e., all channels).
#
# Results:
#       The old format if successful, undef if the new format desription
#       could not be written to all destinations for all channels.
#       Also undef if the new format was undefined.
#
#       TODO: The old format is restored on error, but we do not attempt to
#       reprint the old format lines, which could cause problems.  However,
#       something is presumably very wrong with the logging system if this
#       ever happens anyway.
#
# Side effects:
#       Writes the description of the new format to all destinations
#       currently in use.
#
########################################################################

sub SetFormat
{
   my $self = shift;   # IN/OUT: Invoking instance.
   my $format = shift; # IN: The new format.

   unless (defined $format) {
      return undef;
   }

   my $oldfmt = $self->{format};
   $self->{format} = $format;

   return $oldfmt if $self->{noFormatLines};

   foreach my $dest (values %{$self->{destinations}}) {

      my $ret = print $dest $self->{format}->Description(), "\n";
      unless ($ret) {
         #
         # Try printing error to the internal log.  Even if this was an error
         # on the internal log's destination, there may be other open
         # destinations.  Catch any exceptions in case we're using that
         # policy.  Return undef no matter what, since either way we've failed
         # to write the format.
         #

         eval {
            $self->Log(ID_INTERNAL,
                       "Could not write format description for new " .
                       "format: $!\n");
         };
         $self->{format} = $oldfmt;
         return undef;
      }
   }
   return $oldfmt;
}


########################################################################
#
# LogObj::HandleError --
#
#       Interprets the current error-handling policy settings and
#       behaves appropriately.
#
# Results:
#       undef if it returns at all.
#
# Side effects:
#       May try to write to the internal log or STDERR
#       Uses the package global $Recursing to detect recursive errors.
#       May kill the process (or at least throw a perlish
#       "exception") if ON_ERROR_FATAL is selected.
#
########################################################################

use vars qw($Recursing);
$Recursing = 0;
sub HandleError
{
   my $self = shift;      # IN: Invoking instance
   my $op = shift;        # IN: Action we were trying to perform.
   my $channelID = shift; # IN: Channel that encountered the error.
   my @errors = @_;       # IN: Errors from write.  The contents of
                          #     the array are themselves arrays where
                          #     the first element is the filehandle,
                          #     the second is the string form of $!,
                          #     and the third is the message that was being
                          #     written/read (probably empty on read). 
   my %rhash = reverse %{$self->{destinations}};

   foreach my $error (@errors) {
      my $destID = $rhash{$error->[0]};

      unless (defined $destID) {
         $destID = '<unknown>';
      }

      my $e = "Error ${op} to $destID from channel $channelID:\n" .
              "$error->[1]\n$error->[2]\n";

      if ($self->{errorPolicy} & ON_ERROR_LOG) {
         if ($Recursing) {
            #
            # Give up.
            #
         }
         local $Recursing = 1;
         eval {
            $self->Log(ID_INTERNAL, $e);
         };
      }
      if ($self->{errorPolicy} & ON_ERROR_STDERR) {
         print STDERR $e;
      }
      if ($self->{errorPolicy} & ON_ERROR_FATAL) {
         die $e;
      }
   }
   return undef;
}

1;


