
use strict;
use File::Basename;
use File::Spec;
use IO::Pipe;
use IO::Handle;
use Test;

BEGIN {
   plan tests => 19;
   my $path = File::Spec->catfile(dirname($0), "..", "..", "..");
   push @INC, $path;
   push @INC, ".";
}

use VMware::Config qw(ConfigSet);
use VMware::Config::ConfigObj;
use VMware::Log::Channel;
use VMware::Log::DebugLevel;

########################################################################
# GetPipeHandles
#       Convenince method to get the read and write ends of a pipe.
#       For the purpose of this test we don't need the actual
#       pipe object at all, so ignore it (its just used to attach
#       the handles).  At the time this was written, I was under
#       the delusion that regular pipe() wouldn't work on Windows
#       for some reason, and there's no point in changing it now.
#
########################################################################

sub GetPipeHandles
{
   my $r = IO::Handle->new();
   my $w = IO::Handle->new();
   my $p = new IO::Pipe ($r, $w);
   if (not defined $p) {
      die "Could not construct an IO::Pipe: $!\n";
   }
   return ($r, $w);
}

#
# Turn off all of the formatting options because they mess up comparisions
# and they should be covered by the Format unit tests anyway.
#

my $fmt = new VMware::Log::Format::();
$fmt->ClearDate(0);
$fmt->ClearTime(0);
$fmt->ClearCode(undef);

#
# The basic idea is to write everything to the "write"
# ends of the pipes, close the pipes so we don't deadlock,
# and read the results out the read end.
#
# Focus on destinations for this section.
#

my $id1 = "ID1";
my ($r1, $w1) = GetPipeHandles();
my ($r2, $w2) = GetPipeHandles();

my $c1 = new VMware::Log::Channel::({writer1 => $w1}, $id1, \$fmt, 0);

my $msg1 = "msg1\n";  # With newline
my $msg2 = "msg2";    # Without newline
my $msg3 = "msg3\n";
my $formatted1 = $fmt->Format($id1, $msg1);
my $formatted2 = $fmt->Format($id1, $msg2);
my $formatted3 = $fmt->Format($id1, $msg3);

$c1->Write($msg1);

ok($c1->AddDestination($w2, "writer2"), 1);
ok($c1->AddDestination($w1, "writer2"), 0, "Addition of duplicate ID allowed.");

# This should be written to both pipes.
$c1->Write($msg2);
my %fhs = $c1->GetDestinations();
ok(scalar(keys %fhs), 2, "Wrong number of filehandles.");
ok($fhs{writer1}, $w1);
ok($fhs{writer2}, $w2);

%fhs = ();
%fhs = $c1->ClearDestinations();

ok(scalar(keys %fhs), 2, "Wrong number of cleared filehandles.");
ok($fhs{writer1}, $w1);
ok($fhs{writer2}, $w2);

# This shouldn't show up anywhere.
$c1->Write($msg3);
$c1 = undef;
close($w1) || die "Couldn't close write end of first pipe: $!";
close($w2) || die "Couldn't close write end of second pipe: $!";

ok(<$r1>, $formatted1);
ok(<$r1>, "$formatted2\n");  # re-add newline
ok(not defined <$r1>);

ok(<$r2>, "$formatted2\n");  # re-add newline
ok(not defined <$r2>);

close($r1) || die "Couldn't close read end of first pipe: $!";
close($r2) || die "Couldn't close read end of second pipe: $!";

#
# Same basic idea, but focusing on thresholds this time.
# First try out a fixed (but non-zero) threshold, and
# then test runtime (per-Write-call) thresholds.
#

my $idFixed = "FIXED";
my ($rf, $wf) = GetPipeHandles();

my $fmsg = "this message won't be seen.\n";
my $fixed = VMware::Log::Channel->new({wf => $wf}, $idFixed, \$fmt, 1);
$fixed->Write($fmsg);
$fixed = undef;
close($wf) || die "Couldn't close write end of fixed threshold pipe: $!";

ok(not defined <$rf>);

close($rf) || die "Couldn't close read end of fixed threshold pipe: $!";

my $idRuntime = "RUNTIME";
my ($rr, $wr) = GetPipeHandles();

my $rmsg1 = "First message\n";
my $rmsg2 = "Second message\n";
my $rmsg3 = "Third message\n";

my $rresult1 = $fmt->Format($idRuntime, $rmsg1);
my $rresult2 = $fmt->Format($idRuntime, $rmsg2);
my $rresult3 = $fmt->Format($idRuntime, $rmsg3);

#
# The verbosity setting should be zero for these tests.
#
ConfigSet("log.verbosity", 0);

my $runtime = VMware::Log::Channel->new({wr => $wr},
                                      $idRuntime,
                                      \$fmt,
                                      undef);
$runtime->Write($rmsg1);
$runtime->Write($rmsg2, 1);
$runtime->Write($rmsg3, 0);

#
# Check tag-related verbosity.
#
my $levelConfig = new VMware::Config::ConfigObj::();
$levelConfig->LoadFromHash({"foo" => 1, "bar" => 2});
VMware::Log::DebugLevel::SetObj($levelConfig);

$runtime->Write($rmsg1, 2, "foo");
$runtime->Write($rmsg2, 2, "bar");
$runtime->Write($rmsg3, 2, "foo", "bar");
$runtime->Write($rmsg1, 2, "bar", "foo");
$runtime->Write($rmsg2, 4, "foo", "bar");
$runtime->Write($rmsg3, 4, "zippy");

close($wr) || die "Couldn't close write end of runtime threshold pipe: $!";
$runtime = undef;

ok(<$rr>, $rresult3);
ok(<$rr>, $rresult2);
ok(<$rr>, $rresult3);
ok(<$rr>, $rresult1);
ok(not defined <$rr>);

close($rr) || die "Couldn't close read end of runtime threshold pipe: $!";

#
# TODO:  Implement unit tests for error handling policies once
#        we actually have some.
#

