
use strict;
use File::Basename;
use File::Spec;
use Test;

BEGIN {
   plan tests => 124;
   my $path = File::Spec->catfile(dirname($0), "..", "..", "..");
   push @INC, $path;
   push @INC, ".";
}

use VMware::Log::Format;
use VMware::Log::LogObj;
use VMware::FileManip qw(FileSlurpArray);

########################################################################
# TODO:  Write tests for error handling mechanisms.
#
# TODO:  Write tests for debugging channel verbosity.
#
########################################################################


#
# Turn off all of the formatting options because they mess up comparisions
# and they should be covered by the Format unit tests anyway.
#

my $fmt = new VMware::Log::Format::();
$fmt->ClearDate(0);
$fmt->ClearTime(0);
$fmt->ClearCode(undef);

#
# The basic idea is to write everything to the "write"
# ends of the pipes, close the pipes so we don't deadlock,
# and read the results out the read end.
#

pipe (READER, WRITER) || die "Could not call pipe: $!";
my $reader = \*READER;
my $writer = \*WRITER;

select $writer;
$| = 1;
select STDOUT;

# These are too long to type all the time.
my $infoID = VMware::Log::LogObj::ID_INFO;
my $warnID = VMware::Log::LogObj::ID_WARN;
my $errorID = VMware::Log::LogObj::ID_ERROR;
my $debugID = VMware::Log::LogObj::ID_DEBUG;
my $internalID = VMware::Log::LogObj::ID_INTERNAL;

my $log = VMware::Log::LogObj->new($writer,
                                   "unnamed pipe",
                                   $fmt,
                                   VMware::Log::LogObj::ON_ERROR_FATAL);

#
# Basic write/count tests.
#

ok(<$reader>, ($fmt->Description() . "\n"));
ok($log->GetCount($infoID), 0);
ok($log->GetCount($warnID), 0);
ok($log->GetCount($errorID), 0);
ok($log->GetCount($debugID), 0);
ok($log->GetCount($internalID), 0);


ok($log->Info("Info"), 1);
ok($log->GetCount($infoID), 1);
ok($log->Warn("Warn"), 1);
ok($log->GetCount($warnID), 1);
ok($log->Error("Error"), 1);
ok($log->GetCount($errorID), 1);

ok(<$reader>, ($infoID . ":  Info\n"));
ok(<$reader>, ($warnID . ":  Warn\n"));
ok(<$reader>, ($errorID . ":  Error\n"));

#
# Add/clear destination tests.
#

pipe (R1, W1) || die "Could not make pipe 1: $!";
pipe (R2, W2) || die "Could not make pipe 2: $!";

select W1;
$| = 1;
select W2;
$| = 1;
select STDOUT;

my $w1 = \*W1;
my $r1 = \*R1;
my $w2 = \*W2;
my $r2 = \*R2;

#
# Add to all channels.
#

ok($log->AddDestination($w1, "pipe 1"), 1);
ok($log->AddDestination(\*STDERR, "pipe 1"), 0, "Duplicate destID allowed.");

ok(<$r1>, ($fmt->Description() . "\n"));
ok($log->Info("dest1"), 1);
ok($log->Warn("dest1"), 1);
ok($log->Error("dest1"), 1);
ok($log->Debug("dest1", 0), 1);

ok(<$reader>, ($infoID . ":  dest1\n"));
ok(<$reader>, ($warnID . ":  dest1\n"));
ok(<$reader>, ($errorID . ":  dest1\n"));
ok(<$reader>, ($debugID . ":  dest1\n"));
ok(<$r1>, ($infoID . ":  dest1\n"));
ok(<$r1>, ($warnID . ":  dest1\n"));
ok(<$r1>, ($errorID . ":  dest1\n"));
ok(<$r1>, ($debugID . ":  dest1\n"));

#
# Add to only one channel.
#

ok($log->AddDestination($w2, "pipe 2", $errorID), 1);
ok($log->AddDestination(\*STDERR, "pipe 2", $errorID),
   0,
   "Duplicate destID allowed.");

ok(<$r2>, ($fmt->Description() . "\n"));
ok($log->Info("dest2"), 1);
ok($log->Warn("dest2"), 1);
ok($log->Error("dest2"), 1);

ok(<$reader>, ($infoID . ":  dest2\n"));
ok(<$reader>, ($warnID . ":  dest2\n"));
ok(<$reader>, ($errorID . ":  dest2\n"));
ok(<$r1>, ($infoID . ":  dest2\n"));
ok(<$r1>, ($warnID . ":  dest2\n"));
ok(<$r1>, ($errorID . ":  dest2\n"));
ok(<$r2>, ($errorID . ":  dest2\n"));

#
# Get destinations on one channel.
#

my %fds = $log->GetDestinations($infoID);
ok(scalar(keys %fds), 2);
ok($fds{"unnamed pipe"}, $writer);
ok($fds{"pipe 1"}, $w1);

#
# Clear destinations on one channel.
#

%fds = ();
%fds = $log->ClearDestinations($warnID);
ok(scalar(keys %fds), 2);
ok($fds{"unnamed pipe"}, $writer);
ok($fds{"pipe 1"}, $w1);

ok($log->Info("clear1"), 1);
ok($log->Warn("clear1"), 1);
ok($log->Error("clear1"), 1);

ok(<$reader>, ($infoID . ":  clear1\n"));
ok(<$reader>, ($errorID . ":  clear1\n"));
ok(<$r1>, ($infoID . ":  clear1\n"));
ok(<$r1>, ($errorID . ":  clear1\n"));
ok(<$r2>, ($errorID . ":  clear1\n"));

#
# Test for SetFormat.  Should re-write the format.
#
my $formatTestMsg = "Format test message.\n";
$log->Info($formatTestMsg);
my $oldFmtMsg = <$reader>;
ok($oldFmtMsg, <$r1>);

my $newfmt = new VMware::Log::Format::(1, 1, 1, 1);
my $oldfmt = $log->SetFormat($newfmt);

ok($oldfmt, $fmt);
ok(<$reader>, $newfmt->Description() . "\n");
ok(<$r1>, $newfmt->Description() . "\n");
ok(<$r2>, $newfmt->Description() . "\n");

$log->Info($formatTestMsg);

my $newFmtMsg = <$reader>;
ok ($newFmtMsg, <$r1>);
ok ($oldFmtMsg ne $newFmtMsg);

my $resetfmt = $log->SetFormat($oldfmt);
ok($resetfmt, $newfmt);
ok(<$reader>, $fmt->Description() . "\n");
ok(<$r1>, $fmt->Description() . "\n");
ok(<$r2>, $fmt->Description() . "\n");


#
# Get destinations on all channels.
#

my %fdhash = $log->GetDestinations();
my %infofds = %{$fdhash{$infoID}};
my %warnfds = %{$fdhash{$warnID}};
my %errorfds = %{$fdhash{$errorID}};
my %debugfds = %{$fdhash{$debugID}};
my %internalfds = %{$fdhash{$internalID}};

ok(scalar(keys %infofds), 2);
ok($infofds{"unnamed pipe"}, $writer);
ok($infofds{"pipe 1"}, $w1);
ok(scalar(keys %warnfds), 0);
ok(scalar(keys %errorfds), 3);
ok($errorfds{"unnamed pipe"}, $writer);
ok($errorfds{"pipe 1"}, $w1);
ok($errorfds{"pipe 2"}, $w2);
ok(scalar(keys %debugfds), 2);
ok($debugfds{"unnamed pipe"}, $writer);
ok($debugfds{"pipe 1"}, $w1);
ok(scalar(keys %internalfds), 2);
ok($internalfds{"unnamed pipe"}, $writer);
ok($infofds{"pipe 1"}, $w1);

#
# Test for duplicating a channel.  Note that
# no format description will be written because
# no new destinations are created for a dup.
#

$log->DuplicateChannel($errorID, "Dup", 0);
ok($log->Log("Dup", "Dup Message"), 1);
ok(<$reader>, "Dup:  Dup Message\n");
ok(<$r1>, "Dup:  Dup Message\n");
ok(<$r2>, "Dup:  Dup Message\n");

#
# Clear destinations on all channels.
#

%fdhash = $log->ClearDestinations();
%infofds = %{$fdhash{$infoID}};
%warnfds = %{$fdhash{$warnID}};
%errorfds = %{$fdhash{$errorID}};
%debugfds = %{$fdhash{$debugID}};
%internalfds = %{$fdhash{$internalID}};
my %dupfds = %{$fdhash{"Dup"}};

ok(scalar(keys %infofds), 2);
ok($infofds{"unnamed pipe"}, $writer);
ok($infofds{"pipe 1"}, $w1);
ok(scalar(keys %warnfds), 0);
ok(scalar(keys %errorfds), 3);
ok($errorfds{"unnamed pipe"}, $writer);
ok($errorfds{"pipe 1"}, $w1);
ok($errorfds{"pipe 2"}, $w2);
ok(scalar(keys %debugfds), 2);
ok($debugfds{"unnamed pipe"}, $writer);
ok($debugfds{"pipe 1"}, $w1);
ok(scalar(keys %internalfds), 2);
ok($internalfds{"unnamed pipe"}, $writer);
ok($infofds{"pipe 1"}, $w1);
ok(scalar(keys %dupfds), 3);
ok($dupfds{"unnamed pipe"}, $writer);
ok($dupfds{"pipe 1"}, $w1);
ok($dupfds{"pipe 2"}, $w2);

$log->Info("This goes nowhere");

#
# Duplicate empty channel.
# There's no real test for this- it just shouldn't 
# complian/die/etc.
#

ok($log->DuplicateChannel("Dup", "Dup2", 0));
$log->Log("Dup2", "This also goes nowhere");

#
# Tests for custom channels.
#

$log->CreateChannel("Test", $writer, "unnamed pipe", 0);
ok(<$reader>, ($fmt->Description() . "\n"));

ok($log->Log("Test", "Test Message"), 1);
ok(<$reader>, "Test:  Test Message\n");

#
# Test for the no format lines option.
#

my $noFormatLog =
   new VMware::Log::LogObj::($writer,
                             "nofmt",
                             $fmt,
                             VMware::Log::LogObj::ON_ERROR_FATAL,
                             1);
eval {
   ok($noFormatLog->Info("No format preceded this line."), 1);
   ok(<$reader>, ($infoID . ":  No format preceded this line.\n"));

   ok($noFormatLog->SetFormat(VMware::Log::Format->new()));
   ok($noFormatLog->Info("No format preceded this line."), 1);
   ok(<$reader>, ($infoID . ":  No format preceded this line.\n"));

   ok($noFormatLog->AddDestination($w1, "p1"), 1);
   ok($noFormatLog->Info("No format preceded this line."), 1);
   ok(<$reader>, ($infoID . ":  No format preceded this line.\n"));
   ok(<$r1>, ($infoID . ":  No format preceded this line.\n"));
};
ok($@, "", "Got exception while logging: $@");

close($writer) || die "Cannot close writer: $!\n";
close($w1) || die "Cannot close w1: $!\n";
close($w2) || die "Cannot close w2: $!\n";

ok(not defined <$reader>);
ok(not defined <$r1>);
ok(not defined <$r2>);

close($reader) || die "Cannot close reader: $!\n";
close($r1) || die "Cannot close r1: $!\n";
close($r2) || die "Cannot close r2: $!\n";


########################################################################
# Test using a filename as a destination.
########################################################################

my $testFile = File::Spec->catfile(dirname($0), "test.log");

$log = VMware::Log::LogObj->new($testFile,
                                "file test",
                                $fmt,
                                VMware::Log::LogObj::ON_ERROR_FATAL,
                                1);
my $message = "Test Message";
$log->Info($message);
$log->Error($message);

my $lines = FileSlurpArray($testFile);
foreach my $line (@$lines) {
   chomp $line;
}
ok(scalar(@$lines), 2);
ok($lines->[0] =~ /INFO :\s*$message$/, 1, "Got '$lines->[0]'");
ok($lines->[1] =~ /ERROR:\s*$message$/, 1, "Got '$lines->[1]'");

unlink($testFile) or die "Could not remove test log file: $!\n";


