##########################################################################
# Copyright 2005 VMware, Inc.  All rights reserved. -- VMware Confidential
##########################################################################

#
# Interface.pm --
#
#       This class models the /etc/sysconfig/network-scripts/ifcfg-*
#       and /etc/sysconfig network files, and in particular manipulates
#       common settings that matter to us (such as device,
#       portgroup name, gatewaydev, etc). Information can
#       be loaded from/saved to the files.
#
#       While this class only has specific accessors for certain
#       settings, it handles unknown settings without error which
#       is why it works for multiple types of file.
#

package VMware::Net::Interface;

use VMware::Log qw(:log);
use strict;

use constant BOOTPROTO_DHCP => "dhcp";
use constant BOOTPROTO_STATIC => "static";

########################################################################
#
# Interface::new --
#
#       Constructor for a VMware::Net::Interface object. If a filename
#       is specified, load the interface from that file.
#
# Results:
#       The new object, or undef on error.
#
# Side effects:
#       None.
#
########################################################################

sub new
{
   my $class = shift;      # IN: Invoking class.
   my $filename = shift;   # IN: Filename to load from (optional).

   my $self = {
      comment => undef,    # Interface comment from first line of file
      settings => {},      # Interface settings
      filename => $filename,  # Interface file (for reading, object
                              # may be written to another file).
   };
   
   bless $self => $class;

   if (defined $filename) {
      unless ($self->LoadInterface($filename)) {
         return undef;
      }
   }

   return $self;
}


########################################################################
#
# Interface::Copy --
#
#       Constructs a new Interface instance based on this instance.
#
# Results:
#       A new Interface identical to the original except for the filename,
#       which is not copied.
#
# Side effects:
#       None.
#
########################################################################

sub Copy
{
   my $self = shift;  # IN: Invoking instance (base for the copy).
   my $class = ref($self);

   my $copy = $class->new();
   foreach my $key (keys %{$self->{settings}}) {
      %{$copy->{settings}->{$key}} = %{$self->{settings}->{$key}};
   }
   $copy->{comment} = $self->{comment};

   return $copy;
}


########################################################################
#
# Interface::Get --
#
#       Retrieves a setting from the interface.
#
# Results:
#       The value of the setting, or "" if there is no such setting.
#
# Side effects:
#       None.
#
########################################################################

sub Get
{
   my $self = shift;    # IN: Invoking instance.
   my $setting = shift; # IN: The setting to retrieve.

   my $value = $self->{settings}->{$setting}->{value};
   return defined($value) ? $value : "";
}


########################################################################
#
# Interface::Set --
#
#       Sets the value of a setting from the interface. If the value
#       is undef. the setting will not be saved when the interface is
#       written out to a file.
#
# Results:
#       None.
#
# Side effects:
#       None.
#
########################################################################

sub Set
{
   my $self = shift;    # IN: Invoking instance.
   my $setting = shift; # IN: The setting to retrieve.
   my $value = shift;   # IN: The new setting value.

   if (defined $value) {
      $self->{settings}->{$setting}->{value} = $value;
   } else {
      my $entry = $self->{settings}->{$setting};
      if (defined $entry && defined $entry->{idx}) {
         $self->{contents}->[$entry->{idx}] = undef;
      }

      delete $self->{settings}->{$setting};
   }
}


########################################################################
#
# Interface::Rename --
#
#       Renames a setting in the interface. The new setting will
#       have the same value as the old setting.
#
# Results:
#       None.
#
# Side effects:
#       Removes the old setting key from the interface, and creates
#       a new setting key. May clobber the new setting key if told
#       to do so.
#
########################################################################

sub Rename 
{
   my $self = shift;    # IN: Invoking instance.
   my $old = shift;     # IN: The old setting name.
   my $new = shift;     # IN: The new setting name.
   my $clobber = shift; # IN: Whether or not to clobber an existing setting
                        # of the new name. If false, will preserve
                        # the setting associated with the new name if its
                        # value is a non-empty string.
   
   if (not $clobber and
       $self->Get($new) ne "") {
      return;
   }

   my $value = $self->Get($old);
   $self->Set($new, $value);
   $self->Set($old, undef);
}


#
# Trivial Accessor Functions
#

sub GetDevice
{
   my $self = shift;
   return $self->Get('DEVICE');
}

sub SetDevice
{
   my $self = shift;
   my $value = shift;
   return $self->Set('DEVICE', $value);
}

sub GetPortGroup
{
   my $self = shift;
   return $self->Get('PORTGROUP');
}

sub SetPortGroup
{
   my $self = shift;
   my $value = shift;
   return $self->Set('PORTGROUP', $value);
}

sub GetGatewayDev
{
   my $self = shift;
   return $self->Get('GATEWAYDEV');
}

sub SetGatewayDev
{
   my $self = shift;
   my $value = shift;
   return $self->Set('GATEWAYDEV', $value);
}
   
sub GetAutogenerated
{
   my $self = shift;
   return $self->Get('AUTOGENERATED');
}

sub SetAutogenerated
{
   my $self = shift;
   my $value = shift;
   return $self->Set('AUTOGENERATED', $value);
}

sub GetOnBoot
{
   my $self = shift;
   return $self->Get('ONBOOT');
}

sub SetOnBoot
{
   my $self = shift;
   my $value = shift;
   return $self->Set('ONBOOT', $value);
}

sub GetBootProto
{
   my $self = shift;
   return $self->Get('BOOTPROTO');
}

sub SetBootProto
{
   my $self = shift;
   my $value = shift;
   return $self->Set('BOOTPROTO', $value);
}

sub GetIPAddr
{
   my $self = shift;
   return $self->Get('IPADDR');
}

sub SetIPAddr
{
   my $self = shift;
   my $value = shift;
   return $self->Set('IPADDR', $value);
}

sub GetNetmask
{
   my $self = shift;
   return $self->Get('NETMASK');
}

sub SetNetmask
{
   my $self = shift;
   my $value = shift;
   return $self->Set('NETMASK', $value);
}

sub GetBroadcast
{
   my $self = shift;
   return $self->Get('BROADCAST');
}

sub SetBroadcast
{
   my $self = shift;
   my $value = shift;
   return $self->Set('BROADCAST', $value);
}

# Alias for GetDevice
sub GetName
{
   my $self = shift;
   return $self->GetDevice();
}

# Alias for SetDevice
sub SetName
{
   my $self = shift;
   my $value = shift;
   return $self->SetDevice($value);
}

sub GetNumber
{
   my $self = shift;
   return ($self->GetName() =~ /(\d+)$/)[0];
}

sub GetComment
{
   my $self = shift;
   return $self->{comment};
}

sub SetComment
{
   my $self = shift;
   my $value = shift;
   $self->{comment} = $value;
}


#
# Filename cannot be set.  An alternate filename can be supplied
# to SaveInterface to write the data to a different location.
#

sub GetFilename
{
   my $self = shift;
   return $self->{filename};
}


########################################################################
#
# LoadInterface --
#       Load an interface file. If the first line of the file is a
#       comment, also sets the interface comment to the comment text.
#
# Results:
#       1 for success, 0 otherwise.
#
# Side effects:
#       Reads the specified interface file, and resets all settings in
#       the Interface object to those in the file.
#
########################################################################

sub LoadInterface
{
   my $self = shift;       # IN: Invoking instance.
   my $filename = shift;   # IN: The interface file to read.

   my $fh;
   unless (open($fh, "<$filename")) {
      LogError("Could not open file $filename: $!");
      return 0;
   }

   my $comment = undef;
   my %settings = ();

   my $lineno = -1;

   while (<$fh>) {
      my $line = $_;
      $lineno++;

      push(@{$self->{contents}}, $line);
      # Remove newline and comments from this copy.
      chomp($line);
      if ($lineno == 0 and $line =~ /^#\s+(.*)/) {
         $comment = $1;
         next;
      } else {
         $line =~ s/\#.*//;
      }

      if ($line =~ /^(\w+)\=(.*)$/) {
         my ($key, $value) = ($1, $2);
         $value =~ s/^"([^"]*)"$/$1/;
         $settings{$key}->{value} = $value;
         $settings{$key}->{idx} = $lineno;
      }
   }

   unless (close($fh)) {
      LogError("Could not close file $filename: $!");
      return 0;
   }

   $self->SetComment($comment);
   $self->{settings} = \%settings;
   return 1;
}


########################################################################
#
# SaveInterface --
#       Writes an interface to the specified file.  If no file
#       is supplied, writes it back to the file from which it was read.
#
# Results:
#       On success, writes out the file and returns 1.
#       Returns 0 otherwise.
#
# Side effects:
#       Outputs all settings to the specified file, overwriting it
#       if necessary.
#
########################################################################

sub SaveInterface
{
   my $self = shift;       # IN: Invoking instance.
   my $filename = shift;   # IN: The interface file to write.

   $filename = $self->{filename} if not defined $filename;

   my $fh;
   unless (open($fh, ">$filename")) {
      LogError("Could not open file $filename: $!");
      return 0;
   }

   my $comment = $self->GetComment();
   if ($comment) {
      print $fh "# $comment\n";
   }

   my @tail = ();
   foreach my $key (sort(keys %{$self->{settings}})) {
      my $value = $self->Get($key);
      if ($value =~ / /) {
         $value = "\"$value\"";
      }

      #
      # Specific autogenerated lines will not retain comments,
      # but that seems to be reasonable behavior, and we preserve
      # all other comments if people really want to write notes.
      # New settings will lack an index, and can go at the end.
      #

      my $line = "$key=$value\n";
      my $idx = $self->{settings}->{$key}->{idx};
      if (defined $idx) {
         $self->{contents}->[$idx] = $line;
      } else {
         push(@tail, $line);
      }
   }
   push(@{$self->{contents}}, @tail);

   foreach my $line (@{$self->{contents}}) {
      #
      # We can get an undef if we deleted a line out.  That's OK.
      #
      if (defined $line) {
         print $fh $line;
      }
   }

   unless (close($fh)) {
      LogError("Could not open file $filename: $!");
      return 0;
   }

   return 1;
}

1;
