##########################################################################
# Copyright 2005 VMware, Inc.  All rights reserved. -- VMware Confidential
##########################################################################

#
# NetworkManager.pm --
#
#       This class manages the network config tree and the
#       system network interfaces listed in the network-scripts
#       directory.

package VMware::Net::NetworkManager;

use VMware::Log qw(:log);
use VMware::FileSys::StandardLocator qw(StdFile StdDir);
use VMware::Net::Interface;
use VMware::Net::VSwitch;
use VMware::Net::VMKNic;
use VMware::Net::PortGroup;

use strict;


########################################################################
#
# NetworkManager::new --
#
#       Constructor.  Optionally takes the configuration tree
#       representing all networking info from the main config file and builds
#       its internal set of network objects from the tree.
#
# Results:
#       The new instance, or undef on error.
#
# Side effects:
#       None.
#
########################################################################

sub new
{
   my $class = shift;            # IN: Invoking class name.
   my $netcfg = (shift || {});   # IN: Config tree, defaults to empty.

   my $self = {
      interfaces => {},          # Interface (ifcfg) objects from disk
      addedInterfaces => {},     # Interfaces to save to disk  
      removedInterfaces => {},   # Interfaces to remove from disk 
      networkFile => undef,      # Contents of /etc/syconfig/network
      vswifs => [],              # A list of keys into interfaces
      vmknics => [],             # VMkernel nic objecs
      vswitches => [],           # VSwitch objects
      vswitchNameSequence => 0,  # Counter for assigning vSwitchN names.
   };
   bless $self => $class;

   #
   # Preserve additional config fields by storing the whole tree.
   # We will selectively override the portions that we manage specially
   # before retrieving the configuration data, so any unknown settings
   # will pass through unchanged.
   #

   $self->{cfgTree} = $netcfg;

   $self->LoadInterfaces() or return undef;
   $self->LoadNetworkFile() or return undef;
   $self->LoadVswifs() or return undef;
   $self->LoadVSwitches() or return undef;
   $self->LoadVMKNics() or return undef;

   return $self;
}


########################################################################
#
# NetworkManager::LoadInterfaces --
#
#       Private method used by the constructor. Loads all interfaces
#       in the network-scripts directory.
#
# Results:
#       1 on success, 0 otherwise.
#
# Side effects:
#       Reads interface files from disk into the interfaces member.
#
########################################################################

sub LoadInterfaces
{
   my $self = shift;    # IN: Invoking instance.

   my $ifcfgDir = StdDir("network_scripts", "all");
   my @filenames = glob("$ifcfgDir/ifcfg-*");
   foreach my $filename (@filenames) {
      unless (-r $filename) {
         # XXX: Warn here? By default Escher upgrade will have a
         # broken symlink here.
         next;
      }

      my $interface = new VMware::Net::Interface($filename);
      if ($interface) {
         $self->{interfaces}->{$interface->GetDevice()} = $interface;
      } else {
         LogWarn("Could not load interface $filename");
         next;
      }
   }

   return 1;
}


########################################################################
#
# NetworkManager::LoadNetworkFile --
#
#       Private method used by the constructor. Loads the
#       /etc/sysconfig/network file as an Interface object (the format
#       is the same).
#
# Results:
#       1 on success, 0 otherwise.
#
# Side effects:
#       Reads interface files from disk into the interfaces member.
#
########################################################################

sub LoadNetworkFile
{
   my $self = shift;  # IN/OUT: Invoking instance.

   my $filename = StdFile("network", "all");
   $self->{networkFile} = new VMware::Net::Interface($filename);
   unless ($self->{networkFile}) {
      LogWarn("Could not load $filename");
   }
}


########################################################################
#
# NetworkManager::LoadVswifs --
#
#       Private method used by the constructor. Load the list of vswif
#       names from the config tree, and validate it against the list
#       of known interfaces.
#
# Results:
#       1 on success, 0 otherwise.
#
# Side effects:
#       Initializes the vswifs member to the list of valid vswif names.
#
########################################################################

sub LoadVswifs
{
   my $self = shift;    # IN/OUT: Invoking instance.

   unless (defined $self->{cfgTree}->{vswifs}) {
      return 1;
   }

   foreach my $key (keys %{$self->{cfgTree}->{vswifs}}) {
      if (defined $self->{interfaces}->{$key}) {
         push @{$self->{vswifs}}, $key;
      } else {
         LogWarn("Vswif $key found in config but not on disk");
         next;
      }
   }
   return 1;
}


########################################################################
#
# NetworkManager::LoadVSwitches --
#
#       Private method used by the constructor. Load the list of
#       vswitches from the config tree.
#
# Results:
#       1 on success, 0 otherwise.
#
# Side effects:
#       Initializes the vswitches member to the list of vswitch objects.
#
########################################################################

sub LoadVSwitches
{
   my $self = shift;    # IN/OUT: Invoking instance.

   unless (defined $self->{cfgTree}->{vswitch}->{child}) {
      return 1;
   }

   my $maxVSwitchNum = -1;
   foreach my $child (@{$self->{cfgTree}->{vswitch}->{child}}) {
      # Skip empty indicies, but maintain index order.
      unless (defined $child) {
         push @{$self->{vswitches}}, undef;
         next;
      }

      my $currVSwitchNum = ($child->{name} =~ /^vSwitch(\d+)$/)[0];
      if (defined($currVSwitchNum) && ($currVSwitchNum > $maxVSwitchNum)) {
         $maxVSwitchNum = $currVSwitchNum;
      }

      my $vswitch = new VMware::Net::VSwitch($child);
      unless ($vswitch) {
         LogError("Could not construct vswitch from config.");
         return 0;
      }

      push @{$self->{vswitches}}, $vswitch;
   }

   $self->{vswitchNameSequence} = $maxVSwitchNum + 1;
   return 1;
}


########################################################################
#
# NetworkManager::LoadVMKNics --
#
#       Private method used by the constructor.  Load the list of
#       vmkernel nics from the config tree.
#
# Results:
#       1 on success, 0 otherwise.
#
# Side effects:
#       Initializes the vmknics member ot the list of vmknic objects.
#
########################################################################

sub LoadVMKNics
{
   my $self = shift;  # IN/OUT: Invoking instance.

   unless (defined $self->{cfgTree}->{vmkernelnic}->{child}) {
      return 1;
   }

   foreach my $child (@{$self->{cfgTree}->{vmkernelnic}->{child}}) {
      # Skip empty indicies, but maintain index order.
      unless (defined $child) {
         push @{$self->{vmknics}}, undef;
         next;
      }

      my $vmknic = new VMware::Net::VMKNic($child);
      unless ($vmknic) {
         LogError("Could not construct vmknic from config.");
         return 0;
      }

      push @{$self->{vmknics}}, $vmknic;
   }

   return 1;
}


########################################################################
#
# NetworkManager::GetVswifs --
#
#       List the vswifs on the system.
#
# Results:
#       Returns a list of VMware::Net::Interface objects corresponding
#       to the system vswifs. The list will be sorted in increasing order
#       of vswif number.
#
# Side effects:
#       None.
#
########################################################################

sub GetVswifs
{
   my $self = shift;    # IN: Invoking instance.

   # Vswifs are stored as keys into the interfaces hash.
   my @vswifs = map { $self->{interfaces}->{$_} } @{$self->{vswifs}};

   # Sort by number (ie vswif0 < vswif1)
   @vswifs = sort { $a->GetNumber() <=> $b->GetNumber() } @vswifs;

   return @vswifs;
}


########################################################################
#
# NetworkManager::GetAutoFilenameForInterface --
#
#       Determine a suitable filename for an interface object based on its
#       name. This is a private method used for vswif creation.
#
# Results:
#       The filename for the interface.
#
# Side effects:
#       None.
#
########################################################################

sub GetAutoFilenameForInterface
{
   my $self = shift;       # IN: Invoking instance.
   my $interface = shift;  # IN: The VMware::Net::Interface in question

   my $ifcfgDir = StdDir("network_scripts", "all");
   return "$ifcfgDir/ifcfg-" . $interface->GetDevice();
}


########################################################################
#
# NetworkManager::AddVswif --
#
#       Tell the network manager that it should track a new vswif
#       amongst its interfaces.
#
# Results:
#       True on success, false on failure.
#
# Side effects:
#       Adds a vswif to the tracking list.
#
########################################################################

sub AddVswif
{
   my $self = shift;   # IN/OUT: Invoking instance.
   my $vswif = shift;  # IN: VMware::Net::Interface for the vswif to add.

   unless ($self->AddInterface($vswif)) {
      return 0;
   }

   my $name = $vswif->GetName();
   my $filename = $self->GetAutoFilenameForInterface($vswif);
   $self->{cfgTree}->{vswifs}->{$name} = $filename;
   push @{$self->{vswifs}}, $name;

   return 1;
}


########################################################################
#
# NetworkManager::RemoveVswif --
#
#       Tell the network manager not to track a particular
#       vswif among its interfaces anymore.  If it was not tracking
#       the vswif this is a silent noop.
#
# Results:
#       None.
#
# Side effects:
#       Removes a vswif from the tracking list.
#
########################################################################

sub RemoveVswif
{
   my $self = shift;   # IN/OUT: Invoking instace.
   my $name = shift;  # IN: Vswif name (i.e. 'vswif0') to remove.

   $self->RemoveInterface($name);

   for (my $i = 0; $i < @{$self->{vswifs}}; ++$i) {
      if ($self->{vswifs}->[$i] eq $name) {
         splice(@{$self->{vswifs}}, $i, 1);
      }
   }
   delete $self->{cfgTree}->{vswifs}->{$name};
}


########################################################################
#
# NetworkManager::GetVSwitches --
#
#       List the virtual switches on the system.
#
# Results:
#       Returns a list of VMware::Net::VSwitch objects corresponding
#       to the system virtual switches. The list will be sorted in
#       increasing order of virtual switch number.
#
# Side effects:
#       None.
#
########################################################################

sub GetVSwitches
{
   my $self = shift;    # IN: Invoking instance.

   # The vswitch list can have undef placeholders, filter them out.
   my @vswitches = grep { defined $_ } @{$self->{vswitches}};
   @vswitches = sort { $a->GetNumber() <=> $b->GetNumber() } @vswitches;
   return @vswitches;
}


########################################################################
#
# NetworkManager::GetVSwitchByPortGroup --
#
#       Find the virtual switch which has the given portgroup name.
#
# Results:
#       Returns the VSwitch object if one is found, or undef
#       otherwise.
#
# Side effects:
#       None.
#
########################################################################

sub GetVSwitchByPortGroup
{
   my $self = shift;          # IN: Invoking instance.
   my $name = shift;          # IN: PortGroup name
   my $matches = [];

   foreach my $vswitch ($self->GetVSwitches()) {
      my @portgroups = $vswitch->GetPortGroups(); 
      if (grep { $name eq $_->GetName() } @portgroups) {
         push @$matches, $vswitch;
      }
   }

   #
   # There should be only one or zero, but if things are sufficiently wierd
   # that we have duplicates, a warning and returning the first one seems
   # like the best option.
   # XXX: Is this really true?
   #

   if (scalar(@$matches) > 1) {
      LogWarn("Multiple VSwitches have a port group with the name '$name'." .
              "This should not be possible.  Returning first match.");
   }

   return $matches->[0];
}


########################################################################
#
# NetworkManager::GetPortGroupByName --
#
#       Find the portgroup which has the given name.
#
# Results:
#       Returns the portgroup object if one is found, or undef
#       otherwise.
#
# Side effects:
#       None.
#
########################################################################

sub GetPortGroupByName
{
   my $self = shift;    # IN: Invoking instance.
   my $name = shift;    # IN: PortGroup name

   foreach my $vswitch ($self->GetVSwitches()) {
      foreach my $portgroup ($vswitch->GetPortGroups()) {
         if ($portgroup->GetName() eq $name) {
            return $portgroup;
         }
      }
   }

   return undef;
}


########################################################################
#
# NetworkManger::AddVSwitch --
#
#       Add a VSwitch object to the networking configuration.
#       The object is assumed to have a sane configuration,
#       (no duplicate names, etc.) so this method does not check it.
#
# Results:
#       None.
#
# Side effects:
#       Adds the vswitch to @self->{vswitches}.
#
########################################################################

sub AddVSwitch
{
   my $self = shift;  # IN: Invoking instance.
   my $vswitch = shift;  # IN: VMware::Net::VSwitch instance to add.

   push(@{$self->{vswitches}}, $vswitch);

   my $num = $vswitch->GetNumber();
   if ($num > $self->{vswitchNameSequence}) {
      $self->{vswitchNameSequence} = $num;
   }
}


########################################################################
#
# NetworkManger::GetNextVSwitchName --
#
#       Determine and return the next available vswitch name
#       (i.e. vSwitch0, vSwitch1, etc.) and increment the count.
#
# Results:
#       A vswitch name that can be given to a vswitch that will be
#       added to this network manager.
#
# Side effects:
#       Increments our running vswitch name count, which is local
#       to this object and not stored in the config data.
#
########################################################################

sub GetNextVSwitchName
{
   my $self = shift;  # IN/OUT: Invoking instance.

   return VMware::Net::VSwitch::VSWITCH_NAME_BASE .
          $self->{vswitchNameSequence}++;
}


########################################################################
#
# NetworkManager::GetVMKNics --
#
#       Returns a list of VMKNic objects representing the configured
#       vmkernel network adapters.
#
# Results:
#       The list of adapters.
#
# Side effects:
#       None.
#
########################################################################

sub GetVMKNics
{
   my $self = shift;    # IN: Invoking instance.

   # The vmknic list can have undef placeholders, filter them out.
   return grep { defined $_ } @{$self->{vmknics}};
}


########################################################################
#
# NetworkManager::AddVMKNic --
#
#       Adds a VMkernel nic.  This method does not verify the
#       configuration of the supplied VMKNic.
#
# Results:
#       None.
#
# Side effects:
#       Adds a VMKNic.
#
########################################################################

sub AddVMKNic
{
   my $self = shift; # IN/OUT: Invoking instance.
   my $vmknic = shift;  # IN: VMware::Net::VMKNic instance to add.

   push(@{$self->{vmknics}}, $vmknic);
}


########################################################################
#
# NetworkManager::SetVMKernelGateway --
#
#       Set the gateway for vmkernel network adapters.
#       This is not part of the VMKNic object, rather it goes under
#       /routes so we handle it here directly.
#
# Results:
#       None.
#
# Side effects:
#       Adds a vmkernel route to the config data.
#
########################################################################

sub SetVMKernelGateway
{
   my $self = shift; # IN/OUT: Invoking instance.
   my $gateway = shift; # IN: Gateway IP address string.

   $self->{cfgTree}->{routes}->{kernel}->{gateway} = $gateway;
}


########################################################################
#
# NetworkManager::GetInterface --
#
#       Find the interface of the given name.
#
# Results:
#       Returns the Interface object if one is found, or undef
#       otherwise.
#
# Side effects:
#       None.
#
########################################################################

sub GetInterface
{
   my $self = shift;    # IN: Invoking instance.
   my $name = shift;    # IN: Interface name

   return $self->{interfaces}->{$name};
}


########################################################################
#
# NetworkManager::AddInterface --
#
#       Add a new network interface to the system.
#
# Results:
#       Returns 1 if the interface was added successfully, or 0 if there
#       was an error or if the interface already exists.
#
# Side effects:
#       None.
#
########################################################################

sub AddInterface
{
   my $self = shift;       # IN: Invoking instance.
   my $interface = shift;  # IN: Interface to add

   my $device = $interface->GetDevice();
   if (defined $self->{interfaces}->{$device}) {
      LogError( "Attempting to add interface $device which already exists");
      return 0;
   }

   # Queue for saving, and remove from the deletion queue
   delete $self->{removedInterfaces}->{$device};
   $self->{addedInterfaces}->{$device} = $interface;

   $self->{interfaces}->{$device} = $interface;
   return 1;
}


########################################################################
#
# NetworkManager::RemoveInterface --
#
#       Removes a new network interface from the system.
#
# Results:
#       Returns 1 if the interface was removed successfully, or 0 if there
#       was an error or if there is no such interface.
#
# Side effects:
#       None.
#
########################################################################

sub RemoveInterface
{
   my $self = shift;       # IN: Invoking instance.
   my $name = shift;       # IN: Name of the interface to remove

   my $interface = $self->{interfaces}->{$name};
   unless (defined $interface) {
      LogError("Attempting to remove interface $name which does not exist");
      return 0;
   }

   # Queue for deleting, and remove from the save queue
   delete $self->{addedInterfaces}->{$name};
   $self->{removedInterfaces}->{$name} = $interface;

   delete $self->{interfaces}->{$name};
   return 1;
}


########################################################################
#
# NetworkManager::GetInterfaces --
#
#       List the network interfaces on the system.
#
# Results:
#       Returns a list of VMware::Net::Interface objects corresponding
#       to the system network interfaces. The list will be sorted
#       alphabetically by interface name.
#
# Side effects:
#       None.
#
########################################################################

sub GetInterfaces
{
   my $self = shift;       # IN: Invoking instance.

   my @interfaces = values %{$self->{interfaces}};
   @interfaces = sort { $a->GetName() cmp $b->GetName() } @interfaces;
   return @interfaces;
}


########################################################################
#
# NetworkManager::CommitInterfaces --
#
#       Commits pending interface changes to disk.
#
# Results:
#       1 for success, 0 otherwise.
#
# Side effects:
#       For each added interface, creates an ifcfg file for the
#       interface and adds it to the network-scripts directory.
#       For each removed interface, deletes the interface ifcfg
#       file from the network-scripts directory.
#
########################################################################

sub CommitInterfaces
{
   my $self = shift;       # IN: Invoking instance.

   # Remove interfaces
   foreach my $interface (values %{$self->{removedInterfaces}}) {
      my $filename = $self->GetAutoFilenameForInterface($interface);
      unless (unlink($filename)) {
         LogError("Failed to remove $filename: $!");
         return 0;
      }
   }

   # Add interfaces
   foreach my $interface (values %{$self->{addedInterfaces}}) {
      my $filename = $self->GetAutoFilenameForInterface($interface);
      unless ($interface->SaveInterface($filename)) {
         return 0;
      }
   }

   return 1;
}


########################################################################
#
# NetworkManager::GetNetworkFile --
#
#       Get a reference to the Interface object representing
#       /etc/sysconfig/network.
#
# Results:
#       The reference, which may be undef if the Interface object
#       could not be constructed.
#
# Side effects:
#       None.
#
########################################################################

sub GetNetworkFile
{
   my $self = shift;  # IN: Invoking instance.
   return $self->{networkFile};
}


########################################################################
#
# NetworkManager::CommitNetworkFile --
#
#       Commits pending network file changes to disk.
#
# Results:
#       1 for success, 0 otherwise.
#
# Side effects:
#       Writes data back to the /etc/sysconfig/network file.
#
########################################################################

sub CommitNetworkFile
{
   my $self = shift;  # IN: Invoking instance.

   return 0 unless defined $self->{networkFile};
   return $self->{networkFile}->SaveInterface(StdFile("network", "all"));
}


########################################################################
#
# NetworkManager::GetConfigData --
#
#       Return the config file structure representing the managed
#       network settings.
#
# Results:
#       The structure (see VMware::Config::VmacoreConfigObj).
#
# Side effects:
#       Calls GetConfigData() on all lower life forms.
#
########################################################################

sub GetConfigData
{
   my $self = shift;  # IN/OUT: Invoking instance.

   my @vswitches = ();
   foreach my $vswitch (@{$self->{vswitches}}) {
      if (defined $vswitch) {
         push(@vswitches, $vswitch->GetConfigData());
      } else {
         push(@vswitches, undef);
      }
   }
   $self->{cfgTree}->{vswitch}->{child} = \@vswitches;

   my @vmknics = ();
   foreach my $vmknic (@{$self->{vmknics}}) {
      if (defined $vmknic) {
         push(@vmknics, $vmknic->GetConfigData());
      } else {
         push(@vmknics, undef);
      }
   }
   $self->{cfgTree}->{vmkernelnic}->{child} = \@vmknics;

   return $self->{cfgTree};
}

1;
