##########################################################################
# Copyright 2005 VMware, Inc.  All rights reserved. -- VMware Confidential
##########################################################################

#
# PortGroup.pm --
#
#       A basic class that contains information representing a portgroup.
#       Information can be loaded from a configuration tree.
#

package VMware::Net::PortGroup;

use VMware::Net::TeamPolicy;

use strict;

use constant PORT_GROUP_NAME_BASE => "portgroup";

########################################################################
#
# PortGroup::new --
#
#       Constructor for a VMware::Net::PortGroup object.
#
# Results:
#       The new object, or undef on error.
#
# Side effects:
#       None.
#
########################################################################

sub new
{
   my $class = shift;            # IN: Invoking class.
   my $cfgTree = (shift || {});  # IN: Tree structure of data from the
                                 #     config file.

   my $self = {
      cfgTree => $cfgTree,
   };
   if (defined $cfgTree->{teamPolicy}) {
      $self->{teamPolicy} = new VMware::Net::TeamPolicy($cfgTree->{teamPolicy});
   }

   bless $self => $class;
   return $self;
}


########################################################################
#
# PortGroup::GetConfigData --
#
#       Get the config information for this port group in the
#       vmacore config file data structure format.
#
# Results:
#       The data structure.
#
# Side effects:
#       None.
#
########################################################################

sub GetConfigData
{
   my $self = shift; # IN: Invoking instance.

   if (defined $self->{teamPolicy}) {
      $self->{cfgTree}->{teamPolicy} = $self->{teamPolicy}->GetConfigData();
   }
   return $self->{cfgTree};
}


#
# Trivial Accessor Functions
#

sub GetName
{
   my $self = shift;
   return $self->{cfgTree}->{name};
}

sub GetVlanId
{
   my $self = shift;
   return $self->{cfgTree}->{vlanId};
}

sub GetTeamPolicy
{
   my $self = shift;
   return $self->{teamPolicy};
}


#
# Trivial Mutator Functions
#

sub SetName
{
   my $self = shift;
   $self->{cfgTree}->{name} = shift;
}

sub SetVlanId
{
   my $self = shift;
   $self->{cfgTree}->{vlanId} = shift;
}

########################################################################
#
# PortGroup::SetTeamPolicy --
#
#       Sets the TeamPolicy for the specific PortGroup.  This
#       overrides any TeamPolicy setting at the VSwitch level.
#       Setting the TeamPolicy completely replaces all TeamPolicy
#       configuration for this PortGroup.  If you want to change
#       part of the policy, call GetTeamPolicy and edit the TeamPolicy
#       object.
#
# Results:
#       None.
#
# Side effects:
#       Sets TeamPolicy.
#
########################################################################

sub SetTeamPolicy
{
   my $self = shift;  # IN/OUT: Invoking instance.
   my $policy = shift;  # IN: New VMware::Net::TeamPolicy.

   $self->{teamPolicy} = $policy;
}


1;

