##########################################################################
# Copyright 2005 VMware, Inc.  All rights reserved. -- VMware Confidential
##########################################################################

#
# TeamPolicy.pm --
#
#       This class models the team policy information that can
#       be associated with either a VSwitch or a PortGroup.
#

package VMware::Net::TeamPolicy;

use VMware::Log qw(:log);
use VMware::Net::Uplink;

use strict;

use constant TEAM_FO_EXPLICIT => "fo_explicit";
use constant TEAM_LB_SRCID => "lb_srcid";
use constant TEAM_LB_SRCMAC => "lb_srcmac";
use constant TEAM_LB_IP => "lb_ip";


########################################################################
#
# TeamPolicy::new --
#
#       Constructor for a VMware::Net::TeamPolicy object.
#
# Results:
#       The new object, or undef on error.
#
# Side effects:
#       None.
#
########################################################################

sub new
{
   my $class = shift;            # IN: Invoking class.
   my $cfgTree = (shift || {});  # IN: Tree structure of data from the
                                 #     config file.

   my $self = {
      cfgTree => $cfgTree,
      uplinks => [],
   };

   #
   # Yes, uplinks itself is a list, not uplinks->{child}.
   # This is different from VSwitch uplinks.
   #

   if (defined $cfgTree->{uplinks}) {
      foreach my $child (@{$cfgTree->{uplinks}}) {
         # Skip empty indicies, but maintain index order.
         unless (defined $child) {
            push @{$self->{uplinks}}, undef;
            next;
         }

         my $uplink = new VMware::Net::Uplink($child);
         if ($uplink) {
            push @{$self->{uplinks}}, $uplink;
         } else {
            LogError("Could not build uplink object for team policy.");
            return undef;
         }
      }
   }
   $self->{maxActive} = 1;
  
   bless $self => $class;
   return $self;
}


########################################################################
#
# TeamPolicy::GetConfigData --
#
#       Return a config file data structure of this object's info.
#
# Results:
#       The data structure.
#
# Side effects:
#       Updates $self->{cfgTree} from other data members.
#
########################################################################

sub GetConfigData
{
   my $self = shift;  # IN/OUT: Invoking instance.

   $self->{cfgTree}->{uplinks} = [
      map { defined($_) ? $_->GetConfigData() : undef; } @{$self->{uplinks}}
   ];
   $self->{cfgTree}->{maxActive} = $self->{maxActive};

   return $self->{cfgTree};
}


########################################################################
#
# TeamPolicy::GetTeam --
#
#       Get the "team" (load balance/failover) setting in the policy.
#       Valid values for this setting are given by the TEAM_ constants.
#
# Results:
#       The team setting.
#
# Side effects:
#       None.
#
########################################################################

sub GetTeam
{
   my $self = shift; # IN: Invoking instance.
   return $self->{cfgTree}->{team};
}


########################################################################
#
# TeamPolicy::GetUplinks --
#
#       List the uplinks in the team.
#
# Results:
#       Returns a list of VMware::Net::Uplink objects corresponding
#       to the virtual switch uplinks. The list will be sorted in
#       increasing order of pnic number.
#
# Side effects:
#       None.
#
########################################################################

sub GetUplinks
{
   my $self = shift; # IN: Invoking instance.

   # The uplink list can have undef placeholders, filter them out.
   my @uplinks = grep { defined $_ } @{$self->{uplinks}};
   @uplinks = sort { $a->GetPnicNumber() <=> $b->GetPnicNumber() } @uplinks;
   return @uplinks;
}


########################################################################
#
# TeamPolicy::SetTeam --
#
#    Sets the "team" entry of the uplink policy.  This entry
#    actually determines how the team load balances and/or fails over.
#    The TEAM_ constants in this module give the valid values.
#
# Results:
#    None.
#
# Side effects:
#    Sets the team entry in $self->{cfgTree}.
#
########################################################################

sub SetTeam
{
   my $self = shift; # IN/OUT: Invoking instance.
   $self->{cfgTree}->{team} = shift;  # IN: The new team value.
}


########################################################################
#
# TeamPolicy::AddUplink --
#
#       Add an Uplink object to this TeamPolicy.
#
# Results:
#       None.
#
# Side effects:
#       Adds an uplink.
#
########################################################################

sub AddUplink
{
   my $self = shift; # IN/OUT: Invoking instance.
   my $uplink = shift; # IN: VMware::Net::Uplink instance.
   my $first = shift;  # IN: Make this the first (index 0) uplink.
                       #     This can be important for failover order.

   if ($first) {
      unshift(@{$self->{uplinks}}, $uplink);
      $self->{maxActive} = 1;
   } else {
      push(@{$self->{uplinks}}, $uplink);
   }
}


1;
